#if defined(QJS_BUILD_LIBC) && defined(__linux__) && !defined(_GNU_SOURCE)
#define _GNU_SOURCE
#endif
/*
 * QuickJS C atomics definitions
 *
 * Copyright (c) 2023 Marcin Kolny
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#if (defined(__GNUC__) || defined(__GNUG__)) && !defined(__clang__)
   // Use GCC builtins for version < 4.9
#  if((__GNUC__ << 16) + __GNUC_MINOR__ < ((4) << 16) + 9)
#    define GCC_BUILTIN_ATOMICS
#  endif
#endif

#ifdef GCC_BUILTIN_ATOMICS
#define atomic_fetch_add(obj, arg) \
    __atomic_fetch_add(obj, arg, __ATOMIC_SEQ_CST)
#define atomic_compare_exchange_strong(obj, expected, desired) \
    __atomic_compare_exchange_n(obj, expected, desired, 0, __ATOMIC_SEQ_CST, __ATOMIC_SEQ_CST)
#define atomic_exchange(obj, desired) \
    __atomic_exchange_n (obj, desired, __ATOMIC_SEQ_CST)
#define atomic_load(obj) \
    __atomic_load_n(obj, __ATOMIC_SEQ_CST)
#define atomic_store(obj, desired) \
    __atomic_store_n(obj, desired, __ATOMIC_SEQ_CST)
#define atomic_fetch_or(obj, arg) \
    __atomic_fetch_or(obj, arg, __ATOMIC_SEQ_CST)
#define atomic_fetch_xor(obj, arg) \
    __atomic_fetch_xor(obj, arg, __ATOMIC_SEQ_CST)
#define atomic_fetch_and(obj, arg) \
    __atomic_fetch_and(obj, arg, __ATOMIC_SEQ_CST)
#define atomic_fetch_sub(obj, arg) \
    __atomic_fetch_sub(obj, arg, __ATOMIC_SEQ_CST)
#define _Atomic
#else
#include <stdatomic.h>
#endif
/*
 * C utilities
 *
 * Copyright (c) 2017 Fabrice Bellard
 * Copyright (c) 2018 Charlie Gordon
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#ifndef CUTILS_H
#define CUTILS_H

#include <stdbool.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include <math.h>

#ifdef __cplusplus
extern "C" {
#endif

#if defined(_MSC_VER)
#include <winsock2.h>
#include <malloc.h>
#define alloca _alloca
#define ssize_t ptrdiff_t
#endif
#if defined(__APPLE__)
#include <malloc/malloc.h>
#elif defined(__linux__) || defined(__ANDROID__) || defined(__CYGWIN__) || defined(__GLIBC__)
#include <malloc.h>
#elif defined(__FreeBSD__)
#include <malloc_np.h>
#elif defined(_WIN32)
#include <windows.h>
#endif
#if !defined(_WIN32) && !defined(EMSCRIPTEN) && !defined(__wasi__)
#include <errno.h>
#include <pthread.h>
#endif
#if !defined(_WIN32)
#include <limits.h>
#include <unistd.h>
#endif

#if defined(_MSC_VER) && !defined(__clang__)
#  define likely(x)       (x)
#  define unlikely(x)     (x)
#  define force_inline __forceinline
#  define no_inline __declspec(noinline)
#  define __maybe_unused
#  define __attribute__(x)
#  define __attribute(x)
#else
#  define likely(x)       __builtin_expect(!!(x), 1)
#  define unlikely(x)     __builtin_expect(!!(x), 0)
#  define force_inline inline __attribute__((always_inline))
#  define no_inline __attribute__((noinline))
#  define __maybe_unused __attribute__((unused))
#endif

#if defined(_MSC_VER) && !defined(__clang__)
#include <math.h>
#define INF INFINITY
#define NEG_INF -INFINITY
#else
#define INF (1.0/0.0)
#define NEG_INF (-1.0/0.0)
#endif

#ifndef offsetof
#define offsetof(type, field) ((size_t) &((type *)0)->field)
#endif
#ifndef countof
#define countof(x) (sizeof(x) / sizeof((x)[0]))
#ifndef endof
#define endof(x) ((x) + countof(x))
#endif
#endif
#ifndef container_of
/* return the pointer of type 'type *' containing 'ptr' as field 'member' */
#define container_of(ptr, type, member) ((type *)((uint8_t *)(ptr) - offsetof(type, member)))
#endif

#if defined(_MSC_VER) || defined(__cplusplus)
#define minimum_length(n) n
#else
#define minimum_length(n) static n
#endif

/* Borrowed from Folly */
#ifndef JS_PRINTF_FORMAT
#ifdef _MSC_VER
#include <sal.h>
#define JS_PRINTF_FORMAT _Printf_format_string_
#define JS_PRINTF_FORMAT_ATTR(format_param, dots_param)
#else
#define JS_PRINTF_FORMAT
#if !defined(__clang__) && defined(__GNUC__)
#define JS_PRINTF_FORMAT_ATTR(format_param, dots_param) \
  __attribute__((format(gnu_printf, format_param, dots_param)))
#else
#define JS_PRINTF_FORMAT_ATTR(format_param, dots_param) \
  __attribute__((format(printf, format_param, dots_param)))
#endif
#endif
#endif

#if defined(PATH_MAX)
# define JS__PATH_MAX PATH_MAX
#elif defined(_WIN32)
# define JS__PATH_MAX 32767
#else
# define JS__PATH_MAX 8192
#endif

void js__pstrcpy(char *buf, int buf_size, const char *str);
char *js__pstrcat(char *buf, int buf_size, const char *s);
int js__strstart(const char *str, const char *val, const char **ptr);
int js__has_suffix(const char *str, const char *suffix);

static inline uint8_t is_be(void) {
    union {
        uint16_t a;
        uint8_t  b;
    } u = { 0x100 };
    return u.b;
}

static inline int max_int(int a, int b)
{
    if (a > b)
        return a;
    else
        return b;
}

static inline int min_int(int a, int b)
{
    if (a < b)
        return a;
    else
        return b;
}

static inline uint32_t max_uint32(uint32_t a, uint32_t b)
{
    if (a > b)
        return a;
    else
        return b;
}

static inline uint32_t min_uint32(uint32_t a, uint32_t b)
{
    if (a < b)
        return a;
    else
        return b;
}

static inline int64_t max_int64(int64_t a, int64_t b)
{
    if (a > b)
        return a;
    else
        return b;
}

static inline int64_t min_int64(int64_t a, int64_t b)
{
    if (a < b)
        return a;
    else
        return b;
}

/* WARNING: undefined if a = 0 */
static inline int clz32(unsigned int a)
{
#if defined(_MSC_VER) && !defined(__clang__)
    unsigned long index;
    _BitScanReverse(&index, a);
    return 31 - index;
#else
    return __builtin_clz(a);
#endif
}

/* WARNING: undefined if a = 0 */
static inline int clz64(uint64_t a)
{
#if defined(_MSC_VER) && !defined(__clang__)
#if INTPTR_MAX == INT64_MAX
    unsigned long index;
    _BitScanReverse64(&index, a);
    return 63 - index;
#else
    if (a >> 32)
        return clz32((unsigned)(a >> 32));
    else
        return clz32((unsigned)a) + 32;
#endif
#else
    return __builtin_clzll(a);
#endif
}

/* WARNING: undefined if a = 0 */
static inline int ctz32(unsigned int a)
{
#if defined(_MSC_VER) && !defined(__clang__)
    unsigned long index;
    _BitScanForward(&index, a);
    return index;
#else
    return __builtin_ctz(a);
#endif
}

/* WARNING: undefined if a = 0 */
static inline int ctz64(uint64_t a)
{
#if defined(_MSC_VER) && !defined(__clang__)
    unsigned long index;
    _BitScanForward64(&index, a);
    return index;
#else
    return __builtin_ctzll(a);
#endif
}

static inline uint64_t get_u64(const uint8_t *tab)
{
    uint64_t v;
    memcpy(&v, tab, sizeof(v));
    return v;
}

static inline int64_t get_i64(const uint8_t *tab)
{
    int64_t v;
    memcpy(&v, tab, sizeof(v));
    return v;
}

static inline void put_u64(uint8_t *tab, uint64_t val)
{
    memcpy(tab, &val, sizeof(val));
}

static inline uint32_t get_u32(const uint8_t *tab)
{
    uint32_t v;
    memcpy(&v, tab, sizeof(v));
    return v;
}

static inline int32_t get_i32(const uint8_t *tab)
{
    int32_t v;
    memcpy(&v, tab, sizeof(v));
    return v;
}

static inline void put_u32(uint8_t *tab, uint32_t val)
{
    memcpy(tab, &val, sizeof(val));
}

static inline uint32_t get_u16(const uint8_t *tab)
{
    uint16_t v;
    memcpy(&v, tab, sizeof(v));
    return v;
}

static inline int32_t get_i16(const uint8_t *tab)
{
    int16_t v;
    memcpy(&v, tab, sizeof(v));
    return v;
}

static inline void put_u16(uint8_t *tab, uint16_t val)
{
    memcpy(tab, &val, sizeof(val));
}

static inline uint32_t get_u8(const uint8_t *tab)
{
    return *tab;
}

static inline int32_t get_i8(const uint8_t *tab)
{
    return (int8_t)*tab;
}

static inline void put_u8(uint8_t *tab, uint8_t val)
{
    *tab = val;
}

#ifndef bswap16
static inline uint16_t bswap16(uint16_t x)
{
    return (x >> 8) | (x << 8);
}
#endif

#ifndef bswap32
static inline uint32_t bswap32(uint32_t v)
{
    return ((v & 0xff000000) >> 24) | ((v & 0x00ff0000) >>  8) |
        ((v & 0x0000ff00) <<  8) | ((v & 0x000000ff) << 24);
}
#endif

#ifndef bswap64
static inline uint64_t bswap64(uint64_t v)
{
    return ((v & ((uint64_t)0xff << (7 * 8))) >> (7 * 8)) |
        ((v & ((uint64_t)0xff << (6 * 8))) >> (5 * 8)) |
        ((v & ((uint64_t)0xff << (5 * 8))) >> (3 * 8)) |
        ((v & ((uint64_t)0xff << (4 * 8))) >> (1 * 8)) |
        ((v & ((uint64_t)0xff << (3 * 8))) << (1 * 8)) |
        ((v & ((uint64_t)0xff << (2 * 8))) << (3 * 8)) |
        ((v & ((uint64_t)0xff << (1 * 8))) << (5 * 8)) |
        ((v & ((uint64_t)0xff << (0 * 8))) << (7 * 8));
}
#endif

static inline void inplace_bswap16(uint8_t *tab) {
    put_u16(tab, bswap16(get_u16(tab)));
}

static inline void inplace_bswap32(uint8_t *tab) {
    put_u32(tab, bswap32(get_u32(tab)));
}

static inline double fromfp16(uint16_t v) {
    double d, s;
    int e;
    if ((v & 0x7C00) == 0x7C00) {
        d = (v & 0x3FF) ? NAN : INFINITY;
    } else {
        d = (v & 0x3FF) / 1024.;
        e = (v & 0x7C00) >> 10;
        if (e == 0) {
            e = -14;
        } else {
            d += 1;
            e -= 15;
        }
        d = scalbn(d, e);
    }
    s = (v & 0x8000) ? -1.0 : 1.0;
    return d * s;
}

static inline uint16_t tofp16(double d) {
    uint16_t f, s;
    double t;
    int e;
    s = 0;
    if (copysign(1, d) < 0) { // preserve sign when |d| is negative zero
        d = -d;
        s = 0x8000;
    }
    if (isinf(d))
        return s | 0x7C00;
    if (isnan(d))
        return s | 0x7C01;
    if (d == 0)
        return s | 0;
    d = 2 * frexp(d, &e);
    e--;
    if (e > 15)
        return s | 0x7C00; // out of range, return +/-infinity
    if (e < -25) {
        d = 0;
        e = 0;
    } else if (e < -14) {
        d = scalbn(d, e + 14);
        e = 0;
    } else {
        d -= 1;
        e += 15;
    }
    d *= 1024.;
    f = (uint16_t)d;
    t = d - f;
    if (t < 0.5)
        goto done;
    if (t == 0.5)
        if ((f & 1) == 0)
            goto done;
    // adjust for rounding
    if (++f == 1024) {
        f = 0;
        if (++e == 31)
            return s | 0x7C00; // out of range, return +/-infinity
    }
done:
    return s | (e << 10) | f;
}

static inline int isfp16nan(uint16_t v) {
    return (v & 0x7FFF) > 0x7C00;
}

static inline int isfp16zero(uint16_t v) {
    return (v & 0x7FFF) == 0;
}

/* XXX: should take an extra argument to pass slack information to the caller */
typedef void *DynBufReallocFunc(void *opaque, void *ptr, size_t size);

typedef struct DynBuf {
    uint8_t *buf;
    size_t size;
    size_t allocated_size;
    bool error; /* true if a memory allocation error occurred */
    DynBufReallocFunc *realloc_func;
    void *opaque; /* for realloc_func */
} DynBuf;

void dbuf_init(DynBuf *s);
void dbuf_init2(DynBuf *s, void *opaque, DynBufReallocFunc *realloc_func);
int dbuf_realloc(DynBuf *s, size_t new_size);
int dbuf_write(DynBuf *s, size_t offset, const void *data, size_t len);
int dbuf_put(DynBuf *s, const void *data, size_t len);
int dbuf_put_self(DynBuf *s, size_t offset, size_t len);
int dbuf_putc(DynBuf *s, uint8_t c);
int dbuf_putstr(DynBuf *s, const char *str);
static inline int dbuf_put_u16(DynBuf *s, uint16_t val)
{
    return dbuf_put(s, (uint8_t *)&val, 2);
}
static inline int dbuf_put_u32(DynBuf *s, uint32_t val)
{
    return dbuf_put(s, (uint8_t *)&val, 4);
}
static inline int dbuf_put_u64(DynBuf *s, uint64_t val)
{
    return dbuf_put(s, (uint8_t *)&val, 8);
}
int JS_PRINTF_FORMAT_ATTR(2, 3) dbuf_printf(DynBuf *s, JS_PRINTF_FORMAT const char *fmt, ...);
void dbuf_free(DynBuf *s);
static inline bool dbuf_error(DynBuf *s) {
    return s->error;
}
static inline void dbuf_set_error(DynBuf *s)
{
    s->error = true;
}

/*---- UTF-8 and UTF-16 handling ----*/

#define UTF8_CHAR_LEN_MAX 4

enum {
    UTF8_PLAIN_ASCII  = 0,  // 7-bit ASCII plain text
    UTF8_NON_ASCII    = 1,  // has non ASCII code points (8-bit or more)
    UTF8_HAS_16BIT    = 2,  // has 16-bit code points
    UTF8_HAS_NON_BMP1 = 4,  // has non-BMP1 code points, needs UTF-16 surrogate pairs
    UTF8_HAS_ERRORS   = 8,  // has encoding errors
};
int utf8_scan(const char *buf, size_t len, size_t *plen);
size_t utf8_encode_len(uint32_t c);
size_t utf8_encode(uint8_t buf[minimum_length(UTF8_CHAR_LEN_MAX)], uint32_t c);
uint32_t utf8_decode_len(const uint8_t *p, size_t max_len, const uint8_t **pp);
uint32_t utf8_decode(const uint8_t *p, const uint8_t **pp);
size_t utf8_decode_buf8(uint8_t *dest, size_t dest_len, const char *src, size_t src_len);
size_t utf8_decode_buf16(uint16_t *dest, size_t dest_len, const char *src, size_t src_len);
size_t utf8_encode_buf8(char *dest, size_t dest_len, const uint8_t *src, size_t src_len);
size_t utf8_encode_buf16(char *dest, size_t dest_len, const uint16_t *src, size_t src_len);

static inline bool is_surrogate(uint32_t c)
{
    return (c >> 11) == (0xD800 >> 11); // 0xD800-0xDFFF
}

static inline bool is_hi_surrogate(uint32_t c)
{
    return (c >> 10) == (0xD800 >> 10); // 0xD800-0xDBFF
}

static inline bool is_lo_surrogate(uint32_t c)
{
    return (c >> 10) == (0xDC00 >> 10); // 0xDC00-0xDFFF
}

static inline uint32_t get_hi_surrogate(uint32_t c)
{
    return (c >> 10) - (0x10000 >> 10) + 0xD800;
}

static inline uint32_t get_lo_surrogate(uint32_t c)
{
    return (c & 0x3FF) | 0xDC00;
}

static inline uint32_t from_surrogate(uint32_t hi, uint32_t lo)
{
    return 65536 + 1024 * (hi & 1023) + (lo & 1023);
}

static inline int from_hex(int c)
{
    if (c >= '0' && c <= '9')
        return c - '0';
    else if (c >= 'A' && c <= 'F')
        return c - 'A' + 10;
    else if (c >= 'a' && c <= 'f')
        return c - 'a' + 10;
    else
        return -1;
}

static inline uint8_t is_upper_ascii(uint8_t c) {
    return c >= 'A' && c <= 'Z';
}

static inline uint8_t to_upper_ascii(uint8_t c) {
    return c >= 'a' && c <= 'z' ? c - 'a' + 'A' : c;
}

void rqsort(void *base, size_t nmemb, size_t size,
            int (*cmp)(const void *, const void *, void *),
            void *arg);

static inline uint64_t float64_as_uint64(double d)
{
    union {
        double d;
        uint64_t u64;
    } u;
    u.d = d;
    return u.u64;
}

static inline double uint64_as_float64(uint64_t u64)
{
    union {
        double d;
        uint64_t u64;
    } u;
    u.u64 = u64;
    return u.d;
}

int64_t js__gettimeofday_us(void);
uint64_t js__hrtime_ns(void);

static inline size_t js__malloc_usable_size(const void *ptr)
{
#if defined(__APPLE__)
    return malloc_size(ptr);
#elif defined(_WIN32)
    return _msize((void *)ptr);
#elif defined(__linux__) || defined(__ANDROID__) || defined(__CYGWIN__) || defined(__FreeBSD__) || defined(__GLIBC__)
    return malloc_usable_size((void *)ptr);
#else
    return 0;
#endif
}

int js_exepath(char* buffer, size_t* size);

/* Cross-platform threading APIs. */

#if defined(EMSCRIPTEN) || defined(__wasi__)

#define JS_HAVE_THREADS 0

#else

#define JS_HAVE_THREADS 1

#if defined(_WIN32)
#define JS_ONCE_INIT INIT_ONCE_STATIC_INIT
typedef INIT_ONCE js_once_t;
typedef CRITICAL_SECTION js_mutex_t;
typedef CONDITION_VARIABLE js_cond_t;
typedef HANDLE js_thread_t;
#else
#define JS_ONCE_INIT PTHREAD_ONCE_INIT
typedef pthread_once_t js_once_t;
typedef pthread_mutex_t js_mutex_t;
typedef pthread_cond_t js_cond_t;
typedef pthread_t js_thread_t;
#endif

void js_once(js_once_t *guard, void (*callback)(void));

void js_mutex_init(js_mutex_t *mutex);
void js_mutex_destroy(js_mutex_t *mutex);
void js_mutex_lock(js_mutex_t *mutex);
void js_mutex_unlock(js_mutex_t *mutex);

void js_cond_init(js_cond_t *cond);
void js_cond_destroy(js_cond_t *cond);
void js_cond_signal(js_cond_t *cond);
void js_cond_broadcast(js_cond_t *cond);
void js_cond_wait(js_cond_t *cond, js_mutex_t *mutex);
int js_cond_timedwait(js_cond_t *cond, js_mutex_t *mutex, uint64_t timeout);

enum {
    JS_THREAD_CREATE_DETACHED = 1,
};

// creates threads with 2 MB stacks (glibc default)
int js_thread_create(js_thread_t *thrd, void (*start)(void *), void *arg,
                     int flags);
int js_thread_join(js_thread_t thrd);

#endif /* !defined(EMSCRIPTEN) && !defined(__wasi__) */

#ifdef __cplusplus
} /* extern "C" { */
#endif

#endif  /* CUTILS_H */
/*
 * Tiny float64 printing and parsing library
 *
 * Copyright (c) 2024 Fabrice Bellard
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

//#define JS_DTOA_DUMP_STATS

/* maximum number of digits for fixed and frac formats */
#define JS_DTOA_MAX_DIGITS 101

/* radix != 10 is only supported with flags = JS_DTOA_FORMAT_FREE */
/* use as many digits as necessary */
#define JS_DTOA_FORMAT_FREE  (0 << 0)
/* use n_digits significant digits (1 <= n_digits <= JS_DTOA_MAX_DIGITS) */
#define JS_DTOA_FORMAT_FIXED (1 << 0)
/* force fractional format: [-]dd.dd with n_digits fractional digits.
   0 <= n_digits <= JS_DTOA_MAX_DIGITS */
#define JS_DTOA_FORMAT_FRAC  (2 << 0)
#define JS_DTOA_FORMAT_MASK  (3 << 0)

/* select exponential notation either in fixed or free format */
#define JS_DTOA_EXP_AUTO     (0 << 2)
#define JS_DTOA_EXP_ENABLED  (1 << 2)
#define JS_DTOA_EXP_DISABLED (2 << 2)
#define JS_DTOA_EXP_MASK     (3 << 2)

#define JS_DTOA_MINUS_ZERO   (1 << 4) /* show the minus sign for -0 */

/* only accepts integers (no dot, no exponent) */
#define JS_ATOD_INT_ONLY       (1 << 0)
/* accept Oo and Ob prefixes in addition to 0x prefix if radix = 0 */
#define JS_ATOD_ACCEPT_BIN_OCT (1 << 1)
/* accept O prefix as octal if radix == 0 and properly formed (Annex B) */
#define JS_ATOD_ACCEPT_LEGACY_OCTAL  (1 << 2)
/* accept _ between digits as a digit separator */
#define JS_ATOD_ACCEPT_UNDERSCORES  (1 << 3)

typedef struct {
    uint64_t mem[37];
} JSDTOATempMem;

typedef struct {
    uint64_t mem[27];
} JSATODTempMem;

/* return a maximum bound of the string length */
int js_dtoa_max_len(double d, int radix, int n_digits, int flags);
/* return the string length */
int js_dtoa(char *buf, double d, int radix, int n_digits, int flags,
            JSDTOATempMem *tmp_mem);
double js_atod(const char *str, const char **pnext, int radix, int flags,
               JSATODTempMem *tmp_mem);

#ifdef JS_DTOA_DUMP_STATS
void js_dtoa_dump_stats(void);
#endif

/* additional exported functions */
size_t u32toa(char *buf, uint32_t n);
size_t i32toa(char *buf, int32_t n);
size_t u64toa(char *buf, uint64_t n);
size_t i64toa(char *buf, int64_t n);
size_t u64toa_radix(char *buf, uint64_t n, unsigned int radix);
size_t i64toa_radix(char *buf, int64_t n, unsigned int radix);
/*
 * Linux klist like system
 *
 * Copyright (c) 2016-2017 Fabrice Bellard
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#ifndef LIST_H
#define LIST_H

#ifndef NULL
#include <stddef.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif

struct list_head {
    struct list_head *prev;
    struct list_head *next;
};

#define LIST_HEAD_INIT(el) { &(el), &(el) }

/* return the pointer of type 'type *' containing 'el' as field 'member' */
#define list_entry(el, type, member) container_of(el, type, member)

static inline void init_list_head(struct list_head *head)
{
    head->prev = head;
    head->next = head;
}

/* insert 'el' between 'prev' and 'next' */
static inline void __list_add(struct list_head *el,
                              struct list_head *prev, struct list_head *next)
{
    prev->next = el;
    el->prev = prev;
    el->next = next;
    next->prev = el;
}

/* add 'el' at the head of the list 'head' (= after element head) */
static inline void list_add(struct list_head *el, struct list_head *head)
{
    __list_add(el, head, head->next);
}

/* add 'el' at the end of the list 'head' (= before element head) */
static inline void list_add_tail(struct list_head *el, struct list_head *head)
{
    __list_add(el, head->prev, head);
}

static inline void list_del(struct list_head *el)
{
    struct list_head *prev, *next;
    prev = el->prev;
    next = el->next;
    prev->next = next;
    next->prev = prev;
    el->prev = NULL; /* fail safe */
    el->next = NULL; /* fail safe */
}

static inline int list_empty(struct list_head *el)
{
    return el->next == el;
}

#define list_for_each(el, head) \
  for(el = (head)->next; el != (head); el = el->next)

#define list_for_each_safe(el, el1, head)                \
    for(el = (head)->next, el1 = el->next; el != (head); \
        el = el1, el1 = el->next)

#define list_for_each_prev(el, head) \
  for(el = (head)->prev; el != (head); el = el->prev)

#define list_for_each_prev_safe(el, el1, head)           \
    for(el = (head)->prev, el1 = el->prev; el != (head); \
        el = el1, el1 = el->prev)

#ifdef __cplusplus
} /* extern "C" { */
#endif

#endif /* LIST_H */
/*
 * Unicode utilities
 *
 * Copyright (c) 2017-2018 Fabrice Bellard
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#ifndef LIBUNICODE_H
#define LIBUNICODE_H

#include <stdbool.h>
#include <stddef.h>
#include <inttypes.h>

#ifdef __cplusplus
extern "C" {
#endif

#define LRE_CC_RES_LEN_MAX 3

typedef enum {
    UNICODE_NFC,
    UNICODE_NFD,
    UNICODE_NFKC,
    UNICODE_NFKD,
} UnicodeNormalizationEnum;

int lre_case_conv(uint32_t *res, uint32_t c, int conv_type);
int lre_canonicalize(uint32_t c, bool is_unicode);
bool lre_is_cased(uint32_t c);
bool lre_is_case_ignorable(uint32_t c);

/* char ranges */

typedef struct {
    int len; /* in points, always even */
    int size;
    uint32_t *points; /* points sorted by increasing value */
    void *mem_opaque;
    void *(*realloc_func)(void *opaque, void *ptr, size_t size);
} CharRange;

typedef enum {
    CR_OP_UNION,
    CR_OP_INTER,
    CR_OP_XOR,
} CharRangeOpEnum;

void cr_init(CharRange *cr, void *mem_opaque, void *(*realloc_func)(void *opaque, void *ptr, size_t size));
void cr_free(CharRange *cr);
int cr_realloc(CharRange *cr, int size);
int cr_copy(CharRange *cr, const CharRange *cr1);

static inline int cr_add_point(CharRange *cr, uint32_t v)
{
    if (cr->len >= cr->size) {
        if (cr_realloc(cr, cr->len + 1))
            return -1;
    }
    cr->points[cr->len++] = v;
    return 0;
}

static inline int cr_add_interval(CharRange *cr, uint32_t c1, uint32_t c2)
{
    if ((cr->len + 2) > cr->size) {
        if (cr_realloc(cr, cr->len + 2))
            return -1;
    }
    cr->points[cr->len++] = c1;
    cr->points[cr->len++] = c2;
    return 0;
}

int cr_union1(CharRange *cr, const uint32_t *b_pt, int b_len);

static inline int cr_union_interval(CharRange *cr, uint32_t c1, uint32_t c2)
{
    uint32_t b_pt[2];
    b_pt[0] = c1;
    b_pt[1] = c2 + 1;
    return cr_union1(cr, b_pt, 2);
}

int cr_op(CharRange *cr, const uint32_t *a_pt, int a_len,
          const uint32_t *b_pt, int b_len, int op);

int cr_invert(CharRange *cr);
int cr_regexp_canonicalize(CharRange *cr, bool is_unicode);

bool lre_is_id_start(uint32_t c);
bool lre_is_id_continue(uint32_t c);
bool lre_is_white_space(uint32_t c);

int unicode_normalize(uint32_t **pdst, const uint32_t *src, int src_len,
                      UnicodeNormalizationEnum n_type,
                      void *opaque, void *(*realloc_func)(void *opaque, void *ptr, size_t size));

/* Unicode character range functions */

int unicode_script(CharRange *cr,
                   const char *script_name, bool is_ext);
int unicode_general_category(CharRange *cr, const char *gc_name);
int unicode_prop(CharRange *cr, const char *prop_name);

#ifdef __cplusplus
} /* extern "C" { */
#endif

#endif /* LIBUNICODE_H */
/*
 * Regular Expression Engine
 *
 * Copyright (c) 2017-2018 Fabrice Bellard
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#ifndef LIBREGEXP_H
#define LIBREGEXP_H

#include <stdbool.h>
#include <stddef.h>



#ifdef __cplusplus
extern "C" {
#endif

#define LRE_FLAG_GLOBAL     (1 << 0)
#define LRE_FLAG_IGNORECASE (1 << 1)
#define LRE_FLAG_MULTILINE  (1 << 2)
#define LRE_FLAG_DOTALL     (1 << 3)
#define LRE_FLAG_UNICODE    (1 << 4)
#define LRE_FLAG_STICKY     (1 << 5)
#define LRE_FLAG_INDICES    (1 << 6) /* Unused by libregexp, just recorded. */
#define LRE_FLAG_NAMED_GROUPS (1 << 7) /* named groups are present in the regexp */
#define LRE_FLAG_UNICODE_SETS (1 << 8)

#define LRE_RET_MEMORY_ERROR (-1)
#define LRE_RET_TIMEOUT      (-2)

uint8_t *lre_compile(int *plen, char *error_msg, int error_msg_size,
                     const char *buf, size_t buf_len, int re_flags,
                     void *opaque);
int lre_get_capture_count(const uint8_t *bc_buf);
int lre_get_flags(const uint8_t *bc_buf);
const char *lre_get_groupnames(const uint8_t *bc_buf);
int lre_exec(uint8_t **capture,
             const uint8_t *bc_buf, const uint8_t *cbuf, int cindex, int clen,
             int cbuf_type, void *opaque);

int lre_parse_escape(const uint8_t **pp, int allow_utf16);
bool lre_is_space(int c);

void lre_byte_swap(uint8_t *buf, size_t len, bool is_byte_swapped);

/* must be provided by the user */
bool lre_check_stack_overflow(void *opaque, size_t alloca_size);
/* must be provided by the user, return non zero if time out */
int lre_check_timeout(void *opaque);
void *lre_realloc(void *opaque, void *ptr, size_t size);

/* JS identifier test */
extern uint32_t const lre_id_start_table_ascii[4];
extern uint32_t const lre_id_continue_table_ascii[4];

static inline int lre_js_is_ident_first(int c)
{
    if ((uint32_t)c < 128) {
        return (lre_id_start_table_ascii[c >> 5] >> (c & 31)) & 1;
    } else {
        return lre_is_id_start(c);
    }
}

static inline int lre_js_is_ident_next(int c)
{
    if ((uint32_t)c < 128) {
        return (lre_id_continue_table_ascii[c >> 5] >> (c & 31)) & 1;
    } else {
        /* ZWNJ and ZWJ are accepted in identifiers */
        return lre_is_id_continue(c) || c == 0x200C || c == 0x200D;
    }
}

#ifdef __cplusplus
} /* extern "C" { */
#endif

#endif /* LIBREGEXP_H */
/* Compressed unicode tables */
/* Automatically generated file - do not edit */

#include <stdint.h>

static const uint32_t case_conv_table1[378] = {
    0x00209a30, 0x00309a00, 0x005a8173, 0x00601730,
    0x006c0730, 0x006f81b3, 0x00701700, 0x007c0700,
    0x007f8100, 0x00803040, 0x009801c3, 0x00988190,
    0x00990640, 0x009c9040, 0x00a481b4, 0x00a52e40,
    0x00bc0130, 0x00bc8640, 0x00bf8170, 0x00c00100,
    0x00c08130, 0x00c10440, 0x00c30130, 0x00c38240,
    0x00c48230, 0x00c58240, 0x00c70130, 0x00c78130,
    0x00c80130, 0x00c88240, 0x00c98130, 0x00ca0130,
    0x00ca8100, 0x00cb0130, 0x00cb8130, 0x00cc0240,
    0x00cd0100, 0x00cd8101, 0x00ce0130, 0x00ce8130,
    0x00cf0100, 0x00cf8130, 0x00d00640, 0x00d30130,
    0x00d38240, 0x00d48130, 0x00d60240, 0x00d70130,
    0x00d78240, 0x00d88230, 0x00d98440, 0x00db8130,
    0x00dc0240, 0x00de0240, 0x00df8100, 0x00e20350,
    0x00e38350, 0x00e50350, 0x00e69040, 0x00ee8100,
    0x00ef1240, 0x00f801b4, 0x00f88350, 0x00fa0240,
    0x00fb0130, 0x00fb8130, 0x00fc2840, 0x01100130,
    0x01111240, 0x011d0131, 0x011d8240, 0x011e8130,
    0x011f0131, 0x011f8201, 0x01208240, 0x01218130,
    0x01220130, 0x01228130, 0x01230a40, 0x01280101,
    0x01288101, 0x01290101, 0x01298100, 0x012a0100,
    0x012b0200, 0x012c8100, 0x012d8100, 0x012e0101,
    0x01300100, 0x01308101, 0x01318100, 0x01320101,
    0x01328101, 0x01330101, 0x01340100, 0x01348100,
    0x01350101, 0x01358101, 0x01360101, 0x01378100,
    0x01388101, 0x01390100, 0x013a8100, 0x013e8101,
    0x01400100, 0x01410101, 0x01418100, 0x01438101,
    0x01440100, 0x01448100, 0x01450200, 0x01460100,
    0x01490100, 0x014e8101, 0x014f0101, 0x01a28173,
    0x01b80440, 0x01bb0240, 0x01bd8300, 0x01bf8130,
    0x01c30130, 0x01c40330, 0x01c60130, 0x01c70230,
    0x01c801d0, 0x01c89130, 0x01d18930, 0x01d60100,
    0x01d68300, 0x01d801d3, 0x01d89100, 0x01e10173,
    0x01e18900, 0x01e60100, 0x01e68200, 0x01e78130,
    0x01e80173, 0x01e88173, 0x01ea8173, 0x01eb0173,
    0x01eb8100, 0x01ec1840, 0x01f80173, 0x01f88173,
    0x01f90100, 0x01f98100, 0x01fa01a0, 0x01fa8173,
    0x01fb8240, 0x01fc8130, 0x01fd0240, 0x01fe8330,
    0x02001030, 0x02082030, 0x02182000, 0x02281000,
    0x02302240, 0x02453640, 0x02600130, 0x02608e40,
    0x02678100, 0x02686040, 0x0298a630, 0x02b0a600,
    0x02c381b5, 0x08502631, 0x08638131, 0x08668131,
    0x08682b00, 0x087e8300, 0x09d05011, 0x09f80610,
    0x09fc0620, 0x0e400174, 0x0e408174, 0x0e410174,
    0x0e418174, 0x0e420174, 0x0e428174, 0x0e430174,
    0x0e438180, 0x0e440180, 0x0e448240, 0x0e482b30,
    0x0e5e8330, 0x0ebc8101, 0x0ebe8101, 0x0ec70101,
    0x0f007e40, 0x0f3f1840, 0x0f4b01b5, 0x0f4b81b6,
    0x0f4c01b6, 0x0f4c81b6, 0x0f4d01b7, 0x0f4d8180,
    0x0f4f0130, 0x0f506040, 0x0f800800, 0x0f840830,
    0x0f880600, 0x0f8c0630, 0x0f900800, 0x0f940830,
    0x0f980800, 0x0f9c0830, 0x0fa00600, 0x0fa40630,
    0x0fa801b0, 0x0fa88100, 0x0fa901d3, 0x0fa98100,
    0x0faa01d3, 0x0faa8100, 0x0fab01d3, 0x0fab8100,
    0x0fac8130, 0x0fad8130, 0x0fae8130, 0x0faf8130,
    0x0fb00800, 0x0fb40830, 0x0fb80200, 0x0fb90400,
    0x0fbb0201, 0x0fbc0201, 0x0fbd0201, 0x0fbe0201,
    0x0fc008b7, 0x0fc40867, 0x0fc808b8, 0x0fcc0868,
    0x0fd008b8, 0x0fd40868, 0x0fd80200, 0x0fd901b9,
    0x0fd981b1, 0x0fda01b9, 0x0fdb01b1, 0x0fdb81d7,
    0x0fdc0230, 0x0fdd0230, 0x0fde0161, 0x0fdf0173,
    0x0fe101b9, 0x0fe181b2, 0x0fe201ba, 0x0fe301b2,
    0x0fe381d8, 0x0fe40430, 0x0fe60162, 0x0fe80201,
    0x0fe901d0, 0x0fe981d0, 0x0feb01b0, 0x0feb81d0,
    0x0fec0230, 0x0fed0230, 0x0ff00201, 0x0ff101d3,
    0x0ff181d3, 0x0ff201ba, 0x0ff28101, 0x0ff301b0,
    0x0ff381d3, 0x0ff40231, 0x0ff50230, 0x0ff60131,
    0x0ff901ba, 0x0ff981b2, 0x0ffa01bb, 0x0ffb01b2,
    0x0ffb81d9, 0x0ffc0230, 0x0ffd0230, 0x0ffe0162,
    0x109301a0, 0x109501a0, 0x109581a0, 0x10990131,
    0x10a70101, 0x10b01031, 0x10b81001, 0x10c18240,
    0x125b1a31, 0x12681a01, 0x16003031, 0x16183001,
    0x16300240, 0x16310130, 0x16318130, 0x16320130,
    0x16328100, 0x16330100, 0x16338640, 0x16368130,
    0x16370130, 0x16378130, 0x16380130, 0x16390240,
    0x163a8240, 0x163f0230, 0x16406440, 0x16758440,
    0x16790240, 0x16802600, 0x16938100, 0x16968100,
    0x53202e40, 0x53401c40, 0x53910e40, 0x53993e40,
    0x53bc8440, 0x53be8130, 0x53bf0a40, 0x53c58240,
    0x53c68130, 0x53c80440, 0x53ca0101, 0x53cb1440,
    0x53d50130, 0x53d58130, 0x53d60130, 0x53d68130,
    0x53d70130, 0x53d80130, 0x53d88130, 0x53d90130,
    0x53d98131, 0x53da1040, 0x53e20131, 0x53e28130,
    0x53e30130, 0x53e38440, 0x53e58130, 0x53e60240,
    0x53e80240, 0x53eb0640, 0x53ee0130, 0x53fa8240,
    0x55a98101, 0x55b85020, 0x7d8001b2, 0x7d8081b2,
    0x7d8101b2, 0x7d8181da, 0x7d8201da, 0x7d8281b3,
    0x7d8301b3, 0x7d8981bb, 0x7d8a01bb, 0x7d8a81bb,
    0x7d8b01bc, 0x7d8b81bb, 0x7f909a31, 0x7fa09a01,
    0x82002831, 0x82142801, 0x82582431, 0x826c2401,
    0x82b80b31, 0x82be0f31, 0x82c60731, 0x82ca0231,
    0x82cb8b01, 0x82d18f01, 0x82d98701, 0x82dd8201,
    0x86403331, 0x86603301, 0x86a81631, 0x86b81601,
    0x8c502031, 0x8c602001, 0xb7202031, 0xb7302001,
    0xf4802231, 0xf4912201,
};

static const uint8_t case_conv_table2[378] = {
    0x01, 0x00, 0x9c, 0x06, 0x07, 0x4d, 0x03, 0x04,
    0x10, 0x00, 0x8f, 0x0b, 0x00, 0x00, 0x11, 0x00,
    0x08, 0x00, 0x53, 0x4b, 0x52, 0x00, 0x53, 0x00,
    0x54, 0x00, 0x3b, 0x55, 0x56, 0x00, 0x58, 0x5a,
    0x40, 0x5f, 0x5e, 0x00, 0x47, 0x52, 0x63, 0x65,
    0x43, 0x66, 0x00, 0x68, 0x00, 0x6a, 0x00, 0x6c,
    0x00, 0x6e, 0x00, 0x70, 0x00, 0x00, 0x41, 0x00,
    0x00, 0x00, 0x00, 0x1a, 0x00, 0x93, 0x00, 0x00,
    0x20, 0x36, 0x00, 0x28, 0x00, 0x24, 0x00, 0x24,
    0x25, 0x2d, 0x00, 0x13, 0x6d, 0x6f, 0x00, 0x29,
    0x27, 0x2a, 0x14, 0x16, 0x18, 0x1b, 0x1c, 0x41,
    0x1e, 0x42, 0x1f, 0x4e, 0x3c, 0x40, 0x22, 0x21,
    0x44, 0x21, 0x43, 0x26, 0x28, 0x27, 0x29, 0x23,
    0x2b, 0x4b, 0x2d, 0x46, 0x2f, 0x4c, 0x31, 0x4d,
    0x33, 0x47, 0x45, 0x99, 0x00, 0x00, 0x97, 0x91,
    0x7f, 0x80, 0x85, 0x86, 0x12, 0x82, 0x84, 0x78,
    0x79, 0x12, 0x7d, 0xa3, 0x7e, 0x7a, 0x7b, 0x8c,
    0x92, 0x98, 0xa6, 0xa0, 0x87, 0x00, 0x9a, 0xa1,
    0x95, 0x77, 0x33, 0x95, 0x00, 0x90, 0x00, 0x76,
    0x9b, 0x9a, 0x99, 0x98, 0x00, 0x00, 0xa0, 0x00,
    0x9e, 0x00, 0xa3, 0xa2, 0x15, 0x31, 0x32, 0x33,
    0xb7, 0xb8, 0x55, 0xac, 0xab, 0x12, 0x14, 0x1e,
    0x21, 0x22, 0x22, 0x2a, 0x34, 0x35, 0x00, 0xa8,
    0xa9, 0x39, 0x22, 0x4c, 0x00, 0x00, 0x97, 0x01,
    0x5a, 0xda, 0x1d, 0x36, 0x05, 0x00, 0xc7, 0xc6,
    0xc9, 0xc8, 0xcb, 0xca, 0xcd, 0xcc, 0xcf, 0xce,
    0xc4, 0xd8, 0x45, 0xd9, 0x42, 0xda, 0x46, 0xdb,
    0xd1, 0xd3, 0xd5, 0xd7, 0xdd, 0xdc, 0xf1, 0xf9,
    0x01, 0x11, 0x0a, 0x12, 0x80, 0x9f, 0x00, 0x21,
    0x80, 0xa3, 0xf0, 0x00, 0xc0, 0x40, 0xc6, 0x60,
    0xea, 0xde, 0xe6, 0x99, 0xc0, 0x00, 0x00, 0x06,
    0x60, 0xdf, 0x29, 0x00, 0x15, 0x12, 0x06, 0x16,
    0xfb, 0xe0, 0x09, 0x15, 0x12, 0x84, 0x0b, 0xc6,
    0x16, 0x02, 0xe2, 0x06, 0xc0, 0x40, 0x00, 0x46,
    0x60, 0xe1, 0xe3, 0x6d, 0x37, 0x38, 0x39, 0x18,
    0x17, 0x1a, 0x19, 0x00, 0x1d, 0x1c, 0x1f, 0x1e,
    0x00, 0x61, 0xba, 0x67, 0x45, 0x48, 0x00, 0x50,
    0x64, 0x4f, 0x51, 0x00, 0x00, 0x49, 0x00, 0x00,
    0x00, 0xa5, 0xa6, 0xa7, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xb9, 0x00, 0x00, 0x5c, 0x00, 0x4a, 0x00,
    0x5d, 0x57, 0x59, 0x62, 0x60, 0x72, 0x6b, 0x71,
    0x54, 0x00, 0x3e, 0x69, 0xbb, 0x00, 0x5b, 0x00,
    0x00, 0x00, 0x25, 0x00, 0x48, 0xaa, 0x8a, 0x8b,
    0x8c, 0xab, 0xac, 0x58, 0x58, 0xaf, 0x94, 0xb0,
    0x6f, 0xb2, 0x63, 0x62, 0x65, 0x64, 0x67, 0x66,
    0x6c, 0x6d, 0x6e, 0x6f, 0x68, 0x69, 0x6a, 0x6b,
    0x71, 0x70, 0x73, 0x72, 0x75, 0x74, 0x77, 0x76,
    0x79, 0x78,
};

static const uint16_t case_conv_ext[58] = {
    0x0399, 0x0308, 0x0301, 0x03a5, 0x0313, 0x0300, 0x0342, 0x0391,
    0x0397, 0x03a9, 0x0046, 0x0049, 0x004c, 0x0053, 0x0069, 0x0307,
    0x02bc, 0x004e, 0x004a, 0x030c, 0x0535, 0x0552, 0x0048, 0x0331,
    0x0054, 0x0057, 0x030a, 0x0059, 0x0041, 0x02be, 0x1f08, 0x1f80,
    0x1f28, 0x1f90, 0x1f68, 0x1fa0, 0x1fba, 0x0386, 0x1fb3, 0x1fca,
    0x0389, 0x1fc3, 0x03a1, 0x1ffa, 0x038f, 0x1ff3, 0x0544, 0x0546,
    0x053b, 0x054e, 0x053d, 0x03b8, 0x0462, 0xa64a, 0x1e60, 0x03c9,
    0x006b, 0x00e5,
};

static const uint8_t unicode_prop_Cased1_table[193] = {
    0x40, 0xa9, 0x80, 0x8e, 0x80, 0xfc, 0x80, 0xd3,
    0x80, 0x9b, 0x81, 0x8d, 0x02, 0x80, 0xe1, 0x80,
    0x91, 0x85, 0x9a, 0x01, 0x00, 0x01, 0x11, 0x03,
    0x04, 0x08, 0x01, 0x08, 0x30, 0x08, 0x01, 0x15,
    0x20, 0x00, 0x39, 0x99, 0x31, 0x9d, 0x84, 0x40,
    0x94, 0x80, 0xd6, 0x82, 0xa6, 0x80, 0x41, 0x62,
    0x80, 0xa6, 0x80, 0x4b, 0x72, 0x80, 0x4c, 0x02,
    0xf8, 0x02, 0x80, 0x8f, 0x80, 0xb0, 0x40, 0xdb,
    0x08, 0x80, 0x41, 0xd0, 0x80, 0x8c, 0x80, 0x8f,
    0x8c, 0xe4, 0x03, 0x01, 0x89, 0x00, 0x14, 0x28,
    0x10, 0x11, 0x02, 0x01, 0x18, 0x0b, 0x24, 0x4b,
    0x26, 0x01, 0x01, 0x86, 0xe5, 0x80, 0x60, 0x79,
    0xb6, 0x81, 0x40, 0x91, 0x81, 0xbd, 0x88, 0x94,
    0x05, 0x80, 0x98, 0x80, 0xa2, 0x00, 0x80, 0x9b,
    0x12, 0x82, 0x43, 0x34, 0xa2, 0x06, 0x80, 0x8d,
    0x60, 0x5c, 0x15, 0x01, 0x10, 0xa9, 0x80, 0x88,
    0x60, 0xcc, 0x44, 0xd4, 0x80, 0xc6, 0x01, 0x08,
    0x09, 0x0b, 0x80, 0x8b, 0x00, 0x06, 0x80, 0xc0,
    0x03, 0x0f, 0x06, 0x80, 0x9b, 0x03, 0x04, 0x00,
    0x16, 0x80, 0x41, 0x53, 0x81, 0x98, 0x80, 0x98,
    0x80, 0x9e, 0x80, 0x98, 0x80, 0x9e, 0x80, 0x98,
    0x80, 0x9e, 0x80, 0x98, 0x80, 0x9e, 0x80, 0x98,
    0x07, 0x47, 0x33, 0x89, 0x80, 0x93, 0x2d, 0x41,
    0x04, 0xbd, 0x50, 0xc1, 0x99, 0x85, 0x99, 0x85,
    0x99,
};

static const uint8_t unicode_prop_Cased1_index[18] = {
    0xb9, 0x02, 0x80, 0xa0, 0x1e, 0x40, 0x9e, 0xa6,
    0x40, 0xbb, 0x07, 0x01, 0xdb, 0xd6, 0x01, 0x8a,
    0xf1, 0x01,
};

static const uint8_t unicode_prop_Case_Ignorable_table[764] = {
    0xa6, 0x05, 0x80, 0x8a, 0x80, 0xa2, 0x00, 0x80,
    0xc6, 0x03, 0x00, 0x03, 0x01, 0x81, 0x41, 0xf6,
    0x40, 0xbf, 0x19, 0x18, 0x88, 0x08, 0x80, 0x40,
    0xfa, 0x86, 0x40, 0xce, 0x04, 0x80, 0xb0, 0xac,
    0x00, 0x01, 0x01, 0x00, 0xab, 0x80, 0x8a, 0x85,
    0x89, 0x8a, 0x00, 0xa2, 0x80, 0x89, 0x94, 0x8f,
    0x80, 0xe4, 0x38, 0x89, 0x03, 0xa0, 0x00, 0x80,
    0x9d, 0x9a, 0xda, 0x8a, 0xb9, 0x8a, 0x18, 0x08,
    0x97, 0x97, 0xaa, 0x82, 0xab, 0x06, 0x0c, 0x88,
    0xa8, 0xb9, 0xb6, 0x00, 0x03, 0x3b, 0x02, 0x86,
    0x89, 0x81, 0x8c, 0x80, 0x8e, 0x80, 0xb9, 0x03,
    0x1f, 0x80, 0x93, 0x81, 0x99, 0x01, 0x81, 0xb8,
    0x03, 0x0b, 0x09, 0x12, 0x80, 0x9d, 0x0a, 0x80,
    0x8a, 0x81, 0xb8, 0x03, 0x20, 0x0b, 0x80, 0x93,
    0x81, 0x95, 0x28, 0x80, 0xb9, 0x01, 0x00, 0x1f,
    0x06, 0x81, 0x8a, 0x81, 0x9d, 0x80, 0xbc, 0x80,
    0x8b, 0x80, 0xb1, 0x02, 0x80, 0xb6, 0x00, 0x14,
    0x10, 0x1e, 0x81, 0x8a, 0x81, 0x9c, 0x80, 0xb9,
    0x01, 0x05, 0x04, 0x81, 0x93, 0x81, 0x9b, 0x81,
    0xb8, 0x0b, 0x1f, 0x80, 0x93, 0x81, 0x9c, 0x80,
    0xc7, 0x06, 0x10, 0x80, 0xd9, 0x01, 0x86, 0x8a,
    0x88, 0xe1, 0x01, 0x88, 0x88, 0x00, 0x86, 0xc8,
    0x81, 0x9a, 0x00, 0x00, 0x80, 0xb6, 0x8d, 0x04,
    0x01, 0x84, 0x8a, 0x80, 0xa3, 0x88, 0x80, 0xe5,
    0x18, 0x28, 0x09, 0x81, 0x98, 0x0b, 0x82, 0x8f,
    0x83, 0x8c, 0x01, 0x0d, 0x80, 0x8e, 0x80, 0xdd,
    0x80, 0x42, 0x5f, 0x82, 0x43, 0xb1, 0x82, 0x9c,
    0x81, 0x9d, 0x81, 0x9d, 0x81, 0xbf, 0x08, 0x37,
    0x01, 0x8a, 0x10, 0x20, 0xac, 0x84, 0xb2, 0x80,
    0xc0, 0x81, 0xa1, 0x80, 0xf5, 0x13, 0x81, 0x88,
    0x05, 0x82, 0x40, 0xda, 0x09, 0x80, 0xb9, 0x00,
    0x30, 0x00, 0x01, 0x3d, 0x89, 0x08, 0xa6, 0x07,
    0x9e, 0xb0, 0x83, 0xaf, 0x00, 0x20, 0x04, 0x80,
    0xa7, 0x88, 0x8b, 0x81, 0x9f, 0x19, 0x08, 0x82,
    0xb7, 0x00, 0x0a, 0x00, 0x82, 0xb9, 0x39, 0x81,
    0xbf, 0x85, 0xd1, 0x10, 0x8c, 0x06, 0x18, 0x28,
    0x11, 0xb1, 0xbe, 0x8c, 0x80, 0xa1, 0xe4, 0x41,
    0xbc, 0x00, 0x82, 0x8a, 0x82, 0x8c, 0x82, 0x8c,
    0x82, 0x8c, 0x81, 0x8b, 0x27, 0x81, 0x89, 0x01,
    0x01, 0x84, 0xb0, 0x20, 0x89, 0x00, 0x8c, 0x80,
    0x8f, 0x8c, 0xb2, 0xa0, 0x4b, 0x8a, 0x81, 0xf0,
    0x82, 0xfc, 0x80, 0x8e, 0x80, 0xdf, 0x9f, 0xae,
    0x80, 0x41, 0xd4, 0x80, 0xa3, 0x1a, 0x24, 0x80,
    0xdc, 0x85, 0xdc, 0x82, 0x60, 0x6f, 0x15, 0x80,
    0x44, 0xe1, 0x85, 0x41, 0x0d, 0x80, 0xe1, 0x18,
    0x89, 0x00, 0x9b, 0x83, 0xcf, 0x81, 0x8d, 0xa1,
    0xcd, 0x80, 0x96, 0x82, 0xe6, 0x12, 0x0f, 0x02,
    0x03, 0x80, 0x98, 0x0c, 0x80, 0x40, 0x96, 0x81,
    0x99, 0x91, 0x8c, 0x80, 0xa5, 0x87, 0x98, 0x8a,
    0xad, 0x82, 0xaf, 0x01, 0x19, 0x81, 0x90, 0x80,
    0x94, 0x81, 0xc1, 0x29, 0x09, 0x81, 0x8b, 0x07,
    0x80, 0xa2, 0x80, 0x8a, 0x80, 0xb2, 0x00, 0x11,
    0x0c, 0x08, 0x80, 0x9a, 0x80, 0x8d, 0x0c, 0x08,
    0x80, 0xe3, 0x84, 0x88, 0x82, 0xf8, 0x01, 0x03,
    0x80, 0x60, 0x4f, 0x2f, 0x80, 0x40, 0x92, 0x90,
    0x42, 0x3c, 0x8f, 0x10, 0x8b, 0x8f, 0xa1, 0x01,
    0x80, 0x40, 0xa8, 0x06, 0x05, 0x80, 0x8a, 0x80,
    0xa2, 0x00, 0x80, 0xae, 0x80, 0xac, 0x81, 0xc2,
    0x80, 0x94, 0x82, 0x42, 0x00, 0x80, 0x40, 0xe1,
    0x80, 0x40, 0x94, 0x84, 0x44, 0x04, 0x28, 0xa9,
    0x80, 0x88, 0x42, 0x45, 0x10, 0x0c, 0x83, 0xa7,
    0x13, 0x80, 0x40, 0xa4, 0x81, 0x42, 0x3c, 0x83,
    0xa5, 0x80, 0x99, 0x20, 0x80, 0x41, 0x3a, 0x81,
    0xce, 0x83, 0xc5, 0x8a, 0xb0, 0x83, 0xfa, 0x80,
    0xb5, 0x8e, 0xa8, 0x01, 0x81, 0x89, 0x82, 0xb0,
    0x19, 0x09, 0x03, 0x80, 0x89, 0x80, 0xb1, 0x82,
    0xa3, 0x20, 0x87, 0xbd, 0x80, 0x8b, 0x81, 0xb3,
    0x88, 0x89, 0x19, 0x80, 0xde, 0x11, 0x00, 0x0d,
    0x01, 0x80, 0x40, 0x9c, 0x02, 0x87, 0x94, 0x81,
    0xb8, 0x0a, 0x80, 0xa4, 0x32, 0x84, 0xc5, 0x85,
    0x8c, 0x00, 0x00, 0x80, 0x8d, 0x81, 0xd4, 0x39,
    0x10, 0x80, 0x96, 0x80, 0xd3, 0x28, 0x03, 0x08,
    0x81, 0x40, 0xed, 0x1d, 0x08, 0x81, 0x9a, 0x81,
    0xd4, 0x39, 0x00, 0x81, 0xe9, 0x00, 0x01, 0x28,
    0x80, 0xe4, 0x00, 0x01, 0x18, 0x84, 0x41, 0x02,
    0x88, 0x01, 0x40, 0xff, 0x08, 0x03, 0x80, 0x40,
    0x8f, 0x19, 0x0b, 0x80, 0x9f, 0x89, 0xa7, 0x29,
    0x1f, 0x80, 0x88, 0x29, 0x82, 0xad, 0x8c, 0x01,
    0x41, 0x95, 0x30, 0x28, 0x80, 0xd1, 0x95, 0x0e,
    0x01, 0x01, 0xf9, 0x2a, 0x00, 0x08, 0x30, 0x80,
    0xc7, 0x0a, 0x00, 0x80, 0x41, 0x5a, 0x81, 0x8a,
    0x81, 0xb3, 0x24, 0x00, 0x80, 0x96, 0x80, 0x54,
    0xd4, 0x90, 0x85, 0x8e, 0x60, 0x2c, 0xc7, 0x8b,
    0x12, 0x49, 0xbf, 0x84, 0xba, 0x86, 0x88, 0x83,
    0x41, 0xfb, 0x82, 0xa7, 0x81, 0x41, 0xe1, 0x80,
    0xbe, 0x90, 0xbf, 0x08, 0x81, 0x60, 0x40, 0x0a,
    0x18, 0x30, 0x81, 0x4c, 0x9d, 0x08, 0x83, 0x52,
    0x5b, 0xad, 0x81, 0x96, 0x42, 0x1f, 0x82, 0x88,
    0x8f, 0x0e, 0x9d, 0x83, 0x40, 0x93, 0x82, 0x47,
    0xba, 0xb6, 0x83, 0xb1, 0x38, 0x8d, 0x80, 0x95,
    0x20, 0x8e, 0x45, 0x4f, 0x30, 0x90, 0x0e, 0x01,
    0x04, 0x84, 0xbd, 0xa0, 0x80, 0x40, 0x9f, 0x8d,
    0x41, 0x6f, 0x80, 0xbc, 0x83, 0x41, 0xfa, 0x84,
    0x40, 0xfd, 0x81, 0x42, 0xdf, 0x86, 0xec, 0x87,
    0x4a, 0xae, 0x84, 0x6c, 0x0c, 0x00, 0x80, 0x9d,
    0xdf, 0xff, 0x40, 0xef,
};

static const uint8_t unicode_prop_Case_Ignorable_index[72] = {
    0xbe, 0x05, 0x00, 0xfe, 0x07, 0x00, 0x52, 0x0a,
    0xa0, 0xc1, 0x0b, 0x00, 0x82, 0x0d, 0x00, 0x3f,
    0x10, 0x80, 0xd4, 0x17, 0x40, 0xcf, 0x1a, 0x20,
    0xf5, 0x1c, 0x00, 0x80, 0x20, 0x00, 0x16, 0xa0,
    0x00, 0xc6, 0xa8, 0x00, 0xc2, 0xaa, 0x60, 0x56,
    0xfe, 0x20, 0xb1, 0x07, 0x01, 0x02, 0x10, 0x01,
    0x42, 0x12, 0x41, 0xc4, 0x14, 0x21, 0xe1, 0x19,
    0x81, 0x48, 0x1d, 0x01, 0x44, 0x6b, 0x01, 0x83,
    0xd1, 0x21, 0x3e, 0xe1, 0x01, 0xf0, 0x01, 0x0e,
};

static const uint8_t unicode_prop_ID_Start_table[1133] = {
    0xc0, 0x99, 0x85, 0x99, 0xae, 0x80, 0x89, 0x03,
    0x04, 0x96, 0x80, 0x9e, 0x80, 0x41, 0xc9, 0x83,
    0x8b, 0x8d, 0x26, 0x00, 0x80, 0x40, 0x80, 0x20,
    0x09, 0x18, 0x05, 0x00, 0x10, 0x00, 0x93, 0x80,
    0xd2, 0x80, 0x40, 0x8a, 0x87, 0x40, 0xa5, 0x80,
    0xa5, 0x08, 0x85, 0xa8, 0xc6, 0x9a, 0x1b, 0xac,
    0xaa, 0xa2, 0x08, 0xe2, 0x00, 0x8e, 0x0e, 0x81,
    0x89, 0x11, 0x80, 0x8f, 0x00, 0x9d, 0x9c, 0xd8,
    0x8a, 0x80, 0x97, 0xa0, 0x88, 0x0b, 0x04, 0x95,
    0x18, 0x88, 0x02, 0x80, 0x96, 0x98, 0x86, 0x8a,
    0x84, 0x97, 0x05, 0x90, 0xa9, 0xb9, 0xb5, 0x10,
    0x91, 0x06, 0x89, 0x8e, 0x8f, 0x1f, 0x09, 0x81,
    0x95, 0x06, 0x00, 0x13, 0x10, 0x8f, 0x80, 0x8c,
    0x08, 0x82, 0x8d, 0x81, 0x89, 0x07, 0x2b, 0x09,
    0x95, 0x06, 0x01, 0x01, 0x01, 0x9e, 0x18, 0x80,
    0x92, 0x82, 0x8f, 0x88, 0x02, 0x80, 0x95, 0x06,
    0x01, 0x04, 0x10, 0x91, 0x80, 0x8e, 0x81, 0x96,
    0x80, 0x8a, 0x39, 0x09, 0x95, 0x06, 0x01, 0x04,
    0x10, 0x9d, 0x08, 0x82, 0x8e, 0x80, 0x90, 0x00,
    0x2a, 0x10, 0x1a, 0x08, 0x00, 0x0a, 0x0a, 0x12,
    0x8b, 0x95, 0x80, 0xb3, 0x38, 0x10, 0x96, 0x80,
    0x8f, 0x10, 0x99, 0x11, 0x01, 0x81, 0x9d, 0x03,
    0x38, 0x10, 0x96, 0x80, 0x89, 0x04, 0x10, 0x9e,
    0x08, 0x81, 0x8e, 0x81, 0x90, 0x88, 0x02, 0x80,
    0xa8, 0x08, 0x8f, 0x04, 0x17, 0x82, 0x97, 0x2c,
    0x91, 0x82, 0x97, 0x80, 0x88, 0x00, 0x0e, 0xb9,
    0xaf, 0x01, 0x8b, 0x86, 0xb9, 0x08, 0x00, 0x20,
    0x97, 0x00, 0x80, 0x89, 0x01, 0x88, 0x01, 0x20,
    0x80, 0x94, 0x83, 0x9f, 0x80, 0xbe, 0x38, 0xa3,
    0x9a, 0x84, 0xf2, 0xaa, 0x93, 0x80, 0x8f, 0x2b,
    0x1a, 0x02, 0x0e, 0x13, 0x8c, 0x8b, 0x80, 0x90,
    0xa5, 0x00, 0x20, 0x81, 0xaa, 0x80, 0x41, 0x4c,
    0x03, 0x0e, 0x00, 0x03, 0x81, 0xa8, 0x03, 0x81,
    0xa0, 0x03, 0x0e, 0x00, 0x03, 0x81, 0x8e, 0x80,
    0xb8, 0x03, 0x81, 0xc2, 0xa4, 0x8f, 0x8f, 0xd5,
    0x0d, 0x82, 0x42, 0x6b, 0x81, 0x90, 0x80, 0x99,
    0x84, 0xca, 0x82, 0x8a, 0x86, 0x91, 0x8c, 0x92,
    0x8d, 0x91, 0x8d, 0x8c, 0x02, 0x8e, 0xb3, 0xa2,
    0x03, 0x80, 0xc2, 0xd8, 0x86, 0xa8, 0x00, 0x84,
    0xc5, 0x89, 0x9e, 0xb0, 0x9d, 0x0c, 0x8a, 0xab,
    0x83, 0x99, 0xb5, 0x96, 0x88, 0xb4, 0xd1, 0x80,
    0xdc, 0xae, 0x90, 0x87, 0xb5, 0x9d, 0x8c, 0x81,
    0x89, 0xab, 0x99, 0xa3, 0xa8, 0x82, 0x89, 0xa3,
    0x81, 0x8a, 0x84, 0xaa, 0x0a, 0xa8, 0x18, 0x28,
    0x0a, 0x04, 0x40, 0xbf, 0xbf, 0x41, 0x15, 0x0d,
    0x81, 0xa5, 0x0d, 0x0f, 0x00, 0x00, 0x00, 0x80,
    0x9e, 0x81, 0xb4, 0x06, 0x00, 0x12, 0x06, 0x13,
    0x0d, 0x83, 0x8c, 0x22, 0x06, 0xf3, 0x80, 0x8c,
    0x80, 0x8f, 0x8c, 0xe4, 0x03, 0x01, 0x89, 0x00,
    0x0d, 0x28, 0x00, 0x00, 0x80, 0x8f, 0x0b, 0x24,
    0x18, 0x90, 0xa8, 0x4a, 0x76, 0x40, 0xe4, 0x2b,
    0x11, 0x8b, 0xa5, 0x00, 0x20, 0x81, 0xb7, 0x30,
    0x8f, 0x96, 0x88, 0x30, 0x30, 0x30, 0x30, 0x30,
    0x30, 0x30, 0x86, 0x42, 0x25, 0x82, 0x98, 0x88,
    0x34, 0x0c, 0x83, 0xd5, 0x1c, 0x80, 0xd9, 0x03,
    0x84, 0xaa, 0x80, 0xdd, 0x90, 0x9f, 0xaf, 0x8f,
    0x41, 0xff, 0x59, 0xbf, 0xbf, 0x60, 0x56, 0x8c,
    0xc2, 0xad, 0x81, 0x41, 0x0c, 0x82, 0x8f, 0x89,
    0x81, 0x93, 0xae, 0x8f, 0x9e, 0x81, 0xcf, 0xa6,
    0x88, 0x81, 0xe6, 0x81, 0xc2, 0x09, 0x00, 0x07,
    0x94, 0x8f, 0x02, 0x03, 0x80, 0x96, 0x9c, 0xb3,
    0x8d, 0xb1, 0xbd, 0x2a, 0x00, 0x81, 0x8a, 0x9b,
    0x89, 0x96, 0x98, 0x9c, 0x86, 0xae, 0x9b, 0x80,
    0x8f, 0x20, 0x89, 0x89, 0x20, 0xa8, 0x96, 0x10,
    0x87, 0x93, 0x96, 0x10, 0x82, 0xb1, 0x00, 0x11,
    0x0c, 0x08, 0x00, 0x97, 0x11, 0x8a, 0x32, 0x8b,
    0x29, 0x29, 0x85, 0x88, 0x30, 0x30, 0xaa, 0x80,
    0x8d, 0x85, 0xf2, 0x9c, 0x60, 0x2b, 0xa3, 0x8b,
    0x96, 0x83, 0xb0, 0x60, 0x21, 0x03, 0x41, 0x6d,
    0x81, 0xe9, 0xa5, 0x86, 0x8b, 0x24, 0x00, 0x89,
    0x80, 0x8c, 0x04, 0x00, 0x01, 0x01, 0x80, 0xeb,
    0xa0, 0x41, 0x6a, 0x91, 0xbf, 0x81, 0xb5, 0xa7,
    0x8b, 0xf3, 0x20, 0x40, 0x86, 0xa3, 0x99, 0x85,
    0x99, 0x8a, 0xd8, 0x15, 0x0d, 0x0d, 0x0a, 0xa2,
    0x8b, 0x80, 0x99, 0x80, 0x92, 0x01, 0x80, 0x8e,
    0x81, 0x8d, 0xa1, 0xfa, 0xc4, 0xb4, 0x41, 0x0a,
    0x9c, 0x82, 0xb0, 0xae, 0x9f, 0x8c, 0x9d, 0x84,
    0xa5, 0x89, 0x9d, 0x81, 0xa3, 0x1f, 0x04, 0xa9,
    0x40, 0x9d, 0x91, 0xa3, 0x83, 0xa3, 0x83, 0xa7,
    0x87, 0xb3, 0x8b, 0x8a, 0x80, 0x8e, 0x06, 0x01,
    0x80, 0x8a, 0x80, 0x8e, 0x06, 0x01, 0x82, 0xb3,
    0x8b, 0x41, 0x36, 0x88, 0x95, 0x89, 0x87, 0x97,
    0x28, 0xa9, 0x80, 0x88, 0xc4, 0x29, 0x00, 0xab,
    0x01, 0x10, 0x81, 0x96, 0x89, 0x96, 0x88, 0x9e,
    0xc0, 0x92, 0x01, 0x89, 0x95, 0x89, 0x99, 0xc5,
    0xb7, 0x29, 0xbf, 0x80, 0x8e, 0x18, 0x10, 0x9c,
    0xa9, 0x9c, 0x82, 0x9c, 0xa2, 0x38, 0x9b, 0x9a,
    0xb5, 0x89, 0x95, 0x89, 0x92, 0x8c, 0x91, 0xed,
    0xc8, 0xb6, 0xb2, 0x8c, 0xb2, 0x8c, 0xa3, 0xa5,
    0x9b, 0x88, 0x96, 0x40, 0xf9, 0xa9, 0x29, 0x8f,
    0x82, 0xba, 0x9c, 0x89, 0x07, 0x95, 0xa9, 0x91,
    0xad, 0x94, 0x9a, 0x96, 0x8b, 0xb4, 0xb8, 0x09,
    0x80, 0x8c, 0xac, 0x9f, 0x98, 0x99, 0xa3, 0x9c,
    0x01, 0x07, 0xa2, 0x10, 0x8b, 0xaf, 0x8d, 0x83,
    0x94, 0x00, 0x80, 0xa2, 0x91, 0x80, 0x98, 0x92,
    0x81, 0xbe, 0x30, 0x00, 0x18, 0x8e, 0x80, 0x89,
    0x86, 0xae, 0xa5, 0x39, 0x09, 0x95, 0x06, 0x01,
    0x04, 0x10, 0x91, 0x80, 0x8b, 0x84, 0x9d, 0x89,
    0x00, 0x08, 0x80, 0xa5, 0x00, 0x98, 0x00, 0x80,
    0xab, 0xb4, 0x91, 0x83, 0x93, 0x82, 0x9d, 0xaf,
    0x93, 0x08, 0x80, 0x40, 0xb7, 0xae, 0xa8, 0x83,
    0xa3, 0xaf, 0x93, 0x80, 0xba, 0xaa, 0x8c, 0x80,
    0xc6, 0x9a, 0xa4, 0x86, 0x40, 0xb8, 0xab, 0xf3,
    0xbf, 0x9e, 0x39, 0x01, 0x38, 0x08, 0x97, 0x8e,
    0x00, 0x80, 0xdd, 0x39, 0xa6, 0x8f, 0x00, 0x80,
    0x9b, 0x80, 0x89, 0xa7, 0x30, 0x94, 0x80, 0x8a,
    0xad, 0x92, 0x80, 0x91, 0xc8, 0x40, 0xc6, 0xa0,
    0x9e, 0x88, 0x80, 0xa4, 0x90, 0x80, 0xb0, 0x9d,
    0xef, 0x30, 0x08, 0xa5, 0x94, 0x80, 0x98, 0x28,
    0x08, 0x9f, 0x8d, 0x80, 0x41, 0x46, 0x92, 0x8e,
    0x00, 0x8c, 0x80, 0xa1, 0xfb, 0x80, 0xce, 0x43,
    0x99, 0xe5, 0xee, 0x90, 0x40, 0xc3, 0x4a, 0x4b,
    0xe0, 0x8e, 0x44, 0x2f, 0x90, 0x85, 0x98, 0x4f,
    0x9a, 0x84, 0x42, 0x46, 0x5a, 0xb8, 0x9d, 0x46,
    0xe1, 0x42, 0x38, 0x86, 0x9e, 0x90, 0xce, 0x90,
    0x9d, 0x91, 0xaf, 0x8f, 0x83, 0x9e, 0x94, 0x84,
    0x92, 0x41, 0xaf, 0xac, 0x40, 0xd2, 0xbf, 0xff,
    0xca, 0x20, 0xc1, 0x8c, 0xbf, 0x08, 0x80, 0x9b,
    0x57, 0xf7, 0x87, 0x44, 0xd5, 0xa8, 0x89, 0x60,
    0x22, 0xe6, 0x18, 0x30, 0x08, 0x41, 0x22, 0x8e,
    0x80, 0x9c, 0x11, 0x80, 0x8d, 0x1f, 0x41, 0x8b,
    0x49, 0x03, 0xea, 0x84, 0x8c, 0x82, 0x88, 0x86,
    0x89, 0x57, 0x65, 0xd4, 0x80, 0xc6, 0x01, 0x08,
    0x09, 0x0b, 0x80, 0x8b, 0x00, 0x06, 0x80, 0xc0,
    0x03, 0x0f, 0x06, 0x80, 0x9b, 0x03, 0x04, 0x00,
    0x16, 0x80, 0x41, 0x53, 0x81, 0x98, 0x80, 0x98,
    0x80, 0x9e, 0x80, 0x98, 0x80, 0x9e, 0x80, 0x98,
    0x80, 0x9e, 0x80, 0x98, 0x80, 0x9e, 0x80, 0x98,
    0x07, 0x47, 0x33, 0x9e, 0x2d, 0x41, 0x04, 0xbd,
    0x40, 0x91, 0xac, 0x89, 0x86, 0x8f, 0x80, 0x41,
    0x40, 0x9d, 0x91, 0xab, 0x41, 0xe3, 0x9b, 0x40,
    0xe3, 0x9d, 0x08, 0x41, 0xee, 0x30, 0x18, 0x08,
    0x8e, 0x80, 0x40, 0xc4, 0xba, 0xc3, 0x30, 0x44,
    0xb3, 0x18, 0x9a, 0x01, 0x00, 0x08, 0x80, 0x89,
    0x03, 0x00, 0x00, 0x28, 0x18, 0x00, 0x00, 0x02,
    0x01, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x0b, 0x06, 0x03, 0x03, 0x00, 0x80, 0x89,
    0x80, 0x90, 0x22, 0x04, 0x80, 0x90, 0x51, 0x43,
    0x60, 0xa6, 0xdf, 0x9f, 0x50, 0x39, 0x85, 0x40,
    0xdd, 0x81, 0x56, 0x81, 0x8d, 0x5d, 0x30, 0x8e,
    0x42, 0x6d, 0x49, 0xa1, 0x42, 0x1d, 0x45, 0xe1,
    0x53, 0x4a, 0x84, 0x50, 0x5f,
};

static const uint8_t unicode_prop_ID_Start_index[108] = {
    0xf6, 0x03, 0x20, 0xa6, 0x07, 0x00, 0xa9, 0x09,
    0x20, 0xb1, 0x0a, 0x00, 0xba, 0x0b, 0x20, 0x3b,
    0x0d, 0x20, 0xc7, 0x0e, 0x20, 0x49, 0x12, 0x00,
    0x9b, 0x16, 0x00, 0xac, 0x19, 0x00, 0xc0, 0x1d,
    0x80, 0x80, 0x20, 0x20, 0x70, 0x2d, 0x00, 0x00,
    0x32, 0x00, 0xdd, 0xa7, 0x00, 0x4c, 0xaa, 0x20,
    0xc7, 0xd7, 0x20, 0xfc, 0xfd, 0x20, 0x9d, 0x02,
    0x21, 0x96, 0x05, 0x01, 0x9f, 0x08, 0x01, 0x49,
    0x0c, 0x21, 0x76, 0x10, 0x21, 0xa9, 0x12, 0x01,
    0xb0, 0x14, 0x01, 0x42, 0x19, 0x41, 0x90, 0x1c,
    0x01, 0xf1, 0x2f, 0x21, 0x90, 0x6b, 0x21, 0x33,
    0xb1, 0x21, 0x06, 0xd5, 0x01, 0xc3, 0xd7, 0x01,
    0xff, 0xe7, 0x21, 0x63, 0xee, 0x01, 0x5e, 0xee,
    0x42, 0xb0, 0x23, 0x03,
};

static const uint8_t unicode_prop_ID_Continue1_table[695] = {
    0xaf, 0x89, 0xa4, 0x80, 0xd6, 0x80, 0x42, 0x47,
    0xef, 0x96, 0x80, 0x40, 0xfa, 0x84, 0x41, 0x08,
    0xac, 0x00, 0x01, 0x01, 0x00, 0xc7, 0x8a, 0xaf,
    0x9e, 0x28, 0xe4, 0x31, 0x29, 0x08, 0x19, 0x89,
    0x96, 0x80, 0x9d, 0x9a, 0xda, 0x8a, 0x8e, 0x89,
    0xa0, 0x88, 0x88, 0x80, 0x97, 0x18, 0x88, 0x02,
    0x04, 0xaa, 0x82, 0xba, 0x88, 0xa9, 0x97, 0x80,
    0xa0, 0xb5, 0x10, 0x91, 0x06, 0x89, 0x09, 0x89,
    0x90, 0x82, 0xb7, 0x00, 0x31, 0x09, 0x82, 0x88,
    0x80, 0x89, 0x09, 0x89, 0x8d, 0x01, 0x82, 0xb7,
    0x00, 0x23, 0x09, 0x12, 0x80, 0x93, 0x8b, 0x10,
    0x8a, 0x82, 0xb7, 0x00, 0x38, 0x10, 0x82, 0x93,
    0x09, 0x89, 0x89, 0x28, 0x82, 0xb7, 0x00, 0x31,
    0x09, 0x16, 0x82, 0x89, 0x09, 0x89, 0x91, 0x80,
    0xba, 0x22, 0x10, 0x83, 0x88, 0x80, 0x8d, 0x89,
    0x8f, 0x84, 0xb6, 0x00, 0x30, 0x10, 0x1e, 0x81,
    0x8a, 0x09, 0x89, 0x90, 0x82, 0xb7, 0x00, 0x30,
    0x10, 0x1e, 0x81, 0x8a, 0x09, 0x89, 0x10, 0x8b,
    0x83, 0xb6, 0x08, 0x30, 0x10, 0x83, 0x88, 0x80,
    0x89, 0x09, 0x89, 0x90, 0x82, 0xc5, 0x03, 0x28,
    0x00, 0x3d, 0x89, 0x09, 0xbc, 0x01, 0x86, 0x8b,
    0x38, 0x89, 0xd6, 0x01, 0x88, 0x8a, 0x30, 0x89,
    0xbd, 0x0d, 0x89, 0x8a, 0x00, 0x00, 0x03, 0x81,
    0xb0, 0x93, 0x01, 0x84, 0x8a, 0x80, 0xa3, 0x88,
    0x80, 0xe3, 0x93, 0x80, 0x89, 0x8b, 0x1b, 0x10,
    0x11, 0x32, 0x83, 0x8c, 0x8b, 0x80, 0x8e, 0x42,
    0xbe, 0x82, 0x88, 0x88, 0x43, 0x9f, 0x83, 0x9b,
    0x82, 0x9c, 0x81, 0x9d, 0x81, 0xbf, 0x9f, 0x88,
    0x01, 0x89, 0xa0, 0x10, 0x8a, 0x40, 0x8e, 0x80,
    0xf5, 0x8b, 0x83, 0x8b, 0x89, 0x89, 0xff, 0x8a,
    0xbb, 0x84, 0xb8, 0x89, 0x80, 0x9c, 0x81, 0x8a,
    0x85, 0x89, 0x95, 0x8d, 0x80, 0x8f, 0xb0, 0x84,
    0xae, 0x90, 0x8a, 0x89, 0x90, 0x88, 0x8b, 0x82,
    0x9d, 0x8c, 0x81, 0x89, 0xab, 0x8d, 0xaf, 0x93,
    0x87, 0x89, 0x85, 0x89, 0xf5, 0x10, 0x94, 0x18,
    0x28, 0x0a, 0x40, 0xc5, 0xbf, 0x42, 0x0b, 0x81,
    0xb0, 0x81, 0x92, 0x80, 0xfa, 0x8c, 0x18, 0x82,
    0x8b, 0x4b, 0xfd, 0x82, 0x40, 0x8c, 0x80, 0xdf,
    0x9f, 0x42, 0x29, 0x85, 0xe8, 0x81, 0xdf, 0x80,
    0x60, 0x75, 0x23, 0x89, 0xc4, 0x03, 0x89, 0x9f,
    0x81, 0xcf, 0x81, 0x41, 0x0f, 0x02, 0x03, 0x80,
    0x96, 0x23, 0x80, 0xd2, 0x81, 0xb1, 0x91, 0x89,
    0x89, 0x85, 0x91, 0x8c, 0x8a, 0x9b, 0x87, 0x98,
    0x8c, 0xab, 0x83, 0xae, 0x8d, 0x8e, 0x89, 0x8a,
    0x80, 0x89, 0x89, 0xae, 0x8d, 0x8b, 0x07, 0x09,
    0x89, 0xa0, 0x82, 0xb1, 0x00, 0x11, 0x0c, 0x08,
    0x80, 0xa8, 0x24, 0x81, 0x40, 0xeb, 0x38, 0x09,
    0x89, 0x60, 0x4f, 0x23, 0x80, 0x42, 0xe0, 0x8f,
    0x8f, 0x8f, 0x11, 0x97, 0x82, 0x40, 0xbf, 0x89,
    0xa4, 0x80, 0xa4, 0x80, 0x42, 0x96, 0x80, 0x40,
    0xe1, 0x80, 0x40, 0x94, 0x84, 0x41, 0x24, 0x89,
    0x45, 0x56, 0x10, 0x0c, 0x83, 0xa7, 0x13, 0x80,
    0x40, 0xa4, 0x81, 0x42, 0x3c, 0x1f, 0x89, 0x85,
    0x89, 0x9e, 0x84, 0x41, 0x3c, 0x81, 0xce, 0x83,
    0xc5, 0x8a, 0xb0, 0x83, 0xf9, 0x82, 0xb4, 0x8e,
    0x9e, 0x8a, 0x09, 0x89, 0x83, 0xac, 0x8a, 0x30,
    0xac, 0x89, 0x2a, 0xa3, 0x8d, 0x80, 0x89, 0x21,
    0xab, 0x80, 0x8b, 0x82, 0xaf, 0x8d, 0x3b, 0x80,
    0x8b, 0xd1, 0x8b, 0x28, 0x08, 0x40, 0x9c, 0x8b,
    0x84, 0x89, 0x2b, 0xb6, 0x08, 0x31, 0x09, 0x82,
    0x88, 0x80, 0x89, 0x09, 0x32, 0x84, 0xc2, 0x88,
    0x00, 0x08, 0x03, 0x04, 0x00, 0x8d, 0x81, 0xd1,
    0x91, 0x88, 0x89, 0x18, 0xd0, 0x93, 0x8b, 0x89,
    0x40, 0xd4, 0x31, 0x88, 0x9a, 0x81, 0xd1, 0x90,
    0x8e, 0x89, 0xd0, 0x8c, 0x87, 0x89, 0x85, 0x93,
    0xb8, 0x8e, 0x83, 0x89, 0x40, 0xf1, 0x8e, 0x40,
    0xa4, 0x89, 0xc5, 0x28, 0x09, 0x18, 0x00, 0x81,
    0x8b, 0x89, 0xf6, 0x31, 0x32, 0x80, 0x9b, 0x89,
    0xa7, 0x30, 0x1f, 0x80, 0x88, 0x8a, 0xad, 0x8f,
    0x41, 0x55, 0x89, 0xb4, 0x38, 0x87, 0x8f, 0x89,
    0xb7, 0x95, 0x80, 0x8d, 0xf9, 0x2a, 0x00, 0x08,
    0x30, 0x07, 0x89, 0xaf, 0x20, 0x08, 0x27, 0x89,
    0x41, 0x48, 0x83, 0x88, 0x08, 0x80, 0xaf, 0x32,
    0x84, 0x8c, 0x8a, 0x54, 0xe4, 0x05, 0x8e, 0x60,
    0x2c, 0xc7, 0x9b, 0x49, 0x25, 0x89, 0xd5, 0x89,
    0xa5, 0x84, 0xba, 0x86, 0x98, 0x89, 0x42, 0x15,
    0x89, 0x41, 0xd4, 0x00, 0xb6, 0x33, 0xd0, 0x80,
    0x8a, 0x81, 0x60, 0x4c, 0xaa, 0x81, 0x50, 0x50,
    0x89, 0x42, 0x05, 0xad, 0x81, 0x96, 0x42, 0x1d,
    0x22, 0x2f, 0x39, 0x86, 0x9d, 0x83, 0x40, 0x93,
    0x82, 0x45, 0x88, 0xb1, 0x41, 0xff, 0xb6, 0x83,
    0xb1, 0x38, 0x8d, 0x80, 0x95, 0x20, 0x8e, 0x45,
    0x4f, 0x30, 0x90, 0x0e, 0x01, 0x04, 0xe3, 0x80,
    0x40, 0x9f, 0x86, 0x88, 0x89, 0x41, 0x63, 0x80,
    0xbc, 0x8d, 0x41, 0xf1, 0x8d, 0x40, 0xf3, 0x08,
    0x89, 0x42, 0xd4, 0x86, 0xec, 0x34, 0x89, 0x52,
    0x95, 0x89, 0x6c, 0x05, 0x05, 0x40, 0xef,
};

static const uint8_t unicode_prop_ID_Continue1_index[66] = {
    0xfa, 0x06, 0x00, 0x70, 0x09, 0x00, 0xf0, 0x0a,
    0x40, 0x57, 0x0c, 0x00, 0xf0, 0x0d, 0x60, 0xc7,
    0x0f, 0x20, 0xea, 0x17, 0x40, 0x05, 0x1b, 0x00,
    0x0e, 0x20, 0x00, 0xa0, 0xa6, 0x20, 0xe6, 0xa9,
    0x20, 0x10, 0xfe, 0x00, 0x40, 0x0a, 0x01, 0xc3,
    0x10, 0x01, 0x4e, 0x13, 0x01, 0x41, 0x16, 0x01,
    0x0b, 0x1a, 0x01, 0xaa, 0x1d, 0x01, 0x7a, 0x6d,
    0x21, 0x45, 0xd2, 0x21, 0xaf, 0xe2, 0x01, 0xf0,
    0x01, 0x0e,
};

static const uint8_t unicode_prop_White_Space_table[22] = {
    0x88, 0x84, 0x91, 0x80, 0xe3, 0x80, 0x99, 0x80,
    0x55, 0xde, 0x80, 0x49, 0x7e, 0x8a, 0x9c, 0x0c,
    0x80, 0xae, 0x80, 0x4f, 0x9f, 0x80,
};

static const uint8_t unicode_prop_White_Space_index[3] = {
    0x01, 0x30, 0x00,
};

static const uint8_t unicode_cc_table[916] = {
    0xb2, 0xcf, 0xd4, 0x00, 0xe8, 0x03, 0xdc, 0x00,
    0xe8, 0x00, 0xd8, 0x04, 0xdc, 0x01, 0xca, 0x03,
    0xdc, 0x01, 0xca, 0x0a, 0xdc, 0x04, 0x01, 0x03,
    0xdc, 0xc7, 0x00, 0xf0, 0xc0, 0x02, 0xdc, 0xc2,
    0x01, 0xdc, 0x80, 0xc2, 0x03, 0xdc, 0xc0, 0x00,
    0xe8, 0x01, 0xdc, 0xc0, 0x41, 0xe9, 0x00, 0xea,
    0x41, 0xe9, 0x00, 0xea, 0x00, 0xe9, 0xcc, 0xb0,
    0xe2, 0xc4, 0xb0, 0xd8, 0x00, 0xdc, 0xc3, 0x00,
    0xdc, 0xc2, 0x00, 0xde, 0x00, 0xdc, 0xc5, 0x05,
    0xdc, 0xc1, 0x00, 0xdc, 0xc1, 0x00, 0xde, 0x00,
    0xe4, 0xc0, 0x49, 0x0a, 0x43, 0x13, 0x80, 0x00,
    0x17, 0x80, 0x41, 0x18, 0x80, 0xc0, 0x00, 0xdc,
    0x80, 0x00, 0x12, 0xb0, 0x17, 0xc7, 0x42, 0x1e,
    0xaf, 0x47, 0x1b, 0xc1, 0x01, 0xdc, 0xc4, 0x00,
    0xdc, 0xc1, 0x00, 0xdc, 0x8f, 0x00, 0x23, 0xb0,
    0x34, 0xc6, 0x81, 0xc3, 0x00, 0xdc, 0xc0, 0x81,
    0xc1, 0x80, 0x00, 0xdc, 0xc1, 0x00, 0xdc, 0xa2,
    0x00, 0x24, 0x9d, 0xc0, 0x00, 0xdc, 0xc1, 0x00,
    0xdc, 0xc1, 0x02, 0xdc, 0xc0, 0x01, 0xdc, 0xc0,
    0x00, 0xdc, 0xc2, 0x00, 0xdc, 0xc0, 0x00, 0xdc,
    0xc0, 0x00, 0xdc, 0xc0, 0x00, 0xdc, 0xc1, 0xb0,
    0x6f, 0xc6, 0x00, 0xdc, 0xc0, 0x88, 0x00, 0xdc,
    0x97, 0xc3, 0x80, 0xc8, 0x80, 0xc2, 0x80, 0xc4,
    0xaa, 0x02, 0xdc, 0xb0, 0x0a, 0xc1, 0x02, 0xdc,
    0xc3, 0xa9, 0xc4, 0x04, 0xdc, 0xcd, 0x80, 0x00,
    0xdc, 0xc1, 0x00, 0xdc, 0xc1, 0x00, 0xdc, 0xc2,
    0x02, 0xdc, 0x42, 0x1b, 0xc2, 0x00, 0xdc, 0xc1,
    0x01, 0xdc, 0xc4, 0xb0, 0x0b, 0x00, 0x07, 0x8f,
    0x00, 0x09, 0x82, 0xc0, 0x00, 0xdc, 0xc1, 0xb0,
    0x36, 0x00, 0x07, 0x8f, 0x00, 0x09, 0xaf, 0xc0,
    0xb0, 0x0c, 0x00, 0x07, 0x8f, 0x00, 0x09, 0xb0,
    0x3d, 0x00, 0x07, 0x8f, 0x00, 0x09, 0xb0, 0x3d,
    0x00, 0x07, 0x8f, 0x00, 0x09, 0xb0, 0x4e, 0x00,
    0x09, 0xb0, 0x3d, 0x00, 0x07, 0x8f, 0x00, 0x09,
    0x86, 0x00, 0x54, 0x00, 0x5b, 0xb0, 0x34, 0x00,
    0x07, 0x8f, 0x00, 0x09, 0xb0, 0x3c, 0x01, 0x09,
    0x8f, 0x00, 0x09, 0xb0, 0x4b, 0x00, 0x09, 0xb0,
    0x3c, 0x01, 0x67, 0x00, 0x09, 0x8c, 0x03, 0x6b,
    0xb0, 0x3b, 0x01, 0x76, 0x00, 0x09, 0x8c, 0x03,
    0x7a, 0xb0, 0x1b, 0x01, 0xdc, 0x9a, 0x00, 0xdc,
    0x80, 0x00, 0xdc, 0x80, 0x00, 0xd8, 0xb0, 0x06,
    0x41, 0x81, 0x80, 0x00, 0x84, 0x84, 0x03, 0x82,
    0x81, 0x00, 0x82, 0x80, 0xc1, 0x00, 0x09, 0x80,
    0xc1, 0xb0, 0x0d, 0x00, 0xdc, 0xb0, 0x3f, 0x00,
    0x07, 0x80, 0x01, 0x09, 0xb0, 0x21, 0x00, 0xdc,
    0xb2, 0x9e, 0xc2, 0xb3, 0x83, 0x01, 0x09, 0x9d,
    0x00, 0x09, 0xb0, 0x6c, 0x00, 0x09, 0x89, 0xc0,
    0xb0, 0x9a, 0x00, 0xe4, 0xb0, 0x5e, 0x00, 0xde,
    0xc0, 0x00, 0xdc, 0xb0, 0xaa, 0xc0, 0x00, 0xdc,
    0xb0, 0x16, 0x00, 0x09, 0x93, 0xc7, 0x81, 0x00,
    0xdc, 0xaf, 0xc4, 0x05, 0xdc, 0xc1, 0x00, 0xdc,
    0x80, 0x01, 0xdc, 0xc1, 0x01, 0xdc, 0xc4, 0x00,
    0xdc, 0xc3, 0xb0, 0x34, 0x00, 0x07, 0x8e, 0x00,
    0x09, 0xa5, 0xc0, 0x00, 0xdc, 0xc6, 0xb0, 0x05,
    0x01, 0x09, 0xb0, 0x09, 0x00, 0x07, 0x8a, 0x01,
    0x09, 0xb0, 0x12, 0x00, 0x07, 0xb0, 0x67, 0xc2,
    0x41, 0x00, 0x04, 0xdc, 0xc1, 0x03, 0xdc, 0xc0,
    0x41, 0x00, 0x05, 0x01, 0x83, 0x00, 0xdc, 0x85,
    0xc0, 0x82, 0xc1, 0xb0, 0x95, 0xc1, 0x00, 0xdc,
    0xc6, 0x00, 0xdc, 0xc1, 0x00, 0xea, 0x00, 0xd6,
    0x00, 0xdc, 0x00, 0xca, 0xe4, 0x00, 0xe8, 0x01,
    0xe4, 0x00, 0xdc, 0x00, 0xda, 0xc0, 0x00, 0xe9,
    0x00, 0xdc, 0xc0, 0x00, 0xdc, 0xb2, 0x9f, 0xc1,
    0x01, 0x01, 0xc3, 0x02, 0x01, 0xc1, 0x83, 0xc0,
    0x82, 0x01, 0x01, 0xc0, 0x00, 0xdc, 0xc0, 0x01,
    0x01, 0x03, 0xdc, 0xc0, 0xb8, 0x03, 0xcd, 0xc2,
    0xb0, 0x5c, 0x00, 0x09, 0xb0, 0x2f, 0xdf, 0xb1,
    0xf9, 0x00, 0xda, 0x00, 0xe4, 0x00, 0xe8, 0x00,
    0xde, 0x01, 0xe0, 0xb0, 0x38, 0x01, 0x08, 0xb8,
    0x6d, 0xa3, 0xc0, 0x83, 0xc9, 0x9f, 0xc1, 0xb0,
    0x1f, 0xc1, 0xb0, 0xe3, 0x00, 0x09, 0xa4, 0x00,
    0x09, 0xb0, 0x66, 0x00, 0x09, 0x9a, 0xd1, 0xb0,
    0x08, 0x02, 0xdc, 0xa4, 0x00, 0x09, 0xb0, 0x2e,
    0x00, 0x07, 0x8b, 0x00, 0x09, 0xb0, 0xbe, 0xc0,
    0x80, 0xc1, 0x00, 0xdc, 0x81, 0xc1, 0x84, 0xc1,
    0x80, 0xc0, 0xb0, 0x03, 0x00, 0x09, 0xb0, 0xc5,
    0x00, 0x09, 0xb8, 0x46, 0xff, 0x00, 0x1a, 0xb2,
    0xd0, 0xc6, 0x06, 0xdc, 0xc1, 0xb3, 0x9c, 0x00,
    0xdc, 0xb0, 0xb1, 0x00, 0xdc, 0xb0, 0x64, 0xc4,
    0xb6, 0x61, 0x00, 0xdc, 0x80, 0xc0, 0xa7, 0xc0,
    0x00, 0x01, 0x00, 0xdc, 0x83, 0x00, 0x09, 0xb0,
    0x74, 0xc0, 0x00, 0xdc, 0xb2, 0x0c, 0xc3, 0xb0,
    0x10, 0xc4, 0xb1, 0x0c, 0xc1, 0xb0, 0x1f, 0x02,
    0xdc, 0xb0, 0x15, 0x01, 0xdc, 0xc2, 0x00, 0xdc,
    0xc0, 0x03, 0xdc, 0xb0, 0x00, 0xc0, 0x00, 0xdc,
    0xc0, 0x00, 0xdc, 0xb0, 0x8f, 0x00, 0x09, 0xa8,
    0x00, 0x09, 0x8d, 0x00, 0x09, 0xb0, 0x08, 0x00,
    0x09, 0x00, 0x07, 0xb0, 0x14, 0xc2, 0xaf, 0x01,
    0x09, 0xb0, 0x0d, 0x00, 0x07, 0xb0, 0x1b, 0x00,
    0x09, 0x88, 0x00, 0x07, 0xb0, 0x39, 0x00, 0x09,
    0x00, 0x07, 0xb0, 0x81, 0x00, 0x07, 0x00, 0x09,
    0xb0, 0x1f, 0x01, 0x07, 0x8f, 0x00, 0x09, 0x97,
    0xc6, 0x82, 0xc4, 0xb0, 0x28, 0x02, 0x09, 0xb0,
    0x40, 0x00, 0x09, 0x82, 0x00, 0x07, 0x96, 0xc0,
    0xb0, 0x32, 0x00, 0x09, 0x00, 0x07, 0xb0, 0xca,
    0x00, 0x09, 0x00, 0x07, 0xb0, 0x4d, 0x00, 0x09,
    0xb0, 0x45, 0x00, 0x09, 0x00, 0x07, 0xb0, 0x42,
    0x00, 0x09, 0xb0, 0xdc, 0x00, 0x09, 0x00, 0x07,
    0xb0, 0xd1, 0x01, 0x09, 0x83, 0x00, 0x07, 0xb0,
    0x6b, 0x00, 0x09, 0xb0, 0x22, 0x00, 0x09, 0x91,
    0x00, 0x09, 0xb0, 0x20, 0x00, 0x09, 0xb1, 0x74,
    0x00, 0x09, 0xb0, 0xd1, 0x00, 0x07, 0x80, 0x01,
    0x09, 0xb0, 0x20, 0x00, 0x09, 0xb1, 0x78, 0x01,
    0x09, 0xb8, 0x39, 0xbb, 0x00, 0x09, 0xb8, 0x01,
    0x8f, 0x04, 0x01, 0xb0, 0x0a, 0xc6, 0xb4, 0x88,
    0x01, 0x06, 0xb8, 0x44, 0x7b, 0x00, 0x01, 0xb8,
    0x0c, 0x95, 0x01, 0xd8, 0x02, 0x01, 0x82, 0x00,
    0xe2, 0x04, 0xd8, 0x87, 0x07, 0xdc, 0x81, 0xc4,
    0x01, 0xdc, 0x9d, 0xc3, 0xb0, 0x63, 0xc2, 0xb8,
    0x05, 0x8a, 0xc6, 0x80, 0xd0, 0x81, 0xc6, 0x80,
    0xc1, 0x80, 0xc4, 0xb0, 0x33, 0xc0, 0xb0, 0x6f,
    0xc6, 0xb1, 0x46, 0xc0, 0xb0, 0x0c, 0xc3, 0xb1,
    0xcb, 0x01, 0xe8, 0x00, 0xdc, 0xc0, 0xb0, 0xcd,
    0xc0, 0x00, 0xdc, 0xb2, 0xaf, 0x06, 0xdc, 0xb0,
    0x3c, 0xc5, 0x00, 0x07,
};

static const uint8_t unicode_cc_index[87] = {
    0x4d, 0x03, 0x00, 0x97, 0x05, 0x20, 0xc6, 0x05,
    0x00, 0xe7, 0x06, 0x00, 0x45, 0x07, 0x00, 0x9c,
    0x08, 0x00, 0x4d, 0x09, 0x00, 0x3c, 0x0b, 0x00,
    0x3d, 0x0d, 0x00, 0x36, 0x0f, 0x00, 0x38, 0x10,
    0x20, 0x3a, 0x19, 0x00, 0xcb, 0x1a, 0x20, 0xd3,
    0x1c, 0x00, 0xcf, 0x1d, 0x00, 0xe2, 0x20, 0x00,
    0x2e, 0x30, 0x20, 0x2b, 0xa9, 0x20, 0xed, 0xab,
    0x00, 0x39, 0x0a, 0x01, 0x4c, 0x0f, 0x01, 0x35,
    0x11, 0x21, 0x66, 0x13, 0x01, 0x40, 0x16, 0x01,
    0x47, 0x1a, 0x01, 0xf0, 0x6a, 0x21, 0x8a, 0xd1,
    0x01, 0xec, 0xe4, 0x21, 0x4b, 0xe9, 0x01,
};

static const uint32_t unicode_decomp_table1[709] = {
    0x00280081, 0x002a0097, 0x002a8081, 0x002bc097,
    0x002c8115, 0x002d0097, 0x002d4081, 0x002e0097,
    0x002e4115, 0x002f0199, 0x00302016, 0x00400842,
    0x00448a42, 0x004a0442, 0x004c0096, 0x004c8117,
    0x004d0242, 0x004e4342, 0x004fc12f, 0x0050c342,
    0x005240bf, 0x00530342, 0x00550942, 0x005a0842,
    0x005e0096, 0x005e4342, 0x005fc081, 0x00680142,
    0x006bc142, 0x00710185, 0x0071c317, 0x00734844,
    0x00778344, 0x00798342, 0x007b02be, 0x007c4197,
    0x007d0142, 0x007e0444, 0x00800e42, 0x00878142,
    0x00898744, 0x00ac0483, 0x00b60317, 0x00b80283,
    0x00d00214, 0x00d10096, 0x00dd0080, 0x00de8097,
    0x00df8080, 0x00e10097, 0x00e1413e, 0x00e1c080,
    0x00e204be, 0x00ea83ae, 0x00f282ae, 0x00f401ad,
    0x00f4c12e, 0x00f54103, 0x00fc0303, 0x00fe4081,
    0x0100023e, 0x0101c0be, 0x010301be, 0x010640be,
    0x010e40be, 0x0114023e, 0x0115c0be, 0x011701be,
    0x011d8144, 0x01304144, 0x01340244, 0x01358144,
    0x01368344, 0x01388344, 0x013a8644, 0x013e0144,
    0x0161c085, 0x018882ae, 0x019d422f, 0x01b00184,
    0x01b4c084, 0x024a4084, 0x024c4084, 0x024d0084,
    0x0256042e, 0x0272c12e, 0x02770120, 0x0277c084,
    0x028cc084, 0x028d8084, 0x029641ae, 0x02978084,
    0x02d20084, 0x02d2c12e, 0x02d70120, 0x02e50084,
    0x02f281ae, 0x03120084, 0x03300084, 0x0331c122,
    0x0332812e, 0x035281ae, 0x03768084, 0x037701ae,
    0x038cc085, 0x03acc085, 0x03b7012f, 0x03c30081,
    0x03d0c084, 0x03d34084, 0x03d48084, 0x03d5c084,
    0x03d70084, 0x03da4084, 0x03dcc084, 0x03dd412e,
    0x03ddc085, 0x03de0084, 0x03de4085, 0x03e04084,
    0x03e4c084, 0x03e74084, 0x03e88084, 0x03e9c084,
    0x03eb0084, 0x03ee4084, 0x04098084, 0x043f0081,
    0x06c18484, 0x06c48084, 0x06cec184, 0x06d00120,
    0x06d0c084, 0x074b0383, 0x074cc41f, 0x074f1783,
    0x075e0081, 0x0766d283, 0x07801d44, 0x078e8942,
    0x07931844, 0x079f0d42, 0x07a58216, 0x07a68085,
    0x07a6c0be, 0x07a80d44, 0x07aea044, 0x07c00122,
    0x07c08344, 0x07c20122, 0x07c28344, 0x07c40122,
    0x07c48244, 0x07c60122, 0x07c68244, 0x07c8113e,
    0x07d08244, 0x07d20122, 0x07d28244, 0x07d40122,
    0x07d48344, 0x07d64c3e, 0x07dc4080, 0x07dc80be,
    0x07dcc080, 0x07dd00be, 0x07dd4080, 0x07dd80be,
    0x07ddc080, 0x07de00be, 0x07de4080, 0x07de80be,
    0x07dec080, 0x07df00be, 0x07df4080, 0x07e00820,
    0x07e40820, 0x07e80820, 0x07ec05be, 0x07eec080,
    0x07ef00be, 0x07ef4097, 0x07ef8080, 0x07efc117,
    0x07f0443e, 0x07f24080, 0x07f280be, 0x07f2c080,
    0x07f303be, 0x07f4c080, 0x07f582ae, 0x07f6c080,
    0x07f7433e, 0x07f8c080, 0x07f903ae, 0x07fac080,
    0x07fb013e, 0x07fb8102, 0x07fc83be, 0x07fe4080,
    0x07fe80be, 0x07fec080, 0x07ff00be, 0x07ff4080,
    0x07ff8097, 0x0800011e, 0x08008495, 0x08044081,
    0x0805c097, 0x08090081, 0x08094097, 0x08098099,
    0x080bc081, 0x080cc085, 0x080d00b1, 0x080d8085,
    0x080dc0b1, 0x080f0197, 0x0811c197, 0x0815c0b3,
    0x0817c081, 0x081c0595, 0x081ec081, 0x081f0215,
    0x0820051f, 0x08228583, 0x08254415, 0x082a0097,
    0x08400119, 0x08408081, 0x0840c0bf, 0x08414119,
    0x0841c081, 0x084240bf, 0x0842852d, 0x08454081,
    0x08458097, 0x08464295, 0x08480097, 0x08484099,
    0x08488097, 0x08490081, 0x08498080, 0x084a0081,
    0x084a8102, 0x084b0495, 0x084d421f, 0x084e4081,
    0x084ec099, 0x084f0283, 0x08514295, 0x08540119,
    0x0854809b, 0x0854c619, 0x0857c097, 0x08580081,
    0x08584097, 0x08588099, 0x0858c097, 0x08590081,
    0x08594097, 0x08598099, 0x0859c09b, 0x085a0097,
    0x085a4081, 0x085a8097, 0x085ac099, 0x085b0295,
    0x085c4097, 0x085c8099, 0x085cc097, 0x085d0081,
    0x085d4097, 0x085d8099, 0x085dc09b, 0x085e0097,
    0x085e4081, 0x085e8097, 0x085ec099, 0x085f0215,
    0x08624099, 0x0866813e, 0x086b80be, 0x087341be,
    0x088100be, 0x088240be, 0x088300be, 0x088901be,
    0x088b0085, 0x088b40b1, 0x088bc085, 0x088c00b1,
    0x089040be, 0x089100be, 0x0891c1be, 0x089801be,
    0x089b42be, 0x089d0144, 0x089e0144, 0x08a00144,
    0x08a10144, 0x08a20144, 0x08ab023e, 0x08b80244,
    0x08ba8220, 0x08ca411e, 0x0918049f, 0x091a4523,
    0x091cc097, 0x091d04a5, 0x091f452b, 0x0921c09b,
    0x092204a1, 0x09244525, 0x0926c099, 0x09270d25,
    0x092d8d1f, 0x09340d1f, 0x093a8081, 0x0a8300b3,
    0x0a9d0099, 0x0a9d4097, 0x0a9d8099, 0x0ab700be,
    0x0b1f0115, 0x0b5bc081, 0x0ba7c081, 0x0bbcc081,
    0x0bc004ad, 0x0bc244ad, 0x0bc484ad, 0x0bc6f383,
    0x0be0852d, 0x0be31d03, 0x0bf1882d, 0x0c000081,
    0x0c0d8283, 0x0c130b84, 0x0c194284, 0x0c1c0122,
    0x0c1cc122, 0x0c1d8122, 0x0c1e4122, 0x0c1f0122,
    0x0c250084, 0x0c26c123, 0x0c278084, 0x0c27c085,
    0x0c2b0b84, 0x0c314284, 0x0c340122, 0x0c34c122,
    0x0c358122, 0x0c364122, 0x0c370122, 0x0c3d0084,
    0x0c3dc220, 0x0c3f8084, 0x0c3fc085, 0x0c4c4a2d,
    0x0c51451f, 0x0c53ca9f, 0x0c5915ad, 0x0c648703,
    0x0c800741, 0x0c838089, 0x0c83c129, 0x0c8441a9,
    0x0c850089, 0x0c854129, 0x0c85c2a9, 0x0c870089,
    0x0c87408f, 0x0c87808d, 0x0c881241, 0x0c910203,
    0x0c940099, 0x0c9444a3, 0x0c968323, 0x0c98072d,
    0x0c9b84af, 0x0c9dc2a1, 0x0c9f00b5, 0x0c9f40b3,
    0x0c9f8085, 0x0ca01883, 0x0cac4223, 0x0cad4523,
    0x0cafc097, 0x0cb004a1, 0x0cb241a5, 0x0cb30097,
    0x0cb34099, 0x0cb38097, 0x0cb3c099, 0x0cb417ad,
    0x0cbfc085, 0x0cc001b3, 0x0cc0c0b1, 0x0cc100b3,
    0x0cc14131, 0x0cc1c0b5, 0x0cc200b3, 0x0cc241b1,
    0x0cc30133, 0x0cc38131, 0x0cc40085, 0x0cc440b1,
    0x0cc48133, 0x0cc50085, 0x0cc540b5, 0x0cc580b7,
    0x0cc5c0b5, 0x0cc600b1, 0x0cc64135, 0x0cc6c0b3,
    0x0cc701b1, 0x0cc7c0b3, 0x0cc800b5, 0x0cc840b3,
    0x0cc881b1, 0x0cc9422f, 0x0cca4131, 0x0ccac0b5,
    0x0ccb00b1, 0x0ccb40b3, 0x0ccb80b5, 0x0ccbc0b1,
    0x0ccc012f, 0x0ccc80b5, 0x0cccc0b3, 0x0ccd00b5,
    0x0ccd40b1, 0x0ccd80b5, 0x0ccdc085, 0x0cce02b1,
    0x0ccf40b3, 0x0ccf80b1, 0x0ccfc085, 0x0cd001b1,
    0x0cd0c0b3, 0x0cd101b1, 0x0cd1c0b5, 0x0cd200b3,
    0x0cd24085, 0x0cd280b5, 0x0cd2c085, 0x0cd30133,
    0x0cd381b1, 0x0cd440b3, 0x0cd48085, 0x0cd4c0b1,
    0x0cd500b3, 0x0cd54085, 0x0cd580b5, 0x0cd5c0b1,
    0x0cd60521, 0x0cd88525, 0x0cdb02a5, 0x0cdc4099,
    0x0cdc8117, 0x0cdd0099, 0x0cdd4197, 0x0cde0127,
    0x0cde8285, 0x0cdfc089, 0x0ce0043f, 0x0ce20099,
    0x0ce2409b, 0x0ce283bf, 0x0ce44219, 0x0ce54205,
    0x0ce6433f, 0x0ce7c131, 0x0ce84085, 0x0ce881b1,
    0x0ce94085, 0x0ce98107, 0x0cea0089, 0x0cea4097,
    0x0cea8219, 0x0ceb809d, 0x0cebc08d, 0x0cec083f,
    0x0cf00105, 0x0cf0809b, 0x0cf0c197, 0x0cf1809b,
    0x0cf1c099, 0x0cf20517, 0x0cf48099, 0x0cf4c117,
    0x0cf54119, 0x0cf5c097, 0x0cf6009b, 0x0cf64099,
    0x0cf68217, 0x0cf78119, 0x0cf804a1, 0x0cfa4525,
    0x0cfcc525, 0x0cff4125, 0x0cffc099, 0x29a70103,
    0x29dc0081, 0x29fc8195, 0x29fe0103, 0x2ad70203,
    0x2ada4081, 0x3e401482, 0x3e4a7f82, 0x3e6a3f82,
    0x3e8aa102, 0x3e9b0110, 0x3e9c2f82, 0x3eb3c590,
    0x3ec00197, 0x3ec0c119, 0x3ec1413f, 0x3ec4c2af,
    0x3ec74184, 0x3ec804ad, 0x3eca4081, 0x3eca8304,
    0x3ecc03a0, 0x3ece02a0, 0x3ecf8084, 0x3ed00120,
    0x3ed0c120, 0x3ed184ae, 0x3ed3c085, 0x3ed4312d,
    0x3ef4cbad, 0x3efa892f, 0x3eff022d, 0x3f002f2f,
    0x3f1782a5, 0x3f18c0b1, 0x3f1907af, 0x3f1cffaf,
    0x3f3c81a5, 0x3f3d64af, 0x3f542031, 0x3f649b31,
    0x3f7c0131, 0x3f7c83b3, 0x3f7e40b1, 0x3f7e80bd,
    0x3f7ec0bb, 0x3f7f00b3, 0x3f840503, 0x3f8c01ad,
    0x3f8cc315, 0x3f8e462d, 0x3f91cc03, 0x3f97c695,
    0x3f9c01af, 0x3f9d0085, 0x3f9d852f, 0x3fa03aad,
    0x3fbd442f, 0x3fc06f1f, 0x3fd7c11f, 0x3fd85fad,
    0x3fe80081, 0x3fe84f1f, 0x3ff0831f, 0x3ff2831f,
    0x3ff4831f, 0x3ff6819f, 0x3ff80783, 0x41724092,
    0x41790092, 0x41e04d83, 0x41e70f91, 0x44268192,
    0x442ac092, 0x444b8112, 0x44d2c112, 0x44e0c192,
    0x44e38092, 0x44e44092, 0x44f14212, 0x452ec212,
    0x456e8112, 0x464e0092, 0x58484412, 0x5b5a0192,
    0x73358d1f, 0x733c051f, 0x74578392, 0x746ec312,
    0x75000d1f, 0x75068d1f, 0x750d0d1f, 0x7513839f,
    0x7515891f, 0x751a0d1f, 0x75208d1f, 0x75271015,
    0x752f439f, 0x7531459f, 0x75340d1f, 0x753a8d1f,
    0x75410395, 0x7543441f, 0x7545839f, 0x75478d1f,
    0x754e0795, 0x7552839f, 0x75548d1f, 0x755b0d1f,
    0x75618d1f, 0x75680d1f, 0x756e8d1f, 0x75750d1f,
    0x757b8d1f, 0x75820d1f, 0x75888d1f, 0x758f0d1f,
    0x75958d1f, 0x759c0d1f, 0x75a28d1f, 0x75a90103,
    0x75aa089f, 0x75ae4081, 0x75ae839f, 0x75b04081,
    0x75b08c9f, 0x75b6c081, 0x75b7032d, 0x75b8889f,
    0x75bcc081, 0x75bd039f, 0x75bec081, 0x75bf0c9f,
    0x75c54081, 0x75c5832d, 0x75c7089f, 0x75cb4081,
    0x75cb839f, 0x75cd4081, 0x75cd8c9f, 0x75d3c081,
    0x75d4032d, 0x75d5889f, 0x75d9c081, 0x75da039f,
    0x75dbc081, 0x75dc0c9f, 0x75e24081, 0x75e2832d,
    0x75e4089f, 0x75e84081, 0x75e8839f, 0x75ea4081,
    0x75ea8c9f, 0x75f0c081, 0x75f1042d, 0x75f3851f,
    0x75f6051f, 0x75f8851f, 0x75fb051f, 0x75fd851f,
    0x780c049f, 0x780e419f, 0x780f059f, 0x7811c203,
    0x7812d0ad, 0x781b0103, 0x7b80022d, 0x7b814dad,
    0x7b884203, 0x7b89c081, 0x7b8a452d, 0x7b8d0403,
    0x7b908081, 0x7b91dc03, 0x7ba0052d, 0x7ba2c8ad,
    0x7ba84483, 0x7baac8ad, 0x7c400097, 0x7c404521,
    0x7c440d25, 0x7c4a8087, 0x7c4ac115, 0x7c4b4117,
    0x7c4c0d1f, 0x7c528217, 0x7c538099, 0x7c53c097,
    0x7c5a8197, 0x7c640097, 0x7c80012f, 0x7c808081,
    0x7c841603, 0x7c9004c1, 0x7c940103, 0x7efc051f,
    0xbe0001ac, 0xbe00d110, 0xbe0947ac, 0xbe0d3910,
    0xbe29872c, 0xbe2d022c, 0xbe2e3790, 0xbe49ff90,
    0xbe69bc10,
};

static const uint16_t unicode_decomp_table2[709] = {
    0x0020, 0x0000, 0x0061, 0x0002, 0x0004, 0x0006, 0x03bc, 0x0008,
    0x000a, 0x000c, 0x0015, 0x0095, 0x00a5, 0x00b9, 0x00c1, 0x00c3,
    0x00c7, 0x00cb, 0x00d1, 0x00d7, 0x00dd, 0x00e0, 0x00e6, 0x00f8,
    0x0108, 0x010a, 0x0073, 0x0110, 0x0112, 0x0114, 0x0120, 0x012c,
    0x0144, 0x014d, 0x0153, 0x0162, 0x0168, 0x016a, 0x0176, 0x0192,
    0x0194, 0x01a9, 0x01bb, 0x01c7, 0x01d1, 0x01d5, 0x02b9, 0x01d7,
    0x003b, 0x01d9, 0x01db, 0x00b7, 0x01e1, 0x01fc, 0x020c, 0x0218,
    0x021d, 0x0223, 0x0227, 0x03a3, 0x0233, 0x023f, 0x0242, 0x024b,
    0x024e, 0x0251, 0x025d, 0x0260, 0x0269, 0x026c, 0x026f, 0x0275,
    0x0278, 0x0281, 0x028a, 0x029c, 0x029f, 0x02a3, 0x02af, 0x02b9,
    0x02c5, 0x02c9, 0x02cd, 0x02d1, 0x02d5, 0x02e7, 0x02ed, 0x02f1,
    0x02f5, 0x02f9, 0x02fd, 0x0305, 0x0309, 0x030d, 0x0313, 0x0317,
    0x031b, 0x0323, 0x0327, 0x032b, 0x032f, 0x0335, 0x033d, 0x0341,
    0x0349, 0x034d, 0x0351, 0x0f0b, 0x0357, 0x035b, 0x035f, 0x0363,
    0x0367, 0x036b, 0x036f, 0x0373, 0x0379, 0x037d, 0x0381, 0x0385,
    0x0389, 0x038d, 0x0391, 0x0395, 0x0399, 0x039d, 0x03a1, 0x10dc,
    0x03a5, 0x03c9, 0x03cd, 0x03d9, 0x03dd, 0x03e1, 0x03ef, 0x03f1,
    0x043d, 0x044f, 0x0499, 0x04f0, 0x0502, 0x054a, 0x0564, 0x056c,
    0x0570, 0x0573, 0x059a, 0x05fa, 0x05fe, 0x0607, 0x060b, 0x0614,
    0x0618, 0x061e, 0x0622, 0x0628, 0x068e, 0x0694, 0x0698, 0x069e,
    0x06a2, 0x06ab, 0x03ac, 0x06f3, 0x03ad, 0x06f6, 0x03ae, 0x06f9,
    0x03af, 0x06fc, 0x03cc, 0x06ff, 0x03cd, 0x0702, 0x03ce, 0x0705,
    0x0709, 0x070d, 0x0711, 0x0386, 0x0732, 0x0735, 0x03b9, 0x0737,
    0x073b, 0x0388, 0x0753, 0x0389, 0x0756, 0x0390, 0x076b, 0x038a,
    0x0777, 0x03b0, 0x0789, 0x038e, 0x0799, 0x079f, 0x07a3, 0x038c,
    0x07b8, 0x038f, 0x07bb, 0x00b4, 0x07be, 0x07c0, 0x07c2, 0x2010,
    0x07cb, 0x002e, 0x07cd, 0x07cf, 0x0020, 0x07d2, 0x07d6, 0x07db,
    0x07df, 0x07e4, 0x07ea, 0x07f0, 0x0020, 0x07f6, 0x2212, 0x0801,
    0x0805, 0x0807, 0x081d, 0x0825, 0x0827, 0x0043, 0x082d, 0x0830,
    0x0190, 0x0836, 0x0839, 0x004e, 0x0845, 0x0847, 0x084c, 0x084e,
    0x0851, 0x005a, 0x03a9, 0x005a, 0x0853, 0x0857, 0x0860, 0x0069,
    0x0862, 0x0865, 0x086f, 0x0874, 0x087a, 0x087e, 0x08a2, 0x0049,
    0x08a4, 0x08a6, 0x08a9, 0x0056, 0x08ab, 0x08ad, 0x08b0, 0x08b4,
    0x0058, 0x08b6, 0x08b8, 0x08bb, 0x08c0, 0x08c2, 0x08c5, 0x0076,
    0x08c7, 0x08c9, 0x08cc, 0x08d0, 0x0078, 0x08d2, 0x08d4, 0x08d7,
    0x08db, 0x08de, 0x08e4, 0x08e7, 0x08f0, 0x08f3, 0x08f6, 0x08f9,
    0x0902, 0x0906, 0x090b, 0x090f, 0x0914, 0x0917, 0x091a, 0x0923,
    0x092c, 0x093b, 0x093e, 0x0941, 0x0944, 0x0947, 0x094a, 0x0956,
    0x095c, 0x0960, 0x0962, 0x0964, 0x0968, 0x096a, 0x0970, 0x0978,
    0x097c, 0x0980, 0x0986, 0x0989, 0x098f, 0x0991, 0x0030, 0x0993,
    0x0999, 0x099c, 0x099e, 0x09a1, 0x09a4, 0x2d61, 0x6bcd, 0x9f9f,
    0x09a6, 0x09b1, 0x09bc, 0x09c7, 0x0a95, 0x0aa1, 0x0b15, 0x0020,
    0x0b27, 0x0b31, 0x0b8d, 0x0ba1, 0x0ba5, 0x0ba9, 0x0bad, 0x0bb1,
    0x0bb5, 0x0bb9, 0x0bbd, 0x0bc1, 0x0bc5, 0x0c21, 0x0c35, 0x0c39,
    0x0c3d, 0x0c41, 0x0c45, 0x0c49, 0x0c4d, 0x0c51, 0x0c55, 0x0c59,
    0x0c6f, 0x0c71, 0x0c73, 0x0ca0, 0x0cbc, 0x0cdc, 0x0ce4, 0x0cec,
    0x0cf4, 0x0cfc, 0x0d04, 0x0d0c, 0x0d14, 0x0d22, 0x0d2e, 0x0d7a,
    0x0d82, 0x0d85, 0x0d89, 0x0d8d, 0x0d9d, 0x0db1, 0x0db5, 0x0dbc,
    0x0dc2, 0x0dc6, 0x0e28, 0x0e2c, 0x0e30, 0x0e32, 0x0e36, 0x0e3c,
    0x0e3e, 0x0e41, 0x0e43, 0x0e46, 0x0e77, 0x0e7b, 0x0e89, 0x0e8e,
    0x0e94, 0x0e9c, 0x0ea3, 0x0ea9, 0x0eb4, 0x0ebe, 0x0ec6, 0x0eca,
    0x0ecf, 0x0ed9, 0x0edd, 0x0ee4, 0x0eec, 0x0ef3, 0x0ef8, 0x0f04,
    0x0f0a, 0x0f15, 0x0f1b, 0x0f22, 0x0f28, 0x0f33, 0x0f3d, 0x0f45,
    0x0f4c, 0x0f51, 0x0f57, 0x0f5e, 0x0f63, 0x0f69, 0x0f70, 0x0f76,
    0x0f7d, 0x0f82, 0x0f89, 0x0f8d, 0x0f9e, 0x0fa4, 0x0fa9, 0x0fad,
    0x0fb8, 0x0fbe, 0x0fc9, 0x0fd0, 0x0fd6, 0x0fda, 0x0fe1, 0x0fe5,
    0x0fef, 0x0ffa, 0x1000, 0x1004, 0x1009, 0x100f, 0x1013, 0x101a,
    0x101f, 0x1023, 0x1029, 0x102f, 0x1032, 0x1036, 0x1039, 0x103f,
    0x1045, 0x1059, 0x1061, 0x1079, 0x107c, 0x1080, 0x1095, 0x10a1,
    0x10b1, 0x10c3, 0x10cb, 0x10cf, 0x10da, 0x10de, 0x10ea, 0x10f2,
    0x10f4, 0x1100, 0x1105, 0x1111, 0x1141, 0x1149, 0x114d, 0x1153,
    0x1157, 0x115a, 0x116e, 0x1171, 0x1175, 0x117b, 0x117d, 0x1181,
    0x1184, 0x118c, 0x1192, 0x1196, 0x119c, 0x11a2, 0x11a8, 0x11ab,
    0xa76f, 0x11af, 0x11b2, 0x11b6, 0x028d, 0x11be, 0x1210, 0x130e,
    0x140c, 0x1490, 0x1495, 0x1553, 0x156c, 0x1572, 0x1578, 0x157e,
    0x158a, 0x1596, 0x002b, 0x15a1, 0x15b9, 0x15bd, 0x15c1, 0x15c5,
    0x15c9, 0x15cd, 0x15e1, 0x15e5, 0x1649, 0x1662, 0x1688, 0x168e,
    0x174c, 0x1752, 0x1757, 0x1777, 0x1877, 0x187d, 0x1911, 0x19d3,
    0x1a77, 0x1a7f, 0x1a9d, 0x1aa2, 0x1ab6, 0x1ac0, 0x1ac6, 0x1ada,
    0x1adf, 0x1ae5, 0x1af3, 0x1b23, 0x1b30, 0x1b38, 0x1b3c, 0x1b52,
    0x1bc9, 0x1bdb, 0x1bdd, 0x1bdf, 0x3164, 0x1c20, 0x1c22, 0x1c24,
    0x1c26, 0x1c28, 0x1c2a, 0x1c48, 0x1c4d, 0x1c52, 0x1c88, 0x1cce,
    0x1cdc, 0x1ce1, 0x1cea, 0x1cf3, 0x1d01, 0x1d06, 0x1d0b, 0x1d1d,
    0x1d2f, 0x1d38, 0x1d3d, 0x1d61, 0x1d6f, 0x1d71, 0x1d73, 0x1d93,
    0x1dae, 0x1db0, 0x1db2, 0x1db4, 0x1db6, 0x1db8, 0x1dba, 0x1dbc,
    0x1ddc, 0x1dde, 0x1de0, 0x1de2, 0x1de4, 0x1deb, 0x1ded, 0x1def,
    0x1df1, 0x1e00, 0x1e02, 0x1e04, 0x1e06, 0x1e08, 0x1e0a, 0x1e0c,
    0x1e0e, 0x1e10, 0x1e12, 0x1e14, 0x1e16, 0x1e18, 0x1e1a, 0x1e1c,
    0x1e20, 0x03f4, 0x1e22, 0x2207, 0x1e24, 0x2202, 0x1e26, 0x1e2e,
    0x03f4, 0x1e30, 0x2207, 0x1e32, 0x2202, 0x1e34, 0x1e3c, 0x03f4,
    0x1e3e, 0x2207, 0x1e40, 0x2202, 0x1e42, 0x1e4a, 0x03f4, 0x1e4c,
    0x2207, 0x1e4e, 0x2202, 0x1e50, 0x1e58, 0x03f4, 0x1e5a, 0x2207,
    0x1e5c, 0x2202, 0x1e5e, 0x1e68, 0x1e6a, 0x1e6c, 0x1e6e, 0x1e70,
    0x1e72, 0x1e74, 0x1e76, 0x1e78, 0x1e80, 0x1ea3, 0x1ea7, 0x1ead,
    0x1eca, 0x062d, 0x1ed2, 0x1ede, 0x062c, 0x1eee, 0x1f5e, 0x1f6a,
    0x1f7d, 0x1f8f, 0x1fa2, 0x1fa4, 0x1fa8, 0x1fae, 0x1fb4, 0x1fb6,
    0x1fba, 0x1fbc, 0x1fc4, 0x1fc7, 0x1fc9, 0x1fcf, 0x1fd1, 0x30b5,
    0x1fd7, 0x202f, 0x2045, 0x2049, 0x204b, 0x2050, 0x209d, 0x20ae,
    0x21af, 0x21bf, 0x21c5, 0x22bf, 0x23dd,
};

static const uint8_t unicode_decomp_data[9451] = {
    0x20, 0x88, 0x20, 0x84, 0x32, 0x33, 0x20, 0x81,
    0x20, 0xa7, 0x31, 0x6f, 0x31, 0xd0, 0x34, 0x31,
    0xd0, 0x32, 0x33, 0xd0, 0x34, 0x41, 0x80, 0x41,
    0x81, 0x41, 0x82, 0x41, 0x83, 0x41, 0x88, 0x41,
    0x8a, 0x00, 0x00, 0x43, 0xa7, 0x45, 0x80, 0x45,
    0x81, 0x45, 0x82, 0x45, 0x88, 0x49, 0x80, 0x49,
    0x81, 0x49, 0x82, 0x49, 0x88, 0x00, 0x00, 0x4e,
    0x83, 0x4f, 0x80, 0x4f, 0x81, 0x4f, 0x82, 0x4f,
    0x83, 0x4f, 0x88, 0x00, 0x00, 0x00, 0x00, 0x55,
    0x80, 0x55, 0x81, 0x55, 0x82, 0x55, 0x88, 0x59,
    0x81, 0x00, 0x00, 0x00, 0x00, 0x61, 0x80, 0x61,
    0x81, 0x61, 0x82, 0x61, 0x83, 0x61, 0x88, 0x61,
    0x8a, 0x00, 0x00, 0x63, 0xa7, 0x65, 0x80, 0x65,
    0x81, 0x65, 0x82, 0x65, 0x88, 0x69, 0x80, 0x69,
    0x81, 0x69, 0x82, 0x69, 0x88, 0x00, 0x00, 0x6e,
    0x83, 0x6f, 0x80, 0x6f, 0x81, 0x6f, 0x82, 0x6f,
    0x83, 0x6f, 0x88, 0x00, 0x00, 0x00, 0x00, 0x75,
    0x80, 0x75, 0x81, 0x75, 0x82, 0x75, 0x88, 0x79,
    0x81, 0x00, 0x00, 0x79, 0x88, 0x41, 0x84, 0x41,
    0x86, 0x41, 0xa8, 0x43, 0x81, 0x43, 0x82, 0x43,
    0x87, 0x43, 0x8c, 0x44, 0x8c, 0x45, 0x84, 0x45,
    0x86, 0x45, 0x87, 0x45, 0xa8, 0x45, 0x8c, 0x47,
    0x82, 0x47, 0x86, 0x47, 0x87, 0x47, 0xa7, 0x48,
    0x82, 0x49, 0x83, 0x49, 0x84, 0x49, 0x86, 0x49,
    0xa8, 0x49, 0x87, 0x49, 0x4a, 0x69, 0x6a, 0x4a,
    0x82, 0x4b, 0xa7, 0x4c, 0x81, 0x4c, 0xa7, 0x4c,
    0x8c, 0x4c, 0x00, 0x00, 0x6b, 0x20, 0x6b, 0x4e,
    0x81, 0x4e, 0xa7, 0x4e, 0x8c, 0xbc, 0x02, 0x6e,
    0x4f, 0x84, 0x4f, 0x86, 0x4f, 0x8b, 0x52, 0x81,
    0x52, 0xa7, 0x52, 0x8c, 0x53, 0x81, 0x53, 0x82,
    0x53, 0xa7, 0x53, 0x8c, 0x54, 0xa7, 0x54, 0x8c,
    0x55, 0x83, 0x55, 0x84, 0x55, 0x86, 0x55, 0x8a,
    0x55, 0x8b, 0x55, 0xa8, 0x57, 0x82, 0x59, 0x82,
    0x59, 0x88, 0x5a, 0x81, 0x5a, 0x87, 0x5a, 0x8c,
    0x4f, 0x9b, 0x55, 0x9b, 0x44, 0x00, 0x7d, 0x01,
    0x44, 0x00, 0x7e, 0x01, 0x64, 0x00, 0x7e, 0x01,
    0x4c, 0x4a, 0x4c, 0x6a, 0x6c, 0x6a, 0x4e, 0x4a,
    0x4e, 0x6a, 0x6e, 0x6a, 0x41, 0x00, 0x8c, 0x49,
    0x00, 0x8c, 0x4f, 0x00, 0x8c, 0x55, 0x00, 0x8c,
    0xdc, 0x00, 0x84, 0xdc, 0x00, 0x81, 0xdc, 0x00,
    0x8c, 0xdc, 0x00, 0x80, 0xc4, 0x00, 0x84, 0x26,
    0x02, 0x84, 0xc6, 0x00, 0x84, 0x47, 0x8c, 0x4b,
    0x8c, 0x4f, 0xa8, 0xea, 0x01, 0x84, 0xeb, 0x01,
    0x84, 0xb7, 0x01, 0x8c, 0x92, 0x02, 0x8c, 0x6a,
    0x00, 0x8c, 0x44, 0x5a, 0x44, 0x7a, 0x64, 0x7a,
    0x47, 0x81, 0x4e, 0x00, 0x80, 0xc5, 0x00, 0x81,
    0xc6, 0x00, 0x81, 0xd8, 0x00, 0x81, 0x41, 0x8f,
    0x41, 0x91, 0x45, 0x8f, 0x45, 0x91, 0x49, 0x8f,
    0x49, 0x91, 0x4f, 0x8f, 0x4f, 0x91, 0x52, 0x8f,
    0x52, 0x91, 0x55, 0x8f, 0x55, 0x91, 0x53, 0xa6,
    0x54, 0xa6, 0x48, 0x8c, 0x41, 0x00, 0x87, 0x45,
    0x00, 0xa7, 0xd6, 0x00, 0x84, 0xd5, 0x00, 0x84,
    0x4f, 0x00, 0x87, 0x2e, 0x02, 0x84, 0x59, 0x00,
    0x84, 0x68, 0x00, 0x66, 0x02, 0x6a, 0x00, 0x72,
    0x00, 0x79, 0x02, 0x7b, 0x02, 0x81, 0x02, 0x77,
    0x00, 0x79, 0x00, 0x20, 0x86, 0x20, 0x87, 0x20,
    0x8a, 0x20, 0xa8, 0x20, 0x83, 0x20, 0x8b, 0x63,
    0x02, 0x6c, 0x00, 0x73, 0x00, 0x78, 0x00, 0x95,
    0x02, 0x80, 0x81, 0x00, 0x93, 0x88, 0x81, 0x20,
    0xc5, 0x20, 0x81, 0xa8, 0x00, 0x81, 0x91, 0x03,
    0x81, 0x95, 0x03, 0x81, 0x97, 0x03, 0x81, 0x99,
    0x03, 0x81, 0x00, 0x00, 0x00, 0x9f, 0x03, 0x81,
    0x00, 0x00, 0x00, 0xa5, 0x03, 0x81, 0xa9, 0x03,
    0x81, 0xca, 0x03, 0x81, 0x01, 0x03, 0x98, 0x07,
    0xa4, 0x07, 0xb0, 0x00, 0xb4, 0x00, 0xb6, 0x00,
    0xb8, 0x00, 0xca, 0x00, 0x01, 0x03, 0xb8, 0x07,
    0xc4, 0x07, 0xbe, 0x00, 0xc4, 0x00, 0xc8, 0x00,
    0xa5, 0x03, 0x0d, 0x13, 0x00, 0x01, 0x03, 0xd1,
    0x00, 0xd1, 0x07, 0xc6, 0x03, 0xc0, 0x03, 0xba,
    0x03, 0xc1, 0x03, 0xc2, 0x03, 0x00, 0x00, 0x98,
    0x03, 0xb5, 0x03, 0x15, 0x04, 0x80, 0x15, 0x04,
    0x88, 0x00, 0x00, 0x00, 0x13, 0x04, 0x81, 0x06,
    0x04, 0x88, 0x1a, 0x04, 0x81, 0x18, 0x04, 0x80,
    0x23, 0x04, 0x86, 0x18, 0x04, 0x86, 0x38, 0x04,
    0x86, 0x35, 0x04, 0x80, 0x35, 0x04, 0x88, 0x00,
    0x00, 0x00, 0x33, 0x04, 0x81, 0x56, 0x04, 0x88,
    0x3a, 0x04, 0x81, 0x38, 0x04, 0x80, 0x43, 0x04,
    0x86, 0x74, 0x04, 0x8f, 0x16, 0x04, 0x86, 0x10,
    0x04, 0x86, 0x10, 0x04, 0x88, 0x15, 0x04, 0x86,
    0xd8, 0x04, 0x88, 0x16, 0x04, 0x88, 0x17, 0x04,
    0x88, 0x18, 0x04, 0x84, 0x18, 0x04, 0x88, 0x1e,
    0x04, 0x88, 0xe8, 0x04, 0x88, 0x2d, 0x04, 0x88,
    0x23, 0x04, 0x84, 0x23, 0x04, 0x88, 0x23, 0x04,
    0x8b, 0x27, 0x04, 0x88, 0x2b, 0x04, 0x88, 0x65,
    0x05, 0x82, 0x05, 0x27, 0x06, 0x00, 0x2c, 0x00,
    0x2d, 0x21, 0x2d, 0x00, 0x2e, 0x23, 0x2d, 0x27,
    0x06, 0x00, 0x4d, 0x21, 0x4d, 0xa0, 0x4d, 0x23,
    0x4d, 0xd5, 0x06, 0x54, 0x06, 0x00, 0x00, 0x00,
    0x00, 0xc1, 0x06, 0x54, 0x06, 0xd2, 0x06, 0x54,
    0x06, 0x28, 0x09, 0x3c, 0x09, 0x30, 0x09, 0x3c,
    0x09, 0x33, 0x09, 0x3c, 0x09, 0x15, 0x09, 0x00,
    0x27, 0x01, 0x27, 0x02, 0x27, 0x07, 0x27, 0x0c,
    0x27, 0x0d, 0x27, 0x16, 0x27, 0x1a, 0x27, 0xbe,
    0x09, 0x09, 0x00, 0x09, 0x19, 0xa1, 0x09, 0xbc,
    0x09, 0xaf, 0x09, 0xbc, 0x09, 0x32, 0x0a, 0x3c,
    0x0a, 0x38, 0x0a, 0x3c, 0x0a, 0x16, 0x0a, 0x00,
    0x26, 0x01, 0x26, 0x06, 0x26, 0x2b, 0x0a, 0x3c,
    0x0a, 0x47, 0x0b, 0x56, 0x0b, 0x3e, 0x0b, 0x09,
    0x00, 0x09, 0x19, 0x21, 0x0b, 0x3c, 0x0b, 0x92,
    0x0b, 0xd7, 0x0b, 0xbe, 0x0b, 0x08, 0x00, 0x09,
    0x00, 0x08, 0x19, 0x46, 0x0c, 0x56, 0x0c, 0xbf,
    0x0c, 0xd5, 0x0c, 0xc6, 0x0c, 0xd5, 0x0c, 0xc2,
    0x0c, 0x04, 0x00, 0x08, 0x13, 0x3e, 0x0d, 0x08,
    0x00, 0x09, 0x00, 0x08, 0x19, 0xd9, 0x0d, 0xca,
    0x0d, 0xca, 0x0d, 0x0f, 0x05, 0x12, 0x00, 0x0f,
    0x15, 0x4d, 0x0e, 0x32, 0x0e, 0xcd, 0x0e, 0xb2,
    0x0e, 0x99, 0x0e, 0x12, 0x00, 0x12, 0x08, 0x42,
    0x0f, 0xb7, 0x0f, 0x4c, 0x0f, 0xb7, 0x0f, 0x51,
    0x0f, 0xb7, 0x0f, 0x56, 0x0f, 0xb7, 0x0f, 0x5b,
    0x0f, 0xb7, 0x0f, 0x40, 0x0f, 0xb5, 0x0f, 0x71,
    0x0f, 0x72, 0x0f, 0x71, 0x0f, 0x00, 0x03, 0x41,
    0x0f, 0xb2, 0x0f, 0x81, 0x0f, 0xb3, 0x0f, 0x80,
    0x0f, 0xb3, 0x0f, 0x81, 0x0f, 0x71, 0x0f, 0x80,
    0x0f, 0x92, 0x0f, 0xb7, 0x0f, 0x9c, 0x0f, 0xb7,
    0x0f, 0xa1, 0x0f, 0xb7, 0x0f, 0xa6, 0x0f, 0xb7,
    0x0f, 0xab, 0x0f, 0xb7, 0x0f, 0x90, 0x0f, 0xb5,
    0x0f, 0x25, 0x10, 0x2e, 0x10, 0x05, 0x1b, 0x35,
    0x1b, 0x00, 0x00, 0x00, 0x00, 0x07, 0x1b, 0x35,
    0x1b, 0x00, 0x00, 0x00, 0x00, 0x09, 0x1b, 0x35,
    0x1b, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x1b, 0x35,
    0x1b, 0x00, 0x00, 0x00, 0x00, 0x0d, 0x1b, 0x35,
    0x1b, 0x11, 0x1b, 0x35, 0x1b, 0x3a, 0x1b, 0x35,
    0x1b, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x1b, 0x35,
    0x1b, 0x3e, 0x1b, 0x35, 0x1b, 0x42, 0x1b, 0x35,
    0x1b, 0x41, 0x00, 0xc6, 0x00, 0x42, 0x00, 0x00,
    0x00, 0x44, 0x00, 0x45, 0x00, 0x8e, 0x01, 0x47,
    0x00, 0x4f, 0x00, 0x22, 0x02, 0x50, 0x00, 0x52,
    0x00, 0x54, 0x00, 0x55, 0x00, 0x57, 0x00, 0x61,
    0x00, 0x50, 0x02, 0x51, 0x02, 0x02, 0x1d, 0x62,
    0x00, 0x64, 0x00, 0x65, 0x00, 0x59, 0x02, 0x5b,
    0x02, 0x5c, 0x02, 0x67, 0x00, 0x00, 0x00, 0x6b,
    0x00, 0x6d, 0x00, 0x4b, 0x01, 0x6f, 0x00, 0x54,
    0x02, 0x16, 0x1d, 0x17, 0x1d, 0x70, 0x00, 0x74,
    0x00, 0x75, 0x00, 0x1d, 0x1d, 0x6f, 0x02, 0x76,
    0x00, 0x25, 0x1d, 0xb2, 0x03, 0xb3, 0x03, 0xb4,
    0x03, 0xc6, 0x03, 0xc7, 0x03, 0x69, 0x00, 0x72,
    0x00, 0x75, 0x00, 0x76, 0x00, 0xb2, 0x03, 0xb3,
    0x03, 0xc1, 0x03, 0xc6, 0x03, 0xc7, 0x03, 0x52,
    0x02, 0x63, 0x00, 0x55, 0x02, 0xf0, 0x00, 0x5c,
    0x02, 0x66, 0x00, 0x5f, 0x02, 0x61, 0x02, 0x65,
    0x02, 0x68, 0x02, 0x69, 0x02, 0x6a, 0x02, 0x7b,
    0x1d, 0x9d, 0x02, 0x6d, 0x02, 0x85, 0x1d, 0x9f,
    0x02, 0x71, 0x02, 0x70, 0x02, 0x72, 0x02, 0x73,
    0x02, 0x74, 0x02, 0x75, 0x02, 0x78, 0x02, 0x82,
    0x02, 0x83, 0x02, 0xab, 0x01, 0x89, 0x02, 0x8a,
    0x02, 0x1c, 0x1d, 0x8b, 0x02, 0x8c, 0x02, 0x7a,
    0x00, 0x90, 0x02, 0x91, 0x02, 0x92, 0x02, 0xb8,
    0x03, 0x41, 0x00, 0xa5, 0x42, 0x00, 0x87, 0x42,
    0x00, 0xa3, 0x42, 0x00, 0xb1, 0xc7, 0x00, 0x81,
    0x44, 0x00, 0x87, 0x44, 0x00, 0xa3, 0x44, 0x00,
    0xb1, 0x44, 0x00, 0xa7, 0x44, 0x00, 0xad, 0x12,
    0x01, 0x80, 0x12, 0x01, 0x81, 0x45, 0x00, 0xad,
    0x45, 0x00, 0xb0, 0x28, 0x02, 0x86, 0x46, 0x00,
    0x87, 0x47, 0x00, 0x84, 0x48, 0x00, 0x87, 0x48,
    0x00, 0xa3, 0x48, 0x00, 0x88, 0x48, 0x00, 0xa7,
    0x48, 0x00, 0xae, 0x49, 0x00, 0xb0, 0xcf, 0x00,
    0x81, 0x4b, 0x00, 0x81, 0x4b, 0x00, 0xa3, 0x4b,
    0x00, 0xb1, 0x4c, 0x00, 0xa3, 0x36, 0x1e, 0x84,
    0x4c, 0xb1, 0x4c, 0xad, 0x4d, 0x81, 0x4d, 0x87,
    0x4d, 0xa3, 0x4e, 0x87, 0x4e, 0xa3, 0x4e, 0xb1,
    0x4e, 0xad, 0xd5, 0x00, 0x81, 0xd5, 0x00, 0x88,
    0x4c, 0x01, 0x80, 0x4c, 0x01, 0x81, 0x50, 0x00,
    0x81, 0x50, 0x00, 0x87, 0x52, 0x00, 0x87, 0x52,
    0x00, 0xa3, 0x5a, 0x1e, 0x84, 0x52, 0x00, 0xb1,
    0x53, 0x00, 0x87, 0x53, 0x00, 0xa3, 0x5a, 0x01,
    0x87, 0x60, 0x01, 0x87, 0x62, 0x1e, 0x87, 0x54,
    0x00, 0x87, 0x54, 0x00, 0xa3, 0x54, 0x00, 0xb1,
    0x54, 0x00, 0xad, 0x55, 0x00, 0xa4, 0x55, 0x00,
    0xb0, 0x55, 0x00, 0xad, 0x68, 0x01, 0x81, 0x6a,
    0x01, 0x88, 0x56, 0x83, 0x56, 0xa3, 0x57, 0x80,
    0x57, 0x81, 0x57, 0x88, 0x57, 0x87, 0x57, 0xa3,
    0x58, 0x87, 0x58, 0x88, 0x59, 0x87, 0x5a, 0x82,
    0x5a, 0xa3, 0x5a, 0xb1, 0x68, 0xb1, 0x74, 0x88,
    0x77, 0x8a, 0x79, 0x8a, 0x61, 0x00, 0xbe, 0x02,
    0x7f, 0x01, 0x87, 0x41, 0x00, 0xa3, 0x41, 0x00,
    0x89, 0xc2, 0x00, 0x81, 0xc2, 0x00, 0x80, 0xc2,
    0x00, 0x89, 0xc2, 0x00, 0x83, 0xa0, 0x1e, 0x82,
    0x02, 0x01, 0x81, 0x02, 0x01, 0x80, 0x02, 0x01,
    0x89, 0x02, 0x01, 0x83, 0xa0, 0x1e, 0x86, 0x45,
    0x00, 0xa3, 0x45, 0x00, 0x89, 0x45, 0x00, 0x83,
    0xca, 0x00, 0x81, 0xca, 0x00, 0x80, 0xca, 0x00,
    0x89, 0xca, 0x00, 0x83, 0xb8, 0x1e, 0x82, 0x49,
    0x00, 0x89, 0x49, 0x00, 0xa3, 0x4f, 0x00, 0xa3,
    0x4f, 0x00, 0x89, 0xd4, 0x00, 0x81, 0xd4, 0x00,
    0x80, 0xd4, 0x00, 0x89, 0xd4, 0x00, 0x83, 0xcc,
    0x1e, 0x82, 0xa0, 0x01, 0x81, 0xa0, 0x01, 0x80,
    0xa0, 0x01, 0x89, 0xa0, 0x01, 0x83, 0xa0, 0x01,
    0xa3, 0x55, 0x00, 0xa3, 0x55, 0x00, 0x89, 0xaf,
    0x01, 0x81, 0xaf, 0x01, 0x80, 0xaf, 0x01, 0x89,
    0xaf, 0x01, 0x83, 0xaf, 0x01, 0xa3, 0x59, 0x00,
    0x80, 0x59, 0x00, 0xa3, 0x59, 0x00, 0x89, 0x59,
    0x00, 0x83, 0xb1, 0x03, 0x13, 0x03, 0x00, 0x1f,
    0x80, 0x00, 0x1f, 0x81, 0x00, 0x1f, 0xc2, 0x91,
    0x03, 0x13, 0x03, 0x08, 0x1f, 0x80, 0x08, 0x1f,
    0x81, 0x08, 0x1f, 0xc2, 0xb5, 0x03, 0x13, 0x03,
    0x10, 0x1f, 0x80, 0x10, 0x1f, 0x81, 0x95, 0x03,
    0x13, 0x03, 0x18, 0x1f, 0x80, 0x18, 0x1f, 0x81,
    0xb7, 0x03, 0x93, 0xb7, 0x03, 0x94, 0x20, 0x1f,
    0x80, 0x21, 0x1f, 0x80, 0x20, 0x1f, 0x81, 0x21,
    0x1f, 0x81, 0x20, 0x1f, 0xc2, 0x21, 0x1f, 0xc2,
    0x97, 0x03, 0x93, 0x97, 0x03, 0x94, 0x28, 0x1f,
    0x80, 0x29, 0x1f, 0x80, 0x28, 0x1f, 0x81, 0x29,
    0x1f, 0x81, 0x28, 0x1f, 0xc2, 0x29, 0x1f, 0xc2,
    0xb9, 0x03, 0x93, 0xb9, 0x03, 0x94, 0x30, 0x1f,
    0x80, 0x31, 0x1f, 0x80, 0x30, 0x1f, 0x81, 0x31,
    0x1f, 0x81, 0x30, 0x1f, 0xc2, 0x31, 0x1f, 0xc2,
    0x99, 0x03, 0x93, 0x99, 0x03, 0x94, 0x38, 0x1f,
    0x80, 0x39, 0x1f, 0x80, 0x38, 0x1f, 0x81, 0x39,
    0x1f, 0x81, 0x38, 0x1f, 0xc2, 0x39, 0x1f, 0xc2,
    0xbf, 0x03, 0x93, 0xbf, 0x03, 0x94, 0x40, 0x1f,
    0x80, 0x40, 0x1f, 0x81, 0x9f, 0x03, 0x13, 0x03,
    0x48, 0x1f, 0x80, 0x48, 0x1f, 0x81, 0xc5, 0x03,
    0x13, 0x03, 0x50, 0x1f, 0x80, 0x50, 0x1f, 0x81,
    0x50, 0x1f, 0xc2, 0xa5, 0x03, 0x94, 0x00, 0x00,
    0x00, 0x59, 0x1f, 0x80, 0x00, 0x00, 0x00, 0x59,
    0x1f, 0x81, 0x00, 0x00, 0x00, 0x59, 0x1f, 0xc2,
    0xc9, 0x03, 0x93, 0xc9, 0x03, 0x94, 0x60, 0x1f,
    0x80, 0x61, 0x1f, 0x80, 0x60, 0x1f, 0x81, 0x61,
    0x1f, 0x81, 0x60, 0x1f, 0xc2, 0x61, 0x1f, 0xc2,
    0xa9, 0x03, 0x93, 0xa9, 0x03, 0x94, 0x68, 0x1f,
    0x80, 0x69, 0x1f, 0x80, 0x68, 0x1f, 0x81, 0x69,
    0x1f, 0x81, 0x68, 0x1f, 0xc2, 0x69, 0x1f, 0xc2,
    0xb1, 0x03, 0x80, 0xb5, 0x03, 0x80, 0xb7, 0x03,
    0x80, 0xb9, 0x03, 0x80, 0xbf, 0x03, 0x80, 0xc5,
    0x03, 0x80, 0xc9, 0x03, 0x80, 0x00, 0x1f, 0x45,
    0x03, 0x20, 0x1f, 0x45, 0x03, 0x60, 0x1f, 0x45,
    0x03, 0xb1, 0x03, 0x86, 0xb1, 0x03, 0x84, 0x70,
    0x1f, 0xc5, 0xb1, 0x03, 0xc5, 0xac, 0x03, 0xc5,
    0x00, 0x00, 0x00, 0xb1, 0x03, 0xc2, 0xb6, 0x1f,
    0xc5, 0x91, 0x03, 0x86, 0x91, 0x03, 0x84, 0x91,
    0x03, 0x80, 0x91, 0x03, 0xc5, 0x20, 0x93, 0x20,
    0x93, 0x20, 0xc2, 0xa8, 0x00, 0xc2, 0x74, 0x1f,
    0xc5, 0xb7, 0x03, 0xc5, 0xae, 0x03, 0xc5, 0x00,
    0x00, 0x00, 0xb7, 0x03, 0xc2, 0xc6, 0x1f, 0xc5,
    0x95, 0x03, 0x80, 0x97, 0x03, 0x80, 0x97, 0x03,
    0xc5, 0xbf, 0x1f, 0x80, 0xbf, 0x1f, 0x81, 0xbf,
    0x1f, 0xc2, 0xb9, 0x03, 0x86, 0xb9, 0x03, 0x84,
    0xca, 0x03, 0x80, 0x00, 0x03, 0xb9, 0x42, 0xca,
    0x42, 0x99, 0x06, 0x99, 0x04, 0x99, 0x00, 0xfe,
    0x1f, 0x80, 0xfe, 0x1f, 0x81, 0xfe, 0x1f, 0xc2,
    0xc5, 0x03, 0x86, 0xc5, 0x03, 0x84, 0xcb, 0x03,
    0x80, 0x00, 0x03, 0xc1, 0x13, 0xc1, 0x14, 0xc5,
    0x42, 0xcb, 0x42, 0xa5, 0x06, 0xa5, 0x04, 0xa5,
    0x00, 0xa1, 0x03, 0x94, 0xa8, 0x00, 0x80, 0x85,
    0x03, 0x60, 0x00, 0x7c, 0x1f, 0xc5, 0xc9, 0x03,
    0xc5, 0xce, 0x03, 0xc5, 0x00, 0x00, 0x00, 0xc9,
    0x03, 0xc2, 0xf6, 0x1f, 0xc5, 0x9f, 0x03, 0x80,
    0xa9, 0x03, 0x80, 0xa9, 0x03, 0xc5, 0x20, 0x94,
    0x02, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
    0x20, 0x20, 0x20, 0x20, 0xb3, 0x2e, 0x2e, 0x2e,
    0x2e, 0x2e, 0x32, 0x20, 0x32, 0x20, 0x32, 0x20,
    0x00, 0x00, 0x00, 0x35, 0x20, 0x35, 0x20, 0x35,
    0x20, 0x00, 0x00, 0x00, 0x21, 0x21, 0x00, 0x00,
    0x20, 0x85, 0x3f, 0x3f, 0x3f, 0x21, 0x21, 0x3f,
    0x32, 0x20, 0x00, 0x00, 0x00, 0x00, 0x30, 0x69,
    0x00, 0x00, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,
    0x2b, 0x3d, 0x28, 0x29, 0x6e, 0x30, 0x00, 0x2b,
    0x00, 0x12, 0x22, 0x3d, 0x00, 0x28, 0x00, 0x29,
    0x00, 0x00, 0x00, 0x61, 0x00, 0x65, 0x00, 0x6f,
    0x00, 0x78, 0x00, 0x59, 0x02, 0x68, 0x6b, 0x6c,
    0x6d, 0x6e, 0x70, 0x73, 0x74, 0x52, 0x73, 0x61,
    0x2f, 0x63, 0x61, 0x2f, 0x73, 0xb0, 0x00, 0x43,
    0x63, 0x2f, 0x6f, 0x63, 0x2f, 0x75, 0xb0, 0x00,
    0x46, 0x48, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x20,
    0xdf, 0x01, 0x01, 0x04, 0x24, 0x4e, 0x6f, 0x50,
    0x51, 0x52, 0x52, 0x52, 0x53, 0x4d, 0x54, 0x45,
    0x4c, 0x54, 0x4d, 0x4b, 0x00, 0xc5, 0x00, 0x42,
    0x43, 0x00, 0x65, 0x45, 0x46, 0x00, 0x4d, 0x6f,
    0xd0, 0x05, 0x46, 0x41, 0x58, 0xc0, 0x03, 0xb3,
    0x03, 0x93, 0x03, 0xa0, 0x03, 0x11, 0x22, 0x44,
    0x64, 0x65, 0x69, 0x6a, 0x31, 0xd0, 0x37, 0x31,
    0xd0, 0x39, 0x31, 0xd0, 0x31, 0x30, 0x31, 0xd0,
    0x33, 0x32, 0xd0, 0x33, 0x31, 0xd0, 0x35, 0x32,
    0xd0, 0x35, 0x33, 0xd0, 0x35, 0x34, 0xd0, 0x35,
    0x31, 0xd0, 0x36, 0x35, 0xd0, 0x36, 0x31, 0xd0,
    0x38, 0x33, 0xd0, 0x38, 0x35, 0xd0, 0x38, 0x37,
    0xd0, 0x38, 0x31, 0xd0, 0x49, 0x49, 0x49, 0x49,
    0x49, 0x49, 0x56, 0x56, 0x49, 0x56, 0x49, 0x49,
    0x56, 0x49, 0x49, 0x49, 0x49, 0x58, 0x58, 0x49,
    0x58, 0x49, 0x49, 0x4c, 0x43, 0x44, 0x4d, 0x69,
    0x69, 0x69, 0x69, 0x69, 0x69, 0x69, 0x76, 0x76,
    0x69, 0x76, 0x69, 0x69, 0x76, 0x69, 0x69, 0x69,
    0x69, 0x78, 0x78, 0x69, 0x78, 0x69, 0x69, 0x6c,
    0x63, 0x64, 0x6d, 0x30, 0xd0, 0x33, 0x90, 0x21,
    0xb8, 0x92, 0x21, 0xb8, 0x94, 0x21, 0xb8, 0xd0,
    0x21, 0xb8, 0xd4, 0x21, 0xb8, 0xd2, 0x21, 0xb8,
    0x03, 0x22, 0xb8, 0x08, 0x22, 0xb8, 0x0b, 0x22,
    0xb8, 0x23, 0x22, 0xb8, 0x00, 0x00, 0x00, 0x25,
    0x22, 0xb8, 0x2b, 0x22, 0x2b, 0x22, 0x2b, 0x22,
    0x00, 0x00, 0x00, 0x2e, 0x22, 0x2e, 0x22, 0x2e,
    0x22, 0x00, 0x00, 0x00, 0x3c, 0x22, 0xb8, 0x43,
    0x22, 0xb8, 0x45, 0x22, 0xb8, 0x00, 0x00, 0x00,
    0x48, 0x22, 0xb8, 0x3d, 0x00, 0xb8, 0x00, 0x00,
    0x00, 0x61, 0x22, 0xb8, 0x4d, 0x22, 0xb8, 0x3c,
    0x00, 0xb8, 0x3e, 0x00, 0xb8, 0x64, 0x22, 0xb8,
    0x65, 0x22, 0xb8, 0x72, 0x22, 0xb8, 0x76, 0x22,
    0xb8, 0x7a, 0x22, 0xb8, 0x82, 0x22, 0xb8, 0x86,
    0x22, 0xb8, 0xa2, 0x22, 0xb8, 0xa8, 0x22, 0xb8,
    0xa9, 0x22, 0xb8, 0xab, 0x22, 0xb8, 0x7c, 0x22,
    0xb8, 0x91, 0x22, 0xb8, 0xb2, 0x22, 0x38, 0x03,
    0x08, 0x30, 0x31, 0x00, 0x31, 0x00, 0x30, 0x00,
    0x32, 0x30, 0x28, 0x00, 0x31, 0x00, 0x29, 0x00,
    0x28, 0x00, 0x31, 0x00, 0x30, 0x00, 0x29, 0x00,
    0x28, 0x32, 0x30, 0x29, 0x31, 0x00, 0x2e, 0x00,
    0x31, 0x00, 0x30, 0x00, 0x2e, 0x00, 0x32, 0x30,
    0x2e, 0x28, 0x00, 0x61, 0x00, 0x29, 0x00, 0x41,
    0x00, 0x61, 0x00, 0x2b, 0x22, 0x00, 0x00, 0x00,
    0x00, 0x3a, 0x3a, 0x3d, 0x3d, 0x3d, 0x3d, 0x3d,
    0x3d, 0xdd, 0x2a, 0xb8, 0x6a, 0x56, 0x00, 0x4e,
    0x00, 0x28, 0x36, 0x3f, 0x59, 0x85, 0x8c, 0xa0,
    0xba, 0x3f, 0x51, 0x00, 0x26, 0x2c, 0x43, 0x57,
    0x6c, 0xa1, 0xb6, 0xc1, 0x9b, 0x52, 0x00, 0x5e,
    0x7a, 0x7f, 0x9d, 0xa6, 0xc1, 0xce, 0xe7, 0xb6,
    0x53, 0xc8, 0x53, 0xe3, 0x53, 0xd7, 0x56, 0x1f,
    0x57, 0xeb, 0x58, 0x02, 0x59, 0x0a, 0x59, 0x15,
    0x59, 0x27, 0x59, 0x73, 0x59, 0x50, 0x5b, 0x80,
    0x5b, 0xf8, 0x5b, 0x0f, 0x5c, 0x22, 0x5c, 0x38,
    0x5c, 0x6e, 0x5c, 0x71, 0x5c, 0xdb, 0x5d, 0xe5,
    0x5d, 0xf1, 0x5d, 0xfe, 0x5d, 0x72, 0x5e, 0x7a,
    0x5e, 0x7f, 0x5e, 0xf4, 0x5e, 0xfe, 0x5e, 0x0b,
    0x5f, 0x13, 0x5f, 0x50, 0x5f, 0x61, 0x5f, 0x73,
    0x5f, 0xc3, 0x5f, 0x08, 0x62, 0x36, 0x62, 0x4b,
    0x62, 0x2f, 0x65, 0x34, 0x65, 0x87, 0x65, 0x97,
    0x65, 0xa4, 0x65, 0xb9, 0x65, 0xe0, 0x65, 0xe5,
    0x65, 0xf0, 0x66, 0x08, 0x67, 0x28, 0x67, 0x20,
    0x6b, 0x62, 0x6b, 0x79, 0x6b, 0xb3, 0x6b, 0xcb,
    0x6b, 0xd4, 0x6b, 0xdb, 0x6b, 0x0f, 0x6c, 0x14,
    0x6c, 0x34, 0x6c, 0x6b, 0x70, 0x2a, 0x72, 0x36,
    0x72, 0x3b, 0x72, 0x3f, 0x72, 0x47, 0x72, 0x59,
    0x72, 0x5b, 0x72, 0xac, 0x72, 0x84, 0x73, 0x89,
    0x73, 0xdc, 0x74, 0xe6, 0x74, 0x18, 0x75, 0x1f,
    0x75, 0x28, 0x75, 0x30, 0x75, 0x8b, 0x75, 0x92,
    0x75, 0x76, 0x76, 0x7d, 0x76, 0xae, 0x76, 0xbf,
    0x76, 0xee, 0x76, 0xdb, 0x77, 0xe2, 0x77, 0xf3,
    0x77, 0x3a, 0x79, 0xb8, 0x79, 0xbe, 0x79, 0x74,
    0x7a, 0xcb, 0x7a, 0xf9, 0x7a, 0x73, 0x7c, 0xf8,
    0x7c, 0x36, 0x7f, 0x51, 0x7f, 0x8a, 0x7f, 0xbd,
    0x7f, 0x01, 0x80, 0x0c, 0x80, 0x12, 0x80, 0x33,
    0x80, 0x7f, 0x80, 0x89, 0x80, 0xe3, 0x81, 0x00,
    0x07, 0x10, 0x19, 0x29, 0x38, 0x3c, 0x8b, 0x8f,
    0x95, 0x4d, 0x86, 0x6b, 0x86, 0x40, 0x88, 0x4c,
    0x88, 0x63, 0x88, 0x7e, 0x89, 0x8b, 0x89, 0xd2,
    0x89, 0x00, 0x8a, 0x37, 0x8c, 0x46, 0x8c, 0x55,
    0x8c, 0x78, 0x8c, 0x9d, 0x8c, 0x64, 0x8d, 0x70,
    0x8d, 0xb3, 0x8d, 0xab, 0x8e, 0xca, 0x8e, 0x9b,
    0x8f, 0xb0, 0x8f, 0xb5, 0x8f, 0x91, 0x90, 0x49,
    0x91, 0xc6, 0x91, 0xcc, 0x91, 0xd1, 0x91, 0x77,
    0x95, 0x80, 0x95, 0x1c, 0x96, 0xb6, 0x96, 0xb9,
    0x96, 0xe8, 0x96, 0x51, 0x97, 0x5e, 0x97, 0x62,
    0x97, 0x69, 0x97, 0xcb, 0x97, 0xed, 0x97, 0xf3,
    0x97, 0x01, 0x98, 0xa8, 0x98, 0xdb, 0x98, 0xdf,
    0x98, 0x96, 0x99, 0x99, 0x99, 0xac, 0x99, 0xa8,
    0x9a, 0xd8, 0x9a, 0xdf, 0x9a, 0x25, 0x9b, 0x2f,
    0x9b, 0x32, 0x9b, 0x3c, 0x9b, 0x5a, 0x9b, 0xe5,
    0x9c, 0x75, 0x9e, 0x7f, 0x9e, 0xa5, 0x9e, 0x00,
    0x16, 0x1e, 0x28, 0x2c, 0x54, 0x58, 0x69, 0x6e,
    0x7b, 0x96, 0xa5, 0xad, 0xe8, 0xf7, 0xfb, 0x12,
    0x30, 0x00, 0x00, 0x41, 0x53, 0x44, 0x53, 0x45,
    0x53, 0x4b, 0x30, 0x99, 0x30, 0x00, 0x00, 0x00,
    0x00, 0x4d, 0x30, 0x99, 0x30, 0x00, 0x00, 0x00,
    0x00, 0x4f, 0x30, 0x99, 0x30, 0x00, 0x00, 0x00,
    0x00, 0x51, 0x30, 0x99, 0x30, 0x00, 0x00, 0x00,
    0x00, 0x53, 0x30, 0x99, 0x30, 0x00, 0x00, 0x00,
    0x00, 0x55, 0x30, 0x99, 0x30, 0x00, 0x00, 0x00,
    0x00, 0x57, 0x30, 0x99, 0x30, 0x00, 0x00, 0x00,
    0x00, 0x59, 0x30, 0x99, 0x30, 0x00, 0x00, 0x00,
    0x00, 0x5b, 0x30, 0x99, 0x30, 0x00, 0x00, 0x00,
    0x00, 0x5d, 0x30, 0x99, 0x30, 0x00, 0x00, 0x00,
    0x00, 0x5f, 0x30, 0x99, 0x30, 0x00, 0x00, 0x00,
    0x00, 0x61, 0x30, 0x99, 0x30, 0x64, 0x30, 0x99,
    0x30, 0x00, 0x00, 0x00, 0x00, 0x66, 0x30, 0x99,
    0x30, 0x00, 0x00, 0x00, 0x00, 0x68, 0x30, 0x99,
    0x30, 0x6f, 0x30, 0x99, 0x30, 0x72, 0x30, 0x99,
    0x30, 0x75, 0x30, 0x99, 0x30, 0x78, 0x30, 0x99,
    0x30, 0x7b, 0x30, 0x99, 0x30, 0x46, 0x30, 0x99,
    0x30, 0x20, 0x00, 0x99, 0x30, 0x9d, 0x30, 0x99,
    0x30, 0x88, 0x30, 0x8a, 0x30, 0xab, 0x30, 0x99,
    0x30, 0x00, 0x00, 0x00, 0x00, 0xad, 0x30, 0x99,
    0x30, 0x00, 0x00, 0x00, 0x00, 0xaf, 0x30, 0x99,
    0x30, 0x00, 0x00, 0x00, 0x00, 0xb1, 0x30, 0x99,
    0x30, 0x00, 0x00, 0x00, 0x00, 0xb3, 0x30, 0x99,
    0x30, 0x00, 0x00, 0x00, 0x00, 0xb5, 0x30, 0x99,
    0x30, 0x00, 0x00, 0x00, 0x00, 0xb7, 0x30, 0x99,
    0x30, 0x00, 0x00, 0x00, 0x00, 0xb9, 0x30, 0x99,
    0x30, 0x00, 0x00, 0x00, 0x00, 0xbb, 0x30, 0x99,
    0x30, 0x00, 0x00, 0x00, 0x00, 0xbd, 0x30, 0x99,
    0x30, 0x00, 0x00, 0x00, 0x00, 0xbf, 0x30, 0x99,
    0x30, 0x00, 0x00, 0x00, 0x00, 0xc1, 0x30, 0x99,
    0x30, 0xc4, 0x30, 0x99, 0x30, 0x00, 0x00, 0x00,
    0x00, 0xc6, 0x30, 0x99, 0x30, 0x00, 0x00, 0x00,
    0x00, 0xc8, 0x30, 0x99, 0x30, 0xcf, 0x30, 0x99,
    0x30, 0xd2, 0x30, 0x99, 0x30, 0xd5, 0x30, 0x99,
    0x30, 0xd8, 0x30, 0x99, 0x30, 0xdb, 0x30, 0x99,
    0x30, 0xa6, 0x30, 0x99, 0x30, 0xef, 0x30, 0x99,
    0x30, 0xfd, 0x30, 0x99, 0x30, 0xb3, 0x30, 0xc8,
    0x30, 0x00, 0x11, 0x00, 0x01, 0xaa, 0x02, 0xac,
    0xad, 0x03, 0x04, 0x05, 0xb0, 0xb1, 0xb2, 0xb3,
    0xb4, 0xb5, 0x1a, 0x06, 0x07, 0x08, 0x21, 0x09,
    0x11, 0x61, 0x11, 0x14, 0x11, 0x4c, 0x00, 0x01,
    0xb3, 0xb4, 0xb8, 0xba, 0xbf, 0xc3, 0xc5, 0x08,
    0xc9, 0xcb, 0x09, 0x0a, 0x0c, 0x0e, 0x0f, 0x13,
    0x15, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1e, 0x22,
    0x2c, 0x33, 0x38, 0xdd, 0xde, 0x43, 0x44, 0x45,
    0x70, 0x71, 0x74, 0x7d, 0x7e, 0x80, 0x8a, 0x8d,
    0x00, 0x4e, 0x8c, 0x4e, 0x09, 0x4e, 0xdb, 0x56,
    0x0a, 0x4e, 0x2d, 0x4e, 0x0b, 0x4e, 0x32, 0x75,
    0x59, 0x4e, 0x19, 0x4e, 0x01, 0x4e, 0x29, 0x59,
    0x30, 0x57, 0xba, 0x4e, 0x28, 0x00, 0x29, 0x00,
    0x00, 0x11, 0x02, 0x11, 0x03, 0x11, 0x05, 0x11,
    0x06, 0x11, 0x07, 0x11, 0x09, 0x11, 0x0b, 0x11,
    0x0c, 0x11, 0x0e, 0x11, 0x0f, 0x11, 0x10, 0x11,
    0x11, 0x11, 0x12, 0x11, 0x28, 0x00, 0x00, 0x11,
    0x61, 0x11, 0x29, 0x00, 0x28, 0x00, 0x02, 0x11,
    0x61, 0x11, 0x29, 0x00, 0x28, 0x00, 0x05, 0x11,
    0x61, 0x11, 0x29, 0x00, 0x28, 0x00, 0x09, 0x11,
    0x61, 0x11, 0x29, 0x00, 0x28, 0x00, 0x0b, 0x11,
    0x61, 0x11, 0x29, 0x00, 0x28, 0x00, 0x0e, 0x11,
    0x61, 0x11, 0x29, 0x00, 0x28, 0x00, 0x0c, 0x11,
    0x6e, 0x11, 0x29, 0x00, 0x28, 0x00, 0x0b, 0x11,
    0x69, 0x11, 0x0c, 0x11, 0x65, 0x11, 0xab, 0x11,
    0x29, 0x00, 0x28, 0x00, 0x0b, 0x11, 0x69, 0x11,
    0x12, 0x11, 0x6e, 0x11, 0x29, 0x00, 0x28, 0x00,
    0x29, 0x00, 0x00, 0x4e, 0x8c, 0x4e, 0x09, 0x4e,
    0xdb, 0x56, 0x94, 0x4e, 0x6d, 0x51, 0x03, 0x4e,
    0x6b, 0x51, 0x5d, 0x4e, 0x41, 0x53, 0x08, 0x67,
    0x6b, 0x70, 0x34, 0x6c, 0x28, 0x67, 0xd1, 0x91,
    0x1f, 0x57, 0xe5, 0x65, 0x2a, 0x68, 0x09, 0x67,
    0x3e, 0x79, 0x0d, 0x54, 0x79, 0x72, 0xa1, 0x8c,
    0x5d, 0x79, 0xb4, 0x52, 0xe3, 0x4e, 0x7c, 0x54,
    0x66, 0x5b, 0xe3, 0x76, 0x01, 0x4f, 0xc7, 0x8c,
    0x54, 0x53, 0x6d, 0x79, 0x11, 0x4f, 0xea, 0x81,
    0xf3, 0x81, 0x4f, 0x55, 0x7c, 0x5e, 0x87, 0x65,
    0x8f, 0x7b, 0x50, 0x54, 0x45, 0x32, 0x00, 0x31,
    0x00, 0x33, 0x00, 0x30, 0x00, 0x00, 0x11, 0x00,
    0x02, 0x03, 0x05, 0x06, 0x07, 0x09, 0x0b, 0x0c,
    0x0e, 0x0f, 0x10, 0x11, 0x12, 0x00, 0x11, 0x00,
    0x61, 0x02, 0x61, 0x03, 0x61, 0x05, 0x61, 0x06,
    0x61, 0x07, 0x61, 0x09, 0x61, 0x0b, 0x61, 0x0c,
    0x61, 0x0e, 0x11, 0x61, 0x11, 0x00, 0x11, 0x0e,
    0x61, 0xb7, 0x00, 0x69, 0x0b, 0x11, 0x01, 0x63,
    0x00, 0x69, 0x0b, 0x11, 0x6e, 0x11, 0x00, 0x4e,
    0x8c, 0x4e, 0x09, 0x4e, 0xdb, 0x56, 0x94, 0x4e,
    0x6d, 0x51, 0x03, 0x4e, 0x6b, 0x51, 0x5d, 0x4e,
    0x41, 0x53, 0x08, 0x67, 0x6b, 0x70, 0x34, 0x6c,
    0x28, 0x67, 0xd1, 0x91, 0x1f, 0x57, 0xe5, 0x65,
    0x2a, 0x68, 0x09, 0x67, 0x3e, 0x79, 0x0d, 0x54,
    0x79, 0x72, 0xa1, 0x8c, 0x5d, 0x79, 0xb4, 0x52,
    0xd8, 0x79, 0x37, 0x75, 0x73, 0x59, 0x69, 0x90,
    0x2a, 0x51, 0x70, 0x53, 0xe8, 0x6c, 0x05, 0x98,
    0x11, 0x4f, 0x99, 0x51, 0x63, 0x6b, 0x0a, 0x4e,
    0x2d, 0x4e, 0x0b, 0x4e, 0xe6, 0x5d, 0xf3, 0x53,
    0x3b, 0x53, 0x97, 0x5b, 0x66, 0x5b, 0xe3, 0x76,
    0x01, 0x4f, 0xc7, 0x8c, 0x54, 0x53, 0x1c, 0x59,
    0x33, 0x00, 0x36, 0x00, 0x34, 0x00, 0x30, 0x00,
    0x35, 0x30, 0x31, 0x00, 0x08, 0x67, 0x31, 0x00,
    0x30, 0x00, 0x08, 0x67, 0x48, 0x67, 0x65, 0x72,
    0x67, 0x65, 0x56, 0x4c, 0x54, 0x44, 0xa2, 0x30,
    0x00, 0x02, 0x04, 0x06, 0x08, 0x09, 0x0b, 0x0d,
    0x0f, 0x11, 0x13, 0x15, 0x17, 0x19, 0x1b, 0x1d,
    0x1f, 0x22, 0x24, 0x26, 0x28, 0x29, 0x2a, 0x2b,
    0x2c, 0x2d, 0x30, 0x33, 0x36, 0x39, 0x3c, 0x3d,
    0x3e, 0x3f, 0x40, 0x42, 0x44, 0x46, 0x47, 0x48,
    0x49, 0x4a, 0x4b, 0x4d, 0x4e, 0x4f, 0x50, 0xe4,
    0x4e, 0x8c, 0x54, 0xa1, 0x30, 0x01, 0x30, 0x5b,
    0x27, 0x01, 0x4a, 0x34, 0x00, 0x01, 0x52, 0x39,
    0x01, 0xa2, 0x30, 0x00, 0x5a, 0x49, 0xa4, 0x30,
    0x00, 0x27, 0x4f, 0x0c, 0xa4, 0x30, 0x00, 0x4f,
    0x1d, 0x02, 0x05, 0x4f, 0xa8, 0x30, 0x00, 0x11,
    0x07, 0x54, 0x21, 0xa8, 0x30, 0x00, 0x54, 0x03,
    0x54, 0xa4, 0x30, 0x06, 0x4f, 0x15, 0x06, 0x58,
    0x3c, 0x07, 0x00, 0x46, 0xab, 0x30, 0x00, 0x3e,
    0x18, 0x1d, 0x00, 0x42, 0x3f, 0x51, 0xac, 0x30,
    0x00, 0x41, 0x47, 0x00, 0x47, 0x32, 0xae, 0x30,
    0xac, 0x30, 0xae, 0x30, 0x00, 0x1d, 0x4e, 0xad,
    0x30, 0x00, 0x38, 0x3d, 0x4f, 0x01, 0x3e, 0x13,
    0x4f, 0xad, 0x30, 0xed, 0x30, 0xad, 0x30, 0x00,
    0x40, 0x03, 0x3c, 0x33, 0xad, 0x30, 0x00, 0x40,
    0x34, 0x4f, 0x1b, 0x3e, 0xad, 0x30, 0x00, 0x40,
    0x42, 0x16, 0x1b, 0xb0, 0x30, 0x00, 0x39, 0x30,
    0xa4, 0x30, 0x0c, 0x45, 0x3c, 0x24, 0x4f, 0x0b,
    0x47, 0x18, 0x00, 0x49, 0xaf, 0x30, 0x00, 0x3e,
    0x4d, 0x1e, 0xb1, 0x30, 0x00, 0x4b, 0x08, 0x02,
    0x3a, 0x19, 0x02, 0x4b, 0x2c, 0xa4, 0x30, 0x11,
    0x00, 0x0b, 0x47, 0xb5, 0x30, 0x00, 0x3e, 0x0c,
    0x47, 0x2b, 0xb0, 0x30, 0x07, 0x3a, 0x43, 0x00,
    0xb9, 0x30, 0x02, 0x3a, 0x08, 0x02, 0x3a, 0x0f,
    0x07, 0x43, 0x00, 0xb7, 0x30, 0x10, 0x00, 0x12,
    0x34, 0x11, 0x3c, 0x13, 0x17, 0xa4, 0x30, 0x2a,
    0x1f, 0x24, 0x2b, 0x00, 0x20, 0xbb, 0x30, 0x16,
    0x41, 0x00, 0x38, 0x0d, 0xc4, 0x30, 0x0d, 0x38,
    0x00, 0xd0, 0x30, 0x00, 0x2c, 0x1c, 0x1b, 0xa2,
    0x30, 0x32, 0x00, 0x17, 0x26, 0x49, 0xaf, 0x30,
    0x25, 0x00, 0x3c, 0xb3, 0x30, 0x21, 0x00, 0x20,
    0x38, 0xa1, 0x30, 0x34, 0x00, 0x48, 0x22, 0x28,
    0xa3, 0x30, 0x32, 0x00, 0x59, 0x25, 0xa7, 0x30,
    0x2f, 0x1c, 0x10, 0x00, 0x44, 0xd5, 0x30, 0x00,
    0x14, 0x1e, 0xaf, 0x30, 0x29, 0x00, 0x10, 0x4d,
    0x3c, 0xda, 0x30, 0xbd, 0x30, 0xb8, 0x30, 0x22,
    0x13, 0x1a, 0x20, 0x33, 0x0c, 0x22, 0x3b, 0x01,
    0x22, 0x44, 0x00, 0x21, 0x44, 0x07, 0xa4, 0x30,
    0x39, 0x00, 0x4f, 0x24, 0xc8, 0x30, 0x14, 0x23,
    0x00, 0xdb, 0x30, 0xf3, 0x30, 0xc9, 0x30, 0x14,
    0x2a, 0x00, 0x12, 0x33, 0x22, 0x12, 0x33, 0x2a,
    0xa4, 0x30, 0x3a, 0x00, 0x0b, 0x49, 0xa4, 0x30,
    0x3a, 0x00, 0x47, 0x3a, 0x1f, 0x2b, 0x3a, 0x47,
    0x0b, 0xb7, 0x30, 0x27, 0x3c, 0x00, 0x30, 0x3c,
    0xaf, 0x30, 0x30, 0x00, 0x3e, 0x44, 0xdf, 0x30,
    0xea, 0x30, 0xd0, 0x30, 0x0f, 0x1a, 0x00, 0x2c,
    0x1b, 0xe1, 0x30, 0xac, 0x30, 0xac, 0x30, 0x35,
    0x00, 0x1c, 0x47, 0x35, 0x50, 0x1c, 0x3f, 0xa2,
    0x30, 0x42, 0x5a, 0x27, 0x42, 0x5a, 0x49, 0x44,
    0x00, 0x51, 0xc3, 0x30, 0x27, 0x00, 0x05, 0x28,
    0xea, 0x30, 0xe9, 0x30, 0xd4, 0x30, 0x17, 0x00,
    0x28, 0xd6, 0x30, 0x15, 0x26, 0x00, 0x15, 0xec,
    0x30, 0xe0, 0x30, 0xb2, 0x30, 0x3a, 0x41, 0x16,
    0x00, 0x41, 0xc3, 0x30, 0x2c, 0x00, 0x05, 0x30,
    0x00, 0xb9, 0x70, 0x31, 0x00, 0x30, 0x00, 0xb9,
    0x70, 0x32, 0x00, 0x30, 0x00, 0xb9, 0x70, 0x68,
    0x50, 0x61, 0x64, 0x61, 0x41, 0x55, 0x62, 0x61,
    0x72, 0x6f, 0x56, 0x70, 0x63, 0x64, 0x6d, 0x64,
    0x00, 0x6d, 0x00, 0xb2, 0x00, 0x49, 0x00, 0x55,
    0x00, 0x73, 0x5e, 0x10, 0x62, 0x2d, 0x66, 0x8c,
    0x54, 0x27, 0x59, 0x63, 0x6b, 0x0e, 0x66, 0xbb,
    0x6c, 0x2a, 0x68, 0x0f, 0x5f, 0x1a, 0x4f, 0x3e,
    0x79, 0x70, 0x00, 0x41, 0x6e, 0x00, 0x41, 0xbc,
    0x03, 0x41, 0x6d, 0x00, 0x41, 0x6b, 0x00, 0x41,
    0x4b, 0x00, 0x42, 0x4d, 0x00, 0x42, 0x47, 0x00,
    0x42, 0x63, 0x61, 0x6c, 0x6b, 0x63, 0x61, 0x6c,
    0x70, 0x00, 0x46, 0x6e, 0x00, 0x46, 0xbc, 0x03,
    0x46, 0xbc, 0x03, 0x67, 0x6d, 0x00, 0x67, 0x6b,
    0x00, 0x67, 0x48, 0x00, 0x7a, 0x6b, 0x48, 0x7a,
    0x4d, 0x48, 0x7a, 0x47, 0x48, 0x7a, 0x54, 0x48,
    0x7a, 0xbc, 0x03, 0x13, 0x21, 0x6d, 0x00, 0x13,
    0x21, 0x64, 0x00, 0x13, 0x21, 0x6b, 0x00, 0x13,
    0x21, 0x66, 0x00, 0x6d, 0x6e, 0x00, 0x6d, 0xbc,
    0x03, 0x6d, 0x6d, 0x00, 0x6d, 0x63, 0x00, 0x6d,
    0x6b, 0x00, 0x6d, 0x63, 0x00, 0x0a, 0x0a, 0x4f,
    0x00, 0x0a, 0x4f, 0x6d, 0x00, 0xb2, 0x00, 0x63,
    0x00, 0x08, 0x0a, 0x4f, 0x0a, 0x0a, 0x50, 0x00,
    0x0a, 0x50, 0x6d, 0x00, 0xb3, 0x00, 0x6b, 0x00,
    0x6d, 0x00, 0xb3, 0x00, 0x6d, 0x00, 0x15, 0x22,
    0x73, 0x00, 0x6d, 0x00, 0x15, 0x22, 0x73, 0x00,
    0xb2, 0x00, 0x50, 0x61, 0x6b, 0x50, 0x61, 0x4d,
    0x50, 0x61, 0x47, 0x50, 0x61, 0x72, 0x61, 0x64,
    0x72, 0x61, 0x64, 0xd1, 0x73, 0x72, 0x00, 0x61,
    0x00, 0x64, 0x00, 0x15, 0x22, 0x73, 0x00, 0xb2,
    0x00, 0x70, 0x00, 0x73, 0x6e, 0x00, 0x73, 0xbc,
    0x03, 0x73, 0x6d, 0x00, 0x73, 0x70, 0x00, 0x56,
    0x6e, 0x00, 0x56, 0xbc, 0x03, 0x56, 0x6d, 0x00,
    0x56, 0x6b, 0x00, 0x56, 0x4d, 0x00, 0x56, 0x70,
    0x00, 0x57, 0x6e, 0x00, 0x57, 0xbc, 0x03, 0x57,
    0x6d, 0x00, 0x57, 0x6b, 0x00, 0x57, 0x4d, 0x00,
    0x57, 0x6b, 0x00, 0xa9, 0x03, 0x4d, 0x00, 0xa9,
    0x03, 0x61, 0x2e, 0x6d, 0x2e, 0x42, 0x71, 0x63,
    0x63, 0x63, 0x64, 0x43, 0xd1, 0x6b, 0x67, 0x43,
    0x6f, 0x2e, 0x64, 0x42, 0x47, 0x79, 0x68, 0x61,
    0x48, 0x50, 0x69, 0x6e, 0x4b, 0x4b, 0x4b, 0x4d,
    0x6b, 0x74, 0x6c, 0x6d, 0x6c, 0x6e, 0x6c, 0x6f,
    0x67, 0x6c, 0x78, 0x6d, 0x62, 0x6d, 0x69, 0x6c,
    0x6d, 0x6f, 0x6c, 0x50, 0x48, 0x70, 0x2e, 0x6d,
    0x2e, 0x50, 0x50, 0x4d, 0x50, 0x52, 0x73, 0x72,
    0x53, 0x76, 0x57, 0x62, 0x56, 0xd1, 0x6d, 0x41,
    0xd1, 0x6d, 0x31, 0x00, 0xe5, 0x65, 0x31, 0x00,
    0x30, 0x00, 0xe5, 0x65, 0x32, 0x00, 0x30, 0x00,
    0xe5, 0x65, 0x33, 0x00, 0x30, 0x00, 0xe5, 0x65,
    0x67, 0x61, 0x6c, 0x4a, 0x04, 0x4c, 0x04, 0x43,
    0x46, 0x51, 0x26, 0x01, 0x53, 0x01, 0x27, 0xa7,
    0x37, 0xab, 0x6b, 0x02, 0x52, 0xab, 0x48, 0x8c,
    0xf4, 0x66, 0xca, 0x8e, 0xc8, 0x8c, 0xd1, 0x6e,
    0x32, 0x4e, 0xe5, 0x53, 0x9c, 0x9f, 0x9c, 0x9f,
    0x51, 0x59, 0xd1, 0x91, 0x87, 0x55, 0x48, 0x59,
    0xf6, 0x61, 0x69, 0x76, 0x85, 0x7f, 0x3f, 0x86,
    0xba, 0x87, 0xf8, 0x88, 0x8f, 0x90, 0x02, 0x6a,
    0x1b, 0x6d, 0xd9, 0x70, 0xde, 0x73, 0x3d, 0x84,
    0x6a, 0x91, 0xf1, 0x99, 0x82, 0x4e, 0x75, 0x53,
    0x04, 0x6b, 0x1b, 0x72, 0x2d, 0x86, 0x1e, 0x9e,
    0x50, 0x5d, 0xeb, 0x6f, 0xcd, 0x85, 0x64, 0x89,
    0xc9, 0x62, 0xd8, 0x81, 0x1f, 0x88, 0xca, 0x5e,
    0x17, 0x67, 0x6a, 0x6d, 0xfc, 0x72, 0xce, 0x90,
    0x86, 0x4f, 0xb7, 0x51, 0xde, 0x52, 0xc4, 0x64,
    0xd3, 0x6a, 0x10, 0x72, 0xe7, 0x76, 0x01, 0x80,
    0x06, 0x86, 0x5c, 0x86, 0xef, 0x8d, 0x32, 0x97,
    0x6f, 0x9b, 0xfa, 0x9d, 0x8c, 0x78, 0x7f, 0x79,
    0xa0, 0x7d, 0xc9, 0x83, 0x04, 0x93, 0x7f, 0x9e,
    0xd6, 0x8a, 0xdf, 0x58, 0x04, 0x5f, 0x60, 0x7c,
    0x7e, 0x80, 0x62, 0x72, 0xca, 0x78, 0xc2, 0x8c,
    0xf7, 0x96, 0xd8, 0x58, 0x62, 0x5c, 0x13, 0x6a,
    0xda, 0x6d, 0x0f, 0x6f, 0x2f, 0x7d, 0x37, 0x7e,
    0x4b, 0x96, 0xd2, 0x52, 0x8b, 0x80, 0xdc, 0x51,
    0xcc, 0x51, 0x1c, 0x7a, 0xbe, 0x7d, 0xf1, 0x83,
    0x75, 0x96, 0x80, 0x8b, 0xcf, 0x62, 0x02, 0x6a,
    0xfe, 0x8a, 0x39, 0x4e, 0xe7, 0x5b, 0x12, 0x60,
    0x87, 0x73, 0x70, 0x75, 0x17, 0x53, 0xfb, 0x78,
    0xbf, 0x4f, 0xa9, 0x5f, 0x0d, 0x4e, 0xcc, 0x6c,
    0x78, 0x65, 0x22, 0x7d, 0xc3, 0x53, 0x5e, 0x58,
    0x01, 0x77, 0x49, 0x84, 0xaa, 0x8a, 0xba, 0x6b,
    0xb0, 0x8f, 0x88, 0x6c, 0xfe, 0x62, 0xe5, 0x82,
    0xa0, 0x63, 0x65, 0x75, 0xae, 0x4e, 0x69, 0x51,
    0xc9, 0x51, 0x81, 0x68, 0xe7, 0x7c, 0x6f, 0x82,
    0xd2, 0x8a, 0xcf, 0x91, 0xf5, 0x52, 0x42, 0x54,
    0x73, 0x59, 0xec, 0x5e, 0xc5, 0x65, 0xfe, 0x6f,
    0x2a, 0x79, 0xad, 0x95, 0x6a, 0x9a, 0x97, 0x9e,
    0xce, 0x9e, 0x9b, 0x52, 0xc6, 0x66, 0x77, 0x6b,
    0x62, 0x8f, 0x74, 0x5e, 0x90, 0x61, 0x00, 0x62,
    0x9a, 0x64, 0x23, 0x6f, 0x49, 0x71, 0x89, 0x74,
    0xca, 0x79, 0xf4, 0x7d, 0x6f, 0x80, 0x26, 0x8f,
    0xee, 0x84, 0x23, 0x90, 0x4a, 0x93, 0x17, 0x52,
    0xa3, 0x52, 0xbd, 0x54, 0xc8, 0x70, 0xc2, 0x88,
    0xaa, 0x8a, 0xc9, 0x5e, 0xf5, 0x5f, 0x7b, 0x63,
    0xae, 0x6b, 0x3e, 0x7c, 0x75, 0x73, 0xe4, 0x4e,
    0xf9, 0x56, 0xe7, 0x5b, 0xba, 0x5d, 0x1c, 0x60,
    0xb2, 0x73, 0x69, 0x74, 0x9a, 0x7f, 0x46, 0x80,
    0x34, 0x92, 0xf6, 0x96, 0x48, 0x97, 0x18, 0x98,
    0x8b, 0x4f, 0xae, 0x79, 0xb4, 0x91, 0xb8, 0x96,
    0xe1, 0x60, 0x86, 0x4e, 0xda, 0x50, 0xee, 0x5b,
    0x3f, 0x5c, 0x99, 0x65, 0x02, 0x6a, 0xce, 0x71,
    0x42, 0x76, 0xfc, 0x84, 0x7c, 0x90, 0x8d, 0x9f,
    0x88, 0x66, 0x2e, 0x96, 0x89, 0x52, 0x7b, 0x67,
    0xf3, 0x67, 0x41, 0x6d, 0x9c, 0x6e, 0x09, 0x74,
    0x59, 0x75, 0x6b, 0x78, 0x10, 0x7d, 0x5e, 0x98,
    0x6d, 0x51, 0x2e, 0x62, 0x78, 0x96, 0x2b, 0x50,
    0x19, 0x5d, 0xea, 0x6d, 0x2a, 0x8f, 0x8b, 0x5f,
    0x44, 0x61, 0x17, 0x68, 0x87, 0x73, 0x86, 0x96,
    0x29, 0x52, 0x0f, 0x54, 0x65, 0x5c, 0x13, 0x66,
    0x4e, 0x67, 0xa8, 0x68, 0xe5, 0x6c, 0x06, 0x74,
    0xe2, 0x75, 0x79, 0x7f, 0xcf, 0x88, 0xe1, 0x88,
    0xcc, 0x91, 0xe2, 0x96, 0x3f, 0x53, 0xba, 0x6e,
    0x1d, 0x54, 0xd0, 0x71, 0x98, 0x74, 0xfa, 0x85,
    0xa3, 0x96, 0x57, 0x9c, 0x9f, 0x9e, 0x97, 0x67,
    0xcb, 0x6d, 0xe8, 0x81, 0xcb, 0x7a, 0x20, 0x7b,
    0x92, 0x7c, 0xc0, 0x72, 0x99, 0x70, 0x58, 0x8b,
    0xc0, 0x4e, 0x36, 0x83, 0x3a, 0x52, 0x07, 0x52,
    0xa6, 0x5e, 0xd3, 0x62, 0xd6, 0x7c, 0x85, 0x5b,
    0x1e, 0x6d, 0xb4, 0x66, 0x3b, 0x8f, 0x4c, 0x88,
    0x4d, 0x96, 0x8b, 0x89, 0xd3, 0x5e, 0x40, 0x51,
    0xc0, 0x55, 0x00, 0x00, 0x00, 0x00, 0x5a, 0x58,
    0x00, 0x00, 0x74, 0x66, 0x00, 0x00, 0x00, 0x00,
    0xde, 0x51, 0x2a, 0x73, 0xca, 0x76, 0x3c, 0x79,
    0x5e, 0x79, 0x65, 0x79, 0x8f, 0x79, 0x56, 0x97,
    0xbe, 0x7c, 0xbd, 0x7f, 0x00, 0x00, 0x12, 0x86,
    0x00, 0x00, 0xf8, 0x8a, 0x00, 0x00, 0x00, 0x00,
    0x38, 0x90, 0xfd, 0x90, 0xef, 0x98, 0xfc, 0x98,
    0x28, 0x99, 0xb4, 0x9d, 0xde, 0x90, 0xb7, 0x96,
    0xae, 0x4f, 0xe7, 0x50, 0x4d, 0x51, 0xc9, 0x52,
    0xe4, 0x52, 0x51, 0x53, 0x9d, 0x55, 0x06, 0x56,
    0x68, 0x56, 0x40, 0x58, 0xa8, 0x58, 0x64, 0x5c,
    0x6e, 0x5c, 0x94, 0x60, 0x68, 0x61, 0x8e, 0x61,
    0xf2, 0x61, 0x4f, 0x65, 0xe2, 0x65, 0x91, 0x66,
    0x85, 0x68, 0x77, 0x6d, 0x1a, 0x6e, 0x22, 0x6f,
    0x6e, 0x71, 0x2b, 0x72, 0x22, 0x74, 0x91, 0x78,
    0x3e, 0x79, 0x49, 0x79, 0x48, 0x79, 0x50, 0x79,
    0x56, 0x79, 0x5d, 0x79, 0x8d, 0x79, 0x8e, 0x79,
    0x40, 0x7a, 0x81, 0x7a, 0xc0, 0x7b, 0xf4, 0x7d,
    0x09, 0x7e, 0x41, 0x7e, 0x72, 0x7f, 0x05, 0x80,
    0xed, 0x81, 0x79, 0x82, 0x79, 0x82, 0x57, 0x84,
    0x10, 0x89, 0x96, 0x89, 0x01, 0x8b, 0x39, 0x8b,
    0xd3, 0x8c, 0x08, 0x8d, 0xb6, 0x8f, 0x38, 0x90,
    0xe3, 0x96, 0xff, 0x97, 0x3b, 0x98, 0x75, 0x60,
    0xee, 0x42, 0x18, 0x82, 0x02, 0x26, 0x4e, 0xb5,
    0x51, 0x68, 0x51, 0x80, 0x4f, 0x45, 0x51, 0x80,
    0x51, 0xc7, 0x52, 0xfa, 0x52, 0x9d, 0x55, 0x55,
    0x55, 0x99, 0x55, 0xe2, 0x55, 0x5a, 0x58, 0xb3,
    0x58, 0x44, 0x59, 0x54, 0x59, 0x62, 0x5a, 0x28,
    0x5b, 0xd2, 0x5e, 0xd9, 0x5e, 0x69, 0x5f, 0xad,
    0x5f, 0xd8, 0x60, 0x4e, 0x61, 0x08, 0x61, 0x8e,
    0x61, 0x60, 0x61, 0xf2, 0x61, 0x34, 0x62, 0xc4,
    0x63, 0x1c, 0x64, 0x52, 0x64, 0x56, 0x65, 0x74,
    0x66, 0x17, 0x67, 0x1b, 0x67, 0x56, 0x67, 0x79,
    0x6b, 0xba, 0x6b, 0x41, 0x6d, 0xdb, 0x6e, 0xcb,
    0x6e, 0x22, 0x6f, 0x1e, 0x70, 0x6e, 0x71, 0xa7,
    0x77, 0x35, 0x72, 0xaf, 0x72, 0x2a, 0x73, 0x71,
    0x74, 0x06, 0x75, 0x3b, 0x75, 0x1d, 0x76, 0x1f,
    0x76, 0xca, 0x76, 0xdb, 0x76, 0xf4, 0x76, 0x4a,
    0x77, 0x40, 0x77, 0xcc, 0x78, 0xb1, 0x7a, 0xc0,
    0x7b, 0x7b, 0x7c, 0x5b, 0x7d, 0xf4, 0x7d, 0x3e,
    0x7f, 0x05, 0x80, 0x52, 0x83, 0xef, 0x83, 0x79,
    0x87, 0x41, 0x89, 0x86, 0x89, 0x96, 0x89, 0xbf,
    0x8a, 0xf8, 0x8a, 0xcb, 0x8a, 0x01, 0x8b, 0xfe,
    0x8a, 0xed, 0x8a, 0x39, 0x8b, 0x8a, 0x8b, 0x08,
    0x8d, 0x38, 0x8f, 0x72, 0x90, 0x99, 0x91, 0x76,
    0x92, 0x7c, 0x96, 0xe3, 0x96, 0x56, 0x97, 0xdb,
    0x97, 0xff, 0x97, 0x0b, 0x98, 0x3b, 0x98, 0x12,
    0x9b, 0x9c, 0x9f, 0x4a, 0x28, 0x44, 0x28, 0xd5,
    0x33, 0x9d, 0x3b, 0x18, 0x40, 0x39, 0x40, 0x49,
    0x52, 0xd0, 0x5c, 0xd3, 0x7e, 0x43, 0x9f, 0x8e,
    0x9f, 0x2a, 0xa0, 0x02, 0x66, 0x66, 0x66, 0x69,
    0x66, 0x6c, 0x66, 0x66, 0x69, 0x66, 0x66, 0x6c,
    0x7f, 0x01, 0x74, 0x73, 0x00, 0x74, 0x65, 0x05,
    0x0f, 0x11, 0x0f, 0x00, 0x0f, 0x06, 0x19, 0x11,
    0x0f, 0x08, 0xd9, 0x05, 0xb4, 0x05, 0x00, 0x00,
    0x00, 0x00, 0xf2, 0x05, 0xb7, 0x05, 0xd0, 0x05,
    0x12, 0x00, 0x03, 0x04, 0x0b, 0x0c, 0x0d, 0x18,
    0x1a, 0xe9, 0x05, 0xc1, 0x05, 0xe9, 0x05, 0xc2,
    0x05, 0x49, 0xfb, 0xc1, 0x05, 0x49, 0xfb, 0xc2,
    0x05, 0xd0, 0x05, 0xb7, 0x05, 0xd0, 0x05, 0xb8,
    0x05, 0xd0, 0x05, 0xbc, 0x05, 0xd8, 0x05, 0xbc,
    0x05, 0xde, 0x05, 0xbc, 0x05, 0xe0, 0x05, 0xbc,
    0x05, 0xe3, 0x05, 0xbc, 0x05, 0xb9, 0x05, 0x2d,
    0x03, 0x2e, 0x03, 0x2f, 0x03, 0x30, 0x03, 0x31,
    0x03, 0x1c, 0x00, 0x18, 0x06, 0x22, 0x06, 0x2b,
    0x06, 0xd0, 0x05, 0xdc, 0x05, 0x71, 0x06, 0x00,
    0x00, 0x0a, 0x0a, 0x0a, 0x0a, 0x0d, 0x0d, 0x0d,
    0x0d, 0x0f, 0x0f, 0x0f, 0x0f, 0x09, 0x09, 0x09,
    0x09, 0x0e, 0x0e, 0x0e, 0x0e, 0x08, 0x08, 0x08,
    0x08, 0x33, 0x33, 0x33, 0x33, 0x35, 0x35, 0x35,
    0x35, 0x13, 0x13, 0x13, 0x13, 0x12, 0x12, 0x12,
    0x12, 0x15, 0x15, 0x15, 0x15, 0x16, 0x16, 0x16,
    0x16, 0x1c, 0x1c, 0x1b, 0x1b, 0x1d, 0x1d, 0x17,
    0x17, 0x27, 0x27, 0x20, 0x20, 0x38, 0x38, 0x38,
    0x38, 0x3e, 0x3e, 0x3e, 0x3e, 0x42, 0x42, 0x42,
    0x42, 0x40, 0x40, 0x40, 0x40, 0x49, 0x49, 0x4a,
    0x4a, 0x4a, 0x4a, 0x4f, 0x4f, 0x50, 0x50, 0x50,
    0x50, 0x4d, 0x4d, 0x4d, 0x4d, 0x61, 0x61, 0x62,
    0x62, 0x49, 0x06, 0x64, 0x64, 0x64, 0x64, 0x7e,
    0x7e, 0x7d, 0x7d, 0x7f, 0x7f, 0x2e, 0x82, 0x82,
    0x7c, 0x7c, 0x80, 0x80, 0x87, 0x87, 0x87, 0x87,
    0x00, 0x00, 0x26, 0x06, 0x00, 0x01, 0x00, 0x01,
    0x00, 0xaf, 0x00, 0xaf, 0x00, 0x22, 0x00, 0x22,
    0x00, 0xa1, 0x00, 0xa1, 0x00, 0xa0, 0x00, 0xa0,
    0x00, 0xa2, 0x00, 0xa2, 0x00, 0xaa, 0x00, 0xaa,
    0x00, 0xaa, 0x00, 0x23, 0x00, 0x23, 0x00, 0x23,
    0xcc, 0x06, 0x00, 0x00, 0x00, 0x00, 0x26, 0x06,
    0x00, 0x06, 0x00, 0x07, 0x00, 0x1f, 0x00, 0x23,
    0x00, 0x24, 0x02, 0x06, 0x02, 0x07, 0x02, 0x08,
    0x02, 0x1f, 0x02, 0x23, 0x02, 0x24, 0x04, 0x06,
    0x04, 0x07, 0x04, 0x08, 0x04, 0x1f, 0x04, 0x23,
    0x04, 0x24, 0x05, 0x06, 0x05, 0x1f, 0x05, 0x23,
    0x05, 0x24, 0x06, 0x07, 0x06, 0x1f, 0x07, 0x06,
    0x07, 0x1f, 0x08, 0x06, 0x08, 0x07, 0x08, 0x1f,
    0x0d, 0x06, 0x0d, 0x07, 0x0d, 0x08, 0x0d, 0x1f,
    0x0f, 0x07, 0x0f, 0x1f, 0x10, 0x06, 0x10, 0x07,
    0x10, 0x08, 0x10, 0x1f, 0x11, 0x07, 0x11, 0x1f,
    0x12, 0x1f, 0x13, 0x06, 0x13, 0x1f, 0x14, 0x06,
    0x14, 0x1f, 0x1b, 0x06, 0x1b, 0x07, 0x1b, 0x08,
    0x1b, 0x1f, 0x1b, 0x23, 0x1b, 0x24, 0x1c, 0x07,
    0x1c, 0x1f, 0x1c, 0x23, 0x1c, 0x24, 0x1d, 0x01,
    0x1d, 0x06, 0x1d, 0x07, 0x1d, 0x08, 0x1d, 0x1e,
    0x1d, 0x1f, 0x1d, 0x23, 0x1d, 0x24, 0x1e, 0x06,
    0x1e, 0x07, 0x1e, 0x08, 0x1e, 0x1f, 0x1e, 0x23,
    0x1e, 0x24, 0x1f, 0x06, 0x1f, 0x07, 0x1f, 0x08,
    0x1f, 0x1f, 0x1f, 0x23, 0x1f, 0x24, 0x20, 0x06,
    0x20, 0x07, 0x20, 0x08, 0x20, 0x1f, 0x20, 0x23,
    0x20, 0x24, 0x21, 0x06, 0x21, 0x1f, 0x21, 0x23,
    0x21, 0x24, 0x24, 0x06, 0x24, 0x07, 0x24, 0x08,
    0x24, 0x1f, 0x24, 0x23, 0x24, 0x24, 0x0a, 0x4a,
    0x0b, 0x4a, 0x23, 0x4a, 0x20, 0x00, 0x4c, 0x06,
    0x51, 0x06, 0x51, 0x06, 0xff, 0x00, 0x1f, 0x26,
    0x06, 0x00, 0x0b, 0x00, 0x0c, 0x00, 0x1f, 0x00,
    0x20, 0x00, 0x23, 0x00, 0x24, 0x02, 0x0b, 0x02,
    0x0c, 0x02, 0x1f, 0x02, 0x20, 0x02, 0x23, 0x02,
    0x24, 0x04, 0x0b, 0x04, 0x0c, 0x04, 0x1f, 0x26,
    0x06, 0x04, 0x20, 0x04, 0x23, 0x04, 0x24, 0x05,
    0x0b, 0x05, 0x0c, 0x05, 0x1f, 0x05, 0x20, 0x05,
    0x23, 0x05, 0x24, 0x1b, 0x23, 0x1b, 0x24, 0x1c,
    0x23, 0x1c, 0x24, 0x1d, 0x01, 0x1d, 0x1e, 0x1d,
    0x1f, 0x1d, 0x23, 0x1d, 0x24, 0x1e, 0x1f, 0x1e,
    0x23, 0x1e, 0x24, 0x1f, 0x01, 0x1f, 0x1f, 0x20,
    0x0b, 0x20, 0x0c, 0x20, 0x1f, 0x20, 0x20, 0x20,
    0x23, 0x20, 0x24, 0x23, 0x4a, 0x24, 0x0b, 0x24,
    0x0c, 0x24, 0x1f, 0x24, 0x20, 0x24, 0x23, 0x24,
    0x24, 0x00, 0x06, 0x00, 0x07, 0x00, 0x08, 0x00,
    0x1f, 0x00, 0x21, 0x02, 0x06, 0x02, 0x07, 0x02,
    0x08, 0x02, 0x1f, 0x02, 0x21, 0x04, 0x06, 0x04,
    0x07, 0x04, 0x08, 0x04, 0x1f, 0x04, 0x21, 0x05,
    0x1f, 0x06, 0x07, 0x06, 0x1f, 0x07, 0x06, 0x07,
    0x1f, 0x08, 0x06, 0x08, 0x1f, 0x0d, 0x06, 0x0d,
    0x07, 0x0d, 0x08, 0x0d, 0x1f, 0x0f, 0x07, 0x0f,
    0x08, 0x0f, 0x1f, 0x10, 0x06, 0x10, 0x07, 0x10,
    0x08, 0x10, 0x1f, 0x11, 0x07, 0x12, 0x1f, 0x13,
    0x06, 0x13, 0x1f, 0x14, 0x06, 0x14, 0x1f, 0x1b,
    0x06, 0x1b, 0x07, 0x1b, 0x08, 0x1b, 0x1f, 0x1c,
    0x07, 0x1c, 0x1f, 0x1d, 0x06, 0x1d, 0x07, 0x1d,
    0x08, 0x1d, 0x1e, 0x1d, 0x1f, 0x1e, 0x06, 0x1e,
    0x07, 0x1e, 0x08, 0x1e, 0x1f, 0x1e, 0x21, 0x1f,
    0x06, 0x1f, 0x07, 0x1f, 0x08, 0x1f, 0x1f, 0x20,
    0x06, 0x20, 0x07, 0x20, 0x08, 0x20, 0x1f, 0x20,
    0x21, 0x21, 0x06, 0x21, 0x1f, 0x21, 0x4a, 0x24,
    0x06, 0x24, 0x07, 0x24, 0x08, 0x24, 0x1f, 0x24,
    0x21, 0x00, 0x1f, 0x00, 0x21, 0x02, 0x1f, 0x02,
    0x21, 0x04, 0x1f, 0x04, 0x21, 0x05, 0x1f, 0x05,
    0x21, 0x0d, 0x1f, 0x0d, 0x21, 0x0e, 0x1f, 0x0e,
    0x21, 0x1d, 0x1e, 0x1d, 0x1f, 0x1e, 0x1f, 0x20,
    0x1f, 0x20, 0x21, 0x24, 0x1f, 0x24, 0x21, 0x40,
    0x06, 0x4e, 0x06, 0x51, 0x06, 0x27, 0x06, 0x10,
    0x22, 0x10, 0x23, 0x12, 0x22, 0x12, 0x23, 0x13,
    0x22, 0x13, 0x23, 0x0c, 0x22, 0x0c, 0x23, 0x0d,
    0x22, 0x0d, 0x23, 0x06, 0x22, 0x06, 0x23, 0x05,
    0x22, 0x05, 0x23, 0x07, 0x22, 0x07, 0x23, 0x0e,
    0x22, 0x0e, 0x23, 0x0f, 0x22, 0x0f, 0x23, 0x0d,
    0x05, 0x0d, 0x06, 0x0d, 0x07, 0x0d, 0x1e, 0x0d,
    0x0a, 0x0c, 0x0a, 0x0e, 0x0a, 0x0f, 0x0a, 0x10,
    0x22, 0x10, 0x23, 0x12, 0x22, 0x12, 0x23, 0x13,
    0x22, 0x13, 0x23, 0x0c, 0x22, 0x0c, 0x23, 0x0d,
    0x22, 0x0d, 0x23, 0x06, 0x22, 0x06, 0x23, 0x05,
    0x22, 0x05, 0x23, 0x07, 0x22, 0x07, 0x23, 0x0e,
    0x22, 0x0e, 0x23, 0x0f, 0x22, 0x0f, 0x23, 0x0d,
    0x05, 0x0d, 0x06, 0x0d, 0x07, 0x0d, 0x1e, 0x0d,
    0x0a, 0x0c, 0x0a, 0x0e, 0x0a, 0x0f, 0x0a, 0x0d,
    0x05, 0x0d, 0x06, 0x0d, 0x07, 0x0d, 0x1e, 0x0c,
    0x20, 0x0d, 0x20, 0x10, 0x1e, 0x0c, 0x05, 0x0c,
    0x06, 0x0c, 0x07, 0x0d, 0x05, 0x0d, 0x06, 0x0d,
    0x07, 0x10, 0x1e, 0x11, 0x1e, 0x00, 0x24, 0x00,
    0x24, 0x2a, 0x06, 0x00, 0x02, 0x1b, 0x00, 0x03,
    0x02, 0x00, 0x03, 0x02, 0x00, 0x03, 0x1b, 0x00,
    0x04, 0x1b, 0x00, 0x1b, 0x02, 0x00, 0x1b, 0x03,
    0x00, 0x1b, 0x04, 0x02, 0x1b, 0x03, 0x02, 0x1b,
    0x03, 0x03, 0x1b, 0x20, 0x03, 0x1b, 0x1f, 0x09,
    0x03, 0x02, 0x09, 0x02, 0x03, 0x09, 0x02, 0x1f,
    0x09, 0x1b, 0x03, 0x09, 0x1b, 0x03, 0x09, 0x1b,
    0x02, 0x09, 0x1b, 0x1b, 0x09, 0x1b, 0x1b, 0x0b,
    0x03, 0x03, 0x0b, 0x03, 0x03, 0x0b, 0x1b, 0x1b,
    0x0a, 0x03, 0x1b, 0x0a, 0x03, 0x1b, 0x0a, 0x02,
    0x20, 0x0a, 0x1b, 0x04, 0x0a, 0x1b, 0x04, 0x0a,
    0x1b, 0x1b, 0x0a, 0x1b, 0x1b, 0x0c, 0x03, 0x1f,
    0x0c, 0x04, 0x1b, 0x0c, 0x04, 0x1b, 0x0d, 0x1b,
    0x03, 0x0d, 0x1b, 0x03, 0x0d, 0x1b, 0x1b, 0x0d,
    0x1b, 0x20, 0x0f, 0x02, 0x1b, 0x0f, 0x1b, 0x1b,
    0x0f, 0x1b, 0x1b, 0x0f, 0x1b, 0x1f, 0x10, 0x1b,
    0x1b, 0x10, 0x1b, 0x20, 0x10, 0x1b, 0x1f, 0x17,
    0x04, 0x1b, 0x17, 0x04, 0x1b, 0x18, 0x1b, 0x03,
    0x18, 0x1b, 0x1b, 0x1a, 0x03, 0x1b, 0x1a, 0x03,
    0x20, 0x1a, 0x03, 0x1f, 0x1a, 0x02, 0x02, 0x1a,
    0x02, 0x02, 0x1a, 0x04, 0x1b, 0x1a, 0x04, 0x1b,
    0x1a, 0x1b, 0x03, 0x1a, 0x1b, 0x03, 0x1b, 0x03,
    0x02, 0x1b, 0x03, 0x1b, 0x1b, 0x03, 0x20, 0x1b,
    0x02, 0x03, 0x1b, 0x02, 0x1b, 0x1b, 0x04, 0x02,
    0x1b, 0x04, 0x1b, 0x28, 0x06, 0x1d, 0x04, 0x06,
    0x1f, 0x1d, 0x04, 0x1f, 0x1d, 0x1d, 0x1e, 0x05,
    0x1d, 0x1e, 0x05, 0x21, 0x1e, 0x04, 0x1d, 0x1e,
    0x04, 0x1d, 0x1e, 0x04, 0x21, 0x1e, 0x1d, 0x22,
    0x1e, 0x1d, 0x21, 0x22, 0x1d, 0x1d, 0x22, 0x1d,
    0x1d, 0x00, 0x06, 0x22, 0x02, 0x04, 0x22, 0x02,
    0x04, 0x21, 0x02, 0x06, 0x22, 0x02, 0x06, 0x21,
    0x02, 0x1d, 0x22, 0x02, 0x1d, 0x21, 0x04, 0x1d,
    0x22, 0x04, 0x05, 0x21, 0x04, 0x1d, 0x21, 0x0b,
    0x06, 0x21, 0x0d, 0x05, 0x22, 0x0c, 0x05, 0x22,
    0x0e, 0x05, 0x22, 0x1c, 0x04, 0x22, 0x1c, 0x1d,
    0x22, 0x22, 0x05, 0x22, 0x22, 0x04, 0x22, 0x22,
    0x1d, 0x22, 0x1d, 0x1d, 0x22, 0x1a, 0x1d, 0x22,
    0x1e, 0x05, 0x22, 0x1a, 0x1d, 0x05, 0x1c, 0x05,
    0x1d, 0x11, 0x1d, 0x22, 0x1b, 0x1d, 0x22, 0x1e,
    0x04, 0x05, 0x1d, 0x06, 0x22, 0x1c, 0x04, 0x1d,
    0x1b, 0x1d, 0x1d, 0x1c, 0x04, 0x1d, 0x1e, 0x04,
    0x05, 0x04, 0x05, 0x22, 0x05, 0x04, 0x22, 0x1d,
    0x04, 0x22, 0x19, 0x1d, 0x22, 0x00, 0x05, 0x22,
    0x1b, 0x1d, 0x1d, 0x11, 0x04, 0x1d, 0x0d, 0x1d,
    0x1d, 0x0b, 0x06, 0x22, 0x1e, 0x04, 0x22, 0x35,
    0x06, 0x00, 0x0f, 0x9d, 0x0d, 0x0f, 0x9d, 0x27,
    0x06, 0x00, 0x1d, 0x1d, 0x20, 0x00, 0x1c, 0x01,
    0x0a, 0x1e, 0x06, 0x1e, 0x08, 0x0e, 0x1d, 0x12,
    0x1e, 0x0a, 0x0c, 0x21, 0x1d, 0x12, 0x1d, 0x23,
    0x20, 0x21, 0x0c, 0x1d, 0x1e, 0x35, 0x06, 0x00,
    0x0f, 0x14, 0x27, 0x06, 0x0e, 0x1d, 0x22, 0xff,
    0x00, 0x1d, 0x1d, 0x20, 0xff, 0x12, 0x1d, 0x23,
    0x20, 0xff, 0x21, 0x0c, 0x1d, 0x1e, 0x27, 0x06,
    0x05, 0x1d, 0xff, 0x05, 0x1d, 0x00, 0x1d, 0x20,
    0x27, 0x06, 0x0a, 0xa5, 0x00, 0x1d, 0x2c, 0x00,
    0x01, 0x30, 0x02, 0x30, 0x3a, 0x00, 0x3b, 0x00,
    0x21, 0x00, 0x3f, 0x00, 0x16, 0x30, 0x17, 0x30,
    0x26, 0x20, 0x13, 0x20, 0x12, 0x01, 0x00, 0x5f,
    0x5f, 0x28, 0x29, 0x7b, 0x7d, 0x08, 0x30, 0x0c,
    0x0d, 0x08, 0x09, 0x02, 0x03, 0x00, 0x01, 0x04,
    0x05, 0x06, 0x07, 0x5b, 0x00, 0x5d, 0x00, 0x3e,
    0x20, 0x3e, 0x20, 0x3e, 0x20, 0x3e, 0x20, 0x5f,
    0x00, 0x5f, 0x00, 0x5f, 0x00, 0x2c, 0x00, 0x01,
    0x30, 0x2e, 0x00, 0x00, 0x00, 0x3b, 0x00, 0x3a,
    0x00, 0x3f, 0x00, 0x21, 0x00, 0x14, 0x20, 0x28,
    0x00, 0x29, 0x00, 0x7b, 0x00, 0x7d, 0x00, 0x14,
    0x30, 0x15, 0x30, 0x23, 0x26, 0x2a, 0x2b, 0x2d,
    0x3c, 0x3e, 0x3d, 0x00, 0x5c, 0x24, 0x25, 0x40,
    0x40, 0x06, 0xff, 0x0b, 0x00, 0x0b, 0xff, 0x0c,
    0x20, 0x00, 0x4d, 0x06, 0x40, 0x06, 0xff, 0x0e,
    0x00, 0x0e, 0xff, 0x0f, 0x00, 0x0f, 0xff, 0x10,
    0x00, 0x10, 0xff, 0x11, 0x00, 0x11, 0xff, 0x12,
    0x00, 0x12, 0x21, 0x06, 0x00, 0x01, 0x01, 0x02,
    0x02, 0x03, 0x03, 0x04, 0x04, 0x05, 0x05, 0x05,
    0x05, 0x06, 0x06, 0x07, 0x07, 0x07, 0x07, 0x08,
    0x08, 0x09, 0x09, 0x09, 0x09, 0x0a, 0x0a, 0x0a,
    0x0a, 0x0b, 0x0b, 0x0b, 0x0b, 0x0c, 0x0c, 0x0c,
    0x0c, 0x0d, 0x0d, 0x0d, 0x0d, 0x0e, 0x0e, 0x0f,
    0x0f, 0x10, 0x10, 0x11, 0x11, 0x12, 0x12, 0x12,
    0x12, 0x13, 0x13, 0x13, 0x13, 0x14, 0x14, 0x14,
    0x14, 0x15, 0x15, 0x15, 0x15, 0x16, 0x16, 0x16,
    0x16, 0x17, 0x17, 0x17, 0x17, 0x18, 0x18, 0x18,
    0x18, 0x19, 0x19, 0x19, 0x19, 0x20, 0x20, 0x20,
    0x20, 0x21, 0x21, 0x21, 0x21, 0x22, 0x22, 0x22,
    0x22, 0x23, 0x23, 0x23, 0x23, 0x24, 0x24, 0x24,
    0x24, 0x25, 0x25, 0x25, 0x25, 0x26, 0x26, 0x26,
    0x26, 0x27, 0x27, 0x28, 0x28, 0x29, 0x29, 0x29,
    0x29, 0x22, 0x06, 0x22, 0x00, 0x22, 0x00, 0x22,
    0x01, 0x22, 0x01, 0x22, 0x03, 0x22, 0x03, 0x22,
    0x05, 0x22, 0x05, 0x21, 0x00, 0x85, 0x29, 0x01,
    0x30, 0x01, 0x0b, 0x0c, 0x00, 0xfa, 0xf1, 0xa0,
    0xa2, 0xa4, 0xa6, 0xa8, 0xe2, 0xe4, 0xe6, 0xc2,
    0xfb, 0xa1, 0xa3, 0xa5, 0xa7, 0xa9, 0xaa, 0xac,
    0xae, 0xb0, 0xb2, 0xb4, 0xb6, 0xb8, 0xba, 0xbc,
    0xbe, 0xc0, 0xc3, 0xc5, 0xc7, 0xc9, 0xca, 0xcb,
    0xcc, 0xcd, 0xce, 0xd1, 0xd4, 0xd7, 0xda, 0xdd,
    0xde, 0xdf, 0xe0, 0xe1, 0xe3, 0xe5, 0xe7, 0xe8,
    0xe9, 0xea, 0xeb, 0xec, 0xee, 0xf2, 0x98, 0x99,
    0x31, 0x31, 0x4f, 0x31, 0x55, 0x31, 0x5b, 0x31,
    0x61, 0x31, 0xa2, 0x00, 0xa3, 0x00, 0xac, 0x00,
    0xaf, 0x00, 0xa6, 0x00, 0xa5, 0x00, 0xa9, 0x20,
    0x00, 0x00, 0x02, 0x25, 0x90, 0x21, 0x91, 0x21,
    0x92, 0x21, 0x93, 0x21, 0xa0, 0x25, 0xcb, 0x25,
    0xd2, 0x05, 0x07, 0x03, 0x01, 0xda, 0x05, 0x07,
    0x03, 0x01, 0xd0, 0x02, 0xd1, 0x02, 0xe6, 0x00,
    0x99, 0x02, 0x53, 0x02, 0x00, 0x00, 0xa3, 0x02,
    0x66, 0xab, 0xa5, 0x02, 0xa4, 0x02, 0x56, 0x02,
    0x57, 0x02, 0x91, 0x1d, 0x58, 0x02, 0x5e, 0x02,
    0xa9, 0x02, 0x64, 0x02, 0x62, 0x02, 0x60, 0x02,
    0x9b, 0x02, 0x27, 0x01, 0x9c, 0x02, 0x67, 0x02,
    0x84, 0x02, 0xaa, 0x02, 0xab, 0x02, 0x6c, 0x02,
    0x04, 0xdf, 0x8e, 0xa7, 0x6e, 0x02, 0x05, 0xdf,
    0x8e, 0x02, 0x06, 0xdf, 0xf8, 0x00, 0x76, 0x02,
    0x77, 0x02, 0x71, 0x00, 0x7a, 0x02, 0x08, 0xdf,
    0x7d, 0x02, 0x7e, 0x02, 0x80, 0x02, 0xa8, 0x02,
    0xa6, 0x02, 0x67, 0xab, 0xa7, 0x02, 0x88, 0x02,
    0x71, 0x2c, 0x00, 0x00, 0x8f, 0x02, 0xa1, 0x02,
    0xa2, 0x02, 0x98, 0x02, 0xc0, 0x01, 0xc1, 0x01,
    0xc2, 0x01, 0x0a, 0xdf, 0x1e, 0xdf, 0x41, 0x04,
    0x40, 0x00, 0x00, 0x00, 0x00, 0x14, 0x99, 0x10,
    0xba, 0x10, 0x00, 0x00, 0x00, 0x00, 0x9b, 0x10,
    0xba, 0x10, 0x05, 0x05, 0xa5, 0x10, 0xba, 0x10,
    0x05, 0x31, 0x11, 0x27, 0x11, 0x32, 0x11, 0x27,
    0x11, 0x55, 0x47, 0x13, 0x3e, 0x13, 0x47, 0x13,
    0x57, 0x13, 0x55, 0x82, 0x13, 0xc9, 0x13, 0x00,
    0x00, 0x00, 0x00, 0x84, 0x13, 0xbb, 0x13, 0x05,
    0x05, 0x8b, 0x13, 0xc2, 0x13, 0x05, 0x90, 0x13,
    0xc9, 0x13, 0x05, 0xc2, 0x13, 0xc2, 0x13, 0x00,
    0x00, 0x00, 0x00, 0xc2, 0x13, 0xb8, 0x13, 0xc2,
    0x13, 0xc9, 0x13, 0x05, 0x55, 0xb9, 0x14, 0xba,
    0x14, 0xb9, 0x14, 0xb0, 0x14, 0x00, 0x00, 0x00,
    0x00, 0xb9, 0x14, 0xbd, 0x14, 0x55, 0x50, 0xb8,
    0x15, 0xaf, 0x15, 0xb9, 0x15, 0xaf, 0x15, 0x55,
    0x35, 0x19, 0x30, 0x19, 0x05, 0x1e, 0x61, 0x1e,
    0x61, 0x1e, 0x61, 0x29, 0x61, 0x1e, 0x61, 0x1f,
    0x61, 0x29, 0x61, 0x1f, 0x61, 0x1e, 0x61, 0x20,
    0x61, 0x21, 0x61, 0x1f, 0x61, 0x22, 0x61, 0x1f,
    0x61, 0x21, 0x61, 0x20, 0x61, 0x55, 0x55, 0x55,
    0x55, 0x67, 0x6d, 0x67, 0x6d, 0x63, 0x6d, 0x67,
    0x6d, 0x69, 0x6d, 0x67, 0x6d, 0x55, 0x05, 0x41,
    0x00, 0x30, 0x00, 0x57, 0xd1, 0x65, 0xd1, 0x58,
    0xd1, 0x65, 0xd1, 0x5f, 0xd1, 0x6e, 0xd1, 0x5f,
    0xd1, 0x6f, 0xd1, 0x5f, 0xd1, 0x70, 0xd1, 0x5f,
    0xd1, 0x71, 0xd1, 0x5f, 0xd1, 0x72, 0xd1, 0x55,
    0x55, 0x55, 0x05, 0xb9, 0xd1, 0x65, 0xd1, 0xba,
    0xd1, 0x65, 0xd1, 0xbb, 0xd1, 0x6e, 0xd1, 0xbc,
    0xd1, 0x6e, 0xd1, 0xbb, 0xd1, 0x6f, 0xd1, 0xbc,
    0xd1, 0x6f, 0xd1, 0x55, 0x55, 0x55, 0x41, 0x00,
    0x61, 0x00, 0x41, 0x00, 0x61, 0x00, 0x69, 0x00,
    0x41, 0x00, 0x61, 0x00, 0x41, 0x00, 0x43, 0x44,
    0x00, 0x00, 0x47, 0x00, 0x00, 0x4a, 0x4b, 0x00,
    0x00, 0x4e, 0x4f, 0x50, 0x51, 0x00, 0x53, 0x54,
    0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x61, 0x62,
    0x63, 0x64, 0x00, 0x66, 0x68, 0x00, 0x70, 0x00,
    0x41, 0x00, 0x61, 0x00, 0x41, 0x42, 0x00, 0x44,
    0x45, 0x46, 0x47, 0x4a, 0x00, 0x53, 0x00, 0x61,
    0x00, 0x41, 0x42, 0x00, 0x44, 0x45, 0x46, 0x47,
    0x00, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x00, 0x4f,
    0x53, 0x00, 0x61, 0x00, 0x41, 0x00, 0x61, 0x00,
    0x41, 0x00, 0x61, 0x00, 0x41, 0x00, 0x61, 0x00,
    0x41, 0x00, 0x61, 0x00, 0x41, 0x00, 0x61, 0x00,
    0x41, 0x00, 0x61, 0x00, 0x31, 0x01, 0x37, 0x02,
    0x91, 0x03, 0xa3, 0x03, 0xb1, 0x03, 0xd1, 0x03,
    0x24, 0x00, 0x1f, 0x04, 0x20, 0x05, 0x91, 0x03,
    0xa3, 0x03, 0xb1, 0x03, 0xd1, 0x03, 0x24, 0x00,
    0x1f, 0x04, 0x20, 0x05, 0x91, 0x03, 0xa3, 0x03,
    0xb1, 0x03, 0xd1, 0x03, 0x24, 0x00, 0x1f, 0x04,
    0x20, 0x05, 0x91, 0x03, 0xa3, 0x03, 0xb1, 0x03,
    0xd1, 0x03, 0x24, 0x00, 0x1f, 0x04, 0x20, 0x05,
    0x91, 0x03, 0xa3, 0x03, 0xb1, 0x03, 0xd1, 0x03,
    0x24, 0x00, 0x1f, 0x04, 0x20, 0x05, 0x0b, 0x0c,
    0x30, 0x00, 0x30, 0x00, 0x30, 0x00, 0x30, 0x00,
    0x30, 0x00, 0x30, 0x04, 0x3a, 0x04, 0x3e, 0x04,
    0x4b, 0x04, 0x4d, 0x04, 0x4e, 0x04, 0x89, 0xa6,
    0x30, 0x04, 0xa9, 0x26, 0x28, 0xb9, 0x7f, 0x9f,
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x08, 0x0a, 0x0b, 0x0e, 0x0f, 0x11, 0x13, 0x14,
    0x15, 0x16, 0x17, 0x18, 0x1a, 0x1b, 0x61, 0x26,
    0x25, 0x2f, 0x7b, 0x51, 0xa6, 0xb1, 0x04, 0x27,
    0x06, 0x00, 0x01, 0x05, 0x08, 0x2a, 0x06, 0x1e,
    0x08, 0x03, 0x0d, 0x20, 0x19, 0x1a, 0x1b, 0x1c,
    0x09, 0x0f, 0x17, 0x0b, 0x18, 0x07, 0x0a, 0x00,
    0x01, 0x04, 0x06, 0x0c, 0x0e, 0x10, 0x44, 0x90,
    0x77, 0x45, 0x28, 0x06, 0x2c, 0x06, 0x00, 0x00,
    0x47, 0x06, 0x33, 0x06, 0x17, 0x10, 0x11, 0x12,
    0x13, 0x00, 0x06, 0x0e, 0x02, 0x0f, 0x34, 0x06,
    0x2a, 0x06, 0x2b, 0x06, 0x2e, 0x06, 0x00, 0x00,
    0x36, 0x06, 0x00, 0x00, 0x3a, 0x06, 0x2d, 0x06,
    0x00, 0x00, 0x4a, 0x06, 0x00, 0x00, 0x44, 0x06,
    0x00, 0x00, 0x46, 0x06, 0x33, 0x06, 0x39, 0x06,
    0x00, 0x00, 0x35, 0x06, 0x42, 0x06, 0x00, 0x00,
    0x34, 0x06, 0x00, 0x00, 0x00, 0x00, 0x2e, 0x06,
    0x00, 0x00, 0x36, 0x06, 0x00, 0x00, 0x3a, 0x06,
    0x00, 0x00, 0xba, 0x06, 0x00, 0x00, 0x6f, 0x06,
    0x00, 0x00, 0x28, 0x06, 0x2c, 0x06, 0x00, 0x00,
    0x47, 0x06, 0x00, 0x00, 0x00, 0x00, 0x2d, 0x06,
    0x37, 0x06, 0x4a, 0x06, 0x43, 0x06, 0x00, 0x00,
    0x45, 0x06, 0x46, 0x06, 0x33, 0x06, 0x39, 0x06,
    0x41, 0x06, 0x35, 0x06, 0x42, 0x06, 0x00, 0x00,
    0x34, 0x06, 0x2a, 0x06, 0x2b, 0x06, 0x2e, 0x06,
    0x00, 0x00, 0x36, 0x06, 0x38, 0x06, 0x3a, 0x06,
    0x6e, 0x06, 0x00, 0x00, 0xa1, 0x06, 0x27, 0x06,
    0x00, 0x01, 0x05, 0x08, 0x20, 0x21, 0x0b, 0x06,
    0x10, 0x23, 0x2a, 0x06, 0x1a, 0x1b, 0x1c, 0x09,
    0x0f, 0x17, 0x0b, 0x18, 0x07, 0x0a, 0x00, 0x01,
    0x04, 0x06, 0x0c, 0x0e, 0x10, 0x28, 0x06, 0x2c,
    0x06, 0x2f, 0x06, 0x00, 0x00, 0x48, 0x06, 0x32,
    0x06, 0x2d, 0x06, 0x37, 0x06, 0x4a, 0x06, 0x2a,
    0x06, 0x1a, 0x1b, 0x1c, 0x09, 0x0f, 0x17, 0x0b,
    0x18, 0x07, 0x0a, 0x00, 0x01, 0x04, 0x06, 0x0c,
    0x0e, 0x10, 0x30, 0x2e, 0x30, 0x00, 0x2c, 0x00,
    0x28, 0x00, 0x41, 0x00, 0x29, 0x00, 0x14, 0x30,
    0x53, 0x00, 0x15, 0x30, 0x43, 0x52, 0x43, 0x44,
    0x57, 0x5a, 0x41, 0x00, 0x48, 0x56, 0x4d, 0x56,
    0x53, 0x44, 0x53, 0x53, 0x50, 0x50, 0x56, 0x57,
    0x43, 0x4d, 0x43, 0x4d, 0x44, 0x4d, 0x52, 0x44,
    0x4a, 0x4b, 0x30, 0x30, 0x00, 0x68, 0x68, 0x4b,
    0x62, 0x57, 0x5b, 0xcc, 0x53, 0xc7, 0x30, 0x8c,
    0x4e, 0x1a, 0x59, 0xe3, 0x89, 0x29, 0x59, 0xa4,
    0x4e, 0x20, 0x66, 0x21, 0x71, 0x99, 0x65, 0x4d,
    0x52, 0x8c, 0x5f, 0x8d, 0x51, 0xb0, 0x65, 0x1d,
    0x52, 0x42, 0x7d, 0x1f, 0x75, 0xa9, 0x8c, 0xf0,
    0x58, 0x39, 0x54, 0x14, 0x6f, 0x95, 0x62, 0x55,
    0x63, 0x00, 0x4e, 0x09, 0x4e, 0x4a, 0x90, 0xe6,
    0x5d, 0x2d, 0x4e, 0xf3, 0x53, 0x07, 0x63, 0x70,
    0x8d, 0x53, 0x62, 0x81, 0x79, 0x7a, 0x7a, 0x08,
    0x54, 0x80, 0x6e, 0x09, 0x67, 0x08, 0x67, 0x33,
    0x75, 0x72, 0x52, 0xb6, 0x55, 0x4d, 0x91, 0x14,
    0x30, 0x15, 0x30, 0x2c, 0x67, 0x09, 0x4e, 0x8c,
    0x4e, 0x89, 0x5b, 0xb9, 0x70, 0x53, 0x62, 0xd7,
    0x76, 0xdd, 0x52, 0x57, 0x65, 0x97, 0x5f, 0xef,
    0x53, 0x30, 0x00, 0x38, 0x4e, 0x05, 0x00, 0x09,
    0x22, 0x01, 0x60, 0x4f, 0xae, 0x4f, 0xbb, 0x4f,
    0x02, 0x50, 0x7a, 0x50, 0x99, 0x50, 0xe7, 0x50,
    0xcf, 0x50, 0x9e, 0x34, 0x3a, 0x06, 0x4d, 0x51,
    0x54, 0x51, 0x64, 0x51, 0x77, 0x51, 0x1c, 0x05,
    0xb9, 0x34, 0x67, 0x51, 0x8d, 0x51, 0x4b, 0x05,
    0x97, 0x51, 0xa4, 0x51, 0xcc, 0x4e, 0xac, 0x51,
    0xb5, 0x51, 0xdf, 0x91, 0xf5, 0x51, 0x03, 0x52,
    0xdf, 0x34, 0x3b, 0x52, 0x46, 0x52, 0x72, 0x52,
    0x77, 0x52, 0x15, 0x35, 0x02, 0x00, 0x20, 0x80,
    0x80, 0x00, 0x08, 0x00, 0x00, 0xc7, 0x52, 0x00,
    0x02, 0x1d, 0x33, 0x3e, 0x3f, 0x50, 0x82, 0x8a,
    0x93, 0xac, 0xb6, 0xb8, 0xb8, 0xb8, 0x2c, 0x0a,
    0x70, 0x70, 0xca, 0x53, 0xdf, 0x53, 0x63, 0x0b,
    0xeb, 0x53, 0xf1, 0x53, 0x06, 0x54, 0x9e, 0x54,
    0x38, 0x54, 0x48, 0x54, 0x68, 0x54, 0xa2, 0x54,
    0xf6, 0x54, 0x10, 0x55, 0x53, 0x55, 0x63, 0x55,
    0x84, 0x55, 0x84, 0x55, 0x99, 0x55, 0xab, 0x55,
    0xb3, 0x55, 0xc2, 0x55, 0x16, 0x57, 0x06, 0x56,
    0x17, 0x57, 0x51, 0x56, 0x74, 0x56, 0x07, 0x52,
    0xee, 0x58, 0xce, 0x57, 0xf4, 0x57, 0x0d, 0x58,
    0x8b, 0x57, 0x32, 0x58, 0x31, 0x58, 0xac, 0x58,
    0xe4, 0x14, 0xf2, 0x58, 0xf7, 0x58, 0x06, 0x59,
    0x1a, 0x59, 0x22, 0x59, 0x62, 0x59, 0xa8, 0x16,
    0xea, 0x16, 0xec, 0x59, 0x1b, 0x5a, 0x27, 0x5a,
    0xd8, 0x59, 0x66, 0x5a, 0xee, 0x36, 0xfc, 0x36,
    0x08, 0x5b, 0x3e, 0x5b, 0x3e, 0x5b, 0xc8, 0x19,
    0xc3, 0x5b, 0xd8, 0x5b, 0xe7, 0x5b, 0xf3, 0x5b,
    0x18, 0x1b, 0xff, 0x5b, 0x06, 0x5c, 0x53, 0x5f,
    0x22, 0x5c, 0x81, 0x37, 0x60, 0x5c, 0x6e, 0x5c,
    0xc0, 0x5c, 0x8d, 0x5c, 0xe4, 0x1d, 0x43, 0x5d,
    0xe6, 0x1d, 0x6e, 0x5d, 0x6b, 0x5d, 0x7c, 0x5d,
    0xe1, 0x5d, 0xe2, 0x5d, 0x2f, 0x38, 0xfd, 0x5d,
    0x28, 0x5e, 0x3d, 0x5e, 0x69, 0x5e, 0x62, 0x38,
    0x83, 0x21, 0x7c, 0x38, 0xb0, 0x5e, 0xb3, 0x5e,
    0xb6, 0x5e, 0xca, 0x5e, 0x92, 0xa3, 0xfe, 0x5e,
    0x31, 0x23, 0x31, 0x23, 0x01, 0x82, 0x22, 0x5f,
    0x22, 0x5f, 0xc7, 0x38, 0xb8, 0x32, 0xda, 0x61,
    0x62, 0x5f, 0x6b, 0x5f, 0xe3, 0x38, 0x9a, 0x5f,
    0xcd, 0x5f, 0xd7, 0x5f, 0xf9, 0x5f, 0x81, 0x60,
    0x3a, 0x39, 0x1c, 0x39, 0x94, 0x60, 0xd4, 0x26,
    0xc7, 0x60, 0x02, 0x02, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x08, 0x00, 0x0a, 0x00, 0x00,
    0x02, 0x08, 0x00, 0x80, 0x08, 0x00, 0x00, 0x08,
    0x80, 0x28, 0x80, 0x02, 0x00, 0x00, 0x02, 0x48,
    0x61, 0x00, 0x04, 0x06, 0x04, 0x32, 0x46, 0x6a,
    0x5c, 0x67, 0x96, 0xaa, 0xae, 0xc8, 0xd3, 0x5d,
    0x62, 0x00, 0x54, 0x77, 0xf3, 0x0c, 0x2b, 0x3d,
    0x63, 0xfc, 0x62, 0x68, 0x63, 0x83, 0x63, 0xe4,
    0x63, 0xf1, 0x2b, 0x22, 0x64, 0xc5, 0x63, 0xa9,
    0x63, 0x2e, 0x3a, 0x69, 0x64, 0x7e, 0x64, 0x9d,
    0x64, 0x77, 0x64, 0x6c, 0x3a, 0x4f, 0x65, 0x6c,
    0x65, 0x0a, 0x30, 0xe3, 0x65, 0xf8, 0x66, 0x49,
    0x66, 0x19, 0x3b, 0x91, 0x66, 0x08, 0x3b, 0xe4,
    0x3a, 0x92, 0x51, 0x95, 0x51, 0x00, 0x67, 0x9c,
    0x66, 0xad, 0x80, 0xd9, 0x43, 0x17, 0x67, 0x1b,
    0x67, 0x21, 0x67, 0x5e, 0x67, 0x53, 0x67, 0xc3,
    0x33, 0x49, 0x3b, 0xfa, 0x67, 0x85, 0x67, 0x52,
    0x68, 0x85, 0x68, 0x6d, 0x34, 0x8e, 0x68, 0x1f,
    0x68, 0x14, 0x69, 0x9d, 0x3b, 0x42, 0x69, 0xa3,
    0x69, 0xea, 0x69, 0xa8, 0x6a, 0xa3, 0x36, 0xdb,
    0x6a, 0x18, 0x3c, 0x21, 0x6b, 0xa7, 0x38, 0x54,
    0x6b, 0x4e, 0x3c, 0x72, 0x6b, 0x9f, 0x6b, 0xba,
    0x6b, 0xbb, 0x6b, 0x8d, 0x3a, 0x0b, 0x1d, 0xfa,
    0x3a, 0x4e, 0x6c, 0xbc, 0x3c, 0xbf, 0x6c, 0xcd,
    0x6c, 0x67, 0x6c, 0x16, 0x6d, 0x3e, 0x6d, 0x77,
    0x6d, 0x41, 0x6d, 0x69, 0x6d, 0x78, 0x6d, 0x85,
    0x6d, 0x1e, 0x3d, 0x34, 0x6d, 0x2f, 0x6e, 0x6e,
    0x6e, 0x33, 0x3d, 0xcb, 0x6e, 0xc7, 0x6e, 0xd1,
    0x3e, 0xf9, 0x6d, 0x6e, 0x6f, 0x5e, 0x3f, 0x8e,
    0x3f, 0xc6, 0x6f, 0x39, 0x70, 0x1e, 0x70, 0x1b,
    0x70, 0x96, 0x3d, 0x4a, 0x70, 0x7d, 0x70, 0x77,
    0x70, 0xad, 0x70, 0x25, 0x05, 0x45, 0x71, 0x63,
    0x42, 0x9c, 0x71, 0xab, 0x43, 0x28, 0x72, 0x35,
    0x72, 0x50, 0x72, 0x08, 0x46, 0x80, 0x72, 0x95,
    0x72, 0x35, 0x47, 0x02, 0x20, 0x00, 0x00, 0x20,
    0x00, 0x00, 0x00, 0x00, 0x08, 0x80, 0x00, 0x00,
    0x02, 0x02, 0x80, 0x8a, 0x00, 0x00, 0x20, 0x00,
    0x08, 0x0a, 0x00, 0x80, 0x88, 0x80, 0x20, 0x14,
    0x48, 0x7a, 0x73, 0x8b, 0x73, 0xac, 0x3e, 0xa5,
    0x73, 0xb8, 0x3e, 0xb8, 0x3e, 0x47, 0x74, 0x5c,
    0x74, 0x71, 0x74, 0x85, 0x74, 0xca, 0x74, 0x1b,
    0x3f, 0x24, 0x75, 0x36, 0x4c, 0x3e, 0x75, 0x92,
    0x4c, 0x70, 0x75, 0x9f, 0x21, 0x10, 0x76, 0xa1,
    0x4f, 0xb8, 0x4f, 0x44, 0x50, 0xfc, 0x3f, 0x08,
    0x40, 0xf4, 0x76, 0xf3, 0x50, 0xf2, 0x50, 0x19,
    0x51, 0x33, 0x51, 0x1e, 0x77, 0x1f, 0x77, 0x1f,
    0x77, 0x4a, 0x77, 0x39, 0x40, 0x8b, 0x77, 0x46,
    0x40, 0x96, 0x40, 0x1d, 0x54, 0x4e, 0x78, 0x8c,
    0x78, 0xcc, 0x78, 0xe3, 0x40, 0x26, 0x56, 0x56,
    0x79, 0x9a, 0x56, 0xc5, 0x56, 0x8f, 0x79, 0xeb,
    0x79, 0x2f, 0x41, 0x40, 0x7a, 0x4a, 0x7a, 0x4f,
    0x7a, 0x7c, 0x59, 0xa7, 0x5a, 0xa7, 0x5a, 0xee,
    0x7a, 0x02, 0x42, 0xab, 0x5b, 0xc6, 0x7b, 0xc9,
    0x7b, 0x27, 0x42, 0x80, 0x5c, 0xd2, 0x7c, 0xa0,
    0x42, 0xe8, 0x7c, 0xe3, 0x7c, 0x00, 0x7d, 0x86,
    0x5f, 0x63, 0x7d, 0x01, 0x43, 0xc7, 0x7d, 0x02,
    0x7e, 0x45, 0x7e, 0x34, 0x43, 0x28, 0x62, 0x47,
    0x62, 0x59, 0x43, 0xd9, 0x62, 0x7a, 0x7f, 0x3e,
    0x63, 0x95, 0x7f, 0xfa, 0x7f, 0x05, 0x80, 0xda,
    0x64, 0x23, 0x65, 0x60, 0x80, 0xa8, 0x65, 0x70,
    0x80, 0x5f, 0x33, 0xd5, 0x43, 0xb2, 0x80, 0x03,
    0x81, 0x0b, 0x44, 0x3e, 0x81, 0xb5, 0x5a, 0xa7,
    0x67, 0xb5, 0x67, 0x93, 0x33, 0x9c, 0x33, 0x01,
    0x82, 0x04, 0x82, 0x9e, 0x8f, 0x6b, 0x44, 0x91,
    0x82, 0x8b, 0x82, 0x9d, 0x82, 0xb3, 0x52, 0xb1,
    0x82, 0xb3, 0x82, 0xbd, 0x82, 0xe6, 0x82, 0x3c,
    0x6b, 0xe5, 0x82, 0x1d, 0x83, 0x63, 0x83, 0xad,
    0x83, 0x23, 0x83, 0xbd, 0x83, 0xe7, 0x83, 0x57,
    0x84, 0x53, 0x83, 0xca, 0x83, 0xcc, 0x83, 0xdc,
    0x83, 0x36, 0x6c, 0x6b, 0x6d, 0x02, 0x00, 0x00,
    0x20, 0x22, 0x2a, 0xa0, 0x0a, 0x00, 0x20, 0x80,
    0x28, 0x00, 0xa8, 0x20, 0x20, 0x00, 0x02, 0x80,
    0x22, 0x02, 0x8a, 0x08, 0x00, 0xaa, 0x00, 0x00,
    0x00, 0x02, 0x00, 0x00, 0x28, 0xd5, 0x6c, 0x2b,
    0x45, 0xf1, 0x84, 0xf3, 0x84, 0x16, 0x85, 0xca,
    0x73, 0x64, 0x85, 0x2c, 0x6f, 0x5d, 0x45, 0x61,
    0x45, 0xb1, 0x6f, 0xd2, 0x70, 0x6b, 0x45, 0x50,
    0x86, 0x5c, 0x86, 0x67, 0x86, 0x69, 0x86, 0xa9,
    0x86, 0x88, 0x86, 0x0e, 0x87, 0xe2, 0x86, 0x79,
    0x87, 0x28, 0x87, 0x6b, 0x87, 0x86, 0x87, 0xd7,
    0x45, 0xe1, 0x87, 0x01, 0x88, 0xf9, 0x45, 0x60,
    0x88, 0x63, 0x88, 0x67, 0x76, 0xd7, 0x88, 0xde,
    0x88, 0x35, 0x46, 0xfa, 0x88, 0xbb, 0x34, 0xae,
    0x78, 0x66, 0x79, 0xbe, 0x46, 0xc7, 0x46, 0xa0,
    0x8a, 0xed, 0x8a, 0x8a, 0x8b, 0x55, 0x8c, 0xa8,
    0x7c, 0xab, 0x8c, 0xc1, 0x8c, 0x1b, 0x8d, 0x77,
    0x8d, 0x2f, 0x7f, 0x04, 0x08, 0xcb, 0x8d, 0xbc,
    0x8d, 0xf0, 0x8d, 0xde, 0x08, 0xd4, 0x8e, 0x38,
    0x8f, 0xd2, 0x85, 0xed, 0x85, 0x94, 0x90, 0xf1,
    0x90, 0x11, 0x91, 0x2e, 0x87, 0x1b, 0x91, 0x38,
    0x92, 0xd7, 0x92, 0xd8, 0x92, 0x7c, 0x92, 0xf9,
    0x93, 0x15, 0x94, 0xfa, 0x8b, 0x8b, 0x95, 0x95,
    0x49, 0xb7, 0x95, 0x77, 0x8d, 0xe6, 0x49, 0xc3,
    0x96, 0xb2, 0x5d, 0x23, 0x97, 0x45, 0x91, 0x1a,
    0x92, 0x6e, 0x4a, 0x76, 0x4a, 0xe0, 0x97, 0x0a,
    0x94, 0xb2, 0x4a, 0x96, 0x94, 0x0b, 0x98, 0x0b,
    0x98, 0x29, 0x98, 0xb6, 0x95, 0xe2, 0x98, 0x33,
    0x4b, 0x29, 0x99, 0xa7, 0x99, 0xc2, 0x99, 0xfe,
    0x99, 0xce, 0x4b, 0x30, 0x9b, 0x12, 0x9b, 0x40,
    0x9c, 0xfd, 0x9c, 0xce, 0x4c, 0xed, 0x4c, 0x67,
    0x9d, 0xce, 0xa0, 0xf8, 0x4c, 0x05, 0xa1, 0x0e,
    0xa2, 0x91, 0xa2, 0xbb, 0x9e, 0x56, 0x4d, 0xf9,
    0x9e, 0xfe, 0x9e, 0x05, 0x9f, 0x0f, 0x9f, 0x16,
    0x9f, 0x3b, 0x9f, 0x00, 0xa6, 0x02, 0x88, 0xa0,
    0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x28, 0x00,
    0x08, 0xa0, 0x80, 0xa0, 0x80, 0x00, 0x80, 0x80,
    0x00, 0x0a, 0x88, 0x80, 0x00, 0x80, 0x00, 0x20,
    0x2a, 0x00, 0x80,
};

static const uint16_t unicode_comp_table[965] = {
    0x4a01, 0x49c0, 0x4a02, 0x0280, 0x0281, 0x0282, 0x0283, 0x02c0,
    0x02c2, 0x0a00, 0x0284, 0x2442, 0x0285, 0x07c0, 0x0980, 0x0982,
    0x2440, 0x2280, 0x02c4, 0x2282, 0x2284, 0x2286, 0x02c6, 0x02c8,
    0x02ca, 0x02cc, 0x0287, 0x228a, 0x02ce, 0x228c, 0x2290, 0x2292,
    0x228e, 0x0288, 0x0289, 0x028a, 0x2482, 0x0300, 0x0302, 0x0304,
    0x028b, 0x2480, 0x0308, 0x0984, 0x0986, 0x2458, 0x0a02, 0x0306,
    0x2298, 0x229a, 0x229e, 0x0900, 0x030a, 0x22a0, 0x030c, 0x030e,
    0x0840, 0x0310, 0x0312, 0x22a2, 0x22a6, 0x09c0, 0x22a4, 0x22a8,
    0x22aa, 0x028c, 0x028d, 0x028e, 0x0340, 0x0342, 0x0344, 0x0380,
    0x028f, 0x248e, 0x07c2, 0x0988, 0x098a, 0x2490, 0x0346, 0x22ac,
    0x0400, 0x22b0, 0x0842, 0x22b2, 0x0402, 0x22b4, 0x0440, 0x0444,
    0x22b6, 0x0442, 0x22c2, 0x22c0, 0x22c4, 0x22c6, 0x22c8, 0x0940,
    0x04c0, 0x0291, 0x22ca, 0x04c4, 0x22cc, 0x04c2, 0x22d0, 0x22ce,
    0x0292, 0x0293, 0x0294, 0x0295, 0x0540, 0x0542, 0x0a08, 0x0296,
    0x2494, 0x0544, 0x07c4, 0x098c, 0x098e, 0x06c0, 0x2492, 0x0844,
    0x2308, 0x230a, 0x0580, 0x230c, 0x0584, 0x0990, 0x0992, 0x230e,
    0x0582, 0x2312, 0x0586, 0x0588, 0x2314, 0x058c, 0x2316, 0x0998,
    0x058a, 0x231e, 0x0590, 0x2320, 0x099a, 0x058e, 0x2324, 0x2322,
    0x0299, 0x029a, 0x029b, 0x05c0, 0x05c2, 0x05c4, 0x029c, 0x24ac,
    0x05c6, 0x05c8, 0x07c6, 0x0994, 0x0996, 0x0700, 0x24aa, 0x2326,
    0x05ca, 0x232a, 0x2328, 0x2340, 0x2342, 0x2344, 0x2346, 0x05cc,
    0x234a, 0x2348, 0x234c, 0x234e, 0x2350, 0x24b8, 0x029d, 0x05ce,
    0x24be, 0x0a0c, 0x2352, 0x0600, 0x24bc, 0x24ba, 0x0640, 0x2354,
    0x0642, 0x0644, 0x2356, 0x2358, 0x02a0, 0x02a1, 0x02a2, 0x02a3,
    0x02c1, 0x02c3, 0x0a01, 0x02a4, 0x2443, 0x02a5, 0x07c1, 0x0981,
    0x0983, 0x2441, 0x2281, 0x02c5, 0x2283, 0x2285, 0x2287, 0x02c7,
    0x02c9, 0x02cb, 0x02cd, 0x02a7, 0x228b, 0x02cf, 0x228d, 0x2291,
    0x2293, 0x228f, 0x02a8, 0x02a9, 0x02aa, 0x2483, 0x0301, 0x0303,
    0x0305, 0x02ab, 0x2481, 0x0309, 0x0985, 0x0987, 0x2459, 0x0a03,
    0x0307, 0x2299, 0x229b, 0x229f, 0x0901, 0x030b, 0x22a1, 0x030d,
    0x030f, 0x0841, 0x0311, 0x0313, 0x22a3, 0x22a7, 0x09c1, 0x22a5,
    0x22a9, 0x22ab, 0x2380, 0x02ac, 0x02ad, 0x02ae, 0x0341, 0x0343,
    0x0345, 0x02af, 0x248f, 0x07c3, 0x0989, 0x098b, 0x2491, 0x0347,
    0x22ad, 0x0401, 0x0884, 0x22b1, 0x0843, 0x22b3, 0x0403, 0x22b5,
    0x0441, 0x0445, 0x22b7, 0x0443, 0x22c3, 0x22c1, 0x22c5, 0x22c7,
    0x22c9, 0x0941, 0x04c1, 0x02b1, 0x22cb, 0x04c5, 0x22cd, 0x04c3,
    0x22d1, 0x22cf, 0x02b2, 0x02b3, 0x02b4, 0x02b5, 0x0541, 0x0543,
    0x0a09, 0x02b6, 0x2495, 0x0545, 0x07c5, 0x098d, 0x098f, 0x06c1,
    0x2493, 0x0845, 0x2309, 0x230b, 0x0581, 0x230d, 0x0585, 0x0991,
    0x0993, 0x230f, 0x0583, 0x2313, 0x0587, 0x0589, 0x2315, 0x058d,
    0x2317, 0x0999, 0x058b, 0x231f, 0x2381, 0x0591, 0x2321, 0x099b,
    0x058f, 0x2325, 0x2323, 0x02b9, 0x02ba, 0x02bb, 0x05c1, 0x05c3,
    0x05c5, 0x02bc, 0x24ad, 0x05c7, 0x05c9, 0x07c7, 0x0995, 0x0997,
    0x0701, 0x24ab, 0x2327, 0x05cb, 0x232b, 0x2329, 0x2341, 0x2343,
    0x2345, 0x2347, 0x05cd, 0x234b, 0x2349, 0x2382, 0x234d, 0x234f,
    0x2351, 0x24b9, 0x02bd, 0x05cf, 0x24bf, 0x0a0d, 0x2353, 0x02bf,
    0x24bd, 0x2383, 0x24bb, 0x0641, 0x2355, 0x0643, 0x0645, 0x2357,
    0x2359, 0x3101, 0x0c80, 0x2e00, 0x2446, 0x2444, 0x244a, 0x2448,
    0x0800, 0x0942, 0x0944, 0x0804, 0x2288, 0x2486, 0x2484, 0x248a,
    0x2488, 0x22ae, 0x2498, 0x2496, 0x249c, 0x249a, 0x2300, 0x0a06,
    0x2302, 0x0a04, 0x0946, 0x07ce, 0x07ca, 0x07c8, 0x07cc, 0x2447,
    0x2445, 0x244b, 0x2449, 0x0801, 0x0943, 0x0945, 0x0805, 0x2289,
    0x2487, 0x2485, 0x248b, 0x2489, 0x22af, 0x2499, 0x2497, 0x249d,
    0x249b, 0x2301, 0x0a07, 0x2303, 0x0a05, 0x0947, 0x07cf, 0x07cb,
    0x07c9, 0x07cd, 0x2450, 0x244e, 0x2454, 0x2452, 0x2451, 0x244f,
    0x2455, 0x2453, 0x2294, 0x2296, 0x2295, 0x2297, 0x2304, 0x2306,
    0x2305, 0x2307, 0x2318, 0x2319, 0x231a, 0x231b, 0x232c, 0x232d,
    0x232e, 0x232f, 0x2400, 0x24a2, 0x24a0, 0x24a6, 0x24a4, 0x24a8,
    0x24a3, 0x24a1, 0x24a7, 0x24a5, 0x24a9, 0x24b0, 0x24ae, 0x24b4,
    0x24b2, 0x24b6, 0x24b1, 0x24af, 0x24b5, 0x24b3, 0x24b7, 0x0882,
    0x0880, 0x0881, 0x0802, 0x0803, 0x229c, 0x229d, 0x0a0a, 0x0a0b,
    0x0883, 0x0b40, 0x2c8a, 0x0c81, 0x2c89, 0x2c88, 0x2540, 0x2541,
    0x2d00, 0x2e07, 0x0d00, 0x2640, 0x2641, 0x2e80, 0x0d01, 0x26c8,
    0x26c9, 0x2f00, 0x2f84, 0x0d02, 0x2f83, 0x2f82, 0x0d40, 0x26d8,
    0x26d9, 0x3186, 0x0d04, 0x2740, 0x2741, 0x3100, 0x3086, 0x0d06,
    0x3085, 0x3084, 0x0d41, 0x2840, 0x3200, 0x0d07, 0x284f, 0x2850,
    0x3280, 0x2c84, 0x2e03, 0x2857, 0x0d42, 0x2c81, 0x2c80, 0x24c0,
    0x24c1, 0x2c86, 0x2c83, 0x28c0, 0x0d43, 0x25c0, 0x25c1, 0x2940,
    0x0d44, 0x26c0, 0x26c1, 0x2e05, 0x2e02, 0x29c0, 0x0d45, 0x2f05,
    0x2f04, 0x0d80, 0x26d0, 0x26d1, 0x2f80, 0x2a40, 0x0d82, 0x26e0,
    0x26e1, 0x3080, 0x3081, 0x2ac0, 0x0d83, 0x3004, 0x3003, 0x0d81,
    0x27c0, 0x27c1, 0x3082, 0x2b40, 0x0d84, 0x2847, 0x2848, 0x3184,
    0x3181, 0x2f06, 0x0d08, 0x2f81, 0x3005, 0x0d46, 0x3083, 0x3182,
    0x0e00, 0x0e01, 0x0f40, 0x1180, 0x1182, 0x0f03, 0x0f00, 0x11c0,
    0x0f01, 0x1140, 0x1202, 0x1204, 0x0f81, 0x1240, 0x0fc0, 0x1242,
    0x0f80, 0x1244, 0x1284, 0x0f82, 0x1286, 0x1288, 0x128a, 0x12c0,
    0x1282, 0x1181, 0x1183, 0x1043, 0x1040, 0x11c1, 0x1041, 0x1141,
    0x1203, 0x1205, 0x10c1, 0x1241, 0x1000, 0x1243, 0x10c0, 0x1245,
    0x1285, 0x10c2, 0x1287, 0x1289, 0x128b, 0x12c1, 0x1283, 0x1080,
    0x1100, 0x1101, 0x1200, 0x1201, 0x1280, 0x1281, 0x1340, 0x1341,
    0x1343, 0x1342, 0x1344, 0x13c2, 0x1400, 0x13c0, 0x1440, 0x1480,
    0x14c0, 0x1540, 0x1541, 0x1740, 0x1700, 0x1741, 0x17c0, 0x1800,
    0x1802, 0x1801, 0x1840, 0x1880, 0x1900, 0x18c0, 0x18c1, 0x1901,
    0x1940, 0x1942, 0x1941, 0x1980, 0x19c0, 0x19c2, 0x19c1, 0x1c80,
    0x1cc0, 0x1dc0, 0x1f80, 0x2000, 0x2002, 0x2004, 0x2006, 0x2008,
    0x2040, 0x2080, 0x2082, 0x20c0, 0x20c1, 0x2100, 0x22b8, 0x22b9,
    0x2310, 0x2311, 0x231c, 0x231d, 0x244c, 0x2456, 0x244d, 0x2457,
    0x248c, 0x248d, 0x249e, 0x249f, 0x2500, 0x2502, 0x2504, 0x2bc0,
    0x2501, 0x2503, 0x2505, 0x2bc1, 0x2bc2, 0x2bc3, 0x2bc4, 0x2bc5,
    0x2bc6, 0x2bc7, 0x2580, 0x2582, 0x2584, 0x2bc8, 0x2581, 0x2583,
    0x2585, 0x2bc9, 0x2bca, 0x2bcb, 0x2bcc, 0x2bcd, 0x2bce, 0x2bcf,
    0x2600, 0x2602, 0x2601, 0x2603, 0x2680, 0x2682, 0x2681, 0x2683,
    0x26c2, 0x26c4, 0x26c6, 0x2c00, 0x26c3, 0x26c5, 0x26c7, 0x2c01,
    0x2c02, 0x2c03, 0x2c04, 0x2c05, 0x2c06, 0x2c07, 0x26ca, 0x26cc,
    0x26ce, 0x2c08, 0x26cb, 0x26cd, 0x26cf, 0x2c09, 0x2c0a, 0x2c0b,
    0x2c0c, 0x2c0d, 0x2c0e, 0x2c0f, 0x26d2, 0x26d4, 0x26d6, 0x26d3,
    0x26d5, 0x26d7, 0x26da, 0x26dc, 0x26de, 0x26db, 0x26dd, 0x26df,
    0x2700, 0x2702, 0x2701, 0x2703, 0x2780, 0x2782, 0x2781, 0x2783,
    0x2800, 0x2802, 0x2804, 0x2801, 0x2803, 0x2805, 0x2842, 0x2844,
    0x2846, 0x2849, 0x284b, 0x284d, 0x2c40, 0x284a, 0x284c, 0x284e,
    0x2c41, 0x2c42, 0x2c43, 0x2c44, 0x2c45, 0x2c46, 0x2c47, 0x2851,
    0x2853, 0x2855, 0x2c48, 0x2852, 0x2854, 0x2856, 0x2c49, 0x2c4a,
    0x2c4b, 0x2c4c, 0x2c4d, 0x2c4e, 0x2c4f, 0x2c82, 0x2e01, 0x3180,
    0x2c87, 0x2f01, 0x2f02, 0x2f03, 0x2e06, 0x3185, 0x3000, 0x3001,
    0x3002, 0x4640, 0x4641, 0x4680, 0x46c0, 0x46c2, 0x46c1, 0x4700,
    0x4740, 0x4780, 0x47c0, 0x47c2, 0x4900, 0x4940, 0x4980, 0x4982,
    0x4a00, 0x49c2, 0x4a03, 0x4a04, 0x4a40, 0x4a41, 0x4a80, 0x4a81,
    0x4ac0, 0x4ac1, 0x4bc0, 0x4bc1, 0x4b00, 0x4b01, 0x4b40, 0x4b41,
    0x4bc2, 0x4bc3, 0x4b80, 0x4b81, 0x4b82, 0x4b83, 0x4c00, 0x4c01,
    0x4c02, 0x4c03, 0x5600, 0x5440, 0x5442, 0x5444, 0x5446, 0x5448,
    0x544a, 0x544c, 0x544e, 0x5450, 0x5452, 0x5454, 0x5456, 0x5480,
    0x5482, 0x5484, 0x54c0, 0x54c1, 0x5500, 0x5501, 0x5540, 0x5541,
    0x5580, 0x5581, 0x55c0, 0x55c1, 0x5680, 0x58c0, 0x5700, 0x5702,
    0x5704, 0x5706, 0x5708, 0x570a, 0x570c, 0x570e, 0x5710, 0x5712,
    0x5714, 0x5716, 0x5740, 0x5742, 0x5744, 0x5780, 0x5781, 0x57c0,
    0x57c1, 0x5800, 0x5801, 0x5840, 0x5841, 0x5880, 0x5881, 0x5900,
    0x5901, 0x5902, 0x5903, 0x5940, 0x8ec0, 0x8f00, 0x8fc0, 0x8fc2,
    0x9000, 0x9040, 0x9041, 0x9080, 0x9081, 0x90c0, 0x90c2, 0x9100,
    0x9140, 0x9182, 0x9180, 0x9183, 0x91c1, 0x91c0, 0x91c3, 0x9200,
    0x9201, 0x9240, 0x9280, 0x9282, 0x9284, 0x9281, 0x9285, 0x9287,
    0x9286, 0x9283, 0x92c1, 0x92c0, 0x92c2,
};

typedef enum {
    UNICODE_GC_Cn,
    UNICODE_GC_Lu,
    UNICODE_GC_Ll,
    UNICODE_GC_Lt,
    UNICODE_GC_Lm,
    UNICODE_GC_Lo,
    UNICODE_GC_Mn,
    UNICODE_GC_Mc,
    UNICODE_GC_Me,
    UNICODE_GC_Nd,
    UNICODE_GC_Nl,
    UNICODE_GC_No,
    UNICODE_GC_Sm,
    UNICODE_GC_Sc,
    UNICODE_GC_Sk,
    UNICODE_GC_So,
    UNICODE_GC_Pc,
    UNICODE_GC_Pd,
    UNICODE_GC_Ps,
    UNICODE_GC_Pe,
    UNICODE_GC_Pi,
    UNICODE_GC_Pf,
    UNICODE_GC_Po,
    UNICODE_GC_Zs,
    UNICODE_GC_Zl,
    UNICODE_GC_Zp,
    UNICODE_GC_Cc,
    UNICODE_GC_Cf,
    UNICODE_GC_Cs,
    UNICODE_GC_Co,
    UNICODE_GC_LC,
    UNICODE_GC_L,
    UNICODE_GC_M,
    UNICODE_GC_N,
    UNICODE_GC_S,
    UNICODE_GC_P,
    UNICODE_GC_Z,
    UNICODE_GC_C,
    UNICODE_GC_COUNT,
} UnicodeGCEnum;

static const char unicode_gc_name_table[] =
    "Cn,Unassigned"            "\0"
    "Lu,Uppercase_Letter"      "\0"
    "Ll,Lowercase_Letter"      "\0"
    "Lt,Titlecase_Letter"      "\0"
    "Lm,Modifier_Letter"       "\0"
    "Lo,Other_Letter"          "\0"
    "Mn,Nonspacing_Mark"       "\0"
    "Mc,Spacing_Mark"          "\0"
    "Me,Enclosing_Mark"        "\0"
    "Nd,Decimal_Number,digit"  "\0"
    "Nl,Letter_Number"         "\0"
    "No,Other_Number"          "\0"
    "Sm,Math_Symbol"           "\0"
    "Sc,Currency_Symbol"       "\0"
    "Sk,Modifier_Symbol"       "\0"
    "So,Other_Symbol"          "\0"
    "Pc,Connector_Punctuation" "\0"
    "Pd,Dash_Punctuation"      "\0"
    "Ps,Open_Punctuation"      "\0"
    "Pe,Close_Punctuation"     "\0"
    "Pi,Initial_Punctuation"   "\0"
    "Pf,Final_Punctuation"     "\0"
    "Po,Other_Punctuation"     "\0"
    "Zs,Space_Separator"       "\0"
    "Zl,Line_Separator"        "\0"
    "Zp,Paragraph_Separator"   "\0"
    "Cc,Control,cntrl"         "\0"
    "Cf,Format"                "\0"
    "Cs,Surrogate"             "\0"
    "Co,Private_Use"           "\0"
    "LC,Cased_Letter"          "\0"
    "L,Letter"                 "\0"
    "M,Mark,Combining_Mark"    "\0"
    "N,Number"                 "\0"
    "S,Symbol"                 "\0"
    "P,Punctuation,punct"      "\0"
    "Z,Separator"              "\0"
    "C,Other"                  "\0"
;

static const uint8_t unicode_gc_table[4070] = {
    0xfa, 0x18, 0x17, 0x56, 0x0d, 0x56, 0x12, 0x13,
    0x16, 0x0c, 0x16, 0x11, 0x36, 0xe9, 0x02, 0x36,
    0x4c, 0x36, 0xe1, 0x12, 0x12, 0x16, 0x13, 0x0e,
    0x10, 0x0e, 0xe2, 0x12, 0x12, 0x0c, 0x13, 0x0c,
    0xfa, 0x19, 0x17, 0x16, 0x6d, 0x0f, 0x16, 0x0e,
    0x0f, 0x05, 0x14, 0x0c, 0x1b, 0x0f, 0x0e, 0x0f,
    0x0c, 0x2b, 0x0e, 0x02, 0x36, 0x0e, 0x0b, 0x05,
    0x15, 0x4b, 0x16, 0xe1, 0x0f, 0x0c, 0xc1, 0xe2,
    0x10, 0x0c, 0xe2, 0x00, 0xff, 0x30, 0x02, 0xff,
    0x08, 0x02, 0xff, 0x27, 0xbf, 0x22, 0x21, 0x02,
    0x5f, 0x5f, 0x21, 0x22, 0x61, 0x02, 0x21, 0x02,
    0x41, 0x42, 0x21, 0x02, 0x21, 0x02, 0x9f, 0x7f,
    0x02, 0x5f, 0x5f, 0x21, 0x02, 0x5f, 0x3f, 0x02,
    0x05, 0x3f, 0x22, 0x65, 0x01, 0x03, 0x02, 0x01,
    0x03, 0x02, 0x01, 0x03, 0x02, 0xff, 0x08, 0x02,
    0xff, 0x0a, 0x02, 0x01, 0x03, 0x02, 0x5f, 0x21,
    0x02, 0xff, 0x32, 0xa2, 0x21, 0x02, 0x21, 0x22,
    0x5f, 0x41, 0x02, 0xff, 0x00, 0xe2, 0x3c, 0x05,
    0xe2, 0x13, 0xe4, 0x0a, 0x6e, 0xe4, 0x04, 0xee,
    0x06, 0x84, 0xce, 0x04, 0x0e, 0x04, 0xee, 0x09,
    0xe6, 0x68, 0x7f, 0x04, 0x0e, 0x3f, 0x20, 0x04,
    0x42, 0x16, 0x01, 0x60, 0x2e, 0x01, 0x16, 0x41,
    0x00, 0x01, 0x00, 0x21, 0x02, 0xe1, 0x09, 0x00,
    0xe1, 0x01, 0xe2, 0x1b, 0x3f, 0x02, 0x41, 0x42,
    0xff, 0x10, 0x62, 0x3f, 0x0c, 0x5f, 0x3f, 0x02,
    0xe1, 0x2b, 0xe2, 0x28, 0xff, 0x1a, 0x0f, 0x86,
    0x28, 0xff, 0x2f, 0xff, 0x06, 0x02, 0xff, 0x58,
    0x00, 0xe1, 0x1e, 0x20, 0x04, 0xb6, 0xe2, 0x21,
    0x16, 0x11, 0x20, 0x2f, 0x0d, 0x00, 0xe6, 0x25,
    0x11, 0x06, 0x16, 0x26, 0x16, 0x26, 0x16, 0x06,
    0xe0, 0x00, 0xe5, 0x13, 0x60, 0x65, 0x36, 0xe0,
    0x03, 0xbb, 0x4c, 0x36, 0x0d, 0x36, 0x2f, 0xe6,
    0x03, 0x16, 0x1b, 0x56, 0xe5, 0x18, 0x04, 0xe5,
    0x02, 0xe6, 0x0d, 0xe9, 0x02, 0x76, 0x25, 0x06,
    0xe5, 0x5b, 0x16, 0x05, 0xc6, 0x1b, 0x0f, 0xa6,
    0x24, 0x26, 0x0f, 0x66, 0x25, 0xe9, 0x02, 0x45,
    0x2f, 0x05, 0xf6, 0x06, 0x00, 0x1b, 0x05, 0x06,
    0xe5, 0x16, 0xe6, 0x13, 0x20, 0xe5, 0x51, 0xe6,
    0x03, 0x05, 0xe0, 0x06, 0xe9, 0x02, 0xe5, 0x19,
    0xe6, 0x01, 0x24, 0x0f, 0x56, 0x04, 0x20, 0x06,
    0x2d, 0xe5, 0x0e, 0x66, 0x04, 0xe6, 0x01, 0x04,
    0x46, 0x04, 0x86, 0x20, 0xf6, 0x07, 0x00, 0xe5,
    0x11, 0x46, 0x20, 0x16, 0x00, 0xe5, 0x03, 0x80,
    0xe5, 0x10, 0x0e, 0xa5, 0x00, 0x3b, 0x80, 0xe6,
    0x01, 0xe5, 0x21, 0x04, 0xe6, 0x10, 0x1b, 0xe6,
    0x18, 0x07, 0xe5, 0x2e, 0x06, 0x07, 0x06, 0x05,
    0x47, 0xe6, 0x00, 0x67, 0x06, 0x27, 0x05, 0xc6,
    0xe5, 0x02, 0x26, 0x36, 0xe9, 0x02, 0x16, 0x04,
    0xe5, 0x07, 0x06, 0x27, 0x00, 0xe5, 0x00, 0x20,
    0x25, 0x20, 0xe5, 0x0e, 0x00, 0xc5, 0x00, 0x05,
    0x40, 0x65, 0x20, 0x06, 0x05, 0x47, 0x66, 0x20,
    0x27, 0x20, 0x27, 0x06, 0x05, 0xe0, 0x00, 0x07,
    0x60, 0x25, 0x00, 0x45, 0x26, 0x20, 0xe9, 0x02,
    0x25, 0x2d, 0xab, 0x0f, 0x0d, 0x05, 0x16, 0x06,
    0x20, 0x26, 0x07, 0x00, 0xa5, 0x60, 0x25, 0x20,
    0xe5, 0x0e, 0x00, 0xc5, 0x00, 0x25, 0x00, 0x25,
    0x00, 0x25, 0x20, 0x06, 0x00, 0x47, 0x26, 0x60,
    0x26, 0x20, 0x46, 0x40, 0x06, 0xc0, 0x65, 0x00,
    0x05, 0xc0, 0xe9, 0x02, 0x26, 0x45, 0x06, 0x16,
    0xe0, 0x02, 0x26, 0x07, 0x00, 0xe5, 0x01, 0x00,
    0x45, 0x00, 0xe5, 0x0e, 0x00, 0xc5, 0x00, 0x25,
    0x00, 0x85, 0x20, 0x06, 0x05, 0x47, 0x86, 0x00,
    0x26, 0x07, 0x00, 0x27, 0x06, 0x20, 0x05, 0xe0,
    0x07, 0x25, 0x26, 0x20, 0xe9, 0x02, 0x16, 0x0d,
    0xc0, 0x05, 0xa6, 0x00, 0x06, 0x27, 0x00, 0xe5,
    0x00, 0x20, 0x25, 0x20, 0xe5, 0x0e, 0x00, 0xc5,
    0x00, 0x25, 0x00, 0x85, 0x20, 0x06, 0x05, 0x07,
    0x06, 0x07, 0x66, 0x20, 0x27, 0x20, 0x27, 0x06,
    0xc0, 0x26, 0x07, 0x60, 0x25, 0x00, 0x45, 0x26,
    0x20, 0xe9, 0x02, 0x0f, 0x05, 0xab, 0xe0, 0x02,
    0x06, 0x05, 0x00, 0xa5, 0x40, 0x45, 0x00, 0x65,
    0x40, 0x25, 0x00, 0x05, 0x00, 0x25, 0x40, 0x25,
    0x40, 0x45, 0x40, 0xe5, 0x04, 0x60, 0x27, 0x06,
    0x27, 0x40, 0x47, 0x00, 0x47, 0x06, 0x20, 0x05,
    0xa0, 0x07, 0xe0, 0x06, 0xe9, 0x02, 0x4b, 0xaf,
    0x0d, 0x0f, 0x80, 0x06, 0x47, 0x06, 0xe5, 0x00,
    0x00, 0x45, 0x00, 0xe5, 0x0f, 0x00, 0xe5, 0x08,
    0x20, 0x06, 0x05, 0x46, 0x67, 0x00, 0x46, 0x00,
    0x66, 0xc0, 0x26, 0x00, 0x45, 0x20, 0x05, 0x20,
    0x25, 0x26, 0x20, 0xe9, 0x02, 0xc0, 0x16, 0xcb,
    0x0f, 0x05, 0x06, 0x27, 0x16, 0xe5, 0x00, 0x00,
    0x45, 0x00, 0xe5, 0x0f, 0x00, 0xe5, 0x02, 0x00,
    0x85, 0x20, 0x06, 0x05, 0x07, 0x06, 0x87, 0x00,
    0x06, 0x27, 0x00, 0x27, 0x26, 0xc0, 0x27, 0xa0,
    0x25, 0x00, 0x25, 0x26, 0x20, 0xe9, 0x02, 0x00,
    0x25, 0x07, 0xe0, 0x04, 0x26, 0x27, 0xe5, 0x01,
    0x00, 0x45, 0x00, 0xe5, 0x21, 0x26, 0x05, 0x47,
    0x66, 0x00, 0x47, 0x00, 0x47, 0x06, 0x05, 0x0f,
    0x60, 0x45, 0x07, 0xcb, 0x45, 0x26, 0x20, 0xe9,
    0x02, 0xeb, 0x01, 0x0f, 0xa5, 0x00, 0x06, 0x27,
    0x00, 0xe5, 0x0a, 0x40, 0xe5, 0x10, 0x00, 0xe5,
    0x01, 0x00, 0x05, 0x20, 0xc5, 0x40, 0x06, 0x60,
    0x47, 0x46, 0x00, 0x06, 0x00, 0xe7, 0x00, 0xa0,
    0xe9, 0x02, 0x20, 0x27, 0x16, 0xe0, 0x04, 0xe5,
    0x28, 0x06, 0x25, 0xc6, 0x60, 0x0d, 0xa5, 0x04,
    0xe6, 0x00, 0x16, 0xe9, 0x02, 0x36, 0xe0, 0x1d,
    0x25, 0x00, 0x05, 0x00, 0x85, 0x00, 0xe5, 0x10,
    0x00, 0x05, 0x00, 0xe5, 0x02, 0x06, 0x25, 0xe6,
    0x01, 0x05, 0x20, 0x85, 0x00, 0x04, 0x00, 0xc6,
    0x00, 0xe9, 0x02, 0x20, 0x65, 0xe0, 0x18, 0x05,
    0x4f, 0xf6, 0x07, 0x0f, 0x16, 0x4f, 0x26, 0xaf,
    0xe9, 0x02, 0xeb, 0x02, 0x0f, 0x06, 0x0f, 0x06,
    0x0f, 0x06, 0x12, 0x13, 0x12, 0x13, 0x27, 0xe5,
    0x00, 0x00, 0xe5, 0x1c, 0x60, 0xe6, 0x06, 0x07,
    0x86, 0x16, 0x26, 0x85, 0xe6, 0x03, 0x00, 0xe6,
    0x1c, 0x00, 0xef, 0x00, 0x06, 0xaf, 0x00, 0x2f,
    0x96, 0x6f, 0x36, 0xe0, 0x1d, 0xe5, 0x23, 0x27,
    0x66, 0x07, 0xa6, 0x07, 0x26, 0x27, 0x26, 0x05,
    0xe9, 0x02, 0xb6, 0xa5, 0x27, 0x26, 0x65, 0x46,
    0x05, 0x47, 0x25, 0xc7, 0x45, 0x66, 0xe5, 0x05,
    0x06, 0x27, 0x26, 0xa7, 0x06, 0x05, 0x07, 0xe9,
    0x02, 0x47, 0x06, 0x2f, 0xe1, 0x1e, 0x00, 0x01,
    0x80, 0x01, 0x20, 0xe2, 0x23, 0x16, 0x04, 0x42,
    0xe5, 0x80, 0xc1, 0x00, 0x65, 0x20, 0xc5, 0x00,
    0x05, 0x00, 0x65, 0x20, 0xe5, 0x21, 0x00, 0x65,
    0x20, 0xe5, 0x19, 0x00, 0x65, 0x20, 0xc5, 0x00,
    0x05, 0x00, 0x65, 0x20, 0xe5, 0x07, 0x00, 0xe5,
    0x31, 0x00, 0x65, 0x20, 0xe5, 0x3b, 0x20, 0x46,
    0xf6, 0x01, 0xeb, 0x0c, 0x40, 0xe5, 0x08, 0xef,
    0x02, 0xa0, 0xe1, 0x4e, 0x20, 0xa2, 0x20, 0x11,
    0xe5, 0x81, 0xe4, 0x0f, 0x16, 0xe5, 0x09, 0x17,
    0xe5, 0x12, 0x12, 0x13, 0x40, 0xe5, 0x43, 0x56,
    0x4a, 0xe5, 0x00, 0xc0, 0xe5, 0x0a, 0x46, 0x07,
    0xe0, 0x01, 0xe5, 0x0b, 0x26, 0x07, 0x36, 0xe0,
    0x01, 0xe5, 0x0a, 0x26, 0xe0, 0x04, 0xe5, 0x05,
    0x00, 0x45, 0x00, 0x26, 0xe0, 0x04, 0xe5, 0x2c,
    0x26, 0x07, 0xc6, 0xe7, 0x00, 0x06, 0x27, 0xe6,
    0x03, 0x56, 0x04, 0x56, 0x0d, 0x05, 0x06, 0x20,
    0xe9, 0x02, 0xa0, 0xeb, 0x02, 0xa0, 0xb6, 0x11,
    0x76, 0x46, 0x1b, 0x06, 0xe9, 0x02, 0xa0, 0xe5,
    0x1b, 0x04, 0xe5, 0x2d, 0xc0, 0x85, 0x26, 0xe5,
    0x1a, 0x06, 0x05, 0x80, 0xe5, 0x3e, 0xe0, 0x02,
    0xe5, 0x17, 0x00, 0x46, 0x67, 0x26, 0x47, 0x60,
    0x27, 0x06, 0xa7, 0x46, 0x60, 0x0f, 0x40, 0x36,
    0xe9, 0x02, 0xe5, 0x16, 0x20, 0x85, 0xe0, 0x03,
    0xe5, 0x24, 0x60, 0xe5, 0x12, 0xa0, 0xe9, 0x02,
    0x0b, 0x40, 0xef, 0x1a, 0xe5, 0x0f, 0x26, 0x27,
    0x06, 0x20, 0x36, 0xe5, 0x2d, 0x07, 0x06, 0x07,
    0xc6, 0x00, 0x06, 0x07, 0x06, 0x27, 0xe6, 0x00,
    0xa7, 0xe6, 0x02, 0x20, 0x06, 0xe9, 0x02, 0xa0,
    0xe9, 0x02, 0xa0, 0xd6, 0x04, 0xb6, 0x20, 0xe6,
    0x06, 0x08, 0xe6, 0x08, 0xe0, 0x29, 0x66, 0x07,
    0xe5, 0x27, 0x06, 0x07, 0x86, 0x07, 0x06, 0x87,
    0x06, 0x27, 0xe5, 0x00, 0x00, 0x36, 0xe9, 0x02,
    0xd6, 0xef, 0x02, 0xe6, 0x01, 0xef, 0x01, 0x56,
    0x26, 0x07, 0xe5, 0x16, 0x07, 0x66, 0x27, 0x26,
    0x07, 0x46, 0x25, 0xe9, 0x02, 0xe5, 0x24, 0x06,
    0x07, 0x26, 0x47, 0x06, 0x07, 0x46, 0x27, 0xe0,
    0x00, 0x76, 0xe5, 0x1c, 0xe7, 0x00, 0xe6, 0x00,
    0x27, 0x26, 0x40, 0x96, 0xe9, 0x02, 0x40, 0x45,
    0xe9, 0x02, 0xe5, 0x16, 0xa4, 0x36, 0xe2, 0x01,
    0x3f, 0x80, 0xe1, 0x23, 0x20, 0x41, 0xf6, 0x00,
    0xe0, 0x00, 0x46, 0x16, 0xe6, 0x05, 0x07, 0xc6,
    0x65, 0x06, 0xa5, 0x06, 0x25, 0x07, 0x26, 0x05,
    0x80, 0xe2, 0x24, 0xe4, 0x37, 0xe2, 0x05, 0x04,
    0xe2, 0x1a, 0xe4, 0x1d, 0xe6, 0x38, 0xff, 0x80,
    0x0e, 0xe2, 0x00, 0xff, 0x5a, 0xe2, 0x00, 0xe1,
    0x00, 0xa2, 0x20, 0xa1, 0x20, 0xe2, 0x00, 0xe1,
    0x00, 0xe2, 0x00, 0xe1, 0x00, 0xa2, 0x20, 0xa1,
    0x20, 0xe2, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00,
    0x01, 0x00, 0x3f, 0xc2, 0xe1, 0x00, 0xe2, 0x06,
    0x20, 0xe2, 0x00, 0xe3, 0x00, 0xe2, 0x00, 0xe3,
    0x00, 0xe2, 0x00, 0xe3, 0x00, 0x82, 0x00, 0x22,
    0x61, 0x03, 0x0e, 0x02, 0x4e, 0x42, 0x00, 0x22,
    0x61, 0x03, 0x4e, 0x62, 0x20, 0x22, 0x61, 0x00,
    0x4e, 0xe2, 0x00, 0x81, 0x4e, 0x20, 0x42, 0x00,
    0x22, 0x61, 0x03, 0x2e, 0x00, 0xf7, 0x03, 0x9b,
    0xb1, 0x36, 0x14, 0x15, 0x12, 0x34, 0x15, 0x12,
    0x14, 0xf6, 0x00, 0x18, 0x19, 0x9b, 0x17, 0xf6,
    0x01, 0x14, 0x15, 0x76, 0x30, 0x56, 0x0c, 0x12,
    0x13, 0xf6, 0x03, 0x0c, 0x16, 0x10, 0xf6, 0x02,
    0x17, 0x9b, 0x00, 0xfb, 0x02, 0x0b, 0x04, 0x20,
    0xab, 0x4c, 0x12, 0x13, 0x04, 0xeb, 0x02, 0x4c,
    0x12, 0x13, 0x00, 0xe4, 0x05, 0x40, 0xed, 0x19,
    0xe0, 0x07, 0xe6, 0x05, 0x68, 0x06, 0x48, 0xe6,
    0x04, 0xe0, 0x07, 0x2f, 0x01, 0x6f, 0x01, 0x2f,
    0x02, 0x41, 0x22, 0x41, 0x02, 0x0f, 0x01, 0x2f,
    0x0c, 0x81, 0xaf, 0x01, 0x0f, 0x01, 0x0f, 0x01,
    0x0f, 0x61, 0x0f, 0x02, 0x61, 0x02, 0x65, 0x02,
    0x2f, 0x22, 0x21, 0x8c, 0x3f, 0x42, 0x0f, 0x0c,
    0x2f, 0x02, 0x0f, 0xeb, 0x08, 0xea, 0x1b, 0x3f,
    0x6a, 0x0b, 0x2f, 0x60, 0x8c, 0x8f, 0x2c, 0x6f,
    0x0c, 0x2f, 0x0c, 0x2f, 0x0c, 0xcf, 0x0c, 0xef,
    0x17, 0x2c, 0x2f, 0x0c, 0x0f, 0x0c, 0xef, 0x17,
    0xec, 0x80, 0x84, 0xef, 0x00, 0x12, 0x13, 0x12,
    0x13, 0xef, 0x0c, 0x2c, 0xcf, 0x12, 0x13, 0xef,
    0x49, 0x0c, 0xef, 0x16, 0xec, 0x11, 0xef, 0x20,
    0xac, 0xef, 0x40, 0xe0, 0x0e, 0xef, 0x03, 0xe0,
    0x0d, 0xeb, 0x34, 0xef, 0x46, 0xeb, 0x0e, 0xef,
    0x80, 0x2f, 0x0c, 0xef, 0x01, 0x0c, 0xef, 0x2e,
    0xec, 0x00, 0xef, 0x67, 0x0c, 0xef, 0x80, 0x70,
    0x12, 0x13, 0x12, 0x13, 0x12, 0x13, 0x12, 0x13,
    0x12, 0x13, 0x12, 0x13, 0x12, 0x13, 0xeb, 0x16,
    0xef, 0x24, 0x8c, 0x12, 0x13, 0xec, 0x17, 0x12,
    0x13, 0x12, 0x13, 0x12, 0x13, 0x12, 0x13, 0x12,
    0x13, 0xec, 0x08, 0xef, 0x80, 0x78, 0xec, 0x7b,
    0x12, 0x13, 0x12, 0x13, 0x12, 0x13, 0x12, 0x13,
    0x12, 0x13, 0x12, 0x13, 0x12, 0x13, 0x12, 0x13,
    0x12, 0x13, 0x12, 0x13, 0x12, 0x13, 0xec, 0x37,
    0x12, 0x13, 0x12, 0x13, 0xec, 0x18, 0x12, 0x13,
    0xec, 0x80, 0x7a, 0xef, 0x28, 0xec, 0x0d, 0x2f,
    0xac, 0xef, 0x1f, 0x20, 0xef, 0x18, 0x00, 0xef,
    0x61, 0xe1, 0x28, 0xe2, 0x28, 0x5f, 0x21, 0x22,
    0xdf, 0x41, 0x02, 0x3f, 0x02, 0x3f, 0x82, 0x24,
    0x41, 0x02, 0xff, 0x5a, 0x02, 0xaf, 0x7f, 0x46,
    0x3f, 0x80, 0x76, 0x0b, 0x36, 0xe2, 0x1e, 0x00,
    0x02, 0x80, 0x02, 0x20, 0xe5, 0x30, 0xc0, 0x04,
    0x16, 0xe0, 0x06, 0x06, 0xe5, 0x0f, 0xe0, 0x01,
    0xc5, 0x00, 0xc5, 0x00, 0xc5, 0x00, 0xc5, 0x00,
    0xc5, 0x00, 0xc5, 0x00, 0xc5, 0x00, 0xc5, 0x00,
    0xe6, 0x18, 0x36, 0x14, 0x15, 0x14, 0x15, 0x56,
    0x14, 0x15, 0x16, 0x14, 0x15, 0xf6, 0x01, 0x11,
    0x36, 0x11, 0x16, 0x14, 0x15, 0x36, 0x14, 0x15,
    0x12, 0x13, 0x12, 0x13, 0x12, 0x13, 0x12, 0x13,
    0x96, 0x04, 0xf6, 0x02, 0x31, 0x76, 0x11, 0x16,
    0x12, 0xf6, 0x05, 0x2f, 0x56, 0x12, 0x13, 0x12,
    0x13, 0x12, 0x13, 0x12, 0x13, 0x11, 0xe0, 0x1a,
    0xef, 0x12, 0x00, 0xef, 0x51, 0xe0, 0x04, 0xef,
    0x80, 0x4e, 0xe0, 0x12, 0xef, 0x08, 0x17, 0x56,
    0x0f, 0x04, 0x05, 0x0a, 0x12, 0x13, 0x12, 0x13,
    0x12, 0x13, 0x12, 0x13, 0x12, 0x13, 0x2f, 0x12,
    0x13, 0x12, 0x13, 0x12, 0x13, 0x12, 0x13, 0x11,
    0x12, 0x33, 0x0f, 0xea, 0x01, 0x66, 0x27, 0x11,
    0x84, 0x2f, 0x4a, 0x04, 0x05, 0x16, 0x2f, 0x00,
    0xe5, 0x4e, 0x20, 0x26, 0x2e, 0x24, 0x05, 0x11,
    0xe5, 0x52, 0x16, 0x44, 0x05, 0x80, 0xe5, 0x23,
    0x00, 0xe5, 0x56, 0x00, 0x2f, 0x6b, 0xef, 0x02,
    0xe5, 0x18, 0xef, 0x1e, 0xe0, 0x01, 0x0f, 0xe5,
    0x08, 0xef, 0x17, 0x00, 0xeb, 0x02, 0xef, 0x16,
    0xeb, 0x00, 0x0f, 0xeb, 0x07, 0xef, 0x18, 0xeb,
    0x02, 0xef, 0x1f, 0xeb, 0x07, 0xef, 0x80, 0xb8,
    0xe5, 0x99, 0x38, 0xef, 0x38, 0xe5, 0xc0, 0x11,
    0x8d, 0x04, 0xe5, 0x83, 0xef, 0x40, 0xef, 0x2f,
    0xe0, 0x01, 0xe5, 0x20, 0xa4, 0x36, 0xe5, 0x80,
    0x84, 0x04, 0x56, 0xe5, 0x08, 0xe9, 0x02, 0x25,
    0xe0, 0x0c, 0xff, 0x26, 0x05, 0x06, 0x48, 0x16,
    0xe6, 0x02, 0x16, 0x04, 0xff, 0x14, 0x24, 0x26,
    0xe5, 0x3e, 0xea, 0x02, 0x26, 0xb6, 0xe0, 0x00,
    0xee, 0x0f, 0xe4, 0x01, 0x2e, 0xff, 0x06, 0x22,
    0xff, 0x36, 0x04, 0xe2, 0x00, 0x9f, 0xff, 0x02,
    0x04, 0x2e, 0x7f, 0x05, 0x7f, 0x22, 0xff, 0x0d,
    0x61, 0x02, 0x81, 0x02, 0xff, 0x07, 0x41, 0x02,
    0x5f, 0x3f, 0x20, 0x3f, 0x00, 0x02, 0x00, 0x02,
    0xdf, 0xe0, 0x0d, 0x44, 0x3f, 0x05, 0x24, 0x02,
    0xc5, 0x06, 0x45, 0x06, 0x65, 0x06, 0xe5, 0x0f,
    0x27, 0x26, 0x07, 0x6f, 0x06, 0x40, 0xab, 0x2f,
    0x0d, 0x0f, 0xa0, 0xe5, 0x2c, 0x76, 0xe0, 0x00,
    0x27, 0xe5, 0x2a, 0xe7, 0x08, 0x26, 0xe0, 0x00,
    0x36, 0xe9, 0x02, 0xa0, 0xe6, 0x0a, 0xa5, 0x56,
    0x05, 0x16, 0x25, 0x06, 0xe9, 0x02, 0xe5, 0x14,
    0xe6, 0x00, 0x36, 0xe5, 0x0f, 0xe6, 0x03, 0x27,
    0xe0, 0x03, 0x16, 0xe5, 0x15, 0x40, 0x46, 0x07,
    0xe5, 0x27, 0x06, 0x27, 0x66, 0x27, 0x26, 0x47,
    0xf6, 0x05, 0x00, 0x04, 0xe9, 0x02, 0x60, 0x36,
    0x85, 0x06, 0x04, 0xe5, 0x01, 0xe9, 0x02, 0x85,
    0x00, 0xe5, 0x21, 0xa6, 0x27, 0x26, 0x27, 0x26,
    0xe0, 0x01, 0x45, 0x06, 0xe5, 0x00, 0x06, 0x07,
    0x20, 0xe9, 0x02, 0x20, 0x76, 0xe5, 0x08, 0x04,
    0xa5, 0x4f, 0x05, 0x07, 0x06, 0x07, 0xe5, 0x2a,
    0x06, 0x05, 0x46, 0x25, 0x26, 0x85, 0x26, 0x05,
    0x06, 0x05, 0xe0, 0x10, 0x25, 0x04, 0x36, 0xe5,
    0x03, 0x07, 0x26, 0x27, 0x36, 0x05, 0x24, 0x07,
    0x06, 0xe0, 0x02, 0xa5, 0x20, 0xa5, 0x20, 0xa5,
    0xe0, 0x01, 0xc5, 0x00, 0xc5, 0x00, 0xe2, 0x23,
    0x0e, 0x64, 0xe2, 0x01, 0x04, 0x2e, 0x60, 0xe2,
    0x48, 0xe5, 0x1b, 0x27, 0x06, 0x27, 0x06, 0x27,
    0x16, 0x07, 0x06, 0x20, 0xe9, 0x02, 0xa0, 0xe5,
    0xab, 0x1c, 0xe0, 0x04, 0xe5, 0x0f, 0x60, 0xe5,
    0x29, 0x60, 0xfc, 0x87, 0x78, 0xfd, 0x98, 0x78,
    0xe5, 0x80, 0xe6, 0x20, 0xe5, 0x62, 0xe0, 0x1e,
    0xc2, 0xe0, 0x04, 0x82, 0x80, 0x05, 0x06, 0xe5,
    0x02, 0x0c, 0xe5, 0x05, 0x00, 0x85, 0x00, 0x05,
    0x00, 0x25, 0x00, 0x25, 0x00, 0xe5, 0x64, 0xee,
    0x09, 0xe0, 0x08, 0xe5, 0x80, 0xe3, 0x13, 0x12,
    0xef, 0x08, 0xe5, 0x38, 0x20, 0xe5, 0x2e, 0xc0,
    0x0f, 0xe0, 0x18, 0xe5, 0x04, 0x0d, 0x4f, 0xe6,
    0x08, 0xd6, 0x12, 0x13, 0x16, 0xa0, 0xe6, 0x08,
    0x16, 0x31, 0x30, 0x12, 0x13, 0x12, 0x13, 0x12,
    0x13, 0x12, 0x13, 0x12, 0x13, 0x12, 0x13, 0x12,
    0x13, 0x12, 0x13, 0x36, 0x12, 0x13, 0x76, 0x50,
    0x56, 0x00, 0x76, 0x11, 0x12, 0x13, 0x12, 0x13,
    0x12, 0x13, 0x56, 0x0c, 0x11, 0x4c, 0x00, 0x16,
    0x0d, 0x36, 0x60, 0x85, 0x00, 0xe5, 0x7f, 0x20,
    0x1b, 0x00, 0x56, 0x0d, 0x56, 0x12, 0x13, 0x16,
    0x0c, 0x16, 0x11, 0x36, 0xe9, 0x02, 0x36, 0x4c,
    0x36, 0xe1, 0x12, 0x12, 0x16, 0x13, 0x0e, 0x10,
    0x0e, 0xe2, 0x12, 0x12, 0x0c, 0x13, 0x0c, 0x12,
    0x13, 0x16, 0x12, 0x13, 0x36, 0xe5, 0x02, 0x04,
    0xe5, 0x25, 0x24, 0xe5, 0x17, 0x40, 0xa5, 0x20,
    0xa5, 0x20, 0xa5, 0x20, 0x45, 0x40, 0x2d, 0x0c,
    0x0e, 0x0f, 0x2d, 0x00, 0x0f, 0x6c, 0x2f, 0xe0,
    0x02, 0x5b, 0x2f, 0x20, 0xe5, 0x04, 0x00, 0xe5,
    0x12, 0x00, 0xe5, 0x0b, 0x00, 0x25, 0x00, 0xe5,
    0x07, 0x20, 0xe5, 0x06, 0xe0, 0x1a, 0xe5, 0x73,
    0x80, 0x56, 0x60, 0xeb, 0x25, 0x40, 0xef, 0x01,
    0xea, 0x2d, 0x6b, 0xef, 0x09, 0x2b, 0x4f, 0x00,
    0xef, 0x05, 0x40, 0x0f, 0xe0, 0x27, 0xef, 0x25,
    0x06, 0xe0, 0x7a, 0xe5, 0x15, 0x40, 0xe5, 0x29,
    0xe0, 0x07, 0x06, 0xeb, 0x13, 0x60, 0xe5, 0x18,
    0x6b, 0xe0, 0x01, 0xe5, 0x0c, 0x0a, 0xe5, 0x00,
    0x0a, 0x80, 0xe5, 0x1e, 0x86, 0x80, 0xe5, 0x16,
    0x00, 0x16, 0xe5, 0x1c, 0x60, 0xe5, 0x00, 0x16,
    0x8a, 0xe0, 0x22, 0xe1, 0x20, 0xe2, 0x20, 0xe5,
    0x46, 0x20, 0xe9, 0x02, 0xa0, 0xe1, 0x1c, 0x60,
    0xe2, 0x1c, 0x60, 0xe5, 0x20, 0xe0, 0x00, 0xe5,
    0x2c, 0xe0, 0x03, 0x16, 0xe1, 0x03, 0x00, 0xe1,
    0x07, 0x00, 0xc1, 0x00, 0x21, 0x00, 0xe2, 0x03,
    0x00, 0xe2, 0x07, 0x00, 0xc2, 0x00, 0x22, 0x40,
    0xe5, 0x2c, 0xe0, 0x04, 0xe5, 0x80, 0xaf, 0xe0,
    0x01, 0xe5, 0x0e, 0xe0, 0x02, 0xe5, 0x00, 0xe0,
    0x10, 0xa4, 0x00, 0xe4, 0x22, 0x00, 0xe4, 0x01,
    0xe0, 0x3d, 0xa5, 0x20, 0x05, 0x00, 0xe5, 0x24,
    0x00, 0x25, 0x40, 0x05, 0x20, 0xe5, 0x0f, 0x00,
    0x16, 0xeb, 0x00, 0xe5, 0x0f, 0x2f, 0xcb, 0xe5,
    0x17, 0xe0, 0x00, 0xeb, 0x01, 0xe0, 0x28, 0xe5,
    0x0b, 0x00, 0x25, 0x80, 0x8b, 0xe5, 0x0e, 0xab,
    0x40, 0x16, 0xe5, 0x12, 0x80, 0x16, 0xe0, 0x38,
    0xe5, 0x30, 0x60, 0x2b, 0x25, 0xeb, 0x08, 0x20,
    0xeb, 0x26, 0x05, 0x46, 0x00, 0x26, 0x80, 0x66,
    0x65, 0x00, 0x45, 0x00, 0xe5, 0x15, 0x20, 0x46,
    0x60, 0x06, 0xeb, 0x01, 0xc0, 0xf6, 0x01, 0xc0,
    0xe5, 0x15, 0x2b, 0x16, 0xe5, 0x15, 0x4b, 0xe0,
    0x18, 0xe5, 0x00, 0x0f, 0xe5, 0x14, 0x26, 0x60,
    0x8b, 0xd6, 0xe0, 0x01, 0xe5, 0x2e, 0x40, 0xd6,
    0xe5, 0x0e, 0x20, 0xeb, 0x00, 0xe5, 0x0b, 0x80,
    0xeb, 0x00, 0xe5, 0x0a, 0xc0, 0x76, 0xe0, 0x04,
    0xcb, 0xe0, 0x48, 0xe5, 0x41, 0xe0, 0x2f, 0xe1,
    0x2b, 0xe0, 0x05, 0xe2, 0x2b, 0xc0, 0xab, 0xe5,
    0x1c, 0x66, 0xe0, 0x00, 0xe9, 0x02, 0xa0, 0xe9,
    0x02, 0x65, 0x04, 0x05, 0xe1, 0x0e, 0x40, 0x86,
    0x11, 0x04, 0xe2, 0x0e, 0xe0, 0x00, 0x2c, 0xe0,
    0x80, 0x48, 0xeb, 0x17, 0x00, 0xe5, 0x22, 0x00,
    0x26, 0x11, 0x20, 0x25, 0xe0, 0x08, 0x45, 0xe0,
    0x2f, 0x66, 0xe5, 0x15, 0xeb, 0x02, 0x05, 0xe0,
    0x00, 0xe5, 0x0e, 0xe6, 0x03, 0x6b, 0x96, 0xe0,
    0x0e, 0xe5, 0x0a, 0x66, 0x76, 0xe0, 0x1e, 0xe5,
    0x0d, 0xcb, 0xe0, 0x0c, 0xe5, 0x0f, 0xe0, 0x01,
    0x07, 0x06, 0x07, 0xe5, 0x2d, 0xe6, 0x07, 0xd6,
    0x60, 0xeb, 0x0c, 0xe9, 0x02, 0x06, 0x25, 0x26,
    0x05, 0xe0, 0x01, 0x46, 0x07, 0xe5, 0x25, 0x47,
    0x66, 0x27, 0x26, 0x36, 0x1b, 0x76, 0x06, 0xe0,
    0x02, 0x1b, 0x20, 0xe5, 0x11, 0xc0, 0xe9, 0x02,
    0xa0, 0x46, 0xe5, 0x1c, 0x86, 0x07, 0xe6, 0x00,
    0x00, 0xe9, 0x02, 0x76, 0x05, 0x27, 0x05, 0xe0,
    0x00, 0xe5, 0x1b, 0x06, 0x36, 0x05, 0xe0, 0x01,
    0x26, 0x07, 0xe5, 0x28, 0x47, 0xe6, 0x01, 0x27,
    0x65, 0x76, 0x66, 0x16, 0x07, 0x06, 0xe9, 0x02,
    0x05, 0x16, 0x05, 0x56, 0x00, 0xeb, 0x0c, 0xe0,
    0x03, 0xe5, 0x0a, 0x00, 0xe5, 0x11, 0x47, 0x46,
    0x27, 0x06, 0x07, 0x26, 0xb6, 0x06, 0x25, 0x06,
    0xe0, 0x36, 0xc5, 0x00, 0x05, 0x00, 0x65, 0x00,
    0xe5, 0x07, 0x00, 0xe5, 0x02, 0x16, 0xa0, 0xe5,
    0x27, 0x06, 0x47, 0xe6, 0x00, 0x80, 0xe9, 0x02,
    0xa0, 0x26, 0x27, 0x00, 0xe5, 0x00, 0x20, 0x25,
    0x20, 0xe5, 0x0e, 0x00, 0xc5, 0x00, 0x25, 0x00,
    0x85, 0x00, 0x26, 0x05, 0x27, 0x06, 0x67, 0x20,
    0x27, 0x20, 0x47, 0x20, 0x05, 0xa0, 0x07, 0x80,
    0x85, 0x27, 0x20, 0xc6, 0x40, 0x86, 0xe0, 0x03,
    0xe5, 0x02, 0x00, 0x05, 0x20, 0x05, 0x00, 0xe5,
    0x1e, 0x00, 0x05, 0x47, 0xa6, 0x00, 0x07, 0x20,
    0x07, 0x00, 0x67, 0x00, 0x27, 0x06, 0x07, 0x06,
    0x05, 0x06, 0x05, 0x36, 0x00, 0x36, 0xe0, 0x00,
    0x26, 0xe0, 0x15, 0xe5, 0x2d, 0x47, 0xe6, 0x00,
    0x27, 0x46, 0x07, 0x06, 0x65, 0x96, 0xe9, 0x02,
    0x36, 0x00, 0x16, 0x06, 0x45, 0xe0, 0x16, 0xe5,
    0x28, 0x47, 0xa6, 0x07, 0x06, 0x67, 0x26, 0x07,
    0x26, 0x25, 0x16, 0x05, 0xe0, 0x00, 0xe9, 0x02,
    0xe0, 0x80, 0x1e, 0xe5, 0x27, 0x47, 0x66, 0x20,
    0x67, 0x26, 0x07, 0x26, 0xf6, 0x0f, 0x65, 0x26,
    0xe0, 0x1a, 0xe5, 0x28, 0x47, 0xe6, 0x00, 0x27,
    0x06, 0x07, 0x26, 0x56, 0x05, 0xe0, 0x03, 0xe9,
    0x02, 0xa0, 0xf6, 0x05, 0xe0, 0x0b, 0xe5, 0x23,
    0x06, 0x07, 0x06, 0x27, 0xa6, 0x07, 0x06, 0x05,
    0x16, 0xa0, 0xe9, 0x02, 0xa0, 0xe9, 0x0c, 0xe0,
    0x14, 0xe5, 0x13, 0x20, 0x06, 0x07, 0x06, 0x27,
    0x66, 0x07, 0x86, 0x60, 0xe9, 0x02, 0x2b, 0x56,
    0x0f, 0xc5, 0xe0, 0x80, 0x31, 0xe5, 0x24, 0x47,
    0xe6, 0x01, 0x07, 0x26, 0x16, 0xe0, 0x5c, 0xe1,
    0x18, 0xe2, 0x18, 0xe9, 0x02, 0xeb, 0x01, 0xe0,
    0x04, 0xe5, 0x00, 0x20, 0x05, 0x20, 0xe5, 0x00,
    0x00, 0x25, 0x00, 0xe5, 0x10, 0xa7, 0x00, 0x27,
    0x20, 0x26, 0x07, 0x06, 0x05, 0x07, 0x05, 0x07,
    0x06, 0x56, 0xe0, 0x01, 0xe9, 0x02, 0xe0, 0x3e,
    0xe5, 0x00, 0x20, 0xe5, 0x1f, 0x47, 0x66, 0x20,
    0x26, 0x67, 0x06, 0x05, 0x16, 0x05, 0x07, 0xe0,
    0x13, 0x05, 0xe6, 0x02, 0xe5, 0x20, 0xa6, 0x07,
    0x05, 0x66, 0xf6, 0x00, 0x06, 0xe0, 0x00, 0x05,
    0xa6, 0x27, 0x46, 0xe5, 0x26, 0xe6, 0x05, 0x07,
    0x26, 0x56, 0x05, 0x96, 0xe0, 0x05, 0xe5, 0x41,
    0xc0, 0xf6, 0x02, 0xe0, 0x80, 0x2e, 0xe5, 0x19,
    0x16, 0xe0, 0x06, 0xe9, 0x02, 0xa0, 0xe5, 0x01,
    0x00, 0xe5, 0x1d, 0x07, 0xc6, 0x00, 0xa6, 0x07,
    0x06, 0x05, 0x96, 0xe0, 0x02, 0xe9, 0x02, 0xeb,
    0x0b, 0x40, 0x36, 0xe5, 0x16, 0x20, 0xe6, 0x0e,
    0x00, 0x07, 0xc6, 0x07, 0x26, 0x07, 0x26, 0xe0,
    0x41, 0xc5, 0x00, 0x25, 0x00, 0xe5, 0x1e, 0xa6,
    0x40, 0x06, 0x00, 0x26, 0x00, 0xc6, 0x05, 0x06,
    0xe0, 0x00, 0xe9, 0x02, 0xa0, 0xa5, 0x00, 0x25,
    0x00, 0xe5, 0x18, 0x87, 0x00, 0x26, 0x00, 0x27,
    0x06, 0x07, 0x06, 0x05, 0xc0, 0xe9, 0x02, 0xe0,
    0x80, 0xae, 0xe5, 0x0b, 0x26, 0x27, 0x36, 0xc0,
    0x26, 0x05, 0x07, 0xe5, 0x05, 0x00, 0xe5, 0x1a,
    0x27, 0x86, 0x40, 0x27, 0x06, 0x07, 0x06, 0xf6,
    0x05, 0xe9, 0x02, 0x06, 0xe0, 0x4d, 0x05, 0xe0,
    0x07, 0xeb, 0x0d, 0xef, 0x00, 0x6d, 0xef, 0x09,
    0xe0, 0x05, 0x16, 0xe5, 0x83, 0x12, 0xe0, 0x5e,
    0xea, 0x67, 0x00, 0x96, 0xe0, 0x03, 0xe5, 0x80,
    0x3c, 0xe0, 0x89, 0xc4, 0xe5, 0x59, 0x36, 0xe0,
    0x05, 0xe5, 0x83, 0xa8, 0xfb, 0x08, 0x06, 0xa5,
    0xe6, 0x07, 0xe0, 0x02, 0xe5, 0x8f, 0x13, 0x80,
    0xe5, 0x81, 0xbf, 0xe0, 0x9a, 0x31, 0xe5, 0x16,
    0xe6, 0x04, 0x47, 0x46, 0xe9, 0x02, 0xe0, 0x86,
    0x3e, 0xe5, 0x81, 0xb1, 0xc0, 0xe5, 0x17, 0x00,
    0xe9, 0x02, 0x60, 0x36, 0xe5, 0x47, 0x00, 0xe9,
    0x02, 0xa0, 0xe5, 0x16, 0x20, 0x86, 0x16, 0xe0,
    0x02, 0xe5, 0x28, 0xc6, 0x96, 0x6f, 0x64, 0x16,
    0x0f, 0xe0, 0x02, 0xe9, 0x02, 0x00, 0xcb, 0x00,
    0xe5, 0x0d, 0x80, 0xe5, 0x0b, 0xe0, 0x81, 0x28,
    0x44, 0xe5, 0x20, 0x24, 0x56, 0xe9, 0x02, 0xe0,
    0x80, 0x3e, 0xe1, 0x18, 0xe2, 0x18, 0xeb, 0x0f,
    0x76, 0xe0, 0x5d, 0xe5, 0x43, 0x60, 0x06, 0x05,
    0xe7, 0x2f, 0xc0, 0x66, 0xe4, 0x05, 0xe0, 0x38,
    0x24, 0x16, 0x04, 0x06, 0xe0, 0x03, 0x27, 0xe0,
    0x06, 0xe5, 0x97, 0x70, 0xe0, 0x00, 0xe5, 0x84,
    0x4e, 0xe0, 0x21, 0xe5, 0x02, 0xe0, 0xa2, 0x5f,
    0x64, 0x00, 0xc4, 0x00, 0x24, 0x00, 0xe5, 0x80,
    0x9b, 0xe0, 0x07, 0x05, 0xe0, 0x15, 0x45, 0x20,
    0x05, 0xe0, 0x06, 0x65, 0xe0, 0x00, 0xe5, 0x81,
    0x04, 0xe0, 0x88, 0x7c, 0xe5, 0x63, 0x80, 0xe5,
    0x05, 0x40, 0xe5, 0x01, 0xc0, 0xe5, 0x02, 0x20,
    0x0f, 0x26, 0x16, 0x7b, 0xe0, 0x8e, 0xd4, 0xef,
    0x80, 0x68, 0xe9, 0x02, 0xa0, 0xef, 0x81, 0x2c,
    0xe0, 0x44, 0xe6, 0x26, 0x20, 0xe6, 0x0f, 0xe0,
    0x01, 0xef, 0x6c, 0xe0, 0x34, 0xef, 0x80, 0x6e,
    0xe0, 0x02, 0xef, 0x1f, 0x20, 0xef, 0x34, 0x27,
    0x46, 0x4f, 0xa7, 0xfb, 0x00, 0xe6, 0x00, 0x2f,
    0xc6, 0xef, 0x16, 0x66, 0xef, 0x35, 0xe0, 0x0d,
    0xef, 0x3a, 0x46, 0x0f, 0xe0, 0x72, 0xeb, 0x0c,
    0xe0, 0x04, 0xeb, 0x0c, 0xe0, 0x04, 0xef, 0x4f,
    0xe0, 0x01, 0xeb, 0x11, 0xe0, 0x7f, 0xe1, 0x12,
    0xe2, 0x12, 0xe1, 0x12, 0xc2, 0x00, 0xe2, 0x0a,
    0xe1, 0x12, 0xe2, 0x12, 0x01, 0x00, 0x21, 0x20,
    0x01, 0x20, 0x21, 0x20, 0x61, 0x00, 0xe1, 0x00,
    0x62, 0x00, 0x02, 0x00, 0xc2, 0x00, 0xe2, 0x03,
    0xe1, 0x12, 0xe2, 0x12, 0x21, 0x00, 0x61, 0x20,
    0xe1, 0x00, 0x00, 0xc1, 0x00, 0xe2, 0x12, 0x21,
    0x00, 0x61, 0x00, 0x81, 0x00, 0x01, 0x40, 0xc1,
    0x00, 0xe2, 0x12, 0xe1, 0x12, 0xe2, 0x12, 0xe1,
    0x12, 0xe2, 0x12, 0xe1, 0x12, 0xe2, 0x12, 0xe1,
    0x12, 0xe2, 0x12, 0xe1, 0x12, 0xe2, 0x12, 0xe1,
    0x12, 0xe2, 0x14, 0x20, 0xe1, 0x11, 0x0c, 0xe2,
    0x11, 0x0c, 0xa2, 0xe1, 0x11, 0x0c, 0xe2, 0x11,
    0x0c, 0xa2, 0xe1, 0x11, 0x0c, 0xe2, 0x11, 0x0c,
    0xa2, 0xe1, 0x11, 0x0c, 0xe2, 0x11, 0x0c, 0xa2,
    0xe1, 0x11, 0x0c, 0xe2, 0x11, 0x0c, 0xa2, 0x3f,
    0x20, 0xe9, 0x2a, 0xef, 0x81, 0x78, 0xe6, 0x2f,
    0x6f, 0xe6, 0x2a, 0xef, 0x00, 0x06, 0xef, 0x06,
    0x06, 0x2f, 0x96, 0xe0, 0x07, 0x86, 0x00, 0xe6,
    0x07, 0xe0, 0x83, 0xc8, 0xe2, 0x02, 0x05, 0xe2,
    0x0c, 0xa0, 0xa2, 0xe0, 0x80, 0x4d, 0xc6, 0x00,
    0xe6, 0x09, 0x20, 0xc6, 0x00, 0x26, 0x00, 0x86,
    0x80, 0xe4, 0x36, 0xe0, 0x19, 0x06, 0xe0, 0x68,
    0xe5, 0x25, 0x40, 0xc6, 0xc4, 0x20, 0xe9, 0x02,
    0x60, 0x05, 0x0f, 0xe0, 0x80, 0xb8, 0xe5, 0x16,
    0x06, 0xe0, 0x09, 0xe5, 0x24, 0x66, 0xe9, 0x02,
    0x80, 0x0d, 0xe0, 0x81, 0x48, 0xe5, 0x13, 0x04,
    0x66, 0xe9, 0x02, 0xe0, 0x80, 0x4e, 0xe5, 0x16,
    0x26, 0x05, 0xe9, 0x02, 0x60, 0x16, 0xe0, 0x81,
    0x58, 0xc5, 0x00, 0x65, 0x00, 0x25, 0x00, 0xe5,
    0x07, 0x00, 0xe5, 0x80, 0x3d, 0x20, 0xeb, 0x01,
    0xc6, 0xe0, 0x21, 0xe1, 0x1a, 0xe2, 0x1a, 0xc6,
    0x04, 0x60, 0xe9, 0x02, 0x60, 0x36, 0xe0, 0x82,
    0x89, 0xeb, 0x33, 0x0f, 0x4b, 0x0d, 0x6b, 0xe0,
    0x44, 0xeb, 0x25, 0x0f, 0xeb, 0x07, 0xe0, 0x80,
    0x3a, 0x65, 0x00, 0xe5, 0x13, 0x00, 0x25, 0x00,
    0x05, 0x20, 0x05, 0x00, 0xe5, 0x02, 0x00, 0x65,
    0x00, 0x05, 0x00, 0x05, 0xa0, 0x05, 0x60, 0x05,
    0x00, 0x05, 0x00, 0x05, 0x00, 0x45, 0x00, 0x25,
    0x00, 0x05, 0x20, 0x05, 0x00, 0x05, 0x00, 0x05,
    0x00, 0x05, 0x00, 0x05, 0x00, 0x25, 0x00, 0x05,
    0x20, 0x65, 0x00, 0xc5, 0x00, 0x65, 0x00, 0x65,
    0x00, 0x05, 0x00, 0xe5, 0x02, 0x00, 0xe5, 0x09,
    0x80, 0x45, 0x00, 0x85, 0x00, 0xe5, 0x09, 0xe0,
    0x2c, 0x2c, 0xe0, 0x80, 0x86, 0xef, 0x24, 0x60,
    0xef, 0x5c, 0xe0, 0x04, 0xef, 0x07, 0x20, 0xef,
    0x07, 0x00, 0xef, 0x07, 0x00, 0xef, 0x1d, 0xe0,
    0x02, 0xeb, 0x05, 0xef, 0x80, 0x19, 0xe0, 0x30,
    0xef, 0x15, 0xe0, 0x05, 0xef, 0x24, 0x60, 0xef,
    0x01, 0xc0, 0x2f, 0xe0, 0x06, 0xaf, 0xe0, 0x80,
    0x12, 0xef, 0x80, 0x73, 0x8e, 0xef, 0x82, 0x50,
    0x60, 0xef, 0x09, 0x40, 0xef, 0x05, 0x40, 0xef,
    0x6f, 0x60, 0xef, 0x57, 0xa0, 0xef, 0x04, 0x60,
    0x0f, 0xe0, 0x07, 0xef, 0x04, 0x60, 0xef, 0x30,
    0xe0, 0x00, 0xef, 0x02, 0xa0, 0xef, 0x20, 0xe0,
    0x00, 0xef, 0x16, 0x20, 0xef, 0x04, 0x60, 0x2f,
    0xe0, 0x36, 0xef, 0x80, 0xcc, 0xe0, 0x04, 0xef,
    0x06, 0x20, 0xef, 0x05, 0x40, 0xef, 0x02, 0x80,
    0xef, 0x30, 0xc0, 0xef, 0x07, 0x20, 0xef, 0x03,
    0xa0, 0xef, 0x01, 0xc0, 0xef, 0x80, 0x0b, 0x00,
    0xef, 0x54, 0xe9, 0x02, 0xe0, 0x83, 0x7e, 0xe5,
    0xc0, 0x66, 0x58, 0xe0, 0x18, 0xe5, 0x8f, 0xb2,
    0xa0, 0xe5, 0x80, 0x56, 0x20, 0xe5, 0x95, 0xfa,
    0xe0, 0x06, 0xe5, 0x9c, 0xa9, 0xe0, 0x07, 0xe5,
    0x81, 0xe6, 0xe0, 0x89, 0x1a, 0xe5, 0x81, 0x96,
    0xe0, 0x85, 0x5a, 0xe5, 0x92, 0xc3, 0x80, 0xe5,
    0x8f, 0xd8, 0xe0, 0xca, 0x9b, 0xc9, 0x1b, 0xe0,
    0x16, 0xfb, 0x58, 0xe0, 0x78, 0xe6, 0x80, 0x68,
    0xe0, 0xc0, 0xbd, 0x88, 0xfd, 0xc0, 0xbf, 0x76,
    0x20, 0xfd, 0xc0, 0xbf, 0x76, 0x20,
};

typedef enum {
    UNICODE_SCRIPT_Unknown,
    UNICODE_SCRIPT_Adlam,
    UNICODE_SCRIPT_Ahom,
    UNICODE_SCRIPT_Anatolian_Hieroglyphs,
    UNICODE_SCRIPT_Arabic,
    UNICODE_SCRIPT_Armenian,
    UNICODE_SCRIPT_Avestan,
    UNICODE_SCRIPT_Balinese,
    UNICODE_SCRIPT_Bamum,
    UNICODE_SCRIPT_Bassa_Vah,
    UNICODE_SCRIPT_Batak,
    UNICODE_SCRIPT_Bengali,
    UNICODE_SCRIPT_Bhaiksuki,
    UNICODE_SCRIPT_Bopomofo,
    UNICODE_SCRIPT_Brahmi,
    UNICODE_SCRIPT_Braille,
    UNICODE_SCRIPT_Buginese,
    UNICODE_SCRIPT_Buhid,
    UNICODE_SCRIPT_Canadian_Aboriginal,
    UNICODE_SCRIPT_Carian,
    UNICODE_SCRIPT_Caucasian_Albanian,
    UNICODE_SCRIPT_Chakma,
    UNICODE_SCRIPT_Cham,
    UNICODE_SCRIPT_Cherokee,
    UNICODE_SCRIPT_Chorasmian,
    UNICODE_SCRIPT_Common,
    UNICODE_SCRIPT_Coptic,
    UNICODE_SCRIPT_Cuneiform,
    UNICODE_SCRIPT_Cypriot,
    UNICODE_SCRIPT_Cyrillic,
    UNICODE_SCRIPT_Cypro_Minoan,
    UNICODE_SCRIPT_Deseret,
    UNICODE_SCRIPT_Devanagari,
    UNICODE_SCRIPT_Dives_Akuru,
    UNICODE_SCRIPT_Dogra,
    UNICODE_SCRIPT_Duployan,
    UNICODE_SCRIPT_Egyptian_Hieroglyphs,
    UNICODE_SCRIPT_Elbasan,
    UNICODE_SCRIPT_Elymaic,
    UNICODE_SCRIPT_Ethiopic,
    UNICODE_SCRIPT_Georgian,
    UNICODE_SCRIPT_Glagolitic,
    UNICODE_SCRIPT_Gothic,
    UNICODE_SCRIPT_Garay,
    UNICODE_SCRIPT_Grantha,
    UNICODE_SCRIPT_Greek,
    UNICODE_SCRIPT_Gujarati,
    UNICODE_SCRIPT_Gunjala_Gondi,
    UNICODE_SCRIPT_Gurmukhi,
    UNICODE_SCRIPT_Gurung_Khema,
    UNICODE_SCRIPT_Han,
    UNICODE_SCRIPT_Hangul,
    UNICODE_SCRIPT_Hanifi_Rohingya,
    UNICODE_SCRIPT_Hanunoo,
    UNICODE_SCRIPT_Hatran,
    UNICODE_SCRIPT_Hebrew,
    UNICODE_SCRIPT_Hiragana,
    UNICODE_SCRIPT_Imperial_Aramaic,
    UNICODE_SCRIPT_Inherited,
    UNICODE_SCRIPT_Inscriptional_Pahlavi,
    UNICODE_SCRIPT_Inscriptional_Parthian,
    UNICODE_SCRIPT_Javanese,
    UNICODE_SCRIPT_Kaithi,
    UNICODE_SCRIPT_Kannada,
    UNICODE_SCRIPT_Katakana,
    UNICODE_SCRIPT_Kawi,
    UNICODE_SCRIPT_Kayah_Li,
    UNICODE_SCRIPT_Kharoshthi,
    UNICODE_SCRIPT_Khmer,
    UNICODE_SCRIPT_Khojki,
    UNICODE_SCRIPT_Khitan_Small_Script,
    UNICODE_SCRIPT_Khudawadi,
    UNICODE_SCRIPT_Kirat_Rai,
    UNICODE_SCRIPT_Lao,
    UNICODE_SCRIPT_Latin,
    UNICODE_SCRIPT_Lepcha,
    UNICODE_SCRIPT_Limbu,
    UNICODE_SCRIPT_Linear_A,
    UNICODE_SCRIPT_Linear_B,
    UNICODE_SCRIPT_Lisu,
    UNICODE_SCRIPT_Lycian,
    UNICODE_SCRIPT_Lydian,
    UNICODE_SCRIPT_Makasar,
    UNICODE_SCRIPT_Mahajani,
    UNICODE_SCRIPT_Malayalam,
    UNICODE_SCRIPT_Mandaic,
    UNICODE_SCRIPT_Manichaean,
    UNICODE_SCRIPT_Marchen,
    UNICODE_SCRIPT_Masaram_Gondi,
    UNICODE_SCRIPT_Medefaidrin,
    UNICODE_SCRIPT_Meetei_Mayek,
    UNICODE_SCRIPT_Mende_Kikakui,
    UNICODE_SCRIPT_Meroitic_Cursive,
    UNICODE_SCRIPT_Meroitic_Hieroglyphs,
    UNICODE_SCRIPT_Miao,
    UNICODE_SCRIPT_Modi,
    UNICODE_SCRIPT_Mongolian,
    UNICODE_SCRIPT_Mro,
    UNICODE_SCRIPT_Multani,
    UNICODE_SCRIPT_Myanmar,
    UNICODE_SCRIPT_Nabataean,
    UNICODE_SCRIPT_Nag_Mundari,
    UNICODE_SCRIPT_Nandinagari,
    UNICODE_SCRIPT_New_Tai_Lue,
    UNICODE_SCRIPT_Newa,
    UNICODE_SCRIPT_Nko,
    UNICODE_SCRIPT_Nushu,
    UNICODE_SCRIPT_Nyiakeng_Puachue_Hmong,
    UNICODE_SCRIPT_Ogham,
    UNICODE_SCRIPT_Ol_Chiki,
    UNICODE_SCRIPT_Ol_Onal,
    UNICODE_SCRIPT_Old_Hungarian,
    UNICODE_SCRIPT_Old_Italic,
    UNICODE_SCRIPT_Old_North_Arabian,
    UNICODE_SCRIPT_Old_Permic,
    UNICODE_SCRIPT_Old_Persian,
    UNICODE_SCRIPT_Old_Sogdian,
    UNICODE_SCRIPT_Old_South_Arabian,
    UNICODE_SCRIPT_Old_Turkic,
    UNICODE_SCRIPT_Old_Uyghur,
    UNICODE_SCRIPT_Oriya,
    UNICODE_SCRIPT_Osage,
    UNICODE_SCRIPT_Osmanya,
    UNICODE_SCRIPT_Pahawh_Hmong,
    UNICODE_SCRIPT_Palmyrene,
    UNICODE_SCRIPT_Pau_Cin_Hau,
    UNICODE_SCRIPT_Phags_Pa,
    UNICODE_SCRIPT_Phoenician,
    UNICODE_SCRIPT_Psalter_Pahlavi,
    UNICODE_SCRIPT_Rejang,
    UNICODE_SCRIPT_Runic,
    UNICODE_SCRIPT_Samaritan,
    UNICODE_SCRIPT_Saurashtra,
    UNICODE_SCRIPT_Sharada,
    UNICODE_SCRIPT_Shavian,
    UNICODE_SCRIPT_Siddham,
    UNICODE_SCRIPT_SignWriting,
    UNICODE_SCRIPT_Sinhala,
    UNICODE_SCRIPT_Sogdian,
    UNICODE_SCRIPT_Sora_Sompeng,
    UNICODE_SCRIPT_Soyombo,
    UNICODE_SCRIPT_Sundanese,
    UNICODE_SCRIPT_Sunuwar,
    UNICODE_SCRIPT_Syloti_Nagri,
    UNICODE_SCRIPT_Syriac,
    UNICODE_SCRIPT_Tagalog,
    UNICODE_SCRIPT_Tagbanwa,
    UNICODE_SCRIPT_Tai_Le,
    UNICODE_SCRIPT_Tai_Tham,
    UNICODE_SCRIPT_Tai_Viet,
    UNICODE_SCRIPT_Takri,
    UNICODE_SCRIPT_Tamil,
    UNICODE_SCRIPT_Tangut,
    UNICODE_SCRIPT_Telugu,
    UNICODE_SCRIPT_Thaana,
    UNICODE_SCRIPT_Thai,
    UNICODE_SCRIPT_Tibetan,
    UNICODE_SCRIPT_Tifinagh,
    UNICODE_SCRIPT_Tirhuta,
    UNICODE_SCRIPT_Tangsa,
    UNICODE_SCRIPT_Todhri,
    UNICODE_SCRIPT_Toto,
    UNICODE_SCRIPT_Tulu_Tigalari,
    UNICODE_SCRIPT_Ugaritic,
    UNICODE_SCRIPT_Vai,
    UNICODE_SCRIPT_Vithkuqi,
    UNICODE_SCRIPT_Wancho,
    UNICODE_SCRIPT_Warang_Citi,
    UNICODE_SCRIPT_Yezidi,
    UNICODE_SCRIPT_Yi,
    UNICODE_SCRIPT_Zanabazar_Square,
    UNICODE_SCRIPT_COUNT,
} UnicodeScriptEnum;

static const char unicode_script_name_table[] =
    "Adlam,Adlm"                  "\0"
    "Ahom,Ahom"                   "\0"
    "Anatolian_Hieroglyphs,Hluw"  "\0"
    "Arabic,Arab"                 "\0"
    "Armenian,Armn"               "\0"
    "Avestan,Avst"                "\0"
    "Balinese,Bali"               "\0"
    "Bamum,Bamu"                  "\0"
    "Bassa_Vah,Bass"              "\0"
    "Batak,Batk"                  "\0"
    "Bengali,Beng"                "\0"
    "Bhaiksuki,Bhks"              "\0"
    "Bopomofo,Bopo"               "\0"
    "Brahmi,Brah"                 "\0"
    "Braille,Brai"                "\0"
    "Buginese,Bugi"               "\0"
    "Buhid,Buhd"                  "\0"
    "Canadian_Aboriginal,Cans"    "\0"
    "Carian,Cari"                 "\0"
    "Caucasian_Albanian,Aghb"     "\0"
    "Chakma,Cakm"                 "\0"
    "Cham,Cham"                   "\0"
    "Cherokee,Cher"               "\0"
    "Chorasmian,Chrs"             "\0"
    "Common,Zyyy"                 "\0"
    "Coptic,Copt,Qaac"            "\0"
    "Cuneiform,Xsux"              "\0"
    "Cypriot,Cprt"                "\0"
    "Cyrillic,Cyrl"               "\0"
    "Cypro_Minoan,Cpmn"           "\0"
    "Deseret,Dsrt"                "\0"
    "Devanagari,Deva"             "\0"
    "Dives_Akuru,Diak"            "\0"
    "Dogra,Dogr"                  "\0"
    "Duployan,Dupl"               "\0"
    "Egyptian_Hieroglyphs,Egyp"   "\0"
    "Elbasan,Elba"                "\0"
    "Elymaic,Elym"                "\0"
    "Ethiopic,Ethi"               "\0"
    "Georgian,Geor"               "\0"
    "Glagolitic,Glag"             "\0"
    "Gothic,Goth"                 "\0"
    "Garay,Gara"                  "\0"
    "Grantha,Gran"                "\0"
    "Greek,Grek"                  "\0"
    "Gujarati,Gujr"               "\0"
    "Gunjala_Gondi,Gong"          "\0"
    "Gurmukhi,Guru"               "\0"
    "Gurung_Khema,Gukh"           "\0"
    "Han,Hani"                    "\0"
    "Hangul,Hang"                 "\0"
    "Hanifi_Rohingya,Rohg"        "\0"
    "Hanunoo,Hano"                "\0"
    "Hatran,Hatr"                 "\0"
    "Hebrew,Hebr"                 "\0"
    "Hiragana,Hira"               "\0"
    "Imperial_Aramaic,Armi"       "\0"
    "Inherited,Zinh,Qaai"         "\0"
    "Inscriptional_Pahlavi,Phli"  "\0"
    "Inscriptional_Parthian,Prti" "\0"
    "Javanese,Java"               "\0"
    "Kaithi,Kthi"                 "\0"
    "Kannada,Knda"                "\0"
    "Katakana,Kana"               "\0"
    "Kawi,Kawi"                   "\0"
    "Kayah_Li,Kali"               "\0"
    "Kharoshthi,Khar"             "\0"
    "Khmer,Khmr"                  "\0"
    "Khojki,Khoj"                 "\0"
    "Khitan_Small_Script,Kits"    "\0"
    "Khudawadi,Sind"              "\0"
    "Kirat_Rai,Krai"              "\0"
    "Lao,Laoo"                    "\0"
    "Latin,Latn"                  "\0"
    "Lepcha,Lepc"                 "\0"
    "Limbu,Limb"                  "\0"
    "Linear_A,Lina"               "\0"
    "Linear_B,Linb"               "\0"
    "Lisu,Lisu"                   "\0"
    "Lycian,Lyci"                 "\0"
    "Lydian,Lydi"                 "\0"
    "Makasar,Maka"                "\0"
    "Mahajani,Mahj"               "\0"
    "Malayalam,Mlym"              "\0"
    "Mandaic,Mand"                "\0"
    "Manichaean,Mani"             "\0"
    "Marchen,Marc"                "\0"
    "Masaram_Gondi,Gonm"          "\0"
    "Medefaidrin,Medf"            "\0"
    "Meetei_Mayek,Mtei"           "\0"
    "Mende_Kikakui,Mend"          "\0"
    "Meroitic_Cursive,Merc"       "\0"
    "Meroitic_Hieroglyphs,Mero"   "\0"
    "Miao,Plrd"                   "\0"
    "Modi,Modi"                   "\0"
    "Mongolian,Mong"              "\0"
    "Mro,Mroo"                    "\0"
    "Multani,Mult"                "\0"
    "Myanmar,Mymr"                "\0"
    "Nabataean,Nbat"              "\0"
    "Nag_Mundari,Nagm"            "\0"
    "Nandinagari,Nand"            "\0"
    "New_Tai_Lue,Talu"            "\0"
    "Newa,Newa"                   "\0"
    "Nko,Nkoo"                    "\0"
    "Nushu,Nshu"                  "\0"
    "Nyiakeng_Puachue_Hmong,Hmnp" "\0"
    "Ogham,Ogam"                  "\0"
    "Ol_Chiki,Olck"               "\0"
    "Ol_Onal,Onao"                "\0"
    "Old_Hungarian,Hung"          "\0"
    "Old_Italic,Ital"             "\0"
    "Old_North_Arabian,Narb"      "\0"
    "Old_Permic,Perm"             "\0"
    "Old_Persian,Xpeo"            "\0"
    "Old_Sogdian,Sogo"            "\0"
    "Old_South_Arabian,Sarb"      "\0"
    "Old_Turkic,Orkh"             "\0"
    "Old_Uyghur,Ougr"             "\0"
    "Oriya,Orya"                  "\0"
    "Osage,Osge"                  "\0"
    "Osmanya,Osma"                "\0"
    "Pahawh_Hmong,Hmng"           "\0"
    "Palmyrene,Palm"              "\0"
    "Pau_Cin_Hau,Pauc"            "\0"
    "Phags_Pa,Phag"               "\0"
    "Phoenician,Phnx"             "\0"
    "Psalter_Pahlavi,Phlp"        "\0"
    "Rejang,Rjng"                 "\0"
    "Runic,Runr"                  "\0"
    "Samaritan,Samr"              "\0"
    "Saurashtra,Saur"             "\0"
    "Sharada,Shrd"                "\0"
    "Shavian,Shaw"                "\0"
    "Siddham,Sidd"                "\0"
    "SignWriting,Sgnw"            "\0"
    "Sinhala,Sinh"                "\0"
    "Sogdian,Sogd"                "\0"
    "Sora_Sompeng,Sora"           "\0"
    "Soyombo,Soyo"                "\0"
    "Sundanese,Sund"              "\0"
    "Sunuwar,Sunu"                "\0"
    "Syloti_Nagri,Sylo"           "\0"
    "Syriac,Syrc"                 "\0"
    "Tagalog,Tglg"                "\0"
    "Tagbanwa,Tagb"               "\0"
    "Tai_Le,Tale"                 "\0"
    "Tai_Tham,Lana"               "\0"
    "Tai_Viet,Tavt"               "\0"
    "Takri,Takr"                  "\0"
    "Tamil,Taml"                  "\0"
    "Tangut,Tang"                 "\0"
    "Telugu,Telu"                 "\0"
    "Thaana,Thaa"                 "\0"
    "Thai,Thai"                   "\0"
    "Tibetan,Tibt"                "\0"
    "Tifinagh,Tfng"               "\0"
    "Tirhuta,Tirh"                "\0"
    "Tangsa,Tnsa"                 "\0"
    "Todhri,Todr"                 "\0"
    "Toto,Toto"                   "\0"
    "Tulu_Tigalari,Tutg"          "\0"
    "Ugaritic,Ugar"               "\0"
    "Vai,Vaii"                    "\0"
    "Vithkuqi,Vith"               "\0"
    "Wancho,Wcho"                 "\0"
    "Warang_Citi,Wara"            "\0"
    "Yezidi,Yezi"                 "\0"
    "Yi,Yiii"                     "\0"
    "Zanabazar_Square,Zanb"       "\0"
;

static const uint8_t unicode_script_table[2803] = {
    0xc0, 0x19, 0x99, 0x4a, 0x85, 0x19, 0x99, 0x4a,
    0xae, 0x19, 0x80, 0x4a, 0x8e, 0x19, 0x80, 0x4a,
    0x84, 0x19, 0x96, 0x4a, 0x80, 0x19, 0x9e, 0x4a,
    0x80, 0x19, 0xe1, 0x60, 0x4a, 0xa6, 0x19, 0x84,
    0x4a, 0x84, 0x19, 0x81, 0x0d, 0x93, 0x19, 0xe0,
    0x0f, 0x3a, 0x83, 0x2d, 0x80, 0x19, 0x82, 0x2d,
    0x01, 0x83, 0x2d, 0x80, 0x19, 0x80, 0x2d, 0x03,
    0x80, 0x2d, 0x80, 0x19, 0x80, 0x2d, 0x80, 0x19,
    0x82, 0x2d, 0x00, 0x80, 0x2d, 0x00, 0x93, 0x2d,
    0x00, 0xbe, 0x2d, 0x8d, 0x1a, 0x8f, 0x2d, 0xe0,
    0x24, 0x1d, 0x81, 0x3a, 0xe0, 0x48, 0x1d, 0x00,
    0xa5, 0x05, 0x01, 0xb1, 0x05, 0x01, 0x82, 0x05,
    0x00, 0xb6, 0x37, 0x07, 0x9a, 0x37, 0x03, 0x85,
    0x37, 0x0a, 0x84, 0x04, 0x80, 0x19, 0x85, 0x04,
    0x80, 0x19, 0x8d, 0x04, 0x80, 0x19, 0x82, 0x04,
    0x80, 0x19, 0x9f, 0x04, 0x80, 0x19, 0x89, 0x04,
    0x8a, 0x3a, 0x99, 0x04, 0x80, 0x3a, 0xe0, 0x0b,
    0x04, 0x80, 0x19, 0xa1, 0x04, 0x8d, 0x90, 0x00,
    0xbb, 0x90, 0x01, 0x82, 0x90, 0xaf, 0x04, 0xb1,
    0x9a, 0x0d, 0xba, 0x69, 0x01, 0x82, 0x69, 0xad,
    0x83, 0x01, 0x8e, 0x83, 0x00, 0x9b, 0x55, 0x01,
    0x80, 0x55, 0x00, 0x8a, 0x90, 0x04, 0x9e, 0x04,
    0x00, 0x81, 0x04, 0x04, 0xca, 0x04, 0x80, 0x19,
    0x9c, 0x04, 0xd0, 0x20, 0x83, 0x3a, 0x8e, 0x20,
    0x81, 0x19, 0x99, 0x20, 0x83, 0x0b, 0x00, 0x87,
    0x0b, 0x01, 0x81, 0x0b, 0x01, 0x95, 0x0b, 0x00,
    0x86, 0x0b, 0x00, 0x80, 0x0b, 0x02, 0x83, 0x0b,
    0x01, 0x88, 0x0b, 0x01, 0x81, 0x0b, 0x01, 0x83,
    0x0b, 0x07, 0x80, 0x0b, 0x03, 0x81, 0x0b, 0x00,
    0x84, 0x0b, 0x01, 0x98, 0x0b, 0x01, 0x82, 0x30,
    0x00, 0x85, 0x30, 0x03, 0x81, 0x30, 0x01, 0x95,
    0x30, 0x00, 0x86, 0x30, 0x00, 0x81, 0x30, 0x00,
    0x81, 0x30, 0x00, 0x81, 0x30, 0x01, 0x80, 0x30,
    0x00, 0x84, 0x30, 0x03, 0x81, 0x30, 0x01, 0x82,
    0x30, 0x02, 0x80, 0x30, 0x06, 0x83, 0x30, 0x00,
    0x80, 0x30, 0x06, 0x90, 0x30, 0x09, 0x82, 0x2e,
    0x00, 0x88, 0x2e, 0x00, 0x82, 0x2e, 0x00, 0x95,
    0x2e, 0x00, 0x86, 0x2e, 0x00, 0x81, 0x2e, 0x00,
    0x84, 0x2e, 0x01, 0x89, 0x2e, 0x00, 0x82, 0x2e,
    0x00, 0x82, 0x2e, 0x01, 0x80, 0x2e, 0x0e, 0x83,
    0x2e, 0x01, 0x8b, 0x2e, 0x06, 0x86, 0x2e, 0x00,
    0x82, 0x78, 0x00, 0x87, 0x78, 0x01, 0x81, 0x78,
    0x01, 0x95, 0x78, 0x00, 0x86, 0x78, 0x00, 0x81,
    0x78, 0x00, 0x84, 0x78, 0x01, 0x88, 0x78, 0x01,
    0x81, 0x78, 0x01, 0x82, 0x78, 0x06, 0x82, 0x78,
    0x03, 0x81, 0x78, 0x00, 0x84, 0x78, 0x01, 0x91,
    0x78, 0x09, 0x81, 0x97, 0x00, 0x85, 0x97, 0x02,
    0x82, 0x97, 0x00, 0x83, 0x97, 0x02, 0x81, 0x97,
    0x00, 0x80, 0x97, 0x00, 0x81, 0x97, 0x02, 0x81,
    0x97, 0x02, 0x82, 0x97, 0x02, 0x8b, 0x97, 0x03,
    0x84, 0x97, 0x02, 0x82, 0x97, 0x00, 0x83, 0x97,
    0x01, 0x80, 0x97, 0x05, 0x80, 0x97, 0x0d, 0x94,
    0x97, 0x04, 0x8c, 0x99, 0x00, 0x82, 0x99, 0x00,
    0x96, 0x99, 0x00, 0x8f, 0x99, 0x01, 0x88, 0x99,
    0x00, 0x82, 0x99, 0x00, 0x83, 0x99, 0x06, 0x81,
    0x99, 0x00, 0x82, 0x99, 0x01, 0x80, 0x99, 0x01,
    0x83, 0x99, 0x01, 0x89, 0x99, 0x06, 0x88, 0x99,
    0x8c, 0x3f, 0x00, 0x82, 0x3f, 0x00, 0x96, 0x3f,
    0x00, 0x89, 0x3f, 0x00, 0x84, 0x3f, 0x01, 0x88,
    0x3f, 0x00, 0x82, 0x3f, 0x00, 0x83, 0x3f, 0x06,
    0x81, 0x3f, 0x05, 0x81, 0x3f, 0x00, 0x83, 0x3f,
    0x01, 0x89, 0x3f, 0x00, 0x82, 0x3f, 0x0b, 0x8c,
    0x54, 0x00, 0x82, 0x54, 0x00, 0xb2, 0x54, 0x00,
    0x82, 0x54, 0x00, 0x85, 0x54, 0x03, 0x8f, 0x54,
    0x01, 0x99, 0x54, 0x00, 0x82, 0x89, 0x00, 0x91,
    0x89, 0x02, 0x97, 0x89, 0x00, 0x88, 0x89, 0x00,
    0x80, 0x89, 0x01, 0x86, 0x89, 0x02, 0x80, 0x89,
    0x03, 0x85, 0x89, 0x00, 0x80, 0x89, 0x00, 0x87,
    0x89, 0x05, 0x89, 0x89, 0x01, 0x82, 0x89, 0x0b,
    0xb9, 0x9b, 0x03, 0x80, 0x19, 0x9b, 0x9b, 0x24,
    0x81, 0x49, 0x00, 0x80, 0x49, 0x00, 0x84, 0x49,
    0x00, 0x97, 0x49, 0x00, 0x80, 0x49, 0x00, 0x96,
    0x49, 0x01, 0x84, 0x49, 0x00, 0x80, 0x49, 0x00,
    0x86, 0x49, 0x00, 0x89, 0x49, 0x01, 0x83, 0x49,
    0x1f, 0xc7, 0x9c, 0x00, 0xa3, 0x9c, 0x03, 0xa6,
    0x9c, 0x00, 0xa3, 0x9c, 0x00, 0x8e, 0x9c, 0x00,
    0x86, 0x9c, 0x83, 0x19, 0x81, 0x9c, 0x24, 0xe0,
    0x3f, 0x63, 0xa5, 0x28, 0x00, 0x80, 0x28, 0x04,
    0x80, 0x28, 0x01, 0xaa, 0x28, 0x80, 0x19, 0x83,
    0x28, 0xe0, 0x9f, 0x33, 0xc8, 0x27, 0x00, 0x83,
    0x27, 0x01, 0x86, 0x27, 0x00, 0x80, 0x27, 0x00,
    0x83, 0x27, 0x01, 0xa8, 0x27, 0x00, 0x83, 0x27,
    0x01, 0xa0, 0x27, 0x00, 0x83, 0x27, 0x01, 0x86,
    0x27, 0x00, 0x80, 0x27, 0x00, 0x83, 0x27, 0x01,
    0x8e, 0x27, 0x00, 0xb8, 0x27, 0x00, 0x83, 0x27,
    0x01, 0xc2, 0x27, 0x01, 0x9f, 0x27, 0x02, 0x99,
    0x27, 0x05, 0xd5, 0x17, 0x01, 0x85, 0x17, 0x01,
    0xe2, 0x1f, 0x12, 0x9c, 0x6c, 0x02, 0xca, 0x82,
    0x82, 0x19, 0x8a, 0x82, 0x06, 0x95, 0x91, 0x08,
    0x80, 0x91, 0x94, 0x35, 0x81, 0x19, 0x08, 0x93,
    0x11, 0x0b, 0x8c, 0x92, 0x00, 0x82, 0x92, 0x00,
    0x81, 0x92, 0x0b, 0xdd, 0x44, 0x01, 0x89, 0x44,
    0x05, 0x89, 0x44, 0x05, 0x81, 0x60, 0x81, 0x19,
    0x80, 0x60, 0x80, 0x19, 0x93, 0x60, 0x05, 0xd8,
    0x60, 0x06, 0xaa, 0x60, 0x04, 0xc5, 0x12, 0x09,
    0x9e, 0x4c, 0x00, 0x8b, 0x4c, 0x03, 0x8b, 0x4c,
    0x03, 0x80, 0x4c, 0x02, 0x8b, 0x4c, 0x9d, 0x93,
    0x01, 0x84, 0x93, 0x0a, 0xab, 0x67, 0x03, 0x99,
    0x67, 0x05, 0x8a, 0x67, 0x02, 0x81, 0x67, 0x9f,
    0x44, 0x9b, 0x10, 0x01, 0x81, 0x10, 0xbe, 0x94,
    0x00, 0x9c, 0x94, 0x01, 0x8a, 0x94, 0x05, 0x89,
    0x94, 0x05, 0x8d, 0x94, 0x01, 0x9e, 0x3a, 0x30,
    0xcc, 0x07, 0x00, 0xb1, 0x07, 0xbf, 0x8d, 0xb3,
    0x0a, 0x07, 0x83, 0x0a, 0xb7, 0x4b, 0x02, 0x8e,
    0x4b, 0x02, 0x82, 0x4b, 0xaf, 0x6d, 0x8a, 0x1d,
    0x04, 0xaa, 0x28, 0x01, 0x82, 0x28, 0x87, 0x8d,
    0x07, 0x82, 0x3a, 0x80, 0x19, 0x8c, 0x3a, 0x80,
    0x19, 0x86, 0x3a, 0x83, 0x19, 0x80, 0x3a, 0x85,
    0x19, 0x80, 0x3a, 0x82, 0x19, 0x81, 0x3a, 0x80,
    0x19, 0x04, 0xa5, 0x4a, 0x84, 0x2d, 0x80, 0x1d,
    0xb0, 0x4a, 0x84, 0x2d, 0x83, 0x4a, 0x84, 0x2d,
    0x8c, 0x4a, 0x80, 0x1d, 0xc5, 0x4a, 0x80, 0x2d,
    0xbf, 0x3a, 0xe0, 0x9f, 0x4a, 0x95, 0x2d, 0x01,
    0x85, 0x2d, 0x01, 0xa5, 0x2d, 0x01, 0x85, 0x2d,
    0x01, 0x87, 0x2d, 0x00, 0x80, 0x2d, 0x00, 0x80,
    0x2d, 0x00, 0x80, 0x2d, 0x00, 0x9e, 0x2d, 0x01,
    0xb4, 0x2d, 0x00, 0x8e, 0x2d, 0x00, 0x8d, 0x2d,
    0x01, 0x85, 0x2d, 0x00, 0x92, 0x2d, 0x01, 0x82,
    0x2d, 0x00, 0x88, 0x2d, 0x00, 0x8b, 0x19, 0x81,
    0x3a, 0xd6, 0x19, 0x00, 0x8a, 0x19, 0x80, 0x4a,
    0x01, 0x8a, 0x19, 0x80, 0x4a, 0x8e, 0x19, 0x00,
    0x8c, 0x4a, 0x02, 0xa0, 0x19, 0x0e, 0xa0, 0x3a,
    0x0e, 0xa5, 0x19, 0x80, 0x2d, 0x82, 0x19, 0x81,
    0x4a, 0x85, 0x19, 0x80, 0x4a, 0x9a, 0x19, 0x80,
    0x4a, 0x90, 0x19, 0xa8, 0x4a, 0x82, 0x19, 0x03,
    0xe2, 0x39, 0x19, 0x15, 0x8a, 0x19, 0x14, 0xe3,
    0x3f, 0x19, 0xe0, 0x9f, 0x0f, 0xe2, 0x13, 0x19,
    0x01, 0x9f, 0x19, 0x00, 0xe0, 0x08, 0x19, 0xdf,
    0x29, 0x9f, 0x4a, 0xe0, 0x13, 0x1a, 0x04, 0x86,
    0x1a, 0xa5, 0x28, 0x00, 0x80, 0x28, 0x04, 0x80,
    0x28, 0x01, 0xb7, 0x9d, 0x06, 0x81, 0x9d, 0x0d,
    0x80, 0x9d, 0x96, 0x27, 0x08, 0x86, 0x27, 0x00,
    0x86, 0x27, 0x00, 0x86, 0x27, 0x00, 0x86, 0x27,
    0x00, 0x86, 0x27, 0x00, 0x86, 0x27, 0x00, 0x86,
    0x27, 0x00, 0x86, 0x27, 0x00, 0x9f, 0x1d, 0xdd,
    0x19, 0x21, 0x99, 0x32, 0x00, 0xd8, 0x32, 0x0b,
    0xe0, 0x75, 0x32, 0x19, 0x94, 0x19, 0x80, 0x32,
    0x80, 0x19, 0x80, 0x32, 0x98, 0x19, 0x88, 0x32,
    0x83, 0x3a, 0x81, 0x33, 0x87, 0x19, 0x83, 0x32,
    0x83, 0x19, 0x00, 0xd5, 0x38, 0x01, 0x81, 0x3a,
    0x81, 0x19, 0x82, 0x38, 0x80, 0x19, 0xd9, 0x40,
    0x81, 0x19, 0x82, 0x40, 0x04, 0xaa, 0x0d, 0x00,
    0xdd, 0x33, 0x00, 0x8f, 0x19, 0x9f, 0x0d, 0xa5,
    0x19, 0x08, 0x80, 0x19, 0x8f, 0x40, 0x9e, 0x33,
    0x00, 0xbf, 0x19, 0x9e, 0x33, 0xd0, 0x19, 0xae,
    0x40, 0x80, 0x19, 0xd7, 0x40, 0xe0, 0x47, 0x19,
    0xf0, 0x09, 0x5f, 0x32, 0xbf, 0x19, 0xf0, 0x41,
    0x9f, 0x32, 0xe4, 0x2c, 0xa9, 0x02, 0xb6, 0xa9,
    0x08, 0xaf, 0x4f, 0xe0, 0xcb, 0xa4, 0x13, 0xdf,
    0x1d, 0xd7, 0x08, 0x07, 0xa1, 0x19, 0xe0, 0x05,
    0x4a, 0x82, 0x19, 0xc2, 0x4a, 0x01, 0x81, 0x4a,
    0x00, 0x80, 0x4a, 0x00, 0x87, 0x4a, 0x14, 0x8d,
    0x4a, 0xac, 0x8f, 0x02, 0x89, 0x19, 0x05, 0xb7,
    0x7e, 0x07, 0xc5, 0x84, 0x07, 0x8b, 0x84, 0x05,
    0x9f, 0x20, 0xad, 0x42, 0x80, 0x19, 0x80, 0x42,
    0xa3, 0x81, 0x0a, 0x80, 0x81, 0x9c, 0x33, 0x02,
    0xcd, 0x3d, 0x00, 0x80, 0x19, 0x89, 0x3d, 0x03,
    0x81, 0x3d, 0x9e, 0x63, 0x00, 0xb6, 0x16, 0x08,
    0x8d, 0x16, 0x01, 0x89, 0x16, 0x01, 0x83, 0x16,
    0x9f, 0x63, 0xc2, 0x95, 0x17, 0x84, 0x95, 0x96,
    0x5a, 0x09, 0x85, 0x27, 0x01, 0x85, 0x27, 0x01,
    0x85, 0x27, 0x08, 0x86, 0x27, 0x00, 0x86, 0x27,
    0x00, 0xaa, 0x4a, 0x80, 0x19, 0x88, 0x4a, 0x80,
    0x2d, 0x83, 0x4a, 0x81, 0x19, 0x03, 0xcf, 0x17,
    0xad, 0x5a, 0x01, 0x89, 0x5a, 0x05, 0xf0, 0x1b,
    0x43, 0x33, 0x0b, 0x96, 0x33, 0x03, 0xb0, 0x33,
    0x70, 0x10, 0xa3, 0xe1, 0x0d, 0x32, 0x01, 0xe0,
    0x09, 0x32, 0x25, 0x86, 0x4a, 0x0b, 0x84, 0x05,
    0x04, 0x99, 0x37, 0x00, 0x84, 0x37, 0x00, 0x80,
    0x37, 0x00, 0x81, 0x37, 0x00, 0x81, 0x37, 0x00,
    0x89, 0x37, 0xe0, 0x12, 0x04, 0x0f, 0xe1, 0x0a,
    0x04, 0x81, 0x19, 0xcf, 0x04, 0x01, 0xb5, 0x04,
    0x06, 0x80, 0x04, 0x1f, 0x8f, 0x04, 0x8f, 0x3a,
    0x89, 0x19, 0x05, 0x8d, 0x3a, 0x81, 0x1d, 0xa2,
    0x19, 0x00, 0x92, 0x19, 0x00, 0x83, 0x19, 0x03,
    0x84, 0x04, 0x00, 0xe0, 0x26, 0x04, 0x01, 0x80,
    0x19, 0x00, 0x9f, 0x19, 0x99, 0x4a, 0x85, 0x19,
    0x99, 0x4a, 0x8a, 0x19, 0x89, 0x40, 0x80, 0x19,
    0xac, 0x40, 0x81, 0x19, 0x9e, 0x33, 0x02, 0x85,
    0x33, 0x01, 0x85, 0x33, 0x01, 0x85, 0x33, 0x01,
    0x82, 0x33, 0x02, 0x86, 0x19, 0x00, 0x86, 0x19,
    0x09, 0x84, 0x19, 0x01, 0x8b, 0x4e, 0x00, 0x99,
    0x4e, 0x00, 0x92, 0x4e, 0x00, 0x81, 0x4e, 0x00,
    0x8e, 0x4e, 0x01, 0x8d, 0x4e, 0x21, 0xe0, 0x1a,
    0x4e, 0x04, 0x82, 0x19, 0x03, 0xac, 0x19, 0x02,
    0x88, 0x19, 0xce, 0x2d, 0x00, 0x8c, 0x19, 0x02,
    0x80, 0x2d, 0x2e, 0xac, 0x19, 0x80, 0x3a, 0x60,
    0x21, 0x9c, 0x50, 0x02, 0xb0, 0x13, 0x0e, 0x80,
    0x3a, 0x9a, 0x19, 0x03, 0xa3, 0x70, 0x08, 0x82,
    0x70, 0x9a, 0x2a, 0x04, 0xaa, 0x72, 0x04, 0x9d,
    0xa3, 0x00, 0x80, 0xa3, 0xa3, 0x73, 0x03, 0x8d,
    0x73, 0x29, 0xcf, 0x1f, 0xaf, 0x86, 0x9d, 0x7a,
    0x01, 0x89, 0x7a, 0x05, 0xa3, 0x79, 0x03, 0xa3,
    0x79, 0x03, 0xa7, 0x25, 0x07, 0xb3, 0x14, 0x0a,
    0x80, 0x14, 0x8a, 0xa5, 0x00, 0x8e, 0xa5, 0x00,
    0x86, 0xa5, 0x00, 0x81, 0xa5, 0x00, 0x8a, 0xa5,
    0x00, 0x8e, 0xa5, 0x00, 0x86, 0xa5, 0x00, 0x81,
    0xa5, 0x02, 0xb3, 0xa0, 0x0b, 0xe0, 0xd6, 0x4d,
    0x08, 0x95, 0x4d, 0x09, 0x87, 0x4d, 0x17, 0x85,
    0x4a, 0x00, 0xa9, 0x4a, 0x00, 0x88, 0x4a, 0x44,
    0x85, 0x1c, 0x01, 0x80, 0x1c, 0x00, 0xab, 0x1c,
    0x00, 0x81, 0x1c, 0x02, 0x80, 0x1c, 0x01, 0x80,
    0x1c, 0x95, 0x39, 0x00, 0x88, 0x39, 0x9f, 0x7c,
    0x9e, 0x64, 0x07, 0x88, 0x64, 0x2f, 0x92, 0x36,
    0x00, 0x81, 0x36, 0x04, 0x84, 0x36, 0x9b, 0x7f,
    0x02, 0x80, 0x7f, 0x99, 0x51, 0x04, 0x80, 0x51,
    0x3f, 0x9f, 0x5d, 0x97, 0x5c, 0x03, 0x93, 0x5c,
    0x01, 0xad, 0x5c, 0x83, 0x43, 0x00, 0x81, 0x43,
    0x04, 0x87, 0x43, 0x00, 0x82, 0x43, 0x00, 0x9c,
    0x43, 0x01, 0x82, 0x43, 0x03, 0x89, 0x43, 0x06,
    0x88, 0x43, 0x06, 0x9f, 0x75, 0x9f, 0x71, 0x1f,
    0xa6, 0x56, 0x03, 0x8b, 0x56, 0x08, 0xb5, 0x06,
    0x02, 0x86, 0x06, 0x95, 0x3c, 0x01, 0x87, 0x3c,
    0x92, 0x3b, 0x04, 0x87, 0x3b, 0x91, 0x80, 0x06,
    0x83, 0x80, 0x0b, 0x86, 0x80, 0x4f, 0xc8, 0x76,
    0x36, 0xb2, 0x6f, 0x0c, 0xb2, 0x6f, 0x06, 0x85,
    0x6f, 0xa7, 0x34, 0x07, 0x89, 0x34, 0x05, 0xa5,
    0x2b, 0x02, 0x9c, 0x2b, 0x07, 0x81, 0x2b, 0x60,
    0x6f, 0x9e, 0x04, 0x00, 0xa9, 0xa8, 0x00, 0x82,
    0xa8, 0x01, 0x81, 0xa8, 0x0f, 0x82, 0x04, 0x36,
    0x83, 0x04, 0xa7, 0x74, 0x07, 0xa9, 0x8a, 0x15,
    0x99, 0x77, 0x25, 0x9b, 0x18, 0x13, 0x96, 0x26,
    0x08, 0xcd, 0x0e, 0x03, 0xa3, 0x0e, 0x08, 0x80,
    0x0e, 0xc2, 0x3e, 0x09, 0x80, 0x3e, 0x01, 0x98,
    0x8b, 0x06, 0x89, 0x8b, 0x05, 0xb4, 0x15, 0x00,
    0x91, 0x15, 0x07, 0xa6, 0x53, 0x08, 0xdf, 0x85,
    0x00, 0x93, 0x89, 0x0a, 0x91, 0x45, 0x00, 0xae,
    0x45, 0x3d, 0x86, 0x62, 0x00, 0x80, 0x62, 0x00,
    0x83, 0x62, 0x00, 0x8e, 0x62, 0x00, 0x8a, 0x62,
    0x05, 0xba, 0x47, 0x04, 0x89, 0x47, 0x05, 0x83,
    0x2c, 0x00, 0x87, 0x2c, 0x01, 0x81, 0x2c, 0x01,
    0x95, 0x2c, 0x00, 0x86, 0x2c, 0x00, 0x81, 0x2c,
    0x00, 0x84, 0x2c, 0x00, 0x80, 0x3a, 0x88, 0x2c,
    0x01, 0x81, 0x2c, 0x01, 0x82, 0x2c, 0x01, 0x80,
    0x2c, 0x05, 0x80, 0x2c, 0x04, 0x86, 0x2c, 0x01,
    0x86, 0x2c, 0x02, 0x84, 0x2c, 0x0a, 0x89, 0xa2,
    0x00, 0x80, 0xa2, 0x01, 0x80, 0xa2, 0x00, 0xa5,
    0xa2, 0x00, 0x89, 0xa2, 0x00, 0x80, 0xa2, 0x01,
    0x80, 0xa2, 0x00, 0x83, 0xa2, 0x00, 0x89, 0xa2,
    0x00, 0x81, 0xa2, 0x07, 0x81, 0xa2, 0x1c, 0xdb,
    0x68, 0x00, 0x84, 0x68, 0x1d, 0xc7, 0x9e, 0x07,
    0x89, 0x9e, 0x60, 0x45, 0xb5, 0x87, 0x01, 0xa5,
    0x87, 0x21, 0xc4, 0x5f, 0x0a, 0x89, 0x5f, 0x05,
    0x8c, 0x60, 0x12, 0xb9, 0x96, 0x05, 0x89, 0x96,
    0x05, 0x93, 0x63, 0x1b, 0x9a, 0x02, 0x01, 0x8e,
    0x02, 0x03, 0x96, 0x02, 0x60, 0x58, 0xbb, 0x22,
    0x60, 0x03, 0xd2, 0xa7, 0x0b, 0x80, 0xa7, 0x86,
    0x21, 0x01, 0x80, 0x21, 0x01, 0x87, 0x21, 0x00,
    0x81, 0x21, 0x00, 0x9d, 0x21, 0x00, 0x81, 0x21,
    0x01, 0x8b, 0x21, 0x08, 0x89, 0x21, 0x45, 0x87,
    0x66, 0x01, 0xad, 0x66, 0x01, 0x8a, 0x66, 0x1a,
    0xc7, 0xaa, 0x07, 0xd2, 0x8c, 0x0c, 0x8f, 0x12,
    0xb8, 0x7d, 0x06, 0x89, 0x20, 0x60, 0x55, 0xa1,
    0x8e, 0x0d, 0x89, 0x8e, 0x05, 0x88, 0x0c, 0x00,
    0xac, 0x0c, 0x00, 0x8d, 0x0c, 0x09, 0x9c, 0x0c,
    0x02, 0x9f, 0x57, 0x01, 0x95, 0x57, 0x00, 0x8d,
    0x57, 0x48, 0x86, 0x58, 0x00, 0x81, 0x58, 0x00,
    0xab, 0x58, 0x02, 0x80, 0x58, 0x00, 0x81, 0x58,
    0x00, 0x88, 0x58, 0x07, 0x89, 0x58, 0x05, 0x85,
    0x2f, 0x00, 0x81, 0x2f, 0x00, 0xa4, 0x2f, 0x00,
    0x81, 0x2f, 0x00, 0x85, 0x2f, 0x06, 0x89, 0x2f,
    0x60, 0xd5, 0x98, 0x52, 0x06, 0x90, 0x41, 0x00,
    0xa8, 0x41, 0x02, 0x9c, 0x41, 0x54, 0x80, 0x4f,
    0x0e, 0xb1, 0x97, 0x0c, 0x80, 0x97, 0xe3, 0x39,
    0x1b, 0x60, 0x05, 0xe0, 0x0e, 0x1b, 0x00, 0x84,
    0x1b, 0x0a, 0xe0, 0x63, 0x1b, 0x69, 0xeb, 0xe0,
    0x02, 0x1e, 0x0c, 0xe3, 0xf5, 0x24, 0x09, 0xef,
    0x3a, 0x24, 0x04, 0xe1, 0xe6, 0x03, 0x70, 0x0a,
    0x58, 0xb9, 0x31, 0x66, 0x65, 0xe1, 0xd8, 0x08,
    0x06, 0x9e, 0x61, 0x00, 0x89, 0x61, 0x03, 0x81,
    0x61, 0xce, 0x9f, 0x00, 0x89, 0x9f, 0x05, 0x9d,
    0x09, 0x01, 0x85, 0x09, 0x09, 0xc5, 0x7b, 0x09,
    0x89, 0x7b, 0x00, 0x86, 0x7b, 0x00, 0x94, 0x7b,
    0x04, 0x92, 0x7b, 0x61, 0x4f, 0xb9, 0x48, 0x60,
    0x65, 0xda, 0x59, 0x60, 0x04, 0xca, 0x5e, 0x03,
    0xb8, 0x5e, 0x06, 0x90, 0x5e, 0x3f, 0x80, 0x98,
    0x80, 0x6a, 0x81, 0x32, 0x80, 0x46, 0x0a, 0x81,
    0x32, 0x0d, 0xf0, 0x07, 0x97, 0x98, 0x07, 0xe2,
    0x9f, 0x98, 0xe1, 0x75, 0x46, 0x28, 0x80, 0x46,
    0x88, 0x98, 0x70, 0x12, 0x86, 0x83, 0x40, 0x00,
    0x86, 0x40, 0x00, 0x81, 0x40, 0x00, 0x80, 0x40,
    0xe0, 0xbe, 0x38, 0x82, 0x40, 0x0e, 0x80, 0x38,
    0x1c, 0x82, 0x38, 0x01, 0x80, 0x40, 0x0d, 0x83,
    0x40, 0x07, 0xe1, 0x2b, 0x6a, 0x68, 0xa3, 0xe0,
    0x0a, 0x23, 0x04, 0x8c, 0x23, 0x02, 0x88, 0x23,
    0x06, 0x89, 0x23, 0x01, 0x83, 0x23, 0x83, 0x19,
    0x6e, 0xfb, 0xe0, 0x99, 0x19, 0x05, 0xe1, 0x53,
    0x19, 0x4b, 0xad, 0x3a, 0x01, 0x96, 0x3a, 0x08,
    0xe0, 0x13, 0x19, 0x3b, 0xe0, 0x95, 0x19, 0x09,
    0xa6, 0x19, 0x01, 0xbd, 0x19, 0x82, 0x3a, 0x90,
    0x19, 0x87, 0x3a, 0x81, 0x19, 0x86, 0x3a, 0x9d,
    0x19, 0x83, 0x3a, 0xbc, 0x19, 0x14, 0xc5, 0x2d,
    0x60, 0x19, 0x93, 0x19, 0x0b, 0x93, 0x19, 0x0b,
    0xd6, 0x19, 0x08, 0x98, 0x19, 0x60, 0x26, 0xd4,
    0x19, 0x00, 0xc6, 0x19, 0x00, 0x81, 0x19, 0x01,
    0x80, 0x19, 0x01, 0x81, 0x19, 0x01, 0x83, 0x19,
    0x00, 0x8b, 0x19, 0x00, 0x80, 0x19, 0x00, 0x86,
    0x19, 0x00, 0xc0, 0x19, 0x00, 0x83, 0x19, 0x01,
    0x87, 0x19, 0x00, 0x86, 0x19, 0x00, 0x9b, 0x19,
    0x00, 0x83, 0x19, 0x00, 0x84, 0x19, 0x00, 0x80,
    0x19, 0x02, 0x86, 0x19, 0x00, 0xe0, 0xf3, 0x19,
    0x01, 0xe0, 0xc3, 0x19, 0x01, 0xb1, 0x19, 0xe2,
    0x2b, 0x88, 0x0e, 0x84, 0x88, 0x00, 0x8e, 0x88,
    0x63, 0xef, 0x9e, 0x4a, 0x05, 0x85, 0x4a, 0x60,
    0x74, 0x86, 0x29, 0x00, 0x90, 0x29, 0x01, 0x86,
    0x29, 0x00, 0x81, 0x29, 0x00, 0x84, 0x29, 0x04,
    0xbd, 0x1d, 0x20, 0x80, 0x1d, 0x60, 0x0f, 0xac,
    0x6b, 0x02, 0x8d, 0x6b, 0x01, 0x89, 0x6b, 0x03,
    0x81, 0x6b, 0x60, 0xdf, 0x9e, 0xa1, 0x10, 0xb9,
    0xa6, 0x04, 0x80, 0xa6, 0x61, 0x6f, 0xa9, 0x65,
    0x60, 0x75, 0xaa, 0x6e, 0x03, 0x80, 0x6e, 0x61,
    0x7f, 0x86, 0x27, 0x00, 0x83, 0x27, 0x00, 0x81,
    0x27, 0x00, 0x8e, 0x27, 0x00, 0xe0, 0x64, 0x5b,
    0x01, 0x8f, 0x5b, 0x28, 0xcb, 0x01, 0x03, 0x89,
    0x01, 0x03, 0x81, 0x01, 0x62, 0xb0, 0xc3, 0x19,
    0x4b, 0xbc, 0x19, 0x60, 0x61, 0x83, 0x04, 0x00,
    0x9a, 0x04, 0x00, 0x81, 0x04, 0x00, 0x80, 0x04,
    0x01, 0x80, 0x04, 0x00, 0x89, 0x04, 0x00, 0x83,
    0x04, 0x00, 0x80, 0x04, 0x00, 0x80, 0x04, 0x05,
    0x80, 0x04, 0x03, 0x80, 0x04, 0x00, 0x80, 0x04,
    0x00, 0x80, 0x04, 0x00, 0x82, 0x04, 0x00, 0x81,
    0x04, 0x00, 0x80, 0x04, 0x01, 0x80, 0x04, 0x00,
    0x80, 0x04, 0x00, 0x80, 0x04, 0x00, 0x80, 0x04,
    0x00, 0x80, 0x04, 0x00, 0x81, 0x04, 0x00, 0x80,
    0x04, 0x01, 0x83, 0x04, 0x00, 0x86, 0x04, 0x00,
    0x83, 0x04, 0x00, 0x83, 0x04, 0x00, 0x80, 0x04,
    0x00, 0x89, 0x04, 0x00, 0x90, 0x04, 0x04, 0x82,
    0x04, 0x00, 0x84, 0x04, 0x00, 0x90, 0x04, 0x33,
    0x81, 0x04, 0x60, 0xad, 0xab, 0x19, 0x03, 0xe0,
    0x03, 0x19, 0x0b, 0x8e, 0x19, 0x01, 0x8e, 0x19,
    0x00, 0x8e, 0x19, 0x00, 0xa4, 0x19, 0x09, 0xe0,
    0x4d, 0x19, 0x37, 0x99, 0x19, 0x80, 0x38, 0x81,
    0x19, 0x0c, 0xab, 0x19, 0x03, 0x88, 0x19, 0x06,
    0x81, 0x19, 0x0d, 0x85, 0x19, 0x60, 0x39, 0xe3,
    0x77, 0x19, 0x03, 0x90, 0x19, 0x02, 0x8c, 0x19,
    0x02, 0xe0, 0x16, 0x19, 0x03, 0xde, 0x19, 0x05,
    0x8b, 0x19, 0x03, 0x80, 0x19, 0x0e, 0x8b, 0x19,
    0x03, 0xb7, 0x19, 0x07, 0x89, 0x19, 0x05, 0xa7,
    0x19, 0x07, 0x9d, 0x19, 0x01, 0x8b, 0x19, 0x03,
    0x81, 0x19, 0x3d, 0xe0, 0xf3, 0x19, 0x0b, 0x8d,
    0x19, 0x01, 0x8c, 0x19, 0x02, 0x89, 0x19, 0x04,
    0xb7, 0x19, 0x06, 0x8e, 0x19, 0x01, 0x8a, 0x19,
    0x05, 0x88, 0x19, 0x06, 0xe0, 0x32, 0x19, 0x00,
    0xe0, 0x05, 0x19, 0x63, 0xa5, 0xf0, 0x96, 0x7f,
    0x32, 0x1f, 0xef, 0xd9, 0x32, 0x05, 0xe0, 0x7d,
    0x32, 0x01, 0xf0, 0x06, 0x21, 0x32, 0x0d, 0xf0,
    0x0c, 0xd0, 0x32, 0x0e, 0xe2, 0x0d, 0x32, 0x69,
    0x41, 0xe1, 0xbd, 0x32, 0x65, 0x81, 0xf0, 0x02,
    0xea, 0x32, 0x04, 0xef, 0xff, 0x32, 0x7a, 0xcb,
    0xf0, 0x80, 0x19, 0x1d, 0xdf, 0x19, 0x60, 0x1f,
    0xe0, 0x8f, 0x3a,
};

static const uint8_t unicode_script_ext_table[1253] = {
    0x80, 0x36, 0x00, 0x00, 0x10, 0x06, 0x13, 0x1a,
    0x23, 0x25, 0x28, 0x29, 0x2f, 0x2a, 0x2d, 0x32,
    0x4a, 0x51, 0x53, 0x72, 0x86, 0x81, 0x83, 0x00,
    0x00, 0x07, 0x0b, 0x1d, 0x20, 0x4a, 0x4f, 0x9b,
    0xa1, 0x09, 0x00, 0x00, 0x02, 0x0d, 0x4a, 0x00,
    0x00, 0x02, 0x02, 0x0d, 0x4a, 0x00, 0x00, 0x00,
    0x02, 0x4a, 0x4f, 0x08, 0x00, 0x00, 0x02, 0x4a,
    0x9b, 0x00, 0x00, 0x00, 0x02, 0x0d, 0x4a, 0x25,
    0x00, 0x00, 0x08, 0x17, 0x1a, 0x1d, 0x2d, 0x4a,
    0x72, 0x8e, 0x93, 0x00, 0x08, 0x17, 0x1d, 0x2d,
    0x4a, 0x79, 0x8e, 0x93, 0xa0, 0x00, 0x04, 0x17,
    0x1d, 0x4a, 0x9d, 0x00, 0x05, 0x29, 0x4a, 0x8e,
    0x90, 0x9b, 0x00, 0x0b, 0x14, 0x17, 0x1a, 0x1d,
    0x2a, 0x2d, 0x4a, 0x79, 0x90, 0x9d, 0xa0, 0x00,
    0x06, 0x1a, 0x25, 0x29, 0x2a, 0x40, 0x4a, 0x00,
    0x04, 0x1d, 0x2d, 0x4a, 0x72, 0x00, 0x09, 0x1a,
    0x23, 0x37, 0x4a, 0x72, 0x90, 0x93, 0x9d, 0xa0,
    0x00, 0x0a, 0x05, 0x1d, 0x23, 0x2a, 0x2d, 0x37,
    0x4a, 0x72, 0x90, 0x93, 0x00, 0x02, 0x4a, 0x9d,
    0x00, 0x03, 0x23, 0x4a, 0x90, 0x00, 0x04, 0x17,
    0x1d, 0x4a, 0x79, 0x00, 0x03, 0x17, 0x4a, 0x93,
    0x00, 0x02, 0x4a, 0x8e, 0x00, 0x02, 0x27, 0x4a,
    0x00, 0x00, 0x00, 0x02, 0x4a, 0x8e, 0x00, 0x03,
    0x1d, 0x4a, 0xa0, 0x00, 0x00, 0x00, 0x04, 0x2d,
    0x4a, 0x72, 0xa0, 0x0b, 0x00, 0x00, 0x02, 0x4a,
    0x90, 0x01, 0x00, 0x00, 0x05, 0x17, 0x23, 0x40,
    0x4a, 0x90, 0x00, 0x04, 0x17, 0x23, 0x4a, 0x90,
    0x00, 0x02, 0x4a, 0x90, 0x06, 0x00, 0x00, 0x03,
    0x4a, 0x8e, 0x90, 0x00, 0x02, 0x4a, 0x90, 0x00,
    0x00, 0x00, 0x03, 0x17, 0x4a, 0x90, 0x00, 0x06,
    0x14, 0x17, 0x2a, 0x4a, 0x8e, 0x9b, 0x0f, 0x00,
    0x00, 0x01, 0x2d, 0x01, 0x00, 0x00, 0x01, 0x2d,
    0x11, 0x00, 0x00, 0x02, 0x4a, 0x79, 0x04, 0x00,
    0x00, 0x03, 0x14, 0x4a, 0xa0, 0x03, 0x00, 0x0c,
    0x01, 0x4a, 0x03, 0x00, 0x01, 0x02, 0x1a, 0x2d,
    0x80, 0x8c, 0x00, 0x00, 0x02, 0x1d, 0x72, 0x00,
    0x02, 0x1d, 0x29, 0x01, 0x02, 0x1d, 0x4a, 0x00,
    0x02, 0x1d, 0x29, 0x80, 0x80, 0x00, 0x00, 0x03,
    0x05, 0x28, 0x29, 0x80, 0x01, 0x00, 0x00, 0x07,
    0x04, 0x2b, 0x69, 0x34, 0x90, 0x9a, 0xa8, 0x0d,
    0x00, 0x00, 0x07, 0x04, 0x2b, 0x69, 0x34, 0x90,
    0x9a, 0xa8, 0x00, 0x03, 0x04, 0x90, 0x9a, 0x01,
    0x00, 0x00, 0x08, 0x01, 0x04, 0x2b, 0x69, 0x34,
    0x90, 0x9a, 0xa8, 0x1f, 0x00, 0x00, 0x09, 0x01,
    0x04, 0x55, 0x56, 0x77, 0x80, 0x34, 0x8a, 0x90,
    0x09, 0x00, 0x0a, 0x02, 0x04, 0x90, 0x09, 0x00,
    0x09, 0x03, 0x04, 0x9a, 0xa8, 0x05, 0x00, 0x00,
    0x02, 0x04, 0x90, 0x62, 0x00, 0x00, 0x02, 0x04,
    0x34, 0x81, 0xfb, 0x00, 0x00, 0x0d, 0x0b, 0x20,
    0x2c, 0x2e, 0x30, 0x3f, 0x4a, 0x54, 0x78, 0x85,
    0x97, 0x99, 0x9e, 0x00, 0x0c, 0x0b, 0x20, 0x2c,
    0x2e, 0x30, 0x3f, 0x4a, 0x54, 0x78, 0x97, 0x99,
    0x9e, 0x10, 0x00, 0x00, 0x15, 0x0b, 0x20, 0x22,
    0x2f, 0x58, 0x2c, 0x2e, 0x30, 0x3f, 0x53, 0x54,
    0x66, 0x6e, 0x78, 0x47, 0x89, 0x8f, 0x96, 0x97,
    0x99, 0x9e, 0x00, 0x17, 0x0b, 0x20, 0x22, 0x2f,
    0x58, 0x2c, 0x2e, 0x31, 0x30, 0x3f, 0x4c, 0x53,
    0x54, 0x66, 0x6e, 0x78, 0x47, 0x89, 0x8f, 0x96,
    0x97, 0x99, 0x9e, 0x09, 0x04, 0x20, 0x22, 0x3e,
    0x53, 0x75, 0x00, 0x09, 0x03, 0x0b, 0x15, 0x8f,
    0x75, 0x00, 0x09, 0x02, 0x30, 0x62, 0x75, 0x00,
    0x09, 0x02, 0x2e, 0x45, 0x80, 0x75, 0x00, 0x0d,
    0x02, 0x2c, 0x97, 0x80, 0x71, 0x00, 0x09, 0x03,
    0x3f, 0x66, 0xa2, 0x82, 0xcf, 0x00, 0x09, 0x03,
    0x15, 0x63, 0x93, 0x80, 0x30, 0x00, 0x00, 0x03,
    0x28, 0x29, 0x4a, 0x85, 0x6e, 0x00, 0x02, 0x01,
    0x82, 0x46, 0x00, 0x01, 0x04, 0x11, 0x35, 0x92,
    0x91, 0x80, 0x4a, 0x00, 0x01, 0x02, 0x60, 0x7e,
    0x00, 0x00, 0x00, 0x02, 0x60, 0x7e, 0x84, 0x49,
    0x00, 0x00, 0x04, 0x0b, 0x20, 0x2c, 0x3f, 0x00,
    0x01, 0x20, 0x00, 0x04, 0x0b, 0x20, 0x2c, 0x3f,
    0x00, 0x03, 0x20, 0x2c, 0x3f, 0x00, 0x01, 0x20,
    0x01, 0x02, 0x0b, 0x20, 0x00, 0x02, 0x20, 0x85,
    0x00, 0x02, 0x0b, 0x20, 0x00, 0x02, 0x20, 0x85,
    0x00, 0x06, 0x20, 0x3f, 0x54, 0x78, 0x97, 0x99,
    0x00, 0x01, 0x20, 0x01, 0x02, 0x20, 0x85, 0x01,
    0x01, 0x20, 0x00, 0x02, 0x20, 0x85, 0x00, 0x02,
    0x0b, 0x20, 0x06, 0x01, 0x20, 0x00, 0x02, 0x20,
    0x66, 0x00, 0x02, 0x0b, 0x20, 0x01, 0x01, 0x20,
    0x00, 0x02, 0x0b, 0x20, 0x03, 0x01, 0x20, 0x00,
    0x0b, 0x0b, 0x20, 0x2c, 0x3f, 0x54, 0x66, 0x78,
    0x89, 0x99, 0x9e, 0xa2, 0x00, 0x02, 0x20, 0x2c,
    0x00, 0x04, 0x20, 0x2c, 0x3f, 0xa2, 0x01, 0x02,
    0x0b, 0x20, 0x00, 0x01, 0x0b, 0x01, 0x02, 0x20,
    0x2c, 0x00, 0x01, 0x66, 0x80, 0x44, 0x00, 0x01,
    0x01, 0x2d, 0x35, 0x00, 0x00, 0x03, 0x1d, 0x4a,
    0x90, 0x00, 0x00, 0x00, 0x01, 0x90, 0x81, 0xb3,
    0x00, 0x00, 0x03, 0x4a, 0x60, 0x7e, 0x1e, 0x00,
    0x00, 0x02, 0x01, 0x04, 0x09, 0x00, 0x00, 0x06,
    0x13, 0x28, 0x29, 0x6f, 0x50, 0x76, 0x01, 0x00,
    0x00, 0x04, 0x13, 0x2d, 0x6f, 0x5d, 0x80, 0x11,
    0x00, 0x00, 0x03, 0x20, 0x2c, 0x4a, 0x8c, 0xa5,
    0x00, 0x00, 0x02, 0x1a, 0x4a, 0x17, 0x00, 0x00,
    0x02, 0x06, 0x76, 0x00, 0x07, 0x06, 0x13, 0x28,
    0x6f, 0x3e, 0x51, 0x83, 0x09, 0x00, 0x00, 0x01,
    0x23, 0x03, 0x00, 0x00, 0x03, 0x01, 0x04, 0x6f,
    0x00, 0x00, 0x00, 0x02, 0x1d, 0x29, 0x81, 0x2b,
    0x00, 0x0f, 0x02, 0x32, 0x98, 0x00, 0x00, 0x00,
    0x07, 0x0d, 0x33, 0x32, 0x38, 0x40, 0x60, 0xa9,
    0x00, 0x08, 0x0d, 0x33, 0x32, 0x38, 0x40, 0x60,
    0x7e, 0xa9, 0x00, 0x05, 0x0d, 0x33, 0x32, 0x38,
    0x40, 0x01, 0x00, 0x00, 0x01, 0x32, 0x00, 0x00,
    0x01, 0x08, 0x0d, 0x33, 0x32, 0x38, 0x40, 0x60,
    0x9c, 0xa9, 0x01, 0x09, 0x0d, 0x33, 0x32, 0x38,
    0x40, 0x4f, 0x60, 0x9c, 0xa9, 0x05, 0x06, 0x0d,
    0x33, 0x32, 0x38, 0x40, 0xa9, 0x00, 0x00, 0x00,
    0x05, 0x0d, 0x33, 0x32, 0x38, 0x40, 0x07, 0x06,
    0x0d, 0x33, 0x32, 0x38, 0x40, 0xa9, 0x03, 0x05,
    0x0d, 0x33, 0x32, 0x38, 0x40, 0x09, 0x00, 0x03,
    0x02, 0x0d, 0x32, 0x01, 0x00, 0x00, 0x05, 0x0d,
    0x33, 0x32, 0x38, 0x40, 0x04, 0x02, 0x38, 0x40,
    0x00, 0x00, 0x00, 0x05, 0x0d, 0x33, 0x32, 0x38,
    0x40, 0x03, 0x00, 0x01, 0x03, 0x32, 0x38, 0x40,
    0x01, 0x01, 0x32, 0x58, 0x00, 0x03, 0x02, 0x38,
    0x40, 0x02, 0x00, 0x00, 0x02, 0x38, 0x40, 0x59,
    0x00, 0x00, 0x06, 0x0d, 0x33, 0x32, 0x38, 0x40,
    0xa9, 0x00, 0x02, 0x38, 0x40, 0x80, 0x12, 0x00,
    0x0f, 0x01, 0x32, 0x1f, 0x00, 0x25, 0x01, 0x32,
    0x08, 0x00, 0x00, 0x02, 0x32, 0x98, 0x2f, 0x00,
    0x27, 0x01, 0x32, 0x37, 0x00, 0x30, 0x01, 0x32,
    0x0e, 0x00, 0x0b, 0x01, 0x32, 0x32, 0x00, 0x00,
    0x01, 0x32, 0x57, 0x00, 0x18, 0x01, 0x32, 0x09,
    0x00, 0x04, 0x01, 0x32, 0x5f, 0x00, 0x1e, 0x01,
    0x32, 0xc0, 0x31, 0xef, 0x00, 0x00, 0x02, 0x1d,
    0x29, 0x80, 0x0f, 0x00, 0x07, 0x02, 0x32, 0x4a,
    0x80, 0xa7, 0x00, 0x02, 0x10, 0x20, 0x22, 0x2e,
    0x30, 0x45, 0x3f, 0x3e, 0x53, 0x54, 0x5f, 0x66,
    0x85, 0x47, 0x96, 0x9e, 0xa2, 0x02, 0x0f, 0x20,
    0x22, 0x2e, 0x30, 0x45, 0x3f, 0x3e, 0x53, 0x5f,
    0x66, 0x85, 0x47, 0x96, 0x9e, 0xa2, 0x01, 0x0b,
    0x20, 0x22, 0x2e, 0x30, 0x45, 0x3e, 0x53, 0x5f,
    0x47, 0x96, 0x9e, 0x00, 0x0c, 0x20, 0x22, 0x2e,
    0x30, 0x45, 0x3e, 0x53, 0x5f, 0x85, 0x47, 0x96,
    0x9e, 0x00, 0x0b, 0x20, 0x22, 0x2e, 0x30, 0x45,
    0x3e, 0x53, 0x5f, 0x47, 0x96, 0x9e, 0x80, 0x36,
    0x00, 0x00, 0x03, 0x0b, 0x20, 0xa2, 0x00, 0x00,
    0x00, 0x02, 0x20, 0x97, 0x39, 0x00, 0x00, 0x03,
    0x42, 0x4a, 0x63, 0x80, 0x1f, 0x00, 0x00, 0x02,
    0x10, 0x3d, 0xc0, 0x12, 0xed, 0x00, 0x01, 0x02,
    0x04, 0x69, 0x80, 0x31, 0x00, 0x00, 0x02, 0x04,
    0x9a, 0x09, 0x00, 0x00, 0x02, 0x04, 0x9a, 0x46,
    0x00, 0x01, 0x05, 0x0d, 0x33, 0x32, 0x38, 0x40,
    0x80, 0x99, 0x00, 0x04, 0x06, 0x0d, 0x33, 0x32,
    0x38, 0x40, 0xa9, 0x09, 0x00, 0x00, 0x02, 0x38,
    0x40, 0x2c, 0x00, 0x01, 0x02, 0x38, 0x40, 0x80,
    0xdf, 0x00, 0x01, 0x03, 0x1e, 0x1c, 0x4e, 0x00,
    0x02, 0x1c, 0x4e, 0x03, 0x00, 0x2c, 0x03, 0x1c,
    0x4d, 0x4e, 0x02, 0x00, 0x08, 0x02, 0x1c, 0x4e,
    0x81, 0x1f, 0x00, 0x1b, 0x02, 0x04, 0x1a, 0x87,
    0x75, 0x00, 0x00, 0x02, 0x56, 0x77, 0x87, 0x8d,
    0x00, 0x00, 0x02, 0x2c, 0x97, 0x00, 0x00, 0x00,
    0x02, 0x2c, 0x97, 0x36, 0x00, 0x01, 0x02, 0x2c,
    0x97, 0x8c, 0x12, 0x00, 0x01, 0x02, 0x2c, 0x97,
    0x00, 0x00, 0x00, 0x02, 0x2c, 0x97, 0xc0, 0x5c,
    0x4b, 0x00, 0x03, 0x01, 0x23, 0x96, 0x3b, 0x00,
    0x11, 0x01, 0x32, 0x9e, 0x5d, 0x00, 0x01, 0x01,
    0x32, 0xce, 0xcd, 0x2d, 0x00,
};

static const uint8_t unicode_prop_Hyphen_table[28] = {
    0xac, 0x80, 0xfe, 0x80, 0x44, 0xdb, 0x80, 0x52,
    0x7a, 0x80, 0x48, 0x08, 0x81, 0x4e, 0x04, 0x80,
    0x42, 0xe2, 0x80, 0x60, 0xcd, 0x66, 0x80, 0x40,
    0xa8, 0x80, 0xd6, 0x80,
};

static const uint8_t unicode_prop_Other_Math_table[200] = {
    0xdd, 0x80, 0x43, 0x70, 0x11, 0x80, 0x99, 0x09,
    0x81, 0x5c, 0x1f, 0x80, 0x9a, 0x82, 0x8a, 0x80,
    0x9f, 0x83, 0x97, 0x81, 0x8d, 0x81, 0xc0, 0x8c,
    0x18, 0x11, 0x1c, 0x91, 0x03, 0x01, 0x89, 0x00,
    0x14, 0x28, 0x11, 0x09, 0x02, 0x05, 0x13, 0x24,
    0xca, 0x21, 0x18, 0x08, 0x08, 0x00, 0x21, 0x0b,
    0x0b, 0x91, 0x09, 0x00, 0x06, 0x00, 0x29, 0x41,
    0x21, 0x83, 0x40, 0xa7, 0x08, 0x80, 0x97, 0x80,
    0x90, 0x80, 0x41, 0xbc, 0x81, 0x8b, 0x88, 0x24,
    0x21, 0x09, 0x14, 0x8d, 0x00, 0x01, 0x85, 0x97,
    0x81, 0xb8, 0x00, 0x80, 0x9c, 0x83, 0x88, 0x81,
    0x41, 0x55, 0x81, 0x9e, 0x89, 0x41, 0x92, 0x95,
    0xbe, 0x83, 0x9f, 0x81, 0x60, 0xd4, 0x62, 0x00,
    0x03, 0x80, 0x40, 0xd2, 0x00, 0x80, 0x60, 0xd4,
    0xc0, 0xd4, 0x80, 0xc6, 0x01, 0x08, 0x09, 0x0b,
    0x80, 0x8b, 0x00, 0x06, 0x80, 0xc0, 0x03, 0x0f,
    0x06, 0x80, 0x9b, 0x03, 0x04, 0x00, 0x16, 0x80,
    0x41, 0x53, 0x81, 0x98, 0x80, 0x98, 0x80, 0x9e,
    0x80, 0x98, 0x80, 0x9e, 0x80, 0x98, 0x80, 0x9e,
    0x80, 0x98, 0x80, 0x9e, 0x80, 0x98, 0x07, 0x81,
    0xb1, 0x55, 0xff, 0x18, 0x9a, 0x01, 0x00, 0x08,
    0x80, 0x89, 0x03, 0x00, 0x00, 0x28, 0x18, 0x00,
    0x00, 0x02, 0x01, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x0b, 0x06, 0x03, 0x03, 0x00,
    0x80, 0x89, 0x80, 0x90, 0x22, 0x04, 0x80, 0x90,
};

static const uint8_t unicode_prop_Other_Alphabetic_table[443] = {
    0x43, 0x44, 0x80, 0x9c, 0x8c, 0x42, 0x3f, 0x8d,
    0x00, 0x01, 0x01, 0x00, 0xc7, 0x8a, 0xaf, 0x8c,
    0x06, 0x8f, 0x80, 0xe4, 0x33, 0x19, 0x0b, 0x80,
    0xa2, 0x80, 0x9d, 0x8f, 0xe5, 0x8a, 0xe4, 0x0a,
    0x88, 0x02, 0x03, 0xe9, 0x80, 0xbb, 0x8b, 0x16,
    0x85, 0x93, 0xb5, 0x09, 0x8e, 0x01, 0x22, 0x89,
    0x81, 0x9c, 0x82, 0xb9, 0x31, 0x09, 0x81, 0x89,
    0x80, 0x89, 0x81, 0x9c, 0x82, 0xb9, 0x23, 0x09,
    0x0b, 0x80, 0x9d, 0x0a, 0x80, 0x8a, 0x82, 0xb9,
    0x38, 0x10, 0x81, 0x94, 0x81, 0x95, 0x13, 0x82,
    0xb9, 0x31, 0x09, 0x81, 0x88, 0x81, 0x89, 0x81,
    0x9d, 0x80, 0xba, 0x22, 0x10, 0x82, 0x89, 0x80,
    0xa7, 0x84, 0xb8, 0x30, 0x10, 0x17, 0x81, 0x8a,
    0x81, 0x9c, 0x82, 0xb9, 0x30, 0x10, 0x17, 0x81,
    0x8a, 0x81, 0x8e, 0x80, 0x8b, 0x83, 0xb9, 0x30,
    0x10, 0x82, 0x89, 0x80, 0x89, 0x81, 0x9c, 0x82,
    0xca, 0x28, 0x00, 0x87, 0x91, 0x81, 0xbc, 0x01,
    0x86, 0x91, 0x80, 0xe2, 0x01, 0x28, 0x81, 0x8f,
    0x80, 0x40, 0xa2, 0x92, 0x88, 0x8a, 0x80, 0xa3,
    0xed, 0x8b, 0x00, 0x0b, 0x96, 0x1b, 0x10, 0x11,
    0x32, 0x83, 0x8c, 0x8b, 0x00, 0x89, 0x83, 0x46,
    0x73, 0x81, 0x9d, 0x81, 0x9d, 0x81, 0x9d, 0x81,
    0xc1, 0x92, 0x40, 0xbb, 0x81, 0xa1, 0x80, 0xf5,
    0x8b, 0x83, 0x88, 0x40, 0xdd, 0x84, 0xb8, 0x89,
    0x81, 0x93, 0xc9, 0x81, 0x8a, 0x82, 0xb0, 0x84,
    0xaf, 0x8e, 0xbb, 0x82, 0x9d, 0x88, 0x09, 0xb8,
    0x8a, 0xb1, 0x92, 0x41, 0x9b, 0xa1, 0x46, 0xc0,
    0xb3, 0x48, 0xf5, 0x9f, 0x60, 0x78, 0x73, 0x87,
    0xa1, 0x81, 0x41, 0x61, 0x07, 0x80, 0x96, 0x84,
    0xd7, 0x81, 0xb1, 0x8f, 0x00, 0xb8, 0x80, 0xa5,
    0x84, 0x9b, 0x8b, 0xac, 0x83, 0xaf, 0x8b, 0xa4,
    0x80, 0xc2, 0x8d, 0x8b, 0x07, 0x81, 0xac, 0x82,
    0xb1, 0x00, 0x11, 0x0c, 0x80, 0xab, 0x24, 0x80,
    0x40, 0xec, 0x87, 0x60, 0x4f, 0x32, 0x80, 0x48,
    0x56, 0x84, 0x46, 0x85, 0x10, 0x0c, 0x83, 0x43,
    0x13, 0x83, 0xc0, 0x80, 0x41, 0x40, 0x81, 0xce,
    0x80, 0x41, 0x02, 0x82, 0xb4, 0x8d, 0xac, 0x81,
    0x8a, 0x82, 0xac, 0x88, 0x88, 0x80, 0xbc, 0x82,
    0xa3, 0x8b, 0x91, 0x81, 0xb8, 0x82, 0xaf, 0x8c,
    0x8d, 0x81, 0xdb, 0x88, 0x08, 0x28, 0x08, 0x40,
    0x9c, 0x89, 0x96, 0x83, 0xb9, 0x31, 0x09, 0x81,
    0x89, 0x80, 0x89, 0x81, 0xd3, 0x88, 0x00, 0x08,
    0x03, 0x01, 0xe6, 0x8c, 0x02, 0xe9, 0x91, 0x40,
    0xec, 0x31, 0x86, 0x9c, 0x81, 0xd1, 0x8e, 0x00,
    0xe9, 0x8a, 0xe6, 0x8d, 0x41, 0x00, 0x8c, 0x40,
    0xf6, 0x28, 0x09, 0x0a, 0x00, 0x80, 0x40, 0x8d,
    0x31, 0x2b, 0x80, 0x9b, 0x89, 0xa9, 0x20, 0x83,
    0x91, 0x8a, 0xad, 0x8d, 0x41, 0x96, 0x38, 0x86,
    0xd2, 0x95, 0x80, 0x8d, 0xf9, 0x2a, 0x00, 0x08,
    0x10, 0x02, 0x80, 0xc1, 0x20, 0x08, 0x83, 0x41,
    0x5b, 0x83, 0x88, 0x08, 0x80, 0xaf, 0x32, 0x82,
    0x60, 0x41, 0xdc, 0x90, 0x4e, 0x1f, 0x00, 0xb6,
    0x33, 0xdc, 0x81, 0x60, 0x4c, 0xab, 0x80, 0x60,
    0x23, 0x60, 0x30, 0x90, 0x0e, 0x01, 0x04, 0xe3,
    0x80, 0x48, 0xb6, 0x80, 0x47, 0xe7, 0x99, 0x85,
    0x99, 0x85, 0x99,
};

static const uint8_t unicode_prop_Other_Lowercase_table[69] = {
    0x40, 0xa9, 0x80, 0x8e, 0x80, 0x41, 0xf4, 0x88,
    0x31, 0x9d, 0x84, 0xdf, 0x80, 0xb3, 0x80, 0x4d,
    0x80, 0x80, 0x4c, 0x2e, 0xbe, 0x8c, 0x80, 0xa1,
    0xa4, 0x42, 0xb0, 0x80, 0x8c, 0x80, 0x8f, 0x8c,
    0x40, 0xd2, 0x8f, 0x43, 0x4f, 0x99, 0x47, 0x91,
    0x81, 0x60, 0x7a, 0x1d, 0x81, 0x40, 0xd1, 0x80,
    0x40, 0x80, 0x12, 0x81, 0x43, 0x61, 0x83, 0x88,
    0x80, 0x60, 0x5c, 0x15, 0x01, 0x10, 0xa9, 0x80,
    0x88, 0x60, 0xd8, 0x74, 0xbd,
};

static const uint8_t unicode_prop_Other_Uppercase_table[15] = {
    0x60, 0x21, 0x5f, 0x8f, 0x43, 0x45, 0x99, 0x61,
    0xcc, 0x5f, 0x99, 0x85, 0x99, 0x85, 0x99,
};

static const uint8_t unicode_prop_Other_Grapheme_Extend_table[112] = {
    0x49, 0xbd, 0x80, 0x97, 0x80, 0x41, 0x65, 0x80,
    0x97, 0x80, 0xe5, 0x80, 0x97, 0x80, 0x40, 0xe7,
    0x00, 0x03, 0x08, 0x81, 0x88, 0x81, 0xe6, 0x80,
    0x97, 0x80, 0xf6, 0x80, 0x8e, 0x80, 0x49, 0x34,
    0x80, 0x9d, 0x80, 0x43, 0xff, 0x04, 0x00, 0x04,
    0x81, 0xe4, 0x80, 0xc6, 0x81, 0x44, 0x17, 0x80,
    0x50, 0x20, 0x81, 0x60, 0x79, 0x22, 0x80, 0xeb,
    0x80, 0x60, 0x55, 0xdc, 0x81, 0x52, 0x1f, 0x80,
    0xf3, 0x80, 0x41, 0x07, 0x80, 0x8d, 0x80, 0x88,
    0x80, 0xdf, 0x80, 0x88, 0x01, 0x00, 0x14, 0x80,
    0x40, 0xdf, 0x80, 0x8b, 0x80, 0x40, 0xf0, 0x80,
    0x41, 0x05, 0x80, 0x42, 0x78, 0x80, 0x8b, 0x80,
    0x46, 0x02, 0x80, 0x60, 0x50, 0xad, 0x81, 0x60,
    0x61, 0x72, 0x0d, 0x85, 0x6c, 0x2e, 0xac, 0xdf,
};

static const uint8_t unicode_prop_Other_Default_Ignorable_Code_Point_table[32] = {
    0x43, 0x4e, 0x80, 0x4e, 0x0e, 0x81, 0x46, 0x52,
    0x81, 0x48, 0xae, 0x80, 0x50, 0xfd, 0x80, 0x60,
    0xce, 0x3a, 0x80, 0xce, 0x88, 0x6d, 0x00, 0x06,
    0x00, 0x9d, 0xdf, 0xff, 0x40, 0xef, 0x4e, 0x0f,
};

static const uint8_t unicode_prop_Other_ID_Start_table[11] = {
    0x58, 0x84, 0x81, 0x48, 0x90, 0x80, 0x94, 0x80,
    0x4f, 0x6b, 0x81,
};

static const uint8_t unicode_prop_Other_ID_Continue_table[22] = {
    0x40, 0xb6, 0x80, 0x42, 0xce, 0x80, 0x4f, 0xe0,
    0x88, 0x46, 0x67, 0x80, 0x46, 0x30, 0x81, 0x50,
    0xec, 0x80, 0x60, 0xce, 0x68, 0x80,
};

static const uint8_t unicode_prop_Prepended_Concatenation_Mark_table[19] = {
    0x45, 0xff, 0x85, 0x40, 0xd6, 0x80, 0xb0, 0x80,
    0x41, 0x7f, 0x81, 0xcf, 0x80, 0x61, 0x07, 0xd9,
    0x80, 0x8e, 0x80,
};

static const uint8_t unicode_prop_XID_Start1_table[31] = {
    0x43, 0x79, 0x80, 0x4a, 0xb7, 0x80, 0xfe, 0x80,
    0x60, 0x21, 0xe6, 0x81, 0x60, 0xcb, 0xc0, 0x85,
    0x41, 0x95, 0x81, 0xf3, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x80, 0x41, 0x1e, 0x81,
};

static const uint8_t unicode_prop_XID_Continue1_table[23] = {
    0x43, 0x79, 0x80, 0x60, 0x2d, 0x1f, 0x81, 0x60,
    0xcb, 0xc0, 0x85, 0x41, 0x95, 0x81, 0xf3, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
};

static const uint8_t unicode_prop_Changes_When_Titlecased1_table[22] = {
    0x41, 0xc3, 0x08, 0x08, 0x81, 0xa4, 0x81, 0x4e,
    0xdc, 0xaa, 0x0a, 0x4e, 0x87, 0x3f, 0x3f, 0x87,
    0x8b, 0x80, 0x8e, 0x80, 0xae, 0x80,
};

static const uint8_t unicode_prop_Changes_When_Casefolded1_table[29] = {
    0x41, 0xef, 0x80, 0x41, 0x9e, 0x80, 0x9e, 0x80,
    0x5a, 0xe4, 0x83, 0x40, 0xb5, 0x00, 0x00, 0x00,
    0x80, 0xde, 0x06, 0x06, 0x80, 0x8a, 0x09, 0x81,
    0x89, 0x10, 0x81, 0x8d, 0x80,
};

static const uint8_t unicode_prop_Changes_When_NFKC_Casefolded1_table[450] = {
    0x40, 0x9f, 0x06, 0x00, 0x01, 0x00, 0x01, 0x12,
    0x10, 0x82, 0xf3, 0x80, 0x8b, 0x80, 0x40, 0x84,
    0x01, 0x01, 0x80, 0xa2, 0x01, 0x80, 0x40, 0xbb,
    0x88, 0x9e, 0x29, 0x84, 0xda, 0x08, 0x81, 0x89,
    0x80, 0xa3, 0x04, 0x02, 0x04, 0x08, 0x07, 0x80,
    0x9e, 0x80, 0xa0, 0x82, 0x9c, 0x80, 0x42, 0x28,
    0x80, 0xd7, 0x83, 0x42, 0xde, 0x87, 0xfb, 0x08,
    0x80, 0xd2, 0x01, 0x80, 0xa1, 0x11, 0x80, 0x40,
    0xfc, 0x81, 0x42, 0xd4, 0x80, 0xfe, 0x80, 0xa7,
    0x81, 0xad, 0x80, 0xb5, 0x80, 0x88, 0x03, 0x03,
    0x03, 0x80, 0x8b, 0x80, 0x88, 0x00, 0x26, 0x80,
    0x90, 0x80, 0x88, 0x03, 0x03, 0x03, 0x80, 0x8b,
    0x80, 0x41, 0x41, 0x80, 0xe1, 0x81, 0x46, 0x52,
    0x81, 0xd4, 0x84, 0x45, 0x1b, 0x10, 0x8a, 0x80,
    0x91, 0x80, 0x9b, 0x8c, 0x80, 0xa1, 0xa4, 0x40,
    0xd5, 0x83, 0x40, 0xb5, 0x00, 0x00, 0x00, 0x80,
    0x99, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
    0xb7, 0x05, 0x00, 0x13, 0x05, 0x11, 0x02, 0x0c,
    0x11, 0x00, 0x00, 0x0c, 0x15, 0x05, 0x08, 0x8f,
    0x00, 0x20, 0x8b, 0x12, 0x2a, 0x08, 0x0b, 0x00,
    0x07, 0x82, 0x8c, 0x06, 0x92, 0x81, 0x9a, 0x80,
    0x8c, 0x8a, 0x80, 0xd6, 0x18, 0x10, 0x8a, 0x01,
    0x0c, 0x0a, 0x00, 0x10, 0x11, 0x02, 0x06, 0x05,
    0x1c, 0x85, 0x8f, 0x8f, 0x8f, 0x88, 0x80, 0x40,
    0xa1, 0x08, 0x81, 0x40, 0xf7, 0x81, 0x41, 0x34,
    0xd5, 0x99, 0x9a, 0x45, 0x20, 0x80, 0xe6, 0x82,
    0xe4, 0x80, 0x41, 0x9e, 0x81, 0x40, 0xf0, 0x80,
    0x41, 0x2e, 0x80, 0xd2, 0x80, 0x8b, 0x40, 0xd5,
    0xa9, 0x80, 0xb4, 0x00, 0x82, 0xdf, 0x09, 0x80,
    0xde, 0x80, 0xb0, 0xdd, 0x82, 0x8d, 0xdf, 0x9e,
    0x80, 0xa7, 0x87, 0xae, 0x80, 0x41, 0x7f, 0x60,
    0x72, 0x9b, 0x81, 0x40, 0xd1, 0x80, 0x40, 0x80,
    0x12, 0x81, 0x43, 0x61, 0x83, 0x88, 0x80, 0x60,
    0x4d, 0x95, 0x41, 0x0d, 0x08, 0x00, 0x81, 0x89,
    0x00, 0x00, 0x09, 0x82, 0xc3, 0x81, 0xe9, 0xc2,
    0x00, 0x97, 0x04, 0x00, 0x01, 0x01, 0x80, 0xeb,
    0xa0, 0x41, 0x6a, 0x91, 0xbf, 0x81, 0xb5, 0xa7,
    0x8c, 0x82, 0x99, 0x95, 0x94, 0x81, 0x8b, 0x80,
    0x92, 0x03, 0x1a, 0x00, 0x80, 0x40, 0x86, 0x08,
    0x80, 0x9f, 0x99, 0x40, 0x83, 0x15, 0x0d, 0x0d,
    0x0a, 0x16, 0x06, 0x80, 0x88, 0x47, 0x87, 0x20,
    0xa9, 0x80, 0x88, 0x60, 0xb4, 0xe4, 0x83, 0x50,
    0x31, 0xa3, 0x44, 0x63, 0x86, 0x8d, 0x87, 0xbf,
    0x85, 0x42, 0x3e, 0xd4, 0x80, 0xc6, 0x01, 0x08,
    0x09, 0x0b, 0x80, 0x8b, 0x00, 0x06, 0x80, 0xc0,
    0x03, 0x0f, 0x06, 0x80, 0x9b, 0x03, 0x04, 0x00,
    0x16, 0x80, 0x41, 0x53, 0x81, 0x41, 0x23, 0x81,
    0xb1, 0x48, 0x2f, 0xbd, 0x4d, 0x91, 0x18, 0x9a,
    0x01, 0x00, 0x08, 0x80, 0x89, 0x03, 0x00, 0x00,
    0x28, 0x18, 0x00, 0x00, 0x02, 0x01, 0x00, 0x08,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x0b, 0x06,
    0x03, 0x03, 0x00, 0x80, 0x89, 0x80, 0x90, 0x22,
    0x04, 0x80, 0x90, 0x42, 0x43, 0x8a, 0x84, 0x9e,
    0x80, 0x9f, 0x99, 0x82, 0xa2, 0x80, 0xee, 0x82,
    0x8c, 0xab, 0x83, 0x88, 0x31, 0x49, 0x9d, 0x89,
    0x60, 0xfc, 0x05, 0x42, 0x1d, 0x6b, 0x05, 0xe1,
    0x4f, 0xff,
};

static const uint8_t unicode_prop_ASCII_Hex_Digit_table[5] = {
    0xaf, 0x89, 0x35, 0x99, 0x85,
};

static const uint8_t unicode_prop_Bidi_Control_table[10] = {
    0x46, 0x1b, 0x80, 0x59, 0xf0, 0x81, 0x99, 0x84,
    0xb6, 0x83,
};

static const uint8_t unicode_prop_Dash_table[58] = {
    0xac, 0x80, 0x45, 0x5b, 0x80, 0xb2, 0x80, 0x4e,
    0x40, 0x80, 0x44, 0x04, 0x80, 0x48, 0x08, 0x85,
    0xbc, 0x80, 0xa6, 0x80, 0x8e, 0x80, 0x41, 0x85,
    0x80, 0x4c, 0x03, 0x01, 0x80, 0x9e, 0x0b, 0x80,
    0x9b, 0x80, 0x41, 0xbd, 0x80, 0x92, 0x80, 0xee,
    0x80, 0x60, 0xcd, 0x8f, 0x81, 0xa4, 0x80, 0x89,
    0x80, 0x40, 0xa8, 0x80, 0x4e, 0x5f, 0x80, 0x41,
    0x3d, 0x80,
};

static const uint8_t unicode_prop_Deprecated_table[23] = {
    0x41, 0x48, 0x80, 0x45, 0x28, 0x80, 0x49, 0x02,
    0x00, 0x80, 0x48, 0x28, 0x81, 0x48, 0xc4, 0x85,
    0x42, 0xb8, 0x81, 0x6d, 0xdc, 0xd5, 0x80,
};

static const uint8_t unicode_prop_Diacritic_table[438] = {
    0xdd, 0x00, 0x80, 0xc6, 0x05, 0x03, 0x01, 0x81,
    0x41, 0xf6, 0x40, 0x9e, 0x07, 0x25, 0x90, 0x0b,
    0x80, 0x88, 0x81, 0x40, 0xfc, 0x84, 0x40, 0xd0,
    0x80, 0xb6, 0x90, 0x80, 0x9a, 0x00, 0x01, 0x00,
    0x40, 0x85, 0x3b, 0x81, 0x40, 0x85, 0x0b, 0x0a,
    0x82, 0xc2, 0x9a, 0xda, 0x8a, 0xb9, 0x8a, 0xa1,
    0x81, 0xfd, 0x87, 0xa8, 0x89, 0x8f, 0x9b, 0xbc,
    0x80, 0x8f, 0x02, 0x83, 0x9b, 0x80, 0xc9, 0x80,
    0x8f, 0x80, 0xed, 0x80, 0x8f, 0x80, 0xed, 0x80,
    0x8f, 0x80, 0xae, 0x82, 0xbb, 0x80, 0x8f, 0x06,
    0x80, 0xf6, 0x80, 0xed, 0x80, 0x8f, 0x80, 0xed,
    0x80, 0x8f, 0x80, 0xec, 0x81, 0x8f, 0x80, 0xfb,
    0x80, 0xee, 0x80, 0x8b, 0x28, 0x80, 0xea, 0x80,
    0x8c, 0x84, 0xca, 0x81, 0x9a, 0x00, 0x00, 0x03,
    0x81, 0xc1, 0x10, 0x81, 0xbd, 0x80, 0xef, 0x00,
    0x81, 0xa7, 0x0b, 0x84, 0x98, 0x30, 0x80, 0x89,
    0x81, 0x42, 0xc0, 0x82, 0x43, 0xb3, 0x81, 0x9d,
    0x80, 0x40, 0x93, 0x8a, 0x88, 0x80, 0x41, 0x5a,
    0x82, 0x41, 0x23, 0x80, 0x93, 0x39, 0x80, 0xaf,
    0x8e, 0x81, 0x8a, 0xe7, 0x80, 0x8e, 0x80, 0xa5,
    0x88, 0xb5, 0x81, 0xb9, 0x80, 0x8a, 0x81, 0xc1,
    0x81, 0xbf, 0x85, 0xd1, 0x98, 0x18, 0x28, 0x0a,
    0xb1, 0xbe, 0xd8, 0x8b, 0xa4, 0x8a, 0x41, 0xbc,
    0x00, 0x82, 0x8a, 0x82, 0x8c, 0x82, 0x8c, 0x82,
    0x8c, 0x81, 0x4c, 0xef, 0x82, 0x41, 0x3c, 0x80,
    0x41, 0xf9, 0x85, 0xe8, 0x83, 0xde, 0x80, 0x60,
    0x75, 0x71, 0x80, 0x8b, 0x08, 0x80, 0x9b, 0x81,
    0xd1, 0x81, 0x8d, 0xa1, 0xe5, 0x82, 0xec, 0x81,
    0x8b, 0x80, 0xa4, 0x80, 0x40, 0x96, 0x80, 0x9a,
    0x91, 0xb8, 0x83, 0xa3, 0x80, 0xde, 0x80, 0x8b,
    0x80, 0xa3, 0x80, 0x40, 0x94, 0x82, 0xc0, 0x83,
    0xb2, 0x80, 0xe3, 0x84, 0x88, 0x82, 0xff, 0x81,
    0x60, 0x4f, 0x2f, 0x80, 0x43, 0x00, 0x8f, 0x41,
    0x0d, 0x00, 0x80, 0xae, 0x80, 0xac, 0x81, 0xc2,
    0x80, 0x42, 0xfb, 0x80, 0x44, 0x9e, 0x28, 0xa9,
    0x80, 0x88, 0x42, 0x7c, 0x13, 0x80, 0x40, 0xa4,
    0x81, 0x42, 0x3a, 0x85, 0xa5, 0x80, 0x99, 0x84,
    0x41, 0x8e, 0x82, 0xc5, 0x8a, 0xb0, 0x83, 0x40,
    0xbf, 0x80, 0xa8, 0x80, 0xc7, 0x81, 0xf7, 0x81,
    0xbd, 0x80, 0xcb, 0x80, 0x88, 0x82, 0xe7, 0x81,
    0x40, 0xb1, 0x81, 0xcf, 0x81, 0x8f, 0x80, 0x97,
    0x32, 0x84, 0xd8, 0x10, 0x81, 0x8c, 0x81, 0xde,
    0x02, 0x80, 0xfa, 0x81, 0x40, 0xfa, 0x81, 0xfd,
    0x80, 0xf5, 0x81, 0xf2, 0x80, 0x41, 0x0c, 0x81,
    0x41, 0x01, 0x0b, 0x80, 0x40, 0x9b, 0x80, 0xd2,
    0x80, 0x91, 0x80, 0xd0, 0x80, 0x41, 0xa4, 0x80,
    0x41, 0x01, 0x00, 0x81, 0xd0, 0x80, 0x41, 0xa8,
    0x81, 0x96, 0x80, 0x54, 0xeb, 0x8e, 0x60, 0x2c,
    0xd8, 0x80, 0x49, 0xbf, 0x84, 0xba, 0x86, 0x42,
    0x33, 0x81, 0x42, 0x21, 0x90, 0xcf, 0x81, 0x60,
    0x3f, 0xfd, 0x18, 0x30, 0x81, 0x5f, 0x00, 0xad,
    0x81, 0x96, 0x42, 0x1f, 0x12, 0x2f, 0x39, 0x86,
    0x9d, 0x83, 0x4e, 0x81, 0xbd, 0x40, 0xc1, 0x86,
    0x41, 0x76, 0x80, 0xbc, 0x83, 0x42, 0xfd, 0x81,
    0x42, 0xdf, 0x86, 0xec, 0x10, 0x82,
};

static const uint8_t unicode_prop_Extender_table[111] = {
    0x40, 0xb6, 0x80, 0x42, 0x17, 0x81, 0x43, 0x6d,
    0x80, 0x41, 0xb8, 0x80, 0x42, 0x75, 0x80, 0x40,
    0x88, 0x80, 0xd8, 0x80, 0x42, 0xef, 0x80, 0xfe,
    0x80, 0x49, 0x42, 0x80, 0xb7, 0x80, 0x42, 0x62,
    0x80, 0x41, 0x8d, 0x80, 0xc3, 0x80, 0x53, 0x88,
    0x80, 0xaa, 0x84, 0xe6, 0x81, 0xdc, 0x82, 0x60,
    0x6f, 0x15, 0x80, 0x45, 0xf5, 0x80, 0x43, 0xc1,
    0x80, 0x95, 0x80, 0x40, 0x88, 0x80, 0xeb, 0x80,
    0x94, 0x81, 0x60, 0x54, 0x7a, 0x80, 0x48, 0x0f,
    0x81, 0x45, 0xca, 0x80, 0x9a, 0x03, 0x80, 0x44,
    0xc6, 0x80, 0x41, 0x24, 0x80, 0xf3, 0x81, 0x41,
    0xf1, 0x82, 0x44, 0xce, 0x80, 0x60, 0x50, 0xa8,
    0x81, 0x44, 0x9b, 0x08, 0x80, 0x60, 0x71, 0x57,
    0x81, 0x44, 0xb0, 0x80, 0x43, 0x53, 0x82,
};

static const uint8_t unicode_prop_Hex_Digit_table[12] = {
    0xaf, 0x89, 0x35, 0x99, 0x85, 0x60, 0xfe, 0xa8,
    0x89, 0x35, 0x99, 0x85,
};

static const uint8_t unicode_prop_IDS_Unary_Operator_table[4] = {
    0x60, 0x2f, 0xfd, 0x81,
};

static const uint8_t unicode_prop_IDS_Binary_Operator_table[8] = {
    0x60, 0x2f, 0xef, 0x09, 0x89, 0x41, 0xf0, 0x80,
};

static const uint8_t unicode_prop_IDS_Trinary_Operator_table[4] = {
    0x60, 0x2f, 0xf1, 0x81,
};

static const uint8_t unicode_prop_Ideographic_table[72] = {
    0x60, 0x30, 0x05, 0x81, 0x98, 0x88, 0x8d, 0x82,
    0x43, 0xc4, 0x59, 0xbf, 0xbf, 0x60, 0x51, 0xff,
    0x60, 0x58, 0xff, 0x41, 0x6d, 0x81, 0xe9, 0x60,
    0x75, 0x09, 0x80, 0x9a, 0x57, 0xf7, 0x87, 0x44,
    0xd5, 0xa8, 0x89, 0x60, 0x24, 0x66, 0x41, 0x8b,
    0x60, 0x4d, 0x03, 0x60, 0xa6, 0xdf, 0x9f, 0x50,
    0x39, 0x85, 0x40, 0xdd, 0x81, 0x56, 0x81, 0x8d,
    0x5d, 0x30, 0x8e, 0x42, 0x6d, 0x49, 0xa1, 0x42,
    0x1d, 0x45, 0xe1, 0x53, 0x4a, 0x84, 0x50, 0x5f,
};

static const uint8_t unicode_prop_Join_Control_table[4] = {
    0x60, 0x20, 0x0b, 0x81,
};

static const uint8_t unicode_prop_Logical_Order_Exception_table[15] = {
    0x4e, 0x3f, 0x84, 0xfa, 0x84, 0x4a, 0xef, 0x11,
    0x80, 0x60, 0x90, 0xf9, 0x09, 0x00, 0x81,
};

static const uint8_t unicode_prop_Modifier_Combining_Mark_table[16] = {
    0x46, 0x53, 0x09, 0x80, 0x40, 0x82, 0x05, 0x02,
    0x81, 0x41, 0xe0, 0x08, 0x12, 0x80, 0x9e, 0x80,
};

static const uint8_t unicode_prop_Noncharacter_Code_Point_table[71] = {
    0x60, 0xfd, 0xcf, 0x9f, 0x42, 0x0d, 0x81, 0x60,
    0xff, 0xfd, 0x81, 0x60, 0xff, 0xfd, 0x81, 0x60,
    0xff, 0xfd, 0x81, 0x60, 0xff, 0xfd, 0x81, 0x60,
    0xff, 0xfd, 0x81, 0x60, 0xff, 0xfd, 0x81, 0x60,
    0xff, 0xfd, 0x81, 0x60, 0xff, 0xfd, 0x81, 0x60,
    0xff, 0xfd, 0x81, 0x60, 0xff, 0xfd, 0x81, 0x60,
    0xff, 0xfd, 0x81, 0x60, 0xff, 0xfd, 0x81, 0x60,
    0xff, 0xfd, 0x81, 0x60, 0xff, 0xfd, 0x81, 0x60,
    0xff, 0xfd, 0x81, 0x60, 0xff, 0xfd, 0x81,
};

static const uint8_t unicode_prop_Pattern_Syntax_table[58] = {
    0xa0, 0x8e, 0x89, 0x86, 0x99, 0x18, 0x80, 0x99,
    0x83, 0xa1, 0x30, 0x00, 0x08, 0x00, 0x0b, 0x03,
    0x02, 0x80, 0x96, 0x80, 0x9e, 0x80, 0x5f, 0x17,
    0x97, 0x87, 0x8e, 0x81, 0x92, 0x80, 0x89, 0x41,
    0x30, 0x42, 0xcf, 0x40, 0x9f, 0x42, 0x75, 0x9d,
    0x44, 0x6b, 0x41, 0xff, 0xff, 0x41, 0x80, 0x13,
    0x98, 0x8e, 0x80, 0x60, 0xcd, 0x0c, 0x81, 0x41,
    0x04, 0x81,
};

static const uint8_t unicode_prop_Pattern_White_Space_table[11] = {
    0x88, 0x84, 0x91, 0x80, 0xe3, 0x80, 0x5f, 0x87,
    0x81, 0x97, 0x81,
};

static const uint8_t unicode_prop_Quotation_Mark_table[31] = {
    0xa1, 0x03, 0x80, 0x40, 0x82, 0x80, 0x8e, 0x80,
    0x5f, 0x5b, 0x87, 0x98, 0x81, 0x4e, 0x06, 0x80,
    0x41, 0xc8, 0x83, 0x8c, 0x82, 0x60, 0xce, 0x20,
    0x83, 0x40, 0xbc, 0x03, 0x80, 0xd9, 0x81,
};

static const uint8_t unicode_prop_Radical_table[9] = {
    0x60, 0x2e, 0x7f, 0x99, 0x80, 0xd8, 0x8b, 0x40,
    0xd5,
};

static const uint8_t unicode_prop_Regional_Indicator_table[4] = {
    0x61, 0xf1, 0xe5, 0x99,
};

static const uint8_t unicode_prop_Sentence_Terminal_table[213] = {
    0xa0, 0x80, 0x8b, 0x80, 0x8f, 0x80, 0x45, 0x48,
    0x80, 0x40, 0x92, 0x82, 0x40, 0xb3, 0x80, 0xaa,
    0x82, 0x40, 0xf5, 0x80, 0xbc, 0x00, 0x02, 0x81,
    0x41, 0x24, 0x81, 0x46, 0xe3, 0x81, 0x43, 0x15,
    0x03, 0x81, 0x43, 0x04, 0x80, 0x40, 0xc5, 0x81,
    0x40, 0x9c, 0x81, 0xac, 0x04, 0x80, 0x41, 0x39,
    0x81, 0x41, 0x61, 0x83, 0x40, 0xa1, 0x81, 0x89,
    0x09, 0x81, 0x9c, 0x82, 0x40, 0xba, 0x81, 0xc0,
    0x81, 0x43, 0xa3, 0x80, 0x96, 0x81, 0x88, 0x82,
    0x4c, 0xae, 0x82, 0x41, 0x31, 0x80, 0x8c, 0x80,
    0x95, 0x81, 0x41, 0xac, 0x80, 0x60, 0x74, 0xfb,
    0x80, 0x41, 0x0d, 0x81, 0x40, 0xe2, 0x02, 0x80,
    0x41, 0x7d, 0x81, 0xd5, 0x81, 0xde, 0x80, 0x40,
    0x97, 0x81, 0x40, 0x92, 0x82, 0x40, 0x8f, 0x81,
    0x40, 0xf8, 0x80, 0x60, 0x52, 0x25, 0x01, 0x81,
    0xba, 0x02, 0x81, 0x40, 0xa8, 0x80, 0x8b, 0x80,
    0x8f, 0x80, 0xc0, 0x80, 0x4a, 0xf3, 0x81, 0x44,
    0xfc, 0x84, 0xab, 0x83, 0x40, 0xbc, 0x81, 0xf4,
    0x83, 0xfe, 0x82, 0x40, 0x80, 0x0d, 0x80, 0x8f,
    0x81, 0xd7, 0x08, 0x81, 0xeb, 0x80, 0x41, 0x29,
    0x81, 0xf4, 0x81, 0x41, 0x74, 0x0c, 0x8e, 0xe8,
    0x81, 0x40, 0xf8, 0x82, 0x42, 0x04, 0x00, 0x80,
    0x40, 0xfa, 0x81, 0xd6, 0x81, 0x41, 0xa3, 0x81,
    0x42, 0xb3, 0x81, 0xc9, 0x81, 0x60, 0x4b, 0x28,
    0x81, 0x40, 0x84, 0x80, 0xc0, 0x81, 0x8a, 0x80,
    0x42, 0x28, 0x81, 0x41, 0x27, 0x80, 0x60, 0x4e,
    0x05, 0x80, 0x5d, 0xe7, 0x80,
};

static const uint8_t unicode_prop_Soft_Dotted_table[79] = {
    0xe8, 0x81, 0x40, 0xc3, 0x80, 0x41, 0x18, 0x80,
    0x9d, 0x80, 0xb3, 0x80, 0x93, 0x80, 0x41, 0x3f,
    0x80, 0xe1, 0x00, 0x80, 0x59, 0x08, 0x80, 0xb2,
    0x80, 0x8c, 0x02, 0x80, 0x40, 0x83, 0x80, 0x40,
    0x9c, 0x80, 0x41, 0xa4, 0x80, 0x40, 0xd5, 0x81,
    0x4b, 0x31, 0x80, 0x61, 0xa7, 0xa4, 0x81, 0xb1,
    0x81, 0xb1, 0x81, 0xb1, 0x81, 0xb1, 0x81, 0xb1,
    0x81, 0xb1, 0x81, 0xb1, 0x81, 0xb1, 0x81, 0xb1,
    0x81, 0xb1, 0x81, 0xb1, 0x81, 0xb1, 0x81, 0x48,
    0x85, 0x80, 0x41, 0x30, 0x81, 0x99, 0x80,
};

static const uint8_t unicode_prop_Terminal_Punctuation_table[264] = {
    0xa0, 0x80, 0x89, 0x00, 0x80, 0x8a, 0x0a, 0x80,
    0x43, 0x3d, 0x07, 0x80, 0x42, 0x00, 0x80, 0xb8,
    0x80, 0xc7, 0x80, 0x8d, 0x00, 0x82, 0x40, 0xb3,
    0x80, 0xaa, 0x8a, 0x00, 0x40, 0xea, 0x81, 0xb5,
    0x28, 0x87, 0x9e, 0x80, 0x41, 0x04, 0x81, 0x44,
    0xf3, 0x81, 0x40, 0xab, 0x03, 0x85, 0x41, 0x36,
    0x81, 0x43, 0x14, 0x87, 0x43, 0x04, 0x80, 0xfb,
    0x82, 0xc6, 0x81, 0x40, 0x9c, 0x12, 0x80, 0xa6,
    0x19, 0x81, 0x41, 0x39, 0x81, 0x41, 0x61, 0x83,
    0x40, 0xa1, 0x81, 0x89, 0x08, 0x82, 0x9c, 0x82,
    0x40, 0xba, 0x84, 0xbd, 0x81, 0x43, 0xa3, 0x80,
    0x96, 0x81, 0x88, 0x82, 0x4c, 0xae, 0x82, 0x41,
    0x31, 0x80, 0x8c, 0x03, 0x80, 0x89, 0x00, 0x0a,
    0x81, 0x41, 0xab, 0x81, 0x60, 0x74, 0xfa, 0x81,
    0x41, 0x0c, 0x82, 0x40, 0xe2, 0x84, 0x41, 0x7d,
    0x81, 0xd5, 0x81, 0xde, 0x80, 0x40, 0x96, 0x82,
    0x40, 0x92, 0x82, 0xfe, 0x80, 0x8f, 0x81, 0x40,
    0xf8, 0x80, 0x60, 0x52, 0x25, 0x01, 0x81, 0xb8,
    0x10, 0x83, 0x40, 0xa8, 0x80, 0x89, 0x00, 0x80,
    0x8a, 0x0a, 0x80, 0xc0, 0x01, 0x80, 0x44, 0x39,
    0x80, 0xaf, 0x80, 0x44, 0x85, 0x80, 0x40, 0xc6,
    0x80, 0x41, 0x35, 0x81, 0x40, 0x97, 0x85, 0xc3,
    0x85, 0xd8, 0x83, 0x43, 0xb7, 0x84, 0xab, 0x83,
    0x40, 0xbc, 0x86, 0xef, 0x83, 0xfe, 0x82, 0x40,
    0x80, 0x0d, 0x80, 0x8f, 0x81, 0xd7, 0x84, 0xeb,
    0x80, 0x41, 0x29, 0x81, 0xf4, 0x82, 0x8b, 0x81,
    0x41, 0x65, 0x1a, 0x8e, 0xe8, 0x81, 0x40, 0xf8,
    0x82, 0x42, 0x04, 0x00, 0x80, 0x40, 0xfa, 0x81,
    0xd6, 0x0b, 0x81, 0x41, 0x9d, 0x82, 0xac, 0x80,
    0x42, 0x84, 0x81, 0xc9, 0x81, 0x45, 0x2a, 0x84,
    0x60, 0x45, 0xf8, 0x81, 0x40, 0x84, 0x80, 0xc0,
    0x82, 0x89, 0x80, 0x42, 0x28, 0x81, 0x41, 0x26,
    0x81, 0x60, 0x4e, 0x05, 0x80, 0x5d, 0xe6, 0x83,
};

static const uint8_t unicode_prop_Unified_Ideograph_table[48] = {
    0x60, 0x33, 0xff, 0x59, 0xbf, 0xbf, 0x60, 0x51,
    0xff, 0x60, 0x5a, 0x0d, 0x08, 0x00, 0x81, 0x89,
    0x00, 0x00, 0x09, 0x82, 0x61, 0x05, 0xd5, 0x60,
    0xa6, 0xdf, 0x9f, 0x50, 0x39, 0x85, 0x40, 0xdd,
    0x81, 0x56, 0x81, 0x8d, 0x5d, 0x30, 0x8e, 0x42,
    0x6d, 0x51, 0xa1, 0x53, 0x4a, 0x84, 0x50, 0x5f,
};

static const uint8_t unicode_prop_Variation_Selector_table[13] = {
    0x58, 0x0a, 0x10, 0x80, 0x60, 0xe5, 0xef, 0x8f,
    0x6d, 0x02, 0xef, 0x40, 0xef,
};

static const uint8_t unicode_prop_Bidi_Mirrored_table[173] = {
    0xa7, 0x81, 0x91, 0x00, 0x80, 0x9b, 0x00, 0x80,
    0x9c, 0x00, 0x80, 0xac, 0x80, 0x8e, 0x80, 0x4e,
    0x7d, 0x83, 0x47, 0x5c, 0x81, 0x49, 0x9b, 0x81,
    0x89, 0x81, 0xb5, 0x81, 0x8d, 0x81, 0x40, 0xb0,
    0x80, 0x40, 0xbf, 0x1a, 0x2a, 0x02, 0x0a, 0x18,
    0x18, 0x00, 0x03, 0x88, 0x20, 0x80, 0x91, 0x23,
    0x88, 0x08, 0x00, 0x38, 0x9f, 0x0b, 0x20, 0x88,
    0x09, 0x92, 0x21, 0x88, 0x21, 0x0b, 0x97, 0x81,
    0x8f, 0x3b, 0x93, 0x0e, 0x81, 0x44, 0x3c, 0x8d,
    0xc9, 0x01, 0x18, 0x08, 0x14, 0x1c, 0x12, 0x8d,
    0x41, 0x92, 0x95, 0x0d, 0x80, 0x8d, 0x38, 0x35,
    0x10, 0x1c, 0x01, 0x0c, 0x18, 0x02, 0x09, 0x89,
    0x29, 0x81, 0x8b, 0x92, 0x03, 0x08, 0x00, 0x08,
    0x03, 0x21, 0x2a, 0x97, 0x81, 0x8a, 0x0b, 0x18,
    0x09, 0x0b, 0xaa, 0x0f, 0x80, 0xa7, 0x20, 0x00,
    0x14, 0x22, 0x18, 0x14, 0x00, 0x40, 0xff, 0x80,
    0x42, 0x02, 0x1a, 0x08, 0x81, 0x8d, 0x09, 0x89,
    0xaa, 0x87, 0x41, 0xaa, 0x89, 0x0f, 0x60, 0xce,
    0x3c, 0x2c, 0x81, 0x40, 0xa1, 0x81, 0x91, 0x00,
    0x80, 0x9b, 0x00, 0x80, 0x9c, 0x00, 0x00, 0x08,
    0x81, 0x60, 0xd7, 0x76, 0x80, 0xb8, 0x80, 0xb8,
    0x80, 0xb8, 0x80, 0xb8, 0x80,
};

static const uint8_t unicode_prop_Emoji_table[238] = {
    0xa2, 0x05, 0x04, 0x89, 0xee, 0x03, 0x80, 0x5f,
    0x8c, 0x80, 0x8b, 0x80, 0x40, 0xd7, 0x80, 0x95,
    0x80, 0xd9, 0x85, 0x8e, 0x81, 0x41, 0x6e, 0x81,
    0x8b, 0x80, 0x40, 0xa5, 0x80, 0x98, 0x8a, 0x1a,
    0x40, 0xc6, 0x80, 0x40, 0xe6, 0x81, 0x89, 0x80,
    0x88, 0x80, 0xb9, 0x18, 0x84, 0x88, 0x01, 0x01,
    0x09, 0x03, 0x01, 0x00, 0x09, 0x02, 0x02, 0x0f,
    0x14, 0x00, 0x04, 0x8b, 0x8a, 0x09, 0x00, 0x08,
    0x80, 0x91, 0x01, 0x81, 0x91, 0x28, 0x00, 0x0a,
    0x0c, 0x01, 0x0b, 0x81, 0x8a, 0x0c, 0x09, 0x04,
    0x08, 0x00, 0x81, 0x93, 0x0c, 0x28, 0x19, 0x03,
    0x01, 0x01, 0x28, 0x01, 0x00, 0x00, 0x05, 0x02,
    0x05, 0x80, 0x89, 0x81, 0x8e, 0x01, 0x03, 0x00,
    0x03, 0x10, 0x80, 0x8a, 0x81, 0xaf, 0x82, 0x88,
    0x80, 0x8d, 0x80, 0x8d, 0x80, 0x41, 0x73, 0x81,
    0x41, 0xce, 0x82, 0x92, 0x81, 0xb2, 0x03, 0x80,
    0x44, 0xd9, 0x80, 0x8b, 0x80, 0x42, 0x58, 0x00,
    0x80, 0x61, 0xbd, 0x69, 0x80, 0x40, 0xc9, 0x80,
    0x40, 0x9f, 0x81, 0x8b, 0x81, 0x8d, 0x01, 0x89,
    0xca, 0x99, 0x01, 0x96, 0x80, 0x93, 0x01, 0x88,
    0x94, 0x81, 0x40, 0xad, 0xa1, 0x81, 0xef, 0x09,
    0x02, 0x81, 0xd2, 0x0a, 0x80, 0x41, 0x06, 0x80,
    0xbe, 0x8a, 0x28, 0x97, 0x31, 0x0f, 0x8b, 0x01,
    0x19, 0x03, 0x81, 0x8c, 0x09, 0x07, 0x81, 0x88,
    0x04, 0x82, 0x8b, 0x17, 0x11, 0x00, 0x03, 0x05,
    0x02, 0x05, 0xd5, 0xaf, 0xc5, 0x27, 0x0a, 0x83,
    0x89, 0x10, 0x01, 0x10, 0x81, 0x89, 0x40, 0xe2,
    0x8b, 0x18, 0x41, 0x1a, 0xae, 0x80, 0x89, 0x80,
    0x40, 0xb8, 0xef, 0x8c, 0x82, 0x89, 0x84, 0xb7,
    0x86, 0x8e, 0x81, 0x8a, 0x85, 0x88,
};

static const uint8_t unicode_prop_Emoji_Component_table[28] = {
    0xa2, 0x05, 0x04, 0x89, 0x5f, 0xd2, 0x80, 0x40,
    0xd4, 0x80, 0x60, 0xdd, 0x2a, 0x80, 0x60, 0xf3,
    0xd5, 0x99, 0x41, 0xfa, 0x84, 0x45, 0xaf, 0x83,
    0x6c, 0x06, 0x6b, 0xdf,
};

static const uint8_t unicode_prop_Emoji_Modifier_table[4] = {
    0x61, 0xf3, 0xfa, 0x84,
};

static const uint8_t unicode_prop_Emoji_Modifier_Base_table[71] = {
    0x60, 0x26, 0x1c, 0x80, 0x40, 0xda, 0x80, 0x8f,
    0x83, 0x61, 0xcc, 0x76, 0x80, 0xbb, 0x11, 0x01,
    0x82, 0xf4, 0x09, 0x8a, 0x94, 0x92, 0x10, 0x1a,
    0x02, 0x30, 0x00, 0x97, 0x80, 0x40, 0xc8, 0x0b,
    0x80, 0x94, 0x03, 0x81, 0x40, 0xad, 0x12, 0x84,
    0xd2, 0x80, 0x8f, 0x82, 0x88, 0x80, 0x8a, 0x80,
    0x42, 0x3e, 0x01, 0x07, 0x3d, 0x80, 0x88, 0x89,
    0x0a, 0xb7, 0x80, 0xbc, 0x08, 0x08, 0x80, 0x90,
    0x10, 0x8c, 0x40, 0xe4, 0x82, 0xa9, 0x88,
};

static const uint8_t unicode_prop_Emoji_Presentation_table[144] = {
    0x60, 0x23, 0x19, 0x81, 0x40, 0xcc, 0x1a, 0x01,
    0x80, 0x42, 0x08, 0x81, 0x94, 0x81, 0xb1, 0x8b,
    0xaa, 0x80, 0x92, 0x80, 0x8c, 0x07, 0x81, 0x90,
    0x0c, 0x0f, 0x04, 0x80, 0x94, 0x06, 0x08, 0x03,
    0x01, 0x06, 0x03, 0x81, 0x9b, 0x80, 0xa2, 0x00,
    0x03, 0x10, 0x80, 0xbc, 0x82, 0x97, 0x80, 0x8d,
    0x80, 0x43, 0x5a, 0x81, 0xb2, 0x03, 0x80, 0x61,
    0xc4, 0xad, 0x80, 0x40, 0xc9, 0x80, 0x40, 0xbd,
    0x01, 0x89, 0xca, 0x99, 0x00, 0x97, 0x80, 0x93,
    0x01, 0x20, 0x82, 0x94, 0x81, 0x40, 0xad, 0xa0,
    0x8b, 0x88, 0x80, 0xc5, 0x80, 0x95, 0x8b, 0xaa,
    0x1c, 0x8b, 0x90, 0x10, 0x82, 0xc6, 0x00, 0x80,
    0x40, 0xba, 0x81, 0xbe, 0x8c, 0x18, 0x97, 0x91,
    0x80, 0x99, 0x81, 0x8c, 0x80, 0xd5, 0xd4, 0xaf,
    0xc5, 0x28, 0x12, 0x0a, 0x1b, 0x8a, 0x0e, 0x88,
    0x40, 0xe2, 0x8b, 0x18, 0x41, 0x1a, 0xae, 0x80,
    0x89, 0x80, 0x40, 0xb8, 0xef, 0x8c, 0x82, 0x89,
    0x84, 0xb7, 0x86, 0x8e, 0x81, 0x8a, 0x85, 0x88,
};

static const uint8_t unicode_prop_Extended_Pictographic_table[156] = {
    0x40, 0xa8, 0x03, 0x80, 0x5f, 0x8c, 0x80, 0x8b,
    0x80, 0x40, 0xd7, 0x80, 0x95, 0x80, 0xd9, 0x85,
    0x8e, 0x81, 0x41, 0x6e, 0x81, 0x8b, 0x80, 0xde,
    0x80, 0xc5, 0x80, 0x98, 0x8a, 0x1a, 0x40, 0xc6,
    0x80, 0x40, 0xe6, 0x81, 0x89, 0x80, 0x88, 0x80,
    0xb9, 0x18, 0x28, 0x8b, 0x80, 0xf1, 0x89, 0xf5,
    0x81, 0x8a, 0x00, 0x00, 0x28, 0x10, 0x28, 0x89,
    0x81, 0x8e, 0x01, 0x03, 0x00, 0x03, 0x10, 0x80,
    0x8a, 0x84, 0xac, 0x82, 0x88, 0x80, 0x8d, 0x80,
    0x8d, 0x80, 0x41, 0x73, 0x81, 0x41, 0xce, 0x82,
    0x92, 0x81, 0xb2, 0x03, 0x80, 0x44, 0xd9, 0x80,
    0x8b, 0x80, 0x42, 0x58, 0x00, 0x80, 0x61, 0xbd,
    0x65, 0x40, 0xff, 0x8c, 0x82, 0x9e, 0x80, 0xbb,
    0x85, 0x8b, 0x81, 0x8d, 0x01, 0x89, 0x91, 0xb8,
    0x9a, 0x8e, 0x89, 0x80, 0x93, 0x01, 0x88, 0x03,
    0x88, 0x41, 0xb1, 0x84, 0x41, 0x3d, 0x87, 0x41,
    0x09, 0xaf, 0xff, 0xf3, 0x8b, 0xd4, 0xaa, 0x8b,
    0x83, 0xb7, 0x87, 0x89, 0x85, 0xa7, 0x87, 0x9d,
    0xd1, 0x8b, 0xae, 0x80, 0x89, 0x80, 0x41, 0xb8,
    0x40, 0xff, 0x43, 0xfd,
};

static const uint8_t unicode_prop_Default_Ignorable_Code_Point_table[51] = {
    0x40, 0xac, 0x80, 0x42, 0xa0, 0x80, 0x42, 0xcb,
    0x80, 0x4b, 0x41, 0x81, 0x46, 0x52, 0x81, 0xd4,
    0x84, 0x47, 0xfa, 0x84, 0x99, 0x84, 0xb0, 0x8f,
    0x50, 0xf3, 0x80, 0x60, 0xcc, 0x9a, 0x8f, 0x40,
    0xee, 0x80, 0x40, 0x9f, 0x80, 0xce, 0x88, 0x60,
    0xbc, 0xa6, 0x83, 0x54, 0xce, 0x87, 0x6c, 0x2e,
    0x84, 0x4f, 0xff,
};

typedef enum {
    UNICODE_PROP_Hyphen,
    UNICODE_PROP_Other_Math,
    UNICODE_PROP_Other_Alphabetic,
    UNICODE_PROP_Other_Lowercase,
    UNICODE_PROP_Other_Uppercase,
    UNICODE_PROP_Other_Grapheme_Extend,
    UNICODE_PROP_Other_Default_Ignorable_Code_Point,
    UNICODE_PROP_Other_ID_Start,
    UNICODE_PROP_Other_ID_Continue,
    UNICODE_PROP_Prepended_Concatenation_Mark,
    UNICODE_PROP_ID_Continue1,
    UNICODE_PROP_XID_Start1,
    UNICODE_PROP_XID_Continue1,
    UNICODE_PROP_Changes_When_Titlecased1,
    UNICODE_PROP_Changes_When_Casefolded1,
    UNICODE_PROP_Changes_When_NFKC_Casefolded1,
    UNICODE_PROP_ASCII_Hex_Digit,
    UNICODE_PROP_Bidi_Control,
    UNICODE_PROP_Dash,
    UNICODE_PROP_Deprecated,
    UNICODE_PROP_Diacritic,
    UNICODE_PROP_Extender,
    UNICODE_PROP_Hex_Digit,
    UNICODE_PROP_IDS_Unary_Operator,
    UNICODE_PROP_IDS_Binary_Operator,
    UNICODE_PROP_IDS_Trinary_Operator,
    UNICODE_PROP_Ideographic,
    UNICODE_PROP_Join_Control,
    UNICODE_PROP_Logical_Order_Exception,
    UNICODE_PROP_Modifier_Combining_Mark,
    UNICODE_PROP_Noncharacter_Code_Point,
    UNICODE_PROP_Pattern_Syntax,
    UNICODE_PROP_Pattern_White_Space,
    UNICODE_PROP_Quotation_Mark,
    UNICODE_PROP_Radical,
    UNICODE_PROP_Regional_Indicator,
    UNICODE_PROP_Sentence_Terminal,
    UNICODE_PROP_Soft_Dotted,
    UNICODE_PROP_Terminal_Punctuation,
    UNICODE_PROP_Unified_Ideograph,
    UNICODE_PROP_Variation_Selector,
    UNICODE_PROP_White_Space,
    UNICODE_PROP_Bidi_Mirrored,
    UNICODE_PROP_Emoji,
    UNICODE_PROP_Emoji_Component,
    UNICODE_PROP_Emoji_Modifier,
    UNICODE_PROP_Emoji_Modifier_Base,
    UNICODE_PROP_Emoji_Presentation,
    UNICODE_PROP_Extended_Pictographic,
    UNICODE_PROP_Default_Ignorable_Code_Point,
    UNICODE_PROP_ID_Start,
    UNICODE_PROP_Case_Ignorable,
    UNICODE_PROP_ASCII,
    UNICODE_PROP_Alphabetic,
    UNICODE_PROP_Any,
    UNICODE_PROP_Assigned,
    UNICODE_PROP_Cased,
    UNICODE_PROP_Changes_When_Casefolded,
    UNICODE_PROP_Changes_When_Casemapped,
    UNICODE_PROP_Changes_When_Lowercased,
    UNICODE_PROP_Changes_When_NFKC_Casefolded,
    UNICODE_PROP_Changes_When_Titlecased,
    UNICODE_PROP_Changes_When_Uppercased,
    UNICODE_PROP_Grapheme_Base,
    UNICODE_PROP_Grapheme_Extend,
    UNICODE_PROP_ID_Continue,
    UNICODE_PROP_ID_Compat_Math_Start,
    UNICODE_PROP_ID_Compat_Math_Continue,
    UNICODE_PROP_Lowercase,
    UNICODE_PROP_Math,
    UNICODE_PROP_Uppercase,
    UNICODE_PROP_XID_Continue,
    UNICODE_PROP_XID_Start,
    UNICODE_PROP_Cased1,
    UNICODE_PROP_InCB,
    UNICODE_PROP_COUNT,
} UnicodePropertyEnum;

static const char unicode_prop_name_table[] =
    "ASCII_Hex_Digit,AHex"               "\0"
    "Bidi_Control,Bidi_C"                "\0"
    "Dash"                               "\0"
    "Deprecated,Dep"                     "\0"
    "Diacritic,Dia"                      "\0"
    "Extender,Ext"                       "\0"
    "Hex_Digit,Hex"                      "\0"
    "IDS_Unary_Operator,IDSU"            "\0"
    "IDS_Binary_Operator,IDSB"           "\0"
    "IDS_Trinary_Operator,IDST"          "\0"
    "Ideographic,Ideo"                   "\0"
    "Join_Control,Join_C"                "\0"
    "Logical_Order_Exception,LOE"        "\0"
    "Modifier_Combining_Mark,MCM"        "\0"
    "Noncharacter_Code_Point,NChar"      "\0"
    "Pattern_Syntax,Pat_Syn"             "\0"
    "Pattern_White_Space,Pat_WS"         "\0"
    "Quotation_Mark,QMark"               "\0"
    "Radical"                            "\0"
    "Regional_Indicator,RI"              "\0"
    "Sentence_Terminal,STerm"            "\0"
    "Soft_Dotted,SD"                     "\0"
    "Terminal_Punctuation,Term"          "\0"
    "Unified_Ideograph,UIdeo"            "\0"
    "Variation_Selector,VS"              "\0"
    "White_Space,space"                  "\0"
    "Bidi_Mirrored,Bidi_M"               "\0"
    "Emoji"                              "\0"
    "Emoji_Component,EComp"              "\0"
    "Emoji_Modifier,EMod"                "\0"
    "Emoji_Modifier_Base,EBase"          "\0"
    "Emoji_Presentation,EPres"           "\0"
    "Extended_Pictographic,ExtPict"      "\0"
    "Default_Ignorable_Code_Point,DI"    "\0"
    "ID_Start,IDS"                       "\0"
    "Case_Ignorable,CI"                  "\0"
    "ASCII"                              "\0"
    "Alphabetic,Alpha"                   "\0"
    "Any"                                "\0"
    "Assigned"                           "\0"
    "Cased"                              "\0"
    "Changes_When_Casefolded,CWCF"       "\0"
    "Changes_When_Casemapped,CWCM"       "\0"
    "Changes_When_Lowercased,CWL"        "\0"
    "Changes_When_NFKC_Casefolded,CWKCF" "\0"
    "Changes_When_Titlecased,CWT"        "\0"
    "Changes_When_Uppercased,CWU"        "\0"
    "Grapheme_Base,Gr_Base"              "\0"
    "Grapheme_Extend,Gr_Ext"             "\0"
    "ID_Continue,IDC"                    "\0"
    "ID_Compat_Math_Start"               "\0"
    "ID_Compat_Math_Continue"            "\0"
    "Lowercase,Lower"                    "\0"
    "Math"                               "\0"
    "Uppercase,Upper"                    "\0"
    "XID_Continue,XIDC"                  "\0"
    "XID_Start,XIDS"                     "\0"
;

static const uint8_t * const unicode_prop_table[] = {
    unicode_prop_Hyphen_table,
    unicode_prop_Other_Math_table,
    unicode_prop_Other_Alphabetic_table,
    unicode_prop_Other_Lowercase_table,
    unicode_prop_Other_Uppercase_table,
    unicode_prop_Other_Grapheme_Extend_table,
    unicode_prop_Other_Default_Ignorable_Code_Point_table,
    unicode_prop_Other_ID_Start_table,
    unicode_prop_Other_ID_Continue_table,
    unicode_prop_Prepended_Concatenation_Mark_table,
    unicode_prop_ID_Continue1_table,
    unicode_prop_XID_Start1_table,
    unicode_prop_XID_Continue1_table,
    unicode_prop_Changes_When_Titlecased1_table,
    unicode_prop_Changes_When_Casefolded1_table,
    unicode_prop_Changes_When_NFKC_Casefolded1_table,
    unicode_prop_ASCII_Hex_Digit_table,
    unicode_prop_Bidi_Control_table,
    unicode_prop_Dash_table,
    unicode_prop_Deprecated_table,
    unicode_prop_Diacritic_table,
    unicode_prop_Extender_table,
    unicode_prop_Hex_Digit_table,
    unicode_prop_IDS_Unary_Operator_table,
    unicode_prop_IDS_Binary_Operator_table,
    unicode_prop_IDS_Trinary_Operator_table,
    unicode_prop_Ideographic_table,
    unicode_prop_Join_Control_table,
    unicode_prop_Logical_Order_Exception_table,
    unicode_prop_Modifier_Combining_Mark_table,
    unicode_prop_Noncharacter_Code_Point_table,
    unicode_prop_Pattern_Syntax_table,
    unicode_prop_Pattern_White_Space_table,
    unicode_prop_Quotation_Mark_table,
    unicode_prop_Radical_table,
    unicode_prop_Regional_Indicator_table,
    unicode_prop_Sentence_Terminal_table,
    unicode_prop_Soft_Dotted_table,
    unicode_prop_Terminal_Punctuation_table,
    unicode_prop_Unified_Ideograph_table,
    unicode_prop_Variation_Selector_table,
    unicode_prop_White_Space_table,
    unicode_prop_Bidi_Mirrored_table,
    unicode_prop_Emoji_table,
    unicode_prop_Emoji_Component_table,
    unicode_prop_Emoji_Modifier_table,
    unicode_prop_Emoji_Modifier_Base_table,
    unicode_prop_Emoji_Presentation_table,
    unicode_prop_Extended_Pictographic_table,
    unicode_prop_Default_Ignorable_Code_Point_table,
    unicode_prop_ID_Start_table,
    unicode_prop_Case_Ignorable_table,
};

static const uint16_t unicode_prop_len_table[] = {
    countof(unicode_prop_Hyphen_table),
    countof(unicode_prop_Other_Math_table),
    countof(unicode_prop_Other_Alphabetic_table),
    countof(unicode_prop_Other_Lowercase_table),
    countof(unicode_prop_Other_Uppercase_table),
    countof(unicode_prop_Other_Grapheme_Extend_table),
    countof(unicode_prop_Other_Default_Ignorable_Code_Point_table),
    countof(unicode_prop_Other_ID_Start_table),
    countof(unicode_prop_Other_ID_Continue_table),
    countof(unicode_prop_Prepended_Concatenation_Mark_table),
    countof(unicode_prop_ID_Continue1_table),
    countof(unicode_prop_XID_Start1_table),
    countof(unicode_prop_XID_Continue1_table),
    countof(unicode_prop_Changes_When_Titlecased1_table),
    countof(unicode_prop_Changes_When_Casefolded1_table),
    countof(unicode_prop_Changes_When_NFKC_Casefolded1_table),
    countof(unicode_prop_ASCII_Hex_Digit_table),
    countof(unicode_prop_Bidi_Control_table),
    countof(unicode_prop_Dash_table),
    countof(unicode_prop_Deprecated_table),
    countof(unicode_prop_Diacritic_table),
    countof(unicode_prop_Extender_table),
    countof(unicode_prop_Hex_Digit_table),
    countof(unicode_prop_IDS_Unary_Operator_table),
    countof(unicode_prop_IDS_Binary_Operator_table),
    countof(unicode_prop_IDS_Trinary_Operator_table),
    countof(unicode_prop_Ideographic_table),
    countof(unicode_prop_Join_Control_table),
    countof(unicode_prop_Logical_Order_Exception_table),
    countof(unicode_prop_Modifier_Combining_Mark_table),
    countof(unicode_prop_Noncharacter_Code_Point_table),
    countof(unicode_prop_Pattern_Syntax_table),
    countof(unicode_prop_Pattern_White_Space_table),
    countof(unicode_prop_Quotation_Mark_table),
    countof(unicode_prop_Radical_table),
    countof(unicode_prop_Regional_Indicator_table),
    countof(unicode_prop_Sentence_Terminal_table),
    countof(unicode_prop_Soft_Dotted_table),
    countof(unicode_prop_Terminal_Punctuation_table),
    countof(unicode_prop_Unified_Ideograph_table),
    countof(unicode_prop_Variation_Selector_table),
    countof(unicode_prop_White_Space_table),
    countof(unicode_prop_Bidi_Mirrored_table),
    countof(unicode_prop_Emoji_table),
    countof(unicode_prop_Emoji_Component_table),
    countof(unicode_prop_Emoji_Modifier_table),
    countof(unicode_prop_Emoji_Modifier_Base_table),
    countof(unicode_prop_Emoji_Presentation_table),
    countof(unicode_prop_Extended_Pictographic_table),
    countof(unicode_prop_Default_Ignorable_Code_Point_table),
    countof(unicode_prop_ID_Start_table),
    countof(unicode_prop_Case_Ignorable_table),
};

/*
 * QuickJS Javascript Engine
 *
 * Copyright (c) 2017-2024 Fabrice Bellard
 * Copyright (c) 2017-2024 Charlie Gordon
 * Copyright (c) 2023-2025 Ben Noordhuis
 * Copyright (c) 2023-2025 Saúl Ibarra Corretgé
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#ifndef QUICKJS_H
#define QUICKJS_H

#include <stdbool.h>
#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <math.h>

#ifdef __cplusplus
extern "C" {
#endif

#define QUICKJS_NG 1

#if defined(__GNUC__) || defined(__clang__)
#define js_force_inline       inline __attribute__((always_inline))
#define JS_EXTERN __attribute__((visibility("default")))
#else
#define js_force_inline  inline
#define JS_EXTERN /* nothing */
#endif

/* Borrowed from Folly */
#ifndef JS_PRINTF_FORMAT
#ifdef _MSC_VER
#include <sal.h>
#define JS_PRINTF_FORMAT _Printf_format_string_
#define JS_PRINTF_FORMAT_ATTR(format_param, dots_param)
#else
#define JS_PRINTF_FORMAT
#if !defined(__clang__) && defined(__GNUC__)
#define JS_PRINTF_FORMAT_ATTR(format_param, dots_param) \
  __attribute__((format(gnu_printf, format_param, dots_param)))
#else
#define JS_PRINTF_FORMAT_ATTR(format_param, dots_param) \
  __attribute__((format(printf, format_param, dots_param)))
#endif
#endif
#endif

typedef struct JSRuntime JSRuntime;
typedef struct JSContext JSContext;
typedef struct JSObject JSObject;
typedef struct JSClass JSClass;
typedef uint32_t JSClassID;
typedef uint32_t JSAtom;

/* Unless documented otherwise, C string pointers (`char *` or `const char *`)
   are assumed to verify these constraints:
   - unless a length is passed separately, the string has a null terminator
   - string contents is either pure ASCII or is UTF-8 encoded.
 */

/* Overridable purely for testing purposes; don't touch. */
#ifndef JS_NAN_BOXING
#if INTPTR_MAX < INT64_MAX
#define JS_NAN_BOXING 1 /* Use NAN boxing for 32bit builds. */
#endif
#endif

enum {
    /* all tags with a reference count are negative */
    JS_TAG_FIRST       = -9, /* first negative tag */
    JS_TAG_BIG_INT     = -9,
    JS_TAG_SYMBOL      = -8,
    JS_TAG_STRING      = -7,
    JS_TAG_MODULE      = -3, /* used internally */
    JS_TAG_FUNCTION_BYTECODE = -2, /* used internally */
    JS_TAG_OBJECT      = -1,

    JS_TAG_INT         = 0,
    JS_TAG_BOOL        = 1,
    JS_TAG_NULL        = 2,
    JS_TAG_UNDEFINED   = 3,
    JS_TAG_UNINITIALIZED = 4,
    JS_TAG_CATCH_OFFSET = 5,
    JS_TAG_EXCEPTION   = 6,
    JS_TAG_SHORT_BIG_INT = 7,
    JS_TAG_FLOAT64     = 8,
    /* any larger tag is FLOAT64 if JS_NAN_BOXING */
};

#if !defined(JS_CHECK_JSVALUE)
#define JSValueConst JSValue
#endif

// JS_CHECK_JSVALUE build mode does not produce working code but is here to
// help catch reference counting bugs at compile time, by making it harder
// to mix up JSValue and JSValueConst
//
// rules:
//
// - a function with a JSValue parameter takes ownership;
//   caller must *not* call JS_FreeValue
//
// - a function with a JSValueConst parameter does not take ownership;
//   caller *must* call JS_FreeValue
//
// - a function returning a JSValue transfers ownership to caller;
//   caller *must* call JS_FreeValue
//
// - a function returning a JSValueConst does *not* transfer ownership;
//   caller must *not* call JS_FreeValue
#if defined(JS_CHECK_JSVALUE)

typedef struct JSValue *JSValue;
typedef const struct JSValue *JSValueConst;

#define JS_MKVAL(tag, val)       ((JSValue)((tag) | (intptr_t)(val) << 4))
#define JS_MKPTR(tag, ptr)       ((JSValue)((tag) | (intptr_t)(ptr)))
#define JS_VALUE_GET_NORM_TAG(v) ((int)((intptr_t)(v) & 15))
#define JS_VALUE_GET_TAG(v)      ((int)((intptr_t)(v) & 15))
#define JS_VALUE_GET_SHORT_BIG_INT(v) JS_VALUE_GET_INT(v)
#define JS_VALUE_GET_PTR(v)      ((void *)((intptr_t)(v) & ~15))
#define JS_VALUE_GET_INT(v)      ((int)((intptr_t)(v) >> 4))
#define JS_VALUE_GET_BOOL(v)     ((int)((intptr_t)(v) >> 4))
#define JS_VALUE_GET_FLOAT64(v)  ((double)((intptr_t)(v) >> 4))
#define JS_TAG_IS_FLOAT64(tag)   ((int)(tag) == JS_TAG_FLOAT64)
#define JS_NAN                   JS_MKVAL(JS_TAG_FLOAT64, 0)

static inline JSValue __JS_NewFloat64(double d)
{
    return JS_MKVAL(JS_TAG_FLOAT64, (int)d);
}

static inline JSValue __JS_NewShortBigInt(JSContext *ctx, int32_t d)
{
    (void)&ctx;
    return JS_MKVAL(JS_TAG_SHORT_BIG_INT, d);
}

static inline bool JS_VALUE_IS_NAN(JSValue v)
{
    (void)&v;
    return false;
}

#elif defined(JS_NAN_BOXING) && JS_NAN_BOXING

typedef uint64_t JSValue;

#define JS_VALUE_GET_TAG(v) (int)((v) >> 32)
#define JS_VALUE_GET_INT(v) (int)(v)
#define JS_VALUE_GET_BOOL(v) (int)(v)
#define JS_VALUE_GET_SHORT_BIG_INT(v) (int)(v)
#define JS_VALUE_GET_PTR(v) (void *)(intptr_t)(v)

#define JS_MKVAL(tag, val) (((uint64_t)(tag) << 32) | (uint32_t)(val))
#define JS_MKPTR(tag, ptr) (((uint64_t)(tag) << 32) | (uintptr_t)(ptr))

#define JS_FLOAT64_TAG_ADDEND (0x7ff80000 - JS_TAG_FIRST + 1) /* quiet NaN encoding */

static inline double JS_VALUE_GET_FLOAT64(JSValue v)
{
    union {
        JSValue v;
        double d;
    } u;
    u.v = v;
    u.v += (uint64_t)JS_FLOAT64_TAG_ADDEND << 32;
    return u.d;
}

#define JS_NAN (0x7ff8000000000000 - ((uint64_t)JS_FLOAT64_TAG_ADDEND << 32))

static inline JSValue __JS_NewFloat64(double d)
{
    union {
        double d;
        uint64_t u64;
    } u;
    JSValue v;
    u.d = d;
    /* normalize NaN */
    if ((u.u64 & 0x7fffffffffffffff) > 0x7ff0000000000000)
        v = JS_NAN;
    else
        v = u.u64 - ((uint64_t)JS_FLOAT64_TAG_ADDEND << 32);
    return v;
}

static inline JSValue __JS_NewShortBigInt(JSContext *ctx, int32_t d)
{
    (void)&ctx;
    return JS_MKVAL(JS_TAG_SHORT_BIG_INT, d);
}

#define JS_TAG_IS_FLOAT64(tag) ((unsigned)((tag) - JS_TAG_FIRST) >= (JS_TAG_FLOAT64 - JS_TAG_FIRST))

/* same as JS_VALUE_GET_TAG, but return JS_TAG_FLOAT64 with NaN boxing */
static inline int JS_VALUE_GET_NORM_TAG(JSValue v)
{
    uint32_t tag;
    tag = JS_VALUE_GET_TAG(v);
    if (JS_TAG_IS_FLOAT64(tag))
        return JS_TAG_FLOAT64;
    else
        return tag;
}

static inline bool JS_VALUE_IS_NAN(JSValue v)
{
    uint32_t tag;
    tag = JS_VALUE_GET_TAG(v);
    return tag == (JS_NAN >> 32);
}

#else /* !JS_NAN_BOXING */

typedef union JSValueUnion {
    int32_t int32;
    double float64;
    void *ptr;
    int32_t short_big_int;
} JSValueUnion;

typedef struct JSValue {
    JSValueUnion u;
    int64_t tag;
} JSValue;

#define JS_VALUE_GET_TAG(v) ((int32_t)(v).tag)
/* same as JS_VALUE_GET_TAG, but return JS_TAG_FLOAT64 with NaN boxing */
#define JS_VALUE_GET_NORM_TAG(v) JS_VALUE_GET_TAG(v)
#define JS_VALUE_GET_INT(v) ((v).u.int32)
#define JS_VALUE_GET_BOOL(v) ((v).u.int32)
#define JS_VALUE_GET_FLOAT64(v) ((v).u.float64)
#define JS_VALUE_GET_SHORT_BIG_INT(v) ((v).u.short_big_int)
#define JS_VALUE_GET_PTR(v) ((v).u.ptr)

/* msvc doesn't understand designated initializers without /std:c++20 */
#ifdef __cplusplus
static inline JSValue JS_MKPTR(int64_t tag, void *ptr)
{
    JSValue v;
    v.u.ptr = ptr;
    v.tag = tag;
    return v;
}
static inline JSValue JS_MKVAL(int64_t tag, int32_t int32)
{
    JSValue v;
    v.u.int32 = int32;
    v.tag = tag;
    return v;
}
static inline JSValue JS_MKNAN(void)
{
    JSValue v;
    v.u.float64 = NAN;
    v.tag = JS_TAG_FLOAT64;
    return v;
}
/* provide as macros for consistency and backward compat reasons */
#define JS_MKPTR(tag, ptr) JS_MKPTR(tag, ptr)
#define JS_MKVAL(tag, val) JS_MKVAL(tag, val)
#define JS_NAN             JS_MKNAN() /* alas, not a constant expression */
#else
#define JS_MKPTR(tag, p)   (JSValue){ (JSValueUnion){ .ptr = p }, tag }
#define JS_MKVAL(tag, val) (JSValue){ (JSValueUnion){ .int32 = val }, tag }
#define JS_NAN             (JSValue){ (JSValueUnion){ .float64 = NAN }, JS_TAG_FLOAT64 }
#endif

#define JS_TAG_IS_FLOAT64(tag) ((unsigned)(tag) == JS_TAG_FLOAT64)

static inline JSValue __JS_NewFloat64(double d)
{
    JSValue v;
    v.tag = JS_TAG_FLOAT64;
    v.u.float64 = d;
    return v;
}

static inline JSValue __JS_NewShortBigInt(JSContext *ctx, int64_t d)
{
    (void)&ctx;
    JSValue v;
    v.tag = JS_TAG_SHORT_BIG_INT;
    v.u.short_big_int = d;
    return v;
}

static inline bool JS_VALUE_IS_NAN(JSValue v)
{
    union {
        double d;
        uint64_t u64;
    } u;
    if (v.tag != JS_TAG_FLOAT64)
        return 0;
    u.d = v.u.float64;
    return (u.u64 & 0x7fffffffffffffff) > 0x7ff0000000000000;
}

#endif /* !JS_NAN_BOXING */

#define JS_VALUE_IS_BOTH_INT(v1, v2) ((JS_VALUE_GET_TAG(v1) | JS_VALUE_GET_TAG(v2)) == 0)
#define JS_VALUE_IS_BOTH_FLOAT(v1, v2) (JS_TAG_IS_FLOAT64(JS_VALUE_GET_TAG(v1)) && JS_TAG_IS_FLOAT64(JS_VALUE_GET_TAG(v2)))

#define JS_VALUE_GET_OBJ(v) ((JSObject *)JS_VALUE_GET_PTR(v))
#define JS_VALUE_HAS_REF_COUNT(v) ((unsigned)JS_VALUE_GET_TAG(v) >= (unsigned)JS_TAG_FIRST)

/* special values */
#define JS_NULL      JS_MKVAL(JS_TAG_NULL, 0)
#define JS_UNDEFINED JS_MKVAL(JS_TAG_UNDEFINED, 0)
#define JS_FALSE     JS_MKVAL(JS_TAG_BOOL, 0)
#define JS_TRUE      JS_MKVAL(JS_TAG_BOOL, 1)
#define JS_EXCEPTION JS_MKVAL(JS_TAG_EXCEPTION, 0)
#define JS_UNINITIALIZED JS_MKVAL(JS_TAG_UNINITIALIZED, 0)

/* flags for object properties */
#define JS_PROP_CONFIGURABLE  (1 << 0)
#define JS_PROP_WRITABLE      (1 << 1)
#define JS_PROP_ENUMERABLE    (1 << 2)
#define JS_PROP_C_W_E         (JS_PROP_CONFIGURABLE | JS_PROP_WRITABLE | JS_PROP_ENUMERABLE)
#define JS_PROP_LENGTH        (1 << 3) /* used internally in Arrays */
#define JS_PROP_TMASK         (3 << 4) /* mask for NORMAL, GETSET, VARREF, AUTOINIT */
#define JS_PROP_NORMAL         (0 << 4)
#define JS_PROP_GETSET         (1 << 4)
#define JS_PROP_VARREF         (2 << 4) /* used internally */
#define JS_PROP_AUTOINIT       (3 << 4) /* used internally */

/* flags for JS_DefineProperty */
#define JS_PROP_HAS_SHIFT        8
#define JS_PROP_HAS_CONFIGURABLE (1 << 8)
#define JS_PROP_HAS_WRITABLE     (1 << 9)
#define JS_PROP_HAS_ENUMERABLE   (1 << 10)
#define JS_PROP_HAS_GET          (1 << 11)
#define JS_PROP_HAS_SET          (1 << 12)
#define JS_PROP_HAS_VALUE        (1 << 13)

/* throw an exception if false would be returned
   (JS_DefineProperty/JS_SetProperty) */
#define JS_PROP_THROW            (1 << 14)
/* throw an exception if false would be returned in strict mode
   (JS_SetProperty) */
#define JS_PROP_THROW_STRICT     (1 << 15)

#define JS_PROP_NO_ADD           (1 << 16) /* internal use */
#define JS_PROP_NO_EXOTIC        (1 << 17) /* internal use */
#define JS_PROP_DEFINE_PROPERTY  (1 << 18) /* internal use */
#define JS_PROP_REFLECT_DEFINE_PROPERTY (1 << 19) /* internal use */

#ifndef JS_DEFAULT_STACK_SIZE
#define JS_DEFAULT_STACK_SIZE (1024 * 1024)
#endif

/* JS_Eval() flags */
#define JS_EVAL_TYPE_GLOBAL   (0 << 0) /* global code (default) */
#define JS_EVAL_TYPE_MODULE   (1 << 0) /* module code */
#define JS_EVAL_TYPE_DIRECT   (2 << 0) /* direct call (internal use) */
#define JS_EVAL_TYPE_INDIRECT (3 << 0) /* indirect call (internal use) */
#define JS_EVAL_TYPE_MASK     (3 << 0)

#define JS_EVAL_FLAG_STRICT   (1 << 3) /* force 'strict' mode */
#define JS_EVAL_FLAG_UNUSED   (1 << 4) /* unused */
/* compile but do not run. The result is an object with a
   JS_TAG_FUNCTION_BYTECODE or JS_TAG_MODULE tag. It can be executed
   with JS_EvalFunction(). */
#define JS_EVAL_FLAG_COMPILE_ONLY (1 << 5)
/* don't include the stack frames before this eval in the Error() backtraces */
#define JS_EVAL_FLAG_BACKTRACE_BARRIER (1 << 6)
/* allow top-level await in normal script. JS_Eval() returns a
   promise. Only allowed with JS_EVAL_TYPE_GLOBAL */
#define JS_EVAL_FLAG_ASYNC (1 << 7)

typedef JSValue JSCFunction(JSContext *ctx, JSValueConst this_val, int argc, JSValueConst *argv);
typedef JSValue JSCFunctionMagic(JSContext *ctx, JSValueConst this_val, int argc, JSValueConst *argv, int magic);
typedef JSValue JSCFunctionData(JSContext *ctx, JSValueConst this_val, int argc, JSValueConst *argv, int magic, JSValueConst *func_data);

typedef struct JSMallocFunctions {
    void *(*js_calloc)(void *opaque, size_t count, size_t size);
    void *(*js_malloc)(void *opaque, size_t size);
    void (*js_free)(void *opaque, void *ptr);
    void *(*js_realloc)(void *opaque, void *ptr, size_t size);
    size_t (*js_malloc_usable_size)(const void *ptr);
} JSMallocFunctions;

// Debug trace system: the debug output will be produced to the dump stream (currently
// stdout) if dumps are enabled and JS_SetDumpFlags is invoked with the corresponding
// bit set.
#define JS_DUMP_BYTECODE_FINAL   0x01  /* dump pass 3 final byte code */
#define JS_DUMP_BYTECODE_PASS2   0x02  /* dump pass 2 code */
#define JS_DUMP_BYTECODE_PASS1   0x04  /* dump pass 1 code */
#define JS_DUMP_BYTECODE_HEX     0x10  /* dump bytecode in hex */
#define JS_DUMP_BYTECODE_PC2LINE 0x20  /* dump line number table */
#define JS_DUMP_BYTECODE_STACK   0x40  /* dump compute_stack_size */
#define JS_DUMP_BYTECODE_STEP    0x80  /* dump executed bytecode */
#define JS_DUMP_READ_OBJECT     0x100  /* dump the marshalled objects at load time */
#define JS_DUMP_FREE            0x200  /* dump every object free */
#define JS_DUMP_GC              0x400  /* dump the occurrence of the automatic GC */
#define JS_DUMP_GC_FREE         0x800  /* dump objects freed by the GC */
#define JS_DUMP_MODULE_RESOLVE 0x1000  /* dump module resolution steps */
#define JS_DUMP_PROMISE        0x2000  /* dump promise steps */
#define JS_DUMP_LEAKS          0x4000  /* dump leaked objects and strings in JS_FreeRuntime */
#define JS_DUMP_ATOM_LEAKS     0x8000  /* dump leaked atoms in JS_FreeRuntime */
#define JS_DUMP_MEM           0x10000  /* dump memory usage in JS_FreeRuntime */
#define JS_DUMP_OBJECTS       0x20000  /* dump objects in JS_FreeRuntime */
#define JS_DUMP_ATOMS         0x40000  /* dump atoms in JS_FreeRuntime */
#define JS_DUMP_SHAPES        0x80000  /* dump shapes in JS_FreeRuntime */

// Finalizers run in LIFO order at the very end of JS_FreeRuntime.
// Intended for cleanup of associated resources; the runtime itself
// is no longer usable.
typedef void JSRuntimeFinalizer(JSRuntime *rt, void *arg);

typedef struct JSGCObjectHeader JSGCObjectHeader;

JS_EXTERN JSRuntime *JS_NewRuntime(void);
/* info lifetime must exceed that of rt */
JS_EXTERN void JS_SetRuntimeInfo(JSRuntime *rt, const char *info);
/* use 0 to disable memory limit */
JS_EXTERN void JS_SetMemoryLimit(JSRuntime *rt, size_t limit);
JS_EXTERN void JS_SetDumpFlags(JSRuntime *rt, uint64_t flags);
JS_EXTERN uint64_t JS_GetDumpFlags(JSRuntime *rt);
JS_EXTERN size_t JS_GetGCThreshold(JSRuntime *rt);
JS_EXTERN void JS_SetGCThreshold(JSRuntime *rt, size_t gc_threshold);
/* use 0 to disable maximum stack size check */
JS_EXTERN void JS_SetMaxStackSize(JSRuntime *rt, size_t stack_size);
/* should be called when changing thread to update the stack top value
   used to check stack overflow. */
JS_EXTERN void JS_UpdateStackTop(JSRuntime *rt);
JS_EXTERN JSRuntime *JS_NewRuntime2(const JSMallocFunctions *mf, void *opaque);
JS_EXTERN void JS_FreeRuntime(JSRuntime *rt);
JS_EXTERN void *JS_GetRuntimeOpaque(JSRuntime *rt);
JS_EXTERN void JS_SetRuntimeOpaque(JSRuntime *rt, void *opaque);
JS_EXTERN int JS_AddRuntimeFinalizer(JSRuntime *rt,
                                     JSRuntimeFinalizer *finalizer, void *arg);
typedef void JS_MarkFunc(JSRuntime *rt, JSGCObjectHeader *gp);
JS_EXTERN void JS_MarkValue(JSRuntime *rt, JSValueConst val,
                            JS_MarkFunc *mark_func);
JS_EXTERN void JS_RunGC(JSRuntime *rt);
JS_EXTERN bool JS_IsLiveObject(JSRuntime *rt, JSValueConst obj);

JS_EXTERN JSContext *JS_NewContext(JSRuntime *rt);
JS_EXTERN void JS_FreeContext(JSContext *s);
JS_EXTERN JSContext *JS_DupContext(JSContext *ctx);
JS_EXTERN void *JS_GetContextOpaque(JSContext *ctx);
JS_EXTERN void JS_SetContextOpaque(JSContext *ctx, void *opaque);
JS_EXTERN JSRuntime *JS_GetRuntime(JSContext *ctx);
JS_EXTERN void JS_SetClassProto(JSContext *ctx, JSClassID class_id, JSValue obj);
JS_EXTERN JSValue JS_GetClassProto(JSContext *ctx, JSClassID class_id);
JS_EXTERN JSValue JS_GetFunctionProto(JSContext *ctx);

/* the following functions are used to select the intrinsic object to
   save memory */
JS_EXTERN JSContext *JS_NewContextRaw(JSRuntime *rt);
JS_EXTERN void JS_AddIntrinsicBaseObjects(JSContext *ctx);
JS_EXTERN void JS_AddIntrinsicDate(JSContext *ctx);
JS_EXTERN void JS_AddIntrinsicEval(JSContext *ctx);
JS_EXTERN void JS_AddIntrinsicRegExpCompiler(JSContext *ctx);
JS_EXTERN void JS_AddIntrinsicRegExp(JSContext *ctx);
JS_EXTERN void JS_AddIntrinsicJSON(JSContext *ctx);
JS_EXTERN void JS_AddIntrinsicProxy(JSContext *ctx);
JS_EXTERN void JS_AddIntrinsicMapSet(JSContext *ctx);
JS_EXTERN void JS_AddIntrinsicTypedArrays(JSContext *ctx);
JS_EXTERN void JS_AddIntrinsicPromise(JSContext *ctx);
JS_EXTERN void JS_AddIntrinsicBigInt(JSContext *ctx);
JS_EXTERN void JS_AddIntrinsicWeakRef(JSContext *ctx);
JS_EXTERN void JS_AddPerformance(JSContext *ctx);
JS_EXTERN void JS_AddIntrinsicDOMException(JSContext *ctx);

/* for equality comparisons and sameness */
JS_EXTERN int JS_IsEqual(JSContext *ctx, JSValueConst op1, JSValueConst op2);
JS_EXTERN bool JS_IsStrictEqual(JSContext *ctx, JSValueConst op1, JSValueConst op2);
JS_EXTERN bool JS_IsSameValue(JSContext *ctx, JSValueConst op1, JSValueConst op2);
/* Similar to same-value equality, but +0 and -0 are considered equal. */
JS_EXTERN bool JS_IsSameValueZero(JSContext *ctx, JSValueConst op1, JSValueConst op2);

/* Only used for running 262 tests. TODO(saghul) add build time flag. */
JS_EXTERN JSValue js_string_codePointRange(JSContext *ctx, JSValueConst this_val,
                                           int argc, JSValueConst *argv);

JS_EXTERN void *js_calloc_rt(JSRuntime *rt, size_t count, size_t size);
JS_EXTERN void *js_malloc_rt(JSRuntime *rt, size_t size);
JS_EXTERN void js_free_rt(JSRuntime *rt, void *ptr);
JS_EXTERN void *js_realloc_rt(JSRuntime *rt, void *ptr, size_t size);
JS_EXTERN size_t js_malloc_usable_size_rt(JSRuntime *rt, const void *ptr);
JS_EXTERN void *js_mallocz_rt(JSRuntime *rt, size_t size);

JS_EXTERN void *js_calloc(JSContext *ctx, size_t count, size_t size);
JS_EXTERN void *js_malloc(JSContext *ctx, size_t size);
JS_EXTERN void js_free(JSContext *ctx, void *ptr);
JS_EXTERN void *js_realloc(JSContext *ctx, void *ptr, size_t size);
JS_EXTERN size_t js_malloc_usable_size(JSContext *ctx, const void *ptr);
JS_EXTERN void *js_realloc2(JSContext *ctx, void *ptr, size_t size, size_t *pslack);
JS_EXTERN void *js_mallocz(JSContext *ctx, size_t size);
JS_EXTERN char *js_strdup(JSContext *ctx, const char *str);
JS_EXTERN char *js_strndup(JSContext *ctx, const char *s, size_t n);

typedef struct JSMemoryUsage {
    int64_t malloc_size, malloc_limit, memory_used_size;
    int64_t malloc_count;
    int64_t memory_used_count;
    int64_t atom_count, atom_size;
    int64_t str_count, str_size;
    int64_t obj_count, obj_size;
    int64_t prop_count, prop_size;
    int64_t shape_count, shape_size;
    int64_t js_func_count, js_func_size, js_func_code_size;
    int64_t js_func_pc2line_count, js_func_pc2line_size;
    int64_t c_func_count, array_count;
    int64_t fast_array_count, fast_array_elements;
    int64_t binary_object_count, binary_object_size;
} JSMemoryUsage;

JS_EXTERN void JS_ComputeMemoryUsage(JSRuntime *rt, JSMemoryUsage *s);
JS_EXTERN void JS_DumpMemoryUsage(FILE *fp, const JSMemoryUsage *s, JSRuntime *rt);

/* atom support */
#define JS_ATOM_NULL 0

JS_EXTERN JSAtom JS_NewAtomLen(JSContext *ctx, const char *str, size_t len);
JS_EXTERN JSAtom JS_NewAtom(JSContext *ctx, const char *str);
JS_EXTERN JSAtom JS_NewAtomUInt32(JSContext *ctx, uint32_t n);
JS_EXTERN JSAtom JS_DupAtom(JSContext *ctx, JSAtom v);
JS_EXTERN void JS_FreeAtom(JSContext *ctx, JSAtom v);
JS_EXTERN void JS_FreeAtomRT(JSRuntime *rt, JSAtom v);
JS_EXTERN JSValue JS_AtomToValue(JSContext *ctx, JSAtom atom);
JS_EXTERN JSValue JS_AtomToString(JSContext *ctx, JSAtom atom);
JS_EXTERN const char *JS_AtomToCStringLen(JSContext *ctx, size_t *plen, JSAtom atom);
static inline const char *JS_AtomToCString(JSContext *ctx, JSAtom atom) 
{
    return JS_AtomToCStringLen(ctx, NULL, atom);
}
JS_EXTERN JSAtom JS_ValueToAtom(JSContext *ctx, JSValueConst val);

/* object class support */

typedef struct JSPropertyEnum {
    bool is_enumerable;
    JSAtom atom;
} JSPropertyEnum;

typedef struct JSPropertyDescriptor {
    int flags;
    JSValue value;
    JSValue getter;
    JSValue setter;
} JSPropertyDescriptor;

typedef struct JSClassExoticMethods {
    /* Return -1 if exception (can only happen in case of Proxy object),
       false if the property does not exists, true if it exists. If 1 is
       returned, the property descriptor 'desc' is filled if != NULL. */
    int (*get_own_property)(JSContext *ctx, JSPropertyDescriptor *desc,
                             JSValueConst obj, JSAtom prop);
    /* '*ptab' should hold the '*plen' property keys. Return 0 if OK,
       -1 if exception. The 'is_enumerable' field is ignored.
    */
    int (*get_own_property_names)(JSContext *ctx, JSPropertyEnum **ptab,
                                  uint32_t *plen, JSValueConst obj);
    /* return < 0 if exception, or true/false */
    int (*delete_property)(JSContext *ctx, JSValueConst obj, JSAtom prop);
    /* return < 0 if exception or true/false */
    int (*define_own_property)(JSContext *ctx, JSValueConst this_obj,
                               JSAtom prop, JSValueConst val,
                               JSValueConst getter, JSValueConst setter,
                               int flags);
    /* The following methods can be emulated with the previous ones,
       so they are usually not needed */
    /* return < 0 if exception or true/false */
    int (*has_property)(JSContext *ctx, JSValueConst obj, JSAtom atom);
    JSValue (*get_property)(JSContext *ctx, JSValueConst obj, JSAtom atom,
                            JSValueConst receiver);
    /* return < 0 if exception or true/false */
    int (*set_property)(JSContext *ctx, JSValueConst obj, JSAtom atom,
                        JSValueConst value, JSValueConst receiver, int flags);
} JSClassExoticMethods;

typedef void JSClassFinalizer(JSRuntime *rt, JSValueConst val);
typedef void JSClassGCMark(JSRuntime *rt, JSValueConst val,
                           JS_MarkFunc *mark_func);
#define JS_CALL_FLAG_CONSTRUCTOR (1 << 0)
typedef JSValue JSClassCall(JSContext *ctx, JSValueConst func_obj,
                            JSValueConst this_val, int argc,
                            JSValueConst *argv, int flags);

typedef struct JSClassDef {
    const char *class_name; /* pure ASCII only! */
    JSClassFinalizer *finalizer;
    JSClassGCMark *gc_mark;
    /* if call != NULL, the object is a function. If (flags &
       JS_CALL_FLAG_CONSTRUCTOR) != 0, the function is called as a
       constructor. In this case, 'this_val' is new.target. A
       constructor call only happens if the object constructor bit is
       set (see JS_SetConstructorBit()). */
    JSClassCall *call;
    /* XXX: suppress this indirection ? It is here only to save memory
       because only a few classes need these methods */
    JSClassExoticMethods *exotic;
} JSClassDef;

#define JS_EVAL_OPTIONS_VERSION 1

typedef struct JSEvalOptions {
  int version;
  int eval_flags;
  const char *filename;
  int line_num;
  // can add new fields in ABI-compatible manner by incrementing JS_EVAL_OPTIONS_VERSION
} JSEvalOptions;

#define JS_INVALID_CLASS_ID 0
JS_EXTERN JSClassID JS_NewClassID(JSRuntime *rt, JSClassID *pclass_id);
/* Returns the class ID if `v` is an object, otherwise returns JS_INVALID_CLASS_ID. */
JS_EXTERN JSClassID JS_GetClassID(JSValueConst v);
JS_EXTERN int JS_NewClass(JSRuntime *rt, JSClassID class_id, const JSClassDef *class_def);
JS_EXTERN bool JS_IsRegisteredClass(JSRuntime *rt, JSClassID class_id);

/* value handling */

static js_force_inline JSValue JS_NewBool(JSContext *ctx, bool val)
{
    (void)&ctx;
    return JS_MKVAL(JS_TAG_BOOL, (val != 0));
}

static js_force_inline JSValue JS_NewInt32(JSContext *ctx, int32_t val)
{
    (void)&ctx;
    return JS_MKVAL(JS_TAG_INT, val);
}

static js_force_inline JSValue JS_NewFloat64(JSContext *ctx, double val)
{
    (void)&ctx;
    return __JS_NewFloat64(val);
}

static js_force_inline JSValue JS_NewCatchOffset(JSContext *ctx, int32_t val)
{
    (void)&ctx;
    return JS_MKVAL(JS_TAG_CATCH_OFFSET, val);
}

static js_force_inline JSValue JS_NewInt64(JSContext *ctx, int64_t val)
{
    JSValue v;
    if (val >= INT32_MIN && val <= INT32_MAX) {
        v = JS_NewInt32(ctx, (int32_t)val);
    } else {
        v = JS_NewFloat64(ctx, (double)val);
    }
    return v;
}

static js_force_inline JSValue JS_NewUint32(JSContext *ctx, uint32_t val)
{
    JSValue v;
    if (val <= INT32_MAX) {
        v = JS_NewInt32(ctx, (int32_t)val);
    } else {
        v = JS_NewFloat64(ctx, (double)val);
    }
    return v;
}

JS_EXTERN JSValue JS_NewNumber(JSContext *ctx, double d);
JS_EXTERN JSValue JS_NewBigInt64(JSContext *ctx, int64_t v);
JS_EXTERN JSValue JS_NewBigUint64(JSContext *ctx, uint64_t v);

static inline bool JS_IsNumber(JSValueConst v)
{
    int tag = JS_VALUE_GET_TAG(v);
    return tag == JS_TAG_INT || JS_TAG_IS_FLOAT64(tag);
}

static inline bool JS_IsBigInt(JSValueConst v)
{
    int tag = JS_VALUE_GET_TAG(v);
    return tag == JS_TAG_BIG_INT || tag == JS_TAG_SHORT_BIG_INT;
}

static inline bool JS_IsBool(JSValueConst v)
{
    return JS_VALUE_GET_TAG(v) == JS_TAG_BOOL;
}

static inline bool JS_IsNull(JSValueConst v)
{
    return JS_VALUE_GET_TAG(v) == JS_TAG_NULL;
}

static inline bool JS_IsUndefined(JSValueConst v)
{
    return JS_VALUE_GET_TAG(v) == JS_TAG_UNDEFINED;
}

static inline bool JS_IsException(JSValueConst v)
{
    return JS_VALUE_GET_TAG(v) == JS_TAG_EXCEPTION;
}

static inline bool JS_IsUninitialized(JSValueConst v)
{
    return JS_VALUE_GET_TAG(v) == JS_TAG_UNINITIALIZED;
}

static inline bool JS_IsString(JSValueConst v)
{
    return JS_VALUE_GET_TAG(v) == JS_TAG_STRING;
}

static inline bool JS_IsSymbol(JSValueConst v)
{
    return JS_VALUE_GET_TAG(v) == JS_TAG_SYMBOL;
}

static inline bool JS_IsObject(JSValueConst v)
{
    return JS_VALUE_GET_TAG(v) == JS_TAG_OBJECT;
}

static inline bool JS_IsModule(JSValueConst v)
{
    return JS_VALUE_GET_TAG(v) == JS_TAG_MODULE;
}

JS_EXTERN JSValue JS_Throw(JSContext *ctx, JSValue obj);
JS_EXTERN JSValue JS_GetException(JSContext *ctx);
JS_EXTERN bool JS_HasException(JSContext *ctx);
JS_EXTERN bool JS_IsError(JSValueConst val);
JS_EXTERN bool JS_IsUncatchableError(JSValueConst val);
JS_EXTERN void JS_SetUncatchableError(JSContext *ctx, JSValueConst val);
JS_EXTERN void JS_ClearUncatchableError(JSContext *ctx, JSValueConst val);
// Shorthand for:
//  JSValue exc = JS_GetException(ctx);
//  JS_ClearUncatchableError(ctx, exc);
//  JS_Throw(ctx, exc);
JS_EXTERN void JS_ResetUncatchableError(JSContext *ctx);
JS_EXTERN JSValue JS_NewError(JSContext *ctx);
JS_EXTERN JSValue JS_PRINTF_FORMAT_ATTR(2, 3) JS_NewInternalError(JSContext *ctx, JS_PRINTF_FORMAT const char *fmt, ...);
JS_EXTERN JSValue JS_PRINTF_FORMAT_ATTR(2, 3) JS_NewPlainError(JSContext *ctx, JS_PRINTF_FORMAT const char *fmt, ...);
JS_EXTERN JSValue JS_PRINTF_FORMAT_ATTR(2, 3) JS_NewRangeError(JSContext *ctx, JS_PRINTF_FORMAT const char *fmt, ...);
JS_EXTERN JSValue JS_PRINTF_FORMAT_ATTR(2, 3) JS_NewReferenceError(JSContext *ctx, JS_PRINTF_FORMAT const char *fmt, ...);
JS_EXTERN JSValue JS_PRINTF_FORMAT_ATTR(2, 3) JS_NewSyntaxError(JSContext *ctx, JS_PRINTF_FORMAT const char *fmt, ...);
JS_EXTERN JSValue JS_PRINTF_FORMAT_ATTR(2, 3) JS_NewTypeError(JSContext *ctx, JS_PRINTF_FORMAT const char *fmt, ...);
JS_EXTERN JSValue JS_PRINTF_FORMAT_ATTR(2, 3) JS_ThrowInternalError(JSContext *ctx, JS_PRINTF_FORMAT const char *fmt, ...);
JS_EXTERN JSValue JS_PRINTF_FORMAT_ATTR(2, 3) JS_ThrowPlainError(JSContext *ctx, JS_PRINTF_FORMAT const char *fmt, ...);
JS_EXTERN JSValue JS_PRINTF_FORMAT_ATTR(2, 3) JS_ThrowRangeError(JSContext *ctx, JS_PRINTF_FORMAT const char *fmt, ...);
JS_EXTERN JSValue JS_PRINTF_FORMAT_ATTR(2, 3) JS_ThrowReferenceError(JSContext *ctx, JS_PRINTF_FORMAT const char *fmt, ...);
JS_EXTERN JSValue JS_PRINTF_FORMAT_ATTR(2, 3) JS_ThrowSyntaxError(JSContext *ctx, JS_PRINTF_FORMAT const char *fmt, ...);
JS_EXTERN JSValue JS_PRINTF_FORMAT_ATTR(2, 3) JS_ThrowTypeError(JSContext *ctx, JS_PRINTF_FORMAT const char *fmt, ...);
JS_EXTERN JSValue JS_PRINTF_FORMAT_ATTR(3, 4) JS_ThrowDOMException(JSContext *ctx, const char *name, JS_PRINTF_FORMAT const char *fmt, ...);
JS_EXTERN JSValue JS_ThrowOutOfMemory(JSContext *ctx);
JS_EXTERN void JS_FreeValue(JSContext *ctx, JSValue v);
JS_EXTERN void JS_FreeValueRT(JSRuntime *rt, JSValue v);
JS_EXTERN JSValue JS_DupValue(JSContext *ctx, JSValueConst v);
JS_EXTERN JSValue JS_DupValueRT(JSRuntime *rt, JSValueConst v);
JS_EXTERN int JS_ToBool(JSContext *ctx, JSValueConst val); /* return -1 for JS_EXCEPTION */
static inline JSValue JS_ToBoolean(JSContext *ctx, JSValueConst val)
{
    return JS_NewBool(ctx, JS_ToBool(ctx, val));
}
JS_EXTERN JSValue JS_ToNumber(JSContext *ctx, JSValueConst val);
JS_EXTERN int JS_ToInt32(JSContext *ctx, int32_t *pres, JSValueConst val);
static inline int JS_ToUint32(JSContext *ctx, uint32_t *pres, JSValueConst val)
{
    return JS_ToInt32(ctx, (int32_t*)pres, val);
}
JS_EXTERN int JS_ToInt64(JSContext *ctx, int64_t *pres, JSValueConst val);
JS_EXTERN int JS_ToIndex(JSContext *ctx, uint64_t *plen, JSValueConst val);
JS_EXTERN int JS_ToFloat64(JSContext *ctx, double *pres, JSValueConst val);
/* return an exception if 'val' is a Number */
JS_EXTERN int JS_ToBigInt64(JSContext *ctx, int64_t *pres, JSValueConst val);
JS_EXTERN int JS_ToBigUint64(JSContext *ctx, uint64_t *pres, JSValueConst val);
/* same as JS_ToInt64() but allow BigInt */
JS_EXTERN int JS_ToInt64Ext(JSContext *ctx, int64_t *pres, JSValueConst val);

JS_EXTERN JSValue JS_NewStringLen(JSContext *ctx, const char *str1, size_t len1);
static inline JSValue JS_NewString(JSContext *ctx, const char *str) {
    return JS_NewStringLen(ctx, str, strlen(str));
}
// makes a copy of the input; does not check if the input is valid UTF-16,
// that is the responsibility of the caller
JS_EXTERN JSValue JS_NewTwoByteString(JSContext *ctx, const uint16_t *buf,
                                      size_t len);
JS_EXTERN JSValue JS_NewAtomString(JSContext *ctx, const char *str);
JS_EXTERN JSValue JS_ToString(JSContext *ctx, JSValueConst val);
JS_EXTERN JSValue JS_ToPropertyKey(JSContext *ctx, JSValueConst val);
JS_EXTERN const char *JS_ToCStringLen2(JSContext *ctx, size_t *plen, JSValueConst val1, bool cesu8);
static inline const char *JS_ToCStringLen(JSContext *ctx, size_t *plen, JSValueConst val1)
{
    return JS_ToCStringLen2(ctx, plen, val1, 0);
}
static inline const char *JS_ToCString(JSContext *ctx, JSValueConst val1)
{
    return JS_ToCStringLen2(ctx, NULL, val1, 0);
}
JS_EXTERN void JS_FreeCString(JSContext *ctx, const char *ptr);

JS_EXTERN JSValue JS_NewObjectProtoClass(JSContext *ctx, JSValueConst proto,
                                         JSClassID class_id);
JS_EXTERN JSValue JS_NewObjectClass(JSContext *ctx, JSClassID class_id);
JS_EXTERN JSValue JS_NewObjectProto(JSContext *ctx, JSValueConst proto);
JS_EXTERN JSValue JS_NewObject(JSContext *ctx);
// takes ownership of the values
JS_EXTERN JSValue JS_NewObjectFrom(JSContext *ctx, int count,
                                   const JSAtom *props,
                                   const JSValue *values);
// takes ownership of the values
JS_EXTERN JSValue JS_NewObjectFromStr(JSContext *ctx, int count,
                                      const char **props,
                                      const JSValue *values);
JS_EXTERN JSValue JS_ToObject(JSContext *ctx, JSValueConst val);
JS_EXTERN JSValue JS_ToObjectString(JSContext *ctx, JSValueConst val);

JS_EXTERN bool JS_IsFunction(JSContext* ctx, JSValueConst val);
JS_EXTERN bool JS_IsConstructor(JSContext* ctx, JSValueConst val);
JS_EXTERN bool JS_SetConstructorBit(JSContext *ctx, JSValueConst func_obj, bool val);

JS_EXTERN bool JS_IsRegExp(JSValueConst val);
JS_EXTERN bool JS_IsMap(JSValueConst val);
JS_EXTERN bool JS_IsSet(JSValueConst val);
JS_EXTERN bool JS_IsWeakRef(JSValueConst val);
JS_EXTERN bool JS_IsWeakSet(JSValueConst val);
JS_EXTERN bool JS_IsWeakMap(JSValueConst val);
JS_EXTERN bool JS_IsDataView(JSValueConst val);

JS_EXTERN JSValue JS_NewArray(JSContext *ctx);
// takes ownership of the values
JS_EXTERN JSValue JS_NewArrayFrom(JSContext *ctx, int count,
                                  const JSValue *values);
// reader beware: JS_IsArray used to "punch" through proxies and check
// if the target object is an array but it no longer does; use JS_IsProxy
// and JS_GetProxyTarget instead, and remember that the target itself can
// also be a proxy, ad infinitum
JS_EXTERN bool JS_IsArray(JSValueConst val);

JS_EXTERN bool JS_IsProxy(JSValueConst val);
JS_EXTERN JSValue JS_GetProxyTarget(JSContext *ctx, JSValueConst proxy);
JS_EXTERN JSValue JS_GetProxyHandler(JSContext *ctx, JSValueConst proxy);

JS_EXTERN JSValue JS_NewDate(JSContext *ctx, double epoch_ms);
JS_EXTERN bool JS_IsDate(JSValueConst v);

JS_EXTERN JSValue JS_GetProperty(JSContext *ctx, JSValueConst this_obj, JSAtom prop);
JS_EXTERN JSValue JS_GetPropertyUint32(JSContext *ctx, JSValueConst this_obj,
                                       uint32_t idx);
JS_EXTERN JSValue JS_GetPropertyInt64(JSContext *ctx, JSValueConst this_obj,
                                      int64_t idx);
JS_EXTERN JSValue JS_GetPropertyStr(JSContext *ctx, JSValueConst this_obj,
                                    const char *prop);

JS_EXTERN int JS_SetProperty(JSContext *ctx, JSValueConst this_obj,
                             JSAtom prop, JSValue val);
JS_EXTERN int JS_SetPropertyUint32(JSContext *ctx, JSValueConst this_obj,
                                   uint32_t idx, JSValue val);
JS_EXTERN int JS_SetPropertyInt64(JSContext *ctx, JSValueConst this_obj,
                                  int64_t idx, JSValue val);
JS_EXTERN int JS_SetPropertyStr(JSContext *ctx, JSValueConst this_obj,
                                const char *prop, JSValue val);
JS_EXTERN int JS_HasProperty(JSContext *ctx, JSValueConst this_obj, JSAtom prop);
JS_EXTERN int JS_IsExtensible(JSContext *ctx, JSValueConst obj);
JS_EXTERN int JS_PreventExtensions(JSContext *ctx, JSValueConst obj);
JS_EXTERN int JS_DeleteProperty(JSContext *ctx, JSValueConst obj, JSAtom prop, int flags);
JS_EXTERN int JS_SetPrototype(JSContext *ctx, JSValueConst obj, JSValueConst proto_val);
JS_EXTERN JSValue JS_GetPrototype(JSContext *ctx, JSValueConst val);
JS_EXTERN int JS_GetLength(JSContext *ctx, JSValueConst obj, int64_t *pres);
JS_EXTERN int JS_SetLength(JSContext *ctx, JSValueConst obj, int64_t len);
JS_EXTERN int JS_SealObject(JSContext *ctx, JSValueConst obj);
JS_EXTERN int JS_FreezeObject(JSContext *ctx, JSValueConst obj);

#define JS_GPN_STRING_MASK  (1 << 0)
#define JS_GPN_SYMBOL_MASK  (1 << 1)
#define JS_GPN_PRIVATE_MASK (1 << 2)
/* only include the enumerable properties */
#define JS_GPN_ENUM_ONLY    (1 << 4)
/* set theJSPropertyEnum.is_enumerable field */
#define JS_GPN_SET_ENUM     (1 << 5)

JS_EXTERN int JS_GetOwnPropertyNames(JSContext *ctx, JSPropertyEnum **ptab,
                                     uint32_t *plen, JSValueConst obj,
                                     int flags);
JS_EXTERN int JS_GetOwnProperty(JSContext *ctx, JSPropertyDescriptor *desc,
                                JSValueConst obj, JSAtom prop);
JS_EXTERN void JS_FreePropertyEnum(JSContext *ctx, JSPropertyEnum *tab,
                                   uint32_t len);

JS_EXTERN JSValue JS_Call(JSContext *ctx, JSValueConst func_obj,
                          JSValueConst this_obj, int argc, JSValueConst *argv);
JS_EXTERN JSValue JS_Invoke(JSContext *ctx, JSValueConst this_val, JSAtom atom,
                            int argc, JSValueConst *argv);
JS_EXTERN JSValue JS_CallConstructor(JSContext *ctx, JSValueConst func_obj,
                                     int argc, JSValueConst *argv);
JS_EXTERN JSValue JS_CallConstructor2(JSContext *ctx, JSValueConst func_obj,
                                      JSValueConst new_target,
                                      int argc, JSValueConst *argv);
/* Try to detect if the input is a module. Returns true if parsing the input
 * as a module produces no syntax errors. It's a naive approach that is not
 * wholly infallible: non-strict classic scripts may _parse_ okay as a module
 * but not _execute_ as one (different runtime semantics.) Use with caution.
 * |input| can be either ASCII or UTF-8 encoded source code.
 * Returns false if QuickJS was built with -DQJS_DISABLE_PARSER.
 */
JS_EXTERN bool JS_DetectModule(const char *input, size_t input_len);
/* 'input' must be zero terminated i.e. input[input_len] = '\0'. */
JS_EXTERN JSValue JS_Eval(JSContext *ctx, const char *input, size_t input_len,
                          const char *filename, int eval_flags);
JS_EXTERN JSValue JS_Eval2(JSContext *ctx, const char *input, size_t input_len,
                           JSEvalOptions *options);
JS_EXTERN JSValue JS_EvalThis(JSContext *ctx, JSValueConst this_obj,
                              const char *input, size_t input_len,
                              const char *filename, int eval_flags);
JS_EXTERN JSValue JS_EvalThis2(JSContext *ctx, JSValueConst this_obj,
                              const char *input, size_t input_len,
                              JSEvalOptions *options);
JS_EXTERN JSValue JS_GetGlobalObject(JSContext *ctx);
JS_EXTERN int JS_IsInstanceOf(JSContext *ctx, JSValueConst val, JSValueConst obj);
JS_EXTERN int JS_DefineProperty(JSContext *ctx, JSValueConst this_obj,
                                JSAtom prop, JSValueConst val,
                                JSValueConst getter, JSValueConst setter,
                                int flags);
JS_EXTERN int JS_DefinePropertyValue(JSContext *ctx, JSValueConst this_obj,
                                     JSAtom prop, JSValue val, int flags);
JS_EXTERN int JS_DefinePropertyValueUint32(JSContext *ctx, JSValueConst this_obj,
                                           uint32_t idx, JSValue val, int flags);
JS_EXTERN int JS_DefinePropertyValueStr(JSContext *ctx, JSValueConst this_obj,
                                        const char *prop, JSValue val, int flags);
JS_EXTERN int JS_DefinePropertyGetSet(JSContext *ctx, JSValueConst this_obj,
                                      JSAtom prop, JSValue getter, JSValue setter,
                                      int flags);
/* Only supported for custom classes, returns 0 on success < 0 otherwise. */
JS_EXTERN int JS_SetOpaque(JSValueConst obj, void *opaque);
JS_EXTERN void *JS_GetOpaque(JSValueConst obj, JSClassID class_id);
JS_EXTERN void *JS_GetOpaque2(JSContext *ctx, JSValueConst obj, JSClassID class_id);
JS_EXTERN void *JS_GetAnyOpaque(JSValueConst obj, JSClassID *class_id);

/* 'buf' must be zero terminated i.e. buf[buf_len] = '\0'. */
JS_EXTERN JSValue JS_ParseJSON(JSContext *ctx, const char *buf, size_t buf_len,
                               const char *filename);
JS_EXTERN JSValue JS_JSONStringify(JSContext *ctx, JSValueConst obj,
                                   JSValueConst replacer, JSValueConst space0);

typedef void JSFreeArrayBufferDataFunc(JSRuntime *rt, void *opaque, void *ptr);
JS_EXTERN JSValue JS_NewArrayBuffer(JSContext *ctx, uint8_t *buf, size_t len,
                                    JSFreeArrayBufferDataFunc *free_func, void *opaque,
                                    bool is_shared);
JS_EXTERN JSValue JS_NewArrayBufferCopy(JSContext *ctx, const uint8_t *buf, size_t len);
JS_EXTERN void JS_DetachArrayBuffer(JSContext *ctx, JSValueConst obj);
JS_EXTERN uint8_t *JS_GetArrayBuffer(JSContext *ctx, size_t *psize, JSValueConst obj);
JS_EXTERN bool JS_IsArrayBuffer(JSValueConst obj);
JS_EXTERN uint8_t *JS_GetUint8Array(JSContext *ctx, size_t *psize, JSValueConst obj);

typedef enum JSTypedArrayEnum {
    JS_TYPED_ARRAY_UINT8C = 0,
    JS_TYPED_ARRAY_INT8,
    JS_TYPED_ARRAY_UINT8,
    JS_TYPED_ARRAY_INT16,
    JS_TYPED_ARRAY_UINT16,
    JS_TYPED_ARRAY_INT32,
    JS_TYPED_ARRAY_UINT32,
    JS_TYPED_ARRAY_BIG_INT64,
    JS_TYPED_ARRAY_BIG_UINT64,
    JS_TYPED_ARRAY_FLOAT16,
    JS_TYPED_ARRAY_FLOAT32,
    JS_TYPED_ARRAY_FLOAT64,
} JSTypedArrayEnum;

JS_EXTERN JSValue JS_NewTypedArray(JSContext *ctx, int argc, JSValueConst *argv,
                                   JSTypedArrayEnum array_type);
JS_EXTERN JSValue JS_GetTypedArrayBuffer(JSContext *ctx, JSValueConst obj,
                                         size_t *pbyte_offset,
                                         size_t *pbyte_length,
                                         size_t *pbytes_per_element);
JS_EXTERN JSValue JS_NewUint8Array(JSContext *ctx, uint8_t *buf, size_t len,
                                   JSFreeArrayBufferDataFunc *free_func, void *opaque,
                                   bool is_shared);
/* returns -1 if not a typed array otherwise return a JSTypedArrayEnum value */
JS_EXTERN int JS_GetTypedArrayType(JSValueConst obj);
JS_EXTERN JSValue JS_NewUint8ArrayCopy(JSContext *ctx, const uint8_t *buf, size_t len);
typedef struct {
    void *(*sab_alloc)(void *opaque, size_t size);
    void (*sab_free)(void *opaque, void *ptr);
    void (*sab_dup)(void *opaque, void *ptr);
    void *sab_opaque;
} JSSharedArrayBufferFunctions;
JS_EXTERN void JS_SetSharedArrayBufferFunctions(JSRuntime *rt, const JSSharedArrayBufferFunctions *sf);

typedef enum JSPromiseStateEnum {
    JS_PROMISE_PENDING,
    JS_PROMISE_FULFILLED,
    JS_PROMISE_REJECTED,
} JSPromiseStateEnum;

JS_EXTERN JSValue JS_NewPromiseCapability(JSContext *ctx, JSValue *resolving_funcs);
JS_EXTERN JSPromiseStateEnum JS_PromiseState(JSContext *ctx,
                                             JSValueConst promise);
JS_EXTERN JSValue JS_PromiseResult(JSContext *ctx, JSValueConst promise);
JS_EXTERN bool JS_IsPromise(JSValueConst val);

JS_EXTERN JSValue JS_NewSymbol(JSContext *ctx, const char *description, bool is_global);

typedef enum JSPromiseHookType {
    JS_PROMISE_HOOK_INIT,     // emitted when a new promise is created
    JS_PROMISE_HOOK_BEFORE,   // runs right before promise.then is invoked
    JS_PROMISE_HOOK_AFTER,    // runs right after promise.then is invoked
    JS_PROMISE_HOOK_RESOLVE,  // not emitted for rejected promises
} JSPromiseHookType;

// parent_promise is only passed in when type == JS_PROMISE_HOOK_INIT and
// is then either a promise object or JS_UNDEFINED if the new promise does
// not have a parent promise; only promises created with promise.then have
// a parent promise
typedef void JSPromiseHook(JSContext *ctx, JSPromiseHookType type,
                           JSValueConst promise, JSValueConst parent_promise,
                           void *opaque);
JS_EXTERN void JS_SetPromiseHook(JSRuntime *rt, JSPromiseHook promise_hook,
                                 void *opaque);

/* is_handled = true means that the rejection is handled */
typedef void JSHostPromiseRejectionTracker(JSContext *ctx, JSValueConst promise,
                                           JSValueConst reason,
                                           bool is_handled, void *opaque);
JS_EXTERN void JS_SetHostPromiseRejectionTracker(JSRuntime *rt, JSHostPromiseRejectionTracker *cb, void *opaque);

/* return != 0 if the JS code needs to be interrupted */
typedef int JSInterruptHandler(JSRuntime *rt, void *opaque);
JS_EXTERN void JS_SetInterruptHandler(JSRuntime *rt, JSInterruptHandler *cb, void *opaque);
/* if can_block is true, Atomics.wait() can be used */
JS_EXTERN void JS_SetCanBlock(JSRuntime *rt, bool can_block);
/* set the [IsHTMLDDA] internal slot */
JS_EXTERN void JS_SetIsHTMLDDA(JSContext *ctx, JSValueConst obj);

typedef struct JSModuleDef JSModuleDef;

/* return the module specifier (allocated with js_malloc()) or NULL if
   exception */
typedef char *JSModuleNormalizeFunc(JSContext *ctx,
                                    const char *module_base_name,
                                    const char *module_name, void *opaque);
typedef JSModuleDef *JSModuleLoaderFunc(JSContext *ctx,
                                        const char *module_name, void *opaque);

/* module_normalize = NULL is allowed and invokes the default module
   filename normalizer */
JS_EXTERN void JS_SetModuleLoaderFunc(JSRuntime *rt,
                                      JSModuleNormalizeFunc *module_normalize,
                                      JSModuleLoaderFunc *module_loader, void *opaque);
/* return the import.meta object of a module */
JS_EXTERN JSValue JS_GetImportMeta(JSContext *ctx, JSModuleDef *m);
JS_EXTERN JSAtom JS_GetModuleName(JSContext *ctx, JSModuleDef *m);
JS_EXTERN JSValue JS_GetModuleNamespace(JSContext *ctx, JSModuleDef *m);

/* JS Job support */

typedef JSValue JSJobFunc(JSContext *ctx, int argc, JSValueConst *argv);
JS_EXTERN int JS_EnqueueJob(JSContext *ctx, JSJobFunc *job_func,
                            int argc, JSValueConst *argv);

JS_EXTERN bool JS_IsJobPending(JSRuntime *rt);
JS_EXTERN int JS_ExecutePendingJob(JSRuntime *rt, JSContext **pctx);

/* Structure to retrieve (de)serialized SharedArrayBuffer objects. */
typedef struct JSSABTab {
    uint8_t **tab;
    size_t len;
} JSSABTab;

/* Object Writer/Reader (currently only used to handle precompiled code) */
#define JS_WRITE_OBJ_BYTECODE  (1 << 0) /* allow function/module */
#define JS_WRITE_OBJ_BSWAP     (0)      /* byte swapped output (obsolete, handled transparently) */
#define JS_WRITE_OBJ_SAB       (1 << 2) /* allow SharedArrayBuffer */
#define JS_WRITE_OBJ_REFERENCE (1 << 3) /* allow object references to encode arbitrary object graph */
#define JS_WRITE_OBJ_STRIP_SOURCE  (1 << 4) /* do not write source code information */
#define JS_WRITE_OBJ_STRIP_DEBUG   (1 << 5) /* do not write debug information */
JS_EXTERN uint8_t *JS_WriteObject(JSContext *ctx, size_t *psize, JSValueConst obj, int flags);
JS_EXTERN uint8_t *JS_WriteObject2(JSContext *ctx, size_t *psize, JSValueConst obj,
                                   int flags, JSSABTab *psab_tab);

#define JS_READ_OBJ_BYTECODE  (1 << 0) /* allow function/module */
#define JS_READ_OBJ_ROM_DATA  (0)      /* avoid duplicating 'buf' data (obsolete, broken by ICs) */
#define JS_READ_OBJ_SAB       (1 << 2) /* allow SharedArrayBuffer */
#define JS_READ_OBJ_REFERENCE (1 << 3) /* allow object references */
JS_EXTERN JSValue JS_ReadObject(JSContext *ctx, const uint8_t *buf, size_t buf_len, int flags);
JS_EXTERN JSValue JS_ReadObject2(JSContext *ctx, const uint8_t *buf, size_t buf_len,
                                 int flags, JSSABTab *psab_tab);
/* instantiate and evaluate a bytecode function. Only used when
   reading a script or module with JS_ReadObject() */
JS_EXTERN JSValue JS_EvalFunction(JSContext *ctx, JSValue fun_obj);
/* load the dependencies of the module 'obj'. Useful when JS_ReadObject()
   returns a module. */
JS_EXTERN int JS_ResolveModule(JSContext *ctx, JSValueConst obj);

/* only exported for os.Worker() */
JS_EXTERN JSAtom JS_GetScriptOrModuleName(JSContext *ctx, int n_stack_levels);
/* only exported for os.Worker() */
JS_EXTERN JSValue JS_LoadModule(JSContext *ctx, const char *basename,
                                const char *filename);

/* C function definition */
typedef enum JSCFunctionEnum {  /* XXX: should rename for namespace isolation */
    JS_CFUNC_generic,
    JS_CFUNC_generic_magic,
    JS_CFUNC_constructor,
    JS_CFUNC_constructor_magic,
    JS_CFUNC_constructor_or_func,
    JS_CFUNC_constructor_or_func_magic,
    JS_CFUNC_f_f,
    JS_CFUNC_f_f_f,
    JS_CFUNC_getter,
    JS_CFUNC_setter,
    JS_CFUNC_getter_magic,
    JS_CFUNC_setter_magic,
    JS_CFUNC_iterator_next,
} JSCFunctionEnum;

typedef union JSCFunctionType {
    JSCFunction *generic;
    JSValue (*generic_magic)(JSContext *ctx, JSValueConst this_val, int argc, JSValueConst *argv, int magic);
    JSCFunction *constructor;
    JSValue (*constructor_magic)(JSContext *ctx, JSValueConst new_target, int argc, JSValueConst *argv, int magic);
    JSCFunction *constructor_or_func;
    double (*f_f)(double);
    double (*f_f_f)(double, double);
    JSValue (*getter)(JSContext *ctx, JSValueConst this_val);
    JSValue (*setter)(JSContext *ctx, JSValueConst this_val, JSValueConst val);
    JSValue (*getter_magic)(JSContext *ctx, JSValueConst this_val, int magic);
    JSValue (*setter_magic)(JSContext *ctx, JSValueConst this_val, JSValueConst val, int magic);
    JSValue (*iterator_next)(JSContext *ctx, JSValueConst this_val,
                             int argc, JSValueConst *argv, int *pdone, int magic);
} JSCFunctionType;

JS_EXTERN JSValue JS_NewCFunction2(JSContext *ctx, JSCFunction *func,
                                   const char *name,
                                   int length, JSCFunctionEnum cproto, int magic);
JS_EXTERN JSValue JS_NewCFunction3(JSContext *ctx, JSCFunction *func,
                                   const char *name,
                                   int length, JSCFunctionEnum cproto, int magic,
                                   JSValueConst proto_val);
JS_EXTERN JSValue JS_NewCFunctionData(JSContext *ctx, JSCFunctionData *func,
                                      int length, int magic, int data_len,
                                      JSValueConst *data);
JS_EXTERN JSValue JS_NewCFunctionData2(JSContext *ctx, JSCFunctionData *func,
                                       const char *name,
                                       int length, int magic, int data_len,
                                       JSValueConst *data);

static inline JSValue JS_NewCFunction(JSContext *ctx, JSCFunction *func,
                                      const char *name, int length)
{
    return JS_NewCFunction2(ctx, func, name, length, JS_CFUNC_generic, 0);
}

static inline JSValue JS_NewCFunctionMagic(JSContext *ctx, JSCFunctionMagic *func,
                                           const char *name, int length,
                                           JSCFunctionEnum cproto, int magic)
{
    /* Used to squelch a -Wcast-function-type warning. */
    JSCFunctionType ft;
    ft.generic_magic = func;
    return JS_NewCFunction2(ctx, ft.generic, name, length, cproto, magic);
}
JS_EXTERN void JS_SetConstructor(JSContext *ctx, JSValueConst func_obj,
                                 JSValueConst proto);

/* C property definition */

typedef struct JSCFunctionListEntry {
    const char *name;       /* pure ASCII or UTF-8 encoded */
    uint8_t prop_flags;
    uint8_t def_type;
    int16_t magic;
    union {
        struct {
            uint8_t length; /* XXX: should move outside union */
            uint8_t cproto; /* XXX: should move outside union */
            JSCFunctionType cfunc;
        } func;
        struct {
            JSCFunctionType get;
            JSCFunctionType set;
        } getset;
        struct {
            const char *name;
            int base;
        } alias;
        struct {
            const struct JSCFunctionListEntry *tab;
            int len;
        } prop_list;
        const char *str;    /* pure ASCII or UTF-8 encoded */
        int32_t i32;
        int64_t i64;
        uint64_t u64;
        double f64;
    } u;
} JSCFunctionListEntry;

#define JS_DEF_CFUNC          0
#define JS_DEF_CGETSET        1
#define JS_DEF_CGETSET_MAGIC  2
#define JS_DEF_PROP_STRING    3
#define JS_DEF_PROP_INT32     4
#define JS_DEF_PROP_INT64     5
#define JS_DEF_PROP_DOUBLE    6
#define JS_DEF_PROP_UNDEFINED 7
#define JS_DEF_OBJECT         8
#define JS_DEF_ALIAS          9

/* Note: c++ does not like nested designators */
#define JS_CFUNC_DEF(name, length, func1) { name, JS_PROP_WRITABLE | JS_PROP_CONFIGURABLE, JS_DEF_CFUNC, 0, { .func = { length, JS_CFUNC_generic, { .generic = func1 } } } }
#define JS_CFUNC_DEF2(name, length, func1, prop_flags) { name, prop_flags, JS_DEF_CFUNC, 0, { .func = { length, JS_CFUNC_generic, { .generic = func1 } } } }
#define JS_CFUNC_MAGIC_DEF(name, length, func1, magic) { name, JS_PROP_WRITABLE | JS_PROP_CONFIGURABLE, JS_DEF_CFUNC, magic, { .func = { length, JS_CFUNC_generic_magic, { .generic_magic = func1 } } } }
#define JS_CFUNC_SPECIAL_DEF(name, length, cproto, func1) { name, JS_PROP_WRITABLE | JS_PROP_CONFIGURABLE, JS_DEF_CFUNC, 0, { .func = { length, JS_CFUNC_ ## cproto, { .cproto = func1 } } } }
#define JS_ITERATOR_NEXT_DEF(name, length, func1, magic) { name, JS_PROP_WRITABLE | JS_PROP_CONFIGURABLE, JS_DEF_CFUNC, magic, { .func = { length, JS_CFUNC_iterator_next, { .iterator_next = func1 } } } }
#define JS_CGETSET_DEF(name, fgetter, fsetter) { name, JS_PROP_CONFIGURABLE, JS_DEF_CGETSET, 0, { .getset = { .get = { .getter = fgetter }, .set = { .setter = fsetter } } } }
#define JS_CGETSET_DEF2(name, fgetter, fsetter, prop_flags) { name, prop_flags, JS_DEF_CGETSET, 0, { .getset = { .get = { .getter = fgetter }, .set = { .setter = fsetter } } } }
#define JS_CGETSET_MAGIC_DEF(name, fgetter, fsetter, magic) { name, JS_PROP_CONFIGURABLE, JS_DEF_CGETSET_MAGIC, magic, { .getset = { .get = { .getter_magic = fgetter }, .set = { .setter_magic = fsetter } } } }
#define JS_PROP_STRING_DEF(name, cstr, prop_flags) { name, prop_flags, JS_DEF_PROP_STRING, 0, { .str = cstr } }
#define JS_PROP_INT32_DEF(name, val, prop_flags) { name, prop_flags, JS_DEF_PROP_INT32, 0, { .i32 = val } }
#define JS_PROP_INT64_DEF(name, val, prop_flags) { name, prop_flags, JS_DEF_PROP_INT64, 0, { .i64 = val } }
#define JS_PROP_DOUBLE_DEF(name, val, prop_flags) { name, prop_flags, JS_DEF_PROP_DOUBLE, 0, { .f64 = val } }
#define JS_PROP_U2D_DEF(name, val, prop_flags) { name, prop_flags, JS_DEF_PROP_DOUBLE, 0, { .u64 = val } }
#define JS_PROP_UNDEFINED_DEF(name, prop_flags) { name, prop_flags, JS_DEF_PROP_UNDEFINED, 0, { .i32 = 0 } }
#define JS_OBJECT_DEF(name, tab, len, prop_flags) { name, prop_flags, JS_DEF_OBJECT, 0, { .prop_list = { tab, len } } }
#define JS_ALIAS_DEF(name, from) { name, JS_PROP_WRITABLE | JS_PROP_CONFIGURABLE, JS_DEF_ALIAS, 0, { .alias = { from, -1 } } }
#define JS_ALIAS_BASE_DEF(name, from, base) { name, JS_PROP_WRITABLE | JS_PROP_CONFIGURABLE, JS_DEF_ALIAS, 0, { .alias = { from, base } } }

JS_EXTERN int JS_SetPropertyFunctionList(JSContext *ctx, JSValueConst obj,
                                          const JSCFunctionListEntry *tab,
                                          int len);

/* C module definition */

typedef int JSModuleInitFunc(JSContext *ctx, JSModuleDef *m);

JS_EXTERN JSModuleDef *JS_NewCModule(JSContext *ctx, const char *name_str,
                                     JSModuleInitFunc *func);
/* can only be called before the module is instantiated */
JS_EXTERN int JS_AddModuleExport(JSContext *ctx, JSModuleDef *m, const char *name_str);
JS_EXTERN int JS_AddModuleExportList(JSContext *ctx, JSModuleDef *m,
                                      const JSCFunctionListEntry *tab, int len);
/* can only be called after the module is instantiated */
JS_EXTERN int JS_SetModuleExport(JSContext *ctx, JSModuleDef *m, const char *export_name,
                                 JSValue val);
JS_EXTERN int JS_SetModuleExportList(JSContext *ctx, JSModuleDef *m,
                                     const JSCFunctionListEntry *tab, int len);

/* Version */

#define QJS_VERSION_MAJOR 0
#define QJS_VERSION_MINOR 11
#define QJS_VERSION_PATCH 0
#define QJS_VERSION_SUFFIX ""

JS_EXTERN const char* JS_GetVersion(void);

/* Integration point for quickjs-libc.c, not for public use. */
JS_EXTERN uintptr_t js_std_cmd(int cmd, ...);

#undef JS_EXTERN
#undef js_force_inline

#ifdef __cplusplus
} /* extern "C" { */
#endif

#endif /* QUICKJS_H */
/*
 * QuickJS Javascript Engine
 *
 * Copyright (c) 2017-2025 Fabrice Bellard
 * Copyright (c) 2017-2025 Charlie Gordon
 * Copyright (c) 2023-2025 Ben Noordhuis
 * Copyright (c) 2023-2025 Saúl Ibarra Corretgé
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <inttypes.h>
#include <string.h>
#include <assert.h>
#if !defined(_MSC_VER)
#include <sys/time.h>
#if defined(_WIN32)
#include <timezoneapi.h>
#endif
#endif
#if defined(_WIN32)
#include <intrin.h>
#endif
#include <time.h>
#include <fenv.h>
#include <math.h>







#if defined(EMSCRIPTEN) || defined(_MSC_VER)
#define DIRECT_DISPATCH  0
#else
#define DIRECT_DISPATCH  1
#endif

#if defined(__APPLE__)
#define MALLOC_OVERHEAD  0
#else
#define MALLOC_OVERHEAD  8
#endif

#if defined(__NEWLIB__)
#define NO_TM_GMTOFF
#endif

// atomic_store etc. are completely busted in recent versions of tcc;
// somehow the compiler forgets to load |ptr| into %rdi when calling
// the __atomic_*() helpers in its lib/stdatomic.c and lib/atomic.S
#if !defined(__TINYC__) && !defined(EMSCRIPTEN) && !defined(__wasi__) && !__STDC_NO_ATOMICS__

#define CONFIG_ATOMICS
#endif

#ifndef __GNUC__
#define __extension__
#endif

#ifndef NDEBUG
#define ENABLE_DUMPS
#endif

//#define FORCE_GC_AT_MALLOC  /* test the GC by forcing it before each object allocation */

#define check_dump_flag(rt, flag)  ((rt->dump_flags & (flag +0)) == (flag +0))

#define STRINGIFY_(x) #x
#define STRINGIFY(x)  STRINGIFY_(x)

#define QJS_VERSION_STRING \
    STRINGIFY(QJS_VERSION_MAJOR) "." STRINGIFY(QJS_VERSION_MINOR) "." STRINGIFY(QJS_VERSION_PATCH) QJS_VERSION_SUFFIX

const char* JS_GetVersion(void) {
    return QJS_VERSION_STRING;
}

#undef STRINFIGY_
#undef STRINGIFY

static inline JSValueConst *vc(JSValue *vals)
{
    return (JSValueConst *)vals;
}

static inline JSValue unsafe_unconst(JSValueConst v)
{
#ifdef JS_CHECK_JSVALUE
    return (JSValue)v;
#else
    return v;
#endif
}

static inline JSValueConst safe_const(JSValue v)
{
#ifdef JS_CHECK_JSVALUE
    return (JSValueConst)v;
#else
    return v;
#endif
}

enum {
    /* classid tag        */    /* union usage   | properties */
    JS_CLASS_OBJECT = 1,        /* must be first */
    JS_CLASS_ARRAY,             /* u.array       | length */
    JS_CLASS_ERROR,
    JS_CLASS_NUMBER,            /* u.object_data */
    JS_CLASS_STRING,            /* u.object_data */
    JS_CLASS_BOOLEAN,           /* u.object_data */
    JS_CLASS_SYMBOL,            /* u.object_data */
    JS_CLASS_ARGUMENTS,         /* u.array       | length */
    JS_CLASS_MAPPED_ARGUMENTS,  /*               | length */
    JS_CLASS_DATE,              /* u.object_data */
    JS_CLASS_MODULE_NS,
    JS_CLASS_C_FUNCTION,        /* u.cfunc */
    JS_CLASS_BYTECODE_FUNCTION, /* u.func */
    JS_CLASS_BOUND_FUNCTION,    /* u.bound_function */
    JS_CLASS_C_FUNCTION_DATA,   /* u.c_function_data_record */
    JS_CLASS_GENERATOR_FUNCTION, /* u.func */
    JS_CLASS_FOR_IN_ITERATOR,   /* u.for_in_iterator */
    JS_CLASS_REGEXP,            /* u.regexp */
    JS_CLASS_ARRAY_BUFFER,      /* u.array_buffer */
    JS_CLASS_SHARED_ARRAY_BUFFER, /* u.array_buffer */
    JS_CLASS_UINT8C_ARRAY,      /* u.array (typed_array) */
    JS_CLASS_INT8_ARRAY,        /* u.array (typed_array) */
    JS_CLASS_UINT8_ARRAY,       /* u.array (typed_array) */
    JS_CLASS_INT16_ARRAY,       /* u.array (typed_array) */
    JS_CLASS_UINT16_ARRAY,      /* u.array (typed_array) */
    JS_CLASS_INT32_ARRAY,       /* u.array (typed_array) */
    JS_CLASS_UINT32_ARRAY,      /* u.array (typed_array) */
    JS_CLASS_BIG_INT64_ARRAY,   /* u.array (typed_array) */
    JS_CLASS_BIG_UINT64_ARRAY,  /* u.array (typed_array) */
    JS_CLASS_FLOAT16_ARRAY,     /* u.array (typed_array) */
    JS_CLASS_FLOAT32_ARRAY,     /* u.array (typed_array) */
    JS_CLASS_FLOAT64_ARRAY,     /* u.array (typed_array) */
    JS_CLASS_DATAVIEW,          /* u.typed_array */
    JS_CLASS_BIG_INT,           /* u.object_data */
    JS_CLASS_MAP,               /* u.map_state */
    JS_CLASS_SET,               /* u.map_state */
    JS_CLASS_WEAKMAP,           /* u.map_state */
    JS_CLASS_WEAKSET,           /* u.map_state */
    JS_CLASS_ITERATOR,
    JS_CLASS_ITERATOR_CONCAT,   /* u.iterator_concat_data */
    JS_CLASS_ITERATOR_HELPER,   /* u.iterator_helper_data */
    JS_CLASS_ITERATOR_WRAP,     /* u.iterator_wrap_data */
    JS_CLASS_MAP_ITERATOR,      /* u.map_iterator_data */
    JS_CLASS_SET_ITERATOR,      /* u.map_iterator_data */
    JS_CLASS_ARRAY_ITERATOR,    /* u.array_iterator_data */
    JS_CLASS_STRING_ITERATOR,   /* u.array_iterator_data */
    JS_CLASS_REGEXP_STRING_ITERATOR,   /* u.regexp_string_iterator_data */
    JS_CLASS_GENERATOR,         /* u.generator_data */
    JS_CLASS_PROXY,             /* u.proxy_data */
    JS_CLASS_PROMISE,           /* u.promise_data */
    JS_CLASS_PROMISE_RESOLVE_FUNCTION,  /* u.promise_function_data */
    JS_CLASS_PROMISE_REJECT_FUNCTION,   /* u.promise_function_data */
    JS_CLASS_ASYNC_FUNCTION,            /* u.func */
    JS_CLASS_ASYNC_FUNCTION_RESOLVE,    /* u.async_function_data */
    JS_CLASS_ASYNC_FUNCTION_REJECT,     /* u.async_function_data */
    JS_CLASS_ASYNC_FROM_SYNC_ITERATOR,  /* u.async_from_sync_iterator_data */
    JS_CLASS_ASYNC_GENERATOR_FUNCTION,  /* u.func */
    JS_CLASS_ASYNC_GENERATOR,   /* u.async_generator_data */
    JS_CLASS_WEAK_REF,
    JS_CLASS_FINALIZATION_REGISTRY,
    JS_CLASS_DOM_EXCEPTION,
    JS_CLASS_CALL_SITE,

    JS_CLASS_INIT_COUNT, /* last entry for predefined classes */
};

/* number of typed array types */
#define JS_TYPED_ARRAY_COUNT  (JS_CLASS_FLOAT64_ARRAY - JS_CLASS_UINT8C_ARRAY + 1)
static uint8_t const typed_array_size_log2[JS_TYPED_ARRAY_COUNT];
#define typed_array_size_log2(classid)  (typed_array_size_log2[(classid)- JS_CLASS_UINT8C_ARRAY])

typedef enum JSErrorEnum {
    JS_EVAL_ERROR,
    JS_RANGE_ERROR,
    JS_REFERENCE_ERROR,
    JS_SYNTAX_ERROR,
    JS_TYPE_ERROR,
    JS_URI_ERROR,
    JS_INTERNAL_ERROR,
    JS_AGGREGATE_ERROR,

    JS_NATIVE_ERROR_COUNT, /* number of different NativeError objects */
    JS_PLAIN_ERROR = JS_NATIVE_ERROR_COUNT
} JSErrorEnum;

#define JS_MAX_LOCAL_VARS 65535
#define JS_STACK_SIZE_MAX 65534
#define JS_STRING_LEN_MAX ((1 << 30) - 1)
// 1,024 bytes is about the cutoff point where it starts getting
// more profitable to ref slice than to copy
#define JS_STRING_SLICE_LEN_MAX 1024 // in bytes

#define __exception __attribute__((warn_unused_result))

typedef struct JSShape JSShape;
typedef struct JSString JSString;
typedef struct JSString JSAtomStruct;

#define JS_VALUE_GET_STRING(v) ((JSString *)JS_VALUE_GET_PTR(v))

typedef enum {
    JS_GC_PHASE_NONE,
    JS_GC_PHASE_DECREF,
    JS_GC_PHASE_REMOVE_CYCLES,
} JSGCPhaseEnum;

typedef struct JSMallocState {
    size_t malloc_count;
    size_t malloc_size;
    size_t malloc_limit;
    void *opaque; /* user opaque */
} JSMallocState;

typedef struct JSRuntimeFinalizerState {
    struct JSRuntimeFinalizerState *next;
    JSRuntimeFinalizer *finalizer;
    void *arg;
} JSRuntimeFinalizerState;

typedef struct JSValueLink {
    struct JSValueLink *next;
    JSValueConst value;
} JSValueLink;

struct JSRuntime {
    JSMallocFunctions mf;
    JSMallocState malloc_state;
    const char *rt_info;

    int atom_hash_size; /* power of two */
    int atom_count;
    int atom_size;
    int atom_count_resize; /* resize hash table at this count */
    uint32_t *atom_hash;
    JSAtomStruct **atom_array;
    int atom_free_index; /* 0 = none */

    JSClassID js_class_id_alloc; /* counter for user defined classes */
    int class_count;    /* size of class_array */
    JSClass *class_array;

    struct list_head context_list; /* list of JSContext.link */
    /* list of JSGCObjectHeader.link. List of allocated GC objects (used
       by the garbage collector) */
    struct list_head gc_obj_list;
    /* list of JSGCObjectHeader.link. Used during JS_FreeValueRT() */
    struct list_head gc_zero_ref_count_list;
    struct list_head tmp_obj_list; /* used during GC */
    JSGCPhaseEnum gc_phase : 8;
    size_t malloc_gc_threshold;
#ifdef ENABLE_DUMPS // JS_DUMP_LEAKS
    struct list_head string_list; /* list of JSString.link */
#endif
    /* stack limitation */
    uintptr_t stack_size; /* in bytes, 0 if no limit */
    uintptr_t stack_top;
    uintptr_t stack_limit; /* lower stack limit */

    JSValue current_exception;
    /* true if inside an out of memory error, to avoid recursing */
    bool in_out_of_memory;
    /* true if inside build_backtrace, to avoid recursing */
    bool in_build_stack_trace;
    /* true if inside JS_FreeRuntime */
    bool in_free;

    struct JSStackFrame *current_stack_frame;

    JSInterruptHandler *interrupt_handler;
    void *interrupt_opaque;

    JSPromiseHook *promise_hook;
    void *promise_hook_opaque;
    // for smuggling the parent promise from js_promise_then
    // to js_promise_constructor
    JSValueLink *parent_promise;

    JSHostPromiseRejectionTracker *host_promise_rejection_tracker;
    void *host_promise_rejection_tracker_opaque;

    struct list_head job_list; /* list of JSJobEntry.link */

    JSModuleNormalizeFunc *module_normalize_func;
    JSModuleLoaderFunc *module_loader_func;
    void *module_loader_opaque;
    /* timestamp for internal use in module evaluation */
    int64_t module_async_evaluation_next_timestamp;

    /* used to allocate, free and clone SharedArrayBuffers */
    JSSharedArrayBufferFunctions sab_funcs;

    bool can_block; /* true if Atomics.wait can block */
    uint32_t dump_flags : 24;

    /* Shape hash table */
    int shape_hash_bits;
    int shape_hash_size;
    int shape_hash_count; /* number of hashed shapes */
    JSShape **shape_hash;
    void *user_opaque;
    void *libc_opaque;
    JSRuntimeFinalizerState *finalizers;
};

struct JSClass {
    uint32_t class_id; /* 0 means free entry */
    JSAtom class_name;
    JSClassFinalizer *finalizer;
    JSClassGCMark *gc_mark;
    JSClassCall *call;
    /* pointers for exotic behavior, can be NULL if none are present */
    const JSClassExoticMethods *exotic;
};

typedef struct JSStackFrame {
    struct JSStackFrame *prev_frame; /* NULL if first stack frame */
    JSValue cur_func; /* current function, JS_UNDEFINED if the frame is detached */
    JSValue *arg_buf; /* arguments */
    JSValue *var_buf; /* variables */
    struct list_head var_ref_list; /* list of JSVarRef.link */
    uint8_t *cur_pc; /* only used in bytecode functions : PC of the
                        instruction after the call */
    uint32_t arg_count : 31;
    uint32_t is_strict_mode : 1;
    /* only used in generators. Current stack pointer value. NULL if
       the function is running. */
    JSValue *cur_sp;
} JSStackFrame;

typedef enum {
    JS_GC_OBJ_TYPE_JS_OBJECT,
    JS_GC_OBJ_TYPE_FUNCTION_BYTECODE,
    JS_GC_OBJ_TYPE_SHAPE,
    JS_GC_OBJ_TYPE_VAR_REF,
    JS_GC_OBJ_TYPE_ASYNC_FUNCTION,
    JS_GC_OBJ_TYPE_JS_CONTEXT,
} JSGCObjectTypeEnum;

/* header for GC objects. GC objects are C data structures with a
   reference count that can reference other GC objects. JS Objects are
   a particular type of GC object. */
struct JSGCObjectHeader {
    int ref_count; /* must come first, 32-bit */
    JSGCObjectTypeEnum gc_obj_type : 4;
    uint8_t mark : 4; /* used by the GC */
    uint8_t dummy1; /* not used by the GC */
    uint16_t dummy2; /* not used by the GC */
    struct list_head link;
};

typedef struct JSVarRef {
    union {
        JSGCObjectHeader header; /* must come first */
        struct {
            int __gc_ref_count; /* corresponds to header.ref_count */
            uint8_t __gc_mark; /* corresponds to header.mark/gc_obj_type */
            uint8_t is_detached;
        };
    };
    JSValue *pvalue; /* pointer to the value, either on the stack or
                        to 'value' */
    JSValue value; /* used when the variable is no longer on the stack */
} JSVarRef;

typedef struct JSRefCountHeader {
    int ref_count;
} JSRefCountHeader;

/* bigint */
typedef int32_t js_slimb_t;
typedef uint32_t js_limb_t;
typedef int64_t js_sdlimb_t;
typedef uint64_t js_dlimb_t;

#define JS_LIMB_DIGITS 9

/* Must match the size of short_big_int in JSValueUnion */
#define JS_LIMB_BITS 32
#define JS_SHORT_BIG_INT_BITS JS_LIMB_BITS
#define JS_BIGINT_MAX_SIZE ((1024 * 1024) / JS_LIMB_BITS) /* in limbs */
#define JS_SHORT_BIG_INT_MIN INT32_MIN
#define JS_SHORT_BIG_INT_MAX INT32_MAX


typedef struct JSBigInt {
    JSRefCountHeader header; /* must come first, 32-bit */
    uint32_t len; /* number of limbs, >= 1 */
    js_limb_t tab[]; /* two's complement representation, always
                        normalized so that 'len' is the minimum
                        possible length >= 1 */
} JSBigInt;

/* this bigint structure can hold a 64 bit integer */
typedef struct {
    js_limb_t big_int_buf[sizeof(JSBigInt) / sizeof(js_limb_t)]; /* for JSBigInt */
    /* must come just after */
    js_limb_t tab[(64 + JS_LIMB_BITS - 1) / JS_LIMB_BITS];
} JSBigIntBuf;

typedef enum {
    JS_AUTOINIT_ID_PROTOTYPE,
    JS_AUTOINIT_ID_MODULE_NS,
    JS_AUTOINIT_ID_PROP,
    JS_AUTOINIT_ID_BYTECODE,
} JSAutoInitIDEnum;

enum {
    JS_BUILTIN_ARRAY_FROMASYNC = 1,
};

/* must be large enough to have a negligible runtime cost and small
   enough to call the interrupt callback often. */
#define JS_INTERRUPT_COUNTER_INIT 10000

struct JSContext {
    JSGCObjectHeader header; /* must come first */
    JSRuntime *rt;
    struct list_head link;

    uint16_t binary_object_count;
    int binary_object_size;

    JSShape *array_shape;   /* initial shape for Array objects */

    JSValue *class_proto;
    JSValue function_proto;
    JSValue function_ctor;
    JSValue array_ctor;
    JSValue regexp_ctor;
    JSValue promise_ctor;
    JSValue native_error_proto[JS_NATIVE_ERROR_COUNT];
    JSValue error_ctor;
    JSValue error_back_trace;
    JSValue error_prepare_stack;
    JSValue error_stack_trace_limit;
    JSValue iterator_ctor;
    JSValue iterator_ctor_getset;
    JSValue iterator_proto;
    JSValue async_iterator_proto;
    JSValue array_proto_values;
    JSValue throw_type_error;
    JSValue eval_obj;

    JSValue global_obj; /* global object */
    JSValue global_var_obj; /* contains the global let/const definitions */

    double time_origin;

    uint64_t random_state;

    /* when the counter reaches zero, JSRutime.interrupt_handler is called */
    int interrupt_counter;

    struct list_head loaded_modules; /* list of JSModuleDef.link */

    /* if NULL, RegExp compilation is not supported */
    JSValue (*compile_regexp)(JSContext *ctx, JSValueConst pattern,
                              JSValueConst flags);
    /* if NULL, eval is not supported */
    JSValue (*eval_internal)(JSContext *ctx, JSValueConst this_obj,
                             const char *input, size_t input_len,
                             const char *filename, int line, int flags, int scope_idx);
    void *user_opaque;
};

typedef union JSFloat64Union {
    double d;
    uint64_t u64;
    uint32_t u32[2];
} JSFloat64Union;

typedef enum {
    JS_WEAK_REF_KIND_MAP,
    JS_WEAK_REF_KIND_WEAK_REF,
    JS_WEAK_REF_KIND_FINALIZATION_REGISTRY_ENTRY,
} JSWeakRefKindEnum;

typedef struct JSWeakRefRecord {
    JSWeakRefKindEnum kind;
    struct JSWeakRefRecord *next_weak_ref;
    union {
        struct JSMapRecord *map_record;
        struct JSWeakRefData *weak_ref_data;
        struct JSFinRecEntry *fin_rec_entry;
    } u;
} JSWeakRefRecord;

typedef struct JSMapRecord {
    int ref_count; /* used during enumeration to avoid freeing the record */
    bool empty; /* true if the record is deleted */
    struct JSMapState *map;
    struct list_head link;
    struct list_head hash_link;
    JSValue key;
    JSValue value;
} JSMapRecord;

typedef struct JSMapState {
    bool is_weak; /* true if WeakSet/WeakMap */
    struct list_head records; /* list of JSMapRecord.link */
    uint32_t record_count;
    struct list_head *hash_table;
    uint32_t hash_size; /* must be a power of two */
    uint32_t record_count_threshold; /* count at which a hash table
                                        resize is needed */
} JSMapState;

enum
{
    JS_TO_STRING_IS_PROPERTY_KEY = 1 << 0,
    JS_TO_STRING_NO_SIDE_EFFECTS = 1 << 1,
};

enum {
    JS_ATOM_TYPE_STRING = 1,
    JS_ATOM_TYPE_GLOBAL_SYMBOL,
    JS_ATOM_TYPE_SYMBOL,
    JS_ATOM_TYPE_PRIVATE,
};

enum {
    JS_ATOM_HASH_SYMBOL,
    JS_ATOM_HASH_PRIVATE,
};

typedef enum {
    JS_ATOM_KIND_STRING,
    JS_ATOM_KIND_SYMBOL,
    JS_ATOM_KIND_PRIVATE,
} JSAtomKindEnum;

typedef enum {
    JS_STRING_KIND_NORMAL,
    JS_STRING_KIND_SLICE,
} JSStringKind;

#define JS_ATOM_HASH_MASK  ((1 << 29) - 1)

struct JSString {
    JSRefCountHeader header; /* must come first, 32-bit */
    uint32_t len : 31;
    uint8_t is_wide_char : 1; /* 0 = 8 bits, 1 = 16 bits characters */
    /* for JS_ATOM_TYPE_SYMBOL: hash = 0, atom_type = 3,
       for JS_ATOM_TYPE_PRIVATE: hash = 1, atom_type = 3
       XXX: could change encoding to have one more bit in hash */
    uint32_t hash : 29;
    uint8_t kind : 1;
    uint8_t atom_type : 2; /* != 0 if atom, JS_ATOM_TYPE_x */
    uint32_t hash_next; /* atom_index for JS_ATOM_TYPE_SYMBOL */
    JSWeakRefRecord *first_weak_ref;
#ifdef ENABLE_DUMPS // JS_DUMP_LEAKS
    struct list_head link; /* string list */
#endif
};

typedef struct JSStringSlice {
    JSString *parent;
    uint32_t start; // in bytes, not characters
} JSStringSlice;

static inline void *strv(JSString *p)
{
    JSStringSlice *slice;

    switch (p->kind) {
    case JS_STRING_KIND_NORMAL:
        return (void *)&p[1];
    case JS_STRING_KIND_SLICE:
        slice = (void *)&p[1];
        return (char *)&slice->parent[1] + slice->start;
    }
    abort();
    return NULL;
}

static inline uint8_t *str8(JSString *p)
{
    return strv(p);
}

static inline uint16_t *str16(JSString *p)
{
    return strv(p);
}

typedef struct JSClosureVar {
    uint8_t is_local : 1;
    uint8_t is_arg : 1;
    uint8_t is_const : 1;
    uint8_t is_lexical : 1;
    uint8_t var_kind : 4; /* see JSVarKindEnum */
    /* 8 bits available */
    uint16_t var_idx; /* is_local = true: index to a normal variable of the
                    parent function. otherwise: index to a closure
                    variable of the parent function */
    JSAtom var_name;
} JSClosureVar;

#define ARG_SCOPE_INDEX 1
#define ARG_SCOPE_END (-2)

typedef struct JSVarScope {
    int parent;  /* index into fd->scopes of the enclosing scope */
    int first;   /* index into fd->vars of the last variable in this scope */
} JSVarScope;

typedef enum {
    /* XXX: add more variable kinds here instead of using bit fields */
    JS_VAR_NORMAL,
    JS_VAR_FUNCTION_DECL, /* lexical var with function declaration */
    JS_VAR_NEW_FUNCTION_DECL, /* lexical var with async/generator
                                 function declaration */
    JS_VAR_CATCH,
    JS_VAR_FUNCTION_NAME, /* function expression name */
    JS_VAR_PRIVATE_FIELD,
    JS_VAR_PRIVATE_METHOD,
    JS_VAR_PRIVATE_GETTER,
    JS_VAR_PRIVATE_SETTER, /* must come after JS_VAR_PRIVATE_GETTER */
    JS_VAR_PRIVATE_GETTER_SETTER, /* must come after JS_VAR_PRIVATE_SETTER */
} JSVarKindEnum;

/* XXX: could use a different structure in bytecode functions to save
   memory */
typedef struct JSVarDef {
    JSAtom var_name;
    /* index into fd->scopes of this variable lexical scope */
    int scope_level;
    /* during compilation:
        - if scope_level = 0: scope in which the variable is defined
        - if scope_level != 0: index into fd->vars of the next
          variable in the same or enclosing lexical scope
       in a bytecode function:
       index into fd->vars of the next
       variable in the same or enclosing lexical scope
    */
    int scope_next;
    uint8_t is_const : 1;
    uint8_t is_lexical : 1;
    uint8_t is_captured : 1;
    uint8_t is_static_private : 1; /* only used during private class field parsing */
    uint8_t var_kind : 4; /* see JSVarKindEnum */
    /* only used during compilation: function pool index for lexical
       variables with var_kind =
       JS_VAR_FUNCTION_DECL/JS_VAR_NEW_FUNCTION_DECL or scope level of
       the definition of the 'var' variables (they have scope_level =
       0) */
    int func_pool_idx : 24; /* only used during compilation : index in
                               the constant pool for hoisted function
                               definition */
} JSVarDef;

/* for the encoding of the pc2line table */
#define PC2LINE_BASE     (-1)
#define PC2LINE_RANGE    5
#define PC2LINE_OP_FIRST 1
#define PC2LINE_DIFF_PC_MAX ((255 - PC2LINE_OP_FIRST) / PC2LINE_RANGE)

typedef enum JSFunctionKindEnum {
    JS_FUNC_NORMAL = 0,
    JS_FUNC_GENERATOR = (1 << 0),
    JS_FUNC_ASYNC = (1 << 1),
    JS_FUNC_ASYNC_GENERATOR = (JS_FUNC_GENERATOR | JS_FUNC_ASYNC),
} JSFunctionKindEnum;

typedef struct JSFunctionBytecode {
    JSGCObjectHeader header; /* must come first */
    uint8_t is_strict_mode : 1;
    uint8_t has_prototype : 1; /* true if a prototype field is necessary */
    uint8_t has_simple_parameter_list : 1;
    uint8_t is_derived_class_constructor : 1;
    /* true if home_object needs to be initialized */
    uint8_t need_home_object : 1;
    uint8_t func_kind : 2;
    uint8_t new_target_allowed : 1;
    uint8_t super_call_allowed : 1;
    uint8_t super_allowed : 1;
    uint8_t arguments_allowed : 1;
    uint8_t backtrace_barrier : 1; /* stop backtrace on this function */
    /* XXX: 5 bits available */
    uint8_t *byte_code_buf; /* (self pointer) */
    int byte_code_len;
    JSAtom func_name;
    JSVarDef *vardefs; /* arguments + local variables (arg_count + var_count) (self pointer) */
    JSClosureVar *closure_var; /* list of variables in the closure (self pointer) */
    uint16_t arg_count;
    uint16_t var_count;
    uint16_t defined_arg_count; /* for length function property */
    uint16_t stack_size; /* maximum stack size */
    JSContext *realm; /* function realm */
    JSValue *cpool; /* constant pool (self pointer) */
    int cpool_count;
    int closure_var_count;
    JSAtom filename;
    int line_num;
    int col_num;
    int source_len;
    int pc2line_len;
    uint8_t *pc2line_buf;
    char *source;
} JSFunctionBytecode;

typedef struct JSBoundFunction {
    JSValue func_obj;
    JSValue this_val;
    int argc;
    JSValue argv[];
} JSBoundFunction;

typedef enum JSIteratorKindEnum {
    JS_ITERATOR_KIND_KEY,
    JS_ITERATOR_KIND_VALUE,
    JS_ITERATOR_KIND_KEY_AND_VALUE,
} JSIteratorKindEnum;

typedef enum JSIteratorHelperKindEnum {
    JS_ITERATOR_HELPER_KIND_DROP,
    JS_ITERATOR_HELPER_KIND_EVERY,
    JS_ITERATOR_HELPER_KIND_FILTER,
    JS_ITERATOR_HELPER_KIND_FIND,
    JS_ITERATOR_HELPER_KIND_FLAT_MAP,
    JS_ITERATOR_HELPER_KIND_FOR_EACH,
    JS_ITERATOR_HELPER_KIND_MAP,
    JS_ITERATOR_HELPER_KIND_SOME,
    JS_ITERATOR_HELPER_KIND_TAKE,
} JSIteratorHelperKindEnum;

typedef struct JSForInIterator {
    JSValue obj;
    bool is_array;
    uint32_t array_length;
    uint32_t idx;
} JSForInIterator;

typedef struct JSRegExp {
    JSString *pattern;
    JSString *bytecode; /* also contains the flags */
} JSRegExp;

typedef struct JSProxyData {
    JSValue target;
    JSValue handler;
    uint8_t is_func;
    uint8_t is_revoked;
} JSProxyData;

typedef struct JSArrayBuffer {
    int byte_length; /* 0 if detached */
    int max_byte_length; /* -1 if not resizable; >= byte_length otherwise */
    uint8_t detached;
    uint8_t shared; /* if shared, the array buffer cannot be detached */
    uint8_t *data; /* NULL if detached */
    struct list_head array_list;
    void *opaque;
    JSFreeArrayBufferDataFunc *free_func;
} JSArrayBuffer;

typedef struct JSTypedArray {
    struct list_head link; /* link to arraybuffer */
    JSObject *obj; /* back pointer to the TypedArray/DataView object */
    JSObject *buffer; /* based array buffer */
    uint32_t offset; /* byte offset in the array buffer */
    uint32_t length; /* byte length in the array buffer */
    bool track_rab; /* auto-track length of backing array buffer */
} JSTypedArray;

typedef struct JSAsyncFunctionState {
    JSValue this_val; /* 'this' generator argument */
    int argc; /* number of function arguments */
    bool throw_flag; /* used to throw an exception in JS_CallInternal() */
    JSStackFrame frame;
} JSAsyncFunctionState;

/* XXX: could use an object instead to avoid the
   JS_TAG_ASYNC_FUNCTION tag for the GC */
typedef struct JSAsyncFunctionData {
    JSGCObjectHeader header; /* must come first */
    JSValue resolving_funcs[2];
    bool is_active; /* true if the async function state is valid */
    JSAsyncFunctionState func_state;
} JSAsyncFunctionData;

typedef struct JSReqModuleEntry {
    JSAtom module_name;
    JSModuleDef *module; /* used using resolution */
} JSReqModuleEntry;

typedef enum JSExportTypeEnum {
    JS_EXPORT_TYPE_LOCAL,
    JS_EXPORT_TYPE_INDIRECT,
} JSExportTypeEnum;

typedef struct JSExportEntry {
    union {
        struct {
            int var_idx; /* closure variable index */
            JSVarRef *var_ref; /* if != NULL, reference to the variable */
        } local; /* for local export */
        int req_module_idx; /* module for indirect export */
    } u;
    JSExportTypeEnum export_type;
    JSAtom local_name; /* '*' if export ns from. not used for local
                          export after compilation */
    JSAtom export_name; /* exported variable name */
} JSExportEntry;

typedef struct JSStarExportEntry {
    int req_module_idx; /* in req_module_entries */
} JSStarExportEntry;

typedef struct JSImportEntry {
    int var_idx; /* closure variable index */
    JSAtom import_name;
    int req_module_idx; /* in req_module_entries */
} JSImportEntry;

typedef enum {
    JS_MODULE_STATUS_UNLINKED,
    JS_MODULE_STATUS_LINKING,
    JS_MODULE_STATUS_LINKED,
    JS_MODULE_STATUS_EVALUATING,
    JS_MODULE_STATUS_EVALUATING_ASYNC,
    JS_MODULE_STATUS_EVALUATED,
} JSModuleStatus;

struct JSModuleDef {
    JSRefCountHeader header; /* must come first, 32-bit */
    JSAtom module_name;
    struct list_head link;

    JSReqModuleEntry *req_module_entries;
    int req_module_entries_count;
    int req_module_entries_size;

    JSExportEntry *export_entries;
    int export_entries_count;
    int export_entries_size;

    JSStarExportEntry *star_export_entries;
    int star_export_entries_count;
    int star_export_entries_size;

    JSImportEntry *import_entries;
    int import_entries_count;
    int import_entries_size;

    JSValue module_ns;
    JSValue func_obj; /* only used for JS modules */
    JSModuleInitFunc *init_func; /* only used for C modules */
    bool has_tla; /* true if func_obj contains await */
    bool resolved;
    bool func_created;
    JSModuleStatus status : 8;
    /* temp use during js_module_link() & js_module_evaluate() */
    int dfs_index, dfs_ancestor_index;
    JSModuleDef *stack_prev;
    /* temp use during js_module_evaluate() */
    JSModuleDef **async_parent_modules;
    int async_parent_modules_count;
    int async_parent_modules_size;
    int pending_async_dependencies;
    bool async_evaluation;
    int64_t async_evaluation_timestamp;
    JSModuleDef *cycle_root;
    JSValue promise; /* corresponds to spec field: capability */
    JSValue resolving_funcs[2]; /* corresponds to spec field: capability */
    /* true if evaluation yielded an exception. It is saved in
       eval_exception */
    bool eval_has_exception;
    JSValue eval_exception;
    JSValue meta_obj; /* for import.meta */
};

typedef struct JSJobEntry {
    struct list_head link;
    JSContext *ctx;
    JSJobFunc *job_func;
    int argc;
    JSValue argv[];
} JSJobEntry;

typedef struct JSProperty {
    union {
        JSValue value;      /* JS_PROP_NORMAL */
        struct {            /* JS_PROP_GETSET */
            JSObject *getter; /* NULL if undefined */
            JSObject *setter; /* NULL if undefined */
        } getset;
        JSVarRef *var_ref;  /* JS_PROP_VARREF */
        struct {            /* JS_PROP_AUTOINIT */
            /* in order to use only 2 pointers, we compress the realm
               and the init function pointer */
            uintptr_t realm_and_id; /* realm and init_id (JS_AUTOINIT_ID_x)
                                       in the 2 low bits */
            void *opaque;
        } init;
    } u;
} JSProperty;

#define JS_PROP_INITIAL_SIZE 2
#define JS_PROP_INITIAL_HASH_SIZE 4 /* must be a power of two */
#define JS_ARRAY_INITIAL_SIZE 2

typedef struct JSShapeProperty {
    uint32_t hash_next : 26; /* 0 if last in list */
    uint32_t flags : 6;   /* JS_PROP_XXX */
    JSAtom atom; /* JS_ATOM_NULL = free property entry */
} JSShapeProperty;

struct JSShape {
    /* hash table of size hash_mask + 1 before the start of the
       structure (see prop_hash_end()). */
    JSGCObjectHeader header;
    /* true if the shape is inserted in the shape hash table. If not,
       JSShape.hash is not valid */
    uint8_t is_hashed;
    /* If true, the shape may have small array index properties 'n' with 0
       <= n <= 2^31-1. If false, the shape is guaranteed not to have
       small array index properties */
    uint8_t has_small_array_index;
    uint32_t hash; /* current hash value */
    uint32_t prop_hash_mask;
    int prop_size; /* allocated properties */
    int prop_count; /* include deleted properties */
    int deleted_prop_count;
    JSShape *shape_hash_next; /* in JSRuntime.shape_hash[h] list */
    JSObject *proto;
    JSShapeProperty prop[]; /* prop_size elements */
};

struct JSObject {
    union {
        JSGCObjectHeader header;
        struct {
            int __gc_ref_count; /* corresponds to header.ref_count */
            uint8_t __gc_mark; /* corresponds to header.mark/gc_obj_type */

            uint8_t extensible : 1;
            uint8_t free_mark : 1; /* only used when freeing objects with cycles */
            uint8_t is_exotic : 1; /* true if object has exotic property handlers */
            uint8_t fast_array : 1; /* true if u.array is used for get/put (for JS_CLASS_ARRAY, JS_CLASS_ARGUMENTS and typed arrays) */
            uint8_t is_constructor : 1; /* true if object is a constructor function */
            uint8_t is_uncatchable_error : 1; /* if true, error is not catchable */
            uint8_t tmp_mark : 1; /* used in JS_WriteObjectRec() */
            uint8_t is_HTMLDDA : 1; /* specific annex B IsHtmlDDA behavior */
            uint16_t class_id; /* see JS_CLASS_x */
        };
    };
    /* byte offsets: 16/24 */
    JSShape *shape; /* prototype and property names + flag */
    JSProperty *prop; /* array of properties */
    /* byte offsets: 24/40 */
    JSWeakRefRecord *first_weak_ref;
    /* byte offsets: 28/48 */
    union {
        void *opaque;
        struct JSBoundFunction *bound_function; /* JS_CLASS_BOUND_FUNCTION */
        struct JSCFunctionDataRecord *c_function_data_record; /* JS_CLASS_C_FUNCTION_DATA */
        struct JSForInIterator *for_in_iterator; /* JS_CLASS_FOR_IN_ITERATOR */
        struct JSArrayBuffer *array_buffer; /* JS_CLASS_ARRAY_BUFFER, JS_CLASS_SHARED_ARRAY_BUFFER */
        struct JSTypedArray *typed_array; /* JS_CLASS_UINT8C_ARRAY..JS_CLASS_DATAVIEW */
        struct JSMapState *map_state;   /* JS_CLASS_MAP..JS_CLASS_WEAKSET */
        struct JSMapIteratorData *map_iterator_data; /* JS_CLASS_MAP_ITERATOR, JS_CLASS_SET_ITERATOR */
        struct JSArrayIteratorData *array_iterator_data; /* JS_CLASS_ARRAY_ITERATOR, JS_CLASS_STRING_ITERATOR */
        struct JSRegExpStringIteratorData *regexp_string_iterator_data; /* JS_CLASS_REGEXP_STRING_ITERATOR */
        struct JSGeneratorData *generator_data; /* JS_CLASS_GENERATOR */
        struct JSIteratorConcatData *iterator_concat_data; /* JS_CLASS_ITERATOR_CONCAT */
        struct JSIteratorHelperData *iterator_helper_data; /* JS_CLASS_ITERATOR_HELPER */
        struct JSIteratorWrapData *iterator_wrap_data; /* JS_CLASS_ITERATOR_WRAP */
        struct JSProxyData *proxy_data; /* JS_CLASS_PROXY */
        struct JSPromiseData *promise_data; /* JS_CLASS_PROMISE */
        struct JSPromiseFunctionData *promise_function_data; /* JS_CLASS_PROMISE_RESOLVE_FUNCTION, JS_CLASS_PROMISE_REJECT_FUNCTION */
        struct JSAsyncFunctionData *async_function_data; /* JS_CLASS_ASYNC_FUNCTION_RESOLVE, JS_CLASS_ASYNC_FUNCTION_REJECT */
        struct JSAsyncFromSyncIteratorData *async_from_sync_iterator_data; /* JS_CLASS_ASYNC_FROM_SYNC_ITERATOR */
        struct JSAsyncGeneratorData *async_generator_data; /* JS_CLASS_ASYNC_GENERATOR */
        struct { /* JS_CLASS_BYTECODE_FUNCTION: 12/24 bytes */
            /* also used by JS_CLASS_GENERATOR_FUNCTION, JS_CLASS_ASYNC_FUNCTION and JS_CLASS_ASYNC_GENERATOR_FUNCTION */
            struct JSFunctionBytecode *function_bytecode;
            JSVarRef **var_refs;
            JSObject *home_object; /* for 'super' access */
        } func;
        struct { /* JS_CLASS_C_FUNCTION: 12/20 bytes */
            JSContext *realm;
            JSCFunctionType c_function;
            uint8_t length;
            uint8_t cproto;
            int16_t magic;
        } cfunc;
        /* array part for fast arrays and typed arrays */
        struct { /* JS_CLASS_ARRAY, JS_CLASS_ARGUMENTS, JS_CLASS_UINT8C_ARRAY..JS_CLASS_FLOAT64_ARRAY */
            union {
                uint32_t size;          /* JS_CLASS_ARRAY, JS_CLASS_ARGUMENTS */
                struct JSTypedArray *typed_array; /* JS_CLASS_UINT8C_ARRAY..JS_CLASS_FLOAT64_ARRAY */
            } u1;
            union {
                JSValue *values;        /* JS_CLASS_ARRAY, JS_CLASS_ARGUMENTS */
                void *ptr;              /* JS_CLASS_UINT8C_ARRAY..JS_CLASS_FLOAT64_ARRAY */
                int8_t *int8_ptr;       /* JS_CLASS_INT8_ARRAY */
                uint8_t *uint8_ptr;     /* JS_CLASS_UINT8_ARRAY, JS_CLASS_UINT8C_ARRAY */
                int16_t *int16_ptr;     /* JS_CLASS_INT16_ARRAY */
                uint16_t *uint16_ptr;   /* JS_CLASS_UINT16_ARRAY */
                int32_t *int32_ptr;     /* JS_CLASS_INT32_ARRAY */
                uint32_t *uint32_ptr;   /* JS_CLASS_UINT32_ARRAY */
                int64_t *int64_ptr;     /* JS_CLASS_INT64_ARRAY */
                uint64_t *uint64_ptr;   /* JS_CLASS_UINT64_ARRAY */
                uint16_t *fp16_ptr;     /* JS_CLASS_FLOAT16_ARRAY */
                float *float_ptr;       /* JS_CLASS_FLOAT32_ARRAY */
                double *double_ptr;     /* JS_CLASS_FLOAT64_ARRAY */
            } u;
            uint32_t count; /* <= 2^31-1. 0 for a detached typed array */
        } array;    /* 12/20 bytes */
        JSRegExp regexp;    /* JS_CLASS_REGEXP: 8/16 bytes */
        JSValue object_data;    /* for JS_SetObjectData(): 8/16/16 bytes */
    } u;
    /* byte sizes: 40/48/72 */
};

typedef struct JSCallSiteData {
    JSValue filename;
    JSValue func;
    JSValue func_name;
    bool native;
    int line_num;
    int col_num;
} JSCallSiteData;

enum {
    __JS_ATOM_NULL = JS_ATOM_NULL,
#define DEF(name, str) JS_ATOM_ ## name,
/*
 * QuickJS atom definitions
 *
 * Copyright (c) 2017-2018 Fabrice Bellard
 * Copyright (c) 2017-2018 Charlie Gordon
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifdef DEF

/* Note: first atoms are considered as keywords in the parser */
DEF(null, "null") /* must be first */
DEF(false, "false")
DEF(true, "true")
DEF(if, "if")
DEF(else, "else")
DEF(return, "return")
DEF(var, "var")
DEF(this, "this")
DEF(delete, "delete")
DEF(void, "void")
DEF(typeof, "typeof")
DEF(new, "new")
DEF(in, "in")
DEF(instanceof, "instanceof")
DEF(do, "do")
DEF(while, "while")
DEF(for, "for")
DEF(break, "break")
DEF(continue, "continue")
DEF(switch, "switch")
DEF(case, "case")
DEF(default, "default")
DEF(throw, "throw")
DEF(try, "try")
DEF(catch, "catch")
DEF(finally, "finally")
DEF(function, "function")
DEF(debugger, "debugger")
DEF(with, "with")
/* FutureReservedWord */
DEF(class, "class")
DEF(const, "const")
DEF(enum, "enum")
DEF(export, "export")
DEF(extends, "extends")
DEF(import, "import")
DEF(super, "super")
/* FutureReservedWords when parsing strict mode code */
DEF(implements, "implements")
DEF(interface, "interface")
DEF(let, "let")
DEF(package, "package")
DEF(private, "private")
DEF(protected, "protected")
DEF(public, "public")
DEF(static, "static")
DEF(yield, "yield")
DEF(await, "await")

/* empty string */
DEF(empty_string, "")
/* identifiers */
DEF(keys, "keys")
DEF(size, "size")
DEF(length, "length")
DEF(message, "message")
DEF(cause, "cause")
DEF(errors, "errors")
DEF(stack, "stack")
DEF(name, "name")
DEF(toString, "toString")
DEF(toLocaleString, "toLocaleString")
DEF(valueOf, "valueOf")
DEF(eval, "eval")
DEF(prototype, "prototype")
DEF(constructor, "constructor")
DEF(configurable, "configurable")
DEF(writable, "writable")
DEF(enumerable, "enumerable")
DEF(value, "value")
DEF(get, "get")
DEF(set, "set")
DEF(of, "of")
DEF(__proto__, "__proto__")
DEF(undefined, "undefined")
DEF(number, "number")
DEF(boolean, "boolean")
DEF(string, "string")
DEF(object, "object")
DEF(symbol, "symbol")
DEF(integer, "integer")
DEF(unknown, "unknown")
DEF(arguments, "arguments")
DEF(callee, "callee")
DEF(caller, "caller")
DEF(_eval_, "<eval>")
DEF(_ret_, "<ret>")
DEF(_var_, "<var>")
DEF(_arg_var_, "<arg_var>")
DEF(_with_, "<with>")
DEF(lastIndex, "lastIndex")
DEF(target, "target")
DEF(index, "index")
DEF(input, "input")
DEF(defineProperties, "defineProperties")
DEF(apply, "apply")
DEF(join, "join")
DEF(concat, "concat")
DEF(split, "split")
DEF(construct, "construct")
DEF(getPrototypeOf, "getPrototypeOf")
DEF(setPrototypeOf, "setPrototypeOf")
DEF(isExtensible, "isExtensible")
DEF(preventExtensions, "preventExtensions")
DEF(has, "has")
DEF(deleteProperty, "deleteProperty")
DEF(defineProperty, "defineProperty")
DEF(getOwnPropertyDescriptor, "getOwnPropertyDescriptor")
DEF(ownKeys, "ownKeys")
DEF(add, "add")
DEF(done, "done")
DEF(next, "next")
DEF(values, "values")
DEF(source, "source")
DEF(flags, "flags")
DEF(global, "global")
DEF(unicode, "unicode")
DEF(raw, "raw")
DEF(new_target, "new.target")
DEF(this_active_func, "this.active_func")
DEF(home_object, "<home_object>")
DEF(computed_field, "<computed_field>")
DEF(static_computed_field, "<static_computed_field>") /* must come after computed_fields */
DEF(class_fields_init, "<class_fields_init>")
DEF(brand, "<brand>")
DEF(hash_constructor, "#constructor")
DEF(as, "as")
DEF(from, "from")
DEF(fromAsync, "fromAsync")
DEF(meta, "meta")
DEF(_default_, "*default*")
DEF(_star_, "*")
DEF(Module, "Module")
DEF(then, "then")
DEF(resolve, "resolve")
DEF(reject, "reject")
DEF(promise, "promise")
DEF(proxy, "proxy")
DEF(revoke, "revoke")
DEF(async, "async")
DEF(exec, "exec")
DEF(groups, "groups")
DEF(indices, "indices")
DEF(status, "status")
DEF(reason, "reason")
DEF(globalThis, "globalThis")
DEF(bigint, "bigint")
DEF(not_equal, "not-equal")
DEF(timed_out, "timed-out")
DEF(ok, "ok")
DEF(toJSON, "toJSON")
DEF(maxByteLength, "maxByteLength")
/* class names */
DEF(Object, "Object")
DEF(Array, "Array")
DEF(Error, "Error")
DEF(Number, "Number")
DEF(String, "String")
DEF(Boolean, "Boolean")
DEF(Symbol, "Symbol")
DEF(Arguments, "Arguments")
DEF(Math, "Math")
DEF(JSON, "JSON")
DEF(Date, "Date")
DEF(Function, "Function")
DEF(GeneratorFunction, "GeneratorFunction")
DEF(ForInIterator, "ForInIterator")
DEF(RegExp, "RegExp")
DEF(ArrayBuffer, "ArrayBuffer")
DEF(SharedArrayBuffer, "SharedArrayBuffer")
/* must keep same order as class IDs for typed arrays */
DEF(Uint8ClampedArray, "Uint8ClampedArray")
DEF(Int8Array, "Int8Array")
DEF(Uint8Array, "Uint8Array")
DEF(Int16Array, "Int16Array")
DEF(Uint16Array, "Uint16Array")
DEF(Int32Array, "Int32Array")
DEF(Uint32Array, "Uint32Array")
DEF(BigInt64Array, "BigInt64Array")
DEF(BigUint64Array, "BigUint64Array")
DEF(Float16Array, "Float16Array")
DEF(Float32Array, "Float32Array")
DEF(Float64Array, "Float64Array")
DEF(DataView, "DataView")
DEF(BigInt, "BigInt")
DEF(WeakRef, "WeakRef")
DEF(FinalizationRegistry, "FinalizationRegistry")
DEF(Map, "Map")
DEF(Set, "Set") /* Map + 1 */
DEF(WeakMap, "WeakMap") /* Map + 2 */
DEF(WeakSet, "WeakSet") /* Map + 3 */
DEF(Iterator, "Iterator")
DEF(IteratorConcat, "Iterator Concat")
DEF(IteratorHelper, "Iterator Helper")
DEF(IteratorWrap, "Iterator Wrap")
DEF(Map_Iterator, "Map Iterator")
DEF(Set_Iterator, "Set Iterator")
DEF(Array_Iterator, "Array Iterator")
DEF(String_Iterator, "String Iterator")
DEF(RegExp_String_Iterator, "RegExp String Iterator")
DEF(Generator, "Generator")
DEF(Proxy, "Proxy")
DEF(Promise, "Promise")
DEF(PromiseResolveFunction, "PromiseResolveFunction")
DEF(PromiseRejectFunction, "PromiseRejectFunction")
DEF(AsyncFunction, "AsyncFunction")
DEF(AsyncFunctionResolve, "AsyncFunctionResolve")
DEF(AsyncFunctionReject, "AsyncFunctionReject")
DEF(AsyncGeneratorFunction, "AsyncGeneratorFunction")
DEF(AsyncGenerator, "AsyncGenerator")
DEF(EvalError, "EvalError")
DEF(RangeError, "RangeError")
DEF(ReferenceError, "ReferenceError")
DEF(SyntaxError, "SyntaxError")
DEF(TypeError, "TypeError")
DEF(URIError, "URIError")
DEF(InternalError, "InternalError")
DEF(DOMException, "DOMException")
DEF(CallSite, "CallSite")
/* private symbols */
DEF(Private_brand, "<brand>")
/* symbols */
DEF(Symbol_toPrimitive, "Symbol.toPrimitive")
DEF(Symbol_iterator, "Symbol.iterator")
DEF(Symbol_match, "Symbol.match")
DEF(Symbol_matchAll, "Symbol.matchAll")
DEF(Symbol_replace, "Symbol.replace")
DEF(Symbol_search, "Symbol.search")
DEF(Symbol_split, "Symbol.split")
DEF(Symbol_toStringTag, "Symbol.toStringTag")
DEF(Symbol_isConcatSpreadable, "Symbol.isConcatSpreadable")
DEF(Symbol_hasInstance, "Symbol.hasInstance")
DEF(Symbol_species, "Symbol.species")
DEF(Symbol_unscopables, "Symbol.unscopables")
DEF(Symbol_asyncIterator, "Symbol.asyncIterator")

#endif /* DEF */

#undef DEF
    JS_ATOM_END,
};
#define JS_ATOM_LAST_KEYWORD JS_ATOM_super
#define JS_ATOM_LAST_STRICT_KEYWORD JS_ATOM_yield

static const char js_atom_init[] =
#define DEF(name, str) str "\0"
/*
 * QuickJS atom definitions
 *
 * Copyright (c) 2017-2018 Fabrice Bellard
 * Copyright (c) 2017-2018 Charlie Gordon
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifdef DEF

/* Note: first atoms are considered as keywords in the parser */
DEF(null, "null") /* must be first */
DEF(false, "false")
DEF(true, "true")
DEF(if, "if")
DEF(else, "else")
DEF(return, "return")
DEF(var, "var")
DEF(this, "this")
DEF(delete, "delete")
DEF(void, "void")
DEF(typeof, "typeof")
DEF(new, "new")
DEF(in, "in")
DEF(instanceof, "instanceof")
DEF(do, "do")
DEF(while, "while")
DEF(for, "for")
DEF(break, "break")
DEF(continue, "continue")
DEF(switch, "switch")
DEF(case, "case")
DEF(default, "default")
DEF(throw, "throw")
DEF(try, "try")
DEF(catch, "catch")
DEF(finally, "finally")
DEF(function, "function")
DEF(debugger, "debugger")
DEF(with, "with")
/* FutureReservedWord */
DEF(class, "class")
DEF(const, "const")
DEF(enum, "enum")
DEF(export, "export")
DEF(extends, "extends")
DEF(import, "import")
DEF(super, "super")
/* FutureReservedWords when parsing strict mode code */
DEF(implements, "implements")
DEF(interface, "interface")
DEF(let, "let")
DEF(package, "package")
DEF(private, "private")
DEF(protected, "protected")
DEF(public, "public")
DEF(static, "static")
DEF(yield, "yield")
DEF(await, "await")

/* empty string */
DEF(empty_string, "")
/* identifiers */
DEF(keys, "keys")
DEF(size, "size")
DEF(length, "length")
DEF(message, "message")
DEF(cause, "cause")
DEF(errors, "errors")
DEF(stack, "stack")
DEF(name, "name")
DEF(toString, "toString")
DEF(toLocaleString, "toLocaleString")
DEF(valueOf, "valueOf")
DEF(eval, "eval")
DEF(prototype, "prototype")
DEF(constructor, "constructor")
DEF(configurable, "configurable")
DEF(writable, "writable")
DEF(enumerable, "enumerable")
DEF(value, "value")
DEF(get, "get")
DEF(set, "set")
DEF(of, "of")
DEF(__proto__, "__proto__")
DEF(undefined, "undefined")
DEF(number, "number")
DEF(boolean, "boolean")
DEF(string, "string")
DEF(object, "object")
DEF(symbol, "symbol")
DEF(integer, "integer")
DEF(unknown, "unknown")
DEF(arguments, "arguments")
DEF(callee, "callee")
DEF(caller, "caller")
DEF(_eval_, "<eval>")
DEF(_ret_, "<ret>")
DEF(_var_, "<var>")
DEF(_arg_var_, "<arg_var>")
DEF(_with_, "<with>")
DEF(lastIndex, "lastIndex")
DEF(target, "target")
DEF(index, "index")
DEF(input, "input")
DEF(defineProperties, "defineProperties")
DEF(apply, "apply")
DEF(join, "join")
DEF(concat, "concat")
DEF(split, "split")
DEF(construct, "construct")
DEF(getPrototypeOf, "getPrototypeOf")
DEF(setPrototypeOf, "setPrototypeOf")
DEF(isExtensible, "isExtensible")
DEF(preventExtensions, "preventExtensions")
DEF(has, "has")
DEF(deleteProperty, "deleteProperty")
DEF(defineProperty, "defineProperty")
DEF(getOwnPropertyDescriptor, "getOwnPropertyDescriptor")
DEF(ownKeys, "ownKeys")
DEF(add, "add")
DEF(done, "done")
DEF(next, "next")
DEF(values, "values")
DEF(source, "source")
DEF(flags, "flags")
DEF(global, "global")
DEF(unicode, "unicode")
DEF(raw, "raw")
DEF(new_target, "new.target")
DEF(this_active_func, "this.active_func")
DEF(home_object, "<home_object>")
DEF(computed_field, "<computed_field>")
DEF(static_computed_field, "<static_computed_field>") /* must come after computed_fields */
DEF(class_fields_init, "<class_fields_init>")
DEF(brand, "<brand>")
DEF(hash_constructor, "#constructor")
DEF(as, "as")
DEF(from, "from")
DEF(fromAsync, "fromAsync")
DEF(meta, "meta")
DEF(_default_, "*default*")
DEF(_star_, "*")
DEF(Module, "Module")
DEF(then, "then")
DEF(resolve, "resolve")
DEF(reject, "reject")
DEF(promise, "promise")
DEF(proxy, "proxy")
DEF(revoke, "revoke")
DEF(async, "async")
DEF(exec, "exec")
DEF(groups, "groups")
DEF(indices, "indices")
DEF(status, "status")
DEF(reason, "reason")
DEF(globalThis, "globalThis")
DEF(bigint, "bigint")
DEF(not_equal, "not-equal")
DEF(timed_out, "timed-out")
DEF(ok, "ok")
DEF(toJSON, "toJSON")
DEF(maxByteLength, "maxByteLength")
/* class names */
DEF(Object, "Object")
DEF(Array, "Array")
DEF(Error, "Error")
DEF(Number, "Number")
DEF(String, "String")
DEF(Boolean, "Boolean")
DEF(Symbol, "Symbol")
DEF(Arguments, "Arguments")
DEF(Math, "Math")
DEF(JSON, "JSON")
DEF(Date, "Date")
DEF(Function, "Function")
DEF(GeneratorFunction, "GeneratorFunction")
DEF(ForInIterator, "ForInIterator")
DEF(RegExp, "RegExp")
DEF(ArrayBuffer, "ArrayBuffer")
DEF(SharedArrayBuffer, "SharedArrayBuffer")
/* must keep same order as class IDs for typed arrays */
DEF(Uint8ClampedArray, "Uint8ClampedArray")
DEF(Int8Array, "Int8Array")
DEF(Uint8Array, "Uint8Array")
DEF(Int16Array, "Int16Array")
DEF(Uint16Array, "Uint16Array")
DEF(Int32Array, "Int32Array")
DEF(Uint32Array, "Uint32Array")
DEF(BigInt64Array, "BigInt64Array")
DEF(BigUint64Array, "BigUint64Array")
DEF(Float16Array, "Float16Array")
DEF(Float32Array, "Float32Array")
DEF(Float64Array, "Float64Array")
DEF(DataView, "DataView")
DEF(BigInt, "BigInt")
DEF(WeakRef, "WeakRef")
DEF(FinalizationRegistry, "FinalizationRegistry")
DEF(Map, "Map")
DEF(Set, "Set") /* Map + 1 */
DEF(WeakMap, "WeakMap") /* Map + 2 */
DEF(WeakSet, "WeakSet") /* Map + 3 */
DEF(Iterator, "Iterator")
DEF(IteratorConcat, "Iterator Concat")
DEF(IteratorHelper, "Iterator Helper")
DEF(IteratorWrap, "Iterator Wrap")
DEF(Map_Iterator, "Map Iterator")
DEF(Set_Iterator, "Set Iterator")
DEF(Array_Iterator, "Array Iterator")
DEF(String_Iterator, "String Iterator")
DEF(RegExp_String_Iterator, "RegExp String Iterator")
DEF(Generator, "Generator")
DEF(Proxy, "Proxy")
DEF(Promise, "Promise")
DEF(PromiseResolveFunction, "PromiseResolveFunction")
DEF(PromiseRejectFunction, "PromiseRejectFunction")
DEF(AsyncFunction, "AsyncFunction")
DEF(AsyncFunctionResolve, "AsyncFunctionResolve")
DEF(AsyncFunctionReject, "AsyncFunctionReject")
DEF(AsyncGeneratorFunction, "AsyncGeneratorFunction")
DEF(AsyncGenerator, "AsyncGenerator")
DEF(EvalError, "EvalError")
DEF(RangeError, "RangeError")
DEF(ReferenceError, "ReferenceError")
DEF(SyntaxError, "SyntaxError")
DEF(TypeError, "TypeError")
DEF(URIError, "URIError")
DEF(InternalError, "InternalError")
DEF(DOMException, "DOMException")
DEF(CallSite, "CallSite")
/* private symbols */
DEF(Private_brand, "<brand>")
/* symbols */
DEF(Symbol_toPrimitive, "Symbol.toPrimitive")
DEF(Symbol_iterator, "Symbol.iterator")
DEF(Symbol_match, "Symbol.match")
DEF(Symbol_matchAll, "Symbol.matchAll")
DEF(Symbol_replace, "Symbol.replace")
DEF(Symbol_search, "Symbol.search")
DEF(Symbol_split, "Symbol.split")
DEF(Symbol_toStringTag, "Symbol.toStringTag")
DEF(Symbol_isConcatSpreadable, "Symbol.isConcatSpreadable")
DEF(Symbol_hasInstance, "Symbol.hasInstance")
DEF(Symbol_species, "Symbol.species")
DEF(Symbol_unscopables, "Symbol.unscopables")
DEF(Symbol_asyncIterator, "Symbol.asyncIterator")

#endif /* DEF */

#undef DEF
;

typedef enum OPCodeFormat {
#define FMT(f) OP_FMT_ ## f,
#define DEF(id, size, n_pop, n_push, f)
/*
 * QuickJS opcode definitions
 *
 * Copyright (c) 2017-2018 Fabrice Bellard
 * Copyright (c) 2017-2018 Charlie Gordon
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifdef FMT
FMT(none)
FMT(none_int)
FMT(none_loc)
FMT(none_arg)
FMT(none_var_ref)
FMT(u8)
FMT(i8)
FMT(loc8)
FMT(const8)
FMT(label8)
FMT(u16)
FMT(i16)
FMT(label16)
FMT(npop)
FMT(npopx)
FMT(npop_u16)
FMT(loc)
FMT(arg)
FMT(var_ref)
FMT(u32)
FMT(u32x2)
FMT(i32)
FMT(const)
FMT(label)
FMT(atom)
FMT(atom_u8)
FMT(atom_u16)
FMT(atom_label_u8)
FMT(atom_label_u16)
FMT(label_u16)
#undef FMT
#endif /* FMT */

#ifdef DEF

#ifndef def
#define def(id, size, n_pop, n_push, f) DEF(id, size, n_pop, n_push, f)
#endif

DEF(invalid, 1, 0, 0, none) /* never emitted */

/* push values */
DEF(       push_i32, 5, 0, 1, i32)
DEF(     push_const, 5, 0, 1, const)
DEF(       fclosure, 5, 0, 1, const) /* must follow push_const */
DEF(push_atom_value, 5, 0, 1, atom)
DEF( private_symbol, 5, 0, 1, atom)
DEF(      undefined, 1, 0, 1, none)
DEF(           null, 1, 0, 1, none)
DEF(      push_this, 1, 0, 1, none) /* only used at the start of a function */
DEF(     push_false, 1, 0, 1, none)
DEF(      push_true, 1, 0, 1, none)
DEF(         object, 1, 0, 1, none)
DEF( special_object, 2, 0, 1, u8) /* only used at the start of a function */
DEF(           rest, 3, 0, 1, u16) /* only used at the start of a function */

DEF(           drop, 1, 1, 0, none) /* a -> */
DEF(            nip, 1, 2, 1, none) /* a b -> b */
DEF(           nip1, 1, 3, 2, none) /* a b c -> b c */
DEF(            dup, 1, 1, 2, none) /* a -> a a */
DEF(           dup1, 1, 2, 3, none) /* a b -> a a b */
DEF(           dup2, 1, 2, 4, none) /* a b -> a b a b */
DEF(           dup3, 1, 3, 6, none) /* a b c -> a b c a b c */
DEF(        insert2, 1, 2, 3, none) /* obj a -> a obj a (dup_x1) */
DEF(        insert3, 1, 3, 4, none) /* obj prop a -> a obj prop a (dup_x2) */
DEF(        insert4, 1, 4, 5, none) /* this obj prop a -> a this obj prop a */
DEF(          perm3, 1, 3, 3, none) /* obj a b -> a obj b */
DEF(          perm4, 1, 4, 4, none) /* obj prop a b -> a obj prop b */
DEF(          perm5, 1, 5, 5, none) /* this obj prop a b -> a this obj prop b */
DEF(           swap, 1, 2, 2, none) /* a b -> b a */
DEF(          swap2, 1, 4, 4, none) /* a b c d -> c d a b */
DEF(          rot3l, 1, 3, 3, none) /* x a b -> a b x */
DEF(          rot3r, 1, 3, 3, none) /* a b x -> x a b */
DEF(          rot4l, 1, 4, 4, none) /* x a b c -> a b c x */
DEF(          rot5l, 1, 5, 5, none) /* x a b c d -> a b c d x */

DEF(call_constructor, 3, 2, 1, npop) /* func new.target args -> ret. arguments are not counted in n_pop */
DEF(           call, 3, 1, 1, npop) /* arguments are not counted in n_pop */
DEF(      tail_call, 3, 1, 0, npop) /* arguments are not counted in n_pop */
DEF(    call_method, 3, 2, 1, npop) /* arguments are not counted in n_pop */
DEF(tail_call_method, 3, 2, 0, npop) /* arguments are not counted in n_pop */
DEF(     array_from, 3, 0, 1, npop) /* arguments are not counted in n_pop */
DEF(          apply, 3, 3, 1, u16)
DEF(         return, 1, 1, 0, none)
DEF(   return_undef, 1, 0, 0, none)
DEF(check_ctor_return, 1, 1, 2, none)
DEF(     check_ctor, 1, 0, 0, none)
DEF(      init_ctor, 1, 0, 1, none)
DEF(    check_brand, 1, 2, 2, none) /* this_obj func -> this_obj func */
DEF(      add_brand, 1, 2, 0, none) /* this_obj home_obj -> */
DEF(   return_async, 1, 1, 0, none)
DEF(          throw, 1, 1, 0, none)
DEF(    throw_error, 6, 0, 0, atom_u8)
DEF(           eval, 5, 1, 1, npop_u16) /* func args... -> ret_val */
DEF(     apply_eval, 3, 2, 1, u16) /* func array -> ret_eval */
DEF(         regexp, 1, 2, 1, none) /* create a RegExp object from the pattern and a
                                       bytecode string */
DEF(      get_super, 1, 1, 1, none)
DEF(         import, 1, 1, 1, none) /* dynamic module import */

DEF(      check_var, 5, 0, 1, atom) /* check if a variable exists */
DEF(  get_var_undef, 5, 0, 1, atom) /* push undefined if the variable does not exist */
DEF(        get_var, 5, 0, 1, atom) /* throw an exception if the variable does not exist */
DEF(        put_var, 5, 1, 0, atom) /* must come after get_var */
DEF(   put_var_init, 5, 1, 0, atom) /* must come after put_var. Used to initialize a global lexical variable */
DEF( put_var_strict, 5, 2, 0, atom) /* for strict mode variable write */

DEF(  get_ref_value, 1, 2, 3, none)
DEF(  put_ref_value, 1, 3, 0, none)

DEF(     define_var, 6, 0, 0, atom_u8)
DEF(check_define_var, 6, 0, 0, atom_u8)
DEF(    define_func, 6, 1, 0, atom_u8)

// order matters, see IC counterparts
DEF(      get_field, 5, 1, 1, atom)
DEF(     get_field2, 5, 1, 2, atom)
DEF(      put_field, 5, 2, 0, atom)

DEF( get_private_field, 1, 2, 1, none) /* obj prop -> value */
DEF( put_private_field, 1, 3, 0, none) /* obj value prop -> */
DEF(define_private_field, 1, 3, 1, none) /* obj prop value -> obj */
DEF(   get_array_el, 1, 2, 1, none)
DEF(  get_array_el2, 1, 2, 2, none) /* obj prop -> obj value */
DEF(   put_array_el, 1, 3, 0, none)
DEF(get_super_value, 1, 3, 1, none) /* this obj prop -> value */
DEF(put_super_value, 1, 4, 0, none) /* this obj prop value -> */
DEF(   define_field, 5, 2, 1, atom)
DEF(       set_name, 5, 1, 1, atom)
DEF(set_name_computed, 1, 2, 2, none)
DEF(      set_proto, 1, 2, 1, none)
DEF(set_home_object, 1, 2, 2, none)
DEF(define_array_el, 1, 3, 2, none)
DEF(         append, 1, 3, 2, none) /* append enumerated object, update length */
DEF(copy_data_properties, 2, 3, 3, u8)
DEF(  define_method, 6, 2, 1, atom_u8)
DEF(define_method_computed, 2, 3, 1, u8) /* must come after define_method */
DEF(   define_class, 6, 2, 2, atom_u8) /* parent ctor -> ctor proto */
DEF(   define_class_computed, 6, 3, 3, atom_u8) /* field_name parent ctor -> field_name ctor proto (class with computed name) */

DEF(        get_loc, 3, 0, 1, loc)
DEF(        put_loc, 3, 1, 0, loc) /* must come after get_loc */
DEF(        set_loc, 3, 1, 1, loc) /* must come after put_loc */
DEF(        get_arg, 3, 0, 1, arg)
DEF(        put_arg, 3, 1, 0, arg) /* must come after get_arg */
DEF(        set_arg, 3, 1, 1, arg) /* must come after put_arg */
DEF(    get_var_ref, 3, 0, 1, var_ref)
DEF(    put_var_ref, 3, 1, 0, var_ref) /* must come after get_var_ref */
DEF(    set_var_ref, 3, 1, 1, var_ref) /* must come after put_var_ref */
DEF(set_loc_uninitialized, 3, 0, 0, loc)
DEF(  get_loc_check, 3, 0, 1, loc)
DEF(  put_loc_check, 3, 1, 0, loc) /* must come after get_loc_check */
DEF(  put_loc_check_init, 3, 1, 0, loc)
DEF(get_var_ref_check, 3, 0, 1, var_ref)
DEF(put_var_ref_check, 3, 1, 0, var_ref) /* must come after get_var_ref_check */
DEF(put_var_ref_check_init, 3, 1, 0, var_ref)
DEF(      close_loc, 3, 0, 0, loc)
DEF(       if_false, 5, 1, 0, label)
DEF(        if_true, 5, 1, 0, label) /* must come after if_false */
DEF(           goto, 5, 0, 0, label) /* must come after if_true */
DEF(          catch, 5, 0, 1, label)
DEF(          gosub, 5, 0, 0, label) /* used to execute the finally block */
DEF(            ret, 1, 1, 0, none) /* used to return from the finally block */
DEF(      nip_catch, 1, 2, 1, none) /* catch ... a -> a */

DEF(      to_object, 1, 1, 1, none)
//DEF(      to_string, 1, 1, 1, none)
DEF(     to_propkey, 1, 1, 1, none)
DEF(    to_propkey2, 1, 2, 2, none)

DEF(   with_get_var, 10, 1, 0, atom_label_u8)     /* must be in the same order as scope_xxx */
DEF(   with_put_var, 10, 2, 1, atom_label_u8)     /* must be in the same order as scope_xxx */
DEF(with_delete_var, 10, 1, 0, atom_label_u8)     /* must be in the same order as scope_xxx */
DEF(  with_make_ref, 10, 1, 0, atom_label_u8)     /* must be in the same order as scope_xxx */
DEF(   with_get_ref, 10, 1, 0, atom_label_u8)     /* must be in the same order as scope_xxx */
DEF(with_get_ref_undef, 10, 1, 0, atom_label_u8)

DEF(   make_loc_ref, 7, 0, 2, atom_u16)
DEF(   make_arg_ref, 7, 0, 2, atom_u16)
DEF(make_var_ref_ref, 7, 0, 2, atom_u16)
DEF(   make_var_ref, 5, 0, 2, atom)

DEF(   for_in_start, 1, 1, 1, none)
DEF(   for_of_start, 1, 1, 3, none)
DEF(for_await_of_start, 1, 1, 3, none)
DEF(    for_in_next, 1, 1, 3, none)
DEF(    for_of_next, 2, 3, 5, u8)
DEF(iterator_check_object, 1, 1, 1, none)
DEF(iterator_get_value_done, 1, 1, 2, none)
DEF( iterator_close, 1, 3, 0, none)
DEF(  iterator_next, 1, 4, 4, none)
DEF(  iterator_call, 2, 4, 5, u8)
DEF(  initial_yield, 1, 0, 0, none)
DEF(          yield, 1, 1, 2, none)
DEF(     yield_star, 1, 1, 2, none)
DEF(async_yield_star, 1, 1, 2, none)
DEF(          await, 1, 1, 1, none)

/* arithmetic/logic operations */
DEF(            neg, 1, 1, 1, none)
DEF(           plus, 1, 1, 1, none)
DEF(            dec, 1, 1, 1, none)
DEF(            inc, 1, 1, 1, none)
DEF(       post_dec, 1, 1, 2, none)
DEF(       post_inc, 1, 1, 2, none)
DEF(        dec_loc, 2, 0, 0, loc8)
DEF(        inc_loc, 2, 0, 0, loc8)
DEF(        add_loc, 2, 1, 0, loc8)
DEF(            not, 1, 1, 1, none)
DEF(           lnot, 1, 1, 1, none)
DEF(         typeof, 1, 1, 1, none)
DEF(         delete, 1, 2, 1, none)
DEF(     delete_var, 5, 0, 1, atom)

/* warning: order matters (see js_parse_assign_expr) */
DEF(            mul, 1, 2, 1, none)
DEF(            div, 1, 2, 1, none)
DEF(            mod, 1, 2, 1, none)
DEF(            add, 1, 2, 1, none)
DEF(            sub, 1, 2, 1, none)
DEF(            shl, 1, 2, 1, none)
DEF(            sar, 1, 2, 1, none)
DEF(            shr, 1, 2, 1, none)
DEF(            and, 1, 2, 1, none)
DEF(            xor, 1, 2, 1, none)
DEF(             or, 1, 2, 1, none)
DEF(            pow, 1, 2, 1, none)

DEF(             lt, 1, 2, 1, none)
DEF(            lte, 1, 2, 1, none)
DEF(             gt, 1, 2, 1, none)
DEF(            gte, 1, 2, 1, none)
DEF(     instanceof, 1, 2, 1, none)
DEF(             in, 1, 2, 1, none)
DEF(             eq, 1, 2, 1, none)
DEF(            neq, 1, 2, 1, none)
DEF(      strict_eq, 1, 2, 1, none)
DEF(     strict_neq, 1, 2, 1, none)
DEF(is_undefined_or_null, 1, 1, 1, none)
DEF(     private_in, 1, 2, 1, none)
DEF(push_bigint_i32, 5, 0, 1, i32)
/* must be the last non short and non temporary opcode */
DEF(            nop, 1, 0, 0, none)

/* temporary opcodes: never emitted in the final bytecode */

def(    enter_scope, 3, 0, 0, u16)  /* emitted in phase 1, removed in phase 2 */
def(    leave_scope, 3, 0, 0, u16)  /* emitted in phase 1, removed in phase 2 */

def(          label, 5, 0, 0, label) /* emitted in phase 1, removed in phase 3 */

def(scope_get_var_undef, 7, 0, 1, atom_u16) /* emitted in phase 1, removed in phase 2 */
def(  scope_get_var, 7, 0, 1, atom_u16) /* emitted in phase 1, removed in phase 2 */
def(  scope_put_var, 7, 1, 0, atom_u16) /* emitted in phase 1, removed in phase 2 */
def(scope_delete_var, 7, 0, 1, atom_u16) /* emitted in phase 1, removed in phase 2 */
def( scope_make_ref, 11, 0, 2, atom_label_u16) /* emitted in phase 1, removed in phase 2 */
def(  scope_get_ref, 7, 0, 2, atom_u16) /* emitted in phase 1, removed in phase 2 */
def(scope_put_var_init, 7, 0, 2, atom_u16) /* emitted in phase 1, removed in phase 2 */
def(scope_get_private_field, 7, 1, 1, atom_u16) /* obj -> value, emitted in phase 1, removed in phase 2 */
def(scope_get_private_field2, 7, 1, 2, atom_u16) /* obj -> obj value, emitted in phase 1, removed in phase 2 */
def(scope_put_private_field, 7, 2, 0, atom_u16) /* obj value ->, emitted in phase 1, removed in phase 2 */
def(scope_in_private_field, 7, 1, 1, atom_u16) /* obj -> res emitted in phase 1, removed in phase 2 */
def(get_field_opt_chain, 5, 1, 1, atom) /* emitted in phase 1, removed in phase 2 */
def(get_array_el_opt_chain, 1, 2, 1, none) /* emitted in phase 1, removed in phase 2 */
def( set_class_name, 5, 1, 1, u32) /* emitted in phase 1, removed in phase 2 */

def(     source_loc, 9, 0, 0, u32x2) /* emitted in phase 1, removed in phase 3 */

DEF(    push_minus1, 1, 0, 1, none_int)
DEF(         push_0, 1, 0, 1, none_int)
DEF(         push_1, 1, 0, 1, none_int)
DEF(         push_2, 1, 0, 1, none_int)
DEF(         push_3, 1, 0, 1, none_int)
DEF(         push_4, 1, 0, 1, none_int)
DEF(         push_5, 1, 0, 1, none_int)
DEF(         push_6, 1, 0, 1, none_int)
DEF(         push_7, 1, 0, 1, none_int)
DEF(        push_i8, 2, 0, 1, i8)
DEF(       push_i16, 3, 0, 1, i16)
DEF(    push_const8, 2, 0, 1, const8)
DEF(      fclosure8, 2, 0, 1, const8) /* must follow push_const8 */
DEF(push_empty_string, 1, 0, 1, none)

DEF(       get_loc8, 2, 0, 1, loc8)
DEF(       put_loc8, 2, 1, 0, loc8)
DEF(       set_loc8, 2, 1, 1, loc8)

DEF(  get_loc0_loc1, 1, 0, 2, none_loc)
DEF(       get_loc0, 1, 0, 1, none_loc)
DEF(       get_loc1, 1, 0, 1, none_loc)
DEF(       get_loc2, 1, 0, 1, none_loc)
DEF(       get_loc3, 1, 0, 1, none_loc)
DEF(       put_loc0, 1, 1, 0, none_loc)
DEF(       put_loc1, 1, 1, 0, none_loc)
DEF(       put_loc2, 1, 1, 0, none_loc)
DEF(       put_loc3, 1, 1, 0, none_loc)
DEF(       set_loc0, 1, 1, 1, none_loc)
DEF(       set_loc1, 1, 1, 1, none_loc)
DEF(       set_loc2, 1, 1, 1, none_loc)
DEF(       set_loc3, 1, 1, 1, none_loc)
DEF(       get_arg0, 1, 0, 1, none_arg)
DEF(       get_arg1, 1, 0, 1, none_arg)
DEF(       get_arg2, 1, 0, 1, none_arg)
DEF(       get_arg3, 1, 0, 1, none_arg)
DEF(       put_arg0, 1, 1, 0, none_arg)
DEF(       put_arg1, 1, 1, 0, none_arg)
DEF(       put_arg2, 1, 1, 0, none_arg)
DEF(       put_arg3, 1, 1, 0, none_arg)
DEF(       set_arg0, 1, 1, 1, none_arg)
DEF(       set_arg1, 1, 1, 1, none_arg)
DEF(       set_arg2, 1, 1, 1, none_arg)
DEF(       set_arg3, 1, 1, 1, none_arg)
DEF(   get_var_ref0, 1, 0, 1, none_var_ref)
DEF(   get_var_ref1, 1, 0, 1, none_var_ref)
DEF(   get_var_ref2, 1, 0, 1, none_var_ref)
DEF(   get_var_ref3, 1, 0, 1, none_var_ref)
DEF(   put_var_ref0, 1, 1, 0, none_var_ref)
DEF(   put_var_ref1, 1, 1, 0, none_var_ref)
DEF(   put_var_ref2, 1, 1, 0, none_var_ref)
DEF(   put_var_ref3, 1, 1, 0, none_var_ref)
DEF(   set_var_ref0, 1, 1, 1, none_var_ref)
DEF(   set_var_ref1, 1, 1, 1, none_var_ref)
DEF(   set_var_ref2, 1, 1, 1, none_var_ref)
DEF(   set_var_ref3, 1, 1, 1, none_var_ref)

DEF(     get_length, 1, 1, 1, none)

DEF(      if_false8, 2, 1, 0, label8)
DEF(       if_true8, 2, 1, 0, label8) /* must come after if_false8 */
DEF(          goto8, 2, 0, 0, label8) /* must come after if_true8 */
DEF(         goto16, 3, 0, 0, label16)

DEF(          call0, 1, 1, 1, npopx)
DEF(          call1, 1, 1, 1, npopx)
DEF(          call2, 1, 1, 1, npopx)
DEF(          call3, 1, 1, 1, npopx)

DEF(   is_undefined, 1, 1, 1, none)
DEF(        is_null, 1, 1, 1, none)
DEF(typeof_is_undefined, 1, 1, 1, none)
DEF( typeof_is_function, 1, 1, 1, none)

#undef DEF
#undef def
#endif  /* DEF */

#undef DEF
#undef FMT
} OPCodeFormat;

typedef enum OPCodeEnum {
#define FMT(f)
#define DEF(id, size, n_pop, n_push, f) OP_ ## id,
#define def(id, size, n_pop, n_push, f)
/*
 * QuickJS opcode definitions
 *
 * Copyright (c) 2017-2018 Fabrice Bellard
 * Copyright (c) 2017-2018 Charlie Gordon
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifdef FMT
FMT(none)
FMT(none_int)
FMT(none_loc)
FMT(none_arg)
FMT(none_var_ref)
FMT(u8)
FMT(i8)
FMT(loc8)
FMT(const8)
FMT(label8)
FMT(u16)
FMT(i16)
FMT(label16)
FMT(npop)
FMT(npopx)
FMT(npop_u16)
FMT(loc)
FMT(arg)
FMT(var_ref)
FMT(u32)
FMT(u32x2)
FMT(i32)
FMT(const)
FMT(label)
FMT(atom)
FMT(atom_u8)
FMT(atom_u16)
FMT(atom_label_u8)
FMT(atom_label_u16)
FMT(label_u16)
#undef FMT
#endif /* FMT */

#ifdef DEF

#ifndef def
#define def(id, size, n_pop, n_push, f) DEF(id, size, n_pop, n_push, f)
#endif

DEF(invalid, 1, 0, 0, none) /* never emitted */

/* push values */
DEF(       push_i32, 5, 0, 1, i32)
DEF(     push_const, 5, 0, 1, const)
DEF(       fclosure, 5, 0, 1, const) /* must follow push_const */
DEF(push_atom_value, 5, 0, 1, atom)
DEF( private_symbol, 5, 0, 1, atom)
DEF(      undefined, 1, 0, 1, none)
DEF(           null, 1, 0, 1, none)
DEF(      push_this, 1, 0, 1, none) /* only used at the start of a function */
DEF(     push_false, 1, 0, 1, none)
DEF(      push_true, 1, 0, 1, none)
DEF(         object, 1, 0, 1, none)
DEF( special_object, 2, 0, 1, u8) /* only used at the start of a function */
DEF(           rest, 3, 0, 1, u16) /* only used at the start of a function */

DEF(           drop, 1, 1, 0, none) /* a -> */
DEF(            nip, 1, 2, 1, none) /* a b -> b */
DEF(           nip1, 1, 3, 2, none) /* a b c -> b c */
DEF(            dup, 1, 1, 2, none) /* a -> a a */
DEF(           dup1, 1, 2, 3, none) /* a b -> a a b */
DEF(           dup2, 1, 2, 4, none) /* a b -> a b a b */
DEF(           dup3, 1, 3, 6, none) /* a b c -> a b c a b c */
DEF(        insert2, 1, 2, 3, none) /* obj a -> a obj a (dup_x1) */
DEF(        insert3, 1, 3, 4, none) /* obj prop a -> a obj prop a (dup_x2) */
DEF(        insert4, 1, 4, 5, none) /* this obj prop a -> a this obj prop a */
DEF(          perm3, 1, 3, 3, none) /* obj a b -> a obj b */
DEF(          perm4, 1, 4, 4, none) /* obj prop a b -> a obj prop b */
DEF(          perm5, 1, 5, 5, none) /* this obj prop a b -> a this obj prop b */
DEF(           swap, 1, 2, 2, none) /* a b -> b a */
DEF(          swap2, 1, 4, 4, none) /* a b c d -> c d a b */
DEF(          rot3l, 1, 3, 3, none) /* x a b -> a b x */
DEF(          rot3r, 1, 3, 3, none) /* a b x -> x a b */
DEF(          rot4l, 1, 4, 4, none) /* x a b c -> a b c x */
DEF(          rot5l, 1, 5, 5, none) /* x a b c d -> a b c d x */

DEF(call_constructor, 3, 2, 1, npop) /* func new.target args -> ret. arguments are not counted in n_pop */
DEF(           call, 3, 1, 1, npop) /* arguments are not counted in n_pop */
DEF(      tail_call, 3, 1, 0, npop) /* arguments are not counted in n_pop */
DEF(    call_method, 3, 2, 1, npop) /* arguments are not counted in n_pop */
DEF(tail_call_method, 3, 2, 0, npop) /* arguments are not counted in n_pop */
DEF(     array_from, 3, 0, 1, npop) /* arguments are not counted in n_pop */
DEF(          apply, 3, 3, 1, u16)
DEF(         return, 1, 1, 0, none)
DEF(   return_undef, 1, 0, 0, none)
DEF(check_ctor_return, 1, 1, 2, none)
DEF(     check_ctor, 1, 0, 0, none)
DEF(      init_ctor, 1, 0, 1, none)
DEF(    check_brand, 1, 2, 2, none) /* this_obj func -> this_obj func */
DEF(      add_brand, 1, 2, 0, none) /* this_obj home_obj -> */
DEF(   return_async, 1, 1, 0, none)
DEF(          throw, 1, 1, 0, none)
DEF(    throw_error, 6, 0, 0, atom_u8)
DEF(           eval, 5, 1, 1, npop_u16) /* func args... -> ret_val */
DEF(     apply_eval, 3, 2, 1, u16) /* func array -> ret_eval */
DEF(         regexp, 1, 2, 1, none) /* create a RegExp object from the pattern and a
                                       bytecode string */
DEF(      get_super, 1, 1, 1, none)
DEF(         import, 1, 1, 1, none) /* dynamic module import */

DEF(      check_var, 5, 0, 1, atom) /* check if a variable exists */
DEF(  get_var_undef, 5, 0, 1, atom) /* push undefined if the variable does not exist */
DEF(        get_var, 5, 0, 1, atom) /* throw an exception if the variable does not exist */
DEF(        put_var, 5, 1, 0, atom) /* must come after get_var */
DEF(   put_var_init, 5, 1, 0, atom) /* must come after put_var. Used to initialize a global lexical variable */
DEF( put_var_strict, 5, 2, 0, atom) /* for strict mode variable write */

DEF(  get_ref_value, 1, 2, 3, none)
DEF(  put_ref_value, 1, 3, 0, none)

DEF(     define_var, 6, 0, 0, atom_u8)
DEF(check_define_var, 6, 0, 0, atom_u8)
DEF(    define_func, 6, 1, 0, atom_u8)

// order matters, see IC counterparts
DEF(      get_field, 5, 1, 1, atom)
DEF(     get_field2, 5, 1, 2, atom)
DEF(      put_field, 5, 2, 0, atom)

DEF( get_private_field, 1, 2, 1, none) /* obj prop -> value */
DEF( put_private_field, 1, 3, 0, none) /* obj value prop -> */
DEF(define_private_field, 1, 3, 1, none) /* obj prop value -> obj */
DEF(   get_array_el, 1, 2, 1, none)
DEF(  get_array_el2, 1, 2, 2, none) /* obj prop -> obj value */
DEF(   put_array_el, 1, 3, 0, none)
DEF(get_super_value, 1, 3, 1, none) /* this obj prop -> value */
DEF(put_super_value, 1, 4, 0, none) /* this obj prop value -> */
DEF(   define_field, 5, 2, 1, atom)
DEF(       set_name, 5, 1, 1, atom)
DEF(set_name_computed, 1, 2, 2, none)
DEF(      set_proto, 1, 2, 1, none)
DEF(set_home_object, 1, 2, 2, none)
DEF(define_array_el, 1, 3, 2, none)
DEF(         append, 1, 3, 2, none) /* append enumerated object, update length */
DEF(copy_data_properties, 2, 3, 3, u8)
DEF(  define_method, 6, 2, 1, atom_u8)
DEF(define_method_computed, 2, 3, 1, u8) /* must come after define_method */
DEF(   define_class, 6, 2, 2, atom_u8) /* parent ctor -> ctor proto */
DEF(   define_class_computed, 6, 3, 3, atom_u8) /* field_name parent ctor -> field_name ctor proto (class with computed name) */

DEF(        get_loc, 3, 0, 1, loc)
DEF(        put_loc, 3, 1, 0, loc) /* must come after get_loc */
DEF(        set_loc, 3, 1, 1, loc) /* must come after put_loc */
DEF(        get_arg, 3, 0, 1, arg)
DEF(        put_arg, 3, 1, 0, arg) /* must come after get_arg */
DEF(        set_arg, 3, 1, 1, arg) /* must come after put_arg */
DEF(    get_var_ref, 3, 0, 1, var_ref)
DEF(    put_var_ref, 3, 1, 0, var_ref) /* must come after get_var_ref */
DEF(    set_var_ref, 3, 1, 1, var_ref) /* must come after put_var_ref */
DEF(set_loc_uninitialized, 3, 0, 0, loc)
DEF(  get_loc_check, 3, 0, 1, loc)
DEF(  put_loc_check, 3, 1, 0, loc) /* must come after get_loc_check */
DEF(  put_loc_check_init, 3, 1, 0, loc)
DEF(get_var_ref_check, 3, 0, 1, var_ref)
DEF(put_var_ref_check, 3, 1, 0, var_ref) /* must come after get_var_ref_check */
DEF(put_var_ref_check_init, 3, 1, 0, var_ref)
DEF(      close_loc, 3, 0, 0, loc)
DEF(       if_false, 5, 1, 0, label)
DEF(        if_true, 5, 1, 0, label) /* must come after if_false */
DEF(           goto, 5, 0, 0, label) /* must come after if_true */
DEF(          catch, 5, 0, 1, label)
DEF(          gosub, 5, 0, 0, label) /* used to execute the finally block */
DEF(            ret, 1, 1, 0, none) /* used to return from the finally block */
DEF(      nip_catch, 1, 2, 1, none) /* catch ... a -> a */

DEF(      to_object, 1, 1, 1, none)
//DEF(      to_string, 1, 1, 1, none)
DEF(     to_propkey, 1, 1, 1, none)
DEF(    to_propkey2, 1, 2, 2, none)

DEF(   with_get_var, 10, 1, 0, atom_label_u8)     /* must be in the same order as scope_xxx */
DEF(   with_put_var, 10, 2, 1, atom_label_u8)     /* must be in the same order as scope_xxx */
DEF(with_delete_var, 10, 1, 0, atom_label_u8)     /* must be in the same order as scope_xxx */
DEF(  with_make_ref, 10, 1, 0, atom_label_u8)     /* must be in the same order as scope_xxx */
DEF(   with_get_ref, 10, 1, 0, atom_label_u8)     /* must be in the same order as scope_xxx */
DEF(with_get_ref_undef, 10, 1, 0, atom_label_u8)

DEF(   make_loc_ref, 7, 0, 2, atom_u16)
DEF(   make_arg_ref, 7, 0, 2, atom_u16)
DEF(make_var_ref_ref, 7, 0, 2, atom_u16)
DEF(   make_var_ref, 5, 0, 2, atom)

DEF(   for_in_start, 1, 1, 1, none)
DEF(   for_of_start, 1, 1, 3, none)
DEF(for_await_of_start, 1, 1, 3, none)
DEF(    for_in_next, 1, 1, 3, none)
DEF(    for_of_next, 2, 3, 5, u8)
DEF(iterator_check_object, 1, 1, 1, none)
DEF(iterator_get_value_done, 1, 1, 2, none)
DEF( iterator_close, 1, 3, 0, none)
DEF(  iterator_next, 1, 4, 4, none)
DEF(  iterator_call, 2, 4, 5, u8)
DEF(  initial_yield, 1, 0, 0, none)
DEF(          yield, 1, 1, 2, none)
DEF(     yield_star, 1, 1, 2, none)
DEF(async_yield_star, 1, 1, 2, none)
DEF(          await, 1, 1, 1, none)

/* arithmetic/logic operations */
DEF(            neg, 1, 1, 1, none)
DEF(           plus, 1, 1, 1, none)
DEF(            dec, 1, 1, 1, none)
DEF(            inc, 1, 1, 1, none)
DEF(       post_dec, 1, 1, 2, none)
DEF(       post_inc, 1, 1, 2, none)
DEF(        dec_loc, 2, 0, 0, loc8)
DEF(        inc_loc, 2, 0, 0, loc8)
DEF(        add_loc, 2, 1, 0, loc8)
DEF(            not, 1, 1, 1, none)
DEF(           lnot, 1, 1, 1, none)
DEF(         typeof, 1, 1, 1, none)
DEF(         delete, 1, 2, 1, none)
DEF(     delete_var, 5, 0, 1, atom)

/* warning: order matters (see js_parse_assign_expr) */
DEF(            mul, 1, 2, 1, none)
DEF(            div, 1, 2, 1, none)
DEF(            mod, 1, 2, 1, none)
DEF(            add, 1, 2, 1, none)
DEF(            sub, 1, 2, 1, none)
DEF(            shl, 1, 2, 1, none)
DEF(            sar, 1, 2, 1, none)
DEF(            shr, 1, 2, 1, none)
DEF(            and, 1, 2, 1, none)
DEF(            xor, 1, 2, 1, none)
DEF(             or, 1, 2, 1, none)
DEF(            pow, 1, 2, 1, none)

DEF(             lt, 1, 2, 1, none)
DEF(            lte, 1, 2, 1, none)
DEF(             gt, 1, 2, 1, none)
DEF(            gte, 1, 2, 1, none)
DEF(     instanceof, 1, 2, 1, none)
DEF(             in, 1, 2, 1, none)
DEF(             eq, 1, 2, 1, none)
DEF(            neq, 1, 2, 1, none)
DEF(      strict_eq, 1, 2, 1, none)
DEF(     strict_neq, 1, 2, 1, none)
DEF(is_undefined_or_null, 1, 1, 1, none)
DEF(     private_in, 1, 2, 1, none)
DEF(push_bigint_i32, 5, 0, 1, i32)
/* must be the last non short and non temporary opcode */
DEF(            nop, 1, 0, 0, none)

/* temporary opcodes: never emitted in the final bytecode */

def(    enter_scope, 3, 0, 0, u16)  /* emitted in phase 1, removed in phase 2 */
def(    leave_scope, 3, 0, 0, u16)  /* emitted in phase 1, removed in phase 2 */

def(          label, 5, 0, 0, label) /* emitted in phase 1, removed in phase 3 */

def(scope_get_var_undef, 7, 0, 1, atom_u16) /* emitted in phase 1, removed in phase 2 */
def(  scope_get_var, 7, 0, 1, atom_u16) /* emitted in phase 1, removed in phase 2 */
def(  scope_put_var, 7, 1, 0, atom_u16) /* emitted in phase 1, removed in phase 2 */
def(scope_delete_var, 7, 0, 1, atom_u16) /* emitted in phase 1, removed in phase 2 */
def( scope_make_ref, 11, 0, 2, atom_label_u16) /* emitted in phase 1, removed in phase 2 */
def(  scope_get_ref, 7, 0, 2, atom_u16) /* emitted in phase 1, removed in phase 2 */
def(scope_put_var_init, 7, 0, 2, atom_u16) /* emitted in phase 1, removed in phase 2 */
def(scope_get_private_field, 7, 1, 1, atom_u16) /* obj -> value, emitted in phase 1, removed in phase 2 */
def(scope_get_private_field2, 7, 1, 2, atom_u16) /* obj -> obj value, emitted in phase 1, removed in phase 2 */
def(scope_put_private_field, 7, 2, 0, atom_u16) /* obj value ->, emitted in phase 1, removed in phase 2 */
def(scope_in_private_field, 7, 1, 1, atom_u16) /* obj -> res emitted in phase 1, removed in phase 2 */
def(get_field_opt_chain, 5, 1, 1, atom) /* emitted in phase 1, removed in phase 2 */
def(get_array_el_opt_chain, 1, 2, 1, none) /* emitted in phase 1, removed in phase 2 */
def( set_class_name, 5, 1, 1, u32) /* emitted in phase 1, removed in phase 2 */

def(     source_loc, 9, 0, 0, u32x2) /* emitted in phase 1, removed in phase 3 */

DEF(    push_minus1, 1, 0, 1, none_int)
DEF(         push_0, 1, 0, 1, none_int)
DEF(         push_1, 1, 0, 1, none_int)
DEF(         push_2, 1, 0, 1, none_int)
DEF(         push_3, 1, 0, 1, none_int)
DEF(         push_4, 1, 0, 1, none_int)
DEF(         push_5, 1, 0, 1, none_int)
DEF(         push_6, 1, 0, 1, none_int)
DEF(         push_7, 1, 0, 1, none_int)
DEF(        push_i8, 2, 0, 1, i8)
DEF(       push_i16, 3, 0, 1, i16)
DEF(    push_const8, 2, 0, 1, const8)
DEF(      fclosure8, 2, 0, 1, const8) /* must follow push_const8 */
DEF(push_empty_string, 1, 0, 1, none)

DEF(       get_loc8, 2, 0, 1, loc8)
DEF(       put_loc8, 2, 1, 0, loc8)
DEF(       set_loc8, 2, 1, 1, loc8)

DEF(  get_loc0_loc1, 1, 0, 2, none_loc)
DEF(       get_loc0, 1, 0, 1, none_loc)
DEF(       get_loc1, 1, 0, 1, none_loc)
DEF(       get_loc2, 1, 0, 1, none_loc)
DEF(       get_loc3, 1, 0, 1, none_loc)
DEF(       put_loc0, 1, 1, 0, none_loc)
DEF(       put_loc1, 1, 1, 0, none_loc)
DEF(       put_loc2, 1, 1, 0, none_loc)
DEF(       put_loc3, 1, 1, 0, none_loc)
DEF(       set_loc0, 1, 1, 1, none_loc)
DEF(       set_loc1, 1, 1, 1, none_loc)
DEF(       set_loc2, 1, 1, 1, none_loc)
DEF(       set_loc3, 1, 1, 1, none_loc)
DEF(       get_arg0, 1, 0, 1, none_arg)
DEF(       get_arg1, 1, 0, 1, none_arg)
DEF(       get_arg2, 1, 0, 1, none_arg)
DEF(       get_arg3, 1, 0, 1, none_arg)
DEF(       put_arg0, 1, 1, 0, none_arg)
DEF(       put_arg1, 1, 1, 0, none_arg)
DEF(       put_arg2, 1, 1, 0, none_arg)
DEF(       put_arg3, 1, 1, 0, none_arg)
DEF(       set_arg0, 1, 1, 1, none_arg)
DEF(       set_arg1, 1, 1, 1, none_arg)
DEF(       set_arg2, 1, 1, 1, none_arg)
DEF(       set_arg3, 1, 1, 1, none_arg)
DEF(   get_var_ref0, 1, 0, 1, none_var_ref)
DEF(   get_var_ref1, 1, 0, 1, none_var_ref)
DEF(   get_var_ref2, 1, 0, 1, none_var_ref)
DEF(   get_var_ref3, 1, 0, 1, none_var_ref)
DEF(   put_var_ref0, 1, 1, 0, none_var_ref)
DEF(   put_var_ref1, 1, 1, 0, none_var_ref)
DEF(   put_var_ref2, 1, 1, 0, none_var_ref)
DEF(   put_var_ref3, 1, 1, 0, none_var_ref)
DEF(   set_var_ref0, 1, 1, 1, none_var_ref)
DEF(   set_var_ref1, 1, 1, 1, none_var_ref)
DEF(   set_var_ref2, 1, 1, 1, none_var_ref)
DEF(   set_var_ref3, 1, 1, 1, none_var_ref)

DEF(     get_length, 1, 1, 1, none)

DEF(      if_false8, 2, 1, 0, label8)
DEF(       if_true8, 2, 1, 0, label8) /* must come after if_false8 */
DEF(          goto8, 2, 0, 0, label8) /* must come after if_true8 */
DEF(         goto16, 3, 0, 0, label16)

DEF(          call0, 1, 1, 1, npopx)
DEF(          call1, 1, 1, 1, npopx)
DEF(          call2, 1, 1, 1, npopx)
DEF(          call3, 1, 1, 1, npopx)

DEF(   is_undefined, 1, 1, 1, none)
DEF(        is_null, 1, 1, 1, none)
DEF(typeof_is_undefined, 1, 1, 1, none)
DEF( typeof_is_function, 1, 1, 1, none)

#undef DEF
#undef def
#endif  /* DEF */

#undef def
#undef DEF
#undef FMT
    OP_COUNT, /* excluding temporary opcodes */
    /* temporary opcodes : overlap with the short opcodes */
    OP_TEMP_START = OP_nop + 1,
    OP___dummy = OP_TEMP_START - 1,
#define FMT(f)
#define DEF(id, size, n_pop, n_push, f)
#define def(id, size, n_pop, n_push, f) OP_ ## id,
/*
 * QuickJS opcode definitions
 *
 * Copyright (c) 2017-2018 Fabrice Bellard
 * Copyright (c) 2017-2018 Charlie Gordon
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifdef FMT
FMT(none)
FMT(none_int)
FMT(none_loc)
FMT(none_arg)
FMT(none_var_ref)
FMT(u8)
FMT(i8)
FMT(loc8)
FMT(const8)
FMT(label8)
FMT(u16)
FMT(i16)
FMT(label16)
FMT(npop)
FMT(npopx)
FMT(npop_u16)
FMT(loc)
FMT(arg)
FMT(var_ref)
FMT(u32)
FMT(u32x2)
FMT(i32)
FMT(const)
FMT(label)
FMT(atom)
FMT(atom_u8)
FMT(atom_u16)
FMT(atom_label_u8)
FMT(atom_label_u16)
FMT(label_u16)
#undef FMT
#endif /* FMT */

#ifdef DEF

#ifndef def
#define def(id, size, n_pop, n_push, f) DEF(id, size, n_pop, n_push, f)
#endif

DEF(invalid, 1, 0, 0, none) /* never emitted */

/* push values */
DEF(       push_i32, 5, 0, 1, i32)
DEF(     push_const, 5, 0, 1, const)
DEF(       fclosure, 5, 0, 1, const) /* must follow push_const */
DEF(push_atom_value, 5, 0, 1, atom)
DEF( private_symbol, 5, 0, 1, atom)
DEF(      undefined, 1, 0, 1, none)
DEF(           null, 1, 0, 1, none)
DEF(      push_this, 1, 0, 1, none) /* only used at the start of a function */
DEF(     push_false, 1, 0, 1, none)
DEF(      push_true, 1, 0, 1, none)
DEF(         object, 1, 0, 1, none)
DEF( special_object, 2, 0, 1, u8) /* only used at the start of a function */
DEF(           rest, 3, 0, 1, u16) /* only used at the start of a function */

DEF(           drop, 1, 1, 0, none) /* a -> */
DEF(            nip, 1, 2, 1, none) /* a b -> b */
DEF(           nip1, 1, 3, 2, none) /* a b c -> b c */
DEF(            dup, 1, 1, 2, none) /* a -> a a */
DEF(           dup1, 1, 2, 3, none) /* a b -> a a b */
DEF(           dup2, 1, 2, 4, none) /* a b -> a b a b */
DEF(           dup3, 1, 3, 6, none) /* a b c -> a b c a b c */
DEF(        insert2, 1, 2, 3, none) /* obj a -> a obj a (dup_x1) */
DEF(        insert3, 1, 3, 4, none) /* obj prop a -> a obj prop a (dup_x2) */
DEF(        insert4, 1, 4, 5, none) /* this obj prop a -> a this obj prop a */
DEF(          perm3, 1, 3, 3, none) /* obj a b -> a obj b */
DEF(          perm4, 1, 4, 4, none) /* obj prop a b -> a obj prop b */
DEF(          perm5, 1, 5, 5, none) /* this obj prop a b -> a this obj prop b */
DEF(           swap, 1, 2, 2, none) /* a b -> b a */
DEF(          swap2, 1, 4, 4, none) /* a b c d -> c d a b */
DEF(          rot3l, 1, 3, 3, none) /* x a b -> a b x */
DEF(          rot3r, 1, 3, 3, none) /* a b x -> x a b */
DEF(          rot4l, 1, 4, 4, none) /* x a b c -> a b c x */
DEF(          rot5l, 1, 5, 5, none) /* x a b c d -> a b c d x */

DEF(call_constructor, 3, 2, 1, npop) /* func new.target args -> ret. arguments are not counted in n_pop */
DEF(           call, 3, 1, 1, npop) /* arguments are not counted in n_pop */
DEF(      tail_call, 3, 1, 0, npop) /* arguments are not counted in n_pop */
DEF(    call_method, 3, 2, 1, npop) /* arguments are not counted in n_pop */
DEF(tail_call_method, 3, 2, 0, npop) /* arguments are not counted in n_pop */
DEF(     array_from, 3, 0, 1, npop) /* arguments are not counted in n_pop */
DEF(          apply, 3, 3, 1, u16)
DEF(         return, 1, 1, 0, none)
DEF(   return_undef, 1, 0, 0, none)
DEF(check_ctor_return, 1, 1, 2, none)
DEF(     check_ctor, 1, 0, 0, none)
DEF(      init_ctor, 1, 0, 1, none)
DEF(    check_brand, 1, 2, 2, none) /* this_obj func -> this_obj func */
DEF(      add_brand, 1, 2, 0, none) /* this_obj home_obj -> */
DEF(   return_async, 1, 1, 0, none)
DEF(          throw, 1, 1, 0, none)
DEF(    throw_error, 6, 0, 0, atom_u8)
DEF(           eval, 5, 1, 1, npop_u16) /* func args... -> ret_val */
DEF(     apply_eval, 3, 2, 1, u16) /* func array -> ret_eval */
DEF(         regexp, 1, 2, 1, none) /* create a RegExp object from the pattern and a
                                       bytecode string */
DEF(      get_super, 1, 1, 1, none)
DEF(         import, 1, 1, 1, none) /* dynamic module import */

DEF(      check_var, 5, 0, 1, atom) /* check if a variable exists */
DEF(  get_var_undef, 5, 0, 1, atom) /* push undefined if the variable does not exist */
DEF(        get_var, 5, 0, 1, atom) /* throw an exception if the variable does not exist */
DEF(        put_var, 5, 1, 0, atom) /* must come after get_var */
DEF(   put_var_init, 5, 1, 0, atom) /* must come after put_var. Used to initialize a global lexical variable */
DEF( put_var_strict, 5, 2, 0, atom) /* for strict mode variable write */

DEF(  get_ref_value, 1, 2, 3, none)
DEF(  put_ref_value, 1, 3, 0, none)

DEF(     define_var, 6, 0, 0, atom_u8)
DEF(check_define_var, 6, 0, 0, atom_u8)
DEF(    define_func, 6, 1, 0, atom_u8)

// order matters, see IC counterparts
DEF(      get_field, 5, 1, 1, atom)
DEF(     get_field2, 5, 1, 2, atom)
DEF(      put_field, 5, 2, 0, atom)

DEF( get_private_field, 1, 2, 1, none) /* obj prop -> value */
DEF( put_private_field, 1, 3, 0, none) /* obj value prop -> */
DEF(define_private_field, 1, 3, 1, none) /* obj prop value -> obj */
DEF(   get_array_el, 1, 2, 1, none)
DEF(  get_array_el2, 1, 2, 2, none) /* obj prop -> obj value */
DEF(   put_array_el, 1, 3, 0, none)
DEF(get_super_value, 1, 3, 1, none) /* this obj prop -> value */
DEF(put_super_value, 1, 4, 0, none) /* this obj prop value -> */
DEF(   define_field, 5, 2, 1, atom)
DEF(       set_name, 5, 1, 1, atom)
DEF(set_name_computed, 1, 2, 2, none)
DEF(      set_proto, 1, 2, 1, none)
DEF(set_home_object, 1, 2, 2, none)
DEF(define_array_el, 1, 3, 2, none)
DEF(         append, 1, 3, 2, none) /* append enumerated object, update length */
DEF(copy_data_properties, 2, 3, 3, u8)
DEF(  define_method, 6, 2, 1, atom_u8)
DEF(define_method_computed, 2, 3, 1, u8) /* must come after define_method */
DEF(   define_class, 6, 2, 2, atom_u8) /* parent ctor -> ctor proto */
DEF(   define_class_computed, 6, 3, 3, atom_u8) /* field_name parent ctor -> field_name ctor proto (class with computed name) */

DEF(        get_loc, 3, 0, 1, loc)
DEF(        put_loc, 3, 1, 0, loc) /* must come after get_loc */
DEF(        set_loc, 3, 1, 1, loc) /* must come after put_loc */
DEF(        get_arg, 3, 0, 1, arg)
DEF(        put_arg, 3, 1, 0, arg) /* must come after get_arg */
DEF(        set_arg, 3, 1, 1, arg) /* must come after put_arg */
DEF(    get_var_ref, 3, 0, 1, var_ref)
DEF(    put_var_ref, 3, 1, 0, var_ref) /* must come after get_var_ref */
DEF(    set_var_ref, 3, 1, 1, var_ref) /* must come after put_var_ref */
DEF(set_loc_uninitialized, 3, 0, 0, loc)
DEF(  get_loc_check, 3, 0, 1, loc)
DEF(  put_loc_check, 3, 1, 0, loc) /* must come after get_loc_check */
DEF(  put_loc_check_init, 3, 1, 0, loc)
DEF(get_var_ref_check, 3, 0, 1, var_ref)
DEF(put_var_ref_check, 3, 1, 0, var_ref) /* must come after get_var_ref_check */
DEF(put_var_ref_check_init, 3, 1, 0, var_ref)
DEF(      close_loc, 3, 0, 0, loc)
DEF(       if_false, 5, 1, 0, label)
DEF(        if_true, 5, 1, 0, label) /* must come after if_false */
DEF(           goto, 5, 0, 0, label) /* must come after if_true */
DEF(          catch, 5, 0, 1, label)
DEF(          gosub, 5, 0, 0, label) /* used to execute the finally block */
DEF(            ret, 1, 1, 0, none) /* used to return from the finally block */
DEF(      nip_catch, 1, 2, 1, none) /* catch ... a -> a */

DEF(      to_object, 1, 1, 1, none)
//DEF(      to_string, 1, 1, 1, none)
DEF(     to_propkey, 1, 1, 1, none)
DEF(    to_propkey2, 1, 2, 2, none)

DEF(   with_get_var, 10, 1, 0, atom_label_u8)     /* must be in the same order as scope_xxx */
DEF(   with_put_var, 10, 2, 1, atom_label_u8)     /* must be in the same order as scope_xxx */
DEF(with_delete_var, 10, 1, 0, atom_label_u8)     /* must be in the same order as scope_xxx */
DEF(  with_make_ref, 10, 1, 0, atom_label_u8)     /* must be in the same order as scope_xxx */
DEF(   with_get_ref, 10, 1, 0, atom_label_u8)     /* must be in the same order as scope_xxx */
DEF(with_get_ref_undef, 10, 1, 0, atom_label_u8)

DEF(   make_loc_ref, 7, 0, 2, atom_u16)
DEF(   make_arg_ref, 7, 0, 2, atom_u16)
DEF(make_var_ref_ref, 7, 0, 2, atom_u16)
DEF(   make_var_ref, 5, 0, 2, atom)

DEF(   for_in_start, 1, 1, 1, none)
DEF(   for_of_start, 1, 1, 3, none)
DEF(for_await_of_start, 1, 1, 3, none)
DEF(    for_in_next, 1, 1, 3, none)
DEF(    for_of_next, 2, 3, 5, u8)
DEF(iterator_check_object, 1, 1, 1, none)
DEF(iterator_get_value_done, 1, 1, 2, none)
DEF( iterator_close, 1, 3, 0, none)
DEF(  iterator_next, 1, 4, 4, none)
DEF(  iterator_call, 2, 4, 5, u8)
DEF(  initial_yield, 1, 0, 0, none)
DEF(          yield, 1, 1, 2, none)
DEF(     yield_star, 1, 1, 2, none)
DEF(async_yield_star, 1, 1, 2, none)
DEF(          await, 1, 1, 1, none)

/* arithmetic/logic operations */
DEF(            neg, 1, 1, 1, none)
DEF(           plus, 1, 1, 1, none)
DEF(            dec, 1, 1, 1, none)
DEF(            inc, 1, 1, 1, none)
DEF(       post_dec, 1, 1, 2, none)
DEF(       post_inc, 1, 1, 2, none)
DEF(        dec_loc, 2, 0, 0, loc8)
DEF(        inc_loc, 2, 0, 0, loc8)
DEF(        add_loc, 2, 1, 0, loc8)
DEF(            not, 1, 1, 1, none)
DEF(           lnot, 1, 1, 1, none)
DEF(         typeof, 1, 1, 1, none)
DEF(         delete, 1, 2, 1, none)
DEF(     delete_var, 5, 0, 1, atom)

/* warning: order matters (see js_parse_assign_expr) */
DEF(            mul, 1, 2, 1, none)
DEF(            div, 1, 2, 1, none)
DEF(            mod, 1, 2, 1, none)
DEF(            add, 1, 2, 1, none)
DEF(            sub, 1, 2, 1, none)
DEF(            shl, 1, 2, 1, none)
DEF(            sar, 1, 2, 1, none)
DEF(            shr, 1, 2, 1, none)
DEF(            and, 1, 2, 1, none)
DEF(            xor, 1, 2, 1, none)
DEF(             or, 1, 2, 1, none)
DEF(            pow, 1, 2, 1, none)

DEF(             lt, 1, 2, 1, none)
DEF(            lte, 1, 2, 1, none)
DEF(             gt, 1, 2, 1, none)
DEF(            gte, 1, 2, 1, none)
DEF(     instanceof, 1, 2, 1, none)
DEF(             in, 1, 2, 1, none)
DEF(             eq, 1, 2, 1, none)
DEF(            neq, 1, 2, 1, none)
DEF(      strict_eq, 1, 2, 1, none)
DEF(     strict_neq, 1, 2, 1, none)
DEF(is_undefined_or_null, 1, 1, 1, none)
DEF(     private_in, 1, 2, 1, none)
DEF(push_bigint_i32, 5, 0, 1, i32)
/* must be the last non short and non temporary opcode */
DEF(            nop, 1, 0, 0, none)

/* temporary opcodes: never emitted in the final bytecode */

def(    enter_scope, 3, 0, 0, u16)  /* emitted in phase 1, removed in phase 2 */
def(    leave_scope, 3, 0, 0, u16)  /* emitted in phase 1, removed in phase 2 */

def(          label, 5, 0, 0, label) /* emitted in phase 1, removed in phase 3 */

def(scope_get_var_undef, 7, 0, 1, atom_u16) /* emitted in phase 1, removed in phase 2 */
def(  scope_get_var, 7, 0, 1, atom_u16) /* emitted in phase 1, removed in phase 2 */
def(  scope_put_var, 7, 1, 0, atom_u16) /* emitted in phase 1, removed in phase 2 */
def(scope_delete_var, 7, 0, 1, atom_u16) /* emitted in phase 1, removed in phase 2 */
def( scope_make_ref, 11, 0, 2, atom_label_u16) /* emitted in phase 1, removed in phase 2 */
def(  scope_get_ref, 7, 0, 2, atom_u16) /* emitted in phase 1, removed in phase 2 */
def(scope_put_var_init, 7, 0, 2, atom_u16) /* emitted in phase 1, removed in phase 2 */
def(scope_get_private_field, 7, 1, 1, atom_u16) /* obj -> value, emitted in phase 1, removed in phase 2 */
def(scope_get_private_field2, 7, 1, 2, atom_u16) /* obj -> obj value, emitted in phase 1, removed in phase 2 */
def(scope_put_private_field, 7, 2, 0, atom_u16) /* obj value ->, emitted in phase 1, removed in phase 2 */
def(scope_in_private_field, 7, 1, 1, atom_u16) /* obj -> res emitted in phase 1, removed in phase 2 */
def(get_field_opt_chain, 5, 1, 1, atom) /* emitted in phase 1, removed in phase 2 */
def(get_array_el_opt_chain, 1, 2, 1, none) /* emitted in phase 1, removed in phase 2 */
def( set_class_name, 5, 1, 1, u32) /* emitted in phase 1, removed in phase 2 */

def(     source_loc, 9, 0, 0, u32x2) /* emitted in phase 1, removed in phase 3 */

DEF(    push_minus1, 1, 0, 1, none_int)
DEF(         push_0, 1, 0, 1, none_int)
DEF(         push_1, 1, 0, 1, none_int)
DEF(         push_2, 1, 0, 1, none_int)
DEF(         push_3, 1, 0, 1, none_int)
DEF(         push_4, 1, 0, 1, none_int)
DEF(         push_5, 1, 0, 1, none_int)
DEF(         push_6, 1, 0, 1, none_int)
DEF(         push_7, 1, 0, 1, none_int)
DEF(        push_i8, 2, 0, 1, i8)
DEF(       push_i16, 3, 0, 1, i16)
DEF(    push_const8, 2, 0, 1, const8)
DEF(      fclosure8, 2, 0, 1, const8) /* must follow push_const8 */
DEF(push_empty_string, 1, 0, 1, none)

DEF(       get_loc8, 2, 0, 1, loc8)
DEF(       put_loc8, 2, 1, 0, loc8)
DEF(       set_loc8, 2, 1, 1, loc8)

DEF(  get_loc0_loc1, 1, 0, 2, none_loc)
DEF(       get_loc0, 1, 0, 1, none_loc)
DEF(       get_loc1, 1, 0, 1, none_loc)
DEF(       get_loc2, 1, 0, 1, none_loc)
DEF(       get_loc3, 1, 0, 1, none_loc)
DEF(       put_loc0, 1, 1, 0, none_loc)
DEF(       put_loc1, 1, 1, 0, none_loc)
DEF(       put_loc2, 1, 1, 0, none_loc)
DEF(       put_loc3, 1, 1, 0, none_loc)
DEF(       set_loc0, 1, 1, 1, none_loc)
DEF(       set_loc1, 1, 1, 1, none_loc)
DEF(       set_loc2, 1, 1, 1, none_loc)
DEF(       set_loc3, 1, 1, 1, none_loc)
DEF(       get_arg0, 1, 0, 1, none_arg)
DEF(       get_arg1, 1, 0, 1, none_arg)
DEF(       get_arg2, 1, 0, 1, none_arg)
DEF(       get_arg3, 1, 0, 1, none_arg)
DEF(       put_arg0, 1, 1, 0, none_arg)
DEF(       put_arg1, 1, 1, 0, none_arg)
DEF(       put_arg2, 1, 1, 0, none_arg)
DEF(       put_arg3, 1, 1, 0, none_arg)
DEF(       set_arg0, 1, 1, 1, none_arg)
DEF(       set_arg1, 1, 1, 1, none_arg)
DEF(       set_arg2, 1, 1, 1, none_arg)
DEF(       set_arg3, 1, 1, 1, none_arg)
DEF(   get_var_ref0, 1, 0, 1, none_var_ref)
DEF(   get_var_ref1, 1, 0, 1, none_var_ref)
DEF(   get_var_ref2, 1, 0, 1, none_var_ref)
DEF(   get_var_ref3, 1, 0, 1, none_var_ref)
DEF(   put_var_ref0, 1, 1, 0, none_var_ref)
DEF(   put_var_ref1, 1, 1, 0, none_var_ref)
DEF(   put_var_ref2, 1, 1, 0, none_var_ref)
DEF(   put_var_ref3, 1, 1, 0, none_var_ref)
DEF(   set_var_ref0, 1, 1, 1, none_var_ref)
DEF(   set_var_ref1, 1, 1, 1, none_var_ref)
DEF(   set_var_ref2, 1, 1, 1, none_var_ref)
DEF(   set_var_ref3, 1, 1, 1, none_var_ref)

DEF(     get_length, 1, 1, 1, none)

DEF(      if_false8, 2, 1, 0, label8)
DEF(       if_true8, 2, 1, 0, label8) /* must come after if_false8 */
DEF(          goto8, 2, 0, 0, label8) /* must come after if_true8 */
DEF(         goto16, 3, 0, 0, label16)

DEF(          call0, 1, 1, 1, npopx)
DEF(          call1, 1, 1, 1, npopx)
DEF(          call2, 1, 1, 1, npopx)
DEF(          call3, 1, 1, 1, npopx)

DEF(   is_undefined, 1, 1, 1, none)
DEF(        is_null, 1, 1, 1, none)
DEF(typeof_is_undefined, 1, 1, 1, none)
DEF( typeof_is_function, 1, 1, 1, none)

#undef DEF
#undef def
#endif  /* DEF */

#undef def
#undef DEF
#undef FMT
    OP_TEMP_END,
} OPCodeEnum;

static int JS_InitAtoms(JSRuntime *rt);
static JSAtom __JS_NewAtomInit(JSRuntime *rt, const char *str, int len,
                               int atom_type);
static void JS_FreeAtomStruct(JSRuntime *rt, JSAtomStruct *p);
static void free_function_bytecode(JSRuntime *rt, JSFunctionBytecode *b);
static JSValue js_call_c_function(JSContext *ctx, JSValueConst func_obj,
                                  JSValueConst this_obj,
                                  int argc, JSValueConst *argv, int flags);
static JSValue js_call_bound_function(JSContext *ctx, JSValueConst func_obj,
                                      JSValueConst this_obj,
                                      int argc, JSValueConst *argv, int flags);
static JSValue JS_CallInternal(JSContext *ctx, JSValueConst func_obj,
                               JSValueConst this_obj, JSValueConst new_target,
                               int argc, JSValueConst *argv, int flags);
static JSValue JS_CallConstructorInternal(JSContext *ctx,
                                          JSValueConst func_obj,
                                          JSValueConst new_target,
                                          int argc, JSValueConst *argv, int flags);
static JSValue JS_CallFree(JSContext *ctx, JSValue func_obj, JSValueConst this_obj,
                           int argc, JSValueConst *argv);
static JSValue JS_InvokeFree(JSContext *ctx, JSValue this_val, JSAtom atom,
                             int argc, JSValueConst *argv);
static __exception int JS_ToArrayLengthFree(JSContext *ctx, uint32_t *plen,
                                            JSValue val, bool is_array_ctor);
static JSValue JS_EvalObject(JSContext *ctx, JSValueConst this_obj,
                             JSValueConst val, int flags, int scope_idx);
static __maybe_unused void JS_DumpString(JSRuntime *rt, JSString *p);
static __maybe_unused void JS_DumpObjectHeader(JSRuntime *rt);
static __maybe_unused void JS_DumpObject(JSRuntime *rt, JSObject *p);
static __maybe_unused void JS_DumpGCObject(JSRuntime *rt, JSGCObjectHeader *p);
static __maybe_unused void JS_DumpValue(JSRuntime *rt, JSValueConst val);
static __maybe_unused void JS_DumpAtoms(JSRuntime *rt);
static __maybe_unused void JS_DumpShapes(JSRuntime *rt);

static JSValue js_function_apply(JSContext *ctx, JSValueConst this_val,
                                 int argc, JSValueConst *argv, int magic);
static void js_array_finalizer(JSRuntime *rt, JSValueConst val);
static void js_array_mark(JSRuntime *rt, JSValueConst val,
                          JS_MarkFunc *mark_func);
static void js_object_data_finalizer(JSRuntime *rt, JSValueConst val);
static void js_object_data_mark(JSRuntime *rt, JSValueConst val,
                                JS_MarkFunc *mark_func);
static void js_c_function_finalizer(JSRuntime *rt, JSValueConst val);
static void js_c_function_mark(JSRuntime *rt, JSValueConst val,
                               JS_MarkFunc *mark_func);
static void js_bytecode_function_finalizer(JSRuntime *rt, JSValueConst val);
static void js_bytecode_function_mark(JSRuntime *rt, JSValueConst val,
                                JS_MarkFunc *mark_func);
static void js_bound_function_finalizer(JSRuntime *rt, JSValueConst val);
static void js_bound_function_mark(JSRuntime *rt, JSValueConst val,
                                JS_MarkFunc *mark_func);
static void js_for_in_iterator_finalizer(JSRuntime *rt, JSValueConst val);
static void js_for_in_iterator_mark(JSRuntime *rt, JSValueConst val,
                                JS_MarkFunc *mark_func);
static void js_regexp_finalizer(JSRuntime *rt, JSValueConst val);
static void js_array_buffer_finalizer(JSRuntime *rt, JSValueConst val);
static void js_typed_array_finalizer(JSRuntime *rt, JSValueConst val);
static void js_typed_array_mark(JSRuntime *rt, JSValueConst val,
                                JS_MarkFunc *mark_func);
static void js_proxy_finalizer(JSRuntime *rt, JSValueConst val);
static void js_proxy_mark(JSRuntime *rt, JSValueConst val,
                                JS_MarkFunc *mark_func);
static void js_map_finalizer(JSRuntime *rt, JSValueConst val);
static void js_map_mark(JSRuntime *rt, JSValueConst val,
                                JS_MarkFunc *mark_func);
static void js_map_iterator_finalizer(JSRuntime *rt, JSValueConst val);
static void js_map_iterator_mark(JSRuntime *rt, JSValueConst val,
                                JS_MarkFunc *mark_func);
static void js_array_iterator_finalizer(JSRuntime *rt, JSValueConst val);
static void js_array_iterator_mark(JSRuntime *rt, JSValueConst val,
                                JS_MarkFunc *mark_func);
static void js_iterator_concat_finalizer(JSRuntime *rt, JSValueConst val);
static void js_iterator_concat_mark(JSRuntime *rt, JSValueConst val,
                                    JS_MarkFunc *mark_func);
static void js_iterator_helper_finalizer(JSRuntime *rt, JSValueConst val);
static void js_iterator_helper_mark(JSRuntime *rt, JSValueConst val,
                                    JS_MarkFunc *mark_func);
static void js_iterator_wrap_finalizer(JSRuntime *rt, JSValueConst val);
static void js_iterator_wrap_mark(JSRuntime *rt, JSValueConst val,
                                  JS_MarkFunc *mark_func);
static void js_regexp_string_iterator_finalizer(JSRuntime *rt,
                                                JSValueConst val);
static void js_regexp_string_iterator_mark(JSRuntime *rt, JSValueConst val,
                                JS_MarkFunc *mark_func);
static void js_generator_finalizer(JSRuntime *rt, JSValueConst val);
static void js_generator_mark(JSRuntime *rt, JSValueConst val,
                                JS_MarkFunc *mark_func);
static void js_promise_finalizer(JSRuntime *rt, JSValueConst val);
static void js_promise_mark(JSRuntime *rt, JSValueConst val,
                                JS_MarkFunc *mark_func);
static void js_promise_resolve_function_finalizer(JSRuntime *rt, JSValueConst val);
static void js_promise_resolve_function_mark(JSRuntime *rt, JSValueConst val,
                                JS_MarkFunc *mark_func);

#define HINT_STRING  0
#define HINT_NUMBER  1
#define HINT_NONE    2
#define HINT_FORCE_ORDINARY (1 << 4) // don't try Symbol.toPrimitive
static JSValue JS_ToPrimitiveFree(JSContext *ctx, JSValue val, int hint);
static JSValue JS_ToStringFree(JSContext *ctx, JSValue val);
static int JS_ToBoolFree(JSContext *ctx, JSValue val);
static int JS_ToInt32Free(JSContext *ctx, int32_t *pres, JSValue val);
static int JS_ToFloat64Free(JSContext *ctx, double *pres, JSValue val);
static int JS_ToUint8ClampFree(JSContext *ctx, int32_t *pres, JSValue val);
static JSValue JS_ToPropertyKeyInternal(JSContext *ctx, JSValueConst val,
                                        int flags);
static JSValue js_new_string8_len(JSContext *ctx, const char *buf, int len);
static JSValue js_compile_regexp(JSContext *ctx, JSValueConst pattern,
                                 JSValueConst flags);
static JSValue js_regexp_constructor_internal(JSContext *ctx, JSValueConst ctor,
                                              JSValue pattern, JSValue bc);
static void gc_decref(JSRuntime *rt);
static int JS_NewClass1(JSRuntime *rt, JSClassID class_id,
                        const JSClassDef *class_def, JSAtom name);
static JSValue js_array_push(JSContext *ctx, JSValueConst this_val,
                             int argc, JSValueConst *argv, int unshift);
static JSValue js_array_constructor(JSContext *ctx, JSValueConst new_target,
                                    int argc, JSValueConst *argv);
static JSValue js_error_constructor(JSContext *ctx, JSValueConst new_target,
                                    int argc, JSValueConst *argv, int magic);
static JSValue js_object_defineProperty(JSContext *ctx, JSValueConst this_val,
                                        int argc, JSValueConst *argv, int magic);

typedef enum JSStrictEqModeEnum {
    JS_EQ_STRICT,
    JS_EQ_SAME_VALUE,
    JS_EQ_SAME_VALUE_ZERO,
} JSStrictEqModeEnum;

static bool js_strict_eq2(JSContext *ctx, JSValue op1, JSValue op2,
                          JSStrictEqModeEnum eq_mode);
static bool js_strict_eq(JSContext *ctx, JSValue op1, JSValue op2);
static bool js_same_value(JSContext *ctx, JSValueConst op1, JSValueConst op2);
static bool js_same_value_zero(JSContext *ctx, JSValueConst op1, JSValueConst op2);
static JSValue JS_ToObjectFree(JSContext *ctx, JSValue val);
static JSProperty *add_property(JSContext *ctx,
                                JSObject *p, JSAtom prop, int prop_flags);
static int JS_ToBigInt64Free(JSContext *ctx, int64_t *pres, JSValue val);
static JSValue JS_ThrowStackOverflow(JSContext *ctx);
static JSValue JS_ThrowTypeErrorRevokedProxy(JSContext *ctx);
static JSValue js_proxy_getPrototypeOf(JSContext *ctx, JSValueConst obj);
static int js_proxy_setPrototypeOf(JSContext *ctx, JSValueConst obj,
                                   JSValueConst proto_val, bool throw_flag);
static int js_proxy_isExtensible(JSContext *ctx, JSValueConst obj);
static int js_proxy_preventExtensions(JSContext *ctx, JSValueConst obj);
static int js_proxy_isArray(JSContext *ctx, JSValueConst obj);
static int JS_CreateProperty(JSContext *ctx, JSObject *p,
                             JSAtom prop, JSValueConst val,
                             JSValueConst getter, JSValueConst setter,
                             int flags);
static int js_string_memcmp(JSString *p1, JSString *p2, int len);
static void reset_weak_ref(JSRuntime *rt, JSWeakRefRecord **first_weak_ref);
static bool is_valid_weakref_target(JSValueConst val);
static void insert_weakref_record(JSValueConst target,
                                  struct JSWeakRefRecord *wr);
static JSValue js_array_buffer_constructor3(JSContext *ctx,
                                            JSValueConst new_target,
                                            uint64_t len, uint64_t *max_len,
                                            JSClassID class_id,
                                            uint8_t *buf,
                                            JSFreeArrayBufferDataFunc *free_func,
                                            void *opaque, bool alloc_flag);
static void js_array_buffer_free(JSRuntime *rt, void *opaque, void *ptr);
static JSArrayBuffer *js_get_array_buffer(JSContext *ctx, JSValueConst obj);
static bool array_buffer_is_resizable(const JSArrayBuffer *abuf);
static JSValue js_typed_array_constructor(JSContext *ctx,
                                          JSValueConst this_val,
                                          int argc, JSValueConst *argv,
                                          int classid);
static JSValue js_typed_array_constructor_ta(JSContext *ctx,
                                             JSValueConst new_target,
                                             JSValueConst src_obj,
                                             int classid, uint32_t len);
static bool is_typed_array(JSClassID class_id);
static bool typed_array_is_oob(JSObject *p);
static uint32_t typed_array_length(JSObject *p);
static JSValue JS_ThrowTypeErrorDetachedArrayBuffer(JSContext *ctx);
static JSValue JS_ThrowTypeErrorArrayBufferOOB(JSContext *ctx);
static JSVarRef *get_var_ref(JSContext *ctx, JSStackFrame *sf, int var_idx,
                             bool is_arg);
static JSValue js_call_generator_function(JSContext *ctx, JSValueConst func_obj,
                                          JSValueConst this_obj,
                                          int argc, JSValueConst *argv,
                                          int flags);
static void js_async_function_resolve_finalizer(JSRuntime *rt,
                                                JSValueConst val);
static void js_async_function_resolve_mark(JSRuntime *rt, JSValueConst val,
                                           JS_MarkFunc *mark_func);
static JSValue JS_EvalInternal(JSContext *ctx, JSValueConst this_obj,
                               const char *input, size_t input_len,
                               const char *filename, int line, int flags, int scope_idx);
static void js_free_module_def(JSContext *ctx, JSModuleDef *m);
static void js_mark_module_def(JSRuntime *rt, JSModuleDef *m,
                               JS_MarkFunc *mark_func);
static JSValue js_import_meta(JSContext *ctx);
static JSValue js_dynamic_import(JSContext *ctx, JSValueConst specifier);
static void free_var_ref(JSRuntime *rt, JSVarRef *var_ref);
static JSValue js_new_promise_capability(JSContext *ctx,
                                         JSValue *resolving_funcs,
                                         JSValueConst ctor);
static __exception int perform_promise_then(JSContext *ctx,
                                            JSValueConst promise,
                                            JSValueConst *resolve_reject,
                                            JSValueConst *cap_resolving_funcs);
static JSValue js_promise_resolve(JSContext *ctx, JSValueConst this_val,
                                  int argc, JSValueConst *argv, int magic);
static JSValue js_promise_then(JSContext *ctx, JSValueConst this_val,
                               int argc, JSValueConst *argv);
static JSValue js_promise_resolve_thenable_job(JSContext *ctx,
                                               int argc, JSValueConst *argv);
static bool js_string_eq(JSString *p1, JSString *p2);
static int js_string_compare(JSString *p1, JSString *p2);
static int JS_SetPropertyValue(JSContext *ctx, JSValueConst this_obj,
                               JSValue prop, JSValue val, int flags);
static int JS_NumberIsInteger(JSContext *ctx, JSValueConst val);
static bool JS_NumberIsNegativeOrMinusZero(JSContext *ctx, JSValueConst val);
static JSValue JS_ToNumberFree(JSContext *ctx, JSValue val);
static int JS_GetOwnPropertyInternal(JSContext *ctx, JSPropertyDescriptor *desc,
                                     JSObject *p, JSAtom prop);
static void js_free_desc(JSContext *ctx, JSPropertyDescriptor *desc);
static void async_func_mark(JSRuntime *rt, JSAsyncFunctionState *s,
                            JS_MarkFunc *mark_func);
static void JS_AddIntrinsicBasicObjects(JSContext *ctx);
static void js_free_shape(JSRuntime *rt, JSShape *sh);
static void js_free_shape_null(JSRuntime *rt, JSShape *sh);
static int js_shape_prepare_update(JSContext *ctx, JSObject *p,
                                   JSShapeProperty **pprs);
static int init_shape_hash(JSRuntime *rt);
static __exception int js_get_length32(JSContext *ctx, uint32_t *pres,
                                       JSValueConst obj);
static __exception int js_get_length64(JSContext *ctx, int64_t *pres,
                                       JSValueConst obj);
static __exception int js_set_length64(JSContext *ctx, JSValueConst obj,
                                       int64_t len);
static void free_arg_list(JSContext *ctx, JSValue *tab, uint32_t len);
static JSValue *build_arg_list(JSContext *ctx, uint32_t *plen,
                               JSValueConst array_arg);
static JSValue js_create_array(JSContext *ctx, int len, JSValueConst *tab);
static bool js_get_fast_array(JSContext *ctx, JSValue obj,
                              JSValue **arrpp, uint32_t *countp);
static int expand_fast_array(JSContext *ctx, JSObject *p, uint32_t new_len);
static JSValue JS_CreateAsyncFromSyncIterator(JSContext *ctx,
                                              JSValue sync_iter);
static void js_c_function_data_finalizer(JSRuntime *rt, JSValueConst val);
static void js_c_function_data_mark(JSRuntime *rt, JSValueConst val,
                                    JS_MarkFunc *mark_func);
static JSValue js_call_c_function_data(JSContext *ctx, JSValueConst func_obj,
                                       JSValueConst this_val,
                                       int argc, JSValueConst *argv, int flags);
static JSAtom js_symbol_to_atom(JSContext *ctx, JSValueConst val);
static void add_gc_object(JSRuntime *rt, JSGCObjectHeader *h,
                          JSGCObjectTypeEnum type);
static void remove_gc_object(JSGCObjectHeader *h);
static void js_async_function_free0(JSRuntime *rt, JSAsyncFunctionData *s);
static JSValue js_instantiate_prototype(JSContext *ctx, JSObject *p, JSAtom atom, void *opaque);
static JSValue js_module_ns_autoinit(JSContext *ctx, JSObject *p, JSAtom atom,
                                 void *opaque);
static JSValue JS_InstantiateFunctionListItem2(JSContext *ctx, JSObject *p,
                                               JSAtom atom, void *opaque);

static void js_set_uncatchable_error(JSContext *ctx, JSValueConst val,
                                     bool flag);

static JSValue js_new_callsite(JSContext *ctx, JSCallSiteData *csd);
static void js_new_callsite_data(JSContext *ctx, JSCallSiteData *csd, JSStackFrame *sf);
static void js_new_callsite_data2(JSContext *ctx, JSCallSiteData *csd, const char *filename, int line_num, int col_num);
static void _JS_AddIntrinsicCallSite(JSContext *ctx);

static void JS_SetOpaqueInternal(JSValueConst obj, void *opaque);

static const JSClassExoticMethods js_arguments_exotic_methods;
static const JSClassExoticMethods js_string_exotic_methods;
static const JSClassExoticMethods js_proxy_exotic_methods;
static const JSClassExoticMethods js_module_ns_exotic_methods;

static inline bool double_is_int32(double d)
{
    uint64_t u, e;
    JSFloat64Union t;

    t.d = d;
    u = t.u64;

    e = ((u >> 52) & 0x7FF) - 1023;
    if (e > 30) {
        // accept 0, INT32_MIN, reject too large, too small, nan, inf, -0
        return !u || (u == 0xc1e0000000000000);
    } else {
        // shift out sign, exponent and whole part bits
        // value is fractional if remaining low bits are non-zero
        return !(u << 12 << e);
    }
}

static JSValue js_float64(double d)
{
    return __JS_NewFloat64(d);
}

static int compare_u32(uint32_t a, uint32_t b)
{
    return -(a < b) + (b < a); // -1, 0 or 1
}

static JSValue js_int32(int32_t v)
{
    return JS_MKVAL(JS_TAG_INT, v);
}

static JSValue js_uint32(uint32_t v)
{
    if (v <= INT32_MAX)
        return js_int32(v);
    else
        return js_float64(v);
}

static JSValue js_int64(int64_t v)
{
    if (v >= INT32_MIN && v <= INT32_MAX)
        return js_int32(v);
    else
        return js_float64(v);
}

static JSValue js_number(double d)
{
    if (double_is_int32(d))
        return js_int32((int32_t)d);
    else
        return js_float64(d);
}

JSValue JS_NewNumber(JSContext *ctx, double d)
{
    return js_number(d);
}

static JSValue js_bool(bool v)
{
    return JS_MKVAL(JS_TAG_BOOL, (v != 0));
}

static JSValue js_dup(JSValueConst v)
{
    if (JS_VALUE_HAS_REF_COUNT(v)) {
        JSRefCountHeader *p = (JSRefCountHeader *)JS_VALUE_GET_PTR(v);
        p->ref_count++;
    }
    return unsafe_unconst(v);
}

JSValue JS_DupValue(JSContext *ctx, JSValueConst v)
{
    return js_dup(v);
}

JSValue JS_DupValueRT(JSRuntime *rt, JSValueConst v)
{
    return js_dup(v);
}

static void js_trigger_gc(JSRuntime *rt, size_t size)
{
    bool force_gc;
#ifdef FORCE_GC_AT_MALLOC
    force_gc = true;
#else
    force_gc = ((rt->malloc_state.malloc_size + size) >
                rt->malloc_gc_threshold);
#endif
    if (force_gc) {
#ifdef ENABLE_DUMPS // JS_DUMP_GC
        if (check_dump_flag(rt, JS_DUMP_GC)) {
            printf("GC: size=%zd\n", rt->malloc_state.malloc_size);
        }
#endif
        JS_RunGC(rt);
        rt->malloc_gc_threshold = rt->malloc_state.malloc_size +
            (rt->malloc_state.malloc_size >> 1);
    }
}

static size_t js_malloc_usable_size_unknown(const void *ptr)
{
    return 0;
}

void *js_calloc_rt(JSRuntime *rt, size_t count, size_t size)
{
    void *ptr;
    JSMallocState *s;

    /* Do not allocate zero bytes: behavior is platform dependent */
    assert(count != 0 && size != 0);

    if (size > 0)
        if (unlikely(count != (count * size) / size))
            return NULL;

    s = &rt->malloc_state;
    /* When malloc_limit is 0 (unlimited), malloc_limit - 1 will be SIZE_MAX. */
    if (unlikely(s->malloc_size + (count * size) > s->malloc_limit - 1))
        return NULL;

    ptr = rt->mf.js_calloc(s->opaque, count, size);
    if (!ptr)
        return NULL;

    s->malloc_count++;
    s->malloc_size += rt->mf.js_malloc_usable_size(ptr) + MALLOC_OVERHEAD;
    return ptr;
}

void *js_malloc_rt(JSRuntime *rt, size_t size)
{
    void *ptr;
    JSMallocState *s;

    /* Do not allocate zero bytes: behavior is platform dependent */
    assert(size != 0);

    s = &rt->malloc_state;
    /* When malloc_limit is 0 (unlimited), malloc_limit - 1 will be SIZE_MAX. */
    if (unlikely(s->malloc_size + size > s->malloc_limit - 1))
        return NULL;

    ptr = rt->mf.js_malloc(s->opaque, size);
    if (!ptr)
        return NULL;

    s->malloc_count++;
    s->malloc_size += rt->mf.js_malloc_usable_size(ptr) + MALLOC_OVERHEAD;
    return ptr;
}

void js_free_rt(JSRuntime *rt, void *ptr)
{
    JSMallocState *s;

    if (!ptr)
        return;

    s = &rt->malloc_state;
    s->malloc_count--;
    s->malloc_size -= rt->mf.js_malloc_usable_size(ptr) + MALLOC_OVERHEAD;
    rt->mf.js_free(s->opaque, ptr);
}

void *js_realloc_rt(JSRuntime *rt, void *ptr, size_t size)
{
    size_t old_size;
    JSMallocState *s;

    if (!ptr) {
        if (size == 0)
            return NULL;
        return js_malloc_rt(rt, size);
    }
    if (unlikely(size == 0)) {
        js_free_rt(rt, ptr);
        return NULL;
    }
    old_size = rt->mf.js_malloc_usable_size(ptr);
    s = &rt->malloc_state;
    /* When malloc_limit is 0 (unlimited), malloc_limit - 1 will be SIZE_MAX. */
    if (s->malloc_size + size - old_size > s->malloc_limit - 1)
        return NULL;

    ptr = rt->mf.js_realloc(s->opaque, ptr, size);
    if (!ptr)
        return NULL;

    s->malloc_size += rt->mf.js_malloc_usable_size(ptr) - old_size;
    return ptr;
}

size_t js_malloc_usable_size_rt(JSRuntime *rt, const void *ptr)
{
    return rt->mf.js_malloc_usable_size(ptr);
}

/**
 * This used to be implemented as malloc + memset, but using calloc
 * yields better performance in initial, bursty allocations, something useful
 * for QuickJS.
 *
 * More information: https://github.com/quickjs-ng/quickjs/pull/519
 */
void *js_mallocz_rt(JSRuntime *rt, size_t size)
{
    return js_calloc_rt(rt, 1, size);
}

/* Throw out of memory in case of error */
void *js_calloc(JSContext *ctx, size_t count, size_t size)
{
    void *ptr;
    ptr = js_calloc_rt(ctx->rt, count, size);
    if (unlikely(!ptr)) {
        JS_ThrowOutOfMemory(ctx);
        return NULL;
    }
    return ptr;
}

/* Throw out of memory in case of error */
void *js_malloc(JSContext *ctx, size_t size)
{
    void *ptr;
    ptr = js_malloc_rt(ctx->rt, size);
    if (unlikely(!ptr)) {
        JS_ThrowOutOfMemory(ctx);
        return NULL;
    }
    return ptr;
}

/* Throw out of memory in case of error */
void *js_mallocz(JSContext *ctx, size_t size)
{
    void *ptr;
    ptr = js_mallocz_rt(ctx->rt, size);
    if (unlikely(!ptr)) {
        JS_ThrowOutOfMemory(ctx);
        return NULL;
    }
    return ptr;
}

void js_free(JSContext *ctx, void *ptr)
{
    js_free_rt(ctx->rt, ptr);
}

/* Throw out of memory in case of error */
void *js_realloc(JSContext *ctx, void *ptr, size_t size)
{
    void *ret;
    ret = js_realloc_rt(ctx->rt, ptr, size);
    if (unlikely(!ret && size != 0)) {
        JS_ThrowOutOfMemory(ctx);
        return NULL;
    }
    return ret;
}

/* store extra allocated size in *pslack if successful */
void *js_realloc2(JSContext *ctx, void *ptr, size_t size, size_t *pslack)
{
    void *ret;
    ret = js_realloc_rt(ctx->rt, ptr, size);
    if (unlikely(!ret && size != 0)) {
        JS_ThrowOutOfMemory(ctx);
        return NULL;
    }
    if (pslack) {
        size_t new_size = js_malloc_usable_size_rt(ctx->rt, ret);
        *pslack = (new_size > size) ? new_size - size : 0;
    }
    return ret;
}

size_t js_malloc_usable_size(JSContext *ctx, const void *ptr)
{
    return js_malloc_usable_size_rt(ctx->rt, ptr);
}

/* Throw out of memory exception in case of error */
char *js_strndup(JSContext *ctx, const char *s, size_t n)
{
    char *ptr;
    ptr = js_malloc(ctx, n + 1);
    if (ptr) {
        memcpy(ptr, s, n);
        ptr[n] = '\0';
    }
    return ptr;
}

char *js_strdup(JSContext *ctx, const char *str)
{
    return js_strndup(ctx, str, strlen(str));
}

static no_inline int js_realloc_array(JSContext *ctx, void **parray,
                                      int elem_size, int *psize, int req_size)
{
    int new_size;
    size_t slack;
    void *new_array;
    /* XXX: potential arithmetic overflow */
    new_size = max_int(req_size, *psize * 3 / 2);
    new_array = js_realloc2(ctx, *parray, new_size * elem_size, &slack);
    if (!new_array)
        return -1;
    new_size += slack / elem_size;
    *psize = new_size;
    *parray = new_array;
    return 0;
}

/* resize the array and update its size if req_size > *psize */
static inline int js_resize_array(JSContext *ctx, void **parray, int elem_size,
                                  int *psize, int req_size)
{
    if (unlikely(req_size > *psize))
        return js_realloc_array(ctx, parray, elem_size, psize, req_size);
    else
        return 0;
}

static void *js_dbuf_realloc(void *ctx, void *ptr, size_t size)
{
    return js_realloc(ctx, ptr, size);
}

static inline void js_dbuf_init(JSContext *ctx, DynBuf *s)
{
    dbuf_init2(s, ctx, js_dbuf_realloc);
}

static inline int is_digit(int c) {
    return c >= '0' && c <= '9';
}

static inline int string_get(JSString *p, int idx) {
    return p->is_wide_char ? str16(p)[idx] : str8(p)[idx];
}

typedef struct JSClassShortDef {
    JSAtom class_name;
    JSClassFinalizer *finalizer;
    JSClassGCMark *gc_mark;
} JSClassShortDef;

static JSClassShortDef const js_std_class_def[] = {
    { JS_ATOM_Object, NULL, NULL },                             /* JS_CLASS_OBJECT */
    { JS_ATOM_Array, js_array_finalizer, js_array_mark },       /* JS_CLASS_ARRAY */
    { JS_ATOM_Error, NULL, NULL }, /* JS_CLASS_ERROR */
    { JS_ATOM_Number, js_object_data_finalizer, js_object_data_mark }, /* JS_CLASS_NUMBER */
    { JS_ATOM_String, js_object_data_finalizer, js_object_data_mark }, /* JS_CLASS_STRING */
    { JS_ATOM_Boolean, js_object_data_finalizer, js_object_data_mark }, /* JS_CLASS_BOOLEAN */
    { JS_ATOM_Symbol, js_object_data_finalizer, js_object_data_mark }, /* JS_CLASS_SYMBOL */
    { JS_ATOM_Arguments, js_array_finalizer, js_array_mark },   /* JS_CLASS_ARGUMENTS */
    { JS_ATOM_Arguments, NULL, NULL },                          /* JS_CLASS_MAPPED_ARGUMENTS */
    { JS_ATOM_Date, js_object_data_finalizer, js_object_data_mark }, /* JS_CLASS_DATE */
    { JS_ATOM_Object, NULL, NULL },                             /* JS_CLASS_MODULE_NS */
    { JS_ATOM_Function, js_c_function_finalizer, js_c_function_mark }, /* JS_CLASS_C_FUNCTION */
    { JS_ATOM_Function, js_bytecode_function_finalizer, js_bytecode_function_mark }, /* JS_CLASS_BYTECODE_FUNCTION */
    { JS_ATOM_Function, js_bound_function_finalizer, js_bound_function_mark }, /* JS_CLASS_BOUND_FUNCTION */
    { JS_ATOM_Function, js_c_function_data_finalizer, js_c_function_data_mark }, /* JS_CLASS_C_FUNCTION_DATA */
    { JS_ATOM_GeneratorFunction, js_bytecode_function_finalizer, js_bytecode_function_mark },  /* JS_CLASS_GENERATOR_FUNCTION */
    { JS_ATOM_ForInIterator, js_for_in_iterator_finalizer, js_for_in_iterator_mark },      /* JS_CLASS_FOR_IN_ITERATOR */
    { JS_ATOM_RegExp, js_regexp_finalizer, NULL },                              /* JS_CLASS_REGEXP */
    { JS_ATOM_ArrayBuffer, js_array_buffer_finalizer, NULL },                   /* JS_CLASS_ARRAY_BUFFER */
    { JS_ATOM_SharedArrayBuffer, js_array_buffer_finalizer, NULL },             /* JS_CLASS_SHARED_ARRAY_BUFFER */
    { JS_ATOM_Uint8ClampedArray, js_typed_array_finalizer, js_typed_array_mark }, /* JS_CLASS_UINT8C_ARRAY */
    { JS_ATOM_Int8Array, js_typed_array_finalizer, js_typed_array_mark },       /* JS_CLASS_INT8_ARRAY */
    { JS_ATOM_Uint8Array, js_typed_array_finalizer, js_typed_array_mark },      /* JS_CLASS_UINT8_ARRAY */
    { JS_ATOM_Int16Array, js_typed_array_finalizer, js_typed_array_mark },      /* JS_CLASS_INT16_ARRAY */
    { JS_ATOM_Uint16Array, js_typed_array_finalizer, js_typed_array_mark },     /* JS_CLASS_UINT16_ARRAY */
    { JS_ATOM_Int32Array, js_typed_array_finalizer, js_typed_array_mark },      /* JS_CLASS_INT32_ARRAY */
    { JS_ATOM_Uint32Array, js_typed_array_finalizer, js_typed_array_mark },     /* JS_CLASS_UINT32_ARRAY */
    { JS_ATOM_BigInt64Array, js_typed_array_finalizer, js_typed_array_mark },   /* JS_CLASS_BIG_INT64_ARRAY */
    { JS_ATOM_BigUint64Array, js_typed_array_finalizer, js_typed_array_mark },  /* JS_CLASS_BIG_UINT64_ARRAY */
    { JS_ATOM_Float16Array, js_typed_array_finalizer, js_typed_array_mark },    /* JS_CLASS_FLOAT16_ARRAY */
    { JS_ATOM_Float32Array, js_typed_array_finalizer, js_typed_array_mark },    /* JS_CLASS_FLOAT32_ARRAY */
    { JS_ATOM_Float64Array, js_typed_array_finalizer, js_typed_array_mark },    /* JS_CLASS_FLOAT64_ARRAY */
    { JS_ATOM_DataView, js_typed_array_finalizer, js_typed_array_mark },        /* JS_CLASS_DATAVIEW */
    { JS_ATOM_BigInt, js_object_data_finalizer, js_object_data_mark },      /* JS_CLASS_BIG_INT */
    { JS_ATOM_Map, js_map_finalizer, js_map_mark },             /* JS_CLASS_MAP */
    { JS_ATOM_Set, js_map_finalizer, js_map_mark },             /* JS_CLASS_SET */
    { JS_ATOM_WeakMap, js_map_finalizer, NULL },         /* JS_CLASS_WEAKMAP */
    { JS_ATOM_WeakSet, js_map_finalizer, NULL },         /* JS_CLASS_WEAKSET */
    { JS_ATOM_Iterator, NULL, NULL },                           /* JS_CLASS_ITERATOR */
    { JS_ATOM_IteratorConcat, js_iterator_concat_finalizer, js_iterator_concat_mark }, /* JS_CLASS_ITERATOR_CONCAT */
    { JS_ATOM_IteratorHelper, js_iterator_helper_finalizer, js_iterator_helper_mark }, /* JS_CLASS_ITERATOR_HELPER */
    { JS_ATOM_IteratorWrap, js_iterator_wrap_finalizer, js_iterator_wrap_mark }, /* JS_CLASS_ITERATOR_WRAP */
    { JS_ATOM_Map_Iterator, js_map_iterator_finalizer, js_map_iterator_mark }, /* JS_CLASS_MAP_ITERATOR */
    { JS_ATOM_Set_Iterator, js_map_iterator_finalizer, js_map_iterator_mark }, /* JS_CLASS_SET_ITERATOR */
    { JS_ATOM_Array_Iterator, js_array_iterator_finalizer, js_array_iterator_mark }, /* JS_CLASS_ARRAY_ITERATOR */
    { JS_ATOM_String_Iterator, js_array_iterator_finalizer, js_array_iterator_mark }, /* JS_CLASS_STRING_ITERATOR */
    { JS_ATOM_RegExp_String_Iterator, js_regexp_string_iterator_finalizer, js_regexp_string_iterator_mark }, /* JS_CLASS_REGEXP_STRING_ITERATOR */
    { JS_ATOM_Generator, js_generator_finalizer, js_generator_mark }, /* JS_CLASS_GENERATOR */
};

static int init_class_range(JSRuntime *rt, JSClassShortDef const *tab,
                            int start, int count)
{
    JSClassDef cm_s, *cm = &cm_s;
    int i, class_id;

    for(i = 0; i < count; i++) {
        class_id = i + start;
        memset(cm, 0, sizeof(*cm));
        cm->finalizer = tab[i].finalizer;
        cm->gc_mark = tab[i].gc_mark;
        if (JS_NewClass1(rt, class_id, cm, tab[i].class_name) < 0)
            return -1;
    }
    return 0;
}

/* Uses code from LLVM project. */
static inline uintptr_t js_get_stack_pointer(void)
{
#if defined(__clang__) || defined(__GNUC__)
    return (uintptr_t)__builtin_frame_address(0);
#elif defined(_MSC_VER)
    return (uintptr_t)_AddressOfReturnAddress();
#else
    char CharOnStack = 0;
    // The volatile store here is intended to escape the local variable, to
    // prevent the compiler from optimizing CharOnStack into anything other
    // than a char on the stack.
    //
    // Tested on: MSVC 2015 - 2019, GCC 4.9 - 9, Clang 3.2 - 9, ICC 13 - 19.
    char *volatile Ptr = &CharOnStack;
    return (uintptr_t) Ptr;
#endif
}

static inline bool js_check_stack_overflow(JSRuntime *rt, size_t alloca_size)
{
    uintptr_t sp;
    sp = js_get_stack_pointer() - alloca_size;
    return unlikely(sp < rt->stack_limit);
}

JSRuntime *JS_NewRuntime2(const JSMallocFunctions *mf, void *opaque)
{
    JSRuntime *rt;
    JSMallocState ms;

    memset(&ms, 0, sizeof(ms));
    ms.opaque = opaque;
    ms.malloc_limit = 0;

    rt = mf->js_calloc(opaque, 1, sizeof(JSRuntime));
    if (!rt)
        return NULL;
    rt->mf = *mf;
    if (!rt->mf.js_malloc_usable_size) {
        /* use dummy function if none provided */
        rt->mf.js_malloc_usable_size = js_malloc_usable_size_unknown;
    }
    /* Inline what js_malloc_rt does since we cannot use it here. */
    ms.malloc_count++;
    ms.malloc_size += rt->mf.js_malloc_usable_size(rt) + MALLOC_OVERHEAD;
    rt->malloc_state = ms;
    rt->malloc_gc_threshold = 256 * 1024;

    init_list_head(&rt->context_list);
    init_list_head(&rt->gc_obj_list);
    init_list_head(&rt->gc_zero_ref_count_list);
    rt->gc_phase = JS_GC_PHASE_NONE;

#ifdef ENABLE_DUMPS // JS_DUMP_LEAKS
    init_list_head(&rt->string_list);
#endif
    init_list_head(&rt->job_list);

    if (JS_InitAtoms(rt))
        goto fail;

    /* create the object, array and function classes */
    if (init_class_range(rt, js_std_class_def, JS_CLASS_OBJECT,
                         countof(js_std_class_def)) < 0)
        goto fail;
    rt->class_array[JS_CLASS_ARGUMENTS].exotic = &js_arguments_exotic_methods;
    rt->class_array[JS_CLASS_STRING].exotic = &js_string_exotic_methods;
    rt->class_array[JS_CLASS_MODULE_NS].exotic = &js_module_ns_exotic_methods;

    rt->class_array[JS_CLASS_C_FUNCTION].call = js_call_c_function;
    rt->class_array[JS_CLASS_C_FUNCTION_DATA].call = js_call_c_function_data;
    rt->class_array[JS_CLASS_BOUND_FUNCTION].call = js_call_bound_function;
    rt->class_array[JS_CLASS_GENERATOR_FUNCTION].call = js_call_generator_function;
    if (init_shape_hash(rt))
        goto fail;

    rt->js_class_id_alloc = JS_CLASS_INIT_COUNT;

    rt->stack_size = JS_DEFAULT_STACK_SIZE;
#ifdef __wasi__
    rt->stack_size = 0;
#endif

    JS_UpdateStackTop(rt);

    rt->current_exception = JS_UNINITIALIZED;

    return rt;
 fail:
    JS_FreeRuntime(rt);
    return NULL;
}

void *JS_GetRuntimeOpaque(JSRuntime *rt)
{
    return rt->user_opaque;
}

void JS_SetRuntimeOpaque(JSRuntime *rt, void *opaque)
{
    rt->user_opaque = opaque;
}

int JS_AddRuntimeFinalizer(JSRuntime *rt, JSRuntimeFinalizer *finalizer,
                           void *arg)
{
    JSRuntimeFinalizerState *fs = js_malloc_rt(rt, sizeof(*fs));
    if (!fs)
        return -1;
    fs->next       = rt->finalizers;
    fs->finalizer  = finalizer;
    fs->arg        = arg;
    rt->finalizers = fs;
    return 0;
}

static void *js_def_calloc(void *opaque, size_t count, size_t size)
{
    return calloc(count, size);
}

static void *js_def_malloc(void *opaque, size_t size)
{
    return malloc(size);
}

static void js_def_free(void *opaque, void *ptr)
{
    free(ptr);
}

static void *js_def_realloc(void *opaque, void *ptr, size_t size)
{
    return realloc(ptr, size);
}

static const JSMallocFunctions def_malloc_funcs = {
    js_def_calloc,
    js_def_malloc,
    js_def_free,
    js_def_realloc,
    js__malloc_usable_size
};

JSRuntime *JS_NewRuntime(void)
{
    return JS_NewRuntime2(&def_malloc_funcs, NULL);
}

void JS_SetMemoryLimit(JSRuntime *rt, size_t limit)
{
    rt->malloc_state.malloc_limit = limit;
}

void JS_SetDumpFlags(JSRuntime *rt, uint64_t flags)
{
#ifdef ENABLE_DUMPS
    rt->dump_flags = flags;
#endif
}

uint64_t JS_GetDumpFlags(JSRuntime *rt)
{
#ifdef ENABLE_DUMPS
    return rt->dump_flags;
#else
    return 0;
#endif
}

size_t JS_GetGCThreshold(JSRuntime *rt) {
    return rt->malloc_gc_threshold;
}

/* use -1 to disable automatic GC */
void JS_SetGCThreshold(JSRuntime *rt, size_t gc_threshold)
{
    rt->malloc_gc_threshold = gc_threshold;
}

#define malloc(s) malloc_is_forbidden(s)
#define free(p) free_is_forbidden(p)
#define realloc(p,s) realloc_is_forbidden(p,s)

void JS_SetInterruptHandler(JSRuntime *rt, JSInterruptHandler *cb, void *opaque)
{
    rt->interrupt_handler = cb;
    rt->interrupt_opaque = opaque;
}

void JS_SetCanBlock(JSRuntime *rt, bool can_block)
{
    rt->can_block = can_block;
}

void JS_SetSharedArrayBufferFunctions(JSRuntime *rt,
                                      const JSSharedArrayBufferFunctions *sf)
{
    rt->sab_funcs = *sf;
}

/* return 0 if OK, < 0 if exception */
int JS_EnqueueJob(JSContext *ctx, JSJobFunc *job_func,
                  int argc, JSValueConst *argv)
{
    JSRuntime *rt = ctx->rt;
    JSJobEntry *e;
    int i;

    assert(!rt->in_free);

    e = js_malloc(ctx, sizeof(*e) + argc * sizeof(JSValue));
    if (!e)
        return -1;
    e->ctx = ctx;
    e->job_func = job_func;
    e->argc = argc;
    for(i = 0; i < argc; i++) {
        e->argv[i] = js_dup(argv[i]);
    }
    list_add_tail(&e->link, &rt->job_list);
    return 0;
}

bool JS_IsJobPending(JSRuntime *rt)
{
    return !list_empty(&rt->job_list);
}

/* return < 0 if exception, 0 if no job pending, 1 if a job was
   executed successfully. the context of the job is stored in '*pctx' */
int JS_ExecutePendingJob(JSRuntime *rt, JSContext **pctx)
{
    JSContext *ctx;
    JSJobEntry *e;
    JSValue res;
    int i, ret;

    if (list_empty(&rt->job_list)) {
        *pctx = NULL;
        return 0;
    }

    /* get the first pending job and execute it */
    e = list_entry(rt->job_list.next, JSJobEntry, link);
    list_del(&e->link);
    ctx = e->ctx;
    res = e->job_func(e->ctx, e->argc, vc(e->argv));
    for(i = 0; i < e->argc; i++)
        JS_FreeValue(ctx, e->argv[i]);
    if (JS_IsException(res))
        ret = -1;
    else
        ret = 1;
    JS_FreeValue(ctx, res);
    js_free(ctx, e);
    *pctx = ctx;
    return ret;
}

static inline uint32_t atom_get_free(const JSAtomStruct *p)
{
    return (uintptr_t)p >> 1;
}

static inline bool atom_is_free(const JSAtomStruct *p)
{
    return (uintptr_t)p & 1;
}

static inline JSAtomStruct *atom_set_free(uint32_t v)
{
    return (JSAtomStruct *)(((uintptr_t)v << 1) | 1);
}

/* Note: the string contents are uninitialized */
static JSString *js_alloc_string_rt(JSRuntime *rt, int max_len, int is_wide_char)
{
    JSString *str;
    str = js_malloc_rt(rt, sizeof(JSString) + (max_len << is_wide_char) + 1 - is_wide_char);
    if (unlikely(!str))
        return NULL;
    str->header.ref_count = 1;
    str->is_wide_char = is_wide_char;
    str->len = max_len;
    str->kind = JS_STRING_KIND_NORMAL;
    str->atom_type = 0;
    str->hash = 0;          /* optional but costless */
    str->hash_next = 0;     /* optional */
#ifdef ENABLE_DUMPS // JS_DUMP_LEAKS
    list_add_tail(&str->link, &rt->string_list);
#endif
    return str;
}

static JSString *js_alloc_string(JSContext *ctx, int max_len, int is_wide_char)
{
    JSString *p;
    p = js_alloc_string_rt(ctx->rt, max_len, is_wide_char);
    if (unlikely(!p)) {
        JS_ThrowOutOfMemory(ctx);
        return NULL;
    }
    return p;
}

static inline void js_free_string0(JSRuntime *rt, JSString *str);

/* same as JS_FreeValueRT() but faster */
static inline void js_free_string(JSRuntime *rt, JSString *str)
{
    if (--str->header.ref_count <= 0)
        js_free_string0(rt, str);
}

static inline void js_free_string0(JSRuntime *rt, JSString *str)
{
    if (str->atom_type) {
        JS_FreeAtomStruct(rt, str);
    } else {
#ifdef ENABLE_DUMPS // JS_DUMP_LEAKS
        list_del(&str->link);
#endif
        if (str->kind == JS_STRING_KIND_SLICE) {
            JSStringSlice *slice = (void *)&str[1];
            js_free_string(rt, slice->parent); // safe, recurses only 1 level
        }
        js_free_rt(rt, str);
    }
}

void JS_SetRuntimeInfo(JSRuntime *rt, const char *s)
{
    if (rt)
        rt->rt_info = s;
}

void JS_FreeRuntime(JSRuntime *rt)
{
    struct list_head *el, *el1;
    int i;

    rt->in_free = true;
    JS_FreeValueRT(rt, rt->current_exception);

    list_for_each_safe(el, el1, &rt->job_list) {
        JSJobEntry *e = list_entry(el, JSJobEntry, link);
        for(i = 0; i < e->argc; i++)
            JS_FreeValueRT(rt, e->argv[i]);
        js_free_rt(rt, e);
    }
    init_list_head(&rt->job_list);

    JS_RunGC(rt);

#ifdef ENABLE_DUMPS // JS_DUMP_LEAKS
    /* leaking objects */
    if (check_dump_flag(rt, JS_DUMP_LEAKS)) {
        bool header_done;
        JSGCObjectHeader *p;
        int count;

        /* remove the internal refcounts to display only the object
           referenced externally */
        list_for_each(el, &rt->gc_obj_list) {
            p = list_entry(el, JSGCObjectHeader, link);
            p->mark = 0;
        }
        gc_decref(rt);

        header_done = false;
        list_for_each(el, &rt->gc_obj_list) {
            p = list_entry(el, JSGCObjectHeader, link);
            if (p->ref_count != 0) {
                if (!header_done) {
                    printf("Object leaks:\n");
                    JS_DumpObjectHeader(rt);
                    header_done = true;
                }
                JS_DumpGCObject(rt, p);
            }
        }

        count = 0;
        list_for_each(el, &rt->gc_obj_list) {
            p = list_entry(el, JSGCObjectHeader, link);
            if (p->ref_count == 0) {
                count++;
            }
        }
        if (count != 0)
            printf("Secondary object leaks: %d\n", count);
    }
#endif

    assert(list_empty(&rt->gc_obj_list));

    /* free the classes */
    for(i = 0; i < rt->class_count; i++) {
        JSClass *cl = &rt->class_array[i];
        if (cl->class_id != 0) {
            JS_FreeAtomRT(rt, cl->class_name);
        }
    }
    js_free_rt(rt, rt->class_array);

#ifdef ENABLE_DUMPS // JS_DUMP_ATOM_LEAKS
    /* only the atoms defined in JS_InitAtoms() should be left */
    if (check_dump_flag(rt, JS_DUMP_ATOM_LEAKS)) {
        bool header_done = false;

        for(i = 0; i < rt->atom_size; i++) {
            JSAtomStruct *p = rt->atom_array[i];
            if (!atom_is_free(p) /* && p->str*/) {
                if (i >= JS_ATOM_END || p->header.ref_count != 1) {
                    if (!header_done) {
                        header_done = true;
                        if (rt->rt_info) {
                            printf("%s:1: atom leakage:", rt->rt_info);
                        } else {
                            printf("Atom leaks:\n"
                                   "    %6s %6s %s\n",
                                   "ID", "REFCNT", "NAME");
                        }
                    }
                    if (rt->rt_info) {
                        printf(" ");
                    } else {
                        printf("    %6u %6u ", i, p->header.ref_count);
                    }
                    switch (p->atom_type) {
                    case JS_ATOM_TYPE_STRING:
                        JS_DumpString(rt, p);
                        break;
                    case JS_ATOM_TYPE_GLOBAL_SYMBOL:
                        printf("Symbol.for(");
                        JS_DumpString(rt, p);
                        printf(")");
                        break;
                    case JS_ATOM_TYPE_SYMBOL:
                        if (p->hash == JS_ATOM_HASH_SYMBOL) {
                            printf("Symbol(");
                            JS_DumpString(rt, p);
                            printf(")");
                        } else {
                            printf("Private(");
                            JS_DumpString(rt, p);
                            printf(")");
                        }
                        break;
                    }
                    if (rt->rt_info) {
                        printf(":%u", p->header.ref_count);
                    } else {
                        printf("\n");
                    }
                }
            }
        }
        if (rt->rt_info && header_done)
            printf("\n");
    }
#endif

    /* free the atoms */
    for(i = 0; i < rt->atom_size; i++) {
        JSAtomStruct *p = rt->atom_array[i];
        if (!atom_is_free(p)) {
#ifdef ENABLE_DUMPS // JS_DUMP_LEAKS
            list_del(&p->link);
#endif
            js_free_rt(rt, p);
        }
    }
    js_free_rt(rt, rt->atom_array);
    js_free_rt(rt, rt->atom_hash);
    js_free_rt(rt, rt->shape_hash);
#ifdef ENABLE_DUMPS // JS_DUMP_LEAKS
    if (check_dump_flag(rt, JS_DUMP_LEAKS) && !list_empty(&rt->string_list)) {
        if (rt->rt_info) {
            printf("%s:1: string leakage:", rt->rt_info);
        } else {
            printf("String leaks:\n"
                   "    %6s %s\n",
                   "REFCNT", "VALUE");
        }
        list_for_each_safe(el, el1, &rt->string_list) {
            JSString *str = list_entry(el, JSString, link);
            if (rt->rt_info) {
                printf(" ");
            } else {
                printf("    %6u ", str->header.ref_count);
            }
            JS_DumpString(rt, str);
            if (rt->rt_info) {
                printf(":%u", str->header.ref_count);
            } else {
                printf("\n");
            }
            list_del(&str->link);
            js_free_rt(rt, str);
        }
        if (rt->rt_info)
            printf("\n");
    }
#endif

    while (rt->finalizers) {
        JSRuntimeFinalizerState *fs = rt->finalizers;
        rt->finalizers = fs->next;
        fs->finalizer(rt, fs->arg);
        js_free_rt(rt, fs);
    }

#ifdef ENABLE_DUMPS // JS_DUMP_LEAKS
    if (check_dump_flag(rt, JS_DUMP_LEAKS)) {
        JSMallocState *s = &rt->malloc_state;
        if (s->malloc_count > 1) {
            if (rt->rt_info)
                printf("%s:1: ", rt->rt_info);
            printf("Memory leak: %zd bytes lost in %zd block%s\n",
                   s->malloc_size - sizeof(JSRuntime),
                   s->malloc_count - 1, &"s"[s->malloc_count == 2]);
        }
    }
#endif

    {
        JSMallocState *ms = &rt->malloc_state;
        rt->mf.js_free(ms->opaque, rt);
    }
}

JSContext *JS_NewContextRaw(JSRuntime *rt)
{
    JSContext *ctx;
    int i;

    ctx = js_mallocz_rt(rt, sizeof(JSContext));
    if (!ctx)
        return NULL;
    ctx->header.ref_count = 1;
    add_gc_object(rt, &ctx->header, JS_GC_OBJ_TYPE_JS_CONTEXT);

    ctx->class_proto = js_malloc_rt(rt, sizeof(ctx->class_proto[0]) *
                                    rt->class_count);
    if (!ctx->class_proto) {
        js_free_rt(rt, ctx);
        return NULL;
    }
    ctx->rt = rt;
    list_add_tail(&ctx->link, &rt->context_list);
    for(i = 0; i < rt->class_count; i++)
        ctx->class_proto[i] = JS_NULL;
    ctx->array_ctor = JS_NULL;
    ctx->iterator_ctor = JS_NULL;
    ctx->iterator_ctor_getset = JS_NULL;
    ctx->regexp_ctor = JS_NULL;
    ctx->promise_ctor = JS_NULL;
    ctx->error_ctor = JS_NULL;
    ctx->error_back_trace = JS_UNDEFINED;
    ctx->error_prepare_stack = JS_UNDEFINED;
    ctx->error_stack_trace_limit = js_int32(10);
    init_list_head(&ctx->loaded_modules);

    JS_AddIntrinsicBasicObjects(ctx);
    return ctx;
}

JSContext *JS_NewContext(JSRuntime *rt)
{
    JSContext *ctx;

    ctx = JS_NewContextRaw(rt);
    if (!ctx)
        return NULL;

    JS_AddIntrinsicBaseObjects(ctx);
    JS_AddIntrinsicDate(ctx);
    JS_AddIntrinsicEval(ctx);
    JS_AddIntrinsicRegExp(ctx);
    JS_AddIntrinsicJSON(ctx);
    JS_AddIntrinsicProxy(ctx);
    JS_AddIntrinsicMapSet(ctx);
    JS_AddIntrinsicTypedArrays(ctx);
    JS_AddIntrinsicPromise(ctx);
    JS_AddIntrinsicBigInt(ctx);
    JS_AddIntrinsicWeakRef(ctx);
    JS_AddIntrinsicDOMException(ctx);

    JS_AddPerformance(ctx);

    return ctx;
}

void *JS_GetContextOpaque(JSContext *ctx)
{
    return ctx->user_opaque;
}

void JS_SetContextOpaque(JSContext *ctx, void *opaque)
{
    ctx->user_opaque = opaque;
}

/* set the new value and free the old value after (freeing the value
   can reallocate the object data) */
static inline void set_value(JSContext *ctx, JSValue *pval, JSValue new_val)
{
    JSValue old_val;
    old_val = *pval;
    *pval = new_val;
    JS_FreeValue(ctx, old_val);
}

void JS_SetClassProto(JSContext *ctx, JSClassID class_id, JSValue obj)
{
    assert(class_id < ctx->rt->class_count);
    set_value(ctx, &ctx->class_proto[class_id], obj);
}

JSValue JS_GetClassProto(JSContext *ctx, JSClassID class_id)
{
    assert(class_id < ctx->rt->class_count);
    return js_dup(ctx->class_proto[class_id]);
}

JSValue JS_GetFunctionProto(JSContext *ctx)
{
    return js_dup(ctx->function_proto);
}

typedef enum JSFreeModuleEnum {
    JS_FREE_MODULE_ALL,
    JS_FREE_MODULE_NOT_RESOLVED,
} JSFreeModuleEnum;

/* XXX: would be more efficient with separate module lists */
static void js_free_modules(JSContext *ctx, JSFreeModuleEnum flag)
{
    struct list_head *el, *el1;
    list_for_each_safe(el, el1, &ctx->loaded_modules) {
        JSModuleDef *m = list_entry(el, JSModuleDef, link);
        if (flag == JS_FREE_MODULE_ALL ||
            (flag == JS_FREE_MODULE_NOT_RESOLVED && !m->resolved)) {
            js_free_module_def(ctx, m);
        }
    }
}

JSContext *JS_DupContext(JSContext *ctx)
{
    ctx->header.ref_count++;
    return ctx;
}

/* used by the GC */
static void JS_MarkContext(JSRuntime *rt, JSContext *ctx,
                           JS_MarkFunc *mark_func)
{
    int i;
    struct list_head *el;

    /* modules are not seen by the GC, so we directly mark the objects
       referenced by each module */
    list_for_each(el, &ctx->loaded_modules) {
        JSModuleDef *m = list_entry(el, JSModuleDef, link);
        js_mark_module_def(rt, m, mark_func);
    }

    JS_MarkValue(rt, ctx->global_obj, mark_func);
    JS_MarkValue(rt, ctx->global_var_obj, mark_func);

    JS_MarkValue(rt, ctx->throw_type_error, mark_func);
    JS_MarkValue(rt, ctx->eval_obj, mark_func);

    JS_MarkValue(rt, ctx->array_proto_values, mark_func);
    for(i = 0; i < JS_NATIVE_ERROR_COUNT; i++) {
        JS_MarkValue(rt, ctx->native_error_proto[i], mark_func);
    }
    JS_MarkValue(rt, ctx->error_ctor, mark_func);
    JS_MarkValue(rt, ctx->error_back_trace, mark_func);
    JS_MarkValue(rt, ctx->error_prepare_stack, mark_func);
    JS_MarkValue(rt, ctx->error_stack_trace_limit, mark_func);
    for(i = 0; i < rt->class_count; i++) {
        JS_MarkValue(rt, ctx->class_proto[i], mark_func);
    }
    JS_MarkValue(rt, ctx->iterator_ctor, mark_func);
    JS_MarkValue(rt, ctx->iterator_ctor_getset, mark_func);
    JS_MarkValue(rt, ctx->async_iterator_proto, mark_func);
    JS_MarkValue(rt, ctx->promise_ctor, mark_func);
    JS_MarkValue(rt, ctx->array_ctor, mark_func);
    JS_MarkValue(rt, ctx->regexp_ctor, mark_func);
    JS_MarkValue(rt, ctx->function_ctor, mark_func);
    JS_MarkValue(rt, ctx->function_proto, mark_func);

    if (ctx->array_shape)
        mark_func(rt, &ctx->array_shape->header);
}

void JS_FreeContext(JSContext *ctx)
{
    JSRuntime *rt = ctx->rt;
    int i;

    if (--ctx->header.ref_count > 0)
        return;
    assert(ctx->header.ref_count == 0);

#ifdef ENABLE_DUMPS // JS_DUMP_ATOMS
    if (check_dump_flag(rt, JS_DUMP_ATOMS))
        JS_DumpAtoms(ctx->rt);
#endif
#ifdef ENABLE_DUMPS // JS_DUMP_SHAPES
    if (check_dump_flag(rt, JS_DUMP_SHAPES))
        JS_DumpShapes(ctx->rt);
#endif
#ifdef ENABLE_DUMPS // JS_DUMP_OBJECTS
    if (check_dump_flag(rt, JS_DUMP_OBJECTS)) {
        struct list_head *el;
        JSGCObjectHeader *p;
        printf("JSObjects: {\n");
        JS_DumpObjectHeader(ctx->rt);
        list_for_each(el, &rt->gc_obj_list) {
            p = list_entry(el, JSGCObjectHeader, link);
            JS_DumpGCObject(rt, p);
        }
        printf("}\n");
    }
#endif
#ifdef ENABLE_DUMPS // JS_DUMP_MEM
    if (check_dump_flag(rt, JS_DUMP_MEM)) {
        JSMemoryUsage stats;
        JS_ComputeMemoryUsage(rt, &stats);
        JS_DumpMemoryUsage(stdout, &stats, rt);
    }
#endif

    js_free_modules(ctx, JS_FREE_MODULE_ALL);

    JS_FreeValue(ctx, ctx->global_obj);
    JS_FreeValue(ctx, ctx->global_var_obj);

    JS_FreeValue(ctx, ctx->throw_type_error);
    JS_FreeValue(ctx, ctx->eval_obj);

    JS_FreeValue(ctx, ctx->array_proto_values);
    for(i = 0; i < JS_NATIVE_ERROR_COUNT; i++) {
        JS_FreeValue(ctx, ctx->native_error_proto[i]);
    }
    JS_FreeValue(ctx, ctx->error_ctor);
    JS_FreeValue(ctx, ctx->error_back_trace);
    JS_FreeValue(ctx, ctx->error_prepare_stack);
    JS_FreeValue(ctx, ctx->error_stack_trace_limit);
    for(i = 0; i < rt->class_count; i++) {
        JS_FreeValue(ctx, ctx->class_proto[i]);
    }
    js_free_rt(rt, ctx->class_proto);
    JS_FreeValue(ctx, ctx->iterator_ctor);
    JS_FreeValue(ctx, ctx->iterator_ctor_getset);
    JS_FreeValue(ctx, ctx->async_iterator_proto);
    JS_FreeValue(ctx, ctx->promise_ctor);
    JS_FreeValue(ctx, ctx->array_ctor);
    JS_FreeValue(ctx, ctx->regexp_ctor);
    JS_FreeValue(ctx, ctx->function_ctor);
    JS_FreeValue(ctx, ctx->function_proto);

    js_free_shape_null(ctx->rt, ctx->array_shape);

    list_del(&ctx->link);
    remove_gc_object(&ctx->header);
    js_free_rt(ctx->rt, ctx);
}

JSRuntime *JS_GetRuntime(JSContext *ctx)
{
    return ctx->rt;
}

static void update_stack_limit(JSRuntime *rt)
{
#if defined(__wasi__)
    rt->stack_limit = 0; /* no limit */
#else
    if (rt->stack_size == 0) {
        rt->stack_limit = 0; /* no limit */
    } else {
        rt->stack_limit = rt->stack_top - rt->stack_size;
    }
#endif
}

void JS_SetMaxStackSize(JSRuntime *rt, size_t stack_size)
{
    rt->stack_size = stack_size;
    update_stack_limit(rt);
}

void JS_UpdateStackTop(JSRuntime *rt)
{
    rt->stack_top = js_get_stack_pointer();
    update_stack_limit(rt);
}

static inline bool is_strict_mode(JSContext *ctx)
{
    JSStackFrame *sf = ctx->rt->current_stack_frame;
    return sf && sf->is_strict_mode;
}

/* JSAtom support */

#define JS_ATOM_TAG_INT (1U << 31)
#define JS_ATOM_MAX_INT (JS_ATOM_TAG_INT - 1)
#define JS_ATOM_MAX     ((1U << 30) - 1)

/* return the max count from the hash size */
#define JS_ATOM_COUNT_RESIZE(n) ((n) * 2)

static inline bool __JS_AtomIsConst(JSAtom v)
{
    return (int32_t)v < JS_ATOM_END;
}

static inline bool __JS_AtomIsTaggedInt(JSAtom v)
{
    return (v & JS_ATOM_TAG_INT) != 0;
}

static inline JSAtom __JS_AtomFromUInt32(uint32_t v)
{
    return v | JS_ATOM_TAG_INT;
}

static inline uint32_t __JS_AtomToUInt32(JSAtom atom)
{
    return atom & ~JS_ATOM_TAG_INT;
}

static inline int is_num(int c)
{
    return c >= '0' && c <= '9';
}

/* return true if the string is a number n with 0 <= n <= 2^32-1 */
static inline bool is_num_string(uint32_t *pval, JSString *p)
{
    uint32_t n;
    uint64_t n64;
    int c, i, len;

    len = p->len;
    if (len == 0 || len > 10)
        return false;
    c = string_get(p, 0);
    if (is_num(c)) {
        if (c == '0') {
            if (len != 1)
                return false;
            n = 0;
        } else {
            n = c - '0';
            for(i = 1; i < len; i++) {
                c = string_get(p, i);
                if (!is_num(c))
                    return false;
                n64 = (uint64_t)n * 10 + (c - '0');
                if ((n64 >> 32) != 0)
                    return false;
                n = n64;
            }
        }
        *pval = n;
        return true;
    } else {
        return false;
    }
}

/* XXX: could use faster version ? */
static inline uint32_t hash_string8(const uint8_t *str, size_t len, uint32_t h)
{
    size_t i;

    for(i = 0; i < len; i++)
        h = h * 263 + str[i];
    return h;
}

static inline uint32_t hash_string16(const uint16_t *str,
                                     size_t len, uint32_t h)
{
    size_t i;

    for(i = 0; i < len; i++)
        h = h * 263 + str[i];
    return h;
}

static uint32_t hash_string(JSString *str, uint32_t h)
{
    if (str->is_wide_char)
        h = hash_string16(str16(str), str->len, h);
    else
        h = hash_string8(str8(str), str->len, h);
    return h;
}

static __maybe_unused void JS_DumpString(JSRuntime *rt, JSString *p)
{
    int i, c, sep;

    if (p == NULL) {
        printf("<null>");
        return;
    }
    if (p->header.ref_count != 1)
        printf("%d", p->header.ref_count);
    if (p->is_wide_char)
        putchar('L');
    sep = '\"';
    putchar(sep);
    for(i = 0; i < p->len; i++) {
        c = string_get(p, i);
        if (c == sep || c == '\\') {
            putchar('\\');
            putchar(c);
        } else if (c >= ' ' && c <= 126) {
            putchar(c);
        } else if (c == '\n') {
            putchar('\\');
            putchar('n');
        } else {
            printf("\\u%04x", c);
        }
    }
    putchar(sep);
}

static __maybe_unused void JS_DumpAtoms(JSRuntime *rt)
{
    JSAtomStruct *p;
    int h, i;
    /* This only dumps hashed atoms, not JS_ATOM_TYPE_SYMBOL atoms */
    printf("JSAtom count=%d size=%d hash_size=%d:\n",
           rt->atom_count, rt->atom_size, rt->atom_hash_size);
    printf("JSAtom hash table: {\n");
    for(i = 0; i < rt->atom_hash_size; i++) {
        h = rt->atom_hash[i];
        if (h) {
            printf("  %d:", i);
            while (h) {
                p = rt->atom_array[h];
                printf(" ");
                JS_DumpString(rt, p);
                h = p->hash_next;
            }
            printf("\n");
        }
    }
    printf("}\n");
    printf("JSAtom table: {\n");
    for(i = 0; i < rt->atom_size; i++) {
        p = rt->atom_array[i];
        if (!atom_is_free(p)) {
            printf("  %d: { %d %08x ", i, p->atom_type, p->hash);
            if (!(p->len == 0 && p->is_wide_char != 0))
                JS_DumpString(rt, p);
            printf(" %d }\n", p->hash_next);
        }
    }
    printf("}\n");
}

static int JS_ResizeAtomHash(JSRuntime *rt, int new_hash_size)
{
    JSAtomStruct *p;
    uint32_t new_hash_mask, h, i, hash_next1, j, *new_hash;

    assert((new_hash_size & (new_hash_size - 1)) == 0); /* power of two */
    new_hash_mask = new_hash_size - 1;
    new_hash = js_mallocz_rt(rt, sizeof(rt->atom_hash[0]) * new_hash_size);
    if (!new_hash)
        return -1;
    for(i = 0; i < rt->atom_hash_size; i++) {
        h = rt->atom_hash[i];
        while (h != 0) {
            p = rt->atom_array[h];
            hash_next1 = p->hash_next;
            /* add in new hash table */
            j = p->hash & new_hash_mask;
            p->hash_next = new_hash[j];
            new_hash[j] = h;
            h = hash_next1;
        }
    }
    js_free_rt(rt, rt->atom_hash);
    rt->atom_hash = new_hash;
    rt->atom_hash_size = new_hash_size;
    rt->atom_count_resize = JS_ATOM_COUNT_RESIZE(new_hash_size);
    //    JS_DumpAtoms(rt);
    return 0;
}

static int JS_InitAtoms(JSRuntime *rt)
{
    int i, len, atom_type;
    const char *p;

    rt->atom_hash_size = 0;
    rt->atom_hash = NULL;
    rt->atom_count = 0;
    rt->atom_size = 0;
    rt->atom_free_index = 0;
    if (JS_ResizeAtomHash(rt, 256))     /* there are at least 195 predefined atoms */
        return -1;

    p = js_atom_init;
    for(i = 1; i < JS_ATOM_END; i++) {
        if (i == JS_ATOM_Private_brand)
            atom_type = JS_ATOM_TYPE_PRIVATE;
        else if (i >= JS_ATOM_Symbol_toPrimitive)
            atom_type = JS_ATOM_TYPE_SYMBOL;
        else
            atom_type = JS_ATOM_TYPE_STRING;
        len = strlen(p);
        if (__JS_NewAtomInit(rt, p, len, atom_type) == JS_ATOM_NULL)
            return -1;
        p = p + len + 1;
    }
    return 0;
}

static JSAtom JS_DupAtomRT(JSRuntime *rt, JSAtom v)
{
    JSAtomStruct *p;

    if (!__JS_AtomIsConst(v)) {
        p = rt->atom_array[v];
        p->header.ref_count++;
    }
    return v;
}

JSAtom JS_DupAtom(JSContext *ctx, JSAtom v)
{
    JSRuntime *rt;
    JSAtomStruct *p;

    if (!__JS_AtomIsConst(v)) {
        rt = ctx->rt;
        p = rt->atom_array[v];
        p->header.ref_count++;
    }
    return v;
}

static JSAtomKindEnum JS_AtomGetKind(JSContext *ctx, JSAtom v)
{
    JSRuntime *rt;
    JSAtomStruct *p;

    rt = ctx->rt;
    if (__JS_AtomIsTaggedInt(v))
        return JS_ATOM_KIND_STRING;
    p = rt->atom_array[v];
    switch(p->atom_type) {
    case JS_ATOM_TYPE_STRING:
        return JS_ATOM_KIND_STRING;
    case JS_ATOM_TYPE_GLOBAL_SYMBOL:
        return JS_ATOM_KIND_SYMBOL;
    case JS_ATOM_TYPE_SYMBOL:
        switch(p->hash) {
        case JS_ATOM_HASH_SYMBOL:
            return JS_ATOM_KIND_SYMBOL;
        case JS_ATOM_HASH_PRIVATE:
            return JS_ATOM_KIND_PRIVATE;
        default:
            abort();
        }
    default:
        abort();
    }
    return (JSAtomKindEnum){-1}; // pacify compiler
}

static JSAtom js_get_atom_index(JSRuntime *rt, JSAtomStruct *p)
{
    uint32_t i = p->hash_next;  /* atom_index */
    if (p->atom_type != JS_ATOM_TYPE_SYMBOL) {
        JSAtomStruct *p1;

        i = rt->atom_hash[p->hash & (rt->atom_hash_size - 1)];
        p1 = rt->atom_array[i];
        while (p1 != p) {
            assert(i != 0);
            i = p1->hash_next;
            p1 = rt->atom_array[i];
        }
    }
    return i;
}

/* string case (internal). Return JS_ATOM_NULL if error. 'str' is
   freed. */
static JSAtom __JS_NewAtom(JSRuntime *rt, JSString *str, int atom_type)
{
    uint32_t h, h1, i;
    JSAtomStruct *p;
    int len;

    if (atom_type < JS_ATOM_TYPE_SYMBOL) {
        /* str is not NULL */
        if (str->atom_type == atom_type) {
            /* str is the atom, return its index */
            i = js_get_atom_index(rt, str);
            /* reduce string refcount and increase atom's unless constant */
            if (__JS_AtomIsConst(i))
                str->header.ref_count--;
            return i;
        }
        /* try and locate an already registered atom */
        len = str->len;
        h = hash_string(str, atom_type);
        h &= JS_ATOM_HASH_MASK;
        h1 = h & (rt->atom_hash_size - 1);
        i = rt->atom_hash[h1];
        while (i != 0) {
            p = rt->atom_array[i];
            if (p->hash == h &&
                p->atom_type == atom_type &&
                p->len == len &&
                js_string_memcmp(p, str, len) == 0) {
                if (!__JS_AtomIsConst(i))
                    p->header.ref_count++;
                goto done;
            }
            i = p->hash_next;
        }
    } else {
        h1 = 0; /* avoid warning */
        if (atom_type == JS_ATOM_TYPE_SYMBOL) {
            h = JS_ATOM_HASH_SYMBOL;
        } else {
            h = JS_ATOM_HASH_PRIVATE;
            atom_type = JS_ATOM_TYPE_SYMBOL;
        }
    }

    if (rt->atom_free_index == 0) {
        /* allow new atom entries */
        uint32_t new_size, start;
        JSAtomStruct **new_array;

        /* alloc new with size progression 3/2:
           4 6 9 13 19 28 42 63 94 141 211 316 474 711 1066 1599 2398 3597 5395 8092
           preallocating space for predefined atoms (at least 195).
         */
        new_size = max_int(211, rt->atom_size * 3 / 2);
        if (new_size > JS_ATOM_MAX)
            goto fail;
        /* XXX: should use realloc2 to use slack space */
        new_array = js_realloc_rt(rt, rt->atom_array, sizeof(*new_array) * new_size);
        if (!new_array)
            goto fail;
        /* Note: the atom 0 is not used */
        start = rt->atom_size;
        if (start == 0) {
            /* JS_ATOM_NULL entry */
            p = js_mallocz_rt(rt, sizeof(JSAtomStruct));
            if (!p) {
                js_free_rt(rt, new_array);
                goto fail;
            }
            p->header.ref_count = 1;  /* not refcounted */
            p->atom_type = JS_ATOM_TYPE_SYMBOL;
#ifdef ENABLE_DUMPS // JS_DUMP_LEAKS
            list_add_tail(&p->link, &rt->string_list);
#endif
            new_array[0] = p;
            rt->atom_count++;
            start = 1;
        }
        rt->atom_size = new_size;
        rt->atom_array = new_array;
        rt->atom_free_index = start;
        for(i = start; i < new_size; i++) {
            uint32_t next;
            if (i == (new_size - 1))
                next = 0;
            else
                next = i + 1;
            rt->atom_array[i] = atom_set_free(next);
        }
    }

    if (str) {
        if (str->atom_type == 0) {
            p = str;
            p->atom_type = atom_type;
        } else {
            p = js_malloc_rt(rt, sizeof(JSString) +
                             (str->len << str->is_wide_char) +
                             1 - str->is_wide_char);
            if (unlikely(!p))
                goto fail;
            p->header.ref_count = 1;
            p->is_wide_char = str->is_wide_char;
            p->len = str->len;
            p->kind = JS_STRING_KIND_NORMAL;
#ifdef ENABLE_DUMPS // JS_DUMP_LEAKS
            list_add_tail(&p->link, &rt->string_list);
#endif
            memcpy(str8(p), str8(str), (str->len << str->is_wide_char) +
                   1 - str->is_wide_char);
            js_free_string(rt, str);
        }
    } else {
        p = js_malloc_rt(rt, sizeof(JSAtomStruct)); /* empty wide string */
        if (!p)
            return JS_ATOM_NULL;
        p->header.ref_count = 1;
        p->is_wide_char = 1;    /* Hack to represent NULL as a JSString */
        p->len = 0;
        p->kind = JS_STRING_KIND_NORMAL;
#ifdef ENABLE_DUMPS // JS_DUMP_LEAKS
        list_add_tail(&p->link, &rt->string_list);
#endif
    }

    /* use an already free entry */
    i = rt->atom_free_index;
    rt->atom_free_index = atom_get_free(rt->atom_array[i]);
    rt->atom_array[i] = p;

    p->hash = h;
    p->hash_next = i;   /* atom_index */
    p->atom_type = atom_type;
    p->first_weak_ref = NULL;

    rt->atom_count++;

    if (atom_type != JS_ATOM_TYPE_SYMBOL) {
        p->hash_next = rt->atom_hash[h1];
        rt->atom_hash[h1] = i;
        if (unlikely(rt->atom_count >= rt->atom_count_resize))
            JS_ResizeAtomHash(rt, rt->atom_hash_size * 2);
    }

    //    JS_DumpAtoms(rt);
    return i;

 fail:
    i = JS_ATOM_NULL;
 done:
    if (str)
        js_free_string(rt, str);
    return i;
}

// XXX: `str` must be pure ASCII. No UTF-8 encoded strings
// XXX: `str` must not be the string representation of a small integer
static JSAtom __JS_NewAtomInit(JSRuntime *rt, const char *str, int len,
                               int atom_type)
{
    JSString *p;
    p = js_alloc_string_rt(rt, len, 0);
    if (!p)
        return JS_ATOM_NULL;
    memcpy(str8(p), str, len);
    str8(p)[len] = '\0';
    return __JS_NewAtom(rt, p, atom_type);
}

// XXX: `str` must be raw 8-bit contents. No UTF-8 encoded strings
static JSAtom __JS_FindAtom(JSRuntime *rt, const char *str, size_t len,
                            int atom_type)
{
    uint32_t h, h1, i;
    JSAtomStruct *p;

    h = hash_string8((const uint8_t *)str, len, JS_ATOM_TYPE_STRING);
    h &= JS_ATOM_HASH_MASK;
    h1 = h & (rt->atom_hash_size - 1);
    i = rt->atom_hash[h1];
    while (i != 0) {
        p = rt->atom_array[i];
        if (p->hash == h &&
            p->atom_type == JS_ATOM_TYPE_STRING &&
            p->len == len &&
            p->is_wide_char == 0 &&
            memcmp(str8(p), str, len) == 0) {
            if (!__JS_AtomIsConst(i))
                p->header.ref_count++;
            return i;
        }
        i = p->hash_next;
    }
    return JS_ATOM_NULL;
}

static void JS_FreeAtomStruct(JSRuntime *rt, JSAtomStruct *p)
{
    uint32_t i = p->hash_next;  /* atom_index */
    if (p->atom_type != JS_ATOM_TYPE_SYMBOL) {
        JSAtomStruct *p0, *p1;
        uint32_t h0;

        h0 = p->hash & (rt->atom_hash_size - 1);
        i = rt->atom_hash[h0];
        p1 = rt->atom_array[i];
        if (p1 == p) {
            rt->atom_hash[h0] = p1->hash_next;
        } else {
            for(;;) {
                assert(i != 0);
                p0 = p1;
                i = p1->hash_next;
                p1 = rt->atom_array[i];
                if (p1 == p) {
                    p0->hash_next = p1->hash_next;
                    break;
                }
            }
        }
    }
    /* insert in free atom list */
    rt->atom_array[i] = atom_set_free(rt->atom_free_index);
    rt->atom_free_index = i;
    if (unlikely(p->first_weak_ref)) {
        reset_weak_ref(rt, &p->first_weak_ref);
    }
    /* free the string structure */
#ifdef ENABLE_DUMPS // JS_DUMP_LEAKS
    list_del(&p->link);
#endif
    js_free_rt(rt, p);
    rt->atom_count--;
    assert(rt->atom_count >= 0);
}

static void __JS_FreeAtom(JSRuntime *rt, uint32_t i)
{
    JSAtomStruct *p;

    p = rt->atom_array[i];
    if (--p->header.ref_count > 0)
        return;
    JS_FreeAtomStruct(rt, p);
}

/* Warning: 'p' is freed */
static JSAtom JS_NewAtomStr(JSContext *ctx, JSString *p)
{
    JSRuntime *rt = ctx->rt;
    uint32_t n;
    if (is_num_string(&n, p)) {
        if (n <= JS_ATOM_MAX_INT) {
            js_free_string(rt, p);
            return __JS_AtomFromUInt32(n);
        }
    }
    /* XXX: should generate an exception */
    return __JS_NewAtom(rt, p, JS_ATOM_TYPE_STRING);
}

/* `str` may be pure ASCII or UTF-8 encoded */
JSAtom JS_NewAtomLen(JSContext *ctx, const char *str, size_t len)
{
    JSValue val;

    if (len == 0 || !is_digit(*str)) {
        // TODO(chqrlie): this does not work if `str` has UTF-8 encoded contents
        // bug example: `({ "\u00c3\u00a9": 1 }).\u00e9` evaluates to `1`.
        JSAtom atom = __JS_FindAtom(ctx->rt, str, len, JS_ATOM_TYPE_STRING);
        if (atom)
            return atom;
    }
    val = JS_NewStringLen(ctx, str, len);
    if (JS_IsException(val))
        return JS_ATOM_NULL;
    return JS_NewAtomStr(ctx, JS_VALUE_GET_STRING(val));
}

/* `str` may be pure ASCII or UTF-8 encoded */
JSAtom JS_NewAtom(JSContext *ctx, const char *str)
{
    return JS_NewAtomLen(ctx, str, strlen(str));
}

JSAtom JS_NewAtomUInt32(JSContext *ctx, uint32_t n)
{
    if (n <= JS_ATOM_MAX_INT) {
        return __JS_AtomFromUInt32(n);
    } else {
        char buf[16];
        size_t len = u32toa(buf, n);
        JSValue val = js_new_string8_len(ctx, buf, len);
        if (JS_IsException(val))
            return JS_ATOM_NULL;
        return __JS_NewAtom(ctx->rt, JS_VALUE_GET_STRING(val),
                            JS_ATOM_TYPE_STRING);
    }
}

static JSAtom JS_NewAtomInt64(JSContext *ctx, int64_t n)
{
    if ((uint64_t)n <= JS_ATOM_MAX_INT) {
        return __JS_AtomFromUInt32((uint32_t)n);
    } else {
        char buf[24];
        size_t len = i64toa(buf, n);
        JSValue val = js_new_string8_len(ctx, buf, len);
        if (JS_IsException(val))
            return JS_ATOM_NULL;
        return __JS_NewAtom(ctx->rt, JS_VALUE_GET_STRING(val),
                            JS_ATOM_TYPE_STRING);
    }
}

/* 'p' is freed */
static JSValue JS_NewSymbolInternal(JSContext *ctx, JSString *p, int atom_type)
{
    JSRuntime *rt = ctx->rt;
    JSAtom atom;
    atom = __JS_NewAtom(rt, p, atom_type);
    if (atom == JS_ATOM_NULL)
        return JS_ThrowOutOfMemory(ctx);
    return JS_MKPTR(JS_TAG_SYMBOL, rt->atom_array[atom]);
}

/* descr must be a non-numeric string atom */
static JSValue JS_NewSymbolFromAtom(JSContext *ctx, JSAtom descr,
                                    int atom_type)
{
    JSRuntime *rt = ctx->rt;
    JSString *p;

    assert(!__JS_AtomIsTaggedInt(descr));
    assert(descr < rt->atom_size);
    p = rt->atom_array[descr];
    js_dup(JS_MKPTR(JS_TAG_STRING, p));
    return JS_NewSymbolInternal(ctx, p, atom_type);
}

/* `description` may be pure ASCII or UTF-8 encoded */
JSValue JS_NewSymbol(JSContext *ctx, const char *description, bool is_global)
{
    JSAtom atom = JS_NewAtom(ctx, description);
    if (atom == JS_ATOM_NULL)
        return JS_EXCEPTION;
    return JS_NewSymbolFromAtom(ctx, atom, is_global ? JS_ATOM_TYPE_GLOBAL_SYMBOL : JS_ATOM_TYPE_SYMBOL);
}

#define ATOM_GET_STR_BUF_SIZE 64

static const char *JS_AtomGetStrRT(JSRuntime *rt, char *buf, int buf_size,
                                   JSAtom atom)
{
    if (__JS_AtomIsTaggedInt(atom)) {
        snprintf(buf, buf_size, "%u", __JS_AtomToUInt32(atom));
    } else if (atom == JS_ATOM_NULL) {
        snprintf(buf, buf_size, "<null>");
    } else if (atom >= rt->atom_size) {
        assert(atom < rt->atom_size);
        snprintf(buf, buf_size, "<invalid %x>", atom);
    } else {
        JSAtomStruct *p = rt->atom_array[atom];
        *buf = '\0';
        if (atom_is_free(p)) {
            snprintf(buf, buf_size, "<free %x>", atom);
        } else if (p != NULL) {
            JSString *str = p;
            if (str->is_wide_char) {
                /* encode surrogates correctly */
                utf8_encode_buf16(buf, buf_size, str16(str), str->len);
            } else {
                utf8_encode_buf8(buf, buf_size, str8(str), str->len);
            }
        }
    }
    return buf;
}

static const char *JS_AtomGetStr(JSContext *ctx, char *buf, int buf_size, JSAtom atom)
{
    return JS_AtomGetStrRT(ctx->rt, buf, buf_size, atom);
}

static JSValue __JS_AtomToValue(JSContext *ctx, JSAtom atom, bool force_string)
{
    char buf[ATOM_GET_STR_BUF_SIZE];

    if (__JS_AtomIsTaggedInt(atom)) {
        size_t len = u32toa(buf, __JS_AtomToUInt32(atom));
        return js_new_string8_len(ctx, buf, len);
    } else {
        JSRuntime *rt = ctx->rt;
        JSAtomStruct *p;
        assert(atom < rt->atom_size);
        p = rt->atom_array[atom];
        if (p->atom_type == JS_ATOM_TYPE_STRING) {
            goto ret_string;
        } else if (force_string) {
            if (p->len == 0 && p->is_wide_char != 0) {
                /* no description string */
                p = rt->atom_array[JS_ATOM_empty_string];
            }
        ret_string:
            return js_dup(JS_MKPTR(JS_TAG_STRING, p));
        } else {
            return js_dup(JS_MKPTR(JS_TAG_SYMBOL, p));
        }
    }
}

JSValue JS_AtomToValue(JSContext *ctx, JSAtom atom)
{
    return __JS_AtomToValue(ctx, atom, false);
}

JSValue JS_AtomToString(JSContext *ctx, JSAtom atom)
{
    return __JS_AtomToValue(ctx, atom, true);
}

/* return true if the atom is an array index (i.e. 0 <= index <=
   2^32-2 and return its value */
static bool JS_AtomIsArrayIndex(JSContext *ctx, uint32_t *pval, JSAtom atom)
{
    if (__JS_AtomIsTaggedInt(atom)) {
        *pval = __JS_AtomToUInt32(atom);
        return true;
    } else {
        JSRuntime *rt = ctx->rt;
        JSAtomStruct *p;
        uint32_t val;

        assert(atom < rt->atom_size);
        p = rt->atom_array[atom];
        if (p->atom_type == JS_ATOM_TYPE_STRING &&
            is_num_string(&val, p) && val != -1) {
            *pval = val;
            return true;
        } else {
            *pval = 0;
            return false;
        }
    }
}

/* This test must be fast if atom is not a numeric index (e.g. a
   method name). Return JS_UNDEFINED if not a numeric
   index. JS_EXCEPTION can also be returned. */
static JSValue JS_AtomIsNumericIndex1(JSContext *ctx, JSAtom atom)
{
    JSRuntime *rt = ctx->rt;
    JSAtomStruct *p1;
    JSString *p;
    int c, len, ret;
    JSValue num, str;

    if (__JS_AtomIsTaggedInt(atom))
        return js_int32(__JS_AtomToUInt32(atom));
    assert(atom < rt->atom_size);
    p1 = rt->atom_array[atom];
    if (p1->atom_type != JS_ATOM_TYPE_STRING)
        return JS_UNDEFINED;
    p = p1;
    len = p->len;
    if (p->is_wide_char) {
        const uint16_t *r = str16(p), *r_end = str16(p) + len;
        if (r >= r_end)
            return JS_UNDEFINED;
        c = *r;
        if (c == '-') {
            if (r >= r_end)
                return JS_UNDEFINED;
            r++;
            c = *r;
            /* -0 case is specific */
            if (c == '0' && len == 2)
                goto minus_zero;
        }
        /* XXX: should test NaN, but the tests do not check it */
        if (!is_num(c)) {
            /* XXX: String should be normalized, therefore 8-bit only */
            const uint16_t nfinity16[7] = { 'n', 'f', 'i', 'n', 'i', 't', 'y' };
            if (!(c =='I' && (r_end - r) == 8 &&
                  !memcmp(r + 1, nfinity16, sizeof(nfinity16))))
                return JS_UNDEFINED;
        }
    } else {
        const uint8_t *r = str8(p), *r_end = str8(p) + len;
        if (r >= r_end)
            return JS_UNDEFINED;
        c = *r;
        if (c == '-') {
            if (r >= r_end)
                return JS_UNDEFINED;
            r++;
            c = *r;
            /* -0 case is specific */
            if (c == '0' && len == 2) {
            minus_zero:
                return js_float64(-0.0);
            }
        }
        if (!is_num(c)) {
            if (!(c =='I' && (r_end - r) == 8 &&
                  !memcmp(r + 1, "nfinity", 7)))
                return JS_UNDEFINED;
        }
    }
    /* this is ECMA CanonicalNumericIndexString primitive */
    num = JS_ToNumber(ctx, JS_MKPTR(JS_TAG_STRING, p));
    if (JS_IsException(num))
        return num;
    str = JS_ToString(ctx, num);
    if (JS_IsException(str)) {
        JS_FreeValue(ctx, num);
        return str;
    }
    ret = js_string_eq(p, JS_VALUE_GET_STRING(str));
    JS_FreeValue(ctx, str);
    if (ret) {
        return num;
    } else {
        JS_FreeValue(ctx, num);
        return JS_UNDEFINED;
    }
}

/* return -1 if exception or true/false */
static int JS_AtomIsNumericIndex(JSContext *ctx, JSAtom atom)
{
    JSValue num;
    num = JS_AtomIsNumericIndex1(ctx, atom);
    if (likely(JS_IsUndefined(num)))
        return false;
    if (JS_IsException(num))
        return -1;
    JS_FreeValue(ctx, num);
    return true;
}

void JS_FreeAtom(JSContext *ctx, JSAtom v)
{
    if (!__JS_AtomIsConst(v))
        __JS_FreeAtom(ctx->rt, v);
}

void JS_FreeAtomRT(JSRuntime *rt, JSAtom v)
{
    if (!__JS_AtomIsConst(v))
        __JS_FreeAtom(rt, v);
}

/* return true if 'v' is a symbol with a string description */
static bool JS_AtomSymbolHasDescription(JSContext *ctx, JSAtom v)
{
    JSRuntime *rt;
    JSAtomStruct *p;

    rt = ctx->rt;
    if (__JS_AtomIsTaggedInt(v))
        return false;
    p = rt->atom_array[v];
    return (((p->atom_type == JS_ATOM_TYPE_SYMBOL &&
              p->hash == JS_ATOM_HASH_SYMBOL) ||
             p->atom_type == JS_ATOM_TYPE_GLOBAL_SYMBOL) &&
            !(p->len == 0 && p->is_wide_char != 0));
}

static __maybe_unused void print_atom(JSContext *ctx, JSAtom atom)
{
    char buf[ATOM_GET_STR_BUF_SIZE];
    const char *p;
    int i;

    /* XXX: should handle embedded null characters */
    /* XXX: should move encoding code to JS_AtomGetStr */
    p = JS_AtomGetStr(ctx, buf, sizeof(buf), atom);
    for (i = 0; p[i]; i++) {
        int c = (unsigned char)p[i];
        if (!((c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') ||
              (c == '_' || c == '$') || (c >= '0' && c <= '9' && i > 0)))
            break;
    }
    if (i > 0 && p[i] == '\0') {
        printf("%s", p);
    } else {
        putchar('"');
        printf("%.*s", i, p);
        for (; p[i]; i++) {
            int c = (unsigned char)p[i];
            if (c == '\"' || c == '\\') {
                putchar('\\');
                putchar(c);
            } else if (c >= ' ' && c <= 126) {
                putchar(c);
            } else if (c == '\n') {
                putchar('\\');
                putchar('n');
            } else {
                printf("\\u%04x", c);
            }
        }
        putchar('\"');
    }
}

/* free with JS_FreeCString() */
const char *JS_AtomToCStringLen(JSContext *ctx, size_t *plen, JSAtom atom)
{
    JSValue str;
    const char *cstr;

    str = JS_AtomToString(ctx, atom);
    if (JS_IsException(str)) {
        if (plen)
            *plen = 0;
        return NULL;
    }
    cstr = JS_ToCStringLen(ctx, plen, str);
    JS_FreeValue(ctx, str);
    return cstr;
}

#ifndef QJS_DISABLE_PARSER

/* return a string atom containing name concatenated with str1 */
/* `str1` may be pure ASCII or UTF-8 encoded */
// TODO(chqrlie): use string concatenation instead of UTF-8 conversion
static JSAtom js_atom_concat_str(JSContext *ctx, JSAtom name, const char *str1)
{
    JSValue str;
    JSAtom atom;
    const char *cstr;
    char *cstr2;
    size_t len, len1;

    str = JS_AtomToString(ctx, name);
    if (JS_IsException(str))
        return JS_ATOM_NULL;
    cstr = JS_ToCStringLen(ctx, &len, str);
    if (!cstr)
        goto fail;
    len1 = strlen(str1);
    cstr2 = js_malloc(ctx, len + len1 + 1);
    if (!cstr2)
        goto fail;
    memcpy(cstr2, cstr, len);
    memcpy(cstr2 + len, str1, len1);
    cstr2[len + len1] = '\0';
    atom = JS_NewAtomLen(ctx, cstr2, len + len1);
    js_free(ctx, cstr2);
    JS_FreeCString(ctx, cstr);
    JS_FreeValue(ctx, str);
    return atom;
 fail:
    JS_FreeCString(ctx, cstr);
    JS_FreeValue(ctx, str);
    return JS_ATOM_NULL;
}

static JSAtom js_atom_concat_num(JSContext *ctx, JSAtom name, uint32_t n)
{
    char buf[16];
    size_t len;

    len = u32toa(buf, n);
    buf[len] = '\0';
    return js_atom_concat_str(ctx, name, buf);
}

#endif // QJS_DISABLE_PARSER

static inline bool JS_IsEmptyString(JSValueConst v)
{
    return JS_VALUE_GET_TAG(v) == JS_TAG_STRING && JS_VALUE_GET_STRING(v)->len == 0;
}

/* JSClass support */

/* a new class ID is allocated if *pclass_id == 0, otherwise *pclass_id is left unchanged */
JSClassID JS_NewClassID(JSRuntime *rt, JSClassID *pclass_id)
{
    JSClassID class_id = *pclass_id;
    if (class_id == 0) {
        class_id = rt->js_class_id_alloc++;
        *pclass_id = class_id;
    }
    return class_id;
}

JSClassID JS_GetClassID(JSValueConst v)
{
  JSObject *p;
  if (JS_VALUE_GET_TAG(v) != JS_TAG_OBJECT)
    return JS_INVALID_CLASS_ID;
  p = JS_VALUE_GET_OBJ(v);
  return p->class_id;
}

bool JS_IsRegisteredClass(JSRuntime *rt, JSClassID class_id)
{
    return (class_id < rt->class_count &&
            rt->class_array[class_id].class_id != 0);
}

/* create a new object internal class. Return -1 if error, 0 if
   OK. The finalizer can be NULL if none is needed. */
static int JS_NewClass1(JSRuntime *rt, JSClassID class_id,
                        const JSClassDef *class_def, JSAtom name)
{
    int new_size, i;
    JSClass *cl, *new_class_array;
    struct list_head *el;

    if (class_id >= (1 << 16))
        return -1;
    if (class_id < rt->class_count &&
        rt->class_array[class_id].class_id != 0)
        return -1;

    if (class_id >= rt->class_count) {
        new_size = max_int(JS_CLASS_INIT_COUNT,
                           max_int(class_id + 1, rt->class_count * 3 / 2));

        /* reallocate the context class prototype array, if any */
        list_for_each(el, &rt->context_list) {
            JSContext *ctx = list_entry(el, JSContext, link);
            JSValue *new_tab;
            new_tab = js_realloc_rt(rt, ctx->class_proto,
                                    sizeof(ctx->class_proto[0]) * new_size);
            if (!new_tab)
                return -1;
            for(i = rt->class_count; i < new_size; i++)
                new_tab[i] = JS_NULL;
            ctx->class_proto = new_tab;
        }
        /* reallocate the class array */
        new_class_array = js_realloc_rt(rt, rt->class_array,
                                        sizeof(JSClass) * new_size);
        if (!new_class_array)
            return -1;
        memset(new_class_array + rt->class_count, 0,
               (new_size - rt->class_count) * sizeof(JSClass));
        rt->class_array = new_class_array;
        rt->class_count = new_size;
    }
    cl = &rt->class_array[class_id];
    cl->class_id = class_id;
    cl->class_name = JS_DupAtomRT(rt, name);
    cl->finalizer = class_def->finalizer;
    cl->gc_mark = class_def->gc_mark;
    cl->call = class_def->call;
    cl->exotic = class_def->exotic;
    return 0;
}

int JS_NewClass(JSRuntime *rt, JSClassID class_id, const JSClassDef *class_def)
{
    int ret, len;
    JSAtom name;

    // XXX: class_def->class_name must be raw 8-bit contents. No UTF-8 encoded strings
    len = strlen(class_def->class_name);
    name = __JS_FindAtom(rt, class_def->class_name, len, JS_ATOM_TYPE_STRING);
    if (name == JS_ATOM_NULL) {
        name = __JS_NewAtomInit(rt, class_def->class_name, len, JS_ATOM_TYPE_STRING);
        if (name == JS_ATOM_NULL)
            return -1;
    }
    ret = JS_NewClass1(rt, class_id, class_def, name);
    JS_FreeAtomRT(rt, name);
    return ret;
}

static inline JSValue js_empty_string(JSRuntime *rt)
{
    JSAtomStruct *p = rt->atom_array[JS_ATOM_empty_string];
    return js_dup(JS_MKPTR(JS_TAG_STRING, p));
}

// XXX: `buf` contains raw 8-bit data, no UTF-8 decoding is performed
// XXX: no special case for len == 0
static JSValue js_new_string8_len(JSContext *ctx, const char *buf, int len)
{
    JSString *str;
    str = js_alloc_string(ctx, len, 0);
    if (!str)
        return JS_EXCEPTION;
    memcpy(str8(str), buf, len);
    str8(str)[len] = '\0';
    return JS_MKPTR(JS_TAG_STRING, str);
}

// XXX: `buf` contains raw 8-bit data, no UTF-8 decoding is performed
// XXX: no special case for the empty string
static inline JSValue js_new_string8(JSContext *ctx, const char *str)
{
    return js_new_string8_len(ctx, str, strlen(str));
}

static JSValue js_new_string16_len(JSContext *ctx, const uint16_t *buf, int len)
{
    JSString *str;
    str = js_alloc_string(ctx, len, 1);
    if (!str)
        return JS_EXCEPTION;
    memcpy(str16(str), buf, len * 2);
    return JS_MKPTR(JS_TAG_STRING, str);
}

static JSValue js_new_string_char(JSContext *ctx, uint16_t c)
{
    if (c < 0x100) {
        char ch8 = c;
        return js_new_string8_len(ctx, &ch8, 1);
    } else {
        uint16_t ch16 = c;
        return js_new_string16_len(ctx, &ch16, 1);
    }
}

static JSValue js_sub_string(JSContext *ctx, JSString *p, int start, int end)
{
    JSStringSlice *slice;
    JSString *q;
    int len;

    len = end - start;
    if (start == 0 && end == p->len) {
        return js_dup(JS_MKPTR(JS_TAG_STRING, p));
    }
    if (len <= 0) {
        return js_empty_string(ctx->rt);
    }
    if (len > (JS_STRING_SLICE_LEN_MAX >> p->is_wide_char)) {
        if (p->kind == JS_STRING_KIND_SLICE) {
            slice = (void *)&p[1];
            p = slice->parent;
            start += slice->start >> p->is_wide_char; // bytes -> chars
        }
        // allocate as 16 bit wide string to avoid wastage;
        // js_alloc_string allocates 1 byte extra for 8 bit strings;
        q = js_alloc_string(ctx, sizeof(*slice)/2, /*is_wide_char*/true);
        if (!q)
            return JS_EXCEPTION;
        q->is_wide_char = p->is_wide_char;
        q->kind = JS_STRING_KIND_SLICE;
        q->len = len;
        slice = (void *)&q[1];
        slice->parent = p;
        slice->start = start << p->is_wide_char; // chars -> bytes
        p->header.ref_count++;
        return JS_MKPTR(JS_TAG_STRING, q);
    }
    if (p->is_wide_char) {
        JSString *str;
        int i;
        uint16_t c = 0;
        for (i = start; i < end; i++) {
            c |= str16(p)[i];
        }
        if (c > 0xFF)
            return js_new_string16_len(ctx, str16(p) + start, len);

        str = js_alloc_string(ctx, len, 0);
        if (!str)
            return JS_EXCEPTION;
        for (i = 0; i < len; i++) {
            str8(str)[i] = str16(p)[start + i];
        }
        str8(str)[len] = '\0';
        return JS_MKPTR(JS_TAG_STRING, str);
    } else {
        return js_new_string8_len(ctx, (const char *)(str8(p) + start), len);
    }
}

typedef struct StringBuffer {
    JSContext *ctx;
    JSString *str;
    int len;
    int size;
    int is_wide_char;
    int error_status;
} StringBuffer;

/* It is valid to call string_buffer_end() and all string_buffer functions even
   if string_buffer_init() or another string_buffer function returns an error.
   If the error_status is set, string_buffer_end() returns JS_EXCEPTION.
 */
static int string_buffer_init2(JSContext *ctx, StringBuffer *s, int size,
                               int is_wide)
{
    s->ctx = ctx;
    s->size = size;
    s->len = 0;
    s->is_wide_char = is_wide;
    s->error_status = 0;
    s->str = js_alloc_string(ctx, size, is_wide);
    if (unlikely(!s->str)) {
        s->size = 0;
        return s->error_status = -1;
    }
#ifdef ENABLE_DUMPS // JS_DUMP_LEAKS
    /* the StringBuffer may reallocate the JSString, only link it at the end */
    list_del(&s->str->link);
#endif
    return 0;
}

static inline int string_buffer_init(JSContext *ctx, StringBuffer *s, int size)
{
    return string_buffer_init2(ctx, s, size, 0);
}

static void string_buffer_free(StringBuffer *s)
{
    js_free(s->ctx, s->str);
    s->str = NULL;
}

static int string_buffer_set_error(StringBuffer *s)
{
    js_free(s->ctx, s->str);
    s->str = NULL;
    s->size = 0;
    s->len = 0;
    return s->error_status = -1;
}

static no_inline int string_buffer_widen(StringBuffer *s, int size)
{
    JSString *str;
    size_t slack;
    int i;

    if (s->error_status)
        return -1;

    str = js_realloc2(s->ctx, s->str, sizeof(JSString) + (size << 1), &slack);
    if (!str)
        return string_buffer_set_error(s);
    size += slack >> 1;
    for(i = s->len; i-- > 0;) {
        str16(str)[i] = str8(str)[i];
    }
    s->is_wide_char = 1;
    s->size = size;
    s->str = str;
    return 0;
}

static no_inline int string_buffer_realloc(StringBuffer *s, int new_len, int c)
{
    JSString *new_str;
    int new_size;
    size_t new_size_bytes, slack;

    if (s->error_status)
        return -1;

    if (new_len > JS_STRING_LEN_MAX) {
        JS_ThrowRangeError(s->ctx, "invalid string length");
        return string_buffer_set_error(s);
    }
    new_size = min_int(max_int(new_len, s->size * 3 / 2), JS_STRING_LEN_MAX);
    if (!s->is_wide_char && c >= 0x100) {
        return string_buffer_widen(s, new_size);
    }
    new_size_bytes = sizeof(JSString) + (new_size << s->is_wide_char) + 1 - s->is_wide_char;
    new_str = js_realloc2(s->ctx, s->str, new_size_bytes, &slack);
    if (!new_str)
        return string_buffer_set_error(s);
    new_size = min_int(new_size + (slack >> s->is_wide_char), JS_STRING_LEN_MAX);
    s->size = new_size;
    s->str = new_str;
    return 0;
}

static no_inline int string_buffer_putc_slow(StringBuffer *s, uint32_t c)
{
    if (unlikely(s->len >= s->size)) {
        if (string_buffer_realloc(s, s->len + 1, c))
            return -1;
    }
    if (s->is_wide_char) {
        str16(s->str)[s->len++] = c;
    } else if (c < 0x100) {
        str8(s->str)[s->len++] = c;
    } else {
        if (string_buffer_widen(s, s->size))
            return -1;
        str16(s->str)[s->len++] = c;
    }
    return 0;
}

/* 0 <= c <= 0xff */
static int string_buffer_putc8(StringBuffer *s, uint32_t c)
{
    if (unlikely(s->len >= s->size)) {
        if (string_buffer_realloc(s, s->len + 1, c))
            return -1;
    }
    if (s->is_wide_char) {
        str16(s->str)[s->len++] = c;
    } else {
        str8(s->str)[s->len++] = c;
    }
    return 0;
}

/* 0 <= c <= 0xffff */
static int string_buffer_putc16(StringBuffer *s, uint32_t c)
{
    if (likely(s->len < s->size)) {
        if (s->is_wide_char) {
            str16(s->str)[s->len++] = c;
            return 0;
        } else if (c < 0x100) {
            str8(s->str)[s->len++] = c;
            return 0;
        }
    }
    return string_buffer_putc_slow(s, c);
}

/* 0 <= c <= 0x10ffff */
static int string_buffer_putc(StringBuffer *s, uint32_t c)
{
    if (unlikely(c >= 0x10000)) {
        /* surrogate pair */
        if (string_buffer_putc16(s, get_hi_surrogate(c)))
            return -1;
        c = get_lo_surrogate(c);
    }
    return string_buffer_putc16(s, c);
}

static int string_getc(JSString *p, int *pidx)
{
    int idx, c, c1;
    idx = *pidx;
    if (p->is_wide_char) {
        c = str16(p)[idx++];
        if (is_hi_surrogate(c) && idx < p->len) {
            c1 = str16(p)[idx];
            if (is_lo_surrogate(c1)) {
                c = from_surrogate(c, c1);
                idx++;
            }
        }
    } else {
        c = str8(p)[idx++];
    }
    *pidx = idx;
    return c;
}

static int string_buffer_write8(StringBuffer *s, const uint8_t *p, int len)
{
    int i;

    if (s->len + len > s->size) {
        if (string_buffer_realloc(s, s->len + len, 0))
            return -1;
    }
    if (s->is_wide_char) {
        for (i = 0; i < len; i++) {
            str16(s->str)[s->len + i] = p[i];
        }
        s->len += len;
    } else {
        memcpy(&str8(s->str)[s->len], p, len);
        s->len += len;
    }
    return 0;
}

static int string_buffer_write16(StringBuffer *s, const uint16_t *p, int len)
{
    int c = 0, i;

    for (i = 0; i < len; i++) {
        c |= p[i];
    }
    if (s->len + len > s->size) {
        if (string_buffer_realloc(s, s->len + len, c))
            return -1;
    } else if (!s->is_wide_char && c >= 0x100) {
        if (string_buffer_widen(s, s->size))
            return -1;
    }
    if (s->is_wide_char) {
        memcpy(&str16(s->str)[s->len], p, len << 1);
        s->len += len;
    } else {
        for (i = 0; i < len; i++) {
            str8(s->str)[s->len + i] = p[i];
        }
        s->len += len;
    }
    return 0;
}

/* appending an ASCII string */
static int string_buffer_puts8(StringBuffer *s, const char *str)
{
    return string_buffer_write8(s, (const uint8_t *)str, strlen(str));
}

static int string_buffer_concat(StringBuffer *s, JSString *p,
                                uint32_t from, uint32_t to)
{
    if (to <= from)
        return 0;
    if (p->is_wide_char)
        return string_buffer_write16(s, str16(p) + from, to - from);
    else
        return string_buffer_write8(s, str8(p) + from, to - from);
}

static int string_buffer_concat_value(StringBuffer *s, JSValueConst v)
{
    JSString *p;
    JSValue v1;
    int res;

    if (s->error_status) {
        /* prevent exception overload */
        return -1;
    }
    if (unlikely(JS_VALUE_GET_TAG(v) != JS_TAG_STRING)) {
        v1 = JS_ToString(s->ctx, v);
        if (JS_IsException(v1))
            return string_buffer_set_error(s);
        p = JS_VALUE_GET_STRING(v1);
        res = string_buffer_concat(s, p, 0, p->len);
        JS_FreeValue(s->ctx, v1);
        return res;
    }
    p = JS_VALUE_GET_STRING(v);
    return string_buffer_concat(s, p, 0, p->len);
}

static int string_buffer_concat_value_free(StringBuffer *s, JSValue v)
{
    JSString *p;
    int res;

    if (s->error_status) {
        /* prevent exception overload */
        JS_FreeValue(s->ctx, v);
        return -1;
    }
    if (unlikely(JS_VALUE_GET_TAG(v) != JS_TAG_STRING)) {
        v = JS_ToStringFree(s->ctx, v);
        if (JS_IsException(v))
            return string_buffer_set_error(s);
    }
    p = JS_VALUE_GET_STRING(v);
    res = string_buffer_concat(s, p, 0, p->len);
    JS_FreeValue(s->ctx, v);
    return res;
}

static int string_buffer_fill(StringBuffer *s, int c, int count)
{
    /* XXX: optimize */
    if (s->len + count > s->size) {
        if (string_buffer_realloc(s, s->len + count, c))
            return -1;
    }
    while (count-- > 0) {
        if (string_buffer_putc16(s, c))
            return -1;
    }
    return 0;
}

static JSValue string_buffer_end(StringBuffer *s)
{
    JSString *str;
    str = s->str;
    if (s->error_status)
        return JS_EXCEPTION;
    if (s->len == 0) {
        js_free(s->ctx, str);
        s->str = NULL;
        return js_empty_string(s->ctx->rt);
    }
    if (s->len < s->size) {
        /* smaller size so js_realloc should not fail, but OK if it does */
        /* XXX: should add some slack to avoid unnecessary calls */
        /* XXX: might need to use malloc+free to ensure smaller size */
        str = js_realloc_rt(s->ctx->rt, str, sizeof(JSString) +
                            (s->len << s->is_wide_char) + 1 - s->is_wide_char);
        if (str == NULL)
            str = s->str;
        s->str = str;
    }
    if (!s->is_wide_char)
        str8(str)[s->len] = 0;
#ifdef ENABLE_DUMPS // JS_DUMP_LEAKS
    list_add_tail(&str->link, &s->ctx->rt->string_list);
#endif
    str->is_wide_char = s->is_wide_char;
    str->len = s->len;
    s->str = NULL;
    return JS_MKPTR(JS_TAG_STRING, str);
}

/* create a string from a UTF-8 buffer */
JSValue JS_NewStringLen(JSContext *ctx, const char *buf, size_t buf_len)
{
    JSString *str;
    size_t len;
    int kind;

    if (buf_len <= 0) {
        return js_empty_string(ctx->rt);
    }
    /* Compute string kind and length: 7-bit, 8-bit, 16-bit, 16-bit UTF-16 */
    kind = utf8_scan(buf, buf_len, &len);
    if (len > JS_STRING_LEN_MAX)
        return JS_ThrowRangeError(ctx, "invalid string length");

    switch (kind) {
    case UTF8_PLAIN_ASCII:
        str = js_alloc_string(ctx, len, 0);
        if (!str)
            return JS_EXCEPTION;
        memcpy(str8(str), buf, len);
        str8(str)[len] = '\0';
        break;
    case UTF8_NON_ASCII:
        /* buf contains non-ASCII code-points, but limited to 8-bit values */
        str = js_alloc_string(ctx, len, 0);
        if (!str)
            return JS_EXCEPTION;
        utf8_decode_buf8(str8(str), len + 1, buf, buf_len);
        break;
    default:
        // This causes a potential problem in JS_ThrowError if message is invalid
        //if (kind & UTF8_HAS_ERRORS)
        //    return JS_ThrowRangeError(ctx, "invalid UTF-8 sequence");
        str = js_alloc_string(ctx, len, 1);
        if (!str)
            return JS_EXCEPTION;
        utf8_decode_buf16(str16(str), len, buf, buf_len);
        break;
    }
    return JS_MKPTR(JS_TAG_STRING, str);
}

JSValue JS_NewTwoByteString(JSContext *ctx, const uint16_t *buf, size_t len)
{
    JSString *str;

    if (!len)
        return js_empty_string(ctx->rt);
    str = js_alloc_string(ctx, len, 1);
    if (!str)
        return JS_EXCEPTION;
    memcpy(str16(str), buf, len * sizeof(*buf));
    return JS_MKPTR(JS_TAG_STRING, str);
}

static JSValue JS_ConcatString3(JSContext *ctx, const char *str1,
                                JSValue str2, const char *str3)
{
    StringBuffer b_s, *b = &b_s;
    int len1, len3;
    JSString *p;

    if (unlikely(JS_VALUE_GET_TAG(str2) != JS_TAG_STRING)) {
        str2 = JS_ToStringFree(ctx, str2);
        if (JS_IsException(str2))
            goto fail;
    }
    p = JS_VALUE_GET_STRING(str2);
    len1 = strlen(str1);
    len3 = strlen(str3);

    if (string_buffer_init2(ctx, b, len1 + p->len + len3, p->is_wide_char))
        goto fail;

    string_buffer_write8(b, (const uint8_t *)str1, len1);
    string_buffer_concat(b, p, 0, p->len);
    string_buffer_write8(b, (const uint8_t *)str3, len3);

    JS_FreeValue(ctx, str2);
    return string_buffer_end(b);

 fail:
    JS_FreeValue(ctx, str2);
    return JS_EXCEPTION;
}

/* `str` may be pure ASCII or UTF-8 encoded */
JSValue JS_NewAtomString(JSContext *ctx, const char *str)
{
    JSAtom atom = JS_NewAtom(ctx, str);
    if (atom == JS_ATOM_NULL)
        return JS_EXCEPTION;
    JSValue val = JS_AtomToString(ctx, atom);
    JS_FreeAtom(ctx, atom);
    return val;
}

/* return (NULL, 0) if exception. */
/* return pointer into a JSString with a live ref_count */
/* cesu8 determines if non-BMP1 codepoints are encoded as 1 or 2 utf-8 sequences */
const char *JS_ToCStringLen2(JSContext *ctx, size_t *plen, JSValueConst val1,
                             bool cesu8)
{
    JSValue val;
    JSString *str, *str_new;
    int pos, len, c, c1;
    JSObject *p;
    uint8_t *q;

    if (JS_VALUE_GET_TAG(val1) == JS_TAG_STRING) {
        val = js_dup(val1);
        goto go;
    }

    val = JS_ToString(ctx, val1);
    if (!JS_IsException(val))
        goto go;

    // Stringification can fail when there is an exception pending,
    // e.g. a stack overflow InternalError. Special-case exception
    // objects to make debugging easier, look up the .message property
    // and stringify that.
    if (JS_VALUE_GET_TAG(val1) != JS_TAG_OBJECT)
        goto fail;

    p = JS_VALUE_GET_OBJ(val1);
    if (p->class_id != JS_CLASS_ERROR)
        goto fail;

    val = JS_GetProperty(ctx, val1, JS_ATOM_message);
    if (JS_VALUE_GET_TAG(val) != JS_TAG_STRING) {
        JS_FreeValue(ctx, val);
        goto fail;
    }

go:
    str = JS_VALUE_GET_STRING(val);
    len = str->len;
    if (!str->is_wide_char) {
        const uint8_t *src = str8(str);
        int count;

        /* count the number of non-ASCII characters */
        /* Scanning the whole string is required for ASCII strings,
           and computing the number of non-ASCII bytes is less expensive
           than testing each byte, hence this method is faster for ASCII
           strings, which is the most common case.
         */
        count = 0;
        for (pos = 0; pos < len; pos++) {
            count += src[pos] >> 7;
        }
        if (count == 0 && str->kind == JS_STRING_KIND_NORMAL) {
            if (plen)
                *plen = len;
            return (const char *)src;
        }
        str_new = js_alloc_string(ctx, len + count, 0);
        if (!str_new)
            goto fail;
        q = str8(str_new);
        for (pos = 0; pos < len; pos++) {
            c = src[pos];
            if (c < 0x80) {
                *q++ = c;
            } else {
                *q++ = (c >> 6) | 0xc0;
                *q++ = (c & 0x3f) | 0x80;
            }
        }
    } else {
        const uint16_t *src = str16(str);
        /* Allocate 3 bytes per 16 bit code point. Surrogate pairs may
           produce 4 bytes but use 2 code points.
         */
        str_new = js_alloc_string(ctx, len * 3, 0);
        if (!str_new)
            goto fail;
        q = str8(str_new);
        pos = 0;
        while (pos < len) {
            c = src[pos++];
            if (c < 0x80) {
                *q++ = c;
            } else {
                if (is_hi_surrogate(c)) {
                    if (pos < len && !cesu8) {
                        c1 = src[pos];
                        if (is_lo_surrogate(c1)) {
                            pos++;
                            c = from_surrogate(c, c1);
                        } else {
                            /* Keep unmatched surrogate code points */
                            /* c = 0xfffd; */ /* error */
                        }
                    } else {
                        /* Keep unmatched surrogate code points */
                        /* c = 0xfffd; */ /* error */
                    }
                }
                q += utf8_encode(q, c);
            }
        }
    }

    *q = '\0';
    str_new->len = q - str8(str_new);
    JS_FreeValue(ctx, val);
    if (plen)
        *plen = str_new->len;
    return (const char *)str8(str_new);
 fail:
    if (plen)
        *plen = 0;
    return NULL;
}

void JS_FreeCString(JSContext *ctx, const char *ptr)
{
    if (!ptr)
        return;
    /* purposely removing constness */
    JS_FreeValue(ctx, JS_MKPTR(JS_TAG_STRING, (JSString *)ptr - 1));
}

static int memcmp16_8(const uint16_t *src1, const uint8_t *src2, int len)
{
    int c, i;
    for(i = 0; i < len; i++) {
        c = src1[i] - src2[i];
        if (c != 0)
            return c;
    }
    return 0;
}

static int memcmp16(const uint16_t *src1, const uint16_t *src2, int len)
{
    int c, i;
    for(i = 0; i < len; i++) {
        c = src1[i] - src2[i];
        if (c != 0)
            return c;
    }
    return 0;
}

static int js_string_memcmp(JSString *p1, JSString *p2, int len)
{
    int res;

    if (likely(!p1->is_wide_char)) {
        if (likely(!p2->is_wide_char))
            res = memcmp(str8(p1), str8(p2), len);
        else
            res = -memcmp16_8(str16(p2), str8(p1), len);
    } else {
        if (!p2->is_wide_char)
            res = memcmp16_8(str16(p1), str8(p2), len);
        else
            res = memcmp16(str16(p1), str16(p2), len);
    }
    return res;
}

static bool js_string_eq(JSString *p1, JSString *p2) {
    if (p1->len != p2->len)
        return false;
    return js_string_memcmp(p1, p2, p1->len) == 0;
}

/* return < 0, 0 or > 0 */
static int js_string_compare(JSString *p1, JSString *p2)
{
    int res, len;
    len = min_int(p1->len, p2->len);
    res = js_string_memcmp(p1, p2, len);
    if (res == 0)
        res = compare_u32(p1->len, p2->len);
    return res;
}

static void copy_str16(uint16_t *dst, JSString *p, int offset, int len)
{
    if (p->is_wide_char) {
        memcpy(dst, str16(p) + offset, len * 2);
    } else {
        const uint8_t *src1 = str8(p) + offset;
        int i;

        for(i = 0; i < len; i++)
            dst[i] = src1[i];
    }
}

static JSValue JS_ConcatString1(JSContext *ctx, JSString *p1, JSString *p2)
{
    JSString *p;
    uint32_t len;
    int is_wide_char;

    len = p1->len + p2->len;
    if (len > JS_STRING_LEN_MAX)
        return JS_ThrowRangeError(ctx, "invalid string length");
    is_wide_char = p1->is_wide_char | p2->is_wide_char;
    p = js_alloc_string(ctx, len, is_wide_char);
    if (!p)
        return JS_EXCEPTION;
    if (!is_wide_char) {
        memcpy(str8(p), str8(p1), p1->len);
        memcpy(str8(p) + p1->len, str8(p2), p2->len);
        str8(p)[len] = '\0';
    } else {
        copy_str16(str16(p), p1, 0, p1->len);
        copy_str16(str16(p) + p1->len, p2, 0, p2->len);
    }
    return JS_MKPTR(JS_TAG_STRING, p);
}

/* op1 and op2 are converted to strings. For convience, op1 or op2 =
   JS_EXCEPTION are accepted and return JS_EXCEPTION.  */
static JSValue JS_ConcatString(JSContext *ctx, JSValue op1, JSValue op2)
{
    JSValue ret;
    JSString *p1, *p2;

    if (unlikely(JS_VALUE_GET_TAG(op1) != JS_TAG_STRING)) {
        op1 = JS_ToStringFree(ctx, op1);
        if (JS_IsException(op1)) {
            JS_FreeValue(ctx, op2);
            return JS_EXCEPTION;
        }
    }
    if (unlikely(JS_VALUE_GET_TAG(op2) != JS_TAG_STRING)) {
        op2 = JS_ToStringFree(ctx, op2);
        if (JS_IsException(op2)) {
            JS_FreeValue(ctx, op1);
            return JS_EXCEPTION;
        }
    }
    p1 = JS_VALUE_GET_STRING(op1);
    p2 = JS_VALUE_GET_STRING(op2);

    /* XXX: could also check if p1 is empty */
    if (p2->len == 0) {
        goto ret_op1;
    }
    if (p1->header.ref_count == 1 && p1->is_wide_char == p2->is_wide_char
    &&  js_malloc_usable_size(ctx, p1) >= sizeof(*p1) + ((p1->len + p2->len) << p2->is_wide_char) + 1 - p1->is_wide_char) {
        /* Concatenate in place in available space at the end of p1 */
        if (p1->is_wide_char) {
            memcpy(str16(p1) + p1->len, str16(p2), p2->len << 1);
            p1->len += p2->len;
        } else {
            memcpy(str8(p1) + p1->len, str8(p2), p2->len);
            p1->len += p2->len;
            str8(p1)[p1->len] = '\0';
        }
    ret_op1:
        JS_FreeValue(ctx, op2);
        return op1;
    }
    ret = JS_ConcatString1(ctx, p1, p2);
    JS_FreeValue(ctx, op1);
    JS_FreeValue(ctx, op2);
    return ret;
}

/* Shape support */

static inline size_t get_shape_size(size_t hash_size, size_t prop_size)
{
    return hash_size * sizeof(uint32_t) + sizeof(JSShape) +
        prop_size * sizeof(JSShapeProperty);
}

static inline JSShape *get_shape_from_alloc(void *sh_alloc, size_t hash_size)
{
    return (JSShape *)(void *)((uint32_t *)sh_alloc + hash_size);
}

static inline uint32_t *prop_hash_end(JSShape *sh)
{
    return (uint32_t *)sh;
}

static inline void *get_alloc_from_shape(JSShape *sh)
{
    return prop_hash_end(sh) - ((intptr_t)sh->prop_hash_mask + 1);
}

static inline JSShapeProperty *get_shape_prop(JSShape *sh)
{
    return sh->prop;
}

static int init_shape_hash(JSRuntime *rt)
{
    rt->shape_hash_bits = 6;   /* 64 shapes */
    rt->shape_hash_size = 1 << rt->shape_hash_bits;
    rt->shape_hash_count = 0;
    rt->shape_hash = js_mallocz_rt(rt, sizeof(rt->shape_hash[0]) *
                                   rt->shape_hash_size);
    if (!rt->shape_hash)
        return -1;
    return 0;
}

/* same magic hash multiplier as the Linux kernel */
static uint32_t shape_hash(uint32_t h, uint32_t val)
{
    return (h + val) * 0x9e370001;
}

/* truncate the shape hash to 'hash_bits' bits */
static uint32_t get_shape_hash(uint32_t h, int hash_bits)
{
    return h >> (32 - hash_bits);
}

static uint32_t shape_initial_hash(JSObject *proto)
{
    uint32_t h;
    h = shape_hash(1, (uintptr_t)proto);
    if (sizeof(proto) > 4)
        h = shape_hash(h, (uint64_t)(uintptr_t)proto >> 32);
    return h;
}

static int resize_shape_hash(JSRuntime *rt, int new_shape_hash_bits)
{
    int new_shape_hash_size, i;
    uint32_t h;
    JSShape **new_shape_hash, *sh, *sh_next;

    new_shape_hash_size = 1 << new_shape_hash_bits;
    new_shape_hash = js_mallocz_rt(rt, sizeof(rt->shape_hash[0]) *
                                   new_shape_hash_size);
    if (!new_shape_hash)
        return -1;
    for(i = 0; i < rt->shape_hash_size; i++) {
        for(sh = rt->shape_hash[i]; sh != NULL; sh = sh_next) {
            sh_next = sh->shape_hash_next;
            h = get_shape_hash(sh->hash, new_shape_hash_bits);
            sh->shape_hash_next = new_shape_hash[h];
            new_shape_hash[h] = sh;
        }
    }
    js_free_rt(rt, rt->shape_hash);
    rt->shape_hash_bits = new_shape_hash_bits;
    rt->shape_hash_size = new_shape_hash_size;
    rt->shape_hash = new_shape_hash;
    return 0;
}

static void js_shape_hash_link(JSRuntime *rt, JSShape *sh)
{
    uint32_t h;
    h = get_shape_hash(sh->hash, rt->shape_hash_bits);
    sh->shape_hash_next = rt->shape_hash[h];
    rt->shape_hash[h] = sh;
    rt->shape_hash_count++;
}

static void js_shape_hash_unlink(JSRuntime *rt, JSShape *sh)
{
    uint32_t h;
    JSShape **psh;

    h = get_shape_hash(sh->hash, rt->shape_hash_bits);
    psh = &rt->shape_hash[h];
    while (*psh != sh)
        psh = &(*psh)->shape_hash_next;
    *psh = sh->shape_hash_next;
    rt->shape_hash_count--;
}

/* create a new empty shape with prototype 'proto' */
static no_inline JSShape *js_new_shape2(JSContext *ctx, JSObject *proto,
                                        int hash_size, int prop_size)
{
    JSRuntime *rt = ctx->rt;
    void *sh_alloc;
    JSShape *sh;

    /* resize the shape hash table if necessary */
    if (2 * (rt->shape_hash_count + 1) > rt->shape_hash_size) {
        resize_shape_hash(rt, rt->shape_hash_bits + 1);
    }

    sh_alloc = js_malloc(ctx, get_shape_size(hash_size, prop_size));
    if (!sh_alloc)
        return NULL;
    sh = get_shape_from_alloc(sh_alloc, hash_size);
    sh->header.ref_count = 1;
    add_gc_object(rt, &sh->header, JS_GC_OBJ_TYPE_SHAPE);
    if (proto)
        js_dup(JS_MKPTR(JS_TAG_OBJECT, proto));
    sh->proto = proto;
    memset(prop_hash_end(sh) - hash_size, 0, sizeof(prop_hash_end(sh)[0]) *
           hash_size);
    sh->prop_hash_mask = hash_size - 1;
    sh->prop_size = prop_size;
    sh->prop_count = 0;
    sh->deleted_prop_count = 0;

    /* insert in the hash table */
    sh->hash = shape_initial_hash(proto);
    sh->is_hashed = true;
    sh->has_small_array_index = false;
    js_shape_hash_link(ctx->rt, sh);
    return sh;
}

static JSShape *js_new_shape(JSContext *ctx, JSObject *proto)
{
    return js_new_shape2(ctx, proto, JS_PROP_INITIAL_HASH_SIZE,
                         JS_PROP_INITIAL_SIZE);
}

/* The shape is cloned. The new shape is not inserted in the shape
   hash table */
static JSShape *js_clone_shape(JSContext *ctx, JSShape *sh1)
{
    JSShape *sh;
    void *sh_alloc, *sh_alloc1;
    size_t size;
    JSShapeProperty *pr;
    uint32_t i, hash_size;

    hash_size = sh1->prop_hash_mask + 1;
    size = get_shape_size(hash_size, sh1->prop_size);
    sh_alloc = js_malloc(ctx, size);
    if (!sh_alloc)
        return NULL;
    sh_alloc1 = get_alloc_from_shape(sh1);
    memcpy(sh_alloc, sh_alloc1, size);
    sh = get_shape_from_alloc(sh_alloc, hash_size);
    sh->header.ref_count = 1;
    add_gc_object(ctx->rt, &sh->header, JS_GC_OBJ_TYPE_SHAPE);
    sh->is_hashed = false;
    if (sh->proto) {
        js_dup(JS_MKPTR(JS_TAG_OBJECT, sh->proto));
    }
    for(i = 0, pr = get_shape_prop(sh); i < sh->prop_count; i++, pr++) {
        JS_DupAtom(ctx, pr->atom);
    }
    return sh;
}

static JSShape *js_dup_shape(JSShape *sh)
{
    sh->header.ref_count++;
    return sh;
}

static void js_free_shape0(JSRuntime *rt, JSShape *sh)
{
    uint32_t i;
    JSShapeProperty *pr;

    assert(sh->header.ref_count == 0);
    if (sh->is_hashed)
        js_shape_hash_unlink(rt, sh);
    if (sh->proto != NULL) {
        JS_FreeValueRT(rt, JS_MKPTR(JS_TAG_OBJECT, sh->proto));
    }
    pr = get_shape_prop(sh);
    for(i = 0; i < sh->prop_count; i++) {
        JS_FreeAtomRT(rt, pr->atom);
        pr++;
    }
    remove_gc_object(&sh->header);
    js_free_rt(rt, get_alloc_from_shape(sh));
}

static void js_free_shape(JSRuntime *rt, JSShape *sh)
{
    if (unlikely(--sh->header.ref_count <= 0)) {
        js_free_shape0(rt, sh);
    }
}

static void js_free_shape_null(JSRuntime *rt, JSShape *sh)
{
    if (sh)
        js_free_shape(rt, sh);
}

/* make space to hold at least 'count' properties */
static no_inline int resize_properties(JSContext *ctx, JSShape **psh,
                                       JSObject *p, uint32_t count)
{
    JSShape *sh;
    uint32_t new_size, new_hash_size, new_hash_mask, i;
    JSShapeProperty *pr;
    void *sh_alloc;
    intptr_t h;

    sh = *psh;
    new_size = max_int(count, sh->prop_size * 3 / 2);
    /* Reallocate prop array first to avoid crash or size inconsistency
       in case of memory allocation failure */
    if (p) {
        JSProperty *new_prop;
        new_prop = js_realloc(ctx, p->prop, sizeof(new_prop[0]) * new_size);
        if (unlikely(!new_prop))
            return -1;
        p->prop = new_prop;
    }
    new_hash_size = sh->prop_hash_mask + 1;
    while (new_hash_size < new_size)
        new_hash_size = 2 * new_hash_size;
    if (new_hash_size != (sh->prop_hash_mask + 1)) {
        JSShape *old_sh;
        /* resize the hash table and the properties */
        old_sh = sh;
        sh_alloc = js_malloc(ctx, get_shape_size(new_hash_size, new_size));
        if (!sh_alloc)
            return -1;
        sh = get_shape_from_alloc(sh_alloc, new_hash_size);
        list_del(&old_sh->header.link);
        /* copy all the fields and the properties */
        memcpy(sh, old_sh,
               sizeof(JSShape) + sizeof(sh->prop[0]) * old_sh->prop_count);
        list_add_tail(&sh->header.link, &ctx->rt->gc_obj_list);
        new_hash_mask = new_hash_size - 1;
        sh->prop_hash_mask = new_hash_mask;
        memset(prop_hash_end(sh) - new_hash_size, 0,
               sizeof(prop_hash_end(sh)[0]) * new_hash_size);
        for(i = 0, pr = sh->prop; i < sh->prop_count; i++, pr++) {
            if (pr->atom != JS_ATOM_NULL) {
                h = ((uintptr_t)pr->atom & new_hash_mask);
                pr->hash_next = prop_hash_end(sh)[-h - 1];
                prop_hash_end(sh)[-h - 1] = i + 1;
            }
        }
        js_free(ctx, get_alloc_from_shape(old_sh));
    } else {
        /* only resize the properties */
        list_del(&sh->header.link);
        sh_alloc = js_realloc(ctx, get_alloc_from_shape(sh),
                              get_shape_size(new_hash_size, new_size));
        if (unlikely(!sh_alloc)) {
            /* insert again in the GC list */
            list_add_tail(&sh->header.link, &ctx->rt->gc_obj_list);
            return -1;
        }
        sh = get_shape_from_alloc(sh_alloc, new_hash_size);
        list_add_tail(&sh->header.link, &ctx->rt->gc_obj_list);
    }
    *psh = sh;
    sh->prop_size = new_size;
    return 0;
}

/* remove the deleted properties. */
static int compact_properties(JSContext *ctx, JSObject *p)
{
    JSShape *sh, *old_sh;
    void *sh_alloc;
    intptr_t h;
    uint32_t new_hash_size, i, j, new_hash_mask, new_size;
    JSShapeProperty *old_pr, *pr;
    JSProperty *prop, *new_prop;

    sh = p->shape;
    assert(!sh->is_hashed);

    new_size = max_int(JS_PROP_INITIAL_SIZE,
                       sh->prop_count - sh->deleted_prop_count);
    assert(new_size <= sh->prop_size);

    new_hash_size = sh->prop_hash_mask + 1;
    while ((new_hash_size / 2) >= new_size)
        new_hash_size = new_hash_size / 2;
    new_hash_mask = new_hash_size - 1;

    /* resize the hash table and the properties */
    old_sh = sh;
    sh_alloc = js_malloc(ctx, get_shape_size(new_hash_size, new_size));
    if (!sh_alloc)
        return -1;
    sh = get_shape_from_alloc(sh_alloc, new_hash_size);
    list_del(&old_sh->header.link);
    memcpy(sh, old_sh, sizeof(JSShape));
    list_add_tail(&sh->header.link, &ctx->rt->gc_obj_list);

    memset(prop_hash_end(sh) - new_hash_size, 0,
           sizeof(prop_hash_end(sh)[0]) * new_hash_size);

    j = 0;
    old_pr = old_sh->prop;
    pr = sh->prop;
    prop = p->prop;
    for(i = 0; i < sh->prop_count; i++) {
        if (old_pr->atom != JS_ATOM_NULL) {
            pr->atom = old_pr->atom;
            pr->flags = old_pr->flags;
            h = ((uintptr_t)old_pr->atom & new_hash_mask);
            pr->hash_next = prop_hash_end(sh)[-h - 1];
            prop_hash_end(sh)[-h - 1] = j + 1;
            prop[j] = prop[i];
            j++;
            pr++;
        }
        old_pr++;
    }
    assert(j == (sh->prop_count - sh->deleted_prop_count));
    sh->prop_hash_mask = new_hash_mask;
    sh->prop_size = new_size;
    sh->deleted_prop_count = 0;
    sh->prop_count = j;

    p->shape = sh;
    js_free(ctx, get_alloc_from_shape(old_sh));

    /* reduce the size of the object properties */
    new_prop = js_realloc(ctx, p->prop, sizeof(new_prop[0]) * new_size);
    if (new_prop)
        p->prop = new_prop;
    return 0;
}

static int add_shape_property(JSContext *ctx, JSShape **psh,
                              JSObject *p, JSAtom atom, int prop_flags)
{
    JSRuntime *rt = ctx->rt;
    JSShape *sh = *psh;
    JSShapeProperty *pr, *prop;
    uint32_t hash_mask, new_shape_hash = 0;
    intptr_t h;

    /* update the shape hash */
    if (sh->is_hashed) {
        js_shape_hash_unlink(rt, sh);
        new_shape_hash = shape_hash(shape_hash(sh->hash, atom), prop_flags);
    }

    if (unlikely(sh->prop_count >= sh->prop_size)) {
        if (resize_properties(ctx, psh, p, sh->prop_count + 1)) {
            /* in case of error, reinsert in the hash table.
               sh is still valid if resize_properties() failed */
            if (sh->is_hashed)
                js_shape_hash_link(rt, sh);
            return -1;
        }
        sh = *psh;
    }
    if (sh->is_hashed) {
        sh->hash = new_shape_hash;
        js_shape_hash_link(rt, sh);
    }
    /* Initialize the new shape property.
       The object property at p->prop[sh->prop_count] is uninitialized */
    prop = get_shape_prop(sh);
    pr = &prop[sh->prop_count++];
    pr->atom = JS_DupAtom(ctx, atom);
    pr->flags = prop_flags;
    sh->has_small_array_index |= __JS_AtomIsTaggedInt(atom);
    /* add in hash table */
    hash_mask = sh->prop_hash_mask;
    h = atom & hash_mask;
    pr->hash_next = prop_hash_end(sh)[-h - 1];
    prop_hash_end(sh)[-h - 1] = sh->prop_count;
    return 0;
}

/* find a hashed empty shape matching the prototype. Return NULL if
   not found */
static JSShape *find_hashed_shape_proto(JSRuntime *rt, JSObject *proto)
{
    JSShape *sh1;
    uint32_t h, h1;

    h = shape_initial_hash(proto);
    h1 = get_shape_hash(h, rt->shape_hash_bits);
    for(sh1 = rt->shape_hash[h1]; sh1 != NULL; sh1 = sh1->shape_hash_next) {
        if (sh1->hash == h &&
            sh1->proto == proto &&
            sh1->prop_count == 0) {
            return sh1;
        }
    }
    return NULL;
}

/* find a hashed shape matching sh + (prop, prop_flags). Return NULL if
   not found */
static JSShape *find_hashed_shape_prop(JSRuntime *rt, JSShape *sh,
                                       JSAtom atom, int prop_flags)
{
    JSShape *sh1;
    uint32_t h, h1, i, n;

    h = sh->hash;
    h = shape_hash(h, atom);
    h = shape_hash(h, prop_flags);
    h1 = get_shape_hash(h, rt->shape_hash_bits);
    for(sh1 = rt->shape_hash[h1]; sh1 != NULL; sh1 = sh1->shape_hash_next) {
        /* we test the hash first so that the rest is done only if the
           shapes really match */
        if (sh1->hash == h &&
            sh1->proto == sh->proto &&
            sh1->prop_count == ((n = sh->prop_count) + 1)) {
            for(i = 0; i < n; i++) {
                if (unlikely(sh1->prop[i].atom != sh->prop[i].atom) ||
                    unlikely(sh1->prop[i].flags != sh->prop[i].flags))
                    goto next;
            }
            if (unlikely(sh1->prop[n].atom != atom) ||
                unlikely(sh1->prop[n].flags != prop_flags))
                goto next;
            return sh1;
        }
    next: ;
    }
    return NULL;
}

static __maybe_unused void JS_DumpShape(JSRuntime *rt, int i, JSShape *sh)
{
    char atom_buf[ATOM_GET_STR_BUF_SIZE];
    int j;

    /* XXX: should output readable class prototype */
    printf("%5d %3d%c %14p %5d %5d", i,
           sh->header.ref_count, " *"[sh->is_hashed],
           (void *)sh->proto, sh->prop_size, sh->prop_count);
    for(j = 0; j < sh->prop_count; j++) {
        printf(" %s", JS_AtomGetStrRT(rt, atom_buf, sizeof(atom_buf),
                                      sh->prop[j].atom));
    }
    printf("\n");
}

static __maybe_unused void JS_DumpShapes(JSRuntime *rt)
{
    int i;
    JSShape *sh;
    struct list_head *el;
    JSObject *p;
    JSGCObjectHeader *gp;

    printf("JSShapes: {\n");
    printf("%5s %4s %14s %5s %5s %s\n", "SLOT", "REFS", "PROTO", "SIZE", "COUNT", "PROPS");
    for(i = 0; i < rt->shape_hash_size; i++) {
        for(sh = rt->shape_hash[i]; sh != NULL; sh = sh->shape_hash_next) {
            JS_DumpShape(rt, i, sh);
            assert(sh->is_hashed);
        }
    }
    /* dump non-hashed shapes */
    list_for_each(el, &rt->gc_obj_list) {
        gp = list_entry(el, JSGCObjectHeader, link);
        if (gp->gc_obj_type == JS_GC_OBJ_TYPE_JS_OBJECT) {
            p = (JSObject *)gp;
            if (!p->shape->is_hashed) {
                JS_DumpShape(rt, -1, p->shape);
            }
        }
    }
    printf("}\n");
}

static JSValue JS_NewObjectFromShape(JSContext *ctx, JSShape *sh, JSClassID class_id)
{
    JSObject *p;

    js_trigger_gc(ctx->rt, sizeof(JSObject));
    p = js_malloc(ctx, sizeof(JSObject));
    if (unlikely(!p))
        goto fail;
    p->class_id = class_id;
    p->extensible = true;
    p->free_mark = 0;
    p->is_exotic = 0;
    p->fast_array = 0;
    p->is_constructor = 0;
    p->is_uncatchable_error = 0;
    p->tmp_mark = 0;
    p->is_HTMLDDA = 0;
    p->first_weak_ref = NULL;
    p->u.opaque = NULL;
    p->shape = sh;
    p->prop = js_malloc(ctx, sizeof(JSProperty) * sh->prop_size);
    if (unlikely(!p->prop)) {
        js_free(ctx, p);
    fail:
        js_free_shape(ctx->rt, sh);
        return JS_EXCEPTION;
    }

    switch(class_id) {
    case JS_CLASS_OBJECT:
        break;
    case JS_CLASS_ARRAY:
        {
            JSProperty *pr;
            p->is_exotic = 1;
            p->fast_array = 1;
            p->u.array.u.values = NULL;
            p->u.array.count = 0;
            p->u.array.u1.size = 0;
            /* the length property is always the first one */
            if (likely(sh == ctx->array_shape)) {
                pr = &p->prop[0];
            } else {
                /* only used for the first array */
                /* cannot fail */
                pr = add_property(ctx, p, JS_ATOM_length,
                                  JS_PROP_WRITABLE | JS_PROP_LENGTH);
            }
            pr->u.value = js_int32(0);
        }
        break;
    case JS_CLASS_C_FUNCTION:
        p->prop[0].u.value = JS_UNDEFINED;
        break;
    case JS_CLASS_ARGUMENTS:
    case JS_CLASS_UINT8C_ARRAY:
    case JS_CLASS_INT8_ARRAY:
    case JS_CLASS_UINT8_ARRAY:
    case JS_CLASS_INT16_ARRAY:
    case JS_CLASS_UINT16_ARRAY:
    case JS_CLASS_INT32_ARRAY:
    case JS_CLASS_UINT32_ARRAY:
    case JS_CLASS_BIG_INT64_ARRAY:
    case JS_CLASS_BIG_UINT64_ARRAY:
    case JS_CLASS_FLOAT16_ARRAY:
    case JS_CLASS_FLOAT32_ARRAY:
    case JS_CLASS_FLOAT64_ARRAY:
        p->is_exotic = 1;
        p->fast_array = 1;
        p->u.array.u.ptr = NULL;
        p->u.array.count = 0;
        break;
    case JS_CLASS_DATAVIEW:
        p->u.array.u.ptr = NULL;
        p->u.array.count = 0;
        break;
    case JS_CLASS_NUMBER:
    case JS_CLASS_STRING:
    case JS_CLASS_BOOLEAN:
    case JS_CLASS_SYMBOL:
    case JS_CLASS_DATE:
    case JS_CLASS_BIG_INT:
        p->u.object_data = JS_UNDEFINED;
        goto set_exotic;
    case JS_CLASS_REGEXP:
        p->u.regexp.pattern = NULL;
        p->u.regexp.bytecode = NULL;
        goto set_exotic;
    default:
    set_exotic:
        if (ctx->rt->class_array[class_id].exotic) {
            p->is_exotic = 1;
        }
        break;
    }
    p->header.ref_count = 1;
    add_gc_object(ctx->rt, &p->header, JS_GC_OBJ_TYPE_JS_OBJECT);
    return JS_MKPTR(JS_TAG_OBJECT, p);
}

static JSObject *get_proto_obj(JSValueConst proto_val)
{
    if (JS_VALUE_GET_TAG(proto_val) != JS_TAG_OBJECT)
        return NULL;
    else
        return JS_VALUE_GET_OBJ(proto_val);
}

/* WARNING: proto must be an object or JS_NULL */
JSValue JS_NewObjectProtoClass(JSContext *ctx, JSValueConst proto_val,
                               JSClassID class_id)
{
    JSShape *sh;
    JSObject *proto;

    proto = get_proto_obj(proto_val);
    sh = find_hashed_shape_proto(ctx->rt, proto);
    if (likely(sh)) {
        sh = js_dup_shape(sh);
    } else {
        sh = js_new_shape(ctx, proto);
        if (!sh)
            return JS_EXCEPTION;
    }
    return JS_NewObjectFromShape(ctx, sh, class_id);
}

static int JS_SetObjectData(JSContext *ctx, JSValueConst obj, JSValue val)
{
    JSObject *p;

    if (JS_VALUE_GET_TAG(obj) == JS_TAG_OBJECT) {
        p = JS_VALUE_GET_OBJ(obj);
        switch(p->class_id) {
        case JS_CLASS_NUMBER:
        case JS_CLASS_STRING:
        case JS_CLASS_BOOLEAN:
        case JS_CLASS_SYMBOL:
        case JS_CLASS_DATE:
        case JS_CLASS_BIG_INT:
            JS_FreeValue(ctx, p->u.object_data);
            p->u.object_data = val;
            return 0;
        }
    }
    JS_FreeValue(ctx, val);
    if (!JS_IsException(obj))
        JS_ThrowTypeError(ctx, "invalid object type");
    return -1;
}

JSValue JS_NewObjectClass(JSContext *ctx, JSClassID class_id)
{
    return JS_NewObjectProtoClass(ctx, ctx->class_proto[class_id], class_id);
}

JSValue JS_NewObjectProto(JSContext *ctx, JSValueConst proto)
{
    return JS_NewObjectProtoClass(ctx, proto, JS_CLASS_OBJECT);
}

JSValue JS_NewObjectFrom(JSContext *ctx, int count, const JSAtom *props,
                         const JSValue *values)
{
    JSShapeProperty *pr;
    uint32_t *hash;
    JSRuntime *rt;
    JSObject *p;
    JSShape *sh;
    JSValue obj;
    JSAtom atom;
    intptr_t h;
    int i;

    rt = ctx->rt;
    obj = JS_NewObject(ctx);
    if (JS_IsException(obj))
        return JS_EXCEPTION;
    if (count > 0) {
        p = JS_VALUE_GET_OBJ(obj);
        sh = p->shape;
        assert(sh->is_hashed);
        assert(sh->header.ref_count == 1);
        js_shape_hash_unlink(rt, sh);
        if (resize_properties(ctx, &sh, p, count)) {
            js_shape_hash_link(rt, sh);
            JS_FreeValue(ctx, obj);
            return JS_EXCEPTION;
        }
        p->shape = sh;
        for (i = 0; i < count; i++) {
            atom = props[i];
            pr = &sh->prop[i];
            sh->hash = shape_hash(shape_hash(sh->hash, atom), JS_PROP_C_W_E);
            sh->has_small_array_index |= __JS_AtomIsTaggedInt(atom);
            h = atom & sh->prop_hash_mask;
            hash = &prop_hash_end(sh)[-h - 1];
            pr->hash_next = *hash;
            *hash = i + 1;
            pr->atom = JS_DupAtom(ctx, atom);
            pr->flags = JS_PROP_C_W_E;
            p->prop[i].u.value = values[i];
        }
        js_shape_hash_link(rt, sh);
        sh->prop_count = count;
    }
    return obj;
}

JSValue JS_NewObjectFromStr(JSContext *ctx, int count, const char **props,
                            const JSValue *values)
{
    JSAtom atoms_s[16], *atoms = atoms_s;
    JSValue ret;
    int i;

    i = 0;
    ret = JS_EXCEPTION;
    if (count < 1)
        goto out;
    if (count > (int)countof(atoms_s)) {
        atoms = js_malloc(ctx, count * sizeof(*atoms));
        if (!atoms)
            return JS_EXCEPTION;
    }
    for (i = 0; i < count; i++) {
        atoms[i] = JS_NewAtom(ctx, props[i]);
        if (atoms[i] == JS_ATOM_NULL)
            goto out;
    }
    ret = JS_NewObjectFrom(ctx, count, atoms, values);
out:
    while (i-- > 0)
        JS_FreeAtom(ctx, atoms[i]);
    if (atoms != atoms_s)
        js_free(ctx, atoms);
    return ret;
}

JSValue JS_NewArray(JSContext *ctx)
{
    return JS_NewObjectFromShape(ctx, js_dup_shape(ctx->array_shape),
                                 JS_CLASS_ARRAY);
}

// note: takes ownership of |values|, unlike js_create_array
JSValue JS_NewArrayFrom(JSContext *ctx, int count, const JSValue *values)
{
    JSObject *p;
    JSValue obj;
    int i;

    obj = JS_NewArray(ctx);
    if (JS_IsException(obj))
        goto exception;
    if (count > 0) {
        p = JS_VALUE_GET_OBJ(obj);
        if (expand_fast_array(ctx, p, count)) {
            JS_FreeValue(ctx, obj);
            goto exception;
        }
        p->u.array.count = count;
        p->prop[0].u.value = js_int32(count);
        memcpy(p->u.array.u.values, values, count * sizeof(*values));
    }
    return obj;
exception:
    for (i = 0; i < count; i++)
        JS_FreeValue(ctx, values[i]);
    return JS_EXCEPTION;
}

JSValue JS_NewObject(JSContext *ctx)
{
    /* inline JS_NewObjectClass(ctx, JS_CLASS_OBJECT); */
    return JS_NewObjectProtoClass(ctx, ctx->class_proto[JS_CLASS_OBJECT], JS_CLASS_OBJECT);
}

static void js_function_set_properties(JSContext *ctx, JSValue func_obj,
                                       JSAtom name, int len)
{
    /* ES6 feature non compatible with ES5.1: length is configurable */
    JS_DefinePropertyValue(ctx, func_obj, JS_ATOM_length, js_int32(len),
                           JS_PROP_CONFIGURABLE);
    JS_DefinePropertyValue(ctx, func_obj, JS_ATOM_name,
                           JS_AtomToString(ctx, name), JS_PROP_CONFIGURABLE);
}

static bool js_class_has_bytecode(JSClassID class_id)
{
    return (class_id == JS_CLASS_BYTECODE_FUNCTION ||
            class_id == JS_CLASS_GENERATOR_FUNCTION ||
            class_id == JS_CLASS_ASYNC_FUNCTION ||
            class_id == JS_CLASS_ASYNC_GENERATOR_FUNCTION);
}

/* return NULL without exception if not a function or no bytecode */
static JSFunctionBytecode *JS_GetFunctionBytecode(JSValueConst val)
{
    JSObject *p;
    if (JS_VALUE_GET_TAG(val) != JS_TAG_OBJECT)
        return NULL;
    p = JS_VALUE_GET_OBJ(val);
    if (!js_class_has_bytecode(p->class_id))
        return NULL;
    return p->u.func.function_bytecode;
}

static void js_method_set_home_object(JSContext *ctx, JSValue func_obj,
                                      JSValue home_obj)
{
    JSObject *p, *p1;
    JSFunctionBytecode *b;

    if (JS_VALUE_GET_TAG(func_obj) != JS_TAG_OBJECT)
        return;
    p = JS_VALUE_GET_OBJ(func_obj);
    if (!js_class_has_bytecode(p->class_id))
        return;
    b = p->u.func.function_bytecode;
    if (b->need_home_object) {
        p1 = p->u.func.home_object;
        if (p1) {
            JS_FreeValue(ctx, JS_MKPTR(JS_TAG_OBJECT, p1));
        }
        if (JS_VALUE_GET_TAG(home_obj) == JS_TAG_OBJECT)
            p1 = JS_VALUE_GET_OBJ(js_dup(home_obj));
        else
            p1 = NULL;
        p->u.func.home_object = p1;
    }
}

static JSValue js_get_function_name(JSContext *ctx, JSAtom name)
{
    JSValue name_str;

    name_str = JS_AtomToString(ctx, name);
    if (JS_AtomSymbolHasDescription(ctx, name)) {
        name_str = JS_ConcatString3(ctx, "[", name_str, "]");
    }
    return name_str;
}

/* Modify the name of a method according to the atom and
   'flags'. 'flags' is a bitmask of JS_PROP_HAS_GET and
   JS_PROP_HAS_SET. Also set the home object of the method.
   Return < 0 if exception. */
static int js_method_set_properties(JSContext *ctx, JSValue func_obj,
                                    JSAtom name, int flags, JSValue home_obj)
{
    JSValue name_str;

    name_str = js_get_function_name(ctx, name);
    if (flags & JS_PROP_HAS_GET) {
        name_str = JS_ConcatString3(ctx, "get ", name_str, "");
    } else if (flags & JS_PROP_HAS_SET) {
        name_str = JS_ConcatString3(ctx, "set ", name_str, "");
    }
    if (JS_IsException(name_str))
        return -1;
    if (JS_DefinePropertyValue(ctx, func_obj, JS_ATOM_name, name_str,
                               JS_PROP_CONFIGURABLE) < 0)
        return -1;
    js_method_set_home_object(ctx, func_obj, home_obj);
    return 0;
}

/* Note: at least 'length' arguments will be readable in 'argv' */
/* `name` may be NULL, pure ASCII or UTF-8 encoded */
JSValue JS_NewCFunction3(JSContext *ctx, JSCFunction *func,
                         const char *name,
                         int length, JSCFunctionEnum cproto, int magic,
                         JSValueConst proto_val)
{
    JSValue func_obj;
    JSObject *p;
    JSAtom name_atom;

    func_obj = JS_NewObjectProtoClass(ctx, proto_val, JS_CLASS_C_FUNCTION);
    if (JS_IsException(func_obj))
        return func_obj;
    p = JS_VALUE_GET_OBJ(func_obj);
    p->u.cfunc.realm = JS_DupContext(ctx);
    p->u.cfunc.c_function.generic = func;
    p->u.cfunc.length = length;
    p->u.cfunc.cproto = cproto;
    p->u.cfunc.magic = magic;
    p->is_constructor = (cproto == JS_CFUNC_constructor ||
                         cproto == JS_CFUNC_constructor_magic ||
                         cproto == JS_CFUNC_constructor_or_func ||
                         cproto == JS_CFUNC_constructor_or_func_magic);
    name_atom = JS_ATOM_empty_string;
    if (name && *name) {
        name_atom = JS_NewAtom(ctx, name);
        if (name_atom == JS_ATOM_NULL) {
            JS_FreeValue(ctx, func_obj);
            return JS_EXCEPTION;
        }
    }
    js_function_set_properties(ctx, func_obj, name_atom, length);
    JS_FreeAtom(ctx, name_atom);
    return func_obj;
}

/* Note: at least 'length' arguments will be readable in 'argv' */
JSValue JS_NewCFunction2(JSContext *ctx, JSCFunction *func,
                         const char *name,
                         int length, JSCFunctionEnum cproto, int magic)
{
    return JS_NewCFunction3(ctx, func, name, length, cproto, magic,
                            ctx->function_proto);
}

typedef struct JSCFunctionDataRecord {
    JSCFunctionData *func;
    uint8_t length;
    uint8_t data_len;
    uint16_t magic;
    JSValue data[];
} JSCFunctionDataRecord;

static void js_c_function_data_finalizer(JSRuntime *rt, JSValueConst val)
{
    JSCFunctionDataRecord *s = JS_GetOpaque(val, JS_CLASS_C_FUNCTION_DATA);
    int i;

    if (s) {
        for(i = 0; i < s->data_len; i++) {
            JS_FreeValueRT(rt, s->data[i]);
        }
        js_free_rt(rt, s);
    }
}

static void js_c_function_data_mark(JSRuntime *rt, JSValueConst val,
                                    JS_MarkFunc *mark_func)
{
    JSCFunctionDataRecord *s = JS_GetOpaque(val, JS_CLASS_C_FUNCTION_DATA);
    int i;

    if (s) {
        for(i = 0; i < s->data_len; i++) {
            JS_MarkValue(rt, s->data[i], mark_func);
        }
    }
}

static JSValue js_call_c_function_data(JSContext *ctx, JSValueConst func_obj,
                                       JSValueConst this_val,
                                       int argc, JSValueConst *argv, int flags)
{
    JSRuntime *rt = ctx->rt;
    JSStackFrame sf_s, *sf = &sf_s, *prev_sf;
    JSCFunctionDataRecord *s;
    JSValueConst *arg_buf;
    JSValue ret;
    size_t stack_size;
    int arg_count;
    int i;

    s = JS_GetOpaque(func_obj, JS_CLASS_C_FUNCTION_DATA);
    if (!s)
        return JS_EXCEPTION; // can't really happen
    arg_buf = argv;
    arg_count = s->length;
    if (unlikely(argc < arg_count)) {
        stack_size = arg_count * sizeof(arg_buf[0]);
        if (js_check_stack_overflow(rt, stack_size))
            return JS_ThrowStackOverflow(ctx);
        arg_buf = alloca(stack_size);
        for(i = 0; i < argc; i++)
            arg_buf[i] = argv[i];
        for(i = argc; i < arg_count; i++)
            arg_buf[i] = JS_UNDEFINED;
    }
    prev_sf = rt->current_stack_frame;
    sf->prev_frame = prev_sf;
    rt->current_stack_frame = sf;
    // TODO(bnoordhuis) switch realms like js_call_c_function does
    sf->is_strict_mode = false;
    sf->cur_func = unsafe_unconst(func_obj);
    sf->arg_count = argc;
    ret = s->func(ctx, this_val, argc, arg_buf, s->magic, vc(s->data));
    rt->current_stack_frame = sf->prev_frame;
    return ret;
}

JSValue JS_NewCFunctionData2(JSContext *ctx, JSCFunctionData *func,
                             const char *name,
                             int length, int magic, int data_len,
                             JSValueConst *data)
{
    JSCFunctionDataRecord *s;
    JSAtom name_atom;
    JSValue func_obj;
    int i;

    func_obj = JS_NewObjectProtoClass(ctx, ctx->function_proto,
                                      JS_CLASS_C_FUNCTION_DATA);
    if (JS_IsException(func_obj))
        return func_obj;
    s = js_malloc(ctx, sizeof(*s) + data_len * sizeof(JSValue));
    if (!s) {
        JS_FreeValue(ctx, func_obj);
        return JS_EXCEPTION;
    }
    s->func = func;
    s->length = length;
    s->data_len = data_len;
    s->magic = magic;
    for(i = 0; i < data_len; i++)
        s->data[i] = js_dup(data[i]);
    JS_SetOpaqueInternal(func_obj, s);
    name_atom = JS_ATOM_empty_string;
    if (name && *name) {
        name_atom = JS_NewAtom(ctx, name);
        if (name_atom == JS_ATOM_NULL) {
            JS_FreeValue(ctx, func_obj);
            return JS_EXCEPTION;
        }
    }
    js_function_set_properties(ctx, func_obj, name_atom, length);
    JS_FreeAtom(ctx, name_atom);
    return func_obj;
}

JSValue JS_NewCFunctionData(JSContext *ctx, JSCFunctionData *func,
                            int length, int magic, int data_len,
                            JSValueConst *data)
{
    return JS_NewCFunctionData2(ctx, func, NULL, length, magic, data_len, data);
}

static JSContext *js_autoinit_get_realm(JSProperty *pr)
{
    return (JSContext *)(pr->u.init.realm_and_id & ~3);
}

static JSAutoInitIDEnum js_autoinit_get_id(JSProperty *pr)
{
    return pr->u.init.realm_and_id & 3;
}

static void js_autoinit_free(JSRuntime *rt, JSProperty *pr)
{
    JS_FreeContext(js_autoinit_get_realm(pr));
}

static void js_autoinit_mark(JSRuntime *rt, JSProperty *pr,
                             JS_MarkFunc *mark_func)
{
    mark_func(rt, &js_autoinit_get_realm(pr)->header);
}

static void free_property(JSRuntime *rt, JSProperty *pr, int prop_flags)
{
    if (unlikely(prop_flags & JS_PROP_TMASK)) {
        if ((prop_flags & JS_PROP_TMASK) == JS_PROP_GETSET) {
            if (pr->u.getset.getter)
                JS_FreeValueRT(rt, JS_MKPTR(JS_TAG_OBJECT, pr->u.getset.getter));
            if (pr->u.getset.setter)
                JS_FreeValueRT(rt, JS_MKPTR(JS_TAG_OBJECT, pr->u.getset.setter));
        } else if ((prop_flags & JS_PROP_TMASK) == JS_PROP_VARREF) {
            free_var_ref(rt, pr->u.var_ref);
        } else if ((prop_flags & JS_PROP_TMASK) == JS_PROP_AUTOINIT) {
            js_autoinit_free(rt, pr);
        }
    } else {
        JS_FreeValueRT(rt, pr->u.value);
    }
}

static force_inline JSShapeProperty *find_own_property1(JSObject *p,
                                                        JSAtom atom)
{
    JSShape *sh;
    JSShapeProperty *pr, *prop;
    intptr_t h;
    sh = p->shape;
    h = (uintptr_t)atom & sh->prop_hash_mask;
    h = prop_hash_end(sh)[-h - 1];
    prop = get_shape_prop(sh);
    while (h) {
        pr = &prop[h - 1];
        if (likely(pr->atom == atom)) {
            return pr;
        }
        h = pr->hash_next;
    }
    return NULL;
}

static force_inline JSShapeProperty *find_own_property(JSProperty **ppr,
                                                       JSObject *p,
                                                       JSAtom atom)
{
    JSShape *sh;
    JSShapeProperty *pr, *prop;
    intptr_t h;
    sh = p->shape;
    h = (uintptr_t)atom & sh->prop_hash_mask;
    h = prop_hash_end(sh)[-h - 1];
    prop = get_shape_prop(sh);
    while (h) {
        pr = &prop[h - 1];
        if (likely(pr->atom == atom)) {
            *ppr = &p->prop[h - 1];
            /* the compiler should be able to assume that pr != NULL here */
            return pr;
        }
        h = pr->hash_next;
    }
    *ppr = NULL;
    return NULL;
}

static void free_var_ref(JSRuntime *rt, JSVarRef *var_ref)
{
    if (var_ref) {
        assert(var_ref->header.ref_count > 0);
        if (--var_ref->header.ref_count == 0) {
            if (var_ref->is_detached) {
                JS_FreeValueRT(rt, var_ref->value);
                remove_gc_object(&var_ref->header);
            } else {
                list_del(&var_ref->header.link); /* still on the stack */
            }
            js_free_rt(rt, var_ref);
        }
    }
}

static void js_array_finalizer(JSRuntime *rt, JSValueConst val)
{
    JSObject *p = JS_VALUE_GET_OBJ(val);
    int i;

    for(i = 0; i < p->u.array.count; i++) {
        JS_FreeValueRT(rt, p->u.array.u.values[i]);
    }
    js_free_rt(rt, p->u.array.u.values);
}

static void js_array_mark(JSRuntime *rt, JSValueConst val,
                          JS_MarkFunc *mark_func)
{
    JSObject *p = JS_VALUE_GET_OBJ(val);
    int i;

    for(i = 0; i < p->u.array.count; i++) {
        JS_MarkValue(rt, p->u.array.u.values[i], mark_func);
    }
}

static void js_object_data_finalizer(JSRuntime *rt, JSValueConst val)
{
    JSObject *p = JS_VALUE_GET_OBJ(val);
    JS_FreeValueRT(rt, p->u.object_data);
    p->u.object_data = JS_UNDEFINED;
}

static void js_object_data_mark(JSRuntime *rt, JSValueConst val,
                                JS_MarkFunc *mark_func)
{
    JSObject *p = JS_VALUE_GET_OBJ(val);
    JS_MarkValue(rt, p->u.object_data, mark_func);
}

static void js_c_function_finalizer(JSRuntime *rt, JSValueConst val)
{
    JSObject *p = JS_VALUE_GET_OBJ(val);

    if (p->u.cfunc.realm)
        JS_FreeContext(p->u.cfunc.realm);
}

static void js_c_function_mark(JSRuntime *rt, JSValueConst val,
                               JS_MarkFunc *mark_func)
{
    JSObject *p = JS_VALUE_GET_OBJ(val);

    if (p->u.cfunc.realm)
        mark_func(rt, &p->u.cfunc.realm->header);
}

static void js_bytecode_function_finalizer(JSRuntime *rt, JSValueConst val)
{
    JSObject *p1, *p = JS_VALUE_GET_OBJ(val);
    JSFunctionBytecode *b;
    JSVarRef **var_refs;
    int i;

    p1 = p->u.func.home_object;
    if (p1) {
        JS_FreeValueRT(rt, JS_MKPTR(JS_TAG_OBJECT, p1));
    }
    b = p->u.func.function_bytecode;
    if (b) {
        var_refs = p->u.func.var_refs;
        if (var_refs) {
            for(i = 0; i < b->closure_var_count; i++)
                free_var_ref(rt, var_refs[i]);
            js_free_rt(rt, var_refs);
        }
        JS_FreeValueRT(rt, JS_MKPTR(JS_TAG_FUNCTION_BYTECODE, b));
    }
}

static void js_bytecode_function_mark(JSRuntime *rt, JSValueConst val,
                                      JS_MarkFunc *mark_func)
{
    JSObject *p = JS_VALUE_GET_OBJ(val);
    JSVarRef **var_refs = p->u.func.var_refs;
    JSFunctionBytecode *b = p->u.func.function_bytecode;
    int i;

    if (p->u.func.home_object) {
        JS_MarkValue(rt, JS_MKPTR(JS_TAG_OBJECT, p->u.func.home_object),
                     mark_func);
    }
    if (b) {
        if (var_refs) {
            for(i = 0; i < b->closure_var_count; i++) {
                JSVarRef *var_ref = var_refs[i];
                if (var_ref && var_ref->is_detached) {
                    mark_func(rt, &var_ref->header);
                }
            }
        }
        /* must mark the function bytecode because template objects may be
           part of a cycle */
        JS_MarkValue(rt, JS_MKPTR(JS_TAG_FUNCTION_BYTECODE, b), mark_func);
    }
}

static void js_bound_function_finalizer(JSRuntime *rt, JSValueConst val)
{
    JSObject *p = JS_VALUE_GET_OBJ(val);
    JSBoundFunction *bf = p->u.bound_function;
    int i;

    JS_FreeValueRT(rt, bf->func_obj);
    JS_FreeValueRT(rt, bf->this_val);
    for(i = 0; i < bf->argc; i++) {
        JS_FreeValueRT(rt, bf->argv[i]);
    }
    js_free_rt(rt, bf);
}

static void js_bound_function_mark(JSRuntime *rt, JSValueConst val,
                                JS_MarkFunc *mark_func)
{
    JSObject *p = JS_VALUE_GET_OBJ(val);
    JSBoundFunction *bf = p->u.bound_function;
    int i;

    JS_MarkValue(rt, bf->func_obj, mark_func);
    JS_MarkValue(rt, bf->this_val, mark_func);
    for(i = 0; i < bf->argc; i++)
        JS_MarkValue(rt, bf->argv[i], mark_func);
}

static void js_for_in_iterator_finalizer(JSRuntime *rt, JSValueConst val)
{
    JSObject *p = JS_VALUE_GET_OBJ(val);
    JSForInIterator *it = p->u.for_in_iterator;
    JS_FreeValueRT(rt, it->obj);
    js_free_rt(rt, it);
}

static void js_for_in_iterator_mark(JSRuntime *rt, JSValueConst val,
                                JS_MarkFunc *mark_func)
{
    JSObject *p = JS_VALUE_GET_OBJ(val);
    JSForInIterator *it = p->u.for_in_iterator;
    JS_MarkValue(rt, it->obj, mark_func);
}

static void free_object(JSRuntime *rt, JSObject *p)
{
    int i;
    JSClassFinalizer *finalizer;
    JSShape *sh;
    JSShapeProperty *pr;

    p->free_mark = 1; /* used to tell the object is invalid when
                         freeing cycles */
    /* free all the fields */
    sh = p->shape;
    pr = get_shape_prop(sh);
    for(i = 0; i < sh->prop_count; i++) {
        free_property(rt, &p->prop[i], pr->flags);
        pr++;
    }
    js_free_rt(rt, p->prop);
    /* as an optimization we destroy the shape immediately without
       putting it in gc_zero_ref_count_list */
    js_free_shape(rt, sh);

    /* fail safe */
    p->shape = NULL;
    p->prop = NULL;

    if (unlikely(p->first_weak_ref)) {
        reset_weak_ref(rt, &p->first_weak_ref);
    }

    finalizer = rt->class_array[p->class_id].finalizer;
    if (finalizer)
        (*finalizer)(rt, JS_MKPTR(JS_TAG_OBJECT, p));

    /* fail safe */
    p->class_id = 0;
    p->u.opaque = NULL;
    p->u.func.var_refs = NULL;
    p->u.func.home_object = NULL;

    remove_gc_object(&p->header);
    if (rt->gc_phase == JS_GC_PHASE_REMOVE_CYCLES && p->header.ref_count != 0) {
        list_add_tail(&p->header.link, &rt->gc_zero_ref_count_list);
    } else {
        js_free_rt(rt, p);
    }
}

static void free_gc_object(JSRuntime *rt, JSGCObjectHeader *gp)
{
    switch(gp->gc_obj_type) {
    case JS_GC_OBJ_TYPE_JS_OBJECT:
        free_object(rt, (JSObject *)gp);
        break;
    case JS_GC_OBJ_TYPE_FUNCTION_BYTECODE:
        free_function_bytecode(rt, (JSFunctionBytecode *)gp);
        break;
    default:
        abort();
    }
}

static void free_zero_refcount(JSRuntime *rt)
{
    struct list_head *el;
    JSGCObjectHeader *p;

    rt->gc_phase = JS_GC_PHASE_DECREF;
    for(;;) {
        el = rt->gc_zero_ref_count_list.next;
        if (el == &rt->gc_zero_ref_count_list)
            break;
        p = list_entry(el, JSGCObjectHeader, link);
        assert(p->ref_count == 0);
        free_gc_object(rt, p);
    }
    rt->gc_phase = JS_GC_PHASE_NONE;
}

/* called with the ref_count of 'v' reaches zero. */
static void js_free_value_rt(JSRuntime *rt, JSValue v)
{
    uint32_t tag = JS_VALUE_GET_TAG(v);

#ifdef ENABLE_DUMPS // JS_DUMP_FREE
    if (check_dump_flag(rt, JS_DUMP_FREE)) {
        /* Prevent invalid object access during GC */
        if ((rt->gc_phase != JS_GC_PHASE_REMOVE_CYCLES)
        ||  (tag != JS_TAG_OBJECT && tag != JS_TAG_FUNCTION_BYTECODE)) {
            printf("Freeing ");
            if (tag == JS_TAG_OBJECT) {
                JS_DumpObject(rt, JS_VALUE_GET_OBJ(v));
            } else {
                JS_DumpValue(rt, v);
                printf("\n");
            }
        }
    }
#endif

    switch(tag) {
    case JS_TAG_STRING:
        js_free_string0(rt, JS_VALUE_GET_STRING(v));
        break;
    case JS_TAG_OBJECT:
    case JS_TAG_FUNCTION_BYTECODE:
        {
            JSGCObjectHeader *p = JS_VALUE_GET_PTR(v);
            if (rt->gc_phase != JS_GC_PHASE_REMOVE_CYCLES) {
                list_del(&p->link);
                list_add(&p->link, &rt->gc_zero_ref_count_list);
                if (rt->gc_phase == JS_GC_PHASE_NONE) {
                    free_zero_refcount(rt);
                }
            }
        }
        break;
    case JS_TAG_MODULE:
        abort(); /* never freed here */
        break;
    case JS_TAG_BIG_INT:
        {
            JSBigInt *p = JS_VALUE_GET_PTR(v);
            js_free_rt(rt, p);
        }
        break;
    case JS_TAG_SYMBOL:
        {
            JSAtomStruct *p = JS_VALUE_GET_PTR(v);
            JS_FreeAtomStruct(rt, p);
        }
        break;
    default:
        printf("js_free_value_rt: unknown tag=%d\n", tag);
        abort();
    }
}

void JS_FreeValueRT(JSRuntime *rt, JSValue v)
{
    if (JS_VALUE_HAS_REF_COUNT(v)) {
        JSRefCountHeader *p = (JSRefCountHeader *)JS_VALUE_GET_PTR(v);
        if (--p->ref_count <= 0) {
            js_free_value_rt(rt, v);
        }
    }
}

void JS_FreeValue(JSContext *ctx, JSValue v)
{
    JS_FreeValueRT(ctx->rt, v);
}

/* garbage collection */

static void add_gc_object(JSRuntime *rt, JSGCObjectHeader *h,
                          JSGCObjectTypeEnum type)
{
    h->mark = 0;
    h->gc_obj_type = type;
    list_add_tail(&h->link, &rt->gc_obj_list);
}

static void remove_gc_object(JSGCObjectHeader *h)
{
    list_del(&h->link);
}

void JS_MarkValue(JSRuntime *rt, JSValueConst val, JS_MarkFunc *mark_func)
{
    if (JS_VALUE_HAS_REF_COUNT(val)) {
        switch(JS_VALUE_GET_TAG(val)) {
        case JS_TAG_OBJECT:
        case JS_TAG_FUNCTION_BYTECODE:
            mark_func(rt, JS_VALUE_GET_PTR(val));
            break;
        default:
            break;
        }
    }
}

static void mark_weak_map_value(JSRuntime *rt, JSWeakRefRecord *first_weak_ref, JS_MarkFunc *mark_func) {
    JSWeakRefRecord *wr;
    JSMapRecord *mr;
    JSMapState *s;

    for (wr = first_weak_ref; wr != NULL; wr = wr->next_weak_ref) {
        if (wr->kind == JS_WEAK_REF_KIND_MAP) {
            mr = wr->u.map_record;
            s = mr->map;
            assert(s->is_weak);
            assert(!mr->empty); /* no iterator on WeakMap/WeakSet */
            JS_MarkValue(rt, mr->value, mark_func);
        }
    }
}

static void mark_children(JSRuntime *rt, JSGCObjectHeader *gp,
                          JS_MarkFunc *mark_func)
{
    switch(gp->gc_obj_type) {
    case JS_GC_OBJ_TYPE_JS_OBJECT:
        {
            JSObject *p = (JSObject *)gp;
            JSShapeProperty *prs;
            JSShape *sh;
            int i;
            sh = p->shape;
            mark_func(rt, &sh->header);
            /* mark all the fields */
            prs = get_shape_prop(sh);
            for(i = 0; i < sh->prop_count; i++) {
                JSProperty *pr = &p->prop[i];
                if (prs->atom != JS_ATOM_NULL) {
                    if (prs->flags & JS_PROP_TMASK) {
                        if ((prs->flags & JS_PROP_TMASK) == JS_PROP_GETSET) {
                            if (pr->u.getset.getter)
                                mark_func(rt, &pr->u.getset.getter->header);
                            if (pr->u.getset.setter)
                                mark_func(rt, &pr->u.getset.setter->header);
                        } else if ((prs->flags & JS_PROP_TMASK) == JS_PROP_VARREF) {
                            if (pr->u.var_ref->is_detached) {
                                /* Note: the tag does not matter
                                   provided it is a GC object */
                                mark_func(rt, &pr->u.var_ref->header);
                            }
                        } else if ((prs->flags & JS_PROP_TMASK) == JS_PROP_AUTOINIT) {
                            js_autoinit_mark(rt, pr, mark_func);
                        }
                    } else {
                        JS_MarkValue(rt, pr->u.value, mark_func);
                    }
                }
                prs++;
            }

            if (unlikely(p->first_weak_ref)) {
                mark_weak_map_value(rt, p->first_weak_ref, mark_func);
            }

            if (p->class_id != JS_CLASS_OBJECT) {
                JSClassGCMark *gc_mark;
                gc_mark = rt->class_array[p->class_id].gc_mark;
                if (gc_mark)
                    gc_mark(rt, JS_MKPTR(JS_TAG_OBJECT, p), mark_func);
            }
        }
        break;
    case JS_GC_OBJ_TYPE_FUNCTION_BYTECODE:
        /* the template objects can be part of a cycle */
        {
            JSFunctionBytecode *b = (JSFunctionBytecode *)gp;
            int i;
            for(i = 0; i < b->cpool_count; i++) {
                JS_MarkValue(rt, b->cpool[i], mark_func);
            }
            if (b->realm)
                mark_func(rt, &b->realm->header);
        }
        break;
    case JS_GC_OBJ_TYPE_VAR_REF:
        {
            JSVarRef *var_ref = (JSVarRef *)gp;
            /* only detached variable referenced are taken into account */
            assert(var_ref->is_detached);
            JS_MarkValue(rt, *var_ref->pvalue, mark_func);
        }
        break;
    case JS_GC_OBJ_TYPE_ASYNC_FUNCTION:
        {
            JSAsyncFunctionData *s = (JSAsyncFunctionData *)gp;
            if (s->is_active)
                async_func_mark(rt, &s->func_state, mark_func);
            JS_MarkValue(rt, s->resolving_funcs[0], mark_func);
            JS_MarkValue(rt, s->resolving_funcs[1], mark_func);
        }
        break;
    case JS_GC_OBJ_TYPE_SHAPE:
        {
            JSShape *sh = (JSShape *)gp;
            if (sh->proto != NULL) {
                mark_func(rt, &sh->proto->header);
            }
        }
        break;
    case JS_GC_OBJ_TYPE_JS_CONTEXT:
        {
            JSContext *ctx = (JSContext *)gp;
            JS_MarkContext(rt, ctx, mark_func);
        }
        break;
    default:
        abort();
    }
}

static void gc_decref_child(JSRuntime *rt, JSGCObjectHeader *p)
{
    assert(p->ref_count > 0);
    p->ref_count--;
    if (p->ref_count == 0 && p->mark == 1) {
        list_del(&p->link);
        list_add_tail(&p->link, &rt->tmp_obj_list);
    }
}

static void gc_decref(JSRuntime *rt)
{
    struct list_head *el, *el1;
    JSGCObjectHeader *p;

    init_list_head(&rt->tmp_obj_list);

    /* decrement the refcount of all the children of all the GC
       objects and move the GC objects with zero refcount to
       tmp_obj_list */
    list_for_each_safe(el, el1, &rt->gc_obj_list) {
        p = list_entry(el, JSGCObjectHeader, link);
        assert(p->mark == 0);
        mark_children(rt, p, gc_decref_child);
        p->mark = 1;
        if (p->ref_count == 0) {
            list_del(&p->link);
            list_add_tail(&p->link, &rt->tmp_obj_list);
        }
    }
}

static void gc_scan_incref_child(JSRuntime *rt, JSGCObjectHeader *p)
{
    p->ref_count++;
    if (p->ref_count == 1) {
        /* ref_count was 0: remove from tmp_obj_list and add at the
           end of gc_obj_list */
        list_del(&p->link);
        list_add_tail(&p->link, &rt->gc_obj_list);
        p->mark = 0; /* reset the mark for the next GC call */
    }
}

static void gc_scan_incref_child2(JSRuntime *rt, JSGCObjectHeader *p)
{
    p->ref_count++;
}

static void gc_scan(JSRuntime *rt)
{
    struct list_head *el;
    JSGCObjectHeader *p;

    /* keep the objects with a refcount > 0 and their children. */
    list_for_each(el, &rt->gc_obj_list) {
        p = list_entry(el, JSGCObjectHeader, link);
        assert(p->ref_count > 0);
        p->mark = 0; /* reset the mark for the next GC call */
        mark_children(rt, p, gc_scan_incref_child);
    }

    /* restore the refcount of the objects to be deleted. */
    list_for_each(el, &rt->tmp_obj_list) {
        p = list_entry(el, JSGCObjectHeader, link);
        mark_children(rt, p, gc_scan_incref_child2);
    }
}

static void gc_free_cycles(JSRuntime *rt)
{
    struct list_head *el, *el1;
    JSGCObjectHeader *p;
#ifdef ENABLE_DUMPS // JS_DUMP_GC_FREE
    bool header_done = false;
#endif

    rt->gc_phase = JS_GC_PHASE_REMOVE_CYCLES;

    for(;;) {
        el = rt->tmp_obj_list.next;
        if (el == &rt->tmp_obj_list)
            break;
        p = list_entry(el, JSGCObjectHeader, link);
        /* Only need to free the GC object associated with JS
           values. The rest will be automatically removed because they
           must be referenced by them. */
        switch(p->gc_obj_type) {
        case JS_GC_OBJ_TYPE_JS_OBJECT:
        case JS_GC_OBJ_TYPE_FUNCTION_BYTECODE:
#ifdef ENABLE_DUMPS // JS_DUMP_GC_FREE
            if (check_dump_flag(rt, JS_DUMP_GC_FREE)) {
                if (!header_done) {
                    printf("Freeing cycles:\n");
                    JS_DumpObjectHeader(rt);
                    header_done = true;
                }
                JS_DumpGCObject(rt, p);
            }
#endif
            free_gc_object(rt, p);
            break;
        default:
            list_del(&p->link);
            list_add_tail(&p->link, &rt->gc_zero_ref_count_list);
            break;
        }
    }
    rt->gc_phase = JS_GC_PHASE_NONE;

    list_for_each_safe(el, el1, &rt->gc_zero_ref_count_list) {
        p = list_entry(el, JSGCObjectHeader, link);
        assert(p->gc_obj_type == JS_GC_OBJ_TYPE_JS_OBJECT ||
               p->gc_obj_type == JS_GC_OBJ_TYPE_FUNCTION_BYTECODE);
        js_free_rt(rt, p);
    }

    init_list_head(&rt->gc_zero_ref_count_list);
}

void JS_RunGC(JSRuntime *rt)
{
    /* decrement the reference of the children of each object. mark =
       1 after this pass. */
    gc_decref(rt);

    /* keep the GC objects with a non zero refcount and their childs */
    gc_scan(rt);

    /* free the GC objects in a cycle */
    gc_free_cycles(rt);
}

/* Return false if not an object or if the object has already been
   freed (zombie objects are visible in finalizers when freeing
   cycles). */
bool JS_IsLiveObject(JSRuntime *rt, JSValueConst obj)
{
    JSObject *p;
    if (!JS_IsObject(obj))
        return false;
    p = JS_VALUE_GET_OBJ(obj);
    return !p->free_mark;
}

/* Compute memory used by various object types */
/* XXX: poor man's approach to handling multiply referenced objects */
typedef struct JSMemoryUsage_helper {
    double memory_used_count;
    double str_count;
    double str_size;
    int64_t js_func_count;
    double js_func_size;
    int64_t js_func_code_size;
    int64_t js_func_pc2line_count;
    int64_t js_func_pc2line_size;
} JSMemoryUsage_helper;

static void compute_value_size(JSValue val, JSMemoryUsage_helper *hp);

static void compute_jsstring_size(JSString *str, JSMemoryUsage_helper *hp)
{
    if (!str->atom_type) {  /* atoms are handled separately */
        double s_ref_count = str->header.ref_count;
        hp->str_count += 1 / s_ref_count;
        hp->str_size += ((sizeof(*str) + (str->len << str->is_wide_char) +
                          1 - str->is_wide_char) / s_ref_count);
    }
}

static void compute_bytecode_size(JSFunctionBytecode *b, JSMemoryUsage_helper *hp)
{
    int memory_used_count, js_func_size, i;

    memory_used_count = 0;
    js_func_size = sizeof(*b);
    if (b->vardefs) {
        js_func_size += (b->arg_count + b->var_count) * sizeof(*b->vardefs);
    }
    if (b->cpool) {
        js_func_size += b->cpool_count * sizeof(*b->cpool);
        for (i = 0; i < b->cpool_count; i++) {
            JSValue val = b->cpool[i];
            compute_value_size(val, hp);
        }
    }
    if (b->closure_var) {
        js_func_size += b->closure_var_count * sizeof(*b->closure_var);
    }
    if (b->byte_code_buf) {
        hp->js_func_code_size += b->byte_code_len;
    }
    memory_used_count++;
    js_func_size += b->source_len + 1;
    if (b->pc2line_len) {
        memory_used_count++;
        hp->js_func_pc2line_count += 1;
        hp->js_func_pc2line_size += b->pc2line_len;
    }
    hp->js_func_size += js_func_size;
    hp->js_func_count += 1;
    hp->memory_used_count += memory_used_count;
}

static void compute_value_size(JSValue val, JSMemoryUsage_helper *hp)
{
    switch(JS_VALUE_GET_TAG(val)) {
    case JS_TAG_STRING:
        compute_jsstring_size(JS_VALUE_GET_STRING(val), hp);
        break;
    case JS_TAG_BIG_INT:
        /* should track JSBigInt usage */
        break;
    }
}

void JS_ComputeMemoryUsage(JSRuntime *rt, JSMemoryUsage *s)
{
    struct list_head *el, *el1;
    int i;
    JSMemoryUsage_helper mem = { 0 }, *hp = &mem;

    memset(s, 0, sizeof(*s));
    s->malloc_count = rt->malloc_state.malloc_count;
    s->malloc_size = rt->malloc_state.malloc_size;
    s->malloc_limit = rt->malloc_state.malloc_limit;

    s->memory_used_count = 2; /* rt + rt->class_array */
    s->memory_used_size = sizeof(JSRuntime) + sizeof(JSClass) * rt->class_count;

    list_for_each(el, &rt->context_list) {
        JSContext *ctx = list_entry(el, JSContext, link);
        JSShape *sh = ctx->array_shape;
        s->memory_used_count += 2; /* ctx + ctx->class_proto */
        s->memory_used_size += sizeof(JSContext) +
            sizeof(JSValue) * rt->class_count;
        s->binary_object_count += ctx->binary_object_count;
        s->binary_object_size += ctx->binary_object_size;

        /* the hashed shapes are counted separately */
        if (sh && !sh->is_hashed) {
            int hash_size = sh->prop_hash_mask + 1;
            s->shape_count++;
            s->shape_size += get_shape_size(hash_size, sh->prop_size);
        }
        list_for_each(el1, &ctx->loaded_modules) {
            JSModuleDef *m = list_entry(el1, JSModuleDef, link);
            s->memory_used_count += 1;
            s->memory_used_size += sizeof(*m);
            if (m->req_module_entries) {
                s->memory_used_count += 1;
                s->memory_used_size += m->req_module_entries_count * sizeof(*m->req_module_entries);
            }
            if (m->export_entries) {
                s->memory_used_count += 1;
                s->memory_used_size += m->export_entries_count * sizeof(*m->export_entries);
                for (i = 0; i < m->export_entries_count; i++) {
                    JSExportEntry *me = &m->export_entries[i];
                    if (me->export_type == JS_EXPORT_TYPE_LOCAL && me->u.local.var_ref) {
                        /* potential multiple count */
                        s->memory_used_count += 1;
                        compute_value_size(me->u.local.var_ref->value, hp);
                    }
                }
            }
            if (m->star_export_entries) {
                s->memory_used_count += 1;
                s->memory_used_size += m->star_export_entries_count * sizeof(*m->star_export_entries);
            }
            if (m->import_entries) {
                s->memory_used_count += 1;
                s->memory_used_size += m->import_entries_count * sizeof(*m->import_entries);
            }
            compute_value_size(m->module_ns, hp);
            compute_value_size(m->func_obj, hp);
        }
    }

    list_for_each(el, &rt->gc_obj_list) {
        JSGCObjectHeader *gp = list_entry(el, JSGCObjectHeader, link);
        JSObject *p;
        JSShape *sh;
        JSShapeProperty *prs;

        /* XXX: could count the other GC object types too */
        if (gp->gc_obj_type == JS_GC_OBJ_TYPE_FUNCTION_BYTECODE) {
            compute_bytecode_size((JSFunctionBytecode *)gp, hp);
            continue;
        } else if (gp->gc_obj_type != JS_GC_OBJ_TYPE_JS_OBJECT) {
            continue;
        }
        p = (JSObject *)gp;
        sh = p->shape;
        s->obj_count++;
        if (p->prop) {
            s->memory_used_count++;
            s->prop_size += sh->prop_size * sizeof(*p->prop);
            s->prop_count += sh->prop_count;
            prs = get_shape_prop(sh);
            for(i = 0; i < sh->prop_count; i++) {
                JSProperty *pr = &p->prop[i];
                if (prs->atom != JS_ATOM_NULL && !(prs->flags & JS_PROP_TMASK)) {
                    compute_value_size(pr->u.value, hp);
                }
                prs++;
            }
        }
        /* the hashed shapes are counted separately */
        if (!sh->is_hashed) {
            int hash_size = sh->prop_hash_mask + 1;
            s->shape_count++;
            s->shape_size += get_shape_size(hash_size, sh->prop_size);
        }

        switch(p->class_id) {
        case JS_CLASS_ARRAY:             /* u.array | length */
        case JS_CLASS_ARGUMENTS:         /* u.array | length */
            s->array_count++;
            if (p->fast_array) {
                s->fast_array_count++;
                if (p->u.array.u.values) {
                    s->memory_used_count++;
                    s->memory_used_size += p->u.array.count *
                        sizeof(*p->u.array.u.values);
                    s->fast_array_elements += p->u.array.count;
                    for (i = 0; i < p->u.array.count; i++) {
                        compute_value_size(p->u.array.u.values[i], hp);
                    }
                }
            }
            break;
        case JS_CLASS_NUMBER:            /* u.object_data */
        case JS_CLASS_STRING:            /* u.object_data */
        case JS_CLASS_BOOLEAN:           /* u.object_data */
        case JS_CLASS_SYMBOL:            /* u.object_data */
        case JS_CLASS_DATE:              /* u.object_data */
        case JS_CLASS_BIG_INT:           /* u.object_data */
            compute_value_size(p->u.object_data, hp);
            break;
        case JS_CLASS_C_FUNCTION:        /* u.cfunc */
            s->c_func_count++;
            break;
        case JS_CLASS_BYTECODE_FUNCTION: /* u.func */
            {
                JSFunctionBytecode *b = p->u.func.function_bytecode;
                JSVarRef **var_refs = p->u.func.var_refs;
                /* home_object: object will be accounted for in list scan */
                if (var_refs) {
                    s->memory_used_count++;
                    s->js_func_size += b->closure_var_count * sizeof(*var_refs);
                    for (i = 0; i < b->closure_var_count; i++) {
                        if (var_refs[i]) {
                            double ref_count = var_refs[i]->header.ref_count;
                            s->memory_used_count += 1 / ref_count;
                            s->js_func_size += sizeof(*var_refs[i]) / ref_count;
                            /* handle non object closed values */
                            if (var_refs[i]->pvalue == &var_refs[i]->value) {
                                /* potential multiple count */
                                compute_value_size(var_refs[i]->value, hp);
                            }
                        }
                    }
                }
            }
            break;
        case JS_CLASS_BOUND_FUNCTION:    /* u.bound_function */
            {
                JSBoundFunction *bf = p->u.bound_function;
                /* func_obj and this_val are objects */
                for (i = 0; i < bf->argc; i++) {
                    compute_value_size(bf->argv[i], hp);
                }
                s->memory_used_count += 1;
                s->memory_used_size += sizeof(*bf) + bf->argc * sizeof(*bf->argv);
            }
            break;
        case JS_CLASS_C_FUNCTION_DATA:   /* u.c_function_data_record */
            {
                JSCFunctionDataRecord *fd = p->u.c_function_data_record;
                if (fd) {
                    for (i = 0; i < fd->data_len; i++) {
                        compute_value_size(fd->data[i], hp);
                    }
                    s->memory_used_count += 1;
                    s->memory_used_size += sizeof(*fd) + fd->data_len * sizeof(*fd->data);
                }
            }
            break;
        case JS_CLASS_REGEXP:            /* u.regexp */
            compute_jsstring_size(p->u.regexp.pattern, hp);
            compute_jsstring_size(p->u.regexp.bytecode, hp);
            break;

        case JS_CLASS_FOR_IN_ITERATOR:   /* u.for_in_iterator */
            {
                JSForInIterator *it = p->u.for_in_iterator;
                if (it) {
                    compute_value_size(it->obj, hp);
                    s->memory_used_count += 1;
                    s->memory_used_size += sizeof(*it);
                }
            }
            break;
        case JS_CLASS_ARRAY_BUFFER:      /* u.array_buffer */
        case JS_CLASS_SHARED_ARRAY_BUFFER: /* u.array_buffer */
            {
                JSArrayBuffer *abuf = p->u.array_buffer;
                if (abuf) {
                    s->memory_used_count += 1;
                    s->memory_used_size += sizeof(*abuf);
                    if (abuf->data) {
                        s->memory_used_count += 1;
                        s->memory_used_size += abuf->byte_length;
                    }
                }
            }
            break;
        case JS_CLASS_GENERATOR:         /* u.generator_data */
        case JS_CLASS_UINT8C_ARRAY:      /* u.typed_array / u.array */
        case JS_CLASS_INT8_ARRAY:        /* u.typed_array / u.array */
        case JS_CLASS_UINT8_ARRAY:       /* u.typed_array / u.array */
        case JS_CLASS_INT16_ARRAY:       /* u.typed_array / u.array */
        case JS_CLASS_UINT16_ARRAY:      /* u.typed_array / u.array */
        case JS_CLASS_INT32_ARRAY:       /* u.typed_array / u.array */
        case JS_CLASS_UINT32_ARRAY:      /* u.typed_array / u.array */
        case JS_CLASS_BIG_INT64_ARRAY:   /* u.typed_array / u.array */
        case JS_CLASS_BIG_UINT64_ARRAY:  /* u.typed_array / u.array */
        case JS_CLASS_FLOAT16_ARRAY:     /* u.typed_array / u.array */
        case JS_CLASS_FLOAT32_ARRAY:     /* u.typed_array / u.array */
        case JS_CLASS_FLOAT64_ARRAY:     /* u.typed_array / u.array */
        case JS_CLASS_DATAVIEW:          /* u.typed_array */
        case JS_CLASS_MAP:               /* u.map_state */
        case JS_CLASS_SET:               /* u.map_state */
        case JS_CLASS_WEAKMAP:           /* u.map_state */
        case JS_CLASS_WEAKSET:           /* u.map_state */
        case JS_CLASS_MAP_ITERATOR:      /* u.map_iterator_data */
        case JS_CLASS_SET_ITERATOR:      /* u.map_iterator_data */
        case JS_CLASS_ARRAY_ITERATOR:    /* u.array_iterator_data */
        case JS_CLASS_STRING_ITERATOR:   /* u.array_iterator_data */
        case JS_CLASS_PROXY:             /* u.proxy_data */
        case JS_CLASS_PROMISE:           /* u.promise_data */
        case JS_CLASS_PROMISE_RESOLVE_FUNCTION:  /* u.promise_function_data */
        case JS_CLASS_PROMISE_REJECT_FUNCTION:   /* u.promise_function_data */
        case JS_CLASS_ASYNC_FUNCTION_RESOLVE:    /* u.async_function_data */
        case JS_CLASS_ASYNC_FUNCTION_REJECT:     /* u.async_function_data */
        case JS_CLASS_ASYNC_FROM_SYNC_ITERATOR:  /* u.async_from_sync_iterator_data */
        case JS_CLASS_ASYNC_GENERATOR:   /* u.async_generator_data */
            /* TODO */
        default:
            /* XXX: class definition should have an opaque block size */
            if (p->u.opaque) {
                s->memory_used_count += 1;
            }
            break;
        }
    }
    s->obj_size += s->obj_count * sizeof(JSObject);

    /* hashed shapes */
    s->memory_used_count++; /* rt->shape_hash */
    s->memory_used_size += sizeof(rt->shape_hash[0]) * rt->shape_hash_size;
    for(i = 0; i < rt->shape_hash_size; i++) {
        JSShape *sh;
        for(sh = rt->shape_hash[i]; sh != NULL; sh = sh->shape_hash_next) {
            int hash_size = sh->prop_hash_mask + 1;
            s->shape_count++;
            s->shape_size += get_shape_size(hash_size, sh->prop_size);
        }
    }

    /* atoms */
    s->memory_used_count += 2; /* rt->atom_array, rt->atom_hash */
    s->atom_count = rt->atom_count;
    s->atom_size = sizeof(rt->atom_array[0]) * rt->atom_size +
        sizeof(rt->atom_hash[0]) * rt->atom_hash_size;
    for(i = 0; i < rt->atom_size; i++) {
        JSAtomStruct *p = rt->atom_array[i];
        if (!atom_is_free(p)) {
            s->atom_size += (sizeof(*p) + (p->len << p->is_wide_char) +
                             1 - p->is_wide_char);
        }
    }
    s->str_count = round(mem.str_count);
    s->str_size = round(mem.str_size);
    s->js_func_count = mem.js_func_count;
    s->js_func_size = round(mem.js_func_size);
    s->js_func_code_size = mem.js_func_code_size;
    s->js_func_pc2line_count = mem.js_func_pc2line_count;
    s->js_func_pc2line_size = mem.js_func_pc2line_size;
    s->memory_used_count += round(mem.memory_used_count) +
        s->atom_count + s->str_count +
        s->obj_count + s->shape_count +
        s->js_func_count + s->js_func_pc2line_count;
    s->memory_used_size += s->atom_size + s->str_size +
        s->obj_size + s->prop_size + s->shape_size +
        s->js_func_size + s->js_func_code_size + s->js_func_pc2line_size;
}

void JS_DumpMemoryUsage(FILE *fp, const JSMemoryUsage *s, JSRuntime *rt)
{
    fprintf(fp, "QuickJS-ng memory usage -- %s version, %d-bit, %s Endian, malloc limit: %"PRId64"\n\n",
        JS_GetVersion(), (int)sizeof(void *) * 8, is_be() ? "Big" : "Little", s->malloc_limit);
    if (rt) {
        static const struct {
            const char *name;
            size_t size;
        } object_types[] = {
            { "JSRuntime", sizeof(JSRuntime) },
            { "JSContext", sizeof(JSContext) },
            { "JSObject", sizeof(JSObject) },
            { "JSString", sizeof(JSString) },
            { "JSFunctionBytecode", sizeof(JSFunctionBytecode) },
        };
        int i, usage_size_ok = 0;
        for(i = 0; i < countof(object_types); i++) {
            unsigned int size = object_types[i].size;
            void *p = js_malloc_rt(rt, size);
            if (p) {
                unsigned int size1 = js_malloc_usable_size_rt(rt, p);
                if (size1 >= size) {
                    usage_size_ok = 1;
                    fprintf(fp, "  %3u + %-2u  %s\n",
                            size, size1 - size, object_types[i].name);
                }
                js_free_rt(rt, p);
            }
        }
        if (!usage_size_ok) {
            fprintf(fp, "  malloc_usable_size unavailable\n");
        }
        {
            int obj_classes[JS_CLASS_INIT_COUNT + 1] = { 0 };
            int class_id;
            struct list_head *el;
            list_for_each(el, &rt->gc_obj_list) {
                JSGCObjectHeader *gp = list_entry(el, JSGCObjectHeader, link);
                JSObject *p;
                if (gp->gc_obj_type == JS_GC_OBJ_TYPE_JS_OBJECT) {
                    p = (JSObject *)gp;
                    obj_classes[min_uint32(p->class_id, JS_CLASS_INIT_COUNT)]++;
                }
            }
            fprintf(fp, "\n" "JSObject classes\n");
            if (obj_classes[0])
                fprintf(fp, "  %5d  %2.0d %s\n", obj_classes[0], 0, "none");
            for (class_id = 1; class_id < JS_CLASS_INIT_COUNT; class_id++) {
                if (obj_classes[class_id] && class_id < rt->class_count) {
                    char buf[ATOM_GET_STR_BUF_SIZE];
                    fprintf(fp, "  %5d  %2.0d %s\n", obj_classes[class_id], class_id,
                            JS_AtomGetStrRT(rt, buf, sizeof(buf), rt->class_array[class_id].class_name));
                }
            }
            if (obj_classes[JS_CLASS_INIT_COUNT])
                fprintf(fp, "  %5d  %2.0d %s\n", obj_classes[JS_CLASS_INIT_COUNT], 0, "other");
        }
        fprintf(fp, "\n");
    }
    fprintf(fp, "%-20s %8s %8s\n", "NAME", "COUNT", "SIZE");

    if (s->malloc_count) {
        fprintf(fp, "%-20s %8"PRId64" %8"PRId64"  (%0.1f per block)\n",
                "memory allocated", s->malloc_count, s->malloc_size,
                (double)s->malloc_size / s->malloc_count);
        fprintf(fp, "%-20s %8"PRId64" %8"PRId64"  (%d overhead, %0.1f average slack)\n",
                "memory used", s->memory_used_count, s->memory_used_size,
                MALLOC_OVERHEAD, ((double)(s->malloc_size - s->memory_used_size) /
                                  s->memory_used_count));
    }
    if (s->atom_count) {
        fprintf(fp, "%-20s %8"PRId64" %8"PRId64"  (%0.1f per atom)\n",
                "atoms", s->atom_count, s->atom_size,
                (double)s->atom_size / s->atom_count);
    }
    if (s->str_count) {
        fprintf(fp, "%-20s %8"PRId64" %8"PRId64"  (%0.1f per string)\n",
                "strings", s->str_count, s->str_size,
                (double)s->str_size / s->str_count);
    }
    if (s->obj_count) {
        fprintf(fp, "%-20s %8"PRId64" %8"PRId64"  (%0.1f per object)\n",
                "objects", s->obj_count, s->obj_size,
                (double)s->obj_size / s->obj_count);
        fprintf(fp, "%-20s %8"PRId64" %8"PRId64"  (%0.1f per object)\n",
                "  properties", s->prop_count, s->prop_size,
                (double)s->prop_count / s->obj_count);
        fprintf(fp, "%-20s %8"PRId64" %8"PRId64"  (%0.1f per shape)\n",
                "  shapes", s->shape_count, s->shape_size,
                (double)s->shape_size / s->shape_count);
    }
    if (s->js_func_count) {
        fprintf(fp, "%-20s %8"PRId64" %8"PRId64"\n",
                "bytecode functions", s->js_func_count, s->js_func_size);
        fprintf(fp, "%-20s %8"PRId64" %8"PRId64"  (%0.1f per function)\n",
                "  bytecode", s->js_func_count, s->js_func_code_size,
                (double)s->js_func_code_size / s->js_func_count);
        if (s->js_func_pc2line_count) {
            fprintf(fp, "%-20s %8"PRId64" %8"PRId64"  (%0.1f per function)\n",
                    "  pc2line", s->js_func_pc2line_count,
                    s->js_func_pc2line_size,
                    (double)s->js_func_pc2line_size / s->js_func_pc2line_count);
        }
    }
    if (s->c_func_count) {
        fprintf(fp, "%-20s %8"PRId64"\n", "C functions", s->c_func_count);
    }
    if (s->array_count) {
        fprintf(fp, "%-20s %8"PRId64"\n", "arrays", s->array_count);
        if (s->fast_array_count) {
            fprintf(fp, "%-20s %8"PRId64"\n", "  fast arrays", s->fast_array_count);
            fprintf(fp, "%-20s %8"PRId64" %8"PRId64"  (%0.1f per fast array)\n",
                    "  elements", s->fast_array_elements,
                    s->fast_array_elements * (int)sizeof(JSValue),
                    (double)s->fast_array_elements / s->fast_array_count);
        }
    }
    if (s->binary_object_count) {
        fprintf(fp, "%-20s %8"PRId64" %8"PRId64"\n",
                "binary objects", s->binary_object_count, s->binary_object_size);
    }
}

JSValue JS_GetGlobalObject(JSContext *ctx)
{
    return js_dup(ctx->global_obj);
}

/* WARNING: obj is freed */
JSValue JS_Throw(JSContext *ctx, JSValue obj)
{
    JSRuntime *rt = ctx->rt;
    JS_FreeValue(ctx, rt->current_exception);
    rt->current_exception = obj;
    return JS_EXCEPTION;
}

/* return the pending exception (cannot be called twice). */
JSValue JS_GetException(JSContext *ctx)
{
    JSValue val;
    JSRuntime *rt = ctx->rt;
    val = rt->current_exception;
    rt->current_exception = JS_UNINITIALIZED;
    return val;
}

bool JS_HasException(JSContext *ctx)
{
    return !JS_IsUninitialized(ctx->rt->current_exception);
}

static void dbuf_put_leb128(DynBuf *s, uint32_t v)
{
    uint32_t a;
    for(;;) {
        a = v & 0x7f;
        v >>= 7;
        if (v != 0) {
            dbuf_putc(s, a | 0x80);
        } else {
            dbuf_putc(s, a);
            break;
        }
    }
}

static void dbuf_put_sleb128(DynBuf *s, int32_t v1)
{
    uint32_t v = v1;
    dbuf_put_leb128(s, (2 * v) ^ -(v >> 31));
}

static int get_leb128(uint32_t *pval, const uint8_t *buf,
                      const uint8_t *buf_end)
{
    const uint8_t *ptr = buf;
    uint32_t v, a, i;
    v = 0;
    for(i = 0; i < 5; i++) {
        if (unlikely(ptr >= buf_end))
            break;
        a = *ptr++;
        v |= (a & 0x7f) << (i * 7);
        if (!(a & 0x80)) {
            *pval = v;
            return ptr - buf;
        }
    }
    *pval = 0;
    return -1;
}

static int get_sleb128(int32_t *pval, const uint8_t *buf,
                       const uint8_t *buf_end)
{
    int ret;
    uint32_t val;
    ret = get_leb128(&val, buf, buf_end);
    if (ret < 0) {
        *pval = 0;
        return -1;
    }
    *pval = (val >> 1) ^ -(val & 1);
    return ret;
}

static int find_line_num(JSContext *ctx, JSFunctionBytecode *b,
                         uint32_t pc_value, int *col)
{
    const uint8_t *p_end, *p;
    int new_line_num, new_col_num, line_num, col_num, pc, v, ret;
    unsigned int op;

    *col = 1;
    p = b->pc2line_buf;
    if (!p)
        goto fail;
    p_end = p + b->pc2line_len;
    pc = 0;
    line_num = b->line_num;
    col_num = b->col_num;
    while (p < p_end) {
        op = *p++;
        if (op == 0) {
            uint32_t val;
            ret = get_leb128(&val, p, p_end);
            if (ret < 0)
                goto fail;
            pc += val;
            p += ret;
            ret = get_sleb128(&v, p, p_end);
            if (ret < 0)
                goto fail;
            p += ret;
            new_line_num = line_num + v;
        } else {
            op -= PC2LINE_OP_FIRST;
            pc += (op / PC2LINE_RANGE);
            new_line_num = line_num + (op % PC2LINE_RANGE) + PC2LINE_BASE;
        }
        ret = get_sleb128(&v, p, p_end);
        if (ret < 0)
            goto fail;
        p += ret;
        new_col_num = col_num + v;
        if (pc_value < pc)
            break;
        line_num = new_line_num;
        col_num = new_col_num;
    }
    *col = col_num;
    return line_num;
fail:
    /* should never happen */
    return b->line_num;
}

/* in order to avoid executing arbitrary code during the stack trace
   generation, we only look at simple 'name' properties containing a
   string. */
static const char *get_func_name(JSContext *ctx, JSValueConst func)
{
    JSProperty *pr;
    JSShapeProperty *prs;
    JSValue val;

    if (JS_VALUE_GET_TAG(func) != JS_TAG_OBJECT)
        return NULL;
    prs = find_own_property(&pr, JS_VALUE_GET_OBJ(func), JS_ATOM_name);
    if (!prs)
        return NULL;
    if ((prs->flags & JS_PROP_TMASK) != JS_PROP_NORMAL)
        return NULL;
    val = pr->u.value;
    if (JS_VALUE_GET_TAG(val) != JS_TAG_STRING)
        return NULL;
    return JS_ToCString(ctx, val);
}

/* Note: it is important that no exception is returned by this function */
static bool can_add_backtrace(JSValueConst obj)
{
    JSObject *p;
    if (JS_VALUE_GET_TAG(obj) != JS_TAG_OBJECT)
        return false;
    p = JS_VALUE_GET_OBJ(obj);
    if (p->class_id != JS_CLASS_ERROR && p->class_id != JS_CLASS_DOM_EXCEPTION)
        return false;
    if (find_own_property1(p, JS_ATOM_stack))
        return false;
    return true;
}

#define JS_BACKTRACE_FLAG_SKIP_FIRST_LEVEL (1 << 0)
/* only taken into account if filename is provided */
#define JS_BACKTRACE_FLAG_SINGLE_LEVEL     (1 << 1)
#define JS_BACKTRACE_FLAG_FILTER_FUNC      (1 << 2)

/* if filename != NULL, an additional level is added with the filename
   and line number information (used for parse error). */
static void build_backtrace(JSContext *ctx, JSValueConst error_val,
                            JSValueConst filter_func, const char *filename,
                            int line_num, int col_num, int backtrace_flags)
{
    JSStackFrame *sf, *sf_start;
    JSValue stack, prepare, saved_exception;
    DynBuf dbuf;
    const char *func_name_str;
    const char *str1;
    JSObject *p;
    JSFunctionBytecode *b;
    bool backtrace_barrier, has_prepare, has_filter_func;
    JSRuntime *rt;
    JSCallSiteData csd[64];
    uint32_t i;
    double d;
    int stack_trace_limit;

    rt = ctx->rt;
    if (rt->in_build_stack_trace)
        return;
    rt->in_build_stack_trace = true;

    // Save exception because conversion to double may fail.
    saved_exception = JS_GetException(ctx);

    // Extract stack trace limit.
    // Ignore error since it sets d to NAN anyway.
    // coverity[check_return]
    JS_ToFloat64(ctx, &d, ctx->error_stack_trace_limit);
    if (isnan(d) || d < 0.0)
        stack_trace_limit = 0;
    else if (d > INT32_MAX)
        stack_trace_limit = INT32_MAX;
    else
        stack_trace_limit = fabs(d);

    // Restore current exception.
    JS_Throw(ctx, saved_exception);
    saved_exception = JS_UNINITIALIZED;

    stack_trace_limit = min_int(stack_trace_limit, countof(csd));
    stack_trace_limit = max_int(stack_trace_limit, 0);
    has_prepare = false;
    has_filter_func = backtrace_flags & JS_BACKTRACE_FLAG_FILTER_FUNC;
    i = 0;

    if (!JS_IsNull(ctx->error_ctor)) {
        prepare = js_dup(ctx->error_prepare_stack);
        has_prepare = JS_IsFunction(ctx, prepare);
    }

    if (has_prepare) {
        saved_exception = JS_GetException(ctx);
        if (stack_trace_limit == 0)
            goto done;
        if (filename)
            js_new_callsite_data2(ctx, &csd[i++], filename, line_num, col_num);
    } else {
        js_dbuf_init(ctx, &dbuf);
        if (stack_trace_limit == 0)
            goto done;
        if (filename) {
            i++;
            dbuf_printf(&dbuf, "    at %s", filename);
            if (line_num != -1)
                dbuf_printf(&dbuf, ":%d:%d", line_num, col_num);
            dbuf_putc(&dbuf, '\n');
        }
    }

    if (filename && (backtrace_flags & JS_BACKTRACE_FLAG_SINGLE_LEVEL))
        goto done;

    sf_start = rt->current_stack_frame;

    /* Find the frame we want to start from. Note that when a filter is used the filter
       function will be the first, but we also specify we want to skip the first one. */
    if (has_filter_func) {
        for (sf = sf_start; sf != NULL && i < stack_trace_limit; sf = sf->prev_frame) {
            if (js_same_value(ctx, sf->cur_func, filter_func)) {
                sf_start = sf;
                break;
            }
        }
    }

    for (sf = sf_start; sf != NULL && i < stack_trace_limit; sf = sf->prev_frame) {
        if (backtrace_flags & JS_BACKTRACE_FLAG_SKIP_FIRST_LEVEL) {
            backtrace_flags &= ~JS_BACKTRACE_FLAG_SKIP_FIRST_LEVEL;
            continue;
        }

        p = JS_VALUE_GET_OBJ(sf->cur_func);
        b = NULL;
        backtrace_barrier = false;

        if (js_class_has_bytecode(p->class_id)) {
            b = p->u.func.function_bytecode;
            backtrace_barrier = b->backtrace_barrier;
        }

        if (has_prepare) {
            js_new_callsite_data(ctx, &csd[i], sf);
        } else {
            /* func_name_str is UTF-8 encoded if needed */
            func_name_str = get_func_name(ctx, sf->cur_func);
            if (!func_name_str || func_name_str[0] == '\0')
                str1 = "<anonymous>";
            else
                str1 = func_name_str;
            dbuf_printf(&dbuf, "    at %s", str1);
            JS_FreeCString(ctx, func_name_str);

            if (b && sf->cur_pc) {
                const char *atom_str;
                int line_num1, col_num1;
                uint32_t pc;

                pc = sf->cur_pc - b->byte_code_buf - 1;
                line_num1 = find_line_num(ctx, b, pc, &col_num1);
                atom_str = b->filename ? JS_AtomToCString(ctx, b->filename) : NULL;
                dbuf_printf(&dbuf, " (%s", atom_str ? atom_str : "<null>");
                JS_FreeCString(ctx, atom_str);
                if (line_num1 != -1)
                    dbuf_printf(&dbuf, ":%d:%d", line_num1, col_num1);
                dbuf_putc(&dbuf, ')');
            } else if (b) {
                // FIXME(bnoordhuis) Missing `sf->cur_pc = pc` in bytecode
                // handler in JS_CallInternal. Almost never user observable
                // except with intercepting JS proxies that throw exceptions.
                dbuf_printf(&dbuf, " (missing)");
            } else {
                dbuf_printf(&dbuf, " (native)");
            }
            dbuf_putc(&dbuf, '\n');
        }
        i++;

        /* stop backtrace if JS_EVAL_FLAG_BACKTRACE_BARRIER was used */
        if (backtrace_barrier)
            break;
    }
 done:
    if (has_prepare) {
        int j = 0, k;
        stack = JS_NewArray(ctx);
        if (JS_IsException(stack)) {
            stack = JS_NULL;
        } else {
            for (; j < i; j++) {
                JSValue v = js_new_callsite(ctx, &csd[j]);
                if (JS_IsException(v))
                    break;
                if (JS_DefinePropertyValueUint32(ctx, stack, j, v, JS_PROP_C_W_E) < 0) {
                    JS_FreeValue(ctx, v);
                    break;
                }
            }
        }
        // Clear the csd's we didn't use in case of error.
        for (k = j; k < i; k++) {
            JS_FreeValue(ctx, csd[k].filename);
            JS_FreeValue(ctx, csd[k].func);
            JS_FreeValue(ctx, csd[k].func_name);
        }
        JSValueConst args[] = {
            error_val,
            stack,
        };
        JSValue stack2 = JS_Call(ctx, prepare, ctx->error_ctor, countof(args), args);
        JS_FreeValue(ctx, stack);
        if (JS_IsException(stack2))
            stack = JS_NULL;
        else
            stack = stack2;
        JS_FreeValue(ctx, prepare);
        JS_Throw(ctx, saved_exception);
    } else {
        if (dbuf_error(&dbuf))
            stack = JS_NULL;
        else
            stack = JS_NewStringLen(ctx, (char *)dbuf.buf, dbuf.size);
        dbuf_free(&dbuf);
    }

    if (JS_IsUndefined(ctx->error_back_trace))
        ctx->error_back_trace = js_dup(stack);
    if (has_filter_func || can_add_backtrace(error_val)) {
        JS_DefinePropertyValue(ctx, error_val, JS_ATOM_stack, stack,
                               JS_PROP_WRITABLE | JS_PROP_CONFIGURABLE);
    } else {
        JS_FreeValue(ctx, stack);
    }

    rt->in_build_stack_trace = false;
}

JSValue JS_NewError(JSContext *ctx)
{
    JSValue obj = JS_NewObjectClass(ctx, JS_CLASS_ERROR);
    if (JS_IsException(obj))
        return JS_EXCEPTION;
    build_backtrace(ctx, obj, JS_UNDEFINED, NULL, 0, 0, 0);
    return obj;
}

static JSValue JS_MakeError2(JSContext *ctx, JSErrorEnum error_num,
                             bool add_backtrace, const char *message)
{
    JSValue obj, msg;

    if (error_num == JS_PLAIN_ERROR) {
        obj = JS_NewObjectClass(ctx, JS_CLASS_ERROR);
    } else {
        obj = JS_NewObjectProtoClass(ctx, ctx->native_error_proto[error_num],
                                     JS_CLASS_ERROR);
    }
    if (JS_IsException(obj))
        return JS_EXCEPTION;
    msg = JS_NewString(ctx, message);
    if (JS_IsException(msg))
        msg = JS_NewString(ctx, "Invalid error message");
    if (!JS_IsException(msg)) {
        JS_DefinePropertyValue(ctx, obj, JS_ATOM_message, msg,
                               JS_PROP_WRITABLE | JS_PROP_CONFIGURABLE);
    }
    if (add_backtrace)
        build_backtrace(ctx, obj, JS_UNDEFINED, NULL, 0, 0, 0);
    return obj;
}

static JSValue JS_PRINTF_FORMAT_ATTR(4, 0)
JS_MakeError(JSContext *ctx, JSErrorEnum error_num, bool add_backtrace,
             JS_PRINTF_FORMAT const char *fmt, va_list ap)
{
    char buf[256];

    vsnprintf(buf, sizeof(buf), fmt, ap);
    return JS_MakeError2(ctx, error_num, add_backtrace, buf);
}

/* fmt and arguments may be pure ASCII or UTF-8 encoded contents */
static JSValue JS_PRINTF_FORMAT_ATTR(4, 0)
JS_ThrowError2(JSContext *ctx, JSErrorEnum error_num, bool add_backtrace,
               JS_PRINTF_FORMAT const char *fmt, va_list ap)
{
    JSValue obj;

    obj = JS_MakeError(ctx, error_num, add_backtrace, fmt, ap);
    if (unlikely(JS_IsException(obj))) {
        /* out of memory: throw JS_NULL to avoid recursing */
        obj = JS_NULL;
    }
    return JS_Throw(ctx, obj);
}

static JSValue JS_PRINTF_FORMAT_ATTR(3, 0)
JS_ThrowError(JSContext *ctx, JSErrorEnum error_num,
              JS_PRINTF_FORMAT const char *fmt, va_list ap)
{
    JSRuntime *rt = ctx->rt;
    JSStackFrame *sf;
    bool add_backtrace;

    /* the backtrace is added later if called from a bytecode function */
    sf = rt->current_stack_frame;
    add_backtrace = !rt->in_out_of_memory &&
        (!sf || (JS_GetFunctionBytecode(sf->cur_func) == NULL));
    return JS_ThrowError2(ctx, error_num, add_backtrace, fmt, ap);
}

#define JS_ERROR_MAP(X)     \
    X(Internal, INTERNAL)   \
    X(Plain, PLAIN)         \
    X(Range, RANGE)         \
    X(Reference, REFERENCE) \
    X(Syntax, SYNTAX)       \
    X(Type, TYPE)           \

#define X(lc, uc)   \
    JSValue JS_PRINTF_FORMAT_ATTR(2, 3)                         \
    JS_New##lc##Error(JSContext *ctx,                           \
                      JS_PRINTF_FORMAT const char *fmt, ...)    \
    {                                                           \
        JSValue val;                                            \
        va_list ap;                                             \
                                                                \
        va_start(ap, fmt);                                      \
        val = JS_MakeError(ctx, JS_##uc##_ERROR,                \
                           /*add_backtrace*/true, fmt, ap);     \
        va_end(ap);                                             \
        return val;                                             \
    }                                                           \
    JSValue JS_PRINTF_FORMAT_ATTR(2, 3)                         \
    JS_Throw##lc##Error(JSContext *ctx,                         \
                        JS_PRINTF_FORMAT const char *fmt, ...)  \
    {                                                           \
        JSValue val;                                            \
        va_list ap;                                             \
                                                                \
        va_start(ap, fmt);                                      \
        val = JS_ThrowError(ctx, JS_##uc##_ERROR, fmt, ap);     \
        va_end(ap);                                             \
        return val;                                             \
    }                                                           \

JS_ERROR_MAP(X)

#undef X
#undef JS_ERROR_MAP

static int JS_PRINTF_FORMAT_ATTR(3, 4) JS_ThrowTypeErrorOrFalse(JSContext *ctx, int flags, JS_PRINTF_FORMAT const char *fmt, ...)
{
    va_list ap;

    if ((flags & JS_PROP_THROW) ||
        ((flags & JS_PROP_THROW_STRICT) && is_strict_mode(ctx))) {
        va_start(ap, fmt);
        JS_ThrowError(ctx, JS_TYPE_ERROR, fmt, ap);
        va_end(ap);
        return -1;
    } else {
        return false;
    }
}

#ifdef __GNUC__
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wformat-nonliteral"
#endif // __GNUC__
static JSValue JS_ThrowTypeErrorAtom(JSContext *ctx, const char *fmt, JSAtom atom)
{
    char buf[ATOM_GET_STR_BUF_SIZE];
    JS_AtomGetStr(ctx, buf, sizeof(buf), atom);
    return JS_ThrowTypeError(ctx, fmt, buf);
}

static JSValue JS_ThrowSyntaxErrorAtom(JSContext *ctx, const char *fmt, JSAtom atom)
{
    char buf[ATOM_GET_STR_BUF_SIZE];
    JS_AtomGetStr(ctx, buf, sizeof(buf), atom);
    return JS_ThrowSyntaxError(ctx, fmt, buf);
}
#ifdef __GNUC__
#pragma GCC diagnostic pop // ignored "-Wformat-nonliteral"
#endif // __GNUC__

static int JS_ThrowTypeErrorReadOnly(JSContext *ctx, int flags, JSAtom atom)
{
    if ((flags & JS_PROP_THROW) ||
        ((flags & JS_PROP_THROW_STRICT) && is_strict_mode(ctx))) {
        JS_ThrowTypeErrorAtom(ctx, "'%s' is read-only", atom);
        return -1;
    } else {
        return false;
    }
}

JSValue JS_ThrowOutOfMemory(JSContext *ctx)
{
    JSRuntime *rt = ctx->rt;
    if (!rt->in_out_of_memory) {
        rt->in_out_of_memory = true;
        JS_ThrowInternalError(ctx, "out of memory");
        rt->in_out_of_memory = false;
    }
    return JS_EXCEPTION;
}

static JSValue JS_ThrowStackOverflow(JSContext *ctx)
{
    return JS_ThrowRangeError(ctx, "Maximum call stack size exceeded");
}

static JSValue JS_ThrowTypeErrorNotAConstructor(JSContext *ctx,
                                                JSValueConst func_obj)
{
    JSObject *p;
    JSAtom name;

    if (JS_TAG_OBJECT != JS_VALUE_GET_TAG(func_obj))
        goto fini;
    p = JS_VALUE_GET_OBJ(func_obj);
    if (!js_class_has_bytecode(p->class_id))
        goto fini;
    name = p->u.func.function_bytecode->func_name;
    if (name == JS_ATOM_NULL)
        goto fini;
    return JS_ThrowTypeErrorAtom(ctx, "%s is not a constructor", name);
fini:
    return JS_ThrowTypeError(ctx, "not a constructor");
}

static JSValue JS_ThrowTypeErrorNotAFunction(JSContext *ctx)
{
    return JS_ThrowTypeError(ctx, "not a function");
}

static JSValue JS_ThrowTypeErrorNotAnObject(JSContext *ctx)
{
    return JS_ThrowTypeError(ctx, "not an object");
}

static JSValue JS_ThrowTypeErrorNotASymbol(JSContext *ctx)
{
    return JS_ThrowTypeError(ctx, "not a symbol");
}

static JSValue JS_ThrowReferenceErrorNotDefined(JSContext *ctx, JSAtom name)
{
    char buf[ATOM_GET_STR_BUF_SIZE];
    return JS_ThrowReferenceError(ctx, "%s is not defined",
                                  JS_AtomGetStr(ctx, buf, sizeof(buf), name));
}

static JSValue JS_ThrowReferenceErrorUninitialized(JSContext *ctx, JSAtom name)
{
    char buf[ATOM_GET_STR_BUF_SIZE];
    return JS_ThrowReferenceError(ctx, "%s is not initialized",
                                  name == JS_ATOM_NULL ? "lexical variable" :
                                  JS_AtomGetStr(ctx, buf, sizeof(buf), name));
}

static JSValue JS_ThrowReferenceErrorUninitialized2(JSContext *ctx,
                                                    JSFunctionBytecode *b,
                                                    int idx, bool is_ref)
{
    JSAtom atom = JS_ATOM_NULL;
    if (is_ref) {
        atom = b->closure_var[idx].var_name;
    } else {
        /* not present if the function is stripped and contains no eval() */
        if (b->vardefs)
            atom = b->vardefs[b->arg_count + idx].var_name;
    }
    return JS_ThrowReferenceErrorUninitialized(ctx, atom);
}

static JSValue JS_ThrowTypeErrorInvalidClass(JSContext *ctx, int class_id)
{
    JSRuntime *rt = ctx->rt;
    JSAtom name;
    name = rt->class_array[class_id].class_name;
    return JS_ThrowTypeErrorAtom(ctx, "%s object expected", name);
}

static void JS_ThrowInterrupted(JSContext *ctx)
{
    JS_ThrowInternalError(ctx, "interrupted");
    JS_SetUncatchableError(ctx, ctx->rt->current_exception);
}

static no_inline __exception int __js_poll_interrupts(JSContext *ctx)
{
    JSRuntime *rt = ctx->rt;
    ctx->interrupt_counter = JS_INTERRUPT_COUNTER_INIT;
    if (rt->interrupt_handler) {
        if (rt->interrupt_handler(rt, rt->interrupt_opaque)) {
            JS_ThrowInterrupted(ctx);
            return -1;
        }
    }
    return 0;
}

static inline __exception int js_poll_interrupts(JSContext *ctx)
{
    if (unlikely(--ctx->interrupt_counter <= 0)) {
        return __js_poll_interrupts(ctx);
    } else {
        return 0;
    }
}

/* return -1 (exception) or true/false */
static int JS_SetPrototypeInternal(JSContext *ctx, JSValueConst obj,
                                   JSValueConst proto_val, bool throw_flag)
{
    JSObject *proto, *p, *p1;
    JSShape *sh;

    if (throw_flag) {
        if (JS_VALUE_GET_TAG(obj) == JS_TAG_NULL ||
            JS_VALUE_GET_TAG(obj) == JS_TAG_UNDEFINED)
            goto not_obj;
    } else {
        if (JS_VALUE_GET_TAG(obj) != JS_TAG_OBJECT)
            goto not_obj;
    }
    p = JS_VALUE_GET_OBJ(obj);
    if (JS_VALUE_GET_TAG(proto_val) != JS_TAG_OBJECT) {
        if (JS_VALUE_GET_TAG(proto_val) != JS_TAG_NULL) {
        not_obj:
            JS_ThrowTypeErrorNotAnObject(ctx);
            return -1;
        }
        proto = NULL;
    } else {
        proto = JS_VALUE_GET_OBJ(proto_val);
    }

    if (throw_flag && JS_VALUE_GET_TAG(obj) != JS_TAG_OBJECT)
        return true;

    if (unlikely(p->class_id == JS_CLASS_PROXY))
        return js_proxy_setPrototypeOf(ctx, obj, proto_val, throw_flag);
    sh = p->shape;
    if (sh->proto == proto)
        return true;
    if (p == JS_VALUE_GET_OBJ(ctx->class_proto[JS_CLASS_OBJECT])) {
        if (throw_flag) {
            JS_ThrowTypeError(ctx, "'Immutable prototype object \'Object.prototype\' cannot have their prototype set'");
            return -1;
        }
        return false;
    }
    if (!p->extensible) {
        if (throw_flag) {
            JS_ThrowTypeError(ctx, "object is not extensible");
            return -1;
        } else {
            return false;
        }
    }
    if (proto) {
        /* check if there is a cycle */
        p1 = proto;
        do {
            if (p1 == p) {
                if (throw_flag) {
                    JS_ThrowTypeError(ctx, "circular prototype chain");
                    return -1;
                } else {
                    return false;
                }
            }
            /* Note: for Proxy objects, proto is NULL */
            p1 = p1->shape->proto;
        } while (p1 != NULL);
        js_dup(proto_val);
    }

    if (js_shape_prepare_update(ctx, p, NULL))
        return -1;
    sh = p->shape;
    if (sh->proto)
        JS_FreeValue(ctx, JS_MKPTR(JS_TAG_OBJECT, sh->proto));
    sh->proto = proto;
    return true;
}

/* return -1 (exception) or true/false */
int JS_SetPrototype(JSContext *ctx, JSValueConst obj, JSValueConst proto_val)
{
    return JS_SetPrototypeInternal(ctx, obj, proto_val, true);
}

/* Only works for primitive types, otherwise return JS_NULL. */
static JSValueConst JS_GetPrototypePrimitive(JSContext *ctx, JSValueConst val)
{
    JSValue ret;
    switch(JS_VALUE_GET_NORM_TAG(val)) {
    case JS_TAG_SHORT_BIG_INT:
    case JS_TAG_BIG_INT:
        ret = ctx->class_proto[JS_CLASS_BIG_INT];
        break;
    case JS_TAG_INT:
    case JS_TAG_FLOAT64:
        ret = ctx->class_proto[JS_CLASS_NUMBER];
        break;
    case JS_TAG_BOOL:
        ret = ctx->class_proto[JS_CLASS_BOOLEAN];
        break;
    case JS_TAG_STRING:
        ret = ctx->class_proto[JS_CLASS_STRING];
        break;
    case JS_TAG_SYMBOL:
        ret = ctx->class_proto[JS_CLASS_SYMBOL];
        break;
    case JS_TAG_OBJECT:
    case JS_TAG_NULL:
    case JS_TAG_UNDEFINED:
    default:
        ret = JS_NULL;
        break;
    }
    return ret;
}

/* Return an Object, JS_NULL or JS_EXCEPTION in case of Proxy object. */
JSValue JS_GetPrototype(JSContext *ctx, JSValueConst obj)
{
    JSValue val;
    if (JS_VALUE_GET_TAG(obj) == JS_TAG_OBJECT) {
        JSObject *p;
        p = JS_VALUE_GET_OBJ(obj);
        if (unlikely(p->class_id == JS_CLASS_PROXY)) {
            val = js_proxy_getPrototypeOf(ctx, obj);
        } else {
            p = p->shape->proto;
            if (!p)
                val = JS_NULL;
            else
                val = js_dup(JS_MKPTR(JS_TAG_OBJECT, p));
        }
    } else {
        val = js_dup(JS_GetPrototypePrimitive(ctx, obj));
    }
    return val;
}

static JSValue JS_GetPrototypeFree(JSContext *ctx, JSValue obj)
{
    JSValue obj1;
    obj1 = JS_GetPrototype(ctx, obj);
    JS_FreeValue(ctx, obj);
    return obj1;
}

int JS_GetLength(JSContext *ctx, JSValueConst obj, int64_t *pres) {
    return js_get_length64(ctx, pres, obj);
}

int JS_SetLength(JSContext *ctx, JSValueConst obj, int64_t len) {
    return js_set_length64(ctx, obj, len);
}

/* return true, false or (-1) in case of exception */
static int JS_OrdinaryIsInstanceOf(JSContext *ctx, JSValueConst val,
                                   JSValueConst obj)
{
    JSValue obj_proto;
    JSObject *proto;
    const JSObject *p, *proto1;
    int ret;

    if (!JS_IsFunction(ctx, obj))
        return false;
    p = JS_VALUE_GET_OBJ(obj);
    if (p->class_id == JS_CLASS_BOUND_FUNCTION) {
        JSBoundFunction *s = p->u.bound_function;
        return JS_IsInstanceOf(ctx, val, s->func_obj);
    }

    /* Only explicitly boxed values are instances of constructors */
    if (JS_VALUE_GET_TAG(val) != JS_TAG_OBJECT)
        return false;
    obj_proto = JS_GetProperty(ctx, obj, JS_ATOM_prototype);
    if (JS_VALUE_GET_TAG(obj_proto) != JS_TAG_OBJECT) {
        if (!JS_IsException(obj_proto))
            JS_ThrowTypeError(ctx, "operand 'prototype' property is not an object");
        ret = -1;
        goto done;
    }
    proto = JS_VALUE_GET_OBJ(obj_proto);
    p = JS_VALUE_GET_OBJ(val);
    for(;;) {
        proto1 = p->shape->proto;
        if (!proto1) {
            /* slow case if proxy in the prototype chain */
            if (unlikely(p->class_id == JS_CLASS_PROXY)) {
                JSValue obj1;
                obj1 = js_dup(JS_MKPTR(JS_TAG_OBJECT, (JSObject *)p));
                for(;;) {
                    obj1 = JS_GetPrototypeFree(ctx, obj1);
                    if (JS_IsException(obj1)) {
                        ret = -1;
                        break;
                    }
                    if (JS_IsNull(obj1)) {
                        ret = false;
                        break;
                    }
                    if (proto == JS_VALUE_GET_OBJ(obj1)) {
                        JS_FreeValue(ctx, obj1);
                        ret = true;
                        break;
                    }
                    /* must check for timeout to avoid infinite loop */
                    if (js_poll_interrupts(ctx)) {
                        JS_FreeValue(ctx, obj1);
                        ret = -1;
                        break;
                    }
                }
            } else {
                ret = false;
            }
            break;
        }
        p = proto1;
        if (proto == p) {
            ret = true;
            break;
        }
    }
done:
    JS_FreeValue(ctx, obj_proto);
    return ret;
}

/* return true, false or (-1) in case of exception */
int JS_IsInstanceOf(JSContext *ctx, JSValueConst val, JSValueConst obj)
{
    JSValue method;

    if (!JS_IsObject(obj))
        goto fail;
    method = JS_GetProperty(ctx, obj, JS_ATOM_Symbol_hasInstance);
    if (JS_IsException(method))
        return -1;
    if (!JS_IsNull(method) && !JS_IsUndefined(method)) {
        JSValue ret;
        ret = JS_CallFree(ctx, method, obj, 1, &val);
        return JS_ToBoolFree(ctx, ret);
    }

    /* legacy case */
    if (!JS_IsFunction(ctx, obj)) {
    fail:
        JS_ThrowTypeError(ctx, "invalid 'instanceof' right operand");
        return -1;
    }
    return JS_OrdinaryIsInstanceOf(ctx, val, obj);
}

/* File generated automatically by the QuickJS-ng compiler. */

#include <inttypes.h>

const uint32_t qjsc_builtin_array_fromasync_size = 826;

const uint8_t qjsc_builtin_array_fromasync[826] = {
 0x15, 0x0d, 0x01, 0x1a, 0x61, 0x73, 0x79, 0x6e,
 0x63, 0x49, 0x74, 0x65, 0x72, 0x61, 0x74, 0x6f,
 0x72, 0x01, 0x10, 0x69, 0x74, 0x65, 0x72, 0x61,
 0x74, 0x6f, 0x72, 0x01, 0x12, 0x61, 0x72, 0x72,
 0x61, 0x79, 0x4c, 0x69, 0x6b, 0x65, 0x01, 0x0a,
 0x6d, 0x61, 0x70, 0x46, 0x6e, 0x01, 0x0e, 0x74,
 0x68, 0x69, 0x73, 0x41, 0x72, 0x67, 0x01, 0x0c,
 0x72, 0x65, 0x73, 0x75, 0x6c, 0x74, 0x01, 0x02,
 0x69, 0x01, 0x1a, 0x69, 0x73, 0x43, 0x6f, 0x6e,
 0x73, 0x74, 0x72, 0x75, 0x63, 0x74, 0x6f, 0x72,
 0x01, 0x08, 0x73, 0x79, 0x6e, 0x63, 0x01, 0x0c,
 0x6d, 0x65, 0x74, 0x68, 0x6f, 0x64, 0x01, 0x08,
 0x69, 0x74, 0x65, 0x72, 0x01, 0x1c, 0x6e, 0x6f,
 0x74, 0x20, 0x61, 0x20, 0x66, 0x75, 0x6e, 0x63,
 0x74, 0x69, 0x6f, 0x6e, 0x01, 0x08, 0x63, 0x61,
 0x6c, 0x6c, 0x0c, 0x00, 0x02, 0x00, 0xa2, 0x01,
 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x04, 0x01,
 0xa4, 0x01, 0x00, 0x00, 0x00, 0x0c, 0x43, 0x02,
 0x01, 0x00, 0x05, 0x00, 0x05, 0x01, 0x00, 0x01,
 0x03, 0x05, 0xaa, 0x02, 0x00, 0x01, 0x40, 0xa0,
 0x03, 0x00, 0x01, 0x40, 0xc6, 0x03, 0x00, 0x01,
 0x40, 0xcc, 0x01, 0x00, 0x01, 0x40, 0xc8, 0x03,
 0x00, 0x01, 0x40, 0x0c, 0x60, 0x02, 0x01, 0xf8,
 0x01, 0x03, 0x0e, 0x01, 0x06, 0x05, 0x00, 0x86,
 0x04, 0x11, 0xca, 0x03, 0x00, 0x01, 0x00, 0xcc,
 0x03, 0x00, 0x01, 0x00, 0xce, 0x03, 0x00, 0x01,
 0x00, 0xca, 0x03, 0x01, 0xff, 0xff, 0xff, 0xff,
 0x0f, 0x20, 0xcc, 0x03, 0x01, 0x01, 0x20, 0xce,
 0x03, 0x01, 0x02, 0x20, 0xd0, 0x03, 0x02, 0x00,
 0x20, 0xd2, 0x03, 0x02, 0x04, 0x20, 0xd4, 0x03,
 0x02, 0x05, 0x20, 0xd6, 0x03, 0x02, 0x06, 0x20,
 0xd8, 0x03, 0x02, 0x07, 0x20, 0x64, 0x06, 0x08,
 0x20, 0x82, 0x01, 0x07, 0x09, 0x20, 0xda, 0x03,
 0x0a, 0x08, 0x30, 0x82, 0x01, 0x0d, 0x0b, 0x20,
 0xd4, 0x01, 0x0d, 0x0c, 0x20, 0x10, 0x00, 0x01,
 0x00, 0xa0, 0x03, 0x01, 0x03, 0xc6, 0x03, 0x02,
 0x03, 0xc8, 0x03, 0x04, 0x03, 0xaa, 0x02, 0x00,
 0x03, 0xcc, 0x01, 0x03, 0x03, 0x08, 0xc4, 0x0d,
 0x62, 0x02, 0x00, 0x62, 0x01, 0x00, 0x62, 0x00,
 0x00, 0xd3, 0xcb, 0xd4, 0x11, 0xf4, 0xec, 0x08,
 0x0e, 0x39, 0x46, 0x00, 0x00, 0x00, 0xdc, 0xcc,
 0xd5, 0x11, 0xf4, 0xec, 0x08, 0x0e, 0x39, 0x46,
 0x00, 0x00, 0x00, 0xdd, 0xcd, 0x62, 0x07, 0x00,
 0x62, 0x06, 0x00, 0x62, 0x05, 0x00, 0x62, 0x04,
 0x00, 0x62, 0x03, 0x00, 0xd4, 0x39, 0x46, 0x00,
 0x00, 0x00, 0xb0, 0xec, 0x16, 0xd4, 0x98, 0x04,
 0x1b, 0x00, 0x00, 0x00, 0xb0, 0xec, 0x0c, 0xdf,
 0x11, 0x04, 0xee, 0x00, 0x00, 0x00, 0x21, 0x01,
 0x00, 0x30, 0x06, 0xce, 0xb6, 0xc4, 0x04, 0xc3,
 0x0d, 0xf7, 0xc4, 0x05, 0x09, 0xc4, 0x06, 0xd3,
 0xe0, 0x48, 0xc4, 0x07, 0x63, 0x07, 0x00, 0x07,
 0xad, 0xec, 0x0f, 0x0a, 0x11, 0x64, 0x06, 0x00,
 0x0e, 0xd3, 0xe1, 0x48, 0x11, 0x64, 0x07, 0x00,
 0x0e, 0x63, 0x07, 0x00, 0x07, 0xad, 0x6a, 0xa6,
 0x00, 0x00, 0x00, 0x62, 0x08, 0x00, 0x06, 0x11,
 0xf4, 0xed, 0x0c, 0x71, 0x43, 0x32, 0x00, 0x00,
 0x00, 0xc4, 0x08, 0x0e, 0xee, 0x05, 0x0e, 0xd3,
 0xee, 0xf2, 0x63, 0x08, 0x00, 0x8e, 0x11, 0xed,
 0x03, 0x0e, 0xb6, 0x11, 0x64, 0x08, 0x00, 0x0e,
 0x63, 0x05, 0x00, 0xec, 0x0c, 0xc3, 0x0d, 0x11,
 0x63, 0x08, 0x00, 0x21, 0x01, 0x00, 0xee, 0x06,
 0xe2, 0x63, 0x08, 0x00, 0xf1, 0x11, 0x64, 0x03,
 0x00, 0x0e, 0x63, 0x04, 0x00, 0x63, 0x08, 0x00,
 0xa7, 0x6a, 0x2a, 0x01, 0x00, 0x00, 0x62, 0x09,
 0x00, 0xd3, 0x63, 0x04, 0x00, 0x48, 0xc4, 0x09,
 0x63, 0x06, 0x00, 0xec, 0x0a, 0x63, 0x09, 0x00,
 0x8c, 0x11, 0x64, 0x09, 0x00, 0x0e, 0xd4, 0xec,
 0x17, 0xd4, 0x43, 0xef, 0x00, 0x00, 0x00, 0xd5,
 0x63, 0x09, 0x00, 0x63, 0x04, 0x00, 0x24, 0x03,
 0x00, 0x8c, 0x11, 0x64, 0x09, 0x00, 0x0e, 0x5f,
 0x04, 0x00, 0x63, 0x03, 0x00, 0x63, 0x04, 0x00,
 0x92, 0x64, 0x04, 0x00, 0x0b, 0x63, 0x09, 0x00,
 0x4d, 0x41, 0x00, 0x00, 0x00, 0x0a, 0x4d, 0x3e,
 0x00, 0x00, 0x00, 0x0a, 0x4d, 0x3f, 0x00, 0x00,
 0x00, 0xf3, 0x0e, 0xee, 0x9e, 0x62, 0x0a, 0x00,
 0x63, 0x07, 0x00, 0x43, 0xef, 0x00, 0x00, 0x00,
 0xd3, 0x24, 0x01, 0x00, 0xc4, 0x0a, 0x63, 0x05,
 0x00, 0xec, 0x09, 0xc3, 0x0d, 0x11, 0x21, 0x00,
 0x00, 0xee, 0x03, 0xe2, 0xf0, 0x11, 0x64, 0x03,
 0x00, 0x0e, 0x6d, 0x8c, 0x00, 0x00, 0x00, 0x62,
 0x0c, 0x00, 0x62, 0x0b, 0x00, 0x06, 0x11, 0xf4,
 0xed, 0x13, 0x71, 0x43, 0x41, 0x00, 0x00, 0x00,
 0xc4, 0x0b, 0x43, 0x6a, 0x00, 0x00, 0x00, 0xc4,
 0x0c, 0x0e, 0xee, 0x10, 0x0e, 0x63, 0x0a, 0x00,
 0x43, 0x6b, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00,
 0x8c, 0xee, 0xe0, 0x63, 0x0c, 0x00, 0xed, 0x4e,
 0x63, 0x06, 0x00, 0xec, 0x0a, 0x63, 0x0b, 0x00,
 0x8c, 0x11, 0x64, 0x0b, 0x00, 0x0e, 0xd4, 0xec,
 0x17, 0xd4, 0x43, 0xef, 0x00, 0x00, 0x00, 0xd5,
 0x63, 0x0b, 0x00, 0x63, 0x04, 0x00, 0x24, 0x03,
 0x00, 0x8c, 0x11, 0x64, 0x0b, 0x00, 0x0e, 0x5f,
 0x04, 0x00, 0x63, 0x03, 0x00, 0x63, 0x04, 0x00,
 0x92, 0x64, 0x04, 0x00, 0x0b, 0x63, 0x0b, 0x00,
 0x4d, 0x41, 0x00, 0x00, 0x00, 0x0a, 0x4d, 0x3e,
 0x00, 0x00, 0x00, 0x0a, 0x4d, 0x3f, 0x00, 0x00,
 0x00, 0xf3, 0x0e, 0xee, 0x83, 0x0e, 0x06, 0x6e,
 0x0d, 0x00, 0x00, 0x00, 0x0e, 0xee, 0x1e, 0x6e,
 0x05, 0x00, 0x00, 0x00, 0x30, 0x63, 0x0a, 0x00,
 0x42, 0x06, 0x00, 0x00, 0x00, 0xec, 0x0d, 0x63,
 0x0a, 0x00, 0x43, 0x06, 0x00, 0x00, 0x00, 0x24,
 0x00, 0x00, 0x0e, 0x6f, 0x63, 0x03, 0x00, 0x63,
 0x04, 0x00, 0x44, 0x32, 0x00, 0x00, 0x00, 0x63,
 0x03, 0x00, 0x2f, 0xc1, 0x00, 0x28, 0xc1, 0x00,
 0xcf, 0x28,
};



static JSValue js_bytecode_autoinit(JSContext *ctx, JSObject *p, JSAtom atom,
                                    void *opaque)
{
    switch ((uintptr_t)opaque) {
    default:
        abort();
    case JS_BUILTIN_ARRAY_FROMASYNC:
        {
            JSValue obj = JS_ReadObject(ctx, qjsc_builtin_array_fromasync,
                                        sizeof(qjsc_builtin_array_fromasync),
                                        JS_READ_OBJ_BYTECODE);
            if (JS_IsException(obj))
                return JS_EXCEPTION;
            JSValue fun = JS_EvalFunction(ctx, obj);
            if (JS_IsException(fun))
                return JS_EXCEPTION;
            assert(JS_IsFunction(ctx, fun));
            JSValue args[] = {
                JS_NewCFunction(ctx, js_array_constructor, "Array", 0),
                JS_NewCFunctionMagic(ctx, js_error_constructor, "TypeError", 1,
                                     JS_CFUNC_constructor_or_func_magic,
                                     JS_TYPE_ERROR),
                JS_AtomToValue(ctx, JS_ATOM_Symbol_asyncIterator),
                JS_NewCFunctionMagic(ctx, js_object_defineProperty,
                                     "Object.defineProperty", 3,
                                     JS_CFUNC_generic_magic, 0),
                JS_AtomToValue(ctx, JS_ATOM_Symbol_iterator),
            };
            JSValue result = JS_Call(ctx, fun, JS_UNDEFINED,
                                     countof(args), vc(args));
            for (size_t i = 0; i < countof(args); i++)
                JS_FreeValue(ctx, args[i]);
            JS_FreeValue(ctx, fun);
            if (JS_SetPrototypeInternal(ctx, result, ctx->function_proto,
                                        /*throw_flag*/true) < 0) {
                JS_FreeValue(ctx, result);
                return JS_EXCEPTION;
            }
            return result;
        }
    }
    return JS_UNDEFINED;
}

/* return the value associated to the autoinit property or an exception */
typedef JSValue JSAutoInitFunc(JSContext *ctx, JSObject *p, JSAtom atom, void *opaque);

static JSAutoInitFunc *const js_autoinit_func_table[] = {
    js_instantiate_prototype, /* JS_AUTOINIT_ID_PROTOTYPE */
    js_module_ns_autoinit, /* JS_AUTOINIT_ID_MODULE_NS */
    JS_InstantiateFunctionListItem2, /* JS_AUTOINIT_ID_PROP */
    js_bytecode_autoinit, /* JS_AUTOINIT_ID_BYTECODE */
};

/* warning: 'prs' is reallocated after it */
static int JS_AutoInitProperty(JSContext *ctx, JSObject *p, JSAtom prop,
                               JSProperty *pr, JSShapeProperty *prs)
{
    JSValue val;
    JSContext *realm;
    JSAutoInitFunc *func;

    if (js_shape_prepare_update(ctx, p, &prs))
        return -1;

    realm = js_autoinit_get_realm(pr);
    func = js_autoinit_func_table[js_autoinit_get_id(pr)];
    /* 'func' shall not modify the object properties 'pr' */
    val = func(realm, p, prop, pr->u.init.opaque);
    js_autoinit_free(ctx->rt, pr);
    prs->flags &= ~JS_PROP_TMASK;
    pr->u.value = JS_UNDEFINED;
    if (JS_IsException(val))
        return -1;
    pr->u.value = val;
    return 0;
}

static JSValue JS_GetPropertyInternal(JSContext *ctx, JSValueConst obj,
                                      JSAtom prop, JSValueConst this_obj,
                                      bool throw_ref_error)
{
    JSObject *p;
    JSProperty *pr;
    JSShapeProperty *prs;
    uint32_t tag;

    tag = JS_VALUE_GET_TAG(obj);
    if (unlikely(tag != JS_TAG_OBJECT)) {
        switch(tag) {
        case JS_TAG_NULL:
            return JS_ThrowTypeErrorAtom(ctx, "cannot read property '%s' of null", prop);
        case JS_TAG_UNDEFINED:
            return JS_ThrowTypeErrorAtom(ctx, "cannot read property '%s' of undefined", prop);
        case JS_TAG_EXCEPTION:
            return JS_EXCEPTION;
        case JS_TAG_STRING:
            {
                JSString *p1 = JS_VALUE_GET_STRING(obj);
                if (__JS_AtomIsTaggedInt(prop)) {
                    uint32_t idx, ch;
                    idx = __JS_AtomToUInt32(prop);
                    if (idx < p1->len) {
                        ch = string_get(p1, idx);
                        return js_new_string_char(ctx, ch);
                    }
                } else if (prop == JS_ATOM_length) {
                    return js_int32(p1->len);
                }
            }
            break;
        default:
            break;
        }
        /* cannot raise an exception */
        p = JS_VALUE_GET_OBJ(JS_GetPrototypePrimitive(ctx, obj));
        if (!p)
            return JS_UNDEFINED;
    } else {
        p = JS_VALUE_GET_OBJ(obj);
    }

    for(;;) {
        prs = find_own_property(&pr, p, prop);
        if (prs) {
            /* found */
            if (unlikely(prs->flags & JS_PROP_TMASK)) {
                if ((prs->flags & JS_PROP_TMASK) == JS_PROP_GETSET) {
                    if (unlikely(!pr->u.getset.getter)) {
                        return JS_UNDEFINED;
                    } else {
                        JSValue func = JS_MKPTR(JS_TAG_OBJECT, pr->u.getset.getter);
                        /* Note: the field could be removed in the getter */
                        func = js_dup(func);
                        return JS_CallFree(ctx, func, this_obj, 0, NULL);
                    }
                } else if ((prs->flags & JS_PROP_TMASK) == JS_PROP_VARREF) {
                    JSValue val = *pr->u.var_ref->pvalue;
                    if (unlikely(JS_IsUninitialized(val)))
                        return JS_ThrowReferenceErrorUninitialized(ctx, prs->atom);
                    return js_dup(val);
                } else if ((prs->flags & JS_PROP_TMASK) == JS_PROP_AUTOINIT) {
                    /* Instantiate property and retry */
                    if (JS_AutoInitProperty(ctx, p, prop, pr, prs))
                        return JS_EXCEPTION;
                    continue;
                }
            } else {
                return js_dup(pr->u.value);
            }
        }
        if (unlikely(p->is_exotic)) {
            /* exotic behaviors */
            if (p->fast_array) {
                if (__JS_AtomIsTaggedInt(prop)) {
                    uint32_t idx = __JS_AtomToUInt32(prop);
                    if (idx < p->u.array.count) {
                        /* we avoid duplicating the code */
                        return JS_GetPropertyUint32(ctx, JS_MKPTR(JS_TAG_OBJECT, p), idx);
                    } else if (is_typed_array(p->class_id)) {
                        return JS_UNDEFINED;
                    }
                } else if (is_typed_array(p->class_id)) {
                    int ret;
                    ret = JS_AtomIsNumericIndex(ctx, prop);
                    if (ret != 0) {
                        if (ret < 0)
                            return JS_EXCEPTION;
                        return JS_UNDEFINED;
                    }
                }
            } else {
                const JSClassExoticMethods *em = ctx->rt->class_array[p->class_id].exotic;
                if (em) {
                    if (em->get_property) {
                        JSValue obj1, retval;
                        /* XXX: should pass throw_ref_error */
                        /* Note: if 'p' is a prototype, it can be
                           freed in the called function */
                        obj1 = js_dup(JS_MKPTR(JS_TAG_OBJECT, p));
                        retval = em->get_property(ctx, obj1, prop, this_obj);
                        JS_FreeValue(ctx, obj1);
                        return retval;
                    }
                    if (em->get_own_property) {
                        JSPropertyDescriptor desc;
                        int ret;
                        JSValue obj1;

                        /* Note: if 'p' is a prototype, it can be
                           freed in the called function */
                        obj1 = js_dup(JS_MKPTR(JS_TAG_OBJECT, p));
                        ret = em->get_own_property(ctx, &desc, obj1, prop);
                        JS_FreeValue(ctx, obj1);
                        if (ret < 0)
                            return JS_EXCEPTION;
                        if (ret) {
                            if (desc.flags & JS_PROP_GETSET) {
                                JS_FreeValue(ctx, desc.setter);
                                return JS_CallFree(ctx, desc.getter, this_obj, 0, NULL);
                            } else {
                                return desc.value;
                            }
                        }
                    }
                }
            }
        }
        p = p->shape->proto;
        if (!p)
            break;
    }
    if (unlikely(throw_ref_error)) {
        return JS_ThrowReferenceErrorNotDefined(ctx, prop);
    } else {
        return JS_UNDEFINED;
    }
}

JSValue JS_GetProperty(JSContext *ctx, JSValueConst this_obj, JSAtom prop)
{
    return JS_GetPropertyInternal(ctx, this_obj, prop, this_obj, false);
}

static JSValue JS_ThrowTypeErrorPrivateNotFound(JSContext *ctx, JSAtom atom)
{
    return JS_ThrowTypeErrorAtom(ctx, "private class field '%s' does not exist",
                                 atom);
}

/* Private fields can be added even on non extensible objects or
   Proxies */
static int JS_DefinePrivateField(JSContext *ctx, JSValueConst obj,
                                 JSValue name, JSValue val)
{
    JSObject *p;
    JSShapeProperty *prs;
    JSProperty *pr;
    JSAtom prop;

    if (unlikely(JS_VALUE_GET_TAG(obj) != JS_TAG_OBJECT)) {
        JS_ThrowTypeErrorNotAnObject(ctx);
        goto fail;
    }
    /* safety check */
    if (unlikely(JS_VALUE_GET_TAG(name) != JS_TAG_SYMBOL)) {
        JS_ThrowTypeErrorNotASymbol(ctx);
        goto fail;
    }
    prop = js_symbol_to_atom(ctx, name);
    p = JS_VALUE_GET_OBJ(obj);
    prs = find_own_property(&pr, p, prop);
    if (prs) {
        JS_ThrowTypeErrorAtom(ctx, "private class field '%s' already exists",
                              prop);
        goto fail;
    }
    pr = add_property(ctx, p, prop, JS_PROP_C_W_E);
    if (unlikely(!pr)) {
    fail:
        JS_FreeValue(ctx, val);
        return -1;
    }
    pr->u.value = val;
    return 0;
}

static JSValue JS_GetPrivateField(JSContext *ctx, JSValueConst obj,
                                  JSValueConst name)
{
    JSObject *p;
    JSShapeProperty *prs;
    JSProperty *pr;
    JSAtom prop;

    if (unlikely(JS_VALUE_GET_TAG(obj) != JS_TAG_OBJECT))
        return JS_ThrowTypeErrorNotAnObject(ctx);
    /* safety check */
    if (unlikely(JS_VALUE_GET_TAG(name) != JS_TAG_SYMBOL))
        return JS_ThrowTypeErrorNotASymbol(ctx);
    prop = js_symbol_to_atom(ctx, name);
    p = JS_VALUE_GET_OBJ(obj);
    prs = find_own_property(&pr, p, prop);
    if (!prs) {
        JS_ThrowTypeErrorPrivateNotFound(ctx, prop);
        return JS_EXCEPTION;
    }
    return js_dup(pr->u.value);
}

static int JS_SetPrivateField(JSContext *ctx, JSValueConst obj,
                              JSValueConst name, JSValue val)
{
    JSObject *p;
    JSShapeProperty *prs;
    JSProperty *pr;
    JSAtom prop;

    if (unlikely(JS_VALUE_GET_TAG(obj) != JS_TAG_OBJECT)) {
        JS_ThrowTypeErrorNotAnObject(ctx);
        goto fail;
    }
    /* safety check */
    if (unlikely(JS_VALUE_GET_TAG(name) != JS_TAG_SYMBOL)) {
        JS_ThrowTypeErrorNotASymbol(ctx);
        goto fail;
    }
    prop = js_symbol_to_atom(ctx, name);
    p = JS_VALUE_GET_OBJ(obj);
    prs = find_own_property(&pr, p, prop);
    if (!prs) {
        JS_ThrowTypeErrorPrivateNotFound(ctx, prop);
    fail:
        JS_FreeValue(ctx, val);
        return -1;
    }
    set_value(ctx, &pr->u.value, val);
    return 0;
}

/* add a private brand field to 'home_obj' if not already present and
   if obj is != null add a private brand to it */
static int JS_AddBrand(JSContext *ctx, JSValueConst obj, JSValueConst home_obj)
{
    JSObject *p, *p1;
    JSShapeProperty *prs;
    JSProperty *pr;
    JSValue brand;
    JSAtom brand_atom;

    if (unlikely(JS_VALUE_GET_TAG(home_obj) != JS_TAG_OBJECT)) {
        JS_ThrowTypeErrorNotAnObject(ctx);
        return -1;
    }
    p = JS_VALUE_GET_OBJ(home_obj);
    prs = find_own_property(&pr, p, JS_ATOM_Private_brand);
    if (!prs) {
        /* if the brand is not present, add it */
        brand = JS_NewSymbolFromAtom(ctx, JS_ATOM_brand, JS_ATOM_TYPE_PRIVATE);
        if (JS_IsException(brand))
            return -1;
        pr = add_property(ctx, p, JS_ATOM_Private_brand, JS_PROP_C_W_E);
        if (!pr) {
            JS_FreeValue(ctx, brand);
            return -1;
        }
        pr->u.value = js_dup(brand);
    } else {
        brand = js_dup(pr->u.value);
    }
    brand_atom = js_symbol_to_atom(ctx, brand);

    if (JS_IsObject(obj)) {
        p1 = JS_VALUE_GET_OBJ(obj);
        prs = find_own_property(&pr, p1, brand_atom);
        if (unlikely(prs)) {
            JS_FreeAtom(ctx, brand_atom);
            JS_ThrowTypeError(ctx, "private method is already present");
            return -1;
        }
        pr = add_property(ctx, p1, brand_atom, JS_PROP_C_W_E);
        JS_FreeAtom(ctx, brand_atom);
        if (!pr)
            return -1;
        pr->u.value = JS_UNDEFINED;
    } else {
        JS_FreeAtom(ctx, brand_atom);
    }

    return 0;
}

/* return a boolean telling if the brand of the home object of 'func'
   is present on 'obj' or -1 in case of exception */
static int JS_CheckBrand(JSContext *ctx, JSValue obj, JSValue func)
{
    JSObject *p, *p1, *home_obj;
    JSShapeProperty *prs;
    JSProperty *pr;
    JSValue brand;

    /* get the home object of 'func' */
    if (unlikely(JS_VALUE_GET_TAG(func) != JS_TAG_OBJECT))
        goto not_obj;
    p1 = JS_VALUE_GET_OBJ(func);
    if (!js_class_has_bytecode(p1->class_id))
        goto not_obj;
    home_obj = p1->u.func.home_object;
    if (!home_obj)
        goto not_obj;
    prs = find_own_property(&pr, home_obj, JS_ATOM_Private_brand);
    if (!prs) {
        JS_ThrowTypeError(ctx, "expecting <brand> private field");
        return -1;
    }
    brand = pr->u.value;
    /* safety check */
    if (unlikely(JS_VALUE_GET_TAG(brand) != JS_TAG_SYMBOL))
        goto not_obj;

    /* get the brand array of 'obj' */
    if (unlikely(JS_VALUE_GET_TAG(obj) != JS_TAG_OBJECT)) {
    not_obj:
        JS_ThrowTypeErrorNotAnObject(ctx);
        return -1;
    }
    p = JS_VALUE_GET_OBJ(obj);
    prs = find_own_property(&pr, p, js_symbol_to_atom(ctx, brand));
    return (prs != NULL);
}

static uint32_t js_string_obj_get_length(JSContext *ctx, JSValueConst obj)
{
    JSObject *p;
    JSString *p1;
    uint32_t len = 0;

    /* This is a class exotic method: obj class_id is JS_CLASS_STRING */
    p = JS_VALUE_GET_OBJ(obj);
    if (JS_VALUE_GET_TAG(p->u.object_data) == JS_TAG_STRING) {
        p1 = JS_VALUE_GET_STRING(p->u.object_data);
        len = p1->len;
    }
    return len;
}

static int num_keys_cmp(const void *p1, const void *p2, void *opaque)
{
    JSContext *ctx = opaque;
    JSAtom atom1 = ((const JSPropertyEnum *)p1)->atom;
    JSAtom atom2 = ((const JSPropertyEnum *)p2)->atom;
    uint32_t v1, v2;
    bool atom1_is_integer, atom2_is_integer;

    atom1_is_integer = JS_AtomIsArrayIndex(ctx, &v1, atom1);
    atom2_is_integer = JS_AtomIsArrayIndex(ctx, &v2, atom2);
    assert(atom1_is_integer && atom2_is_integer);
    if (v1 < v2)
        return -1;
    else if (v1 == v2)
        return 0;
    else
        return 1;
}

static void js_free_prop_enum(JSContext *ctx, JSPropertyEnum *tab, uint32_t len)
{
    uint32_t i;
    if (tab) {
        for(i = 0; i < len; i++)
            JS_FreeAtom(ctx, tab[i].atom);
        js_free(ctx, tab);
    }
}

/* return < 0 in case if exception, 0 if OK. ptab and its atoms must
   be freed by the user. */
static int __exception JS_GetOwnPropertyNamesInternal(JSContext *ctx,
                                                      JSPropertyEnum **ptab,
                                                      uint32_t *plen,
                                                      JSObject *p, int flags)
{
    int i, j;
    JSShape *sh;
    JSShapeProperty *prs;
    JSPropertyEnum *tab_atom, *tab_exotic;
    JSAtom atom;
    uint32_t num_keys_count, str_keys_count, sym_keys_count, atom_count;
    uint32_t num_index, str_index, sym_index, exotic_count, exotic_keys_count;
    bool is_enumerable, num_sorted;
    uint32_t num_key;
    JSAtomKindEnum kind;

    /* clear pointer for consistency in case of failure */
    *ptab = NULL;
    *plen = 0;

    /* compute the number of returned properties */
    num_keys_count = 0;
    str_keys_count = 0;
    sym_keys_count = 0;
    exotic_keys_count = 0;
    exotic_count = 0;
    tab_exotic = NULL;
    sh = p->shape;
    for(i = 0, prs = get_shape_prop(sh); i < sh->prop_count; i++, prs++) {
        atom = prs->atom;
        if (atom != JS_ATOM_NULL) {
            is_enumerable = ((prs->flags & JS_PROP_ENUMERABLE) != 0);
            kind = JS_AtomGetKind(ctx, atom);
            if ((!(flags & JS_GPN_ENUM_ONLY) || is_enumerable) &&
                ((flags >> kind) & 1) != 0) {
                /* need to raise an exception in case of the module
                   name space (implicit GetOwnProperty) */
                if (unlikely((prs->flags & JS_PROP_TMASK) == JS_PROP_VARREF) &&
                    (flags & (JS_GPN_SET_ENUM | JS_GPN_ENUM_ONLY))) {
                    JSVarRef *var_ref = p->prop[i].u.var_ref;
                    if (unlikely(JS_IsUninitialized(*var_ref->pvalue))) {
                        JS_ThrowReferenceErrorUninitialized(ctx, prs->atom);
                        return -1;
                    }
                }
                if (JS_AtomIsArrayIndex(ctx, &num_key, atom)) {
                    num_keys_count++;
                } else if (kind == JS_ATOM_KIND_STRING) {
                    str_keys_count++;
                } else {
                    sym_keys_count++;
                }
            }
        }
    }

    if (p->is_exotic) {
        if (p->fast_array) {
            if (flags & JS_GPN_STRING_MASK) {
                num_keys_count += p->u.array.count;
            }
        } else if (p->class_id == JS_CLASS_STRING) {
            if (flags & JS_GPN_STRING_MASK) {
                num_keys_count += js_string_obj_get_length(ctx, JS_MKPTR(JS_TAG_OBJECT, p));
            }
        } else {
            const JSClassExoticMethods *em = ctx->rt->class_array[p->class_id].exotic;
            if (em && em->get_own_property_names) {
                if (em->get_own_property_names(ctx, &tab_exotic, &exotic_count,
                                               JS_MKPTR(JS_TAG_OBJECT, p)))
                    return -1;
                for(i = 0; i < exotic_count; i++) {
                    atom = tab_exotic[i].atom;
                    kind = JS_AtomGetKind(ctx, atom);
                    if (((flags >> kind) & 1) != 0) {
                        is_enumerable = false;
                        if (flags & (JS_GPN_SET_ENUM | JS_GPN_ENUM_ONLY)) {
                            JSPropertyDescriptor desc;
                            int res;
                            /* set the "is_enumerable" field if necessary */
                            res = JS_GetOwnPropertyInternal(ctx, &desc, p, atom);
                            if (res < 0) {
                                js_free_prop_enum(ctx, tab_exotic, exotic_count);
                                return -1;
                            }
                            if (res) {
                                is_enumerable =
                                    ((desc.flags & JS_PROP_ENUMERABLE) != 0);
                                js_free_desc(ctx, &desc);
                            }
                            tab_exotic[i].is_enumerable = is_enumerable;
                        }
                        if (!(flags & JS_GPN_ENUM_ONLY) || is_enumerable) {
                            exotic_keys_count++;
                        }
                    }
                }
            }
        }
    }

    /* fill them */

    atom_count = num_keys_count + str_keys_count + sym_keys_count + exotic_keys_count;
    /* avoid allocating 0 bytes */
    tab_atom = js_malloc(ctx, sizeof(tab_atom[0]) * max_int(atom_count, 1));
    if (!tab_atom) {
        js_free_prop_enum(ctx, tab_exotic, exotic_count);
        return -1;
    }

    num_index = 0;
    str_index = num_keys_count;
    sym_index = str_index + str_keys_count;

    num_sorted = true;
    sh = p->shape;
    for(i = 0, prs = get_shape_prop(sh); i < sh->prop_count; i++, prs++) {
        atom = prs->atom;
        if (atom != JS_ATOM_NULL) {
            is_enumerable = ((prs->flags & JS_PROP_ENUMERABLE) != 0);
            kind = JS_AtomGetKind(ctx, atom);
            if ((!(flags & JS_GPN_ENUM_ONLY) || is_enumerable) &&
                ((flags >> kind) & 1) != 0) {
                if (JS_AtomIsArrayIndex(ctx, &num_key, atom)) {
                    j = num_index++;
                    num_sorted = false;
                } else if (kind == JS_ATOM_KIND_STRING) {
                    j = str_index++;
                } else {
                    j = sym_index++;
                }
                tab_atom[j].atom = JS_DupAtom(ctx, atom);
                tab_atom[j].is_enumerable = is_enumerable;
            }
        }
    }

    if (p->is_exotic) {
        int len;
        if (p->fast_array) {
            if (flags & JS_GPN_STRING_MASK) {
                len = p->u.array.count;
                goto add_array_keys;
            }
        } else if (p->class_id == JS_CLASS_STRING) {
            if (flags & JS_GPN_STRING_MASK) {
                len = js_string_obj_get_length(ctx, JS_MKPTR(JS_TAG_OBJECT, p));
            add_array_keys:
                for(i = 0; i < len; i++) {
                    tab_atom[num_index].atom = __JS_AtomFromUInt32(i);
                    if (tab_atom[num_index].atom == JS_ATOM_NULL) {
                        js_free_prop_enum(ctx, tab_atom, num_index);
                        return -1;
                    }
                    tab_atom[num_index].is_enumerable = true;
                    num_index++;
                }
            }
        } else {
            /* Note: exotic keys are not reordered and comes after the object own properties. */
            for(i = 0; i < exotic_count; i++) {
                atom = tab_exotic[i].atom;
                is_enumerable = tab_exotic[i].is_enumerable;
                kind = JS_AtomGetKind(ctx, atom);
                if ((!(flags & JS_GPN_ENUM_ONLY) || is_enumerable) &&
                    ((flags >> kind) & 1) != 0) {
                    tab_atom[sym_index].atom = atom;
                    tab_atom[sym_index].is_enumerable = is_enumerable;
                    sym_index++;
                } else {
                    JS_FreeAtom(ctx, atom);
                }
            }
            js_free(ctx, tab_exotic);
        }
    }

    assert(num_index == num_keys_count);
    assert(str_index == num_keys_count + str_keys_count);
    assert(sym_index == atom_count);

    if (num_keys_count != 0 && !num_sorted) {
        rqsort(tab_atom, num_keys_count, sizeof(tab_atom[0]), num_keys_cmp,
               ctx);
    }
    *ptab = tab_atom;
    *plen = atom_count;
    return 0;
}

int JS_GetOwnPropertyNames(JSContext *ctx, JSPropertyEnum **ptab,
                           uint32_t *plen, JSValueConst obj, int flags)
{
    if (JS_VALUE_GET_TAG(obj) != JS_TAG_OBJECT) {
        JS_ThrowTypeErrorNotAnObject(ctx);
        return -1;
    }
    return JS_GetOwnPropertyNamesInternal(ctx, ptab, plen,
                                          JS_VALUE_GET_OBJ(obj), flags);
}

/* Return -1 if exception,
   false if the property does not exist, true if it exists. If true is
   returned, the property descriptor 'desc' is filled present. */
static int JS_GetOwnPropertyInternal(JSContext *ctx, JSPropertyDescriptor *desc,
                                     JSObject *p, JSAtom prop)
{
    JSShapeProperty *prs;
    JSProperty *pr;

retry:
    prs = find_own_property(&pr, p, prop);
    if (prs) {
        if (desc) {
            desc->flags = prs->flags & JS_PROP_C_W_E;
            desc->getter = JS_UNDEFINED;
            desc->setter = JS_UNDEFINED;
            desc->value = JS_UNDEFINED;
            if (unlikely(prs->flags & JS_PROP_TMASK)) {
                if ((prs->flags & JS_PROP_TMASK) == JS_PROP_GETSET) {
                    desc->flags |= JS_PROP_GETSET;
                    if (pr->u.getset.getter)
                        desc->getter = js_dup(JS_MKPTR(JS_TAG_OBJECT, pr->u.getset.getter));
                    if (pr->u.getset.setter)
                        desc->setter = js_dup(JS_MKPTR(JS_TAG_OBJECT, pr->u.getset.setter));
                } else if ((prs->flags & JS_PROP_TMASK) == JS_PROP_VARREF) {
                    JSValue val = *pr->u.var_ref->pvalue;
                    if (unlikely(JS_IsUninitialized(val))) {
                        JS_ThrowReferenceErrorUninitialized(ctx, prs->atom);
                        return -1;
                    }
                    desc->value = js_dup(val);
                } else if ((prs->flags & JS_PROP_TMASK) == JS_PROP_AUTOINIT) {
                    /* Instantiate property and retry */
                    if (JS_AutoInitProperty(ctx, p, prop, pr, prs))
                        return -1;
                    goto retry;
                }
            } else {
                desc->value = js_dup(pr->u.value);
            }
        } else {
            /* for consistency, send the exception even if desc is NULL */
            if (unlikely((prs->flags & JS_PROP_TMASK) == JS_PROP_VARREF)) {
                if (unlikely(JS_IsUninitialized(*pr->u.var_ref->pvalue))) {
                    JS_ThrowReferenceErrorUninitialized(ctx, prs->atom);
                    return -1;
                }
            } else if ((prs->flags & JS_PROP_TMASK) == JS_PROP_AUTOINIT) {
                /* nothing to do: delay instantiation until actual value and/or attributes are read */
            }
        }
        return true;
    }
    if (p->is_exotic) {
        if (p->fast_array) {
            /* specific case for fast arrays */
            if (__JS_AtomIsTaggedInt(prop)) {
                uint32_t idx;
                idx = __JS_AtomToUInt32(prop);
                if (idx < p->u.array.count) {
                    if (desc) {
                        desc->flags = JS_PROP_WRITABLE | JS_PROP_ENUMERABLE |
                            JS_PROP_CONFIGURABLE;
                        desc->getter = JS_UNDEFINED;
                        desc->setter = JS_UNDEFINED;
                        desc->value = JS_GetPropertyUint32(ctx, JS_MKPTR(JS_TAG_OBJECT, p), idx);
                    }
                    return true;
                }
            }
        } else {
            const JSClassExoticMethods *em = ctx->rt->class_array[p->class_id].exotic;
            if (em && em->get_own_property) {
                return em->get_own_property(ctx, desc,
                                            JS_MKPTR(JS_TAG_OBJECT, p), prop);
            }
        }
    }
    return false;
}

int JS_GetOwnProperty(JSContext *ctx, JSPropertyDescriptor *desc,
                      JSValueConst obj, JSAtom prop)
{
    if (JS_VALUE_GET_TAG(obj) != JS_TAG_OBJECT) {
        JS_ThrowTypeErrorNotAnObject(ctx);
        return -1;
    }
    return JS_GetOwnPropertyInternal(ctx, desc, JS_VALUE_GET_OBJ(obj), prop);
}

void JS_FreePropertyEnum(JSContext *ctx, JSPropertyEnum *tab,
                         uint32_t len)
{
    js_free_prop_enum(ctx, tab, len);
}

/* return -1 if exception (Proxy object only) or true/false */
int JS_IsExtensible(JSContext *ctx, JSValueConst obj)
{
    JSObject *p;

    if (unlikely(JS_VALUE_GET_TAG(obj) != JS_TAG_OBJECT))
        return false;
    p = JS_VALUE_GET_OBJ(obj);
    if (unlikely(p->class_id == JS_CLASS_PROXY))
        return js_proxy_isExtensible(ctx, obj);
    else
        return p->extensible;
}

/* return -1 if exception (Proxy object only) or true/false */
int JS_PreventExtensions(JSContext *ctx, JSValueConst obj)
{
    JSObject *p;

    if (unlikely(JS_VALUE_GET_TAG(obj) != JS_TAG_OBJECT))
        return false;
    p = JS_VALUE_GET_OBJ(obj);
    if (unlikely(p->class_id == JS_CLASS_PROXY))
        return js_proxy_preventExtensions(ctx, obj);
    p->extensible = false;
    return true;
}

/* return -1 if exception otherwise true or false */
int JS_HasProperty(JSContext *ctx, JSValueConst obj, JSAtom prop)
{
    JSObject *p;
    int ret;
    JSValue obj1;

    if (unlikely(JS_VALUE_GET_TAG(obj) != JS_TAG_OBJECT))
        return false;
    p = JS_VALUE_GET_OBJ(obj);
    for(;;) {
        if (p->is_exotic) {
            const JSClassExoticMethods *em = ctx->rt->class_array[p->class_id].exotic;
            if (em && em->has_property) {
                /* has_property can free the prototype */
                obj1 = js_dup(JS_MKPTR(JS_TAG_OBJECT, p));
                ret = em->has_property(ctx, obj1, prop);
                JS_FreeValue(ctx, obj1);
                return ret;
            }
        }
        /* JS_GetOwnPropertyInternal can free the prototype */
        js_dup(JS_MKPTR(JS_TAG_OBJECT, p));
        ret = JS_GetOwnPropertyInternal(ctx, NULL, p, prop);
        JS_FreeValue(ctx, JS_MKPTR(JS_TAG_OBJECT, p));
        if (ret != 0)
            return ret;
        if (is_typed_array(p->class_id)) {
            ret = JS_AtomIsNumericIndex(ctx, prop);
            if (ret != 0) {
                if (ret < 0)
                    return -1;
                return false;
            }
        }
        p = p->shape->proto;
        if (!p)
            break;
    }
    return false;
}

/* val must be a symbol */
static JSAtom js_symbol_to_atom(JSContext *ctx, JSValueConst val)
{
    JSAtomStruct *p = JS_VALUE_GET_PTR(val);
    return js_get_atom_index(ctx->rt, p);
}

/* return JS_ATOM_NULL in case of exception */
static JSAtom JS_ValueToAtomInternal(JSContext *ctx, JSValueConst val,
                                     int flags)
{
    JSAtom atom;
    uint32_t tag;
    tag = JS_VALUE_GET_TAG(val);
    if (tag == JS_TAG_INT &&
        (uint32_t)JS_VALUE_GET_INT(val) <= JS_ATOM_MAX_INT) {
        /* fast path for integer values */
        atom = __JS_AtomFromUInt32(JS_VALUE_GET_INT(val));
    } else if (tag == JS_TAG_SYMBOL) {
        JSAtomStruct *p = JS_VALUE_GET_PTR(val);
        atom = JS_DupAtom(ctx, js_get_atom_index(ctx->rt, p));
    } else {
        JSValue str;
        str = JS_ToPropertyKeyInternal(ctx, val, flags);
        if (JS_IsException(str))
            return JS_ATOM_NULL;
        if (JS_VALUE_GET_TAG(str) == JS_TAG_SYMBOL) {
            atom = js_symbol_to_atom(ctx, str);
        } else {
            atom = JS_NewAtomStr(ctx, JS_VALUE_GET_STRING(str));
        }
    }
    return atom;
}

JSAtom JS_ValueToAtom(JSContext *ctx, JSValueConst val)
{
    return JS_ValueToAtomInternal(ctx, val, /*flags*/0);
}

static bool js_get_fast_array_element(JSContext *ctx, JSObject *p,
                                      uint32_t idx, JSValue *pval)
{
    switch(p->class_id) {
    case JS_CLASS_ARRAY:
    case JS_CLASS_ARGUMENTS:
        if (unlikely(idx >= p->u.array.count)) return false;
        *pval = js_dup(p->u.array.u.values[idx]);
        return true;
    case JS_CLASS_INT8_ARRAY:
        if (unlikely(idx >= p->u.array.count)) return false;
        *pval = js_int32(p->u.array.u.int8_ptr[idx]);
        return true;
    case JS_CLASS_UINT8C_ARRAY:
    case JS_CLASS_UINT8_ARRAY:
        if (unlikely(idx >= p->u.array.count)) return false;
        *pval = js_int32(p->u.array.u.uint8_ptr[idx]);
        return true;
    case JS_CLASS_INT16_ARRAY:
        if (unlikely(idx >= p->u.array.count)) return false;
        *pval = js_int32(p->u.array.u.int16_ptr[idx]);
        return true;
    case JS_CLASS_UINT16_ARRAY:
        if (unlikely(idx >= p->u.array.count)) return false;
        *pval = js_int32(p->u.array.u.uint16_ptr[idx]);
        return true;
    case JS_CLASS_INT32_ARRAY:
        if (unlikely(idx >= p->u.array.count)) return false;
        *pval = js_int32(p->u.array.u.int32_ptr[idx]);
        return true;
    case JS_CLASS_UINT32_ARRAY:
        if (unlikely(idx >= p->u.array.count)) return false;
        *pval = js_uint32(p->u.array.u.uint32_ptr[idx]);
        return true;
    case JS_CLASS_BIG_INT64_ARRAY:
        if (unlikely(idx >= p->u.array.count)) return false;
        *pval = JS_NewBigInt64(ctx, p->u.array.u.int64_ptr[idx]);
        return true;
    case JS_CLASS_BIG_UINT64_ARRAY:
        if (unlikely(idx >= p->u.array.count)) return false;
        *pval = JS_NewBigUint64(ctx, p->u.array.u.uint64_ptr[idx]);
        return true;
    case JS_CLASS_FLOAT16_ARRAY:
        if (unlikely(idx >= p->u.array.count)) return false;
        *pval = js_float64(fromfp16(p->u.array.u.fp16_ptr[idx]));
        return true;
    case JS_CLASS_FLOAT32_ARRAY:
        if (unlikely(idx >= p->u.array.count)) return false;
        *pval = js_float64(p->u.array.u.float_ptr[idx]);
        return true;
    case JS_CLASS_FLOAT64_ARRAY:
        if (unlikely(idx >= p->u.array.count)) return false;
        *pval = js_float64(p->u.array.u.double_ptr[idx]);
        return true;
    default:
        return false;
    }
}

static JSValue JS_GetPropertyValue(JSContext *ctx, JSValueConst this_obj,
                                   JSValue prop)
{
    JSAtom atom;
    JSValue ret;
    uint32_t tag;

    tag = JS_VALUE_GET_TAG(this_obj);
    if (likely(tag == JS_TAG_OBJECT)) {
        if (JS_VALUE_GET_TAG(prop) == JS_TAG_INT) {
            JSObject *p = JS_VALUE_GET_OBJ(this_obj);
            uint32_t idx = JS_VALUE_GET_INT(prop);
            JSValue val;
            /* fast path for array and typed array access */
            if (js_get_fast_array_element(ctx, p, idx, &val))
                return val;
        }
    } else if (unlikely(tag == JS_TAG_NULL || tag == JS_TAG_UNDEFINED)) {
        // per spec: not allowed to call ToPropertyKey before ToObject
        // so we must ensure to not invoke JS anything that's observable
        // from JS code
        atom = JS_ValueToAtomInternal(ctx, prop, JS_TO_STRING_NO_SIDE_EFFECTS);
        JS_FreeValue(ctx, prop);
        if (unlikely(atom == JS_ATOM_NULL))
            return JS_EXCEPTION;
        if (tag == JS_TAG_NULL) {
            JS_ThrowTypeErrorAtom(ctx, "cannot read property '%s' of null", atom);
        } else {
            JS_ThrowTypeErrorAtom(ctx, "cannot read property '%s' of undefined", atom);
        }
        JS_FreeAtom(ctx, atom);
        return JS_EXCEPTION;
    }
    atom = JS_ValueToAtom(ctx, prop);
    JS_FreeValue(ctx, prop);
    if (unlikely(atom == JS_ATOM_NULL))
        return JS_EXCEPTION;
    ret = JS_GetProperty(ctx, this_obj, atom);
    JS_FreeAtom(ctx, atom);
    return ret;
}

JSValue JS_GetPropertyUint32(JSContext *ctx, JSValueConst this_obj,
                             uint32_t idx)
{
    return JS_GetPropertyInt64(ctx, this_obj, idx);
}

/* Check if an object has a generalized numeric property. Return value:
   -1 for exception, *pval set to JS_EXCEPTION
   true if property exists, stored into *pval,
   false if property does not exist. *pval set to JS_UNDEFINED.
 */
static int JS_TryGetPropertyInt64(JSContext *ctx, JSValueConst obj, int64_t idx, JSValue *pval)
{
    JSValue val;
    JSAtom prop;
    int present;

    if (likely(JS_VALUE_GET_TAG(obj) == JS_TAG_OBJECT &&
               (uint64_t)idx <= INT32_MAX)) {
        /* fast path for array and typed array access */
        JSObject *p = JS_VALUE_GET_OBJ(obj);
        if (js_get_fast_array_element(ctx, p, idx, pval))
            return true;
    }
    val = JS_EXCEPTION;
    present = -1;
    prop = JS_NewAtomInt64(ctx, idx);
    if (likely(prop != JS_ATOM_NULL)) {
        present = JS_HasProperty(ctx, obj, prop);
        if (present > 0) {
            val = JS_GetProperty(ctx, obj, prop);
            if (unlikely(JS_IsException(val)))
                present = -1;
        } else if (present == false) {
            val = JS_UNDEFINED;
        }
        JS_FreeAtom(ctx, prop);
    }
    *pval = val;
    return present;
}

JSValue JS_GetPropertyInt64(JSContext *ctx, JSValueConst obj, int64_t idx)
{
    JSAtom prop;
    JSValue val;

    if (likely(JS_VALUE_GET_TAG(obj) == JS_TAG_OBJECT &&
               (uint64_t)idx <= INT32_MAX)) {
        /* fast path for array and typed array access */
        JSObject *p = JS_VALUE_GET_OBJ(obj);
        if (js_get_fast_array_element(ctx, p, idx, &val))
            return val;
    }
    prop = JS_NewAtomInt64(ctx, idx);
    if (prop == JS_ATOM_NULL)
        return JS_EXCEPTION;

    val = JS_GetProperty(ctx, obj, prop);
    JS_FreeAtom(ctx, prop);
    return val;
}

/* `prop` may be pure ASCII or UTF-8 encoded */
JSValue JS_GetPropertyStr(JSContext *ctx, JSValueConst this_obj,
                          const char *prop)
{
    JSAtom atom;
    JSValue ret;
    atom = JS_NewAtom(ctx, prop);
    if (atom == JS_ATOM_NULL)
        return JS_EXCEPTION;
    ret = JS_GetProperty(ctx, this_obj, atom);
    JS_FreeAtom(ctx, atom);
    return ret;
}

/* Note: the property value is not initialized. Return NULL if memory
   error. */
static JSProperty *add_property(JSContext *ctx,
                                JSObject *p, JSAtom prop, int prop_flags)
{
    JSShape *sh, *new_sh;

    sh = p->shape;
    if (sh->is_hashed) {
        /* try to find an existing shape */
        new_sh = find_hashed_shape_prop(ctx->rt, sh, prop, prop_flags);
        if (new_sh) {
            /* matching shape found: use it */
            /*  the property array may need to be resized */
            if (new_sh->prop_size != sh->prop_size) {
                JSProperty *new_prop;
                new_prop = js_realloc(ctx, p->prop, sizeof(p->prop[0]) *
                                      new_sh->prop_size);
                if (!new_prop)
                    return NULL;
                p->prop = new_prop;
            }
            p->shape = js_dup_shape(new_sh);
            js_free_shape(ctx->rt, sh);
            return &p->prop[new_sh->prop_count - 1];
        } else if (sh->header.ref_count != 1) {
            /* if the shape is shared, clone it */
            new_sh = js_clone_shape(ctx, sh);
            if (!new_sh)
                return NULL;
            /* hash the cloned shape */
            new_sh->is_hashed = true;
            js_shape_hash_link(ctx->rt, new_sh);
            js_free_shape(ctx->rt, p->shape);
            p->shape = new_sh;
        }
    }
    assert(p->shape->header.ref_count == 1);
    if (add_shape_property(ctx, &p->shape, p, prop, prop_flags))
        return NULL;
    return &p->prop[p->shape->prop_count - 1];
}

/* can be called on Array or Arguments objects. return < 0 if
   memory alloc error. */
static no_inline __exception int convert_fast_array_to_array(JSContext *ctx,
                                                             JSObject *p)
{
    JSProperty *pr;
    JSShape *sh;
    JSValue *tab;
    uint32_t i, len, new_count;

    if (js_shape_prepare_update(ctx, p, NULL))
        return -1;
    len = p->u.array.count;
    /* resize the properties once to simplify the error handling */
    sh = p->shape;
    new_count = sh->prop_count + len;
    if (new_count > sh->prop_size) {
        if (resize_properties(ctx, &p->shape, p, new_count))
            return -1;
    }

    tab = p->u.array.u.values;
    for(i = 0; i < len; i++) {
        /* add_property cannot fail here but
           __JS_AtomFromUInt32(i) fails for i > INT32_MAX */
        pr = add_property(ctx, p, __JS_AtomFromUInt32(i), JS_PROP_C_W_E);
        pr->u.value = *tab++;
    }
    js_free(ctx, p->u.array.u.values);
    p->u.array.count = 0;
    p->u.array.u.values = NULL; /* fail safe */
    p->u.array.u1.size = 0;
    p->fast_array = 0;
    return 0;
}

static int delete_property(JSContext *ctx, JSObject *p, JSAtom atom)
{
    JSShape *sh;
    JSShapeProperty *pr, *lpr, *prop;
    JSProperty *pr1;
    uint32_t lpr_idx;
    intptr_t h, h1;

 redo:
    sh = p->shape;
    h1 = atom & sh->prop_hash_mask;
    h = prop_hash_end(sh)[-h1 - 1];
    prop = get_shape_prop(sh);
    lpr = NULL;
    lpr_idx = 0;   /* prevent warning */
    while (h != 0) {
        pr = &prop[h - 1];
        if (likely(pr->atom == atom)) {
            /* found ! */
            if (!(pr->flags & JS_PROP_CONFIGURABLE))
                return false;
            /* realloc the shape if needed */
            if (lpr)
                lpr_idx = lpr - get_shape_prop(sh);
            if (js_shape_prepare_update(ctx, p, &pr))
                return -1;
            sh = p->shape;
            /* remove property */
            if (lpr) {
                lpr = get_shape_prop(sh) + lpr_idx;
                lpr->hash_next = pr->hash_next;
            } else {
                prop_hash_end(sh)[-h1 - 1] = pr->hash_next;
            }
            sh->deleted_prop_count++;
            /* free the entry */
            pr1 = &p->prop[h - 1];
            free_property(ctx->rt, pr1, pr->flags);
            JS_FreeAtom(ctx, pr->atom);
            /* put default values */
            pr->flags = 0;
            pr->atom = JS_ATOM_NULL;
            pr1->u.value = JS_UNDEFINED;

            /* compact the properties if too many deleted properties */
            if (sh->deleted_prop_count >= 8 &&
                sh->deleted_prop_count >= ((unsigned)sh->prop_count / 2)) {
                compact_properties(ctx, p);
            }
            return true;
        }
        lpr = pr;
        h = pr->hash_next;
    }

    if (p->is_exotic) {
        if (p->fast_array) {
            uint32_t idx;
            if (JS_AtomIsArrayIndex(ctx, &idx, atom) &&
                idx < p->u.array.count) {
                if (p->class_id == JS_CLASS_ARRAY ||
                    p->class_id == JS_CLASS_ARGUMENTS) {
                    /* Special case deleting the last element of a fast Array */
                    if (idx == p->u.array.count - 1) {
                        JS_FreeValue(ctx, p->u.array.u.values[idx]);
                        p->u.array.count = idx;
                        return true;
                    }
                    if (convert_fast_array_to_array(ctx, p))
                        return -1;
                    goto redo;
                } else {
                    return false;
                }
            }
        } else {
            const JSClassExoticMethods *em = ctx->rt->class_array[p->class_id].exotic;
            if (em && em->delete_property) {
                return em->delete_property(ctx, JS_MKPTR(JS_TAG_OBJECT, p), atom);
            }
        }
    }
    /* not found */
    return true;
}

static int call_setter(JSContext *ctx, JSObject *setter,
                       JSValueConst this_obj, JSValue val, int flags)
{
    JSValue ret, func;
    if (likely(setter)) {
        func = JS_MKPTR(JS_TAG_OBJECT, setter);
        /* Note: the field could be removed in the setter */
        func = js_dup(func);
        ret = JS_CallFree(ctx, func, this_obj, 1, vc(&val));
        JS_FreeValue(ctx, val);
        if (JS_IsException(ret))
            return -1;
        JS_FreeValue(ctx, ret);
        return true;
    } else {
        JS_FreeValue(ctx, val);
        if ((flags & JS_PROP_THROW) ||
            ((flags & JS_PROP_THROW_STRICT) && is_strict_mode(ctx))) {
            JS_ThrowTypeError(ctx, "no setter for property");
            return -1;
        }
        return false;
    }
}

/* set the array length and remove the array elements if necessary. */
static int set_array_length(JSContext *ctx, JSObject *p, JSValue val,
                            int flags)
{
    uint32_t len, idx, cur_len;
    int i, ret;

    /* Note: this call can reallocate the properties of 'p' */
    ret = JS_ToArrayLengthFree(ctx, &len, val, false);
    if (ret)
        return -1;
    /* JS_ToArrayLengthFree() must be done before the read-only test */
    if (unlikely(!(p->shape->prop[0].flags & JS_PROP_WRITABLE)))
        return JS_ThrowTypeErrorReadOnly(ctx, flags, JS_ATOM_length);

    if (likely(p->fast_array)) {
        uint32_t old_len = p->u.array.count;
        if (len < old_len) {
            for(i = len; i < old_len; i++) {
                JS_FreeValue(ctx, p->u.array.u.values[i]);
            }
            p->u.array.count = len;
        }
        p->prop[0].u.value = js_uint32(len);
    } else {
        /* Note: length is always a uint32 because the object is an
           array */
        JS_ToUint32(ctx, &cur_len, p->prop[0].u.value);
        if (len < cur_len) {
            uint32_t d;
            JSShape *sh;
            JSShapeProperty *pr;

            d = cur_len - len;
            sh = p->shape;
            if (d <= sh->prop_count) {
                JSAtom atom;

                /* faster to iterate */
                while (cur_len > len) {
                    atom = JS_NewAtomUInt32(ctx, cur_len - 1);
                    ret = delete_property(ctx, p, atom);
                    JS_FreeAtom(ctx, atom);
                    if (unlikely(!ret)) {
                        /* unlikely case: property is not
                           configurable */
                        break;
                    }
                    cur_len--;
                }
            } else {
                /* faster to iterate thru all the properties. Need two
                   passes in case one of the property is not
                   configurable */
                cur_len = len;
                for(i = 0, pr = get_shape_prop(sh); i < sh->prop_count;
                    i++, pr++) {
                    if (pr->atom != JS_ATOM_NULL &&
                        JS_AtomIsArrayIndex(ctx, &idx, pr->atom)) {
                        if (idx >= cur_len &&
                            !(pr->flags & JS_PROP_CONFIGURABLE)) {
                            cur_len = idx + 1;
                        }
                    }
                }

                for(i = 0, pr = get_shape_prop(sh); i < sh->prop_count;
                    i++, pr++) {
                    if (pr->atom != JS_ATOM_NULL &&
                        JS_AtomIsArrayIndex(ctx, &idx, pr->atom)) {
                        if (idx >= cur_len) {
                            /* remove the property */
                            delete_property(ctx, p, pr->atom);
                            /* WARNING: the shape may have been modified */
                            sh = p->shape;
                            pr = get_shape_prop(sh) + i;
                        }
                    }
                }
            }
        } else {
            cur_len = len;
        }
        set_value(ctx, &p->prop[0].u.value, js_uint32(cur_len));
        if (unlikely(cur_len > len)) {
            return JS_ThrowTypeErrorOrFalse(ctx, flags, "not configurable");
        }
    }
    return true;
}

/* return -1 if exception */
static int expand_fast_array(JSContext *ctx, JSObject *p, uint32_t new_len)
{
    uint32_t new_size;
    size_t slack;
    JSValue *new_array_prop;
    /* XXX: potential arithmetic overflow */
    new_size = max_int(new_len, p->u.array.u1.size * 3 / 2);
    new_array_prop = js_realloc2(ctx, p->u.array.u.values, sizeof(JSValue) * new_size, &slack);
    if (!new_array_prop)
        return -1;
    new_size += slack / sizeof(*new_array_prop);
    p->u.array.u.values = new_array_prop;
    p->u.array.u1.size = new_size;
    return 0;
}

/* Preconditions: 'p' must be of class JS_CLASS_ARRAY, p->fast_array =
   true and p->extensible = true */
static int add_fast_array_element(JSContext *ctx, JSObject *p,
                                  JSValue val, int flags)
{
    uint32_t new_len, array_len;
    /* extend the array by one */
    /* XXX: convert to slow array if new_len > 2^31-1 elements */
    new_len = p->u.array.count + 1;
    /* update the length if necessary. We assume that if the length is
       not an integer, then if it >= 2^31.  */
    if (likely(JS_VALUE_GET_TAG(p->prop[0].u.value) == JS_TAG_INT)) {
        array_len = JS_VALUE_GET_INT(p->prop[0].u.value);
        if (new_len > array_len) {
            if (unlikely(!(get_shape_prop(p->shape)->flags & JS_PROP_WRITABLE))) {
                JS_FreeValue(ctx, val);
                return JS_ThrowTypeErrorReadOnly(ctx, flags, JS_ATOM_length);
            }
            p->prop[0].u.value = js_int32(new_len);
        }
    }
    if (unlikely(new_len > p->u.array.u1.size)) {
        if (expand_fast_array(ctx, p, new_len)) {
            JS_FreeValue(ctx, val);
            return -1;
        }
    }
    p->u.array.u.values[new_len - 1] = val;
    p->u.array.count = new_len;
    return true;
}

static void js_free_desc(JSContext *ctx, JSPropertyDescriptor *desc)
{
    JS_FreeValue(ctx, desc->getter);
    JS_FreeValue(ctx, desc->setter);
    JS_FreeValue(ctx, desc->value);
}

/* return -1 in case of exception or true or false. Warning: 'val' is
   freed by the function. 'flags' is a bitmask of JS_PROP_NO_ADD,
   JS_PROP_THROW or JS_PROP_THROW_STRICT. If JS_PROP_NO_ADD is set,
   the new property is not added and an error is raised.
   'obj' must be an object when obj != this_obj.
   */
static int JS_SetPropertyInternal2(JSContext *ctx, JSValueConst obj, JSAtom prop,
                                   JSValue val, JSValueConst this_obj, int flags)
{
    JSObject *p, *p1;
    JSShapeProperty *prs;
    JSProperty *pr;
    JSPropertyDescriptor desc;
    int ret;

    switch(JS_VALUE_GET_TAG(this_obj)) {
    case JS_TAG_NULL:
        JS_ThrowTypeErrorAtom(ctx, "cannot set property '%s' of null", prop);
        goto fail;
    case JS_TAG_UNDEFINED:
        JS_ThrowTypeErrorAtom(ctx, "cannot set property '%s' of undefined", prop);
        goto fail;
    case JS_TAG_OBJECT:
        p = JS_VALUE_GET_OBJ(this_obj);
        p1 = JS_VALUE_GET_OBJ(obj);
        if (p == p1)
            break;
        goto retry2;
    default:
        if (JS_VALUE_GET_TAG(obj) != JS_TAG_OBJECT)
            obj = JS_GetPrototypePrimitive(ctx, obj);
        p = NULL;
        p1 = JS_VALUE_GET_OBJ(obj);
        goto prototype_lookup;
    }

retry:
    prs = find_own_property(&pr, p1, prop);
    if (prs) {
        if (likely((prs->flags & (JS_PROP_TMASK | JS_PROP_WRITABLE |
                                  JS_PROP_LENGTH)) == JS_PROP_WRITABLE)) {
            /* fast case */
            set_value(ctx, &pr->u.value, val);
            return true;
        } else if (prs->flags & JS_PROP_LENGTH) {
            assert(p->class_id == JS_CLASS_ARRAY);
            assert(prop == JS_ATOM_length);
            return set_array_length(ctx, p, val, flags);
        } else if ((prs->flags & JS_PROP_TMASK) == JS_PROP_GETSET) {
            return call_setter(ctx, pr->u.getset.setter, this_obj, val, flags);
        } else if ((prs->flags & JS_PROP_TMASK) == JS_PROP_VARREF) {
            /* JS_PROP_WRITABLE is always true for variable
               references, but they are write protected in module name
               spaces. */
            if (p->class_id == JS_CLASS_MODULE_NS)
                goto read_only_prop;
            set_value(ctx, pr->u.var_ref->pvalue, val);
            return true;
        } else if ((prs->flags & JS_PROP_TMASK) == JS_PROP_AUTOINIT) {
            /* Instantiate property and retry (potentially useless) */
            if (JS_AutoInitProperty(ctx, p, prop, pr, prs))
                goto fail;
            goto retry;
        } else {
            goto read_only_prop;
        }
    }

    for(;;) {
        if (p1->is_exotic) {
            if (p1->fast_array) {
                if (__JS_AtomIsTaggedInt(prop)) {
                    uint32_t idx = __JS_AtomToUInt32(prop);
                    if (idx < p1->u.array.count) {
                        if (unlikely(p == p1))
                            return JS_SetPropertyValue(ctx, this_obj, js_int32(idx), val, flags);
                        else
                            break;
                    } else if (is_typed_array(p1->class_id)) {
                        goto typed_array_oob;
                    }
                } else if (is_typed_array(p1->class_id)) {
                    ret = JS_AtomIsNumericIndex(ctx, prop);
                    if (ret != 0) {
                        if (ret < 0)
                            goto fail;
                    typed_array_oob:
                        // per spec: evaluate value for side effects
                        if (p1->class_id == JS_CLASS_BIG_INT64_ARRAY ||
                            p1->class_id == JS_CLASS_BIG_UINT64_ARRAY) {
                            int64_t v;
                            if (JS_ToBigInt64Free(ctx, &v, val))
                                return -1;
                        } else {
                            val = JS_ToNumberFree(ctx, val);
                            JS_FreeValue(ctx, val);
                            if (JS_IsException(val))
                                return -1;
                        }
                        return true;
                    }
                }
            } else {
                const JSClassExoticMethods *em = ctx->rt->class_array[p1->class_id].exotic;
                if (em) {
                    JSValue obj1;
                    if (em->set_property) {
                        /* set_property can free the prototype */
                        obj1 = js_dup(JS_MKPTR(JS_TAG_OBJECT, p1));
                        ret = em->set_property(ctx, obj1, prop,
                                               val, this_obj, flags);
                        JS_FreeValue(ctx, obj1);
                        JS_FreeValue(ctx, val);
                        return ret;
                    }
                    if (em->get_own_property) {
                        /* get_own_property can free the prototype */
                        obj1 = js_dup(JS_MKPTR(JS_TAG_OBJECT, p1));
                        ret = em->get_own_property(ctx, &desc,
                                                   obj1, prop);
                        JS_FreeValue(ctx, obj1);
                        if (ret < 0)
                            goto fail;
                        if (ret) {
                            if (desc.flags & JS_PROP_GETSET) {
                                JSObject *setter;
                                if (JS_IsUndefined(desc.setter))
                                    setter = NULL;
                                else
                                    setter = JS_VALUE_GET_OBJ(desc.setter);
                                ret = call_setter(ctx, setter, this_obj, val, flags);
                                JS_FreeValue(ctx, desc.getter);
                                JS_FreeValue(ctx, desc.setter);
                                return ret;
                            } else {
                                JS_FreeValue(ctx, desc.value);
                                if (!(desc.flags & JS_PROP_WRITABLE))
                                    goto read_only_prop;
                                if (likely(p == p1)) {
                                    ret = JS_DefineProperty(ctx, this_obj, prop, val,
                                                            JS_UNDEFINED, JS_UNDEFINED,
                                                            JS_PROP_HAS_VALUE);
                                    JS_FreeValue(ctx, val);
                                    return ret;
                                } else {
                                    break;
                                }
                            }
                        }
                    }
                }
            }
        }
        p1 = p1->shape->proto;
    prototype_lookup:
        if (!p1)
            break;

    retry2:
        prs = find_own_property(&pr, p1, prop);
        if (prs) {
            if ((prs->flags & JS_PROP_TMASK) == JS_PROP_GETSET) {
                return call_setter(ctx, pr->u.getset.setter, this_obj, val, flags);
            } else if ((prs->flags & JS_PROP_TMASK) == JS_PROP_AUTOINIT) {
                /* Instantiate property and retry (potentially useless) */
                if (JS_AutoInitProperty(ctx, p1, prop, pr, prs))
                    return -1;
                goto retry2;
            } else if (!(prs->flags & JS_PROP_WRITABLE)) {
                goto read_only_prop;
            } else {
                break;
            }
        }
    }

    if (unlikely(flags & JS_PROP_NO_ADD)) {
        JS_ThrowReferenceErrorNotDefined(ctx, prop);
        goto fail;
    }

    if (unlikely(!p)) {
        ret = JS_ThrowTypeErrorOrFalse(ctx, flags, "not an object");
        goto done;
    }

    if (unlikely(!p->extensible)) {
        ret = JS_ThrowTypeErrorOrFalse(ctx, flags, "object is not extensible");
        goto done;
    }

    if (p == JS_VALUE_GET_OBJ(obj)) {
        if (p->is_exotic) {
            if (p->class_id == JS_CLASS_ARRAY && p->fast_array &&
                __JS_AtomIsTaggedInt(prop)) {
                uint32_t idx = __JS_AtomToUInt32(prop);
                if (idx == p->u.array.count) {
                    /* fast case */
                    return add_fast_array_element(ctx, p, val, flags);
                }
            }
            goto generic_create_prop;
        } else {
            pr = add_property(ctx, p, prop, JS_PROP_C_W_E);
            if (!pr)
                goto fail;
            pr->u.value = val;
            return true;
        }
    }

    // TODO(bnoordhuis) return JSProperty slot and update in place
    // when plain property (not is_exotic/setter/etc.) to avoid
    // calling find_own_property() thrice?
    ret = JS_GetOwnPropertyInternal(ctx, &desc, p, prop);
    if (ret < 0)
        goto fail;

    if (ret) {
        JS_FreeValue(ctx, desc.value);
        if (desc.flags & JS_PROP_GETSET) {
            JS_FreeValue(ctx, desc.getter);
            JS_FreeValue(ctx, desc.setter);
            ret = JS_ThrowTypeErrorOrFalse(ctx, flags, "setter is forbidden");
            goto done;
        } else if (!(desc.flags & JS_PROP_WRITABLE) ||
                   p->class_id == JS_CLASS_MODULE_NS) {
        read_only_prop:
            ret = JS_ThrowTypeErrorReadOnly(ctx, flags, prop);
            goto done;
        }
        ret = JS_DefineProperty(ctx, this_obj, prop, val,
                                JS_UNDEFINED, JS_UNDEFINED,
                                JS_PROP_HAS_VALUE);
    } else {
    generic_create_prop:
        ret = JS_CreateProperty(ctx, p, prop, val, JS_UNDEFINED, JS_UNDEFINED,
                                flags |
                                JS_PROP_HAS_VALUE |
                                JS_PROP_HAS_ENUMERABLE |
                                JS_PROP_HAS_WRITABLE |
                                JS_PROP_HAS_CONFIGURABLE |
                                JS_PROP_C_W_E);
    }

done:
    JS_FreeValue(ctx, val);
    return ret;
fail:
    JS_FreeValue(ctx, val);
    return -1;
}

static int JS_SetPropertyInternal(JSContext *ctx, JSValueConst obj, JSAtom prop,
                                  JSValue val, int flags)
{
    return JS_SetPropertyInternal2(ctx, obj, prop, val, obj, flags);
}

int JS_SetProperty(JSContext *ctx, JSValueConst this_obj, JSAtom prop, JSValue val)
{
    return JS_SetPropertyInternal(ctx, this_obj, prop, val, JS_PROP_THROW);
}

/* flags can be JS_PROP_THROW or JS_PROP_THROW_STRICT */
static int JS_SetPropertyValue(JSContext *ctx, JSValueConst this_obj,
                               JSValue prop, JSValue val, int flags)
{
    if (likely(JS_VALUE_GET_TAG(this_obj) == JS_TAG_OBJECT &&
               JS_VALUE_GET_TAG(prop) == JS_TAG_INT)) {
        JSObject *p;
        uint32_t idx;
        double d;
        int32_t v;

        /* fast path for array access */
        p = JS_VALUE_GET_OBJ(this_obj);
        idx = JS_VALUE_GET_INT(prop);
        switch(p->class_id) {
        case JS_CLASS_ARRAY:
            if (unlikely(idx >= (uint32_t)p->u.array.count)) {
                JSObject *p1;
                JSShape *sh1;

                /* fast path to add an element to the array */
                if (idx != (uint32_t)p->u.array.count ||
                    !p->fast_array || !p->extensible)
                    goto slow_path;
                /* check if prototype chain has a numeric property */
                p1 = p->shape->proto;
                while (p1 != NULL) {
                    sh1 = p1->shape;
                    if (p1->class_id == JS_CLASS_ARRAY) {
                        if (unlikely(!p1->fast_array))
                            goto slow_path;
                    } else if (p1->class_id == JS_CLASS_OBJECT) {
                        if (unlikely(sh1->has_small_array_index))
                            goto slow_path;
                    } else {
                        goto slow_path;
                    }
                    p1 = sh1->proto;
                }
                /* add element */
                return add_fast_array_element(ctx, p, val, flags);
            }
            set_value(ctx, &p->u.array.u.values[idx], val);
            break;
        case JS_CLASS_ARGUMENTS:
            if (unlikely(idx >= (uint32_t)p->u.array.count))
                goto slow_path;
            set_value(ctx, &p->u.array.u.values[idx], val);
            break;
        case JS_CLASS_UINT8C_ARRAY:
            if (JS_ToUint8ClampFree(ctx, &v, val))
                goto ta_cvt_fail;
            /* Note: the conversion can detach the typed array, so the
               array bound check must be done after */
            if (unlikely(idx >= (uint32_t)p->u.array.count))
                goto ta_out_of_bound;
            p->u.array.u.uint8_ptr[idx] = v;
            break;
        case JS_CLASS_INT8_ARRAY:
        case JS_CLASS_UINT8_ARRAY:
            if (JS_ToInt32Free(ctx, &v, val))
                goto ta_cvt_fail;
            if (unlikely(idx >= (uint32_t)p->u.array.count))
                goto ta_out_of_bound;
            p->u.array.u.uint8_ptr[idx] = v;
            break;
        case JS_CLASS_INT16_ARRAY:
        case JS_CLASS_UINT16_ARRAY:
            if (JS_ToInt32Free(ctx, &v, val))
                goto ta_cvt_fail;
            if (unlikely(idx >= (uint32_t)p->u.array.count))
                goto ta_out_of_bound;
            p->u.array.u.uint16_ptr[idx] = v;
            break;
        case JS_CLASS_INT32_ARRAY:
        case JS_CLASS_UINT32_ARRAY:
            if (JS_ToInt32Free(ctx, &v, val))
                goto ta_cvt_fail;
            if (unlikely(idx >= (uint32_t)p->u.array.count))
                goto ta_out_of_bound;
            p->u.array.u.uint32_ptr[idx] = v;
            break;
        case JS_CLASS_BIG_INT64_ARRAY:
        case JS_CLASS_BIG_UINT64_ARRAY:
            /* XXX: need specific conversion function */
            {
                int64_t v;
                if (JS_ToBigInt64Free(ctx, &v, val))
                    goto ta_cvt_fail;
                if (unlikely(idx >= (uint32_t)p->u.array.count))
                    goto ta_out_of_bound;
                p->u.array.u.uint64_ptr[idx] = v;
            }
            break;
        case JS_CLASS_FLOAT16_ARRAY:
            if (JS_ToFloat64Free(ctx, &d, val))
                goto ta_cvt_fail;
            if (unlikely(idx >= (uint32_t)p->u.array.count))
                goto ta_out_of_bound;
            p->u.array.u.fp16_ptr[idx] = tofp16(d);
            break;
        case JS_CLASS_FLOAT32_ARRAY:
            if (JS_ToFloat64Free(ctx, &d, val))
                goto ta_cvt_fail;
            if (unlikely(idx >= (uint32_t)p->u.array.count))
                goto ta_out_of_bound;
            p->u.array.u.float_ptr[idx] = d;
            break;
        case JS_CLASS_FLOAT64_ARRAY:
            if (JS_ToFloat64Free(ctx, &d, val)) {
            ta_cvt_fail:
                if (flags & JS_PROP_REFLECT_DEFINE_PROPERTY) {
                    JS_FreeValue(ctx, JS_GetException(ctx));
                    return false;
                }
                return -1;
            }
            if (unlikely(idx >= (uint32_t)p->u.array.count)) {
            ta_out_of_bound:
                if (typed_array_is_oob(p))
                    if (flags & JS_PROP_DEFINE_PROPERTY)
                        return JS_ThrowTypeErrorOrFalse(ctx, flags, "out-of-bound numeric index");
                return true; // per spec: no OOB exception
            }
            p->u.array.u.double_ptr[idx] = d;
            break;
        default:
            goto slow_path;
        }
        return true;
    } else {
        JSAtom atom;
        int ret;
    slow_path:
        atom = JS_ValueToAtom(ctx, prop);
        JS_FreeValue(ctx, prop);
        if (unlikely(atom == JS_ATOM_NULL)) {
            JS_FreeValue(ctx, val);
            return -1;
        }
        ret = JS_SetPropertyInternal(ctx, this_obj, atom, val, flags);
        JS_FreeAtom(ctx, atom);
        return ret;
    }
}

int JS_SetPropertyUint32(JSContext *ctx, JSValueConst this_obj,
                         uint32_t idx, JSValue val)
{
    return JS_SetPropertyValue(ctx, this_obj, js_uint32(idx), val,
                               JS_PROP_THROW);
}

int JS_SetPropertyInt64(JSContext *ctx, JSValueConst this_obj,
                        int64_t idx, JSValue val)
{
    JSAtom prop;
    int res;

    if ((uint64_t)idx <= INT32_MAX) {
        /* fast path for fast arrays */
        return JS_SetPropertyValue(ctx, this_obj, js_int32(idx), val,
                                   JS_PROP_THROW);
    }
    prop = JS_NewAtomInt64(ctx, idx);
    if (prop == JS_ATOM_NULL) {
        JS_FreeValue(ctx, val);
        return -1;
    }
    res = JS_SetProperty(ctx, this_obj, prop, val);
    JS_FreeAtom(ctx, prop);
    return res;
}

/* `prop` may be pure ASCII or UTF-8 encoded */
int JS_SetPropertyStr(JSContext *ctx, JSValueConst this_obj,
                      const char *prop, JSValue val)
{
    JSAtom atom;
    int ret;
    atom = JS_NewAtom(ctx, prop);
    if (atom == JS_ATOM_NULL) {
        JS_FreeValue(ctx, val);
        return -1;
    }
    ret = JS_SetPropertyInternal(ctx, this_obj, atom, val, JS_PROP_THROW);
    JS_FreeAtom(ctx, atom);
    return ret;
}

/* compute the property flags. For each flag: (JS_PROP_HAS_x forces
   it, otherwise def_flags is used)
   Note: makes assumption about the bit pattern of the flags
*/
static int get_prop_flags(int flags, int def_flags)
{
    int mask;
    mask = (flags >> JS_PROP_HAS_SHIFT) & JS_PROP_C_W_E;
    return (flags & mask) | (def_flags & ~mask);
}

static int JS_CreateProperty(JSContext *ctx, JSObject *p,
                             JSAtom prop, JSValueConst val,
                             JSValueConst getter, JSValueConst setter,
                             int flags)
{
    JSProperty *pr;
    int ret, prop_flags;

    /* add a new property or modify an existing exotic one */
    if (p->is_exotic) {
        if (p->class_id == JS_CLASS_ARRAY) {
            uint32_t idx, len;

            if (p->fast_array) {
                if (__JS_AtomIsTaggedInt(prop)) {
                    idx = __JS_AtomToUInt32(prop);
                    if (idx == p->u.array.count) {
                        if (!p->extensible)
                            goto not_extensible;
                        if (flags & (JS_PROP_HAS_GET | JS_PROP_HAS_SET))
                            goto convert_to_array;
                        prop_flags = get_prop_flags(flags, 0);
                        if (prop_flags != JS_PROP_C_W_E)
                            goto convert_to_array;
                        return add_fast_array_element(ctx, p,
                                                      js_dup(val), flags);
                    } else {
                        goto convert_to_array;
                    }
                } else if (JS_AtomIsArrayIndex(ctx, &idx, prop)) {
                    /* convert the fast array to normal array */
                convert_to_array:
                    if (convert_fast_array_to_array(ctx, p))
                        return -1;
                    goto generic_array;
                }
            } else if (JS_AtomIsArrayIndex(ctx, &idx, prop)) {
                JSProperty *plen;
                JSShapeProperty *pslen;
            generic_array:
                /* update the length field */
                plen = &p->prop[0];
                JS_ToUint32(ctx, &len, plen->u.value);
                if ((idx + 1) > len) {
                    pslen = get_shape_prop(p->shape);
                    if (unlikely(!(pslen->flags & JS_PROP_WRITABLE)))
                        return JS_ThrowTypeErrorReadOnly(ctx, flags, JS_ATOM_length);
                    /* XXX: should update the length after defining
                       the property */
                    len = idx + 1;
                    set_value(ctx, &plen->u.value, js_uint32(len));
                }
            }
        } else if (is_typed_array(p->class_id)) {
            ret = JS_AtomIsNumericIndex(ctx, prop);
            if (ret != 0) {
                if (ret < 0)
                    return -1;
                return JS_ThrowTypeErrorOrFalse(ctx, flags, "cannot create numeric index in typed array");
            }
        } else if (!(flags & JS_PROP_NO_EXOTIC)) {
            const JSClassExoticMethods *em = ctx->rt->class_array[p->class_id].exotic;
            if (em) {
                if (em->define_own_property) {
                    return em->define_own_property(ctx, JS_MKPTR(JS_TAG_OBJECT, p),
                                                   prop, val, getter, setter, flags);
                }
                ret = JS_IsExtensible(ctx, JS_MKPTR(JS_TAG_OBJECT, p));
                if (ret < 0)
                    return -1;
                if (!ret)
                    goto not_extensible;
            }
        }
    }

    if (!p->extensible) {
    not_extensible:
        return JS_ThrowTypeErrorOrFalse(ctx, flags, "object is not extensible");
    }

    if (flags & (JS_PROP_HAS_GET | JS_PROP_HAS_SET)) {
        prop_flags = (flags & (JS_PROP_CONFIGURABLE | JS_PROP_ENUMERABLE)) |
            JS_PROP_GETSET;
    } else {
        prop_flags = flags & JS_PROP_C_W_E;
    }
    pr = add_property(ctx, p, prop, prop_flags);
    if (unlikely(!pr))
        return -1;
    if (flags & (JS_PROP_HAS_GET | JS_PROP_HAS_SET)) {
        pr->u.getset.getter = NULL;
        if ((flags & JS_PROP_HAS_GET) && JS_IsFunction(ctx, getter)) {
            pr->u.getset.getter =
                JS_VALUE_GET_OBJ(js_dup(getter));
        }
        pr->u.getset.setter = NULL;
        if ((flags & JS_PROP_HAS_SET) && JS_IsFunction(ctx, setter)) {
            pr->u.getset.setter =
                JS_VALUE_GET_OBJ(js_dup(setter));
        }
    } else {
        if (flags & JS_PROP_HAS_VALUE) {
            pr->u.value = js_dup(val);
        } else {
            pr->u.value = JS_UNDEFINED;
        }
    }
    return true;
}

/* return false if not OK */
static bool check_define_prop_flags(int prop_flags, int flags)
{
    bool has_accessor, is_getset;

    if (!(prop_flags & JS_PROP_CONFIGURABLE)) {
        if ((flags & (JS_PROP_HAS_CONFIGURABLE | JS_PROP_CONFIGURABLE)) ==
            (JS_PROP_HAS_CONFIGURABLE | JS_PROP_CONFIGURABLE)) {
            return false;
        }
        if ((flags & JS_PROP_HAS_ENUMERABLE) &&
            (flags & JS_PROP_ENUMERABLE) != (prop_flags & JS_PROP_ENUMERABLE))
            return false;
    }
    if (flags & (JS_PROP_HAS_VALUE | JS_PROP_HAS_WRITABLE |
                 JS_PROP_HAS_GET | JS_PROP_HAS_SET)) {
        if (!(prop_flags & JS_PROP_CONFIGURABLE)) {
            has_accessor = ((flags & (JS_PROP_HAS_GET | JS_PROP_HAS_SET)) != 0);
            is_getset = ((prop_flags & JS_PROP_TMASK) == JS_PROP_GETSET);
            if (has_accessor != is_getset)
                return false;
            if (!has_accessor && !is_getset && !(prop_flags & JS_PROP_WRITABLE)) {
                /* not writable: cannot set the writable bit */
                if ((flags & (JS_PROP_HAS_WRITABLE | JS_PROP_WRITABLE)) ==
                    (JS_PROP_HAS_WRITABLE | JS_PROP_WRITABLE))
                    return false;
            }
        }
    }
    return true;
}

/* ensure that the shape can be safely modified */
static int js_shape_prepare_update(JSContext *ctx, JSObject *p,
                                   JSShapeProperty **pprs)
{
    JSShape *sh;
    uint32_t idx = 0;    /* prevent warning */

    sh = p->shape;
    if (sh->is_hashed) {
        if (sh->header.ref_count != 1) {
            if (pprs)
                idx = *pprs - get_shape_prop(sh);
            /* clone the shape (the resulting one is no longer hashed) */
            sh = js_clone_shape(ctx, sh);
            if (!sh)
                return -1;
            js_free_shape(ctx->rt, p->shape);
            p->shape = sh;
            if (pprs)
                *pprs = get_shape_prop(sh) + idx;
        } else {
            js_shape_hash_unlink(ctx->rt, sh);
            sh->is_hashed = false;
        }
    }
    return 0;
}

static int js_update_property_flags(JSContext *ctx, JSObject *p,
                                    JSShapeProperty **pprs, int flags)
{
    if (flags != (*pprs)->flags) {
        if (js_shape_prepare_update(ctx, p, pprs))
            return -1;
        (*pprs)->flags = flags;
    }
    return 0;
}

/* allowed flags:
   JS_PROP_CONFIGURABLE, JS_PROP_WRITABLE, JS_PROP_ENUMERABLE
   JS_PROP_HAS_GET, JS_PROP_HAS_SET, JS_PROP_HAS_VALUE,
   JS_PROP_HAS_CONFIGURABLE, JS_PROP_HAS_WRITABLE, JS_PROP_HAS_ENUMERABLE,
   JS_PROP_THROW, JS_PROP_NO_EXOTIC.
   If JS_PROP_THROW is set, return an exception instead of false.
   if JS_PROP_NO_EXOTIC is set, do not call the exotic
   define_own_property callback.
   return -1 (exception), false or true.
*/
int JS_DefineProperty(JSContext *ctx, JSValueConst this_obj,
                      JSAtom prop, JSValueConst val,
                      JSValueConst getter, JSValueConst setter, int flags)
{
    JSObject *p;
    JSShapeProperty *prs;
    JSProperty *pr;
    int mask, res;

    if (JS_VALUE_GET_TAG(this_obj) != JS_TAG_OBJECT) {
        JS_ThrowTypeErrorNotAnObject(ctx);
        return -1;
    }
    p = JS_VALUE_GET_OBJ(this_obj);

 redo_prop_update:
    prs = find_own_property(&pr, p, prop);
    if (prs) {
        /* the range of the Array length property is always tested before */
        if ((prs->flags & JS_PROP_LENGTH) && (flags & JS_PROP_HAS_VALUE)) {
            uint32_t array_length;
            if (JS_ToArrayLengthFree(ctx, &array_length,
                                     js_dup(val), false)) {
                return -1;
            }
            /* this code relies on the fact that Uint32 are never allocated */
            val = js_uint32(array_length);
            /* prs may have been modified */
            prs = find_own_property(&pr, p, prop);
            assert(prs != NULL);
        }
        /* property already exists */
        if (!check_define_prop_flags(prs->flags, flags)) {
        not_configurable:
            return JS_ThrowTypeErrorOrFalse(ctx, flags, "property is not configurable");
        }

        if ((prs->flags & JS_PROP_TMASK) == JS_PROP_AUTOINIT) {
            /* Instantiate property and retry */
            if (JS_AutoInitProperty(ctx, p, prop, pr, prs))
                return -1;
            goto redo_prop_update;
        }

        if (flags & (JS_PROP_HAS_VALUE | JS_PROP_HAS_WRITABLE |
                     JS_PROP_HAS_GET | JS_PROP_HAS_SET)) {
            if (flags & (JS_PROP_HAS_GET | JS_PROP_HAS_SET)) {
                JSObject *new_getter, *new_setter;

                if (JS_IsFunction(ctx, getter)) {
                    new_getter = JS_VALUE_GET_OBJ(getter);
                } else {
                    new_getter = NULL;
                }
                if (JS_IsFunction(ctx, setter)) {
                    new_setter = JS_VALUE_GET_OBJ(setter);
                } else {
                    new_setter = NULL;
                }

                if ((prs->flags & JS_PROP_TMASK) != JS_PROP_GETSET) {
                    if (js_shape_prepare_update(ctx, p, &prs))
                        return -1;
                    /* convert to getset */
                    if ((prs->flags & JS_PROP_TMASK) == JS_PROP_VARREF) {
                        free_var_ref(ctx->rt, pr->u.var_ref);
                    } else {
                        JS_FreeValue(ctx, pr->u.value);
                    }
                    prs->flags = (prs->flags &
                                  (JS_PROP_CONFIGURABLE | JS_PROP_ENUMERABLE)) |
                        JS_PROP_GETSET;
                    pr->u.getset.getter = NULL;
                    pr->u.getset.setter = NULL;
                } else {
                    if (!(prs->flags & JS_PROP_CONFIGURABLE)) {
                        if ((flags & JS_PROP_HAS_GET) &&
                            new_getter != pr->u.getset.getter) {
                            goto not_configurable;
                        }
                        if ((flags & JS_PROP_HAS_SET) &&
                            new_setter != pr->u.getset.setter) {
                            goto not_configurable;
                        }
                    }
                }
                if (flags & JS_PROP_HAS_GET) {
                    if (pr->u.getset.getter)
                        JS_FreeValue(ctx, JS_MKPTR(JS_TAG_OBJECT, pr->u.getset.getter));
                    if (new_getter)
                        js_dup(getter);
                    pr->u.getset.getter = new_getter;
                }
                if (flags & JS_PROP_HAS_SET) {
                    if (pr->u.getset.setter)
                        JS_FreeValue(ctx, JS_MKPTR(JS_TAG_OBJECT, pr->u.getset.setter));
                    if (new_setter)
                        js_dup(setter);
                    pr->u.getset.setter = new_setter;
                }
            } else {
                if ((prs->flags & JS_PROP_TMASK) == JS_PROP_GETSET) {
                    /* convert to data descriptor */
                    if (js_shape_prepare_update(ctx, p, &prs))
                        return -1;
                    if (pr->u.getset.getter)
                        JS_FreeValue(ctx, JS_MKPTR(JS_TAG_OBJECT, pr->u.getset.getter));
                    if (pr->u.getset.setter)
                        JS_FreeValue(ctx, JS_MKPTR(JS_TAG_OBJECT, pr->u.getset.setter));
                    prs->flags &= ~(JS_PROP_TMASK | JS_PROP_WRITABLE);
                    pr->u.value = JS_UNDEFINED;
                } else if ((prs->flags & JS_PROP_TMASK) == JS_PROP_VARREF) {
                    /* Note: JS_PROP_VARREF is always writable */
                } else {
                    if ((prs->flags & (JS_PROP_CONFIGURABLE | JS_PROP_WRITABLE)) == 0 &&
                        (flags & JS_PROP_HAS_VALUE)) {
                        if (!js_same_value(ctx, val, pr->u.value)) {
                            goto not_configurable;
                        } else {
                            return true;
                        }
                    }
                }
                if ((prs->flags & JS_PROP_TMASK) == JS_PROP_VARREF) {
                    if (flags & JS_PROP_HAS_VALUE) {
                        if (p->class_id == JS_CLASS_MODULE_NS) {
                            /* JS_PROP_WRITABLE is always true for variable
                               references, but they are write protected in module name
                               spaces. */
                            if (!js_same_value(ctx, val, *pr->u.var_ref->pvalue))
                                goto not_configurable;
                        }
                        /* update the reference */
                        set_value(ctx, pr->u.var_ref->pvalue, js_dup(val));
                    }
                    /* if writable is set to false, no longer a
                       reference (for mapped arguments) */
                    if ((flags & (JS_PROP_HAS_WRITABLE | JS_PROP_WRITABLE)) == JS_PROP_HAS_WRITABLE) {
                        JSValue val1;
                        if (js_shape_prepare_update(ctx, p, &prs))
                            return -1;
                        val1 = js_dup(*pr->u.var_ref->pvalue);
                        free_var_ref(ctx->rt, pr->u.var_ref);
                        pr->u.value = val1;
                        prs->flags &= ~(JS_PROP_TMASK | JS_PROP_WRITABLE);
                    }
                } else if (prs->flags & JS_PROP_LENGTH) {
                    if (flags & JS_PROP_HAS_VALUE) {
                        /* Note: no JS code is executable because
                           'val' is guaranted to be a Uint32 */
                        res = set_array_length(ctx, p, js_dup(val), flags);
                    } else {
                        res = true;
                    }
                    /* still need to reset the writable flag if
                       needed.  The JS_PROP_LENGTH is kept because the
                       Uint32 test is still done if the length
                       property is read-only. */
                    if ((flags & (JS_PROP_HAS_WRITABLE | JS_PROP_WRITABLE)) ==
                        JS_PROP_HAS_WRITABLE) {
                        prs = get_shape_prop(p->shape);
                        if (js_update_property_flags(ctx, p, &prs,
                                                     prs->flags & ~JS_PROP_WRITABLE))
                            return -1;
                    }
                    return res;
                } else {
                    if (flags & JS_PROP_HAS_VALUE) {
                        JS_FreeValue(ctx, pr->u.value);
                        pr->u.value = js_dup(val);
                    }
                    if (flags & JS_PROP_HAS_WRITABLE) {
                        if (js_update_property_flags(ctx, p, &prs,
                                                     (prs->flags & ~JS_PROP_WRITABLE) |
                                                     (flags & JS_PROP_WRITABLE)))
                            return -1;
                    }
                }
            }
        }
        mask = 0;
        if (flags & JS_PROP_HAS_CONFIGURABLE)
            mask |= JS_PROP_CONFIGURABLE;
        if (flags & JS_PROP_HAS_ENUMERABLE)
            mask |= JS_PROP_ENUMERABLE;
        if (js_update_property_flags(ctx, p, &prs,
                                     (prs->flags & ~mask) | (flags & mask)))
            return -1;
        return true;
    }

    /* handle modification of fast array elements */
    if (p->fast_array) {
        uint32_t idx;
        uint32_t prop_flags;
        if (p->class_id == JS_CLASS_ARRAY) {
            if (__JS_AtomIsTaggedInt(prop)) {
                idx = __JS_AtomToUInt32(prop);
                if (idx < p->u.array.count) {
                    prop_flags = get_prop_flags(flags, JS_PROP_C_W_E);
                    if (prop_flags != JS_PROP_C_W_E)
                        goto convert_to_slow_array;
                    if (flags & (JS_PROP_HAS_GET | JS_PROP_HAS_SET)) {
                    convert_to_slow_array:
                        if (convert_fast_array_to_array(ctx, p))
                            return -1;
                        else
                            goto redo_prop_update;
                    }
                    if (flags & JS_PROP_HAS_VALUE) {
                        set_value(ctx, &p->u.array.u.values[idx], js_dup(val));
                    }
                    return true;
                }
            }
        } else if (is_typed_array(p->class_id)) {
            JSValue num;
            int ret;

            if (!__JS_AtomIsTaggedInt(prop)) {
                /* slow path with to handle all numeric indexes */
                num = JS_AtomIsNumericIndex1(ctx, prop);
                if (JS_IsUndefined(num))
                    goto typed_array_done;
                if (JS_IsException(num))
                    return -1;
                ret = JS_NumberIsInteger(ctx, num);
                if (ret < 0) {
                    JS_FreeValue(ctx, num);
                    return -1;
                }
                if (!ret) {
                    JS_FreeValue(ctx, num);
                    return JS_ThrowTypeErrorOrFalse(ctx, flags, "non integer index in typed array");
                }
                ret = JS_NumberIsNegativeOrMinusZero(ctx, num);
                JS_FreeValue(ctx, num);
                if (ret) {
                    return JS_ThrowTypeErrorOrFalse(ctx, flags, "negative index in typed array");
                }
                if (!__JS_AtomIsTaggedInt(prop))
                    goto typed_array_oob;
            }
            idx = __JS_AtomToUInt32(prop);
            /* if the typed array is detached, p->u.array.count = 0 */
            if (idx >= p->u.array.count) {
            typed_array_oob:
                return JS_ThrowTypeErrorOrFalse(ctx, flags, "out-of-bound index in typed array");
            }
            prop_flags = get_prop_flags(flags, JS_PROP_ENUMERABLE | JS_PROP_WRITABLE | JS_PROP_CONFIGURABLE);
            if (flags & (JS_PROP_HAS_GET | JS_PROP_HAS_SET) ||
                prop_flags != (JS_PROP_ENUMERABLE | JS_PROP_WRITABLE | JS_PROP_CONFIGURABLE)) {
                return JS_ThrowTypeErrorOrFalse(ctx, flags, "invalid descriptor flags");
            }
            if (flags & JS_PROP_HAS_VALUE) {
                return JS_SetPropertyValue(ctx, this_obj, js_int32(idx), js_dup(val), flags);
            }
            return true;
        typed_array_done: ;
        }
    }

    return JS_CreateProperty(ctx, p, prop, val, getter, setter, flags);
}

static int JS_DefineAutoInitProperty(JSContext *ctx, JSValueConst this_obj,
                                     JSAtom prop, JSAutoInitIDEnum id,
                                     void *opaque, int flags)
{
    JSObject *p;
    JSProperty *pr;

    if (JS_VALUE_GET_TAG(this_obj) != JS_TAG_OBJECT)
        return false;

    p = JS_VALUE_GET_OBJ(this_obj);

    if (find_own_property(&pr, p, prop)) {
        /* property already exists */
        abort();
        return false;
    }

    /* Specialized CreateProperty */
    pr = add_property(ctx, p, prop, (flags & JS_PROP_C_W_E) | JS_PROP_AUTOINIT);
    if (unlikely(!pr))
        return -1;
    pr->u.init.realm_and_id = (uintptr_t)JS_DupContext(ctx);
    assert((pr->u.init.realm_and_id & 3) == 0);
    assert(id <= 3);
    pr->u.init.realm_and_id |= id;
    pr->u.init.opaque = opaque;
    return true;
}

/* shortcut to add or redefine a new property value */
int JS_DefinePropertyValue(JSContext *ctx, JSValueConst this_obj,
                           JSAtom prop, JSValue val, int flags)
{
    int ret;
    ret = JS_DefineProperty(ctx, this_obj, prop, val, JS_UNDEFINED, JS_UNDEFINED,
                            flags | JS_PROP_HAS_VALUE | JS_PROP_HAS_CONFIGURABLE | JS_PROP_HAS_WRITABLE | JS_PROP_HAS_ENUMERABLE);
    JS_FreeValue(ctx, val);
    return ret;
}

int JS_DefinePropertyValueValue(JSContext *ctx, JSValueConst this_obj,
                                JSValue prop, JSValue val, int flags)
{
    JSAtom atom;
    int ret;
    atom = JS_ValueToAtom(ctx, prop);
    JS_FreeValue(ctx, prop);
    if (unlikely(atom == JS_ATOM_NULL)) {
        JS_FreeValue(ctx, val);
        return -1;
    }
    ret = JS_DefinePropertyValue(ctx, this_obj, atom, val, flags);
    JS_FreeAtom(ctx, atom);
    return ret;
}

int JS_DefinePropertyValueUint32(JSContext *ctx, JSValueConst this_obj,
                                 uint32_t idx, JSValue val, int flags)
{
    return JS_DefinePropertyValueValue(ctx, this_obj, js_uint32(idx),
                                       val, flags);
}

int JS_DefinePropertyValueInt64(JSContext *ctx, JSValueConst this_obj,
                                int64_t idx, JSValue val, int flags)
{
    return JS_DefinePropertyValueValue(ctx, this_obj, js_int64(idx),
                                       val, flags);
}

/* `prop` may be pure ASCII or UTF-8 encoded */
int JS_DefinePropertyValueStr(JSContext *ctx, JSValueConst this_obj,
                              const char *prop, JSValue val, int flags)
{
    JSAtom atom;
    int ret;
    atom = JS_NewAtom(ctx, prop);
    if (atom == JS_ATOM_NULL) {
        JS_FreeValue(ctx, val);
        return -1;
    }
    ret = JS_DefinePropertyValue(ctx, this_obj, atom, val, flags);
    JS_FreeAtom(ctx, atom);
    return ret;
}

/* shortcut to add getter & setter */
int JS_DefinePropertyGetSet(JSContext *ctx, JSValueConst this_obj,
                            JSAtom prop, JSValue getter, JSValue setter,
                            int flags)
{
    int ret;
    ret = JS_DefineProperty(ctx, this_obj, prop, JS_UNDEFINED, getter, setter,
                            flags | JS_PROP_HAS_GET | JS_PROP_HAS_SET |
                            JS_PROP_HAS_CONFIGURABLE | JS_PROP_HAS_ENUMERABLE);
    JS_FreeValue(ctx, getter);
    JS_FreeValue(ctx, setter);
    return ret;
}

static int JS_CreateDataPropertyUint32(JSContext *ctx, JSValueConst this_obj,
                                       int64_t idx, JSValue val, int flags)
{
    return JS_DefinePropertyValueValue(ctx, this_obj, js_int64(idx),
                                       val, flags | JS_PROP_CONFIGURABLE |
                                       JS_PROP_ENUMERABLE | JS_PROP_WRITABLE);
}


/* return true if 'obj' has a non empty 'name' string */
static bool js_object_has_name(JSContext *ctx, JSValue obj)
{
    JSProperty *pr;
    JSShapeProperty *prs;
    JSValue val;
    JSString *p;

    prs = find_own_property(&pr, JS_VALUE_GET_OBJ(obj), JS_ATOM_name);
    if (!prs)
        return false;
    if ((prs->flags & JS_PROP_TMASK) != JS_PROP_NORMAL)
        return true;
    val = pr->u.value;
    if (JS_VALUE_GET_TAG(val) != JS_TAG_STRING)
        return true;
    p = JS_VALUE_GET_STRING(val);
    return (p->len != 0);
}

static int JS_DefineObjectName(JSContext *ctx, JSValue obj,
                               JSAtom name, int flags)
{
    if (name != JS_ATOM_NULL
    &&  JS_IsObject(obj)
    &&  !js_object_has_name(ctx, obj)
    &&  JS_DefinePropertyValue(ctx, obj, JS_ATOM_name, JS_AtomToString(ctx, name), flags) < 0) {
        return -1;
    }
    return 0;
}

static int JS_DefineObjectNameComputed(JSContext *ctx, JSValue obj,
                                       JSValue str, int flags)
{
    if (JS_IsObject(obj) &&
        !js_object_has_name(ctx, obj)) {
        JSAtom prop;
        JSValue name_str;
        prop = JS_ValueToAtom(ctx, str);
        if (prop == JS_ATOM_NULL)
            return -1;
        name_str = js_get_function_name(ctx, prop);
        JS_FreeAtom(ctx, prop);
        if (JS_IsException(name_str))
            return -1;
        if (JS_DefinePropertyValue(ctx, obj, JS_ATOM_name, name_str, flags) < 0)
            return -1;
    }
    return 0;
}

#define DEFINE_GLOBAL_LEX_VAR (1 << 7)
#define DEFINE_GLOBAL_FUNC_VAR (1 << 6)

static JSValue JS_ThrowSyntaxErrorVarRedeclaration(JSContext *ctx, JSAtom prop)
{
    return JS_ThrowSyntaxErrorAtom(ctx, "redeclaration of '%s'", prop);
}

/* flags is 0, DEFINE_GLOBAL_LEX_VAR or DEFINE_GLOBAL_FUNC_VAR */
/* XXX: could support exotic global object. */
static int JS_CheckDefineGlobalVar(JSContext *ctx, JSAtom prop, int flags)
{
    JSObject *p;
    JSShapeProperty *prs;

    p = JS_VALUE_GET_OBJ(ctx->global_obj);
    prs = find_own_property1(p, prop);
    /* XXX: should handle JS_PROP_AUTOINIT */
    if (flags & DEFINE_GLOBAL_LEX_VAR) {
        if (prs && !(prs->flags & JS_PROP_CONFIGURABLE))
            goto fail_redeclaration;
    } else {
        if (!prs && !p->extensible)
            goto define_error;
        if (flags & DEFINE_GLOBAL_FUNC_VAR) {
            if (prs) {
                if (!(prs->flags & JS_PROP_CONFIGURABLE) &&
                    ((prs->flags & JS_PROP_TMASK) == JS_PROP_GETSET ||
                     ((prs->flags & (JS_PROP_WRITABLE | JS_PROP_ENUMERABLE)) !=
                      (JS_PROP_WRITABLE | JS_PROP_ENUMERABLE)))) {
                define_error:
                    JS_ThrowTypeErrorAtom(ctx, "cannot define variable '%s'",
                                          prop);
                    return -1;
                }
            }
        }
    }
    /* check if there already is a lexical declaration */
    p = JS_VALUE_GET_OBJ(ctx->global_var_obj);
    prs = find_own_property1(p, prop);
    if (prs) {
    fail_redeclaration:
        JS_ThrowSyntaxErrorVarRedeclaration(ctx, prop);
        return -1;
    }
    return 0;
}

/* def_flags is (0, DEFINE_GLOBAL_LEX_VAR) |
   JS_PROP_CONFIGURABLE | JS_PROP_WRITABLE */
/* XXX: could support exotic global object. */
static int JS_DefineGlobalVar(JSContext *ctx, JSAtom prop, int def_flags)
{
    JSObject *p;
    JSShapeProperty *prs;
    JSProperty *pr;
    JSValue val;
    int flags;

    if (def_flags & DEFINE_GLOBAL_LEX_VAR) {
        p = JS_VALUE_GET_OBJ(ctx->global_var_obj);
        flags = JS_PROP_ENUMERABLE | (def_flags & JS_PROP_WRITABLE) |
            JS_PROP_CONFIGURABLE;
        val = JS_UNINITIALIZED;
    } else {
        p = JS_VALUE_GET_OBJ(ctx->global_obj);
        flags = JS_PROP_ENUMERABLE | JS_PROP_WRITABLE |
            (def_flags & JS_PROP_CONFIGURABLE);
        val = JS_UNDEFINED;
    }
    prs = find_own_property1(p, prop);
    if (prs)
        return 0;
    if (!p->extensible)
        return 0;
    pr = add_property(ctx, p, prop, flags);
    if (unlikely(!pr))
        return -1;
    pr->u.value = val;
    return 0;
}

/* 'def_flags' is 0 or JS_PROP_CONFIGURABLE. */
/* XXX: could support exotic global object. */
static int JS_DefineGlobalFunction(JSContext *ctx, JSAtom prop,
                                   JSValue func, int def_flags)
{

    JSObject *p;
    JSShapeProperty *prs;
    int flags;

    p = JS_VALUE_GET_OBJ(ctx->global_obj);
    prs = find_own_property1(p, prop);
    flags = JS_PROP_HAS_VALUE | JS_PROP_THROW;
    if (!prs || (prs->flags & JS_PROP_CONFIGURABLE)) {
        flags |= JS_PROP_ENUMERABLE | JS_PROP_WRITABLE | def_flags |
            JS_PROP_HAS_CONFIGURABLE | JS_PROP_HAS_WRITABLE | JS_PROP_HAS_ENUMERABLE;
    }
    if (JS_DefineProperty(ctx, ctx->global_obj, prop, func,
                          JS_UNDEFINED, JS_UNDEFINED, flags) < 0)
        return -1;
    return 0;
}

static JSValue JS_GetGlobalVar(JSContext *ctx, JSAtom prop,
                               bool throw_ref_error)
{
    JSObject *p;
    JSShapeProperty *prs;
    JSProperty *pr;

    /* no exotic behavior is possible in global_var_obj */
    p = JS_VALUE_GET_OBJ(ctx->global_var_obj);
    prs = find_own_property(&pr, p, prop);
    if (prs) {
        /* XXX: should handle JS_PROP_TMASK properties */
        if (unlikely(JS_IsUninitialized(pr->u.value)))
            return JS_ThrowReferenceErrorUninitialized(ctx, prs->atom);
        return js_dup(pr->u.value);
    }
    return JS_GetPropertyInternal(ctx, ctx->global_obj, prop,
                                 ctx->global_obj, throw_ref_error);
}

/* construct a reference to a global variable */
static int JS_GetGlobalVarRef(JSContext *ctx, JSAtom prop, JSValue *sp)
{
    JSObject *p;
    JSShapeProperty *prs;
    JSProperty *pr;

    /* no exotic behavior is possible in global_var_obj */
    p = JS_VALUE_GET_OBJ(ctx->global_var_obj);
    prs = find_own_property(&pr, p, prop);
    if (prs) {
        /* XXX: should handle JS_PROP_AUTOINIT properties? */
        /* XXX: conformance: do these tests in
           OP_put_var_ref/OP_get_var_ref ? */
        if (unlikely(JS_IsUninitialized(pr->u.value))) {
            JS_ThrowReferenceErrorUninitialized(ctx, prs->atom);
            return -1;
        }
        if (unlikely(!(prs->flags & JS_PROP_WRITABLE))) {
            return JS_ThrowTypeErrorReadOnly(ctx, JS_PROP_THROW, prop);
        }
        sp[0] = js_dup(ctx->global_var_obj);
    } else {
        int ret;
        ret = JS_HasProperty(ctx, ctx->global_obj, prop);
        if (ret < 0)
            return -1;
        if (ret) {
            sp[0] = js_dup(ctx->global_obj);
        } else {
            sp[0] = JS_UNDEFINED;
        }
    }
    sp[1] = JS_AtomToValue(ctx, prop);
    return 0;
}

/* use for strict variable access: test if the variable exists */
static int JS_CheckGlobalVar(JSContext *ctx, JSAtom prop)
{
    JSObject *p;
    JSShapeProperty *prs;
    int ret;

    /* no exotic behavior is possible in global_var_obj */
    p = JS_VALUE_GET_OBJ(ctx->global_var_obj);
    prs = find_own_property1(p, prop);
    if (prs) {
        ret = true;
    } else {
        ret = JS_HasProperty(ctx, ctx->global_obj, prop);
        if (ret < 0)
            return -1;
    }
    return ret;
}

/* flag = 0: normal variable write
   flag = 1: initialize lexical variable
   flag = 2: normal variable write, strict check was done before
*/
static int JS_SetGlobalVar(JSContext *ctx, JSAtom prop, JSValue val,
                           int flag)
{
    JSObject *p;
    JSShapeProperty *prs;
    JSProperty *pr;
    int flags;

    /* no exotic behavior is possible in global_var_obj */
    p = JS_VALUE_GET_OBJ(ctx->global_var_obj);
    prs = find_own_property(&pr, p, prop);
    if (prs) {
        /* XXX: should handle JS_PROP_AUTOINIT properties? */
        if (flag != 1) {
            if (unlikely(JS_IsUninitialized(pr->u.value))) {
                JS_FreeValue(ctx, val);
                JS_ThrowReferenceErrorUninitialized(ctx, prs->atom);
                return -1;
            }
            if (unlikely(!(prs->flags & JS_PROP_WRITABLE))) {
                JS_FreeValue(ctx, val);
                return JS_ThrowTypeErrorReadOnly(ctx, JS_PROP_THROW, prop);
            }
        }
        set_value(ctx, &pr->u.value, val);
        return 0;
    }
    flags = JS_PROP_THROW_STRICT;
    if (is_strict_mode(ctx))
        flags |= JS_PROP_NO_ADD;
    return JS_SetPropertyInternal(ctx, ctx->global_obj, prop, val, flags);
}

/* return -1, false or true */
static int JS_DeleteGlobalVar(JSContext *ctx, JSAtom prop)
{
    JSObject *p;
    JSShapeProperty *prs;
    JSProperty *pr;
    int ret;

    /* 9.1.1.4.7 DeleteBinding ( N ) */
    p = JS_VALUE_GET_OBJ(ctx->global_var_obj);
    prs = find_own_property(&pr, p, prop);
    if (prs)
        return false; /* lexical variables cannot be deleted */
    ret = JS_HasProperty(ctx, ctx->global_obj, prop);
    if (ret < 0)
        return -1;
    if (ret) {
        return JS_DeleteProperty(ctx, ctx->global_obj, prop, 0);
    } else {
        return true;
    }
}

/* return -1, false or true. return false if not configurable or
   invalid object. return -1 in case of exception.
   flags can be 0, JS_PROP_THROW or JS_PROP_THROW_STRICT */
int JS_DeleteProperty(JSContext *ctx, JSValueConst obj, JSAtom prop, int flags)
{
    JSValue obj1;
    JSObject *p;
    int res;

    obj1 = JS_ToObject(ctx, obj);
    if (JS_IsException(obj1))
        return -1;
    p = JS_VALUE_GET_OBJ(obj1);
    res = delete_property(ctx, p, prop);
    JS_FreeValue(ctx, obj1);
    if (res != false)
        return res;
    if ((flags & JS_PROP_THROW) ||
        ((flags & JS_PROP_THROW_STRICT) && is_strict_mode(ctx))) {
        JS_ThrowTypeError(ctx, "could not delete property");
        return -1;
    }
    return false;
}

int JS_DeletePropertyInt64(JSContext *ctx, JSValueConst obj, int64_t idx, int flags)
{
    JSAtom prop;
    int res;

    if ((uint64_t)idx <= JS_ATOM_MAX_INT) {
        /* fast path for fast arrays */
        return JS_DeleteProperty(ctx, obj, __JS_AtomFromUInt32(idx), flags);
    }
    prop = JS_NewAtomInt64(ctx, idx);
    if (prop == JS_ATOM_NULL)
        return -1;
    res = JS_DeleteProperty(ctx, obj, prop, flags);
    JS_FreeAtom(ctx, prop);
    return res;
}

bool JS_IsFunction(JSContext *ctx, JSValueConst val)
{
    JSObject *p;
    if (JS_VALUE_GET_TAG(val) != JS_TAG_OBJECT)
        return false;
    p = JS_VALUE_GET_OBJ(val);
    switch(p->class_id) {
    case JS_CLASS_BYTECODE_FUNCTION:
        return true;
    case JS_CLASS_PROXY:
        return p->u.proxy_data->is_func;
    default:
        return (ctx->rt->class_array[p->class_id].call != NULL);
    }
}

static bool JS_IsCFunction(JSContext *ctx, JSValueConst val, JSCFunction *func,
                           int magic)
{
    JSObject *p;
    if (JS_VALUE_GET_TAG(val) != JS_TAG_OBJECT)
        return false;
    p = JS_VALUE_GET_OBJ(val);
    if (p->class_id == JS_CLASS_C_FUNCTION)
        return (p->u.cfunc.c_function.generic == func && p->u.cfunc.magic == magic);
    else
        return false;
}

bool JS_IsConstructor(JSContext *ctx, JSValueConst val)
{
    JSObject *p;
    if (JS_VALUE_GET_TAG(val) != JS_TAG_OBJECT)
        return false;
    p = JS_VALUE_GET_OBJ(val);
    return p->is_constructor;
}

bool JS_SetConstructorBit(JSContext *ctx, JSValueConst func_obj, bool val)
{
    JSObject *p;
    if (JS_VALUE_GET_TAG(func_obj) != JS_TAG_OBJECT)
        return false;
    p = JS_VALUE_GET_OBJ(func_obj);
    p->is_constructor = val;
    return true;
}

bool JS_IsRegExp(JSValueConst val)
{
    return JS_CLASS_REGEXP == JS_GetClassID(val);
}

bool JS_IsMap(JSValueConst val)
{
    return JS_CLASS_MAP == JS_GetClassID(val);
}

bool JS_IsSet(JSValueConst val)
{
    return JS_CLASS_SET == JS_GetClassID(val);
}

bool JS_IsWeakRef(JSValueConst val)
{
    return JS_CLASS_WEAK_REF == JS_GetClassID(val);
}

bool JS_IsWeakSet(JSValueConst val)
{
    return JS_CLASS_WEAKSET == JS_GetClassID(val);
}

bool JS_IsWeakMap(JSValueConst val)
{
    return JS_CLASS_WEAKMAP == JS_GetClassID(val);
}

bool JS_IsDataView(JSValueConst val)
{
    return JS_CLASS_DATAVIEW == JS_GetClassID(val);
}

bool JS_IsError(JSValueConst val)
{
    return JS_CLASS_ERROR == JS_GetClassID(val);
}

/* used to avoid catching interrupt exceptions */
bool JS_IsUncatchableError(JSValueConst val)
{
    JSObject *p;
    if (JS_VALUE_GET_TAG(val) != JS_TAG_OBJECT)
        return false;
    p = JS_VALUE_GET_OBJ(val);
    return p->class_id == JS_CLASS_ERROR && p->is_uncatchable_error;
}

static void js_set_uncatchable_error(JSContext *ctx, JSValueConst val, bool flag)
{
    JSObject *p;
    if (JS_VALUE_GET_TAG(val) != JS_TAG_OBJECT)
        return;
    p = JS_VALUE_GET_OBJ(val);
    if (p->class_id == JS_CLASS_ERROR)
        p->is_uncatchable_error = flag;
}

void JS_SetUncatchableError(JSContext *ctx, JSValueConst val)
{
    js_set_uncatchable_error(ctx, val, true);
}

void JS_ClearUncatchableError(JSContext *ctx, JSValueConst val)
{
    js_set_uncatchable_error(ctx, val, false);
}

void JS_ResetUncatchableError(JSContext *ctx)
{
    js_set_uncatchable_error(ctx, ctx->rt->current_exception, false);
}

int JS_SetOpaque(JSValueConst obj, void *opaque)
{
    JSObject *p;
    if (JS_VALUE_GET_TAG(obj) == JS_TAG_OBJECT) {
        p = JS_VALUE_GET_OBJ(obj);
        // User code can't set the opaque of internal objects.
        if (p->class_id >= JS_CLASS_INIT_COUNT) {
            p->u.opaque = opaque;
            return 0;
        }
    }

    return -1;
}

/* |obj| must be a JSObject of an internal class. */
static void JS_SetOpaqueInternal(JSValueConst obj, void *opaque)
{
    JSObject *p;
    assert(JS_VALUE_GET_TAG(obj) == JS_TAG_OBJECT);
    p = JS_VALUE_GET_OBJ(obj);
    assert(p->class_id < JS_CLASS_INIT_COUNT);
    p->u.opaque = opaque;
}

/* return NULL if not an object of class class_id */
void *JS_GetOpaque(JSValueConst obj, JSClassID class_id)
{
    JSObject *p;
    if (JS_VALUE_GET_TAG(obj) != JS_TAG_OBJECT)
        return NULL;
    p = JS_VALUE_GET_OBJ(obj);
    if (p->class_id != class_id)
        return NULL;
    return p->u.opaque;
}

void *JS_GetOpaque2(JSContext *ctx, JSValueConst obj, JSClassID class_id)
{
    void *p = JS_GetOpaque(obj, class_id);
    if (unlikely(!p)) {
        JS_ThrowTypeErrorInvalidClass(ctx, class_id);
    }
    return p;
}

void *JS_GetAnyOpaque(JSValueConst obj, JSClassID *class_id)
{
    JSObject *p;
    if (JS_VALUE_GET_TAG(obj) != JS_TAG_OBJECT) {
        *class_id = 0;
        return NULL;
    }
    p = JS_VALUE_GET_OBJ(obj);
    *class_id = p->class_id;
    return p->u.opaque;
}

static JSValue JS_ToPrimitiveFree(JSContext *ctx, JSValue val, int hint)
{
    int i;
    bool force_ordinary;

    JSAtom method_name;
    JSValue method, ret;
    if (JS_VALUE_GET_TAG(val) != JS_TAG_OBJECT)
        return val;
    force_ordinary = hint & HINT_FORCE_ORDINARY;
    hint &= ~HINT_FORCE_ORDINARY;
    if (!force_ordinary) {
        method = JS_GetProperty(ctx, val, JS_ATOM_Symbol_toPrimitive);
        if (JS_IsException(method))
            goto exception;
        /* ECMA says *If exoticToPrim is not undefined* but tests in
           test262 use null as a non callable converter */
        if (!JS_IsUndefined(method) && !JS_IsNull(method)) {
            JSAtom atom;
            JSValue arg;
            switch(hint) {
            case HINT_STRING:
                atom = JS_ATOM_string;
                break;
            case HINT_NUMBER:
                atom = JS_ATOM_number;
                break;
            default:
            case HINT_NONE:
                atom = JS_ATOM_default;
                break;
            }
            arg = JS_AtomToString(ctx, atom);
            ret = JS_CallFree(ctx, method, val, 1, vc(&arg));
            JS_FreeValue(ctx, arg);
            if (JS_IsException(ret))
                goto exception;
            JS_FreeValue(ctx, val);
            if (JS_VALUE_GET_TAG(ret) != JS_TAG_OBJECT)
                return ret;
            JS_FreeValue(ctx, ret);
            return JS_ThrowTypeError(ctx, "toPrimitive");
        }
    }
    if (hint != HINT_STRING)
        hint = HINT_NUMBER;
    for(i = 0; i < 2; i++) {
        if ((i ^ hint) == 0) {
            method_name = JS_ATOM_toString;
        } else {
            method_name = JS_ATOM_valueOf;
        }
        method = JS_GetProperty(ctx, val, method_name);
        if (JS_IsException(method))
            goto exception;
        if (JS_IsFunction(ctx, method)) {
            ret = JS_CallFree(ctx, method, val, 0, NULL);
            if (JS_IsException(ret))
                goto exception;
            if (JS_VALUE_GET_TAG(ret) != JS_TAG_OBJECT) {
                JS_FreeValue(ctx, val);
                return ret;
            }
            JS_FreeValue(ctx, ret);
        } else {
            JS_FreeValue(ctx, method);
        }
    }
    JS_ThrowTypeError(ctx, "toPrimitive");
exception:
    JS_FreeValue(ctx, val);
    return JS_EXCEPTION;
}

static JSValue JS_ToPrimitive(JSContext *ctx, JSValueConst val, int hint)
{
    return JS_ToPrimitiveFree(ctx, js_dup(val), hint);
}

void JS_SetIsHTMLDDA(JSContext *ctx, JSValueConst obj)
{
    JSObject *p;
    if (JS_VALUE_GET_TAG(obj) != JS_TAG_OBJECT)
        return;
    p = JS_VALUE_GET_OBJ(obj);
    p->is_HTMLDDA = true;
}

static inline bool JS_IsHTMLDDA(JSContext *ctx, JSValueConst obj)
{
    JSObject *p;
    if (JS_VALUE_GET_TAG(obj) != JS_TAG_OBJECT)
        return false;
    p = JS_VALUE_GET_OBJ(obj);
    return p->is_HTMLDDA;
}

static int JS_ToBoolFree(JSContext *ctx, JSValue val)
{
    uint32_t tag = JS_VALUE_GET_TAG(val);
    switch(tag) {
    case JS_TAG_INT:
        return JS_VALUE_GET_INT(val) != 0;
    case JS_TAG_BOOL:
    case JS_TAG_NULL:
    case JS_TAG_UNDEFINED:
        return JS_VALUE_GET_INT(val);
    case JS_TAG_EXCEPTION:
        return -1;
    case JS_TAG_STRING:
        {
            bool ret = JS_VALUE_GET_STRING(val)->len != 0;
            JS_FreeValue(ctx, val);
            return ret;
        }
    case JS_TAG_SHORT_BIG_INT:
        return JS_VALUE_GET_SHORT_BIG_INT(val) != 0;
    case JS_TAG_BIG_INT:
        {
            JSBigInt *p = JS_VALUE_GET_PTR(val);
            bool ret;
            int i;

            /* fail safe: we assume it is not necessarily
                normalized. Beginning from the MSB ensures that the
                test is fast. */
            ret = false;
            for(i = p->len - 1; i >= 0; i--) {
                if (p->tab[i] != 0) {
                    ret = true;
                    break;
                }
            }
            JS_FreeValue(ctx, val);
            return ret;
        }
    case JS_TAG_OBJECT:
        {
            JSObject *p = JS_VALUE_GET_OBJ(val);
            bool ret = !p->is_HTMLDDA;
            JS_FreeValue(ctx, val);
            return ret;
        }
        break;
    default:
        if (JS_TAG_IS_FLOAT64(tag)) {
            double d = JS_VALUE_GET_FLOAT64(val);
            return !isnan(d) && d != 0;
        } else {
            JS_FreeValue(ctx, val);
            return true;
        }
    }
}

int JS_ToBool(JSContext *ctx, JSValueConst val)
{
    return JS_ToBoolFree(ctx, js_dup(val));
}

/* pc points to pure ASCII or UTF-8, null terminated contents */
static int skip_spaces(const char *pc)
{
    const uint8_t *p, *p_next, *p_start;
    uint32_t c;

    p = p_start = (const uint8_t *)pc;
    for (;;) {
        c = *p++;
        if (c < 0x80) {
            if (!((c >= 0x09 && c <= 0x0d) || (c == 0x20)))
                break;
        } else {
            c = utf8_decode(p - 1, &p_next);
            /* no need to test for invalid UTF-8, 0xFFFD is not a space */
            if (!lre_is_space(c))
                break;
            p = p_next;
        }
    }
    return p - 1 - p_start;
}

static inline int js_to_digit(int c)
{
    if (c >= '0' && c <= '9')
        return c - '0';
    else if (c >= 'A' && c <= 'Z')
        return c - 'A' + 10;
    else if (c >= 'a' && c <= 'z')
        return c - 'a' + 10;
    else
        return 36;
}

/* bigint support */

#define ADDC(res, carry_out, op1, op2, carry_in)        \
do {                                                    \
    js_limb_t __v, __a, __k, __k1;                      \
    __v = (op1);                                        \
    __a = __v + (op2);                                  \
    __k1 = __a < __v;                                   \
    __k = (carry_in);                                   \
    __a = __a + __k;                                    \
    carry_out = (__a < __k) | __k1;                     \
    res = __a;                                          \
} while (0)

/* a != 0 */
static inline js_limb_t js_limb_clz(js_limb_t a)
{
    if (!a)
        return JS_LIMB_BITS;
    return clz32(a);
}

static js_limb_t js_mp_add(js_limb_t *res, const js_limb_t *op1, const js_limb_t *op2,
                     js_limb_t n, js_limb_t carry)
{
    int i;
    for(i = 0;i < n; i++) {
        ADDC(res[i], carry, op1[i], op2[i], carry);
    }
    return carry;
}

static js_limb_t js_mp_sub(js_limb_t *res, const js_limb_t *op1, const js_limb_t *op2,
                        int n, js_limb_t carry)
{
    int i;
    js_limb_t k, a, v, k1;

    k = carry;
    for(i=0;i<n;i++) {
        v = op1[i];
        a = v - op2[i];
        k1 = a > v;
        v = a - k;
        k = (v > a) | k1;
        res[i] = v;
    }
    return k;
}

/* compute 0 - op2. carry = 0 or 1. */
static js_limb_t js_mp_neg(js_limb_t *res, const js_limb_t *op2, int n)
{
    int i;
    js_limb_t v, carry;

    carry = 1;
    for(i=0;i<n;i++) {
        v = ~op2[i] + carry;
        carry = v < carry;
        res[i] = v;
    }
    return carry;
}

/* tabr[] = taba[] * b + l. Return the high carry */
static js_limb_t js_mp_mul1(js_limb_t *tabr, const js_limb_t *taba, js_limb_t n,
                      js_limb_t b, js_limb_t l)
{
    js_limb_t i;
    js_dlimb_t t;

    for(i = 0; i < n; i++) {
        t = (js_dlimb_t)taba[i] * (js_dlimb_t)b + l;
        tabr[i] = t;
        l = t >> JS_LIMB_BITS;
    }
    return l;
}

static js_limb_t js_mp_div1(js_limb_t *tabr, const js_limb_t *taba, js_limb_t n,
                      js_limb_t b, js_limb_t r)
{
    js_slimb_t i;
    js_dlimb_t a1;
    for(i = n - 1; i >= 0; i--) {
        a1 = ((js_dlimb_t)r << JS_LIMB_BITS) | taba[i];
        tabr[i] = a1 / b;
        r = a1 % b;
    }
    return r;
}

/* tabr[] += taba[] * b, return the high word. */
static js_limb_t js_mp_add_mul1(js_limb_t *tabr, const js_limb_t *taba, js_limb_t n,
                          js_limb_t b)
{
    js_limb_t i, l;
    js_dlimb_t t;

    l = 0;
    for(i = 0; i < n; i++) {
        t = (js_dlimb_t)taba[i] * (js_dlimb_t)b + l + tabr[i];
        tabr[i] = t;
        l = t >> JS_LIMB_BITS;
    }
    return l;
}

/* size of the result : op1_size + op2_size. */
static void js_mp_mul_basecase(js_limb_t *result,
                            const js_limb_t *op1, js_limb_t op1_size,
                            const js_limb_t *op2, js_limb_t op2_size)
{
    int i;
    js_limb_t r;

    result[op1_size] = js_mp_mul1(result, op1, op1_size, op2[0], 0);
    for(i=1;i<op2_size;i++) {
        r = js_mp_add_mul1(result + i, op1, op1_size, op2[i]);
        result[i + op1_size] = r;
    }
}

/* tabr[] -= taba[] * b. Return the value to substract to the high
   word. */
static js_limb_t js_mp_sub_mul1(js_limb_t *tabr, const js_limb_t *taba, js_limb_t n,
                          js_limb_t b)
{
    js_limb_t i, l;
    js_dlimb_t t;

    l = 0;
    for(i = 0; i < n; i++) {
        t = tabr[i] - (js_dlimb_t)taba[i] * (js_dlimb_t)b - l;
        tabr[i] = t;
        l = -(t >> JS_LIMB_BITS);
    }
    return l;
}

/* WARNING: d must be >= 2^(JS_LIMB_BITS-1) */
static inline js_limb_t js_udiv1norm_init(js_limb_t d)
{
    js_limb_t a0, a1;
    a1 = -d - 1;
    a0 = -1;
    return (((js_dlimb_t)a1 << JS_LIMB_BITS) | a0) / d;
}

/* return the quotient and the remainder in '*pr'of 'a1*2^JS_LIMB_BITS+a0
   / d' with 0 <= a1 < d. */
static inline js_limb_t js_udiv1norm(js_limb_t *pr, js_limb_t a1, js_limb_t a0,
                                     js_limb_t d, js_limb_t d_inv)
{
    js_limb_t n1m, n_adj, q, r, ah;
    js_dlimb_t a;
    n1m = ((js_slimb_t)a0 >> (JS_LIMB_BITS - 1));
    n_adj = a0 + (n1m & d);
    a = (js_dlimb_t)d_inv * (a1 - n1m) + n_adj;
    q = (a >> JS_LIMB_BITS) + a1;
    /* compute a - q * r and update q so that the remainder is\
       between 0 and d - 1 */
    a = ((js_dlimb_t)a1 << JS_LIMB_BITS) | a0;
    a = a - (js_dlimb_t)q * d - d;
    ah = a >> JS_LIMB_BITS;
    q += 1 + ah;
    r = (js_limb_t)a + (ah & d);
    *pr = r;
    return q;
}

#define UDIV1NORM_THRESHOLD 3

/* b must be >= 1 << (JS_LIMB_BITS - 1) */
static js_limb_t js_mp_div1norm(js_limb_t *tabr, const js_limb_t *taba, js_limb_t n,
                          js_limb_t b, js_limb_t r)
{
    js_slimb_t i;

    if (n >= UDIV1NORM_THRESHOLD) {
        js_limb_t b_inv;
        b_inv = js_udiv1norm_init(b);
        for(i = n - 1; i >= 0; i--) {
            tabr[i] = js_udiv1norm(&r, r, taba[i], b, b_inv);
        }
    } else {
        js_dlimb_t a1;
        for(i = n - 1; i >= 0; i--) {
            a1 = ((js_dlimb_t)r << JS_LIMB_BITS) | taba[i];
            tabr[i] = a1 / b;
            r = a1 % b;
        }
    }
    return r;
}

/* base case division: divides taba[0..na-1] by tabb[0..nb-1]. tabb[nb
   - 1] must be >= 1 << (JS_LIMB_BITS - 1). na - nb must be >= 0. 'taba'
   is modified and contains the remainder (nb limbs). tabq[0..na-nb]
   contains the quotient with tabq[na - nb] <= 1. */
static void js_mp_divnorm(js_limb_t *tabq, js_limb_t *taba, js_limb_t na,
                       const js_limb_t *tabb, js_limb_t nb)
{
    js_limb_t r, a, c, q, v, b1, b1_inv, n, dummy_r;
    int i, j;

    b1 = tabb[nb - 1];
    if (nb == 1) {
        taba[0] = js_mp_div1norm(tabq, taba, na, b1, 0);
        return;
    }
    n = na - nb;

    if (n >= UDIV1NORM_THRESHOLD)
        b1_inv = js_udiv1norm_init(b1);
    else
        b1_inv = 0;

    /* first iteration: the quotient is only 0 or 1 */
    q = 1;
    for(j = nb - 1; j >= 0; j--) {
        if (taba[n + j] != tabb[j]) {
            if (taba[n + j] < tabb[j])
                q = 0;
            break;
        }
    }
    tabq[n] = q;
    if (q) {
        js_mp_sub(taba + n, taba + n, tabb, nb, 0);
    }

    for(i = n - 1; i >= 0; i--) {
        if (unlikely(taba[i + nb] >= b1)) {
            q = -1;
        } else if (b1_inv) {
            q = js_udiv1norm(&dummy_r, taba[i + nb], taba[i + nb - 1], b1, b1_inv);
        } else {
            js_dlimb_t al;
            al = ((js_dlimb_t)taba[i + nb] << JS_LIMB_BITS) | taba[i + nb - 1];
            q = al / b1;
            r = al % b1;
        }
        r = js_mp_sub_mul1(taba + i, tabb, nb, q);

        v = taba[i + nb];
        a = v - r;
        c = (a > v);
        taba[i + nb] = a;

        if (c != 0) {
            /* negative result */
            for(;;) {
                q--;
                c = js_mp_add(taba + i, taba + i, tabb, nb, 0);
                /* propagate carry and test if positive result */
                if (c != 0) {
                    if (++taba[i + nb] == 0) {
                        break;
                    }
                }
            }
        }
        tabq[i] = q;
    }
}

/* 1 <= shift <= JS_LIMB_BITS - 1 */
static js_limb_t js_mp_shl(js_limb_t *tabr, const js_limb_t *taba, int n,
                        int shift)
{
    int i;
    js_limb_t l, v;
    l = 0;
    for(i = 0; i < n; i++) {
        v = taba[i];
        tabr[i] = (v << shift) | l;
        l = v >> (JS_LIMB_BITS - shift);
    }
    return l;
}

/* r = (a + high*B^n) >> shift. Return the remainder r (0 <= r < 2^shift).
   1 <= shift <= LIMB_BITS - 1 */
static js_limb_t js_mp_shr(js_limb_t *tab_r, const js_limb_t *tab, int n,
                        int shift, js_limb_t high)
{
    int i;
    js_limb_t l, a;

    l = high;
    for(i = n - 1; i >= 0; i--) {
        a = tab[i];
        tab_r[i] = (a >> shift) | (l << (JS_LIMB_BITS - shift));
        l = a;
    }
    return l & (((js_limb_t)1 << shift) - 1);
}

static JSBigInt *js_bigint_new(JSContext *ctx, int len)
{
    JSBigInt *r;
    if (len > JS_BIGINT_MAX_SIZE) {
        JS_ThrowRangeError(ctx, "BigInt is too large to allocate");
        return NULL;
    }
    r = js_malloc(ctx, sizeof(JSBigInt) + len * sizeof(js_limb_t));
    if (!r)
        return NULL;
    r->header.ref_count = 1;
    r->len = len;
    return r;
}

static JSBigInt *js_bigint_set_si(JSBigIntBuf *buf, js_slimb_t a)
{
    JSBigInt *r = (JSBigInt *)buf->big_int_buf;
    r->header.ref_count = 0; /* fail safe */
    r->len = 1;
    r->tab[0] = a;
    return r;
}

static JSBigInt *js_bigint_set_si64(JSBigIntBuf *buf, int64_t a)
{
    JSBigInt *r = (JSBigInt *)buf->big_int_buf;
    r->header.ref_count = 0; /* fail safe */
    if (a >= INT32_MIN && a <= INT32_MAX) {
        r->len = 1;
        r->tab[0] = a;
    } else {
        r->len = 2;
        r->tab[0] = a;
        r->tab[1] = a >> JS_LIMB_BITS;
    }
    return r;
}

/* val must be a short big int */
static JSBigInt *js_bigint_set_short(JSBigIntBuf *buf, JSValueConst val)
{
    return js_bigint_set_si(buf, JS_VALUE_GET_SHORT_BIG_INT(val));
}

static __maybe_unused void js_bigint_dump1(JSContext *ctx, const char *str,
                                           const js_limb_t *tab, int len)
{
    int i;
    printf("%s: ", str);
    for(i = len - 1; i >= 0; i--) {
        printf(" %08x", tab[i]);
    }
    printf("\n");
}

static __maybe_unused void js_bigint_dump(JSContext *ctx, const char *str,
                                          const JSBigInt *p)
{
    js_bigint_dump1(ctx, str, p->tab, p->len);
}

static JSBigInt *js_bigint_new_si(JSContext *ctx, js_slimb_t a)
{
    JSBigInt *r;
    r = js_bigint_new(ctx, 1);
    if (!r)
        return NULL;
    r->tab[0] = a;
    return r;
}

static JSBigInt *js_bigint_new_si64(JSContext *ctx, int64_t a)
{
    if (a >= INT32_MIN && a <= INT32_MAX) {
        return js_bigint_new_si(ctx, a);
    } else {
        JSBigInt *r;
        r = js_bigint_new(ctx, 2);
        if (!r)
            return NULL;
        r->tab[0] = a;
        r->tab[1] = a >> 32;
        return r;
    }
}

static JSBigInt *js_bigint_new_ui64(JSContext *ctx, uint64_t a)
{
    if (a <= INT64_MAX) {
        return js_bigint_new_si64(ctx, a);
    } else {
        JSBigInt *r;
        r = js_bigint_new(ctx, (65 + JS_LIMB_BITS - 1) / JS_LIMB_BITS);
        if (!r)
            return NULL;
        r->tab[0] = a;
        r->tab[1] = a >> 32;
        r->tab[2] = 0;
        return r;
    }
}

static JSBigInt *js_bigint_new_di(JSContext *ctx, js_sdlimb_t a)
{
    JSBigInt *r;
    if (a == (js_slimb_t)a) {
        r = js_bigint_new(ctx, 1);
        if (!r)
            return NULL;
        r->tab[0] = a;
    } else {
        r = js_bigint_new(ctx, 2);
        if (!r)
            return NULL;
        r->tab[0] = a;
        r->tab[1] = a >> JS_LIMB_BITS;
    }
    return r;
}

/* Remove redundant high order limbs. Warning: 'a' may be
   reallocated. Can never fail.
*/
static JSBigInt *js_bigint_normalize1(JSContext *ctx, JSBigInt *a, int l)
{
    js_limb_t v;

    assert(a->header.ref_count == 1);
    while (l > 1) {
        v = a->tab[l - 1];
        if ((v != 0 && v != -1) ||
            (v & 1) != (a->tab[l - 2] >> (JS_LIMB_BITS - 1))) {
            break;
        }
        l--;
    }
    if (l != a->len) {
        JSBigInt *a1;
        /* realloc to reduce the size */
        a->len = l;
        a1 = js_realloc(ctx, a, sizeof(JSBigInt) + l * sizeof(js_limb_t));
        if (a1)
            a = a1;
    }
    return a;
}

static JSBigInt *js_bigint_normalize(JSContext *ctx, JSBigInt *a)
{
    return js_bigint_normalize1(ctx, a, a->len);
}

/* return 0 or 1 depending on the sign */
static inline int js_bigint_sign(const JSBigInt *a)
{
    return a->tab[a->len - 1] >> (JS_LIMB_BITS - 1);
}

static js_slimb_t js_bigint_get_si_sat(const JSBigInt *a)
{
    if (a->len == 1) {
        return a->tab[0];
    } else {
        if (js_bigint_sign(a))
            return INT32_MIN;
        else
            return INT32_MAX;
    }
}

/* add the op1 limb */
static JSBigInt *js_bigint_extend(JSContext *ctx, JSBigInt *r,
                                  js_limb_t op1)
{
    int n2 = r->len;
    if ((op1 != 0 && op1 != -1) ||
        (op1 & 1) != r->tab[n2 - 1] >> (JS_LIMB_BITS - 1)) {
        JSBigInt *r1;
        r1 = js_realloc(ctx, r,
                        sizeof(JSBigInt) + (n2 + 1) * sizeof(js_limb_t));
        if (!r1) {
            js_free(ctx, r);
            return NULL;
        }
        r = r1;
        r->len = n2 + 1;
        r->tab[n2] = op1;
    } else {
        /* otherwise still need to normalize the result */
        r = js_bigint_normalize(ctx, r);
    }
    return r;
}

/* return NULL in case of error. Compute a + b (b_neg = 0) or a - b
   (b_neg = 1) */
/* XXX: optimize */
static JSBigInt *js_bigint_add(JSContext *ctx, const JSBigInt *a,
                               const JSBigInt *b, int b_neg)
{
    JSBigInt *r;
    int n1, n2, i;
    js_limb_t carry, op1, op2, a_sign, b_sign;

    n2 = max_int(a->len, b->len);
    n1 = min_int(a->len, b->len);
    r = js_bigint_new(ctx, n2);
    if (!r)
        return NULL;
    /* XXX: optimize */
    /* common part */
    carry = b_neg;
    for(i = 0; i < n1; i++) {
        op1 = a->tab[i];
        op2 = b->tab[i] ^ (-b_neg);
        ADDC(r->tab[i], carry, op1, op2, carry);
    }
    a_sign = -js_bigint_sign(a);
    b_sign = (-js_bigint_sign(b)) ^ (-b_neg);
    /* part with sign extension of one operand  */
    if (a->len > b->len) {
        for(i = n1; i < n2; i++) {
            op1 = a->tab[i];
            ADDC(r->tab[i], carry, op1, b_sign, carry);
        }
    } else if (a->len < b->len) {
        for(i = n1; i < n2; i++) {
            op2 = b->tab[i] ^ (-b_neg);
            ADDC(r->tab[i], carry, a_sign, op2, carry);
        }
    }

    /* part with sign extension for both operands. Extend the result
       if necessary */
    return js_bigint_extend(ctx, r, a_sign + b_sign + carry);
}

/* XXX: optimize */
static JSBigInt *js_bigint_neg(JSContext *ctx, const JSBigInt *a)
{
    JSBigIntBuf buf;
    JSBigInt *b;
    b = js_bigint_set_si(&buf, 0);
    return js_bigint_add(ctx, b, a, 1);
}

static JSBigInt *js_bigint_mul(JSContext *ctx, const JSBigInt *a,
                               const JSBigInt *b)
{
    JSBigInt *r;

    r = js_bigint_new(ctx, a->len + b->len);
    if (!r)
        return NULL;
    js_mp_mul_basecase(r->tab, a->tab, a->len, b->tab, b->len);
    /* correct the result if negative operands (no overflow is
       possible) */
    if (js_bigint_sign(a))
        js_mp_sub(r->tab + a->len, r->tab + a->len, b->tab, b->len, 0);
    if (js_bigint_sign(b))
        js_mp_sub(r->tab + b->len, r->tab + b->len, a->tab, a->len, 0);
    return js_bigint_normalize(ctx, r);
}

/* return the division or the remainder. 'b' must be != 0. return NULL
   in case of exception (division by zero or memory error) */
static JSBigInt *js_bigint_divrem(JSContext *ctx, const JSBigInt *a,
                                  const JSBigInt *b, bool is_rem)
{
    JSBigInt *r, *q;
    js_limb_t *tabb, h;
    int na, nb, a_sign, b_sign, shift;

    if (b->len == 1 && b->tab[0] == 0) {
        JS_ThrowRangeError(ctx, "BigInt division by zero");
        return NULL;
    }

    a_sign = js_bigint_sign(a);
    b_sign = js_bigint_sign(b);
    na = a->len;
    nb = b->len;

    r = js_bigint_new(ctx, na + 2);
    if (!r)
        return NULL;
    if (a_sign) {
        js_mp_neg(r->tab, a->tab, na);
    } else {
        memcpy(r->tab, a->tab, na * sizeof(a->tab[0]));
    }
    /* normalize */
    while (na > 1 && r->tab[na - 1] == 0)
        na--;

    tabb = js_malloc(ctx, nb * sizeof(tabb[0]));
    if (!tabb) {
        js_free(ctx, r);
        return NULL;
    }
    if (b_sign) {
        js_mp_neg(tabb, b->tab, nb);
    } else {
        memcpy(tabb, b->tab, nb * sizeof(tabb[0]));
    }
    /* normalize */
    while (nb > 1 && tabb[nb - 1] == 0)
        nb--;

    /* trivial case if 'a' is small */
    if (na < nb) {
        js_free(ctx, r);
        js_free(ctx, tabb);
        if (is_rem) {
            /* r = a */
            r = js_bigint_new(ctx, a->len);
            if (!r)
                return NULL;
            memcpy(r->tab, a->tab, a->len * sizeof(a->tab[0]));
            return r;
        } else {
            /* q = 0 */
            return js_bigint_new_si(ctx, 0);
        }
    }

    /* normalize 'b' */
    shift = js_limb_clz(tabb[nb - 1]);
    if (shift != 0) {
        js_mp_shl(tabb, tabb, nb, shift);
        h = js_mp_shl(r->tab, r->tab, na, shift);
        if (h != 0)
            r->tab[na++] = h;
    }

    q = js_bigint_new(ctx, na - nb + 2); /* one more limb for the sign */
    if (!q) {
        js_free(ctx, r);
        js_free(ctx, tabb);
        return NULL;
    }

    //    js_bigint_dump1(ctx, "a", r->tab, na);
    //    js_bigint_dump1(ctx, "b", tabb, nb);
    js_mp_divnorm(q->tab, r->tab, na, tabb, nb);
    js_free(ctx, tabb);

    if (is_rem) {
        js_free(ctx, q);
        if (shift != 0)
            js_mp_shr(r->tab, r->tab, nb, shift, 0);
        r->tab[nb++] = 0;
        if (a_sign)
            js_mp_neg(r->tab, r->tab, nb);
        r = js_bigint_normalize1(ctx, r, nb);
        return r;
    } else {
        js_free(ctx, r);
        q->tab[na - nb + 1] = 0;
        if (a_sign ^ b_sign) {
            js_mp_neg(q->tab, q->tab, q->len);
        }
        q = js_bigint_normalize(ctx, q);
        return q;
    }
}

/* and, or, xor */
static JSBigInt *js_bigint_logic(JSContext *ctx, const JSBigInt *a,
                                 const JSBigInt *b, OPCodeEnum op)
{
    JSBigInt *r;
    js_limb_t b_sign;
    int a_len, b_len, i;

    if (a->len < b->len) {
        const JSBigInt *tmp;
        tmp = a;
        a = b;
        b = tmp;
    }
    /* a_len >= b_len */
    a_len = a->len;
    b_len = b->len;
    b_sign = -js_bigint_sign(b);

    r = js_bigint_new(ctx, a_len);
    if (!r)
        return NULL;
    switch(op) {
    case OP_or:
        for(i = 0; i < b_len; i++) {
            r->tab[i] = a->tab[i] | b->tab[i];
        }
        for(i = b_len; i < a_len; i++) {
            r->tab[i] = a->tab[i] | b_sign;
        }
        break;
    case OP_and:
        for(i = 0; i < b_len; i++) {
            r->tab[i] = a->tab[i] & b->tab[i];
        }
        for(i = b_len; i < a_len; i++) {
            r->tab[i] = a->tab[i] & b_sign;
        }
        break;
    case OP_xor:
        for(i = 0; i < b_len; i++) {
            r->tab[i] = a->tab[i] ^ b->tab[i];
        }
        for(i = b_len; i < a_len; i++) {
            r->tab[i] = a->tab[i] ^ b_sign;
        }
        break;
    default:
        abort();
    }
    return js_bigint_normalize(ctx, r);
}

static JSBigInt *js_bigint_not(JSContext *ctx, const JSBigInt *a)
{
    JSBigInt *r;
    int i;

    r = js_bigint_new(ctx, a->len);
    if (!r)
        return NULL;
    for(i = 0; i < a->len; i++) {
        r->tab[i] = ~a->tab[i];
    }
    /* no normalization is needed */
    return r;
}

static JSBigInt *js_bigint_shl(JSContext *ctx, const JSBigInt *a,
                               unsigned int shift1)
{
    int d, i, shift;
    JSBigInt *r;
    js_limb_t l;

    if (a->len == 1 && a->tab[0] == 0)
        return js_bigint_new_si(ctx, 0); /* zero case */
    d = shift1 / JS_LIMB_BITS;
    shift = shift1 % JS_LIMB_BITS;
    r = js_bigint_new(ctx, a->len + d);
    if (!r)
        return NULL;
    for(i = 0; i < d; i++)
        r->tab[i] = 0;
    if (shift == 0) {
        for(i = 0; i < a->len; i++) {
            r->tab[i + d] = a->tab[i];
        }
    } else {
        l = js_mp_shl(r->tab + d, a->tab, a->len, shift);
        if (js_bigint_sign(a))
            l |= (js_limb_t)(-1) << shift;
        r = js_bigint_extend(ctx, r, l);
    }
    return r;
}

static JSBigInt *js_bigint_shr(JSContext *ctx, const JSBigInt *a,
                               unsigned int shift1)
{
    int d, i, shift, a_sign, n1;
    JSBigInt *r;

    d = shift1 / JS_LIMB_BITS;
    shift = shift1 % JS_LIMB_BITS;
    a_sign = js_bigint_sign(a);
    if (d >= a->len)
        return js_bigint_new_si(ctx, -a_sign);
    n1 = a->len - d;
    r = js_bigint_new(ctx, n1);
    if (!r)
        return NULL;
    if (shift == 0) {
        for(i = 0; i < n1; i++) {
            r->tab[i] = a->tab[i + d];
        }
        /* no normalization is needed */
    } else {
        js_mp_shr(r->tab, a->tab + d, n1, shift, -a_sign);
        r = js_bigint_normalize(ctx, r);
    }
    return r;
}

static JSBigInt *js_bigint_pow(JSContext *ctx, const JSBigInt *a, JSBigInt *b)
{
    uint32_t e;
    int n_bits, i;
    JSBigInt *r, *r1;

    /* b must be >= 0 */
    if (js_bigint_sign(b)) {
        JS_ThrowRangeError(ctx, "BigInt negative exponent");
        return NULL;
    }
    if (b->len == 1 && b->tab[0] == 0) {
        /* a^0 = 1 */
        return js_bigint_new_si(ctx, 1);
    } else if (a->len == 1) {
        js_limb_t v;
        bool is_neg;

        v = a->tab[0];
        if (v <= 1)
            return js_bigint_new_si(ctx, v);
        else if (v == -1)
            return js_bigint_new_si(ctx, 1 - 2 * (b->tab[0] & 1));
        is_neg = (js_slimb_t)v < 0;
        if (is_neg)
            v = -v;
        if ((v & (v - 1)) == 0) {
            uint64_t e1;
            int n;
            /* v = 2^n */
            n = JS_LIMB_BITS - 1 - js_limb_clz(v);
            if (b->len > 1)
                goto overflow;
            if (b->tab[0] > INT32_MAX)
                goto overflow;
            e = b->tab[0];
            e1 = (uint64_t)e * n;
            if (e1 > JS_BIGINT_MAX_SIZE * JS_LIMB_BITS)
                goto overflow;
            e = e1;
            if (is_neg)
                is_neg = b->tab[0] & 1;
            r = js_bigint_new(ctx,
                              (e + JS_LIMB_BITS + 1 - is_neg) / JS_LIMB_BITS);
            if (!r)
                return NULL;
            memset(r->tab, 0, sizeof(r->tab[0]) * r->len);
            r->tab[e / JS_LIMB_BITS] =
                (js_limb_t)(1 - 2 * is_neg) << (e % JS_LIMB_BITS);
            return r;
        }
    }
    if (b->len > 1)
        goto overflow;
    if (b->tab[0] > INT32_MAX)
        goto overflow;
    e = b->tab[0];
    n_bits = 32 - clz32(e);

    r = js_bigint_new(ctx, a->len);
    if (!r)
        return NULL;
    memcpy(r->tab, a->tab, a->len * sizeof(a->tab[0]));
    for(i = n_bits - 2; i >= 0; i--) {
        r1 = js_bigint_mul(ctx, r, r);
        if (!r1)
            return NULL;
        js_free(ctx, r);
        r = r1;
        if ((e >> i) & 1) {
            r1 = js_bigint_mul(ctx, r, a);
            if (!r1)
                return NULL;
            js_free(ctx, r);
            r = r1;
        }
    }
    return r;
 overflow:
    JS_ThrowRangeError(ctx, "BigInt is too large");
    return NULL;
}

/* return (mant, exp) so that abs(a) ~ mant*2^(exp - (limb_bits -
   1). a must be != 0. */
static uint64_t js_bigint_get_mant_exp(JSContext *ctx,
                                       int *pexp, const JSBigInt *a)
{
    js_limb_t t[4 - JS_LIMB_BITS / 32], carry, v, low_bits;
    int n1, n2, sgn, shift, i, j, e;
    uint64_t a1, a0;

    n2 = 4 - JS_LIMB_BITS / 32;
    n1 = a->len - n2;
    sgn = js_bigint_sign(a);

    /* low_bits != 0 if there are a non zero low bit in abs(a) */
    low_bits = 0;
    carry = sgn;
    for(i = 0; i < n1; i++) {
        v = (a->tab[i] ^ (-sgn)) + carry;
        carry = v < carry;
        low_bits |= v;
    }
    /* get the n2 high limbs of abs(a) */
    for(j = 0; j < n2; j++) {
        i = j + n1;
        if (i < 0) {
            v = 0;
        } else {
            v = (a->tab[i] ^ (-sgn)) + carry;
            carry = v < carry;
        }
        t[j] = v;
    }

    a1 = ((uint64_t)t[2] << 32) | t[1];
    a0 = (uint64_t)t[0] << 32;
    a0 |= (low_bits != 0);
    /* normalize */
    {
        shift = clz64(a1);
        if (shift != 0) {
            a1 = (a1 << shift) | (a0 >> (64 - shift));
            a0 <<= shift;
        }
    }
    a1 |= (a0 != 0); /* keep the bits for the final rounding */
    /* compute the exponent */
    e = a->len * JS_LIMB_BITS - shift - 1;
    *pexp = e;
    return a1;
}

/* shift left with round to nearest, ties to even. n >= 1 */
static uint64_t shr_rndn(uint64_t a, int n)
{
    uint64_t addend = ((a >> n) & 1) + ((1 << (n - 1)) - 1);
    return (a + addend) >> n;
}

/* convert to float64 with round to nearest, ties to even. Return
   +/-infinity if too large. */
static double js_bigint_to_float64(JSContext *ctx, const JSBigInt *a)
{
    int sgn, e;
    uint64_t mant;

    if (a->len == 1) {
        /* fast case, including zero */
        return (double)(js_slimb_t)a->tab[0];
    }

    sgn = js_bigint_sign(a);
    mant = js_bigint_get_mant_exp(ctx, &e, a);
    if (e > 1023) {
        /* overflow: return infinity */
        mant = 0;
        e = 1024;
    } else {
        mant = (mant >> 1) | (mant & 1); /* avoid overflow in rounding */
        mant = shr_rndn(mant, 10);
        /* rounding can cause an overflow */
        if (mant >= ((uint64_t)1 << 53)) {
            mant >>= 1;
            e++;
        }
        mant &= (((uint64_t)1 << 52) - 1);
    }
    return uint64_as_float64(((uint64_t)sgn << 63) |
                             ((uint64_t)(e + 1023) << 52) |
                             mant);
}

/* return (1, NULL) if not an integer, (2, NULL) if NaN or Infinity,
   (0, n) if an integer, (0, NULL) in case of memory error */
static JSBigInt *js_bigint_from_float64(JSContext *ctx, int *pres, double a1)
{
    uint64_t a = float64_as_uint64(a1);
    int sgn, e, shift;
    uint64_t mant;
    JSBigIntBuf buf;
    JSBigInt *r;

    sgn = a >> 63;
    e = (a >> 52) & ((1 << 11) - 1);
    mant = a & (((uint64_t)1 << 52) - 1);
    if (e == 2047) {
        /* NaN, Infinity */
        *pres = 2;
        return NULL;
    }
    if (e == 0 && mant == 0) {
        /* zero */
        *pres = 0;
        return js_bigint_new_si(ctx, 0);
    }
    e -= 1023;
    /* 0 < a < 1 : not an integer */
    if (e < 0)
        goto not_an_integer;
    mant |= (uint64_t)1 << 52;
    if (e < 52) {
        shift = 52 - e;
        /* check that there is no fractional part */
        if (mant & (((uint64_t)1 << shift) - 1)) {
        not_an_integer:
            *pres = 1;
            return NULL;
        }
        mant >>= shift;
        e = 0;
    } else {
        e -= 52;
    }
    if (sgn)
        mant = -mant;
    /* the integer is mant*2^e */
    r = js_bigint_set_si64(&buf, (int64_t)mant);
    *pres = 0;
    return js_bigint_shl(ctx, r, e);
}

/* return -1, 0, 1 or (2) (unordered) */
static int js_bigint_float64_cmp(JSContext *ctx, const JSBigInt *a,
                                 double b)
{
    int b_sign, a_sign, e, f;
    uint64_t mant, b1, a_mant;

    b1 = float64_as_uint64(b);
    b_sign = b1 >> 63;
    e = (b1 >> 52) & ((1 << 11) - 1);
    mant = b1 & (((uint64_t)1 << 52) - 1);
    a_sign = js_bigint_sign(a);
    if (e == 2047) {
        if (mant != 0) {
            /* NaN */
            return 2;
        } else {
            /* +/- infinity */
            return 2 * b_sign - 1;
        }
    } else if (e == 0 && mant == 0) {
        /* b = +/-0 */
        if (a->len == 1 && a->tab[0] == 0)
            return 0;
        else
            return 1 - 2 * a_sign;
    } else if (a->len == 1 && a->tab[0] == 0) {
        /* a = 0, b != 0 */
        return 2 * b_sign - 1;
    } else if (a_sign != b_sign) {
        return 1 - 2 * a_sign;
    } else {
        e -= 1023;
        /* Note: handling denormals is not necessary because we
           compare to integers hence f >= 0 */
        /* compute f so that 2^f <= abs(a) < 2^(f+1) */
        a_mant = js_bigint_get_mant_exp(ctx, &f, a);
        if (f != e) {
            if (f < e)
                return -1;
            else
                return 1;
        } else {
            mant = (mant | ((uint64_t)1 << 52)) << 11; /* align to a_mant */
            if (a_mant < mant)
                return 2 * a_sign - 1;
            else if (a_mant > mant)
                return 1 - 2 * a_sign;
            else
                return 0;
        }
    }
}

/* return -1, 0 or 1 */
static int js_bigint_cmp(JSContext *ctx, const JSBigInt *a,
                         const JSBigInt *b)
{
    int a_sign, b_sign, res, i;
    a_sign = js_bigint_sign(a);
    b_sign = js_bigint_sign(b);
    if (a_sign != b_sign) {
        res = 1 - 2 * a_sign;
    } else {
        /* we assume the numbers are normalized */
        if (a->len != b->len) {
            if (a->len < b->len)
                res = 2 * a_sign - 1;
            else
                res = 1 - 2 * a_sign;
        } else {
            res = 0;
            for(i = a->len -1; i >= 0; i--) {
                if (a->tab[i] != b->tab[i]) {
                    if (a->tab[i] < b->tab[i])
                        res = -1;
                    else
                        res = 1;
                    break;
                }
            }
        }
    }
    return res;
}

/* contains 10^i */
static const js_limb_t js_pow_dec[JS_LIMB_DIGITS + 1] = {
    1U,
    10U,
    100U,
    1000U,
    10000U,
    100000U,
    1000000U,
    10000000U,
    100000000U,
    1000000000U,
};

/* syntax: [-]digits in base radix. Return NULL if memory error. radix
   = 10, 2, 8 or 16. */
static JSBigInt *js_bigint_from_string(JSContext *ctx,
                                       const char *str, int radix)
{
    const char *p = str;
    size_t n_digits1;
    int is_neg, n_digits, n_limbs, len, log2_radix, n_bits, i;
    JSBigInt *r;
    js_limb_t v, c, h;

    is_neg = 0;
    if (*p == '-') {
        is_neg = 1;
        p++;
    }
    while (*p == '0')
        p++;
    n_digits1 = strlen(p);
    /* the real check for overflox is done js_bigint_new(). Here
       we just avoid integer overflow */
    if (n_digits1 > JS_BIGINT_MAX_SIZE * JS_LIMB_BITS) {
        JS_ThrowRangeError(ctx, "BigInt is too large to allocate");
        return NULL;
    }
    n_digits = n_digits1;
    log2_radix = 32 - clz32(radix - 1); /* ceil(log2(radix)) */
    /* compute the maximum number of limbs */
    if (radix == 10) {
        n_bits = (n_digits * 27 + 7) / 8; /* >= ceil(n_digits * log2(10)) */
    } else {
        n_bits = n_digits * log2_radix;
    }
    /* we add one extra bit for the sign */
    n_limbs = max_int(1, n_bits / JS_LIMB_BITS + 1);
    r = js_bigint_new(ctx, n_limbs);
    if (!r)
        return NULL;
    if (radix == 10) {
        int digits_per_limb = JS_LIMB_DIGITS;
        len = 1;
        r->tab[0] = 0;
        for(;;) {
            /* XXX: slow */
            v = 0;
            for(i = 0; i < digits_per_limb; i++) {
                c = js_to_digit(*p);
                if (c >= radix)
                    break;
                p++;
                v = v * 10 + c;
            }
            if (i == 0)
                break;
            if (len == 1 && r->tab[0] == 0) {
                r->tab[0] = v;
            } else {
                h = js_mp_mul1(r->tab, r->tab, len, js_pow_dec[i], v);
                if (h != 0) {
                    r->tab[len++] = h;
                }
            }
        }
        /* add one extra limb to have the correct sign*/
        if ((r->tab[len - 1] >> (JS_LIMB_BITS - 1)) != 0)
            r->tab[len++] = 0;
        r->len = len;
    } else {
        unsigned int bit_pos, shift, pos;

        /* power of two base: no multiplication is needed */
        r->len = n_limbs;
        memset(r->tab, 0, sizeof(r->tab[0]) * n_limbs);
        for(i = 0; i < n_digits; i++) {
            c = js_to_digit(p[n_digits - 1 - i]);
            assert(c < radix);
            bit_pos = i * log2_radix;
            shift = bit_pos & (JS_LIMB_BITS - 1);
            pos = bit_pos / JS_LIMB_BITS;
            r->tab[pos] |= c << shift;
            /* if log2_radix does not divide JS_LIMB_BITS, needed an
               additional op */
            if (shift + log2_radix > JS_LIMB_BITS) {
                r->tab[pos + 1] |= c >> (JS_LIMB_BITS - shift);
            }
        }
    }
    r = js_bigint_normalize(ctx, r);
    /* XXX: could do it in place */
    if (is_neg) {
        JSBigInt *r1;
        r1 = js_bigint_neg(ctx, r);
        js_free(ctx, r);
        r = r1;
    }
    return r;
}

/* 2 <= base <= 36 */
static char const digits[36] = {
    '0','1','2','3','4','5','6','7','8','9',
    'a','b','c','d','e','f','g','h','i','j',
    'k','l','m','n','o','p','q','r','s','t',
    'u','v','w','x','y','z'
};

/* special version going backwards */
/* XXX: use dtoa.c */
static char *js_u64toa(char *q, int64_t n, unsigned int base)
{
    int digit;
    if (base == 10) {
        /* division by known base uses multiplication */
        do {
            digit = (uint64_t)n % 10;
            n = (uint64_t)n / 10;
            *--q = '0' + digit;
        } while (n != 0);
    } else {
        do {
            digit = (uint64_t)n % base;
            n = (uint64_t)n / base;
            *--q = digits[digit];
        } while (n != 0);
    }
    return q;
}

/* len >= 1. 2 <= radix <= 36 */
static char *js_limb_to_a(char *q, js_limb_t n, unsigned int radix, int len)
{
    int digit, i;

    if (radix == 10) {
        /* specific case with constant divisor */
        /* XXX: optimize */
        for(i = 0; i < len; i++) {
            digit = (js_limb_t)n % 10;
            n = (js_limb_t)n / 10;
            *--q = digit + '0';
        }
    } else {
        for(i = 0; i < len; i++) {
            digit = (js_limb_t)n % radix;
            n = (js_limb_t)n / radix;
            *--q = digits[digit];
        }
    }
    return q;
}

#define JS_RADIX_MAX 36

static const uint8_t js_digits_per_limb_table[JS_RADIX_MAX - 1] = {
32,20,16,13,12,11,10,10, 9, 9, 8, 8, 8, 8, 8, 7, 7, 7, 7, 7, 7, 7, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
};

static const js_limb_t js_radix_base_table[JS_RADIX_MAX - 1] = {
 0x00000000, 0xcfd41b91, 0x00000000, 0x48c27395,
 0x81bf1000, 0x75db9c97, 0x40000000, 0xcfd41b91,
 0x3b9aca00, 0x8c8b6d2b, 0x19a10000, 0x309f1021,
 0x57f6c100, 0x98c29b81, 0x00000000, 0x18754571,
 0x247dbc80, 0x3547667b, 0x4c4b4000, 0x6b5a6e1d,
 0x94ace180, 0xcaf18367, 0x0b640000, 0x0e8d4a51,
 0x1269ae40, 0x17179149, 0x1cb91000, 0x23744899,
 0x2b73a840, 0x34e63b41, 0x40000000, 0x4cfa3cc1,
 0x5c13d840, 0x6d91b519, 0x81bf1000,
};

static JSValue js_bigint_to_string1(JSContext *ctx, JSValueConst val, int radix)
{
    if (JS_VALUE_GET_TAG(val) == JS_TAG_SHORT_BIG_INT) {
        char buf[66];
        int len;
        len = i64toa_radix(buf, JS_VALUE_GET_SHORT_BIG_INT(val), radix);
        return js_new_string8_len(ctx, buf, len);
    } else {
        JSBigInt *r, *tmp = NULL;
        char *buf, *q, *buf_end;
        int is_neg, n_bits, log2_radix, n_digits;
        bool is_binary_radix;
        JSValue res;

        assert(JS_VALUE_GET_TAG(val) == JS_TAG_BIG_INT);
        r = JS_VALUE_GET_PTR(val);
        if (r->len == 1 && r->tab[0] == 0) {
            /* '0' case */
            return js_new_string8_len(ctx, "0", 1);
        }
        is_binary_radix = ((radix & (radix - 1)) == 0);
        is_neg = js_bigint_sign(r);
        if (is_neg) {
            tmp = js_bigint_neg(ctx, r);
            if (!tmp)
                return JS_EXCEPTION;
            r = tmp;
        } else if (!is_binary_radix) {
            /* need to modify 'r' */
            tmp = js_bigint_new(ctx, r->len);
            if (!tmp)
                return JS_EXCEPTION;
            memcpy(tmp->tab, r->tab, r->len * sizeof(r->tab[0]));
            r = tmp;
        }
        log2_radix = 31 - clz32(radix); /* floor(log2(radix)) */
        n_bits = r->len * JS_LIMB_BITS - js_limb_clz(r->tab[r->len - 1]);
        /* n_digits is exact only if radix is a power of
           two. Otherwise it is >= the exact number of digits */
        n_digits = (n_bits + log2_radix - 1) / log2_radix;
        /* XXX: could directly build the JSString */
        buf = js_malloc(ctx, n_digits + is_neg + 1);
        if (!buf) {
            js_free(ctx, tmp);
            return JS_EXCEPTION;
        }
        q = buf + n_digits + is_neg + 1;
        *--q = '\0';
        buf_end = q;
        if (!is_binary_radix) {
            int len;
            js_limb_t radix_base, v;
            radix_base = js_radix_base_table[radix - 2];
            len = r->len;
            for(;;) {
                /* remove leading zero limbs */
                while (len > 1 && r->tab[len - 1] == 0)
                    len--;
                if (len == 1 && r->tab[0] < radix_base) {
                    v = r->tab[0];
                    if (v != 0) {
                        q = js_u64toa(q, v, radix);
                    }
                    break;
                } else {
                    v = js_mp_div1(r->tab, r->tab, len, radix_base, 0);
                    q = js_limb_to_a(q, v, radix, js_digits_per_limb_table[radix - 2]);
                }
            }
        } else {
            int i, shift;
            unsigned int bit_pos, pos, c;

            /* radix is a power of two */
            for(i = 0; i < n_digits; i++) {
                bit_pos = i * log2_radix;
                pos = bit_pos / JS_LIMB_BITS;
                shift = bit_pos % JS_LIMB_BITS;
                c = r->tab[pos] >> shift;
                if ((shift + log2_radix) > JS_LIMB_BITS &&
                    (pos + 1) < r->len) {
                    c |= r->tab[pos + 1] << (JS_LIMB_BITS - shift);
                }
                c &= (radix - 1);
                *--q = digits[c];
            }
        }
        if (is_neg)
            *--q = '-';
        js_free(ctx, tmp);
        res = js_new_string8_len(ctx, q, buf_end - q);
        js_free(ctx, buf);
        return res;
    }
}

/* if possible transform a BigInt to short big and free it, otherwise
   return a normal bigint */
static JSValue JS_CompactBigInt(JSContext *ctx, JSBigInt *p)
{
    JSValue res;
    if (p->len == 1) {
        res = __JS_NewShortBigInt(ctx, (js_slimb_t)p->tab[0]);
        js_free(ctx, p);
        return res;
    } else {
        return JS_MKPTR(JS_TAG_BIG_INT, p);
    }
}

#define ATOD_INT_ONLY        (1 << 0)
/* accept Oo and Ob prefixes in addition to 0x prefix if radix = 0 */
#define ATOD_ACCEPT_BIN_OCT  (1 << 2)
/* accept O prefix as octal if radix == 0 and properly formed (Annex B) */
#define ATOD_ACCEPT_LEGACY_OCTAL  (1 << 4)
/* accept _ between digits as a digit separator */
#define ATOD_ACCEPT_UNDERSCORES  (1 << 5)
/* allow a suffix to override the type */
#define ATOD_ACCEPT_SUFFIX    (1 << 6)
/* default type */
#define ATOD_TYPE_MASK        (3 << 7)
#define ATOD_TYPE_FLOAT64     (0 << 7)
#define ATOD_TYPE_BIG_INT     (1 << 7)
/* accept -0x1 */
#define ATOD_ACCEPT_PREFIX_AFTER_SIGN (1 << 10)

/* return an exception in case of memory error. Return JS_NAN if
   invalid syntax */
/* XXX: directly use js_atod() */
static JSValue js_atof(JSContext *ctx, const char *str, const char **pp,
                       int radix, int flags)
{
    const char *p, *p_start;
    int sep, is_neg;
    bool is_float, has_legacy_octal;
    int atod_type = flags & ATOD_TYPE_MASK;
    char buf1[64], *buf;
    int i, j, len;
    bool buf_allocated = false;
    JSValue val;
    JSATODTempMem atod_mem;

    /* optional separator between digits */
    sep = (flags & ATOD_ACCEPT_UNDERSCORES) ? '_' : 256;
    has_legacy_octal = false;

    p = str;
    p_start = p;
    is_neg = 0;
    if (p[0] == '+') {
        p++;
        p_start++;
        if (!(flags & ATOD_ACCEPT_PREFIX_AFTER_SIGN))
            goto no_radix_prefix;
    } else if (p[0] == '-') {
        p++;
        p_start++;
        is_neg = 1;
        if (!(flags & ATOD_ACCEPT_PREFIX_AFTER_SIGN))
            goto no_radix_prefix;
    }
    if (p[0] == '0') {
        if ((p[1] == 'x' || p[1] == 'X') &&
            (radix == 0 || radix == 16)) {
            p += 2;
            radix = 16;
        } else if ((p[1] == 'o' || p[1] == 'O') &&
                   radix == 0 && (flags & ATOD_ACCEPT_BIN_OCT)) {
            p += 2;
            radix = 8;
        } else if ((p[1] == 'b' || p[1] == 'B') &&
                   radix == 0 && (flags & ATOD_ACCEPT_BIN_OCT)) {
            p += 2;
            radix = 2;
        } else if ((p[1] >= '0' && p[1] <= '9') &&
                   radix == 0 && (flags & ATOD_ACCEPT_LEGACY_OCTAL)) {
            int i;
            has_legacy_octal = true;
            sep = 256;
            for (i = 1; (p[i] >= '0' && p[i] <= '7'); i++)
                continue;
            if (p[i] == '8' || p[i] == '9')
                goto no_prefix;
            p += 1;
            radix = 8;
        } else {
            goto no_prefix;
        }
        /* there must be a digit after the prefix */
        if (js_to_digit((uint8_t)*p) >= radix)
            goto fail;
    no_prefix: ;
    } else {
 no_radix_prefix:
        if (!(flags & ATOD_INT_ONLY) &&
            (atod_type == ATOD_TYPE_FLOAT64) &&
            js__strstart(p, "Infinity", &p)) {
            double d = INF;
            if (is_neg)
                d = -d;
            val = js_float64(d);
            goto done;
        }
    }
    if (radix == 0)
        radix = 10;
    is_float = false;
    p_start = p;
    while (js_to_digit((uint8_t)*p) < radix
           ||  (*p == sep && (radix != 10 ||
                              p != p_start + 1 || p[-1] != '0') &&
                js_to_digit((uint8_t)p[1]) < radix)) {
        p++;
    }
    if (!(flags & ATOD_INT_ONLY) && radix == 10) {
        if (*p == '.' && (p > p_start || js_to_digit((uint8_t)p[1]) < radix)) {
            is_float = true;
            p++;
            if (*p == sep)
                goto fail;
            while (js_to_digit((uint8_t)*p) < radix ||
                   (*p == sep && js_to_digit((uint8_t)p[1]) < radix))
                p++;
        }
        if (p > p_start && (*p == 'e' || *p == 'E')) {
            const char *p1 = p + 1;
            is_float = true;
            if (*p1 == '+') {
                p1++;
            } else if (*p1 == '-') {
                p1++;
            }
            if (is_digit((uint8_t)*p1)) {
                p = p1 + 1;
                while (is_digit((uint8_t)*p) || (*p == sep && is_digit((uint8_t)p[1])))
                    p++;
            }
        }
    }
    if (p == p_start)
        goto fail;

    buf = buf1;
    buf_allocated = false;
    len = p - p_start;
    if (unlikely((len + 2) > sizeof(buf1))) {
        buf = js_malloc_rt(ctx->rt, len + 2); /* no exception raised */
        if (!buf)
            goto mem_error;
        buf_allocated = true;
    }
    /* remove the separators and the radix prefixes */
    j = 0;
    if (is_neg)
        buf[j++] = '-';
    for (i = 0; i < len; i++) {
        if (p_start[i] != '_')
            buf[j++] = p_start[i];
    }
    buf[j] = '\0';

    if (flags & ATOD_ACCEPT_SUFFIX) {
        if (*p == 'n') {
            p++;
            atod_type = ATOD_TYPE_BIG_INT;
        }
    }

    switch(atod_type) {
    case ATOD_TYPE_FLOAT64:
        {
            double d;
            d = js_atod(buf, NULL, radix, is_float ? 0 : JS_ATOD_INT_ONLY,
                        &atod_mem);
            /* return int or float64 */
            val = js_number(d);
        }
        break;
    case ATOD_TYPE_BIG_INT:
        {
            JSBigInt *r;
            if (has_legacy_octal || is_float)
                goto fail;
            r = js_bigint_from_string(ctx, buf, radix);
            if (!r) {
                val = JS_EXCEPTION;
                goto done;
            }
            val = JS_CompactBigInt(ctx, r);
        }
        break;
    default:
        abort();
    }

done:
    if (buf_allocated)
        js_free_rt(ctx->rt, buf);
    if (pp)
        *pp = p;
    return val;
 fail:
    val = JS_NAN;
    goto done;
 mem_error:
    val = JS_ThrowOutOfMemory(ctx);
    goto done;
}

typedef enum JSToNumberHintEnum {
    TON_FLAG_NUMBER,
    TON_FLAG_NUMERIC,
} JSToNumberHintEnum;

static JSValue JS_ToNumberHintFree(JSContext *ctx, JSValue val,
                                   JSToNumberHintEnum flag)
{
    uint32_t tag;
    JSValue ret;

 redo:
    tag = JS_VALUE_GET_NORM_TAG(val);
    switch(tag) {
    case JS_TAG_BIG_INT:
    case JS_TAG_SHORT_BIG_INT:
        if (flag != TON_FLAG_NUMERIC) {
            JS_FreeValue(ctx, val);
            return JS_ThrowTypeError(ctx, "cannot convert BigInt to number");
        }
        ret = val;
        break;
    case JS_TAG_FLOAT64:
    case JS_TAG_INT:
    case JS_TAG_EXCEPTION:
        ret = val;
        break;
    case JS_TAG_BOOL:
    case JS_TAG_NULL:
        ret = js_int32(JS_VALUE_GET_INT(val));
        break;
    case JS_TAG_UNDEFINED:
        ret = JS_NAN;
        break;
    case JS_TAG_OBJECT:
        val = JS_ToPrimitiveFree(ctx, val, HINT_NUMBER);
        if (JS_IsException(val))
            return JS_EXCEPTION;
        goto redo;
    case JS_TAG_STRING:
        {
            const char *str;
            const char *p;
            size_t len;

            str = JS_ToCStringLen(ctx, &len, val);
            JS_FreeValue(ctx, val);
            if (!str)
                return JS_EXCEPTION;
            p = str;
            p += skip_spaces(p);
            if ((p - str) == len) {
                ret = JS_NewInt32(ctx, 0);
            } else {
                int flags = ATOD_ACCEPT_BIN_OCT;
                ret = js_atof(ctx, p, &p, 0, flags);
                if (!JS_IsException(ret)) {
                    p += skip_spaces(p);
                    if ((p - str) != len) {
                        JS_FreeValue(ctx, ret);
                        ret = JS_NAN;
                    }
                }
            }
            JS_FreeCString(ctx, str);
        }
        break;
    case JS_TAG_SYMBOL:
        JS_FreeValue(ctx, val);
        return JS_ThrowTypeError(ctx, "cannot convert symbol to number");
    default:
        JS_FreeValue(ctx, val);
        ret = JS_NAN;
        break;
    }
    return ret;
}

static JSValue JS_ToNumberFree(JSContext *ctx, JSValue val)
{
    return JS_ToNumberHintFree(ctx, val, TON_FLAG_NUMBER);
}

static JSValue JS_ToNumericFree(JSContext *ctx, JSValue val)
{
    return JS_ToNumberHintFree(ctx, val, TON_FLAG_NUMERIC);
}

static JSValue JS_ToNumeric(JSContext *ctx, JSValueConst val)
{
    return JS_ToNumericFree(ctx, js_dup(val));
}

static __exception int __JS_ToFloat64Free(JSContext *ctx, double *pres,
                                          JSValue val)
{
    double d;
    uint32_t tag;

    val = JS_ToNumberFree(ctx, val);
    if (JS_IsException(val))
        goto fail;
    tag = JS_VALUE_GET_NORM_TAG(val);
    switch(tag) {
    case JS_TAG_INT:
        d = JS_VALUE_GET_INT(val);
        break;
    case JS_TAG_FLOAT64:
        d = JS_VALUE_GET_FLOAT64(val);
        break;
    default:
        abort();
    }
    *pres = d;
    return 0;
fail:
    *pres = NAN;
    return -1;
}

static inline int JS_ToFloat64Free(JSContext *ctx, double *pres, JSValue val)
{
    uint32_t tag;

    tag = JS_VALUE_GET_TAG(val);
    if (tag <= JS_TAG_NULL) {
        *pres = JS_VALUE_GET_INT(val);
        return 0;
    } else if (JS_TAG_IS_FLOAT64(tag)) {
        *pres = JS_VALUE_GET_FLOAT64(val);
        return 0;
    } else {
        return __JS_ToFloat64Free(ctx, pres, val);
    }
}

int JS_ToFloat64(JSContext *ctx, double *pres, JSValueConst val)
{
    return JS_ToFloat64Free(ctx, pres, js_dup(val));
}

JSValue JS_ToNumber(JSContext *ctx, JSValueConst val)
{
    return JS_ToNumberFree(ctx, js_dup(val));
}

/* same as JS_ToNumber() but return 0 in case of NaN/Undefined */
static __maybe_unused JSValue JS_ToIntegerFree(JSContext *ctx, JSValue val)
{
    uint32_t tag;
    JSValue ret;

 redo:
    tag = JS_VALUE_GET_NORM_TAG(val);
    switch(tag) {
    case JS_TAG_INT:
    case JS_TAG_BOOL:
    case JS_TAG_NULL:
    case JS_TAG_UNDEFINED:
        ret = js_int32(JS_VALUE_GET_INT(val));
        break;
    case JS_TAG_FLOAT64:
        {
            double d = JS_VALUE_GET_FLOAT64(val);
            if (isnan(d)) {
                ret = js_int32(0);
            } else {
                /* convert -0 to +0 */
                d = trunc(d) + 0.0;
                ret = js_number(d);
            }
        }
        break;
    default:
        val = JS_ToNumberFree(ctx, val);
        if (JS_IsException(val))
            return val;
        goto redo;
    }
    return ret;
}

/* Note: the integer value is satured to 32 bits */
static int JS_ToInt32SatFree(JSContext *ctx, int *pres, JSValue val)
{
    uint32_t tag;
    int ret;

 redo:
    tag = JS_VALUE_GET_NORM_TAG(val);
    switch(tag) {
    case JS_TAG_INT:
    case JS_TAG_BOOL:
    case JS_TAG_NULL:
    case JS_TAG_UNDEFINED:
        ret = JS_VALUE_GET_INT(val);
        break;
    case JS_TAG_EXCEPTION:
        *pres = 0;
        return -1;
    case JS_TAG_FLOAT64:
        {
            double d = JS_VALUE_GET_FLOAT64(val);
            if (isnan(d)) {
                ret = 0;
            } else {
                if (d < INT32_MIN)
                    ret = INT32_MIN;
                else if (d > INT32_MAX)
                    ret = INT32_MAX;
                else
                    ret = (int)d;
            }
        }
        break;
    default:
        val = JS_ToNumberFree(ctx, val);
        if (JS_IsException(val)) {
            *pres = 0;
            return -1;
        }
        goto redo;
    }
    *pres = ret;
    return 0;
}

static int JS_ToInt32Sat(JSContext *ctx, int *pres, JSValueConst val)
{
    return JS_ToInt32SatFree(ctx, pres, js_dup(val));
}

static int JS_ToInt32Clamp(JSContext *ctx, int *pres, JSValueConst val,
                           int min, int max, int min_offset)
{
    int res = JS_ToInt32SatFree(ctx, pres, js_dup(val));
    if (res == 0) {
        if (*pres < min) {
            *pres += min_offset;
            if (*pres < min)
                *pres = min;
        } else {
            if (*pres > max)
                *pres = max;
        }
    }
    return res;
}

static int JS_ToInt64SatFree(JSContext *ctx, int64_t *pres, JSValue val)
{
    uint32_t tag;

 redo:
    tag = JS_VALUE_GET_NORM_TAG(val);
    switch(tag) {
    case JS_TAG_INT:
    case JS_TAG_BOOL:
    case JS_TAG_NULL:
    case JS_TAG_UNDEFINED:
        *pres = JS_VALUE_GET_INT(val);
        return 0;
    case JS_TAG_EXCEPTION:
        *pres = 0;
        return -1;
    case JS_TAG_FLOAT64:
        {
            double d = JS_VALUE_GET_FLOAT64(val);
            if (isnan(d)) {
                *pres = 0;
            } else {
                if (d < INT64_MIN)
                    *pres = INT64_MIN;
                else if (d >= 0x1p63)
                    *pres = INT64_MAX;
                else
                    *pres = (int64_t)d;
            }
        }
        return 0;
    default:
        val = JS_ToNumberFree(ctx, val);
        if (JS_IsException(val)) {
            *pres = 0;
            return -1;
        }
        goto redo;
    }
}

int JS_ToInt64Sat(JSContext *ctx, int64_t *pres, JSValueConst val)
{
    return JS_ToInt64SatFree(ctx, pres, js_dup(val));
}

int JS_ToInt64Clamp(JSContext *ctx, int64_t *pres, JSValueConst val,
                    int64_t min, int64_t max, int64_t neg_offset)
{
    int res = JS_ToInt64SatFree(ctx, pres, js_dup(val));
    if (res == 0) {
        if (*pres < 0)
            *pres += neg_offset;
        if (*pres < min)
            *pres = min;
        else if (*pres > max)
            *pres = max;
    }
    return res;
}

/* Same as JS_ToInt32Free() but with a 64 bit result. Return (<0, 0)
   in case of exception */
static int JS_ToInt64Free(JSContext *ctx, int64_t *pres, JSValue val)
{
    uint32_t tag;
    int64_t ret;

 redo:
    tag = JS_VALUE_GET_NORM_TAG(val);
    switch(tag) {
    case JS_TAG_INT:
    case JS_TAG_BOOL:
    case JS_TAG_NULL:
    case JS_TAG_UNDEFINED:
        ret = JS_VALUE_GET_INT(val);
        break;
    case JS_TAG_FLOAT64:
        {
            JSFloat64Union u;
            double d;
            int e;
            d = JS_VALUE_GET_FLOAT64(val);
            u.d = d;
            /* we avoid doing fmod(x, 2^64) */
            e = (u.u64 >> 52) & 0x7ff;
            if (likely(e <= (1023 + 62))) {
                /* fast case */
                ret = (int64_t)d;
            } else if (e <= (1023 + 62 + 53)) {
                uint64_t v;
                /* remainder modulo 2^64 */
                v = (u.u64 & (((uint64_t)1 << 52) - 1)) | ((uint64_t)1 << 52);
                ret = v << ((e - 1023) - 52);
                /* take the sign into account */
                if (u.u64 >> 63)
                    if (ret != INT64_MIN)
                        ret = -ret;
            } else {
                ret = 0; /* also handles NaN and +inf */
            }
        }
        break;
    default:
        val = JS_ToNumberFree(ctx, val);
        if (JS_IsException(val)) {
            *pres = 0;
            return -1;
        }
        goto redo;
    }
    *pres = ret;
    return 0;
}

int JS_ToInt64(JSContext *ctx, int64_t *pres, JSValueConst val)
{
    return JS_ToInt64Free(ctx, pres, js_dup(val));
}

int JS_ToInt64Ext(JSContext *ctx, int64_t *pres, JSValueConst val)
{
    if (JS_IsBigInt(val))
        return JS_ToBigInt64(ctx, pres, val);
    else
        return JS_ToInt64(ctx, pres, val);
}

/* return (<0, 0) in case of exception */
static int JS_ToInt32Free(JSContext *ctx, int32_t *pres, JSValue val)
{
    uint32_t tag;
    int32_t ret;

 redo:
    tag = JS_VALUE_GET_NORM_TAG(val);
    switch(tag) {
    case JS_TAG_INT:
    case JS_TAG_BOOL:
    case JS_TAG_NULL:
    case JS_TAG_UNDEFINED:
        ret = JS_VALUE_GET_INT(val);
        break;
    case JS_TAG_FLOAT64:
        {
            JSFloat64Union u;
            double d;
            int e;
            d = JS_VALUE_GET_FLOAT64(val);
            u.d = d;
            /* we avoid doing fmod(x, 2^32) */
            e = (u.u64 >> 52) & 0x7ff;
            if (likely(e <= (1023 + 30))) {
                /* fast case */
                ret = (int32_t)d;
            } else if (e <= (1023 + 30 + 53)) {
                uint64_t v;
                /* remainder modulo 2^32 */
                v = (u.u64 & (((uint64_t)1 << 52) - 1)) | ((uint64_t)1 << 52);
                v = v << ((e - 1023) - 52 + 32);
                ret = v >> 32;
                /* take the sign into account */
                if (u.u64 >> 63)
                    if (ret != INT32_MIN)
                        ret = -ret;
            } else {
                ret = 0; /* also handles NaN and +inf */
            }
        }
        break;
    default:
        val = JS_ToNumberFree(ctx, val);
        if (JS_IsException(val)) {
            *pres = 0;
            return -1;
        }
        goto redo;
    }
    *pres = ret;
    return 0;
}

int JS_ToInt32(JSContext *ctx, int32_t *pres, JSValueConst val)
{
    return JS_ToInt32Free(ctx, pres, js_dup(val));
}

static inline int JS_ToUint32Free(JSContext *ctx, uint32_t *pres, JSValue val)
{
    return JS_ToInt32Free(ctx, (int32_t *)pres, val);
}

static int JS_ToUint8ClampFree(JSContext *ctx, int32_t *pres, JSValue val)
{
    uint32_t tag;
    int res;

 redo:
    tag = JS_VALUE_GET_NORM_TAG(val);
    switch(tag) {
    case JS_TAG_INT:
    case JS_TAG_BOOL:
    case JS_TAG_NULL:
    case JS_TAG_UNDEFINED:
        res = JS_VALUE_GET_INT(val);
        res = max_int(0, min_int(255, res));
        break;
    case JS_TAG_FLOAT64:
        {
            double d = JS_VALUE_GET_FLOAT64(val);
            if (isnan(d)) {
                res = 0;
            } else {
                if (d < 0)
                    res = 0;
                else if (d > 255)
                    res = 255;
                else
                    res = lrint(d);
            }
        }
        break;
    default:
        val = JS_ToNumberFree(ctx, val);
        if (JS_IsException(val)) {
            *pres = 0;
            return -1;
        }
        goto redo;
    }
    *pres = res;
    return 0;
}

static __exception int JS_ToArrayLengthFree(JSContext *ctx, uint32_t *plen,
                                            JSValue val, bool is_array_ctor)
{
    uint32_t tag, len;

    tag = JS_VALUE_GET_TAG(val);
    switch(tag) {
    case JS_TAG_INT:
    case JS_TAG_BOOL:
    case JS_TAG_NULL:
        {
            int v;
            v = JS_VALUE_GET_INT(val);
            if (v < 0)
                goto fail;
            len = v;
        }
        break;
    default:
        if (JS_TAG_IS_FLOAT64(tag)) {
            double d;
            d = JS_VALUE_GET_FLOAT64(val);
            if (!(d >= 0 && d <= UINT32_MAX))
                goto fail;
            len = (uint32_t)d;
            if (len != d)
                goto fail;
        } else {
            uint32_t len1;

            if (is_array_ctor) {
                val = JS_ToNumberFree(ctx, val);
                if (JS_IsException(val))
                    return -1;
                /* cannot recurse because val is a number */
                if (JS_ToArrayLengthFree(ctx, &len, val, true))
                    return -1;
            } else {
                /* legacy behavior: must do the conversion twice and compare */
                if (JS_ToUint32(ctx, &len, val)) {
                    JS_FreeValue(ctx, val);
                    return -1;
                }
                val = JS_ToNumberFree(ctx, val);
                if (JS_IsException(val))
                    return -1;
                /* cannot recurse because val is a number */
                if (JS_ToArrayLengthFree(ctx, &len1, val, false))
                    return -1;
                if (len1 != len) {
                fail:
                    JS_ThrowRangeError(ctx, "invalid array length");
                    return -1;
                }
            }
        }
        break;
    }
    *plen = len;
    return 0;
}

#define MAX_SAFE_INTEGER (((int64_t)1 << 53) - 1)

static bool is_safe_integer(double d)
{
    return isfinite(d) && floor(d) == d &&
        fabs(d) <= (double)MAX_SAFE_INTEGER;
}

int JS_ToIndex(JSContext *ctx, uint64_t *plen, JSValueConst val)
{
    int64_t v;
    if (JS_ToInt64Sat(ctx, &v, val))
        return -1;
    if (v < 0 || v > MAX_SAFE_INTEGER) {
        JS_ThrowRangeError(ctx, "invalid array index");
        *plen = 0;
        return -1;
    }
    *plen = v;
    return 0;
}

/* convert a value to a length between 0 and MAX_SAFE_INTEGER.
   return -1 for exception */
static __exception int JS_ToLengthFree(JSContext *ctx, int64_t *plen,
                                       JSValue val)
{
    int res = JS_ToInt64Clamp(ctx, plen, val, 0, MAX_SAFE_INTEGER, 0);
    JS_FreeValue(ctx, val);
    return res;
}

/* Note: can return an exception */
static int JS_NumberIsInteger(JSContext *ctx, JSValueConst val)
{
    double d;
    if (!JS_IsNumber(val))
        return false;
    if (unlikely(JS_ToFloat64(ctx, &d, val)))
        return -1;
    return isfinite(d) && floor(d) == d;
}

static bool JS_NumberIsNegativeOrMinusZero(JSContext *ctx, JSValueConst val)
{
    uint32_t tag;

    tag = JS_VALUE_GET_NORM_TAG(val);
    switch(tag) {
    case JS_TAG_INT:
        {
            int v;
            v = JS_VALUE_GET_INT(val);
            return (v < 0);
        }
    case JS_TAG_FLOAT64:
        {
            JSFloat64Union u;
            u.d = JS_VALUE_GET_FLOAT64(val);
            return (u.u64 >> 63);
        }
    case JS_TAG_SHORT_BIG_INT:
        return (JS_VALUE_GET_SHORT_BIG_INT(val) < 0);
    case JS_TAG_BIG_INT:
        {
            JSBigInt *p = JS_VALUE_GET_PTR(val);
            return js_bigint_sign(p);
        }
    default:
        return false;
    }
}

static JSValue js_bigint_to_string(JSContext *ctx, JSValueConst val)
{
    return js_bigint_to_string1(ctx, val, 10);
}

/*---- floating point number to string conversions ----*/

static JSValue js_dtoa2(JSContext *ctx,
                        double d, int radix, int n_digits, int flags)
{
    char static_buf[128], *buf, *tmp_buf;
    int len, len_max;
    JSValue res;
    JSDTOATempMem dtoa_mem;
    len_max = js_dtoa_max_len(d, radix, n_digits, flags);

    /* longer buffer may be used if radix != 10 */
    if (len_max > sizeof(static_buf) - 1) {
        tmp_buf = js_malloc(ctx, len_max + 1);
        if (!tmp_buf)
            return JS_EXCEPTION;
        buf = tmp_buf;
    } else {
        tmp_buf = NULL;
        buf = static_buf;
    }
    len = js_dtoa(buf, d, radix, n_digits, flags, &dtoa_mem);
    res = js_new_string8_len(ctx, buf, len);
    js_free(ctx, tmp_buf);
    return res;
}

static JSValue JS_ToStringInternal(JSContext *ctx, JSValueConst val,
                                   int flags)
{
    uint32_t tag;
    char buf[32];
    size_t len;

    tag = JS_VALUE_GET_NORM_TAG(val);
    switch(tag) {
    case JS_TAG_STRING:
        return js_dup(val);
    case JS_TAG_INT:
        len = i32toa(buf, JS_VALUE_GET_INT(val));
        return js_new_string8_len(ctx, buf, len);
    case JS_TAG_BOOL:
        return JS_AtomToString(ctx, JS_VALUE_GET_BOOL(val) ?
                          JS_ATOM_true : JS_ATOM_false);
    case JS_TAG_NULL:
        return JS_AtomToString(ctx, JS_ATOM_null);
    case JS_TAG_UNDEFINED:
        return JS_AtomToString(ctx, JS_ATOM_undefined);
    case JS_TAG_EXCEPTION:
        return JS_EXCEPTION;
    case JS_TAG_OBJECT:
        if (flags & JS_TO_STRING_NO_SIDE_EFFECTS) {
            return js_new_string8(ctx, "{}");
        } else {
            JSValue val1, ret;
            val1 = JS_ToPrimitive(ctx, val, HINT_STRING);
            if (JS_IsException(val1))
                return val1;
            ret = JS_ToStringInternal(ctx, val1, flags);
            JS_FreeValue(ctx, val1);
            return ret;
        }
        break;
    case JS_TAG_FUNCTION_BYTECODE:
        return js_new_string8(ctx, "[function bytecode]");
    case JS_TAG_SYMBOL:
        if (flags & JS_TO_STRING_IS_PROPERTY_KEY) {
            return js_dup(val);
        } else {
            return JS_ThrowTypeError(ctx, "cannot convert symbol to string");
        }
    case JS_TAG_FLOAT64:
        return js_dtoa2(ctx, JS_VALUE_GET_FLOAT64(val), 10, 0,
                        JS_DTOA_FORMAT_FREE);
    case JS_TAG_SHORT_BIG_INT:
    case JS_TAG_BIG_INT:
        return js_bigint_to_string(ctx, val);
    case JS_TAG_UNINITIALIZED:
        return js_new_string8(ctx, "[uninitialized]");
    default:
        return js_new_string8(ctx, "[unsupported type]");
    }
}

JSValue JS_ToString(JSContext *ctx, JSValueConst val)
{
    return JS_ToStringInternal(ctx, val, /*flags*/0);
}

static JSValue JS_ToStringFree(JSContext *ctx, JSValue val)
{
    JSValue ret;
    ret = JS_ToString(ctx, val);
    JS_FreeValue(ctx, val);
    return ret;
}

static JSValue JS_ToLocaleStringFree(JSContext *ctx, JSValue val)
{
    if (JS_IsUndefined(val) || JS_IsNull(val))
        return JS_ToStringFree(ctx, val);
    return JS_InvokeFree(ctx, val, JS_ATOM_toLocaleString, 0, NULL);
}

static JSValue JS_ToPropertyKeyInternal(JSContext *ctx, JSValueConst val,
                                        int flags)
{
    return JS_ToStringInternal(ctx, val, flags | JS_TO_STRING_IS_PROPERTY_KEY);
}

JSValue JS_ToPropertyKey(JSContext *ctx, JSValueConst val)
{
    return JS_ToPropertyKeyInternal(ctx, val, /*flags*/0);
}

static JSValue JS_ToStringCheckObject(JSContext *ctx, JSValueConst val)
{
    uint32_t tag = JS_VALUE_GET_TAG(val);
    if (tag == JS_TAG_NULL || tag == JS_TAG_UNDEFINED)
        return JS_ThrowTypeError(ctx, "null or undefined are forbidden");
    return JS_ToString(ctx, val);
}

static JSValue JS_ToQuotedString(JSContext *ctx, JSValueConst val1)
{
    JSValue val;
    JSString *p;
    int i;
    uint32_t c;
    StringBuffer b_s, *b = &b_s;
    char buf[16];

    val = JS_ToStringCheckObject(ctx, val1);
    if (JS_IsException(val))
        return val;
    p = JS_VALUE_GET_STRING(val);

    if (string_buffer_init(ctx, b, p->len + 2))
        goto fail;

    if (string_buffer_putc8(b, '\"'))
        goto fail;
    for(i = 0; i < p->len; ) {
        c = string_getc(p, &i);
        switch(c) {
        case '\t':
            c = 't';
            goto quote;
        case '\r':
            c = 'r';
            goto quote;
        case '\n':
            c = 'n';
            goto quote;
        case '\b':
            c = 'b';
            goto quote;
        case '\f':
            c = 'f';
            goto quote;
        case '\"':
        case '\\':
        quote:
            if (string_buffer_putc8(b, '\\'))
                goto fail;
            if (string_buffer_putc8(b, c))
                goto fail;
            break;
        default:
            if (c < 32 || is_surrogate(c)) {
                snprintf(buf, sizeof(buf), "\\u%04x", c);
                if (string_buffer_write8(b, (uint8_t*)buf, 6))
                    goto fail;
            } else {
                if (string_buffer_putc(b, c))
                    goto fail;
            }
            break;
        }
    }
    if (string_buffer_putc8(b, '\"'))
        goto fail;
    JS_FreeValue(ctx, val);
    return string_buffer_end(b);
 fail:
    JS_FreeValue(ctx, val);
    string_buffer_free(b);
    return JS_EXCEPTION;
}

static __maybe_unused void JS_DumpObjectHeader(JSRuntime *rt)
{
    printf("%14s %4s %4s %14s %10s %s\n",
           "ADDRESS", "REFS", "SHRF", "PROTO", "CLASS", "PROPS");
}

/* for debug only: dump an object without side effect */
static __maybe_unused void JS_DumpObject(JSRuntime *rt, JSObject *p)
{
    uint32_t i;
    char atom_buf[ATOM_GET_STR_BUF_SIZE];
    JSShape *sh;
    JSShapeProperty *prs;
    JSProperty *pr;
    bool is_first = true;

    /* XXX: should encode atoms with special characters */
    sh = p->shape; /* the shape can be NULL while freeing an object */
    printf("%14p %4d ",
           (void *)p,
           p->header.ref_count);
    if (sh) {
        printf("%3d%c %14p ",
               sh->header.ref_count,
               " *"[sh->is_hashed],
               (void *)sh->proto);
    } else {
        printf("%3s  %14s ", "-", "-");
    }
    printf("%10s ",
           JS_AtomGetStrRT(rt, atom_buf, sizeof(atom_buf), rt->class_array[p->class_id].class_name));
    if (p->is_exotic && p->fast_array) {
        printf("[ ");
        for(i = 0; i < p->u.array.count; i++) {
            if (i != 0)
                printf(", ");
            switch (p->class_id) {
            case JS_CLASS_ARRAY:
            case JS_CLASS_ARGUMENTS:
                JS_DumpValue(rt, p->u.array.u.values[i]);
                break;
            case JS_CLASS_UINT8C_ARRAY:
            case JS_CLASS_INT8_ARRAY:
            case JS_CLASS_UINT8_ARRAY:
            case JS_CLASS_INT16_ARRAY:
            case JS_CLASS_UINT16_ARRAY:
            case JS_CLASS_INT32_ARRAY:
            case JS_CLASS_UINT32_ARRAY:
            case JS_CLASS_BIG_INT64_ARRAY:
            case JS_CLASS_BIG_UINT64_ARRAY:
            case JS_CLASS_FLOAT16_ARRAY:
            case JS_CLASS_FLOAT32_ARRAY:
            case JS_CLASS_FLOAT64_ARRAY:
                {
                    int size = 1 << typed_array_size_log2(p->class_id);
                    const uint8_t *b = p->u.array.u.uint8_ptr + i * size;
                    while (size-- > 0)
                        printf("%02X", *b++);
                }
                break;
            }
        }
        printf(" ] ");
    }

    if (sh) {
        printf("{ ");
        for(i = 0, prs = get_shape_prop(sh); i < sh->prop_count; i++, prs++) {
            if (prs->atom != JS_ATOM_NULL) {
                pr = &p->prop[i];
                if (!is_first)
                    printf(", ");
                printf("%s: ",
                       JS_AtomGetStrRT(rt, atom_buf, sizeof(atom_buf), prs->atom));
                if ((prs->flags & JS_PROP_TMASK) == JS_PROP_GETSET) {
                    printf("[getset %p %p]", (void *)pr->u.getset.getter,
                           (void *)pr->u.getset.setter);
                } else if ((prs->flags & JS_PROP_TMASK) == JS_PROP_VARREF) {
                    printf("[varref %p]", (void *)pr->u.var_ref);
                } else if ((prs->flags & JS_PROP_TMASK) == JS_PROP_AUTOINIT) {
                    printf("[autoinit %p %d %p]",
                           (void *)js_autoinit_get_realm(pr),
                           js_autoinit_get_id(pr),
                           (void *)pr->u.init.opaque);
                } else {
                    JS_DumpValue(rt, pr->u.value);
                }
                is_first = false;
            }
        }
        printf(" }");
    }

    if (js_class_has_bytecode(p->class_id)) {
        JSFunctionBytecode *b = p->u.func.function_bytecode;
        JSVarRef **var_refs;
        if (b->closure_var_count) {
            var_refs = p->u.func.var_refs;
            printf(" Closure:");
            for(i = 0; i < b->closure_var_count; i++) {
                printf(" ");
                JS_DumpValue(rt, var_refs[i]->value);
            }
            if (p->u.func.home_object) {
                printf(" HomeObject: ");
                JS_DumpValue(rt, JS_MKPTR(JS_TAG_OBJECT, p->u.func.home_object));
            }
        }
    }
    printf("\n");
}

static __maybe_unused void JS_DumpGCObject(JSRuntime *rt, JSGCObjectHeader *p)
{
    if (p->gc_obj_type == JS_GC_OBJ_TYPE_JS_OBJECT) {
        JS_DumpObject(rt, (JSObject *)p);
    } else {
        printf("%14p %4d ",
               (void *)p,
               p->ref_count);
        switch(p->gc_obj_type) {
        case JS_GC_OBJ_TYPE_FUNCTION_BYTECODE:
            printf("[function bytecode]");
            break;
        case JS_GC_OBJ_TYPE_SHAPE:
            printf("[shape]");
            break;
        case JS_GC_OBJ_TYPE_VAR_REF:
            printf("[var_ref]");
            break;
        case JS_GC_OBJ_TYPE_ASYNC_FUNCTION:
            printf("[async_function]");
            break;
        case JS_GC_OBJ_TYPE_JS_CONTEXT:
            printf("[js_context]");
            break;
        default:
            printf("[unknown %d]", p->gc_obj_type);
            break;
        }
        printf("\n");
    }
}

static __maybe_unused void JS_DumpValue(JSRuntime *rt, JSValueConst val)
{
    uint32_t tag = JS_VALUE_GET_NORM_TAG(val);
    const char *str;

    switch(tag) {
    case JS_TAG_INT:
        printf("%d", JS_VALUE_GET_INT(val));
        break;
    case JS_TAG_BOOL:
        if (JS_VALUE_GET_BOOL(val))
            str = "true";
        else
            str = "false";
        goto print_str;
    case JS_TAG_NULL:
        str = "null";
        goto print_str;
    case JS_TAG_EXCEPTION:
        str = "exception";
        goto print_str;
    case JS_TAG_UNINITIALIZED:
        str = "uninitialized";
        goto print_str;
    case JS_TAG_UNDEFINED:
        str = "undefined";
    print_str:
        printf("%s", str);
        break;
    case JS_TAG_FLOAT64:
        printf("%.14g", JS_VALUE_GET_FLOAT64(val));
        break;
    case JS_TAG_SHORT_BIG_INT:
        printf("%" PRId64 "n", (int64_t)JS_VALUE_GET_SHORT_BIG_INT(val));
        break;
    case JS_TAG_BIG_INT:
        {
            JSBigInt *p = JS_VALUE_GET_PTR(val);
            int sgn, i;
            /* In order to avoid allocations we just dump the limbs */
            sgn = js_bigint_sign(p);
            if (sgn)
                printf("BigInt.asIntN(%d,", p->len * JS_LIMB_BITS);
            printf("0x");
            for(i = p->len - 1; i >= 0; i--) {
                if (i != p->len - 1)
                    printf("_");
                printf("%08x", p->tab[i]);
            }
            printf("n");
            if (sgn)
                printf(")");
        }
        break;
    case JS_TAG_STRING:
        {
            JSString *p;
            p = JS_VALUE_GET_STRING(val);
            JS_DumpString(rt, p);
        }
        break;
    case JS_TAG_FUNCTION_BYTECODE:
        {
            JSFunctionBytecode *b = JS_VALUE_GET_PTR(val);
            char buf[ATOM_GET_STR_BUF_SIZE];
            if (b->func_name) {
                printf("[bytecode %s]", JS_AtomGetStrRT(rt, buf, sizeof(buf), b->func_name));
            } else {
                printf("[bytecode (anonymous)]");
            }
        }
        break;
    case JS_TAG_OBJECT:
        {
            JSObject *p = JS_VALUE_GET_OBJ(val);
            JSAtom atom = rt->class_array[p->class_id].class_name;
            char atom_buf[ATOM_GET_STR_BUF_SIZE];
            printf("[%s %p]",
                   JS_AtomGetStrRT(rt, atom_buf, sizeof(atom_buf), atom), (void *)p);
        }
        break;
    case JS_TAG_SYMBOL:
        {
            JSAtomStruct *p = JS_VALUE_GET_PTR(val);
            char atom_buf[ATOM_GET_STR_BUF_SIZE];
            printf("Symbol(%s)",
                   JS_AtomGetStrRT(rt, atom_buf, sizeof(atom_buf), js_get_atom_index(rt, p)));
        }
        break;
    case JS_TAG_MODULE:
        printf("[module]");
        break;
    default:
        printf("[unknown tag %d]", tag);
        break;
    }
}

bool JS_IsArray(JSValueConst val)
{
    if (JS_VALUE_GET_TAG(val) == JS_TAG_OBJECT) {
        JSObject *p = JS_VALUE_GET_OBJ(val);
        return p->class_id == JS_CLASS_ARRAY;
    }
    return false;
}

/* return -1 if exception (proxy case) or true/false */
static int js_is_array(JSContext *ctx, JSValueConst val)
{
    JSObject *p;
    if (JS_VALUE_GET_TAG(val) == JS_TAG_OBJECT) {
        p = JS_VALUE_GET_OBJ(val);
        if (unlikely(p->class_id == JS_CLASS_PROXY))
            return js_proxy_isArray(ctx, val);
        else
            return p->class_id == JS_CLASS_ARRAY;
    } else {
        return false;
    }
}

static double js_math_pow(double a, double b)
{
    if (unlikely(!isfinite(b)) && fabs(a) == 1) {
        /* not compatible with IEEE 754 */
        return NAN;
    } else {
        return pow(a, b);
    }
}

JSValue JS_NewBigInt64(JSContext *ctx, int64_t v)
{
    if (v >= JS_SHORT_BIG_INT_MIN && v <= JS_SHORT_BIG_INT_MAX) {
        return __JS_NewShortBigInt(ctx, v);
    } else {
        JSBigInt *p;
        p = js_bigint_new_si64(ctx, v);
        if (!p)
            return JS_EXCEPTION;
        return JS_MKPTR(JS_TAG_BIG_INT, p);
    }
}

JSValue JS_NewBigUint64(JSContext *ctx, uint64_t v)
{
    if (v <= JS_SHORT_BIG_INT_MAX) {
        return __JS_NewShortBigInt(ctx, v);
    } else {
        JSBigInt *p;
        p = js_bigint_new_ui64(ctx, v);
        if (!p)
            return JS_EXCEPTION;
        return JS_MKPTR(JS_TAG_BIG_INT, p);
    }
}

/* return NaN if bad bigint literal */
static JSValue JS_StringToBigInt(JSContext *ctx, JSValue val)
{
    const char *str, *p;
    size_t len;
    int flags;

    str = JS_ToCStringLen(ctx, &len, val);
    JS_FreeValue(ctx, val);
    if (!str)
        return JS_EXCEPTION;
    p = str;
    p += skip_spaces(p);
    if ((p - str) == len) {
        val = JS_NewBigInt64(ctx, 0);
    } else {
        flags = ATOD_INT_ONLY | ATOD_ACCEPT_BIN_OCT | ATOD_TYPE_BIG_INT;
        val = js_atof(ctx, p, &p, 0, flags);
        p += skip_spaces(p);
        if (!JS_IsException(val)) {
            if ((p - str) != len) {
                JS_FreeValue(ctx, val);
                val = JS_NAN;
            }
        }
    }
    JS_FreeCString(ctx, str);
    return val;
}

static JSValue JS_StringToBigIntErr(JSContext *ctx, JSValue val)
{
    val = JS_StringToBigInt(ctx, val);
    if (JS_VALUE_IS_NAN(val))
        return JS_ThrowSyntaxError(ctx, "invalid BigInt literal");
    return val;
}

/* JS Numbers are not allowed */
static JSValue JS_ToBigIntFree(JSContext *ctx, JSValue val)
{
    uint32_t tag;

 redo:
    tag = JS_VALUE_GET_NORM_TAG(val);
    switch(tag) {
    case JS_TAG_SHORT_BIG_INT:
    case JS_TAG_BIG_INT:
        break;
    case JS_TAG_INT:
    case JS_TAG_NULL:
    case JS_TAG_UNDEFINED:
    case JS_TAG_FLOAT64:
        goto fail;
    case JS_TAG_BOOL:
        val = __JS_NewShortBigInt(ctx, JS_VALUE_GET_INT(val));
        break;
    case JS_TAG_STRING:
        val = JS_StringToBigIntErr(ctx, val);
        if (JS_IsException(val))
            return val;
        goto redo;
    case JS_TAG_OBJECT:
        val = JS_ToPrimitiveFree(ctx, val, HINT_NUMBER);
        if (JS_IsException(val))
            return val;
        goto redo;
    default:
    fail:
        JS_FreeValue(ctx, val);
        return JS_ThrowTypeError(ctx, "cannot convert to bigint");
    }
    return val;
}

static JSValue JS_ToBigInt(JSContext *ctx, JSValueConst val)
{
    return JS_ToBigIntFree(ctx, js_dup(val));
}

/* XXX: merge with JS_ToInt64Free with a specific flag */
static int JS_ToBigInt64Free(JSContext *ctx, int64_t *pres, JSValue val)
{
    uint64_t res;

    val = JS_ToBigIntFree(ctx, val);
    if (JS_IsException(val)) {
        *pres = 0;
        return -1;
    }
    if (JS_VALUE_GET_TAG(val) == JS_TAG_SHORT_BIG_INT) {
        res = JS_VALUE_GET_SHORT_BIG_INT(val);
    } else {
        JSBigInt *p = JS_VALUE_GET_PTR(val);
        /* return the value mod 2^64 */
        res = p->tab[0];
        if (p->len >= 2)
            res |= (uint64_t)p->tab[1] << 32;
        JS_FreeValue(ctx, val);
    }
    *pres = res;
    return 0;
}

int JS_ToBigInt64(JSContext *ctx, int64_t *pres, JSValueConst val)
{
    return JS_ToBigInt64Free(ctx, pres, js_dup(val));
}

int JS_ToBigUint64(JSContext *ctx, uint64_t *pres, JSValueConst val)
{
    return JS_ToBigInt64Free(ctx, (int64_t *)pres, js_dup(val));
}

static no_inline __exception int js_unary_arith_slow(JSContext *ctx,
                                                     JSValue *sp,
                                                     OPCodeEnum op)
{
    JSValue op1;
    int v;
    uint32_t tag;
    JSBigIntBuf buf1;
    JSBigInt *p1;

    op1 = sp[-1];
    /* fast path for float64 */
    if (JS_TAG_IS_FLOAT64(JS_VALUE_GET_TAG(op1)))
        goto handle_float64;
    op1 = JS_ToNumericFree(ctx, op1);
    if (JS_IsException(op1))
        goto exception;
    tag = JS_VALUE_GET_TAG(op1);
    switch(tag) {
    case JS_TAG_INT:
        {
            int64_t v64;
            v64 = JS_VALUE_GET_INT(op1);
            switch(op) {
            case OP_inc:
            case OP_dec:
                v = 2 * (op - OP_dec) - 1;
                v64 += v;
                break;
            case OP_plus:
                break;
            case OP_neg:
                if (v64 == 0) {
                    sp[-1] = js_float64(-0.0);
                    return 0;
                } else {
                    v64 = -v64;
                }
                break;
            default:
                abort();
            }
            sp[-1] = js_int64(v64);
        }
        break;
    case JS_TAG_SHORT_BIG_INT:
        {
            int64_t v;
            v = JS_VALUE_GET_SHORT_BIG_INT(op1);
            switch(op) {
            case OP_plus:
                JS_ThrowTypeError(ctx, "bigint argument with unary +");
                goto exception;
            case OP_inc:
                if (v == JS_SHORT_BIG_INT_MAX)
                    goto bigint_slow_case;
                sp[-1] = __JS_NewShortBigInt(ctx, v + 1);
                break;
            case OP_dec:
                if (v == JS_SHORT_BIG_INT_MIN)
                    goto bigint_slow_case;
                sp[-1] = __JS_NewShortBigInt(ctx, v - 1);
                break;
            case OP_neg:
                v = JS_VALUE_GET_SHORT_BIG_INT(op1);
                if (v == JS_SHORT_BIG_INT_MIN) {
                bigint_slow_case:
                    p1 = js_bigint_set_short(&buf1, op1);
                    goto bigint_slow_case1;
                }
                sp[-1] = __JS_NewShortBigInt(ctx, -v);
                break;
            default:
                abort();
            }
        }
        break;
    case JS_TAG_BIG_INT:
        {
            JSBigInt *r;
            p1 = JS_VALUE_GET_PTR(op1);
        bigint_slow_case1:
            switch(op) {
            case OP_plus:
                JS_ThrowTypeError(ctx, "bigint argument with unary +");
                JS_FreeValue(ctx, op1);
                goto exception;
            case OP_inc:
            case OP_dec:
                {
                    JSBigIntBuf buf2;
                    JSBigInt *p2;
                    p2 = js_bigint_set_si(&buf2, 2 * (op - OP_dec) - 1);
                    r = js_bigint_add(ctx, p1, p2, 0);
                }
                break;
            case OP_neg:
                r = js_bigint_neg(ctx, p1);
                break;
            case OP_not:
                r = js_bigint_not(ctx, p1);
                break;
            default:
                abort();
            }
            JS_FreeValue(ctx, op1);
            if (!r)
                goto exception;
            sp[-1] = JS_CompactBigInt(ctx, r);
        }
        break;
    default:
    handle_float64:
        {
            double d;
            d = JS_VALUE_GET_FLOAT64(op1);
            switch(op) {
            case OP_inc:
            case OP_dec:
                v = 2 * (op - OP_dec) - 1;
                d += v;
                break;
            case OP_plus:
                break;
            case OP_neg:
                d = -d;
                break;
            default:
                abort();
            }
            sp[-1] = js_float64(d);
        }
        break;
    }
    return 0;
 exception:
    sp[-1] = JS_UNDEFINED;
    return -1;
}

static __exception int js_post_inc_slow(JSContext *ctx,
                                        JSValue *sp, OPCodeEnum op)
{
    JSValue op1;

    /* XXX: allow custom operators */
    op1 = sp[-1];
    op1 = JS_ToNumericFree(ctx, op1);
    if (JS_IsException(op1)) {
        sp[-1] = JS_UNDEFINED;
        return -1;
    }
    sp[-1] = op1;
    sp[0] = js_dup(op1);
    return js_unary_arith_slow(ctx, sp + 1, op - OP_post_dec + OP_dec);
}

static no_inline int js_not_slow(JSContext *ctx, JSValue *sp)
{
    JSValue op1;

    op1 = sp[-1];
    op1 = JS_ToNumericFree(ctx, op1);
    if (JS_IsException(op1))
        goto exception;
    if (JS_VALUE_GET_TAG(op1) == JS_TAG_SHORT_BIG_INT) {
        sp[-1] = __JS_NewShortBigInt(ctx, ~JS_VALUE_GET_SHORT_BIG_INT(op1));
    } else if (JS_VALUE_GET_TAG(op1) == JS_TAG_BIG_INT) {
        JSBigInt *r;
        r = js_bigint_not(ctx, JS_VALUE_GET_PTR(op1));
        JS_FreeValue(ctx, op1);
        if (!r)
            goto exception;
        sp[-1] = JS_CompactBigInt(ctx, r);
    } else {
        int32_t v1;
        if (unlikely(JS_ToInt32Free(ctx, &v1, op1)))
            goto exception;
        sp[-1] = js_int32(~v1);
    }
    return 0;
 exception:
    sp[-1] = JS_UNDEFINED;
    return -1;
}

static no_inline __exception int js_binary_arith_slow(JSContext *ctx, JSValue *sp,
                                                      OPCodeEnum op)
{
    JSValue op1, op2;
    uint32_t tag1, tag2;
    double d1, d2;

    op1 = sp[-2];
    op2 = sp[-1];
    tag1 = JS_VALUE_GET_NORM_TAG(op1);
    tag2 = JS_VALUE_GET_NORM_TAG(op2);
    /* fast path for float operations */
    if (tag1 == JS_TAG_FLOAT64 && tag2 == JS_TAG_FLOAT64) {
        d1 = JS_VALUE_GET_FLOAT64(op1);
        d2 = JS_VALUE_GET_FLOAT64(op2);
        goto handle_float64;
    }
    /* fast path for short big int operations */
    if (tag1 == JS_TAG_SHORT_BIG_INT && tag2 == JS_TAG_SHORT_BIG_INT) {
        js_slimb_t v1, v2;
        js_sdlimb_t v;
        v1 = JS_VALUE_GET_SHORT_BIG_INT(op1);
        v2 = JS_VALUE_GET_SHORT_BIG_INT(op2);
        switch(op) {
        case OP_sub:
            v = (js_sdlimb_t)v1 - (js_sdlimb_t)v2;
            break;
        case OP_mul:
            v = (js_sdlimb_t)v1 * (js_sdlimb_t)v2;
            break;
        case OP_div:
            if (v2 == 0 ||
                ((js_limb_t)v1 == (js_limb_t)1 << (JS_LIMB_BITS - 1) &&
                 v2 == -1)) {
                goto slow_big_int;
            }
            sp[-2] = __JS_NewShortBigInt(ctx, v1 / v2);
            return 0;
        case OP_mod:
            if (v2 == 0 ||
                ((js_limb_t)v1 == (js_limb_t)1 << (JS_LIMB_BITS - 1) &&
                 v2 == -1)) {
                goto slow_big_int;
            }
            sp[-2] = __JS_NewShortBigInt(ctx, v1 % v2);
            return 0;
        case OP_pow:
            goto slow_big_int;
        default:
            abort();
        }
        if (likely(v >= JS_SHORT_BIG_INT_MIN && v <= JS_SHORT_BIG_INT_MAX)) {
            sp[-2] = __JS_NewShortBigInt(ctx, v);
        } else {
            JSBigInt *r = js_bigint_new_di(ctx, v);
            if (!r)
                goto exception;
            sp[-2] = JS_MKPTR(JS_TAG_BIG_INT, r);
        }
        return 0;
    }
    op1 = JS_ToNumericFree(ctx, op1);
    if (JS_IsException(op1)) {
        JS_FreeValue(ctx, op2);
        goto exception;
    }
    op2 = JS_ToNumericFree(ctx, op2);
    if (JS_IsException(op2)) {
        JS_FreeValue(ctx, op1);
        goto exception;
    }
    tag1 = JS_VALUE_GET_NORM_TAG(op1);
    tag2 = JS_VALUE_GET_NORM_TAG(op2);

    if (tag1 == JS_TAG_INT && tag2 == JS_TAG_INT) {
        int32_t v1, v2;
        int64_t v;
        v1 = JS_VALUE_GET_INT(op1);
        v2 = JS_VALUE_GET_INT(op2);
        switch(op) {
        case OP_sub:
            v = (int64_t)v1 - (int64_t)v2;
            break;
        case OP_mul:
            v = (int64_t)v1 * (int64_t)v2;
            if (v == 0 && (v1 | v2) < 0) {
                sp[-2] = js_float64(-0.0);
                return 0;
            }
            break;
        case OP_div:
            sp[-2] = js_number((double)v1 / (double)v2);
            return 0;
        case OP_mod:
            if (v1 < 0 || v2 <= 0) {
                sp[-2] = js_number(fmod(v1, v2));
                return 0;
            } else {
                v = (int64_t)v1 % (int64_t)v2;
            }
            break;
        case OP_pow:
            sp[-2] = js_number(js_math_pow(v1, v2));
            return 0;
        default:
            abort();
        }
        sp[-2] = js_int64(v);
    } else if ((tag1 == JS_TAG_SHORT_BIG_INT || tag1 == JS_TAG_BIG_INT) &&
               (tag2 == JS_TAG_SHORT_BIG_INT || tag2 == JS_TAG_BIG_INT)) {
        JSBigInt *p1, *p2, *r;
        JSBigIntBuf buf1, buf2;
    slow_big_int:
        /* bigint result */
        if (JS_VALUE_GET_TAG(op1) == JS_TAG_SHORT_BIG_INT)
            p1 = js_bigint_set_short(&buf1, op1);
        else
            p1 = JS_VALUE_GET_PTR(op1);
        if (JS_VALUE_GET_TAG(op2) == JS_TAG_SHORT_BIG_INT)
            p2 = js_bigint_set_short(&buf2, op2);
        else
            p2 = JS_VALUE_GET_PTR(op2);
        switch(op) {
        case OP_add:
            r = js_bigint_add(ctx, p1, p2, 0);
            break;
        case OP_sub:
            r = js_bigint_add(ctx, p1, p2, 1);
            break;
        case OP_mul:
            r = js_bigint_mul(ctx, p1, p2);
            break;
        case OP_div:
            r = js_bigint_divrem(ctx, p1, p2, false);
            break;
        case OP_mod:
            r = js_bigint_divrem(ctx, p1, p2, true);
            break;
        case OP_pow:
            r = js_bigint_pow(ctx, p1, p2);
            break;
        default:
            abort();
        }
        JS_FreeValue(ctx, op1);
        JS_FreeValue(ctx, op2);
        if (!r)
            goto exception;
        sp[-2] = JS_CompactBigInt(ctx, r);
    } else {
        double dr;
        /* float64 result */
        if (JS_ToFloat64Free(ctx, &d1, op1)) {
            JS_FreeValue(ctx, op2);
            goto exception;
        }
        if (JS_ToFloat64Free(ctx, &d2, op2))
            goto exception;
    handle_float64:
        switch(op) {
        case OP_sub:
            dr = d1 - d2;
            break;
        case OP_mul:
            dr = d1 * d2;
            break;
        case OP_div:
            dr = d1 / d2;
            break;
        case OP_mod:
            dr = fmod(d1, d2);
            break;
        case OP_pow:
            dr = js_math_pow(d1, d2);
            break;
        default:
            abort();
        }
        sp[-2] = js_float64(dr);
    }
    return 0;
 exception:
    sp[-2] = JS_UNDEFINED;
    sp[-1] = JS_UNDEFINED;
    return -1;
}

static no_inline __exception int js_add_slow(JSContext *ctx, JSValue *sp)
{
    JSValue op1, op2;
    uint32_t tag1, tag2;

    op1 = sp[-2];
    op2 = sp[-1];

    tag1 = JS_VALUE_GET_NORM_TAG(op1);
    tag2 = JS_VALUE_GET_NORM_TAG(op2);
    /* fast path for float64 */
    if (tag1 == JS_TAG_FLOAT64 && tag2 == JS_TAG_FLOAT64) {
        double d1, d2;
        d1 = JS_VALUE_GET_FLOAT64(op1);
        d2 = JS_VALUE_GET_FLOAT64(op2);
        sp[-2] = js_float64(d1 + d2);
        return 0;
    }
    /* fast path for short bigint */
    if (tag1 == JS_TAG_SHORT_BIG_INT && tag2 == JS_TAG_SHORT_BIG_INT) {
        js_slimb_t v1, v2;
        js_sdlimb_t v;
        v1 = JS_VALUE_GET_SHORT_BIG_INT(op1);
        v2 = JS_VALUE_GET_SHORT_BIG_INT(op2);
        v = (js_sdlimb_t)v1 + (js_sdlimb_t)v2;
        if (likely(v >= JS_SHORT_BIG_INT_MIN && v <= JS_SHORT_BIG_INT_MAX)) {
            sp[-2] = __JS_NewShortBigInt(ctx, v);
        } else {
            JSBigInt *r = js_bigint_new_di(ctx, v);
            if (!r)
                goto exception;
            sp[-2] = JS_MKPTR(JS_TAG_BIG_INT, r);
        }
        return 0;
    }

    if (tag1 == JS_TAG_OBJECT || tag2 == JS_TAG_OBJECT) {
        op1 = JS_ToPrimitiveFree(ctx, op1, HINT_NONE);
        if (JS_IsException(op1)) {
            JS_FreeValue(ctx, op2);
            goto exception;
        }

        op2 = JS_ToPrimitiveFree(ctx, op2, HINT_NONE);
        if (JS_IsException(op2)) {
            JS_FreeValue(ctx, op1);
            goto exception;
        }
        tag1 = JS_VALUE_GET_NORM_TAG(op1);
        tag2 = JS_VALUE_GET_NORM_TAG(op2);
    }

    if (tag1 == JS_TAG_STRING || tag2 == JS_TAG_STRING) {
        sp[-2] = JS_ConcatString(ctx, op1, op2);
        if (JS_IsException(sp[-2]))
            goto exception;
        return 0;
    }

    op1 = JS_ToNumericFree(ctx, op1);
    if (JS_IsException(op1)) {
        JS_FreeValue(ctx, op2);
        goto exception;
    }
    op2 = JS_ToNumericFree(ctx, op2);
    if (JS_IsException(op2)) {
        JS_FreeValue(ctx, op1);
        goto exception;
    }
    tag1 = JS_VALUE_GET_NORM_TAG(op1);
    tag2 = JS_VALUE_GET_NORM_TAG(op2);

    if (tag1 == JS_TAG_INT && tag2 == JS_TAG_INT) {
        int32_t v1, v2;
        int64_t v;
        v1 = JS_VALUE_GET_INT(op1);
        v2 = JS_VALUE_GET_INT(op2);
        v = (int64_t)v1 + (int64_t)v2;
        sp[-2] = js_int64(v);
    } else if ((tag1 == JS_TAG_BIG_INT || tag1 == JS_TAG_SHORT_BIG_INT) &&
               (tag2 == JS_TAG_BIG_INT || tag2 == JS_TAG_SHORT_BIG_INT)) {
        JSBigInt *p1, *p2, *r;
        JSBigIntBuf buf1, buf2;
        /* bigint result */
        if (JS_VALUE_GET_TAG(op1) == JS_TAG_SHORT_BIG_INT)
            p1 = js_bigint_set_short(&buf1, op1);
        else
            p1 = JS_VALUE_GET_PTR(op1);
        if (JS_VALUE_GET_TAG(op2) == JS_TAG_SHORT_BIG_INT)
            p2 = js_bigint_set_short(&buf2, op2);
        else
            p2 = JS_VALUE_GET_PTR(op2);
        r = js_bigint_add(ctx, p1, p2, 0);
        JS_FreeValue(ctx, op1);
        JS_FreeValue(ctx, op2);
        if (!r)
            goto exception;
        sp[-2] = JS_CompactBigInt(ctx, r);
    } else {
        double d1, d2;
        /* float64 result */
        if (JS_ToFloat64Free(ctx, &d1, op1)) {
            JS_FreeValue(ctx, op2);
            goto exception;
        }
        if (JS_ToFloat64Free(ctx, &d2, op2))
            goto exception;
        sp[-2] = js_float64(d1 + d2);
    }
    return 0;
 exception:
    sp[-2] = JS_UNDEFINED;
    sp[-1] = JS_UNDEFINED;
    return -1;
}

static no_inline __exception int js_binary_logic_slow(JSContext *ctx,
                                                      JSValue *sp,
                                                      OPCodeEnum op)
{
    JSValue op1, op2;
    uint32_t tag1, tag2;
    uint32_t v1, v2, r;

    op1 = sp[-2];
    op2 = sp[-1];
    tag1 = JS_VALUE_GET_NORM_TAG(op1);
    tag2 = JS_VALUE_GET_NORM_TAG(op2);

    if (tag1 == JS_TAG_SHORT_BIG_INT && tag2 == JS_TAG_SHORT_BIG_INT) {
        js_slimb_t v1, v2, v;
        js_sdlimb_t vd;
        v1 = JS_VALUE_GET_SHORT_BIG_INT(op1);
        v2 = JS_VALUE_GET_SHORT_BIG_INT(op2);
        /* bigint fast path */
        switch(op) {
        case OP_and:
            v = v1 & v2;
            break;
        case OP_or:
            v = v1 | v2;
            break;
        case OP_xor:
            v = v1 ^ v2;
            break;
        case OP_sar:
            if (v2 > (JS_LIMB_BITS - 1)) {
                goto slow_big_int;
            } else if (v2 < 0) {
                if (v2 < -(JS_LIMB_BITS - 1))
                    goto slow_big_int;
                v2 = -v2;
                goto bigint_shl;
            }
        bigint_sar:
            v = v1 >> v2;
            break;
        case OP_shl:
            if (v2 > (JS_LIMB_BITS - 1)) {
                goto slow_big_int;
            } else if (v2 < 0) {
                if (v2 < -(JS_LIMB_BITS - 1))
                    goto slow_big_int;
                v2 = -v2;
                goto bigint_sar;
            }
        bigint_shl:
            vd = (js_dlimb_t)v1 << v2;
            if (likely(vd >= JS_SHORT_BIG_INT_MIN &&
                       vd <= JS_SHORT_BIG_INT_MAX)) {
                v = vd;
            } else {
                JSBigInt *r = js_bigint_new_di(ctx, vd);
                if (!r)
                    goto exception;
                sp[-2] = JS_MKPTR(JS_TAG_BIG_INT, r);
                return 0;
            }
            break;
        default:
            abort();
        }
        sp[-2] = __JS_NewShortBigInt(ctx, v);
        return 0;
    }
    op1 = JS_ToNumericFree(ctx, op1);
    if (JS_IsException(op1)) {
        JS_FreeValue(ctx, op2);
        goto exception;
    }
    op2 = JS_ToNumericFree(ctx, op2);
    if (JS_IsException(op2)) {
        JS_FreeValue(ctx, op1);
        goto exception;
    }

    tag1 = JS_VALUE_GET_TAG(op1);
    tag2 = JS_VALUE_GET_TAG(op2);
    if ((tag1 == JS_TAG_BIG_INT || tag1 == JS_TAG_SHORT_BIG_INT) &&
        (tag2 == JS_TAG_BIG_INT || tag2 == JS_TAG_SHORT_BIG_INT)) {
        JSBigInt *p1, *p2, *r;
        JSBigIntBuf buf1, buf2;
    slow_big_int:
        if (JS_VALUE_GET_TAG(op1) == JS_TAG_SHORT_BIG_INT)
            p1 = js_bigint_set_short(&buf1, op1);
        else
            p1 = JS_VALUE_GET_PTR(op1);
        if (JS_VALUE_GET_TAG(op2) == JS_TAG_SHORT_BIG_INT)
            p2 = js_bigint_set_short(&buf2, op2);
        else
            p2 = JS_VALUE_GET_PTR(op2);
        switch(op) {
        case OP_and:
        case OP_or:
        case OP_xor:
            r = js_bigint_logic(ctx, p1, p2, op);
            break;
        case OP_shl:
        case OP_sar:
            {
                js_slimb_t shift;
                shift = js_bigint_get_si_sat(p2);
                if (shift > INT32_MAX)
                    shift = INT32_MAX;
                else if (shift < -INT32_MAX)
                    shift = -INT32_MAX;
                if (op == OP_sar)
                    shift = -shift;
                if (shift >= 0)
                    r = js_bigint_shl(ctx, p1, shift);
                else
                    r = js_bigint_shr(ctx, p1, -shift);
            }
            break;
        default:
            abort();
        }
        JS_FreeValue(ctx, op1);
        JS_FreeValue(ctx, op2);
        if (!r)
            goto exception;
        sp[-2] = JS_CompactBigInt(ctx, r);
    } else {
        if (unlikely(JS_ToInt32Free(ctx, (int32_t *)&v1, op1))) {
            JS_FreeValue(ctx, op2);
            goto exception;
        }
        if (unlikely(JS_ToInt32Free(ctx, (int32_t *)&v2, op2)))
            goto exception;
        switch(op) {
        case OP_shl:
            r = v1 << (v2 & 0x1f);
            break;
        case OP_sar:
            r = (int)v1 >> (v2 & 0x1f);
            break;
        case OP_and:
            r = v1 & v2;
            break;
        case OP_or:
            r = v1 | v2;
            break;
        case OP_xor:
            r = v1 ^ v2;
            break;
        default:
            abort();
        }
        sp[-2] = js_int32(r);
    }
    return 0;
 exception:
    sp[-2] = JS_UNDEFINED;
    sp[-1] = JS_UNDEFINED;
    return -1;
}

/* op1 must be a bigint or int. */
static JSBigInt *JS_ToBigIntBuf(JSContext *ctx, JSBigIntBuf *buf1,
                                JSValue op1)
{
    JSBigInt *p1;

    switch(JS_VALUE_GET_TAG(op1)) {
    case JS_TAG_INT:
        p1 = js_bigint_set_si(buf1, JS_VALUE_GET_INT(op1));
        break;
    case JS_TAG_SHORT_BIG_INT:
        p1 = js_bigint_set_short(buf1, op1);
        break;
    case JS_TAG_BIG_INT:
        p1 = JS_VALUE_GET_PTR(op1);
        break;
    default:
        abort();
    }
    return p1;
}

/* op1 and op2 must be numeric types and at least one must be a
   bigint. No exception is generated. */
static int js_compare_bigint(JSContext *ctx, OPCodeEnum op,
                             JSValue op1, JSValue op2)
{
    int res, val, tag1, tag2;
    JSBigIntBuf buf1, buf2;
    JSBigInt *p1, *p2;

    tag1 = JS_VALUE_GET_NORM_TAG(op1);
    tag2 = JS_VALUE_GET_NORM_TAG(op2);
    if ((tag1 == JS_TAG_SHORT_BIG_INT || tag1 == JS_TAG_INT) &&
        (tag2 == JS_TAG_SHORT_BIG_INT || tag2 == JS_TAG_INT)) {
        /* fast path */
        js_slimb_t v1, v2;
        if (tag1 == JS_TAG_INT)
            v1 = JS_VALUE_GET_INT(op1);
        else
            v1 = JS_VALUE_GET_SHORT_BIG_INT(op1);
        if (tag2 == JS_TAG_INT)
            v2 = JS_VALUE_GET_INT(op2);
        else
            v2 = JS_VALUE_GET_SHORT_BIG_INT(op2);
        val = (v1 > v2) - (v1 < v2);
    } else {
        if (tag1 == JS_TAG_FLOAT64) {
            p2 = JS_ToBigIntBuf(ctx, &buf2, op2);
            val = js_bigint_float64_cmp(ctx, p2, JS_VALUE_GET_FLOAT64(op1));
            if (val == 2)
                goto unordered;
            val = -val;
        } else if (tag2 == JS_TAG_FLOAT64) {
            p1 = JS_ToBigIntBuf(ctx, &buf1, op1);
            val = js_bigint_float64_cmp(ctx, p1, JS_VALUE_GET_FLOAT64(op2));
            if (val == 2) {
            unordered:
                JS_FreeValue(ctx, op1);
                JS_FreeValue(ctx, op2);
                return false;
            }
        } else {
            p1 = JS_ToBigIntBuf(ctx, &buf1, op1);
            p2 = JS_ToBigIntBuf(ctx, &buf2, op2);
            val = js_bigint_cmp(ctx, p1, p2);
        }
        JS_FreeValue(ctx, op1);
        JS_FreeValue(ctx, op2);
    }

    switch(op) {
    case OP_lt:
        res = val < 0;
        break;
    case OP_lte:
        res = val <= 0;
        break;
    case OP_gt:
        res = val > 0;
        break;
    case OP_gte:
        res = val >= 0;
        break;
    case OP_eq:
        res = val == 0;
        break;
    default:
        abort();
    }
    return res;
}

static no_inline int js_relational_slow(JSContext *ctx, JSValue *sp,
                                        OPCodeEnum op)
{
    JSValue op1, op2;
    int res;
    uint32_t tag1, tag2;

    op1 = sp[-2];
    op2 = sp[-1];
    tag1 = JS_VALUE_GET_NORM_TAG(op1);
    tag2 = JS_VALUE_GET_NORM_TAG(op2);

    op1 = JS_ToPrimitiveFree(ctx, op1, HINT_NUMBER);
    if (JS_IsException(op1)) {
        JS_FreeValue(ctx, op2);
        goto exception;
    }
    op2 = JS_ToPrimitiveFree(ctx, op2, HINT_NUMBER);
    if (JS_IsException(op2)) {
        JS_FreeValue(ctx, op1);
        goto exception;
    }
    tag1 = JS_VALUE_GET_NORM_TAG(op1);
    tag2 = JS_VALUE_GET_NORM_TAG(op2);

    if (tag1 == JS_TAG_STRING && tag2 == JS_TAG_STRING) {
        JSString *p1, *p2;
        p1 = JS_VALUE_GET_STRING(op1);
        p2 = JS_VALUE_GET_STRING(op2);
        res = js_string_compare(p1, p2);
        switch(op) {
        case OP_lt:
            res = (res < 0);
            break;
        case OP_lte:
            res = (res <= 0);
            break;
        case OP_gt:
            res = (res > 0);
            break;
        default:
        case OP_gte:
            res = (res >= 0);
            break;
        }
        JS_FreeValue(ctx, op1);
        JS_FreeValue(ctx, op2);
    } else if ((tag1 <= JS_TAG_NULL || tag1 == JS_TAG_FLOAT64) &&
               (tag2 <= JS_TAG_NULL || tag2 == JS_TAG_FLOAT64)) {
        /* fast path for float64/int */
        goto float64_compare;
    } else {
        if ((((tag1 == JS_TAG_BIG_INT || tag1 == JS_TAG_SHORT_BIG_INT) &&
              tag2 == JS_TAG_STRING) ||
             ((tag2 == JS_TAG_BIG_INT || tag2 == JS_TAG_SHORT_BIG_INT) &&
              tag1 == JS_TAG_STRING))) {
            if (tag1 == JS_TAG_STRING) {
                op1 = JS_StringToBigInt(ctx, op1);
                if (JS_VALUE_GET_TAG(op1) != JS_TAG_BIG_INT &&
                    JS_VALUE_GET_TAG(op1) != JS_TAG_SHORT_BIG_INT)
                    goto invalid_bigint_string;
            }
            if (tag2 == JS_TAG_STRING) {
                op2 = JS_StringToBigInt(ctx, op2);
                if (JS_VALUE_GET_TAG(op2) != JS_TAG_BIG_INT &&
                    JS_VALUE_GET_TAG(op2) != JS_TAG_SHORT_BIG_INT) {
                invalid_bigint_string:
                    JS_FreeValue(ctx, op1);
                    JS_FreeValue(ctx, op2);
                    res = false;
                    goto done;
                }
            }
        } else {
            op1 = JS_ToNumericFree(ctx, op1);
            if (JS_IsException(op1)) {
                JS_FreeValue(ctx, op2);
                goto exception;
            }
            op2 = JS_ToNumericFree(ctx, op2);
            if (JS_IsException(op2)) {
                JS_FreeValue(ctx, op1);
                goto exception;
            }
        }

        tag1 = JS_VALUE_GET_NORM_TAG(op1);
        tag2 = JS_VALUE_GET_NORM_TAG(op2);

        if (tag1 == JS_TAG_BIG_INT || tag1 == JS_TAG_SHORT_BIG_INT ||
            tag2 == JS_TAG_BIG_INT || tag2 == JS_TAG_SHORT_BIG_INT) {
            res = js_compare_bigint(ctx, op, op1, op2);
        } else {
            double d1, d2;

        float64_compare:
            /* can use floating point comparison */
            if (tag1 == JS_TAG_FLOAT64) {
                d1 = JS_VALUE_GET_FLOAT64(op1);
            } else {
                d1 = JS_VALUE_GET_INT(op1);
            }
            if (tag2 == JS_TAG_FLOAT64) {
                d2 = JS_VALUE_GET_FLOAT64(op2);
            } else {
                d2 = JS_VALUE_GET_INT(op2);
            }
            switch(op) {
            case OP_lt:
                res = (d1 < d2); /* if NaN return false */
                break;
            case OP_lte:
                res = (d1 <= d2); /* if NaN return false */
                break;
            case OP_gt:
                res = (d1 > d2); /* if NaN return false */
                break;
            default:
            case OP_gte:
                res = (d1 >= d2); /* if NaN return false */
                break;
            }
        }
    }
 done:
    sp[-2] = js_bool(res);
    return 0;
 exception:
    sp[-2] = JS_UNDEFINED;
    sp[-1] = JS_UNDEFINED;
    return -1;
}

static bool tag_is_number(uint32_t tag)
{
    return (tag == JS_TAG_INT ||
            tag == JS_TAG_FLOAT64 ||
            tag == JS_TAG_BIG_INT || tag == JS_TAG_SHORT_BIG_INT);
}

static no_inline __exception int js_eq_slow(JSContext *ctx, JSValue *sp,
                                            bool is_neq)
{
    JSValue op1, op2;
    int res;
    uint32_t tag1, tag2;

    op1 = sp[-2];
    op2 = sp[-1];
 redo:
    tag1 = JS_VALUE_GET_NORM_TAG(op1);
    tag2 = JS_VALUE_GET_NORM_TAG(op2);
    if (tag_is_number(tag1) && tag_is_number(tag2)) {
        if (tag1 == JS_TAG_INT && tag2 == JS_TAG_INT) {
            res = JS_VALUE_GET_INT(op1) == JS_VALUE_GET_INT(op2);
        } else if ((tag1 == JS_TAG_FLOAT64 &&
                    (tag2 == JS_TAG_INT || tag2 == JS_TAG_FLOAT64)) ||
                   (tag2 == JS_TAG_FLOAT64 &&
                    (tag1 == JS_TAG_INT || tag1 == JS_TAG_FLOAT64))) {
            double d1, d2;
            if (tag1 == JS_TAG_FLOAT64) {
                d1 = JS_VALUE_GET_FLOAT64(op1);
            } else {
                d1 = JS_VALUE_GET_INT(op1);
            }
            if (tag2 == JS_TAG_FLOAT64) {
                d2 = JS_VALUE_GET_FLOAT64(op2);
            } else {
                d2 = JS_VALUE_GET_INT(op2);
            }
            res = (d1 == d2);
        } else {
            res = js_compare_bigint(ctx, OP_eq, op1, op2);
            if (res < 0)
                goto exception;
        }
    } else if (tag1 == tag2) {
        res = js_strict_eq2(ctx, op1, op2, JS_EQ_STRICT);
    } else if ((tag1 == JS_TAG_NULL && tag2 == JS_TAG_UNDEFINED) ||
               (tag2 == JS_TAG_NULL && tag1 == JS_TAG_UNDEFINED)) {
        res = true;
    } else if ((tag1 == JS_TAG_STRING && tag_is_number(tag2)) ||
               (tag2 == JS_TAG_STRING && tag_is_number(tag1))) {

        if (tag1 == JS_TAG_BIG_INT || tag1 == JS_TAG_SHORT_BIG_INT ||
            tag2 == JS_TAG_BIG_INT || tag2 == JS_TAG_SHORT_BIG_INT) {
            if (tag1 == JS_TAG_STRING) {
                op1 = JS_StringToBigInt(ctx, op1);
                if (JS_VALUE_GET_TAG(op1) != JS_TAG_BIG_INT &&
                    JS_VALUE_GET_TAG(op1) != JS_TAG_SHORT_BIG_INT)
                    goto invalid_bigint_string;
            }
            if (tag2 == JS_TAG_STRING) {
                op2 = JS_StringToBigInt(ctx, op2);
                if (JS_VALUE_GET_TAG(op2) != JS_TAG_BIG_INT &&
                    JS_VALUE_GET_TAG(op2) != JS_TAG_SHORT_BIG_INT ) {
                invalid_bigint_string:
                    JS_FreeValue(ctx, op1);
                    JS_FreeValue(ctx, op2);
                    res = false;
                    goto done;
                }
            }
        } else {
            op1 = JS_ToNumericFree(ctx, op1);
            if (JS_IsException(op1)) {
                JS_FreeValue(ctx, op2);
                goto exception;
            }
            op2 = JS_ToNumericFree(ctx, op2);
            if (JS_IsException(op2)) {
                JS_FreeValue(ctx, op1);
                goto exception;
            }
        }
        res = js_strict_eq(ctx, op1, op2);
    } else if (tag1 == JS_TAG_BOOL) {
        op1 = js_int32(JS_VALUE_GET_INT(op1));
        goto redo;
    } else if (tag2 == JS_TAG_BOOL) {
        op2 = js_int32(JS_VALUE_GET_INT(op2));
        goto redo;
    } else if ((tag1 == JS_TAG_OBJECT &&
                (tag_is_number(tag2) || tag2 == JS_TAG_STRING || tag2 == JS_TAG_SYMBOL)) ||
               (tag2 == JS_TAG_OBJECT &&
                (tag_is_number(tag1) || tag1 == JS_TAG_STRING || tag1 == JS_TAG_SYMBOL))) {
        op1 = JS_ToPrimitiveFree(ctx, op1, HINT_NONE);
        if (JS_IsException(op1)) {
            JS_FreeValue(ctx, op2);
            goto exception;
        }
        op2 = JS_ToPrimitiveFree(ctx, op2, HINT_NONE);
        if (JS_IsException(op2)) {
            JS_FreeValue(ctx, op1);
            goto exception;
        }
        goto redo;
    } else {
        /* IsHTMLDDA object is equivalent to undefined for '==' and '!=' */
        if ((JS_IsHTMLDDA(ctx, op1) &&
             (tag2 == JS_TAG_NULL || tag2 == JS_TAG_UNDEFINED)) ||
            (JS_IsHTMLDDA(ctx, op2) &&
             (tag1 == JS_TAG_NULL || tag1 == JS_TAG_UNDEFINED))) {
            res = true;
        } else {
            res = false;
        }
        JS_FreeValue(ctx, op1);
        JS_FreeValue(ctx, op2);
    }
 done:
    sp[-2] = js_bool(res ^ is_neq);
    return 0;
 exception:
    sp[-2] = JS_UNDEFINED;
    sp[-1] = JS_UNDEFINED;
    return -1;
}

static no_inline int js_shr_slow(JSContext *ctx, JSValue *sp)
{
    JSValue op1, op2;
    uint32_t v1, v2, r;

    op1 = sp[-2];
    op2 = sp[-1];
    op1 = JS_ToNumericFree(ctx, op1);
    if (JS_IsException(op1)) {
        JS_FreeValue(ctx, op2);
        goto exception;
    }
    op2 = JS_ToNumericFree(ctx, op2);
    if (JS_IsException(op2)) {
        JS_FreeValue(ctx, op1);
        goto exception;
    }

    if (JS_VALUE_GET_TAG(op1) == JS_TAG_BIG_INT ||
        JS_VALUE_GET_TAG(op1) == JS_TAG_SHORT_BIG_INT ||
        JS_VALUE_GET_TAG(op2) == JS_TAG_BIG_INT ||
        JS_VALUE_GET_TAG(op2) == JS_TAG_SHORT_BIG_INT) {
        JS_ThrowTypeError(ctx, "BigInt operands are forbidden for >>>");
        JS_FreeValue(ctx, op1);
        JS_FreeValue(ctx, op2);
        goto exception;
    }
    /* cannot give an exception */
    JS_ToUint32Free(ctx, &v1, op1);
    JS_ToUint32Free(ctx, &v2, op2);
    r = v1 >> (v2 & 0x1f);
    sp[-2] = js_uint32(r);
    return 0;
 exception:
    sp[-2] = JS_UNDEFINED;
    sp[-1] = JS_UNDEFINED;
    return -1;
}

static bool js_strict_eq2(JSContext *ctx, JSValue op1, JSValue op2,
                          JSStrictEqModeEnum eq_mode)
{
    bool res;
    int tag1, tag2;
    double d1, d2;

    tag1 = JS_VALUE_GET_NORM_TAG(op1);
    tag2 = JS_VALUE_GET_NORM_TAG(op2);
    switch(tag1) {
    case JS_TAG_BOOL:
        if (tag1 != tag2) {
            res = false;
        } else {
            res = JS_VALUE_GET_INT(op1) == JS_VALUE_GET_INT(op2);
            goto done_no_free;
        }
        break;
    case JS_TAG_NULL:
    case JS_TAG_UNDEFINED:
        res = (tag1 == tag2);
        break;
    case JS_TAG_STRING:
        {
            JSString *p1, *p2;
            if (tag1 != tag2) {
                res = false;
            } else {
                p1 = JS_VALUE_GET_STRING(op1);
                p2 = JS_VALUE_GET_STRING(op2);
                res = js_string_eq(p1, p2);
            }
        }
        break;
    case JS_TAG_SYMBOL:
        {
            JSAtomStruct *p1, *p2;
            if (tag1 != tag2) {
                res = false;
            } else {
                p1 = JS_VALUE_GET_PTR(op1);
                p2 = JS_VALUE_GET_PTR(op2);
                res = (p1 == p2);
            }
        }
        break;
    case JS_TAG_OBJECT:
        if (tag1 != tag2)
            res = false;
        else
            res = JS_VALUE_GET_OBJ(op1) == JS_VALUE_GET_OBJ(op2);
        break;
    case JS_TAG_INT:
        d1 = JS_VALUE_GET_INT(op1);
        if (tag2 == JS_TAG_INT) {
            d2 = JS_VALUE_GET_INT(op2);
            goto number_test;
        } else if (tag2 == JS_TAG_FLOAT64) {
            d2 = JS_VALUE_GET_FLOAT64(op2);
            goto number_test;
        } else {
            res = false;
        }
        break;
    case JS_TAG_FLOAT64:
        d1 = JS_VALUE_GET_FLOAT64(op1);
        if (tag2 == JS_TAG_FLOAT64) {
            d2 = JS_VALUE_GET_FLOAT64(op2);
        } else if (tag2 == JS_TAG_INT) {
            d2 = JS_VALUE_GET_INT(op2);
        } else {
            res = false;
            break;
        }
    number_test:
        if (unlikely(eq_mode >= JS_EQ_SAME_VALUE)) {
            JSFloat64Union u1, u2;
            /* NaN is not always normalized, so this test is necessary */
            if (isnan(d1) || isnan(d2)) {
                res = isnan(d1) == isnan(d2);
            } else if (eq_mode == JS_EQ_SAME_VALUE_ZERO) {
                res = (d1 == d2); /* +0 == -0 */
            } else {
                u1.d = d1;
                u2.d = d2;
                res = (u1.u64 == u2.u64); /* +0 != -0 */
            }
        } else {
            res = (d1 == d2); /* if NaN return false and +0 == -0 */
        }
        goto done_no_free;
    case JS_TAG_SHORT_BIG_INT:
    case JS_TAG_BIG_INT:
        {
            JSBigIntBuf buf1, buf2;
            JSBigInt *p1, *p2;

            if (tag2 != JS_TAG_SHORT_BIG_INT &&
                tag2 != JS_TAG_BIG_INT) {
                res = false;
                break;
            }

            if (JS_VALUE_GET_TAG(op1) == JS_TAG_SHORT_BIG_INT)
                p1 = js_bigint_set_short(&buf1, op1);
            else
                p1 = JS_VALUE_GET_PTR(op1);
            if (JS_VALUE_GET_TAG(op2) == JS_TAG_SHORT_BIG_INT)
                p2 = js_bigint_set_short(&buf2, op2);
            else
                p2 = JS_VALUE_GET_PTR(op2);
            res = (js_bigint_cmp(ctx, p1, p2) == 0);
        }
        break;
    default:
        res = false;
        break;
    }
    JS_FreeValue(ctx, op1);
    JS_FreeValue(ctx, op2);
 done_no_free:
    return res;
}

static bool js_strict_eq(JSContext *ctx, JSValue op1, JSValue op2)
{
    return js_strict_eq2(ctx, op1, op2, JS_EQ_STRICT);
}

static bool js_same_value(JSContext *ctx, JSValueConst op1, JSValueConst op2)
{
    return js_strict_eq2(ctx, js_dup(op1), js_dup(op2), JS_EQ_SAME_VALUE);
}

static bool js_same_value_zero(JSContext *ctx, JSValueConst op1, JSValueConst op2)
{
    return js_strict_eq2(ctx, js_dup(op1), js_dup(op2), JS_EQ_SAME_VALUE_ZERO);
}

static no_inline int js_strict_eq_slow(JSContext *ctx, JSValue *sp,
                                       bool is_neq)
{
    bool res;
    res = js_strict_eq(ctx, sp[-2], sp[-1]);
    sp[-2] = js_bool(res ^ is_neq);
    return 0;
}

static __exception int js_operator_in(JSContext *ctx, JSValue *sp)
{
    JSValue op1, op2;
    JSAtom atom;
    int ret;

    op1 = sp[-2];
    op2 = sp[-1];

    if (JS_VALUE_GET_TAG(op2) != JS_TAG_OBJECT) {
        JS_ThrowTypeError(ctx, "invalid 'in' operand");
        return -1;
    }
    atom = JS_ValueToAtom(ctx, op1);
    if (unlikely(atom == JS_ATOM_NULL))
        return -1;
    ret = JS_HasProperty(ctx, op2, atom);
    JS_FreeAtom(ctx, atom);
    if (ret < 0)
        return -1;
    JS_FreeValue(ctx, op1);
    JS_FreeValue(ctx, op2);
    sp[-2] = js_bool(ret);
    return 0;
}

static __exception int js_operator_private_in(JSContext *ctx, JSValue *sp)
{
    JSValue op1, op2;
    int ret;
    op1 = sp[-2]; /* object */
    op2 = sp[-1]; /* field name or method function */
    if (JS_VALUE_GET_TAG(op1) != JS_TAG_OBJECT) {
        JS_ThrowTypeError(ctx, "invalid 'in' operand");
        return -1;
    }
    if (JS_IsObject(op2)) {
        /* method: use the brand */
        ret = JS_CheckBrand(ctx, op1, op2);
        if (ret < 0)
            return -1;
    } else {
        JSAtom atom;
        JSObject *p;
        JSShapeProperty *prs;
        JSProperty *pr;
        /* field */
        atom = JS_ValueToAtom(ctx, op2);
        if (unlikely(atom == JS_ATOM_NULL))
            return -1;
        p = JS_VALUE_GET_OBJ(op1);
        prs = find_own_property(&pr, p, atom);
        JS_FreeAtom(ctx, atom);
        ret = (prs != NULL);
    }
    JS_FreeValue(ctx, op1);
    JS_FreeValue(ctx, op2);
    sp[-2] = js_bool(ret);
    return 0;
}

static __exception int js_has_unscopable(JSContext *ctx, JSValue obj,
                                         JSAtom atom)
{
    JSValue arr, val;
    int ret;

    arr = JS_GetProperty(ctx, obj, JS_ATOM_Symbol_unscopables);
    if (JS_IsException(arr))
        return -1;
    ret = 0;
    if (JS_IsObject(arr)) {
        val = JS_GetProperty(ctx, arr, atom);
        ret = JS_ToBoolFree(ctx, val);
    }
    JS_FreeValue(ctx, arr);
    return ret;
}

static __exception int js_operator_instanceof(JSContext *ctx, JSValue *sp)
{
    JSValue op1, op2;
    int ret;

    op1 = sp[-2];
    op2 = sp[-1];
    ret = JS_IsInstanceOf(ctx, op1, op2);
    if (ret < 0)
        return ret;
    JS_FreeValue(ctx, op1);
    JS_FreeValue(ctx, op2);
    sp[-2] = js_bool(ret);
    return 0;
}

static __exception int js_operator_typeof(JSContext *ctx, JSValue op1)
{
    JSAtom atom;
    uint32_t tag;

    tag = JS_VALUE_GET_NORM_TAG(op1);
    switch(tag) {
    case JS_TAG_SHORT_BIG_INT:
    case JS_TAG_BIG_INT:
        atom = JS_ATOM_bigint;
        break;
    case JS_TAG_INT:
    case JS_TAG_FLOAT64:
        atom = JS_ATOM_number;
        break;
    case JS_TAG_UNDEFINED:
        atom = JS_ATOM_undefined;
        break;
    case JS_TAG_BOOL:
        atom = JS_ATOM_boolean;
        break;
    case JS_TAG_STRING:
        atom = JS_ATOM_string;
        break;
    case JS_TAG_OBJECT:
        {
            JSObject *p;
            p = JS_VALUE_GET_OBJ(op1);
            if (unlikely(p->is_HTMLDDA))
                atom = JS_ATOM_undefined;
            else if (JS_IsFunction(ctx, op1))
                atom = JS_ATOM_function;
            else
                goto obj_type;
        }
        break;
    case JS_TAG_NULL:
    obj_type:
        atom = JS_ATOM_object;
        break;
    case JS_TAG_SYMBOL:
        atom = JS_ATOM_symbol;
        break;
    default:
        atom = JS_ATOM_unknown;
        break;
    }
    return atom;
}

static __exception int js_operator_delete(JSContext *ctx, JSValue *sp)
{
    JSValue op1, op2;
    JSAtom atom;
    int ret;

    op1 = sp[-2];
    op2 = sp[-1];
    atom = JS_ValueToAtom(ctx, op2);
    if (unlikely(atom == JS_ATOM_NULL))
        return -1;
    ret = JS_DeleteProperty(ctx, op1, atom, JS_PROP_THROW_STRICT);
    JS_FreeAtom(ctx, atom);
    if (unlikely(ret < 0))
        return -1;
    JS_FreeValue(ctx, op1);
    JS_FreeValue(ctx, op2);
    sp[-2] = js_bool(ret);
    return 0;
}

static JSValue js_throw_type_error(JSContext *ctx, JSValueConst this_val,
                                   int argc, JSValueConst *argv)
{
    JSFunctionBytecode *b = JS_GetFunctionBytecode(this_val);
    if (!b || b->is_strict_mode || !b->has_prototype) {
        return JS_ThrowTypeError(ctx, "invalid property access");
    }
    return JS_UNDEFINED;
}

static JSValue js_function_proto_fileName(JSContext *ctx,
                                          JSValueConst this_val)
{
    JSFunctionBytecode *b = JS_GetFunctionBytecode(this_val);
    if (b) {
        return JS_AtomToString(ctx, b->filename);
    }
    return JS_UNDEFINED;
}

static JSValue js_function_proto_int32(JSContext *ctx,
                                       JSValueConst this_val,
                                       int magic)
{
    JSFunctionBytecode *b = JS_GetFunctionBytecode(this_val);
    if (b) {
        int *field = (int *) ((char *)b + magic);
        return js_int32(*field);
    }
    return JS_UNDEFINED;
}

static int js_arguments_define_own_property(JSContext *ctx,
                                            JSValueConst this_obj,
                                            JSAtom prop, JSValueConst val,
                                            JSValueConst getter,
                                            JSValueConst setter, int flags)
{
    JSObject *p;
    uint32_t idx;
    p = JS_VALUE_GET_OBJ(this_obj);
    /* convert to normal array when redefining an existing numeric field */
    if (p->fast_array && JS_AtomIsArrayIndex(ctx, &idx, prop) &&
        idx < p->u.array.count) {
        if (convert_fast_array_to_array(ctx, p))
            return -1;
    }
    /* run the default define own property */
    return JS_DefineProperty(ctx, this_obj, prop, val, getter, setter,
                             flags | JS_PROP_NO_EXOTIC);
}

static const JSClassExoticMethods js_arguments_exotic_methods = {
    .define_own_property = js_arguments_define_own_property,
};

static JSValue js_build_arguments(JSContext *ctx, int argc, JSValueConst *argv)
{
    JSValue val, *tab;
    JSProperty *pr;
    JSObject *p;
    int i;

    val = JS_NewObjectProtoClass(ctx, ctx->class_proto[JS_CLASS_OBJECT],
                                 JS_CLASS_ARGUMENTS);
    if (JS_IsException(val))
        return val;
    p = JS_VALUE_GET_OBJ(val);

    /* add the length field (cannot fail) */
    pr = add_property(ctx, p, JS_ATOM_length,
                      JS_PROP_WRITABLE | JS_PROP_CONFIGURABLE);
    if (!pr) {
        JS_FreeValue(ctx, val);
        return JS_EXCEPTION;
    }
    pr->u.value = js_int32(argc);

    /* initialize the fast array part */
    tab = NULL;
    if (argc > 0) {
        tab = js_malloc(ctx, sizeof(tab[0]) * argc);
        if (!tab) {
            JS_FreeValue(ctx, val);
            return JS_EXCEPTION;
        }
        for(i = 0; i < argc; i++) {
            tab[i] = js_dup(argv[i]);
        }
    }
    p->u.array.u.values = tab;
    p->u.array.count = argc;

    JS_DefinePropertyValue(ctx, val, JS_ATOM_Symbol_iterator,
                           js_dup(ctx->array_proto_values),
                           JS_PROP_CONFIGURABLE | JS_PROP_WRITABLE);
    /* add callee property to throw a TypeError in strict mode */
    JS_DefineProperty(ctx, val, JS_ATOM_callee, JS_UNDEFINED,
                      ctx->throw_type_error, ctx->throw_type_error,
                      JS_PROP_HAS_GET | JS_PROP_HAS_SET);
    return val;
}

#define GLOBAL_VAR_OFFSET 0x40000000
#define ARGUMENT_VAR_OFFSET 0x20000000

/* legacy arguments object: add references to the function arguments */
static JSValue js_build_mapped_arguments(JSContext *ctx, int argc,
                                         JSValueConst *argv,
                                         JSStackFrame *sf, int arg_count)
{
    JSValue val;
    JSProperty *pr;
    JSObject *p;
    int i;

    val = JS_NewObjectProtoClass(ctx, ctx->class_proto[JS_CLASS_OBJECT],
                                 JS_CLASS_MAPPED_ARGUMENTS);
    if (JS_IsException(val))
        return val;
    p = JS_VALUE_GET_OBJ(val);

    /* add the length field (cannot fail) */
    pr = add_property(ctx, p, JS_ATOM_length,
                      JS_PROP_WRITABLE | JS_PROP_CONFIGURABLE);
    if (!pr)
        goto fail;
    pr->u.value = js_int32(argc);

    for(i = 0; i < arg_count; i++) {
        JSVarRef *var_ref;
        var_ref = get_var_ref(ctx, sf, i, true);
        if (!var_ref)
            goto fail;
        pr = add_property(ctx, p, __JS_AtomFromUInt32(i), JS_PROP_C_W_E | JS_PROP_VARREF);
        if (!pr) {
            free_var_ref(ctx->rt, var_ref);
            goto fail;
        }
        pr->u.var_ref = var_ref;
    }

    /* the arguments not mapped to the arguments of the function can
       be normal properties */
    for(i = arg_count; i < argc; i++) {
        if (JS_DefinePropertyValueUint32(ctx, val, i,
                                         js_dup(argv[i]),
                                         JS_PROP_C_W_E) < 0)
            goto fail;
    }

    JS_DefinePropertyValue(ctx, val, JS_ATOM_Symbol_iterator,
                           js_dup(ctx->array_proto_values),
                           JS_PROP_CONFIGURABLE | JS_PROP_WRITABLE);
    /* callee returns this function in non strict mode */
    JS_DefinePropertyValue(ctx, val, JS_ATOM_callee,
                           js_dup(ctx->rt->current_stack_frame->cur_func),
                           JS_PROP_CONFIGURABLE | JS_PROP_WRITABLE);
    return val;
 fail:
    JS_FreeValue(ctx, val);
    return JS_EXCEPTION;
}

static JSValue build_for_in_iterator(JSContext *ctx, JSValue obj)
{
    JSObject *p;
    JSPropertyEnum *tab_atom;
    int i;
    JSValue enum_obj, obj1;
    JSForInIterator *it;
    uint32_t tag, tab_atom_count;

    tag = JS_VALUE_GET_TAG(obj);
    if (tag != JS_TAG_OBJECT && tag != JS_TAG_NULL && tag != JS_TAG_UNDEFINED) {
        obj = JS_ToObjectFree(ctx, obj);
    }

    it = js_malloc(ctx, sizeof(*it));
    if (!it) {
        JS_FreeValue(ctx, obj);
        return JS_EXCEPTION;
    }
    enum_obj = JS_NewObjectProtoClass(ctx, JS_NULL, JS_CLASS_FOR_IN_ITERATOR);
    if (JS_IsException(enum_obj)) {
        js_free(ctx, it);
        JS_FreeValue(ctx, obj);
        return JS_EXCEPTION;
    }
    it->is_array = false;
    it->obj = obj;
    it->idx = 0;
    p = JS_VALUE_GET_OBJ(enum_obj);
    p->u.for_in_iterator = it;

    if (tag == JS_TAG_NULL || tag == JS_TAG_UNDEFINED)
        return enum_obj;

    /* fast path: assume no enumerable properties in the prototype chain */
    obj1 = js_dup(obj);
    for(;;) {
        obj1 = JS_GetPrototypeFree(ctx, obj1);
        if (JS_IsNull(obj1))
            break;
        if (JS_IsException(obj1))
            goto fail;
        if (JS_GetOwnPropertyNamesInternal(ctx, &tab_atom, &tab_atom_count,
                                           JS_VALUE_GET_OBJ(obj1),
                                           JS_GPN_STRING_MASK | JS_GPN_ENUM_ONLY)) {
            JS_FreeValue(ctx, obj1);
            goto fail;
        }
        js_free_prop_enum(ctx, tab_atom, tab_atom_count);
        if (tab_atom_count != 0) {
            JS_FreeValue(ctx, obj1);
            goto slow_path;
        }
        /* must check for timeout to avoid infinite loop */
        if (js_poll_interrupts(ctx)) {
            JS_FreeValue(ctx, obj1);
            goto fail;
        }
    }

    p = JS_VALUE_GET_OBJ(obj);

    if (p->fast_array) {
        JSShape *sh;
        JSShapeProperty *prs;
        /* check that there are no enumerable normal fields */
        sh = p->shape;
        for(i = 0, prs = get_shape_prop(sh); i < sh->prop_count; i++, prs++) {
            if (prs->flags & JS_PROP_ENUMERABLE)
                goto normal_case;
        }
        /* for fast arrays, we only store the number of elements */
        it->is_array = true;
        it->array_length = p->u.array.count;
    } else {
    normal_case:
        if (JS_GetOwnPropertyNamesInternal(ctx, &tab_atom, &tab_atom_count, p,
                                   JS_GPN_STRING_MASK | JS_GPN_ENUM_ONLY))
            goto fail;
        for(i = 0; i < tab_atom_count; i++) {
            JS_SetPropertyInternal(ctx, enum_obj, tab_atom[i].atom, JS_NULL, 0);
        }
        js_free_prop_enum(ctx, tab_atom, tab_atom_count);
    }
    return enum_obj;

 slow_path:
    /* non enumerable properties hide the enumerables ones in the
       prototype chain */
    obj1 = js_dup(obj);
    for(;;) {
        if (JS_GetOwnPropertyNamesInternal(ctx, &tab_atom, &tab_atom_count,
                                           JS_VALUE_GET_OBJ(obj1),
                                           JS_GPN_STRING_MASK | JS_GPN_SET_ENUM)) {
            JS_FreeValue(ctx, obj1);
            goto fail;
        }
        for(i = 0; i < tab_atom_count; i++) {
            JS_DefinePropertyValue(ctx, enum_obj, tab_atom[i].atom, JS_NULL,
                                   (tab_atom[i].is_enumerable ?
                                    JS_PROP_ENUMERABLE : 0));
        }
        js_free_prop_enum(ctx, tab_atom, tab_atom_count);
        obj1 = JS_GetPrototypeFree(ctx, obj1);
        if (JS_IsNull(obj1))
            break;
        if (JS_IsException(obj1))
            goto fail;
        /* must check for timeout to avoid infinite loop */
        if (js_poll_interrupts(ctx)) {
            JS_FreeValue(ctx, obj1);
            goto fail;
        }
    }
    return enum_obj;

 fail:
    JS_FreeValue(ctx, enum_obj);
    return JS_EXCEPTION;
}

/* obj -> enum_obj */
static __exception int js_for_in_start(JSContext *ctx, JSValue *sp)
{
    sp[-1] = build_for_in_iterator(ctx, sp[-1]);
    if (JS_IsException(sp[-1]))
        return -1;
    return 0;
}

/* enum_obj -> enum_obj value done */
static __exception int js_for_in_next(JSContext *ctx, JSValue *sp)
{
    JSValue enum_obj;
    JSObject *p;
    JSAtom prop;
    JSForInIterator *it;
    int ret;

    enum_obj = sp[-1];
    /* fail safe */
    if (JS_VALUE_GET_TAG(enum_obj) != JS_TAG_OBJECT)
        goto done;
    p = JS_VALUE_GET_OBJ(enum_obj);
    if (p->class_id != JS_CLASS_FOR_IN_ITERATOR)
        goto done;
    it = p->u.for_in_iterator;

    for(;;) {
        if (it->is_array) {
            if (it->idx >= it->array_length)
                goto done;
            prop = __JS_AtomFromUInt32(it->idx);
            it->idx++;
        } else {
            JSShape *sh = p->shape;
            JSShapeProperty *prs;
            if (it->idx >= sh->prop_count)
                goto done;
            prs = get_shape_prop(sh) + it->idx;
            prop = prs->atom;
            it->idx++;
            if (prop == JS_ATOM_NULL || !(prs->flags & JS_PROP_ENUMERABLE))
                continue;
        }
        // check if the property was deleted unless we're dealing with a proxy
        JSValue obj = it->obj;
        if (JS_VALUE_GET_TAG(obj) == JS_TAG_OBJECT) {
            JSObject *p = JS_VALUE_GET_OBJ(obj);
            if (p->class_id == JS_CLASS_PROXY)
                break;
        }
        ret = JS_HasProperty(ctx, obj, prop);
        if (ret < 0)
            return ret;
        if (ret)
            break;
    }
    /* return the property */
    sp[0] = JS_AtomToValue(ctx, prop);
    sp[1] = JS_FALSE;
    return 0;
 done:
    /* return the end */
    sp[0] = JS_UNDEFINED;
    sp[1] = JS_TRUE;
    return 0;
}

static JSValue JS_GetIterator2(JSContext *ctx, JSValueConst obj,
                               JSValueConst method)
{
    JSValue enum_obj;

    enum_obj = JS_Call(ctx, method, obj, 0, NULL);
    if (JS_IsException(enum_obj))
        return enum_obj;
    if (!JS_IsObject(enum_obj)) {
        JS_FreeValue(ctx, enum_obj);
        return JS_ThrowTypeErrorNotAnObject(ctx);
    }
    return enum_obj;
}

static JSValue JS_GetIterator(JSContext *ctx, JSValueConst obj, bool is_async)
{
    JSValue method, ret, sync_iter;

    if (is_async) {
        method = JS_GetProperty(ctx, obj, JS_ATOM_Symbol_asyncIterator);
        if (JS_IsException(method))
            return method;
        if (JS_IsUndefined(method) || JS_IsNull(method)) {
            method = JS_GetProperty(ctx, obj, JS_ATOM_Symbol_iterator);
            if (JS_IsException(method))
                return method;
            sync_iter = JS_GetIterator2(ctx, obj, method);
            JS_FreeValue(ctx, method);
            if (JS_IsException(sync_iter))
                return sync_iter;
            ret = JS_CreateAsyncFromSyncIterator(ctx, sync_iter);
            JS_FreeValue(ctx, sync_iter);
            return ret;
        }
    } else {
        method = JS_GetProperty(ctx, obj, JS_ATOM_Symbol_iterator);
        if (JS_IsException(method))
            return method;
    }
    if (!JS_IsFunction(ctx, method)) {
        JS_FreeValue(ctx, method);
        return JS_ThrowTypeError(ctx, "value is not iterable");
    }
    ret = JS_GetIterator2(ctx, obj, method);
    JS_FreeValue(ctx, method);
    return ret;
}

/* return *pdone = 2 if the iterator object is not parsed */
static JSValue JS_IteratorNext2(JSContext *ctx, JSValueConst enum_obj,
                                JSValueConst method,
                                int argc, JSValueConst *argv, int *pdone)
{
    JSValue obj;

    /* fast path for the built-in iterators (avoid creating the
       intermediate result object) */
    if (JS_IsObject(method)) {
        JSObject *p = JS_VALUE_GET_OBJ(method);
        if (p->class_id == JS_CLASS_C_FUNCTION &&
            p->u.cfunc.cproto == JS_CFUNC_iterator_next) {
            JSCFunctionType func;
            JSValueConst args[1];

            /* in case the function expects one argument */
            if (argc == 0) {
                args[0] = JS_UNDEFINED;
                argv = args;
            }
            func = p->u.cfunc.c_function;
            return func.iterator_next(ctx, enum_obj, argc, argv,
                                      pdone, p->u.cfunc.magic);
        }
    }
    obj = JS_Call(ctx, method, enum_obj, argc, argv);
    if (JS_IsException(obj))
        goto fail;
    if (!JS_IsObject(obj)) {
        JS_FreeValue(ctx, obj);
        JS_ThrowTypeError(ctx, "iterator must return an object");
        goto fail;
    }
    *pdone = 2;
    return obj;
 fail:
    *pdone = false;
    return JS_EXCEPTION;
}

static JSValue JS_IteratorNext(JSContext *ctx, JSValueConst enum_obj,
                               JSValueConst method,
                               int argc, JSValueConst *argv, int *pdone)
{
    JSValue obj, value, done_val;
    int done;

    obj = JS_IteratorNext2(ctx, enum_obj, method, argc, argv, &done);
    if (JS_IsException(obj))
        goto fail;
    if (likely(done == 0)) {
        *pdone = false;
        return obj;
    } else if (done != 2) {
        JS_FreeValue(ctx, obj);
        *pdone = true;
        return JS_UNDEFINED;
    } else {
        done_val = JS_GetProperty(ctx, obj, JS_ATOM_done);
        if (JS_IsException(done_val))
            goto fail;
        *pdone = JS_ToBoolFree(ctx, done_val);
        value = JS_UNDEFINED;
        if (!*pdone) {
            value = JS_GetProperty(ctx, obj, JS_ATOM_value);
        }
        JS_FreeValue(ctx, obj);
        return value;
    }
 fail:
    JS_FreeValue(ctx, obj);
    *pdone = false;
    return JS_EXCEPTION;
}

/* return < 0 in case of exception */
static int JS_IteratorClose(JSContext *ctx, JSValueConst enum_obj,
                            bool is_exception_pending)
{
    JSValue method, ret, ex_obj;
    int res;

    if (is_exception_pending) {
        ex_obj = ctx->rt->current_exception;
        ctx->rt->current_exception = JS_UNINITIALIZED;
        res = -1;
    } else {
        ex_obj = JS_UNDEFINED;
        res = 0;
    }
    method = JS_GetProperty(ctx, enum_obj, JS_ATOM_return);
    if (JS_IsException(method)) {
        res = -1;
        goto done;
    }
    if (JS_IsUndefined(method) || JS_IsNull(method)) {
        goto done;
    }
    ret = JS_CallFree(ctx, method, enum_obj, 0, NULL);
    if (!is_exception_pending) {
        if (JS_IsException(ret)) {
            res = -1;
        } else if (!JS_IsObject(ret)) {
            JS_ThrowTypeErrorNotAnObject(ctx);
            res = -1;
        }
    }
    JS_FreeValue(ctx, ret);
 done:
    if (is_exception_pending) {
        JS_Throw(ctx, ex_obj);
    }
    return res;
}

/* obj -> enum_rec (3 slots) */
static __exception int js_for_of_start(JSContext *ctx, JSValue *sp,
                                       bool is_async)
{
    JSValue op1, obj, method;
    op1 = sp[-1];
    obj = JS_GetIterator(ctx, op1, is_async);
    if (JS_IsException(obj))
        return -1;
    JS_FreeValue(ctx, op1);
    sp[-1] = obj;
    method = JS_GetProperty(ctx, obj, JS_ATOM_next);
    if (JS_IsException(method))
        return -1;
    sp[0] = method;
    return 0;
}

/* enum_rec [objs] -> enum_rec [objs] value done. There are 'offset'
   objs. If 'done' is true or in case of exception, 'enum_rec' is set
   to undefined. If 'done' is true, 'value' is always set to
   undefined. */
static __exception int js_for_of_next(JSContext *ctx, JSValue *sp, int offset)
{
    JSValue value = JS_UNDEFINED;
    int done = 1;

    if (likely(!JS_IsUndefined(sp[offset]))) {
        value = JS_IteratorNext(ctx, sp[offset], sp[offset + 1], 0, NULL, &done);
        if (JS_IsException(value))
            done = -1;
        if (done) {
            /* value is JS_UNDEFINED or JS_EXCEPTION */
            /* replace the iteration object with undefined */
            JS_FreeValue(ctx, sp[offset]);
            sp[offset] = JS_UNDEFINED;
            if (done < 0) {
                return -1;
            } else {
                JS_FreeValue(ctx, value);
                value = JS_UNDEFINED;
            }
        }
    }
    sp[0] = value;
    sp[1] = js_bool(done);
    return 0;
}

static JSValue JS_IteratorGetCompleteValue(JSContext *ctx, JSValue obj,
                                           int *pdone)
{
    JSValue done_val, value;
    int done;
    done_val = JS_GetProperty(ctx, obj, JS_ATOM_done);
    if (JS_IsException(done_val))
        goto fail;
    done = JS_ToBoolFree(ctx, done_val);
    value = JS_GetProperty(ctx, obj, JS_ATOM_value);
    if (JS_IsException(value))
        goto fail;
    *pdone = done;
    return value;
 fail:
    *pdone = false;
    return JS_EXCEPTION;
}

static __exception int js_iterator_get_value_done(JSContext *ctx, JSValue *sp)
{
    JSValue obj, value;
    int done;
    obj = sp[-1];
    if (!JS_IsObject(obj)) {
        JS_ThrowTypeError(ctx, "iterator must return an object");
        return -1;
    }
    value = JS_IteratorGetCompleteValue(ctx, obj, &done);
    if (JS_IsException(value))
        return -1;
    JS_FreeValue(ctx, obj);
    sp[-1] = value;
    sp[0] = js_bool(done);
    return 0;
}

static JSValue js_create_iterator_result(JSContext *ctx,
                                         JSValue val,
                                         bool done)
{
    JSValue obj;
    obj = JS_NewObject(ctx);
    if (JS_IsException(obj)) {
        JS_FreeValue(ctx, val);
        return obj;
    }
    if (JS_DefinePropertyValue(ctx, obj, JS_ATOM_value,
                               val, JS_PROP_C_W_E) < 0) {
        goto fail;
    }
    if (JS_DefinePropertyValue(ctx, obj, JS_ATOM_done,
                               js_bool(done), JS_PROP_C_W_E) < 0) {
    fail:
        JS_FreeValue(ctx, obj);
        return JS_EXCEPTION;
    }
    return obj;
}

static JSValue js_array_iterator_next(JSContext *ctx, JSValueConst this_val,
                                      int argc, JSValueConst *argv,
                                      int *pdone, int magic);

static JSValue js_create_array_iterator(JSContext *ctx, JSValueConst this_val,
                                        int argc, JSValueConst *argv, int magic);

static bool js_is_fast_array(JSContext *ctx, JSValue obj)
{
    /* Try and handle fast arrays explicitly */
    if (JS_VALUE_GET_TAG(obj) == JS_TAG_OBJECT) {
        JSObject *p = JS_VALUE_GET_OBJ(obj);
        if (p->class_id == JS_CLASS_ARRAY && p->fast_array) {
            return true;
        }
    }
    return false;
}

/* Access an Array's internal JSValue array if available */
static bool js_get_fast_array(JSContext *ctx, JSValue obj,
                              JSValue **arrpp, uint32_t *countp)
{
    /* Try and handle fast arrays explicitly */
    if (JS_VALUE_GET_TAG(obj) == JS_TAG_OBJECT) {
        JSObject *p = JS_VALUE_GET_OBJ(obj);
        if (p->class_id == JS_CLASS_ARRAY && p->fast_array) {
            *countp = p->u.array.count;
            *arrpp = p->u.array.u.values;
            return true;
        }
    }
    return false;
}

static __exception int js_append_enumerate(JSContext *ctx, JSValue *sp)
{
    JSValue iterator, enumobj, method, value;
    int is_array_iterator;
    JSValue *arrp;
    uint32_t i, count32, pos;

    if (JS_VALUE_GET_TAG(sp[-2]) != JS_TAG_INT) {
        JS_ThrowInternalError(ctx, "invalid index for append");
        return -1;
    }

    pos = JS_VALUE_GET_INT(sp[-2]);

    /* XXX: further optimisations:
       - use ctx->array_proto_values?
       - check if array_iterator_prototype next method is built-in and
         avoid constructing actual iterator object?
       - build this into js_for_of_start and use in all `for (x of o)` loops
     */
    iterator = JS_GetProperty(ctx, sp[-1], JS_ATOM_Symbol_iterator);
    if (JS_IsException(iterator))
        return -1;
    /* Used to squelch a -Wcast-function-type warning. */
    JSCFunctionType ft = { .generic_magic = js_create_array_iterator };
    is_array_iterator = JS_IsCFunction(ctx, iterator,
                                       ft.generic,
                                       JS_ITERATOR_KIND_VALUE);
    JS_FreeValue(ctx, iterator);

    enumobj = JS_GetIterator(ctx, sp[-1], false);
    if (JS_IsException(enumobj))
        return -1;
    method = JS_GetProperty(ctx, enumobj, JS_ATOM_next);
    if (JS_IsException(method)) {
        JS_FreeValue(ctx, enumobj);
        return -1;
    }
    /* Used to squelch a -Wcast-function-type warning. */
    JSCFunctionType ft2 = { .iterator_next = js_array_iterator_next };
    if (is_array_iterator
            &&  JS_IsCFunction(ctx, method, ft2.generic, 0)
            &&  js_get_fast_array(ctx, sp[-1], &arrp, &count32)) {
        uint32_t len;
        if (js_get_length32(ctx, &len, sp[-1]))
            goto exception;
        /* if len > count32, the elements >= count32 might be read in
           the prototypes and might have side effects */
        if (len != count32)
            goto general_case;
        /* Handle fast arrays explicitly */
        for (i = 0; i < count32; i++) {
            if (JS_DefinePropertyValueUint32(ctx, sp[-3], pos++,
                                             js_dup(arrp[i]), JS_PROP_C_W_E) < 0)
                goto exception;
        }
    } else {
    general_case:
        for (;;) {
            int done;
            value = JS_IteratorNext(ctx, enumobj, method, 0, NULL, &done);
            if (JS_IsException(value))
                goto exception;
            if (done) {
                /* value is JS_UNDEFINED */
                break;
            }
            if (JS_DefinePropertyValueUint32(ctx, sp[-3], pos++, value, JS_PROP_C_W_E) < 0)
                goto exception;
        }
    }
    /* Note: could raise an error if too many elements */
    sp[-2] = js_int32(pos);
    JS_FreeValue(ctx, enumobj);
    JS_FreeValue(ctx, method);
    return 0;

exception:
    JS_IteratorClose(ctx, enumobj, true);
    JS_FreeValue(ctx, enumobj);
    JS_FreeValue(ctx, method);
    return -1;
}

static __exception int JS_CopyDataProperties(JSContext *ctx,
                                             JSValue target,
                                             JSValue source,
                                             JSValue excluded,
                                             bool setprop)
{
    JSPropertyEnum *tab_atom;
    JSValue val;
    uint32_t i, tab_atom_count;
    JSObject *p;
    JSObject *pexcl = NULL;
    int ret, gpn_flags;
    JSPropertyDescriptor desc;
    bool is_enumerable;

    if (JS_VALUE_GET_TAG(source) != JS_TAG_OBJECT)
        return 0;

    if (JS_VALUE_GET_TAG(excluded) == JS_TAG_OBJECT)
        pexcl = JS_VALUE_GET_OBJ(excluded);

    p = JS_VALUE_GET_OBJ(source);

    gpn_flags = JS_GPN_STRING_MASK | JS_GPN_SYMBOL_MASK | JS_GPN_ENUM_ONLY;
    if (p->is_exotic) {
        const JSClassExoticMethods *em = ctx->rt->class_array[p->class_id].exotic;
        /* cannot use JS_GPN_ENUM_ONLY with e.g. proxies because it
           introduces a visible change */
        if (em && em->get_own_property_names) {
            gpn_flags &= ~JS_GPN_ENUM_ONLY;
        }
    }
    if (JS_GetOwnPropertyNamesInternal(ctx, &tab_atom, &tab_atom_count, p,
                                       gpn_flags))
        return -1;

    for (i = 0; i < tab_atom_count; i++) {
        if (pexcl) {
            ret = JS_GetOwnPropertyInternal(ctx, NULL, pexcl, tab_atom[i].atom);
            if (ret) {
                if (ret < 0)
                    goto exception;
                continue;
            }
        }
        if (!(gpn_flags & JS_GPN_ENUM_ONLY)) {
            /* test if the property is enumerable */
            ret = JS_GetOwnPropertyInternal(ctx, &desc, p, tab_atom[i].atom);
            if (ret < 0)
                goto exception;
            if (!ret)
                continue;
            is_enumerable = (desc.flags & JS_PROP_ENUMERABLE) != 0;
            js_free_desc(ctx, &desc);
            if (!is_enumerable)
                continue;
        }
        val = JS_GetProperty(ctx, source, tab_atom[i].atom);
        if (JS_IsException(val))
            goto exception;
        if (setprop)
            ret = JS_SetProperty(ctx, target, tab_atom[i].atom, val);
        else
            ret = JS_DefinePropertyValue(ctx, target, tab_atom[i].atom, val,
                                         JS_PROP_C_W_E);
        if (ret < 0)
            goto exception;
    }
    js_free_prop_enum(ctx, tab_atom, tab_atom_count);
    return 0;
 exception:
    js_free_prop_enum(ctx, tab_atom, tab_atom_count);
    return -1;
}

/* only valid inside C functions */
static JSValueConst JS_GetActiveFunction(JSContext *ctx)
{
    return ctx->rt->current_stack_frame->cur_func;
}

static JSVarRef *get_var_ref(JSContext *ctx, JSStackFrame *sf,
                             int var_idx, bool is_arg)
{
    JSVarRef *var_ref;
    struct list_head *el;
    JSValue *pvalue;

    if (is_arg)
        pvalue = &sf->arg_buf[var_idx];
    else
        pvalue = &sf->var_buf[var_idx];

    list_for_each(el, &sf->var_ref_list) {
        var_ref = list_entry(el, JSVarRef, header.link);
        if (var_ref->pvalue == pvalue) {
            var_ref->header.ref_count++;
            return var_ref;
        }
    }
    /* create a new one */
    var_ref = js_malloc(ctx, sizeof(JSVarRef));
    if (!var_ref)
        return NULL;
    var_ref->header.ref_count = 1;
    var_ref->is_detached = false;
    list_add_tail(&var_ref->header.link, &sf->var_ref_list);
    var_ref->pvalue = pvalue;
    var_ref->value = JS_UNDEFINED;
    return var_ref;
}

static JSValue js_closure2(JSContext *ctx, JSValue func_obj,
                           JSFunctionBytecode *b,
                           JSVarRef **cur_var_refs,
                           JSStackFrame *sf)
{
    JSObject *p;
    JSVarRef **var_refs;
    int i;

    p = JS_VALUE_GET_OBJ(func_obj);
    p->u.func.function_bytecode = b;
    p->u.func.home_object = NULL;
    p->u.func.var_refs = NULL;
    if (b->closure_var_count) {
        var_refs = js_mallocz(ctx, sizeof(var_refs[0]) * b->closure_var_count);
        if (!var_refs)
            goto fail;
        p->u.func.var_refs = var_refs;
        for(i = 0; i < b->closure_var_count; i++) {
            JSClosureVar *cv = &b->closure_var[i];
            JSVarRef *var_ref;
            if (cv->is_local) {
                /* reuse the existing variable reference if it already exists */
                var_ref = get_var_ref(ctx, sf, cv->var_idx, cv->is_arg);
                if (!var_ref)
                    goto fail;
            } else {
                var_ref = cur_var_refs[cv->var_idx];
                var_ref->header.ref_count++;
            }
            var_refs[i] = var_ref;
        }
    }
    return func_obj;
 fail:
    /* bfunc is freed when func_obj is freed */
    JS_FreeValue(ctx, func_obj);
    return JS_EXCEPTION;
}

static JSValue js_instantiate_prototype(JSContext *ctx, JSObject *p, JSAtom atom, void *opaque)
{
    JSValue obj, this_val;
    int ret;

    this_val = JS_MKPTR(JS_TAG_OBJECT, p);
    obj = JS_NewObject(ctx);
    if (JS_IsException(obj))
        return JS_EXCEPTION;
    ret = JS_DefinePropertyValue(ctx, obj, JS_ATOM_constructor,
                                 js_dup(this_val),
                                 JS_PROP_WRITABLE | JS_PROP_CONFIGURABLE);
    if (ret < 0) {
        JS_FreeValue(ctx, obj);
        return JS_EXCEPTION;
    }
    return obj;
}

static const uint16_t func_kind_to_class_id[] = {
    [JS_FUNC_NORMAL] = JS_CLASS_BYTECODE_FUNCTION,
    [JS_FUNC_GENERATOR] = JS_CLASS_GENERATOR_FUNCTION,
    [JS_FUNC_ASYNC] = JS_CLASS_ASYNC_FUNCTION,
    [JS_FUNC_ASYNC_GENERATOR] = JS_CLASS_ASYNC_GENERATOR_FUNCTION,
};

static JSValue js_closure(JSContext *ctx, JSValue bfunc,
                          JSVarRef **cur_var_refs,
                          JSStackFrame *sf)
{
    JSFunctionBytecode *b;
    JSValue func_obj;
    JSAtom name_atom;

    b = JS_VALUE_GET_PTR(bfunc);
    func_obj = JS_NewObjectClass(ctx, func_kind_to_class_id[b->func_kind]);
    if (JS_IsException(func_obj)) {
        JS_FreeValue(ctx, bfunc);
        return JS_EXCEPTION;
    }
    func_obj = js_closure2(ctx, func_obj, b, cur_var_refs, sf);
    if (JS_IsException(func_obj)) {
        /* bfunc has been freed */
        goto fail;
    }
    name_atom = b->func_name;
    if (name_atom == JS_ATOM_NULL)
        name_atom = JS_ATOM_empty_string;
    js_function_set_properties(ctx, func_obj, name_atom,
                               b->defined_arg_count);

    if (b->func_kind & JS_FUNC_GENERATOR) {
        JSValue proto;
        int proto_class_id;
        /* generators have a prototype field which is used as
           prototype for the generator object */
        if (b->func_kind == JS_FUNC_ASYNC_GENERATOR)
            proto_class_id = JS_CLASS_ASYNC_GENERATOR;
        else
            proto_class_id = JS_CLASS_GENERATOR;
        proto = JS_NewObjectProto(ctx, ctx->class_proto[proto_class_id]);
        if (JS_IsException(proto))
            goto fail;
        JS_DefinePropertyValue(ctx, func_obj, JS_ATOM_prototype, proto,
                               JS_PROP_WRITABLE);
    } else if (b->has_prototype) {
        /* add the 'prototype' property: delay instantiation to avoid
           creating cycles for every javascript function. The prototype
           object is created on the fly when first accessed */
        JS_SetConstructorBit(ctx, func_obj, true);
        JS_DefineAutoInitProperty(ctx, func_obj, JS_ATOM_prototype,
                                  JS_AUTOINIT_ID_PROTOTYPE, NULL,
                                  JS_PROP_WRITABLE);
    }
    return func_obj;
 fail:
    /* bfunc is freed when func_obj is freed */
    JS_FreeValue(ctx, func_obj);
    return JS_EXCEPTION;
}

#define JS_DEFINE_CLASS_HAS_HERITAGE     (1 << 0)

static int js_op_define_class(JSContext *ctx, JSValue *sp,
                              JSAtom class_name, int class_flags,
                              JSVarRef **cur_var_refs,
                              JSStackFrame *sf, bool is_computed_name)
{
    JSValue bfunc, parent_class, proto = JS_UNDEFINED;
    JSValue ctor = JS_UNDEFINED, parent_proto = JS_UNDEFINED;
    JSFunctionBytecode *b;

    parent_class = sp[-2];
    bfunc = sp[-1];

    if (class_flags & JS_DEFINE_CLASS_HAS_HERITAGE) {
        if (JS_IsNull(parent_class)) {
            parent_proto = JS_NULL;
            parent_class = js_dup(ctx->function_proto);
        } else {
            if (!JS_IsConstructor(ctx, parent_class)) {
                JS_ThrowTypeError(ctx, "parent class must be constructor");
                goto fail;
            }
            parent_proto = JS_GetProperty(ctx, parent_class, JS_ATOM_prototype);
            if (JS_IsException(parent_proto))
                goto fail;
            if (!JS_IsNull(parent_proto) && !JS_IsObject(parent_proto)) {
                JS_ThrowTypeError(ctx, "parent prototype must be an object or null");
                goto fail;
            }
        }
    } else {
        /* parent_class is JS_UNDEFINED in this case */
        parent_proto = js_dup(ctx->class_proto[JS_CLASS_OBJECT]);
        parent_class = js_dup(ctx->function_proto);
    }
    proto = JS_NewObjectProto(ctx, parent_proto);
    if (JS_IsException(proto))
        goto fail;

    b = JS_VALUE_GET_PTR(bfunc);
    assert(b->func_kind == JS_FUNC_NORMAL);
    ctor = JS_NewObjectProtoClass(ctx, parent_class,
                                  JS_CLASS_BYTECODE_FUNCTION);
    if (JS_IsException(ctor))
        goto fail;
    ctor = js_closure2(ctx, ctor, b, cur_var_refs, sf);
    bfunc = JS_UNDEFINED;
    if (JS_IsException(ctor))
        goto fail;
    js_method_set_home_object(ctx, ctor, proto);
    JS_SetConstructorBit(ctx, ctor, true);

    JS_DefinePropertyValue(ctx, ctor, JS_ATOM_length,
                           js_int32(b->defined_arg_count),
                           JS_PROP_CONFIGURABLE);

    if (is_computed_name) {
        if (JS_DefineObjectNameComputed(ctx, ctor, sp[-3],
                                        JS_PROP_CONFIGURABLE) < 0)
            goto fail;
    } else {
        if (JS_DefineObjectName(ctx, ctor, class_name, JS_PROP_CONFIGURABLE) < 0)
            goto fail;
    }

    /* the constructor property must be first. It can be overriden by
       computed property names */
    if (JS_DefinePropertyValue(ctx, proto, JS_ATOM_constructor,
                               js_dup(ctor),
                               JS_PROP_CONFIGURABLE |
                               JS_PROP_WRITABLE | JS_PROP_THROW) < 0)
        goto fail;
    /* set the prototype property */
    if (JS_DefinePropertyValue(ctx, ctor, JS_ATOM_prototype,
                               js_dup(proto), JS_PROP_THROW) < 0)
        goto fail;

    JS_FreeValue(ctx, parent_proto);
    JS_FreeValue(ctx, parent_class);

    sp[-2] = ctor;
    sp[-1] = proto;
    return 0;
 fail:
    JS_FreeValue(ctx, parent_class);
    JS_FreeValue(ctx, parent_proto);
    JS_FreeValue(ctx, bfunc);
    JS_FreeValue(ctx, proto);
    JS_FreeValue(ctx, ctor);
    sp[-2] = JS_UNDEFINED;
    sp[-1] = JS_UNDEFINED;
    return -1;
}

static void close_var_refs(JSRuntime *rt, JSStackFrame *sf)
{
    struct list_head *el, *el1;
    JSVarRef *var_ref;

    list_for_each_safe(el, el1, &sf->var_ref_list) {
        var_ref = list_entry(el, JSVarRef, header.link);
        var_ref->value = js_dup(*var_ref->pvalue);
        var_ref->pvalue = &var_ref->value;
        /* the reference is no longer to a local variable */
        var_ref->is_detached = true;
        add_gc_object(rt, &var_ref->header, JS_GC_OBJ_TYPE_VAR_REF);
    }
}

static void close_lexical_var(JSContext *ctx, JSStackFrame *sf, int var_idx)
{
    JSValue *pvalue;
    struct list_head *el, *el1;
    JSVarRef *var_ref;

    pvalue = &sf->var_buf[var_idx];
    list_for_each_safe(el, el1, &sf->var_ref_list) {
        var_ref = list_entry(el, JSVarRef, header.link);
        if (var_ref->pvalue == pvalue) {
            var_ref->value = js_dup(*var_ref->pvalue);
            var_ref->pvalue = &var_ref->value;
            list_del(&var_ref->header.link);
            /* the reference is no longer to a local variable */
            var_ref->is_detached = true;
            add_gc_object(ctx->rt, &var_ref->header, JS_GC_OBJ_TYPE_VAR_REF);
        }
    }
}

#define JS_CALL_FLAG_COPY_ARGV   (1 << 1)
#define JS_CALL_FLAG_GENERATOR   (1 << 2)

static JSValue js_call_c_function(JSContext *ctx, JSValueConst func_obj,
                                  JSValueConst this_obj,
                                  int argc, JSValueConst *argv, int flags)
{
    JSRuntime *rt = ctx->rt;
    JSCFunctionType func;
    JSObject *p;
    JSStackFrame sf_s, *sf = &sf_s, *prev_sf;
    JSValue ret_val;
    JSValueConst *arg_buf;
    int arg_count, i;
    JSCFunctionEnum cproto;

    p = JS_VALUE_GET_OBJ(func_obj);
    cproto = p->u.cfunc.cproto;
    arg_count = p->u.cfunc.length;

    /* better to always check stack overflow */
    if (js_check_stack_overflow(rt, sizeof(arg_buf[0]) * arg_count))
        return JS_ThrowStackOverflow(ctx);

    prev_sf = rt->current_stack_frame;
    sf->prev_frame = prev_sf;
    rt->current_stack_frame = sf;
    ctx = p->u.cfunc.realm; /* change the current realm */

    sf->is_strict_mode = false;
    sf->cur_func = unsafe_unconst(func_obj);
    sf->arg_count = argc;
    arg_buf = argv;

    if (unlikely(argc < arg_count)) {
        /* ensure that at least argc_count arguments are readable */
        arg_buf = alloca(sizeof(arg_buf[0]) * arg_count);
        for(i = 0; i < argc; i++)
            arg_buf[i] = argv[i];
        for(i = argc; i < arg_count; i++)
            arg_buf[i] = JS_UNDEFINED;
        sf->arg_count = arg_count;
    }
    sf->arg_buf = (JSValue *)arg_buf;

    func = p->u.cfunc.c_function;
    switch(cproto) {
    case JS_CFUNC_constructor:
    case JS_CFUNC_constructor_or_func:
        if (!(flags & JS_CALL_FLAG_CONSTRUCTOR)) {
            if (cproto == JS_CFUNC_constructor) {
            not_a_constructor:
                ret_val = JS_ThrowTypeError(ctx, "must be called with new");
                break;
            } else {
                this_obj = JS_UNDEFINED;
            }
        }
        /* here this_obj is new_target */
        /* fall thru */
    case JS_CFUNC_generic:
        ret_val = func.generic(ctx, this_obj, argc, arg_buf);
        break;
    case JS_CFUNC_constructor_magic:
    case JS_CFUNC_constructor_or_func_magic:
        if (!(flags & JS_CALL_FLAG_CONSTRUCTOR)) {
            if (cproto == JS_CFUNC_constructor_magic) {
                goto not_a_constructor;
            } else {
                this_obj = JS_UNDEFINED;
            }
        }
        /* fall thru */
    case JS_CFUNC_generic_magic:
        ret_val = func.generic_magic(ctx, this_obj, argc, arg_buf,
                                     p->u.cfunc.magic);
        break;
    case JS_CFUNC_getter:
        ret_val = func.getter(ctx, this_obj);
        break;
    case JS_CFUNC_setter:
        ret_val = func.setter(ctx, this_obj, arg_buf[0]);
        break;
    case JS_CFUNC_getter_magic:
        ret_val = func.getter_magic(ctx, this_obj, p->u.cfunc.magic);
        break;
    case JS_CFUNC_setter_magic:
        ret_val = func.setter_magic(ctx, this_obj, arg_buf[0], p->u.cfunc.magic);
        break;
    case JS_CFUNC_f_f:
        {
            double d1;

            if (unlikely(JS_ToFloat64(ctx, &d1, arg_buf[0]))) {
                ret_val = JS_EXCEPTION;
                break;
            }
            ret_val = js_number(func.f_f(d1));
        }
        break;
    case JS_CFUNC_f_f_f:
        {
            double d1, d2;

            if (unlikely(JS_ToFloat64(ctx, &d1, arg_buf[0]))) {
                ret_val = JS_EXCEPTION;
                break;
            }
            if (unlikely(JS_ToFloat64(ctx, &d2, arg_buf[1]))) {
                ret_val = JS_EXCEPTION;
                break;
            }
            ret_val = js_number(func.f_f_f(d1, d2));
        }
        break;
    case JS_CFUNC_iterator_next:
        {
            int done;
            ret_val = func.iterator_next(ctx, this_obj, argc, arg_buf,
                                         &done, p->u.cfunc.magic);
            if (!JS_IsException(ret_val) && done != 2) {
                ret_val = js_create_iterator_result(ctx, ret_val, done);
            }
        }
        break;
    default:
        abort();
    }

    rt->current_stack_frame = sf->prev_frame;
    return ret_val;
}

static JSValue js_call_bound_function(JSContext *ctx, JSValueConst func_obj,
                                      JSValueConst this_obj,
                                      int argc, JSValueConst *argv, int flags)
{
    JSObject *p;
    JSBoundFunction *bf;
    JSValueConst *arg_buf, new_target;
    int arg_count, i;

    p = JS_VALUE_GET_OBJ(func_obj);
    bf = p->u.bound_function;
    arg_count = bf->argc + argc;
    if (js_check_stack_overflow(ctx->rt, sizeof(JSValue) * arg_count))
        return JS_ThrowStackOverflow(ctx);
    arg_buf = alloca(sizeof(JSValue) * arg_count);
    for(i = 0; i < bf->argc; i++) {
        arg_buf[i] = bf->argv[i];
    }
    for(i = 0; i < argc; i++) {
        arg_buf[bf->argc + i] = argv[i];
    }
    if (flags & JS_CALL_FLAG_CONSTRUCTOR) {
        new_target = this_obj;
        if (js_same_value(ctx, func_obj, new_target))
            new_target = bf->func_obj;
        return JS_CallConstructor2(ctx, bf->func_obj, new_target,
                                   arg_count, arg_buf);
    } else {
        return JS_Call(ctx, bf->func_obj, bf->this_val,
                       arg_count, arg_buf);
    }
}

/* argument of OP_special_object */
typedef enum {
    OP_SPECIAL_OBJECT_ARGUMENTS,
    OP_SPECIAL_OBJECT_MAPPED_ARGUMENTS,
    OP_SPECIAL_OBJECT_THIS_FUNC,
    OP_SPECIAL_OBJECT_NEW_TARGET,
    OP_SPECIAL_OBJECT_HOME_OBJECT,
    OP_SPECIAL_OBJECT_VAR_OBJECT,
    OP_SPECIAL_OBJECT_IMPORT_META,
    OP_SPECIAL_OBJECT_NULL_PROTO,
} OPSpecialObjectEnum;

#define FUNC_RET_AWAIT      0
#define FUNC_RET_YIELD      1
#define FUNC_RET_YIELD_STAR 2

#ifdef ENABLE_DUMPS // JS_DUMP_BYTECODE_*
static void dump_single_byte_code(JSContext *ctx, const uint8_t *pc,
                                  JSFunctionBytecode *b, int start_pos);
static void print_func_name(JSFunctionBytecode *b);
#endif

static bool needs_backtrace(JSValue exc)
{
    JSObject *p;

    if (JS_VALUE_GET_TAG(exc) != JS_TAG_OBJECT)
        return false;
    p = JS_VALUE_GET_OBJ(exc);
    if (p->class_id != JS_CLASS_ERROR)
        return false;
    return !find_own_property1(p, JS_ATOM_stack);
}

/* argv[] is modified if (flags & JS_CALL_FLAG_COPY_ARGV) = 0. */
static JSValue JS_CallInternal(JSContext *caller_ctx, JSValueConst func_obj,
                               JSValueConst this_obj, JSValueConst new_target,
                               int argc, JSValueConst *argv, int flags)
{
    JSRuntime *rt = caller_ctx->rt;
    JSContext *ctx;
    JSObject *p;
    JSFunctionBytecode *b;
    JSStackFrame sf_s, *sf = &sf_s;
    uint8_t *pc;
    int opcode, arg_allocated_size, i;
    JSValue *local_buf, *stack_buf, *var_buf, *arg_buf, *sp, ret_val, *pval;
    JSVarRef **var_refs;
    size_t alloca_size;

#ifdef ENABLE_DUMPS // JS_DUMP_BYTECODE_STEP
#define DUMP_BYTECODE_OR_DONT(pc) \
    if (check_dump_flag(ctx->rt, JS_DUMP_BYTECODE_STEP)) dump_single_byte_code(ctx, pc, b, 0);
#else
#define DUMP_BYTECODE_OR_DONT(pc)
#endif

#if !DIRECT_DISPATCH
#define SWITCH(pc)      DUMP_BYTECODE_OR_DONT(pc) switch (opcode = *pc++)
#define CASE(op)        case op
#define DEFAULT         default
#define BREAK           break
#else
    __extension__ static const void * const dispatch_table[256] = {
#define DEF(id, size, n_pop, n_push, f) && case_OP_ ## id,
#define def(id, size, n_pop, n_push, f)
/*
 * QuickJS opcode definitions
 *
 * Copyright (c) 2017-2018 Fabrice Bellard
 * Copyright (c) 2017-2018 Charlie Gordon
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifdef FMT
FMT(none)
FMT(none_int)
FMT(none_loc)
FMT(none_arg)
FMT(none_var_ref)
FMT(u8)
FMT(i8)
FMT(loc8)
FMT(const8)
FMT(label8)
FMT(u16)
FMT(i16)
FMT(label16)
FMT(npop)
FMT(npopx)
FMT(npop_u16)
FMT(loc)
FMT(arg)
FMT(var_ref)
FMT(u32)
FMT(u32x2)
FMT(i32)
FMT(const)
FMT(label)
FMT(atom)
FMT(atom_u8)
FMT(atom_u16)
FMT(atom_label_u8)
FMT(atom_label_u16)
FMT(label_u16)
#undef FMT
#endif /* FMT */

#ifdef DEF

#ifndef def
#define def(id, size, n_pop, n_push, f) DEF(id, size, n_pop, n_push, f)
#endif

DEF(invalid, 1, 0, 0, none) /* never emitted */

/* push values */
DEF(       push_i32, 5, 0, 1, i32)
DEF(     push_const, 5, 0, 1, const)
DEF(       fclosure, 5, 0, 1, const) /* must follow push_const */
DEF(push_atom_value, 5, 0, 1, atom)
DEF( private_symbol, 5, 0, 1, atom)
DEF(      undefined, 1, 0, 1, none)
DEF(           null, 1, 0, 1, none)
DEF(      push_this, 1, 0, 1, none) /* only used at the start of a function */
DEF(     push_false, 1, 0, 1, none)
DEF(      push_true, 1, 0, 1, none)
DEF(         object, 1, 0, 1, none)
DEF( special_object, 2, 0, 1, u8) /* only used at the start of a function */
DEF(           rest, 3, 0, 1, u16) /* only used at the start of a function */

DEF(           drop, 1, 1, 0, none) /* a -> */
DEF(            nip, 1, 2, 1, none) /* a b -> b */
DEF(           nip1, 1, 3, 2, none) /* a b c -> b c */
DEF(            dup, 1, 1, 2, none) /* a -> a a */
DEF(           dup1, 1, 2, 3, none) /* a b -> a a b */
DEF(           dup2, 1, 2, 4, none) /* a b -> a b a b */
DEF(           dup3, 1, 3, 6, none) /* a b c -> a b c a b c */
DEF(        insert2, 1, 2, 3, none) /* obj a -> a obj a (dup_x1) */
DEF(        insert3, 1, 3, 4, none) /* obj prop a -> a obj prop a (dup_x2) */
DEF(        insert4, 1, 4, 5, none) /* this obj prop a -> a this obj prop a */
DEF(          perm3, 1, 3, 3, none) /* obj a b -> a obj b */
DEF(          perm4, 1, 4, 4, none) /* obj prop a b -> a obj prop b */
DEF(          perm5, 1, 5, 5, none) /* this obj prop a b -> a this obj prop b */
DEF(           swap, 1, 2, 2, none) /* a b -> b a */
DEF(          swap2, 1, 4, 4, none) /* a b c d -> c d a b */
DEF(          rot3l, 1, 3, 3, none) /* x a b -> a b x */
DEF(          rot3r, 1, 3, 3, none) /* a b x -> x a b */
DEF(          rot4l, 1, 4, 4, none) /* x a b c -> a b c x */
DEF(          rot5l, 1, 5, 5, none) /* x a b c d -> a b c d x */

DEF(call_constructor, 3, 2, 1, npop) /* func new.target args -> ret. arguments are not counted in n_pop */
DEF(           call, 3, 1, 1, npop) /* arguments are not counted in n_pop */
DEF(      tail_call, 3, 1, 0, npop) /* arguments are not counted in n_pop */
DEF(    call_method, 3, 2, 1, npop) /* arguments are not counted in n_pop */
DEF(tail_call_method, 3, 2, 0, npop) /* arguments are not counted in n_pop */
DEF(     array_from, 3, 0, 1, npop) /* arguments are not counted in n_pop */
DEF(          apply, 3, 3, 1, u16)
DEF(         return, 1, 1, 0, none)
DEF(   return_undef, 1, 0, 0, none)
DEF(check_ctor_return, 1, 1, 2, none)
DEF(     check_ctor, 1, 0, 0, none)
DEF(      init_ctor, 1, 0, 1, none)
DEF(    check_brand, 1, 2, 2, none) /* this_obj func -> this_obj func */
DEF(      add_brand, 1, 2, 0, none) /* this_obj home_obj -> */
DEF(   return_async, 1, 1, 0, none)
DEF(          throw, 1, 1, 0, none)
DEF(    throw_error, 6, 0, 0, atom_u8)
DEF(           eval, 5, 1, 1, npop_u16) /* func args... -> ret_val */
DEF(     apply_eval, 3, 2, 1, u16) /* func array -> ret_eval */
DEF(         regexp, 1, 2, 1, none) /* create a RegExp object from the pattern and a
                                       bytecode string */
DEF(      get_super, 1, 1, 1, none)
DEF(         import, 1, 1, 1, none) /* dynamic module import */

DEF(      check_var, 5, 0, 1, atom) /* check if a variable exists */
DEF(  get_var_undef, 5, 0, 1, atom) /* push undefined if the variable does not exist */
DEF(        get_var, 5, 0, 1, atom) /* throw an exception if the variable does not exist */
DEF(        put_var, 5, 1, 0, atom) /* must come after get_var */
DEF(   put_var_init, 5, 1, 0, atom) /* must come after put_var. Used to initialize a global lexical variable */
DEF( put_var_strict, 5, 2, 0, atom) /* for strict mode variable write */

DEF(  get_ref_value, 1, 2, 3, none)
DEF(  put_ref_value, 1, 3, 0, none)

DEF(     define_var, 6, 0, 0, atom_u8)
DEF(check_define_var, 6, 0, 0, atom_u8)
DEF(    define_func, 6, 1, 0, atom_u8)

// order matters, see IC counterparts
DEF(      get_field, 5, 1, 1, atom)
DEF(     get_field2, 5, 1, 2, atom)
DEF(      put_field, 5, 2, 0, atom)

DEF( get_private_field, 1, 2, 1, none) /* obj prop -> value */
DEF( put_private_field, 1, 3, 0, none) /* obj value prop -> */
DEF(define_private_field, 1, 3, 1, none) /* obj prop value -> obj */
DEF(   get_array_el, 1, 2, 1, none)
DEF(  get_array_el2, 1, 2, 2, none) /* obj prop -> obj value */
DEF(   put_array_el, 1, 3, 0, none)
DEF(get_super_value, 1, 3, 1, none) /* this obj prop -> value */
DEF(put_super_value, 1, 4, 0, none) /* this obj prop value -> */
DEF(   define_field, 5, 2, 1, atom)
DEF(       set_name, 5, 1, 1, atom)
DEF(set_name_computed, 1, 2, 2, none)
DEF(      set_proto, 1, 2, 1, none)
DEF(set_home_object, 1, 2, 2, none)
DEF(define_array_el, 1, 3, 2, none)
DEF(         append, 1, 3, 2, none) /* append enumerated object, update length */
DEF(copy_data_properties, 2, 3, 3, u8)
DEF(  define_method, 6, 2, 1, atom_u8)
DEF(define_method_computed, 2, 3, 1, u8) /* must come after define_method */
DEF(   define_class, 6, 2, 2, atom_u8) /* parent ctor -> ctor proto */
DEF(   define_class_computed, 6, 3, 3, atom_u8) /* field_name parent ctor -> field_name ctor proto (class with computed name) */

DEF(        get_loc, 3, 0, 1, loc)
DEF(        put_loc, 3, 1, 0, loc) /* must come after get_loc */
DEF(        set_loc, 3, 1, 1, loc) /* must come after put_loc */
DEF(        get_arg, 3, 0, 1, arg)
DEF(        put_arg, 3, 1, 0, arg) /* must come after get_arg */
DEF(        set_arg, 3, 1, 1, arg) /* must come after put_arg */
DEF(    get_var_ref, 3, 0, 1, var_ref)
DEF(    put_var_ref, 3, 1, 0, var_ref) /* must come after get_var_ref */
DEF(    set_var_ref, 3, 1, 1, var_ref) /* must come after put_var_ref */
DEF(set_loc_uninitialized, 3, 0, 0, loc)
DEF(  get_loc_check, 3, 0, 1, loc)
DEF(  put_loc_check, 3, 1, 0, loc) /* must come after get_loc_check */
DEF(  put_loc_check_init, 3, 1, 0, loc)
DEF(get_var_ref_check, 3, 0, 1, var_ref)
DEF(put_var_ref_check, 3, 1, 0, var_ref) /* must come after get_var_ref_check */
DEF(put_var_ref_check_init, 3, 1, 0, var_ref)
DEF(      close_loc, 3, 0, 0, loc)
DEF(       if_false, 5, 1, 0, label)
DEF(        if_true, 5, 1, 0, label) /* must come after if_false */
DEF(           goto, 5, 0, 0, label) /* must come after if_true */
DEF(          catch, 5, 0, 1, label)
DEF(          gosub, 5, 0, 0, label) /* used to execute the finally block */
DEF(            ret, 1, 1, 0, none) /* used to return from the finally block */
DEF(      nip_catch, 1, 2, 1, none) /* catch ... a -> a */

DEF(      to_object, 1, 1, 1, none)
//DEF(      to_string, 1, 1, 1, none)
DEF(     to_propkey, 1, 1, 1, none)
DEF(    to_propkey2, 1, 2, 2, none)

DEF(   with_get_var, 10, 1, 0, atom_label_u8)     /* must be in the same order as scope_xxx */
DEF(   with_put_var, 10, 2, 1, atom_label_u8)     /* must be in the same order as scope_xxx */
DEF(with_delete_var, 10, 1, 0, atom_label_u8)     /* must be in the same order as scope_xxx */
DEF(  with_make_ref, 10, 1, 0, atom_label_u8)     /* must be in the same order as scope_xxx */
DEF(   with_get_ref, 10, 1, 0, atom_label_u8)     /* must be in the same order as scope_xxx */
DEF(with_get_ref_undef, 10, 1, 0, atom_label_u8)

DEF(   make_loc_ref, 7, 0, 2, atom_u16)
DEF(   make_arg_ref, 7, 0, 2, atom_u16)
DEF(make_var_ref_ref, 7, 0, 2, atom_u16)
DEF(   make_var_ref, 5, 0, 2, atom)

DEF(   for_in_start, 1, 1, 1, none)
DEF(   for_of_start, 1, 1, 3, none)
DEF(for_await_of_start, 1, 1, 3, none)
DEF(    for_in_next, 1, 1, 3, none)
DEF(    for_of_next, 2, 3, 5, u8)
DEF(iterator_check_object, 1, 1, 1, none)
DEF(iterator_get_value_done, 1, 1, 2, none)
DEF( iterator_close, 1, 3, 0, none)
DEF(  iterator_next, 1, 4, 4, none)
DEF(  iterator_call, 2, 4, 5, u8)
DEF(  initial_yield, 1, 0, 0, none)
DEF(          yield, 1, 1, 2, none)
DEF(     yield_star, 1, 1, 2, none)
DEF(async_yield_star, 1, 1, 2, none)
DEF(          await, 1, 1, 1, none)

/* arithmetic/logic operations */
DEF(            neg, 1, 1, 1, none)
DEF(           plus, 1, 1, 1, none)
DEF(            dec, 1, 1, 1, none)
DEF(            inc, 1, 1, 1, none)
DEF(       post_dec, 1, 1, 2, none)
DEF(       post_inc, 1, 1, 2, none)
DEF(        dec_loc, 2, 0, 0, loc8)
DEF(        inc_loc, 2, 0, 0, loc8)
DEF(        add_loc, 2, 1, 0, loc8)
DEF(            not, 1, 1, 1, none)
DEF(           lnot, 1, 1, 1, none)
DEF(         typeof, 1, 1, 1, none)
DEF(         delete, 1, 2, 1, none)
DEF(     delete_var, 5, 0, 1, atom)

/* warning: order matters (see js_parse_assign_expr) */
DEF(            mul, 1, 2, 1, none)
DEF(            div, 1, 2, 1, none)
DEF(            mod, 1, 2, 1, none)
DEF(            add, 1, 2, 1, none)
DEF(            sub, 1, 2, 1, none)
DEF(            shl, 1, 2, 1, none)
DEF(            sar, 1, 2, 1, none)
DEF(            shr, 1, 2, 1, none)
DEF(            and, 1, 2, 1, none)
DEF(            xor, 1, 2, 1, none)
DEF(             or, 1, 2, 1, none)
DEF(            pow, 1, 2, 1, none)

DEF(             lt, 1, 2, 1, none)
DEF(            lte, 1, 2, 1, none)
DEF(             gt, 1, 2, 1, none)
DEF(            gte, 1, 2, 1, none)
DEF(     instanceof, 1, 2, 1, none)
DEF(             in, 1, 2, 1, none)
DEF(             eq, 1, 2, 1, none)
DEF(            neq, 1, 2, 1, none)
DEF(      strict_eq, 1, 2, 1, none)
DEF(     strict_neq, 1, 2, 1, none)
DEF(is_undefined_or_null, 1, 1, 1, none)
DEF(     private_in, 1, 2, 1, none)
DEF(push_bigint_i32, 5, 0, 1, i32)
/* must be the last non short and non temporary opcode */
DEF(            nop, 1, 0, 0, none)

/* temporary opcodes: never emitted in the final bytecode */

def(    enter_scope, 3, 0, 0, u16)  /* emitted in phase 1, removed in phase 2 */
def(    leave_scope, 3, 0, 0, u16)  /* emitted in phase 1, removed in phase 2 */

def(          label, 5, 0, 0, label) /* emitted in phase 1, removed in phase 3 */

def(scope_get_var_undef, 7, 0, 1, atom_u16) /* emitted in phase 1, removed in phase 2 */
def(  scope_get_var, 7, 0, 1, atom_u16) /* emitted in phase 1, removed in phase 2 */
def(  scope_put_var, 7, 1, 0, atom_u16) /* emitted in phase 1, removed in phase 2 */
def(scope_delete_var, 7, 0, 1, atom_u16) /* emitted in phase 1, removed in phase 2 */
def( scope_make_ref, 11, 0, 2, atom_label_u16) /* emitted in phase 1, removed in phase 2 */
def(  scope_get_ref, 7, 0, 2, atom_u16) /* emitted in phase 1, removed in phase 2 */
def(scope_put_var_init, 7, 0, 2, atom_u16) /* emitted in phase 1, removed in phase 2 */
def(scope_get_private_field, 7, 1, 1, atom_u16) /* obj -> value, emitted in phase 1, removed in phase 2 */
def(scope_get_private_field2, 7, 1, 2, atom_u16) /* obj -> obj value, emitted in phase 1, removed in phase 2 */
def(scope_put_private_field, 7, 2, 0, atom_u16) /* obj value ->, emitted in phase 1, removed in phase 2 */
def(scope_in_private_field, 7, 1, 1, atom_u16) /* obj -> res emitted in phase 1, removed in phase 2 */
def(get_field_opt_chain, 5, 1, 1, atom) /* emitted in phase 1, removed in phase 2 */
def(get_array_el_opt_chain, 1, 2, 1, none) /* emitted in phase 1, removed in phase 2 */
def( set_class_name, 5, 1, 1, u32) /* emitted in phase 1, removed in phase 2 */

def(     source_loc, 9, 0, 0, u32x2) /* emitted in phase 1, removed in phase 3 */

DEF(    push_minus1, 1, 0, 1, none_int)
DEF(         push_0, 1, 0, 1, none_int)
DEF(         push_1, 1, 0, 1, none_int)
DEF(         push_2, 1, 0, 1, none_int)
DEF(         push_3, 1, 0, 1, none_int)
DEF(         push_4, 1, 0, 1, none_int)
DEF(         push_5, 1, 0, 1, none_int)
DEF(         push_6, 1, 0, 1, none_int)
DEF(         push_7, 1, 0, 1, none_int)
DEF(        push_i8, 2, 0, 1, i8)
DEF(       push_i16, 3, 0, 1, i16)
DEF(    push_const8, 2, 0, 1, const8)
DEF(      fclosure8, 2, 0, 1, const8) /* must follow push_const8 */
DEF(push_empty_string, 1, 0, 1, none)

DEF(       get_loc8, 2, 0, 1, loc8)
DEF(       put_loc8, 2, 1, 0, loc8)
DEF(       set_loc8, 2, 1, 1, loc8)

DEF(  get_loc0_loc1, 1, 0, 2, none_loc)
DEF(       get_loc0, 1, 0, 1, none_loc)
DEF(       get_loc1, 1, 0, 1, none_loc)
DEF(       get_loc2, 1, 0, 1, none_loc)
DEF(       get_loc3, 1, 0, 1, none_loc)
DEF(       put_loc0, 1, 1, 0, none_loc)
DEF(       put_loc1, 1, 1, 0, none_loc)
DEF(       put_loc2, 1, 1, 0, none_loc)
DEF(       put_loc3, 1, 1, 0, none_loc)
DEF(       set_loc0, 1, 1, 1, none_loc)
DEF(       set_loc1, 1, 1, 1, none_loc)
DEF(       set_loc2, 1, 1, 1, none_loc)
DEF(       set_loc3, 1, 1, 1, none_loc)
DEF(       get_arg0, 1, 0, 1, none_arg)
DEF(       get_arg1, 1, 0, 1, none_arg)
DEF(       get_arg2, 1, 0, 1, none_arg)
DEF(       get_arg3, 1, 0, 1, none_arg)
DEF(       put_arg0, 1, 1, 0, none_arg)
DEF(       put_arg1, 1, 1, 0, none_arg)
DEF(       put_arg2, 1, 1, 0, none_arg)
DEF(       put_arg3, 1, 1, 0, none_arg)
DEF(       set_arg0, 1, 1, 1, none_arg)
DEF(       set_arg1, 1, 1, 1, none_arg)
DEF(       set_arg2, 1, 1, 1, none_arg)
DEF(       set_arg3, 1, 1, 1, none_arg)
DEF(   get_var_ref0, 1, 0, 1, none_var_ref)
DEF(   get_var_ref1, 1, 0, 1, none_var_ref)
DEF(   get_var_ref2, 1, 0, 1, none_var_ref)
DEF(   get_var_ref3, 1, 0, 1, none_var_ref)
DEF(   put_var_ref0, 1, 1, 0, none_var_ref)
DEF(   put_var_ref1, 1, 1, 0, none_var_ref)
DEF(   put_var_ref2, 1, 1, 0, none_var_ref)
DEF(   put_var_ref3, 1, 1, 0, none_var_ref)
DEF(   set_var_ref0, 1, 1, 1, none_var_ref)
DEF(   set_var_ref1, 1, 1, 1, none_var_ref)
DEF(   set_var_ref2, 1, 1, 1, none_var_ref)
DEF(   set_var_ref3, 1, 1, 1, none_var_ref)

DEF(     get_length, 1, 1, 1, none)

DEF(      if_false8, 2, 1, 0, label8)
DEF(       if_true8, 2, 1, 0, label8) /* must come after if_false8 */
DEF(          goto8, 2, 0, 0, label8) /* must come after if_true8 */
DEF(         goto16, 3, 0, 0, label16)

DEF(          call0, 1, 1, 1, npopx)
DEF(          call1, 1, 1, 1, npopx)
DEF(          call2, 1, 1, 1, npopx)
DEF(          call3, 1, 1, 1, npopx)

DEF(   is_undefined, 1, 1, 1, none)
DEF(        is_null, 1, 1, 1, none)
DEF(typeof_is_undefined, 1, 1, 1, none)
DEF( typeof_is_function, 1, 1, 1, none)

#undef DEF
#undef def
#endif  /* DEF */

        [ OP_COUNT ... 255 ] = &&case_default
    };
#define SWITCH(pc)      DUMP_BYTECODE_OR_DONT(pc) __extension__ ({ goto *dispatch_table[opcode = *pc++]; });
#define CASE(op)        case_ ## op
#define DEFAULT         case_default
#define BREAK           SWITCH(pc)
#endif

    if (js_poll_interrupts(caller_ctx))
        return JS_EXCEPTION;
    if (unlikely(JS_VALUE_GET_TAG(func_obj) != JS_TAG_OBJECT)) {
        if (flags & JS_CALL_FLAG_GENERATOR) {
            JSAsyncFunctionState *s = JS_VALUE_GET_PTR(func_obj);
            /* func_obj get contains a pointer to JSFuncAsyncState */
            /* the stack frame is already allocated */
            sf = &s->frame;
            p = JS_VALUE_GET_OBJ(sf->cur_func);
            b = p->u.func.function_bytecode;
            ctx = b->realm;
            var_refs = p->u.func.var_refs;
            local_buf = arg_buf = sf->arg_buf;
            var_buf = sf->var_buf;
            stack_buf = sf->var_buf + b->var_count;
            sp = sf->cur_sp;
            sf->cur_sp = NULL; /* cur_sp is NULL if the function is running */
            pc = sf->cur_pc;
            sf->prev_frame = rt->current_stack_frame;
            rt->current_stack_frame = sf;
            if (s->throw_flag)
                goto exception;
            else
                goto restart;
        } else {
            goto not_a_function;
        }
    }
    p = JS_VALUE_GET_OBJ(func_obj);
    if (unlikely(p->class_id != JS_CLASS_BYTECODE_FUNCTION)) {
        JSClassCall *call_func;
        call_func = rt->class_array[p->class_id].call;
        if (!call_func) {
        not_a_function:
            return JS_ThrowTypeErrorNotAFunction(caller_ctx);
        }
        return call_func(caller_ctx, func_obj, this_obj, argc,
                         argv, flags);
    }
    b = p->u.func.function_bytecode;

    if (unlikely(argc < b->arg_count || (flags & JS_CALL_FLAG_COPY_ARGV))) {
        arg_allocated_size = b->arg_count;
    } else {
        arg_allocated_size = 0;
    }

    alloca_size = sizeof(JSValue) * (arg_allocated_size + b->var_count +
                                     b->stack_size);
    if (js_check_stack_overflow(rt, alloca_size))
        return JS_ThrowStackOverflow(caller_ctx);

    sf->is_strict_mode = b->is_strict_mode;
    arg_buf = (JSValue *)argv;
    sf->arg_count = argc;
    sf->cur_func = unsafe_unconst(func_obj);
    init_list_head(&sf->var_ref_list);
    var_refs = p->u.func.var_refs;

    local_buf = alloca(alloca_size);
    if (unlikely(arg_allocated_size)) {
        int n = min_int(argc, b->arg_count);
        arg_buf = local_buf;
        for(i = 0; i < n; i++)
            arg_buf[i] = js_dup(argv[i]);
        for(; i < b->arg_count; i++)
            arg_buf[i] = JS_UNDEFINED;
        sf->arg_count = b->arg_count;
    }
    var_buf = local_buf + arg_allocated_size;
    sf->var_buf = var_buf;
    sf->arg_buf = arg_buf;

    for(i = 0; i < b->var_count; i++)
        var_buf[i] = JS_UNDEFINED;

    stack_buf = var_buf + b->var_count;
    sp = stack_buf;
    pc = b->byte_code_buf;
    /* sf->cur_pc must we set to pc before any recursive calls to JS_CallInternal. */
    sf->cur_pc = NULL;
    sf->prev_frame = rt->current_stack_frame;
    rt->current_stack_frame = sf;
    ctx = b->realm; /* set the current realm */

#ifdef ENABLE_DUMPS // JS_DUMP_BYTECODE_STEP
    if (check_dump_flag(ctx->rt, JS_DUMP_BYTECODE_STEP))
        print_func_name(b);
#endif

 restart:
    for(;;) {
        int call_argc;
        JSValue *call_argv;

        SWITCH(pc) {
        CASE(OP_push_i32):
            *sp++ = js_int32(get_u32(pc));
            pc += 4;
            BREAK;
        CASE(OP_push_bigint_i32):
            *sp++ = __JS_NewShortBigInt(ctx, (int)get_u32(pc));
            pc += 4;
            BREAK;
        CASE(OP_push_const):
            *sp++ = js_dup(b->cpool[get_u32(pc)]);
            pc += 4;
            BREAK;
        CASE(OP_push_minus1):
        CASE(OP_push_0):
        CASE(OP_push_1):
        CASE(OP_push_2):
        CASE(OP_push_3):
        CASE(OP_push_4):
        CASE(OP_push_5):
        CASE(OP_push_6):
        CASE(OP_push_7):
            *sp++ = js_int32(opcode - OP_push_0);
            BREAK;
        CASE(OP_push_i8):
            *sp++ = js_int32(get_i8(pc));
            pc += 1;
            BREAK;
        CASE(OP_push_i16):
            *sp++ = js_int32(get_i16(pc));
            pc += 2;
            BREAK;
        CASE(OP_push_const8):
            *sp++ = js_dup(b->cpool[*pc++]);
            BREAK;
        CASE(OP_fclosure8):
            *sp++ = js_closure(ctx, js_dup(b->cpool[*pc++]), var_refs, sf);
            if (unlikely(JS_IsException(sp[-1])))
                goto exception;
            BREAK;
        CASE(OP_push_empty_string):
            *sp++ = js_empty_string(rt);
            BREAK;
        CASE(OP_get_length):
            {
                JSValue val;

                sf->cur_pc = pc;
                val = JS_GetProperty(ctx, sp[-1], JS_ATOM_length);
                if (unlikely(JS_IsException(val)))
                    goto exception;
                JS_FreeValue(ctx, sp[-1]);
                sp[-1] = val;
            }
            BREAK;
        CASE(OP_push_atom_value):
            *sp++ = JS_AtomToValue(ctx, get_u32(pc));
            pc += 4;
            BREAK;
        CASE(OP_undefined):
            *sp++ = JS_UNDEFINED;
            BREAK;
        CASE(OP_null):
            *sp++ = JS_NULL;
            BREAK;
        CASE(OP_push_this):
            /* OP_push_this is only called at the start of a function */
            {
                JSValue val;
                if (!b->is_strict_mode) {
                    uint32_t tag = JS_VALUE_GET_TAG(this_obj);
                    if (likely(tag == JS_TAG_OBJECT))
                        goto normal_this;
                    if (tag == JS_TAG_NULL || tag == JS_TAG_UNDEFINED) {
                        val = js_dup(ctx->global_obj);
                    } else {
                        val = JS_ToObject(ctx, this_obj);
                        if (JS_IsException(val))
                            goto exception;
                    }
                } else {
                normal_this:
                    val = js_dup(this_obj);
                }
                *sp++ = val;
            }
            BREAK;
        CASE(OP_push_false):
            *sp++ = JS_FALSE;
            BREAK;
        CASE(OP_push_true):
            *sp++ = JS_TRUE;
            BREAK;
        CASE(OP_object):
            *sp++ = JS_NewObject(ctx);
            if (unlikely(JS_IsException(sp[-1])))
                goto exception;
            BREAK;
        CASE(OP_special_object):
            {
                int arg = *pc++;
                switch(arg) {
                case OP_SPECIAL_OBJECT_ARGUMENTS:
                    *sp++ = js_build_arguments(ctx, argc, argv);
                    if (unlikely(JS_IsException(sp[-1])))
                        goto exception;
                    break;
                case OP_SPECIAL_OBJECT_MAPPED_ARGUMENTS:
                    *sp++ = js_build_mapped_arguments(ctx, argc, argv,
                                                      sf, min_int(argc, b->arg_count));
                    if (unlikely(JS_IsException(sp[-1])))
                        goto exception;
                    break;
                case OP_SPECIAL_OBJECT_THIS_FUNC:
                    *sp++ = js_dup(sf->cur_func);
                    break;
                case OP_SPECIAL_OBJECT_NEW_TARGET:
                    *sp++ = js_dup(new_target);
                    break;
                case OP_SPECIAL_OBJECT_HOME_OBJECT:
                    {
                        JSObject *p1;
                        p1 = p->u.func.home_object;
                        if (unlikely(!p1))
                            *sp++ = JS_UNDEFINED;
                        else
                            *sp++ = js_dup(JS_MKPTR(JS_TAG_OBJECT, p1));
                    }
                    break;
                case OP_SPECIAL_OBJECT_VAR_OBJECT:
                    *sp++ = JS_NewObjectProto(ctx, JS_NULL);
                    if (unlikely(JS_IsException(sp[-1])))
                        goto exception;
                    break;
                case OP_SPECIAL_OBJECT_IMPORT_META:
                    *sp++ = js_import_meta(ctx);
                    if (unlikely(JS_IsException(sp[-1])))
                        goto exception;
                    break;
                case OP_SPECIAL_OBJECT_NULL_PROTO:
                    *sp++ = JS_NewObjectProtoClass(ctx, JS_NULL, JS_CLASS_OBJECT);
                    if (unlikely(JS_IsException(sp[-1])))
                        goto exception;
                    break;
                default:
                    abort();
                }
            }
            BREAK;
        CASE(OP_rest):
            {
                int i, n, first = get_u16(pc);
                pc += 2;
                i = min_int(first, argc);
                n = argc - i;
                *sp++ = js_create_array(ctx, n, &argv[i]);
                if (unlikely(JS_IsException(sp[-1])))
                    goto exception;
            }
            BREAK;

        CASE(OP_drop):
            JS_FreeValue(ctx, sp[-1]);
            sp--;
            BREAK;
        CASE(OP_nip):
            JS_FreeValue(ctx, sp[-2]);
            sp[-2] = sp[-1];
            sp--;
            BREAK;
        CASE(OP_nip1): /* a b c -> b c */
            JS_FreeValue(ctx, sp[-3]);
            sp[-3] = sp[-2];
            sp[-2] = sp[-1];
            sp--;
            BREAK;
        CASE(OP_dup):
            sp[0] = js_dup(sp[-1]);
            sp++;
            BREAK;
        CASE(OP_dup2): /* a b -> a b a b */
            sp[0] = js_dup(sp[-2]);
            sp[1] = js_dup(sp[-1]);
            sp += 2;
            BREAK;
        CASE(OP_dup3): /* a b c -> a b c a b c */
            sp[0] = js_dup(sp[-3]);
            sp[1] = js_dup(sp[-2]);
            sp[2] = js_dup(sp[-1]);
            sp += 3;
            BREAK;
        CASE(OP_dup1): /* a b -> a a b */
            sp[0] = sp[-1];
            sp[-1] = js_dup(sp[-2]);
            sp++;
            BREAK;
        CASE(OP_insert2): /* obj a -> a obj a (dup_x1) */
            sp[0] = sp[-1];
            sp[-1] = sp[-2];
            sp[-2] = js_dup(sp[0]);
            sp++;
            BREAK;
        CASE(OP_insert3): /* obj prop a -> a obj prop a (dup_x2) */
            sp[0] = sp[-1];
            sp[-1] = sp[-2];
            sp[-2] = sp[-3];
            sp[-3] = js_dup(sp[0]);
            sp++;
            BREAK;
        CASE(OP_insert4): /* this obj prop a -> a this obj prop a */
            sp[0] = sp[-1];
            sp[-1] = sp[-2];
            sp[-2] = sp[-3];
            sp[-3] = sp[-4];
            sp[-4] = js_dup(sp[0]);
            sp++;
            BREAK;
        CASE(OP_perm3): /* obj a b -> a obj b (213) */
            {
                JSValue tmp;
                tmp = sp[-2];
                sp[-2] = sp[-3];
                sp[-3] = tmp;
            }
            BREAK;
        CASE(OP_rot3l): /* x a b -> a b x (231) */
            {
                JSValue tmp;
                tmp = sp[-3];
                sp[-3] = sp[-2];
                sp[-2] = sp[-1];
                sp[-1] = tmp;
            }
            BREAK;
        CASE(OP_rot4l): /* x a b c -> a b c x */
            {
                JSValue tmp;
                tmp = sp[-4];
                sp[-4] = sp[-3];
                sp[-3] = sp[-2];
                sp[-2] = sp[-1];
                sp[-1] = tmp;
            }
            BREAK;
        CASE(OP_rot5l): /* x a b c d -> a b c d x */
            {
                JSValue tmp;
                tmp = sp[-5];
                sp[-5] = sp[-4];
                sp[-4] = sp[-3];
                sp[-3] = sp[-2];
                sp[-2] = sp[-1];
                sp[-1] = tmp;
            }
            BREAK;
        CASE(OP_rot3r): /* a b x -> x a b (312) */
            {
                JSValue tmp;
                tmp = sp[-1];
                sp[-1] = sp[-2];
                sp[-2] = sp[-3];
                sp[-3] = tmp;
            }
            BREAK;
        CASE(OP_perm4): /* obj prop a b -> a obj prop b */
            {
                JSValue tmp;
                tmp = sp[-2];
                sp[-2] = sp[-3];
                sp[-3] = sp[-4];
                sp[-4] = tmp;
            }
            BREAK;
        CASE(OP_perm5): /* this obj prop a b -> a this obj prop b */
            {
                JSValue tmp;
                tmp = sp[-2];
                sp[-2] = sp[-3];
                sp[-3] = sp[-4];
                sp[-4] = sp[-5];
                sp[-5] = tmp;
            }
            BREAK;
        CASE(OP_swap): /* a b -> b a */
            {
                JSValue tmp;
                tmp = sp[-2];
                sp[-2] = sp[-1];
                sp[-1] = tmp;
            }
            BREAK;
        CASE(OP_swap2): /* a b c d -> c d a b */
            {
                JSValue tmp1, tmp2;
                tmp1 = sp[-4];
                tmp2 = sp[-3];
                sp[-4] = sp[-2];
                sp[-3] = sp[-1];
                sp[-2] = tmp1;
                sp[-1] = tmp2;
            }
            BREAK;

        CASE(OP_fclosure):
            {
                JSValue bfunc = js_dup(b->cpool[get_u32(pc)]);
                pc += 4;
                *sp++ = js_closure(ctx, bfunc, var_refs, sf);
                if (unlikely(JS_IsException(sp[-1])))
                    goto exception;
            }
            BREAK;
        CASE(OP_call0):
        CASE(OP_call1):
        CASE(OP_call2):
        CASE(OP_call3):
            call_argc = opcode - OP_call0;
            goto has_call_argc;
        CASE(OP_call):
        CASE(OP_tail_call):
            {
                call_argc = get_u16(pc);
                pc += 2;
                goto has_call_argc;
            has_call_argc:
                call_argv = sp - call_argc;
                sf->cur_pc = pc;
                ret_val = JS_CallInternal(ctx, call_argv[-1], JS_UNDEFINED,
                                          JS_UNDEFINED, call_argc,
                                          vc(call_argv), 0);
                if (unlikely(JS_IsException(ret_val)))
                    goto exception;
                if (opcode == OP_tail_call)
                    goto done;
                for(i = -1; i < call_argc; i++)
                    JS_FreeValue(ctx, call_argv[i]);
                sp -= call_argc + 1;
                *sp++ = ret_val;
            }
            BREAK;
        CASE(OP_call_constructor):
            {
                call_argc = get_u16(pc);
                pc += 2;
                call_argv = sp - call_argc;
                sf->cur_pc = pc;
                ret_val = JS_CallConstructorInternal(ctx, call_argv[-2],
                                                     call_argv[-1], call_argc,
                                                     vc(call_argv), 0);
                if (unlikely(JS_IsException(ret_val)))
                    goto exception;
                for(i = -2; i < call_argc; i++)
                    JS_FreeValue(ctx, call_argv[i]);
                sp -= call_argc + 2;
                *sp++ = ret_val;
            }
            BREAK;
        CASE(OP_call_method):
        CASE(OP_tail_call_method):
            {
                call_argc = get_u16(pc);
                pc += 2;
                call_argv = sp - call_argc;
                sf->cur_pc = pc;
                ret_val = JS_CallInternal(ctx, call_argv[-1], call_argv[-2],
                                          JS_UNDEFINED, call_argc,
                                          vc(call_argv), 0);
                if (unlikely(JS_IsException(ret_val)))
                    goto exception;
                if (opcode == OP_tail_call_method)
                    goto done;
                for(i = -2; i < call_argc; i++)
                    JS_FreeValue(ctx, call_argv[i]);
                sp -= call_argc + 2;
                *sp++ = ret_val;
            }
            BREAK;
        CASE(OP_array_from):
            {
                call_argc = get_u16(pc);
                pc += 2;
                call_argv = sp - call_argc;
                ret_val = JS_NewArrayFrom(ctx, call_argc, call_argv);
                sp -= call_argc;
                if (unlikely(JS_IsException(ret_val)))
                    goto exception;
                *sp++ = ret_val;
            }
            BREAK;

        CASE(OP_apply):
            {
                int magic;
                magic = get_u16(pc);
                pc += 2;
                sf->cur_pc = pc;

                ret_val = js_function_apply(ctx, sp[-3], 2, vc(&sp[-2]), magic);
                if (unlikely(JS_IsException(ret_val)))
                    goto exception;
                JS_FreeValue(ctx, sp[-3]);
                JS_FreeValue(ctx, sp[-2]);
                JS_FreeValue(ctx, sp[-1]);
                sp -= 3;
                *sp++ = ret_val;
            }
            BREAK;
        CASE(OP_return):
            ret_val = *--sp;
            goto done;
        CASE(OP_return_undef):
            ret_val = JS_UNDEFINED;
            goto done;

        CASE(OP_check_ctor_return):
            /* return true if 'this' should be returned */
            if (!JS_IsObject(sp[-1])) {
                if (!JS_IsUndefined(sp[-1])) {
                    JS_ThrowTypeError(caller_ctx, "derived class constructor must return an object or undefined");
                    goto exception;
                }
                sp[0] = JS_TRUE;
            } else {
                sp[0] = JS_FALSE;
            }
            sp++;
            BREAK;
        CASE(OP_check_ctor):
            if (JS_IsUndefined(new_target)) {
            non_ctor_call:
                JS_ThrowTypeError(ctx, "class constructors must be invoked with 'new'");
                goto exception;
            }
            BREAK;
        CASE(OP_init_ctor):
            {
                JSValue super, ret;
                sf->cur_pc = pc;
                if (JS_IsUndefined(new_target))
                    goto non_ctor_call;
                super = JS_GetPrototype(ctx, func_obj);
                if (JS_IsException(super))
                    goto exception;
                ret = JS_CallConstructor2(ctx, super, new_target, argc, argv);
                JS_FreeValue(ctx, super);
                if (JS_IsException(ret))
                    goto exception;
                *sp++ = ret;
            }
            BREAK;
        CASE(OP_check_brand):
            {
                int ret = JS_CheckBrand(ctx, sp[-2], sp[-1]);
                if (ret < 0)
                    goto exception;
                if (!ret) {
                    JS_ThrowTypeError(ctx, "invalid brand on object");
                    goto exception;
                }
            }
            BREAK;
        CASE(OP_add_brand):
            if (JS_AddBrand(ctx, sp[-2], sp[-1]) < 0)
                goto exception;
            JS_FreeValue(ctx, sp[-2]);
            JS_FreeValue(ctx, sp[-1]);
            sp -= 2;
            BREAK;

        CASE(OP_throw):
            JS_Throw(ctx, *--sp);
            goto exception;

        CASE(OP_throw_error):
#define JS_THROW_VAR_RO             0
#define JS_THROW_VAR_REDECL         1
#define JS_THROW_VAR_UNINITIALIZED  2
#define JS_THROW_ERROR_DELETE_SUPER   3
#define JS_THROW_ERROR_ITERATOR_THROW 4
            {
                JSAtom atom;
                int type;
                atom = get_u32(pc);
                type = pc[4];
                pc += 5;
                if (type == JS_THROW_VAR_RO)
                    JS_ThrowTypeErrorReadOnly(ctx, JS_PROP_THROW, atom);
                else
                if (type == JS_THROW_VAR_REDECL)
                    JS_ThrowSyntaxErrorVarRedeclaration(ctx, atom);
                else
                if (type == JS_THROW_VAR_UNINITIALIZED)
                    JS_ThrowReferenceErrorUninitialized(ctx, atom);
                else
                if (type == JS_THROW_ERROR_DELETE_SUPER)
                    JS_ThrowReferenceError(ctx, "unsupported reference to 'super'");
                else
                if (type == JS_THROW_ERROR_ITERATOR_THROW)
                    JS_ThrowTypeError(ctx, "iterator does not have a throw method");
                else
                    JS_ThrowInternalError(ctx, "invalid throw var type %d", type);
            }
            goto exception;

        CASE(OP_eval):
            {
                JSValue obj;
                int scope_idx;
                call_argc = get_u16(pc);
                scope_idx = get_u16(pc + 2) - 1;
                pc += 4;
                call_argv = sp - call_argc;
                sf->cur_pc = pc;
                if (js_same_value(ctx, call_argv[-1], ctx->eval_obj)) {
                    if (call_argc >= 1)
                        obj = call_argv[0];
                    else
                        obj = JS_UNDEFINED;
                    ret_val = JS_EvalObject(ctx, JS_UNDEFINED, obj,
                                            JS_EVAL_TYPE_DIRECT, scope_idx);
                } else {
                    ret_val = JS_CallInternal(ctx, call_argv[-1], JS_UNDEFINED,
                                              JS_UNDEFINED, call_argc,
                                              vc(call_argv), 0);
                }
                if (unlikely(JS_IsException(ret_val)))
                    goto exception;
                for(i = -1; i < call_argc; i++)
                    JS_FreeValue(ctx, call_argv[i]);
                sp -= call_argc + 1;
                *sp++ = ret_val;
            }
            BREAK;
            /* could merge with OP_apply */
        CASE(OP_apply_eval):
            {
                int scope_idx;
                uint32_t len;
                JSValue *tab;
                JSValue obj;

                scope_idx = get_u16(pc) - 1;
                pc += 2;
                sf->cur_pc = pc;
                tab = build_arg_list(ctx, &len, sp[-1]);
                if (!tab)
                    goto exception;
                if (js_same_value(ctx, sp[-2], ctx->eval_obj)) {
                    if (len >= 1)
                        obj = tab[0];
                    else
                        obj = JS_UNDEFINED;
                    ret_val = JS_EvalObject(ctx, JS_UNDEFINED, obj,
                                            JS_EVAL_TYPE_DIRECT, scope_idx);
                } else {
                    ret_val = JS_Call(ctx, sp[-2], JS_UNDEFINED, len, vc(tab));
                }
                free_arg_list(ctx, tab, len);
                if (unlikely(JS_IsException(ret_val)))
                    goto exception;
                JS_FreeValue(ctx, sp[-2]);
                JS_FreeValue(ctx, sp[-1]);
                sp -= 2;
                *sp++ = ret_val;
            }
            BREAK;

        CASE(OP_regexp):
            {
                sp[-2] = js_regexp_constructor_internal(ctx, JS_UNDEFINED,
                                                        sp[-2], sp[-1]);
                sp--;
            }
            BREAK;

        CASE(OP_get_super):
            {
                JSValue proto;
                proto = JS_GetPrototype(ctx, sp[-1]);
                if (JS_IsException(proto))
                    goto exception;
                JS_FreeValue(ctx, sp[-1]);
                sp[-1] = proto;
            }
            BREAK;

        CASE(OP_import):
            {
                JSValue val;
                sf->cur_pc = pc;
                val = js_dynamic_import(ctx, sp[-1]);
                if (JS_IsException(val))
                    goto exception;
                JS_FreeValue(ctx, sp[-1]);
                sp[-1] = val;
            }
            BREAK;

        CASE(OP_check_var):
            {
                int ret;
                JSAtom atom;
                atom = get_u32(pc);
                pc += 4;

                ret = JS_CheckGlobalVar(ctx, atom);
                if (ret < 0)
                    goto exception;
                *sp++ = js_bool(ret);
            }
            BREAK;

        CASE(OP_get_var_undef):
        CASE(OP_get_var):
            {
                JSValue val;
                JSAtom atom;
                atom = get_u32(pc);
                pc += 4;
                sf->cur_pc = pc;

                val = JS_GetGlobalVar(ctx, atom, opcode - OP_get_var_undef);
                if (unlikely(JS_IsException(val)))
                    goto exception;
                *sp++ = val;
            }
            BREAK;

        CASE(OP_put_var):
        CASE(OP_put_var_init):
            {
                int ret;
                JSAtom atom;
                atom = get_u32(pc);
                pc += 4;
                sf->cur_pc = pc;

                ret = JS_SetGlobalVar(ctx, atom, sp[-1], opcode - OP_put_var);
                sp--;
                if (unlikely(ret < 0))
                    goto exception;
            }
            BREAK;

        CASE(OP_put_var_strict):
            {
                int ret;
                JSAtom atom;
                atom = get_u32(pc);
                pc += 4;
                sf->cur_pc = pc;

                /* sp[-2] is JS_TRUE or JS_FALSE */
                if (unlikely(!JS_VALUE_GET_INT(sp[-2]))) {
                    JS_ThrowReferenceErrorNotDefined(ctx, atom);
                    goto exception;
                }
                ret = JS_SetGlobalVar(ctx, atom, sp[-1], 2);
                sp -= 2;
                if (unlikely(ret < 0))
                    goto exception;
            }
            BREAK;

        CASE(OP_check_define_var):
            {
                JSAtom atom;
                int flags;
                atom = get_u32(pc);
                flags = pc[4];
                pc += 5;
                if (JS_CheckDefineGlobalVar(ctx, atom, flags))
                    goto exception;
            }
            BREAK;
        CASE(OP_define_var):
            {
                JSAtom atom;
                int flags;
                atom = get_u32(pc);
                flags = pc[4];
                pc += 5;
                if (JS_DefineGlobalVar(ctx, atom, flags))
                    goto exception;
            }
            BREAK;
        CASE(OP_define_func):
            {
                JSAtom atom;
                int flags;
                atom = get_u32(pc);
                flags = pc[4];
                pc += 5;
                if (JS_DefineGlobalFunction(ctx, atom, sp[-1], flags))
                    goto exception;
                JS_FreeValue(ctx, sp[-1]);
                sp--;
            }
            BREAK;

        CASE(OP_get_loc):
            {
                int idx;
                idx = get_u16(pc);
                pc += 2;
                sp[0] = js_dup(var_buf[idx]);
                sp++;
            }
            BREAK;
        CASE(OP_put_loc):
            {
                int idx;
                idx = get_u16(pc);
                pc += 2;
                set_value(ctx, &var_buf[idx], sp[-1]);
                sp--;
            }
            BREAK;
        CASE(OP_set_loc):
            {
                int idx;
                idx = get_u16(pc);
                pc += 2;
                set_value(ctx, &var_buf[idx], js_dup(sp[-1]));
            }
            BREAK;
        CASE(OP_get_arg):
            {
                int idx;
                idx = get_u16(pc);
                pc += 2;
                sp[0] = js_dup(arg_buf[idx]);
                sp++;
            }
            BREAK;
        CASE(OP_put_arg):
            {
                int idx;
                idx = get_u16(pc);
                pc += 2;
                set_value(ctx, &arg_buf[idx], sp[-1]);
                sp--;
            }
            BREAK;
        CASE(OP_set_arg):
            {
                int idx;
                idx = get_u16(pc);
                pc += 2;
                set_value(ctx, &arg_buf[idx], js_dup(sp[-1]));
            }
            BREAK;

        CASE(OP_get_loc8): *sp++ = js_dup(var_buf[*pc++]); BREAK;
        CASE(OP_put_loc8): set_value(ctx, &var_buf[*pc++], *--sp); BREAK;
        CASE(OP_set_loc8): set_value(ctx, &var_buf[*pc++], js_dup(sp[-1])); BREAK;

        // Observation: get_loc0 and get_loc1 are individually very
        // frequent opcodes _and_ they are very often paired together,
        // making them ideal candidates for opcode fusion.
        CASE(OP_get_loc0_loc1):
            *sp++ = js_dup(var_buf[0]);
            *sp++ = js_dup(var_buf[1]);
            BREAK;

        CASE(OP_get_loc0): *sp++ = js_dup(var_buf[0]); BREAK;
        CASE(OP_get_loc1): *sp++ = js_dup(var_buf[1]); BREAK;
        CASE(OP_get_loc2): *sp++ = js_dup(var_buf[2]); BREAK;
        CASE(OP_get_loc3): *sp++ = js_dup(var_buf[3]); BREAK;
        CASE(OP_put_loc0): set_value(ctx, &var_buf[0], *--sp); BREAK;
        CASE(OP_put_loc1): set_value(ctx, &var_buf[1], *--sp); BREAK;
        CASE(OP_put_loc2): set_value(ctx, &var_buf[2], *--sp); BREAK;
        CASE(OP_put_loc3): set_value(ctx, &var_buf[3], *--sp); BREAK;
        CASE(OP_set_loc0): set_value(ctx, &var_buf[0], js_dup(sp[-1])); BREAK;
        CASE(OP_set_loc1): set_value(ctx, &var_buf[1], js_dup(sp[-1])); BREAK;
        CASE(OP_set_loc2): set_value(ctx, &var_buf[2], js_dup(sp[-1])); BREAK;
        CASE(OP_set_loc3): set_value(ctx, &var_buf[3], js_dup(sp[-1])); BREAK;
        CASE(OP_get_arg0): *sp++ = js_dup(arg_buf[0]); BREAK;
        CASE(OP_get_arg1): *sp++ = js_dup(arg_buf[1]); BREAK;
        CASE(OP_get_arg2): *sp++ = js_dup(arg_buf[2]); BREAK;
        CASE(OP_get_arg3): *sp++ = js_dup(arg_buf[3]); BREAK;
        CASE(OP_put_arg0): set_value(ctx, &arg_buf[0], *--sp); BREAK;
        CASE(OP_put_arg1): set_value(ctx, &arg_buf[1], *--sp); BREAK;
        CASE(OP_put_arg2): set_value(ctx, &arg_buf[2], *--sp); BREAK;
        CASE(OP_put_arg3): set_value(ctx, &arg_buf[3], *--sp); BREAK;
        CASE(OP_set_arg0): set_value(ctx, &arg_buf[0], js_dup(sp[-1])); BREAK;
        CASE(OP_set_arg1): set_value(ctx, &arg_buf[1], js_dup(sp[-1])); BREAK;
        CASE(OP_set_arg2): set_value(ctx, &arg_buf[2], js_dup(sp[-1])); BREAK;
        CASE(OP_set_arg3): set_value(ctx, &arg_buf[3], js_dup(sp[-1])); BREAK;
        CASE(OP_get_var_ref0): *sp++ = js_dup(*var_refs[0]->pvalue); BREAK;
        CASE(OP_get_var_ref1): *sp++ = js_dup(*var_refs[1]->pvalue); BREAK;
        CASE(OP_get_var_ref2): *sp++ = js_dup(*var_refs[2]->pvalue); BREAK;
        CASE(OP_get_var_ref3): *sp++ = js_dup(*var_refs[3]->pvalue); BREAK;
        CASE(OP_put_var_ref0): set_value(ctx, var_refs[0]->pvalue, *--sp); BREAK;
        CASE(OP_put_var_ref1): set_value(ctx, var_refs[1]->pvalue, *--sp); BREAK;
        CASE(OP_put_var_ref2): set_value(ctx, var_refs[2]->pvalue, *--sp); BREAK;
        CASE(OP_put_var_ref3): set_value(ctx, var_refs[3]->pvalue, *--sp); BREAK;
        CASE(OP_set_var_ref0): set_value(ctx, var_refs[0]->pvalue, js_dup(sp[-1])); BREAK;
        CASE(OP_set_var_ref1): set_value(ctx, var_refs[1]->pvalue, js_dup(sp[-1])); BREAK;
        CASE(OP_set_var_ref2): set_value(ctx, var_refs[2]->pvalue, js_dup(sp[-1])); BREAK;
        CASE(OP_set_var_ref3): set_value(ctx, var_refs[3]->pvalue, js_dup(sp[-1])); BREAK;

        CASE(OP_get_var_ref):
            {
                int idx;
                JSValue val;
                idx = get_u16(pc);
                pc += 2;
                val = *var_refs[idx]->pvalue;
                sp[0] = js_dup(val);
                sp++;
            }
            BREAK;
        CASE(OP_put_var_ref):
            {
                int idx;
                idx = get_u16(pc);
                pc += 2;
                set_value(ctx, var_refs[idx]->pvalue, sp[-1]);
                sp--;
            }
            BREAK;
        CASE(OP_set_var_ref):
            {
                int idx;
                idx = get_u16(pc);
                pc += 2;
                set_value(ctx, var_refs[idx]->pvalue, js_dup(sp[-1]));
            }
            BREAK;
        CASE(OP_get_var_ref_check):
            {
                int idx;
                JSValue val;
                idx = get_u16(pc);
                pc += 2;
                val = *var_refs[idx]->pvalue;
                if (unlikely(JS_IsUninitialized(val))) {
                    JS_ThrowReferenceErrorUninitialized2(ctx, b, idx, true);
                    goto exception;
                }
                sp[0] = js_dup(val);
                sp++;
            }
            BREAK;
        CASE(OP_put_var_ref_check):
            {
                int idx;
                idx = get_u16(pc);
                pc += 2;
                if (unlikely(JS_IsUninitialized(*var_refs[idx]->pvalue))) {
                    JS_ThrowReferenceErrorUninitialized2(ctx, b, idx, true);
                    goto exception;
                }
                set_value(ctx, var_refs[idx]->pvalue, sp[-1]);
                sp--;
            }
            BREAK;
        CASE(OP_put_var_ref_check_init):
            {
                int idx;
                idx = get_u16(pc);
                pc += 2;
                if (unlikely(!JS_IsUninitialized(*var_refs[idx]->pvalue))) {
                    JS_ThrowReferenceErrorUninitialized2(ctx, b, idx, true);
                    goto exception;
                }
                set_value(ctx, var_refs[idx]->pvalue, sp[-1]);
                sp--;
            }
            BREAK;
        CASE(OP_set_loc_uninitialized):
            {
                int idx;
                idx = get_u16(pc);
                pc += 2;
                set_value(ctx, &var_buf[idx], JS_UNINITIALIZED);
            }
            BREAK;
        CASE(OP_get_loc_check):
            {
                int idx;
                idx = get_u16(pc);
                pc += 2;
                if (unlikely(JS_IsUninitialized(var_buf[idx]))) {
                    JS_ThrowReferenceErrorUninitialized2(caller_ctx, b, idx,
                                                         false);
                    goto exception;
                }
                sp[0] = js_dup(var_buf[idx]);
                sp++;
            }
            BREAK;
        CASE(OP_put_loc_check):
            {
                int idx;
                idx = get_u16(pc);
                pc += 2;
                if (unlikely(JS_IsUninitialized(var_buf[idx]))) {
                    JS_ThrowReferenceErrorUninitialized2(caller_ctx, b, idx,
                                                         false);
                    goto exception;
                }
                set_value(ctx, &var_buf[idx], sp[-1]);
                sp--;
            }
            BREAK;
        CASE(OP_put_loc_check_init):
            {
                int idx;
                idx = get_u16(pc);
                pc += 2;
                if (unlikely(!JS_IsUninitialized(var_buf[idx]))) {
                    JS_ThrowReferenceError(caller_ctx,
                                           "'this' can be initialized only once");
                    goto exception;
                }
                set_value(ctx, &var_buf[idx], sp[-1]);
                sp--;
            }
            BREAK;
        CASE(OP_close_loc):
            {
                int idx;
                idx = get_u16(pc);
                pc += 2;
                close_lexical_var(ctx, sf, idx);
            }
            BREAK;

        CASE(OP_make_loc_ref):
        CASE(OP_make_arg_ref):
        CASE(OP_make_var_ref_ref):
            {
                JSVarRef *var_ref;
                JSProperty *pr;
                JSAtom atom;
                int idx;
                atom = get_u32(pc);
                idx = get_u16(pc + 4);
                pc += 6;
                *sp++ = JS_NewObjectProto(ctx, JS_NULL);
                if (unlikely(JS_IsException(sp[-1])))
                    goto exception;
                if (opcode == OP_make_var_ref_ref) {
                    var_ref = var_refs[idx];
                    var_ref->header.ref_count++;
                } else {
                    var_ref = get_var_ref(ctx, sf, idx, opcode == OP_make_arg_ref);
                    if (!var_ref)
                        goto exception;
                }
                pr = add_property(ctx, JS_VALUE_GET_OBJ(sp[-1]), atom,
                                  JS_PROP_WRITABLE | JS_PROP_VARREF);
                if (!pr) {
                    free_var_ref(rt, var_ref);
                    goto exception;
                }
                pr->u.var_ref = var_ref;
                *sp++ = JS_AtomToValue(ctx, atom);
            }
            BREAK;
        CASE(OP_make_var_ref):
            {
                JSAtom atom;
                atom = get_u32(pc);
                pc += 4;

                if (JS_GetGlobalVarRef(ctx, atom, sp))
                    goto exception;
                sp += 2;
            }
            BREAK;

        CASE(OP_goto):
            pc += (int32_t)get_u32(pc);
            if (unlikely(js_poll_interrupts(ctx)))
                goto exception;
            BREAK;
        CASE(OP_goto16):
            pc += (int16_t)get_u16(pc);
            if (unlikely(js_poll_interrupts(ctx)))
                goto exception;
            BREAK;
        CASE(OP_goto8):
            pc += (int8_t)pc[0];
            if (unlikely(js_poll_interrupts(ctx)))
                goto exception;
            BREAK;
        CASE(OP_if_true):
            {
                int res;
                JSValue op1;

                op1 = sp[-1];
                pc += 4;
                if ((uint32_t)JS_VALUE_GET_TAG(op1) <= JS_TAG_UNDEFINED) {
                    res = JS_VALUE_GET_INT(op1);
                } else {
                    res = JS_ToBoolFree(ctx, op1);
                }
                sp--;
                if (res) {
                    pc += (int32_t)get_u32(pc - 4) - 4;
                }
                if (unlikely(js_poll_interrupts(ctx)))
                    goto exception;
            }
            BREAK;
        CASE(OP_if_false):
            {
                int res;
                JSValue op1;

                op1 = sp[-1];
                pc += 4;
                if ((uint32_t)JS_VALUE_GET_TAG(op1) <= JS_TAG_UNDEFINED) {
                    res = JS_VALUE_GET_INT(op1);
                } else {
                    res = JS_ToBoolFree(ctx, op1);
                }
                sp--;
                if (!res) {
                    pc += (int32_t)get_u32(pc - 4) - 4;
                }
                if (unlikely(js_poll_interrupts(ctx)))
                    goto exception;
            }
            BREAK;
        CASE(OP_if_true8):
            {
                int res;
                JSValue op1;

                op1 = sp[-1];
                pc += 1;
                if ((uint32_t)JS_VALUE_GET_TAG(op1) <= JS_TAG_UNDEFINED) {
                    res = JS_VALUE_GET_INT(op1);
                } else {
                    res = JS_ToBoolFree(ctx, op1);
                }
                sp--;
                if (res) {
                    pc += (int8_t)pc[-1] - 1;
                }
                if (unlikely(js_poll_interrupts(ctx)))
                    goto exception;
            }
            BREAK;
        CASE(OP_if_false8):
            {
                int res;
                JSValue op1;

                op1 = sp[-1];
                pc += 1;
                if ((uint32_t)JS_VALUE_GET_TAG(op1) <= JS_TAG_UNDEFINED) {
                    res = JS_VALUE_GET_INT(op1);
                } else {
                    res = JS_ToBoolFree(ctx, op1);
                }
                sp--;
                if (!res) {
                    pc += (int8_t)pc[-1] - 1;
                }
                if (unlikely(js_poll_interrupts(ctx)))
                    goto exception;
            }
            BREAK;
        CASE(OP_catch):
            {
                int32_t diff;
                diff = get_u32(pc);
                sp[0] = JS_NewCatchOffset(ctx, pc + diff - b->byte_code_buf);
                sp++;
                pc += 4;
            }
            BREAK;
        CASE(OP_gosub):
            {
                int32_t diff;
                diff = get_u32(pc);
                /* XXX: should have a different tag to avoid security flaw */
                sp[0] = js_int32(pc + 4 - b->byte_code_buf);
                sp++;
                pc += diff;
            }
            BREAK;
        CASE(OP_ret):
            {
                JSValue op1;
                uint32_t pos;
                op1 = sp[-1];
                if (unlikely(JS_VALUE_GET_TAG(op1) != JS_TAG_INT))
                    goto ret_fail;
                pos = JS_VALUE_GET_INT(op1);
                if (unlikely(pos >= b->byte_code_len)) {
                ret_fail:
                    JS_ThrowInternalError(ctx, "invalid ret value");
                    goto exception;
                }
                sp--;
                pc = b->byte_code_buf + pos;
            }
            BREAK;

        CASE(OP_for_in_start):
            sf->cur_pc = pc;
            if (js_for_in_start(ctx, sp))
                goto exception;
            BREAK;
        CASE(OP_for_in_next):
            sf->cur_pc = pc;
            if (js_for_in_next(ctx, sp))
                goto exception;
            sp += 2;
            BREAK;
        CASE(OP_for_of_start):
            sf->cur_pc = pc;
            if (js_for_of_start(ctx, sp, false))
                goto exception;
            sp += 1;
            *sp++ = JS_NewCatchOffset(ctx, 0);
            BREAK;
        CASE(OP_for_of_next):
            {
                int offset = -3 - pc[0];
                pc += 1;
                sf->cur_pc = pc;
                if (js_for_of_next(ctx, sp, offset))
                    goto exception;
                sp += 2;
            }
            BREAK;
        CASE(OP_for_await_of_start):
            sf->cur_pc = pc;
            if (js_for_of_start(ctx, sp, true))
                goto exception;
            sp += 1;
            *sp++ = JS_NewCatchOffset(ctx, 0);
            BREAK;
        CASE(OP_iterator_get_value_done):
            sf->cur_pc = pc;
            if (js_iterator_get_value_done(ctx, sp))
                goto exception;
            sp += 1;
            BREAK;
        CASE(OP_iterator_check_object):
            if (unlikely(!JS_IsObject(sp[-1]))) {
                JS_ThrowTypeError(ctx, "iterator must return an object");
                goto exception;
            }
            BREAK;

        CASE(OP_iterator_close):
            /* iter_obj next catch_offset -> */
            sp--; /* drop the catch offset to avoid getting caught by exception */
            JS_FreeValue(ctx, sp[-1]); /* drop the next method */
            sp--;
            if (!JS_IsUndefined(sp[-1])) {
                sf->cur_pc = pc;
                if (JS_IteratorClose(ctx, sp[-1], false))
                    goto exception;
                JS_FreeValue(ctx, sp[-1]);
            }
            sp--;
            BREAK;
        CASE(OP_nip_catch):
            {
                JSValue ret_val;
                /* catch_offset ... ret_val -> ret_eval */
                ret_val = *--sp;
                while (sp > stack_buf &&
                       JS_VALUE_GET_TAG(sp[-1]) != JS_TAG_CATCH_OFFSET) {
                    JS_FreeValue(ctx, *--sp);
                }
                if (unlikely(sp == stack_buf)) {
                    JS_ThrowInternalError(ctx, "nip_catch");
                    JS_FreeValue(ctx, ret_val);
                    goto exception;
                }
                sp[-1] = ret_val;
            }
            BREAK;

        CASE(OP_iterator_next):
            /* stack: iter_obj next catch_offset val */
            {
                JSValue ret;
                sf->cur_pc = pc;
                ret = JS_Call(ctx, sp[-3], sp[-4], 1, vc(sp - 1));
                if (JS_IsException(ret))
                    goto exception;
                JS_FreeValue(ctx, sp[-1]);
                sp[-1] = ret;
            }
            BREAK;

        CASE(OP_iterator_call):
            /* stack: iter_obj next catch_offset val */
            {
                JSValue method, ret;
                bool ret_flag;
                int flags;
                flags = *pc++;
                sf->cur_pc = pc;
                method = JS_GetProperty(ctx, sp[-4], (flags & 1) ?
                                        JS_ATOM_throw : JS_ATOM_return);
                if (JS_IsException(method))
                    goto exception;
                if (JS_IsUndefined(method) || JS_IsNull(method)) {
                    ret_flag = true;
                } else {
                    if (flags & 2) {
                        /* no argument */
                        ret = JS_CallFree(ctx, method, sp[-4],
                                          0, NULL);
                    } else {
                        ret = JS_CallFree(ctx, method, sp[-4],
                                          1, vc(sp - 1));
                    }
                    if (JS_IsException(ret))
                        goto exception;
                    JS_FreeValue(ctx, sp[-1]);
                    sp[-1] = ret;
                    ret_flag = false;
                }
                sp[0] = js_bool(ret_flag);
                sp += 1;
            }
            BREAK;

        CASE(OP_lnot):
            {
                int res;
                JSValue op1;

                op1 = sp[-1];
                if ((uint32_t)JS_VALUE_GET_TAG(op1) <= JS_TAG_UNDEFINED) {
                    res = JS_VALUE_GET_INT(op1) != 0;
                } else {
                    res = JS_ToBoolFree(ctx, op1);
                }
                sp[-1] = js_bool(!res);
            }
            BREAK;

        CASE(OP_get_field):
            {
                JSValue val;
                JSAtom atom;
                atom = get_u32(pc);
                pc += 4;
                sf->cur_pc = pc;
                val = JS_GetPropertyInternal(ctx, sp[-1], atom, sp[-1], false);
                if (unlikely(JS_IsException(val)))
                    goto exception;
                JS_FreeValue(ctx, sp[-1]);
                sp[-1] = val;
            }
            BREAK;

        CASE(OP_get_field2):
            {
                JSValue val;
                JSAtom atom;
                atom = get_u32(pc);
                pc += 4;
                sf->cur_pc = pc;
                val = JS_GetPropertyInternal(ctx, sp[-1], atom, sp[-1], false);
                if (unlikely(JS_IsException(val)))
                    goto exception;
                *sp++ = val;
          }
          BREAK;

        CASE(OP_put_field):
            {
                int ret;
                JSAtom atom;
                atom = get_u32(pc);
                pc += 4;
                sf->cur_pc = pc;
                ret = JS_SetPropertyInternal2(ctx,
                                              sp[-2], atom,
                                              sp[-1], sp[-2],
                                              JS_PROP_THROW_STRICT);
                JS_FreeValue(ctx, sp[-2]);
                sp -= 2;
                if (unlikely(ret < 0))
                    goto exception;
            }
            BREAK;

        CASE(OP_private_symbol):
            {
                JSAtom atom;
                JSValue val;

                atom = get_u32(pc);
                pc += 4;
                val = JS_NewSymbolFromAtom(ctx, atom, JS_ATOM_TYPE_PRIVATE);
                if (JS_IsException(val))
                    goto exception;
                *sp++ = val;
            }
            BREAK;

        CASE(OP_get_private_field):
            {
                JSValue val;
                sf->cur_pc = pc;
                val = JS_GetPrivateField(ctx, sp[-2], sp[-1]);
                JS_FreeValue(ctx, sp[-1]);
                JS_FreeValue(ctx, sp[-2]);
                sp[-2] = val;
                sp--;
                if (unlikely(JS_IsException(val)))
                    goto exception;
            }
            BREAK;

        CASE(OP_put_private_field):
            {
                int ret;
                sf->cur_pc = pc;
                ret = JS_SetPrivateField(ctx, sp[-3], sp[-1], sp[-2]);
                JS_FreeValue(ctx, sp[-3]);
                JS_FreeValue(ctx, sp[-1]);
                sp -= 3;
                if (unlikely(ret < 0))
                    goto exception;
            }
            BREAK;

        CASE(OP_define_private_field):
            {
                int ret;
                ret = JS_DefinePrivateField(ctx, sp[-3], sp[-2], sp[-1]);
                JS_FreeValue(ctx, sp[-2]);
                sp -= 2;
                if (unlikely(ret < 0))
                    goto exception;
            }
            BREAK;

        CASE(OP_define_field):
            {
                int ret;
                JSAtom atom;
                atom = get_u32(pc);
                pc += 4;

                ret = JS_DefinePropertyValue(ctx, sp[-2], atom, sp[-1],
                                             JS_PROP_C_W_E | JS_PROP_THROW);
                sp--;
                if (unlikely(ret < 0))
                    goto exception;
            }
            BREAK;

        CASE(OP_set_name):
            {
                int ret;
                JSAtom atom;
                atom = get_u32(pc);
                pc += 4;

                ret = JS_DefineObjectName(ctx, sp[-1], atom, JS_PROP_CONFIGURABLE);
                if (unlikely(ret < 0))
                    goto exception;
            }
            BREAK;
        CASE(OP_set_name_computed):
            {
                int ret;
                ret = JS_DefineObjectNameComputed(ctx, sp[-1], sp[-2], JS_PROP_CONFIGURABLE);
                if (unlikely(ret < 0))
                    goto exception;
            }
            BREAK;
        CASE(OP_set_proto):
            {
                JSValue proto;
                proto = sp[-1];
                if (JS_IsObject(proto) || JS_IsNull(proto)) {
                    if (JS_SetPrototypeInternal(ctx, sp[-2], proto, true) < 0)
                        goto exception;
                }
                JS_FreeValue(ctx, proto);
                sp--;
            }
            BREAK;
        CASE(OP_set_home_object):
            js_method_set_home_object(ctx, sp[-1], sp[-2]);
            BREAK;
        CASE(OP_define_method):
        CASE(OP_define_method_computed):
            {
                JSValue getter, setter, value;
                JSValue obj;
                JSAtom atom;
                int flags, ret, op_flags;
                bool is_computed;
#define OP_DEFINE_METHOD_METHOD 0
#define OP_DEFINE_METHOD_GETTER 1
#define OP_DEFINE_METHOD_SETTER 2
#define OP_DEFINE_METHOD_ENUMERABLE 4

                is_computed = (opcode == OP_define_method_computed);
                if (is_computed) {
                    atom = JS_ValueToAtom(ctx, sp[-2]);
                    if (unlikely(atom == JS_ATOM_NULL))
                        goto exception;
                    opcode += OP_define_method - OP_define_method_computed;
                } else {
                    atom = get_u32(pc);
                    pc += 4;
                }
                op_flags = *pc++;

                obj = sp[-2 - is_computed];
                flags = JS_PROP_HAS_CONFIGURABLE | JS_PROP_CONFIGURABLE |
                    JS_PROP_HAS_ENUMERABLE | JS_PROP_THROW;
                if (op_flags & OP_DEFINE_METHOD_ENUMERABLE)
                    flags |= JS_PROP_ENUMERABLE;
                op_flags &= 3;
                value = JS_UNDEFINED;
                getter = JS_UNDEFINED;
                setter = JS_UNDEFINED;
                if (op_flags == OP_DEFINE_METHOD_METHOD) {
                    value = sp[-1];
                    flags |= JS_PROP_HAS_VALUE | JS_PROP_HAS_WRITABLE | JS_PROP_WRITABLE;
                } else if (op_flags == OP_DEFINE_METHOD_GETTER) {
                    getter = sp[-1];
                    flags |= JS_PROP_HAS_GET;
                } else {
                    setter = sp[-1];
                    flags |= JS_PROP_HAS_SET;
                }
                ret = js_method_set_properties(ctx, sp[-1], atom, flags, obj);
                if (ret >= 0) {
                    ret = JS_DefineProperty(ctx, obj, atom, value,
                                            getter, setter, flags);
                }
                JS_FreeValue(ctx, sp[-1]);
                if (is_computed) {
                    JS_FreeAtom(ctx, atom);
                    JS_FreeValue(ctx, sp[-2]);
                }
                sp -= 1 + is_computed;
                if (unlikely(ret < 0))
                    goto exception;
            }
            BREAK;

        CASE(OP_define_class):
        CASE(OP_define_class_computed):
            {
                int class_flags;
                JSAtom atom;

                atom = get_u32(pc);
                class_flags = pc[4];
                pc += 5;
                if (js_op_define_class(ctx, sp, atom, class_flags,
                                       var_refs, sf,
                                       (opcode == OP_define_class_computed)) < 0)
                    goto exception;
            }
            BREAK;

        CASE(OP_get_array_el):
            {
                JSValue val;

                sf->cur_pc = pc;
                val = JS_GetPropertyValue(ctx, sp[-2], sp[-1]);
                JS_FreeValue(ctx, sp[-2]);
                sp[-2] = val;
                sp--;
                if (unlikely(JS_IsException(val)))
                    goto exception;
            }
            BREAK;

        CASE(OP_get_array_el2):
            {
                JSValue val;

                sf->cur_pc = pc;
                val = JS_GetPropertyValue(ctx, sp[-2], sp[-1]);
                sp[-1] = val;
                if (unlikely(JS_IsException(val)))
                    goto exception;
            }
            BREAK;

        CASE(OP_get_ref_value):
            {
                JSValue val;
                sf->cur_pc = pc;
                if (unlikely(JS_IsUndefined(sp[-2]))) {
                    JSAtom atom = JS_ValueToAtom(ctx, sp[-1]);
                    if (atom != JS_ATOM_NULL) {
                        JS_ThrowReferenceErrorNotDefined(ctx, atom);
                        JS_FreeAtom(ctx, atom);
                    }
                    goto exception;
                }
                val = JS_GetPropertyValue(ctx, sp[-2],
                                          js_dup(sp[-1]));
                if (unlikely(JS_IsException(val)))
                    goto exception;
                sp[0] = val;
                sp++;
            }
            BREAK;

        CASE(OP_get_super_value):
            {
                JSValue val;
                JSAtom atom;
                sf->cur_pc = pc;
                atom = JS_ValueToAtom(ctx, sp[-1]);
                if (unlikely(atom == JS_ATOM_NULL))
                    goto exception;
                val = JS_GetPropertyInternal(ctx, sp[-2], atom, sp[-3], false);
                JS_FreeAtom(ctx, atom);
                if (unlikely(JS_IsException(val)))
                    goto exception;
                JS_FreeValue(ctx, sp[-1]);
                JS_FreeValue(ctx, sp[-2]);
                JS_FreeValue(ctx, sp[-3]);
                sp[-3] = val;
                sp -= 2;
            }
            BREAK;

        CASE(OP_put_array_el):
            {
                int ret;
                sf->cur_pc = pc;
                ret = JS_SetPropertyValue(ctx, sp[-3], sp[-2], sp[-1], JS_PROP_THROW_STRICT);
                JS_FreeValue(ctx, sp[-3]);
                sp -= 3;
                if (unlikely(ret < 0))
                    goto exception;
            }
            BREAK;

        CASE(OP_put_ref_value):
            {
                int ret, flags;
                sf->cur_pc = pc;
                flags = JS_PROP_THROW_STRICT;
                if (unlikely(JS_IsUndefined(sp[-3]))) {
                    if (is_strict_mode(ctx)) {
                        JSAtom atom = JS_ValueToAtom(ctx, sp[-2]);
                        if (atom != JS_ATOM_NULL) {
                            JS_ThrowReferenceErrorNotDefined(ctx, atom);
                            JS_FreeAtom(ctx, atom);
                        }
                        goto exception;
                    } else {
                        sp[-3] = js_dup(ctx->global_obj);
                    }
                } else {
                    if (is_strict_mode(ctx))
                        flags |= JS_PROP_NO_ADD;
                }
                ret = JS_SetPropertyValue(ctx, sp[-3], sp[-2], sp[-1], flags);
                JS_FreeValue(ctx, sp[-3]);
                sp -= 3;
                if (unlikely(ret < 0))
                    goto exception;
            }
            BREAK;

        CASE(OP_put_super_value):
            {
                int ret;
                JSAtom atom;
                sf->cur_pc = pc;
                if (JS_VALUE_GET_TAG(sp[-3]) != JS_TAG_OBJECT) {
                    JS_ThrowTypeErrorNotAnObject(ctx);
                    goto exception;
                }
                atom = JS_ValueToAtom(ctx, sp[-2]);
                if (unlikely(atom == JS_ATOM_NULL))
                    goto exception;
                ret = JS_SetPropertyInternal2(ctx,
                                              sp[-3], atom,
                                              sp[-1], sp[-4],
                                              JS_PROP_THROW_STRICT);
                JS_FreeAtom(ctx, atom);
                JS_FreeValue(ctx, sp[-4]);
                JS_FreeValue(ctx, sp[-3]);
                JS_FreeValue(ctx, sp[-2]);
                sp -= 4;
                if (ret < 0)
                    goto exception;
            }
            BREAK;

        CASE(OP_define_array_el):
            {
                int ret;
                ret = JS_DefinePropertyValueValue(ctx, sp[-3], js_dup(sp[-2]), sp[-1],
                                                  JS_PROP_C_W_E | JS_PROP_THROW);
                sp -= 1;
                if (unlikely(ret < 0))
                    goto exception;
            }
            BREAK;

        CASE(OP_append):    /* array pos enumobj -- array pos */
            {
                sf->cur_pc = pc;
                if (js_append_enumerate(ctx, sp))
                    goto exception;
                JS_FreeValue(ctx, *--sp);
            }
            BREAK;

        CASE(OP_copy_data_properties):    /* target source excludeList */
            {
                /* stack offsets (-1 based):
                   2 bits for target,
                   3 bits for source,
                   2 bits for exclusionList */
                int mask;

                mask = *pc++;
                sf->cur_pc = pc;
                if (JS_CopyDataProperties(ctx, sp[-1 - (mask & 3)],
                                          sp[-1 - ((mask >> 2) & 7)],
                                          sp[-1 - ((mask >> 5) & 7)], 0))
                    goto exception;
            }
            BREAK;

        CASE(OP_add):
            {
                JSValue op1, op2;
                op1 = sp[-2];
                op2 = sp[-1];
                if (likely(JS_VALUE_IS_BOTH_INT(op1, op2))) {
                    int64_t r;
                    r = (int64_t)JS_VALUE_GET_INT(op1) + JS_VALUE_GET_INT(op2);
                    if (unlikely((int)r != r))
                        goto add_slow;
                    sp[-2] = js_int32(r);
                    sp--;
                } else if (JS_VALUE_IS_BOTH_FLOAT(op1, op2)) {
                    sp[-2] = js_float64(JS_VALUE_GET_FLOAT64(op1) +
                                        JS_VALUE_GET_FLOAT64(op2));
                    sp--;
                } else {
                add_slow:
                    sf->cur_pc = pc;
                    if (js_add_slow(ctx, sp))
                        goto exception;
                    sp--;
                }
            }
            BREAK;
        CASE(OP_add_loc):
            {
                JSValue *pv;
                int idx;
                idx = *pc;
                pc += 1;

                pv = &var_buf[idx];
                if (likely(JS_VALUE_IS_BOTH_INT(*pv, sp[-1]))) {
                    int64_t r;
                    r = (int64_t)JS_VALUE_GET_INT(*pv) +
                        JS_VALUE_GET_INT(sp[-1]);
                    if (unlikely((int)r != r))
                        goto add_loc_slow;
                    *pv = js_int32(r);
                    sp--;
                } else if (JS_VALUE_GET_TAG(*pv) == JS_TAG_STRING) {
                    JSValue op1;
                    op1 = sp[-1];
                    sp--;
                    sf->cur_pc = pc;
                    op1 = JS_ToPrimitiveFree(ctx, op1, HINT_NONE);
                    if (JS_IsException(op1))
                        goto exception;
                    op1 = JS_ConcatString(ctx, js_dup(*pv), op1);
                    if (JS_IsException(op1))
                        goto exception;
                    set_value(ctx, pv, op1);
                } else {
                    JSValue ops[2];
                add_loc_slow:
                    /* In case of exception, js_add_slow frees ops[0]
                       and ops[1], so we must duplicate *pv */
                    sf->cur_pc = pc;
                    ops[0] = js_dup(*pv);
                    ops[1] = sp[-1];
                    sp--;
                    if (js_add_slow(ctx, ops + 2))
                        goto exception;
                    set_value(ctx, pv, ops[0]);
                }
            }
            BREAK;
        CASE(OP_sub):
            {
                JSValue op1, op2;
                op1 = sp[-2];
                op2 = sp[-1];
                if (likely(JS_VALUE_IS_BOTH_INT(op1, op2))) {
                    int64_t r;
                    r = (int64_t)JS_VALUE_GET_INT(op1) - JS_VALUE_GET_INT(op2);
                    if (unlikely((int)r != r))
                        goto binary_arith_slow;
                    sp[-2] = js_int32(r);
                    sp--;
                } else if (JS_VALUE_IS_BOTH_FLOAT(op1, op2)) {
                    sp[-2] = js_float64(JS_VALUE_GET_FLOAT64(op1) -
                                        JS_VALUE_GET_FLOAT64(op2));
                    sp--;
                } else {
                    goto binary_arith_slow;
                }
            }
            BREAK;
        CASE(OP_mul):
            {
                JSValue op1, op2;
                double d;
                op1 = sp[-2];
                op2 = sp[-1];
                if (likely(JS_VALUE_IS_BOTH_INT(op1, op2))) {
                    int32_t v1, v2;
                    int64_t r;
                    v1 = JS_VALUE_GET_INT(op1);
                    v2 = JS_VALUE_GET_INT(op2);
                    r = (int64_t)v1 * v2;
                    if (unlikely((int)r != r)) {
                        d = (double)r;
                        goto mul_fp_res;
                    }
                    /* need to test zero case for -0 result */
                    if (unlikely(r == 0 && (v1 | v2) < 0)) {
                        d = -0.0;
                        goto mul_fp_res;
                    }
                    sp[-2] = js_int32(r);
                    sp--;
                } else if (JS_VALUE_IS_BOTH_FLOAT(op1, op2)) {
                    d = JS_VALUE_GET_FLOAT64(op1) * JS_VALUE_GET_FLOAT64(op2);
                mul_fp_res:
                    sp[-2] = js_float64(d);
                    sp--;
                } else {
                    goto binary_arith_slow;
                }
            }
            BREAK;
        CASE(OP_div):
            {
                JSValue op1, op2;
                op1 = sp[-2];
                op2 = sp[-1];
                if (likely(JS_VALUE_IS_BOTH_INT(op1, op2))) {
                    int v1, v2;
                    v1 = JS_VALUE_GET_INT(op1);
                    v2 = JS_VALUE_GET_INT(op2);
                    sp[-2] = js_number((double)v1 / (double)v2);
                    sp--;
                } else {
                    goto binary_arith_slow;
                }
            }
            BREAK;
        CASE(OP_mod):
            {
                JSValue op1, op2;
                op1 = sp[-2];
                op2 = sp[-1];
                if (likely(JS_VALUE_IS_BOTH_INT(op1, op2))) {
                    int v1, v2, r;
                    v1 = JS_VALUE_GET_INT(op1);
                    v2 = JS_VALUE_GET_INT(op2);
                    /* We must avoid v2 = 0, v1 = INT32_MIN and v2 =
                       -1 and the cases where the result is -0. */
                    if (unlikely(v1 < 0 || v2 <= 0))
                        goto binary_arith_slow;
                    r = v1 % v2;
                    sp[-2] = js_int32(r);
                    sp--;
                } else {
                    goto binary_arith_slow;
                }
            }
            BREAK;
        CASE(OP_pow):
        binary_arith_slow:
            sf->cur_pc = pc;
            if (js_binary_arith_slow(ctx, sp, opcode))
                goto exception;
            sp--;
            BREAK;

        CASE(OP_plus):
            {
                JSValue op1;
                uint32_t tag;
                op1 = sp[-1];
                tag = JS_VALUE_GET_TAG(op1);
                if (tag == JS_TAG_INT || JS_TAG_IS_FLOAT64(tag)) {
                } else {
                    sf->cur_pc = pc;
                    if (js_unary_arith_slow(ctx, sp, opcode))
                        goto exception;
                }
            }
            BREAK;
        CASE(OP_neg):
            {
                JSValue op1;
                uint32_t tag;
                int val;
                double d;
                op1 = sp[-1];
                tag = JS_VALUE_GET_TAG(op1);
                if (tag == JS_TAG_INT) {
                    val = JS_VALUE_GET_INT(op1);
                    /* Note: -0 cannot be expressed as integer */
                    if (unlikely(val == 0)) {
                        d = -0.0;
                        goto neg_fp_res;
                    }
                    if (unlikely(val == INT32_MIN)) {
                        d = -(double)val;
                        goto neg_fp_res;
                    }
                    sp[-1] = js_int32(-val);
                } else if (JS_TAG_IS_FLOAT64(tag)) {
                    d = -JS_VALUE_GET_FLOAT64(op1);
                neg_fp_res:
                    sp[-1] = js_float64(d);
                } else {
                    sf->cur_pc = pc;
                    if (js_unary_arith_slow(ctx, sp, opcode))
                        goto exception;
                }
            }
            BREAK;
        CASE(OP_inc):
            {
                JSValue op1;
                int val;
                op1 = sp[-1];
                if (JS_VALUE_GET_TAG(op1) == JS_TAG_INT) {
                    val = JS_VALUE_GET_INT(op1);
                    if (unlikely(val == INT32_MAX))
                        goto inc_slow;
                    sp[-1] = js_int32(val + 1);
                } else {
                inc_slow:
                    sf->cur_pc = pc;
                    if (js_unary_arith_slow(ctx, sp, opcode))
                        goto exception;
                }
            }
            BREAK;
        CASE(OP_dec):
            {
                JSValue op1;
                int val;
                op1 = sp[-1];
                if (JS_VALUE_GET_TAG(op1) == JS_TAG_INT) {
                    val = JS_VALUE_GET_INT(op1);
                    if (unlikely(val == INT32_MIN))
                        goto dec_slow;
                    sp[-1] = js_int32(val - 1);
                } else {
                dec_slow:
                    sf->cur_pc = pc;
                    if (js_unary_arith_slow(ctx, sp, opcode))
                        goto exception;
                }
            }
            BREAK;
        CASE(OP_post_inc):
        CASE(OP_post_dec):
            sf->cur_pc = pc;
            if (js_post_inc_slow(ctx, sp, opcode))
                goto exception;
            sp++;
            BREAK;
        CASE(OP_inc_loc):
            {
                JSValue op1;
                int val;
                int idx;
                idx = *pc;
                pc += 1;

                op1 = var_buf[idx];
                if (JS_VALUE_GET_TAG(op1) == JS_TAG_INT) {
                    val = JS_VALUE_GET_INT(op1);
                    if (unlikely(val == INT32_MAX))
                        goto inc_loc_slow;
                    var_buf[idx] = js_int32(val + 1);
                } else {
                inc_loc_slow:
                    sf->cur_pc = pc;
                    /* must duplicate otherwise the variable value may
                       be destroyed before JS code accesses it */
                    op1 = js_dup(op1);
                    if (js_unary_arith_slow(ctx, &op1 + 1, OP_inc))
                        goto exception;
                    set_value(ctx, &var_buf[idx], op1);
                }
            }
            BREAK;
        CASE(OP_dec_loc):
            {
                JSValue op1;
                int val;
                int idx;
                idx = *pc;
                pc += 1;

                op1 = var_buf[idx];
                if (JS_VALUE_GET_TAG(op1) == JS_TAG_INT) {
                    val = JS_VALUE_GET_INT(op1);
                    if (unlikely(val == INT32_MIN))
                        goto dec_loc_slow;
                    var_buf[idx] = js_int32(val - 1);
                } else {
                dec_loc_slow:
                    sf->cur_pc = pc;
                    /* must duplicate otherwise the variable value may
                       be destroyed before JS code accesses it */
                    op1 = js_dup(op1);
                    if (js_unary_arith_slow(ctx, &op1 + 1, OP_dec))
                        goto exception;
                    set_value(ctx, &var_buf[idx], op1);
                }
            }
            BREAK;
        CASE(OP_not):
            {
                JSValue op1;
                op1 = sp[-1];
                if (JS_VALUE_GET_TAG(op1) == JS_TAG_INT) {
                    sp[-1] = js_int32(~JS_VALUE_GET_INT(op1));
                } else {
                    sf->cur_pc = pc;
                    if (js_not_slow(ctx, sp))
                        goto exception;
                }
            }
            BREAK;

        CASE(OP_shl):
            {
                JSValue op1, op2;
                op1 = sp[-2];
                op2 = sp[-1];
                if (likely(JS_VALUE_IS_BOTH_INT(op1, op2))) {
                    uint32_t v1, v2;
                    v1 = JS_VALUE_GET_INT(op1);
                    v2 = JS_VALUE_GET_INT(op2) & 0x1f;
                    sp[-2] = js_int32(v1 << v2);
                    sp--;
                } else {
                    sf->cur_pc = pc;
                    if (js_binary_logic_slow(ctx, sp, opcode))
                        goto exception;
                    sp--;
                }
            }
            BREAK;
        CASE(OP_shr):
            {
                JSValue op1, op2;
                op1 = sp[-2];
                op2 = sp[-1];
                if (likely(JS_VALUE_IS_BOTH_INT(op1, op2))) {
                    uint32_t v2;
                    v2 = JS_VALUE_GET_INT(op2);
                    v2 &= 0x1f;
                    sp[-2] = js_uint32((uint32_t)JS_VALUE_GET_INT(op1) >> v2);
                    sp--;
                } else {
                    sf->cur_pc = pc;
                    if (js_shr_slow(ctx, sp))
                        goto exception;
                    sp--;
                }
            }
            BREAK;
        CASE(OP_sar):
            {
                JSValue op1, op2;
                op1 = sp[-2];
                op2 = sp[-1];
                if (likely(JS_VALUE_IS_BOTH_INT(op1, op2))) {
                    uint32_t v2;
                    v2 = JS_VALUE_GET_INT(op2);
                    if (unlikely(v2 > 0x1f)) {
                        v2 &= 0x1f;
                    }
                    sp[-2] = js_int32((int)JS_VALUE_GET_INT(op1) >> v2);
                    sp--;
                } else {
                    sf->cur_pc = pc;
                    if (js_binary_logic_slow(ctx, sp, opcode))
                        goto exception;
                    sp--;
                }
            }
            BREAK;
        CASE(OP_and):
            {
                JSValue op1, op2;
                op1 = sp[-2];
                op2 = sp[-1];
                if (likely(JS_VALUE_IS_BOTH_INT(op1, op2))) {
                    sp[-2] = js_int32(JS_VALUE_GET_INT(op1) & JS_VALUE_GET_INT(op2));
                    sp--;
                } else {
                    sf->cur_pc = pc;
                    if (js_binary_logic_slow(ctx, sp, opcode))
                        goto exception;
                    sp--;
                }
            }
            BREAK;
        CASE(OP_or):
            {
                JSValue op1, op2;
                op1 = sp[-2];
                op2 = sp[-1];
                if (likely(JS_VALUE_IS_BOTH_INT(op1, op2))) {
                    sp[-2] = js_int32(JS_VALUE_GET_INT(op1) | JS_VALUE_GET_INT(op2));
                    sp--;
                } else {
                    sf->cur_pc = pc;
                    if (js_binary_logic_slow(ctx, sp, opcode))
                        goto exception;
                    sp--;
                }
            }
            BREAK;
        CASE(OP_xor):
            {
                JSValue op1, op2;
                op1 = sp[-2];
                op2 = sp[-1];
                if (likely(JS_VALUE_IS_BOTH_INT(op1, op2))) {
                    sp[-2] = js_int32(JS_VALUE_GET_INT(op1) ^ JS_VALUE_GET_INT(op2));
                    sp--;
                } else {
                    sf->cur_pc = pc;
                    if (js_binary_logic_slow(ctx, sp, opcode))
                        goto exception;
                    sp--;
                }
            }
            BREAK;


#define OP_CMP(opcode, binary_op, slow_call)              \
            CASE(opcode):                                 \
                {                                         \
                JSValue op1, op2;                         \
                op1 = sp[-2];                             \
                op2 = sp[-1];                                   \
                if (likely(JS_VALUE_IS_BOTH_INT(op1, op2))) {           \
                    sp[-2] = js_bool(JS_VALUE_GET_INT(op1) binary_op JS_VALUE_GET_INT(op2)); \
                    sp--;                                               \
                } else {                                                \
                    sf->cur_pc = pc;                                    \
                    if (slow_call)                                      \
                        goto exception;                                 \
                    sp--;                                               \
                }                                                       \
                }                                                       \
            BREAK

            OP_CMP(OP_lt, <, js_relational_slow(ctx, sp, opcode));
            OP_CMP(OP_lte, <=, js_relational_slow(ctx, sp, opcode));
            OP_CMP(OP_gt, >, js_relational_slow(ctx, sp, opcode));
            OP_CMP(OP_gte, >=, js_relational_slow(ctx, sp, opcode));
            OP_CMP(OP_eq, ==, js_eq_slow(ctx, sp, 0));
            OP_CMP(OP_neq, !=, js_eq_slow(ctx, sp, 1));
            OP_CMP(OP_strict_eq, ==, js_strict_eq_slow(ctx, sp, 0));
            OP_CMP(OP_strict_neq, !=, js_strict_eq_slow(ctx, sp, 1));

        CASE(OP_in):
            sf->cur_pc = pc;
            if (js_operator_in(ctx, sp))
                goto exception;
            sp--;
            BREAK;
        CASE(OP_private_in):
            if (js_operator_private_in(ctx, sp))
                goto exception;
            sp--;
            BREAK;
        CASE(OP_instanceof):
            sf->cur_pc = pc;
            if (js_operator_instanceof(ctx, sp))
                goto exception;
            sp--;
            BREAK;
        CASE(OP_typeof):
            {
                JSValue op1;
                JSAtom atom;

                op1 = sp[-1];
                atom = js_operator_typeof(ctx, op1);
                JS_FreeValue(ctx, op1);
                sp[-1] = JS_AtomToString(ctx, atom);
            }
            BREAK;
        CASE(OP_delete):
            sf->cur_pc = pc;
            if (js_operator_delete(ctx, sp))
                goto exception;
            sp--;
            BREAK;
        CASE(OP_delete_var):
            {
                JSAtom atom;
                int ret;

                atom = get_u32(pc);
                pc += 4;

                sf->cur_pc = pc;
                ret = JS_DeleteGlobalVar(ctx, atom);
                if (unlikely(ret < 0))
                    goto exception;
                *sp++ = js_bool(ret);
            }
            BREAK;

        CASE(OP_to_object):
            if (JS_VALUE_GET_TAG(sp[-1]) != JS_TAG_OBJECT) {
                sf->cur_pc = pc;
                ret_val = JS_ToObject(ctx, sp[-1]);
                if (JS_IsException(ret_val))
                    goto exception;
                JS_FreeValue(ctx, sp[-1]);
                sp[-1] = ret_val;
            }
            BREAK;

        CASE(OP_to_propkey):
            switch (JS_VALUE_GET_TAG(sp[-1])) {
            case JS_TAG_INT:
            case JS_TAG_STRING:
            case JS_TAG_SYMBOL:
                break;
            default:
                sf->cur_pc = pc;
                ret_val = JS_ToPropertyKey(ctx, sp[-1]);
                if (JS_IsException(ret_val))
                    goto exception;
                JS_FreeValue(ctx, sp[-1]);
                sp[-1] = ret_val;
                break;
            }
            BREAK;

        CASE(OP_to_propkey2):
            /* must be tested first */
            if (unlikely(JS_IsUndefined(sp[-2]) || JS_IsNull(sp[-2]))) {
                JS_ThrowTypeError(ctx, "value has no property");
                goto exception;
            }
            switch (JS_VALUE_GET_TAG(sp[-1])) {
            case JS_TAG_INT:
            case JS_TAG_STRING:
            case JS_TAG_SYMBOL:
                break;
            default:
                sf->cur_pc = pc;
                ret_val = JS_ToPropertyKey(ctx, sp[-1]);
                if (JS_IsException(ret_val))
                    goto exception;
                JS_FreeValue(ctx, sp[-1]);
                sp[-1] = ret_val;
                break;
            }
            BREAK;
        CASE(OP_with_get_var):
        CASE(OP_with_put_var):
        CASE(OP_with_delete_var):
        CASE(OP_with_make_ref):
        CASE(OP_with_get_ref):
        CASE(OP_with_get_ref_undef):
            {
                JSAtom atom;
                int32_t diff;
                JSValue obj, val;
                int ret, is_with;
                atom = get_u32(pc);
                diff = get_u32(pc + 4);
                is_with = pc[8];
                pc += 9;
                sf->cur_pc = pc;

                obj = sp[-1];
                ret = JS_HasProperty(ctx, obj, atom);
                if (unlikely(ret < 0))
                    goto exception;
                if (ret) {
                    if (is_with) {
                        ret = js_has_unscopable(ctx, obj, atom);
                        if (unlikely(ret < 0))
                            goto exception;
                        if (ret)
                            goto no_with;
                    }
                    switch (opcode) {
                    case OP_with_get_var:
                        val = JS_GetProperty(ctx, obj, atom);
                        if (unlikely(JS_IsException(val)))
                            goto exception;
                        set_value(ctx, &sp[-1], val);
                        break;
                    case OP_with_put_var:
                        /* XXX: check if strict mode */
                        ret = JS_SetPropertyInternal(ctx, obj, atom, sp[-2],
                                                     JS_PROP_THROW_STRICT);
                        JS_FreeValue(ctx, sp[-1]);
                        sp -= 2;
                        if (unlikely(ret < 0))
                            goto exception;
                        break;
                    case OP_with_delete_var:
                        ret = JS_DeleteProperty(ctx, obj, atom, 0);
                        if (unlikely(ret < 0))
                            goto exception;
                        JS_FreeValue(ctx, sp[-1]);
                        sp[-1] = js_bool(ret);
                        break;
                    case OP_with_make_ref:
                        /* produce a pair object/propname on the stack */
                        *sp++ = JS_AtomToValue(ctx, atom);
                        break;
                    case OP_with_get_ref:
                        /* produce a pair object/method on the stack */
                        val = JS_GetProperty(ctx, obj, atom);
                        if (unlikely(JS_IsException(val)))
                            goto exception;
                        *sp++ = val;
                        break;
                    case OP_with_get_ref_undef:
                        /* produce a pair undefined/function on the stack */
                        val = JS_GetProperty(ctx, obj, atom);
                        if (unlikely(JS_IsException(val)))
                            goto exception;
                        JS_FreeValue(ctx, sp[-1]);
                        sp[-1] = JS_UNDEFINED;
                        *sp++ = val;
                        break;
                    }
                    pc += diff - 5;
                } else {
                no_with:
                    /* if not jumping, drop the object argument */
                    JS_FreeValue(ctx, sp[-1]);
                    sp--;
                }
            }
            BREAK;

        CASE(OP_await):
            ret_val = js_int32(FUNC_RET_AWAIT);
            goto done_generator;
        CASE(OP_yield):
            ret_val = js_int32(FUNC_RET_YIELD);
            goto done_generator;
        CASE(OP_yield_star):
        CASE(OP_async_yield_star):
            ret_val = js_int32(FUNC_RET_YIELD_STAR);
            goto done_generator;
        CASE(OP_return_async):
        CASE(OP_initial_yield):
            ret_val = JS_UNDEFINED;
            goto done_generator;

        CASE(OP_nop):
            BREAK;
        CASE(OP_is_undefined_or_null):
            if (JS_VALUE_GET_TAG(sp[-1]) == JS_TAG_UNDEFINED ||
                JS_VALUE_GET_TAG(sp[-1]) == JS_TAG_NULL) {
                goto set_true;
            } else {
                goto free_and_set_false;
            }
        CASE(OP_is_undefined):
            if (JS_VALUE_GET_TAG(sp[-1]) == JS_TAG_UNDEFINED) {
                goto set_true;
            } else {
                goto free_and_set_false;
            }
        CASE(OP_is_null):
            if (JS_VALUE_GET_TAG(sp[-1]) == JS_TAG_NULL) {
                goto set_true;
            } else {
                goto free_and_set_false;
            }
            /* XXX: could merge to a single opcode */
        CASE(OP_typeof_is_undefined):
            /* different from OP_is_undefined because of isHTMLDDA */
            if (js_operator_typeof(ctx, sp[-1]) == JS_ATOM_undefined) {
                goto free_and_set_true;
            } else {
                goto free_and_set_false;
            }
        CASE(OP_typeof_is_function):
            if (js_operator_typeof(ctx, sp[-1]) == JS_ATOM_function) {
                goto free_and_set_true;
            } else {
                goto free_and_set_false;
            }
        free_and_set_true:
            JS_FreeValue(ctx, sp[-1]);
        set_true:
            sp[-1] = JS_TRUE;
            BREAK;
        free_and_set_false:
            JS_FreeValue(ctx, sp[-1]);
            sp[-1] = JS_FALSE;
            BREAK;
        CASE(OP_invalid):
        DEFAULT:
            JS_ThrowInternalError(ctx, "invalid opcode: pc=%u opcode=0x%02x",
                                  (int)(pc - b->byte_code_buf - 1), opcode);
            goto exception;
        }
    }
 exception:
    if (needs_backtrace(rt->current_exception)
    || JS_IsUndefined(ctx->error_back_trace)) {
        sf->cur_pc = pc;
        build_backtrace(ctx, rt->current_exception, JS_UNDEFINED,
                        NULL, 0, 0, 0);
    }
    if (!JS_IsUncatchableError(rt->current_exception)) {
        while (sp > stack_buf) {
            JSValue val = *--sp;
            JS_FreeValue(ctx, val);
            if (JS_VALUE_GET_TAG(val) == JS_TAG_CATCH_OFFSET) {
                int pos = JS_VALUE_GET_INT(val);
                if (pos == 0) {
                    /* enumerator: close it with a throw */
                    JS_FreeValue(ctx, sp[-1]); /* drop the next method */
                    sp--;
                    JS_IteratorClose(ctx, sp[-1], true);
                } else {
                    *sp++ = rt->current_exception;
                    rt->current_exception = JS_UNINITIALIZED;
                    JS_FreeValueRT(rt, ctx->error_back_trace);
                    ctx->error_back_trace = JS_UNDEFINED;
                    pc = b->byte_code_buf + pos;
                    goto restart;
                }
            }
        }
    }
    ret_val = JS_EXCEPTION;
    /* the local variables are freed by the caller in the generator
       case. Hence the label 'done' should never be reached in a
       generator function. */
    if (b->func_kind != JS_FUNC_NORMAL) {
    done_generator:
        sf->cur_pc = pc;
        sf->cur_sp = sp;
    } else {
    done:
        if (unlikely(!list_empty(&sf->var_ref_list))) {
            /* variable references reference the stack: must close them */
            close_var_refs(rt, sf);
        }
        /* free the local variables and stack */
        for(pval = local_buf; pval < sp; pval++) {
            JS_FreeValue(ctx, *pval);
        }
    }
    rt->current_stack_frame = sf->prev_frame;
    return ret_val;
}

JSValue JS_Call(JSContext *ctx, JSValueConst func_obj, JSValueConst this_obj,
                int argc, JSValueConst *argv)
{
    return JS_CallInternal(ctx, func_obj, this_obj, JS_UNDEFINED,
                           argc, argv, JS_CALL_FLAG_COPY_ARGV);
}

static JSValue JS_CallFree(JSContext *ctx, JSValue func_obj, JSValueConst this_obj,
                           int argc, JSValueConst *argv)
{
    JSValue res = JS_CallInternal(ctx, func_obj, this_obj, JS_UNDEFINED,
                                  argc, argv, JS_CALL_FLAG_COPY_ARGV);
    JS_FreeValue(ctx, func_obj);
    return res;
}

/* warning: the refcount of the context is not incremented. Return
   NULL in case of exception (case of revoked proxy only) */
static JSContext *JS_GetFunctionRealm(JSContext *ctx, JSValueConst func_obj)
{
    JSObject *p;
    JSContext *realm;

    if (JS_VALUE_GET_TAG(func_obj) != JS_TAG_OBJECT)
        return ctx;
    p = JS_VALUE_GET_OBJ(func_obj);
    switch(p->class_id) {
    case JS_CLASS_C_FUNCTION:
        realm = p->u.cfunc.realm;
 