// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/UnstableParticles.hh"
#include "Rivet/Projections/ChargedFinalState.hh"
#include "Rivet/Projections/Beam.hh"

namespace Rivet {


  /// @brief pi, K and p spectra at 34 and 44 GeV
  class TASSO_1989_I267755 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(TASSO_1989_I267755);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      // Initialise and register projections
      declare(Beam(), "Beams");
      declare(ChargedFinalState(), "FS");
      declare(UnstableParticles(), "UFS");

      // Book histograms
      size_t ih = 0;
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;
        book(_c[en], "_sumW_"+en);
        book(_h[en+"x_pi"], 3*ih+7, 1, 1);
        book(_h[en+"x_K"],  3*ih+8, 1, 1);
        book(_h[en+"x_p"],  3*ih+9, 1, 1);
        if (en == "44"s)  book(_h[en+"x_pi0"], 13, 1, 1);
        book(_h[en+"n_pi"], "TMP/n_pi"+en, refData(3*ih+1, 1, 1));
        book(_h[en+"d_pi"], "TMP/d_pi"+en, refData(3*ih+1, 1, 1));
        book(_h[en+"n_K"],  "TMP/n_K"+en,  refData(3*ih+2, 1, 1));
        book(_h[en+"d_K"],  "TMP/d_K"+en,  refData(3*ih+2, 1, 1));
        book(_h[en+"n_p"],  "TMP/n_p"+en,  refData(3*ih+3, 1, 1));
        book(_h[en+"d_p"],  "TMP/d_p"+en,  refData(3*ih+3, 1, 1));
        ++ih;
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      _c[_sqs]->fill();
      // First, veto on leptonic events by requiring at least 4 charged FS particles
      const ChargedFinalState& fs = apply<ChargedFinalState>(event, "FS");
      const size_t numParticles = fs.particles().size();

      // Even if we only generate hadronic events, we still need a cut on numCharged >= 2.
      if (numParticles < 2) vetoEvent;

      // Get beams and average beam momentum
      const ParticlePair& beams = apply<Beam>(event, "Beams").beams();
      const double meanBeamMom = 0.5*( beams.first.p3().mod() + beams.second.p3().mod());
      MSG_DEBUG("Avg beam momentum = " << meanBeamMom);

      for (const Particle& p : fs.particles()) {
      	double xP = p.p3().mod()/meanBeamMom;
      	_h[_sqs+"d_pi"]->fill(xP);
      	_h[_sqs+"d_K"]->fill(xP);
      	_h[_sqs+"d_p"]->fill(xP);
      	if (p.abspid()==211) {
      	  _h[_sqs+"x_pi"]->fill(xP);
      	  _h[_sqs+"n_pi"]->fill(xP);
      	}
      	else if (p.abspid()==321) {
      	  _h[_sqs+"x_K"]->fill(xP);
      	  _h[_sqs+"n_K"]->fill(xP);
      	}
      	else if (p.abspid()==2212) {
      	  _h[_sqs+"x_p"]->fill(xP);
      	  _h[_sqs+"n_p"]->fill(xP);
      	}
      }
      if (_sqs == "44"s) {
      	for (const Particle& p : apply<UnstableParticles>(event, "UFS").particles(Cuts::pid==111)) {
      	  _h[_sqs+"x_pi0"]->fill(p.p3().mod()/meanBeamMom);
      	}
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_c, crossSectionPerEvent());
      scale(_h, crossSectionPerEvent());
      for (auto& item : _h) {
        const double w = _c[item.first.substr(0,2)]->sumW();
        if (!isZero(w))  scale(item.second, 1.0/w);
      }
      size_t ih = 0;
      Estimate1DPtr temp;
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal));
        book(temp, 3*ih+1, 1, 1);
        divide(_h[en+"n_pi"], _h[en+"d_pi"], temp);
        book(temp, 3*ih+2, 1, 1);
        divide(_h[en+"n_K"], _h[en+"d_K"], temp);
        book(temp, 3*ih+3, 1, 1);
        divide(_h[en+"n_p"], _h[en+"d_p"], temp);
        ++ih;
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,CounterPtr> _c;
    map<string,Histo1DPtr> _h;
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(TASSO_1989_I267755);
}
