/***************************************************************************
 *   copyright       : (C) 2003-2024 by Pascal Brachet                     *
 *   https://www.xm1math.net/texmaker/                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   @license GPL-2.0+ <https://spdx.org/licenses/GPL-2.0+.html>           *
 ***************************************************************************/

#include "githelper.h"
#include "cmdparser.h"

#include <QProcess>
#include <QRegularExpression>
#include <QDebug>


GitHelper::GitHelper(QString filename,QString gitPath)
{
    QStringList args;
    args << "diff";
    args << "--ignore-cr-at-eol";
    args << filename;

    //add git to path
    QString extraPath = gitPath;
    QProcessEnvironment env = QProcessEnvironment::systemEnvironment();
#ifdef OS_WINDOWS
    if (!extraPath.isEmpty())
    {
        env.insert("PATH", env.value("PATH") + ";"+extraPath);
        _process.setProcessEnvironment(env);
    }
#endif
#ifdef OS_LINUX
    if (!extraPath.isEmpty())
    {
        env.insert("PATH", env.value("PATH") + ":"+extraPath);
        _process.setProcessEnvironment(env);
    }
#endif
    //Very Important, everything is in there because process->setEnvironement is for child process
    qputenv("PATH", env.value("PATH").toLatin1());



    connect(&_process, SIGNAL(finished(int,QProcess::ExitStatus)), this, SLOT(parseUnified(int,QProcess::ExitStatus)));
#ifdef OS_WINDOWS
    _process.start("git.exe", args);
#else
    _process.start("git", args);
#endif

}

GitHelper::~GitHelper()
{

}

void GitHelper::parseUnified(int exitCode, QProcess::ExitStatus exitStatus)
{
    if(exitCode == QProcess::CrashExit)
    {
        //qDebug()<<"git error: "<<_process.errorString();
        this->deleteLater();
        return;
    }
    _process.readLine();// "Index: ...
    _process.readLine();// "====== ...
    _process.readLine();// "--- ...
    _process.readLine();// "+++ ...


    while(!_process.atEnd())
    {
        QString c = _process.read(1);
        if(c == "\\")
        {
            break;
        }
        QString position = _process.readLine();
        QRegularExpression positionRegex("^\\@ \\-(\\d+),(\\d+) \\+(\\d+),(\\d+) \\@\\@");
        QRegularExpressionMatch match=positionRegex.match(position);
        if(!match.hasMatch())
        {
            //qDebug()<<position;
            qWarning("Wrong diff output. Expecting unified format");
            return;
        }
        int line1 = match.captured(1).toInt();
        int length1 = match.captured(2).toInt();
        int line2 = match.captured(3).toInt();
        int length2 = match.captured(4).toInt();
        while(length2 > 0 || length1 > 0)
        {
            c = _process.read(1);
            _process.readLine();
            if(c == " ")
            {
                length1--;
                line1++;
                length2--;
                line2++;
                continue;
            }
            if(c == "+")
            {
                this->_uncommitLines << line2;
                length2--;
                line2++;
                continue;
            }
            if(c == "-")
            {
                length1--;
                line1++;
                continue;
            }

        }
    }
    emit uncommittedLines(_uncommitLines);
    this->deleteLater();
}

