// Licensed to the Software Freedom Conservancy (SFC) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The SFC licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.
package org.openqa.selenium.bidi.browsingcontext;

import static org.assertj.core.api.Assertions.assertThat;
import static org.openqa.selenium.testing.drivers.Browser.FIREFOX;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import org.junit.jupiter.api.Test;
import org.openqa.selenium.bidi.module.Script;
import org.openqa.selenium.bidi.script.EvaluateResult;
import org.openqa.selenium.bidi.script.EvaluateResultSuccess;
import org.openqa.selenium.bidi.script.LocalValue;
import org.openqa.selenium.bidi.script.NodeProperties;
import org.openqa.selenium.bidi.script.RemoteReference;
import org.openqa.selenium.bidi.script.RemoteValue;
import org.openqa.selenium.bidi.script.ResultOwnership;
import org.openqa.selenium.testing.JupiterTestBase;
import org.openqa.selenium.testing.NeedsFreshDriver;
import org.openqa.selenium.testing.NotYetImplemented;

public class LocateNodesTest extends JupiterTestBase {

  @Test
  @NeedsFreshDriver
  void canLocateNodes() {
    BrowsingContext browsingContext = new BrowsingContext(driver, driver.getWindowHandle());
    assertThat(browsingContext.getId()).isNotEmpty();

    driver.get(pages.xhtmlTestPage);

    LocateNodeParameters parameters = new LocateNodeParameters(Locator.css("div"));

    List<RemoteValue> elements = browsingContext.locateNodes(parameters);
    assertThat(elements).hasSize(13);
  }

  @Test
  @NeedsFreshDriver
  void canLocateNodesWithJustLocator() {
    BrowsingContext browsingContext = new BrowsingContext(driver, driver.getWindowHandle());
    assertThat(browsingContext.getId()).isNotEmpty();

    driver.get(pages.xhtmlTestPage);

    List<RemoteValue> elements = browsingContext.locateNodes(Locator.css("div"));
    assertThat(elements).hasSize(13);
  }

  @Test
  @NeedsFreshDriver
  void canLocateNode() {
    BrowsingContext browsingContext = new BrowsingContext(driver, driver.getWindowHandle());
    assertThat(browsingContext.getId()).isNotEmpty();

    driver.get(pages.xhtmlTestPage);

    RemoteValue element = browsingContext.locateNode(Locator.css("div"));
    assertThat(element.getType()).isEqualTo("node");
  }

  @Test
  @NeedsFreshDriver
  void canLocateNodesWithCSSLocator() {
    BrowsingContext browsingContext = new BrowsingContext(driver, driver.getWindowHandle());
    assertThat(browsingContext.getId()).isNotEmpty();

    driver.get(pages.xhtmlTestPage);

    LocateNodeParameters parameters =
        new LocateNodeParameters(Locator.css("div.extraDiv, div.content")).setMaxNodeCount(1);

    List<RemoteValue> elements = browsingContext.locateNodes(parameters);
    assertThat(elements.size()).isGreaterThanOrEqualTo(1);

    RemoteValue value = elements.get(0);
    assertThat(value.getType()).isEqualTo("node");
    assertThat(value.getValue()).isPresent();
    NodeProperties properties = (NodeProperties) value.getValue().get();
    assertThat(properties.getLocalName()).hasValue("div");
    assertThat(properties.getAttributes()).hasValue(Map.of("class", "content"));
  }

  @Test
  @NeedsFreshDriver
  void canLocateNodesWithXPathLocator() {
    BrowsingContext browsingContext = new BrowsingContext(driver, driver.getWindowHandle());
    assertThat(browsingContext.getId()).isNotEmpty();

    driver.get(pages.xhtmlTestPage);

    LocateNodeParameters parameters =
        new LocateNodeParameters(Locator.xpath("/html/body/div[2]")).setMaxNodeCount(1);

    List<RemoteValue> elements = browsingContext.locateNodes(parameters);
    assertThat(elements.size()).isGreaterThanOrEqualTo(1);

    RemoteValue value = elements.get(0);
    assertThat(value.getType()).isEqualTo("node");
    assertThat(value.getValue()).isPresent();
    NodeProperties properties = (NodeProperties) value.getValue().get();
    assertThat(properties.getLocalName()).hasValue("div");
    assertThat(properties.getAttributes()).hasValue(Map.of("class", "content"));
  }

  @Test
  @NeedsFreshDriver
  @NotYetImplemented(FIREFOX)
  void canLocateNodesWithInnerText() {
    BrowsingContext browsingContext = new BrowsingContext(driver, driver.getWindowHandle());
    assertThat(browsingContext.getId()).isNotEmpty();

    driver.get(pages.xhtmlTestPage);

    LocateNodeParameters parameters =
        new LocateNodeParameters(Locator.innerText("Spaced out")).setMaxNodeCount(1);

    List<RemoteValue> elements = browsingContext.locateNodes(parameters);
    assertThat(elements.size()).isGreaterThanOrEqualTo(1);

    RemoteValue value = elements.get(0);
    assertThat(value.getType()).isEqualTo("node");
    assertThat(value.getValue()).isPresent();
  }

  @Test
  @NeedsFreshDriver
  void canLocateNodesWithMaxNodeCount() {
    BrowsingContext browsingContext = new BrowsingContext(driver, driver.getWindowHandle());
    assertThat(browsingContext.getId()).isNotEmpty();

    driver.get(pages.xhtmlTestPage);

    LocateNodeParameters parameters =
        new LocateNodeParameters(Locator.css("div")).setMaxNodeCount(4);

    List<RemoteValue> elements = browsingContext.locateNodes(parameters);
    assertThat(elements).hasSize(4);
  }

  @Test
  @NeedsFreshDriver
  void canLocateNodesGivenStartNodes() {
    String handle = driver.getWindowHandle();
    BrowsingContext browsingContext = new BrowsingContext(driver, handle);
    assertThat(browsingContext.getId()).isNotEmpty();

    driver.get(pages.formPage);

    Script script = new Script(driver);
    EvaluateResult result =
        script.evaluateFunctionInBrowsingContext(
            handle,
            "document.querySelectorAll(\"form\")",
            false,
            Optional.of(ResultOwnership.ROOT));

    assertThat(result.getResultType()).isEqualTo(EvaluateResult.Type.SUCCESS);

    EvaluateResultSuccess resultSuccess = (EvaluateResultSuccess) result;
    List<RemoteReference> startNodes = new ArrayList<>();

    RemoteValue remoteValue = resultSuccess.getResult();
    List<RemoteValue> remoteValues = (List<RemoteValue>) remoteValue.getValue().get();

    remoteValues.forEach(
        value ->
            startNodes.add(
                new RemoteReference(RemoteReference.Type.SHARED_ID, value.getSharedId().get())));

    LocateNodeParameters parameters =
        new LocateNodeParameters(Locator.css("input"))
            .setStartNodes(startNodes)
            .setMaxNodeCount(50);

    List<RemoteValue> elements = browsingContext.locateNodes(parameters);
    assertThat(elements).hasSize(35);
  }

  @Test
  @NeedsFreshDriver
  void canLocateNodesInAGivenSandbox() {
    String sandbox = "sandbox";
    BrowsingContext browsingContext = new BrowsingContext(driver, driver.getWindowHandle());
    assertThat(browsingContext.getId()).isNotEmpty();

    browsingContext.navigate(pages.xhtmlTestPage, ReadinessState.COMPLETE);

    LocateNodeParameters parameters =
        new LocateNodeParameters(Locator.css("div")).setSandbox(sandbox).setMaxNodeCount(1);

    List<RemoteValue> elements = browsingContext.locateNodes(parameters);
    assertThat(elements).hasSize(1);

    String nodeId = elements.get(0).getSharedId().get();

    List<LocalValue> arguments = new ArrayList<>();

    LocalValue value = LocalValue.mapValue(Map.of("sharedId", LocalValue.stringValue(nodeId)));
    arguments.add(value);

    Script script = new Script(driver);

    // Since the node was present in the sandbox, the script run in the same sandbox should be able
    // to retrieve it
    EvaluateResult result =
        script.callFunctionInBrowsingContext(
            driver.getWindowHandle(),
            sandbox,
            "function(){ return arguments[0]; }",
            true,
            Optional.of(arguments),
            Optional.empty(),
            Optional.empty());

    assertThat(result.getResultType()).isEqualTo(EvaluateResult.Type.SUCCESS);
    Map<String, Object> sharedIdMap =
        (Map<String, Object>) ((EvaluateResultSuccess) result).getResult().getValue().get();

    String sharedId = (String) ((RemoteValue) sharedIdMap.get("sharedId")).getValue().get();
    assertThat(sharedId).isEqualTo(nodeId);
  }
}
