using System;
using System.Collections.Generic;
using System.Collections.Immutable;
using System.Linq;
using System.Threading;
using IntervalTree;
using Microsoft.Extensions.Logging;
using OmniSharp.Extensions.LanguageServer.Protocol;
using OmniSharp.Extensions.LanguageServer.Protocol.Models;
using Range = OmniSharp.Extensions.LanguageServer.Protocol.Models.Range;

namespace Microsoft.Dafny.LanguageServer.Workspace;

public class SymbolTable {

  public static SymbolTable CreateFrom(ILogger logger, Program program, CancellationToken cancellationToken) {
    var visited = program.Visit(a => true, b => { });

    var references = visited.OfType<IHasReferences>()
      .SelectMany(r => {
        var usages = r.GetReferences();
        if (usages == null) {
          logger.LogError($"Value of type {r.GetType()} returned a null for ${nameof(IHasReferences.GetReferences)}");
          usages = Array.Empty<Reference>();
        }
        return usages.Where(d => !AutoGeneratedOrigin.Is(d.Referer) && d.Referred != null);
      }).ToList();

    var definitions = visited.OfType<IHasNavigationToken>();

    return new SymbolTable(references, definitions.ToList());
  }

  public static SymbolTable Empty() {
    return new SymbolTable();
  }

  private SymbolTable() {
    NodeToReferences = ImmutableDictionary<IOrigin, ISet<IOrigin>>.Empty;
    ReferenceToNode = ImmutableDictionary<IOrigin, IHasNavigationToken>.Empty;
    Symbols = ImmutableList<ISymbol>.Empty;
    Declarations = ImmutableDictionary<IOrigin, ISymbol>.Empty;
  }

  private SymbolTable(IReadOnlyList<Reference> references, IReadOnlyList<IHasNavigationToken> hasNavigationTokens) {
    var safeReferences = references.Where(k => k.Referer.Uri != null).ToImmutableList();

    var referencesAsKeyValuePairs = safeReferences.Select(k => KeyValuePair.Create(k.Referer, k.Referred));
    ReferenceToNode = referencesAsKeyValuePairs.DistinctBy(pair => pair.Key).ToImmutableDictionary();

    NodeToReferences = safeReferences.GroupBy(u => u.Referred.NavigationToken).ToImmutableDictionary(
      g => g.Key,
      g => (ISet<IOrigin>)g.Select(k => k.Referer).ToHashSet());

    Declarations = hasNavigationTokens.OfType<ISymbol>().Where(h => h.NavigationToken.IsValid && !AutoGeneratedOrigin.Is(h.Origin)
        && (h is not AliasModuleDecl alias /* see AliasModuleDecl.NavigationToken for explanation */ || alias.HasAlias)).
      DistinctBy(h => h.NavigationToken).
      ToImmutableDictionary(h => h.NavigationToken, h => h);

    var relevantDafnySymbolKinds = new HashSet<SymbolKind> {
      SymbolKind.Function,
      SymbolKind.Class,
      SymbolKind.Enum,
      SymbolKind.Method,
      SymbolKind.EnumMember,
      SymbolKind.Struct,
      SymbolKind.Interface,
      SymbolKind.Namespace,
    };
    // Since these definitions are checked for whether they
    // contain substrings when answering workspace/resolve queries,
    // performance can be improved by storing their names in a
    // data structure that makes this operation cheaper, such as
    // a suffix tree.
    Symbols = hasNavigationTokens.OfType<ISymbol>().
      Where(symbol => symbol.Kind.HasValue && relevantDafnySymbolKinds.Contains(symbol.Kind.Value)).
      ToImmutableList();

    var symbols = hasNavigationTokens.Select(h => h.NavigationToken)
      .Concat(safeReferences.Select(r => r.Referer))
      .Concat(safeReferences.Select(r => r.Referred.NavigationToken))
      .Where(s => !AutoGeneratedOrigin.Is(s)).Distinct();
    var symbolsByFile = symbols.GroupBy(s => s.Uri);
    foreach (var symbolsForFile in symbolsByFile) {
      if (symbolsForFile.Key == null) {
        continue;
      }
      var nodePositionsForFile = new IntervalTree<Position, IOrigin>();
      navigationRanges.Add(symbolsForFile.Key, nodePositionsForFile);
      foreach (var symbolForFile in symbolsForFile) {
        var range = symbolForFile.GetLspRange();
        nodePositionsForFile.Add(range.Start, range.End, symbolForFile);
      }
    }
  }

  private readonly Dictionary<Uri, IIntervalTree<Position, IOrigin>> navigationRanges = new();

  /// <summary>
  /// Maps each symbol declaration to itself, and each symbol usage to the symbol's declaration.
  /// </summary>
  public ImmutableDictionary<IOrigin, IHasNavigationToken> ReferenceToNode { get; }
  public ImmutableDictionary<IOrigin, ISymbol> Declarations { get; }

  /// <summary>
  /// Maps each symbol declaration to usages of the symbol, not including the declaration itself.
  /// </summary>
  public ImmutableDictionary<IOrigin, ISet<IOrigin>> NodeToReferences { get; }

  /// <summary>
  ///  A list of all definitions, such as methods, classes, functions, etc., used for workspace-wide symbol
  /// lookup.
  /// </summary>
  public ImmutableList<ISymbol> Symbols { get; }

  public IEnumerable<Location> GetReferences(Uri uri, Position position) {
    if (navigationRanges.TryGetValue(uri, out var rangesForFile)) {
      return rangesForFile.Query(position).
        SelectMany(node => NodeToReferences.GetOrDefault(node, () => (ISet<IOrigin>)new HashSet<IOrigin>())).
        Select(u => new Location { Uri = u.Filepath, Range = u.GetLspRange() }).Distinct();
    }
    return [];
  }

  public Location? GetDeclaration(Uri uri, Position position) {
    var node = GetDeclarationNode(uri, position);
    return node == null ? null : NodeToLocation(node);
  }

  internal static Location? NodeToLocation(IHasNavigationToken node) {
    var origin = node.NavigationToken;
    if (origin.Uri == null) {
      return null;
    }
    return new Location {
      Uri = DocumentUri.From(origin.Uri),
      Range = origin.GetLspRange()
    };
  }

  public IHasNavigationToken? GetDeclarationNode(Uri uri, Position position) {
    if (!navigationRanges.TryGetValue(uri, out var forFile)) {
      return null;
    }

    var character = new Range(position, new Position(position.Line, position.Character + 1));
    var options = forFile.Query(position)
      .Where(n => n.ToLspRange().Contains(character)).OrderBy(o => o.Length);
    return options
      .Select(node => Declarations.TryGetValue(node, out var declaration)
        ? declaration : ReferenceToNode.GetValueOrDefault(node))
      .FirstOrDefault(x => x != null);
  }
}
