// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System.Collections.Generic;
using Xunit;

namespace System.Numerics.Tests
{
    public class BigIntegerAddTests
    {
        public static IEnumerable<object[]> UnaryPlus_TestData()
        {
            yield return new object[] { BigInteger.Parse("123123123123123123123123123123123123123123123123123123123123123123123123123123123123123123123123") };
            yield return new object[] { BigInteger.Parse("-123123123123123123123123123123123123123123123123123123123123123123123123123123123123123123123123") };
            yield return new object[] { new BigInteger(123) };
            yield return new object[] { new BigInteger(-123) };
            yield return new object[] { new BigInteger(0) };
            yield return new object[] { new BigInteger(-1) };
            yield return new object[] { new BigInteger(1) };
            yield return new object[] { new BigInteger(int.MinValue) };
            yield return new object[] { new BigInteger(int.MaxValue) };
            yield return new object[] { new BigInteger(long.MaxValue) };
            yield return new object[] { new BigInteger(long.MinValue) };
        }

        [Theory]
        [MemberData(nameof(UnaryPlus_TestData))]
        public static void UnaryPlus(BigInteger bigInteger)
        {
            Assert.Equal(bigInteger, +bigInteger);
        }

        public static IEnumerable<object[]> BinaryPlus_TestData()
        {
            yield return new object[] { new BigInteger(int.MinValue), new BigInteger(-1), new BigInteger((long)int.MinValue - 1) };
            yield return new object[] { new BigInteger(int.MinValue), new BigInteger(1), new BigInteger(int.MinValue + 1) };
            yield return new object[] { new BigInteger(int.MaxValue), new BigInteger(-1), new BigInteger(int.MaxValue - 1) };
            yield return new object[] { new BigInteger(int.MaxValue), new BigInteger(1), new BigInteger((long)int.MaxValue + 1) };

            yield return new object[] { new BigInteger(long.MinValue), new BigInteger(-1), new BigInteger(new byte[] { 255, 255, 255, 255, 255, 255, 255, 127, 255 }) };
            yield return new object[] { new BigInteger(long.MinValue), new BigInteger(1), new BigInteger(long.MinValue + 1) };
            yield return new object[] { new BigInteger(long.MaxValue), new BigInteger(-1), new BigInteger(long.MaxValue - 1) };
            yield return new object[] { new BigInteger(long.MaxValue), new BigInteger(1), new BigInteger((ulong)long.MaxValue + 1) };

            BigInteger largePositiveBigInt = BigInteger.Parse("123123123123123123123123123123123123123123123123123123123123123123123123123123123123123");
            BigInteger largeNegativeBigInt = BigInteger.Parse("-123123123123123123123123123123123123123123123123123123123123123123123123123123123123123");

            // Big + Big
            yield return new object[] { largePositiveBigInt, BigInteger.Parse("234234234234234234234234234234234234234234234234234234234234234234234234"), BigInteger.Parse("123123123123123357357357357357357357357357357357357357357357357357357357357357357357357") };
            yield return new object[] { largePositiveBigInt, BigInteger.Parse("-234234234234234234234234234234234234234234234234234234234234234234234234"), BigInteger.Parse("123123123123122888888888888888888888888888888888888888888888888888888888888888888888889") };

            // Big + Small
            yield return new object[] { largePositiveBigInt, new BigInteger(123), BigInteger.Parse("123123123123123123123123123123123123123123123123123123123123123123123123123123123123246") };
            yield return new object[] { largePositiveBigInt, new BigInteger(-123), BigInteger.Parse("123123123123123123123123123123123123123123123123123123123123123123123123123123123123000") };

            // Plus Zero
            yield return new object[] { largePositiveBigInt, BigInteger.Zero, largePositiveBigInt };
            yield return new object[] { largeNegativeBigInt, BigInteger.Zero, largeNegativeBigInt };
            yield return new object[] { new BigInteger(123), BigInteger.Zero, new BigInteger(123) };
            yield return new object[] { new BigInteger(-123), BigInteger.Zero, new BigInteger(-123) };
            yield return new object[] { BigInteger.Zero, BigInteger.Zero, BigInteger.Zero };

            // Boundaries
            yield return new object[] { new BigInteger(Math.Pow(2, 31) + Math.Pow(2, 30)), new BigInteger(Math.Pow(2, 31) + Math.Pow(2, 30)), new BigInteger(6442450944) };
            yield return new object[] { BigInteger.Zero, new BigInteger(Math.Pow(2, 32)), new BigInteger(Math.Pow(2, 32)) };
            yield return new object[] { new BigInteger(Math.Pow(2, 31)), new BigInteger(Math.Pow(2, 32) + Math.Pow(2, 31)), new BigInteger(8589934592) };
            yield return new object[] { new BigInteger(Math.Pow(2, 32)), new BigInteger(Math.Pow(2, 32)), new BigInteger(8589934592) };
            yield return new object[] { new BigInteger(Math.Pow(2, 32) + Math.Pow(2, 31)), new BigInteger(Math.Pow(2, 32) + Math.Pow(2, 31)), new BigInteger(12884901888) };

            // Very large + very large
            yield return new object[]
            {
                new BigInteger(new byte[] { 86, 40, 240, 54, 176, 173, 86, 246, 44, 114, 227, 38, 121, 29, 118, 9, 250, 39, 166, 53, 89, 182, 206, 34, 18, 186, 194, 31, 31, 100, 147, 122, 139, 172, 194, 117, 20, 197, 208, 5, 106, 23, 124, 19, 116, 173, 97, 202, 241, 114, 49, 31, 25, 186, 193, 159, 89, 206, 35, 246, 235, 55, 214, 200, 223, 105, 154, 204, 91, 0, 245, 53, 86, 160, 202, 66, 58, 187, 173, 101, 89, 247, 45, 173, 50, 116, 96, 209, 234, 34, 28, 70, 161, 218, 26, 51, 64, 180, 52, 11, 18, 148, 111, 240, 124, 252, 200, 121, 239, 127, 100, 158, 32, 190, 156, 243, 9, 41, 149, 106, 8, 201, 225, 57, 228, 175, 147, 123, 134, 151, 1, 133, 134, 161, 83, 197, 136, 60, 48, 54, 171, 231, 225, 107, 189, 125, 37, 227, 62, 38, 41, 23, 30, 201, 3, 72, 178, 53, 12, 205, 104, 76, 242, 88, 126, 223, 23, 126, 107, 214, 106, 204, 248, 95, 190, 32, 231, 118, 123, 102, 137, 175, 61, 95, 110, 233, 102, 189, 158, 10, 19, 82, 48, 99, 205, 95, 244, 137, 237, 222, 102, 166, 120, 104, 188, 92, 31, 191, 10, 226, 96, 59, 186, 165, 67, 185, 15, 146, 233, 148, 120, 90, 224, 96, 194, 23, 156, 148, 146, 95, 43, 94, 137, 156, 0, 227, 54, 213, 163, 17, 202, 167, 178, 187, 169, 216, 151, 138, 31, 20, 79, 98, 159, 89, 102, 11, 197, 24, 165, 164, 192, 138, 44, 171, 182, 2, 178, 30, 165, 96, 130, 58, 239, 216, 201, 208, 168, 36, 183, 234, 127, 17, 116, 126, 15, 201, 190, 47, 54, 244, 29, 30, 47, 255, 81, 64, 122, 6, 4, 166, 37, 121, 229, 191, 145, 21, 114, 199, 143, 149, 99, 160, 96, 186, 36, 174, 21, 174, 155, 237, 68, 130, 232, 176, 66, 99, 10, 240, 134, 137, 85, 161, 31, 17, 196, 6, 44, 180, 236, 250, 87, 247, 160, 160, 144, 124, 190, 116, 218, 162, 42, 204, 107, 23, 98, 163, 145, 52, 125, 46, 11, 130, 64, 6, 64, 194, 129, 79, 245, 30, 130, 96, 194, 161, 149, 76, 226, 71, 32, 197, 165, 149, 21, 228, 94, 137, 54, 7, 174, 33, 116, 247, 55, 189, 238, 212, 182, 153, 95, 206, 251, 111, 126, 188, 31, 201, 9, 202, 129, 21, 192, 73, 20, 183, 136, 117, 108, 92, 168, 183, 140, 121, 160, 212, 169, 139, 41, 5, 178, 193, 150, 73, 232, 81, 202, 54, 22, 88, 196, 149, 128, 108, 134, 152, 97, 43, 227, 128, 52, 121, 104, 89, 241, 168, 66, 130, 206, 170, 19, 148, 160, 4, 23, 192, 126, 119, 97, 194, 237, 82, 95, 19, 152, 18, 55, 32, 243, 7, 115, 126, 167, 169, 208, 57, 89, 60, 87, 63, 15, 72, 104, 107, 69, 47, 244, 123, 84, 197, 25, 233, 201, 32, 146, 226, 25, 84, 193, 222, 113, 34, 143, 198, 55, 149, 236, 246, 52, 221, 103, 65, 31, 34, 179, 165, 234, 243, 206, 104, 29, 187, 203, 45, 237, 137, 180, 135, 22, 238, 31, 4, 123, 121, 205, 237, 184, 161, 52, 175, 67, 254, 71, 118, 93, 216, 201, 167, 109, 237, 247, 38, 133, 89, 192, 182, 86, 97, 216, 173, 225, 101, 224, 69, 190, 98, 198, 166, 84, 198, 236, 72, 190, 30, 36, 31, 115, 84, 207, 21, 192, 13, 25, 40, 246, 248, 126, 33, 184, 23, 150, 87, 201, 134, 206, 222, 29, 1, 184, 250, 17, 34, 82, 167, 1, 174, 104, 103, 52, 160, 66, 2, 145, 194, 237, 211, 75, 184, 79, 197, 228, 164, 238, 61, 48, 149, 127, 56, 79, 70, 2, 113, 155, 213, 4, 157, 186, 114, 39, 72, 144, 23, 236, 23, 212, 148, 197, 7, 153, 250, 146, 101, 178, 243, 22, 108, 52, 20, 119, 108, 47, 48, 23, 238, 158, 208, 102, 188, 189, 79, 25, 217, 145, 6, 52, 205, 184, 214, 105, 155, 207, 120, 159, 84, 180, 157, 191, 168, 190, 152, 105, 165, 251, 186, 25, 96, 255, 254, 121, 3, 45, 8, 73, 73, 225, 185, 212, 171, 190, 73, 180, 94, 171, 219, 156, 182, 144, 194, 57, 1, 17, 189, 253, 37, 230, 115, 221, 214, 237, 212, 205, 185, 92, 102, 109, 198, 47, 85, 115, 250, 201, 19, 234, 117, 227, 11, 250, 32, 30, 98, 207, 60, 196, 120, 241, 112, 11, 35, 99, 109, 219, 253, 189, 233, 124, 250, 1, 69, 109, 213, 135, 59, 79, 62, 55, 253, 210, 136, 65, 218, 120, 189, 183, 139, 5, 144, 244, 227, 116, 253, 187, 11, 241, 5, 31, 203, 85, 0, 249, 233, 9, 181, 178, 72, 218, 147, 237, 133, 64, 244, 157, 130, 226, 136, 245, 30, 70, 50, 228, 235, 5, 29, 209, 111, 25, 124, 74, 73, 31, 108, 167, 210, 143, 88, 209, 195, 10, 110, 176, 113, 163, 111, 163, 127, 95, 57, 41, 104, 16, 0, 133, 35, 28, 13, 227, 235, 228, 67, 46, 94, 73, 163, 101, 211, 76, 203, 242, 137, 183, 49, 20, 215, 129, 115, 71, 150, 122, 81, 16, 224, 83, 225, 57, 31, 231, 132, 182, 129, 116, 229, 222, 199, 63, 81, 39, 255, 9, 207, 51, 148, 70, 159, 176, 222, 235, 174, 161, 168, 197, 77, 113, 147, 51, 78, 11, 104, 106, 70, 236, 199, 21, 113, 21, 210, 82, 77, 151, 48, 76, 72, 140, 171, 29, 123, 154, 180, 48, 90, 51, 57, 227, 13, 187, 39, 116, 75, 148, 241, 70, 190, 149, 195, 226, 107, 127, 213, 137, 222, 139, 158, 23, 113, 92, 125, 192, 198, 117, 183, 180, 87, 56, 15, 18, 178, 228, 7, 181, 238, 94, 7, 185, 165, 233, 141 }),
                new BigInteger(new byte[] { 201, 200, 177, 252, 196, 42, 81, 206, 148, 34, 113, 177, 62, 102, 173, 213, 220, 61, 239, 134, 134, 176, 47, 135, 119, 32, 155, 128, 132, 229, 113, 135, 207, 203, 246, 253, 195, 187, 239, 44, 230, 64, 125, 144, 114, 86, 118, 135, 176, 49, 220, 103, 249, 6, 44, 25, 152, 42, 132, 164, 142, 209, 74, 175, 176, 233, 226, 115, 112, 176, 18, 33, 78, 194, 160, 69, 50, 159, 20, 33, 169, 20, 207, 82, 8, 9, 142, 200, 159, 221, 101, 152, 55, 74, 158, 20, 245, 206, 223, 136, 115, 3, 22, 0, 31, 187, 25, 55, 101, 230, 230, 249, 22, 98, 251, 121, 1, 247, 167, 167, 91, 25, 211, 146, 75, 121, 233, 3, 36, 139, 80, 100, 191, 139, 174, 165, 253, 207, 51, 76, 240, 86, 99, 185, 246, 107, 130, 212, 79, 36, 19, 254, 39, 56, 45, 90, 47, 131, 180, 165, 209, 21, 19, 217, 149, 130, 57, 139, 180, 85, 123, 49, 195, 90, 182, 5, 182, 26, 155, 223, 247, 21, 180, 255, 120, 81, 61, 134, 94, 83, 118, 122, 26, 142, 122, 218, 67, 137, 35, 116, 49, 247, 32, 250, 168, 225, 58, 204, 130, 39, 163, 20, 232, 213, 174, 188, 97, 19, 97, 67, 68, 179, 44, 96, 223, 1, 22, 53, 223, 219, 193, 44, 247, 38, 173, 0, 244, 186, 87, 151, 23, 113, 4, 54, 175, 97, 146, 69, 223, 189, 121, 48, 40, 223, 48, 126, 202, 192, 27, 166, 202, 4, 236, 166, 102, 118, 28, 193, 39, 174, 200, 167, 188, 202, 44, 211, 174, 246, 176, 125, 111, 209, 85, 34, 98, 145, 4, 24, 246, 46, 54, 52, 158, 176, 204, 19, 132, 94, 208, 57, 68, 208, 29, 49, 245, 210, 116, 94, 178, 92, 208, 211, 15, 157, 55, 249, 174, 202, 67, 212, 114, 4, 202, 249, 120, 148, 8, 11, 253, 25, 79, 79, 38, 119, 57, 158, 179, 36, 44, 144, 29, 166, 102, 153, 93, 98, 36, 1, 121, 211, 100, 186, 26, 251, 198, 64, 6, 36, 185, 97, 201, 104, 96, 153, 13, 128, 173, 152, 100, 152, 69, 138, 157, 179, 183, 203, 157, 48, 155, 21, 111, 7, 146, 42, 180, 149, 52, 42, 176, 249, 152, 143, 106, 203, 198, 180, 70, 204, 140, 221, 181, 140, 156, 225, 141, 218, 28, 103, 68, 251, 162, 214, 136, 164, 241, 194, 214, 53, 228, 120, 75, 130, 231, 107, 0, 181, 31, 210, 236, 2, 133, 209, 163, 190, 95, 227, 106, 177, 156, 218, 120, 205, 230, 181, 86, 194 }),
                new BigInteger(new byte[] { 31, 241, 161, 51, 117, 216, 167, 196, 193, 148, 84, 216, 183, 131, 35, 223, 214, 101, 149, 188, 223, 102, 254, 169, 137, 218, 93, 160, 163, 73, 5, 2, 91, 120, 185, 115, 216, 128, 192, 50, 80, 88, 249, 163, 230, 3, 216, 81, 162, 164, 13, 135, 18, 193, 237, 184, 241, 248, 167, 154, 122, 9, 33, 120, 144, 83, 125, 64, 204, 176, 7, 87, 164, 98, 107, 136, 108, 90, 194, 134, 2, 12, 253, 255, 58, 125, 238, 153, 138, 0, 130, 222, 216, 36, 185, 71, 53, 131, 20, 148, 133, 151, 133, 240, 155, 183, 226, 176, 84, 102, 75, 152, 55, 32, 152, 109, 11, 32, 61, 18, 100, 226, 180, 204, 47, 41, 125, 127, 170, 34, 82, 233, 69, 45, 2, 107, 134, 12, 100, 130, 155, 62, 69, 37, 180, 233, 167, 183, 142, 74, 60, 21, 70, 1, 49, 162, 225, 184, 192, 114, 58, 98, 5, 50, 20, 98, 81, 9, 32, 44, 230, 253, 187, 186, 116, 38, 157, 145, 22, 70, 129, 197, 241, 94, 231, 58, 164, 67, 253, 93, 137, 204, 74, 241, 71, 58, 56, 19, 17, 83, 152, 157, 153, 98, 101, 62, 90, 139, 141, 9, 4, 80, 162, 123, 242, 117, 113, 165, 74, 216, 188, 13, 13, 193, 161, 25, 178, 201, 113, 59, 237, 138, 128, 195, 173, 227, 42, 144, 251, 168, 225, 24, 183, 241, 88, 58, 42, 208, 254, 209, 200, 146, 199, 56, 151, 137, 143, 217, 192, 74, 139, 143, 24, 82, 29, 121, 206, 223, 204, 14, 75, 226, 171, 163, 246, 163, 87, 27, 104, 104, 239, 226, 201, 160, 113, 90, 195, 71, 44, 35, 84, 82, 205, 175, 30, 84, 254, 100, 212, 223, 105, 73, 3, 241, 134, 232, 230, 37, 66, 242, 51, 116, 112, 87, 92, 167, 196, 120, 223, 193, 183, 134, 178, 170, 187, 247, 18, 251, 131, 163, 164, 240, 69, 136, 253, 164, 223, 216, 24, 139, 117, 157, 7, 58, 238, 222, 226, 117, 83, 118, 143, 134, 134, 18, 41, 228, 151, 88, 54, 144, 212, 234, 160, 159, 77, 66, 47, 232, 89, 183, 199, 234, 95, 85, 77, 24, 128, 120, 187, 218, 20, 157, 167, 14, 19, 31, 107, 49, 94, 27, 13, 135, 162, 136, 181, 137, 253, 101, 236, 171, 177, 252, 26, 158, 173, 163, 38, 49, 198, 16, 99, 32, 157, 91, 122, 56, 67, 146, 140, 48, 216, 251, 135, 64, 170, 64, 73, 215, 158, 196, 27, 27, 140, 16, 42, 26, 129, 9, 97, 112, 249, 57, 109, 78, 184, 237, 226, 128, 52, 121, 104, 89, 241, 168, 66, 130, 206, 170, 19, 148, 160, 4, 23, 192, 126, 119, 97, 194, 237, 82, 95, 19, 152, 18, 55, 32, 243, 7, 115, 126, 167, 169, 208, 57, 89, 60, 87, 63, 15, 72, 104, 107, 69, 47, 244, 123, 84, 197, 25, 233, 201, 32, 146, 226, 25, 84, 193, 222, 113, 34, 143, 198, 55, 149, 236, 246, 52, 221, 103, 65, 31, 34, 179, 165, 234, 243, 206, 104, 29, 187, 203, 45, 237, 137, 180, 135, 22, 238, 31, 4, 123, 121, 205, 237, 184, 161, 52, 175, 67, 254, 71, 118, 93, 216, 201, 167, 109, 237, 247, 38, 133, 89, 192, 182, 86, 97, 216, 173, 225, 101, 224, 69, 190, 98, 198, 166, 84, 198, 236, 72, 190, 30, 36, 31, 115, 84, 207, 21, 192, 13, 25, 40, 246, 248, 126, 33, 184, 23, 150, 87, 201, 134, 206, 222, 29, 1, 184, 250, 17, 34, 82, 167, 1, 174, 104, 103, 52, 160, 66, 2, 145, 194, 237, 211, 75, 184, 79, 197, 228, 164, 238, 61, 48, 149, 127, 56, 79, 70, 2, 113, 155, 213, 4, 157, 186, 114, 39, 72, 144, 23, 236, 23, 212, 148, 197, 7, 153, 250, 146, 101, 178, 243, 22, 108, 52, 20, 119, 108, 47, 48, 23, 238, 158, 208, 102, 188, 189, 79, 25, 217, 145, 6, 52, 205, 184, 214, 105, 155, 207, 120, 159, 84, 180, 157, 191, 168, 190, 152, 105, 165, 251, 186, 25, 96, 255, 254, 121, 3, 45, 8, 73, 73, 225, 185, 212, 171, 190, 73, 180, 94, 171, 219, 156, 182, 144, 194, 57, 1, 17, 189, 253, 37, 230, 115, 221, 214, 237, 212, 205, 185, 92, 102, 109, 198, 47, 85, 115, 250, 201, 19, 234, 117, 227, 11, 250, 32, 30, 98, 207, 60, 196, 120, 241, 112, 11, 35, 99, 109, 219, 253, 189, 233, 124, 250, 1, 69, 109, 213, 135, 59, 79, 62, 55, 253, 210, 136, 65, 218, 120, 189, 183, 139, 5, 144, 244, 227, 116, 253, 187, 11, 241, 5, 31, 203, 85, 0, 249, 233, 9, 181, 178, 72, 218, 147, 237, 133, 64, 244, 157, 130, 226, 136, 245, 30, 70, 50, 228, 235, 5, 29, 209, 111, 25, 124, 74, 73, 31, 108, 167, 210, 143, 88, 209, 195, 10, 110, 176, 113, 163, 111, 163, 127, 95, 57, 41, 104, 16, 0, 133, 35, 28, 13, 227, 235, 228, 67, 46, 94, 73, 163, 101, 211, 76, 203, 242, 137, 183, 49, 20, 215, 129, 115, 71, 150, 122, 81, 16, 224, 83, 225, 57, 31, 231, 132, 182, 129, 116, 229, 222, 199, 63, 81, 39, 255, 9, 207, 51, 148, 70, 159, 176, 222, 235, 174, 161, 168, 197, 77, 113, 147, 51, 78, 11, 104, 106, 70, 236, 199, 21, 113, 21, 210, 82, 77, 151, 48, 76, 72, 140, 171, 29, 123, 154, 180, 48, 90, 51, 57, 227, 13, 187, 39, 116, 75, 148, 241, 70, 190, 149, 195, 226, 107, 127, 213, 137, 222, 139, 158, 23, 113, 92, 125, 192, 198, 117, 183, 180, 87, 56, 15, 18, 178, 228, 7, 181, 238, 94, 7, 185, 165, 233, 141 })
            };

            yield return new object[]
            {
                new BigInteger(new byte[] { 198, 217, 4, 222, 16, 231, 19, 145, 51, 164, 49, 225, 215, 187, 74, 123, 65, 118, 208, 50, 202, 68, 14, 87, 37, 153, 10, 0, 126, 155, 50, 20, 61, 175, 88, 248, 231, 35, 116, 233, 144, 45, 44, 63, 57, 207, 196, 31, 46, 81, 164, 63, 133, 143, 60, 130, 203, 173, 185, 119, 221, 19, 18, 152, 113, 29, 163, 40, 98, 81, 147, 30, 1, 231, 162, 157, 23, 207, 30, 85, 212, 235, 209, 45, 246, 243, 142, 174, 115, 213, 45, 58, 105, 253, 12, 124, 26, 148, 205, 28, 44, 155, 188, 220, 190, 134, 61, 158, 237, 158, 106, 252, 143, 100, 162, 241, 65, 229, 161, 126, 142, 244, 181, 141, 36, 89, 180, 3, 218, 38, 130, 131, 1, 1, 40, 56, 47, 245, 110, 111, 181, 239, 192, 212, 106, 48, 171, 5, 90, 26, 58, 53, 242, 79, 141, 55, 230, 47, 183, 101, 210, 58, 196, 199, 27, 231, 250, 141, 60, 106, 194, 76, 118, 49, 200, 159, 52, 225, 34, 243, 174, 174, 168, 191, 235, 77, 145, 178, 116, 240, 82, 255, 62, 9, 157, 123, 42, 248, 184, 131, 54, 29, 201, 240, 88, 237, 190, 192, 134, 238, 3, 4, 12, 195, 182, 35, 145, 5, 219, 206, 86, 72, 24, 75, 23, 194, 14, 108, 147, 77, 116, 59, 40, 158, 189, 144, 229, 183, 103, 253, 142, 208, 43, 134, 237, 77, 243, 122, 82, 122, 234, 37, 28, 234, 45, 238, 4, 125, 217, 149, 223, 82, 44, 57, 122, 199, 220, 109, 6, 38, 61, 217, 157, 221, 63, 134, 28, 146, 94, 202, 207, 147, 25, 25, 99, 78, 30, 62, 113, 207, 139, 104, 222, 209, 29, 60, 163, 242, 12, 37, 113, 133, 115, 44, 61, 16, 135, 62, 170, 167, 210, 114, 30, 15, 197, 76, 56, 19, 214, 43, 169, 158, 252, 55, 154, 213, 250, 204, 192, 3, 226, 41, 134, 57, 88, 74, 31, 237, 220, 82, 198, 223, 147, 202, 252, 25, 73, 207, 12, 209, 3, 143, 28, 224, 124, 211, 136, 60, 145, 103, 22, 187, 126, 166, 196, 168, 235, 229, 182, 123, 44, 74, 55, 132, 101, 201, 10, 49, 58, 129, 140, 42, 192, 238, 0, 82, 12, 166, 189, 133, 193, 227, 91, 116, 59, 10, 96, 237, 5, 84, 45, 99, 25, 144, 165, 184, 88, 152, 122, 178, 200, 87, 1, 16, 219, 235, 251, 144, 166, 113, 156, 69, 40, 49, 185, 72, 239, 194, 72, 90, 104, 29, 43, 230, 84, 40, 17, 47, 72, 71, 250, 116, 43, 10, 188, 106, 225, 75, 153, 80, 14, 207, 70, 148, 184, 231, 59, 232, 215, 93, 201, 150, 80, 31, 74, 171, 43, 27, 188, 178, 217, 204, 72, 95, 119, 40, 26, 30, 116, 79, 102, 164, 41, 248, 76, 152, 214, 151, 46, 108, 236, 41, 88, 234, 125, 100, 35, 11, 191, 16, 63, 197, 142, 230, 118, 65, 135, 231, 29, 143, 204, 29, 115, 135, 246, 37, 108, 88, 211, 177, 212, 147, 237, 79, 198, 176, 116, 93, 225, 19, 4, 15, 94, 100, 16, 161, 21, 66, 129, 10, 17, 239, 122, 158, 160, 207, 182, 208, 244, 88, 247, 203, 55, 14, 59, 171, 216, 63, 32, 198, 205, 41, 5, 10, 139, 188, 190, 15, 119, 85, 16, 42, 214, 201, 159, 229, 25, 79, 175, 246, 249, 164, 105, 137, 27, 56, 216, 80, 40, 100, 200, 214, 34, 187, 61, 232, 234, 112, 18, 227, 16, 167, 88, 159, 72, 205, 245, 110, 110, 86, 146, 137, 143, 41, 204, 41, 192, 123, 238, 83, 228, 110, 230, 18, 140, 58, 7, 26, 140, 183, 250, 168, 60, 203, 229, 82, 75, 201, 65, 78, 66, 105, 225, 209, 210, 63, 70, 248, 238, 19, 39, 110, 129, 190, 254, 162, 192, 70, 18, 13, 197, 218, 226, 182, 92, 233, 76, 195, 68, 122, 221, 246, 58, 160, 17, 123, 140, 21, 64, 77, 243, 15, 43, 120, 143, 139, 39, 125, 205, 67, 146, 5, 182, 51, 137, 103, 135, 43, 117, 233, 242, 130, 179, 116, 54, 49, 52, 225, 173, 131, 21, 170, 5, 197, 26, 181, 174, 106, 38, 208, 194, 206, 246, 1, 231, 216, 63, 6, 23, 169, 243, 97, 159, 87, 90, 113, 91, 183 }),
                new BigInteger(new byte[] { 94, 87, 69, 208, 73, 45, 231, 255, 241, 25, 211, 184, 28, 4, 165, 39, 36, 139, 178, 140, 93, 241, 45, 202, 4, 144, 180, 11, 174, 107, 168, 90, 247, 40, 200, 120, 124, 254, 173, 211, 0, 217, 78, 237, 67, 240, 214, 92, 250, 48, 53, 230, 168, 42, 228, 108, 42, 122, 204, 185, 121, 220, 81, 133, 113, 120 }),
                new BigInteger(new byte[] { 36, 49, 74, 174, 90, 20, 251, 144, 37, 190, 4, 154, 244, 191, 239, 162, 101, 1, 131, 191, 39, 54, 60, 33, 42, 41, 191, 11, 44, 7, 219, 110, 52, 216, 32, 113, 100, 34, 34, 189, 145, 6, 123, 44, 125, 191, 155, 124, 40, 130, 217, 37, 46, 186, 32, 239, 245, 39, 134, 49, 87, 240, 99, 29, 227, 149, 163, 40, 98, 81, 147, 30, 1, 231, 162, 157, 23, 207, 30, 85, 212, 235, 209, 45, 246, 243, 142, 174, 115, 213, 45, 58, 105, 253, 12, 124, 26, 148, 205, 28, 44, 155, 188, 220, 190, 134, 61, 158, 237, 158, 106, 252, 143, 100, 162, 241, 65, 229, 161, 126, 142, 244, 181, 141, 36, 89, 180, 3, 218, 38, 130, 131, 1, 1, 40, 56, 47, 245, 110, 111, 181, 239, 192, 212, 106, 48, 171, 5, 90, 26, 58, 53, 242, 79, 141, 55, 230, 47, 183, 101, 210, 58, 196, 199, 27, 231, 250, 141, 60, 106, 194, 76, 118, 49, 200, 159, 52, 225, 34, 243, 174, 174, 168, 191, 235, 77, 145, 178, 116, 240, 82, 255, 62, 9, 157, 123, 42, 248, 184, 131, 54, 29, 201, 240, 88, 237, 190, 192, 134, 238, 3, 4, 12, 195, 182, 35, 145, 5, 219, 206, 86, 72, 24, 75, 23, 194, 14, 108, 147, 77, 116, 59, 40, 158, 189, 144, 229, 183, 103, 253, 142, 208, 43, 134, 237, 77, 243, 122, 82, 122, 234, 37, 28, 234, 45, 238, 4, 125, 217, 149, 223, 82, 44, 57, 122, 199, 220, 109, 6, 38, 61, 217, 157, 221, 63, 134, 28, 146, 94, 202, 207, 147, 25, 25, 99, 78, 30, 62, 113, 207, 139, 104, 222, 209, 29, 60, 163, 242, 12, 37, 113, 133, 115, 44, 61, 16, 135, 62, 170, 167, 210, 114, 30, 15, 197, 76, 56, 19, 214, 43, 169, 158, 252, 55, 154, 213, 250, 204, 192, 3, 226, 41, 134, 57, 88, 74, 31, 237, 220, 82, 198, 223, 147, 202, 252, 25, 73, 207, 12, 209, 3, 143, 28, 224, 124, 211, 136, 60, 145, 103, 22, 187, 126, 166, 196, 168, 235, 229, 182, 123, 44, 74, 55, 132, 101, 201, 10, 49, 58, 129, 140, 42, 192, 238, 0, 82, 12, 166, 189, 133, 193, 227, 91, 116, 59, 10, 96, 237, 5, 84, 45, 99, 25, 144, 165, 184, 88, 152, 122, 178, 200, 87, 1, 16, 219, 235, 251, 144, 166, 113, 156, 69, 40, 49, 185, 72, 239, 194, 72, 90, 104, 29, 43, 230, 84, 40, 17, 47, 72, 71, 250, 116, 43, 10, 188, 106, 225, 75, 153, 80, 14, 207, 70, 148, 184, 231, 59, 232, 215, 93, 201, 150, 80, 31, 74, 171, 43, 27, 188, 178, 217, 204, 72, 95, 119, 40, 26, 30, 116, 79, 102, 164, 41, 248, 76, 152, 214, 151, 46, 108, 236, 41, 88, 234, 125, 100, 35, 11, 191, 16, 63, 197, 142, 230, 118, 65, 135, 231, 29, 143, 204, 29, 115, 135, 246, 37, 108, 88, 211, 177, 212, 147, 237, 79, 198, 176, 116, 93, 225, 19, 4, 15, 94, 100, 16, 161, 21, 66, 129, 10, 17, 239, 122, 158, 160, 207, 182, 208, 244, 88, 247, 203, 55, 14, 59, 171, 216, 63, 32, 198, 205, 41, 5, 10, 139, 188, 190, 15, 119, 85, 16, 42, 214, 201, 159, 229, 25, 79, 175, 246, 249, 164, 105, 137, 27, 56, 216, 80, 40, 100, 200, 214, 34, 187, 61, 232, 234, 112, 18, 227, 16, 167, 88, 159, 72, 205, 245, 110, 110, 86, 146, 137, 143, 41, 204, 41, 192, 123, 238, 83, 228, 110, 230, 18, 140, 58, 7, 26, 140, 183, 250, 168, 60, 203, 229, 82, 75, 201, 65, 78, 66, 105, 225, 209, 210, 63, 70, 248, 238, 19, 39, 110, 129, 190, 254, 162, 192, 70, 18, 13, 197, 218, 226, 182, 92, 233, 76, 195, 68, 122, 221, 246, 58, 160, 17, 123, 140, 21, 64, 77, 243, 15, 43, 120, 143, 139, 39, 125, 205, 67, 146, 5, 182, 51, 137, 103, 135, 43, 117, 233, 242, 130, 179, 116, 54, 49, 52, 225, 173, 131, 21, 170, 5, 197, 26, 181, 174, 106, 38, 208, 194, 206, 246, 1, 231, 216, 63, 6, 23, 169, 243, 97, 159, 87, 90, 113, 91, 183 })
            };

            yield return new object[]
            {
                new BigInteger(new byte[] { 244, 59, 44, 170, 37, 221, 223, 139, 132, 163, 64, 134, 19, 185, 87, 16, 126, 53, 194, 177, 204, 67, 92, 78, 2, 50, 244, 90, 36, 135, 200, 124, 203, 47, 226, 32, 205, 133, 15, 8 }),
                new BigInteger(new byte[] { 158, 96, 200, 233, 243, 184, 0, 33, 203, 210, 80, 174, 198, 244, 177, 223, 221, 168, 243, 233, 133, 103, 252, 219, 195, 187, 227, 215, 54, 66, 248, 37, 186, 232, 45, 227, 147, 100, 14, 121, 244, 56, 89, 181, 120, 205, 4, 59, 48, 65, 239, 221, 28, 30, 68, 55, 99, 237, 38, 56, 213, 40, 234, 136, 218, 42, 244, 222, 198, 205, 75, 121, 153, 122, 245, 76, 13, 71, 98, 246, 182, 167, 166, 245, 82, 27, 9, 155, 164, 246, 127, 166, 62, 213, 164, 203, 78, 208, 219, 77, 217, 38, 117, 99, 245, 118, 67, 161, 41, 247, 41, 28, 138, 48, 129, 45, 130, 244, 54, 190, 33, 89, 57, 208, 78, 165, 58, 196, 213, 41, 253, 61, 108, 21, 28, 143, 50, 67, 255, 219, 215, 103, 113, 172, 204, 99, 27, 14, 83, 119, 72, 89, 154, 28, 44, 128, 237, 165, 20, 80, 60, 127, 204, 0, 250, 236, 176, 117, 19, 242, 250, 63, 244, 90, 231, 186, 231, 140, 3, 234, 137, 44, 112, 94, 224, 163, 162, 80, 232, 156, 162, 141, 46, 175, 159, 88, 155, 113, 178, 224 }),
                new BigInteger(new byte[] { 146, 156, 244, 147, 25, 150, 224, 172, 79, 118, 145, 52, 218, 173, 9, 240, 91, 222, 181, 155, 82, 171, 88, 42, 198, 237, 215, 50, 91, 201, 192, 162, 133, 24, 16, 4, 97, 234, 29, 129, 244, 56, 89, 181, 120, 205, 4, 59, 48, 65, 239, 221, 28, 30, 68, 55, 99, 237, 38, 56, 213, 40, 234, 136, 218, 42, 244, 222, 198, 205, 75, 121, 153, 122, 245, 76, 13, 71, 98, 246, 182, 167, 166, 245, 82, 27, 9, 155, 164, 246, 127, 166, 62, 213, 164, 203, 78, 208, 219, 77, 217, 38, 117, 99, 245, 118, 67, 161, 41, 247, 41, 28, 138, 48, 129, 45, 130, 244, 54, 190, 33, 89, 57, 208, 78, 165, 58, 196, 213, 41, 253, 61, 108, 21, 28, 143, 50, 67, 255, 219, 215, 103, 113, 172, 204, 99, 27, 14, 83, 119, 72, 89, 154, 28, 44, 128, 237, 165, 20, 80, 60, 127, 204, 0, 250, 236, 176, 117, 19, 242, 250, 63, 244, 90, 231, 186, 231, 140, 3, 234, 137, 44, 112, 94, 224, 163, 162, 80, 232, 156, 162, 141, 46, 175, 159, 88, 155, 113, 178, 224 })
            };

            yield return new object[]
            {
                new BigInteger(new byte[] { 165, 251, 97, 124, 9, 100, 64, 52, 114, 152, 96, 161, 41, 198, 16, 15, 109, 31, 86, 73, 95, 140, 118, 79, 108, 17, 69, 186, 143, 30, 118, 218, 35, 55, 227, 48, 14, 225, 215, 63, 98, 27, 118, 4, 66, 236, 4, 232, 143, 73, 45, 81, 92, 150, 83, 25, 132, 18, 16, 248, 31, 134, 165, 83, 33, 134, 126, 242, 227, 223, 0, 140, 72, 23, 230, 68, 22, 113, 13, 128, 12, 92, 42, 70, 241, 36, 125, 140, 228, 202, 171, 252, 209, 222, 31, 220, 118, 34, 226, 187, 110, 18, 5, 176, 72, 161, 9, 69, 176, 14, 2, 19, 4, 143, 235, 194, 91, 94, 97, 252, 8, 241, 13, 24, 51, 4, 186, 105, 247, 10, 206, 243, 142, 81, 18, 250, 87, 121, 253, 80, 27, 56, 220, 213, 200, 152, 247, 65 }),
                new BigInteger(new byte[] { 92, 128, 23, 192, 229, 178, 124, 4, 236, 125, 68, 230, 160, 77, 166, 64, 14, 132, 179, 125, 241, 107, 225, 97, 17, 225, 208, 21, 55, 80, 154, 12, 120, 117, 154, 137, 182, 50, 169, 158, 149, 245, 106, 252, 185, 105, 122, 234, 54, 149, 185, 20, 68, 196, 2, 241, 159, 182, 175, 4, 226, 103, 204, 156, 227, 55, 109, 42, 178, 28, 137, 219, 219, 20, 231, 251, 0, 229, 167, 167, 103, 229, 223, 162, 150, 133, 200, 180, 115, 169, 234, 0, 131, 165, 188, 46, 41, 206, 25, 129, 252, 80, 55, 25, 11, 221, 57, 47, 221, 6, 240, 185, 14, 7, 156, 253, 136, 42, 5, 93, 111, 185, 182, 9, 50, 137, 88, 53, 88, 184, 210, 50, 203, 38, 171, 23, 173, 198, 151, 185, 75, 152, 215, 109, 184, 48, 242, 124, 8, 190, 166, 255, 201, 188, 18, 67, 153, 46, 230, 130, 133, 3, 215, 36, 52, 190, 238, 232, 91, 133, 80, 193, 146, 76, 17, 214, 226, 73, 80, 2, 150, 219, 45, 153, 18, 210, 105, 15, 19, 104, 125, 126, 223, 20, 51, 72, 193, 178, 56, 250, 65, 9, 33, 130, 110, 228, 108, 124, 170, 60, 97, 79, 221, 227, 235, 170, 214, 61, 17, 97, 85, 222, 212, 242, 7, 209, 226, 126, 24, 201, 21, 47, 16, 86, 192, 141, 186, 171, 43, 45, 102, 236, 101, 215, 7, 45, 161, 252, 41, 136, 114, 131, 161, 215, 165, 99, 53, 46, 122, 157, 2, 237, 100, 225, 39, 49, 62, 135, 34 }),
                new BigInteger(new byte[] { 1, 124, 121, 60, 239, 22, 189, 56, 94, 22, 165, 135, 202, 19, 183, 79, 123, 163, 9, 199, 80, 248, 87, 177, 125, 242, 21, 208, 198, 110, 16, 231, 155, 172, 125, 186, 196, 19, 129, 222, 247, 16, 225, 0, 252, 85, 127, 210, 198, 222, 230, 101, 160, 90, 86, 10, 36, 201, 191, 252, 1, 238, 113, 240, 4, 190, 235, 28, 150, 252, 137, 103, 36, 44, 205, 64, 23, 86, 181, 39, 116, 65, 10, 233, 135, 170, 69, 65, 88, 116, 150, 253, 84, 132, 220, 10, 160, 240, 251, 60, 107, 99, 60, 201, 83, 126, 67, 116, 141, 21, 242, 204, 18, 150, 135, 192, 228, 136, 102, 89, 120, 170, 196, 33, 101, 141, 18, 159, 79, 195, 160, 38, 90, 120, 189, 17, 5, 64, 149, 10, 103, 208, 179, 67, 129, 201, 233, 190, 8, 190, 166, 255, 201, 188, 18, 67, 153, 46, 230, 130, 133, 3, 215, 36, 52, 190, 238, 232, 91, 133, 80, 193, 146, 76, 17, 214, 226, 73, 80, 2, 150, 219, 45, 153, 18, 210, 105, 15, 19, 104, 125, 126, 223, 20, 51, 72, 193, 178, 56, 250, 65, 9, 33, 130, 110, 228, 108, 124, 170, 60, 97, 79, 221, 227, 235, 170, 214, 61, 17, 97, 85, 222, 212, 242, 7, 209, 226, 126, 24, 201, 21, 47, 16, 86, 192, 141, 186, 171, 43, 45, 102, 236, 101, 215, 7, 45, 161, 252, 41, 136, 114, 131, 161, 215, 165, 99, 53, 46, 122, 157, 2, 237, 100, 225, 39, 49, 62, 135, 34 })
            };

            yield return new object[]
            {
                new BigInteger(new byte[] { 240, 43, 18, 227, 250, 104, 121, 253, 235, 217, 48, 229, 84, 143, 87, 145, 142, 57, 177, 93, 88, 139, 48, 142, 43, 120, 136, 131, 176, 213, 238, 25, 217, 202, 152, 71, 113, 189, 220, 60, 234, 55, 66, 160, 108, 28, 152, 212, 138, 149, 163, 5, 214, 197, 6, 100, 250, 131, 184, 129, 224, 245, 76, 21, 234, 22, 12, 138, 247, 16, 31, 208, 93, 116, 115, 32, 73, 143, 33, 14, 223, 180, 249, 26, 49, 233, 67, 20, 196, 52, 76, 238, 4, 90, 91, 244, 234, 45, 181, 86, 16, 14, 220, 121, 118, 211, 167, 97, 81, 229, 27, 107, 98, 169, 161, 44, 251, 50, 228, 0, 211, 247, 197, 194, 195, 49, 204, 3, 24, 126, 53, 28, 217, 203, 253, 209, 215, 127, 163, 94, 66, 225, 194, 222, 25, 224, 139, 90, 184, 47, 249, 184, 72, 180, 130, 25, 73, 26, 182, 69, 6, 164, 72, 211, 176, 255, 146, 151, 172, 208, 85, 123, 142, 133, 190, 241, 53, 230, 169, 227, 165, 113, 74, 233, 132, 124, 211, 36, 72, 228, 135, 189, 201, 94, 87, 158, 153, 238, 46, 26, 78, 244, 174, 231, 217, 125, 42, 195, 246, 145, 228, 98, 112, 211, 160, 149, 89, 95, 78, 51, 43, 109, 78, 199, 73, 152, 177, 48, 45, 31, 88, 70, 184, 143, 148, 176, 194, 99, 15, 7, 82, 15, 45, 182, 0, 36, 76, 246, 189, 193, 69, 57, 160, 250, 238, 225, 165, 230, 158, 65, 204, 250, 149, 214, 56, 157, 170, 225, 62, 239, 211, 245, 79, 71, 225, 28, 63, 151, 199, 36, 75, 187, 114, 107, 218, 30, 165, 189, 161, 179, 11, 219, 197, 206, 58, 88, 122, 86, 126, 98, 122, 107, 183, 206, 238, 79, 234, 88, 25, 210, 161, 13, 31, 121, 245, 17, 135, 41, 252, 26, 247, 236, 63, 139, 228, 247, 48, 129, 12, 137, 161, 232, 25, 100, 170, 224, 3, 163, 124, 138, 52, 77, 164, 207, 17, 253, 188, 75, 216, 41, 208, 80, 90, 100, 106, 141, 43, 44, 233, 247, 31 }),
                new BigInteger(new byte[] { 76, 144, 48, 184, 244, 187, 206, 20, 13, 227, 172, 113, 229, 170, 71, 112, 122, 231, 58, 228, 168, 226, 184, 81, 142, 181, 249, 29, 181, 215, 185, 133, 107, 202, 0, 20, 158, 89, 9, 1, 125, 195, 28, 236, 163, 232, 126, 192, 69, 186, 178, 2, 189, 58, 192, 105, 215, 223, 42, 62, 193, 177, 95, 54, 41, 39, 5, 27, 169, 51, 209, 32, 222, 57, 103, 229, 198, 203, 173, 165, 55, 57, 215, 250, 37, 183, 199, 49, 10, 151, 61, 191, 46, 202, 63, 203, 100, 229, 194, 124, 226, 99, 22, 18, 232, 145, 36, 132, 210, 218, 163, 12, 50, 132, 7, 136, 217, 100, 17, 114, 95, 212, 16, 18, 246, 17, 241, 19, 249, 156, 129, 224, 8, 48, 194, 212, 34, 197, 18, 148, 146, 67, 95, 47, 243, 48, 141, 170, 195, 183, 242, 255, 212, 80, 28, 14, 82, 3, 28, 214, 159, 17, 139, 54, 89, 195, 3, 1, 194, 50, 101, 19, 82, 124, 223, 28, 117, 224, 31, 197, 132, 71, 79, 65, 170, 129, 11, 184, 20, 180, 129, 31, 169, 59, 244, 129, 184, 40, 214, 48, 45, 140, 231, 144, 82, 222, 173, 87, 112, 0, 153, 113, 228, 86, 82, 87, 193, 73, 140, 215, 183, 74, 236, 14, 177, 69, 106, 23, 135, 94, 100, 76, 10, 238, 151, 247, 96, 191, 238, 203, 212, 179, 72, 20, 27, 16, 59, 82, 232, 157, 192, 110, 156, 255, 121, 226, 159, 217, 223, 12, 115, 150, 207, 18, 155, 76, 103, 245, 191 }),
                new BigInteger(new byte[] { 60, 188, 66, 155, 239, 36, 72, 18, 249, 188, 221, 86, 58, 58, 159, 1, 9, 33, 236, 65, 1, 110, 233, 223, 185, 45, 130, 161, 101, 173, 168, 159, 68, 149, 153, 91, 15, 23, 230, 61, 103, 251, 94, 140, 16, 5, 23, 149, 208, 79, 86, 8, 147, 0, 199, 205, 209, 99, 227, 191, 161, 167, 172, 75, 19, 62, 17, 165, 160, 68, 240, 240, 59, 174, 218, 5, 16, 91, 207, 179, 22, 238, 208, 21, 87, 160, 11, 70, 206, 203, 137, 173, 51, 36, 155, 191, 79, 19, 120, 211, 242, 113, 242, 139, 94, 101, 204, 229, 35, 192, 191, 119, 148, 45, 169, 180, 212, 151, 245, 114, 50, 204, 214, 212, 185, 67, 189, 23, 17, 27, 183, 252, 225, 251, 191, 166, 250, 68, 182, 242, 212, 36, 34, 14, 13, 17, 25, 5, 124, 231, 235, 184, 29, 5, 159, 39, 155, 29, 210, 27, 166, 181, 211, 9, 10, 195, 150, 152, 110, 3, 187, 142, 224, 1, 158, 14, 171, 198, 201, 168, 42, 185, 153, 42, 47, 254, 222, 220, 92, 152, 9, 221, 114, 154, 75, 32, 82, 23, 5, 75, 123, 128, 150, 120, 44, 92, 216, 26, 103, 146, 125, 212, 84, 42, 243, 236, 26, 169, 218, 10, 227, 183, 58, 214, 250, 221, 27, 72, 180, 125, 188, 146, 194, 125, 44, 168, 35, 35, 254, 210, 38, 195, 117, 202, 27, 52, 135, 72, 166, 95, 6, 168, 60, 250, 104, 196, 69, 192, 126, 78, 63, 145, 101, 233, 211, 233, 17, 215, 254, 238, 211, 245, 79, 71, 225, 28, 63, 151, 199, 36, 75, 187, 114, 107, 218, 30, 165, 189, 161, 179, 11, 219, 197, 206, 58, 88, 122, 86, 126, 98, 122, 107, 183, 206, 238, 79, 234, 88, 25, 210, 161, 13, 31, 121, 245, 17, 135, 41, 252, 26, 247, 236, 63, 139, 228, 247, 48, 129, 12, 137, 161, 232, 25, 100, 170, 224, 3, 163, 124, 138, 52, 77, 164, 207, 17, 253, 188, 75, 216, 41, 208, 80, 90, 100, 106, 141, 43, 44, 233, 247, 31 })
            };

            yield return new object[]
            {
                new BigInteger(new byte[] { 247, 1, 90, 192, 3, 3, 2, 216, 242, 160, 10, 23, 46, 234, 157, 245, 176, 11, 117, 181, 88, 134, 223, 226, 88, 62, 227, 53, 120, 206, 169, 70, 147, 141, 166, 190, 19, 220, 252, 117, 223, 199, 230, 62, 2, 92, 103, 229, 116, 166, 156, 92, 127, 10, 3, 112, 33, 120, 104, 196, 31, 205, 96, 36, 247, 195, 132, 161, 67, 223, 225, 212, 14, 0, 213, 144, 159, 161, 223, 252, 214, 254, 193, 209, 50, 76, 198, 137, 137, 53, 156, 155, 115, 55, 85, 18, 102, 194, 70, 63, 215, 197, 161, 149, 196, 200, 77, 127, 52, 114, 209, 128, 153, 108, 237, 33, 11, 142, 241, 170, 253, 251, 23, 14, 66, 70, 96, 215, 170, 194, 42, 220, 90, 160, 36, 17, 92, 43, 12, 106, 255, 71, 84, 23, 99, 28, 1, 7, 73, 51, 6, 216, 208, 155, 66, 220, 173, 147, 83, 126, 103, 117, 212, 114, 71, 245, 38, 249, 71, 220, 197, 224, 129, 135, 171, 181, 166, 0, 170, 38, 68, 88, 142, 118, 187, 81, 11, 190, 94, 71, 100, 200, 216, 142, 2, 137, 114, 225, 207, 110, 246, 8 }),
                new BigInteger(new byte[] { 108, 110, 38, 86, 73, 240, 140, 53, 173, 233, 44, 58, 14, 231, 93, 182, 245, 233, 103, 155, 0, 120, 252, 7, 243, 132, 33, 67, 196, 48, 59, 47, 59, 153, 33, 8, 148, 251, 194, 205, 210, 17, 182, 223, 171, 84, 243, 162, 20, 42, 217, 133, 142, 109, 191, 244, 114, 31, 99, 196, 206, 72, 112, 124, 174, 253, 254, 117, 197, 85, 34, 250, 39, 153, 201, 238, 193, 29, 91, 158, 144, 126, 46, 154, 86, 181, 161, 206, 217, 45, 149, 117, 151, 253, 255, 137, 160, 58, 49, 174, 85, 126, 221, 190, 8, 223, 94, 244, 164, 208, 50, 85, 196, 196, 51, 80, 26, 214, 37, 248, 91, 48, 155, 151, 192, 173, 99, 42, 154, 63, 77, 135, 235, 172, 73, 18, 161, 111, 145, 119, 87, 172, 41, 8, 250, 64, 176, 211, 201, 175, 110, 201, 20, 57, 82, 37, 226, 69, 254, 91, 124, 51, 90, 100, 131, 171, 105, 23, 123, 33, 174, 170, 69, 174, 160, 175, 6, 147, 157, 127, 252, 144, 96, 235, 209, 131, 114, 178, 89, 35, 48, 91, 113, 53, 251, 36, 82, 196, 133, 79, 214, 152, 87, 168, 0, 196, 132, 102, 153, 163, 30, 79, 168, 126, 95, 236, 211, 110, 147, 255, 57, 183, 189, 87, 240, 83, 57, 15, 178, 124, 93, 66, 13, 78, 169, 99, 56, 184, 235, 12, 255, 12, 25, 182, 5, 224, 178, 242, 214, 15, 80, 227, 48, 133, 21, 31, 219, 0, 183, 172, 138, 116, 180, 29, 69, 220, 65, 90, 213, 251, 179, 26, 131, 135, 0, 45, 158, 7, 82, 15, 161, 71, 57, 162, 43, 122, 17, 135, 56, 137, 136, 56, 1, 63, 129, 138, 87, 252, 112, 41, 159, 88, 28, 218, 91, 54, 95, 169, 133, 232, 129, 211, 173, 167, 11, 67, 59, 17, 241, 203, 202, 186, 34, 76, 115, 123, 24, 68, 5, 117, 213, 162, 150, 41, 112, 72, 43, 95, 70, 244, 26, 103, 1, 79, 8, 180, 138, 90, 51, 62, 79, 70, 10, 164, 94, 229, 53, 208, 102, 224, 29, 27, 164, 15, 19, 222, 60, 132, 54, 194, 24, 220, 202, 253, 177, 99, 184, 210, 67, 191, 241, 190, 16, 198, 38, 142, 151, 241, 203, 139, 19, 91, 248, 102, 214, 254, 194, 241, 60, 41, 119, 5, 35, 113, 38, 114, 124, 12, 243, 250, 44, 98, 140, 167, 239, 95, 10, 221, 233, 132, 71, 75, 185, 170, 175, 189, 227, 100, 38, 178, 160, 198, 150, 25, 71, 235, 154, 159, 160, 179, 18, 138, 254, 208, 94, 176, 206, 80, 119, 119, 243, 229, 8, 183, 226, 44, 75, 120, 194, 105, 143, 23, 230, 65, 89, 102, 203, 115, 96, 3, 196, 107, 61, 53, 114, 188, 77, 232, 76, 254, 238, 147, 236, 175, 190, 187, 189, 223, 55, 26, 159, 33, 221, 55, 36, 250, 164, 188, 118, 247, 228, 91, 240, 115, 178, 136, 167, 210, 68, 38, 223, 98, 44, 195, 122, 251, 43, 55, 130, 157, 168, 236, 60, 69, 99, 163, 142, 6, 17, 119, 23, 144, 114, 84, 26, 147, 162, 120, 252, 54, 20, 235, 154, 16, 59, 61, 245, 25, 188, 40, 207, 108, 57, 216, 78, 248, 31, 171, 16, 15, 250, 161, 192, 205, 14, 79, 240, 215, 169, 231, 89, 178, 58, 76, 137, 189, 81, 43, 9, 38, 173, 116, 73, 233, 167, 170, 86, 153, 5, 34, 116, 247, 104, 236, 60, 114, 42, 205, 2, 236, 76, 29, 112, 213, 207, 29, 49, 237, 79, 145, 166, 243, 161, 234, 97, 133, 87, 215, 37, 99, 248, 87, 210, 135, 114, 98, 73, 78, 16, 23, 147, 131, 41, 29, 217, 144, 4, 116, 26, 138, 121, 105, 73, 116, 123, 128, 85, 125, 139, 182, 26, 82, 168, 158, 243, 244, 74, 233, 98, 108, 211, 227, 220, 55, 253, 34, 202, 131, 16, 209, 129, 227, 189, 154, 234, 143, 13, 94 }),
                new BigInteger(new byte[] { 99, 112, 128, 22, 77, 243, 142, 13, 160, 138, 55, 81, 60, 209, 251, 171, 166, 245, 220, 80, 89, 254, 219, 234, 75, 195, 4, 121, 60, 255, 228, 117, 206, 38, 200, 198, 167, 215, 191, 67, 178, 217, 156, 30, 174, 176, 90, 136, 137, 208, 117, 226, 13, 120, 194, 100, 148, 151, 203, 136, 238, 21, 209, 160, 165, 193, 131, 23, 9, 53, 4, 207, 54, 153, 158, 127, 97, 191, 58, 155, 103, 125, 240, 107, 137, 1, 104, 88, 99, 99, 49, 17, 11, 53, 85, 156, 6, 253, 119, 237, 44, 68, 127, 84, 205, 167, 172, 115, 217, 66, 4, 214, 93, 49, 33, 114, 37, 100, 23, 163, 89, 44, 179, 165, 2, 244, 195, 1, 69, 2, 120, 99, 70, 77, 110, 35, 253, 154, 157, 225, 86, 244, 125, 31, 93, 93, 177, 218, 18, 227, 116, 161, 229, 212, 148, 1, 144, 217, 81, 218, 227, 168, 46, 215, 202, 160, 144, 16, 195, 253, 115, 139, 199, 53, 76, 101, 173, 147, 71, 166, 64, 233, 238, 97, 141, 213, 125, 112, 184, 106, 148, 35, 74, 196, 253, 173, 196, 165, 85, 190, 204, 161, 87, 168, 0, 196, 132, 102, 153, 163, 30, 79, 168, 126, 95, 236, 211, 110, 147, 255, 57, 183, 189, 87, 240, 83, 57, 15, 178, 124, 93, 66, 13, 78, 169, 99, 56, 184, 235, 12, 255, 12, 25, 182, 5, 224, 178, 242, 214, 15, 80, 227, 48, 133, 21, 31, 219, 0, 183, 172, 138, 116, 180, 29, 69, 220, 65, 90, 213, 251, 179, 26, 131, 135, 0, 45, 158, 7, 82, 15, 161, 71, 57, 162, 43, 122, 17, 135, 56, 137, 136, 56, 1, 63, 129, 138, 87, 252, 112, 41, 159, 88, 28, 218, 91, 54, 95, 169, 133, 232, 129, 211, 173, 167, 11, 67, 59, 17, 241, 203, 202, 186, 34, 76, 115, 123, 24, 68, 5, 117, 213, 162, 150, 41, 112, 72, 43, 95, 70, 244, 26, 103, 1, 79, 8, 180, 138, 90, 51, 62, 79, 70, 10, 164, 94, 229, 53, 208, 102, 224, 29, 27, 164, 15, 19, 222, 60, 132, 54, 194, 24, 220, 202, 253, 177, 99, 184, 210, 67, 191, 241, 190, 16, 198, 38, 142, 151, 241, 203, 139, 19, 91, 248, 102, 214, 254, 194, 241, 60, 41, 119, 5, 35, 113, 38, 114, 124, 12, 243, 250, 44, 98, 140, 167, 239, 95, 10, 221, 233, 132, 71, 75, 185, 170, 175, 189, 227, 100, 38, 178, 160, 198, 150, 25, 71, 235, 154, 159, 160, 179, 18, 138, 254, 208, 94, 176, 206, 80, 119, 119, 243, 229, 8, 183, 226, 44, 75, 120, 194, 105, 143, 23, 230, 65, 89, 102, 203, 115, 96, 3, 196, 107, 61, 53, 114, 188, 77, 232, 76, 254, 238, 147, 236, 175, 190, 187, 189, 223, 55, 26, 159, 33, 221, 55, 36, 250, 164, 188, 118, 247, 228, 91, 240, 115, 178, 136, 167, 210, 68, 38, 223, 98, 44, 195, 122, 251, 43, 55, 130, 157, 168, 236, 60, 69, 99, 163, 142, 6, 17, 119, 23, 144, 114, 84, 26, 147, 162, 120, 252, 54, 20, 235, 154, 16, 59, 61, 245, 25, 188, 40, 207, 108, 57, 216, 78, 248, 31, 171, 16, 15, 250, 161, 192, 205, 14, 79, 240, 215, 169, 231, 89, 178, 58, 76, 137, 189, 81, 43, 9, 38, 173, 116, 73, 233, 167, 170, 86, 153, 5, 34, 116, 247, 104, 236, 60, 114, 42, 205, 2, 236, 76, 29, 112, 213, 207, 29, 49, 237, 79, 145, 166, 243, 161, 234, 97, 133, 87, 215, 37, 99, 248, 87, 210, 135, 114, 98, 73, 78, 16, 23, 147, 131, 41, 29, 217, 144, 4, 116, 26, 138, 121, 105, 73, 116, 123, 128, 85, 125, 139, 182, 26, 82, 168, 158, 243, 244, 74, 233, 98, 108, 211, 227, 220, 55, 253, 34, 202, 131, 16, 209, 129, 227, 189, 154, 234, 143, 13, 94 })
            };

            yield return new object[]
            {
                new BigInteger(new byte[] { 241, 12, 204, 195, 90, 252, 65, 219, 138, 41, 229, 69, 26, 17, 173, 182, 149, 140, 18, 118, 88, 182, 210, 109, 185, 53, 55, 148, 13, 179, 100, 230, 236, 221, 112, 96, 7, 32, 130, 59, 188, 70, 63, 90, 43, 140, 102, 209, 16, 228, 85, 2, 251, 231, 179, 58, 58, 94, 9, 36, 196, 172, 206, 215, 197, 254, 88, 60, 160, 76, 175, 116, 10, 214, 185, 17, 119, 119, 65, 45, 206, 102, 131, 41, 94, 97, 235, 5, 42, 54, 38, 168, 23, 93, 119, 16, 120, 104, 148, 44, 51, 41, 48, 196, 52, 224, 248, 37, 45, 161, 195, 194, 28, 220, 86, 94, 40, 165, 120, 99, 18, 83, 18, 106, 37, 7, 93, 172, 95, 169, 153, 15, 227, 21, 250, 164, 53, 191, 244, 125, 105, 198, 215, 136, 115, 100, 139, 59, 7, 24, 231, 210, 239, 49, 26, 224, 23, 198, 158, 45, 131, 76, 198, 132, 8, 179, 222, 7, 226, 177, 40, 105, 177, 250, 250, 76, 123, 137, 247, 193, 123, 34, 164, 70, 194, 198, 32, 178, 251, 25, 141, 111, 32, 199, 250, 28, 0, 83, 128, 191, 133, 131, 89, 85, 239, 126, 33, 244, 54, 205, 100, 141, 206, 221, 177, 97, 167, 127, 194, 65, 147, 44, 12, 200, 36, 185, 72, 234, 0, 221, 76, 40, 8, 55, 59, 248, 201, 79, 86, 67, 164, 43, 123, 45, 181, 255, 160, 67, 21, 153, 117, 128, 145, 62, 44, 88, 118, 144, 49, 54, 54, 55, 244, 88, 128, 60, 132, 150, 162, 185, 152, 88, 41, 126, 156, 104, 177, 223, 19, 36, 25, 4, 212, 103, 104, 204, 150, 202, 11, 226, 129, 56, 29, 32, 13, 109, 54, 34, 173, 120, 122, 9, 161, 92, 1, 28, 87, 18, 162, 195, 167, 151, 125, 12, 28, 49, 98, 140, 239, 180, 165, 186, 139, 191, 55, 95, 58, 9, 113, 47, 49, 142, 50, 154, 169, 15, 99, 120, 102, 75, 116, 132, 40, 71, 58, 234, 186, 16, 3, 219, 10, 170, 51, 248, 211, 192, 126, 225, 37, 161, 225, 78, 161, 114, 146, 25, 101, 227, 99, 13, 205, 234, 38, 163, 63, 33, 146, 67, 133, 76, 165, 42, 5, 149, 36, 135, 91, 57, 198, 233, 145, 130, 83, 197, 106, 38, 226, 181, 181, 33, 133, 214, 173, 246, 13, 32, 131, 143, 184, 178, 45, 59, 92, 216, 252, 183, 72, 11, 78, 11, 189, 43, 29, 121, 124, 75, 151, 97, 57, 25, 62, 221, 141, 99, 198, 206, 125, 15, 135, 3, 3, 204, 129, 20, 187, 85, 38, 122, 201, 178, 222, 214, 102, 169, 99, 90, 185, 51, 121, 194, 137, 34, 85, 159, 115, 80, 173, 248, 17, 45, 58, 56, 131, 74, 8, 241, 170, 8, 190, 38, 36, 28, 152, 134, 59, 103, 118, 228, 255, 108, 20, 73, 149, 197, 122, 224, 118, 225, 161, 106, 168, 45, 105, 155, 120, 165, 82, 28, 161, 114, 176, 176, 117, 83, 157, 109, 138, 206, 100, 11, 218, 201, 248, 164, 24, 240, 163, 190, 207, 39, 122, 200, 103, 83, 126, 247, 178, 252, 14, 150, 20, 89, 66, 141, 188, 99, 212, 66, 39, 13, 85, 168, 125, 149, 144, 222, 53, 196, 130, 135, 2, 148, 76, 121, 247, 232, 73, 33, 212, 166, 186, 142, 192, 205, 246, 129, 134, 106, 232, 180, 28, 96, 150, 194, 210, 210, 74, 210, 195, 159, 193, 238, 121, 134, 147, 128, 12, 200, 150, 211, 26, 178, 109, 129, 82, 199, 231, 176, 158, 15, 87, 202, 219, 206, 107, 161, 253, 138, 167, 36, 158, 119, 93, 52, 228, 203, 20, 58, 57, 117, 116, 190, 212, 20, 154, 105, 242, 21, 111, 10, 235, 154, 51, 180, 72, 247, 18, 171, 26, 242, 131, 130, 33, 113, 53, 163, 85, 57, 156, 134, 179, 173, 118, 40, 154, 153, 245, 198, 52, 1, 175, 231, 26, 157, 57, 3, 68, 169, 236, 236, 185, 105, 31, 71, 242, 241, 156, 99, 222, 246, 20, 185, 120, 232, 87, 62, 36, 10, 25, 174, 2, 75, 118, 24, 66, 155, 103, 132, 55, 49, 95, 171, 76, 175, 205, 73, 141, 46, 54, 168, 252, 146, 232, 61, 237, 245, 110, 50, 69, 251, 223, 57, 143, 61, 234, 109, 242, 6, 4, 87, 138, 24, 43, 173, 151, 104, 108, 201, 26, 13, 176, 245, 212, 112, 177, 111, 99, 57, 43, 77, 44, 53, 62, 60, 81, 37, 28, 15, 197, 96, 87, 134, 41, 222, 30, 228, 103, 59, 165, 132, 137 }),
                new BigInteger(new byte[] { 23, 56, 74, 234, 101, 30, 139, 122, 9, 184, 219, 183, 43, 84, 218, 238, 91, 114, 76, 93, 3, 186, 182, 42, 204, 245, 82, 205, 8, 40, 148, 166, 196, 134, 30, 6, 241, 103, 191, 253, 131, 75, 87, 158, 170, 113, 178, 201, 10, 120, 223, 50, 56, 43, 162, 92, 129, 32, 29, 111, 204, 189, 113, 224, 36, 53, 243, 165, 54, 212, 253, 243, 178, 79, 217, 184, 99, 24, 127, 90, 66, 136, 195, 144, 72, 97, 110, 152, 228, 193, 132, 208, 74, 79, 48, 197, 217, 118, 121, 117, 126, 13, 147, 53, 122, 236, 127, 233, 82, 233, 249, 104, 160, 194, 44, 68, 250, 224, 151, 195, 199, 91, 193, 116, 79, 44, 168, 99, 30, 19, 222, 236, 158, 10, 220, 53, 244, 141, 22, 92, 225, 132, 70, 250, 200, 28, 47, 135, 34, 236, 203, 248, 223, 182, 173, 34, 76, 31, 230, 77, 67, 28, 202, 62, 11, 13, 202, 150, 230, 247, 79, 108, 202, 59, 225, 66, 21, 199, 172, 50, 252, 33, 64, 49, 72, 229, 12, 231, 92, 185, 232, 213, 166, 200, 24, 196, 186, 7, 238, 186, 81, 153, 160, 42, 156, 95, 46, 164, 213, 107, 13, 133, 114, 179, 80, 34, 220, 152, 246, 37, 0, 226, 57, 44, 15, 122, 197, 2, 35, 28, 136, 13, 216, 241, 224, 99, 128, 21, 149, 233, 183, 104, 18, 241, 172, 98, 98, 242, 119, 245, 232, 33, 17, 201, 185, 111, 96, 115, 27, 33, 153, 43, 129, 184, 227, 255, 173, 128, 211, 237, 161, 198, 3, 13, 109, 152, 207, 72, 128, 172, 24, 211, 138, 45, 52, 102, 252, 14, 55, 113, 3, 12, 249, 115, 79, 139, 231, 90, 13, 172, 181, 225, 31, 137, 84, 33, 29, 3, 91, 33, 160, 23, 243, 110, 9, 198, 160, 83, 131, 124, 2, 158, 73, 97, 234, 148, 204, 143, 189, 225, 177, 116, 58, 211, 247, 54, 179, 0, 217, 18, 73, 248, 179, 22, 209, 235, 25, 138, 105, 137, 235, 147, 214, 145, 169, 22, 152, 189, 159, 191, 84, 141, 70, 121, 112, 43, 221, 32, 119, 211, 18, 160, 122, 113, 51, 10, 235, 51, 144, 254, 123, 65, 38, 51, 246, 29, 157, 168, 41, 224, 158, 245, 97, 26, 189, 187, 178, 57, 165, 165, 14, 249, 19, 150, 119, 74, 24, 101, 94, 159, 43, 101, 45, 160, 67, 19, 103, 18, 131, 82, 141, 52, 246, 150, 52, 29, 62, 96, 179, 119, 88, 177, 141, 235, 240, 130, 46, 143, 188, 171, 5, 56, 74, 138, 181, 56, 200, 192, 214, 170, 83, 159, 182, 83, 24, 217, 2, 124, 97, 90, 12, 184, 178, 230, 71, 121, 216, 66, 175, 192, 228, 216, 86, 215, 77, 85, 234, 108, 225, 251, 84, 125, 137, 8, 36, 184, 251, 57, 210, 22, 127, 177, 45, 98, 159, 76, 133, 100, 66, 59, 96, 134, 16, 21, 198, 123, 73, 223, 5, 195, 239, 150, 155, 101, 5, 142, 47, 169, 194, 142, 126, 158, 111, 152, 64, 50, 171, 244, 55, 0, 207, 133, 41, 166, 155, 206, 108, 115, 66, 169, 111, 28, 51, 15, 39, 233, 22, 199, 92, 17, 29, 220, 162, 180, 188, 194, 23, 119, 213, 147, 208, 85, 167, 4, 243, 105, 108, 245, 202, 54, 33, 187, 102, 24, 31, 98, 106, 95, 113, 87, 28, 227, 151, 38, 227, 242, 227, 117, 233, 216, 183, 245, 158, 175, 216, 26, 116, 46, 28, 189, 124, 33, 252, 38, 239, 97, 117, 137, 149, 90, 88, 183, 6, 125, 119, 236, 189, 128, 33, 0, 133, 247, 11, 242, 127, 44, 29, 183, 64, 4, 237, 59, 67, 180, 218, 250, 231, 113, 54, 130, 125, 90, 224, 143, 128, 255, 238, 32, 56, 236, 92, 244, 13, 27, 55, 132, 21, 10, 166, 195, 67, 190, 72, 144, 108, 28, 116, 81, 162, 124, 5, 76, 73, 158, 131, 42, 23, 124, 113, 127, 61, 252, 126, 83, 167, 249, 45, 39, 124, 85, 229, 220, 202, 115, 62, 190, 17, 79, 35, 99, 139, 156, 125, 188, 231, 15, 170, 111, 152, 12, 109, 153, 153, 39, 133, 70, 63, 245, 232, 114, 239, 76, 213, 76, 151, 40, 130, 214, 95, 196, 24, 199, 89, 13, 244, 160, 127, 194, 108, 152, 79, 190, 143, 111, 232, 75, 67, 163, 237, 247, 127, 28, 102, 176, 116, 63, 199, 48, 37, 190, 141, 212, 15, 211, 243, 165, 63, 139, 51, 229, 199, 124, 136, 80, 35, 48, 188, 101, 100, 75, 229, 223, 232, 104, 68, 252, 97, 168, 153, 180, 216, 78, 157, 174, 10, 123, 174, 187, 123, 66, 11, 183, 112, 8, 47, 2, 147, 21, 141, 67, 130, 98, 204, 217, 217, 173, 235, 167, 143, 247, 68, 151, 242, 126, 12, 121, 223, 218, 69, 167, 130, 1, 233, 34, 63, 45, 110, 223, 56, 65, 104, 75, 27, 85, 50, 118, 129, 195, 212, 93, 143, 7, 44, 132, 254, 116, 197, 242, 252, 182, 35, 185, 207, 229, 193, 128, 120, 170, 154, 171, 63, 12, 110, 77, 142, 249, 123, 156, 75, 149, 247, 27, 5 }),
                new BigInteger(new byte[] { 8, 69, 22, 174, 192, 26, 205, 85, 148, 225, 192, 253, 69, 101, 135, 165, 241, 254, 94, 211, 91, 112, 137, 152, 133, 43, 138, 97, 22, 219, 248, 140, 177, 100, 143, 102, 248, 135, 65, 57, 64, 146, 150, 248, 213, 253, 24, 155, 27, 92, 53, 53, 51, 19, 86, 151, 187, 126, 38, 147, 144, 106, 64, 184, 234, 51, 76, 226, 214, 32, 173, 104, 189, 37, 147, 202, 218, 143, 192, 135, 16, 239, 70, 186, 166, 194, 89, 158, 14, 248, 170, 120, 98, 172, 167, 213, 81, 223, 13, 162, 177, 54, 195, 249, 174, 204, 120, 15, 128, 138, 189, 43, 189, 158, 131, 162, 34, 134, 16, 39, 218, 174, 211, 222, 116, 51, 5, 16, 126, 188, 119, 252, 129, 32, 214, 218, 41, 77, 11, 218, 74, 75, 30, 131, 60, 129, 186, 194, 41, 4, 179, 203, 207, 232, 199, 2, 100, 229, 132, 123, 198, 104, 144, 195, 19, 192, 168, 158, 200, 169, 120, 213, 123, 54, 220, 143, 144, 80, 164, 244, 119, 68, 228, 119, 10, 172, 45, 153, 88, 211, 117, 69, 199, 143, 19, 225, 186, 90, 110, 122, 215, 28, 250, 127, 139, 222, 79, 152, 12, 57, 114, 18, 65, 145, 2, 132, 131, 24, 185, 103, 147, 14, 70, 244, 51, 51, 14, 237, 35, 249, 212, 53, 224, 40, 28, 92, 74, 101, 235, 44, 92, 148, 141, 30, 98, 98, 3, 54, 141, 142, 94, 162, 162, 7, 230, 199, 214, 3, 77, 87, 207, 98, 117, 17, 100, 60, 50, 23, 118, 167, 58, 31, 45, 139, 9, 1, 129, 40, 148, 208, 49, 215, 94, 149, 156, 50, 147, 217, 66, 83, 133, 68, 22, 148, 92, 248, 29, 125, 186, 36, 48, 235, 192, 229, 85, 61, 116, 21, 253, 228, 71, 175, 112, 123, 37, 247, 2, 224, 114, 49, 168, 88, 213, 32, 34, 244, 6, 153, 46, 17, 227, 2, 109, 109, 161, 70, 22, 121, 63, 94, 189, 124, 220, 93, 11, 214, 212, 154, 108, 100, 246, 61, 10, 138, 125, 215, 22, 159, 197, 96, 54, 220, 231, 235, 2, 69, 66, 4, 219, 224, 223, 138, 161, 20, 115, 43, 125, 119, 21, 75, 33, 108, 43, 200, 26, 165, 248, 225, 239, 201, 48, 120, 181, 223, 39, 226, 148, 239, 90, 199, 147, 207, 193, 140, 133, 106, 155, 244, 22, 82, 89, 160, 137, 120, 64, 203, 175, 29, 209, 93, 74, 96, 19, 16, 177, 104, 213, 193, 236, 144, 150, 142, 27, 79, 183, 81, 172, 158, 67, 175, 8, 4, 204, 158, 112, 142, 238, 58, 160, 93, 50, 118, 29, 253, 123, 51, 188, 175, 218, 28, 150, 218, 7, 134, 187, 201, 133, 59, 193, 237, 30, 17, 218, 33, 86, 70, 149, 117, 159, 34, 121, 153, 33, 143, 95, 31, 114, 30, 210, 131, 147, 250, 194, 39, 26, 45, 252, 69, 228, 165, 8, 180, 121, 176, 62, 33, 156, 251, 166, 53, 160, 71, 17, 185, 162, 251, 185, 119, 39, 154, 88, 104, 104, 61, 89, 34, 79, 179, 7, 40, 73, 78, 145, 249, 25, 198, 31, 112, 81, 63, 132, 117, 117, 156, 227, 76, 235, 9, 132, 30, 114, 132, 32, 74, 77, 161, 77, 59, 88, 27, 211, 233, 243, 125, 234, 82, 182, 22, 159, 221, 219, 73, 39, 230, 21, 228, 240, 201, 89, 12, 57, 67, 46, 233, 181, 197, 46, 72, 173, 120, 121, 228, 24, 54, 108, 155, 128, 246, 178, 144, 151, 211, 105, 168, 65, 41, 93, 58, 52, 106, 175, 129, 226, 75, 227, 141, 187, 11, 201, 36, 35, 111, 105, 38, 100, 248, 49, 241, 121, 121, 97, 250, 23, 201, 116, 100, 218, 135, 165, 140, 104, 245, 19, 68, 201, 246, 1, 204, 82, 222, 224, 118, 47, 140, 108, 39, 107, 67, 66, 74, 247, 107, 191, 184, 6, 182, 105, 24, 215, 125, 180, 51, 100, 59, 189, 45, 91, 37, 94, 108, 247, 101, 158, 154, 153, 235, 202, 138, 90, 76, 250, 149, 67, 92, 150, 252, 53, 89, 60, 17, 142, 231, 243, 212, 41, 171, 17, 244, 207, 61, 204, 68, 230, 214, 82, 144, 204, 35, 31, 27, 236, 223, 189, 138, 132, 30, 241, 8, 165, 191, 248, 0, 233, 74, 222, 14, 114, 201, 112, 239, 217, 214, 186, 28, 128, 180, 175, 108, 8, 5, 48, 18, 59, 33, 38, 175, 42, 106, 80, 11, 186, 9, 78, 15, 69, 203, 91, 154, 248, 69, 31, 3, 178, 46, 66, 20, 36, 161, 9, 208, 110, 223, 232, 104, 68, 252, 97, 168, 153, 180, 216, 78, 157, 174, 10, 123, 174, 187, 123, 66, 11, 183, 112, 8, 47, 2, 147, 21, 141, 67, 130, 98, 204, 217, 217, 173, 235, 167, 143, 247, 68, 151, 242, 126, 12, 121, 223, 218, 69, 167, 130, 1, 233, 34, 63, 45, 110, 223, 56, 65, 104, 75, 27, 85, 50, 118, 129, 195, 212, 93, 143, 7, 44, 132, 254, 116, 197, 242, 252, 182, 35, 185, 207, 229, 193, 128, 120, 170, 154, 171, 63, 12, 110, 77, 142, 249, 123, 156, 75, 149, 247, 27, 5 })
            };

            yield return new object[]
            {
                new BigInteger(new byte[] { 24, 122, 212, 101, 36, 202, 103, 124, 227, 130, 29, 175, 123, 53, 148, 178, 254, 215, 41, 41, 30, 131, 152, 223, 245, 4, 248, 40, 167, 102, 190, 8, 179, 231, 205, 71, 172, 19, 134, 207, 67, 194, 40, 236, 47, 223, 125, 70, 8, 152, 151, 31, 241, 220, 202, 148, 225, 245, 111, 31, 209, 63, 213, 125, 195, 21, 252, 148, 103, 76, 6, 234, 81, 90, 229, 91, 91, 171, 175, 21, 135, 178, 32, 14, 206, 158, 23, 214, 28, 56, 101, 182, 163, 102, 253, 4, 237, 170, 41, 26, 227, 233, 222, 187, 146, 172, 206, 151, 247, 111, 57, 53, 69, 89, 151, 31, 30, 199, 174, 37, 253, 37, 119, 47, 231, 79, 71, 234, 92, 224, 110, 176, 130, 149, 50, 158, 212, 100, 123, 33, 208, 128, 223, 173, 255, 48, 112, 73, 207, 222, 229, 37, 251, 4, 28, 190, 113, 175, 212, 67, 101, 227, 58, 22, 0, 137, 178, 175, 39, 249, 74, 186, 75, 141, 84, 125, 69, 202, 48, 182, 222, 253, 27, 126, 250, 72, 180, 126, 121, 115, 44, 37, 144, 56, 187, 237, 70, 201, 172, 117, 125, 192, 34, 213, 147, 43, 218, 110, 175, 158, 121, 166, 126, 29, 100, 104, 199, 187, 28, 184, 212, 52, 54, 179, 204, 37, 42, 19, 209, 103, 55, 124, 30, 186, 57, 29, 218, 70, 234, 206, 110, 70, 207, 218, 250, 133, 166, 114, 211, 83, 37, 138, 172, 243, 161, 73, 138, 111, 8, 110, 1, 198, 157, 71, 103, 253, 136, 196, 228, 70, 142, 129, 209, 77, 74, 142, 100, 91, 184, 71, 126, 183, 63, 126, 65, 173, 207, 42, 24, 240, 147, 107, 61, 123, 205, 167, 169, 135, 114, 253, 252, 225, 142, 141, 197, 164, 184, 95, 169, 19, 228, 46, 182, 193, 240, 73, 134, 31, 5, 186, 45, 94, 171, 244, 179, 34, 68, 85, 127, 162, 228, 221, 233, 187, 74, 156, 41, 178, 184, 157, 244, 111, 128, 5, 11, 100, 181, 123, 139, 132, 32, 138, 130, 184, 207, 158, 54, 154, 218, 162, 96, 98, 224, 7, 46, 7, 69, 251, 119, 83, 31, 211, 102, 103, 197, 190, 222, 165, 232, 78, 109, 200, 202, 250, 129, 90, 91, 48, 235, 172, 102, 142, 216, 22, 61, 145, 142, 121, 215, 4, 31, 186, 4, 56, 100, 76, 35, 26, 242, 17, 191, 144, 177, 139, 52, 151, 152, 229, 133, 211, 172, 20, 15, 203, 236, 144, 251, 79, 42, 52, 47, 101, 206, 228, 46, 179, 195, 145, 150, 52, 54, 64, 61, 218, 236, 214, 221, 77, 225, 165, 248, 169, 243, 3, 87, 10, 170, 57, 75, 211, 81, 212, 239, 190, 225, 90, 193, 205, 92, 129, 212, 7, 79, 81, 156, 107, 214, 53, 222, 22, 179, 174, 110, 132, 59, 134, 113, 202, 140, 36, 9, 138, 70, 194, 226, 98, 81, 127, 248, 145, 20, 15, 240, 96, 209, 241, 90, 162, 103, 235, 51, 116, 90, 52, 31, 163, 102, 106, 131, 91, 233, 246, 64, 56, 120, 73, 192, 140, 110, 58, 26, 87, 60, 77, 2, 164, 189, 13, 179, 73, 44, 207, 99, 160, 240, 148, 48, 18, 163, 63, 252, 230, 251, 157, 168, 29, 207, 183, 231, 136, 49, 205, 52, 44, 209, 220, 55, 12, 50, 123, 229, 88, 183, 58, 46, 25, 147, 159, 71, 228, 25, 174, 232, 46, 61, 51, 91, 159, 165, 228, 213, 5, 37, 43, 23, 94, 155, 54, 207, 20, 92, 35, 223, 40, 89, 163, 46, 193, 110, 142, 138, 47, 112, 3, 110, 130, 228, 5, 239, 158, 128, 152, 103, 77, 166, 223, 50, 140, 34, 208, 126, 92, 208, 51, 136, 246, 206, 192, 213, 153, 4, 153, 48, 22, 89, 165, 149, 34, 188, 149, 121, 149, 71, 49, 147, 195, 187, 146, 130, 122, 113, 162, 159, 158, 15, 46, 95, 60, 123, 120, 180, 35, 48, 85, 153, 231, 103, 81, 243, 0, 73, 15, 208, 141, 86, 233, 20, 159, 160, 196, 58, 60, 62, 91, 40, 98, 249, 146, 6, 43, 70, 89, 167, 67, 224, 86, 101, 242, 151, 171, 92, 106, 158, 129, 48, 97, 210, 18, 72, 38, 25, 191, 155, 182, 61, 119, 229, 213, 45, 55, 237, 213, 74, 200, 185, 240, 104, 140, 172, 0, 131, 34, 7, 245, 104, 208, 157, 189, 42, 198, 144, 230, 73, 224, 17, 134, 190, 240, 6, 105, 113, 144, 196, 96, 189, 134, 32 }),
                new BigInteger(new byte[] { 199, 63, 248, 69, 102, 72, 162, 247, 184, 166, 78, 109, 217, 39, 170, 102, 167, 6, 139, 105, 182, 51, 50, 120, 134, 67, 143, 243, 93, 167, 55, 225, 175, 199, 245, 235, 205, 74, 128, 1, 61, 25, 206, 23, 161, 152, 143, 73, 229, 94, 185, 183, 251, 182, 32, 148, 13, 128, 191, 35, 185, 175, 153, 17, 111, 108, 189, 18, 50, 190, 152, 93, 134, 137, 44, 156, 100, 26, 214, 237, 180, 70, 13, 255, 237, 127, 230, 249, 167, 96, 221, 77, 138, 93, 72, 142, 128, 189, 168, 53, 218, 124, 22, 38, 197, 92, 48, 113, 137, 131, 47, 242, 169, 210, 111, 114, 161, 154, 35, 239, 134, 196, 107, 209, 225, 74, 210, 40, 64, 158, 28, 167, 114, 161, 18, 55, 47, 230, 58, 145, 79, 116, 111, 35, 49, 235, 164, 184, 238, 213, 236, 229, 153, 185, 174, 22, 16, 68, 69, 122, 137, 8, 48, 235, 103, 135, 128, 7, 191, 55, 67, 186, 95, 146, 160, 18, 84, 71, 160, 246, 166, 26, 58, 107, 178, 54, 14, 184, 242, 252, 38, 235, 161, 192, 7, 70, 67, 132, 188, 169, 106, 156, 249, 182, 146, 49, 133, 6, 25, 156, 194, 200, 163, 79, 211, 111, 205, 197, 57, 48, 121, 199, 20, 194, 16, 88, 25, 159, 139, 89, 206, 207, 139, 247, 139, 9, 33, 131, 216, 128, 177, 7, 159, 86, 57, 94, 71, 81, 236, 151, 121, 125, 75, 140, 48, 162, 136, 54, 166, 57, 24, 229, 123, 191, 205, 242, 61, 172, 195, 202, 78, 73, 236, 184, 127, 113, 39, 190, 136, 178, 17, 200, 179, 243, 223, 81, 132, 255, 199, 233, 129, 234, 221, 158, 104, 203, 139, 224, 136, 71, 32, 154, 141, 33, 73, 47, 144, 146, 12, 190, 122, 201, 173, 244, 40, 79, 50, 136, 223, 124, 110, 62, 229, 218, 73, 99, 108, 77, 80, 180, 230, 71, 53, 65, 145, 118, 58, 145, 169, 176, 192, 241, 242, 8, 110, 183, 61, 232, 118, 24, 152, 2, 0, 12, 217, 226, 181, 178, 216, 229, 194, 67, 65, 88, 216, 50, 148, 108, 99, 178, 20, 160, 223, 47, 188, 124, 76, 220, 107, 145, 38, 132, 215, 56, 27, 227, 71, 41, 104, 174, 192, 136, 185, 195, 237, 125, 175, 153, 48, 60, 35, 208, 133, 195, 3, 248, 35, 208, 79, 92, 150, 217, 71, 71, 191, 62, 107, 9, 61, 244, 234, 106, 3, 180, 241, 140, 230, 27, 66, 63, 204, 178, 171, 47, 109, 85, 254, 19, 52, 34, 192, 119, 218, 11, 24, 231, 64, 113, 4, 174, 17, 166, 92, 60, 81, 184, 205, 209, 210, 118, 17, 7, 141, 16, 143, 227, 45, 24, 217, 105, 63, 87, 212, 26, 52, 115, 144, 247, 155, 10, 76, 117, 23, 148, 45, 38, 86, 111, 221, 181, 136, 45, 65, 190, 17, 184, 233, 201, 124, 52, 185, 40, 152, 154, 110, 186, 209, 65, 8, 221, 39, 213, 53, 199, 42, 156, 239, 249, 226, 104, 141, 190, 58, 36, 224, 18, 21, 22, 9, 123, 138, 198, 123, 103, 80, 35, 220, 124, 37, 115, 85, 20, 102, 0, 141, 179, 248, 122, 147, 116, 201, 239, 231, 19, 167, 250, 238, 116, 186, 92, 164, 186, 56, 141, 82, 97, 90, 205, 119, 6, 192, 114, 212, 110, 19, 120, 88, 57, 151, 45, 25, 155, 130, 148, 177, 155, 222, 103, 192, 85, 53, 104, 194, 201, 207, 154 }),
                new BigInteger(new byte[] { 223, 185, 204, 171, 138, 18, 10, 116, 156, 41, 108, 28, 85, 93, 62, 25, 166, 222, 180, 146, 212, 182, 202, 87, 124, 72, 135, 28, 5, 14, 246, 233, 98, 175, 195, 51, 122, 94, 6, 209, 128, 219, 246, 3, 209, 119, 13, 144, 237, 246, 80, 215, 236, 147, 235, 40, 239, 117, 47, 67, 138, 239, 110, 143, 50, 130, 185, 167, 153, 10, 159, 71, 216, 227, 17, 248, 191, 197, 133, 3, 60, 249, 45, 13, 188, 30, 254, 207, 196, 152, 66, 4, 46, 196, 69, 147, 109, 104, 210, 79, 189, 102, 245, 225, 87, 9, 255, 8, 129, 243, 104, 39, 239, 43, 7, 146, 191, 97, 210, 20, 132, 234, 226, 0, 201, 154, 25, 19, 157, 126, 139, 87, 245, 54, 69, 213, 3, 75, 182, 178, 31, 245, 78, 209, 48, 28, 21, 2, 190, 180, 210, 11, 149, 190, 202, 212, 129, 243, 25, 190, 238, 235, 106, 1, 104, 16, 51, 183, 230, 48, 142, 116, 171, 31, 245, 143, 153, 17, 209, 172, 133, 24, 86, 233, 172, 127, 194, 54, 108, 112, 83, 16, 50, 249, 194, 51, 138, 77, 105, 31, 232, 92, 28, 140, 38, 93, 95, 117, 200, 58, 60, 111, 34, 109, 55, 216, 148, 129, 86, 232, 77, 252, 74, 117, 221, 125, 67, 178, 92, 193, 5, 76, 170, 177, 197, 38, 251, 201, 194, 79, 32, 78, 110, 49, 52, 228, 237, 195, 191, 235, 158, 7, 248, 127, 210, 235, 18, 166, 174, 167, 25, 171, 25, 7, 53, 240, 198, 112, 168, 17, 221, 202, 189, 6, 202, 255, 139, 25, 65, 250, 143, 127, 243, 113, 33, 255, 83, 42, 224, 217, 21, 86, 27, 26, 54, 115, 53, 104, 251, 68, 29, 124, 28, 175, 14, 212, 72, 242, 181, 209, 94, 248, 99, 182, 25, 153, 184, 167, 228, 54, 156, 156, 144, 207, 253, 133, 176, 162, 207, 86, 203, 37, 31, 253, 219, 18, 100, 67, 98, 78, 181, 97, 115, 14, 121, 27, 243, 99, 2, 157, 184, 140, 130, 196, 168, 129, 236, 76, 179, 136, 35, 166, 33, 96, 6, 58, 217, 103, 219, 5, 52, 115, 70, 151, 129, 59, 43, 130, 84, 224, 147, 76, 162, 51, 157, 61, 163, 89, 83, 91, 39, 23, 146, 218, 42, 15, 62, 19, 8, 65, 66, 138, 138, 251, 103, 68, 71, 234, 65, 110, 85, 106, 249, 210, 243, 213, 3, 239, 194, 199, 151, 127, 18, 127, 222, 29, 226, 107, 108, 115, 251, 23, 122, 20, 156, 8, 194, 165, 202, 86, 246, 183, 23, 230, 4, 190, 30, 191, 229, 83, 10, 80, 80, 64, 168, 194, 119, 11, 30, 74, 99, 219, 124, 207, 112, 62, 239, 229, 53, 235, 19, 95, 35, 108, 208, 222, 102, 45, 122, 33, 255, 35, 134, 24, 105, 172, 199, 57, 106, 218, 145, 183, 135, 128, 244, 26, 59, 73, 117, 198, 205, 55, 136, 251, 63, 172, 44, 228, 111, 200, 91, 73, 144, 251, 73, 63, 86, 100, 102, 196, 118, 181, 123, 92, 88, 92, 213, 162, 119, 181, 164, 29, 184, 180, 82, 199, 153, 138, 216, 188, 129, 227, 201, 160, 125, 72, 41, 141, 54, 180, 197, 214, 227, 177, 79, 24, 190, 44, 162, 229, 213, 135, 109, 185, 35, 62, 146, 217, 169, 129, 165, 203, 139, 169, 65, 145, 235, 216, 222, 17, 51, 73, 107, 195, 238, 206, 57, 7, 102, 58, 11, 110, 231, 244, 230, 248, 154, 54, 207, 20, 92, 35, 223, 40, 89, 163, 46, 193, 110, 142, 138, 47, 112, 3, 110, 130, 228, 5, 239, 158, 128, 152, 103, 77, 166, 223, 50, 140, 34, 208, 126, 92, 208, 51, 136, 246, 206, 192, 213, 153, 4, 153, 48, 22, 89, 165, 149, 34, 188, 149, 121, 149, 71, 49, 147, 195, 187, 146, 130, 122, 113, 162, 159, 158, 15, 46, 95, 60, 123, 120, 180, 35, 48, 85, 153, 231, 103, 81, 243, 0, 73, 15, 208, 141, 86, 233, 20, 159, 160, 196, 58, 60, 62, 91, 40, 98, 249, 146, 6, 43, 70, 89, 167, 67, 224, 86, 101, 242, 151, 171, 92, 106, 158, 129, 48, 97, 210, 18, 72, 38, 25, 191, 155, 182, 61, 119, 229, 213, 45, 55, 237, 213, 74, 200, 185, 240, 104, 140, 172, 0, 131, 34, 7, 245, 104, 208, 157, 189, 42, 198, 144, 230, 73, 224, 17, 134, 190, 240, 6, 105, 113, 144, 196, 96, 189, 134, 32 })
            };

            yield return new object[]
            {
                new BigInteger(new byte[] { 217, 173, 43, 89, 42, 2, 141, 180, 86, 27, 121, 0, 46, 162, 180, 221, 140, 66, 8, 56, 163, 219, 183, 130, 242, 13, 33, 209, 87, 176, 11, 158, 120, 155, 126, 140, 107, 212, 239, 152, 245, 224, 91, 128, 101, 103, 145, 178, 128, 139, 54, 134, 199, 98, 57, 254, 246, 169, 102, 29, 225, 187, 92, 165, 16, 218, 136, 146, 36, 40, 113, 184, 83, 111, 66, 195, 127, 55, 28, 139, 123, 110, 81, 204, 122, 132, 215, 22, 98, 128, 150, 193, 109, 209, 183, 58, 131, 181, 112, 138, 223, 255, 142, 88, 25, 126, 51, 87, 31, 117, 126, 190, 140, 219, 161, 114, 106, 144, 43, 139, 3, 113, 48, 163, 146, 176, 74, 129, 184, 8, 64, 202, 199, 146, 172, 124, 223, 248, 178, 251, 81, 128, 246, 237, 2, 215, 52, 145, 48, 68, 207, 242, 137, 228, 135, 109, 206, 235, 198, 105, 51, 177, 159, 22, 52, 180, 247, 250, 47, 37, 147, 113, 221, 48, 58, 130, 123, 67, 161, 186, 123, 184, 81, 115, 56, 78, 64, 226, 10, 63, 173, 244, 50, 73, 200, 160, 224, 224, 184, 77, 223, 58, 98, 10, 177, 94, 20, 89, 170, 4, 242, 139, 73, 11, 237, 123, 95, 44, 221, 230, 115, 20, 240, 240, 119, 158, 62, 148, 104, 154, 161, 154, 139, 83, 219, 65, 86, 70, 193, 218, 57, 231, 56, 5, 76, 34, 170, 39, 91, 76, 64, 180, 2, 209, 218, 202, 73, 114, 146, 19, 32, 128, 241, 16, 98, 87, 0, 245, 48, 172, 37, 24, 106, 3, 173, 140, 220, 234, 163, 85, 243, 23, 57, 27, 89, 237, 152, 185, 121, 16, 247, 228, 180, 174, 185, 185, 245, 39, 162, 245, 33, 180, 246, 174, 39, 39, 73, 254, 36, 77, 237, 95, 206, 61, 31, 9, 71, 214, 182, 117, 191, 70, 124, 32, 181, 65, 100, 188, 74, 243, 151, 181, 71, 173, 51, 63, 32, 138, 243, 50, 163, 153, 149, 43, 34, 151, 21, 118, 143, 176, 114, 31, 113, 45, 54, 218, 56, 214, 249, 230, 194, 141, 180, 48, 182, 56, 24, 32, 9, 223, 233, 188, 226, 131, 210, 37, 177, 80, 253, 29, 43, 238, 45, 153, 129, 119, 67, 25, 119, 89, 6, 73, 144, 12, 112, 22, 228, 101, 117, 233, 127, 244, 109, 175, 199, 181, 60, 237, 91, 17, 41, 232, 216, 220, 186, 212, 95, 27, 175, 62, 245, 254, 169, 176, 217, 159, 44, 213, 19, 187, 64, 76, 218, 20, 157, 55, 128, 126, 210, 203, 59, 86, 190, 101, 47, 30, 112, 180, 81, 155, 182, 201, 181, 54, 244, 129, 74, 189, 213, 39, 137, 102, 217, 159, 208, 221, 13, 196, 62, 131, 83, 225, 73, 227, 3, 158, 63, 67, 128, 186, 47, 119, 131, 119, 5, 118, 150, 163, 32, 28, 236, 195, 168, 170, 65, 213, 124, 30, 149, 82, 17, 172, 118, 206, 72, 212, 128, 209, 51, 85, 65, 6, 60, 26, 248, 18, 226, 97, 154, 89, 70, 105, 164, 33, 151, 143, 57, 158, 162, 45, 33, 33, 174, 46, 169, 131, 0, 181, 47, 48, 246, 196, 112, 202, 219, 229, 134, 141, 177, 47, 242, 27, 132, 59, 11, 167, 8, 85, 55, 184, 155, 225, 47, 5, 51, 32, 88, 13, 113, 117, 17, 44, 50, 105, 99, 130, 249, 217, 70, 177, 9, 171, 237, 114, 59, 6, 156, 114, 35, 1, 123, 171, 126, 118, 149, 21, 149, 106, 151, 187, 141, 120, 28, 60, 30, 197, 232, 26, 216, 137, 174, 46, 14, 133, 175, 239, 244, 189, 202, 44, 131, 229, 234, 111, 250, 101, 174, 187, 212, 78, 237, 143, 65, 139, 36, 144, 143, 177, 84, 117, 29, 129, 154, 154, 83, 76, 156, 249, 199, 165, 32, 215, 159, 106, 55, 167, 142, 172, 65, 184, 23, 45, 70, 138, 155, 31, 237, 131, 97, 31, 95, 66, 104, 85, 14, 1, 75, 80, 28, 174, 200, 180, 244, 46, 45, 21, 239, 215, 83, 124, 1, 227, 111, 156, 93, 240, 59, 15, 254, 52, 94, 25, 152, 167, 69, 222, 111, 74, 91, 54, 92, 62, 144, 147, 239, 98, 56, 23, 93, 134, 47, 21, 48, 228, 29, 124, 210, 204, 247, 30, 198, 60, 81, 231, 79, 175, 28, 76, 233, 78, 165, 165, 124, 29, 32, 195, 82, 220, 172, 250, 184, 35, 177, 119, 5, 73, 232, 179 }),
                new BigInteger(new byte[] { 200, 11, 162, 82, 61, 111, 31, 236, 168, 67, 63, 70, 236, 46, 14, 224, 137, 153, 19, 196, 90, 42, 246, 31, 252, 101, 162, 138, 217, 226, 6, 199, 104, 194, 32, 67, 3, 83, 62, 173, 0, 215, 170, 153, 109, 180, 203, 245, 162, 151, 123, 175, 213, 239, 47, 158, 21, 131, 85, 215, 205, 149, 18, 197, 61, 120, 221, 42, 13, 203, 221, 53, 91, 101, 195, 131, 128, 93, 178, 72, 154, 172, 231, 66, 103, 87, 241, 120, 146, 129, 46, 127, 253, 103, 223, 106, 196, 228, 91, 244, 214, 161, 231, 215, 185, 70, 84, 111, 44, 172, 30, 184, 208, 12, 61, 33, 173, 207 }),
                new BigInteger(new byte[] { 161, 185, 205, 171, 103, 113, 172, 160, 255, 94, 184, 70, 26, 209, 194, 189, 22, 220, 27, 252, 253, 5, 174, 162, 238, 115, 195, 91, 49, 147, 18, 101, 225, 93, 159, 207, 110, 39, 46, 70, 246, 183, 6, 26, 211, 27, 93, 168, 35, 35, 178, 53, 157, 82, 105, 156, 12, 45, 188, 244, 174, 81, 111, 106, 78, 82, 102, 189, 49, 243, 78, 238, 174, 212, 5, 71, 0, 149, 206, 211, 21, 27, 57, 15, 226, 219, 200, 143, 244, 1, 197, 64, 107, 57, 151, 165, 71, 154, 204, 126, 182, 161, 118, 48, 211, 196, 135, 198, 75, 33, 157, 118, 93, 232, 222, 147, 23, 96, 43, 139, 3, 113, 48, 163, 146, 176, 74, 129, 184, 8, 64, 202, 199, 146, 172, 124, 223, 248, 178, 251, 81, 128, 246, 237, 2, 215, 52, 145, 48, 68, 207, 242, 137, 228, 135, 109, 206, 235, 198, 105, 51, 177, 159, 22, 52, 180, 247, 250, 47, 37, 147, 113, 221, 48, 58, 130, 123, 67, 161, 186, 123, 184, 81, 115, 56, 78, 64, 226, 10, 63, 173, 244, 50, 73, 200, 160, 224, 224, 184, 77, 223, 58, 98, 10, 177, 94, 20, 89, 170, 4, 242, 139, 73, 11, 237, 123, 95, 44, 221, 230, 115, 20, 240, 240, 119, 158, 62, 148, 104, 154, 161, 154, 139, 83, 219, 65, 86, 70, 193, 218, 57, 231, 56, 5, 76, 34, 170, 39, 91, 76, 64, 180, 2, 209, 218, 202, 73, 114, 146, 19, 32, 128, 241, 16, 98, 87, 0, 245, 48, 172, 37, 24, 106, 3, 173, 140, 220, 234, 163, 85, 243, 23, 57, 27, 89, 237, 152, 185, 121, 16, 247, 228, 180, 174, 185, 185, 245, 39, 162, 245, 33, 180, 246, 174, 39, 39, 73, 254, 36, 77, 237, 95, 206, 61, 31, 9, 71, 214, 182, 117, 191, 70, 124, 32, 181, 65, 100, 188, 74, 243, 151, 181, 71, 173, 51, 63, 32, 138, 243, 50, 163, 153, 149, 43, 34, 151, 21, 118, 143, 176, 114, 31, 113, 45, 54, 218, 56, 214, 249, 230, 194, 141, 180, 48, 182, 56, 24, 32, 9, 223, 233, 188, 226, 131, 210, 37, 177, 80, 253, 29, 43, 238, 45, 153, 129, 119, 67, 25, 119, 89, 6, 73, 144, 12, 112, 22, 228, 101, 117, 233, 127, 244, 109, 175, 199, 181, 60, 237, 91, 17, 41, 232, 216, 220, 186, 212, 95, 27, 175, 62, 245, 254, 169, 176, 217, 159, 44, 213, 19, 187, 64, 76, 218, 20, 157, 55, 128, 126, 210, 203, 59, 86, 190, 101, 47, 30, 112, 180, 81, 155, 182, 201, 181, 54, 244, 129, 74, 189, 213, 39, 137, 102, 217, 159, 208, 221, 13, 196, 62, 131, 83, 225, 73, 227, 3, 158, 63, 67, 128, 186, 47, 119, 131, 119, 5, 118, 150, 163, 32, 28, 236, 195, 168, 170, 65, 213, 124, 30, 149, 82, 17, 172, 118, 206, 72, 212, 128, 209, 51, 85, 65, 6, 60, 26, 248, 18, 226, 97, 154, 89, 70, 105, 164, 33, 151, 143, 57, 158, 162, 45, 33, 33, 174, 46, 169, 131, 0, 181, 47, 48, 246, 196, 112, 202, 219, 229, 134, 141, 177, 47, 242, 27, 132, 59, 11, 167, 8, 85, 55, 184, 155, 225, 47, 5, 51, 32, 88, 13, 113, 117, 17, 44, 50, 105, 99, 130, 249, 217, 70, 177, 9, 171, 237, 114, 59, 6, 156, 114, 35, 1, 123, 171, 126, 118, 149, 21, 149, 106, 151, 187, 141, 120, 28, 60, 30, 197, 232, 26, 216, 137, 174, 46, 14, 133, 175, 239, 244, 189, 202, 44, 131, 229, 234, 111, 250, 101, 174, 187, 212, 78, 237, 143, 65, 139, 36, 144, 143, 177, 84, 117, 29, 129, 154, 154, 83, 76, 156, 249, 199, 165, 32, 215, 159, 106, 55, 167, 142, 172, 65, 184, 23, 45, 70, 138, 155, 31, 237, 131, 97, 31, 95, 66, 104, 85, 14, 1, 75, 80, 28, 174, 200, 180, 244, 46, 45, 21, 239, 215, 83, 124, 1, 227, 111, 156, 93, 240, 59, 15, 254, 52, 94, 25, 152, 167, 69, 222, 111, 74, 91, 54, 92, 62, 144, 147, 239, 98, 56, 23, 93, 134, 47, 21, 48, 228, 29, 124, 210, 204, 247, 30, 198, 60, 81, 231, 79, 175, 28, 76, 233, 78, 165, 165, 124, 29, 32, 195, 82, 220, 172, 250, 184, 35, 177, 119, 5, 73, 232, 179 })
            };

            yield return new object[]
            {
                new BigInteger(new byte[] { 230, 134, 101, 151, 140, 156, 93, 55, 243, 134, 88, 190, 155, 1, 189, 114, 248, 196, 15, 136, 32, 3, 129, 76, 229, 99, 118, 175, 240, 42, 190, 188, 244, 133, 117, 109, 113, 9, 91, 58, 91, 14, 96, 233, 211, 237, 130, 182, 19, 154, 68, 92, 158, 70, 61, 226, 5, 24, 177, 40, 38, 174, 71, 201, 200, 155, 202, 22, 139, 79, 1, 239, 105, 212, 45, 18, 162, 152, 120, 247, 224, 191, 156, 86, 229, 97, 30, 173, 72, 146, 104, 88, 87, 51, 20, 173, 199, 150, 33, 55, 35, 108, 43, 195, 153, 85, 243, 201, 25, 42, 64, 108, 160, 186, 71, 103, 18, 240, 228, 102, 125, 28, 205, 248, 230, 169, 151, 53, 192, 127, 75, 12, 118, 64, 212, 116, 148, 216, 188, 144, 109, 85, 147, 29, 82, 251, 183, 157, 235, 172, 154, 214, 178, 186, 186, 7, 158, 50, 220, 240, 190, 189, 8, 154, 223, 51, 246, 95, 229, 211, 211, 57, 51, 83, 249, 39, 136, 176, 165, 234, 241, 249, 74, 19, 229, 117, 89, 187, 133, 205, 214, 97, 252, 204, 210, 175, 76, 249, 62, 31, 5, 88, 183, 24, 217, 97, 162, 94, 193, 147, 126, 237, 141, 241, 254, 196, 115, 244, 180, 105, 217, 58, 217, 24, 252, 114, 61, 102, 129, 73, 218, 143, 114, 134, 229, 153, 65, 49, 58, 131, 211, 251, 250, 242, 78, 232, 211, 11, 206, 13, 60, 87, 68, 212, 69, 203, 126, 159, 27, 103, 36, 59, 221, 120, 184, 238, 123, 6, 11, 101, 131, 65, 186, 49, 150, 222, 64, 230, 201, 20, 158, 49, 152, 115, 12, 130, 74, 157, 65, 26, 26, 162, 22, 210, 95, 151, 29, 130, 58, 96, 109, 205, 255, 104, 138, 250, 157, 19, 62, 102, 138, 152, 213, 7, 201, 242, 162, 105, 107, 53, 164, 221, 196, 241, 75, 224, 43, 232, 209, 254, 193, 190, 172, 105, 167, 208, 50, 48, 233, 18, 229, 54, 95, 219, 143, 130, 204, 15, 9, 109, 245, 179, 22, 4, 187, 143, 8, 13, 29, 169, 207, 180, 66, 63, 165, 204, 235, 108, 95, 248, 191, 114, 128, 89, 80, 109, 126, 233, 97, 201, 20, 28, 223, 99, 9, 13, 167, 168, 173, 23, 200, 36, 18, 63, 66, 48, 244, 31, 53, 194, 150, 96, 175, 16, 173, 130, 51, 189, 180, 78, 17, 30, 172, 84, 149, 178, 213, 222, 53, 152, 37, 67, 190, 72, 48, 155, 95, 65, 22, 31, 120, 94, 179, 158, 50, 180, 109, 206, 181, 135, 218, 233, 243, 95, 6, 169, 120, 190, 170, 143, 90, 21, 233, 157, 157, 83, 162, 102, 223, 18, 34, 241, 167, 148, 213, 179, 107, 13, 33, 224, 69, 6, 41, 174, 190, 59, 79, 94, 66, 135, 34, 160, 151, 134, 196, 99, 116, 21, 1, 223, 18, 6, 239, 163, 100, 232, 66, 94, 137, 82, 61, 107, 157, 201, 74, 83, 236, 59, 223, 97, 3, 67, 36, 88, 239, 129, 90, 33, 208, 114, 62, 85, 11, 65, 206, 63, 23, 134, 74, 213, 249, 240, 213, 239, 73, 183, 3, 205, 59, 112, 118, 57, 53, 159, 220, 207, 226, 150, 179, 226, 142, 32, 142, 23, 20, 232, 95, 91, 250, 11, 60, 102, 241, 10, 104, 18, 109, 52, 15, 56, 126, 140, 229, 95, 251, 5, 31, 108, 101, 255, 92, 129, 210, 103, 71, 216, 98, 189, 219, 96, 88, 168, 211, 48, 52, 16, 71, 43, 54, 201, 97, 173, 135, 123, 100, 1, 59, 47, 27, 14, 201, 242, 245, 250, 174, 186, 147, 137, 194, 143, 176, 176, 89, 174, 224, 214, 184, 180, 42, 234, 189, 243, 53, 156, 158, 175, 249, 87, 2, 69, 157, 50, 162 }),
                new BigInteger(new byte[] { 151, 101, 178, 216, 130, 121, 125, 189, 105, 166, 25, 177, 83, 217, 155, 142, 72, 134, 4, 48, 80, 213, 189, 89, 92, 254, 193, 59, 135, 124, 242, 125, 13, 140, 22, 122, 35, 224, 49, 177, 109, 255, 138, 142, 130, 237, 91, 32, 188, 115, 253, 22, 45, 113, 123, 194, 167, 88, 124, 131, 184, 65, 54, 236, 180, 155, 163, 199, 194, 32, 106, 104, 85, 83, 195, 81, 74, 46, 214, 110, 162, 160, 127, 19, 127, 220, 79, 156, 16, 84, 210, 203, 99, 173, 248, 44, 201, 157, 218, 231, 74, 148, 93, 156, 9, 148, 193, 218, 174, 42, 119, 121, 129, 13, 225, 24, 194, 35, 109, 215, 75, 6, 182, 110, 77, 159, 4, 167, 91, 150, 136, 173, 83, 239, 36, 14, 66, 227, 10, 234, 27, 117, 238, 230, 221, 89, 73, 86, 204, 224, 106, 165, 48, 3, 156, 67, 222, 239, 78, 105, 144, 26, 126, 23, 162, 202, 37, 146, 233, 210, 213, 223, 24, 130, 187, 213, 24, 207, 145, 243, 86, 174, 218, 122, 44, 226, 124, 80, 83, 224, 47, 82, 148, 161, 89, 1, 246, 215, 68, 18, 52, 183, 108, 249, 11, 138, 141, 173, 71, 198, 43, 14, 94, 91, 19, 226, 63, 4, 234, 191, 77, 212, 62, 8, 163, 130, 74, 119, 41, 185, 223, 64, 139, 126, 191, 158, 65, 224, 111, 161, 85, 206, 8, 141, 180, 33, 244, 57, 141, 119, 193, 242, 237, 132, 197, 95, 120, 99, 86, 136, 64, 21, 132, 142, 231, 234, 130, 224, 155, 116, 161, 95, 148, 72, 106, 127, 204, 177, 84, 129, 141, 17, 233, 177, 248, 236, 83, 7, 185, 95, 38, 221, 137, 231, 97, 63, 74, 122, 165, 201, 158, 20, 157, 25, 214, 98, 94, 164, 25, 70, 147, 198, 15, 212, 250, 46, 43, 210, 150, 250, 151, 123, 39, 59, 78, 196, 214, 172, 230, 42, 53, 81, 11, 138, 227, 121, 116, 208, 219, 150, 141, 174, 238, 114, 204, 95, 205, 180, 236, 51, 20, 119, 228, 171, 50, 35, 237, 97, 202, 17, 139, 177, 190, 238, 17, 65, 186, 133, 241, 128, 185, 90, 247, 255, 109, 232, 141, 180, 111, 238, 246, 33, 192, 178, 22, 189, 109, 96, 87, 192, 140, 18, 6, 201, 10, 220, 240, 255, 97, 139, 164, 71, 195, 107, 121, 185, 127, 228, 61, 74, 150, 56, 242, 219, 27, 105, 241, 11, 216, 83, 211, 89, 46, 49, 244, 167, 84, 45, 245, 144, 248, 141, 82, 227, 73, 175, 94, 84, 57, 92, 62, 137, 35, 13, 42, 83, 31, 42, 174, 160, 235, 228, 39, 14, 183, 75, 24, 146, 119, 210, 101, 82, 238, 173, 81, 8, 230, 15, 145, 107, 11, 156, 210, 123, 21, 74, 54, 33, 7, 160, 135, 41, 126, 85, 165, 19, 102, 67, 44, 254, 150, 204, 221, 103, 246, 235, 154, 117, 187, 33, 109, 16, 153, 66, 149, 78, 18, 171, 249, 108, 20, 247, 138, 214, 74, 60, 211, 153, 7, 62, 161, 162, 226, 146, 213, 62, 190, 106, 132, 41, 176, 192, 101, 229, 51, 119, 144, 60, 191, 86, 0, 186, 74, 192, 234, 158, 65, 7, 29, 186, 24, 0, 110, 125, 127, 202, 46, 6, 109, 86, 144, 167, 39, 208, 187, 83, 145, 165, 163, 211, 172, 151, 217, 176, 61, 231, 88, 34, 168, 54, 253, 183, 77, 202, 17, 176, 82, 232, 101, 105, 73, 137, 206, 105, 198, 88, 147, 121, 4, 150, 12, 156, 100, 73, 13, 129, 38, 190, 63, 152, 113, 11, 93, 54, 89, 217, 90, 92, 190, 11, 1, 169, 63, 58, 138, 34, 200, 112, 234, 229, 83, 222, 30, 18, 41, 96, 83, 3, 188, 144, 137, 147, 169, 205, 247, 61, 44, 151, 16, 236, 254, 54, 186, 248, 148, 99, 93, 41, 191, 234, 184, 181, 211, 90, 68, 146, 237, 75, 13, 248, 134, 86, 159, 46, 129, 109, 0, 113, 66, 76, 246, 48, 96, 253, 230, 28, 36, 37, 125, 197, 3, 41, 67, 210, 215, 212, 192, 78, 136, 101, 79, 224, 138, 23, 102, 167, 151, 140, 119, 39, 99, 235, 231, 115, 243, 136, 85, 214, 115, 40, 110, 200, 206, 71, 245, 93, 18, 205, 87, 173, 173, 177, 233, 221, 253, 21, 165, 73, 151, 62, 133, 152, 173, 86, 98, 24, 66, 208, 222, 78, 126, 233, 218, 96, 246, 134, 18, 66, 33, 9, 149, 201, 52, 202, 121, 178, 2, 9, 118, 222, 226, 11, 63, 53, 162, 214, 27, 119, 229, 221, 26, 66, 238, 9, 46, 195, 214, 136, 230, 118, 2, 98, 39, 141, 169, 187, 99, 227, 220, 176, 22, 56, 90 }),
                new BigInteger(new byte[] { 125, 236, 23, 112, 15, 22, 219, 244, 92, 45, 114, 111, 239, 218, 88, 1, 65, 75, 20, 184, 112, 216, 62, 166, 65, 98, 56, 235, 119, 167, 176, 58, 2, 18, 140, 231, 148, 233, 140, 235, 200, 13, 235, 119, 86, 219, 222, 214, 207, 13, 66, 115, 203, 183, 184, 164, 173, 112, 45, 172, 222, 239, 125, 181, 125, 55, 110, 222, 77, 112, 107, 87, 191, 39, 241, 99, 236, 198, 78, 102, 131, 96, 28, 106, 100, 62, 110, 73, 89, 230, 58, 36, 187, 224, 12, 218, 144, 52, 252, 30, 110, 0, 137, 95, 163, 233, 180, 164, 200, 84, 183, 229, 33, 200, 40, 128, 212, 19, 82, 62, 201, 34, 131, 103, 52, 73, 156, 220, 27, 22, 212, 185, 201, 47, 249, 130, 214, 187, 199, 122, 137, 202, 129, 4, 48, 85, 1, 244, 183, 141, 5, 124, 227, 189, 86, 75, 124, 34, 43, 90, 79, 216, 134, 177, 129, 254, 27, 242, 206, 166, 169, 25, 76, 213, 180, 253, 160, 127, 55, 222, 72, 168, 37, 142, 17, 88, 214, 11, 217, 173, 6, 180, 144, 110, 44, 177, 66, 209, 131, 49, 57, 15, 36, 18, 229, 235, 47, 12, 9, 90, 170, 251, 235, 76, 18, 167, 179, 248, 158, 41, 39, 15, 24, 33, 159, 245, 135, 221, 170, 2, 186, 208, 253, 4, 165, 56, 131, 17, 170, 36, 41, 202, 3, 128, 3, 10, 200, 69, 91, 133, 253, 73, 50, 89, 11, 43, 247, 2, 114, 239, 100, 80, 97, 7, 160, 217, 254, 230, 166, 217, 36, 161, 78, 122, 0, 94, 13, 152, 30, 150, 43, 67, 129, 37, 5, 111, 158, 164, 250, 121, 64, 127, 160, 185, 193, 214, 103, 252, 223, 41, 12, 226, 156, 130, 96, 93, 252, 183, 87, 172, 29, 95, 229, 219, 195, 33, 206, 59, 2, 48, 60, 89, 236, 44, 154, 164, 2, 149, 184, 41, 247, 15, 184, 243, 138, 74, 167, 0, 197, 169, 114, 229, 77, 78, 92, 226, 153, 196, 245, 160, 9, 43, 251, 175, 237, 178, 245, 110, 231, 186, 90, 102, 1, 46, 183, 13, 166, 242, 80, 121, 121, 205, 119, 89, 190, 85, 12, 158, 209, 183, 11, 62, 159, 22, 32, 202, 20, 9, 5, 216, 84, 55, 24, 8, 77, 12, 229, 31, 151, 77, 59, 168, 114, 124, 38, 60, 179, 161, 242, 152, 167, 86, 158, 48, 177, 27, 199, 234, 13, 236, 248, 156, 236, 121, 36, 67, 180, 110, 11, 176, 112, 236, 5, 130, 124, 99, 204, 34, 239, 227, 24, 115, 23, 109, 48, 252, 151, 232, 88, 48, 70, 250, 16, 172, 84, 159, 186, 248, 86, 229, 135, 67, 150, 66, 39, 188, 81, 29, 178, 75, 81, 162, 251, 41, 212, 133, 133, 127, 73, 39, 170, 201, 21, 220, 105, 119, 218, 88, 45, 221, 169, 210, 204, 11, 91, 212, 221, 211, 68, 116, 170, 123, 54, 12, 224, 161, 254, 230, 216, 206, 23, 58, 175, 46, 58, 190, 45, 187, 215, 176, 223, 247, 237, 211, 163, 126, 213, 240, 206, 254, 169, 177, 59, 213, 125, 46, 148, 9, 251, 198, 118, 243, 127, 95, 199, 110, 36, 158, 208, 156, 167, 32, 252, 148, 147, 178, 142, 97, 103, 98, 204, 13, 25, 219, 35, 102, 254, 217, 178, 11, 43, 36, 191, 16, 57, 237, 119, 142, 13, 54, 90, 57, 32, 50, 89, 136, 181, 165, 65, 202, 161, 49, 162, 154, 250, 104, 218, 164, 58, 95, 110, 73, 236, 196, 113, 130, 97, 237, 90, 166, 58, 254, 82, 49, 8, 148, 238, 229, 128, 155, 177, 89, 153, 232, 106, 249, 128, 37, 21, 208, 17, 210, 84, 174, 199, 15, 77, 91, 190, 213, 38, 198, 75, 205, 247, 61, 44, 151, 16, 236, 254, 54, 186, 248, 148, 99, 93, 41, 191, 234, 184, 181, 211, 90, 68, 146, 237, 75, 13, 248, 134, 86, 159, 46, 129, 109, 0, 113, 66, 76, 246, 48, 96, 253, 230, 28, 36, 37, 125, 197, 3, 41, 67, 210, 215, 212, 192, 78, 136, 101, 79, 224, 138, 23, 102, 167, 151, 140, 119, 39, 99, 235, 231, 115, 243, 136, 85, 214, 115, 40, 110, 200, 206, 71, 245, 93, 18, 205, 87, 173, 173, 177, 233, 221, 253, 21, 165, 73, 151, 62, 133, 152, 173, 86, 98, 24, 66, 208, 222, 78, 126, 233, 218, 96, 246, 134, 18, 66, 33, 9, 149, 201, 52, 202, 121, 178, 2, 9, 118, 222, 226, 11, 63, 53, 162, 214, 27, 119, 229, 221, 26, 66, 238, 9, 46, 195, 214, 136, 230, 118, 2, 98, 39, 141, 169, 187, 99, 227, 220, 176, 22, 56, 90 })
            };

            // Very large + very small
            yield return new object[]
            {
                new BigInteger(new byte[] { 107, 114, 170, 165, 189, 32, 107, 154, 93, 68, 238, 82, 218, 43, 155, 182, 220, 121, 34, 0, 219, 128, 228, 177, 140, 231, 114, 56, 155, 53, 199, 226, 232, 238, 157, 187, 24, 48, 203, 198, 10, 103, 5, 227, 159, 86, 224, 56, 71, 38, 211, 156, 21, 89, 36, 235, 142, 249, 25, 214, 173, 51, 255, 40, 124, 12, 106 }),
                new BigInteger(new byte[] { 236, 141 }),
                new BigInteger(new byte[] { 87, 0, 170, 165, 189, 32, 107, 154, 93, 68, 238, 82, 218, 43, 155, 182, 220, 121, 34, 0, 219, 128, 228, 177, 140, 231, 114, 56, 155, 53, 199, 226, 232, 238, 157, 187, 24, 48, 203, 198, 10, 103, 5, 227, 159, 86, 224, 56, 71, 38, 211, 156, 21, 89, 36, 235, 142, 249, 25, 214, 173, 51, 255, 40, 124, 12, 106 })
            };

            yield return new object[]
            {
                new BigInteger(new byte[] { 157, 171, 174, 92, 245, 115, 123, 1, 18, 53, 6, 58, 240, 117, 97, 42, 204, 142, 202, 177, 44, 31, 23, 244, 25, 214, 104, 220, 102, 147, 235, 244, 170, 102, 136, 40, 238, 185, 99, 176, 112, 141, 231, 227, 208, 115, 66, 137, 199, 82, 63, 220, 33, 220, 179, 126, 148, 185, 66, 31, 11, 158, 154, 127, 74, 17, 143, 138, 237, 56, 59, 18, 42, 25, 65, 159, 55, 51, 36, 166, 147, 223, 21, 19, 83, 14, 210, 206, 179, 10, 148, 53, 118, 68, 165, 209, 242, 104, 153, 190, 227, 184, 156, 142, 22, 44, 178, 8, 154, 19, 70, 97, 149, 156, 140, 43, 137, 134, 43, 59, 62, 193, 214, 227, 164, 6, 155, 229, 115, 111, 172, 207, 154, 101, 194, 218, 66, 134, 252, 38, 92, 202, 51, 159, 195, 51, 160, 218, 184, 121, 72, 107, 60, 93, 127, 33, 225, 185, 234, 16, 144, 204, 148, 43, 103, 119, 198, 47, 217, 177, 233, 2, 138, 4, 223, 115, 141, 55, 31, 113, 102, 193, 45, 249, 38, 65, 146, 60, 230, 161, 248, 137, 156, 236, 150, 143, 54, 8, 56, 76, 108, 112, 0, 7, 144, 69, 225, 55, 125, 11, 147, 170, 153, 120, 170, 206, 159, 154, 4, 38, 228, 138, 73, 56, 184, 95, 102, 157, 110, 80, 61, 133, 255, 211, 5, 245, 193, 38, 105, 225, 95, 90, 190, 218, 13, 78, 239, 35, 65, 199, 239, 155, 3, 18, 104, 225, 39, 199, 78, 48, 223, 67, 201, 45, 206, 14, 170, 198, 115, 181, 12, 121, 48, 35, 198, 137, 203, 110, 43, 106, 111, 67, 92, 167, 96, 161, 130, 237, 107, 35, 205, 250, 215, 56, 2, 27, 145, 145, 12, 255, 163, 41, 175, 140, 186, 118, 107, 223, 75, 223, 22, 185, 156, 228, 192, 155, 231, 34, 204, 44, 139, 189, 91, 80, 232, 18, 161, 248, 33, 171, 247, 57, 98, 43, 198, 70, 147, 207, 236, 234, 54, 162, 162, 139, 12, 20, 94, 243, 120, 102, 51, 111, 197, 223, 115, 229, 206, 94, 164, 167, 212, 114, 189, 160, 231, 221, 86, 112, 29, 122, 8, 24, 53, 226, 245, 233, 27, 207, 68, 212, 179, 174, 127, 187, 119, 136, 116, 130, 184, 224, 120, 52, 43, 69, 21, 195, 228, 2, 164, 47, 189, 238, 213, 253, 94, 27, 58, 227, 233, 138, 202, 254, 26, 208, 243, 38, 242, 2, 40, 95, 104, 212, 183, 60, 2, 211, 236, 240, 204, 50, 4, 25, 42, 20, 22, 197, 217, 130, 93, 92, 77, 145, 152, 45, 22, 121, 26, 91, 81, 239, 208, 226, 217, 69, 198, 233, 54, 152, 251, 139, 46, 74, 22, 183, 133, 176 }),
                new BigInteger(new byte[] { 212, 5 }),
                new BigInteger(new byte[] { 113, 177, 174, 92, 245, 115, 123, 1, 18, 53, 6, 58, 240, 117, 97, 42, 204, 142, 202, 177, 44, 31, 23, 244, 25, 214, 104, 220, 102, 147, 235, 244, 170, 102, 136, 40, 238, 185, 99, 176, 112, 141, 231, 227, 208, 115, 66, 137, 199, 82, 63, 220, 33, 220, 179, 126, 148, 185, 66, 31, 11, 158, 154, 127, 74, 17, 143, 138, 237, 56, 59, 18, 42, 25, 65, 159, 55, 51, 36, 166, 147, 223, 21, 19, 83, 14, 210, 206, 179, 10, 148, 53, 118, 68, 165, 209, 242, 104, 153, 190, 227, 184, 156, 142, 22, 44, 178, 8, 154, 19, 70, 97, 149, 156, 140, 43, 137, 134, 43, 59, 62, 193, 214, 227, 164, 6, 155, 229, 115, 111, 172, 207, 154, 101, 194, 218, 66, 134, 252, 38, 92, 202, 51, 159, 195, 51, 160, 218, 184, 121, 72, 107, 60, 93, 127, 33, 225, 185, 234, 16, 144, 204, 148, 43, 103, 119, 198, 47, 217, 177, 233, 2, 138, 4, 223, 115, 141, 55, 31, 113, 102, 193, 45, 249, 38, 65, 146, 60, 230, 161, 248, 137, 156, 236, 150, 143, 54, 8, 56, 76, 108, 112, 0, 7, 144, 69, 225, 55, 125, 11, 147, 170, 153, 120, 170, 206, 159, 154, 4, 38, 228, 138, 73, 56, 184, 95, 102, 157, 110, 80, 61, 133, 255, 211, 5, 245, 193, 38, 105, 225, 95, 90, 190, 218, 13, 78, 239, 35, 65, 199, 239, 155, 3, 18, 104, 225, 39, 199, 78, 48, 223, 67, 201, 45, 206, 14, 170, 198, 115, 181, 12, 121, 48, 35, 198, 137, 203, 110, 43, 106, 111, 67, 92, 167, 96, 161, 130, 237, 107, 35, 205, 250, 215, 56, 2, 27, 145, 145, 12, 255, 163, 41, 175, 140, 186, 118, 107, 223, 75, 223, 22, 185, 156, 228, 192, 155, 231, 34, 204, 44, 139, 189, 91, 80, 232, 18, 161, 248, 33, 171, 247, 57, 98, 43, 198, 70, 147, 207, 236, 234, 54, 162, 162, 139, 12, 20, 94, 243, 120, 102, 51, 111, 197, 223, 115, 229, 206, 94, 164, 167, 212, 114, 189, 160, 231, 221, 86, 112, 29, 122, 8, 24, 53, 226, 245, 233, 27, 207, 68, 212, 179, 174, 127, 187, 119, 136, 116, 130, 184, 224, 120, 52, 43, 69, 21, 195, 228, 2, 164, 47, 189, 238, 213, 253, 94, 27, 58, 227, 233, 138, 202, 254, 26, 208, 243, 38, 242, 2, 40, 95, 104, 212, 183, 60, 2, 211, 236, 240, 204, 50, 4, 25, 42, 20, 22, 197, 217, 130, 93, 92, 77, 145, 152, 45, 22, 121, 26, 91, 81, 239, 208, 226, 217, 69, 198, 233, 54, 152, 251, 139, 46, 74, 22, 183, 133, 176 })
            };

            yield return new object[]
            {
                new BigInteger(new byte[] { 46, 77, 112, 165, 203, 17, 215, 59, 138, 38, 137, 184, 145, 158, 221, 97, 34, 63, 229, 77, 45, 143, 64, 134, 210, 45, 3, 123, 225, 168, 101, 164, 20, 162, 34, 161, 95, 61, 13, 51, 239, 174, 172, 114, 114, 210, 157, 171, 56, 85, 99, 113, 238, 183, 140, 159, 12, 234, 210, 158, 14, 92, 89, 225, 193, 228, 163, 238, 124, 60, 1, 229, 50, 177, 94, 127, 226, 206, 20, 255, 143, 87, 67, 139, 68, 244, 181, 92, 21, 131, 149, 117, 106, 111, 37, 146, 84, 202, 177, 141, 46, 174, 47, 251, 84, 125, 63, 61, 89, 13, 188, 62, 213, 211, 14, 182, 25, 206, 157, 204, 46, 71, 217, 248, 167, 139, 123, 194, 139, 204, 42, 24, 29, 134, 209, 225, 40, 71, 55, 198, 181, 120, 3, 8, 199, 86, 160, 150, 96, 111, 121, 134, 45, 228, 95, 231, 213, 54, 84, 200, 2, 124, 177, 140, 226, 164, 32, 79, 2, 45, 141, 210, 150, 214, 23, 182, 67, 208, 49, 80, 235, 228, 98, 27, 82, 163, 235, 57, 39, 233, 11, 241, 242, 111, 196, 56, 198, 118, 38, 34, 54, 80, 148, 51, 226, 167, 239, 86, 204, 169, 163, 5, 5, 4, 220, 29, 26, 149, 58, 62, 185, 231, 40, 24, 34, 155, 223, 39, 18, 223, 239, 204, 169, 14, 25, 155, 80, 47, 57, 171, 179, 148, 108, 125, 69, 5, 236, 237, 147, 166, 30, 48, 60, 231, 105, 78, 40, 123, 16, 70, 199, 200, 67, 237, 185, 215, 91, 246, 234, 65, 205, 235, 92, 143, 138, 36, 122, 170, 210, 28, 175, 242, 148, 108, 192, 190, 144, 194, 165, 203, 114, 212, 30, 243, 227, 235, 81, 127, 195, 110, 170, 245, 3, 81, 216, 51, 212, 173, 92, 68, 212, 126, 169, 244, 151, 213, 52, 215, 44, 251, 70, 153, 81, 31, 206, 248, 204, 104, 171, 159, 211, 250, 230, 100, 114, 185, 238, 66, 147, 141, 234, 226, 102, 96, 247, 244, 43, 42, 91, 14, 182, 122, 83, 200, 39, 16, 164, 227, 131, 223, 102, 107, 2, 188, 113, 217, 151, 69, 129, 221, 230, 242, 67, 158, 16, 100, 51, 240, 40, 218, 205, 162, 13, 156, 233, 89, 167, 29, 60, 97, 21, 11, 191, 179, 39, 126, 224, 169, 239, 29, 92, 230, 168, 126, 39, 196, 54, 181, 183, 195, 221, 179, 187, 168, 18, 196, 94, 101, 211, 24, 50, 121, 8, 31, 199, 219, 50, 143, 119, 146, 131, 137, 0, 10, 125 }),
                new BigInteger(new byte[] { 231, 249 }),
                new BigInteger(new byte[] { 21, 71, 112, 165, 203, 17, 215, 59, 138, 38, 137, 184, 145, 158, 221, 97, 34, 63, 229, 77, 45, 143, 64, 134, 210, 45, 3, 123, 225, 168, 101, 164, 20, 162, 34, 161, 95, 61, 13, 51, 239, 174, 172, 114, 114, 210, 157, 171, 56, 85, 99, 113, 238, 183, 140, 159, 12, 234, 210, 158, 14, 92, 89, 225, 193, 228, 163, 238, 124, 60, 1, 229, 50, 177, 94, 127, 226, 206, 20, 255, 143, 87, 67, 139, 68, 244, 181, 92, 21, 131, 149, 117, 106, 111, 37, 146, 84, 202, 177, 141, 46, 174, 47, 251, 84, 125, 63, 61, 89, 13, 188, 62, 213, 211, 14, 182, 25, 206, 157, 204, 46, 71, 217, 248, 167, 139, 123, 194, 139, 204, 42, 24, 29, 134, 209, 225, 40, 71, 55, 198, 181, 120, 3, 8, 199, 86, 160, 150, 96, 111, 121, 134, 45, 228, 95, 231, 213, 54, 84, 200, 2, 124, 177, 140, 226, 164, 32, 79, 2, 45, 141, 210, 150, 214, 23, 182, 67, 208, 49, 80, 235, 228, 98, 27, 82, 163, 235, 57, 39, 233, 11, 241, 242, 111, 196, 56, 198, 118, 38, 34, 54, 80, 148, 51, 226, 167, 239, 86, 204, 169, 163, 5, 5, 4, 220, 29, 26, 149, 58, 62, 185, 231, 40, 24, 34, 155, 223, 39, 18, 223, 239, 204, 169, 14, 25, 155, 80, 47, 57, 171, 179, 148, 108, 125, 69, 5, 236, 237, 147, 166, 30, 48, 60, 231, 105, 78, 40, 123, 16, 70, 199, 200, 67, 237, 185, 215, 91, 246, 234, 65, 205, 235, 92, 143, 138, 36, 122, 170, 210, 28, 175, 242, 148, 108, 192, 190, 144, 194, 165, 203, 114, 212, 30, 243, 227, 235, 81, 127, 195, 110, 170, 245, 3, 81, 216, 51, 212, 173, 92, 68, 212, 126, 169, 244, 151, 213, 52, 215, 44, 251, 70, 153, 81, 31, 206, 248, 204, 104, 171, 159, 211, 250, 230, 100, 114, 185, 238, 66, 147, 141, 234, 226, 102, 96, 247, 244, 43, 42, 91, 14, 182, 122, 83, 200, 39, 16, 164, 227, 131, 223, 102, 107, 2, 188, 113, 217, 151, 69, 129, 221, 230, 242, 67, 158, 16, 100, 51, 240, 40, 218, 205, 162, 13, 156, 233, 89, 167, 29, 60, 97, 21, 11, 191, 179, 39, 126, 224, 169, 239, 29, 92, 230, 168, 126, 39, 196, 54, 181, 183, 195, 221, 179, 187, 168, 18, 196, 94, 101, 211, 24, 50, 121, 8, 31, 199, 219, 50, 143, 119, 146, 131, 137, 0, 10, 125 })
            };

            yield return new object[]
            {
                new BigInteger(new byte[] { 117, 36, 34, 242, 102, 120, 131, 97, 79, 22, 161, 98, 32, 123, 213, 215, 235, 227, 221, 137, 182, 76, 145, 167, 61, 37, 64, 84, 178, 177, 42, 43, 202, 208, 239, 174, 245, 63, 19, 143, 155, 102, 139, 27, 238, 20, 113, 7, 179, 41, 30, 154, 9, 112, 215, 40, 146, 122, 180, 64, 56, 47, 174, 158, 236, 117, 151, 79, 140, 39, 225, 171, 207, 77, 237, 80, 192, 118, 185, 40, 179, 57, 160, 137, 146, 144, 33, 90, 248, 198, 27, 122, 139, 211, 87, 108, 183, 237, 47, 120, 124, 33, 123, 140, 71, 114, 202, 187, 131, 134, 104, 28, 193, 139, 140, 254, 143, 37, 12, 91, 84, 61, 87, 197, 11, 102, 32, 252, 245, 128, 153, 211, 70, 64, 172, 145, 190, 19, 65, 31, 198, 101, 215, 114, 94, 255, 185, 255, 70, 88, 220, 146, 225, 211, 36, 63, 202, 182, 129, 225, 82, 205, 198, 2, 237, 148, 214, 129, 223, 250, 64, 123, 227, 236, 149, 238, 101, 229, 102, 11, 173, 32, 181, 156, 163, 7, 242, 114, 28, 108, 199, 8, 145, 96, 205, 248, 159, 213, 132, 201, 40, 56, 31, 76, 24, 97, 174, 244, 62, 54, 218, 229, 79, 117, 51, 49, 23, 41, 94, 229, 162, 99, 100, 115, 51, 56, 233, 130, 31, 157, 78, 144, 96, 156, 227, 117, 0, 105, 132, 66, 88, 253, 52, 230, 146, 225, 184, 28, 44, 155, 224, 118, 119, 159, 39, 196, 211, 172, 25, 223, 119, 44, 213, 161, 231, 73, 133, 178, 146, 190, 48, 174, 164, 27, 141, 164, 47, 126, 225, 81, 128, 205, 85, 132, 188, 216, 24, 193, 117, 30, 161, 84, 80, 165, 203, 44, 40, 146, 254, 72, 92, 5, 137, 110, 106, 49, 209, 91, 17, 130, 148, 85, 203, 199, 95, 170, 214, 81, 228, 15, 49, 195, 61, 116, 28, 220, 94, 255, 80, 96, 124, 157, 127, 152, 245, 122, 68, 231, 25, 138, 6, 189, 175, 242, 137, 76, 136, 136, 70, 32, 86, 215, 174, 239, 23, 152, 199, 21, 81, 142, 211, 210, 11, 176, 6, 247 }),
                new BigInteger(new byte[] { 213, 50 }),
                new BigInteger(new byte[] { 74, 87, 34, 242, 102, 120, 131, 97, 79, 22, 161, 98, 32, 123, 213, 215, 235, 227, 221, 137, 182, 76, 145, 167, 61, 37, 64, 84, 178, 177, 42, 43, 202, 208, 239, 174, 245, 63, 19, 143, 155, 102, 139, 27, 238, 20, 113, 7, 179, 41, 30, 154, 9, 112, 215, 40, 146, 122, 180, 64, 56, 47, 174, 158, 236, 117, 151, 79, 140, 39, 225, 171, 207, 77, 237, 80, 192, 118, 185, 40, 179, 57, 160, 137, 146, 144, 33, 90, 248, 198, 27, 122, 139, 211, 87, 108, 183, 237, 47, 120, 124, 33, 123, 140, 71, 114, 202, 187, 131, 134, 104, 28, 193, 139, 140, 254, 143, 37, 12, 91, 84, 61, 87, 197, 11, 102, 32, 252, 245, 128, 153, 211, 70, 64, 172, 145, 190, 19, 65, 31, 198, 101, 215, 114, 94, 255, 185, 255, 70, 88, 220, 146, 225, 211, 36, 63, 202, 182, 129, 225, 82, 205, 198, 2, 237, 148, 214, 129, 223, 250, 64, 123, 227, 236, 149, 238, 101, 229, 102, 11, 173, 32, 181, 156, 163, 7, 242, 114, 28, 108, 199, 8, 145, 96, 205, 248, 159, 213, 132, 201, 40, 56, 31, 76, 24, 97, 174, 244, 62, 54, 218, 229, 79, 117, 51, 49, 23, 41, 94, 229, 162, 99, 100, 115, 51, 56, 233, 130, 31, 157, 78, 144, 96, 156, 227, 117, 0, 105, 132, 66, 88, 253, 52, 230, 146, 225, 184, 28, 44, 155, 224, 118, 119, 159, 39, 196, 211, 172, 25, 223, 119, 44, 213, 161, 231, 73, 133, 178, 146, 190, 48, 174, 164, 27, 141, 164, 47, 126, 225, 81, 128, 205, 85, 132, 188, 216, 24, 193, 117, 30, 161, 84, 80, 165, 203, 44, 40, 146, 254, 72, 92, 5, 137, 110, 106, 49, 209, 91, 17, 130, 148, 85, 203, 199, 95, 170, 214, 81, 228, 15, 49, 195, 61, 116, 28, 220, 94, 255, 80, 96, 124, 157, 127, 152, 245, 122, 68, 231, 25, 138, 6, 189, 175, 242, 137, 76, 136, 136, 70, 32, 86, 215, 174, 239, 23, 152, 199, 21, 81, 142, 211, 210, 11, 176, 6, 247 })
            };

            yield return new object[]
            {
                new BigInteger(new byte[] { 228, 102, 238, 55, 89, 8, 115, 20, 75, 234, 208, 83, 214, 184, 47, 10, 164, 238, 144, 129, 92, 179, 46, 149, 139, 183, 51, 177, 254, 235, 145, 119, 86, 181, 98, 186, 103, 159, 84, 230, 163, 132, 124, 42, 129, 58, 252, 26, 219, 166, 97, 9, 69, 176, 117, 48, 55, 89, 172, 161, 213, 193, 21, 95, 89, 88, 252, 32, 86, 116, 163, 4, 153, 169, 222, 216, 54, 4, 19, 80, 186, 25, 214, 88, 138, 136, 49, 165, 63, 49, 131, 13, 243, 179, 17, 225, 110, 29, 208, 40, 61, 220, 195, 102, 3, 93, 111, 156, 210, 157, 249, 50, 69, 91, 230, 187, 235, 188, 213, 209, 38, 87, 224, 36, 241, 149, 17, 230, 151, 252, 105, 25, 52, 234, 18, 222, 85, 111, 85, 44, 24, 149, 211, 162, 55, 81, 200, 151, 204, 86, 246, 177, 71, 255, 2, 230, 252, 159, 117, 109, 75, 137, 4, 213, 51, 224, 254, 90, 73, 8, 102, 123, 103, 168, 185, 144, 132, 61, 237, 159, 205, 121, 25, 65, 6, 184, 210, 53, 232, 44, 226, 182, 249, 231, 225, 142, 144, 253, 110, 87, 83, 109, 78, 196, 240, 122, 74, 159, 70, 113, 98, 190, 178, 214, 160, 209, 111, 195, 207, 123, 13, 201, 114, 28, 38, 175, 130, 127, 199, 42, 0, 134, 206, 149, 76, 95, 43, 85, 80, 140, 109, 50, 239, 119, 201, 35, 4, 35, 71, 15, 30, 204, 45, 243, 73, 137, 251, 49, 157, 64, 247, 202, 216, 27, 113, 219, 239, 28, 137, 64, 166, 26, 114, 67, 13, 218, 218, 250, 83, 2, 171, 94, 56, 183, 11, 51, 89, 219, 75, 194, 100, 250, 183, 16, 148, 162, 90, 211, 6, 237, 52, 231, 154, 6, 105, 4, 89, 234, 229, 110, 175, 0, 221, 155, 148, 153, 146, 32, 16, 61, 130, 19, 208, 198, 220, 171, 99, 98, 174, 106, 128, 6, 244, 101, 206, 196, 196, 49, 78, 7, 217, 110, 246, 221, 19, 99, 28, 27, 124, 251, 16, 58, 194, 200, 107, 32, 23, 211, 41, 189, 160, 247, 59, 122, 238, 168, 12, 120, 205, 208, 212, 96, 210, 9, 100, 19, 29, 243, 179, 120, 143, 71, 229, 179, 90, 69, 68, 43, 250, 173, 172, 240, 7, 144, 102, 225, 50, 123, 238, 106, 87, 164, 78, 172, 38, 175, 104, 185, 100, 87, 3, 36, 32, 177, 46, 89, 18, 136, 31, 168, 100, 224, 125, 31, 35, 228, 88, 65, 162, 131, 224, 162, 4, 72, 32, 235, 248, 57, 141, 171, 221, 138, 199, 163, 170, 136, 207, 86, 97, 13, 207, 170, 91, 69, 5, 119, 38, 47, 136, 65, 86, 38, 23, 224, 118, 97, 31, 215, 144, 67, 96, 216, 251, 116, 203, 218, 25, 218, 116, 154, 20, 2, 223, 149, 179, 54, 70, 190, 67, 169, 196, 201, 176, 75, 85, 182, 115, 230, 45, 73, 105, 248, 198, 30, 172, 246, 175, 231, 122, 43, 93, 75, 186, 238, 45, 83, 70, 129, 44, 63, 26, 97, 148, 230, 126, 151, 39, 176, 31, 20, 103, 50, 173, 43, 49, 28, 60, 193, 232, 189, 134, 94, 68 }),
                new BigInteger(new byte[] { 24, 75 }),
                new BigInteger(new byte[] { 252, 177, 238, 55, 89, 8, 115, 20, 75, 234, 208, 83, 214, 184, 47, 10, 164, 238, 144, 129, 92, 179, 46, 149, 139, 183, 51, 177, 254, 235, 145, 119, 86, 181, 98, 186, 103, 159, 84, 230, 163, 132, 124, 42, 129, 58, 252, 26, 219, 166, 97, 9, 69, 176, 117, 48, 55, 89, 172, 161, 213, 193, 21, 95, 89, 88, 252, 32, 86, 116, 163, 4, 153, 169, 222, 216, 54, 4, 19, 80, 186, 25, 214, 88, 138, 136, 49, 165, 63, 49, 131, 13, 243, 179, 17, 225, 110, 29, 208, 40, 61, 220, 195, 102, 3, 93, 111, 156, 210, 157, 249, 50, 69, 91, 230, 187, 235, 188, 213, 209, 38, 87, 224, 36, 241, 149, 17, 230, 151, 252, 105, 25, 52, 234, 18, 222, 85, 111, 85, 44, 24, 149, 211, 162, 55, 81, 200, 151, 204, 86, 246, 177, 71, 255, 2, 230, 252, 159, 117, 109, 75, 137, 4, 213, 51, 224, 254, 90, 73, 8, 102, 123, 103, 168, 185, 144, 132, 61, 237, 159, 205, 121, 25, 65, 6, 184, 210, 53, 232, 44, 226, 182, 249, 231, 225, 142, 144, 253, 110, 87, 83, 109, 78, 196, 240, 122, 74, 159, 70, 113, 98, 190, 178, 214, 160, 209, 111, 195, 207, 123, 13, 201, 114, 28, 38, 175, 130, 127, 199, 42, 0, 134, 206, 149, 76, 95, 43, 85, 80, 140, 109, 50, 239, 119, 201, 35, 4, 35, 71, 15, 30, 204, 45, 243, 73, 137, 251, 49, 157, 64, 247, 202, 216, 27, 113, 219, 239, 28, 137, 64, 166, 26, 114, 67, 13, 218, 218, 250, 83, 2, 171, 94, 56, 183, 11, 51, 89, 219, 75, 194, 100, 250, 183, 16, 148, 162, 90, 211, 6, 237, 52, 231, 154, 6, 105, 4, 89, 234, 229, 110, 175, 0, 221, 155, 148, 153, 146, 32, 16, 61, 130, 19, 208, 198, 220, 171, 99, 98, 174, 106, 128, 6, 244, 101, 206, 196, 196, 49, 78, 7, 217, 110, 246, 221, 19, 99, 28, 27, 124, 251, 16, 58, 194, 200, 107, 32, 23, 211, 41, 189, 160, 247, 59, 122, 238, 168, 12, 120, 205, 208, 212, 96, 210, 9, 100, 19, 29, 243, 179, 120, 143, 71, 229, 179, 90, 69, 68, 43, 250, 173, 172, 240, 7, 144, 102, 225, 50, 123, 238, 106, 87, 164, 78, 172, 38, 175, 104, 185, 100, 87, 3, 36, 32, 177, 46, 89, 18, 136, 31, 168, 100, 224, 125, 31, 35, 228, 88, 65, 162, 131, 224, 162, 4, 72, 32, 235, 248, 57, 141, 171, 221, 138, 199, 163, 170, 136, 207, 86, 97, 13, 207, 170, 91, 69, 5, 119, 38, 47, 136, 65, 86, 38, 23, 224, 118, 97, 31, 215, 144, 67, 96, 216, 251, 116, 203, 218, 25, 218, 116, 154, 20, 2, 223, 149, 179, 54, 70, 190, 67, 169, 196, 201, 176, 75, 85, 182, 115, 230, 45, 73, 105, 248, 198, 30, 172, 246, 175, 231, 122, 43, 93, 75, 186, 238, 45, 83, 70, 129, 44, 63, 26, 97, 148, 230, 126, 151, 39, 176, 31, 20, 103, 50, 173, 43, 49, 28, 60, 193, 232, 189, 134, 94, 68 })
            };

            yield return new object[]
            {
                new BigInteger(new byte[] { 14, 194, 29, 1, 112, 241, 185, 237, 46, 8, 100, 223, 160, 20, 207, 255, 200, 101, 195, 43, 158, 142, 188, 17, 23, 133, 153, 111, 185, 187, 28, 123, 154, 5, 136, 100, 146, 29, 164, 118, 120, 192, 252, 41, 183, 92, 32, 211, 237, 135, 150, 222, 84, 32, 218, 127, 5, 12, 233, 234, 88, 74, 51, 115, 235, 232, 68, 155, 140, 106, 109, 170, 192, 77, 179, 221, 146, 147, 89, 186, 100, 197, 129, 49, 37, 62, 39, 122, 42, 8, 94, 134, 51, 186, 30, 46, 140, 137, 62, 207, 23, 133, 71, 131, 26, 235, 30, 7, 109, 253, 237, 114, 178, 47, 133, 146, 200, 2, 77, 173, 192, 202, 113, 131, 11, 230, 118, 6, 46, 132, 81, 8, 85, 137, 162, 223, 125, 254, 23, 57, 31, 31, 12, 44, 27, 236, 211, 4, 52, 139, 101, 138, 59, 145, 14, 77, 20, 196, 119, 51, 116, 24, 216, 232, 171, 229, 29, 40, 140, 166, 21, 202, 234, 20, 141, 161, 189, 68, 103, 31, 18, 114, 209, 163, 31, 198, 204, 196, 123, 85, 202, 184, 134 }),
                new BigInteger(new byte[] { 228, 196 }),
                new BigInteger(new byte[] { 242, 134, 29, 1, 112, 241, 185, 237, 46, 8, 100, 223, 160, 20, 207, 255, 200, 101, 195, 43, 158, 142, 188, 17, 23, 133, 153, 111, 185, 187, 28, 123, 154, 5, 136, 100, 146, 29, 164, 118, 120, 192, 252, 41, 183, 92, 32, 211, 237, 135, 150, 222, 84, 32, 218, 127, 5, 12, 233, 234, 88, 74, 51, 115, 235, 232, 68, 155, 140, 106, 109, 170, 192, 77, 179, 221, 146, 147, 89, 186, 100, 197, 129, 49, 37, 62, 39, 122, 42, 8, 94, 134, 51, 186, 30, 46, 140, 137, 62, 207, 23, 133, 71, 131, 26, 235, 30, 7, 109, 253, 237, 114, 178, 47, 133, 146, 200, 2, 77, 173, 192, 202, 113, 131, 11, 230, 118, 6, 46, 132, 81, 8, 85, 137, 162, 223, 125, 254, 23, 57, 31, 31, 12, 44, 27, 236, 211, 4, 52, 139, 101, 138, 59, 145, 14, 77, 20, 196, 119, 51, 116, 24, 216, 232, 171, 229, 29, 40, 140, 166, 21, 202, 234, 20, 141, 161, 189, 68, 103, 31, 18, 114, 209, 163, 31, 198, 204, 196, 123, 85, 202, 184, 134 })
            };

            yield return new object[]
            {
                new BigInteger(new byte[] { 71, 36, 129, 54, 207, 171, 120, 141, 117, 155, 160, 38, 3, 109, 143, 208, 92, 88, 32, 2, 70, 52, 201, 229, 24, 89, 173, 54, 219, 92, 68, 5, 80, 134, 89, 153, 195, 49, 80, 103, 249, 67, 45, 132, 38, 165, 192, 13, 198, 250, 91, 45, 195, 193, 192, 18, 90, 155, 29, 117, 253, 65, 177, 24, 86, 154, 214, 125, 19, 245, 13, 43, 8, 184, 9, 2, 7, 69, 190, 114, 152, 159, 112, 225, 0, 22, 65, 143, 197, 71, 62, 39, 19, 218, 105, 183, 145, 20, 45, 140, 207, 66, 249, 208, 237, 48, 37, 11, 184, 189, 11, 21, 221, 171, 220, 93, 209, 208, 23, 64, 88, 71, 183, 204, 182, 229, 23, 45, 79, 81, 113, 242, 24, 50, 162, 86, 166, 159, 244, 208, 241, 54, 214, 7, 59, 41, 73, 104, 253, 11, 229, 193, 252, 236, 51, 136, 138, 44, 26, 7, 24, 247, 187, 103, 76, 25, 253, 171, 8, 134, 183, 50, 220, 71, 79, 33, 112, 176, 144, 141, 156, 175, 48, 67, 107, 175, 245, 43, 254, 26, 203, 121, 133, 236, 183, 249, 122, 111, 187, 34, 43, 158, 95, 118, 84, 179, 228, 181, 157, 17, 23, 218, 156, 140, 107, 105, 199, 120, 251, 156, 36, 209, 172, 237, 187, 61, 173, 240, 144, 86, 167, 0, 244, 110, 97, 253, 79, 185, 238, 183, 203, 63, 142, 237, 2, 240, 221, 248, 129, 77, 50, 2, 115, 30, 254, 89, 242, 113, 187, 22, 5, 244, 37, 71, 106, 23, 229, 45, 42, 110, 25, 13, 137, 68, 209, 228, 67, 191, 235, 203, 96, 181, 111, 66 }),
                new BigInteger(new byte[] { 35, 164 }),
                new BigInteger(new byte[] { 106, 200, 128, 54, 207, 171, 120, 141, 117, 155, 160, 38, 3, 109, 143, 208, 92, 88, 32, 2, 70, 52, 201, 229, 24, 89, 173, 54, 219, 92, 68, 5, 80, 134, 89, 153, 195, 49, 80, 103, 249, 67, 45, 132, 38, 165, 192, 13, 198, 250, 91, 45, 195, 193, 192, 18, 90, 155, 29, 117, 253, 65, 177, 24, 86, 154, 214, 125, 19, 245, 13, 43, 8, 184, 9, 2, 7, 69, 190, 114, 152, 159, 112, 225, 0, 22, 65, 143, 197, 71, 62, 39, 19, 218, 105, 183, 145, 20, 45, 140, 207, 66, 249, 208, 237, 48, 37, 11, 184, 189, 11, 21, 221, 171, 220, 93, 209, 208, 23, 64, 88, 71, 183, 204, 182, 229, 23, 45, 79, 81, 113, 242, 24, 50, 162, 86, 166, 159, 244, 208, 241, 54, 214, 7, 59, 41, 73, 104, 253, 11, 229, 193, 252, 236, 51, 136, 138, 44, 26, 7, 24, 247, 187, 103, 76, 25, 253, 171, 8, 134, 183, 50, 220, 71, 79, 33, 112, 176, 144, 141, 156, 175, 48, 67, 107, 175, 245, 43, 254, 26, 203, 121, 133, 236, 183, 249, 122, 111, 187, 34, 43, 158, 95, 118, 84, 179, 228, 181, 157, 17, 23, 218, 156, 140, 107, 105, 199, 120, 251, 156, 36, 209, 172, 237, 187, 61, 173, 240, 144, 86, 167, 0, 244, 110, 97, 253, 79, 185, 238, 183, 203, 63, 142, 237, 2, 240, 221, 248, 129, 77, 50, 2, 115, 30, 254, 89, 242, 113, 187, 22, 5, 244, 37, 71, 106, 23, 229, 45, 42, 110, 25, 13, 137, 68, 209, 228, 67, 191, 235, 203, 96, 181, 111, 66 }),
            };

            yield return new object[]
            {
                new BigInteger(new byte[] { 214, 111, 8, 10, 221, 254, 216, 177, 214, 26, 71, 130, 235, 10, 175, 205, 18, 51, 37, 120, 47, 76, 25, 22, 51, 134, 239, 182, 80, 133, 226, 35, 197, 137, 15, 190, 34, 99, 60, 14, 177, 146, 250, 202, 252, 61, 255, 192, 176, 231, 92, 14, 254, 44, 93, 137, 86, 242, 215, 86, 14, 33, 97, 80, 55, 35, 188, 97, 251, 241, 171, 175, 247, 22, 198, 157, 81, 79, 25, 246, 135, 47, 5, 105, 41, 212, 36, 153, 44, 133, 103, 48, 139, 229, 255, 143, 48, 169, 146, 216, 128, 157, 197, 191, 59, 172, 22, 232, 102, 192, 55, 6, 216, 225, 207, 223, 28, 247, 39, 99, 254 }),
                new BigInteger(new byte[] { 35, 53 }),
                new BigInteger(new byte[] { 249, 164, 8, 10, 221, 254, 216, 177, 214, 26, 71, 130, 235, 10, 175, 205, 18, 51, 37, 120, 47, 76, 25, 22, 51, 134, 239, 182, 80, 133, 226, 35, 197, 137, 15, 190, 34, 99, 60, 14, 177, 146, 250, 202, 252, 61, 255, 192, 176, 231, 92, 14, 254, 44, 93, 137, 86, 242, 215, 86, 14, 33, 97, 80, 55, 35, 188, 97, 251, 241, 171, 175, 247, 22, 198, 157, 81, 79, 25, 246, 135, 47, 5, 105, 41, 212, 36, 153, 44, 133, 103, 48, 139, 229, 255, 143, 48, 169, 146, 216, 128, 157, 197, 191, 59, 172, 22, 232, 102, 192, 55, 6, 216, 225, 207, 223, 28, 247, 39, 99, 254 })
            };

            yield return new object[]
            {
                new BigInteger(new byte[] { 137, 123, 36, 18, 22, 54, 108, 167, 189, 65, 117, 234, 106, 70, 46, 125, 189, 60, 50, 108, 77, 97, 87, 169, 22, 32, 115, 57, 130, 46, 217, 92, 104, 79, 53, 192, 136, 4, 209, 47, 83, 143, 73, 150, 108, 229, 117, 213, 201, 170, 165, 193, 240, 201, 39, 40, 35, 123, 252, 245, 195, 51, 37, 142, 103, 24, 129, 26, 17, 109, 244, 185, 107, 3, 24, 190, 24, 193, 60, 49, 171, 158, 111, 216, 137, 23, 108, 159, 40, 12, 156, 13, 8, 219, 107, 32, 106, 186, 46, 83, 99, 91, 195, 91, 181, 236, 86, 236, 176, 105, 16, 98, 62, 202, 74, 37, 195, 77, 5, 80, 172, 226, 241, 4, 208, 231, 177, 143, 152, 247, 83, 93, 146, 233, 205, 80, 218, 20, 34, 157, 193, 127, 238, 190, 252, 58, 206, 61, 145, 70, 93, 29, 181, 222, 167, 129, 105, 190, 139, 205, 58, 198, 84, 184, 21, 179, 208, 84, 253, 250, 74, 174, 42, 103, 143, 44, 243, 51, 8, 150, 25, 116, 254, 81, 154, 54, 167, 179, 65, 76, 230, 27, 137, 85, 98, 250, 42, 54, 168, 72, 106, 122, 63, 150, 251, 174, 242, 77, 79, 123, 141, 54, 182, 245, 180, 198, 199, 3, 30, 223, 11, 28, 19, 176, 19, 47, 37, 213, 5, 148, 1, 189, 138, 192, 43, 158, 52, 104, 85, 235, 68, 89, 100, 198, 191, 176, 101, 148, 160, 156, 51, 39, 23, 201, 60, 78, 102, 143, 131, 203, 137, 28, 72, 187, 121, 207, 172, 245, 201, 21, 146, 95, 173, 51, 155, 178, 184, 77, 241, 98, 202, 145, 52, 105, 97, 217, 165, 192, 131, 221, 56, 164, 228, 138, 98, 39, 16, 169, 76, 239, 4, 111, 202, 139, 9, 212, 122, 228, 46, 138, 149, 25, 157, 32, 1, 247, 232, 223, 89, 160, 42, 235, 114, 236, 221, 237, 122, 35, 208, 116, 161, 15, 1, 2, 93, 90, 198, 169, 95, 140, 95, 193, 146, 248, 71, 31, 7, 196, 136, 106, 62, 49, 79, 171, 196, 25, 253, 222, 174, 28, 89, 87, 19, 186, 232, 134, 23, 155, 195, 166, 48, 62, 128, 205, 65, 104, 89, 121, 164, 189, 230, 182, 154, 102, 53, 112, 191, 85, 61, 68, 93, 232, 251, 67, 50, 7, 173, 215, 16, 193, 7, 107, 1, 225, 57, 0, 177, 130, 106, 92, 191, 131, 58, 240, 53, 163, 188, 172, 132, 120, 198, 194, 94, 126, 73, 72, 66, 60, 154, 57, 187, 129, 105, 227, 181, 122, 180, 185, 45, 52, 191, 61, 234, 225, 133, 218, 174, 11, 13, 143, 74, 0, 82, 152, 250, 69, 13, 130, 152, 241, 189, 116, 232, 49, 160, 61, 26, 86, 58, 187, 238, 3, 127, 79, 184, 136, 215, 124, 249, 111, 154, 55, 47, 11, 239, 186, 46, 208, 232, 112, 109, 50, 32, 60, 119, 75, 85, 185, 64, 71, 191, 87, 208, 82, 160, 70, 128, 3, 223, 113, 109, 144, 137, 41, 86, 133, 69, 220, 65, 230, 15, 73, 109, 201, 77, 188, 227, 67, 216, 109, 51, 30, 60, 178, 175, 66 }),
                new BigInteger(new byte[] { 103, 220 }),
                new BigInteger(new byte[] { 240, 87, 36, 18, 22, 54, 108, 167, 189, 65, 117, 234, 106, 70, 46, 125, 189, 60, 50, 108, 77, 97, 87, 169, 22, 32, 115, 57, 130, 46, 217, 92, 104, 79, 53, 192, 136, 4, 209, 47, 83, 143, 73, 150, 108, 229, 117, 213, 201, 170, 165, 193, 240, 201, 39, 40, 35, 123, 252, 245, 195, 51, 37, 142, 103, 24, 129, 26, 17, 109, 244, 185, 107, 3, 24, 190, 24, 193, 60, 49, 171, 158, 111, 216, 137, 23, 108, 159, 40, 12, 156, 13, 8, 219, 107, 32, 106, 186, 46, 83, 99, 91, 195, 91, 181, 236, 86, 236, 176, 105, 16, 98, 62, 202, 74, 37, 195, 77, 5, 80, 172, 226, 241, 4, 208, 231, 177, 143, 152, 247, 83, 93, 146, 233, 205, 80, 218, 20, 34, 157, 193, 127, 238, 190, 252, 58, 206, 61, 145, 70, 93, 29, 181, 222, 167, 129, 105, 190, 139, 205, 58, 198, 84, 184, 21, 179, 208, 84, 253, 250, 74, 174, 42, 103, 143, 44, 243, 51, 8, 150, 25, 116, 254, 81, 154, 54, 167, 179, 65, 76, 230, 27, 137, 85, 98, 250, 42, 54, 168, 72, 106, 122, 63, 150, 251, 174, 242, 77, 79, 123, 141, 54, 182, 245, 180, 198, 199, 3, 30, 223, 11, 28, 19, 176, 19, 47, 37, 213, 5, 148, 1, 189, 138, 192, 43, 158, 52, 104, 85, 235, 68, 89, 100, 198, 191, 176, 101, 148, 160, 156, 51, 39, 23, 201, 60, 78, 102, 143, 131, 203, 137, 28, 72, 187, 121, 207, 172, 245, 201, 21, 146, 95, 173, 51, 155, 178, 184, 77, 241, 98, 202, 145, 52, 105, 97, 217, 165, 192, 131, 221, 56, 164, 228, 138, 98, 39, 16, 169, 76, 239, 4, 111, 202, 139, 9, 212, 122, 228, 46, 138, 149, 25, 157, 32, 1, 247, 232, 223, 89, 160, 42, 235, 114, 236, 221, 237, 122, 35, 208, 116, 161, 15, 1, 2, 93, 90, 198, 169, 95, 140, 95, 193, 146, 248, 71, 31, 7, 196, 136, 106, 62, 49, 79, 171, 196, 25, 253, 222, 174, 28, 89, 87, 19, 186, 232, 134, 23, 155, 195, 166, 48, 62, 128, 205, 65, 104, 89, 121, 164, 189, 230, 182, 154, 102, 53, 112, 191, 85, 61, 68, 93, 232, 251, 67, 50, 7, 173, 215, 16, 193, 7, 107, 1, 225, 57, 0, 177, 130, 106, 92, 191, 131, 58, 240, 53, 163, 188, 172, 132, 120, 198, 194, 94, 126, 73, 72, 66, 60, 154, 57, 187, 129, 105, 227, 181, 122, 180, 185, 45, 52, 191, 61, 234, 225, 133, 218, 174, 11, 13, 143, 74, 0, 82, 152, 250, 69, 13, 130, 152, 241, 189, 116, 232, 49, 160, 61, 26, 86, 58, 187, 238, 3, 127, 79, 184, 136, 215, 124, 249, 111, 154, 55, 47, 11, 239, 186, 46, 208, 232, 112, 109, 50, 32, 60, 119, 75, 85, 185, 64, 71, 191, 87, 208, 82, 160, 70, 128, 3, 223, 113, 109, 144, 137, 41, 86, 133, 69, 220, 65, 230, 15, 73, 109, 201, 77, 188, 227, 67, 216, 109, 51, 30, 60, 178, 175, 66 })
            };

            yield return new object[]
            {
                new BigInteger(new byte[] { 168, 57, 43, 241, 119, 3, 220, 168, 247, 229, 241, 5, 120, 119, 90, 56, 117, 233, 6, 4, 227, 99, 61, 42, 114, 62, 78, 84, 215, 121, 20, 30, 104, 114, 153, 172, 227, 88, 246, 198, 113, 19, 235, 94, 231, 103, 186, 167, 225, 121, 57, 88, 96, 215, 135, 68, 252, 1, 126, 57, 180, 135, 209, 125, 208, 211, 156, 6, 222, 173, 84, 29, 243, 64, 147, 207, 120, 222, 67, 11, 131, 167, 114, 93, 63, 187, 189, 145, 234, 84, 175, 225, 218, 189, 17, 233, 66, 109, 142, 19, 202, 179, 200, 52, 36, 28, 101, 32, 68, 184, 204, 30, 189, 114, 181, 13, 21, 115, 62, 21, 21, 11, 28, 137, 253, 115, 66, 54, 47, 169, 141, 10, 80, 47, 64, 207, 222, 62, 56, 35, 85, 157, 76, 50, 136, 145, 35, 103, 7, 4, 212, 206, 47, 120, 254, 190, 151, 63, 54, 177, 217, 47, 241, 154, 43, 194, 205, 142, 49, 230, 46, 214, 59, 27, 191, 119, 190, 234, 0, 108, 79, 219, 46, 42, 213, 190, 219, 215, 49, 130, 55, 158, 8, 134, 73, 38, 172, 178, 6, 198, 196, 149, 53, 32, 213, 253, 147, 20, 185, 134, 0, 173, 63, 201, 98, 254, 0, 198, 196, 108, 231, 246, 93, 175, 174, 143, 206, 180, 209, 153, 203, 12, 227, 58, 168, 185, 165, 13, 85, 216, 43, 198, 29, 170, 130, 138, 95, 62, 36, 75, 37, 230, 237, 154, 223, 205, 56, 133, 114, 69, 47, 222, 66, 32, 187, 243, 202, 4, 33, 169, 88, 242, 113, 236, 65, 32, 216, 178, 45, 36, 47, 143, 143, 134, 115, 229, 30, 72, 91, 218, 129, 32, 155, 15, 169, 76, 132, 252, 239, 142, 191, 91, 28, 123, 243, 50, 117, 1, 89, 191, 244, 134, 88, 77, 21, 159, 79, 187, 172, 213, 212, 129, 168, 71, 40, 56, 87, 97, 66, 244, 155, 219, 194, 158, 100, 212, 115, 20, 147, 65, 112, 29, 239, 156, 229, 250, 199, 77, 249, 10, 253, 200, 79, 25, 185, 62, 178, 213, 83, 187, 218, 19, 190, 100, 35, 24, 195, 185, 233, 65, 44, 58, 40, 107, 101, 183, 146, 12, 97, 45, 112, 9, 104, 56, 247, 210, 140, 168, 228, 38, 34, 157, 193, 216, 102, 92, 94, 139, 120, 204, 55, 14, 100, 184, 222, 195, 160, 228, 180, 2, 171, 165, 115, 38, 74, 209, 170, 134, 109, 81, 140, 26, 212, 194, 31, 142, 120, 80, 5, 9, 88, 7, 148, 148, 245, 97, 164, 176, 90, 52, 50, 167, 244, 226, 122, 124, 42, 155, 141, 148, 177, 216, 29, 39, 63, 28, 58, 162, 152, 79, 74, 80, 223, 171, 170, 164, 17, 222, 48, 232, 44, 250, 43, 57, 31, 229, 37, 242, 71, 162, 100, 221, 194, 112, 87, 127, 233, 109, 85, 216, 39, 2, 23, 11, 233, 225, 200, 73, 55, 213, 106, 75, 106, 164, 104, 183, 173, 228, 7, 90, 246, 71, 90, 246, 235, 108, 141, 110, 83, 42, 187, 164, 137, 87, 193, 42, 226, 32, 79, 61, 28, 220, 186, 113, 56, 25, 114, 29, 7, 159, 210, 5, 101, 250, 226, 223, 167, 33, 31, 125, 83, 89, 245, 12, 26, 198, 193, 19, 227, 62, 212, 140, 148, 202, 62, 26, 141, 232, 190, 210, 250, 112, 102, 180, 88, 182, 63, 143, 117, 225, 130, 193, 0, 210, 234, 194, 230, 173, 87, 113, 87, 95, 58, 201, 202, 69, 112, 155, 188, 199, 82, 191, 99, 77, 131, 227, 242, 65, 180, 99, 134, 50, 158, 2, 187, 19, 139, 194, 223, 123, 52, 223, 145, 76, 122, 154, 53, 157, 233, 18, 70, 164, 210, 173, 142, 195, 94, 178, 79, 6, 207, 164, 249, 243, 235, 36, 193, 56, 13, 183, 185, 174, 187, 64, 147, 114, 45, 23, 211, 232, 174, 85, 88, 161, 29, 226, 96, 240, 115, 248, 181, 217, 114, 217, 171, 143, 230, 157, 96, 86, 217, 252, 143, 49, 89, 152, 232, 145, 26, 28, 149, 70, 223, 103, 88, 122, 76, 88, 214, 9, 196, 209, 197, 20, 2, 224, 59, 226, 149, 4, 140, 144, 182, 124, 14, 177, 198, 38, 72, 133, 249, 206, 213, 87, 99, 110, 249, 11, 128, 49, 66, 141, 199, 87, 146, 8, 55, 122, 47, 120, 92, 6, 252, 22, 143, 5, 144, 99, 88, 167, 179, 38, 16, 80, 15, 245, 251, 109, 177, 147, 230, 47, 97, 99, 193, 254, 200, 94, 234, 5, 121, 75, 246, 207, 40, 242, 209, 191, 199, 94, 222, 150, 178, 217, 125, 27, 125, 118, 72, 156, 13, 202, 200, 155, 145, 45, 165, 152, 84, 145, 60, 49, 120, 82, 46, 104, 47, 64, 39, 28, 36, 59, 165, 83, 181, 80, 124, 136, 230, 166, 128, 82, 22, 77, 248, 174, 130, 91, 62, 251, 77, 56, 107, 54, 33, 172, 29, 95, 170, 21, 235, 61, 79, 43, 120, 209, 36, 117, 229, 65, 176, 28, 178, 173, 16, 234, 27, 43, 90, 127, 229, 212, 1, 44, 207, 235, 208, 111, 49, 9, 50, 28, 221, 208, 186, 66, 3, 235, 213, 38, 138, 13, 117, 253, 75, 48, 133, 189, 38, 55, 145, 242, 5, 43, 47, 23, 59, 34, 91, 141, 1, 181, 67, 220, 14, 147, 63, 225, 243, 205, 230, 47, 85, 51, 244, 216, 94, 4, 225, 159, 101, 19, 72, 250, 120, 63, 23, 177, 161, 163, 7, 224, 143, 72, 37, 213, 202, 153, 238, 183, 241, 164, 202, 88, 80, 97, 156, 210, 54, 87, 184, 55, 65, 188, 39, 237, 108, 73, 100, 206, 124, 141, 64, 79, 198, 6, 160, 13, 13, 31, 13, 197, 21, 41, 174, 193, 73, 240, 169, 22, 163, 68, 123, 73, 235, 208, 158, 211, 32, 56, 104, 129, 53 }),
                new BigInteger(new byte[] { 32, 58 }),
                new BigInteger(new byte[] { 200, 115, 43, 241, 119, 3, 220, 168, 247, 229, 241, 5, 120, 119, 90, 56, 117, 233, 6, 4, 227, 99, 61, 42, 114, 62, 78, 84, 215, 121, 20, 30, 104, 114, 153, 172, 227, 88, 246, 198, 113, 19, 235, 94, 231, 103, 186, 167, 225, 121, 57, 88, 96, 215, 135, 68, 252, 1, 126, 57, 180, 135, 209, 125, 208, 211, 156, 6, 222, 173, 84, 29, 243, 64, 147, 207, 120, 222, 67, 11, 131, 167, 114, 93, 63, 187, 189, 145, 234, 84, 175, 225, 218, 189, 17, 233, 66, 109, 142, 19, 202, 179, 200, 52, 36, 28, 101, 32, 68, 184, 204, 30, 189, 114, 181, 13, 21, 115, 62, 21, 21, 11, 28, 137, 253, 115, 66, 54, 47, 169, 141, 10, 80, 47, 64, 207, 222, 62, 56, 35, 85, 157, 76, 50, 136, 145, 35, 103, 7, 4, 212, 206, 47, 120, 254, 190, 151, 63, 54, 177, 217, 47, 241, 154, 43, 194, 205, 142, 49, 230, 46, 214, 59, 27, 191, 119, 190, 234, 0, 108, 79, 219, 46, 42, 213, 190, 219, 215, 49, 130, 55, 158, 8, 134, 73, 38, 172, 178, 6, 198, 196, 149, 53, 32, 213, 253, 147, 20, 185, 134, 0, 173, 63, 201, 98, 254, 0, 198, 196, 108, 231, 246, 93, 175, 174, 143, 206, 180, 209, 153, 203, 12, 227, 58, 168, 185, 165, 13, 85, 216, 43, 198, 29, 170, 130, 138, 95, 62, 36, 75, 37, 230, 237, 154, 223, 205, 56, 133, 114, 69, 47, 222, 66, 32, 187, 243, 202, 4, 33, 169, 88, 242, 113, 236, 65, 32, 216, 178, 45, 36, 47, 143, 143, 134, 115, 229, 30, 72, 91, 218, 129, 32, 155, 15, 169, 76, 132, 252, 239, 142, 191, 91, 28, 123, 243, 50, 117, 1, 89, 191, 244, 134, 88, 77, 21, 159, 79, 187, 172, 213, 212, 129, 168, 71, 40, 56, 87, 97, 66, 244, 155, 219, 194, 158, 100, 212, 115, 20, 147, 65, 112, 29, 239, 156, 229, 250, 199, 77, 249, 10, 253, 200, 79, 25, 185, 62, 178, 213, 83, 187, 218, 19, 190, 100, 35, 24, 195, 185, 233, 65, 44, 58, 40, 107, 101, 183, 146, 12, 97, 45, 112, 9, 104, 56, 247, 210, 140, 168, 228, 38, 34, 157, 193, 216, 102, 92, 94, 139, 120, 204, 55, 14, 100, 184, 222, 195, 160, 228, 180, 2, 171, 165, 115, 38, 74, 209, 170, 134, 109, 81, 140, 26, 212, 194, 31, 142, 120, 80, 5, 9, 88, 7, 148, 148, 245, 97, 164, 176, 90, 52, 50, 167, 244, 226, 122, 124, 42, 155, 141, 148, 177, 216, 29, 39, 63, 28, 58, 162, 152, 79, 74, 80, 223, 171, 170, 164, 17, 222, 48, 232, 44, 250, 43, 57, 31, 229, 37, 242, 71, 162, 100, 221, 194, 112, 87, 127, 233, 109, 85, 216, 39, 2, 23, 11, 233, 225, 200, 73, 55, 213, 106, 75, 106, 164, 104, 183, 173, 228, 7, 90, 246, 71, 90, 246, 235, 108, 141, 110, 83, 42, 187, 164, 137, 87, 193, 42, 226, 32, 79, 61, 28, 220, 186, 113, 56, 25, 114, 29, 7, 159, 210, 5, 101, 250, 226, 223, 167, 33, 31, 125, 83, 89, 245, 12, 26, 198, 193, 19, 227, 62, 212, 140, 148, 202, 62, 26, 141, 232, 190, 210, 250, 112, 102, 180, 88, 182, 63, 143, 117, 225, 130, 193, 0, 210, 234, 194, 230, 173, 87, 113, 87, 95, 58, 201, 202, 69, 112, 155, 188, 199, 82, 191, 99, 77, 131, 227, 242, 65, 180, 99, 134, 50, 158, 2, 187, 19, 139, 194, 223, 123, 52, 223, 145, 76, 122, 154, 53, 157, 233, 18, 70, 164, 210, 173, 142, 195, 94, 178, 79, 6, 207, 164, 249, 243, 235, 36, 193, 56, 13, 183, 185, 174, 187, 64, 147, 114, 45, 23, 211, 232, 174, 85, 88, 161, 29, 226, 96, 240, 115, 248, 181, 217, 114, 217, 171, 143, 230, 157, 96, 86, 217, 252, 143, 49, 89, 152, 232, 145, 26, 28, 149, 70, 223, 103, 88, 122, 76, 88, 214, 9, 196, 209, 197, 20, 2, 224, 59, 226, 149, 4, 140, 144, 182, 124, 14, 177, 198, 38, 72, 133, 249, 206, 213, 87, 99, 110, 249, 11, 128, 49, 66, 141, 199, 87, 146, 8, 55, 122, 47, 120, 92, 6, 252, 22, 143, 5, 144, 99, 88, 167, 179, 38, 16, 80, 15, 245, 251, 109, 177, 147, 230, 47, 97, 99, 193, 254, 200, 94, 234, 5, 121, 75, 246, 207, 40, 242, 209, 191, 199, 94, 222, 150, 178, 217, 125, 27, 125, 118, 72, 156, 13, 202, 200, 155, 145, 45, 165, 152, 84, 145, 60, 49, 120, 82, 46, 104, 47, 64, 39, 28, 36, 59, 165, 83, 181, 80, 124, 136, 230, 166, 128, 82, 22, 77, 248, 174, 130, 91, 62, 251, 77, 56, 107, 54, 33, 172, 29, 95, 170, 21, 235, 61, 79, 43, 120, 209, 36, 117, 229, 65, 176, 28, 178, 173, 16, 234, 27, 43, 90, 127, 229, 212, 1, 44, 207, 235, 208, 111, 49, 9, 50, 28, 221, 208, 186, 66, 3, 235, 213, 38, 138, 13, 117, 253, 75, 48, 133, 189, 38, 55, 145, 242, 5, 43, 47, 23, 59, 34, 91, 141, 1, 181, 67, 220, 14, 147, 63, 225, 243, 205, 230, 47, 85, 51, 244, 216, 94, 4, 225, 159, 101, 19, 72, 250, 120, 63, 23, 177, 161, 163, 7, 224, 143, 72, 37, 213, 202, 153, 238, 183, 241, 164, 202, 88, 80, 97, 156, 210, 54, 87, 184, 55, 65, 188, 39, 237, 108, 73, 100, 206, 124, 141, 64, 79, 198, 6, 160, 13, 13, 31, 13, 197, 21, 41, 174, 193, 73, 240, 169, 22, 163, 68, 123, 73, 235, 208, 158, 211, 32, 56, 104, 129, 53 })
            };
        }

        [Theory]
        [MemberData(nameof(BinaryPlus_TestData))]
        public static void BinaryPlus(BigInteger bigInteger1, BigInteger bigInteger2, BigInteger expected)
        {
            Assert.Equal(expected, bigInteger1 + bigInteger2);
            Assert.Equal(expected, bigInteger2 + bigInteger1);

            Assert.Equal(expected, BigInteger.Add(bigInteger1, bigInteger2));
            Assert.Equal(expected, BigInteger.Add(bigInteger2, bigInteger1));
        }
    }
}
