Migration from version 2 to version 3
--------------------------------------------------------------------------------

* [Moving serialization format mixins](#moving-serialization-format-mixins)
* [Removing `use_bytes` parameter](#removing-use_bytes-parameter)
* [Removing `use_enum` parameter](#removing-use_enum-parameter)
* [Removing `use_datetime` parameter](#removing-use_datetime-parameter)
* [Changing `from_json`, `from_msgpack`, `from_yaml` signature](#changing-from_json-from_msgpack-from_yaml-signature)
* [Changing `to_json`, `to_msgpack`, `to_yaml` signature](#changing-to_json-to_msgpack-to_yaml-signature)

### Moving serialization format mixins

You might need to alter your imports if you've used the following mixins:
* `DataClassJSONMixin`
* `DataClassMessagePackMixin`
* `DataClassYAMLMixin`

Tne new imports will look like this:

```python
from mashumaro.mixins.json import DataClassJSONMixin
from mashumaro.mixins.msgpack import DataClassMessagePackMixin
from mashumaro.mixins.yaml import DataClassYAMLMixin
```

### Removing `use_bytes` parameter

Parameter `use_bytes` was removed from `from_dict` / `to_dict` methods.
If you've used it to pass bytes or bytearray values as is, you can do the same
with [dialect](https://github.com/Fatal1ty/mashumaro#dialects) and
[pass_through](https://github.com/Fatal1ty/mashumaro/tree/master#passing-field-values-as-is)
features:

```python
from dataclasses import dataclass

from mashumaro import DataClassDictMixin, pass_through
from mashumaro.config import BaseConfig, ADD_DIALECT_SUPPORT
from mashumaro.dialect import Dialect

class BytesDialect(Dialect):
    serialization_strategy = {
        bytes: pass_through,
        bytearray: pass_through,
    }

@dataclass
class A(DataClassDictMixin):
    bytes: bytes
    bytearray: bytearray

    class Config(BaseConfig):
        code_generation_options = [ADD_DIALECT_SUPPORT]

obj = A(b"\x00", bytearray(b"\x00"))
dct = {"bytes": b"\x00", "bytearray": bytearray(b"\x00")}

assert A.from_dict(dct, dialect=BytesDialect) == obj
assert obj.to_dict(dialect=BytesDialect) == dct
```

### Removing `use_enum` parameter

Parameter `use_enum` was removed from `from_dict` / `to_dict` methods.
If you've used it to pass enum values as is, you can do the same
with [dialect](https://github.com/Fatal1ty/mashumaro#dialects) and
[pass_through](https://github.com/Fatal1ty/mashumaro/tree/master#passing-field-values-as-is)
features:

```python
from dataclasses import dataclass
from enum import Enum

from mashumaro import DataClassDictMixin, pass_through
from mashumaro.config import BaseConfig, ADD_DIALECT_SUPPORT
from mashumaro.dialect import Dialect

class MyEnum(Enum):
    a = 1
    b = 2

class EnumDialect(Dialect):
    serialization_strategy = {
        MyEnum: pass_through,
    }

@dataclass
class A(DataClassDictMixin):
    my_enum: MyEnum

    class Config(BaseConfig):
        code_generation_options = [ADD_DIALECT_SUPPORT]

obj = A(MyEnum.a)
dct = {"my_enum": MyEnum.a}

assert A.from_dict(dct, dialect=EnumDialect) == obj
assert obj.to_dict(dialect=EnumDialect) == dct
```

### Removing `use_datetime` parameter

Parameter `use_datetime` was removed from `from_dict` / `to_dict` methods.
If you've used it to pass datetime, date and time values as is, you can do
the same with [dialect](https://github.com/Fatal1ty/mashumaro#dialects) and
[pass_through](https://github.com/Fatal1ty/mashumaro/tree/master#passing-field-values-as-is)
features:

```python
from dataclasses import dataclass
from datetime import date, datetime, time

from mashumaro import DataClassDictMixin, pass_through
from mashumaro.config import BaseConfig, ADD_DIALECT_SUPPORT
from mashumaro.dialect import Dialect

class DatetimeDialect(Dialect):
    serialization_strategy = {
        date: pass_through,
        datetime: pass_through,
        time: pass_through,
    }

@dataclass
class A(DataClassDictMixin):
    datetime: datetime
    date: date
    time: time

    class Config(BaseConfig):
        code_generation_options = [ADD_DIALECT_SUPPORT]

obj = A(
    datetime=datetime(2022, 2, 9, 12, 0),
    date=date(2022, 2, 9),
    time=time(12, 0),
)
dct = {
    "datetime": datetime(2022, 2, 9, 12, 0),
    "date": date(2022, 2, 9),
    "time": time(12, 0),
}

assert A.from_dict(dct, dialect=DatetimeDialect) == obj
assert obj.to_dict(dialect=DatetimeDialect) == dct
```

### Changing `from_json`, `from_msgpack`, `from_yaml` signature

In version 2 methods `from_json`, `from_msgpack`, `from_yaml` had the following
signature:
```python
@classmethod
def from_*(  # where * is json, msgpack, yaml
    cls,
    data: EncodedData,
    decoder: Decoder = ...,
    dict_params: Mapping = ...,
    **decoder_kwargs,
)
```

In version 3 these methods have a slightly different signature:
```python
@classmethod
def from_*(  # where * is json, msgpack, yaml
    cls,
    data: EncodedData,
    decoder: Decoder = ...,
    **from_dict_kwargs,
)
```

As you can see, the `dict_params` positional argument was removed in order
to pass keyword arguments to underlying `from_dict` method. Decoder parameters
were removed because they can be easily passed to decoder using
a lambda function, a partial object or something else:

```python
A.from_json(
    data,
    decoder=lambda data: json.loads(data, parse_float=decimal.Decimal),
)
```

### Changing `to_json`, `to_msgpack`, `to_yaml` signature

In version 2 methods `to_json`, `to_msgpack`, `to_yaml` had the following
signature:
```python
def to_*(  # where * is json, msgpack, yaml
    self,
    encoder: Encoder = ...
    dict_params: Mapping = ...,
    **encoder_kwargs,
)
```

In version 3 these methods have a slightly different signature:
```python
def to_*(  # where * is json, msgpack, yaml
    self,
    encoder: Encoder = ...,
    **to_dict_kwargs,
)
```

As you can see, the `dict_params` positional argument was removed in order
to pass keyword arguments to underlying `to_dict` method. Encoder parameters
were removed because they can be easily passed to encoder using
a lambda function, a partial object or something else:

```python
dataclass_obj.to_json(
    encoder=lambda data: json.dumps(data, ensure_ascii=False),
)
```
