% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/VaR.R
\name{VaR}
\alias{VaR}
\alias{VaR_qf}
\alias{VaR_cdf}
\alias{VaR.default}
\alias{VaR.numeric}
\alias{VaR.matrix}
\title{Compute Value-at-Risk (VaR)}
\usage{
VaR(dist, p_loss = 0.05, ...)

VaR_qf(
  dist,
  p_loss = 0.05,
  ...,
  intercept = 0,
  slope = 1,
  tol = .Machine$double.eps^0.5,
  x
)

VaR_cdf(
  dist,
  p_loss = 0.05,
  ...,
  intercept = 0,
  slope = 1,
  tol = .Machine$double.eps^0.5,
  x
)

\method{VaR}{default}(
  dist,
  p_loss = 0.05,
  dist.type = "qf",
  ...,
  intercept = 0,
  slope = 1,
  tol = .Machine$double.eps^0.5,
  transf = FALSE,
  x
)

\method{VaR}{numeric}(dist, p_loss = 0.05, ..., intercept = 0, slope = 1, transf = FALSE, x)

\method{VaR}{matrix}(dist, p_loss = 0.05, ..., intercept = 0, slope = 1, transf = FALSE, x)
}
\arguments{
\item{dist}{specifies the distribution whose ES is computed, usually a function or a name
of a function computing quantiles, cdf, pdf, or a random number generator. Can also be
a numeric vector or matrix, representing data, see section \sQuote{Details}.}

\item{p_loss}{level, default is 0.05.}

\item{...}{passed on to \code{dist}.}

\item{intercept, slope}{compute VaR for the linear transformation \code{intercept +
slope*X}, where \code{X} has distribution specified by \code{dist}, see Details.}

\item{tol}{tollerance}

\item{x}{deprecated and will soon be removed. \code{x} was renamed to \code{p_loss},
please use the latter.}

\item{dist.type}{a character string specifying what is computed by \code{dist}, such as
"qf" or "cdf".}

\item{transf}{use only if \code{"dist"} represents log-returns. If \code{TRUE}, give the
result for the returns (not log-returns), see section \sQuote{Details}.}
}
\description{
\code{VaR} is a generic function for computation of Value-at-Risk. The default method
    computes VaR corresponding to a distribution, usually fitted or implied by a fitted
    model.  There are also methods for data (numeric or matrix).
}
\details{
\code{VaR} is S3 generic. The meaning of the parameters for the methods defined in
    \pkg{cvar} and described here are the same as in \code{\link{ES}}, including the
    recycling rules.

    \code{VaR_qf} and \code{VaR_cdf} are streamlined, non-generic, variants for the common
    case when the \code{"..."} parameters are scalar. The parameters \code{p_loss},
    \code{intercept}, and \code{slope} can be vectors, as for \code{VaR}.

    Argument \code{dist} can also be a numeric vector. In that case the ES is computed,
    effectively, for the empirical cumulative distribution function (ecdf) of the
    vector. The ecdf is not created explicitly and the \code{\link[stats]{quantile}}
    function is used instead for the computation of VaR. Arguments in \code{"..."} are
    passed eventually to \code{quantile()} and can be used, for example, to select a
    non-defult method for the computation of quantiles.

    If \code{dist} is of class \code{"matrix"}, then the numeric method is applied to each
    column.

    In practice, we may need to compute VaR associated with data. The distribution comes
    from fitting a model. In the simplest case, we fit a distribution to the data,
    assuming that the sample is i.i.d. For example, a normal distribution \eqn{N(\mu,
    \sigma^2)} can be fitted using the sample mean and sample variance as estimates of the
    unknown parameters \eqn{\mu} and \eqn{\sigma^2}, see section \sQuote{Examples}. For
    other common distributions there are specialised functions to fit their parameters and
    if not, general optimisation routines can be used. More soffisticated models may be
    used, even time series models such as GARCH and mixture autoregressive models.
    
    When argument \code{dist} represents log-returns, VaR is for the log-returns. Use
    \code{transf = TRUE} to return its value for the returns.
}
\note{
We use the traditional definition of VaR as the negated lower quantile. For example,
    if \eqn{X} are returns on an asset, VAR\eqn{{}_\alpha}{_a} = \eqn{-q_\alpha}{-q_a},
    where \eqn{q_\alpha}{-q_a} is the lower \eqn{\alpha}{a} quantile of \eqn{X}.
    Equivalently, VAR\eqn{{}_\alpha}{_a} is equal to the lower \eqn{1-\alpha}{1-a}
    quantile of \eqn{-X}.
}
\examples{
cvar::VaR(qnorm, c(0.01, 0.05), dist.type = "qf")

## the following examples use these values, obtained by fitting a normal distribution to
## some data:
muA <- 0.006408553
sigma2A <- 0.0004018977

## with quantile function, giving the parameters directly in the call:
res1 <- cvar::VaR(qnorm, 0.05, mean = muA, sd = sqrt(sigma2A))
res2 <- cvar::VaR(qnorm, 0.05, intercept = muA, slope = sqrt(sigma2A))
abs((res2 - res1)) # 0, intercept/slope equivalent to mean/sd

## with quantile function, which already knows the parameters:
my_qnorm <- function(p) qnorm(p, mean = muA, sd = sqrt(sigma2A))
res1_alt <- cvar::VaR(my_qnorm, 0.05)
abs((res1_alt - res1))

## with cdf the precision depends on solving an equation
res1a <- cvar::VaR(pnorm, 0.05, dist.type = "cdf", mean = muA, sd = sqrt(sigma2A))
res2a <- cvar::VaR(pnorm, 0.05, dist.type = "cdf", intercept = muA, slope = sqrt(sigma2A))
abs((res1a - res2)) # 3.287939e-09
abs((res2a - res2)) # 5.331195e-11, intercept/slope better numerically

## as above, but increase the precision, this is probably excessive
res1b <- cvar::VaR(pnorm, 0.05, dist.type = "cdf",
                   mean = muA, sd = sqrt(sigma2A), tol = .Machine$double.eps^0.75)
res2b <- cvar::VaR(pnorm, 0.05, dist.type = "cdf",
                   intercept = muA, slope = sqrt(sigma2A), tol = .Machine$double.eps^0.75)
abs((res1b - res2)) # 6.938894e-18 # both within machine precision
abs((res2b - res2)) # 1.040834e-16

## relative precision is also good
abs((res1b - res2)/res2) # 2.6119e-16 # both within machine precision
abs((res2b - res2)/res2) # 3.91785e-15


## an extended example with vector args, if "PerformanceAnalytics" is present
if (requireNamespace("PerformanceAnalytics", quietly = TRUE)) withAutoprint({
    data(edhec, package = "PerformanceAnalytics")
    mu <- apply(edhec, 2, mean)
    sigma2 <- apply(edhec, 2, var)
    musigma2 <- cbind(mu, sigma2)

    ## compute in 2 ways with cvar::VaR
    vAz1 <- cvar::VaR(qnorm, 0.05, mean = mu, sd = sqrt(sigma2))
    vAz2 <- cvar::VaR(qnorm, 0.05, intercept = mu, slope = sqrt(sigma2))

    vAz1a <- cvar::VaR(pnorm, 0.05, dist.type = "cdf",
                       mean = mu, sd = sqrt(sigma2))
    vAz2a <- cvar::VaR(pnorm, 0.05, dist.type = "cdf",
                       intercept = mu, slope = sqrt(sigma2))

    vAz1b <- cvar::VaR(pnorm, 0.05, dist.type = "cdf",
                   mean = mu, sd = sqrt(sigma2),
                   tol = .Machine$double.eps^0.75)
    vAz2b <- cvar::VaR(pnorm, 0.05, dist.type = "cdf",
                   intercept = mu, slope = sqrt(sigma2),
                   tol = .Machine$double.eps^0.75)

    ## analogous calc. with PerformanceAnalytics::VaR
    vPA <- apply(musigma2, 1, function(x)
        PerformanceAnalytics::VaR(p = .95, method = "gaussian", invert = FALSE,
                                  mu = x[1], sigma = x[2], weights = 1))
    ## the results are numerically the same
    max(abs((vPA - vAz1))) # 5.551115e-17
    max(abs((vPA - vAz2))) #   ""

    max(abs((vPA - vAz1a))) # 3.287941e-09
    max(abs((vPA - vAz2a))) #  1.465251e-10, intercept/slope better

    max(abs((vPA - vAz1b))) # 4.374869e-13
    max(abs((vPA - vAz2b))) # 3.330669e-16
})

}
\seealso{
\code{\link{ES}} for ES,

    \code{\link[=predict.garch1c1]{predict}} for examples with fitted models
}
\concept{Value at Risk}
