/*
    SPDX-FileCopyrightText: 2003, 2007-2009 Friedrich W. H. Kossebau <kossebau@kde.org>

    SPDX-License-Identifier: LGPL-2.1-only OR LGPL-3.0-only OR LicenseRef-KDE-Accepted-LGPL
*/

#include "bytearraymodel.hpp"

// lib
#include "bytearraymodel_p.hpp"
#include "arraychangemetricslist.hpp"

namespace Okteta {

ByteArrayModel::ByteArrayModel(const Byte* data, int size, QObject* parent)
    : AbstractByteArrayModel(std::make_unique<ByteArrayModelPrivate>(this, data, size), parent)
{
}

ByteArrayModel::ByteArrayModel(Byte* data, int size, int capacity, QObject* parent)
    : AbstractByteArrayModel(std::make_unique<ByteArrayModelPrivate>(this, data, size, capacity), parent)
{
}

ByteArrayModel::ByteArrayModel(std::unique_ptr<Okteta::Byte[]>&& data, int size, int capacity,
                               QObject* parent)
    : AbstractByteArrayModel(std::make_unique<ByteArrayModelPrivate>(this, std::move(data), size, capacity), parent)
{
}

ByteArrayModel::ByteArrayModel(int size, Byte fillByte, QObject* parent)
    : AbstractByteArrayModel(std::make_unique<ByteArrayModelPrivate>(this, size, fillByte), parent)
{
}

ByteArrayModel::ByteArrayModel(int size, QObject* parent)
    : AbstractByteArrayModel(std::make_unique<ByteArrayModelPrivate>(this, size), parent)
{
}

ByteArrayModel::~ByteArrayModel() = default;

Byte ByteArrayModel::byte(Address offset) const
{
    Q_D(const ByteArrayModel);

    return d->byte(offset);
}

Size ByteArrayModel::size() const
{
    Q_D(const ByteArrayModel);

    return d->size();
}

bool ByteArrayModel::isReadOnly() const
{
    Q_D(const ByteArrayModel);

    return d->isReadOnly();
}

bool ByteArrayModel::isModified() const
{
    Q_D(const ByteArrayModel);

    return d->isModified();
}

void ByteArrayModel::setReadOnly(bool readOnly)
{
    Q_D(ByteArrayModel);

    d->setReadOnly(readOnly);
}

void ByteArrayModel::setModified(bool modified)
{
    Q_D(ByteArrayModel);

    d->setModified(modified);
}

void ByteArrayModel::setData(const Byte* data, int size)
{
    Q_D(ByteArrayModel);

    d->setData(data, size);
}

void ByteArrayModel::setData(Byte* data, int size, int capacity)
{
    Q_D(ByteArrayModel);

    d->setData(data, size, capacity);
}

void ByteArrayModel::setData(std::unique_ptr<Okteta::Byte[]>&& data, int size, int capacity)
{
    Q_D(ByteArrayModel);

    d->setData(std::move(data), size, capacity);
}

std::unique_ptr<Okteta::Byte[]> ByteArrayModel::releaseData()
{
    Q_D(ByteArrayModel);

    return d->releaseData();
}

void ByteArrayModel::setMaxSize(int maxSize)
{
    Q_D(ByteArrayModel);

    d->setMaxSize(maxSize);
}

int ByteArrayModel::maxSize() const
{
    Q_D(const ByteArrayModel);

    return d->maxSize();
}

int ByteArrayModel::capacity() const
{
    Q_D(const ByteArrayModel);

    return d->capacity();
}

void ByteArrayModel::signalContentsChanged(int start, int end)
{
    const int length = end - start + 1;
    Q_EMIT contentsChanged(ArrayChangeMetricsList::oneReplacement(start, length, length));
}

void ByteArrayModel::setByte(Address offset, Byte byte)
{
    Q_D(ByteArrayModel);

    d->setByte(offset, byte);
}

Size ByteArrayModel::insertBytes(Address offset, const Byte* insertData, int insertLength)
{
    Q_D(ByteArrayModel);

    return d->insertBytes(offset, insertData, insertLength);
}

Size ByteArrayModel::removeBytes(const AddressRange& removeRange)
{
    Q_D(ByteArrayModel);

    return d->removeBytes(removeRange);
}

Size ByteArrayModel::replaceBytes(const AddressRange& removeRange, const Byte* insertData, int insertLength)
{
    Q_D(ByteArrayModel);

    return d->replaceBytes(removeRange, insertData, insertLength);
}

bool ByteArrayModel::swapBytes(Address firstStart, const AddressRange& secondRange)
{
    Q_D(ByteArrayModel);

    return d->swapBytes(firstStart, secondRange);
}

Size ByteArrayModel::fill(Byte fillByte, Address offset, Size fillLength)
{
    Q_D(ByteArrayModel);

    return d->fill(fillByte, offset, fillLength);
}

void ByteArrayModel::addBookmarks(const QList<Okteta::Bookmark>& bookmarks)
{
    Q_D(ByteArrayModel);

    d->addBookmarks(bookmarks);
}

void ByteArrayModel::removeBookmarks(const QList<Okteta::Bookmark>& bookmarks)
{
    Q_D(ByteArrayModel);

    d->removeBookmarks(bookmarks);
}

void ByteArrayModel::removeAllBookmarks()
{
    Q_D(ByteArrayModel);

    d->removeAllBookmarks();
}

void ByteArrayModel::setBookmark(unsigned int index, const Okteta::Bookmark& bookmark)
{
    Q_D(ByteArrayModel);

    d->setBookmark(index, bookmark);
}

Okteta::BookmarksConstIterator ByteArrayModel::createBookmarksConstIterator() const
{
    Q_D(const ByteArrayModel);

    return d->createBookmarksConstIterator();
}

const Okteta::Bookmark& ByteArrayModel::bookmarkFor(Address offset) const
{
    Q_D(const ByteArrayModel);

    return d->bookmarkFor(offset);
}

const Okteta::Bookmark& ByteArrayModel::bookmarkAt(unsigned int index) const
{
    Q_D(const ByteArrayModel);

    return d->bookmarkAt(index);
}

bool ByteArrayModel::containsBookmarkFor(Address offset) const
{
    Q_D(const ByteArrayModel);

    return d->containsBookmarkFor(offset);
}

unsigned int ByteArrayModel::bookmarksCount() const
{
    Q_D(const ByteArrayModel);

    return d->bookmarksCount();
}

}

#include "moc_bytearraymodel.cpp"
