/*
    SPDX-FileCopyrightText: 2010, 2011, 2012 Alex Richardson <alex.richardson@gmx.de>

    SPDX-License-Identifier: LGPL-2.1-only OR LGPL-3.0-only OR LicenseRef-KDE-Accepted-LGPL
*/

#include "scripthandler.hpp"

// lib
#include "scriptengineinitializer.hpp"
#include "scriptlogger.hpp"
#include <defaultscriptclass.hpp>
#include <datainformation.hpp>
#include <topleveldatainformation.hpp>
#include <arraydatainformation.hpp>
#include <parserutils.hpp>
// Qt
#include <QStringList>
#include <QScriptValue>
#include <QScriptEngine>
// Std
#include <utility>

ScriptHandler::ScriptHandler(std::unique_ptr<QScriptEngine>&& engine, TopLevelDataInformation* topLevel)
    : mEngine(std::move(engine))
    , mTopLevel(topLevel)
    , mHandlerInfo(mEngine.get(), topLevel->logger())
{
}

ScriptHandler::~ScriptHandler() = default;

QScriptEngine* ScriptHandler::engine() const
{
    return mEngine.get();
}

ScriptHandlerInfo* ScriptHandler::handlerInfo()
{
    return &mHandlerInfo;
}

void ScriptHandler::validateData(DataInformation* data)
{
    Q_CHECK_PTR(data);

    if (data->hasBeenValidated()) {
        return;
    }
    // first validate the children
    for (uint i = 0; i < data->childCount(); ++i) {
        validateData(data->childAt(i));
    }

    // check if has a validation function:
    QScriptValue validationFunc = data->validationFunc();
    if (validationFunc.isValid()) {
        QScriptValue result = callFunction(validationFunc, data, ScriptHandlerInfo::Mode::Validating);
        if (result.isError()) {
            mTopLevel->logger()->error(data) << "Error occurred while validating element: "
                                             << result.toString();
            data->setValidationError(QStringLiteral("Error occurred in validation: ")
                                     + result.toString());
        } else if (mEngine->hasUncaughtException()) {
            mTopLevel->logger()->error(data) << "Error occurred while validating element:"
                                             << result.toString() << "\nBacktrace:" << mEngine->uncaughtExceptionBacktrace();
            data->setValidationError(QStringLiteral("Error occurred in validation: ")
                                     + result.toString());
            mEngine->clearExceptions();
        }
        if (result.isBool()) {
            data->mValidationSuccessful = result.toBool();
        }
        if (result.isString()) {
            // error string
            QString str = result.toString();
            if (!str.isEmpty()) {
                data->setValidationError(str);
            }
        }
        data->mHasBeenValidated = true;
    }
}

void ScriptHandler::updateDataInformation(DataInformation* data)
{
    Q_CHECK_PTR(data);
    // check if has an update function:
    Q_ASSERT(!data->hasBeenUpdated());
    QScriptValue updateFunc = data->updateFunc();
    data->mHasBeenUpdated = true;
    if (updateFunc.isValid()) {
        QString context = data->fullObjectPath(); // we mustn't use data after updateFunc.call(), save context
        QScriptValue result = callFunction(updateFunc, data, ScriptHandlerInfo::Mode::Updating);
        if (result.isError()) {
            mTopLevel->logger()->error(context) << "Error occurred while updating element: "
                                                << result.toString();
        }
        if (mEngine->hasUncaughtException()) {
            mTopLevel->logger()->error(context) << "Error occurred while updating element:"
                                                << result.toString() << "\nBacktrace:" << mEngine->uncaughtExceptionBacktrace();
            mEngine->clearExceptions();
        }
    }
}

void ScriptHandler::updateLength(ArrayDataInformation* array)
{
    QScriptValue lengthFunc = array->lengthFunction();
    if (lengthFunc.isValid()) {
        Q_ASSERT(lengthFunc.isFunction());

        QScriptValue result = callFunction(lengthFunc, array, ScriptHandlerInfo::Mode::DeterminingLength);
        if (mEngine->hasUncaughtException()) {
            mTopLevel->logger()->error(array) << "Error occurred while calculating length:"
                                              << result.toString() << "\nBacktrace:" << mEngine->uncaughtExceptionBacktrace();
            mEngine->clearExceptions();
        }
        ParsedNumber<uint> value = ParserUtils::uintFromScriptValue(result);
        if (value.isValid) {
            std::ignore = array->setArrayLength(value.value);
        } else {
            array->logError() << "Length function did not return a valid number! Result was: " << result.toString();
        }
    }
}

QString ScriptHandler::customToString(const DataInformation* data, const QScriptValue& func)
{
    Q_ASSERT(func.isValid());
    Q_ASSERT(func.isFunction());
    Q_ASSERT(data->wasAbleToRead()); // this should never be called if EOF was reached
    // it is effectively const, since nothing may be modified while mode is CustomToString
    // const_cast is okay in this case
    QScriptValue result = callFunction(func, const_cast<DataInformation*>(data), ScriptHandlerInfo::Mode::CustomToString);
    if (result.isError()) {
        data->logError() << "toStringFunc caused an error:" << result.toString();
    }
    return result.toString();
}

QScriptValue ScriptHandler::callFunction(QScriptValue func, DataInformation* data,
                                         ScriptHandlerInfo::Mode mode)
{
    Q_ASSERT(func.isFunction());
    // value exists, we assume it has been checked to be a function
    QScriptValue thisObject = data->toScriptValue(mEngine.get(), &mHandlerInfo);
    QScriptValue mainStruct = data->mainStructure()->toScriptValue(mEngine.get(), &mHandlerInfo);
    const QScriptValueList args { mainStruct };
    // ensure we get the right properties
    mHandlerInfo.setMode(mode);
    QScriptValue result = func.call(thisObject, args);
    mHandlerInfo.setMode(ScriptHandlerInfo::Mode::None);
    return result;
}
