% Ultralytics 🚀 AGPL-3.0 License - https://ultralytics.com/license

function [name, color, mass, charge] = fcnpid2name(pid)
%returns two strings for one particle, or two cells for multiple particles
%particle colors! sprintf('%.3f %.3f %.3f\n',rand(254,3))
emass   = 0.510998910; %MeV
pmass   = 938.272046; %MeV
nmass   = 939.565378; %MeV
umass   = 931.494061;% 'unified atomic mass unit' http://en.wikipedia.org/wiki/Unified_atomic_mass_unit
X       = fullGEANTdata;  
pidvec  = getpidvec;


n = numel(pid); nu=n; upid=pid;
if n>1 %lots of pids, do only the unique ones then assign to common indices
    [upid, ~, IC] = fcnunique(pid);  nu = numel(upid);
end

x=cell(n,1);  name=x; mass=x; charge=x; color=x;
for i = 1:nu
    p = upid(i);
    j = find(abs(p)==pidvec,1,'first');
    if ~isempty(j) %we found the pid
        x = X(j,:);
        if p>-1
            name1       = x{2}; %name
            charge1     = x{4};
            color1      = [x{7} x{8} x{9}];
        else
            name1       = x{6}; %anti-name
            charge1     =-x{4}; 
            color1      = 1 - [x{7} x{8} x{9}];
        end
        mass1            = x{3}*1000; %MeV/c^2
    elseif p>1E9 %pid is at least 10 digits long, probably a large atom or isotope
        str             = sprintf('%d',p);
        isotopenumber   = eval(str(end-3:end-1));
        atomicnumber    = eval(str(end-7:end-4));
        symbol          = fcnatomicnumber2symbol(atomicnumber);
        name1           = sprintf('%s%d',symbol,isotopenumber);
        mass1           = (pmass+emass)*atomicnumber + (isotopenumber-atomicnumber)*nmass;
        charge1         = 0;
        %s = RandStream('mt19937ar','Seed',sum(name1));
        %color1 = rand(s,1,3);
        if numel(name1)==3
            color1 = double([name1(1) isotopenumber.^2 atomicnumber.^2]);  color1=color1/max(color1)*.9;
        elseif numel(name1)==2
            color1 = double([name1(1) isotopenumber.^2.5 atomicnumber.^2]);  color1=color1/max(color1)*.9;
        else
            color1 = [0 0 0];
        end
    else
       fprintf('WARNING: unknown GEANT particle ID %d detected. Unable to match a name to this PID.\n',p)
        name1           = sprintf('%d pid',p);
        mass1           = 0;
        charge1         = 0;
        color1          = [.7 .7 .7];
    end
    
    if nu<n %only do the unique particles
        j = find(IC==i);
        name(j)         = {name1};
        mass(j)         = {mass1};
        color(j)        = {color1};
        charge(j)       = {charge1};
    else
        name{i}         = name1;
        mass{i}         = mass1;
        color{i}        = color1;
        charge{i}       = charge1;
    end
end

end

function X = fullGEANTdata()
%save -v6 GEANTparticles.mat X
%http://geant4.web.cern.ch/geant4/UserDocumentation/UsersGuides/ForApplicationDeveloper/html/ch05s03.html
%[1.Code (pid)	2.Name	3.Mass	4.Charge	5.Life Time	6.Anti-Particle	7.Color (red)	8.Color (green)	9.Color (blue)]
X ={ ...
    [         5]    'b_quark'           [      4.19]    [-0.333]    [          -1]    'anti_b_quark'           [0.097]    [0.908]    [0.115]
    [         4]    'c_quark'           [      1.29]    [ 0.667]    [          -1]    'anti_c_quark'           [0.076]    [0.987]    [0.626]
    [         1]    'd_quark'           [    0.0049]    [-0.333]    [          -1]    'anti_d_quark'           [0.758]    [ 0.38]    [0.718]
    [      1103]    'dd1_diquark'       [    0.0096]    [-0.667]    [          -1]    'anti_dd1_diquark'       [0.115]    [0.608]    [0.118]
    [        21]    'gluon'             [         0]    [     0]    [          -1]                       []    [0.861]    [0.864]    [0.632]
    [         3]    's_quark'           [       0.1]    [-0.333]    [          -1]    'anti_s_quark'           [0.153]    [ 0.69]    [0.408]
    [      3101]    'sd0_diquark'       [     0.108]    [-0.667]    [          -1]    'anti_sd0_diquark'       [0.638]    [0.146]    [0.564]
    [      3103]    'sd1_diquark'       [    0.1088]    [-0.667]    [          -1]    'anti_sd1_diquark'       [0.817]    [0.655]    [ 0.44]
    [      3303]    'ss1_diquark'       [     0.208]    [-0.667]    [          -1]    'anti_ss1_diquark'       [0.177]    [0.188]    [0.152]
    [      3201]    'su0_diquark'       [    0.1064]    [ 0.333]    [          -1]    'anti_su0_diquark'       [0.713]    [0.127]    [0.616]
    [      3203]    'su1_diquark'       [    0.1064]    [ 0.333]    [          -1]    'anti_su1_diquark'       [0.752]    [0.316]    [0.767]
    [         6]    't_quark'           [     172.9]    [ 0.667]    [          -1]    'anti_t_quark'           [ 0.32]    [0.936]    [0.357]
    [         2]    'u_quark'           [    0.0024]    [ 0.667]    [          -1]    'anti_u_quark'           [0.715]    [0.466]    [ 0.69]
    [      2101]    'ud0_diquark'       [    0.0073]    [ 0.333]    [          -1]    'anti_ud0_diquark'       [0.892]    [0.093]    [0.329]
    [      2103]    'ud1_diquark'       [    0.0072]    [ 0.333]    [          -1]    'anti_ud1_diquark'       [0.402]    [0.814]    [0.482]
    [      2203]    'uu1_diquark'       [    0.0048]    [ 1.333]    [          -1]    'anti_uu1_diquark'       [0.138]    [0.637]    [0.116]
    [        11]    'e-'                [  0.000511]    [    -1]    [          -1]    'e+'                     [    1]    [    0]    [    1]
    [        13]    'mu-'               [  0.105658]    [    -1]    [     2197.03]    'mu+'                    [  0.7]    [  0.7]    [  0.7]
    [        12]    'nu_e'              [         0]    [     0]    [           0]    'anti_nu_e'              [    0]    [    1]    [    0]
    [        14]    'nu_mu'             [         0]    [     0]    [           0]    'anti_nu_mu'             [    1]    [  0.7]    [    1]
    [        16]    'nu_tau'            [         0]    [     0]    [           0]    'anti_nu_tau'            [0.856]    [0.211]    [0.582]
    [        15]    'tau-'              [   1.77682]    [    -1]    [           0]    'tau+'                   [ 0.51]    [0.242]    [0.807]
    [       521]    'B+'                [   5.27917]    [     1]    [       0.002]    'B-'                     [0.577]    [0.448]    [0.731]
    [       511]    'B0'                [    5.2795]    [     0]    [       0.002]    'anti_B0'                [ 0.48]    [0.048]    [0.863]
    [       541]    'Bc+'               [     6.277]    [     1]    [           0]    'Bc-'                    [0.602]    [ 0.77]    [0.093]
    [       531]    'Bs0'               [    5.3663]    [     0]    [       0.001]    'anti_Bs0'               [0.342]    [0.208]    [0.187]
    [       411]    'D+'                [   1.86957]    [     1]    [       0.001]    'D-'                     [ 0.11]    [0.388]    [0.433]
    [       421]    'D0'                [    1.8648]    [     0]    [           0]    'anti_D0'                [0.433]    [0.741]    [0.465]
    [       431]    'Ds+'               [   1.96845]    [     1]    [       0.001]    'Ds-'                    [0.117]    [0.155]    [0.717]
    [       443]    'J/psi'             [  3.096916]    [     0]    [           0]                       []    [0.099]    [0.577]    [0.668]
    [       553]    'Upsiron'           [    9.4603]    [     0]    [           0]                       []    [ 0.71]    [0.017]    [0.239]
    [     10211]    'a0(1450)+'         [     1.474]    [     1]    [           0]    'a0(1450)-'              [0.207]    [0.709]    [0.549]
    [     10111]    'a0(1450)0'         [     1.474]    [     0]    [           0]                       []    [0.398]    [ 0.42]    [0.744]
    [   9000211]    'a0(980)+'          [      0.98]    [     1]    [           0]    'a0(980)-'               [0.988]    [0.029]    [0.876]
    [   9000111]    'a0(980)0'          [      0.98]    [     0]    [           0]                       []    [0.814]    [0.122]    [ 0.87]
    [     20213]    'a1(1260)+'         [      1.23]    [     1]    [           0]    'a1(1260)-'              [0.468]    [0.445]    [ 0.73]
    [     20113]    'a1(1260)0'         [      1.23]    [     0]    [           0]                       []    [ 0.89]    [0.529]    [0.435]
    [       215]    'a2(1320)+'         [    1.3183]    [     1]    [           0]    'a2(1320)-'              [0.074]    [0.053]    [0.293]
    [       115]    'a2(1320)0'         [    1.3183]    [     0]    [           0]                       []    [0.265]    [0.113]    [0.088]
    [     10213]    'b1(1235)+'         [    1.2295]    [     1]    [           0]    'b1(1235)-'              [0.622]    [0.565]    [0.792]
    [     10113]    'b1(1235)0'         [    1.2295]    [     0]    [           0]                       []    [0.509]    [0.244]    [0.824]
    [       221]    'eta'               [  0.547853]    [     0]    [           0]                       []    [0.792]    [0.501]    [0.654]
    [    100221]    'eta(1295)'         [     1.294]    [     0]    [           0]                       []    [0.369]    [0.289]    [0.839]
    [   9020221]    'eta(1405)'         [    1.4098]    [     0]    [           0]                       []    [0.714]    [0.833]    [ 0.88]
    [    100331]    'eta(1475)'         [     1.476]    [     0]    [           0]                       []    [0.921]    [0.046]    [0.532]
    [     10225]    'eta2(1645)'        [     1.617]    [     0]    [           0]                       []    [0.966]    [0.257]    [0.266]
    [     10335]    'eta2(1870)'        [     1.842]    [     0]    [           0]                       []    [0.111]    [0.095]    [0.961]
    [       331]    'eta_prime'         [   0.95778]    [     0]    [           0]                       []    [0.439]    [ 0.96]    [  0.6]
    [       441]    'etac'              [    2.9803]    [     0]    [           0]                       []    [0.153]    [0.045]    [0.478]
    [     10221]    'f0(1370)'          [      1.35]    [     0]    [           0]                       []    [0.501]    [0.333]    [  0.6]
    [   9030221]    'f0(1500)'          [     1.505]    [     0]    [           0]                       []    [0.654]    [0.148]    [0.749]
    [     10331]    'f0(1710)'          [      1.72]    [     0]    [           0]                       []    [0.367]    [0.566]    [0.685]
    [   9000221]    'f0(600)'           [       0.8]    [     0]    [           0]                       []    [0.768]    [0.232]    [0.649]
    [   9010221]    'f0(980)'           [      0.98]    [     0]    [           0]                       []    [0.833]    [0.268]    [0.362]
    [     20223]    'f1(1285)'          [    1.2818]    [     0]    [           0]                       []    [ 0.75]    [0.291]    [0.385]
    [     20333]    'f1(1420)'          [    1.4264]    [     0]    [           0]                       []    [0.121]    [ 0.39]    [0.235]
    [       225]    'f2(1270)'          [    1.2751]    [     0]    [           0]                       []    [0.249]    [0.225]    [0.009]
    [   9030225]    'f2(1810)'          [     1.815]    [     0]    [           0]                       []    [0.351]    [0.696]    [0.931]
    [   9060225]    'f2(2010)'          [      2.01]    [     0]    [           0]                       []    [0.008]    [0.464]    [0.744]
    [       335]    'f2_prime(1525)'    [     1.525]    [     0]    [           0]                       []    [0.512]    [0.537]    [0.977]
    [     10223]    'h1(1170)'          [      1.17]    [     0]    [           0]                       []    [0.995]    [ 0.77]    [0.514]
    [     10333]    'h1(1380)'          [     1.386]    [     0]    [           0]                       []    [0.548]    [0.334]    [0.759]
    [    100321]    'k(1460)+'          [      1.46]    [     1]    [           0]    'k(1460)-'               [0.885]    [ 0.03]    [0.521]
    [    100311]    'k(1460)0'          [      1.46]    [     0]    [           0]    'anti_k(1460)0'          [0.294]    [0.627]    [0.471]
    [     10321]    'k0_star(1430)+'    [      1.43]    [     1]    [           0]    'k0_star(1430)-'         [0.772]    [0.126]    [0.749]
    [     10311]    'k0_star(1430)0'    [      1.43]    [     0]    [           0]    'anti_k0_star(1430)0'    [0.338]    [0.931]    [0.207]
    [     10323]    'k1(1270)+'         [     1.272]    [     1]    [           0]    'k1(1270)-'              [0.617]    [0.948]    [0.655]
    [     10313]    'k1(1270)0'         [     1.272]    [     0]    [           0]    'anti_k1(1270)0'         [0.636]    [0.574]    [0.868]
    [     20323]    'k1(1400)+'         [     1.403]    [     1]    [           0]    'k1(1400)-'              [0.101]    [0.074]    [ 0.25]
    [     20313]    'k1(1400)0'         [     1.403]    [     0]    [           0]    'anti_k1(1400)0'         [ 0.19]    [0.135]    [0.927]
    [     10325]    'k2(1770)+'         [     1.773]    [     1]    [           0]    'k2(1770)-'              [0.555]    [0.329]    [0.752]
    [     10315]    'k2(1770)0'         [     1.773]    [     0]    [           0]    'anti_k2(1770)0'         [0.035]    [0.163]    [0.963]
    [       325]    'k2_star(1430)+'    [    1.4256]    [     1]    [           0]    'k2_star(1430)-'         [0.583]    [  0.6]    [0.012]
    [       315]    'k2_star(1430)0'    [    1.4324]    [     0]    [           0]    'anti_k2_star(1430)0'    [0.777]    [0.626]    [0.855]
    [    100325]    'k2_star(1980)+'    [     1.973]    [     1]    [           0]    'k2_star(1980)-'         [0.792]    [0.229]    [0.984]
    [    100315]    'k2_star(1980)0'    [     1.973]    [     0]    [           0]    'anti_k2_star(1980)0'    [ 0.58]    [0.673]    [0.527]
    [       327]    'k3_star(1780)+'    [     1.776]    [     1]    [           0]    'k3_star(1780)-'         [0.323]    [ 0.58]    [0.958]
    [       317]    'k3_star(1780)0'    [     1.776]    [     0]    [           0]    'anti_k3_star(1780)0'    [0.594]    [0.051]    [0.753]
    [    100323]    'k_star(1410)+'     [     1.414]    [     1]    [           0]    'k_star(1410)-'          [0.601]    [0.026]    [0.893]
    [    100313]    'k_star(1410)0'     [     1.414]    [     0]    [           0]    'anti_k_star(1410)0'     [0.072]    [0.527]    [0.787]
    [     30323]    'k_star(1680)+'     [     1.717]    [     1]    [           0]    'k_star(1680)-'          [0.584]    [  0.6]    [0.398]
    [     30313]    'k_star(1680)0'     [     1.717]    [     0]    [           0]    'anti_k_star(1680)0'     [0.574]    [0.652]    [0.351]
    [       323]    'k_star+'           [   0.89166]    [     1]    [           0]    'k_star-'                [0.897]    [0.784]    [0.057]
    [       313]    'k_star0'           [   0.89594]    [     0]    [           0]    'anti_k_star0'           [0.625]    [0.564]    [0.141]
    [       321]    'kaon+'             [  0.493677]    [     1]    [       12.38]    'kaon-'                  [0.918]    [ 0.66]    [0.941]
    [       311]    'kaon0'             [  0.497614]    [     0]    [           0]    'anti_kaon0'             [0.149]    [0.227]    [0.379]
    [       130]    'kaon0L'            [  0.497614]    [     0]    [       51.16]                       []    [0.551]    [0.995]    [0.666]
    [       310]    'kaon0S'            [  0.497614]    [     0]    [        0.09]                       []    [0.049]    [0.194]    [0.929]
    [       223]    'omega'             [   0.78265]    [     0]    [           0]                       []    [0.151]    [0.654]    [0.626]
    [    100223]    'omega(1420)'       [     1.425]    [     0]    [           0]                       []    [ 0.83]    [0.742]    [0.616]
    [     30223]    'omega(1650)'       [      1.67]    [     0]    [           0]                       []    [0.951]    [0.265]    [0.492]
    [       227]    'omega3(1670)'      [     1.667]    [     0]    [           0]                       []    [ 0.94]    [0.936]    [0.706]
    [       333]    'phi'               [   1.01946]    [     0]    [           0]                       []    [0.557]    [ 0.27]    [0.252]
    [    100333]    'phi(1680)'         [      1.68]    [     0]    [           0]                       []    [0.105]    [  0.2]    [0.304]
    [       337]    'phi3(1850)'        [     1.854]    [     0]    [           0]                       []    [0.066]    [0.046]    [0.279]
    [    100211]    'pi(1300)+'         [       1.3]    [     1]    [           0]    'pi(1300)-'              [0.267]    [0.824]    [0.161]
    [    100111]    'pi(1300)0'         [       1.3]    [     0]    [           0]                       []    [0.167]    [0.837]    [0.433]
    [       211]    'pi+'               [   0.13957]    [     1]    [      26.033]    'pi-'                    [ 0.75]    [0.751]    [0.515]
    [       111]    'pi0'               [  0.134977]    [     0]    [           0]                       []    [0.103]    [0.708]    [0.208]
    [     10215]    'pi2(1670)+'        [    1.6722]    [     1]    [           0]    'pi2(1670)-'             [0.073]    [0.287]    [0.705]
    [     10115]    'pi2(1670)0'        [    1.6722]    [     0]    [           0]                       []    [0.873]    [0.116]    [0.604]
    [    100213]    'rho(1450)+'        [     1.465]    [     1]    [           0]    'rho(1450)-'             [0.275]    [0.329]    [0.148]
    [    100113]    'rho(1450)0'        [     1.465]    [     0]    [           0]                       []    [0.239]    [0.668]    [0.224]
    [     30213]    'rho(1700)+'        [      1.72]    [     1]    [           0]    'rho(1700)-'             [0.026]    [0.608]    [0.676]
    [     30113]    'rho(1700)0'        [      1.72]    [     0]    [           0]                       []    [0.644]    [0.735]    [0.353]
    [       213]    'rho+'              [    0.7758]    [     1]    [           0]    'rho-'                   [0.332]    [0.474]    [ 0.32]
    [       113]    'rho0'              [   0.77549]    [     0]    [           0]                       []    [0.778]    [0.231]    [0.725]
    [       217]    'rho3(1690)+'       [    1.6888]    [     1]    [           0]    'rho3(1690)-'            [0.889]    [0.344]    [0.115]
    [       117]    'rho3(1690)0'       [    1.6888]    [     0]    [           0]                       []    [0.939]    [ 0.72]    [0.093]
    [     12212]    'N(1440)+'          [      1.44]    [     1]    [           0]    'anti_N(1440)+'          [0.602]    [0.189]    [0.617]
    [     12112]    'N(1440)0'          [      1.44]    [     0]    [           0]    'anti_N(1440)0'          [0.049]    [0.685]    [0.968]
    [      2124]    'N(1520)+'          [      1.52]    [     1]    [           0]    'anti_N(1520)+'          [0.347]    [0.275]    [0.809]
    [      1214]    'N(1520)0'          [      1.52]    [     0]    [           0]    'anti_N(1520)0'          [0.551]    [0.761]    [ 0.57]
    [     22212]    'N(1535)+'          [     1.535]    [     1]    [           0]    'anti_N(1535)+'          [0.902]    [ 0.47]    [0.233]
    [     22112]    'N(1535)0'          [     1.535]    [     0]    [           0]    'anti_N(1535)0'          [0.484]    [0.812]    [0.152]
    [     32212]    'N(1650)+'          [     1.655]    [     1]    [           0]    'anti_N(1650)+'          [0.042]    [0.694]    [0.353]
    [     32112]    'N(1650)0'          [     1.655]    [     0]    [           0]    'anti_N(1650)0'          [0.654]    [0.661]    [0.237]
    [      2216]    'N(1675)+'          [     1.675]    [     1]    [           0]    'anti_N(1675)+'          [0.212]    [0.807]    [0.148]
    [      2116]    'N(1675)0'          [     1.675]    [     0]    [           0]    'anti_N(1675)0'          [0.384]    [0.751]    [0.674]
    [     12216]    'N(1680)+'          [     1.685]    [     1]    [           0]    'anti_N(1680)+'          [0.071]    [0.416]    [0.583]
    [     12116]    'N(1680)0'          [     1.685]    [     0]    [           0]    'anti_N(1680)0'          [0.104]    [0.572]    [0.005]
    [     22124]    'N(1700)+'          [       1.7]    [     1]    [           0]    'anti_N(1700)+'          [0.512]    [0.428]    [0.095]
    [     21214]    'N(1700)0'          [       1.7]    [     0]    [           0]    'anti_N(1700)0'          [0.949]    [0.603]    [0.752]
    [     42212]    'N(1710)+'          [      1.71]    [     1]    [           0]    'anti_N(1710)+'          [0.815]    [0.061]    [0.008]
    [     42112]    'N(1710)0'          [      1.71]    [     0]    [           0]    'anti_N(1710)0'          [ 0.34]    [0.673]    [ 0.78]
    [     32124]    'N(1720)+'          [      1.72]    [     1]    [           0]    'anti_N(1720)+'          [0.235]    [0.702]    [0.222]
    [     31214]    'N(1720)0'          [      1.72]    [     0]    [           0]    'anti_N(1720)0'          [0.783]    [0.572]    [ 0.14]
    [     42124]    'N(1900)+'          [       1.9]    [     1]    [           0]    'anti_N(1900)+'          [ 0.21]    [0.816]    [0.119]
    [     41214]    'N(1900)0'          [       1.9]    [     0]    [           0]    'anti_N(1900)0'          [0.611]    [0.834]    [0.264]
    [     12218]    'N(1990)+'          [      1.95]    [     1]    [           0]    'anti_N(1990)+'          [ 0.08]    [0.278]    [0.299]
    [     12118]    'N(1990)0'          [      1.95]    [     0]    [           0]    'anti_N(1990)0'          [0.461]    [0.153]    [0.797]
    [     52214]    'N(2090)+'          [      2.08]    [     1]    [           0]    'anti_N(2090)+'          [0.628]    [0.679]    [0.859]
    [     52114]    'N(2090)0'          [      2.08]    [     0]    [           0]    'anti_N(2090)0'          [0.325]    [0.933]    [0.645]
    [      2128]    'N(2190)+'          [      2.19]    [     1]    [           0]    'anti_N(2190)+'          [0.555]    [0.351]    [0.322]
    [      1218]    'N(2190)0'          [      2.19]    [     0]    [           0]    'anti_N(2190)0'          [0.526]    [0.636]    [0.041]
    [ 100002210]    'N(2220)+'          [      2.25]    [     1]    [           0]    'anti_N(2220)+'          [0.211]    [0.171]    [0.834]
    [ 100002110]    'N(2220)0'          [      2.25]    [     0]    [           0]    'anti_N(2220)0'          [0.614]    [0.559]    [0.398]
    [ 100012210]    'N(2250)+'          [     2.275]    [     1]    [           0]    'anti_N(2250)+'          [0.495]    [0.348]    [0.069]
    [ 100012110]    'N(2250)0'          [     2.275]    [     0]    [           0]    'anti_N(2250)0'          [0.348]    [0.914]    [0.299]
    [     32214]    'delta(1600)+'      [       1.6]    [     1]    [           0]    'anti_delta(1600)+'      [0.149]    [0.559]    [0.273]
    [     32224]    'delta(1600)++'     [       1.6]    [     2]    [           0]    'anti_delta(1600)++'     [0.404]    [0.598]    [0.803]
    [     31114]    'delta(1600)-'      [       1.6]    [    -1]    [           0]    'anti_delta(1600)-'      [  0.1]    [0.528]    [    0]
    [     32114]    'delta(1600)0'      [       1.6]    [     0]    [           0]    'anti_delta(1600)0'      [0.618]    [0.857]    [0.837]
    [      2122]    'delta(1620)+'      [      1.63]    [     1]    [           0]    'anti_delta(1620)+'      [0.003]    [0.953]    [ 0.79]
    [      2222]    'delta(1620)++'     [      1.63]    [     2]    [           0]    'anti_delta(1620)++'     [0.002]    [0.688]    [0.804]
    [      1112]    'delta(1620)-'      [      1.63]    [    -1]    [           0]    'anti_delta(1620)-'      [ 0.75]    [0.045]    [0.401]
    [      1212]    'delta(1620)0'      [      1.63]    [     0]    [           0]    'anti_delta(1620)0'      [0.397]    [0.005]    [0.449]
    [     12214]    'delta(1700)+'      [       1.7]    [     1]    [           0]    'anti_delta(1700)+'      [0.355]    [0.852]    [0.472]
    [     12224]    'delta(1700)++'     [       1.7]    [     2]    [           0]    'anti_delta(1700)++'     [0.151]    [0.623]    [0.005]
    [     11114]    'delta(1700)-'      [       1.7]    [    -1]    [           0]    'anti_delta(1700)-'      [ 0.11]    [0.701]    [0.891]
    [     12114]    'delta(1700)0'      [       1.7]    [     0]    [           0]    'anti_delta(1700)0'      [ 0.53]    [0.553]    [0.465]
    [     12122]    'delta(1900)+'      [       1.9]    [     1]    [           0]    'anti_delta(1900)+'      [0.105]    [0.332]    [0.852]
    [     12222]    'delta(1900)++'     [       1.9]    [     2]    [           0]    'anti_delta(1900)++'     [0.474]    [0.484]    [0.754]
    [     11112]    'delta(1900)-'      [       1.9]    [    -1]    [           0]    'anti_delta(1900)-'      [0.168]    [ 0.07]    [0.159]
    [     11212]    'delta(1900)0'      [       1.9]    [     0]    [           0]    'anti_delta(1900)0'      [ 0.46]    [0.971]    [0.386]
    [      2126]    'delta(1905)+'      [      1.89]    [     1]    [           0]    'anti_delta(1905)+'      [0.092]    [0.563]    [0.655]
    [      2226]    'delta(1905)++'     [      1.89]    [     2]    [           0]    'anti_delta(1905)++'     [0.503]    [0.455]    [0.712]
    [      1116]    'delta(1905)-'      [      1.89]    [    -1]    [           0]    'anti_delta(1905)-'      [ 0.74]    [0.409]    [0.337]
    [      1216]    'delta(1905)0'      [      1.89]    [     0]    [           0]    'anti_delta(1905)0'      [0.397]    [0.387]    [0.173]
    [     22122]    'delta(1910)+'      [      1.91]    [     1]    [           0]    'anti_delta(1910)+'      [0.104]    [0.358]    [0.223]
    [     22222]    'delta(1910)++'     [      1.91]    [     2]    [           0]    'anti_delta(1910)++'     [0.583]    [0.832]    [0.304]
    [     21112]    'delta(1910)-'      [      1.91]    [    -1]    [           0]    'anti_delta(1910)-'      [ 0.94]    [0.887]    [0.026]
    [     21212]    'delta(1910)0'      [      1.91]    [     0]    [           0]    'anti_delta(1910)0'      [0.208]    [0.398]    [0.204]
    [     22214]    'delta(1920)+'      [      1.92]    [     1]    [           0]    'anti_delta(1920)+'      [0.572]    [ 0.45]    [0.693]
    [     22224]    'delta(1920)++'     [      1.92]    [     2]    [           0]    'anti_delta(1920)++'     [0.559]    [0.032]    [0.466]
    [     21114]    'delta(1920)-'      [      1.92]    [    -1]    [           0]    'anti_delta(1920)-'      [0.505]    [0.279]    [0.407]
    [     22114]    'delta(1920)0'      [      1.92]    [     0]    [           0]    'anti_delta(1920)0'      [0.249]    [0.652]    [0.484]
    [     12126]    'delta(1930)+'      [      1.96]    [     1]    [           0]    'anti_delta(1930)+'      [0.951]    [0.053]    [0.104]
    [     12226]    'delta(1930)++'     [      1.96]    [     2]    [           0]    'anti_delta(1930)++'     [0.947]    [0.923]    [0.837]
    [     11116]    'delta(1930)-'      [      1.96]    [    -1]    [           0]    'anti_delta(1930)-'      [0.147]    [0.971]    [0.405]
    [     11216]    'delta(1930)0'      [      1.96]    [     0]    [           0]    'anti_delta(1930)0'      [0.848]    [0.831]    [  0.2]
    [      2218]    'delta(1950)+'      [      1.93]    [     1]    [           0]    'anti_delta(1950)+'      [0.632]    [0.588]    [0.349]
    [      2228]    'delta(1950)++'     [      1.93]    [     2]    [           0]    'anti_delta(1950)++'     [0.541]    [0.332]    [0.897]
    [      1118]    'delta(1950)-'      [      1.93]    [    -1]    [           0]    'anti_delta(1950)-'      [0.582]    [0.296]    [ 0.92]
    [      2118]    'delta(1950)0'      [      1.93]    [     0]    [           0]    'anti_delta(1950)0'      [0.081]    [0.102]    [0.305]
    [      2214]    'delta+'            [     1.232]    [     1]    [           0]    'anti_delta+'            [ 0.86]    [0.888]    [0.469]
    [      2224]    'delta++'           [     1.232]    [     2]    [           0]    'anti_delta++'           [0.712]    [0.233]    [0.073]
    [      1114]    'delta-'            [     1.232]    [    -1]    [           0]    'anti_delta-'            [0.585]    [0.105]    [0.769]
    [      2114]    'delta0'            [     1.232]    [     0]    [           0]    'anti_delta0'            [0.593]    [ 0.85]    [0.034]
    [      3122]    'lambda'            [  1.115683]    [     0]    [       0.263]    'anti_lambda'            [0.329]    [0.156]    [0.733]
    [     13122]    'lambda(1405)'      [    1.4051]    [     0]    [           0]    'anti_lambda(1405)'      [0.504]    [0.779]    [0.307]
    [      3124]    'lambda(1520)'      [    1.5195]    [     0]    [           0]    'anti_lambda(1520)'      [0.559]    [0.663]    [0.944]
    [     23122]    'lambda(1600)'      [       1.6]    [     0]    [           0]    'anti_lambda(1600)'      [0.994]    [0.189]    [0.945]
    [     33122]    'lambda(1670)'      [      1.67]    [     0]    [           0]    'anti_lambda(1670)'      [0.715]    [0.479]    [0.071]
    [     13124]    'lambda(1690)'      [      1.69]    [     0]    [           0]    'anti_lambda(1690)'      [0.538]    [0.092]    [0.926]
    [     43122]    'lambda(1800)'      [       1.8]    [     0]    [           0]    'anti_lambda(1800)'      [0.397]    [0.826]    [  0.1]
    [     53122]    'lambda(1810)'      [      1.81]    [     0]    [           0]    'anti_lambda(1810)'      [0.337]    [0.803]    [ 0.64]
    [      3126]    'lambda(1820)'      [      1.82]    [     0]    [           0]    'anti_lambda(1820)'      [ 0.71]    [ 0.25]    [0.917]
    [     13126]    'lambda(1830)'      [      1.83]    [     0]    [           0]    'anti_lambda(1830)'      [0.873]    [0.477]    [0.668]
    [     23124]    'lambda(1890)'      [      1.89]    [     0]    [           0]    'anti_lambda(1890)'      [0.597]    [0.965]    [0.833]
    [      3128]    'lambda(2100)'      [       2.1]    [     0]    [           0]    'anti_lambda(2100)'      [0.624]    [0.824]    [0.951]
    [     23126]    'lambda(2110)'      [      2.11]    [     0]    [           0]    'anti_lambda(2110)'      [ 0.86]    [ 0.47]    [ 0.85]
    [      5122]    'lambda_b'          [    5.6202]    [     0]    [       0.001]    'anti_lambda_b'          [0.017]    [0.292]    [0.945]
    [      4122]    'lambda_c+'         [   2.28646]    [     1]    [           0]    'anti_lambda_c+'         [0.302]    [0.476]    [0.648]
    [      2112]    'neutron'           [  0.939565]    [     0]    [885700000000]    'anti_neutron'           [    1]    [   .5]    [    0]
    [      3334]    'omega-'            [   1.67245]    [    -1]    [       0.082]    'anti_omega-'            [0.071]    [0.854]    [0.371]
    [      5332]    'omega_b-'          [     6.071]    [    -1]    [       0.001]    'anti_omega_b-'          [0.491]    [0.888]    [0.488]
    [      4332]    'omega_c0'          [    2.6952]    [     0]    [           0]    'anti_omega_c0'          [0.535]    [0.704]    [0.527]
    [      2212]    'proton'            [  0.938272]    [     1]    [          -1]    'anti_proton'            [    0]    [    0]    [    1]
    [      3224]    'sigma(1385)+'      [    1.3828]    [     1]    [           0]    'anti_sigma(1385)+'      [0.023]    [0.235]    [0.669]
    [      3114]    'sigma(1385)-'      [    1.3872]    [    -1]    [           0]    'anti_sigma(1385)-'      [ 0.06]    [0.048]    [0.317]
    [      3214]    'sigma(1385)0'      [    1.3837]    [     0]    [           0]    'anti_sigma(1385)0'      [0.614]    [0.152]    [ 0.18]
    [     13222]    'sigma(1660)+'      [      1.66]    [     1]    [           0]    'anti_sigma(1660)+'      [0.555]    [ 0.67]    [ 0.97]
    [     13112]    'sigma(1660)-'      [      1.66]    [    -1]    [           0]    'anti_sigma(1660)-'      [0.762]    [ 0.37]    [0.213]
    [     13212]    'sigma(1660)0'      [      1.66]    [     0]    [           0]    'anti_sigma(1660)0'      [0.842]    [0.042]    [0.369]
    [     13224]    'sigma(1670)+'      [      1.67]    [     1]    [           0]    'anti_sigma(1670)+'      [0.161]    [0.055]    [0.009]
    [     13114]    'sigma(1670)-'      [      1.67]    [    -1]    [           0]    'anti_sigma(1670)-'      [0.363]    [0.331]    [0.065]
    [     13214]    'sigma(1670)0'      [      1.67]    [     0]    [           0]    'anti_sigma(1670)0'      [0.859]    [0.688]    [0.609]
    [     23222]    'sigma(1750)+'      [      1.75]    [     1]    [           0]    'anti_sigma(1750)+'      [0.549]    [0.341]    [0.453]
    [     23112]    'sigma(1750)-'      [      1.75]    [    -1]    [           0]    'anti_sigma(1750)-'      [0.297]    [0.247]    [0.681]
    [     23212]    'sigma(1750)0'      [      1.75]    [     0]    [           0]    'anti_sigma(1750)0'      [0.607]    [ 0.15]    [0.453]
    [      3226]    'sigma(1775)+'      [     1.775]    [     1]    [           0]    'anti_sigma(1775)+'      [0.938]    [ 0.47]    [0.171]
    [      3116]    'sigma(1775)-'      [     1.775]    [    -1]    [           0]    'anti_sigma(1775)-'      [0.027]    [0.022]    [0.861]
    [      3216]    'sigma(1775)0'      [     1.775]    [     0]    [           0]    'anti_sigma(1775)0'      [0.028]    [0.454]    [0.652]
    [     13226]    'sigma(1915)+'      [     1.915]    [     1]    [           0]    'anti_sigma(1915)+'      [0.385]    [0.063]    [0.972]
    [     13116]    'sigma(1915)-'      [     1.915]    [    -1]    [           0]    'anti_sigma(1915)-'      [0.245]    [0.158]    [0.562]
    [     13216]    'sigma(1915)0'      [     1.915]    [     0]    [           0]    'anti_sigma(1915)0'      [0.063]    [0.391]    [ 0.97]
    [     23224]    'sigma(1940)+'      [      1.94]    [     1]    [           0]    'anti_sigma(1940)+'      [0.484]    [0.376]    [0.833]
    [     23114]    'sigma(1940)-'      [      1.94]    [    -1]    [           0]    'anti_sigma(1940)-'      [0.769]    [0.653]    [0.876]
    [     23214]    'sigma(1940)0'      [      1.94]    [     0]    [           0]    'anti_sigma(1940)0'      [0.983]    [ 0.94]    [0.791]
    [      3228]    'sigma(2030)+'      [      2.03]    [     1]    [           0]    'anti_sigma(2030)+'      [0.226]    [0.101]    [0.415]
    [      3118]    'sigma(2030)-'      [      2.03]    [    -1]    [           0]    'anti_sigma(2030)-'      [0.296]    [0.349]    [0.345]
    [      3218]    'sigma(2030)0'      [      2.03]    [     0]    [           0]    'anti_sigma(2030)0'      [0.833]    [ 0.28]    [0.474]
    [      3222]    'sigma+'            [   1.18937]    [     1]    [        0.08]    'anti_sigma+'            [0.255]    [0.456]    [0.469]
    [      3112]    'sigma-'            [  1.197449]    [    -1]    [       0.148]    'anti_sigma-'            [0.912]    [0.831]    [0.574]
    [      3212]    'sigma0'            [  1.192642]    [     0]    [           0]    'anti_sigma0'            [0.412]    [0.848]    [0.325]
    [      5222]    'sigma_b+'          [    5.8078]    [     1]    [           0]    'anti_sigma_b+'          [0.562]    [0.921]    [0.114]
    [      5112]    'sigma_b-'          [    5.8152]    [    -1]    [           0]    'anti_sigma_b-'          [0.875]    [0.446]    [0.027]
    [      5212]    'sigma_b0'          [    5.8078]    [     0]    [           0]    'anti_sigma_b0'          [  0.5]    [0.692]    [0.352]
    [      4212]    'sigma_c+'          [    2.4529]    [     1]    [           0]    'anti_sigma_c+'          [0.572]    [0.969]    [0.109]
    [      4222]    'sigma_c++'         [   2.45402]    [     2]    [           0]    'anti_sigma_c++'         [0.331]    [0.497]    [0.601]
    [      4112]    'sigma_c0'          [   2.45376]    [     0]    [           0]    'anti_sigma_c0'          [0.748]    [0.661]    [0.329]
    [      3314]    'xi(1530)-'         [     1.535]    [    -1]    [           0]    'anti_xi(1530)-'         [0.058]    [0.388]    [0.948]
    [      3324]    'xi(1530)0'         [    1.5318]    [     0]    [           0]    'anti_xi(1530)0'         [0.017]    [0.753]    [0.384]
    [     23314]    'xi(1690)-'         [      1.69]    [    -1]    [           0]    'anti_xi(1690)-'         [0.809]    [0.668]    [0.784]
    [     23324]    'xi(1690)0'         [      1.69]    [     0]    [           0]    'anti_xi(1690)0'         [0.854]    [0.516]    [0.922]
    [     13314]    'xi(1820)-'         [     1.823]    [    -1]    [           0]    'anti_xi(1820)-'         [0.869]    [ 0.62]    [0.865]
    [     13324]    'xi(1820)0'         [     1.823]    [     0]    [           0]    'anti_xi(1820)0'         [0.136]    [0.158]    [0.813]
    [     33314]    'xi(1950)-'         [      1.95]    [    -1]    [           0]    'anti_xi(1950)-'         [0.592]    [0.449]    [0.822]
    [     33324]    'xi(1950)0'         [      1.95]    [     0]    [           0]    'anti_xi(1950)0'         [0.817]    [0.662]    [0.477]
    [     13316]    'xi(2030)-'         [     2.025]    [    -1]    [           0]    'anti_xi(2030)-'         [0.112]    [0.783]    [0.232]
    [     13326]    'xi(2030)0'         [     2.025]    [     0]    [           0]    'anti_xi(2030)0'         [  0.5]    [0.354]    [0.066]
    [      3312]    'xi-'               [   1.32171]    [    -1]    [       0.164]    'anti_xi-'               [0.965]    [0.248]    [0.639]
    [      3322]    'xi0'               [   1.31486]    [     0]    [        0.29]    'anti_xi0'               [0.242]    [0.291]    [0.411]
    [      5132]    'xi_b-'             [    5.7905]    [    -1]    [       0.002]    'anti_xi_b-'             [0.764]    [0.057]    [ 0.57]
    [      5232]    'xi_b0'             [    5.7905]    [     0]    [       0.001]    'anti_xi_b0'             [0.439]    [0.861]    [0.149]
    [      4232]    'xi_c+'             [    2.4678]    [     1]    [           0]    'anti_xi_c+'             [0.592]    [0.496]    [0.194]
    [      4132]    'xi_c0'             [   2.47088]    [     0]    [           0]    'anti_xi_c0'             [0.475]    [0.138]    [0.005]
    [1000020030]    'He3'               [  2.808391]    [     2]    [          -1]    'anti_He3'               [  0.5]    [    0]    [    1]
    [1000020040]    'alpha'             [  3.727379]    [     2]    [          -1]    'anti_alpha'             [  0.5]    [    0]    [    1]
    [1000010020]    'deuteron'          [  1.875613]    [     1]    [          -1]    'anti_deuteron'          [  0.7]    [    1]    [    0]
    [1000010030]    'triton'            [  2.808921]    [     1]    [3.885235e+17]    'anti_triton'            [  0.7]    [    1]    [    0]
    [        22]    'gamma'             [         0]    [     0]    [           0]                       []    [  0.7]    [  0.7]    [  0.7]
    [         0]    'opticalphoton'     [         0]    [     0]    [           0]                       []    [  0.8]    [  0.9]    [  0.8]
    [1000060120]    'C12'               [ 11.177929]    [     0]    [           0]                       []    [    1]    [  0.7]    [    0]
    [1000060130]    'C13'               [ 12.112543]    [     0]    [           0]                       []    [    1]    [  0.7]    [    0]
    [1000641580]    'Gd158'             [147.105365]    [     0]    [           0]                       []    [    0]    [    1]    [    1]
    [1000030060]    'Li6'               [  5.603051]    [     0]    [           0]                       []    [    0]    [    1]    [    1]
    [1000030070]    'Li7'               [  6.535367]    [     0]    [           0]                       []    [    0]    [    1]    [    1] ...
    }; %#ok<*NBRAK>
end

function pidvec = getpidvec()
%same as pidvec = [X{:,1}]; but faster!
pidvec = [...
                         5
                         4
                         1
                      1103
                        21
                         3
                      3101
                      3103
                      3303
                      3201
                      3203
                         6
                         2
                      2101
                      2103
                      2203
                        11
                        13
                        12
                        14
                        16
                        15
                       521
                       511
                       541
                       531
                       411
                       421
                       431
                       443
                       553
                     10211
                     10111
                   9000211
                   9000111
                     20213
                     20113
                       215
                       115
                     10213
                     10113
                       221
                    100221
                   9020221
                    100331
                     10225
                     10335
                       331
                       441
                     10221
                   9030221
                     10331
                   9000221
                   9010221
                     20223
                     20333
                       225
                   9030225
                   9060225
                       335
                     10223
                     10333
                    100321
                    100311
                     10321
                     10311
                     10323
                     10313
                     20323
                     20313
                     10325
                     10315
                       325
                       315
                    100325
                    100315
                       327
                       317
                    100323
                    100313
                     30323
                     30313
                       323
                       313
                       321
                       311
                       130
                       310
                       223
                    100223
                     30223
                       227
                       333
                    100333
                       337
                    100211
                    100111
                       211
                       111
                     10215
                     10115
                    100213
                    100113
                     30213
                     30113
                       213
                       113
                       217
                       117
                     12212
                     12112
                      2124
                      1214
                     22212
                     22112
                     32212
                     32112
                      2216
                      2116
                     12216
                     12116
                     22124
                     21214
                     42212
                     42112
                     32124
                     31214
                     42124
                     41214
                     12218
                     12118
                     52214
                     52114
                      2128
                      1218
                 100002210
                 100002110
                 100012210
                 100012110
                     32214
                     32224
                     31114
                     32114
                      2122
                      2222
                      1112
                      1212
                     12214
                     12224
                     11114
                     12114
                     12122
                     12222
                     11112
                     11212
                      2126
                      2226
                      1116
                      1216
                     22122
                     22222
                     21112
                     21212
                     22214
                     22224
                     21114
                     22114
                     12126
                     12226
                     11116
                     11216
                      2218
                      2228
                      1118
                      2118
                      2214
                      2224
                      1114
                      2114
                      3122
                     13122
                      3124
                     23122
                     33122
                     13124
                     43122
                     53122
                      3126
                     13126
                     23124
                      3128
                     23126
                      5122
                      4122
                      2112
                      3334
                      5332
                      4332
                      2212
                      3224
                      3114
                      3214
                     13222
                     13112
                     13212
                     13224
                     13114
                     13214
                     23222
                     23112
                     23212
                      3226
                      3116
                      3216
                     13226
                     13116
                     13216
                     23224
                     23114
                     23214
                      3228
                      3118
                      3218
                      3222
                      3112
                      3212
                      5222
                      5112
                      5212
                      4212
                      4222
                      4112
                      3314
                      3324
                     23314
                     23324
                     13314
                     13324
                     33314
                     33324
                     13316
                     13326
                      3312
                      3322
                      5132
                      5232
                      4232
                      4132
                1000020030
                1000020040
                1000010020
                1000010030
                        22
                         0
                1000060120
                1000060130
                1000641580
                1000030060
                1000030070   ...
];
end