//
// Syd: rock-solid application kernel
// src/kernel/chmod.rs: chmod(2), fchmod(2), fchmodat(2), and fchmodat2(2) handlers
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::os::fd::{AsFd, AsRawFd};

use libseccomp::{ScmpNotifResp, ScmpSyscall};
use nix::{
    errno::Errno,
    fcntl::{AtFlags, OFlag},
    sys::stat::{fchmod, Mode},
    NixPath,
};
use once_cell::sync::Lazy;

use crate::{
    config::PROC_FD,
    fs::{safe_open_path, FsFlags},
    hook::{PathArgs, SysArg, UNotifyEventRequest},
    kernel::{syscall_path_handler, to_atflags, to_mode},
    path::XPathBuf,
    sandbox::SandboxGuard,
};

// Note fchmodat2 may not be available,
// and libc::SYS_fchmodat2 may not be defined.
// Therefore we query the number using libseccomp.
static SYS_FCHMODAT2: Lazy<libc::c_long> = Lazy::new(|| {
    ScmpSyscall::from_name("fchmodat2")
        .map(i32::from)
        .map(libc::c_long::from)
        .unwrap_or(-1) // Invalid system call.
});

pub(crate) fn sys_fchmod(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: Reject undefined/invalid mode.
    let mode = match to_mode(req.data.args[1]) {
        Ok(mode) => mode,
        Err(errno) => return request.fail_syscall(errno),
    };

    let argv = &[SysArg {
        dirfd: Some(0),
        ..Default::default()
    }];
    syscall_path_handler(request, "fchmod", argv, |path_args, request, sandbox| {
        // SAFETY:
        // 1. SysArg has one element.
        // 2. SysArg.path is None asserting dir is Some.
        #[allow(clippy::disallowed_methods)]
        let fd = path_args.0.as_ref().unwrap().dir.as_ref().unwrap();

        // SAFETY: We apply force_umask to chmod modes to ensure consistency.
        let mut mode = mode;
        let umask = sandbox.umask.unwrap_or(Mode::empty());
        mode &= !umask;

        fchmod(fd, mode).map(|_| request.return_syscall(0))
    })
}

pub(crate) fn sys_chmod(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: Reject undefined/invalid mode.
    let mode = match to_mode(req.data.args[1]) {
        Ok(mode) => mode,
        Err(errno) => return request.fail_syscall(errno),
    };

    let argv = &[SysArg {
        path: Some(0),
        fsflags: FsFlags::MUST_PATH,
        ..Default::default()
    }];

    syscall_path_handler(request, "chmod", argv, |path_args, request, sandbox| {
        syscall_chmod_handler(request, &sandbox, path_args, mode)
    })
}

pub(crate) fn sys_fchmodat(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: Reject undefined/invalid mode.
    let mode = match to_mode(req.data.args[2]) {
        Ok(mode) => mode,
        Err(errno) => return request.fail_syscall(errno),
    };

    // Note: Unlike fchmodat2, fchmodat always resolves symbolic links.
    let argv = &[SysArg {
        dirfd: Some(0),
        path: Some(1),
        fsflags: FsFlags::MUST_PATH,
        ..Default::default()
    }];

    syscall_path_handler(request, "fchmodat", argv, |path_args, request, sandbox| {
        syscall_chmod_handler(request, &sandbox, path_args, mode)
    })
}

pub(crate) fn sys_fchmodat2(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: Reject undefined/invalid/unused flags.
    let flags = match to_atflags(req.data.args[3], AtFlags::AT_SYMLINK_NOFOLLOW) {
        Ok(flags) => flags,
        Err(errno) => return request.fail_syscall(errno),
    };

    // SAFETY: Reject undefined/invalid mode.
    let mode = match to_mode(req.data.args[2]) {
        Ok(mode) => mode,
        Err(errno) => return request.fail_syscall(errno),
    };

    let mut fsflags = FsFlags::MUST_PATH;
    if flags.contains(AtFlags::AT_SYMLINK_NOFOLLOW) {
        fsflags |= FsFlags::NO_FOLLOW_LAST
    }

    let argv = &[SysArg {
        dirfd: Some(0),
        path: Some(1),
        fsflags,
        ..Default::default()
    }];

    syscall_path_handler(request, "fchmodat2", argv, |path_args, request, sandbox| {
        syscall_chmod_handler(request, &sandbox, path_args, mode)
    })
}

/// A helper function to handle chmod, fchmodat, and fchmodat2 syscalls.
fn syscall_chmod_handler(
    request: &UNotifyEventRequest,
    sandbox: &SandboxGuard,
    args: PathArgs,
    mut mode: Mode,
) -> Result<ScmpNotifResp, Errno> {
    // SAFETY: SysArg has one element.
    #[allow(clippy::disallowed_methods)]
    let path = args.0.as_ref().unwrap();

    // SAFETY: We apply force_umask to chmod modes to ensure consistency.
    let umask = sandbox.umask.unwrap_or(Mode::empty());
    mode &= !umask;

    let fd = path
        .dir
        .as_ref()
        .map(|fd| fd.as_raw_fd())
        .ok_or(Errno::EBADF)?;

    let flags = if path.base.is_empty() {
        // FD-only call, e.g remote-fd transfer due to fchmod(2).
        libc::AT_EMPTY_PATH
    } else {
        // SAFETY: Do not resolve symlinks in base to prevent TOCTTOU.
        libc::AT_SYMLINK_NOFOLLOW
    };

    path.base
        .with_nix_path(|cstr| {
            match if *SYS_FCHMODAT2 > 0 {
                // SAFETY: No libc wrapper for fchmodat2 yet.
                Errno::result(unsafe {
                    libc::syscall(
                        *SYS_FCHMODAT2,
                        fd.as_raw_fd(),
                        cstr.as_ptr(),
                        mode.bits(),
                        flags,
                    )
                })
            } else {
                Err(Errno::ENOSYS)
            } {
                Ok(_) => Ok(()),
                Err(Errno::ENOSYS) if path.base.is_empty() => {
                    // Fallback to `/proc` indirection,
                    //
                    // path to fd is open already!
                    let mut pfd = XPathBuf::from("self/fd");
                    pfd.push_fd(fd.as_raw_fd());
                    pfd.with_nix_path(|cstr| {
                        // SAFETY: We deliberately bypass the libc wrapper here.
                        Errno::result(unsafe {
                            libc::syscall(libc::SYS_fchmodat, PROC_FD(), cstr.as_ptr(), mode.bits())
                        })
                    })?
                    .map(drop)
                }
                Err(Errno::ENOSYS) => {
                    // Fallback to `/proc` indirection.
                    //
                    // open an `O_PATH` fd without following symlinks.
                    let fd = safe_open_path(
                        path.dir.as_ref().map(|fd| fd.as_fd()).ok_or(Errno::EBADF)?,
                        path.base,
                        OFlag::O_NOFOLLOW,
                    )?;
                    let mut pfd = XPathBuf::from("self/fd");
                    pfd.push_fd(fd.as_raw_fd());
                    pfd.with_nix_path(|cstr| {
                        // SAFETY: We deliberately bypass the libc wrapper here.
                        Errno::result(unsafe {
                            libc::syscall(libc::SYS_fchmodat, PROC_FD(), cstr.as_ptr(), mode.bits())
                        })
                    })?
                    .map(drop)
                }
                Err(errno) => Err(errno),
            }
        })?
        .map(|_| request.return_syscall(0))
}
