//
// Syd: rock-solid application kernel
// src/kernel/open.rs: creat(2), open(2), openat(2), and openat2(2) handlers
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::{
    borrow::Cow,
    fs::File,
    io::{Seek, Write},
    os::fd::{AsFd, AsRawFd, IntoRawFd, OwnedFd, RawFd},
};

use libseccomp::ScmpNotifResp;
use nix::{
    errno::Errno,
    fcntl::{OFlag, OpenHow, ResolveFlag},
    sys::stat::{umask, Mode},
    NixPath,
};

use crate::{
    config::{ESYD_SH, MAGIC_PREFIX, NULL_FD, PROC_FILE, ROOT_FILE, SYD_EL},
    cookie::safe_openat2,
    fs::{
        create_memfd, lock_fd, safe_open_how_magicsym, safe_open_magicsym, seal_memfd, unlock_fd,
        CanonicalPath, FileInfo, FileType, FsFlags, MaybeFd, MFD_ALLOW_SEALING,
    },
    hash::aes_ctr_tmp,
    hook::{SysArg, UNotifyEventRequest},
    info, log_enabled,
    path::{XPath, XPathBuf},
    proc::{proc_tty, proc_umask},
    sandbox::{Action, Capability},
    syslog::LogLevel,
    warn,
    workers::aes::{AesMod, AesVal},
};

// `OpenSyscall` represents possible open family system calls.
//
//  The list of open family system calls are: creat(2), open(2),
//  openat(2), and openat2(2).
#[derive(Clone, Copy, Debug, Eq, PartialEq)]
enum OpenSyscall {
    Creat,
    Open,
    Openat,
    Openat2,
}

pub(crate) fn sys_creat(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: Reject invalid mode.
    let mode = match req.data.args[1].try_into() {
        Ok(mode) => mode,
        Err(_) => return request.fail_syscall(Errno::EINVAL),
    };
    let mode = match Mode::from_bits(mode) {
        Some(mode) => mode,
        None => return request.fail_syscall(Errno::EINVAL),
    };

    // SAFETY:
    // 1. Omit MUST_PATH in fsflags as path may not exist yet.
    // 2. Use WANT_BASE to split base which will be opened by the handler.
    let arg = SysArg {
        path: Some(0),
        fsflags: FsFlags::WANT_BASE,
        ..Default::default()
    };
    let flags = OFlag::O_CREAT | OFlag::O_WRONLY | OFlag::O_TRUNC;

    // Path pointer will be validated by open handler
    // after further flag validation.
    syscall_open_handler(request, OpenSyscall::Creat, arg, flags, mode)
}

pub(crate) fn sys_open(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: Truncate undefined/invalid flags.
    let flags = match to_oflag(req.data.args[1], true) {
        Ok(flags) => flags,
        Err(errno) => return request.fail_syscall(errno),
    };

    // SAFETY: Reject undefined/invalid mode.
    let mode = match to_mode(req.data.args[2], flags) {
        Ok(mode) => mode,
        Err(errno) => return request.fail_syscall(errno),
    };

    // Path pointer will be validated by open handler
    // after further flag validation.
    let arg = SysArg::open(flags, false, ResolveFlag::empty());
    syscall_open_handler(request, OpenSyscall::Open, arg, flags, mode)
}

pub(crate) fn sys_openat(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: Truncate undefined/invalid flags.
    let flags = match to_oflag(req.data.args[2], true) {
        Ok(flags) => flags,
        Err(errno) => return request.fail_syscall(errno),
    };

    // SAFETY: Reject undefined/invalid mode.
    let mode = match to_mode(req.data.args[3], flags) {
        Ok(mode) => mode,
        Err(errno) => return request.fail_syscall(errno),
    };

    // Path pointer will be validated by open handler
    // after further flag validation.
    let arg = SysArg::open(flags, true, ResolveFlag::empty());
    syscall_open_handler(request, OpenSyscall::Openat, arg, flags, mode)
}

pub(crate) fn sys_openat2(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: remote_ohow validates address and size.
    let open_how = match request.remote_ohow(req.data.args[2], req.data.args[3]) {
        Ok(open_how) => open_how,
        Err(errno) => {
            return request.fail_syscall(errno);
        }
    };

    // SAFETY: Reject undefined and invalid flags.
    let flags = match to_oflag(open_how.flags, false) {
        Ok(flags) => flags,
        Err(errno) => return request.fail_syscall(errno),
    };

    // SAFETY:
    // 1. Return ENOSYS for valid but unsupported openat2 resolve flags.
    // 2. Return EINVAL for invalid resolve flags.
    let rflags = match ResolveFlag::from_bits(open_how.resolve) {
        Some(rflags) if rflags.contains(ResolveFlag::RESOLVE_IN_ROOT) => {
            return request.fail_syscall(Errno::ENOSYS)
        }
        Some(rflags) => rflags,
        None => return request.fail_syscall(Errno::EINVAL),
    };

    let mode = match to_mode(open_how.mode, flags) {
        Ok(mode) => mode,
        Err(errno) => return request.fail_syscall(errno),
    };

    // Path pointer will be validated by open handler
    // after further flag validation.
    let arg = SysArg::open(flags, true, rflags);
    syscall_open_handler(request, OpenSyscall::Openat2, arg, flags, mode)
}

#[allow(clippy::cognitive_complexity)]
fn syscall_open_handler(
    request: UNotifyEventRequest,
    syscall: OpenSyscall,
    arg: SysArg,
    flags: OFlag,
    mode: Mode,
) -> ScmpNotifResp {
    syscall_handler!(request, |request: UNotifyEventRequest| {
        // SAFETY: When emulating the open call we always open the file
        // descriptor with O_CLOEXEC flag for safety. Here, we mark the
        // state of the O_CLOEXEC flag to use it in seccomp_addfd call.
        // Mutability:
        //   1. o_cloexec is mutable because trace/open_force_cloexec:1
        //      may change its value for O_CLOEXEC.
        //   2. flags is mutable because trace/allow_unsafe_open:0
        //      may change its value for O_PATH. Append-only may
        //      also edit flags.
        let mut o_cloexec = flags.contains(OFlag::O_CLOEXEC);
        let mut flags = flags | OFlag::O_CLOEXEC;

        // Determine capabilities based on access mode:
        //
        // 1. glibc does not include O_PATH to O_ACCMODE.
        // 2. musl defines O_PATH equal to O_EXEC and O_SEARCH,
        //    and O_ACCMODE is defined as O_ACCMODE|O_SEARCH.
        // Here we force the second behaviour by explicitly
        // adding O_PATH into O_ACCMODE. This works on both libcs.
        // See: https://www.openwall.com/lists/musl/2013/02/22/1
        //
        // TODO: Confine O_PATH file descriptors with Stat category,
        // rather than Read category when they can be properly emulated.
        // See: https://bugzilla.kernel.org/show_bug.cgi?id=218501
        let (mut caps, o_path, o_rdwr) = match flags.bits() & (libc::O_ACCMODE | libc::O_PATH) {
            libc::O_RDONLY => (Capability::CAP_READ, false, false),
            libc::O_WRONLY => (Capability::CAP_WRITE, false, false),
            libc::O_RDWR => (Capability::CAP_READ | Capability::CAP_WRITE, false, true),
            libc::O_PATH => (Capability::CAP_READ, true, false),
            _ => return Err(Errno::EINVAL), // Invalid access mode.
        };

        let o_creat = flags.contains(OFlag::O_CREAT);
        let o_tmpfl = flags.contains(OFlag::O_TMPFILE);
        let o_trunc = flags.contains(OFlag::O_TRUNC);
        let mut o_mask = o_creat || o_tmpfl;

        // O_PATH|O_CREAT and O_PATH|O_TMPFILE combinations are invalid.
        if o_path && o_mask {
            return Err(Errno::EINVAL);
        }

        let req = request.scmpreq;
        let sandbox = request.get_sandbox();

        // Check for chroot.
        if sandbox.is_chroot() {
            return Err(Errno::ENOENT);
        }

        let is_lock = sandbox.locked_for(req.pid());

        // We use exactly one of Mktemp, Create, Truncate sandboxing
        // categories to improve usability, ie the priority is
        // O_TMPFILE > O_CREAT > O_TRUNC.
        if o_tmpfl {
            caps.insert(Capability::CAP_MKTEMP);
        } else if o_creat {
            caps.insert(Capability::CAP_CREATE);
        } else if o_trunc {
            caps.insert(Capability::CAP_TRUNCATE);
        }

        let has_write = caps.intersects(Capability::CAP_WRSET);
        if has_write && sandbox.force_ro_open() {
            // SAFETY: Deny write attempt with trace/force_ro_open:1.
            return Err(Errno::EACCES);
        }

        if sandbox.verbose {
            crate::debug!("ctx": "open", "op": "init",
                "cap": caps,
                "sbc": *sandbox.state,
                "cwr": caps.intersects(Capability::CAP_WRSET),
                "arg": format!("{arg:?}"),
                "flg": format!("{flags:?}"),
                "mod": format!("{mode:?}"),
                "req": &request);
        } else {
            crate::debug!("ctx": "open", "op": "init",
                "cap": caps,
                "sbc": *sandbox.state,
                "cwr": caps.intersects(Capability::CAP_WRSET),
                "arg": format!("{arg:?}"),
                "flg": format!("{flags:?}"),
                "mod": format!("{mode:?}"),
                "pid": request.scmpreq.pid);
        }

        // SAFETY: Apply deny_dotdot as necessary for open family.
        let mut arg = arg;
        if sandbox.deny_dotdot() {
            arg.fsflags.insert(FsFlags::NO_RESOLVE_DOTDOT);
        }

        // Read the remote path.
        // If lock is on do not check for magic path.
        let (mut path, magic) = request.read_path(&sandbox, arg, !is_lock)?;

        if sandbox.verbose {
            crate::debug!("ctx": "open", "op": "read_path",
                "path": &path, "magic": magic,
                "req": &request);
        } else {
            crate::debug!("ctx": "open", "op": "read_path",
                "path": &path, "magic": magic,
                "pid": request.scmpreq.pid().as_raw());
        }

        // Handle the special /dev/syd paths.
        if magic {
            if o_path {
                return Err(Errno::EINVAL);
            }

            let fd = if caps.contains(Capability::CAP_READ) {
                let fd = create_memfd(b"syd-box\0", MFD_ALLOW_SEALING)?;
                let mut file = File::from(fd);

                const FLEN: usize = MAGIC_PREFIX.len() + 3; /* .{el,sh} */
                let fack = path.abs().len() == FLEN;
                let fext = path.abs().extension();
                let data = if path.abs().is_equal(MAGIC_PREFIX) {
                    Cow::Owned(serde_json::to_string_pretty(&*sandbox).or(Err(Errno::EINVAL))?)
                } else if fack && fext.map(|ext| ext.is_equal(b"el")).unwrap_or(false) {
                    Cow::Borrowed(SYD_EL)
                } else if fack && fext.map(|ext| ext.is_equal(b"sh")).unwrap_or(false) {
                    Cow::Borrowed(ESYD_SH)
                } else {
                    return Err(Errno::EINVAL);
                };
                drop(sandbox); // release the read-lock.

                file.write_all(data.as_bytes()).or(Err(Errno::EIO))?;
                file.rewind().or(Err(Errno::EIO))?;

                // SAFETY: Deny further writes to the file descriptor.
                seal_memfd(&file)?;

                MaybeFd::Owned(OwnedFd::from(file))
            } else {
                MaybeFd::RawFd(NULL_FD())
            };

            // Send the file descriptor to the process and return the fd no.
            // SAFETY: Randomize FD to prevent reuse.
            return request.send_fd(fd, true, true);
        }

        // Validate file flags based on file type information.
        //
        // is_dir: Used for mask to determine directory override.
        // is_blocking: Used by interrupt handler to unblock FIFOs.
        // may_crypt: Used by Crypt sandboxing.
        let (is_dir, is_blocking, may_crypt) = if let Some(ref file_type) = path.typ {
            if o_creat && !o_tmpfl {
                // Creating open with existing file:
                //
                // Remove Create from capabilities, and add back
                // Truncate capability as necessary to improve
                // usability.
                caps.remove(Capability::CAP_CREATE);
                if o_trunc {
                    caps.insert(Capability::CAP_TRUNCATE);
                }
            }

            match *file_type {
                FileType::Reg => {
                    // SAFETY:
                    // 1. We only ever attempt to encrypt regular files.
                    // 2. We do not support interruptions on regular file blocks.
                    (false, false, true)
                }
                FileType::Dir => {
                    // We know it's a directory, so let's assert it.
                    // Unless path ends with a slash, in that case
                    // we don't want to break expectations.
                    if path.abs().last() != Some(b'/') {
                        flags.insert(OFlag::O_DIRECTORY);
                    }

                    // Change capability from Read to Readdir.
                    caps.remove(Capability::CAP_READ);
                    caps.insert(Capability::CAP_READDIR);

                    (true, false, false)
                }
                FileType::Blk | FileType::Unk => {
                    // SAFETY:
                    // 1. Do not allow access to block devices.
                    // 2. Do not allow access to files with unknown types.
                    // 3. Deny with ENOENT for stealth.
                    return Err(Errno::ENOENT);
                }
                FileType::MagicLnk(_, _) => {
                    // SAFETY/TODO: Implement further restrictions on magic-links here.
                    if file_type.is_magic_dir() {
                        // Change capability from Read to Readdir.
                        caps.remove(Capability::CAP_READ);
                        caps.insert(Capability::CAP_READDIR);
                    }

                    // SAFETY: This may or may not block, better safe than sorry.
                    (
                        false,
                        !(flags.contains(OFlag::O_NONBLOCK) || flags.contains(OFlag::O_NDELAY)),
                        false,
                    )
                }
                FileType::Lnk => {
                    // SAFETY: This may or may not block, better safe than sorry.
                    (
                        false,
                        !(flags.contains(OFlag::O_NONBLOCK) || flags.contains(OFlag::O_NDELAY)),
                        false,
                    )
                }
                FileType::Chr | FileType::Fifo | FileType::Sock => {
                    // SAFETY: character devices, fifos and sockets may block.
                    (
                        false,
                        !(flags.contains(OFlag::O_NONBLOCK) || flags.contains(OFlag::O_NDELAY)),
                        false,
                    )
                }
                FileType::Mfd => {
                    // SAFETY:
                    // 1. We do not encrypt memory fds.
                    // 2. We do not support interruptions on memory fd blocks.
                    (false, false, false)
                }
            }
        } else if !o_creat {
            // Non-creating open on non-existing file.
            return Err(Errno::ENOENT);
        } else {
            // Creating open on non-existing file.
            //
            // Note, adding O_EXCL to flags here to assert file creation
            // may result in a race condition where a fellow Syd thread
            // can race with this one in creating the same file, making
            // the thread losing the race return a confusing EEXIST error.
            // Therefore we avoid doing that, see:
            // https://gitlab.exherbo.org/sydbox/sydbox/-/issues/211
            //
            // flags.insert(OFlag::O_EXCL);

            // Non-existing files are not directories.
            // Non-existing files do not block.
            // Non-existing files may be encrypted from scratch.
            (false, false, true)
        };

        // SAFETY:
        // 1. We must provide safe access to sandbox process' controlling terminal.
        // 2. Both "/dev/tty" and the original tty path are checked for access.
        let pid = req.pid();
        let dev_tty = Cow::Borrowed(XPath::from_bytes(b"/dev/tty"));
        let has_tty = if path.abs().is_equal(dev_tty.as_bytes()) {
            let dev_tty = proc_tty(pid)?;
            if !request.is_valid() {
                return Err(Errno::ESRCH);
            }
            path = CanonicalPath::new_tty(dev_tty)?;

            true // TTY.
        } else {
            false // Not a TTY.
        };

        // SAFETY: Ensure the sandbox process cannot
        // acquire a new controlling terminal other
        // than what they already have.
        // Note, O_NOCTTY is invalid with O_PATH,
        // but we don't care for now since we'll change it
        // to O_RDONLY later as a mitigation due to our
        // inability to emulate them.
        // See: https://bugzilla.kernel.org/show_bug.cgi?id=218501
        flags.insert(OFlag::O_NOCTTY);

        // Sandboxing.
        let orig_caps = caps;
        let caps = sandbox.getcaps(orig_caps); // Get enabled caps.

        let mut action = Action::Allow;
        let mut filter = false;

        let mut path_check = if !caps.is_empty() {
            // Convert /proc/${pid} to /proc/self as necessary.
            let path_check = if let Some(p) = path.abs().split_prefix(b"/proc") {
                let mut buf = itoa::Buffer::new();
                let req = request.scmpreq;
                let pid = buf.format(req.pid);
                if let Some(p) = p.split_prefix(pid.as_bytes()) {
                    let mut pdir = XPathBuf::from("/proc/self");
                    pdir.push(p.as_bytes());
                    Cow::Owned(pdir)
                } else {
                    Cow::Borrowed(path.abs())
                }
            } else {
                Cow::Borrowed(path.abs())
            };

            if !has_tty {
                for cap in caps {
                    let (new_action, new_filter) = sandbox.check_path(cap, &path_check);
                    if new_action >= action {
                        action = new_action;
                    }
                    if !filter && new_filter {
                        filter = true;
                    }
                }
            } else {
                // Both "/dev/tty" and the original tty path are checked for access.
                // The more critical action wins.
                for path in [&path_check, &dev_tty] {
                    for cap in caps {
                        let (new_action, new_filter) = sandbox.check_path(cap, path);
                        if new_action >= action {
                            action = new_action;
                        }
                        if !filter && new_filter {
                            filter = true;
                        }
                    }
                }
            }
            path_check
        } else {
            Cow::Borrowed(path.abs())
        };

        let (hidden, mut crypted) = if action.is_denying() {
            // No need to check for mask when denying.
            // No need to check for encryption when denying.
            (sandbox.is_hidden(&path_check), false)
        } else {
            // No need for hidden check if we're allowing.
            // SAFETY:
            // 1. Check for encrypted path and mark for later.
            // 2. Check for masked path and change path to /dev/null.
            // 3. Check for append-only path and edit flags argument.
            // We perform these check only if we're allowing.
            let crypted = if let Some(mask) = sandbox.is_masked(&path_check) {
                let mask = if let Some(mask_dir) = &mask.mask_dir {
                    // Override mask for directories as necessary.
                    if is_dir {
                        Some(mask_dir)
                    } else {
                        mask.mask_all.as_ref()
                    }
                } else {
                    mask.mask_all.as_ref()
                };
                match mask {
                    None => path = CanonicalPath::new_null(),
                    Some(mask) => path = CanonicalPath::new_mask(mask, &path_check)?,
                };
                // SAFETY: Mask path is not subject to sandbox check!
                // `path_check' is only used for logging and points
                // to the original path.
                path_check = Cow::Borrowed(path.abs());
                false // masked path is not encrypted.
            } else if sandbox.enabled(Capability::CAP_CRYPT) {
                may_crypt && sandbox.is_crypt(&path_check)
            } else {
                false // encryption not enabled for path.
            };

            // Check for append-only path and edit flags argument.
            // Temporary files can not be made append-only.
            if has_write && sandbox.is_append(&path_check) {
                flags.insert(OFlag::O_APPEND);
                flags.remove(OFlag::O_TRUNC);

                if sandbox.verbose {
                    info!("ctx": "open", "op": "set_append_only",
                            "msg": "added O_APPEND and removed O_TRUNC from open flags",
                            "sys": request.syscall, "path": &path_check,
                            "flags": format!("{flags:?}"),
                            "cap": caps,
                            "cap_write_set": Capability::CAP_WRSET,
                            "req": &request);
                } else {
                    info!("ctx": "open", "op": "set_append_only",
                            "msg": "added O_APPEND and removed O_TRUNC from open flags",
                            "sys": request.syscall, "path": &path_check,
                            "flags": format!("{flags:?}"),
                            "cap": caps,
                            "cap_write_set": Capability::CAP_WRSET,
                            "pid": request.scmpreq.pid);
                }
            }

            (false, crypted)
        };

        let force_umask = sandbox.umask;
        let verbose = sandbox.verbose;
        let unsafe_open_path = sandbox.allow_unsafe_open_path();
        let unsafe_open_cdev = sandbox.allow_unsafe_open_cdev();
        let restrict_memfd = !sandbox.allow_unsafe_memfd();
        if sandbox.force_cloexec() {
            o_cloexec = true;
        }
        let o_rand_fd = sandbox.force_rand_fd();
        let setup_fds = sandbox.crypt_setup();
        let crypt_tmp = if crypted {
            sandbox.crypt_tmp.as_ref().map(|fd| fd.as_raw_fd())
        } else {
            None
        };
        drop(sandbox); // release the read-lock.

        // Perform action: allow->emulate, deny->log.
        if !filter && action >= Action::Warn && log_enabled!(LogLevel::Warn) {
            let grp = caps.to_string().to_ascii_lowercase();
            if verbose {
                warn!("ctx": "access", "cap": caps, "act": action,
                    "sys": request.syscall, "path": &path_check,
                    "open_flags": format!("{flags:?}"),
                    "open_mode": format!("{mode:?}"),
                    "tip": format!("configure `allow/{grp}+{path_check}'"),
                    "req": &request);
            } else {
                warn!("ctx": "access", "cap": caps, "act": action,
                    "sys": request.syscall, "path": &path_check,
                    "open_flags": format!("{flags:?}"),
                    "open_mode": format!("{mode:?}"),
                    "tip": format!("configure `allow/{grp}+{path_check}'"),
                    "pid": request.scmpreq.pid);
            }
        }

        // SAFETY: Access check is done, now it is safe to:
        //
        // 1. Return ENOENT if path is hidden.
        // 2. Return EEXIST if options include MISS_LAST.
        // 3. Return ENOTDIR for non-directories with trailing slash.
        // 4. Return EISDIR for write opens on directories.
        // 5. Return ELOOP for symlinks unless O_NOFOLLOW was passed.

        // We check for the actions Deny|Filter here as other actions
        // such as Panic, Stop, Kill are handled afterwards as necessary.
        if hidden && matches!(action, Action::Deny | Action::Filter) {
            return Err(Errno::ENOENT);
        }

        if arg.fsflags.missing() && path.typ.is_some() {
            // Exclusive open for existing file.
            return Err(Errno::EEXIST);
        }

        if let Some(file_type) = &path.typ {
            if !matches!(file_type, FileType::Dir | FileType::MagicLnk(_, _))
                && path.abs().last() == Some(b'/')
            {
                return Err(Errno::ENOTDIR);
            }
        }

        if let Some(FileType::Dir) = &path.typ {
            if !o_tmpfl {
                if orig_caps.can_write() {
                    // Open for write on directory.
                    return Err(Errno::EISDIR);
                } else if o_creat && path.abs().last() == Some(b'/') {
                    // Creating open on directory.
                    return Err(Errno::EISDIR);
                }
            }
        }

        // SAFETY: Return EPERM for {/dev,/proc}/kmsg,
        // so dmesg(1) falls back to syslog(2) which we provide.
        // EPERM is fine as we do this after the access check
        // so path hiding was already done as necessary.
        if !has_tty
            && ((path.typ == Some(FileType::Chr) && path.abs().is_equal(b"/dev/kmsg"))
                || path.abs().is_equal(b"/proc/kmsg"))
        {
            return Err(Errno::EPERM);
        }

        // SAFETY: Do not follow (magic) symlinks after canonicalization.
        // Exception: Last component is allowed with O_NOFOLLOW.
        if let Some(FileType::Lnk) = &path.typ {
            if !flags.contains(OFlag::O_NOFOLLOW) {
                return Err(Errno::ELOOP);
            }
        }

        match action {
            Action::Allow | Action::Warn => {
                // The system call is allowed.
                // To prevent TOCTOU, we open the file ourselves,
                // and put the file descriptor to the process'
                // address space with SECCOMP_IOCTL_NOTIF_ADDFD.
                if o_path {
                    if unsafe_open_path {
                        // SAFETY:
                        // seccomp addfd operation returns EBADF for O_PATH file
                        // descriptors so there's no TOCTOU-free way to emulate
                        // this as of yet. However we did our best by
                        // delaying continue up to this point, thereby
                        // including the open request to the sandbox access
                        // check.
                        return unsafe { Ok(request.continue_syscall()) };
                    }
                    // SAFETY: Turn O_PATH flag to O_RDONLY for successful emulation.
                    flags.remove(OFlag::O_PATH);
                    flags.insert(OFlag::O_RDONLY);
                } else if unsafe_open_cdev
                    && o_rdwr
                    && !o_creat
                    && !o_trunc
                    && !o_tmpfl
                    && path.typ == Some(FileType::Chr)
                {
                    // SAFETY:
                    //
                    // trace/allow_unsafe_open_cdev:true
                    //
                    // 1. Some character devices, such as AMD GPUs,
                    //    require per-application access to the GPU
                    //    device, therefore opening the device in the
                    //    Syd emulator thread and then continuing the
                    //    subsequent ioctl(2) system calls in the
                    //    sandbox process is going to return EBADF.
                    //    Until, Syd has a way to fully emulate the
                    //    ioctl(2) request space and are able to call
                    //    ioctl(2) directly from Syd emulator threads,
                    //    this option may be used to access such
                    //    character devices. Note, setting this option
                    //    opens a TOCTOU attack vector, whereby the
                    //    sandbox process can open an arbitrary file
                    //    instead of the character device in question!
                    // 2. Syd does not CONTINUE the system call if at
                    //    least one of the flags
                    //    O_CREAT|O_TRUNC|O_TMPFILE is set in flags
                    //    argument to limit the scope of the TOCTOU
                    //    attack vector.
                    //  3. Syd CONTINUEs the system call if and only if
                    //     O_RDWR is set in the flags argument to limit
                    //     the scope of the TOCTOU attack vector.
                    //  4. Syd returns ENOSYS for openat2(2) rather than
                    //     CONTINUE'ing the system call to prevent the
                    //     "struct open_how" pointer indirection to
                    //     bypass the restrictions applied to the flags
                    //     argument.
                    //  5. This option may be changed at runtime, so it
                    //     is highly recommended to unset this option
                    //     right after the respective character device
                    //     is open using the syd(2) API to prevent the
                    //     TOCTOU attack vector.
                    return if syscall != OpenSyscall::Openat2 {
                        // SAFETY: See above, stupid clippy.
                        unsafe { Ok(request.continue_syscall()) }
                    } else {
                        Err(Errno::ENOSYS)
                    };
                }

                // SAFETY: We have already resolved the symbolic
                // links in the path as necessary, to prevent a
                // time-of-check to time-of-use vector:
                // 1. Add O_NOFOLLOW to flags.
                // 2. Add RESOLVE_BENEATH to flags.
                // 3. Add RESOLVE_NO_MAGICLINKS | RESOLVE_NO_SYMLINKS to flags.
                // 4. Mode must be 0 if O_CREAT or O_TMPFILE is not in flags.
                // Note, magic symbolic links are an exception here.
                let mut resolve_flags = ResolveFlag::empty();
                if matches!(path.typ, Some(FileType::MagicLnk(_, _))) {
                    if flags.contains(OFlag::O_NOFOLLOW) {
                        // Magic symlink O_PATH fds were continued
                        // as necessary if relevant unsafe options
                        // were set. After this point, we have to
                        // ELOOP.
                        return Err(Errno::ELOOP);
                    }

                    // SAFETY: Ensure we can never acquire a
                    // controlling terminal by misguided magic symlink.
                    flags.insert(OFlag::O_NOCTTY);

                    // SAFETY: Ensure no encryption or file creation
                    // attempts can ever happen for magic symlinks.
                    o_mask = false;
                    crypted = false;
                    flags.remove(OFlag::O_CREAT);
                    flags.remove(OFlag::O_TMPFILE);
                } else {
                    flags.insert(OFlag::O_NOFOLLOW);
                    resolve_flags.insert(ResolveFlag::RESOLVE_BENEATH);
                    resolve_flags.insert(ResolveFlag::RESOLVE_NO_MAGICLINKS);
                    resolve_flags.insert(ResolveFlag::RESOLVE_NO_SYMLINKS);
                }

                let fd: OwnedFd = if crypted {
                    // Handle Encryption.
                    if let Some(fd) = handle_crypt(
                        setup_fds,
                        &request,
                        &path,
                        crypt_tmp,
                        flags,
                        mode,
                        force_umask,
                        o_cloexec,
                        o_rand_fd,
                        restrict_memfd,
                    )? {
                        // read-only encryption.
                        fd
                    } else {
                        // read-write encryption.
                        // We do not need to send a response,
                        // return a dummy response which will be
                        // skipped by the handler.
                        return Ok(ScmpNotifResp::new(0, 0, 0, 0));
                    }
                } else if path.base.is_empty() {
                    // Existing path: Construct path to /proc magic symlink,
                    // or pre-open file descriptor.
                    //
                    // SAFETY: Note, the path may be borrowed here, ie
                    // it can be a fd to the preopen `/`, `/proc` or
                    // `/dev/null` fds. In these cases, we still want to
                    // reopen, because the fd we're going to send will
                    // share the same open file description and we do
                    // not want to mess up file offsets for everyone.
                    let mut pfd = XPathBuf::from("self/fd");

                    // SAFETY: ^^ empty base asserts dir is Some.
                    #[allow(clippy::disallowed_methods)]
                    pfd.push_fd(path.dir.as_ref().map(|fd| fd.as_raw_fd()).unwrap());

                    // Reopen the `O_PATH` path fd with the requested flags.
                    flags.remove(OFlag::O_NOFOLLOW);
                    let how = safe_open_how_magicsym(flags);

                    // SAFETY: Record blocking call so it can get invalidated.
                    if is_blocking {
                        request.cache.add_sys_block(req, false)?;
                    }

                    let result = safe_openat2(PROC_FILE(), &pfd, how);

                    // Remove invalidation record unless interrupted.
                    if is_blocking && !matches!(result, Err(Errno::EINTR)) {
                        request.cache.del_sys_block(req.id);
                    }

                    result?
                } else if o_mask {
                    // Cannot be O_PATH or encrypted fd!
                    // SAFETY: If we're creating the file,
                    // we must fork so we can apply the umask
                    // and still honour POSIX ACLs.
                    handle_creat(&request, &path, flags, mode, force_umask)?
                } else {
                    // Prepare `struct open_how`.
                    // Mode is empty because we handle creation differently.
                    let how = OpenHow::new().flags(flags).resolve(resolve_flags);

                    // Prepare file descriptor, avoid relative pathnames.
                    let fd = path.dir.as_ref().map(|fd| fd.as_fd()).ok_or(Errno::EBADF)?;

                    // SAFETY: Record blocking call so it can get invalidated.
                    if is_blocking {
                        request.cache.add_sys_block(req, false)?;
                    }

                    // All set, open the file.
                    let result = safe_openat2(fd, path.base, how);

                    // Remove invalidation record unless interrupted.
                    if is_blocking && !matches!(result, Err(Errno::EINTR)) {
                        request.cache.del_sys_block(req.id);
                    }

                    result?
                };

                // File opened successfully, return to caller at one go.
                request.send_fd(fd, o_cloexec, o_rand_fd)
            }
            Action::Deny | Action::Filter if hidden => Err(Errno::ENOENT),
            Action::Deny | Action::Filter => Err(Errno::EACCES),
            Action::Panic => panic!(),
            Action::Exit => std::process::exit(libc::EACCES),
            action => {
                // Stop|Kill
                let _ = request.kill(action);
                Err(Errno::EACCES)
            }
        }
    })
}

/// Handle open() calls.
fn do_open(path: &CanonicalPath, mut safe_flags: OFlag) -> Result<OwnedFd, Errno> {
    if path.base.is_empty() {
        // Existing path: Construct path to proc magic symlink.
        let mut pfd = XPathBuf::from("self/fd");

        // SAFETY: ^^ empty base asserts dir is Some.
        #[allow(clippy::disallowed_methods)]
        pfd.push_fd(path.dir.as_ref().map(|fd| fd.as_raw_fd()).unwrap());

        // Reopen the `O_PATH` path fd with the requested flags.
        safe_flags.remove(OFlag::O_NOFOLLOW);

        safe_open_magicsym(PROC_FILE(), &pfd, safe_flags)
    } else {
        // Return a read-only fd to the underlying encrypted file.
        let how = OpenHow::new().flags(safe_flags).resolve(
            ResolveFlag::RESOLVE_BENEATH
                | ResolveFlag::RESOLVE_NO_MAGICLINKS
                | ResolveFlag::RESOLVE_NO_SYMLINKS,
        );

        let (fd, base) = if let Some(ref fd) = path.dir {
            (fd.as_fd(), path.base)
        } else if path.abs().is_proc() {
            (
                PROC_FILE(),
                XPath::from_bytes(&path.abs().as_bytes()[b"/proc/".len()..]),
            )
        } else {
            (
                ROOT_FILE(),
                XPath::from_bytes(&path.abs().as_bytes()[b"/".len()..]),
            )
        };

        safe_openat2(fd, base, how)
    }
}

/// Handle open() calls for encrypted files.
#[allow(clippy::too_many_arguments)]
fn handle_crypt(
    setup_fds: Result<(RawFd, RawFd), Errno>,
    request: &UNotifyEventRequest,
    path: &CanonicalPath,
    tmpdir: Option<RawFd>,
    safe_flags: OFlag,
    mode: Mode,
    force_umask: Option<Mode>,
    o_cloexec: bool,
    o_rand_fd: bool,
    restrict_memfd: bool,
) -> Result<Option<OwnedFd>, Errno> {
    #[allow(clippy::disallowed_methods)]
    let files = request.crypt_map.as_ref().unwrap();
    let my_mode = AesMod::from(safe_flags);

    // Check for concurrent read access to the same crypt fd.
    if my_mode == AesMod::Read {
        // Lock AesMap for read:
        // We keep the read lock until we open the file,
        // to ensure no races occur with a subsequent Syd thread.
        let files = files.read().unwrap_or_else(|err| err.into_inner());

        // Concurrent access is only allowed if both modes are Read.
        let entry = files
            .0
            .get(path.abs())
            .filter(|&entry| entry.mode == AesMod::Read);

        if let Some(entry) = entry {
            // Open a new file description.
            // Lock it for read and pass to sandbox process.
            let mut flags = safe_flags;
            flags.remove(OFlag::O_NOFOLLOW);

            let mut pfd = XPathBuf::from("self/fd");
            pfd.push_fd(entry.crypt_fd);

            match safe_open_magicsym(PROC_FILE(), &pfd, flags) {
                Ok(fd) if lock_fd(&fd, false, false).is_ok() => {
                    // Return shared fd.
                    return Ok(Some(fd));
                }
                Ok(fd) => {
                    // SAFETY: If our attempt to lock fails,
                    // syd_aes thread has already taken over.
                    // We can not use the crypt fd any longer!
                    drop(fd);
                    /* fall through */
                }
                Err(Errno::ENOENT) => {
                    // SAFETY: If our attempt to open via procfs fails,
                    // syd_aes thread had already closed the crypt fd.
                    // We can not use the crypt fd any longer!
                    /* fall through */
                }
                Err(errno) => return Err(errno),
            };
        }
    }

    // Promote O_WRONLY to O_RDWR and drop O_APPEND.
    // SAFETY: This fd is not exposed to sandbox process.
    let mut flags = safe_flags;
    flags.remove(OFlag::O_WRONLY);
    flags.insert(OFlag::O_RDWR);
    flags.remove(OFlag::O_APPEND);

    // Strip O_ASYNC|O_NDELAY|O_NONBLOCK.
    // We want blocking writes to the underlying fd.
    flags.remove(OFlag::O_ASYNC | OFlag::O_NDELAY | OFlag::O_NONBLOCK);

    // Open or create the file as read-write.
    //
    // Note, if `path.base.is_empty()` we have an existing file
    // which do_open is going to reopen using proc magic symlink.
    let create = !path.base.is_empty() && safe_flags.contains(OFlag::O_CREAT);
    let enc_fd = if create {
        handle_creat(request, path, flags, mode, force_umask)
    } else {
        do_open(path, flags)
    }?;

    // Hold a write-lock to ensure no concurrent Syd aes writes.
    //
    // SAFETY: Block until we can acquire the lock to ensure safe
    // concurrent access _unless_ the open was a non-blocking open
    // in which case we happily return EAGAIN so the caller can
    // retry.
    //
    // Similarly, `aes_ctr_tmp` respects non-blocking opens too,
    // and returns a non-blocking fd as necessary.
    let wait = !(safe_flags.contains(OFlag::O_NONBLOCK) || safe_flags.contains(OFlag::O_NDELAY));
    lock_fd(&enc_fd, true, create || wait)?;

    // Acquire encryption sockets.
    let setup_fds = setup_fds?;

    // Map decrypted version to memory/temporary FD.
    // SAFETY: This does not read plaintext into memory!
    // We use zero-copy with splice and pipes.
    // Note, enc_fd is an OwnedFd so in the event of
    // an aes_ctr_tmp error it'll be closed and the lock
    // will be released.
    let (fd, iv) = if let Some((fd, iv)) =
        aes_ctr_tmp(setup_fds, &enc_fd, safe_flags, tmpdir, restrict_memfd)?
    {
        (fd, iv)
    } else {
        // SAFETY:
        // 1. This is not a new file.
        // 2. This is not a Syd encrypted file.
        // Unlock and open as-is.
        unlock_fd(&enc_fd)?;
        return Ok(Some(enc_fd));
    };

    // Reopen a new instance to pass to the sandbox process.
    // This instance points to a different open file description!
    let mut pfd = XPathBuf::from("self/fd");
    pfd.push_fd(fd.as_raw_fd());

    let mut flags = safe_flags;
    flags.remove(
        OFlag::O_WRONLY | OFlag::O_ASYNC | OFlag::O_CREAT | OFlag::O_EXCL | OFlag::O_NOFOLLOW,
    );
    flags.insert(OFlag::O_RDWR);

    let aes_fd = safe_open_magicsym(PROC_FILE(), &pfd, flags)?;

    // Record the file information,
    // so we can answer reliably on fstat.
    let info = FileInfo::from_fd(&aes_fd)?;

    // SAFETY:
    // No need to wait on this lock, as we've just opened the file.
    // Close the send fd to get rid off our copy of the lock.
    lock_fd(&aes_fd, false, false)?;
    request.send_fd(aes_fd, o_cloexec, o_rand_fd)?;

    // Record encryption process information.
    {
        files
            .write()
            .unwrap_or_else(|err| err.into_inner())
            .0
            .insert(
                path.abs().to_owned(),
                AesVal {
                    info,
                    iv: Some(iv),
                    enc_fd: Some(enc_fd),
                    crypt_fd: fd.into_raw_fd(),
                    mode: my_mode,
                    spawned: false,
                },
            );
    }

    Ok(None)
}

/// Handle open() calls that can potentially create files.
fn handle_creat(
    request: &UNotifyEventRequest,
    path: &CanonicalPath,
    safe_flags: OFlag,
    mut mode: Mode,
    force_umask: Option<Mode>,
) -> Result<OwnedFd, Errno> {
    // SAFETY: force_umask overrides POSIX ACLs.
    if let Some(mask) = force_umask {
        mode &= !mask;
    }

    let how = OpenHow::new().flags(safe_flags).mode(mode).resolve(
        ResolveFlag::RESOLVE_BENEATH
            | ResolveFlag::RESOLVE_NO_MAGICLINKS
            | ResolveFlag::RESOLVE_NO_SYMLINKS,
    );

    // Determine process umask to apply in the thread.
    let req = request.scmpreq;
    let mask = proc_umask(req.pid())?;

    // Set umask which is per-thread here.
    umask(mask);

    // All set, make the open call.
    let fd = path.dir.as_ref().map(|fd| fd.as_fd()).ok_or(Errno::EBADF)?;
    safe_openat2(fd, path.base, how)
}

fn to_oflag(arg: u64, truncate: bool) -> Result<OFlag, Errno> {
    // SAFETY: Reject undefined flags.
    let flags = arg.try_into().or(Err(Errno::EINVAL))?;

    // SAFETY: Reject invalid flags.
    const KEEP_OFLAG: OFlag =
        // Keep O_LARGEFILE for old ABI.
        OFlag::from_bits_retain(0x8000);

    let mut flags = OFlag::from_bits_retain(flags);
    let preserved = flags & KEEP_OFLAG;
    flags.remove(KEEP_OFLAG);

    if truncate {
        // open, openat truncates invalid flags.
        return Ok((flags & OFlag::all()) | preserved);
    }

    // openat2 validates invalid flags.
    let unknown_flags = flags.difference(OFlag::all());
    if unknown_flags.bits() != 0 {
        crate::error!("ctx": "open", "op": "invalid_flags",
            "msg": "passed invalid flags to open flags argument",
            "flags": format!("{unknown_flags:?}"));
        return Err(Errno::EINVAL);
    }

    Ok(flags | preserved)
}

#[inline]
fn to_mode(arg: u64, flags: OFlag) -> Result<Mode, Errno> {
    if flags.contains(OFlag::O_CREAT) || flags.contains(OFlag::O_TMPFILE) {
        // SAFETY: Reject invalid mode with creation flags.
        crate::kernel::to_mode(arg)
    } else {
        // SAFETY: mode argument is ignored without creation flags.
        // glibc sets it to 0, musl does not. Both are valid behaviour.
        Ok(Mode::empty())
    }
}
