//
// Syd: rock-solid application kernel
// src/kernel/setid.rs: Set UID/GID syscall handlers
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use libseccomp::ScmpNotifResp;
use nix::{
    errno::Errno,
    unistd::{getresgid, getresuid, Gid, Uid},
};

use crate::{
    caps,
    config::{GID_MIN, UID_MIN},
    hook::UNotifyEventRequest,
    safe_drop_cap, warn,
};

#[allow(clippy::cognitive_complexity)]
pub(crate) fn sys_setuid(request: UNotifyEventRequest) -> ScmpNotifResp {
    syscall_handler!(request, |request: UNotifyEventRequest| {
        let req = request.scmpreq;

        let target_uid =
            Uid::from_raw(libc::uid_t::try_from(req.data.args[0]).or(Err(Errno::EINVAL))?);
        let source_uid = Uid::current();

        if u64::from(target_uid.as_raw()) <= UID_MIN {
            // SAFETY: This is already asserted with the parent
            // seccomp-bpf filter, this is the second layer.
            return Ok(request.fail_syscall(Errno::EACCES));
        } else if source_uid == target_uid {
            // SAFETY: There's no pointer dereference in the access check.
            return unsafe { Ok(request.continue_syscall()) };
        }

        let sandbox = request.get_sandbox();
        let allowed = sandbox.chk_uid_transit(source_uid, target_uid);
        let verbose = sandbox.verbose;
        drop(sandbox); // release the read lock.

        if !allowed {
            if verbose {
                warn!("ctx": "safesetid", "err": libc::EACCES,
                    "sys": request.syscall, "target_uid": target_uid.as_raw(), "source_uid": source_uid.as_raw(),
                    "req": request);
            } else {
                warn!("ctx": "safesetid", "err": libc::EACCES,
                    "sys": request.syscall, "target_uid": target_uid.as_raw(), "source_uid": source_uid.as_raw(),
                    "pid": request.scmpreq.pid);
            }
            return Err(Errno::EACCES);
        }

        // SAFETY: nix version of setuid does not allow -1 as argument.
        if let Err(errno) = Errno::result(unsafe { libc::setuid(target_uid.as_raw()) }) {
            if verbose {
                warn!("ctx": "safesetid", "err": errno as i32,
                    "sys": request.syscall, "target_uid": target_uid.as_raw(), "source_uid": source_uid.as_raw(),
                    "req": request);
            } else {
                warn!("ctx": "safesetid", "err": errno as i32,
                    "sys": request.syscall, "target_uid": target_uid.as_raw(), "source_uid": source_uid.as_raw(),
                    "pid": request.scmpreq.pid);
            }
            return Err(errno);
        } else if safe_drop_cap(caps::Capability::CAP_SETUID).is_err() {
            // SAFETY: We cannot do much on errors,
            // and on panic the thread will be restarted.
            // The best we can do from a security POV is
            // to enter Ghost mode. This is certainly
            // unexpected but it's safe.
            return Err(Errno::EOWNERDEAD);
        }

        // SAFETY: There's no pointer dereference in the access check.
        unsafe { Ok(request.continue_syscall()) }
    })
}

#[allow(clippy::cognitive_complexity)]
pub(crate) fn sys_setgid(request: UNotifyEventRequest) -> ScmpNotifResp {
    syscall_handler!(request, |request: UNotifyEventRequest| {
        let req = request.scmpreq;

        let target_gid =
            Gid::from_raw(libc::gid_t::try_from(req.data.args[0]).or(Err(Errno::EINVAL))?);
        let source_gid = Gid::current();

        if u64::from(target_gid.as_raw()) <= GID_MIN {
            // SAFETY: This is already asserted with the parent
            // seccomp-bpf filter, this is the second layer.
            return Ok(request.fail_syscall(Errno::EACCES));
        } else if source_gid == target_gid {
            // SAFETY: There's no pointer dereference in the access check.
            return unsafe { Ok(request.continue_syscall()) };
        }

        let sandbox = request.get_sandbox();
        let allowed = sandbox.chk_gid_transit(source_gid, target_gid);
        let verbose = sandbox.verbose;
        drop(sandbox); // release the read lock.

        if !allowed {
            if verbose {
                warn!("ctx": "safesetid", "err": libc::EACCES,
                    "sys": request.syscall, "target_gid": target_gid.as_raw(), "source_gid": source_gid.as_raw(),
                    "req": request);
            } else {
                warn!("ctx": "safesetid", "err": libc::EACCES,
                    "sys": request.syscall, "target_gid": target_gid.as_raw(), "source_gid": source_gid.as_raw(),
                    "pid": request.scmpreq.pid);
            }
            return Err(Errno::EACCES);
        }

        // SAFETY: nix version of setgid does not allow -1 as argument.
        if let Err(errno) = Errno::result(unsafe { libc::setgid(target_gid.as_raw()) }) {
            if verbose {
                warn!("ctx": "safesetid", "err": errno as i32,
                    "sys": request.syscall, "target_gid": target_gid.as_raw(), "source_gid": source_gid.as_raw(),
                    "req": request);
            } else {
                warn!("ctx": "safesetid", "err": errno as i32,
                    "sys": request.syscall, "target_gid": target_gid.as_raw(), "source_gid": source_gid.as_raw(),
                    "pid": request.scmpreq.pid);
            }
            return Err(errno);
        } else if safe_drop_cap(caps::Capability::CAP_SETGID).is_err() {
            // SAFETY: We cannot do much on errors,
            // and on panic the thread will be restarted.
            // The best we can do from a security POV is
            // to enter Ghost mode. This is certainly
            // unexpected but it's safe.
            return Err(Errno::EOWNERDEAD);
        }

        // SAFETY: There's no pointer dereference in the access check.
        unsafe { Ok(request.continue_syscall()) }
    })
}

#[allow(clippy::cognitive_complexity)]
pub(crate) fn sys_setreuid(request: UNotifyEventRequest) -> ScmpNotifResp {
    syscall_handler!(request, |request: UNotifyEventRequest| {
        let req = request.scmpreq;

        #[allow(clippy::cast_possible_truncation)]
        #[allow(clippy::cast_possible_wrap)]
        let target_ruid = match req.data.args[0] as i32 {
            -1 => None,
            n if n >= 0 => Some(Uid::from_raw(
                libc::uid_t::try_from(n).or(Err(Errno::EINVAL))?,
            )),
            _ => return Err(Errno::EINVAL),
        };
        #[allow(clippy::cast_possible_truncation)]
        #[allow(clippy::cast_possible_wrap)]
        let target_euid = match req.data.args[1] as i32 {
            -1 => None,
            n if n >= 0 => Some(Uid::from_raw(
                libc::uid_t::try_from(n).or(Err(Errno::EINVAL))?,
            )),
            _ => return Err(Errno::EINVAL),
        };

        if target_ruid.is_none() && target_euid.is_none() {
            return Ok(request.return_syscall(0));
        }

        // getresuid can only fail with EFAULT which should not happen.
        let resuid = getresuid()?;
        let source_ruid = resuid.real;
        let source_euid = resuid.effective;

        let mut change = false;
        if let Some(target_ruid) = target_ruid {
            if u64::from(target_ruid.as_raw()) <= UID_MIN {
                // SAFETY: This is already asserted with the parent
                // seccomp-bpf filter, this is the second layer.
                return Ok(request.fail_syscall(Errno::EACCES));
            } else if source_ruid != target_ruid {
                change = true;
            }
        }
        if let Some(target_euid) = target_euid {
            if u64::from(target_euid.as_raw()) <= UID_MIN {
                // SAFETY: This is already asserted with the parent
                // seccomp-bpf filter, this is the second layer.
                return Ok(request.fail_syscall(Errno::EACCES));
            } else if source_euid != target_euid {
                change = true;
            }
        }

        if !change {
            // SAFETY: There's no pointer dereference in the access check.
            return unsafe { Ok(request.continue_syscall()) };
        }

        let sandbox = request.get_sandbox();
        let verbose = sandbox.verbose;

        // SAFETY: We do not support RUID != EUID
        if let Some(target_ruid) = target_ruid {
            if let Some(target_euid) = target_euid {
                if target_ruid != target_euid {
                    if verbose {
                        warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                            "target_euid": target_euid.as_raw(), "target_ruid": target_ruid.as_raw(),
                            "source_euid": source_euid.as_raw(), "source_ruid": source_ruid.as_raw(),
                            "req": &request);
                    } else {
                        warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                            "target_euid": target_euid.as_raw(), "target_ruid": target_ruid.as_raw(),
                            "source_euid": source_euid.as_raw(), "source_ruid": source_ruid.as_raw(),
                            "pid": request.scmpreq.pid);
                    }
                    return Err(Errno::EACCES);
                }
            }
        }

        let mut allowed = true;
        if let Some(target_ruid) = target_ruid {
            if !sandbox.chk_uid_transit(source_ruid, target_ruid) {
                allowed = false;
            }
        }
        if allowed {
            if let Some(target_euid) = target_euid {
                if !sandbox.chk_uid_transit(source_euid, target_euid) {
                    allowed = false;
                }
            }
        }
        drop(sandbox); // release the read lock.

        let target_ruid = target_ruid.map(|uid| i64::from(uid.as_raw())).unwrap_or(-1);
        let target_euid = target_euid.map(|uid| i64::from(uid.as_raw())).unwrap_or(-1);
        if !allowed {
            if verbose {
                warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                    "target_euid": target_euid, "target_ruid": target_ruid,
                    "source_euid": source_euid.as_raw(), "source_ruid": source_ruid.as_raw(),
                    "req": request);
            } else {
                warn!("ctx": "safesetid", "err": libc::EACCES,
                    "target_euid": target_euid, "target_ruid": target_ruid,
                    "source_euid": source_euid.as_raw(), "source_ruid": source_ruid.as_raw(),
                    "pid": request.scmpreq.pid);
            }
            return Err(Errno::EACCES);
        }

        if let Err(errno) =
            // SAFETY: nix version of setreuid does not allow -1 as argument.
            Errno::result(unsafe {
                libc::syscall(libc::SYS_setreuid, target_ruid, target_euid)
            })
        {
            if verbose {
                warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                    "target_euid": target_euid, "target_ruid": target_ruid,
                    "source_euid": source_euid.as_raw(), "source_ruid": source_ruid.as_raw(),
                    "req": request);
            } else {
                warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                    "target_euid": target_euid, "target_ruid": target_ruid,
                    "source_euid": source_euid.as_raw(), "source_ruid": source_ruid.as_raw(),
                    "pid": request.scmpreq.pid);
            }
            return Err(errno);
        } else if safe_drop_cap(caps::Capability::CAP_SETUID).is_err() {
            // SAFETY: We cannot do much on errors,
            // and on panic the thread will be restarted.
            // The best we can do from a security POV is
            // to enter Ghost mode. This is certainly
            // unexpected but it's safe.
            return Err(Errno::EOWNERDEAD);
        }

        // SAFETY: There's no pointer dereference in the access check.
        unsafe { Ok(request.continue_syscall()) }
    })
}

#[allow(clippy::cognitive_complexity)]
pub(crate) fn sys_setregid(request: UNotifyEventRequest) -> ScmpNotifResp {
    syscall_handler!(request, |request: UNotifyEventRequest| {
        let req = request.scmpreq;

        #[allow(clippy::cast_possible_truncation)]
        #[allow(clippy::cast_possible_wrap)]
        let target_rgid = match req.data.args[0] as i32 {
            -1 => None,
            n if n >= 0 => Some(Gid::from_raw(
                libc::gid_t::try_from(n).or(Err(Errno::EINVAL))?,
            )),
            _ => return Err(Errno::EINVAL),
        };
        #[allow(clippy::cast_possible_truncation)]
        #[allow(clippy::cast_possible_wrap)]
        let target_egid = match req.data.args[1] as i32 {
            -1 => None,
            n if n >= 0 => Some(Gid::from_raw(
                libc::gid_t::try_from(n).or(Err(Errno::EINVAL))?,
            )),
            _ => return Err(Errno::EINVAL),
        };

        if target_rgid.is_none() && target_egid.is_none() {
            return Ok(request.return_syscall(0));
        }

        // getresgid can only fail with EFAULT which should not happen.
        let resgid = getresgid()?;
        let source_rgid = resgid.real;
        let source_egid = resgid.effective;

        let mut change = false;
        if let Some(target_rgid) = target_rgid {
            if u64::from(target_rgid.as_raw()) <= GID_MIN {
                // SAFETY: This is already asserted with the parent
                // seccomp-bpf filter, this is the second layer.
                return Ok(request.fail_syscall(Errno::EACCES));
            } else if source_rgid != target_rgid {
                change = true;
            }
        }
        if let Some(target_egid) = target_egid {
            if u64::from(target_egid.as_raw()) <= GID_MIN {
                // SAFETY: This is already asserted with the parent
                // seccomp-bpf filter, this is the second layer.
                return Ok(request.fail_syscall(Errno::EACCES));
            } else if source_egid != target_egid {
                change = true;
            }
        }

        if !change {
            // SAFETY: There's no pointer dereference in the access check.
            return unsafe { Ok(request.continue_syscall()) };
        }

        let sandbox = request.get_sandbox();
        let verbose = sandbox.verbose;

        // SAFETY: We do not support Rgid != Egid
        if let Some(target_rgid) = target_rgid {
            if let Some(target_egid) = target_egid {
                if target_rgid != target_egid {
                    if verbose {
                        warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                            "target_egid": target_egid.as_raw(), "target_rgid": target_rgid.as_raw(),
                            "source_egid": source_egid.as_raw(), "source_rgid": source_rgid.as_raw(),
                            "req": &request);
                    } else {
                        warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                            "target_egid": target_egid.as_raw(), "target_rgid": target_rgid.as_raw(),
                            "source_egid": source_egid.as_raw(), "source_rgid": source_rgid.as_raw(),
                            "err": request.scmpreq.pid);
                    }
                    return Err(Errno::EACCES);
                }
            }
        }

        let mut allowed = true;
        if let Some(target_rgid) = target_rgid {
            if !sandbox.chk_gid_transit(source_rgid, target_rgid) {
                allowed = false;
            }
        }
        if allowed {
            if let Some(target_egid) = target_egid {
                if !sandbox.chk_gid_transit(source_egid, target_egid) {
                    allowed = false;
                }
            }
        }
        drop(sandbox); // release the read lock.

        let target_rgid = target_rgid.map(|gid| i64::from(gid.as_raw())).unwrap_or(-1);
        let target_egid = target_egid.map(|gid| i64::from(gid.as_raw())).unwrap_or(-1);
        if !allowed {
            if verbose {
                warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                    "target_egid": target_egid, "target_rgid": target_rgid,
                    "source_egid": source_egid.as_raw(), "source_rgid": source_rgid.as_raw(),
                    "req": request);
            } else {
                warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                    "target_egid": target_egid, "target_rgid": target_rgid,
                    "source_egid": source_egid.as_raw(), "source_rgid": source_rgid.as_raw(),
                    "pid": request.scmpreq.pid);
            }
            return Err(Errno::EACCES);
        }

        if let Err(errno) =
            // SAFETY: nix version of setregid does not allow -1 as argument.
            Errno::result(unsafe {
                libc::syscall(libc::SYS_setregid, target_rgid, target_egid)
            })
        {
            if verbose {
                warn!("ctx": "safesetid", "err": errno as i32,
                    "target_egid": target_egid, "target_rgid": target_rgid, "sys": request.syscall,
                    "source_egid": source_egid.as_raw(), "source_rgid": source_rgid.as_raw(),
                    "req": request);
            } else {
                warn!("ctx": "safesetid", "err": errno as i32,
                    "target_egid": target_egid, "target_rgid": target_rgid, "sys": request.syscall,
                    "source_egid": source_egid.as_raw(), "source_rgid": source_rgid.as_raw(),
                    "pid": request.scmpreq.pid);
            }
            return Err(errno);
        } else if safe_drop_cap(caps::Capability::CAP_SETGID).is_err() {
            // SAFETY: We cannot do much on errors,
            // and on panic the thread will be restarted.
            // The best we can do from a security POV is
            // to enter Ghost mode. This is certainly
            // unexpected but it's safe.
            return Err(Errno::EOWNERDEAD);
        }

        // SAFETY: There's no pointer dereference in the access check.
        unsafe { Ok(request.continue_syscall()) }
    })
}

#[allow(clippy::cognitive_complexity)]
pub(crate) fn sys_setresuid(request: UNotifyEventRequest) -> ScmpNotifResp {
    syscall_handler!(request, |request: UNotifyEventRequest| {
        let req = request.scmpreq;

        #[allow(clippy::cast_possible_truncation)]
        #[allow(clippy::cast_possible_wrap)]
        let target_ruid = match req.data.args[0] as i32 {
            -1 => None,
            n if n >= 0 => Some(Uid::from_raw(
                libc::uid_t::try_from(n).or(Err(Errno::EINVAL))?,
            )),
            _ => return Err(Errno::EINVAL),
        };
        #[allow(clippy::cast_possible_truncation)]
        #[allow(clippy::cast_possible_wrap)]
        let target_euid = match req.data.args[1] as i32 {
            -1 => None,
            n if n >= 0 => Some(Uid::from_raw(
                libc::uid_t::try_from(n).or(Err(Errno::EINVAL))?,
            )),
            _ => return Err(Errno::EINVAL),
        };
        #[allow(clippy::cast_possible_truncation)]
        #[allow(clippy::cast_possible_wrap)]
        let target_suid = match req.data.args[2] as i32 {
            -1 => None,
            n if n >= 0 => Some(Uid::from_raw(
                libc::uid_t::try_from(n).or(Err(Errno::EINVAL))?,
            )),
            _ => return Err(Errno::EINVAL),
        };

        if target_ruid.is_none() && target_euid.is_none() && target_suid.is_none() {
            return Ok(request.return_syscall(0));
        }

        // getresuid can only fail with EFAULT which should not happen.
        let resuid = getresuid()?;
        let source_ruid = resuid.real;
        let source_euid = resuid.effective;
        let source_suid = resuid.saved;

        let mut change = false;
        if let Some(target_ruid) = target_ruid {
            if u64::from(target_ruid.as_raw()) <= UID_MIN {
                // SAFETY: This is already asserted with the parent
                // seccomp-bpf filter, this is the second layer.
                return Ok(request.fail_syscall(Errno::EACCES));
            } else if source_ruid != target_ruid {
                change = true;
            }
        }
        if let Some(target_euid) = target_euid {
            if u64::from(target_euid.as_raw()) <= UID_MIN {
                // SAFETY: This is already asserted with the parent
                // seccomp-bpf filter, this is the second layer.
                return Ok(request.fail_syscall(Errno::EACCES));
            } else if source_euid != target_euid {
                change = true;
            }
        }
        if let Some(target_suid) = target_suid {
            if u64::from(target_suid.as_raw()) <= UID_MIN {
                // SAFETY: This is already asserted with the parent
                // seccomp-bpf filter, this is the second layer.
                return Ok(request.fail_syscall(Errno::EACCES));
            } else if source_suid != target_suid {
                change = true;
            }
        }

        if !change {
            // SAFETY: There's no pointer dereference in the access check.
            return unsafe { Ok(request.continue_syscall()) };
        }

        let sandbox = request.get_sandbox();
        let verbose = sandbox.verbose;

        // SAFETY: We do not support RUID != EUID != SUID
        if let Some(target_ruid) = target_ruid {
            if let Some(target_euid) = target_euid {
                if target_ruid != target_euid {
                    if verbose {
                        warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                            "target_suid": target_suid.map(|u| u.as_raw()),
                            "target_euid": target_euid.as_raw(),
                            "target_ruid": target_ruid.as_raw(),
                            "source_euid": source_euid.as_raw(),
                            "source_ruid": source_ruid.as_raw(),
                            "source_suid": source_suid.as_raw(),
                            "req": &request);
                    } else {
                        warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                            "target_suid": target_suid.map(|u| u.as_raw()),
                            "target_euid": target_euid.as_raw(),
                            "target_ruid": target_ruid.as_raw(),
                            "source_euid": source_euid.as_raw(),
                            "source_ruid": source_ruid.as_raw(),
                            "source_suid": source_suid.as_raw(),
                            "pid": request.scmpreq.pid);
                    }
                    return Err(Errno::EACCES);
                }
            }
        }
        if let Some(target_ruid) = target_ruid {
            if let Some(target_suid) = target_suid {
                if target_ruid != target_suid {
                    if verbose {
                        warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                            "target_suid": target_suid.as_raw(),
                            "target_euid": target_euid.map(|u| u.as_raw()),
                            "target_ruid": target_ruid.as_raw(),
                            "source_euid": source_euid.as_raw(),
                            "source_ruid": source_ruid.as_raw(),
                            "source_suid": source_suid.as_raw(),
                            "req": &request);
                    } else {
                        warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                            "target_suid": target_suid.as_raw(),
                            "target_euid": target_euid.map(|u| u.as_raw()),
                            "target_ruid": target_ruid.as_raw(),
                            "source_euid": source_euid.as_raw(),
                            "source_ruid": source_ruid.as_raw(),
                            "source_suid": source_suid.as_raw(),
                            "pid": request.scmpreq.pid);
                    }
                    return Err(Errno::EACCES);
                }
            }
        }
        if let Some(target_euid) = target_euid {
            if let Some(target_suid) = target_suid {
                if target_euid != target_suid {
                    if verbose {
                        warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                            "target_suid": target_suid.as_raw(),
                            "target_euid": target_euid.as_raw(),
                            "target_ruid": target_ruid.map(|u| u.as_raw()),
                            "source_euid": source_euid.as_raw(),
                            "source_ruid": source_ruid.as_raw(),
                            "source_suid": source_suid.as_raw(),
                            "req": &request);
                    } else {
                        warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                            "target_suid": target_suid.as_raw(),
                            "target_euid": target_euid.as_raw(),
                            "target_ruid": target_ruid.map(|u| u.as_raw()),
                            "source_euid": source_euid.as_raw(),
                            "source_ruid": source_ruid.as_raw(),
                            "source_suid": source_suid.as_raw(),
                            "pid": request.scmpreq.pid);
                    }
                    return Err(Errno::EACCES);
                }
            }
        }

        let mut allowed = true;
        if let Some(target_ruid) = target_ruid {
            if !sandbox.chk_uid_transit(source_ruid, target_ruid) {
                allowed = false;
            }
        }
        if allowed {
            if let Some(target_euid) = target_euid {
                if !sandbox.chk_uid_transit(source_euid, target_euid) {
                    allowed = false;
                }
            }
        }
        if allowed {
            if let Some(target_suid) = target_suid {
                if !sandbox.chk_uid_transit(source_suid, target_suid) {
                    allowed = false;
                }
            }
        }
        drop(sandbox); // release the read lock.

        let target_ruid = target_ruid.map(|uid| i64::from(uid.as_raw())).unwrap_or(-1);
        let target_euid = target_euid.map(|uid| i64::from(uid.as_raw())).unwrap_or(-1);
        let target_suid = target_suid.map(|uid| i64::from(uid.as_raw())).unwrap_or(-1);
        if !allowed {
            if verbose {
                warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                    "target_suid": target_suid,
                    "target_euid": target_euid,
                    "target_ruid": target_ruid,
                    "source_euid": source_euid.as_raw(),
                    "source_ruid": source_ruid.as_raw(),
                    "source_suid": source_suid.as_raw(),
                    "req": request);
            } else {
                warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                    "target_suid": target_suid,
                    "target_euid": target_euid,
                    "target_ruid": target_ruid,
                    "source_euid": source_euid.as_raw(),
                    "source_ruid": source_ruid.as_raw(),
                    "source_suid": source_suid.as_raw(),
                    "pid": request.scmpreq.pid);
            }
            return Err(Errno::EACCES);
        }

        // SAFETY: nix version of setresuid does not allow -1 as argument.
        if let Err(errno) = Errno::result(unsafe {
            libc::syscall(libc::SYS_setresuid, target_ruid, target_euid, target_suid)
        }) {
            if verbose {
                warn!("ctx": "safesetid", "err": errno as i32, "sys": request.syscall,
                    "target_suid": target_suid,
                    "target_euid": target_euid,
                    "target_ruid": target_ruid,
                    "source_euid": source_euid.as_raw(),
                    "source_ruid": source_ruid.as_raw(),
                    "source_suid": source_suid.as_raw(),
                    "req": request);
            } else {
                warn!("ctx": "safesetid", "err": errno as i32, "sys": request.syscall,
                    "target_suid": target_suid,
                    "target_euid": target_euid,
                    "target_ruid": target_ruid,
                    "source_euid": source_euid.as_raw(),
                    "source_ruid": source_ruid.as_raw(),
                    "source_suid": source_suid.as_raw(),
                    "pid": request.scmpreq.pid);
            }
            return Err(errno);
        } else if safe_drop_cap(caps::Capability::CAP_SETUID).is_err() {
            // SAFETY: We cannot do much on errors,
            // and on panic the thread will be restarted.
            // The best we can do from a security POV is
            // to enter Ghost mode. This is certainly
            // unexpected but it's safe.
            return Err(Errno::EOWNERDEAD);
        }

        // SAFETY: There's no pointer dereference in the access check.
        unsafe { Ok(request.continue_syscall()) }
    })
}

#[allow(clippy::cognitive_complexity)]
pub(crate) fn sys_setresgid(request: UNotifyEventRequest) -> ScmpNotifResp {
    syscall_handler!(request, |request: UNotifyEventRequest| {
        let req = request.scmpreq;

        #[allow(clippy::cast_possible_truncation)]
        #[allow(clippy::cast_possible_wrap)]
        let target_rgid = match req.data.args[0] as i32 {
            -1 => None,
            n if n >= 0 => Some(Gid::from_raw(
                libc::gid_t::try_from(n).or(Err(Errno::EINVAL))?,
            )),
            _ => return Err(Errno::EINVAL),
        };
        #[allow(clippy::cast_possible_truncation)]
        #[allow(clippy::cast_possible_wrap)]
        let target_egid = match req.data.args[1] as i32 {
            -1 => None,
            n if n >= 0 => Some(Gid::from_raw(
                libc::gid_t::try_from(n).or(Err(Errno::EINVAL))?,
            )),
            _ => return Err(Errno::EINVAL),
        };
        #[allow(clippy::cast_possible_truncation)]
        #[allow(clippy::cast_possible_wrap)]
        let target_sgid = match req.data.args[2] as i32 {
            -1 => None,
            n if n >= 0 => Some(Gid::from_raw(
                libc::gid_t::try_from(n).or(Err(Errno::EINVAL))?,
            )),
            _ => return Err(Errno::EINVAL),
        };

        if target_rgid.is_none() && target_egid.is_none() && target_sgid.is_none() {
            return Ok(request.return_syscall(0));
        }

        // getresgid can only fail with EFAULT which should not happen.
        let resgid = getresgid()?;
        let source_rgid = resgid.real;
        let source_egid = resgid.effective;
        let source_sgid = resgid.saved;

        let mut change = false;
        if let Some(target_rgid) = target_rgid {
            if u64::from(target_rgid.as_raw()) <= GID_MIN {
                // SAFETY: This is already asserted with the parent
                // seccomp-bpf filter, this is the second layer.
                return Ok(request.fail_syscall(Errno::EACCES));
            } else if source_rgid != target_rgid {
                change = true;
            }
        }
        if let Some(target_egid) = target_egid {
            if u64::from(target_egid.as_raw()) <= GID_MIN {
                // SAFETY: This is already asserted with the parent
                // seccomp-bpf filter, this is the second layer.
                return Ok(request.fail_syscall(Errno::EACCES));
            } else if source_egid != target_egid {
                change = true;
            }
        }
        if let Some(target_sgid) = target_sgid {
            if u64::from(target_sgid.as_raw()) <= GID_MIN {
                // SAFETY: This is already asserted with the parent
                // seccomp-bpf filter, this is the second layer.
                return Ok(request.fail_syscall(Errno::EACCES));
            } else if source_sgid != target_sgid {
                change = true;
            }
        }

        if !change {
            // SAFETY: There's no pointer dereference in the access check.
            return unsafe { Ok(request.continue_syscall()) };
        }

        let sandbox = request.get_sandbox();
        let verbose = sandbox.verbose;

        // SAFETY: We do not support Rgid != Egid != Sgid
        if let Some(target_rgid) = target_rgid {
            if let Some(target_egid) = target_egid {
                if target_rgid != target_egid {
                    if verbose {
                        warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                            "target_sgid": target_sgid.map(|u| u.as_raw()),
                            "target_egid": target_egid.as_raw(),
                            "target_rgid": target_rgid.as_raw(),
                            "source_egid": source_egid.as_raw(),
                            "source_rgid": source_rgid.as_raw(),
                            "source_sgid": source_sgid.as_raw(),
                            "req": &request);
                    } else {
                        warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                            "target_sgid": target_sgid.map(|u| u.as_raw()),
                            "target_egid": target_egid.as_raw(),
                            "target_rgid": target_rgid.as_raw(),
                            "source_egid": source_egid.as_raw(),
                            "source_rgid": source_rgid.as_raw(),
                            "source_sgid": source_sgid.as_raw(),
                            "pid": request.scmpreq.pid);
                    }
                    return Err(Errno::EACCES);
                }
            }
        }
        if let Some(target_rgid) = target_rgid {
            if let Some(target_sgid) = target_sgid {
                if target_rgid != target_sgid {
                    if verbose {
                        warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                            "target_sgid": target_sgid.as_raw(),
                            "target_egid": target_egid.map(|u| u.as_raw()),
                            "target_rgid": target_rgid.as_raw(),
                            "source_egid": source_egid.as_raw(),
                            "source_rgid": source_rgid.as_raw(),
                            "source_sgid": source_sgid.as_raw(),
                            "req": &request);
                    } else {
                        warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                            "target_sgid": target_sgid.as_raw(),
                            "target_egid": target_egid.map(|u| u.as_raw()),
                            "target_rgid": target_rgid.as_raw(),
                            "source_egid": source_egid.as_raw(),
                            "source_rgid": source_rgid.as_raw(),
                            "source_sgid": source_sgid.as_raw(),
                            "pid": request.scmpreq.pid);
                    }
                    return Err(Errno::EACCES);
                }
            }
        }
        if let Some(target_egid) = target_egid {
            if let Some(target_sgid) = target_sgid {
                if target_egid != target_sgid {
                    if verbose {
                        warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                            "target_sgid": target_sgid.as_raw(),
                            "target_egid": target_egid.as_raw(),
                            "target_rgid": target_rgid.map(|u| u.as_raw()),
                            "source_egid": source_egid.as_raw(),
                            "source_rgid": source_rgid.as_raw(),
                            "source_sgid": source_sgid.as_raw(),
                            "req": &request);
                    } else {
                        warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                            "target_sgid": target_sgid.as_raw(),
                            "target_egid": target_egid.as_raw(),
                            "target_rgid": target_rgid.map(|u| u.as_raw()),
                            "source_egid": source_egid.as_raw(),
                            "source_rgid": source_rgid.as_raw(),
                            "source_sgid": source_sgid.as_raw(),
                            "pid": request.scmpreq.pid);
                    }
                    return Err(Errno::EACCES);
                }
            }
        }

        let mut allowed = true;
        if let Some(target_rgid) = target_rgid {
            if !sandbox.chk_gid_transit(source_rgid, target_rgid) {
                allowed = false;
            }
        }
        if allowed {
            if let Some(target_egid) = target_egid {
                if !sandbox.chk_gid_transit(source_egid, target_egid) {
                    allowed = false;
                }
            }
        }
        if allowed {
            if let Some(target_sgid) = target_sgid {
                if !sandbox.chk_gid_transit(source_sgid, target_sgid) {
                    allowed = false;
                }
            }
        }
        drop(sandbox); // release the read lock.

        let target_rgid = target_rgid.map(|gid| i64::from(gid.as_raw())).unwrap_or(-1);
        let target_egid = target_egid.map(|gid| i64::from(gid.as_raw())).unwrap_or(-1);
        let target_sgid = target_sgid.map(|gid| i64::from(gid.as_raw())).unwrap_or(-1);
        if !allowed {
            if verbose {
                warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                    "target_sgid": target_sgid,
                    "target_egid": target_egid,
                    "target_rgid": target_rgid,
                    "source_egid": source_egid.as_raw(),
                    "source_rgid": source_rgid.as_raw(),
                    "source_sgid": source_sgid.as_raw(),
                    "req": request);
            } else {
                warn!("ctx": "safesetid", "err": libc::EACCES, "sys": request.syscall,
                    "target_sgid": target_sgid,
                    "target_egid": target_egid,
                    "target_rgid": target_rgid,
                    "source_egid": source_egid.as_raw(),
                    "source_rgid": source_rgid.as_raw(),
                    "source_sgid": source_sgid.as_raw(),
                    "pid": request.scmpreq.pid);
            }
            return Err(Errno::EACCES);
        }

        // SAFETY: nix version of setregid does not allow -1 as argument.
        if let Err(errno) = Errno::result(unsafe {
            libc::syscall(libc::SYS_setresgid, target_rgid, target_egid, target_sgid)
        }) {
            if verbose {
                warn!("ctx": "safesetid", "err": errno as i32, "sys": request.syscall,
                    "target_sgid": target_sgid,
                    "target_egid": target_egid,
                    "target_rgid": target_rgid,
                    "source_egid": source_egid.as_raw(),
                    "source_rgid": source_rgid.as_raw(),
                    "source_sgid": source_sgid.as_raw(),
                    "req": request);
            } else {
                warn!("ctx": "safesetid", "err": errno as i32, "sys": request.syscall,
                    "target_sgid": target_sgid,
                    "target_egid": target_egid,
                    "target_rgid": target_rgid,
                    "source_egid": source_egid.as_raw(),
                    "source_rgid": source_rgid.as_raw(),
                    "source_sgid": source_sgid.as_raw(),
                    "pid": request.scmpreq.pid);
            }
            return Err(errno);
        } else if safe_drop_cap(caps::Capability::CAP_SETGID).is_err() {
            // SAFETY: We cannot do much on errors,
            // and on panic the thread will be restarted.
            // The best we can do from a security POV is
            // to enter Ghost mode. This is certainly
            // unexpected but it's safe.
            return Err(Errno::EOWNERDEAD);
        }

        // SAFETY: There's no pointer dereference in the access check.
        unsafe { Ok(request.continue_syscall()) }
    })
}
