/*
  DB Mixer
  ========
  Description: 
  a DJ Mixer style GUI interface to the DBMix system.

  Copyright (c) 1999, 2000 Robert Michael S Dean

  Author: Robert Michael S Dean
  Version: 1.0


  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public Licensse as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

*/


#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <limits.h>
#include <fcntl.h>
#include <signal.h>
#include <gtk/gtk.h>
#include <sys/shm.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <xmms/plugin.h>
#include <dbchannel.h>
#include <dbaudiolib.h>
#include <dbdebug.h>
#include <dbsoundcard.h>

#include "dbmixer.h"

extern int * channel_indexes;
extern local_channel * local_channels;
extern dbfsd_data * sysdata;
extern channel_widgets * widgets;

extern GtkAdjustment * fader_adj; 

void channel_menu_select(GtkWidget *w, gpointer * data);

/* the following flag is used within the gui callback functions to prevent
   actions from being taken if we are processing a channel update request. For 
   example, pausing a client would call a pause message to be sent during an update. */
int updating_channels_flag = 0;

int update_counter = 0;
float initial_pitch = 0;
int initial_pause = FALSE;

GtkWidget* make_channel_menu(int ch_id)
{
	GtkWidget * menu;
	GtkWidget * item;
	gint i;
	channel_select_data * channeldata;

	menu = gtk_menu_new();
		
	/* add options to channel selector */
	for (i = 0; i < sysdata->num_channels;i++)
	{
		channeldata = (channel_select_data *) malloc(sizeof(channel_select_data));
		/* remember pointers to channeldata for later deletion */
		widgets[ch_id].datalist = g_slist_prepend(widgets[ch_id].datalist,channeldata);

		channeldata->channel = ch_id;
		channeldata->index = i;
		
		item = make_menu_item (local_channels[i].channel_name,
							   GTK_SIGNAL_FUNC(channel_menu_select),
							   channeldata);
		gtk_menu_append (GTK_MENU (menu), item);
	}
	
	gtk_menu_set_active(GTK_MENU(menu),(guint)ch_id);
	channel_indexes[ch_id] = ch_id;

	return menu;
}

/*
  update_channels - updates the gui to reflect the state of the channel structures
  in case another application changed the channel state.
*/
int update_channels(gpointer data)
{
	gint i,a;
	GtkWidget* tempwidget;
	gint doall_flag = 0;
	GSList *templist;

	updating_channels_flag = 1;
	update_counter++;

	if (update_counter == 3)
	{
		update_counter = 0;
		doall_flag = 1;
	}

	for (i = 0; i < sysdata->num_channels; i++)
	{
		/* set state of cue and mute buttons */
		gtk_toggle_button_set_active(widgets[i].mute_button,local_channels[i].mute);
		gtk_toggle_button_set_active(widgets[i].cue_button,local_channels[i].cue);
		gtk_toggle_button_set_active(widgets[i].pause_button,local_channels[i].pause);

		switch (local_channels[i].sampler_state)
		{
			case SAMPLER_OFF:
				/* disable playback buttons */
				gtk_widget_set_sensitive(GTK_WIDGET(widgets[i].playloop_button), FALSE);
				gtk_widget_set_sensitive(GTK_WIDGET(widgets[i].playsingle_button), FALSE);
				break;
			case SAMPLER_READY:
				/* enable plaback buttons */
				gtk_widget_set_sensitive(GTK_WIDGET(widgets[i].playloop_button),TRUE);
				gtk_widget_set_sensitive(GTK_WIDGET(widgets[i].playsingle_button),TRUE);
				gtk_toggle_button_set_active(widgets[i].playloop_button,0);
				/* the single shot button is a pushbutton, and cannot be set active */
				/* 				gtk_toggle_button_set_active(widgets[i].playsingle_button,0); */
				break;
			case SAMPLER_PLAY_LOOP:
			case SAMPLER_PLAY_SINGLE:
				break;
			default:
				
				break;
		}
	
		if (doall_flag)
		{
			/* only update menu if it is not visible/ being used right now */
		   	if (!GTK_WIDGET_VISIBLE(GTK_WIDGET(gtk_option_menu_get_menu(GTK_OPTION_MENU(widgets[i].opt)))))
			{
				/* delete channel data */
				templist = widgets[i].datalist;

				while (templist != NULL)
				{
					free(templist->data);
					templist = g_slist_next(templist);
				}

				g_slist_free(widgets[i].datalist);
				widgets[i].datalist = NULL;

				/* recreate menu in case channel names have changed */
				a = channel_indexes[i];
				tempwidget = gtk_option_menu_get_menu(GTK_OPTION_MENU(widgets[i].opt));
				gtk_option_menu_remove_menu(GTK_OPTION_MENU(widgets[i].opt));
				/* 		gtk_widget_destroy(tempwidget); */
				tempwidget = make_channel_menu(i);
				channel_indexes[i] = a;
				gtk_menu_set_active(GTK_MENU(tempwidget),(guint)channel_indexes[i]);
				gtk_option_menu_set_menu(GTK_OPTION_MENU(widgets[i].opt),tempwidget);	
			}

			/* 		Debug("Setting pitch."); */
			gtk_adjustment_set_value(widgets[i].pitch_adj,
									 local_channels[channel_indexes[i]].user_pitch);
		}
	}
	
	updating_channels_flag = 0;

	return TRUE;
}


/***********************************************************************/
/*             callback functions for dbmix channel widgets            */
/***********************************************************************/

void channel_menu_select(GtkWidget *w, gpointer * data)
{
    channel_select_data * chdata;
	local_channel * ch;

	/* set channel index for this input selector */
	chdata = (channel_select_data *)data;
	channel_indexes[chdata->channel] = chdata->index;

	ch = &(local_channels[channel_indexes[chdata->index]]);

	/* set the gain level for this mixer channel to be that of the 
	   dbfsd channel it refers to */
	gtk_adjustment_set_value(widgets[chdata->channel].adj,
							 normalize_scale(100 - ch->left_gain));
}


void mute_button_clicked(GtkWidget *w, gint * data)
{
	local_channel * ch;
	gint index = (gint)*data;
#if 0
	dbfsd_msg       msg;
#endif

	ch = &(local_channels[channel_indexes[index]]);

	/* Note on removal of use of MUTE messages:
       There is a conflict with using the update_channels function
       and the IPC messages in that the update can occur while 
       the message is in transit. The side effect of this is once
       the channel is muted, it cannot be reliablly unmuted. */

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
	{
#if 0 
		Debug("dbmixer: sending MUTE message");
		msg.msg_type = DBMSG_MUTE;
#endif
		ch->mute = 1;
	}
	else
	{
#if 0
		Debug("dbmixer: sending UNMUTE message.");
		msg.msg_type = DBMSG_UNMUTE;
#endif
		ch->mute = 0;
	}

#if 0
	if (msgsnd(ch->msg_q_id, &msg,sizeof(dbfsd_msg) - sizeof(long int),IPC_NOWAIT) == 0)
	{
		Debug("dbmixer: Message sent.");
	}
	else
	{
	    Error("dbmixer: Message failure.");
	}
#endif
}


void cue_button_clicked(GtkWidget *w, gchar * data)
{
	local_channel * ch;
	gint index = (gint)*data;

	ch = &(local_channels[channel_indexes[index]]);

	ch->cue = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));
}


void playsingle_button_clicked(GtkWidget *w, gint * data)
{
	local_channel * ch;
	gint index = (gint)*data;
	volatile char write_flag;

	ch = &(local_channels[channel_indexes[index]]);

	/* if recording, stop and begin playback */
	if (ch->sampler_state == SAMPLER_RECORD)
	{
		ch->sampler_state = SAMPLER_READY;
	   
		/* spin on writing flag before changing state */

		/* it is possible for DBAudio_Write to exit,
           and reenter without us catching the state change
           of the writing flag. So temporarially pause 
           channel to prevent this. */
		ch->pause = 1;
		write_flag = ch->writing;

		while (write_flag)
		{
			write_flag = ch->writing;
		}

		ch->pause = 0;
	}

	/* if currently playing, this will force the playback to 
       stop at end of next loop iteration */

	/* reset readoffset */
	ch->sampler_readoffset = ch->sampler_startoffset;

	ch->sampler_state = SAMPLER_PLAY_SINGLE;

	Debug("dbmixer: single clicked");
}


void playloop_button_clicked(GtkWidget *w, gint * data)
{
	local_channel * ch;
	gint index = (gint)*data;
	volatile char write_flag;

	ch = &(local_channels[channel_indexes[index]]);

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
	{
		/* if recording, stop and begin playback */
		if (ch->sampler_state == SAMPLER_RECORD)
		{
			ch->sampler_state = SAMPLER_READY;
			
			/* spin on writing flag before changing state */
			
			/* it is possible for DBAudio_Write to exit,
			   and reenter without us catching the state change
			   of the writing flag. So temporarially pause 
			   channel to prevent this. */
			ch->pause = 1;
			write_flag = ch->writing;
			
			while (write_flag)
			{
				write_flag = ch->writing;
			}
			
			ch->pause = 0;
		}

		/* if we are not playing, reset to start of sample */
		if (ch->sampler_state != SAMPLER_PLAY_SINGLE)
		{
			ch->sampler_readoffset = ch->sampler_startoffset;
		}

		ch->sampler_state = SAMPLER_PLAY_LOOP;
	}
	else
	{
		ch->sampler_state = SAMPLER_READY;
	}
}

void reset_pitch_button_clicked(GtkWidget *w, gchar * data)
{
	local_channel * ch;
	gint index = (gint)*data;

	ch = &(local_channels[channel_indexes[index]]);

	ch->user_pitch = 100;

	gtk_adjustment_set_value(widgets[index].pitch_adj,100);
}


void pause_button_clicked(GtkWidget *w, gchar* data)
{
	local_channel * ch;
	dbfsd_msg       msg;
	gint index = (gint)*data;

	if (updating_channels_flag) return;

	ch = &(local_channels[channel_indexes[index]]);

	if (ch->msg_q_id == -1) return;
	
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
	{
		Debug("dbmixer: sending PAUSE message");
		msg.msg_type = DBMSG_PAUSE;
/* 		ch->pause = TRUE; */
	}
	else
	{
		Debug("dbmixer: sending UNPAUSE message.");
		msg.msg_type = DBMSG_UNPAUSE;
/* 		ch->pause = FALSE; */
	}

	if (msgsnd(ch->msg_q_id, &msg,sizeof(dbfsd_msg) - sizeof(long int),IPC_NOWAIT) == 0)
	{
		Debug("dbmixer: Message sent.");
	}
	else
	{
	    Error("dbmixer: Message failure.");
	}
}


/* static void punch_button_pressed(GtkWidget *w, gchar* data) */
/* { */
/* 	local_channel * ch; */
/* 	dbfsd_msg       msg; */
/* 	gint index = (gint)*data; */

/* 	ch = &(local_channels[channel_indexes[index]]);	 */
	
/* 	widgets[index].tempvol = ch->left_gain; */
/* 	ch->left_gain = ch->right_gain =  */
/* 		normalize_scale(100 - widgets[index].adj->value); */
/* } */



/* static void punch_button_released(GtkWidget *w, gchar* data) */
/* { */
/* 	local_channel * ch; */
/* 	dbfsd_msg       msg; */
/* 	gint index = (gint)*data; */

/* 	ch = &(local_channels[channel_indexes[index]]);	 */
	
/* 	ch->left_gain = ch->right_gain = widgets[index].tempvol; */
/* } */


void level_scale_changed(GtkAdjustment * adj)
{
	int i;
	local_channel * ch;
	int value;

	for (i=0; i< sysdata->num_channels; i++)
	{
		if (widgets[i].adj == adj)
		{
			ch = &(local_channels[channel_indexes[i]]);

			value = normalize_scale(100 - adj->value);

			value = (value * DBAUDIO_INTERNAL_MAX_VOLUME) / DBAUDIO_MAX_VOLUME;

			ch->left_gain = ch->right_gain = value;
			ch->cue_left_gain = ch->cue_right_gain = value;
		}
	}

	if (fader_adj != NULL) 
	{
		crossfader_scale_changed(fader_adj); 
	}
}


void pitch_scale_changed(GtkAdjustment * adj)
{
	int i;
	local_channel * ch;

	for (i=0; i< sysdata->num_channels; i++)
	{
		if (widgets[i].pitch_adj == adj)
		{
			ch = &(local_channels[channel_indexes[i]]);
			ch->user_pitch = adj->value;
		}
	}
}

void pitch_plus5_button_clicked(GtkWidget *w, gchar* data)
{
	gint index = (gint)*data;
	float value;

	value = widgets[index].pitch_adj->value + 5.0;

	if (value > PITCH_HIGH_BOUND) value = PITCH_HIGH_BOUND;

	local_channels[channel_indexes[index]].user_pitch = value;

	gtk_adjustment_set_value(widgets[index].pitch_adj,
							 local_channels[channel_indexes[index]].user_pitch);	
}

void pitch_minus5_button_clicked(GtkWidget *w, gchar* data)
{
	gint index = (gint)*data;
	float value;

	value = widgets[index].pitch_adj->value - 5.0;

	if (value < PITCH_LOW_BOUND) value = PITCH_LOW_BOUND;

	local_channels[channel_indexes[index]].user_pitch = value;

	gtk_adjustment_set_value(widgets[index].pitch_adj,
							 local_channels[channel_indexes[index]].user_pitch);
}


void pitch_sync_minus_button_pressed(GtkWidget *w, gchar* data)
{
	local_channel * ch;
	gint index = (gint)*data;

	Debug("pause pressed");

	ch = &(local_channels[channel_indexes[index]]);	

	initial_pitch = ch->user_pitch;
	ch->user_pitch -= PITCH_SYNC_VALUE;
}


void pitch_sync_minus_button_released(GtkWidget *w, gchar* data)
{
	local_channel * ch;
	gint index = (gint)*data;

	ch = &(local_channels[channel_indexes[index]]);	
	
	Debug("pause released");

	ch->user_pitch = initial_pitch;
}

void pitch_sync_pause_button_pressed(GtkWidget *w, gchar* data)
{
	local_channel * ch;
	dbfsd_msg       msg;
	gint index = (gint)*data;

	if (updating_channels_flag) return;

	ch = &(local_channels[channel_indexes[index]]);

	if (ch->msg_q_id == -1) return;

	msg.msg_type = DBMSG_PAUSE;

	if (msgsnd(ch->msg_q_id, &msg,sizeof(dbfsd_msg) - sizeof(long int),IPC_NOWAIT) == 0)
	{
		Debug("dbmixer: Message sent.");
	}
	else
	{
	    Error("dbmixer: Message failure.");
	}

/* 	local_channel * ch; */
/* 	gint index = (gint)*data; */

/* 	ch = &(local_channels[channel_indexes[index]]);	 */

/* 	initial_pause = ch->pause; */
/* 	ch->pause = TRUE; */
}


void pitch_sync_pause_button_released(GtkWidget *w, gchar* data)
{
	local_channel * ch;
	dbfsd_msg       msg;
	gint index = (gint)*data;

	if (updating_channels_flag) return;

	ch = &(local_channels[channel_indexes[index]]);

	if (ch->msg_q_id == -1) return;

	msg.msg_type = DBMSG_UNPAUSE;

	if (msgsnd(ch->msg_q_id, &msg,sizeof(dbfsd_msg) - sizeof(long int),IPC_NOWAIT) == 0)
	{
		Debug("dbmixer: Message sent.");
	}
	else
	{
	    Error("dbmixer: Message failure.");
	}

/* 	local_channel * ch; */
/* 	gint index = (gint)*data; */

/* 	ch = &(local_channels[channel_indexes[index]]);	 */
	
/* 	ch->pause = initial_pause; */
}


void pitch_sync_plus_button_pressed(GtkWidget *w, gchar* data)
{
	local_channel * ch;
	gint index = (gint)*data;

	ch = &(local_channels[channel_indexes[index]]);	

	initial_pitch = ch->user_pitch;
	ch->user_pitch += PITCH_SYNC_VALUE;
}


void pitch_sync_plus_button_released(GtkWidget *w, gchar* data)
{
	local_channel * ch;
	gint index = (gint)*data;

	ch = &(local_channels[channel_indexes[index]]);	
	
	ch->user_pitch = initial_pitch;
}


/***********************************************************************/
/*             widget creation functions for dbmix channels            */
/***********************************************************************/

GtkWidget * create_channel_controls(int ch_id)
{
	/* 	GtkWidget * window; */
	GtkWidget * main_box;           /* main box for the window */
	GtkWidget * other_main_box;
	GtkWidget * channel_box;        /* hbox to hold channel selector */
	GtkWidget * top_button_box;     /* hbox to hold mute pause cue buttons */
	GtkWidget * sampler_box;        /* hbox to hold sampler buttons */
	GtkWidget * scale_box;          /* vbox to hold gain scale and pitch scale box */
	GtkWidget * pitch_box;          /* hbox to hold pitch scale and reset */
	GtkWidget * pitch_buttons_box;  /* hbox to hold pitch buttons */
    GtkWidget * pitch_sync_box;	    /* hbox to hold pitch sync buttons */	
	GtkWidget * menu;               /* pointer to fill in channel selector*/
	GtkWidget * label, * separator;

	Debug("create_channel_controls start. channel id: %d",ch_id);

	/* create main box for the window */
	main_box = (GtkWidget*)gtk_vbox_new(FALSE,0);
	gtk_widget_show(main_box);

	other_main_box = (GtkWidget*)gtk_hbox_new(FALSE,0);
	gtk_widget_show(main_box);

	widgets[ch_id].ch_id = ch_id;
	
	widgets[ch_id].datalist = NULL;

	/* create channel selector */
	{
		channel_box = gtk_hbox_new(TRUE,0);
		gtk_container_set_border_width(GTK_CONTAINER(channel_box),0);

		widgets[ch_id].opt = gtk_option_menu_new();

		menu = make_channel_menu(ch_id);

		/* add option list to channel selector */
		gtk_option_menu_set_menu (GTK_OPTION_MENU (widgets[ch_id].opt), menu);
		gtk_box_pack_start (GTK_BOX (channel_box), widgets[ch_id].opt, FALSE, FALSE, 0);
		gtk_widget_show (widgets[ch_id].opt);
	}

	/* add channel selector to main box*/
	gtk_box_pack_start (GTK_BOX (main_box), channel_box, FALSE, FALSE, 0);
	gtk_widget_show (channel_box);

	separator = gtk_hseparator_new();
	gtk_box_pack_start (GTK_BOX (main_box), separator, FALSE, FALSE, 0);
	gtk_widget_show (separator);


	/* create channel control buttons */
	{
		top_button_box = gtk_hbox_new(FALSE,0);
		gtk_container_set_border_width(GTK_CONTAINER(top_button_box),0);

		/* create mute button */
		widgets[ch_id].mute_button = (GtkToggleButton *)gtk_toggle_button_new();
		label = gtk_label_new(MUTE_STR);
		gtk_container_add(GTK_CONTAINER(widgets[ch_id].mute_button),label);
		gtk_box_pack_start(GTK_BOX(top_button_box),GTK_WIDGET(widgets[ch_id].mute_button),
						   TRUE,FALSE,0);
		gtk_signal_connect (GTK_OBJECT(widgets[ch_id].mute_button), "clicked", 
							GTK_SIGNAL_FUNC(mute_button_clicked),&(widgets[ch_id].ch_id));
		gtk_widget_show(label);
		gtk_widget_show(GTK_WIDGET(widgets[ch_id].mute_button));
   
		gtk_toggle_button_set_active(widgets[ch_id].mute_button,local_channels[ch_id].mute);


		/* create pause button */
		widgets[ch_id].pause_button = (GtkToggleButton *)gtk_toggle_button_new();
		label = gtk_label_new(PAUSE_STR);
		gtk_container_add(GTK_CONTAINER(widgets[ch_id].pause_button),label);
		gtk_box_pack_start(GTK_BOX(top_button_box),GTK_WIDGET(widgets[ch_id].pause_button),
						   TRUE,FALSE,0);
		gtk_signal_connect (GTK_OBJECT(widgets[ch_id].pause_button), "clicked", 
							GTK_SIGNAL_FUNC(pause_button_clicked),&(widgets[ch_id].ch_id));
		gtk_widget_show(label);
		gtk_widget_show(GTK_WIDGET(widgets[ch_id].pause_button));
   
		gtk_toggle_button_set_active(widgets[ch_id].pause_button,local_channels[ch_id].mute);


		/* create cue button */
		widgets[ch_id].cue_button = (GtkToggleButton *)gtk_toggle_button_new();
		label = gtk_label_new(CUE_STR);
		gtk_container_add(GTK_CONTAINER(widgets[ch_id].cue_button),label);
		gtk_box_pack_start(GTK_BOX(top_button_box),(GtkWidget*)widgets[ch_id].cue_button,TRUE,FALSE,0);
		gtk_signal_connect (GTK_OBJECT(widgets[ch_id].cue_button), "clicked", 
							GTK_SIGNAL_FUNC(cue_button_clicked),&(widgets[ch_id].ch_id));

		gtk_toggle_button_set_active(widgets[ch_id].cue_button,local_channels[ch_id].cue);

		gtk_widget_show(label);

		if (!Cue_Enabled(&(local_channels[ch_id])))
		{
			gtk_widget_set_sensitive(GTK_WIDGET(widgets[ch_id].cue_button), FALSE);
		}

		gtk_widget_show((GtkWidget*)widgets[ch_id].cue_button);
	}

	/* add button box to main box */
	gtk_box_pack_start (GTK_BOX (main_box), top_button_box, FALSE, FALSE, 0);
	gtk_widget_show (top_button_box);

	separator = gtk_hseparator_new();
	gtk_box_pack_start (GTK_BOX (main_box), separator, FALSE, FALSE, 0);
	gtk_widget_show (separator);

	/* create sampler buttons */
	{
		sampler_box = gtk_hbox_new(FALSE,0);
		gtk_container_set_border_width(GTK_CONTAINER(sampler_box),0);

		label = gtk_label_new("Samp:");
		gtk_box_pack_start(GTK_BOX(sampler_box),label,TRUE,FALSE,0);
		gtk_widget_show(label);

		/* playsingle button */
		widgets[ch_id].playsingle_button = (GtkButton *)gtk_button_new();
		label = gtk_label_new(PLAYSINGLE_STR);
		gtk_container_add(GTK_CONTAINER(widgets[ch_id].playsingle_button),label);
		gtk_box_pack_start(GTK_BOX(sampler_box),GTK_WIDGET(widgets[ch_id].playsingle_button),
						   TRUE,FALSE,0);
		gtk_signal_connect (GTK_OBJECT(widgets[ch_id].playsingle_button), "clicked", 
							GTK_SIGNAL_FUNC(playsingle_button_clicked),&(widgets[ch_id].ch_id));
		gtk_widget_show(label);
		gtk_widget_show(GTK_WIDGET(widgets[ch_id].playsingle_button));

		/* playloop button */
		widgets[ch_id].playloop_button = (GtkToggleButton *)gtk_toggle_button_new();
		label = gtk_label_new(PLAYLOOP_STR);
		gtk_container_add(GTK_CONTAINER(widgets[ch_id].playloop_button),label);
		gtk_box_pack_start(GTK_BOX(sampler_box),GTK_WIDGET(widgets[ch_id].playloop_button),
						   TRUE,FALSE,0);
		gtk_signal_connect (GTK_OBJECT(widgets[ch_id].playloop_button), "clicked", 
							GTK_SIGNAL_FUNC(playloop_button_clicked),&(widgets[ch_id].ch_id));
		gtk_widget_show(label);
		gtk_widget_show(GTK_WIDGET(widgets[ch_id].playloop_button));
   		
	}

	/* add button box to main box */
	gtk_box_pack_start (GTK_BOX (main_box), sampler_box, FALSE, FALSE, 0);
	gtk_widget_show (sampler_box);

	separator = gtk_hseparator_new();
	gtk_box_pack_start (GTK_BOX (main_box), separator, FALSE, FALSE, 0);
	gtk_widget_show (separator);


	/* create gain and pitch sliders */
	{
		scale_box = gtk_vbox_new(FALSE,0);
		
		gtk_container_set_border_width(GTK_CONTAINER(scale_box),5);

/* 		label = gtk_label_new(GAIN_STR); */
/* 		gtk_box_pack_start(GTK_BOX(scale_box),label,FALSE,FALSE,0); */
		
/* 		gtk_widget_show(label);		 */

		/* create gain scale */
		{
			widgets[ch_id].adj = (GtkAdjustment*) gtk_adjustment_new(0.0,0.0,101.0,1.0,10.0,1.0);
			gtk_signal_connect (GTK_OBJECT(widgets[ch_id].adj), "value_changed", 
								GTK_SIGNAL_FUNC(level_scale_changed),NULL);
			
			widgets[ch_id].level_scale = (GtkVScale*)gtk_vscale_new(GTK_ADJUSTMENT(widgets[ch_id].adj));
			gtk_scale_set_digits((GtkScale*)widgets[ch_id].level_scale,0);
			
			gtk_scale_set_value_pos((GtkScale*)widgets[ch_id].level_scale,GTK_POS_TOP);
			
			gtk_box_pack_start(GTK_BOX(scale_box),(GtkWidget*)widgets[ch_id].level_scale,FALSE,FALSE,0);
			gtk_widget_show((GtkWidget*)widgets[ch_id].level_scale);
			
			if (local_channels[channel_indexes[ch_id]].left_gain > 0)
			{
				gtk_adjustment_set_value(widgets[ch_id].adj,
										 normalize_scale(100 - ((local_channels[channel_indexes[ch_id]].left_gain * DBAUDIO_MAX_VOLUME) / DBAUDIO_INTERNAL_MAX_VOLUME)));
			}
			else
			{
				gtk_adjustment_set_value(widgets[ch_id].adj,DEFAULT_GAIN);
				local_channels[channel_indexes[ch_id]].left_gain = DEFAULT_GAIN;
				local_channels[channel_indexes[ch_id]].right_gain = DEFAULT_GAIN;
			}
			
			gtk_widget_set_usize((GtkWidget*)widgets[ch_id].level_scale,10,110);
		}

		/* add pitch slider and buttons */
		{
			pitch_box = (GtkWidget*) gtk_vbox_new(FALSE,0);
			pitch_buttons_box = (GtkWidget *) gtk_hbox_new(TRUE,0);

			/* create pitch scale */
			widgets[ch_id].pitch_adj = (GtkAdjustment*) gtk_adjustment_new(0.0,PITCH_LOW_BOUND,PITCH_HIGH_BOUND,0.1,1.0,0.0);

			gtk_signal_connect (GTK_OBJECT(widgets[ch_id].pitch_adj), "value_changed", 
								GTK_SIGNAL_FUNC(pitch_scale_changed),NULL);

			widgets[ch_id].pitch_scale = (GtkHScale*)gtk_hscale_new(GTK_ADJUSTMENT(widgets[ch_id].pitch_adj));
			gtk_scale_set_digits((GtkScale*)widgets[ch_id].pitch_scale,1);

			gtk_box_pack_start(GTK_BOX(pitch_box),(GtkWidget*)widgets[ch_id].pitch_scale,TRUE,TRUE,5);
			gtk_widget_show((GtkWidget*)widgets[ch_id].pitch_scale);

			gtk_adjustment_set_value(widgets[ch_id].pitch_adj,
									 local_channels[channel_indexes[ch_id]].user_pitch);

			/* create pitch buttons */
			{
				/* create pitch -5% button */
				widgets[ch_id].pitch_minus5_button = (GtkButton *)gtk_button_new();
				label = gtk_label_new(PITCH_MINUS5_STR);
				gtk_container_add(GTK_CONTAINER(widgets[ch_id].pitch_minus5_button),label);
				gtk_box_pack_start(GTK_BOX(pitch_buttons_box),
								   GTK_WIDGET(widgets[ch_id].pitch_minus5_button),
								   FALSE,FALSE,0);
				gtk_signal_connect (GTK_OBJECT(widgets[ch_id].pitch_minus5_button), "clicked", 
									GTK_SIGNAL_FUNC(pitch_minus5_button_clicked),&(widgets[ch_id].ch_id));
				gtk_widget_show(label);
				gtk_widget_show(GTK_WIDGET(widgets[ch_id].pitch_minus5_button));	

				/* create pitch center button */
				widgets[ch_id].reset_pitch_button = (GtkButton *)gtk_button_new();
				label = gtk_label_new(RESET_PITCH_STR);
				gtk_container_add(GTK_CONTAINER(widgets[ch_id].reset_pitch_button),label);
				gtk_box_pack_start(GTK_BOX(pitch_buttons_box),
								   GTK_WIDGET(widgets[ch_id].reset_pitch_button),
								   FALSE,FALSE,0);
				gtk_signal_connect (GTK_OBJECT(widgets[ch_id].reset_pitch_button), "clicked", 
									GTK_SIGNAL_FUNC(reset_pitch_button_clicked),&(widgets[ch_id].ch_id));
				gtk_widget_show(label);
				gtk_widget_show(GTK_WIDGET(widgets[ch_id].reset_pitch_button));		

				/* create pitch +5% button */
				widgets[ch_id].pitch_plus5_button = (GtkButton *)gtk_button_new();
				label = gtk_label_new(PITCH_PLUS5_STR);
				gtk_container_add(GTK_CONTAINER(widgets[ch_id].pitch_plus5_button),label);
				gtk_box_pack_start(GTK_BOX(pitch_buttons_box),
								   GTK_WIDGET(widgets[ch_id].pitch_plus5_button),
								   FALSE,FALSE,0);
				gtk_signal_connect (GTK_OBJECT(widgets[ch_id].pitch_plus5_button), "clicked", 
									GTK_SIGNAL_FUNC(pitch_plus5_button_clicked),&(widgets[ch_id].ch_id));
				gtk_widget_show(label);
				gtk_widget_show(GTK_WIDGET(widgets[ch_id].pitch_plus5_button));	
			}

			/* create sync buttons */
			{
				pitch_sync_box = (GtkWidget *) gtk_hbox_new(TRUE,0);

				/* create pitch sync minus button */
				widgets[ch_id].pitch_sync_minus_button = (GtkButton *)gtk_button_new();
				label = gtk_label_new(SYNC_MINUS);

				gtk_container_add(GTK_CONTAINER(widgets[ch_id].pitch_sync_minus_button),label);

				gtk_box_pack_start(GTK_BOX(pitch_sync_box),
								   GTK_WIDGET(widgets[ch_id].pitch_sync_minus_button),
								   FALSE,FALSE,0);

				gtk_signal_connect (GTK_OBJECT(widgets[ch_id].pitch_sync_minus_button), "pressed", 
									GTK_SIGNAL_FUNC(pitch_sync_minus_button_pressed),
									&(widgets[ch_id].ch_id));
				
				gtk_signal_connect (GTK_OBJECT(widgets[ch_id].pitch_sync_minus_button), "released", 
									GTK_SIGNAL_FUNC(pitch_sync_minus_button_released),
									&(widgets[ch_id].ch_id));

				gtk_widget_show(label);
				gtk_widget_show(GTK_WIDGET(widgets[ch_id].pitch_sync_minus_button)); 

				/* create pitch sync pause button */
				widgets[ch_id].pitch_sync_pause_button = (GtkButton *)gtk_button_new();
				label = gtk_label_new(SYNC_PAUSE);

				gtk_container_add(GTK_CONTAINER(widgets[ch_id].pitch_sync_pause_button),label);

				gtk_box_pack_start(GTK_BOX(pitch_sync_box),
								   GTK_WIDGET(widgets[ch_id].pitch_sync_pause_button),
								   FALSE,FALSE,0);

				gtk_signal_connect (GTK_OBJECT(widgets[ch_id].pitch_sync_pause_button), "pressed", 
									GTK_SIGNAL_FUNC(pitch_sync_pause_button_pressed),
									&(widgets[ch_id].ch_id));
				
				gtk_signal_connect (GTK_OBJECT(widgets[ch_id].pitch_sync_pause_button), "released", 
									GTK_SIGNAL_FUNC(pitch_sync_pause_button_released),
									&(widgets[ch_id].ch_id));

				gtk_widget_show(label);
				gtk_widget_show(GTK_WIDGET(widgets[ch_id].pitch_sync_pause_button)); 

				/* create pitch sync plus button */
				widgets[ch_id].pitch_sync_plus_button = (GtkButton *)gtk_button_new();
				label = gtk_label_new(SYNC_PLUS);

				gtk_container_add(GTK_CONTAINER(widgets[ch_id].pitch_sync_plus_button),label);

				gtk_box_pack_start(GTK_BOX(pitch_sync_box),
								   GTK_WIDGET(widgets[ch_id].pitch_sync_plus_button),
								   FALSE,FALSE,0);

				gtk_signal_connect (GTK_OBJECT(widgets[ch_id].pitch_sync_plus_button), "pressed", 
									GTK_SIGNAL_FUNC(pitch_sync_plus_button_pressed),
									&(widgets[ch_id].ch_id));
				
				gtk_signal_connect (GTK_OBJECT(widgets[ch_id].pitch_sync_plus_button), "released", 
									GTK_SIGNAL_FUNC(pitch_sync_plus_button_released),
									&(widgets[ch_id].ch_id));

				gtk_widget_show(label);
				gtk_widget_show(GTK_WIDGET(widgets[ch_id].pitch_sync_plus_button));
			}

			/* add to scale box */
			gtk_box_pack_start(GTK_BOX(pitch_box),pitch_buttons_box,FALSE,FALSE,0);
			gtk_box_pack_start(GTK_BOX(scale_box),pitch_box,FALSE,FALSE,0);

			separator = gtk_hseparator_new();
			gtk_box_pack_start (GTK_BOX (scale_box), separator, FALSE, FALSE, 2);
			gtk_widget_show (separator);

			gtk_box_pack_start(GTK_BOX(scale_box),pitch_sync_box,FALSE,FALSE,0);
			gtk_widget_show(pitch_sync_box);
			gtk_widget_show(pitch_box);
			gtk_widget_show(pitch_buttons_box);
		}
	}
	
	/* add scale box to main box */
	gtk_box_pack_start (GTK_BOX (main_box), scale_box, FALSE, FALSE, 0);
	gtk_widget_show (scale_box);

/* 	separator = gtk_hseparator_new(); */
/* 	gtk_box_pack_start (GTK_BOX (main_box), separator, FALSE, FALSE, 0); */
/* 	gtk_widget_show (separator); */

	gtk_box_pack_start(GTK_BOX(other_main_box),main_box,FALSE,FALSE,0);
	separator = gtk_vseparator_new();
	gtk_box_pack_start(GTK_BOX(other_main_box),separator,FALSE,FALSE,0);
	gtk_widget_show(separator);

	Debug("create_channel_controls end");

	return other_main_box;
}

