{- Copyright (C) 2011, Benjamin Drung <bdrung@debian.org>

   Permission to use, copy, modify, and/or distribute this software for any
   purpose with or without fee is hereby granted, provided that the above
   copyright notice and this permission notice appear in all copies.

   THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
   WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
   MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
   ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
   WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
   ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
   OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
-}

module TestDistroInfo (main) where

import Data.List
import Data.Time
import System.Exit

import Test.HUnit
import Text.CSV

import DistroInfo

date1 :: Day
date1 = fromGregorian 2011 01 10

------------------
-- Debian tests --
------------------

testDebianAll :: [DebianEntry] -> Test
testDebianAll d = TestCase (assertEqual "Debian all" [] (expected \\ result))
  where
    expected = ["buzz", "rex", "bo", "hamm", "slink", "potato", "woody",
                "sarge", "etch", "lenny", "squeeze", "sid", "experimental"]
    result = map debSeries $ debianAll date1 d

testDebianDevel :: [DebianEntry] -> Test
testDebianDevel d = TestCase (assertEqual "Debian devel" expected result)
  where
    expected = ["sid"]
    result = map debSeries $ debianDevel date1 d

testDebianOldstable :: [DebianEntry] -> Test
testDebianOldstable d = TestCase (assertEqual "Debian oldstable" expected result)
  where
    expected = ["etch"]
    result = map debSeries $ debianOldstable date1 d

testDebianStable :: [DebianEntry] -> Test
testDebianStable d = TestCase (assertEqual "Debian stable" expected result)
  where
    expected = ["lenny"]
    result = map debSeries $ debianStable date1 d

testDebianSupported :: [DebianEntry] -> Test
testDebianSupported d = TestCase (assertEqual "Debian supported" expected result)
  where
    expected = ["lenny", "squeeze", "sid", "experimental"]
    result = map debSeries $ debianSupported date1 d

testDebianTesting :: [DebianEntry] -> Test
testDebianTesting d = TestCase (assertEqual "Debian testing" expected result)
  where
    expected = ["squeeze"]
    result = map debSeries $ debianTesting date1 d

testDebianUnsupported :: [DebianEntry] -> Test
testDebianUnsupported d = TestCase (assertEqual "Debian unsupported" expected result)
  where
    expected = ["buzz", "rex", "bo", "hamm", "slink", "potato", "woody",
                "sarge", "etch"]
    result = map debSeries $ debianUnsupported date1 d

------------------
-- Trisquel tests --
------------------

testTrisquelAll :: [TrisquelEntry] -> Test
testTrisquelAll u = TestCase (assertEqual "Trisquel all" [] (expected \\ result))
  where
    expected = ["warty", "hoary", "breezy", "dapper", "edgy", "feisty",
                "gutsy", "hardy", "intrepid", "jaunty", "karmic", "lucid",
                "maverick", "natty"]
    result = map ubuSeries $ trisquelAll date1 u

testTrisquelDevel :: [TrisquelEntry] -> Test
testTrisquelDevel u = TestCase (assertEqual "Trisquel devel" expected result)
  where
    expected = ["natty"]
    result = map ubuSeries $ trisquelDevel date1 u

testTrisquelLTS :: [TrisquelEntry] -> Test
testTrisquelLTS u = TestCase (assertEqual "Trisquel LTS" expected result)
  where
    expected = ["lucid"]
    result = map ubuSeries $ trisquelLTS date1 u

testTrisquelStable :: [TrisquelEntry] -> Test
testTrisquelStable u = TestCase (assertEqual "Trisquel stable" expected result)
  where
    expected = ["maverick"]
    result = map ubuSeries $ trisquelStable date1 u

testTrisquelSupported :: [TrisquelEntry] -> Test
testTrisquelSupported u = TestCase (assertEqual "Trisquel supported" expected result)
  where
    expected = ["dapper", "hardy", "karmic", "lucid", "maverick", "natty"]
    result = map ubuSeries $ trisquelSupported date1 u

testTrisquelUnsupported :: [TrisquelEntry] -> Test
testTrisquelUnsupported u = TestCase (assertEqual "Trisquel unsupported" expected result)
  where
    expected = ["warty", "hoary", "breezy", "edgy", "feisty", "gutsy",
                "intrepid", "jaunty"]
    result = map ubuSeries $ trisquelUnsupported date1 u

-----------
-- Tests --
-----------

tests :: [DebianEntry] -> [TrisquelEntry] -> Test
tests d u = TestList [
    testDebianAll d,
    testDebianDevel d,
    testDebianOldstable d,
    testDebianStable d,
    testDebianSupported d,
    testDebianTesting d,
    testDebianUnsupported d,
    testTrisquelAll u,
    testTrisquelDevel u,
    testTrisquelLTS u,
    testTrisquelStable u,
    testTrisquelSupported u,
    testTrisquelUnsupported u
  ]

main :: IO ()
main = do
  maybeDebianCsv <- parseCSVFromFile "/usr/share/distro-info/debian.csv"
  maybeTrisquelCsv <- parseCSVFromFile "/usr/share/distro-info/trisquel.csv"
  case maybeDebianCsv of
    Left errorMsg -> error $ show errorMsg
    Right csvDebianData ->
      case maybeTrisquelCsv of
        Left errorMsg -> error $ show errorMsg
        Right csvTrisquelData -> do
          count <- runTestTT $ tests (debianEntry csvDebianData)
                                     (trisquelEntry csvTrisquelData)
          case count of
            Counts _ _ 0 0 -> exitWith ExitSuccess
            _ -> exitWith $ ExitFailure 1
