# SPDX-License-Identifier: GPL-3.0-or-later
"""
Base worker class to run various commands that build the image.
"""

import logging
import os

from . import internal, library

DISTRO_BRAND = 'Trisquel'

# initramfs-tools is a dependency for the kernel-image package. However, when
# kernel is not installed, as in case of Raspberry Pi image, explicit
# dependency is needed.
BASE_PACKAGES = [
    'initramfs-tools',
]

logger = logging.getLogger(__name__)  # pylint: disable=invalid-name


class ImageBuilder(object):  # pylint: disable=too-many-instance-attributes
    """Base for all image builders."""
    architecture = None
    machine = 'all'
    free = True

    builder_backend = 'internal'
    partition_table_type = 'msdos'
    root_filesystem_type = 'btrfs'
    boot_filesystem_type = None
    boot_size = None
    efi_filesystem_type = None
    efi_size = None
    firmware_filesystem_type = None
    firmware_size = None
    kernel_flavor = 'default'
    debootstrap_variant = None

    extra_storage_size = '1000M'

    @classmethod
    def get_target_name(cls):
        """Return the command line name of target for this builder."""
        return None

    @classmethod
    def get_builder_class(cls, target):
        """Return an builder class given target name."""
        from . import builders  # noqa: F401, pylint: disable=unused-variable

        for subclass in cls.get_subclasses():
            if subclass.get_target_name() == target:
                return subclass

        raise ValueError('No such target')

    @classmethod
    def get_subclasses(cls):
        """Iterate through the subclasses of this class."""
        for subclass in cls.__subclasses__():
            yield subclass
            yield from subclass.get_subclasses()

    def __init__(self, arguments):
        """Initialize object."""
        self.arguments = arguments
        self.packages = BASE_PACKAGES
        self.ram_directory = None

        self.builder_backends = {}
        self.builder_backends['internal'] = internal.InternalBuilderBackend(
            self)

        self.image_file = os.path.join(self.arguments.build_dir,
                                       self._get_image_base_name() + '.img')

    @property
    def release_components(self):
        """Return the Debian release components to use for the build."""
        components = ['main']
        if not self.free:
            components += ['contrib', 'non-free']

        if self.arguments.release_component:
            for component in self.arguments.release_component:
                if component not in components:
                    components.append(component)

        return components

    def build(self):
        """Run the image building process."""
        archive_file = self.image_file + '.xz'
        self.make_image()
        self.compress(archive_file, self.image_file)

        self.sign(archive_file)

    def _get_builder_backend(self):
        """Returns the builder backend."""
        builder = self.builder_backend
        return self.builder_backends[builder]

    def make_image(self):
        """Call a builder backend to create basic image."""
        self._get_builder_backend().make_image()

    def _get_image_base_name(self):
        """Return the base file name of the final image."""
        if self.free:
            # base image is free, but check added components:
            if 'non-free' in self.release_components:
                free_tag = 'nonfree'
            elif 'contrib' in self.release_components:
                free_tag = 'contrib'
            else:
                free_tag = 'libre'
        else:
            free_tag = 'libre'

        build_stamp = self.arguments.build_stamp
        build_stamp = build_stamp + '_' if build_stamp else ''
        return '{distro}freedombox-{distribution}-{free_tag}_{build_stamp}_{machine}' \
            '-{architecture}'.format(
                distro=DISTRO_BRAND.lower() + '-' if DISTRO_BRAND else '',
                distribution=self.arguments.distribution, free_tag=free_tag,
                build_stamp=build_stamp, machine=self.machine,
                architecture=self.architecture)

    def compress(self, archive_file, image_file):
        """Compress the generated image."""
        if not self.arguments.skip_compression:
            library.compress(archive_file, image_file)
        else:
            logger.info('Skipping image compression')

    def sign(self, archive):
        """Sign the final output image."""
        if not self.arguments.sign:
            return

        library.sign(archive)

    @staticmethod
    def _replace_extension(file_name, new_extension):
        """Replace a file's extension with a new extention."""
        return file_name.rsplit('.', maxsplit=1)[0] + new_extension
