package main

import (
	"bufio"
	"bytes"
	"fmt"
	"io"
	"os"
	"strings"
	"text/template"
	"unicode"
	"unicode/utf8"

	"github.com/constabulary/gb/cmd"
)

var helpTemplate = `{{if .Runnable}}usage: gb vendor {{.UsageLine}}

{{end}}{{.Long | trim}}
`

// help implements the 'help' command.
func help(args []string) {
	if len(args) == 0 {
		printUsage(os.Stdout)
		return
	}
	if len(args) != 1 {
		fmt.Fprintf(os.Stderr, "usage: gb vendor help command\n\nToo many arguments given.\n")
		os.Exit(2)
	}

	arg := args[0]

	// 'gb vendor help documentation' generates alldocs.go.
	if arg == "documentation" {
		var buf bytes.Buffer
		printUsage(&buf)
		usage := &cmd.Command{Long: buf.String()}
		f, _ := os.Create("alldocs.go")
		tmpl(f, documentationTemplate, append([]*cmd.Command{usage}, commands...))
		f.Close()
		return
	}

	for _, cmd := range commands {
		if cmd.Name == arg {
			tmpl(os.Stdout, helpTemplate, cmd)
			// not exit 2: succeeded at 'gb help cmd'.
			return
		}
	}

	fmt.Fprintf(os.Stderr, "Unknown help topic %#q. Run 'gb help'.\n", arg)
	os.Exit(2) // failed at 'gb help cmd'
}

var usageTemplate = `gb-vendor, a gb plugin to manage your vendored dependencies.

Usage:

        gb vendor command [arguments]

The commands are:
{{range .}}{{if .Runnable}}
        {{.Name | printf "%-11s"}} {{.Short}}{{end}}{{end}}

Use "gb vendor help [command]" for more information about a command.

Additional help topics:
{{range .}}{{if not .Runnable}}
        {{.Name | printf "%-11s"}} {{.Short}}{{end}}{{end}}

Use "gb vendor help [topic]" for more information about that topic.
`

var documentationTemplate = `// DO NOT EDIT THIS FILE.
//go:generate gb vendor help documentation

/*
{{range .}}{{if .Short}}{{.Short | capitalize}}

{{end}}{{if .Runnable}}Usage:

        gb vendor {{.UsageLine}}

{{end}}{{.Long | trim}}


{{end}}*/
package main
`

// tmpl executes the given template text on data, writing the result to w.
func tmpl(w io.Writer, text string, data interface{}) {
	t := template.New("top")
	t.Funcs(template.FuncMap{"trim": strings.TrimSpace, "capitalize": capitalize})
	template.Must(t.Parse(text))
	if err := t.Execute(w, data); err != nil {
		panic(err)
	}
}

func capitalize(s string) string {
	if s == "" {
		return s
	}
	r, n := utf8.DecodeRuneInString(s)
	return string(unicode.ToTitle(r)) + s[n:]
}

func printUsage(w io.Writer) {
	bw := bufio.NewWriter(w)
	tmpl(bw, usageTemplate, commands)
	bw.Flush()
}

func usage() {
	printUsage(os.Stderr)
	os.Exit(2)
}
