/***************************************************************************
*   Copyright (C) 2007-2010 by Thomas Thelliez aka jblud                  *
*   Contact : <admin.kontrol@gmail.com>                                   *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as               *
*   published by the Free Software Foundation; either version 2.0 of      *
*   the License, or (at your option) any later version.                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
*   General Public License for more details.                              *
*                                                                         *
*   You should have received a copy of the GNU General Public             *
*   License along with this program; if not, write to                     *
*   the Free Software Foundation, Inc.,                                   *
*   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA          *
***************************************************************************/

#include "connection_controller.h"

ConnectionController::ConnectionController(QObject *parent)
    : QObject(parent)
{
    path = QDir::homePath();
#ifdef WIN32
    slash = QString("\\");
    path.replace(QString("/"), QString("\\"));
    Logger::setFileName(QDir::toNativeSeparators(QDir::homePath()) + "\\" + QString(LOG_FILENAME));
#else
    Logger::setFileName(QDir::toNativeSeparators(QDir::homePath()) + "/" + QString(LOG_FILENAME));
    slash = QString("/");
#endif
    xml_creator = new XmlCreatorServer(this);
    settings = new QSettings("OpenNetwork", "KontrolPack", this);
    exec = new Exec("", 0);

    m_logger = new Logger(this);
}

int ConnectionController::start_server(QString password, QString ip, int port, int f_port)
{
    master_server = new auth_server(password);
    if (!master_server->listen(QHostAddress(ip), port)) {
        return 0;
    }
    file_server = new auth_server(password);
    if (!file_server->listen(QHostAddress(ip), f_port)) {
        master_server->stop_server();
        return 0;
    }
    file_manager = new FileManagerServer(file_server, this);
    QObject::connect(file_manager, SIGNAL(tell_client_transfer_state(int, int)), this, SLOT(tell_client_transfer_state_slot(int, int)));
    QObject::connect(master_server, SIGNAL(signals_state(int, int)), this, SLOT(tcp_state(int, int)));
    QObject::connect(master_server, SIGNAL(auth_succeeded(int)), this, SLOT(auth_succeeded_slots(int)));
    QObject::connect(master_server, SIGNAL(received_stream_signal(QString, int)), this, SLOT(received_stream_slot(QString, int)));
    QObject::connect(file_server, SIGNAL(signals_state(int, int)), this, SLOT(tcp_state(int, int)));
    QObject::connect(file_server, SIGNAL(received_stream_signal(QString, int)), file_manager, SLOT(received_stream_files_slot(QString, int)));
    QObject::connect(master_server, SIGNAL(emit_error(int)), this, SLOT(master_client_error_slot(int)));
    QObject::connect(file_server, SIGNAL(emit_error(int)), this, SLOT(file_client_error_slot(int)));
    m_logger->info("Server started : " + ip + " " + QString::number(port) + " " + QString::number(f_port));
    return 1;
}

void ConnectionController::stop_server()
{
    master_server->stop_server();
    file_server->stop_server();
    m_logger->info("Server stopped");
}

/*
    #define TOKEN_WELCOME 0
    #define RSA_PUBLIC_KEY_EXCHANGE 1
    #define PASSWORD_VALIDATION 2
    #define AES_KEY_RECEPTION 3
*/
void ConnectionController::tcp_state(int state, int client_nbr)
{
    switch (state)
    {
    case TOKEN_WELCOME : {
            emit signals_state(state, client_nbr);
            m_logger->debug("TOKEN WELCOME : " + QString::number(client_nbr));
            break;
        }
    case RSA_PUBLIC_KEY_RECEPTION : {
            emit signals_state(state, client_nbr);
            m_logger->debug("RSA PUBLIC KEY RECEPTION : " + QString::number(client_nbr));
            break;
        }
    case PASSWORD_VALIDATION : {
            emit signals_state(state, client_nbr);
            m_logger->debug("PASSWORD VALIDATION : " + QString::number(client_nbr));
            break;
        }
    case AES_KEY_RECEPTION : {
            emit signals_state(state, client_nbr);
            m_logger->debug("AES KEY RECEPTION : " + QString::number(client_nbr));
            break;
        }
    case AUTH_SUCCESS : {
            emit signals_state(state, client_nbr);
            m_logger->debug("AUTH SUCCESS : " + QString::number(client_nbr));
            break;
        }
    case 5 : {
            emit signals_state(state, client_nbr);
            break;
        }
    default:
        ;
    }
}

/*
    Connect the app to the received_stream_signalis not opened
    to be aware of reception and handle streams.
*/
void ConnectionController::received_stream_slot(QString message, int client_nbr)
{
    // Stop thread to not saturate sockets.
    stop_shell_thread();

    // Send parameters : first connexion
    if (message.contains(ASK_PARAMETERS, Qt::CaseInsensitive)) {
        QStringList *list = new QStringList;
        QList<QNetworkInterface> interface_list = QNetworkInterface::allInterfaces();
        for (int k = 0; k < interface_list.size(); k++) {
            if (interface_list.at(k).flags().testFlag(QNetworkInterface::IsRunning))
                foreach (QNetworkAddressEntry entry, interface_list.at(k).addressEntries()) {
                if ( interface_list.at(k).hardwareAddress() != "00:00:00:00:00:00" && entry.ip().toString().contains(".")) {
                    list->append(entry.ip().toString());
                }
            }
        }
        QString homePath = QDir::homePath();
#ifdef WIN32
        homePath.replace(QString("/"), QString("\\"));
#endif
        QString xml = xml_creator->createXmlParameters(homePath,  list);
        master_server->write_encrypted_data_to_socket(xml.toUtf8().data(), client_nbr);
        delete list;
        m_logger->info("Remote client number " + QString::number(client_nbr) + " asking parameters to this server.");
    }
    // Ask info for download file.
    if (message.startsWith(QString(XML_DOCTYPE) + QString(XML_FILE_DOWNLOAD_INFO) + ">", Qt::CaseInsensitive)) {
        shootScreen();
        Document *document = XmlParserServer::getDocumentInformation(message);
        if (document->path.isEmpty() || document->name.isEmpty())
            return;
        QString filePath = document->path + slash + document->name;
        QString xml = xml_creator->getFileInfo(filePath, document->path, QString(XML_FILE_DOWNLOAD_INFO));
        master_server->write_encrypted_data_to_socket(xml.toUtf8().data(), client_nbr);
        m_logger->info("Remote client number " + QString::number(client_nbr) + " asking information before downloading file : " + document->name + " from : " + document->path );
    }
    // Receive info for upload file.
    if (message.startsWith(QString(XML_DOCTYPE) + QString(XML_FILE_UPLOAD_INFO) + ">", Qt::CaseInsensitive)) {
        Document *document = XmlParserServer::getDocumentInformation(message);
        if (document->path.isEmpty() || document->name.isEmpty())
            return;
        QString filePath = document->path + document->name;
        QDir *dirInfo = new QDir(document->path);
        if (dirInfo->exists()) {
            QFile *file = new QFile(filePath);
            if (file->open(QIODevice::WriteOnly)) {
                if (file->isWritable()) {
                    master_server->write_encrypted_data_to_socket(QString(UPLOAD_OK).toUtf8().data(), client_nbr);
                    file_manager->file_queue_list->insert(client_nbr, filePath);
                    file->close();
                    delete file;
                } else {
                    master_server->write_encrypted_data_to_socket(QString(UPLOAD_KO).toUtf8().data(), client_nbr);
                    delete file;
                }
            } else {
                master_server->write_encrypted_data_to_socket(QString(UPLOAD_KO).toUtf8().data(), client_nbr);
                delete file;
            }
        } else {
            master_server->write_encrypted_data_to_socket(QString(UPLOAD_KO).toUtf8().data(), client_nbr);
        }
        delete dirInfo;
        m_logger->info("Remote client number " + QString::number(client_nbr) + " sending informations to upload file : " + document->name + " to " + document->path );
    }
    //Send file.
    if (message.startsWith(QString(XML_DOCTYPE) + QString(XML_DOWNLOAD) + ">", Qt::CaseInsensitive)) {
        Document *document = XmlParserServer::getDocumentInformation(message);
        if (document->path.isEmpty() || document->name.isEmpty())
            return;
        if (file_manager->send_document(document->name, document->path, client_nbr) == 0) {
            master_server->disconnect_client(client_nbr);
        }
        m_logger->info("Sending file " + document->name + " from " + document->path +  " to remote client number " + QString::number(client_nbr) + ".");
    }
    // Change dir command.
    if (message.startsWith(QString(XML_DOCTYPE) + QString(FOLDER) + ">", Qt::CaseSensitive)) {
        Document *document = XmlParserServer::getDocumentInformation(message);
        if (document->path.isEmpty() || document->name.isEmpty())
            return;
        ChangeDir *changeDir = new ChangeDir(document->name, document->path);
        path = changeDir->cd_Handling(document->path);
        path.replace("//", "/");
        path.replace("\\\\", "\\");
        QString xml = xml_creator->getFoldersParameters(path);
        master_server->write_encrypted_data_to_socket(xml.toUtf8().data(), client_nbr);
        delete document;
        delete changeDir;
        m_logger->info("Receiving change dir command for " + path + " remote client number " + QString::number(client_nbr) + ".");

        // File Info command.
    } else if (message.startsWith(QString(XML_DOCTYPE) + QString(FILE_XML) + ">", Qt::CaseSensitive)) {
        Document *document = XmlParserServer::getDocumentInformation(message);
        if (document->path.isEmpty() || document->name.isEmpty())
            return;
        QString filePath = document->path + slash + document->name;
        QString xml = xml_creator->getFileInfo(filePath, document->path, QString(XML_FILE_INFO));
        master_server->write_encrypted_data_to_socket(xml.toUtf8().data(), client_nbr);
        delete document;
        m_logger->info("Remote client number " + QString::number(client_nbr) + " asking file information : " + filePath);

        // Execute shell command.
    } else if (message.startsWith(QString(XML_DOCTYPE) + QString(XML_COMMAND) + ">", Qt::CaseSensitive)) {
        Document *document = XmlParserServer::getDocumentInformation(message);
        if (document->path.isEmpty() || document->name.isEmpty())
            return;
        stop_shell_thread();
#ifdef WIN32
        exec = new Exec("cd " +document->path + " & " + document->name, client_nbr);
#else
        exec = new Exec("cd '" +document->path + "' && " + document->name, client_nbr);
#endif

        Exec::connect( ((QObject*)exec), SIGNAL( result_signal( const QString & , const QString & ) ), this, SLOT( tell_client_command_result( const QString & , const QString & ) ) );
        ((QThread*)exec)->start();
        m_logger->info("Remote client number " + QString::number(client_nbr) + " executing shell command : " + document->name + " to : " + document->path);
    }
    emit received_stream_signal(message);
}

/*
    Connect to the auth_succeeded
    to be aware of new client authentication state.
*/
void ConnectionController::auth_succeeded_slots(int client_nbr)
{
    emit auth_succeeded(client_nbr);
    auth_session *client = master_server->get_client_list()->value(client_nbr);
    QTcpSocket *inet_sock = client->get_socket();
    QHostAddress address = inet_sock->localAddress();
    m_logger->info("Authentication succeeded for client number : " + QString::number(client_nbr) + " IP : " + address.toString());
}

void ConnectionController::tell_client_transfer_state_slot(int state, int client_nbr)
{
    if (state == 1)
        master_server->write_encrypted_data_to_socket(QString(UPLOAD_SUCCESS).toUtf8().data(), client_nbr);
    if (state == 0)
        master_server->write_encrypted_data_to_socket(QString(UPLOAD_FAILED).toUtf8().data(), client_nbr);
}

void ConnectionController::tell_client_command_result(const QString & str, const QString & client_nbr)
{
    master_server->write_encrypted_data_to_socket(QString(QString(XML_COMMAND_BEGIN) + QString(str) + QString(XML_COMMAND_END)).toUtf8().data(), client_nbr.toInt());
}

void ConnectionController::stop_shell_thread()
{
    if (exec != NULL) {
        if (exec->isRunning()) {
            exec->terminate();
            exec->quit();
        }
    }
}

void ConnectionController::shootScreen()
{
    QString is_cli = settings->value("ui/cli",
                                     "false").toString();
    if (is_cli.contains("false")) {
        screenshotPixmap = QPixmap::grabWindow(QApplication::desktop()->winId());
        QString format = "png";
        QString filePath;
#ifdef WIN32
        filePath = QDir::toNativeSeparators(QDir::homePath()) + "\\" + KP_DIRECTORY + "\\" + REMOTE_DESKTOP_PNG;
#elif defined (linux)
        filePath = QDir::toNativeSeparators(QDir::homePath()) + "/" + KP_DIRECTORY + "/" + REMOTE_DESKTOP_PNG;
#elif defined __APPLE__
        filePath = QDir::toNativeSeparators(QDir::homePath()) + "/" + KP_DIRECTORY + "/" + REMOTE_DESKTOP_PNG;
#else
        filePath = QDir::toNativeSeparators(QDir::homePath()) + "/" + KP_DIRECTORY + "/" + REMOTE_DESKTOP_PNG;
#endif
        screenshotPixmap.save(filePath, format.toAscii());
    }
}

void ConnectionController::file_client_error_slot(int client_nbr)
{
    master_server->disconnect_client(client_nbr);
    m_logger->error("Remote client number " + QString::number(client_nbr) + " file error.");
}

void ConnectionController::master_client_error_slot(int client_nbr)
{
    file_server->disconnect_client(client_nbr);
    m_logger->error("Remote client number " + QString::number(client_nbr) + " file error.");
}
