/* SPDX-License-Identifier: GPL-2.0-only OR GPL-3.0-only */
/* Copyright (c) 2022-2025 Brett Sheffield <bacs@librecast.net> */

#ifndef GF256_H
#define GF256_H

#include <lcrq_pvt.h>
#include <stdint.h>
#include <assert.h>

/* GF(256) operations as per RFC6330 (5.7) */

static const uint8_t OCT_EXP[] = {
	1, 2, 4, 8, 16, 32, 64, 128, 29, 58, 116, 232, 205, 135, 19, 38, 76,
	152, 45, 90, 180, 117, 234, 201, 143, 3, 6, 12, 24, 48, 96, 192, 157,
	39, 78, 156, 37, 74, 148, 53, 106, 212, 181, 119, 238, 193, 159, 35,
	70, 140, 5, 10, 20, 40, 80, 160, 93, 186, 105, 210, 185, 111, 222,
	161, 95, 190, 97, 194, 153, 47, 94, 188, 101, 202, 137, 15, 30, 60,
	120, 240, 253, 231, 211, 187, 107, 214, 177, 127, 254, 225, 223, 163,
	91, 182, 113, 226, 217, 175, 67, 134, 17, 34, 68, 136, 13, 26, 52,
	104, 208, 189, 103, 206, 129, 31, 62, 124, 248, 237, 199, 147, 59,
	118, 236, 197, 151, 51, 102, 204, 133, 23, 46, 92, 184, 109, 218,
	169, 79, 158, 33, 66, 132, 21, 42, 84, 168, 77, 154, 41, 82, 164, 85,
	170, 73, 146, 57, 114, 228, 213, 183, 115, 230, 209, 191, 99, 198,
	145, 63, 126, 252, 229, 215, 179, 123, 246, 241, 255, 227, 219, 171,
	75, 150, 49, 98, 196, 149, 55, 110, 220, 165, 87, 174, 65, 130, 25,
	50, 100, 200, 141, 7, 14, 28, 56, 112, 224, 221, 167, 83, 166, 81,
	162, 89, 178, 121, 242, 249, 239, 195, 155, 43, 86, 172, 69, 138, 9,
	18, 36, 72, 144, 61, 122, 244, 245, 247, 243, 251, 235, 203, 139, 11,
	22, 44, 88, 176, 125, 250, 233, 207, 131, 27, 54, 108, 216, 173, 71,
	142, 1, 2, 4, 8, 16, 32, 64, 128, 29, 58, 116, 232, 205, 135, 19, 38,
	76, 152, 45, 90, 180, 117, 234, 201, 143, 3, 6, 12, 24, 48, 96, 192,
	157, 39, 78, 156, 37, 74, 148, 53, 106, 212, 181, 119, 238, 193, 159,
	35, 70, 140, 5, 10, 20, 40, 80, 160, 93, 186, 105, 210, 185, 111,
	222, 161, 95, 190, 97, 194, 153, 47, 94, 188, 101, 202, 137, 15, 30,
	60, 120, 240, 253, 231, 211, 187, 107, 214, 177, 127, 254, 225, 223,
	163, 91, 182, 113, 226, 217, 175, 67, 134, 17, 34, 68, 136, 13, 26,
	52, 104, 208, 189, 103, 206, 129, 31, 62, 124, 248, 237, 199, 147,
	59, 118, 236, 197, 151, 51, 102, 204, 133, 23, 46, 92, 184, 109, 218,
	169, 79, 158, 33, 66, 132, 21, 42, 84, 168, 77, 154, 41, 82, 164, 85,
	170, 73, 146, 57, 114, 228, 213, 183, 115, 230, 209, 191, 99, 198,
	145, 63, 126, 252, 229, 215, 179, 123, 246, 241, 255, 227, 219, 171,
	75, 150, 49, 98, 196, 149, 55, 110, 220, 165, 87, 174, 65, 130, 25,
	50, 100, 200, 141, 7, 14, 28, 56, 112, 224, 221, 167, 83, 166, 81,
	162, 89, 178, 121, 242, 249, 239, 195, 155, 43, 86, 172, 69, 138, 9,
	18, 36, 72, 144, 61, 122, 244, 245, 247, 243, 251, 235, 203, 139, 11,
	22, 44, 88, 176, 125, 250, 233, 207, 131, 27, 54, 108, 216, 173, 71,
	142
};
#ifdef static_assert
static_assert(sizeof(OCT_EXP) == 510, "OCT_EXP table has wrong size.");
#endif

/*
5.7.4.  The Table OCT_LOG

   The table OCT_LOG contains 255 non-negative integers.  The table is
   indexed by octets interpreted as integers.  The octet corresponding
   to the zero element, which is represented by the integer 0, is
   excluded as an index, and thus indexing starts at 1 and ranges up to
   255, and the entries are the following:
*/

static const uint8_t OCT_LOG[] = {
	0, /* bump base index to 1 */
	0, 1, 25, 2, 50, 26, 198, 3, 223, 51, 238, 27, 104, 199, 75, 4, 100,
	224, 14, 52, 141, 239, 129, 28, 193, 105, 248, 200, 8, 76, 113, 5,
	138, 101, 47, 225, 36, 15, 33, 53, 147, 142, 218, 240, 18, 130, 69,
	29, 181, 194, 125, 106, 39, 249, 185, 201, 154, 9, 120, 77, 228, 114,
	166, 6, 191, 139, 98, 102, 221, 48, 253, 226, 152, 37, 179, 16, 145,
	34, 136, 54, 208, 148, 206, 143, 150, 219, 189, 241, 210, 19, 92,
	131, 56, 70, 64, 30, 66, 182, 163, 195, 72, 126, 110, 107, 58, 40,
	84, 250, 133, 186, 61, 202, 94, 155, 159, 10, 21, 121, 43, 78, 212,
	229, 172, 115, 243, 167, 87, 7, 112, 192, 247, 140, 128, 99, 13, 103,
	74, 222, 237, 49, 197, 254, 24, 227, 165, 153, 119, 38, 184, 180,
	124, 17, 68, 146, 217, 35, 32, 137, 46, 55, 63, 209, 91, 149, 188,
	207, 205, 144, 135, 151, 178, 220, 252, 190, 97, 242, 86, 211, 171,
	20, 42, 93, 158, 132, 60, 57, 83, 71, 109, 65, 162, 31, 45, 67, 216,
	183, 123, 164, 118, 196, 23, 73, 236, 127, 12, 111, 246, 108, 161,
	59, 82, 41, 157, 85, 170, 251, 96, 134, 177, 187, 204, 62, 90, 203,
	89, 95, 176, 156, 169, 160, 81, 11, 245, 22, 235, 122, 117, 44, 215,
	79, 174, 213, 233, 230, 231, 173, 232, 116, 214, 244, 234, 168, 80,
	88, 175
};
#ifdef static_assert
static_assert(sizeof(OCT_LOG) == 255 + 1, "OCT_LOG table has wrong size.");
#endif

extern uint8_t GF256LR[256][2][16];

#ifdef USE_SIMD_x86
#define GF256LR_INIT if (GF256LR[7][0][0x0f] != 0x2d) gf256_init()
#else
#define GF256LR_INIT while(0)
#endif

#define GF256ADD(a, b) ((a) ^ (b))

#define GF256EXP(e) OCT_EXP[(e)]

#define GF256LOG(v) OCT_LOG[(v)]

#define GF256INV(v) OCT_EXP[255 - OCT_LOG[(v)]]

#define GF256MUL0(a, b) (GF256EXP((int)GF256LOG(a) + (int)GF256LOG(b)))

#define GF256MUL(a, b) ((a) == 0 || (b) == 0) ? 0 : (GF256EXP((int)GF256LOG(a) + (int)GF256LOG(b)))

#define GF256DIV(a, b) ((a) == 0) ? 0 : OCT_EXP[OCT_LOG[(a)] - OCT_LOG[(b)] + 255]

/* add (or subtract) in GF(256) */
uint8_t gf256_add(const uint8_t a, const uint8_t b);

uint8_t gf256_exp(const int e);

uint8_t gf256_log(const uint8_t v);

uint8_t gf256_inv(const uint8_t v);

uint8_t gf256_mul(const uint8_t a, const uint8_t b);

uint8_t gf256_div(const uint8_t u, const uint8_t v);

#ifdef USE_SIMD_x86
__m512i gf256_mul_512(__m512i A, uint8_t y);
void gf256_mul_512_inplace(uint8_t *d, uint8_t y);
__m256i gf256_mul_256(__m256i A, uint8_t y);
void gf256_mul_256_inplace(uint8_t *d, uint8_t y);
__m128i gf256_mul_128(__m128i A, uint8_t y);
void gf256_mul_128_inplace(uint8_t *d, uint8_t y);
#endif

void gf256_init(void);

#endif /* GF256_H */
