/*
 *                            COPYRIGHT
 *
 *  sch-rnd - modular/flexible schematics editor - tEDAx netlist export
 *  Copyright (C) 2020, 2023, 2025 Tibor 'Igor2' Palinkas
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 31 Milk Street, # 960789 Boston, MA 02196 USA.
 *
 *  Contact:
 *    Project page: http://repo.hu/projects/sch-rnd
 *    contact lead developer: http://www.repo.hu/projects/sch-rnd/contact.html
 *    mailing list: http://www.repo.hu/projects/sch-rnd/contact.html
 */


#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <librnd/core/compat_misc.h>
#include <librnd/core/safe_fs.h>
#include <librnd/core/plugins.h>
#include <librnd/core/error.h>
#include <libcschem/config.h>
#include <libcschem/plug_io.h>

#include <plugins/lib_netlist_exp/lib_netlist_exp.h>
#include <plugins/lib_tedax/write.h>

static csch_plug_io_t etdx_net;

static int tdx_export_prio(const char *fn, const char *fmt, csch_plug_io_type_t type)
{
	if (type != CSCH_IOTYP_NETLIST)
		return 0;
	if ((rnd_strcasecmp(fmt, "tedax") == 0) || (rnd_strcasecmp(fmt, "tdx") == 0))
		return 100;
	if (rnd_strcasecmp(fmt, "net") == 0)
		return 90;
	return 0;
}

static void export_attr(FILE *f, const char *cmd, const char *key, const char *val, const char *val2)
{
	if ((key == NULL) || (val == NULL))
		return;
	fprintf(f, "%s ", cmd);
	tedax_fprint_escape(f, key);
	fputc(' ', f);
	tedax_fprint_escape(f, val);
	if (val2 != NULL) {
		fputc(' ', f);
		tedax_fprint_escape(f, val2);
	}
	fputc('\n', f);
}

/* Export comptag and nettag for attributes that have export_name */
static void export_custom_attrs(FILE *f, const char *name, csch_ahdr_t *obj, const char *cmd)
{
	htsp_entry_t *e;
	for(e = htsp_first(&obj->attr); e != NULL; e = htsp_next(&obj->attr, e)) {
		csch_attrib_t *a = e->value;
		if ((a->export_name != NULL) && (a->val != NULL))
			export_attr(f, cmd, name, a->export_name, a->val);
	}
}

static void tdx_export_comps(FILE *f, csch_abstract_t *abs)
{
	htsp_entry_t *e, *p;
	for(e = htsp_first(&abs->comps); e != NULL; e = htsp_next(&abs->comps, e)) {
		csch_acomp_t *comp = e->value;
		const char *refdes = sch_nle_get_refdes(comp);

		if (refdes == NULL)
			continue;

		if (comp->hdr.omit)
			continue;

		export_attr(f, "footprint", refdes, sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_FOOTPRINT), NULL);
		export_attr(f, "value", refdes, sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_VALUE), NULL);
		export_attr(f, "device", refdes, sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_DEVICE), NULL);
		export_custom_attrs(f, refdes, &comp->hdr, "comptag");
		for(p = htsp_first(&comp->ports); p != NULL; p = htsp_next(&comp->ports, p)) {
			const csch_aport_t *port = p->value;
			const char *pinnums = sch_nle_get_pinnum(port);
			const char *pinname = sch_nle_get_alt_attr(&port->hdr.attr, SCH_NLE_ALTATTR_PINNAME);

			if (pinname == NULL)
				continue;

			/* call export_attr on each pin in pinnums (which can be a list of
			   pin numbers) so that each pinnum -> pinname is written */
			SCH_NLE_FOREACH_PINNUM(pinnums, my_num,
				export_attr(f, "pinname", refdes, my_num, pinname)
			);
		}
	}
}

static void tdx_export_nets(FILE *f, csch_abstract_t *abs)
{
	htsp_entry_t *e;
	long n;

	for(e = htsp_first(&abs->nets); e != NULL; e = htsp_next(&abs->nets, e)) {
		csch_anet_t *net = e->value;
		const char *netname = sch_nle_get_netname(net);
		int net_exported = 0;

		if (net->hdr.omit) continue;

		for(n = 0; n < net->conns.used; n++) {
			csch_aport_t *port = net->conns.array[n];
			const char *refdes = NULL, *pinnums;

			if (port->hdr.type != CSCH_ATYPE_PORT) {
				rnd_message(RND_MSG_ERROR, "tdx: invalid connection (object type)\n");
				continue;
			}

			pinnums = sch_nle_get_pinnum(port);
			if (pinnums == NULL) {
				rnd_message(RND_MSG_ERROR, "tdx: can't determine port's pin number\n");
				continue;
			}

			if (port->parent != NULL) {
				refdes = sch_nle_get_refdes(port->parent);
				if (port->parent->hdr.omit)
					continue; /* omit component */
			}
			if (refdes == NULL) {
				/* This is not an error: no refdes means: do not export (e.g. gnd) */
/*				rnd_message(RND_MSG_ERROR, "tdx: can't determine port's parent component refdes\n");*/
				continue;
			}

			/* split up pinnum at space and create one or more conn lines connecting
			   each pin to the given net */
			SCH_NLE_FOREACH_PINNUM(pinnums, my_num,
				{
					net_exported = 1;

					fprintf(f, "conn ");
					tedax_fprint_escape(f, netname);
					fputc(' ', f);
					tedax_fprint_escape(f, refdes);
					fputc(' ', f);
					tedax_fprint_escape(f, my_num);
					fputc('\n', f);
				}
			);
		}

		/* If the net got exported, all the vital attributes are already printed;
		   now print misc attributes the user wanted to see in the export */
		if (net_exported)
			export_custom_attrs(f, netname, &net->hdr, "nettag");
	}
}


static int tdx_export_project_abst(const char *fn, const char *fmt, csch_abstract_t *abs, rnd_hid_attr_val_t *options)
{
	TODO("get hidlib as an arg")
	rnd_design_t *hidlib = NULL;
	FILE *f = rnd_fopen(hidlib, fn, "w");
	if (f == NULL)
		return -1;

	fprintf(f, "tEDAx v1\n");
	fprintf(f, "begin netlist v1 ");
	tedax_fprint_escape(f, "<TODO: name>");
	fprintf(f, "\n");

	tdx_export_comps(f, abs);
	tdx_export_nets(f, abs);

	fprintf(f, "end netlist\n\n");
	fclose(f);
	return 0;
}

#include "hid_impl.c"

int pplg_check_ver_export_tedax(int ver_needed) { return 0; }

void pplg_uninit_export_tedax(void)
{
	csch_plug_io_unregister(&etdx_net);
	rnd_export_remove_opts_by_cookie(tedax_cookie);
	rnd_hid_remove_hid(&tedax_hid);
}

int pplg_init_export_tedax(void)
{
	RND_API_CHK_VER;

	etdx_net.name = "export to tEDAx";
	etdx_net.export_prio = tdx_export_prio;
	etdx_net.export_project_abst = tdx_export_project_abst;
	etdx_net.ext_export_project = ".tdx";
	csch_plug_io_register(&etdx_net);


	rnd_hid_nogui_init(&tedax_hid);

	tedax_hid.struct_size = sizeof(rnd_hid_t);
	tedax_hid.name = "tedax";
	tedax_hid.description = "Exports project's tEDAx netlist";
	tedax_hid.exporter = 1;

	tedax_hid.get_export_options = tedax_get_export_options;
	tedax_hid.do_export = tedax_do_export;
	tedax_hid.parse_arguments = tedax_parse_arguments;
	tedax_hid.argument_array = tedax_values;

	tedax_hid.usage = tedax_usage;

	rnd_hid_register_hid(&tedax_hid);
	rnd_hid_load_defaults(&tedax_hid, tedax_options, NUM_OPTIONS);


	return 0;
}

