// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief e+e- > light hadrons including KS0
  class BESII_2009_I835937 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BESII_2009_I835937);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(FinalState(), "FS");
      declare(UnstableParticles(), "UFS");

      if (isCompatibleWithSqrtS(3.65))  _sqs = "3.65";
      else if (isCompatibleWithSqrtS(3.773)) {
        _sqs = "3.773";
        book(_sigma[1], 1, 1, 2);
      }
      raiseBeamErrorIf(_sqs.empty());
      for (size_t ix=0; ix<7; ++ix) {
        if (ix>0 && ix<5) continue;
        book(_sigma[ix], 1, 1, 1+ix);
      }
    }

    void findChildren(const Particle& p, map<long,int>& nRes, int& ncount) const {
      for (const Particle& child : p.children()) {
        if (child.children().empty()) {
          nRes[child.pid()]-=1;
          --ncount;
        }
        else {
          findChildren(child,nRes,ncount);
        }
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const FinalState& fs = apply<FinalState>(event, "FS");

      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
        nCount[p.pid()] += 1;
        ++ntotal;
      }
      const FinalState& ufs = apply<FinalState>(event, "UFS");
      // first the purely FS particles
      if (ntotal==5 && nCount[310]==1 &&
        ((nCount[-321]==2 && nCount[ 321]==1 && nCount[ 211]==1) ||
				 (nCount[ 321]==2 && nCount[-321]==1 && nCount[-211]==1))) {
        _sigma[0]->fill(_sqs);
      }

      // loop over unstable particles
      for (const Particle& p : ufs.particles(Cuts::pid==221 ||
                                             Cuts::pid==113 ||
                                             Cuts::abspid==213)) {
        if (p.children().empty()) continue;
      	map<long,int> nRes = nCount;
      	int ncount = ntotal;
      	findChildren(p,nRes,ncount);
        // eta /rho0
        bool matched = false;
        if (p.pid()==221 || p.pid()==113) {
       	  if (ncount==3) {
      	    matched = true;
      	    for (const auto& val : nRes) {
      	      if (abs(val.first)==310) {
                if (val.second!=1) {
                  matched = false;
                  break;
                }
      	      }
              else if (abs(val.first)==211) {
                if (val.second!=1) {
                  matched = false;
                  break;
                }
      	      }
              else if (abs(val.first)==321) {
                if (val.second!=1) {
                  matched = false;
                  break;
                }
      	      }
      	      else if (val.second!=0) {
                matched = false;
                break;
      	      }
            }
            if (matched) {
              if (p.pid()==113) _sigma[6]->fill(_sqs);
              else if (p.pid()==221 && _sqs=="3.773"s) _sigma[1]->fill(_sqs);
            }
          }
        }
        // rho+/-
        else if (p.abspid()==213) {
          if (ncount==2) {
            int sign = p.pid()>0 ? -1 : 1;
            matched=true;
            for (const auto& val : nRes) {
              if (abs(val.first)==310) {
                if (val.second!=1) {
                  matched = false;
                  break;
                }
      	      }
              else if (val.first==321*sign) {
                if (val.second!=1) {
                  matched = false;
                  break;
                }
      	      }
      	      else if (val.second!=0) {
                matched = false;
                break;
      	      }
      	    }
            if (matched) _sigma[5]->fill(_sqs);
          }
        }
        if (matched) break;
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_sigma, crossSection()/ sumOfWeights() /picobarn);
    }

    /// @}


    /// @name Histograms
    /// @{
    map<int,BinnedHistoPtr<string>> _sigma;
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(BESII_2009_I835937);

}
