# Graph llbrary
Simple Graph library

# usage

use_module(graph).


# Specifications / API
1. generate_graph(Vs, Es, G)

Description:
Generates a graph G with vertices Vs and edges Es.

Parameters:

Vs : List of vertices, e.g., [a,b,c]

Es : List of edges, e.g., [e(a,b), e(b,c)]

G : Resulting graph term, e.g., graph([a,b,c],[e(a,b),e(b,c)])

Example:

?- generate_graph([a,b,c],[e(a,b),e(b,c)], G).
G = graph([a,b,c],[e(a,b), e(b,c)]).

## undirected/directed, weighted 

- Es : List of undirected edges with weight, e.g., [ew(a,b,3), ew(b,c,4)]
ew(_,_,weight)
- Es : List of irected edges with weight, e.g., [edw(a,b,3), edw(b,c,4)]
edw(_,_,weight)
- Es : List of directed edges, e.g., [ed(a,b), ed(b,c)]
ed(_,_)


2. vertex(G, V)

Description:
Checks whether V is a vertex in graph G.

Parameters:

G : Graph term

V : Vertex

Example:

?- generate_graph([a,b,c],[e(a,b),e(b,c)], G),
   vertex(G, b).
true.

3. edge(G, U, V)

Description:
Checks whether there is an edge from U to V in graph G.

Parameters:

G : Graph term, graph(_, Es)

U : Start vertex

V : End vertex

Example:

?- generate_graph([a,b,c],[e(a,b),e(b,c)], G),
   edge(G, a, b).
true.

4. adjacent(G, U, V)

Description:
Checks if vertices U and V are adjacent (connected directly by an edge) in graph G.

Example:

?- generate_graph([a,b,c],[e(a,b),e(b,c)], G),
   adjacent(G, a, b).
true.

?- adjacent(G, a, c).
false.

5. connected(G)

Description:
Checks whether the graph G is connected (there is a path between any two vertices).

Example:

?- generate_graph([a,b,c],[e(a,b),e(b,c)], G),
   connected(G).
true.

6. strongly_connected(G)

Description:
Checks whether G is strongly connected (for directed graphs, a path exists from every vertex to every other vertex).

Example:

?- generate_graph([a,b,c],[e(a,b),e(b,c),e(c,a)], G),
   strongly_connected(G).
true.

7. complete(G)

Description:
Checks whether G is a complete graph (every vertex is connected to every other vertex).

Example:

?- generate_graph([a,b,c],[e(a,b),e(a,c),e(b,c),e(b,a),e(c,a),e(c,b)], G),
   complete(G).
true.

8. generate_kn(N, G)

Description:
Generates a complete graph G with N vertices.

Example:

?- generate_kn(3, G).
G = graph([1,2,3],[e(1,2), e(1,3), e(2,1), e(2,3), e(3,1), e(3,2)]).

9. reverse_graph(G,R).

Description:
Generates a reverse graph.

Example:

?- generate_graph([1,2,3],[ed(1,2),ed(2,3),ed(3,1),ed(3,4),ed(4,5),ed(5,6),ed(6,4)],G),reverse_graph(G,R).
G = graph([1,2,3],[ed(1,2),ed(2,3),ed(3,1),ed(3,4),ed(4,5),ed(5,6),ed(6,4)])
R = graph([1,2,3],[ed(2,1),ed(3,2),ed(1,3),ed(4,3),ed(5,4),ed(6,5),ed(4,6)]) .
yes
?- 

10. scc(G,C)

Description:
generate strong connected component.

Example:
generate_graph([1,2,3,4,5,6,7],[ed(1,2),ed(2,3),ed(3,1),ed(3,4),ed(4,1),ed(4,5),ed(5,6),ed(6,7),ed(7,5)],G),
    scc(G,C).

C = [[5,6,7],[1,2,3,4]] .
yes
?- 

11. topsort(G,X)

Description:
generate topological sorted list for DAG graph G.

Example:
?- generate_graph([1,2,3,4],[ed(1,2),ed(1,3),ed(1,4),ed(3,2),ed(4,2),ed(4,3)],G),topsort(G,X).
G = graph([1,2,3,4],[ed(1,2),ed(1,3),ed(1,4),ed(3,2),ed(4,2),ed(4,3)])
X = [1,4,3,2] .
yes
?- 

12. dag(G)

Description:
When G is DAG(directed acyclic grahp) return YES.

Example:
?- generate_graph([1,2,3,4],[ed(1,2),ed(1,3),ed(1,4),ed(3,2),ed(4,2),ed(4,3)],G),dag(G).
G = graph([1,2,3,4],[ed(1,2),ed(1,3),ed(1,4),ed(3,2),ed(4,2),ed(4,3)]) .
yes
?- 

💡 Note:

graph(Vs, Es) is the standard representation used in this library.

e(U,V) denotes a undirected edge from U to V.

For undirected graphs, treat e(U,V) and e(V,U) as equivalent.

13. dijkstra(G,Start,Goal,Path)

G is undirected-wighted graph
e.g.

```
find(P) :-
    generate_graph([a,b,c,d,e,f,g,h],
                   [ew(a,b,1),
                    ew(b,e,2),
                    ew(b,f,4),
                    ew(e,f,1),
                    ew(f,h,6),
                    ew(a,c,7),
                    ew(a,d,2),
                    ew(d,g,5),
                    ew(c,g,3),
                    ew(c,f,2),
                    ew(g,h,2)],G),
    dijkstra(G,a,h,P).

?- find(X).
X = [h,g,d,a] .
yes
?- 

```
![](dijkstra.png)

