/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

package org.mozilla.javascript.tests;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Arrays;
import java.util.List;
import java.util.function.Consumer;
import org.junit.Assert;
import org.junit.Test;
import org.mozilla.javascript.EcmaError;
import org.mozilla.javascript.JavaScriptException;
import org.mozilla.javascript.ScriptableObject;
import org.mozilla.javascript.Undefined;
import org.mozilla.javascript.WrappedException;
import org.mozilla.javascript.testutils.Utils;

/**
 * Unit tests to check error handling. Especially, we expect to get a correct cause, when an error
 * happened in Java.
 *
 * @author Roland Praml
 */
public class ErrorHandlingTest {

    public static void generateJavaError() {
        throw new java.lang.RuntimeException("foo");
    }

    // string contains stack element with correct line number. e.g:
    // ' at
    // org.mozilla.javascript.tests.ErrorHandlingTest.generateJavaError(ErrorHandlingTest.java:30)'
    private static final String EXPECTED_LINE_IN_STACK = findLine();

    @Test
    public void throwError() {
        testIt(
                "\n" // 1
                        + "throw new Error('foo')", // 2
                e -> {
                    Assert.assertEquals(JavaScriptException.class, e.getClass());
                    Assert.assertEquals("Error: foo (myScript.js#2)", e.getMessage());
                });
        testIt(
                "\n" // 1
                        + "throw new EvalError('foo')",
                e -> {
                    Assert.assertEquals(JavaScriptException.class, e.getClass());
                    Assert.assertEquals("EvalError: foo (myScript.js#2)", e.getMessage());
                });
        testIt(
                "try {\n" // 1
                        + " throw new Error('foo')\n" // 2
                        + "} catch (e) {\n" // 3
                        + " throw e\n" // 4
                        + "}", // 5
                e -> {
                    Assert.assertEquals(JavaScriptException.class, e.getClass());
                    Assert.assertEquals("Error: foo (myScript.js#4)", e.getMessage());
                });
        testIt(
                "\n" // 1
                        + "null.toString()", // 2
                e -> {
                    Assert.assertEquals(EcmaError.class, e.getClass());
                    Assert.assertEquals(
                            "TypeError: Cannot call method \"toString\" of null (myScript.js#2)",
                            e.getMessage());
                });
    }

    @Test
    public void throwErrorOnProp() {
        Utils.assertWithAllModes(
                "TypeError: Cannot call method \"toString\" of null\tat test.js:3"
                        + System.lineSeparator(),
                "var ret\n" // 1
                        + "try {\n" // 2
                        + "    null.toString()\n" // 3
                        + "} catch (e) {\n" // 4
                        + "    ret = e + e.stack\n" // 5
                        + "}\n" // 6
                        + "ret"); // 7
    }

    @Test
    public void javaErrorFromInvocation() {

        testIt(
                "org.mozilla.javascript.tests.ErrorHandlingTest.generateJavaError()",
                e -> {
                    Assert.assertEquals(WrappedException.class, e.getClass());
                    Assert.assertEquals(
                            "Wrapped java.lang.RuntimeException: foo (myScript.js#1)",
                            e.getMessage());
                    Assert.assertEquals(RuntimeException.class, e.getCause().getClass());
                    Assert.assertEquals("foo", e.getCause().getMessage());
                    Assert.assertTrue(stackToLines(e).contains(EXPECTED_LINE_IN_STACK));
                });
        testIt(
                "try { org.mozilla.javascript.tests.ErrorHandlingTest.generateJavaError() } catch (e) { throw e }",
                e -> {
                    Assert.assertEquals(JavaScriptException.class, e.getClass());
                    Assert.assertEquals(
                            "JavaException: java.lang.RuntimeException: foo (myScript.js#1)",
                            e.getMessage());
                    Assert.assertEquals(RuntimeException.class, e.getCause().getClass());
                    Assert.assertEquals("foo", e.getCause().getMessage());
                    Assert.assertTrue(stackToLines(e).contains(EXPECTED_LINE_IN_STACK));
                });
    }

    @Test
    public void javaErrorThrown() {

        testIt(
                "throw new java.lang.RuntimeException('foo')",
                e -> {
                    Assert.assertEquals(JavaScriptException.class, e.getClass());
                    Assert.assertEquals(
                            "java.lang.RuntimeException: foo (myScript.js#1)", e.getMessage());
                    Assert.assertEquals(RuntimeException.class, e.getCause().getClass());
                    Assert.assertEquals("foo", e.getCause().getMessage());
                });
        testIt(
                "try { throw new java.lang.RuntimeException('foo') } catch (e) { throw e }",
                e -> {
                    Assert.assertEquals(JavaScriptException.class, e.getClass());
                    Assert.assertEquals(
                            "java.lang.RuntimeException: foo (myScript.js#1)", e.getMessage());
                    Assert.assertEquals(RuntimeException.class, e.getCause().getClass());
                    Assert.assertEquals("foo", e.getCause().getMessage());
                });
    }

    @Test
    public void stackProvider() {
        String nl = System.lineSeparator();
        Utils.assertWithAllModes(Undefined.instance, "Error.stack");
        Utils.assertWithAllModes("\tat test.js:1" + nl, "new Error().stack");
        Utils.assertWithAllModes(Undefined.instance, "EvalError.stack");
        Utils.assertWithAllModes("\tat test.js:1" + nl, "new EvalError('foo').stack");
    }

    private void testIt(final String script, final Consumer<Throwable> exception) {
        Utils.runWithAllModes(
                cx -> {
                    try {
                        final ScriptableObject scope = cx.initStandardObjects();
                        cx.evaluateString(scope, script, "myScript.js", 1, null);
                        Assert.fail("No error was thrown");
                    } catch (final Throwable t) {
                        exception.accept(t);
                    }
                    return null;
                });
    }

    static List<String> stackToLines(Throwable t) {
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        t.printStackTrace(pw);
        String[] tmp = sw.toString().replace("\r\n", "\n").split("\n");
        return Arrays.asList(tmp);
    }

    private static String findLine() {
        try {
            generateJavaError();
        } catch (Throwable t) {
            for (String line : stackToLines(t)) {
                if (line.contains("generateJavaError")) {
                    System.out.println(line);
                    return line;
                }
            }
        }
        throw new UnsupportedOperationException("Did not find expected line");
    }
}
