/*-
 * Copyright (c) 2008, 2009, 2012, 2013  Peter Pentchev
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Ringlet$
 */

#define _GNU_SOURCE

#include <ctype.h>
#include <err.h>
#include <fnmatch.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#ifdef HAVE_PCRE
#include <pcre.h>
#endif

#include "confget.h"
#include "confget_http_get.h"
#include "confget_ini.h"

/* The configuration filename specified by the user. */
const char		*filename;
/* The configuration section specified by the user. */
const char		*section;
/* The number of the variable or pattern parameters after the options. */
static int		 margc;
/* The variables to extract or patterns to match against. */
static char		* const *margv;
/* Display the variable names, or only the values? */
static int		 showvarname;
/* Look for more variables after a match is found? */
static int		 manyvars;
/* The pattern to match the variable value */
static const char	*matchvalue;
/* The prefix to use when displaying the variable name */
static const char	*output_prefix;
/* The postfix to use when displaying the variable name */
static const char	*output_postfix;

/* Scan for more variables after one was found? */
int			 morevars;

static const confget_backend	* const backends[] = {
	&confget_http_get_backend,
	&confget_ini_backend,
};
#define BACKEND_CNT	(sizeof(backends) / sizeof(backends[0]))

static int		 find_backend(const char * const _name);
static int		 init(int _argc, char * const *_argv);
static void		 usage(int _error);
static void		 version(void);

/* The currently used configuration backend (ini, http_get, etc.) */
const confget_backend	*backend;

/* The currently processed input file */
FILE			*conffile;

/* Has the "-c" (check only) option been specified? */
int			 cflag;
/* In check-only mode, has the variable been found? */
int			 cfound;
/* Has the "-l" (list the section) option been specified? */
static int		 lflag;
/* Has the "-L" (match varname against patterns) option been specified? */
static int		 Lflag;
/* Query for the names of the sections in the configuration file? */
int			 qsections;
/* Shell-quote the variable values? */
static int		 Sflag;
#ifdef HAVE_PCRE
/* Treat patterns as regular expressions? */
static int		 xflag;

static const pcre	*x_matchvalue, **x_margv;
static const pcre_extra	*x_matchvalue_extra, **x_margv_extra;
#endif

/***
 * Function:
 *	usage		- display program usage information and exit
 * Inputs:
 *	error		- is this invoked in error?
 * Returns:
 *	Nothing.
 * Modifies:
 *	Writes the usage information to the standard output or standard
 *	error depending on the "error" flag.
 */
void
usage(int error)
{
	const char * const s1 = "Usage:\n"
	    "confget [-cSx] [-N | -n] [-f filename] [-m pattern] [-P postfix]"
	    " [-p prefix]\n"
	    "        [-s section] [-t type] var...\n\n"
	    "confget [-Sx] [-N | -n] [-f filename] [-m pattern] [-P postfix]"
	    " [-p prefix]\n"
	    "        [-s section] [-t type] -L pattern...\n\n"
	    "confget [-Sx] [-N | -n] [-f filename] [-m pattern] [-P postfix]"
	    " [-p prefix]\n"
	    "        [-s section] [-t type] -l\n"
	    "confget [-f filename] -q sections [-t type]\n\n"
	    "confget [-hTV]\n\n";
	const char * const s2 =
	    "\t-c\tcheck if the variable is defined in the file;\n"
	    "\t-f\tspecify the configuration file to read from,\n"
	    "\t\tor \"-\" for standard input;\n"
	    "\t-h\tdisplay usage information and exit;\n"
	    "\t-L\tspecify which variables to display;\n"
	    "\t-l\tlist all variables in the specified section;\n"
	    "\t-m\tonly display values that match the specified pattern;\n"
	    "\t-N\talways display the variable name;\n"
	    "\t-n\tnever display the variable name;\n";
	const char * const s3 =
	    "\t-P\tdisplay this string after the variable name;\n"
	    "\t-p\tdisplay this string before the variable name;\n"
	    "\t-q\tquery for a specific type of information, e.g. the list of\n"
	    "\t\tsections defined in the configuration file;\n"
	    "\t-S\tquote the values suitably for the Bourne shell;\n"
	    "\t-s\tspecify the configuration section to read;\n"
	    "\t-T\tlist the available configuration file types;\n"
	    "\t-t\tspecify the configuration file type;\n"
	    "\t-x\ttreat the match patterns as regular expressions;\n"
	    "\t-V\tdisplay program version information and exit.";

	if (error) {
		fprintf(stderr, "%s%s%s\n", s1, s2, s3);
		exit(1);
	}
	printf("%s%s%s\n", s1, s2, s3);
}

/***
 * Function:
 *	version		- display program version information
 * Inputs:
 *	None.
 * Returns:
 *	Nothing.
 * Modifies:
 *	Nothing; displays program version information on the standard output.
 */
void
version(void)
{
	printf("confget 1.05\n");
}

/***
 * Function:
 *	init		- parse the command-line options
 * Inputs:
 *	argc, argv	- the command-line parameter
 * Returns:
 *	0 on success, -1 on error.
 * Modifies:
 *	Sets section and varname on success.
 *	Writes to the standard error on error.
 */
static int
init(int argc, char * const *argv)
{
	int bidx, ch, do_help, do_list, do_version, show_name, hide_name;
	size_t i;
#ifdef HAVE_PCRE
	const char *pcre_err;
	int pcre_ofs;
	size_t t;
#endif

	conffile = NULL;

	cflag = Lflag = lflag = Sflag = showvarname = manyvars = 0;
	do_help = do_list = do_version = show_name = hide_name = 0;
	matchvalue = NULL;
	output_prefix = output_postfix = "";
	bidx = find_backend("ini");
	if (bidx == -1) {
		fputs("INTERNAL ERROR: no 'ini' backend\n", stderr);
		return (-1);
	}
	while ((ch = getopt(argc, argv, "cf:hLlm:NnP:p:q:Ss:Tt:xV")) != EOF) {
		switch (ch) {
			case 'c':
				cflag = 1;
				manyvars = 0;
				break;

			case 'f':
				filename = optarg;
				break;

			case 'h':
				do_help = 1;
				break;

			case 'L':
				Lflag = 1;
				showvarname = 1;
				manyvars = 1;
				break;

			case 'l':
				lflag = 1;
				showvarname = 1;
				manyvars = 1;
				break;

			case 'm':
				matchvalue = optarg;
				break;

			case 'N':
				show_name = 1;
				break;

			case 'n':
				hide_name = 1;
				break;

			case 'P':
				output_postfix = optarg;
				break;

			case 'p':
				output_prefix = optarg;
				break;

			case 'q':
				/* TODO: Implement better query parsing */
				if (strcmp(optarg, "sections") != 0)
					errx(1, "Only the 'sections' query is supported for the present");
				qsections = 1;
				break;

			case 'S':
				Sflag = 1;
				break;

			case 's':
				section = optarg;
				break;

			case 'T':
				do_list = 1;
				break;

			case 't':
				bidx = find_backend(optarg);
				if (bidx == -1) {
					fprintf(stderr, "No backend '%s'\n",
					    optarg);
					return (-1);
				}
				break;

			case 'x':
#ifdef HAVE_PCRE
				xflag = 1;
				break;
#else
				errx(1, "No regular expression support in this confget build");
#endif

			case 'V':
				do_version = 1;
				break;

			case '?':
			default:
				usage(1);
				break;
		}
	}

	if (do_version)
		version();
	if (do_help)
		usage(0);
	if (do_list) {
		printf("Supported backends:");
		for (i = 0; i < BACKEND_CNT; i++)
			printf(" %s", backends[i]->name);
		puts("");
	}
	if (do_help || do_list || do_version)
		exit(0);

	argc -= optind;
	argv += optind;
	if (cflag && (lflag || Lflag)) {
		warnx("The -c flag may not be used with -l or -L");
		return (-1);
	}
	if (argc == 0 && !lflag && !qsections) {
		warnx("No matching criteria specified");
		return (-1);
	} else if ((argc > 0) && lflag) {
		warnx("Too many matching criteria specified");
		return (-1);
	}
	if (argc > 1 && !lflag) {
		showvarname = 1;
		manyvars = 1;
	}
	if (show_name) {
		if (hide_name) {
			warnx("The -N and -n flags may not be used together");
			return (-1);
		}
		showvarname = 1;
	} else if (hide_name) {
		showvarname = 0;
	}
	margc = argc;
	margv = argv;

#ifdef HAVE_PCRE
	if (xflag) {
		if (matchvalue) {
			x_matchvalue = pcre_compile(matchvalue, 0, &pcre_err,
			    &pcre_ofs, NULL);
			if (pcre_err != NULL)
				errx(1, "Invalid match value: %s", pcre_err);
			x_matchvalue_extra = pcre_study(x_matchvalue, 0,
			    &pcre_err);
			if (pcre_err != NULL)
				errx(1, "Invalid match value: %s", pcre_err);
		} else {
			x_matchvalue = NULL;
		}
		x_margv = malloc(margc * sizeof(*x_margv));
		if (x_margv == NULL)
			errx(1, "Could not allocate memory");
		x_margv_extra = malloc(margc * sizeof(*x_margv_extra));
		if (x_margv_extra == NULL)
			errx(1, "Could not allocate memory");
		for (t = 0; t < (size_t)margc; t++) {
			x_margv[t] = pcre_compile(margv[t], 0, &pcre_err,
			    &pcre_ofs, NULL);
			if (pcre_err != NULL)
				errx(1, "Invalid match pattern: %s", pcre_err);
			x_margv_extra[t] = pcre_study(x_margv[t], 0,
			    &pcre_err);
			if (pcre_err != NULL)
				errx(1, "Invalid match pattern: %s", pcre_err);
		}
	}
#endif

	if (qsections && strcmp(backends[bidx]->name, "ini") != 0)
		errx(1, "The query for sections is only supported for the "
		    "'ini' backend for the present");
	backend = backends[bidx];
	return (0);
}

/***
 * Function:
 *	find_backend		- find a confget backend by name
 * Inputs:
 *	name			- the name of the backend
 * Returns:
 *	The index in backends[] if found, -1 otherwise.
 * Modifies:
 *	Nothing.
 */
static int
find_backend(const char * const name)
{
	size_t i, len;
	int tentative;
	
	len = strlen(name);
	for (i = 0, tentative = -1; i < BACKEND_CNT; i++) {
		if (strncmp(name, backends[i]->name, len))
			continue;
		if (backends[i]->name[len] == '\0')
			return (i);
		if (tentative >= 0)
			return (-1);
		tentative = i;
	}
	return (tentative);
}

/***
 * Function:
 *	openfile		- open the requested file for reading
 * Inputs:
 *	None.
 * Returns:
 *	0 on success, -1 on error.
 * Modifies:
 *	Stores the opened file into fp.
 */
static int
openfile(void)
{
	if (backend->openfile == NULL) {
		fprintf(stderr,
		    "INTERNAL ERROR: backend '%s' does not define a openfile routine\n",
		    backend->name);
		return (-1);
	}
	return (backend->openfile());
}

/***
 * Function:
 *	readfile		- scan an INI file for the requested variable
 * Inputs:
 *	None.
 * Returns:
 *	0 on success, -1 on error.
 * Modifies:
 *	May write to standard output if the variable has been found.
 */
static int
readfile(void)
{

	if (backend->readfile == NULL) {
		fprintf(stderr,
		    "INTERNAL ERROR: backend '%s' does not define a readfile routine\n",
		    backend->name);
		return (-1);
	}
	return (backend->readfile());
}

/***
 * Function:
 *	closefile		- close a scanned INI file
 * Inputs:
 *	None.
 * Returns:
 *	0 on success, -1 on error.
 * Modifies:
 *	Closes the file pointed to by fp.
 */
static int
closefile(void)
{

	if (backend->closefile == NULL) {
		fprintf(stderr,
		    "INTERNAL ERROR: backend '%s' does not define a closefile routine\n",
		    backend->name);
		return (-1);
	}
	return (backend->closefile());
}

/***
 * Function:
 *	foundvar	- process the user-requested variable
 * Inputs:
 *	sec		- the section the variable was found in
 *	name		- the variable name
 *	value		- the variable value
 * Returns:
 *	0 on success, -1 on failure.
 * Modifies:
 *	In check-only mode, sets cfound.
 *	In normal mode, writes to standard output.
 */
int
foundvar(const char * const sec __unused, const char * const name,
	const char * const value)
{
	int i;
#ifdef HAVE_PCRE
	int r;
#endif

	if (!lflag) {
		for (i = 0; i < margc; i++)
			if (Lflag) {
#ifdef HAVE_PCRE
				if (xflag) {
					r = pcre_exec(x_margv[i],
					    x_margv_extra[i], name,
					    strlen(name), 0, 0, NULL, 0);
					if (r == 0)
						break;
					if (r != PCRE_ERROR_NOMATCH)
						errx(1, "Could not match '%s' against the '%s' pattern", name, margv[i]);
				} else
#endif
				if (fnmatch(margv[i], name, 0) == 0)
					break;
			} else {
				if (!strcmp(name, margv[i]))
					break;
			}
		if (i == margc) {
			morevars = 1;
			return (0);
		}
	}

	if (matchvalue != NULL) {
		i = 1;
#ifdef HAVE_PCRE
		if (xflag) {
			r = pcre_exec(x_matchvalue, x_matchvalue_extra, value,
			    strlen(value), 0, 0, NULL, 0);
			if (r == PCRE_ERROR_NOMATCH)
				i = 0;
			else if (r != 0)
				errx(1, "Could not match '%s' against the '%s' pattern", value, margv[i]);
		} else
#endif
		if (fnmatch(matchvalue, value, 0) == FNM_NOMATCH) {
			i = 0;
		}
		if (i == 0) {
			morevars = 1;
			return (0);
		}
	}
	if (cflag) {
		cfound = 1;
		morevars = 0;
	} else {
		if (showvarname)
			printf("%s%s%s=", output_prefix, name, output_postfix);
		if (!Sflag) {
			printf("%s\n", value);
		} else {
			const char *p;

			printf("'");
			p = value;
			while (*p) {
				while (*p && *p != '\'')
					putchar(*p++);
				if (*p == '\0')
					break;
				printf("'\"");
				while (*p == '\'')
					putchar(*p++);
				printf("\"'");
			}
			printf("'\n");
		}
		morevars = manyvars;
	}
	return (0);
}

/***
 * Function:
 *	foundsection	- process a new section header
 * Inputs:
 *	sec		- the name of the new section
 * Returns:
 *	0 on success, -1 on failure.
 * Modifies:
 *	In "-q sections" mode, writes to standard output.
 */
int
foundsection(const char * const _sec)
{
	static size_t alloc = 0, count = 0;
	static char **names = NULL;
	size_t i;

	if (!qsections)
		return (0);

	for (i = 0; i < count; i++)
		if (strcmp(_sec, names[i]) == 0)
			return (0);
	/* A new section! */
	if (alloc == count) {
		size_t nalloc;
		char **newnames;
		
		nalloc = 2 * alloc + 1;
		newnames = realloc(names, nalloc * sizeof(names[0]));
		if (newnames == NULL) {
			warnx("Out of memory for the section names list");
			return (-1);
		}
		names = newnames;
		alloc = nalloc;
	}
	names[count] = strdup(_sec);
	if (names[count] == NULL) {
		warnx("Out of memory for the new section name");
		return (-1);
	}
	count++;

	/* And finally output it :) */
	puts(_sec);
	return (0);
}

/***
 * Main routine
 */
int
main(int argc, char * const *argv)
{

	if (init(argc, argv) == -1)
		return (1);

	cfound = 0;
	if (openfile() == -1)
		return (1);
	if (readfile() == -1) {
		closefile();
		return (1);
	}
	if (closefile() == -1)
		return (1);

	if (cflag)
		return (!cfound);
	return (0);
}
