/*
 * @(#)IScriptMaker.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.uicapture.v1;

import java.io.File;

/**
 * Implementors of this interface accept a collection of UI captured
 * events as they relate to creating a play-back script.  This interface
 * dictates that the contract for all IScriptMaker instances, the
 * {@link #start()} method
 * <b>must</b> be called before any other method, and no other method, besides
 * <tt>toString()</tt>, may be called after invoking the {@link #end()} method.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2003/02/10 22:52:31 $
 * @since     Jan 7, 2002
 */
public interface IScriptMaker
{

    //-------------------------------------------------------------------------
    // Public methods


    /**
     * Starts the script generation.
     *
     * @exception IllegalStateException if <tt>start()</tt> has already been
     *      called.
     */
    public void start();
    
    
    /**
     * Terminates the script generation.
     *
     * @exception IllegalStateException if <tt>start()</tt> has not been called,
     *      or if <tt>end()</tt> has already been called.
     */
    public void end();

    
    /**
     * Cause the script to delay for the given number of milliseconds.
     *
     * @param milliseconds the number of milliseconds in the delay.
     * @exception IllegalStateException if <tt>start()</tt> has not been called,
     *      or if <tt>end()</tt> has already been called.
     */
    public void generateDelay( long milliseconds );

    
    /**
     * Cause the script to rotate the mouse wheel.
     *
     * @param rotate the number of rotations the mouse wheel generated.
     * @exception IllegalStateException if <tt>start()</tt> has not been called,
     *      or if <tt>end()</tt> has already been called.
     * @see java.awt.event.MouseWheelEvent
     */
    public void generateMoveMouseWheel( int rotate );

    
    /**
     * Cause the script to move the mouse.
     *
     * @param x the x-coordinate that the mouse moved to
     * @param y the y-coordinate that the mouse moved to
     * @exception IllegalStateException if <tt>start()</tt> has not been called,
     *      or if <tt>end()</tt> has already been called.
     */
    public void generateMoveMouse( int x, int y );

    
    /**
     * Cause the script to press a mouse button.
     *
     * @param modifier the modifier code for the mouse button pressed.
     * @exception IllegalStateException if <tt>start()</tt> has not been called,
     *      or if <tt>end()</tt> has already been called.
     * @see java.awt.event.MouseEvent
     */
    public void generatePressMouse( int modifer );

    
    /**
     * Cause the script to release a mouse button.
     *
     * @param modifier the modifier code for the mouse button released.
     * @exception IllegalStateException if <tt>start()</tt> has not been called,
     *      or if <tt>end()</tt> has already been called.
     * @see java.awt.event.MouseEvent
     */
    public void generateReleaseMouse( int modifier );

    
    /**
     * Cause the script to press a key.
     *
     * @param keyCode code for the pressed key.
     * @exception IllegalStateException if <tt>start()</tt> has not been called,
     *      or if <tt>end()</tt> has already been called.
     * @see java.awt.event.KeyEvent
     */
    public void generatePressKey( int keyCode );

    
    /**
     * Cause the script to release a key.
     *
     * @param keyCode code for the released key.
     * @exception IllegalStateException if <tt>start()</tt> has not been called,
     *      or if <tt>end()</tt> has already been called.
     * @see java.awt.event.KeyEvent
     */
    public void generateReleaseKey( int keyCode );

    
    /**
     * Cause the script to capture a screen shot.  Probably, it should compare
     * the captured image against the original saved image.
     *
     * @param originalImage file containing the original image for the screen
     *      capture that the generated script will compare against when the
     *      test is run.
     * @param x screen image bounds.
     * @param y screen image bounds.
     * @param width screen image bounds.
     * @param height screen image bounds.
     *
     * @exception IllegalStateException if <tt>start()</tt> has not been called,
     *      or if <tt>end()</tt> has already been called.
     */
    public void generateScreenCapture( File originalImage,
            int x, int y, int width, int height );
}

