/*
 * @(#)BackgroundProcess.java    1.0.0 11/17/2000 - 13:41:07
 *
 * Copyright (C) 2000,,2003 2002 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.util.thread.v1;


import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;

import java.lang.reflect.Method;


/**
 * Creates and executes the given process.  Ensures that all the output
 * is properly read without overflowing or dead-locking the process.
 * Outside of the streaming, this class has an identical API to that
 * of {@link java.lang.Process}.
 * <P>
 * Creation of a background process begins at the creation of this object.
 * <P>
 * <H3>Changes for 0.9.1d</H3>
 * <UL>
 *      <LI>Each IOThreadRunner which handles the background process's
 *          IO have been changed such that they now close the streams when
 *          an EOL is encountered, and have no delay between reads.
 * </UL>
 *
 * @author   Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @since    June 4, 2000 (0.9.1d Alpha)
 * @version  $Date: 2003/02/10 22:52:48 $
 */
public class BackgroundProcess
{
    private Process proc;
    private OutputStream stdIn;
    private PipedInputStream outReader;
    private PipedOutputStream outWriter;
    private PipedInputStream errReader;
    private PipedOutputStream errWriter;
    
    private IOThreadRunner outThread;
    private IOThreadRunner errThread;
    
    
    /**
     * 
     *
     * @see java.lang.Runtime#exec( String )
     */
    public BackgroundProcess( String command )
        throws IOException
    {
        setupProcess( exec( command, null, null ) );
    }
    
    
    /**
     * @see java.lang.Runtime#exec( String[] )
     */
    public BackgroundProcess( String[] cmdarray )
        throws IOException
    {
        setupProcess( exec( cmdarray, null, null ) );
    }
    
    
    /**
     * @see java.lang.Runtime#exec( String[], String[], File )
     */
    public BackgroundProcess( String[] cmdarray, String[] envp, File dir )
        throws IOException
    {
        setupProcess( exec( cmdarray, envp, dir ) );
    }
    
    
    /**
     * @see java.lang.Runtime#exec( String, String[], File )
     */
    public BackgroundProcess( String command, String[] envp, File dir )
        throws IOException
    {
        setupProcess( exec( command, envp, dir ) );
    }
    
    
    /**
     * @see java.lang.Runtime#exec( String[], String[] )
     */
    public BackgroundProcess( String[] cmdarray, String[] envp )
        throws IOException
    {
        setupProcess( exec( cmdarray, envp, null ) );
    }
    
    
    /**
     * @see java.lang.Runtime#exec( String, String[] )
     */
    public BackgroundProcess( String command, String[] envp )
        throws IOException
    {
        setupProcess( exec( command, envp, null ) );
    }
    
    
    /**
     * Initializes the background process with an existing process.
     */
    public BackgroundProcess( Process p )
        throws IOException
    {
        if (p == null)
        {
            throw new IllegalArgumentException("no null args");
        }
        setupProcess( p );
    }
    
    
    /**
     * Get the OutputStream that is sent to the StdIn of the process.
     */
    public OutputStream getStdIn()
    {
        return this.stdIn;
    }
    
    
    /**
     * Get the InputStream that retrieves the data from the StdOut of the
     * process.
     */
    public InputStream getStdOut()
    {
        return this.outReader;
    }
    
    
    /**
     * Get the InputStream that retrieves the data from the StdErr of the
     * process.
     */
    public InputStream getStdErr()
    {
        return this.errReader;
    }
    
    
    /**
     * @see java.lang.Process#destroy()
     */
    public void destroy()
    {
        this.proc.destroy();
    }
    
    
    /**
     * @see java.lang.Process#exitValue()
     */
    public int exitValue()
    {
        return this.proc.exitValue();
    }
    

    /**
     * @see java.lang.Process#waitFor()
     */
    public int waitFor()
            throws InterruptedException
    {
        return this.proc.waitFor();
    }
    
    
    //-------------------------------------------------
    // Protected methods
    
    
    /**
     * Initalize the process for internal use.
     */
    protected void setupProcess( Process p )
            throws IOException
    {
        this.proc = p;
        
        this.stdIn = p.getOutputStream();
        
        this.outReader = new PipedInputStream();
        this.outWriter = new PipedOutputStream( this.outReader );
        
        this.errReader = new PipedInputStream();
        this.errWriter = new PipedOutputStream( this.errReader );
        
        this.outThread =
            new IOThreadRunner( p.getInputStream(), this.outWriter );
        this.outThread.setCloseInputOnStop( true );
        this.outThread.setCloseOutputOnStop( true );
        this.outThread.getThread().setSleepTime( 0 );

        this.errThread =
            new IOThreadRunner( p.getErrorStream(), this.errWriter );
        this.errThread.setCloseInputOnStop( true );
        this.errThread.setCloseOutputOnStop( true );
        this.errThread.getThread().setSleepTime( 0 );
        
        
        this.outThread.start();
        this.errThread.start();
    }
    
    
    
    /**
     * Invoke the correct Exec method for the given parameters.
     */
    protected Process exec( String cmd, String[] env, File dir )
            throws IOException
    {
        Runtime r = Runtime.getRuntime();
        Process p;
        if (dir == null)
        {
            if (env == null)
            {
                p = r.exec( cmd );
            }
            else
            {
                p = r.exec( cmd, env );
            }
        }
        else
        {
            p = reflectExec( cmd, env, dir );
            if (p == null)
            {
                p = r.exec( cmd, setPWD( env, dir ) );
            }
        }
        return p;
    }
    
    
    /**
     * 
     */
    protected static Process exec( String[] cmd, String[] env, File dir )
            throws IOException
    {
        Runtime r = Runtime.getRuntime();
        Process p;
        if (dir == null)
        {
            if (env == null)
            {
                p = r.exec( cmd );
            }
            else
            {
                p = r.exec( cmd, env );
            }
        }
        else
        {
            p = reflectExec( cmd, env, dir );
            if (p == null)
            {
                p = r.exec( cmd, setPWD( env, dir ) );
            }
        }
        return p;
    }
    
    
    /**
     * Attempts to invoke the JDK 1.3 exec method with the given dir.
     * <tt>cmd</tt> is either a String or String[] type.
     *
     * @return the generated Process.  If the process is <tt>null</tt>,
     *      then another technique must be used to execute the command.
     */
    protected static Process reflectExec( Object cmd, String[] env, File dir )
            throws IOException
    {
        try
        {
            Runtime r = Runtime.getRuntime();
            Method m = r.getClass().getMethod( "exec",
                new Class[] { cmd.getClass(), String[].class,
                    File.class }
                );
            if (m == null)
            {
                return null;
            }
            return (Process)m.invoke( r, new Object[] { cmd, env, dir } );
        }
        catch (java.lang.reflect.InvocationTargetException ite)
        {
            Throwable t = ite.getTargetException();
            if (t instanceof IOException)
            {
                throw (IOException)t;
            }
            if (t instanceof RuntimeException)
            {
                throw (RuntimeException)t;
            }
            if (t instanceof Error)
            {
                throw (Error)t;
            }
            // else swallow it
            t.printStackTrace();
        }
        catch (IllegalAccessException e)
        {
            // gulp!
        }
        catch (NoSuchMethodException e)
        {
            // gulp!
        }
        catch (IllegalArgumentException e)
        {
            // gulp!
        }
        /* allow this to be thrown - it indicates a programatic error
        catch (NullPointerException e)
        {
        }
        */
        /* allow this to be thrown
        catch (ExceptionInInitializerError e)
        {
        }
        */
        return null;
    }
    
    
    /**
     * Don't change <tt>env</tt>!
     */
    protected static String[] setPWD( String env[], File dir )
    {
        String tmp[];
        String pwd = "PWD="+dir.getAbsolutePath();
        if (env == null)
        {
            tmp = new String[] { pwd };
        }
        else
        {
            int len = env.length;
            tmp = new String[ len ];
            System.arraycopy( env, 0, tmp, 0, len );
            String s[] = new String[ len + 1 ];
            for (int i = 0; i < len; ++i)
            {
                if (tmp[i].startsWith( "PWD=" ))
                {
                    tmp[i] = pwd;
                    s = null;
                    break;
                }
                else
                {
                    s[i] = tmp[i];
                }
            }
            if (s != null)
            {
                s[ env.length ] = pwd;
                tmp = s;
            }
        }
        return tmp;
    }
}
