/* BEGIN software license
 *
 * msXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2019 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the msXpertSuite project.
 *
 * The msXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// StdLib includes


/////////////////////// Qt includes
#include <QSettings>
#include <QMenuBar>
#include <QMenu>
#include <QDebug>
#include <QDialog>


/////////////////////// pappsomspp includes
#include <pappsomspp/msrun/private/timsmsrunreader.h>

/////////////////////// Local includes
#include "TicChromPreviewerMassDataLoadDlg.hpp"
#include "ProgramWindow.hpp"
#include "Application.hpp"
#include "../nongui/MassDataIntegratorTask.hpp"


namespace msxps
{
namespace minexpert
{


//! Construct an TicChromPreviewerMassDataLoadDlg instance.
TicChromPreviewerMassDataLoadDlg::TicChromPreviewerMassDataLoadDlg(
  QWidget *parent,
  const QString &title,
  const QString &settingsTitle,
  const QString &description,
  pappso::MsRunIdCstSPtr ms_run_id_csp,
  MsRunDataPreviewInfo &ms_run_data_preview_info)
  : QDialog(parent),
    mp_programWindow(static_cast<ProgramWindow *>(parent)),
    m_title(title),
    m_settingsTitle(settingsTitle),
    m_description(description),
    m_msRunIdCstSPtr(ms_run_id_csp),
    m_msRunDataPreviewInfo(ms_run_data_preview_info)
{
  if(parent == Q_NULLPTR)
    qFatal("Programming error.");

  if(ms_run_id_csp == nullptr)
    qFatal("Programming error.");

  m_ui.setupUi(this);

  if(!initialize())
    qFatal("Programming error.");
}


//! Destruct \c this TicChromPreviewerMassDataLoadDlg instance.
TicChromPreviewerMassDataLoadDlg::~TicChromPreviewerMassDataLoadDlg()
{
  // qDebug();
  writeSettings();
}


//! Write the settings to as to restore the window geometry later.
void
TicChromPreviewerMassDataLoadDlg::writeSettings()
{
  QSettings settings(static_cast<Application *>(QCoreApplication::instance())
                       ->getUserConfigSettingsFilePath(),
                     QSettings::IniFormat);

  settings.beginGroup(m_settingsTitle);

  settings.setValue("geometry", saveGeometry());
  // settings.setValue("splitter", m_ui.splitter->saveState());
  settings.setValue("RtTimeUnitSeconds", m_areRtTimeUnitSeconds);

  settings.endGroup();
}


//! Read the settings to as to restore the window geometry.
void
TicChromPreviewerMassDataLoadDlg::readSettings()
{
  QSettings settings(static_cast<Application *>(QCoreApplication::instance())
                       ->getUserConfigSettingsFilePath(),
                     QSettings::IniFormat);

  settings.beginGroup(m_settingsTitle);

  restoreGeometry(settings.value("geometry").toByteArray());
  // m_ui.splitter->restoreState(settings.value("splitter").toByteArray());

  // By default we want retention times in seconds (this is in the Bruker data).
  m_areRtTimeUnitSeconds = settings.value("RtTimeUnitSeconds", true).toBool();

  m_ui.rtInSecondsRadioButton->setChecked(m_areRtTimeUnitSeconds);

  settings.endGroup();
}


//! Handle the close event.
void
TicChromPreviewerMassDataLoadDlg::closeEvent(
  [[maybe_unused]] QCloseEvent *event)
{
  // qDebug();
  writeSettings();
  QDialog::reject();
}


ProgramWindow *
TicChromPreviewerMassDataLoadDlg::getProgramWindow()
{
  return mp_programWindow;
}


//! Initialize the window.
bool
TicChromPreviewerMassDataLoadDlg::initialize()
{
  setWindowTitle(QString("mineXpert2 - %1").arg(m_title));

  setAttribute(Qt::WA_DeleteOnClose);

  // The default window icon.
  QString pixmap_file_name = ":/images/icons/32x32/minexpert2.png";
  QPixmap icon_pixmap(pixmap_file_name);
  QIcon icon(icon_pixmap);
  setWindowIcon(icon);

  // Will read the retention time unit. By default, minutes.
  readSettings();

  /****************** The widgetry  ******************/

  // Now that we have read the settings, check the right radio button.
  m_ui.rtInSecondsRadioButton->setChecked(m_areRtTimeUnitSeconds);
  m_ui.rtInMinutesRadioButton->setChecked(!m_areRtTimeUnitSeconds);

  m_ui.ms1SpectraCountSpinBox->setValue(m_msRunDataPreviewInfo.ms1SpectraCount);
  m_ui.ms2SpectraCountSpinBox->setValue(m_msRunDataPreviewInfo.ms2SpectraCount);

  mp_plotWidget = new pappso::TicXicChromTracePlotWidget(
    this, "Retention time", "Intensity (a.u.)");

  m_ui.qcpTracePlotWidgetHorizontalLayout->addWidget(mp_plotWidget);

  // Set the widest range possible for retention times
  m_ui.rtRangeBeginDoubleSpinBox->setRange(std::numeric_limits<double>::min(),
                                           std::numeric_limits<double>::max());

  connect(
    m_ui.rtRangeBeginDoubleSpinBox, &QDoubleSpinBox::editingFinished, [&]() {
      mp_plotWidget->xAxis->setRange(
        QCPRange(m_ui.rtRangeBeginDoubleSpinBox->value(),
                 mp_plotWidget->xAxis->range().upper));

      // setRange orders lower and upper in  the right  ascending order. If the
      // user enters an upper value < lower value, then setRange fixes this.
      // This is why we take advantage of this and call updateRtRangeSpinBoxes.
      updateRtRangeSpinBoxes();
      mp_plotWidget->replot();
    });

  m_ui.rtRangeEndDoubleSpinBox->setRange(std::numeric_limits<double>::min(),
                                         std::numeric_limits<double>::max());

  connect(
    m_ui.rtRangeEndDoubleSpinBox, &QDoubleSpinBox::editingFinished, [&]() {
      mp_plotWidget->xAxis->setRange(
        QCPRange(mp_plotWidget->xAxis->range().lower,
                 m_ui.rtRangeEndDoubleSpinBox->value()));

      // setRange orders lower and upper in  the right  ascending order. If the
      // user enters an upper value < lower value, then setRange fixes this.
      // This is why we take advantage of this and call updateRtRangeSpinBoxes.
      updateRtRangeSpinBoxes();
      mp_plotWidget->replot();
    });

  connect(m_ui.rtInSecondsRadioButton, &QRadioButton::clicked, [&]() {
    if(!m_areRtTimeUnitSeconds)
      {
        // We are converting the data to seconds.
        convertTicChromatogramRetTimeUnit();
        mp_plotWidget->replot();
        m_areRtTimeUnitSeconds = true;
      }
  });

  connect(m_ui.rtInMinutesRadioButton, &QRadioButton::clicked, [&]() {
    if(m_areRtTimeUnitSeconds)
      {
        // We are converting the data to minutes.
        convertTicChromatogramRetTimeUnit();
        mp_plotWidget->replot();
        m_areRtTimeUnitSeconds = false;
      }
  });

  connect(mp_plotWidget,
          &pappso::TicXicChromTracePlotWidget::plotRangesChangedSignal,
          [&]() {
            m_ui.rtRangeBeginDoubleSpinBox->setValue(
              mp_plotWidget->xAxis->range().lower);
            m_ui.rtRangeEndDoubleSpinBox->setValue(
              mp_plotWidget->xAxis->range().upper);

            qDebug() << "New x axis range:"
                     << mp_plotWidget->xAxis->range().lower << "--"
                     << mp_plotWidget->xAxis->range().upper;
          });

  // Detect if the user wants to flatten the IM spectra of a frame into a single
  // spectrum.
  connect(m_ui.flattenImScansPushButton, &QPushButton::clicked, [&]() {
    qDebug() << "Clicked to flatten IM scans.";

    m_msRunDataPreviewInfo.flattenIonMobilityScans = true;

    if(!validateData())
      return;
    else
      QDialog::accept();
  });

  // Detect if the user wants to keep all the IM spectra of a frame.
  connect(m_ui.keepImScansPushButton, &QPushButton::clicked, [&]() {
    qDebug() << "Clicked to keep IM scans.";

    m_msRunDataPreviewInfo.flattenIonMobilityScans = false;

    if(!validateData())
      return;
    else
      QDialog::accept();
  });

  connect(m_ui.cancelPushButton, &QPushButton::clicked, [&]() {
    qDebug() << "Clicked to cancel.";

    QDialog::reject();
  });

  connect(m_ui.imOneOverKoGroupBox, &QGroupBox::clicked, [&](bool checked) {
    m_ui.imScansGroupBox->setChecked(!checked);
  });
  connect(m_ui.imScansGroupBox, &QGroupBox::clicked, [&](bool checked) {
    m_ui.imOneOverKoGroupBox->setChecked(!checked);
  });

  // Now get the TIC chromatogram out of the file in question.

  pappso::TimsMsRunReader tims_ms_run_reader =
    pappso::TimsMsRunReader(m_msRunIdCstSPtr);

  pappso::Trace tic_chrom_trace = tims_ms_run_reader.getTicChromatogram();

  // Note that the TIC chromatogram from the Bruker timsTOF data files has time
  // unit seconds, not minutes.

  if(!m_areRtTimeUnitSeconds)
    {
      // The user wants unit to be min, so divide the seconds from the Bruker
      // data file by 60.
      std::for_each(tic_chrom_trace.begin(),
                    tic_chrom_trace.end(),
                    [](pappso::DataPoint &dp) { dp.x /= 60; });
    }

  mp_plotWidget->addTrace(tic_chrom_trace, QColor(180, 0, 0));

  updateRtRangeSpinBoxes();

  // Now all the other data.

  m_ui.imOneOverKoRangeBeginDoubleSpinBox->setValue(
    m_msRunDataPreviewInfo.ionMobilityOneOverKoBegin);
  m_ui.imOneOverKoRangeEndDoubleSpinBox->setValue(
    m_msRunDataPreviewInfo.ionMobilityOneOverKoEnd);

  m_ui.imScansRangeBeginSpinBox->setValue(
    m_msRunDataPreviewInfo.ionMobilityScansBegin);
  m_ui.imScansRangeEndSpinBox->setValue(
    m_msRunDataPreviewInfo.ionMobilityScansEnd);

  m_ui.mzRangeBeginDoubleSpinBox->setValue(m_msRunDataPreviewInfo.mzBegin);
  m_ui.mzRangeEndDoubleSpinBox->setValue(m_msRunDataPreviewInfo.mzEnd);

  // It is necessary that m_msRunDataPreviewInfo.mzBinCount be non-naught.
  if(m_msRunDataPreviewInfo.mzBinCount)
    {
      m_ui.binCountSpinBox->setValue(m_msRunDataPreviewInfo.mzBinCount);

      double mz_bin_size =
        (m_msRunDataPreviewInfo.mzEnd - m_msRunDataPreviewInfo.mzBegin) /
        static_cast<double>(m_msRunDataPreviewInfo.mzBinCount);

      m_ui.mzBinSizeDoubleSpinBox->setValue(mz_bin_size);
    }
  else
    m_ui.mzBinSizeDoubleSpinBox->setValue(0);

  // At this point we need to configure the behaviour of the m/z bin radiobutton
  // widgets

  connect(
    m_ui.mzMergeBinsByMzDoubleSpinBox, &QDoubleSpinBox::editingFinished, [&]() {
      double mz_window = m_ui.mzMergeBinsByMzDoubleSpinBox->value();
      m_ui.mzMergeBinsByMzRadioButton->setChecked(true);

      if(m_msRunDataPreviewInfo.mzBinCount)
        {
          double mz_bin_size =
            (m_msRunDataPreviewInfo.mzEnd - m_msRunDataPreviewInfo.mzBegin) /
            static_cast<double>(m_msRunDataPreviewInfo.mzBinCount);

          // Compute the equivalent TOF indices window (that is, the
          // number of bins)
          uint mz_bin_count = mz_window / mz_bin_size;
          m_ui.mzMergeBinsByCountSpinBox->setValue(mz_bin_count);
        }
    });


  connect(
    m_ui.mzMergeBinsByCountSpinBox, &QDoubleSpinBox::editingFinished, [&]() {
      uint mz_bin_count = m_ui.mzMergeBinsByCountSpinBox->value();
      m_ui.mzMergeBinsByCountRadioButton->setChecked(true);

      if(m_msRunDataPreviewInfo.mzBinCount)
        {
          double mz_bin_size =
            (m_msRunDataPreviewInfo.mzEnd - m_msRunDataPreviewInfo.mzBegin) /
            static_cast<double>(m_msRunDataPreviewInfo.mzBinCount);

          // Compute the equivalent m/z window (that is, the
          // size of the m/z bin in which m/z values will be packed)
          double mz_window = mz_bin_count * mz_bin_size;
          m_ui.mzMergeBinsByMzDoubleSpinBox->setValue(mz_window);
        }
    });

  // Now that we have configured the way the m/z-related widgets behave, we can
  // set the default m/z window to the m/z bin size deduced by looking at the
  // sample spectrum.

  if(m_msRunDataPreviewInfo.mzBinCount)
  {
    double mz_bin_size =
    (m_msRunDataPreviewInfo.mzEnd - m_msRunDataPreviewInfo.mzBegin) /
    static_cast<double>(m_msRunDataPreviewInfo.mzBinCount);
  
    m_ui.mzMergeBinsByMzDoubleSpinBox->setValue(mz_bin_size);
  }

  m_ui.renderMzRangeGroupBox->setChecked(false);
  m_ui.mergeMzBinsGroupBox->setChecked(false);

  return true;
}


void
TicChromPreviewerMassDataLoadDlg::show()
{
  QSettings settings(static_cast<Application *>(QCoreApplication::instance())
                       ->getUserConfigSettingsFilePath(),
                     QSettings::IniFormat);
  settings.beginGroup(m_settingsTitle);

  restoreGeometry(settings.value(m_settingsTitle + "_geometry").toByteArray());

  settings.endGroup();

  QDialog::show();
}


void
TicChromPreviewerMassDataLoadDlg::showWindow()
{
  activateWindow();
  raise();
  show();
}

void
TicChromPreviewerMassDataLoadDlg::convertTicChromatogramRetTimeUnit()
{
  QSharedPointer<QCPGraphDataContainer> graph_data_container_p =
    mp_plotWidget->graph(0)->data();

  QCPGraphDataContainer::iterator iter     = graph_data_container_p->begin();
  QCPGraphDataContainer::iterator iter_end = graph_data_container_p->end();
  QCPRange prev_range(mp_plotWidget->xAxis->range());
  QCPRange new_range;

  if(m_areRtTimeUnitSeconds)
    {
      // Convert to minutes.

      while(iter != iter_end)
        {
          iter->key /= 60;
          ++iter;
        }
      mp_plotWidget->xAxis->setRange(
        QCPRange(prev_range.lower / 60, prev_range.upper / 60));
    }
  else
    {
      // Convert to seconds

      while(iter != iter_end)
        {
          iter->key *= 60;
          ++iter;
        }
      mp_plotWidget->xAxis->setRange(
        QCPRange(prev_range.lower * 60, prev_range.upper * 60));
    }

  updateRtRangeSpinBoxes();
}

void
TicChromPreviewerMassDataLoadDlg::updateRtRangeSpinBoxes()
{
  m_ui.rtRangeBeginDoubleSpinBox->setValue(mp_plotWidget->xAxis->range().lower);
  m_ui.rtRangeEndDoubleSpinBox->setValue(mp_plotWidget->xAxis->range().upper);
}

bool
TicChromPreviewerMassDataLoadDlg::validateData()
{
  if(m_ui.retentionTimeRangeGroupBox->isChecked())
    {
      if(m_ui.rtInSecondsRadioButton->isChecked())
        {
          m_msRunDataPreviewInfo.retentionTimeBeginInSeconds =
            m_ui.rtRangeBeginDoubleSpinBox->value();
          m_msRunDataPreviewInfo.retentionTimeEndInSeconds =
            m_ui.rtRangeEndDoubleSpinBox->value();
        }
      else
        {
          m_msRunDataPreviewInfo.retentionTimeBeginInSeconds =
            m_ui.rtRangeBeginDoubleSpinBox->value() * 60;
          m_msRunDataPreviewInfo.retentionTimeEndInSeconds =
            m_ui.rtRangeEndDoubleSpinBox->value() * 60;
        }
    }
  else
    {
      m_msRunDataPreviewInfo.retentionTimeBeginInSeconds = -1;
      m_msRunDataPreviewInfo.retentionTimeEndInSeconds   = -1;
    }

  if(m_ui.ionMobilityRangeGroupBox->isChecked())
    {
      if(m_ui.imOneOverKoGroupBox->isChecked())
        {
          m_msRunDataPreviewInfo.ionMobilityUnit = IonMobilityUnit::ONE_OVER_KO;
          m_msRunDataPreviewInfo.ionMobilityOneOverKoBegin =
            m_ui.imOneOverKoRangeBeginDoubleSpinBox->value();
          m_msRunDataPreviewInfo.ionMobilityOneOverKoEnd =
            m_ui.imOneOverKoRangeEndDoubleSpinBox->value();
        }
      else if(m_ui.imScansGroupBox->isChecked())
        {
          m_msRunDataPreviewInfo.ionMobilityUnit = IonMobilityUnit::SCAN;
          m_msRunDataPreviewInfo.ionMobilityScansBegin =
            m_ui.imScansRangeBeginSpinBox->value();
          m_msRunDataPreviewInfo.ionMobilityScansEnd =
            m_ui.imScansRangeEndSpinBox->value();
        }
      else
        qFatal("Cannot be...");
    }
  else
    {
      m_msRunDataPreviewInfo.ionMobilityUnit = IonMobilityUnit::NONE;
    }

  QString ms_levels_to_render = m_ui.msLevelsLineEdit->text();

  bool ok = false;

  if(!ms_levels_to_render.isEmpty())
    {
      qDebug() << "The MS levels line edit widget contains this text:"
               << ms_levels_to_render;

      QStringList ms_levels_string_list =
        ms_levels_to_render.split(",", Qt::SkipEmptyParts);

      m_msRunDataPreviewInfo.msLevels.clear();

      for(int iter = 0; iter < ms_levels_string_list.size(); ++iter)
        {
          QString ms_level_string = ms_levels_string_list.at(iter).trimmed();

          std::size_t ms_level = ms_level_string.toUInt(&ok);

          if(!ok)
            {
              QMessageBox::warning(this,
                                   "Mass spectrometry file loading preview",
                                   "The MS levels failed to validate fully.");
              return false;
            }


          m_msRunDataPreviewInfo.msLevels.push_back(ms_level);
        }
    }
  else
    {
      m_msRunDataPreviewInfo.msLevels.clear();
    }

  if(m_ui.renderMzRangeGroupBox->isChecked())
    {
      m_msRunDataPreviewInfo.mzBegin = m_ui.mzRangeBeginDoubleSpinBox->value();
      m_msRunDataPreviewInfo.mzEnd   = m_ui.mzRangeEndDoubleSpinBox->value();
    }
  else
    {
      m_msRunDataPreviewInfo.mzBegin = -1;
      m_msRunDataPreviewInfo.mzEnd   = -1;
    }

  if(m_ui.mergeMzBinsGroupBox->isChecked())
    {
      qDebug() << "The merge m/z bins is asked for.";

      // Since the m/z window is computed automatically (connection in the
      // initialize) just read the mz window.
      m_msRunDataPreviewInfo.mzBinsMergeWindow =
        m_ui.mzMergeBinsByMzDoubleSpinBox->value();
    }

  return true;
}

} // namespace minexpert

} // namespace msxps
