% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MCMCpoisson.R
\name{MCMCpoisson}
\alias{MCMCpoisson}
\title{Markov Chain Monte Carlo for Poisson Regression}
\usage{
MCMCpoisson(formula, data = NULL, burnin = 1000, mcmc = 10000, thin = 1,
  tune = 1.1, verbose = 0, seed = NA, beta.start = NA, b0 = 0,
  B0 = 0, marginal.likelihood = c("none", "Laplace"), ...)
}
\arguments{
\item{formula}{Model formula.}

\item{data}{Data frame.}

\item{burnin}{The number of burn-in iterations for the sampler.}

\item{mcmc}{The number of Metropolis iterations for the sampler.}

\item{thin}{The thinning interval used in the simulation.  The number of
mcmc iterations must be divisible by this value.}

\item{tune}{Metropolis tuning parameter. Can be either a positive scalar or
a \eqn{k}-vector, where \eqn{k} is the length of
\eqn{\beta}.Make sure that the acceptance rate is satisfactory
(typically between 0.20 and 0.5) before using the posterior sample for
inference.}

\item{verbose}{A switch which determines whether or not the progress of the
sampler is printed to the screen.  If \code{verbose} is greater than 0 the
iteration number, the current beta vector, and the Metropolis acceptance
rate are printed to the screen every \code{verbose}th iteration.}

\item{seed}{The seed for the random number generator.  If NA, the Mersenne
Twister generator is used with default seed 12345; if an integer is passed
it is used to seed the Mersenne twister.  The user can also pass a list of
length two to use the L'Ecuyer random number generator, which is suitable
for parallel computation.  The first element of the list is the L'Ecuyer
seed, which is a vector of length six or NA (if NA a default seed of
\code{rep(12345,6)} is used).  The second element of list is a positive
substream number. See the MCMCpack specification for more details.}

\item{beta.start}{The starting value for the \eqn{\beta} vector.  This
can either be a scalar or a column vector with dimension equal to the number
of betas. If this takes a scalar value, then that value will serve as the
starting value for all of the betas.  The default value of NA will use the
maximum likelihood estimate of \eqn{\beta} as the starting value.}

\item{b0}{The prior mean of \eqn{\beta}.  This can either be a scalar
or a column vector with dimension equal to the number of betas. If this
takes a scalar value, then that value will serve as the prior mean for all
of the betas.}

\item{B0}{The prior precision of \eqn{\beta}.  This can either be a
scalar or a square matrix with dimensions equal to the number of betas.  If
this takes a scalar value, then that value times an identity matrix serves
as the prior precision of \eqn{\beta}. Default value of 0 is
equivalent to an improper uniform prior for beta.}

\item{marginal.likelihood}{How should the marginal likelihood be calculated?
Options are: \code{none} in which case the marginal likelihood will not be
calculated or \code{Laplace} in which case the Laplace approximation (see
Kass and Raftery, 1995) is used.}

\item{...}{further arguments to be passed.}
}
\value{
An mcmc object that contains the posterior sample.  This object can
be summarized by functions provided by the coda package.
}
\description{
This function generates a sample from the posterior distribution of a
Poisson regression model using a random walk Metropolis algorithm. The user
supplies data and priors, and a sample from the posterior distribution is
returned as an mcmc object, which can be subsequently analyzed with
functions provided in the coda package.
}
\details{
\code{MCMCpoisson} simulates from the posterior distribution of a Poisson
regression model using a random walk Metropolis algorithm. The simulation
proper is done in compiled C++ code to maximize efficiency.  Please consult
the coda documentation for a comprehensive list of functions that can be
used to analyze the posterior sample.

The model takes the following form:

\deqn{y_i \sim \mathcal{P}oisson(\mu_i)}

Where the inverse link
function:

\deqn{\mu_i = \exp(x_i'\beta)}

We assume a multivariate Normal prior on \eqn{\beta}:

\deqn{\beta \sim \mathcal{N}(b_0,B_0^{-1})}

The Metropois proposal distribution is centered at the current value of
\eqn{\theta} and has variance-covariance \eqn{V = T (B_0 + C^{-1})^{-1} T }
where \eqn{T} is a the diagonal positive definite matrix formed from the
\code{tune}, \eqn{B_0} is the prior precision, and \eqn{C} is the
large sample variance-covariance matrix of the MLEs. This last calculation
is done via an initial call to \code{glm}.
}
\examples{

   \dontrun{
   counts <- c(18,17,15,20,10,20,25,13,12)
   outcome <- gl(3,1,9)
   treatment <- gl(3,3)
   posterior <- MCMCpoisson(counts ~ outcome + treatment)
   plot(posterior)
   summary(posterior)
   }

}
\references{
Andrew D. Martin, Kevin M. Quinn, and Jong Hee Park. 2011.
``MCMCpack: Markov Chain Monte Carlo in R.'', \emph{Journal of Statistical
Software}. 42(9): 1-21.  \url{http://www.jstatsoft.org/v42/i09/}.

Daniel Pemstein, Kevin M. Quinn, and Andrew D. Martin.  2007.  \emph{Scythe
Statistical Library 1.0.} \url{http://scythe.wustl.edu}.

Martyn Plummer, Nicky Best, Kate Cowles, and Karen Vines. 2006.  ``Output
Analysis and Diagnostics for MCMC (CODA)'', \emph{R News}. 6(1): 7-11.
\url{https://CRAN.R-project.org/doc/Rnews/Rnews_2006-1.pdf}.
}
\seealso{
\code{\link[coda]{plot.mcmc}},\code{\link[coda]{summary.mcmc}},
\code{\link[stats]{glm}}
}
\keyword{models}
