% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/polyCub.SV.R
\name{polyCub.SV}
\alias{polyCub.SV}
\title{Product Gauss Cubature over Polygonal Domains}
\usage{
polyCub.SV(polyregion, f, ..., nGQ = 20, alpha = NULL,
  rotation = FALSE, engine = "C", plot = FALSE)
}
\arguments{
\item{polyregion}{a polygonal domain.
The following classes are supported:
\code{"\link[spatstat]{owin}"} from package \pkg{spatstat},
\code{"\link[rgeos:gpc.poly-class]{gpc.poly}"} from \pkg{rgeos} (or
\pkg{gpclib}), as well as
\code{"\linkS4class{SpatialPolygons}"}, \code{"\linkS4class{Polygons}"},
and \code{"\linkS4class{Polygon}"} from package \pkg{sp}.
(For these classes, \pkg{polyCub} knows how to get an \code{\link{xylist}}.)}

\item{f}{a two-dimensional real-valued function to be integrated over
\code{polyregion} (or \code{NULL} to only compute nodes and weights).
As its first argument it must take a coordinate matrix, i.e., a
numeric matrix with two columns, and it must return a numeric vector of
length the number of coordinates.}

\item{...}{further arguments for \code{f}.}

\item{nGQ}{degree of the one-dimensional Gauss-Legendre quadrature rule
(default: 20) as implemented in function \code{\link[statmod]{gauss.quad}}
of package \pkg{statmod}. Nodes and weights up to \code{nGQ=60} are cached
in \pkg{polyCub}, for larger degrees \pkg{statmod} is required.}

\item{alpha}{base-line of the (rotated) polygon at \eqn{x = \alpha} (see
Sommariva and Vianello (2007) for an explication). If \code{NULL} (default),
the midpoint of the x-range of each polygon is chosen if no \code{rotation}
is performed, and otherwise the \eqn{x}-coordinate of the rotated point
\code{"P"} (see \code{rotation}). If \code{f} has its maximum value at the
origin \eqn{(0,0)}, e.g., the bivariate Gaussian density with zero mean,
\code{alpha = 0} is a reasonable choice.}

\item{rotation}{logical (default: \code{FALSE}) or a list of points
\code{"P"} and \code{"Q"} describing the preferred direction. If
\code{TRUE}, the polygon is rotated according to the vertices \code{"P"} and
\code{"Q"}, which are farthest apart (see Sommariva and Vianello, 2007). For
convex polygons, this rotation guarantees that all nodes fall inside the
polygon.}

\item{engine}{character string specifying the implementation to use.
Up to \pkg{polyCub} version 0.4-3, the two-dimensional nodes and weights
were computed by \R functions and these are still available by setting
\code{engine = "R"}.
The new C-implementation is now the default (\code{engine = "C"}) and
requires approximately 30\% less computation time.\cr
The special setting \code{engine = "C+reduce"} will discard redundant nodes
at (0,0) with zero weight resulting from edges on the base-line
\eqn{x = \alpha} or orthogonal to it.
This extra cleaning is only worth its cost for computationally intensive
functions \code{f} over polygons which really have some edges on the
baseline or parallel to the x-axis.  Note that the old \R
implementation does not have such unset zero nodes and weights.}

\item{plot}{logical indicating if an illustrative plot of the numerical
integration should be produced.}
}
\value{
The approximated value of the integral of \code{f} over
\code{polyregion}.\cr
In the case \code{f = NULL}, only the computed nodes and weights are
returned in a list of length the number of polygons of \code{polyregion},
where each component is a list with \code{nodes} (a numeric matrix with
two columns), \code{weights} (a numeric vector of length
\code{nrow(nodes)}), the rotation \code{angle}, and \code{alpha}.
}
\description{
Product Gauss cubature over polygons as proposed by
Sommariva and Vianello (2007).
}
\examples{
## a function to integrate (here: isotropic zero-mean Gaussian density)
f <- function (s, sigma = 5)
    exp(-rowSums(s^2)/2/sigma^2) / (2*pi*sigma^2)

## a simple polygon as integration domain
hexagon <- list(
    list(x = c(7.33, 7.33, 3, -1.33, -1.33, 3),
         y = c(-0.5, 4.5, 7, 4.5, -0.5, -3))
)

## image of the function and integration domain
plotpolyf(hexagon, f, xlim = c(-8,8), ylim = c(-8,8))

## use a degree of nGQ = 3 and show the corresponding nodes
polyCub.SV(hexagon, f, nGQ = 3, plot = TRUE)

## extract nodes and weights
nw <- polyCub.SV(hexagon, f = NULL, nGQ = 3)[[1]]
nrow(nw$nodes)

## manually apply the cubature rule
sum(nw$weights * f(nw$nodes))

## use an increasing number of nodes
for (nGQ in c(1:5, 10, 20, 60))
    cat(sprintf("nGQ = \%2i: \%.16f\\n", nGQ,
                polyCub.SV(hexagon, f, nGQ = nGQ)))

## polyCub.SV() is the default method used by the polyCub() wrapper
polyCub(hexagon, f, nGQ = 3)  # calls polyCub.SV()


### now using a simple *rectangular* integration domain

rectangle <- list(list(x = c(-1, 7, 7, -1), y = c(-3, -3, 7, 7)))

## try rotation (may improve accuracy)
opar <- par(mfrow = c(1,3))
polyCub.SV(rectangle, f, nGQ = 4, rotation = FALSE, plot = TRUE)
           title(main = "without rotation")
polyCub.SV(rectangle, f, nGQ = 4, rotation = TRUE,  plot = TRUE)
           title(main = "default rotation")
polyCub.SV(rectangle, f, nGQ = 4,
           rotation = list(P = c(0,0), Q = c(2,-3)), plot = TRUE)
           title(main = "custom rotation")
par(opar)

## comparison with cubature::adaptIntegrate()
if (require("cubature")) {
    fc <- function (s, sigma = 5)
        exp(-sum(s^2)/2/sigma^2) / (2*pi*sigma^2)
    adaptIntegrate(f = fc, lowerLimit = c(-1, -3), upperLimit = c(7, 7))
}
}
\references{
Sommariva, A. and Vianello, M. (2007):
Product Gauss cubature over polygons based on Green's integration formula.
\emph{BIT Numerical Mathematics}, \bold{47} (2), 441-453.\cr
DOI-Link: \url{https://doi.org/10.1007/s10543-007-0131-2}
}
\seealso{
Other polyCub-methods: \code{\link{polyCub.exact.Gauss}},
  \code{\link{polyCub.iso}},
  \code{\link{polyCub.midpoint}}, \code{\link{polyCub}}
}
\author{
Sebastian Meyer\cr
These R and C implementations of product Gauss cubature are based on the
original \acronym{MATLAB} implementation \code{polygauss} by Sommariva and
Vianello (2007), which is available under the GNU GPL (>=2) license from
\url{http://www.math.unipd.it/~alvise/software.html}.
}
\concept{polyCub-methods}
\keyword{math}
\keyword{spatial}
