/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Commons", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Stack;
import java.util.Map;
import java.util.HashMap;

import java.io.StringWriter;
import java.io.Writer;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.BufferedInputStream;
import java.io.IOException;

import java.net.URL;
import java.net.URLConnection;

import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.xml.sax.helpers.DefaultHandler;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

/**
 *  Simple class to hold XML document for easy access from Velocity.
 *  
 *  The idea is to be able to access the document directly using
 *      $foo.bar.woogie
 *  access syntax.
 *
 *  Requires nothing more than JAXP (jaxp.jar and crimson.jar)
 *
 *  @author <a href="mailto:geirm@optonline.net">Geir Magnusson Jr.</a>
 *  @version $Id:$
 */
public class XMLEasyBean extends DefaultHandler
{
	/** root of our document tree */
    private XMLEasyBeanNode root = null;

	/** stack used for SAX parsing */
    private Stack  stack = new Stack();
 
	/** user for SAX parsing to avoid character events */
    private StringWriter saxWriter = null;

    private boolean namespaceAware = false;
    private boolean validating = false;
    
    /**
     *   default CTOR
     */
    public XMLEasyBean()
    {
    }
    
    /**
     *  CTOR that allows you to set the 
     *   namespace and validating flags
     */
    public XMLEasyBean( boolean namespace, boolean validate )
    {
        this.namespaceAware = namespace;
        this.validating = validate;
    }
     
	/**
	 *  loads the object with the XML data from the
	 *  passed in URL.  Invokes the SAX parser.
	 *
	 *  @param url location of the repository.xml
	 */
    public void load( URL url )
    {
        try 
        {
            System.out.println("Loading URL : " + url );
            
            URLConnection conn = url.openConnection();
            InputStream is = conn.getInputStream();

            SAXParserFactory factory = SAXParserFactory.newInstance();
            factory.setNamespaceAware(false);
            factory.setValidating(false);
            
            SAXParser parser = factory.newSAXParser();
 
            parser.parse( new BufferedInputStream(is), (DefaultHandler) this);
        }
        catch ( IOException e) 
        {
            System.out.println("IOException: " + e);
            e.printStackTrace();
        }
        catch ( SAXException e) 
        {
            System.out.println("SAXException: " + e);
            e.printStackTrace();
        }
        catch ( Throwable e) 
        {
            System.out.println("Other Exception: " + e);
            e.printStackTrace();
        }
    }
 
    /**
     *  Returns the root of the document tree
     */
    public XMLEasyBeanNode getRoot()
    {
        return root;
    }
   
    /**
     *  start element event handler for SAX.  
     */
    public void startElement( String namespaceURI, String localName, String qName, Attributes atts)
        throws SAXException
    {
        /*
         *  get the top element on the list
         */
		
		XMLEasyBeanNode parent = null;
		XMLEasyBeanNode element = new XMLEasyBeanNode( qName );
		
        if( root == null)
		{
            root = element;
        }
        else
        {
            /*
             * get parent node
             */

            try
            {
                parent = (XMLEasyBeanNode) stack.peek();
            }
            catch(Exception e)
            {}
        }

        /*
         *  everything is a list
         */

        if (parent != null)
        {    
			parent.addChildNode( element);
        }

        /*
         *  now add our attributes
         */

        for (int i = 0; i < atts.getLength(); i++)
        {
            element.setAttribute( atts.getQName(i), atts.getValue(i) );
        }

        /*
         *  now put this element on the stack for later
         */

        stack.push(element);
        
        saxWriter = new StringWriter();
    }
   

    /**
     *  end element event handler for SAX parsing
     */
    public void endElement( String namespaceURI, String localName, String qName)
        throws SAXException
    {
        try
        {
            XMLEasyBeanNode element = (XMLEasyBeanNode) stack.pop();
            
            String s = saxWriter.toString();

            if ( s.length() > 0)
                element.setValue( s );
        }
        catch( Exception e )
        {
            System.out.println("XMLEasyBean.endElement() : " + e );
        }
    }
    
    /**
     *  our characters...
     */
    public void characters( char arr[], int start, int length )
        throws SAXException
    {
        try
        {
            saxWriter.write( arr, start, length );
        }
        catch( Exception e ) {}    
    }
    
    
    public void fatalError (SAXParseException e) 
      throws SAXException 
    {
        System.out.println("XMLEasyBean.fatalError(): " + e);
        throw e;
    }
    
    /**
     *  used for debugging
     */
    public static void main ( String[] args) 
    {
        XMLEasyBean rep = new XMLEasyBean();

        try
        {
            rep.load( new URL("file://" + args[0] ));
        }
        catch( Exception e )
        { 
            System.out.println(e );
        }

        XMLEasyBeanNode node = rep.getRoot();
    }
    

	/**
	 *  small class to hold our node information for
	 *  XML processing
	 */
	public class XMLEasyBeanNode 
	{
		private String name = null;
		private String value = "";

		private HashMap attributes = new HashMap();
		private HashMap children = new HashMap();

		XMLEasyBeanNode( String name )
		{
			this.name = name;
		}
		
		public String toString()
		{
		    return getValue();
		}

        public Object get( String name )
        {
            List l = getChildren( name );
            
            if (l.size() == 1 )
            {
                return l.get(0);
            }
                           
            return l;
        }
           
		void setValue( String val )
		{
			value = val;
		}
		
		String getValue()
		{
			return value;
		}

		void setAttribute( String key, String val )
		{
			attributes.put( key, val );
		}
		
		public String getAttribute( String key )
		{
			return (String) attributes.get( key );
		}

		void addChildNode( XMLEasyBeanNode node )
		{		    
			ArrayList ar = (ArrayList) children.get( node.name );

			if (ar == null)
			{
				ar = new ArrayList();
            }
            
			ar.add( node );

			children.put( node.name, ar );
		}

		ArrayList getChildren( String name )
		{
			return (ArrayList) children.get( name );
		}
				
	}

}





